"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContainerImageBuilder = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ecr_1 = require("aws-cdk-lib/aws-ecr");
const utils_1 = require("../../utils");
const common_1 = require("../common");
const common_2 = require("./common");
const windows_components_1 = require("./windows-components");
const dockerfileTemplate = `FROM {{{ imagebuilder:parentImage }}}
ENV RUNNER_VERSION=___RUNNER_VERSION___
{{{ imagebuilder:environments }}}
{{{ imagebuilder:components }}}`;
/**
 * Image builder recipe for a Docker container image.
 */
class ContainerRecipe extends common_2.ImageBuilderObjectBase {
    constructor(scope, id, props) {
        super(scope, id);
        const name = common_2.uniqueImageBuilderName(this);
        let components = props.components.map(component => {
            return {
                componentArn: component.arn,
            };
        });
        const recipe = new aws_cdk_lib_1.aws_imagebuilder.CfnContainerRecipe(this, 'Recipe', {
            name: name,
            version: this.version('ContainerRecipe', name, {
                platform: props.platform,
                components,
                dockerfileTemplate,
            }),
            parentImage: props.parentImage ?? 'mcr.microsoft.com/windows/servercore:ltsc2019-amd64',
            components,
            containerType: 'DOCKER',
            targetRepository: {
                service: 'ECR',
                repositoryName: props.targetRepository.repositoryName,
            },
            dockerfileTemplateData: props.dockerfileTemplate,
        });
        this.arn = recipe.attrArn;
        this.name = name;
    }
}
/**
 * An image builder that uses AWS Image Builder to build Docker images pre-baked with all the GitHub Actions runner requirements. Builders can be used with runner providers.
 *
 * The CodeBuild builder is better and faster. Only use this one if you have no choice. For example, if you need Windows containers.
 *
 * Each builder re-runs automatically at a set interval to make sure the images contain the latest versions of everything.
 *
 * You can create an instance of this construct to customize the image used to spin-up runners. Some runner providers may require custom components. Check the runner provider documentation. The default components work with CodeBuild and Fargate.
 *
 * For example, to set a specific runner version, rebuild the image every 2 weeks, and add a few packages for the Fargate provider, use:
 *
 * ```
 * const builder = new ContainerImageBuilder(this, 'Builder', {
 *     runnerVersion: RunnerVersion.specific('2.293.0'),
 *     rebuildInterval: Duration.days(14),
 * });
 * new CodeBuildRunner(this, 'CodeBuild provider', {
 *     label: 'custom-codebuild',
 *     imageBuilder: builder,
 * });
 * ```
 */
class ContainerImageBuilder extends common_2.ImageBuilderBase {
    constructor(scope, id, props) {
        super(scope, id, {
            os: props?.os,
            supportedOs: [common_1.Os.WINDOWS],
            architecture: props?.architecture,
            supportedArchitectures: [common_1.Architecture.X86_64],
            instanceType: props?.instanceType,
            vpc: props?.vpc,
            securityGroups: props?.securityGroup ? [props.securityGroup] : props?.securityGroups,
            subnetSelection: props?.subnetSelection,
            logRemovalPolicy: props?.logRemovalPolicy,
            logRetention: props?.logRetention,
            runnerVersion: props?.runnerVersion,
            rebuildInterval: props?.rebuildInterval,
            imageTypeName: 'image',
        });
        this.parentImage = props?.parentImage;
        // create repository that only keeps one tag
        this.repository = new aws_cdk_lib_1.aws_ecr.Repository(this, 'Repository', {
            imageScanOnPush: true,
            imageTagMutability: aws_ecr_1.TagMutability.MUTABLE,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            lifecycleRules: [
                {
                    description: 'Remove all but the latest image',
                    tagStatus: aws_ecr_1.TagStatus.ANY,
                    maxImageCount: 1,
                },
            ],
        });
        // add all basic components
        this.addBaseWindowsComponents();
    }
    addBaseWindowsComponents() {
        this.addComponent(windows_components_1.WindowsComponents.awsCli(this, 'AWS CLI'));
        this.addComponent(windows_components_1.WindowsComponents.githubCli(this, 'GitHub CLI'));
        this.addComponent(windows_components_1.WindowsComponents.git(this, 'git'));
        this.addComponent(windows_components_1.WindowsComponents.githubRunner(this, 'GitHub Actions Runner', this.runnerVersion));
    }
    /**
     * Add a component to be installed before any other components. Useful for required system settings like certificates or proxy settings.
     * @param component
     */
    prependComponent(component) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        if (component.platform != this.platform) {
            throw new Error('Component platform doesn\'t match builder platform');
        }
        this.components = [component].concat(this.components);
    }
    /**
     * Add a component to be installed.
     * @param component
     */
    addComponent(component) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        if (component.platform != this.platform) {
            throw new Error('Component platform doesn\'t match builder platform');
        }
        this.components.push(component);
    }
    /**
     * Add extra trusted certificates. This helps deal with self-signed certificates for GitHub Enterprise Server.
     *
     * All first party Dockerfiles support this. Others may not.
     *
     * @param path path to directory containing a file called certs.pem containing all the required certificates
     */
    addExtraCertificates(path) {
        this.prependComponent(new common_2.ImageBuilderComponent(this, 'Extra Certs', {
            platform: this.platform,
            displayName: 'GitHub Actions Runner',
            description: 'Install latest version of GitHub Actions Runner',
            commands: [
                '$ErrorActionPreference = \'Stop\'',
                'Import-Certificate -FilePath certs\\certs.pem -CertStoreLocation Cert:\\LocalMachine\\Root',
            ],
            assets: [
                {
                    path: 'certs',
                    asset: new aws_cdk_lib_1.aws_s3_assets.Asset(this, 'Extra Certs Asset', { path }),
                },
            ],
        }));
    }
    /**
     * Called by IRunnerProvider to finalize settings and create the image builder.
     */
    bind() {
        if (this.boundImage) {
            return this.boundImage;
        }
        const dist = new aws_cdk_lib_1.aws_imagebuilder.CfnDistributionConfiguration(this, 'Distribution', {
            name: common_2.uniqueImageBuilderName(this),
            description: this.description,
            distributions: [
                {
                    region: aws_cdk_lib_1.Stack.of(this).region,
                    containerDistributionConfiguration: {
                        ContainerTags: ['latest'],
                        TargetRepository: {
                            Service: 'ECR',
                            RepositoryName: this.repository.repositoryName,
                        },
                    },
                },
            ],
        });
        const recipe = new ContainerRecipe(this, 'Container Recipe', {
            platform: this.platform,
            components: this.components,
            targetRepository: this.repository,
            dockerfileTemplate: dockerfileTemplate.replace('___RUNNER_VERSION___', this.runnerVersion.version),
            parentImage: this.parentImage,
        });
        const log = this.createLog(recipe.name);
        const infra = this.createInfrastructure([
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilderECRContainerBuilds'),
        ]);
        const image = this.createImage(infra, dist, log, undefined, recipe.arn);
        this.createPipeline(infra, dist, log, undefined, recipe.arn);
        this.imageCleaner(image, recipe.name);
        this.boundImage = {
            // There are simpler ways to get the ARN, but we want an image object that depends on the newly built image.
            // We want whoever is using this image to automatically wait for Image Builder to finish building before using the image.
            imageRepository: aws_cdk_lib_1.aws_ecr.Repository.fromRepositoryName(this, 'Dependable Image', 
            // we can't use image.attrName because it comes up with upper case
            cdk.Fn.split(':', cdk.Fn.split('/', image.attrImageUri, 2)[1], 2)[0]),
            imageTag: 'latest',
            os: this.os,
            architecture: this.architecture,
            logGroup: log,
            runnerVersion: this.runnerVersion,
        };
        return this.boundImage;
    }
    imageCleaner(image, recipeName) {
        const crHandler = utils_1.BundledNodejsFunction.singleton(this, 'build-image', {
            description: 'Custom resource handler that triggers CodeBuild to build runner images, and cleans-up images on deletion',
            timeout: cdk.Duration.minutes(3),
        });
        const policy = new aws_cdk_lib_1.aws_iam.Policy(this, 'CR Policy', {
            statements: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['ecr:BatchDeleteImage', 'ecr:ListImages'],
                    resources: [this.repository.repositoryArn],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['imagebuilder:ListImages', 'imagebuilder:ListImageBuildVersions', 'imagebuilder:DeleteImage'],
                    resources: ['*'],
                }),
            ],
        });
        crHandler.role?.attachInlinePolicy(policy);
        const cr = new aws_cdk_lib_1.CustomResource(this, 'Deleter', {
            serviceToken: crHandler.functionArn,
            resourceType: 'Custom::ImageDeleter',
            properties: {
                RepoName: this.repository.repositoryName,
                ImageBuilderName: recipeName,
                DeleteOnly: true,
            },
        });
        // add dependencies to make sure resources are there when we need them
        cr.node.addDependency(image);
        cr.node.addDependency(policy);
        cr.node.addDependency(crHandler);
        return cr;
    }
}
exports.ContainerImageBuilder = ContainerImageBuilder;
_a = JSII_RTTI_SYMBOL_1;
ContainerImageBuilder[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.ContainerImageBuilder", version: "0.7.0" };
//# sourceMappingURL=data:application/json;base64,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