# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['async_btree']

package_data = \
{'': ['*']}

extras_require = \
{'curio': ['curio>=1,<2']}

setup_kwargs = {
    'name': 'async-btree',
    'version': '1.1.0',
    'description': 'Async behavior tree',
    'long_description': '# Async Behaviour Tree for Python\n\n\n[![Unix Build Status](https://img.shields.io/travis/geronimo-iia/async-btree/master.svg?label=unix)](https://travis-ci.com/geronimo-iia/async-btree)\n[![Coverage Status](https://img.shields.io/coveralls/geronimo-iia/async-btree/master.svg)](https://coveralls.io/r/geronimo-iia/async-btree)\n[![Codacy Badge](https://api.codacy.com/project/badge/Grade/fe669a02b4aa46b5b1faf619ba2bf382)](https://www.codacy.com/app/geronimo-iia/async-btree?utm_source=github.com&amp;utm_medium=referral&amp;utm_content=geronimo-iia/async-btree&amp;utm_campaign=Badge_Grade)\n[![Scrutinizer Code Quality](https://img.shields.io/scrutinizer/g/geronimo-iia/async-btree.svg)](https://scrutinizer-ci.com/g/geronimo-iia/async-btree/?branch=master)\n[![PyPI Version](https://img.shields.io/pypi/v/async-btree.svg)](https://pypi.org/project/async-btree)\n[![PyPI License](https://img.shields.io/pypi/l/async-btree.svg)](https://pypi.org/project/async-btree)\n\nVersions following [Semantic Versioning](https://semver.org/)\n\nSee [documentation](https://geronimo-iia.github.io/async-btree).\n\n\n## Overview\n\n\n### What\'s a behavior tree ?\n\n> Unlike a Finite State Machine, a Behaviour Tree is a tree of hierarchical nodes that controls the flow of decision and the execution of "tasks" or, as we will call them further, "Actions".\n> -- <cite>[behaviortree](https://www.behaviortree.dev/bt_basics/)</cite>\n\nIf your new (or not) about behavior tree, you could spend some time on this few links:\n\n- [Behavior trees for AI: How they work](https://www.gamasutra.com/blogs/ChrisSimpson/20140717/221339/Behavior_trees_for_AI_How_they_work.php) by Chris Simpson\n- [Introduction to BTs](https://www.behaviortree.dev/bt_basics/)\n\nFew implementation libraries:\n\n- [task_behavior_engine](https://github.com/ToyotaResearchInstitute/task_behavior_engine) A behavior tree based task engine written in Python\n- [pi_trees](https://github.com/pirobot/pi_trees/) a Python/ROS library for implementing Behavior Trees\n- [pr_behavior_tree](https://github.com/personalrobotics/pr_behavior_tree) A simple python behavior tree library based on coroutines\n- [btsk](https://github.com/aigamedev/btsk) Behavior Tree Starter Kit\n- [behave](https://github.com/fuchen/behave) A behavior tree implementation in Python\n\n\n### Why another library so ?\n\n__SIMPLICITY__\n\nWhen you study behavior tree implementation, reactive node, dynamic change, runtime execution, etc ...\nAt a moment you\'re build more or less something that mimic an evaluator \'eval/apply\' or a compilator, with a complex hierachical set of class.\n\nAll complexity came with internal state management, using tree of blackboard to avoid global variable, multithreading issue, maybe few callback etc ...\n\nThis break the simplicity and beauty of your initial design.\n\nWhat I find usefull with behavior tree:\n\n- clarity of expression\n- node tree representation\n- possibility to reuse behavior\n- add external measure to dynamicaly change a behavior, a first step on observable pattern...\n\nAs I\'ve used OOP for years (very long time), I will try to avoid class tree and prefer using the power of functionnal programming to obtain what I want: add metadata on a sematic construction, deal with closure, use function in parameters or in return value...\n\nAnd a last reason, more personal, it that i would explore python expressivity.\n\n__SO HOW ?__\n\nIn this module, I purpose you to use the concept of coroutines, and their mecanisms to manage the execution flow.\nBy this way:\n\n- we reuse simple language idiom to manage state, parameter, etc\n- no design constraint on action implementation\n- most of language build block could be reused\n\nYou could build expression like this:\n\n```python\n\nasync def a_func():\n    """A great function"""\n    return "a"\n\nasync def b_decorator(child_value, other=""):\n    """A great decorator..."""\n    return f"b{child_value}{other}"\n\nassert run(decorate(a_func, b_decorator)) == "ba"\n\n```\nThis expression apply ```b_decorator``` on function ```a_func```. \nNote that ```decorate(a_func, b_decorator)``` is not an async function, only action, or condition are async function.\n\n\nFew guidelines of this implementation:\n\n- In order to mimic all NodeStatus (success, failure, running), I replace this by truthy/falsy meaning of evaluation value.\n  A special dedicated exception decorate standard exception in order to give them a Falsy meaning (`ControlFlowException`).\n  By default, exception are raised like happen usually until you catch them.\n- Blackboard pattern, act as a manager of context variable for behavior tree.\n  With python 3, please... simply use [contextvars](https://docs.python.org/3/library/contextvars.html) !\n- In order to be able to build a sematic tree, I\'ve introduce a metadata tuple added on function implementation.\n\nThe rest is just implementation details..\n\n\n\nA little note:\n\n> You should not use this until you\'re ready to think about what you\'re doing :)\n\n\n### Note about \'async\' framework\n\nAs we use async function as underlaying mechanism to manage the execution flow, the standard library asyncio is pretty fine.\nBut, (always a but somewhere isn\'t it...), you should read this [amazing blog post}(https://vorpus.org/blog/some-thoughts-on-asynchronous-api-design-in-a-post-asyncawait-world/) by Nathaniel J. Smith.\nAnd next study [curio](https://github.com/dabeaz/curio) framework in deep.\n\nAs curio say:\n> Don\'t Use Curio if You\'re Allergic to Curio\n\nPersonaly, after few time of testing and reading curio code, I\'m pretty addict.\n\n## Installation\n\nInstall this library directly into an activated virtual environment:\n\n```text\n$ pip install async-btree\n```\n\nor add it to your [Poetry](https://poetry.eustace.io/) project:\n\n```text\n$ poetry add async-btree\n```\n\n## Usage\n\nAfter installation, the package can imported:\n\n```text\n$ python\n>>> import async_btree\n>>> async_btree.__version__\n```\n\nSee [API Reference documentation](https://geronimo-iia.github.io/async-btree).\n\n\nWith this framework, you didn\'t find any configuration file, no Xml, no json, no yaml.\n\nThe main reason (oriented and personal point of view) is that you did not need to introduce an extra level of abtraction \nto declare a composition of functions. I think it\'s true for most of main use case (except using an editor to wrote behaviour tree for example).\n\nSo "If you wrote your function with python, wrote composition in python"... \n_(remember that you did not need XML to do SQL, just write good sql...)_\n\n\nSo, the goal is to:\n - define your business function wich implements actions or conditions, with all test case that you wish/need\n - compose them using those provided by this framework like ```sequence```, ```selector```, ...\n - use them as it is or create a well define python module to reuse them\n\n\nWanna style have an abtract tree of our behaviour tree ?\n\nFunctions from async-btree build an abstract tree for you. \nIf you lookup in code, you should see an annotation "node_metadata" on internal implementation. \nThis decorator add basic information like function name, parameters, and children relation ship.\n\nThis abstract tree can be retreived and stringified with ```analyze``` and ```stringify_analyze```.\nHere the profile:\n\n```python\n  def analyze(target: CallableFunction) -> Node: # here we have our "abtract tree code"\n    ...\n```\n\nFor example:\n\n```python\n\n# your behaviour tree, or a sub tree:\nmy_func = alias(child=repeat_until(child=action(hello), condition=success_until_zero), name="btree_1")\n\n# retrieve meta information and build a Node tree\nabstract_tree_tree_1 = analyze(my_func) \n\n# output the tree:\nprint(stringify_analyze(abstract_tree_tree_1))\n```\n\nThis should print:\n\n```text\n --> btree_1:\n     --(child)--> repeat_until:\n         --(condition)--> success_until_zero:\n         --(child)--> action:\n                      target: hello\n```\n\n\nNote about action and condition method:\n\n - you could use sync or async function\n - you could specify a return value with SUCCESS or FAILURE\n - function with no return value will be evaluated as FAILURE until you decorate them with a `always_success`or `always_failure`\n\nSee this [example/tutorial_1.py](https://raw.githubusercontent.com/geronimo-iia/async-btree/master/examples/tutorial_1.py) for more information.',
    'author': 'Jerome Guibert',
    'author_email': 'jguibert@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://pypi.org/project/async_btree',
    'packages': packages,
    'package_data': package_data,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
