/** @file common-api.h
 *  @brief contam common functions API.
 *  @author Brian J. Polidoro (NIST)
 *  @author W. Stuart Dols (NIST)
 *  @date 2022-10-20
 *
 *  Defines for APIs for log file, file reader, getlasterror, name check and common state
 */
#ifndef COMMON_API_H
#define COMMON_API_H

/**
 * @defgroup RETURN_GROUP Function Return Group
 *
 * @{
 */

 /** Function passed */
#define FCN_PASS 0
  /** Function failed due to no state passed in */
#define FCN_NOSTATE -1
  /** Function failed, use cciGetLastError function to get an error message */
#define FCN_FAIL 1

 /** @} */

 /**
 * @defgroup READ_MODE_GROUP Reader Mode Group
 *
 * @{
 */

 /** Read the next word mode */
#define NEXT_WORD_MODE 0
 /** Read first word on the next line mode */
#define FIRST_WORD_NEXT_LINE 1
 /** Read the rest of the current line mode */
#define REST_OF_LINE_MODE 2
 /** Read all of the next line mode */
#define ALL_OF_NEXT_LINE 3

 /** @} */

 /** @brief Used to get a state for use with common APIs.
  *
  *  @return A pointer to a common state.
  */
void* cciGetNewCommonState();

/** @brief Used to delete a state for common APIs. Delete a state when finished with it to avoid a memory leak.
 *
 *  @param[in] commonState This is a pointer to a state for the common APIs.
 *  @return See @ref RETURN_GROUP for return values
 */
int   cciDeleteCommonState(void* commonState);

/** @brief Used to open a file to read with the CONTAM reader.
 *
 *  @param[in] commonState This is a pointer to a state for the common APIs.
 *  @param[in] filePath    This is a path to the file to read.
 *  @return See @ref RETURN_GROUP for return values
 */
int   cciOpenFileToRead(void* commonState, char* filePath);

/** @brief Used to close a file that was opened with the cciOpenFileToRead API.
 *
 *  @param[in] commonState This is a pointer to a state for the common APIs.
 *  @return See @ref RETURN_GROUP for return values
 */
int   cciCloseReadFile(void* commonState);

/** @brief Used to read a string from the file that was opened with the cciOpenFileToRead API.
 *
 *  @param[in] commonState       This is a pointer to a state for the common APIs.
 *  @param[in] mode              This is which mode to read. See READ_MODE_GROUP.
 *  @param[in,out] stringBuffer  This is a char buffer where the content read will be put.
 *  @param[in] stringBufferSize  This is the size of the stringBuffer in bytes.
 *  @return See @ref RETURN_GROUP for return values
 */
int   cciReadString(void* commonState, short mode, char* stringBuffer, int stringBufferSize);

/** @brief Used to read an integer from the file that was opened with the cciOpenFileToRead API.
 *
 *  @param[in] commonState       This is a pointer to a state for the common APIs.
 *  @param[in] mode              This is which mode to read. See READ_MODE_GROUP.
 *  @return the integer read from the file
 */
int   cciReadInt(void* commonState, short mode);

/** @brief Used to read a floating point number from the file that was opened with the cciOpenFileToRead API.
 *
 *  @param[in] commonState       This is a pointer to a state for the common APIs.
 *  @param[in] mode              This is which mode to read. See READ_MODE_GROUP.
 *  @return the float read from the file
 */
float cciReadFloat(void* commonState, short mode);

/** @brief Used to retrieve the last error generated by the library.
 *
 *  @param[in] commonState       This is a pointer to a state for the common APIs.
 *  @param[in,out] errorMessage  This a char buffer where the error message will be placed.
 *  @param[in] errorMessageSize  This the size of the errorMessageBuffer.
 *  @return 0 for succes, -1 for NULL common state, 1 for NULL pointer for errorMessage buffer, 2 for insufficient buffer size
 */
int   cciGetLastError(void* commonState, char* errorMessage, unsigned int errorMessageSize);

/** @brief Used to check if a name is valid for use in CONTAM.
 *
 *  @param[in] commonState       This is a pointer to a state for the common APIs.
 *  @param[in] name              This is the name to check.
 *  @return See @ref RETURN_GROUP for return values
 */
int   cciCheckValidName(void* commonState, char* name);

/** @brief Used to open the log file.
 *
 *  @param[in] commonState       This is a pointer to a state for the common APIs.
 *  @param[in] logfilePath       This is the path to use for a log file.
 *  @return See @ref RETURN_GROUP for return values
 */
int   cciOpenLogFile(void* commonState, const char* const logfilePath);

/** @brief Used to close the log file.
 *
 *  @param[in] commonState       This is a pointer to a state for the common APIs.
 *  @return See @ref RETURN_GROUP for return values
 */
int   cciCloseLogFile(void* commonState);

/** @brief Used to print text to the log file. This function is used like a printf fuction.
 *
 *  @param[in] commonState       This is a pointer to a state for the common APIs.
 *  @param[in] format            This is char array that contains text to be output to the log file.
 *  @param[in] ...               Depending on the format string, the function may expect a sequence of additional arguments, 
 *                               each containing a value to be used to replace a format specifier in the format string.
 *                               There should be at least as many of these arguments as the number of values specified in the format specifiers. 
 *                               Additional arguments are ignored by the function.
 *  @return See @ref RETURN_GROUP for return values
 */
int   cciPrintToLog(void* commonState, const char* format, ...);

#endif