"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobDefinition = exports.PlatformCapabilities = exports.LogDriver = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const batch_generated_1 = require("./batch.generated");
const job_definition_image_config_1 = require("./job-definition-image-config");
/**
 * (experimental) The log driver to use for the container.
 *
 * @experimental
 */
var LogDriver;
(function (LogDriver) {
    LogDriver["AWSLOGS"] = "awslogs";
    LogDriver["FLUENTD"] = "fluentd";
    LogDriver["GELF"] = "gelf";
    LogDriver["JOURNALD"] = "journald";
    LogDriver["LOGENTRIES"] = "logentries";
    LogDriver["JSON_FILE"] = "json-file";
    LogDriver["SPLUNK"] = "splunk";
    LogDriver["SYSLOG"] = "syslog";
})(LogDriver = exports.LogDriver || (exports.LogDriver = {}));
/**
 * (experimental) Platform capabilities.
 *
 * @experimental
 */
var PlatformCapabilities;
(function (PlatformCapabilities) {
    PlatformCapabilities["EC2"] = "EC2";
    PlatformCapabilities["FARGATE"] = "FARGATE";
})(PlatformCapabilities = exports.PlatformCapabilities || (exports.PlatformCapabilities = {}));
/**
 * (experimental) Batch Job Definition.
 *
 * Defines a batch job definition to execute a specific batch job.
 *
 * @experimental
 */
class JobDefinition extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id, {
            physicalName: props.jobDefinitionName,
        });
        this.validateProps(props);
        this.imageConfig = new job_definition_image_config_1.JobDefinitionImageConfig(this, props.container);
        const isFargate = !!((_b = props.platformCapabilities) === null || _b === void 0 ? void 0 : _b.includes(PlatformCapabilities.FARGATE));
        const jobDef = new batch_generated_1.CfnJobDefinition(this, 'Resource', {
            jobDefinitionName: props.jobDefinitionName,
            containerProperties: this.buildJobContainer(props.container, isFargate),
            type: 'container',
            nodeProperties: props.nodeProps
                ? {
                    mainNode: props.nodeProps.mainNode,
                    nodeRangeProperties: this.buildNodeRangeProps(props.nodeProps, isFargate),
                    numNodes: props.nodeProps.count,
                }
                : undefined,
            parameters: props.parameters,
            retryStrategy: {
                attempts: props.retryAttempts || 1,
            },
            timeout: {
                attemptDurationSeconds: props.timeout ? props.timeout.toSeconds() : undefined,
            },
            platformCapabilities: (_c = props.platformCapabilities) !== null && _c !== void 0 ? _c : [PlatformCapabilities.EC2],
        });
        this.jobDefinitionArn = this.getResourceArnAttribute(jobDef.ref, {
            service: 'batch',
            resource: 'job-definition',
            resourceName: this.physicalName,
        });
        this.jobDefinitionName = this.getResourceNameAttribute(jobDef.ref);
    }
    /**
     * (experimental) Imports an existing batch job definition by its amazon resource name.
     *
     * @experimental
     */
    static fromJobDefinitionArn(scope, id, jobDefinitionArn) {
        const stack = core_1.Stack.of(scope);
        const jobDefName = stack.parseArn(jobDefinitionArn).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.jobDefinitionArn = jobDefinitionArn;
                this.jobDefinitionName = jobDefName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Imports an existing batch job definition by its name.
     *
     * If name is specified without a revision then the latest active revision is used.
     *
     * @experimental
     */
    static fromJobDefinitionName(scope, id, jobDefinitionName) {
        const stack = core_1.Stack.of(scope);
        const jobDefArn = stack.formatArn({
            service: 'batch',
            resource: 'job-definition',
            sep: '/',
            resourceName: jobDefinitionName,
        });
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.jobDefinitionArn = jobDefArn;
                this.jobDefinitionName = jobDefinitionName;
            }
        }
        return new Import(scope, id);
    }
    deserializeEnvVariables(env) {
        const vars = new Array();
        if (env === undefined) {
            return undefined;
        }
        Object.keys(env).map((name) => {
            vars.push({ name, value: env[name] });
        });
        return vars;
    }
    /**
     * Validates the properties provided for a new job definition.
     */
    validateProps(props) {
        if (props === undefined) {
            return;
        }
        if (props.platformCapabilities !== undefined && props.platformCapabilities.includes(PlatformCapabilities.FARGATE)
            && props.container.executionRole === undefined) {
            throw new Error('Fargate job must have executionRole set');
        }
        if (props.platformCapabilities !== undefined && props.platformCapabilities.includes(PlatformCapabilities.FARGATE)
            && props.container.gpuCount !== undefined) {
            throw new Error('Fargate job must not have gpuCount set');
        }
        if ((props.platformCapabilities === undefined || props.platformCapabilities.includes(PlatformCapabilities.EC2))
            && props.container.assignPublicIp !== undefined) {
            throw new Error('EC2 job must not have assignPublicIp set');
        }
    }
    buildJobContainer(container, isFargate) {
        if (container === undefined) {
            return undefined;
        }
        return {
            command: container.command,
            environment: this.deserializeEnvVariables(container.environment),
            image: this.imageConfig.imageName,
            instanceType: container.instanceType && container.instanceType.toString(),
            jobRoleArn: container.jobRole && container.jobRole.roleArn,
            executionRoleArn: container.executionRole && container.executionRole.roleArn,
            linuxParameters: container.linuxParams
                ? { devices: container.linuxParams.renderLinuxParameters().devices }
                : undefined,
            logConfiguration: container.logConfiguration ? {
                logDriver: container.logConfiguration.logDriver,
                options: container.logConfiguration.options,
                secretOptions: container.logConfiguration.secretOptions
                    ? this.buildLogConfigurationSecretOptions(container.logConfiguration.secretOptions)
                    : undefined,
            } : undefined,
            mountPoints: container.mountPoints,
            privileged: container.privileged || false,
            networkConfiguration: container.assignPublicIp ? {
                assignPublicIp: container.assignPublicIp ? 'ENABLED' : 'DISABLED',
            } : undefined,
            readonlyRootFilesystem: container.readOnly || false,
            ulimits: container.ulimits,
            user: container.user,
            volumes: container.volumes,
            fargatePlatformConfiguration: container.platformVersion ? {
                platformVersion: container.platformVersion,
            } : undefined,
            resourceRequirements: [
                { type: 'VCPU', value: String(container.vcpus || (isFargate ? 0.25 : 1)) },
                { type: 'MEMORY', value: String(container.memoryLimitMiB || (isFargate ? 512 : 4)) },
            ].concat(container.gpuCount ? [{ type: 'GPU', value: String(container.gpuCount) }] : []),
        };
    }
    buildNodeRangeProps(multiNodeProps, isFargate) {
        const rangeProps = new Array();
        for (const prop of multiNodeProps.rangeProps) {
            rangeProps.push({
                container: this.buildJobContainer(prop.container, isFargate),
                targetNodes: `${prop.fromNodeIndex || 0}:${prop.toNodeIndex || multiNodeProps.count}`,
            });
        }
        return rangeProps;
    }
    buildLogConfigurationSecretOptions(secretOptions) {
        return secretOptions.map(secretOption => {
            return {
                name: secretOption.optionName,
                valueFrom: secretOption.secretArn,
            };
        });
    }
}
exports.JobDefinition = JobDefinition;
_a = JSII_RTTI_SYMBOL_1;
JobDefinition[_a] = { fqn: "@aws-cdk/aws-batch.JobDefinition", version: "1.124.0" };
//# sourceMappingURL=data:application/json;base64,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