"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToKinesisStreams = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const core_1 = require("@aws-cdk/core");
/**
 * @summary The ApiGatewayToKinesisStreams class.
 */
class ApiGatewayToKinesisStreams extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the ApiGatewayToKinesisStreams class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {ApiGatewayToKinesisStreamsProps} props - user provided props for the construct.
     * @since 1.62.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        defaults.CheckProps(props);
        // Setup the Kinesis stream
        this.kinesisStream = defaults.buildKinesisStream(scope, {
            existingStreamObj: props.existingStreamObj,
            kinesisStreamProps: props.kinesisStreamProps
        });
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole, this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, props.apiGatewayProps, props.logGroupProps);
        // Setup the API Gateway role
        this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
        });
        this.kinesisStream.grantWrite(this.apiGatewayRole);
        // Setup API Gateway methods
        const requestValidator = this.apiGateway.addRequestValidator('request-validator', {
            requestValidatorName: 'request-body-validator',
            validateRequestBody: true
        });
        // PutRecord
        const putRecordResource = this.apiGateway.root.addResource('record');
        defaults.addProxyMethodToApiResource({
            service: 'kinesis',
            action: 'PutRecord',
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'POST',
            apiResource: putRecordResource,
            requestTemplate: this.getPutRecordTemplate(props.putRecordRequestTemplate),
            contentType: "'x-amz-json-1.1'",
            requestValidator,
            requestModel: { 'application/json': this.getPutRecordModel(props.putRecordRequestModel) }
        });
        // PutRecords
        const putRecordsResource = this.apiGateway.root.addResource('records');
        defaults.addProxyMethodToApiResource({
            service: 'kinesis',
            action: 'PutRecords',
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'POST',
            apiResource: putRecordsResource,
            requestTemplate: this.getPutRecordsTemplate(props.putRecordsRequestTemplate),
            contentType: "'x-amz-json-1.1'",
            requestValidator,
            requestModel: { 'application/json': this.getPutRecordsModel(props.putRecordsRequestModel) }
        });
        if (props.createCloudWatchAlarms === undefined || props.createCloudWatchAlarms) {
            // Deploy best practices CW Alarms for Kinesis Stream
            this.cloudwatchAlarms = defaults.buildKinesisStreamCWAlarms(this);
        }
    }
    getPutRecordTemplate(putRecordTemplate) {
        if (putRecordTemplate !== undefined) {
            return putRecordTemplate.replace("${StreamName}", this.kinesisStream.streamName);
        }
        return `{ "StreamName": "${this.kinesisStream.streamName}", "Data": "$util.base64Encode($input.json('$.data'))", "PartitionKey": "$input.path('$.partitionKey')" }`;
    }
    getPutRecordModel(putRecordModel) {
        let modelProps;
        if (putRecordModel !== undefined) {
            modelProps = putRecordModel;
        }
        else {
            modelProps = {
                contentType: 'application/json',
                modelName: 'PutRecordModel',
                description: 'PutRecord proxy single-record payload',
                schema: {
                    schema: api.JsonSchemaVersion.DRAFT4,
                    title: 'PutRecord proxy single-record payload',
                    type: api.JsonSchemaType.OBJECT,
                    required: ['data', 'partitionKey'],
                    properties: {
                        data: { type: api.JsonSchemaType.STRING },
                        partitionKey: { type: api.JsonSchemaType.STRING }
                    }
                }
            };
        }
        return this.apiGateway.addModel('PutRecordModel', modelProps);
    }
    getPutRecordsTemplate(putRecordsTemplate) {
        if (putRecordsTemplate !== undefined) {
            return putRecordsTemplate.replace("${StreamName}", this.kinesisStream.streamName);
        }
        return `{ "StreamName": "${this.kinesisStream.streamName}", "Records": [ #foreach($elem in $input.path('$.records')) { "Data": "$util.base64Encode($elem.data)", "PartitionKey": "$elem.partitionKey"}#if($foreach.hasNext),#end #end ] }`;
    }
    getPutRecordsModel(putRecordsModel) {
        let modelProps;
        if (putRecordsModel !== undefined) {
            modelProps = putRecordsModel;
        }
        else {
            modelProps = {
                contentType: 'application/json',
                modelName: 'PutRecordsModel',
                description: 'PutRecords proxy payload data',
                schema: {
                    schema: api.JsonSchemaVersion.DRAFT4,
                    title: 'PutRecords proxy payload data',
                    type: api.JsonSchemaType.OBJECT,
                    required: ['records'],
                    properties: {
                        records: {
                            type: api.JsonSchemaType.ARRAY,
                            items: {
                                type: api.JsonSchemaType.OBJECT,
                                required: ['data', 'partitionKey'],
                                properties: {
                                    data: { type: api.JsonSchemaType.STRING },
                                    partitionKey: { type: api.JsonSchemaType.STRING }
                                }
                            }
                        }
                    }
                }
            };
        }
        return this.apiGateway.addModel('PutRecordsModel', modelProps);
    }
}
exports.ApiGatewayToKinesisStreams = ApiGatewayToKinesisStreams;
_a = JSII_RTTI_SYMBOL_1;
ApiGatewayToKinesisStreams[_a] = { fqn: "@aws-solutions-constructs/aws-apigateway-kinesisstreams.ApiGatewayToKinesisStreams", version: "1.168.0" };
//# sourceMappingURL=data:application/json;base64,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