"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Model = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const cdk = require("aws-cdk-lib");
const aws_sagemaker_1 = require("aws-cdk-lib/aws-sagemaker");
class ModelBase extends cdk.Resource {
    /**
     * An accessor for the Connections object that will fail if this Model does not have a VPC
     * configured.
     */
    get connections() {
        if (!this._connections) {
            throw new Error('Cannot manage network access without configuring a VPC');
        }
        return this._connections;
    }
    /**
     * Adds a statement to the IAM role assumed by the instance.
     */
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPolicy(statement);
    }
}
/**
 * Defines a SageMaker Model.
 */
class Model extends ModelBase {
    /**
     * Imports a Model defined either outside the CDK or in a different CDK stack.
     * @param scope the Construct scope.
     * @param id the resource id.
     * @param modelArn the ARN of the model.
     */
    static fromModelArn(scope, id, modelArn) {
        return Model.fromModelAttributes(scope, id, { modelArn });
    }
    /**
     * Imports a Model defined either outside the CDK or in a different CDK stack.
     * @param scope the Construct scope.
     * @param id the resource id.
     * @param modelName the name of the model.
     */
    static fromModelName(scope, id, modelName) {
        const modelArn = cdk.Stack.of(scope).formatArn({
            service: 'sagemaker',
            resource: 'model',
            resourceName: modelName,
        });
        return Model.fromModelAttributes(scope, id, { modelArn });
    }
    /**
     * Imports a Model defined either outside the CDK or in a different CDK stack.
     * @param scope the Construct scope.
     * @param id the resource id.
     * @param attrs the attributes of the model to import.
     */
    static fromModelAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_sagemaker_alpha_ModelAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromModelAttributes);
            }
            throw error;
        }
        const modelArn = attrs.modelArn;
        const modelName = cdk.Stack.of(scope).splitArn(modelArn, cdk.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        const role = attrs.role;
        class Import extends ModelBase {
            constructor(s, i) {
                super(s, i, {
                    environmentFromArn: attrs.modelArn,
                });
                this.modelArn = modelArn;
                this.modelName = modelName;
                this.role = role;
                this.grantPrincipal = role || new iam.UnknownPrincipal({ resource: this });
                if (attrs.securityGroups) {
                    this._connections = new ec2.Connections({
                        securityGroups: attrs.securityGroups,
                    });
                }
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.modelName,
        });
        this.containers = [];
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_sagemaker_alpha_ModelProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Model);
            }
            throw error;
        }
        this._connections = this.configureNetworking(props);
        this.subnets = (props.vpc) ? props.vpc.selectSubnets(props.vpcSubnets) : undefined;
        // set the sagemaker role or create new one
        this.role = props.role || this.createSageMakerRole();
        this.grantPrincipal = this.role;
        (props.containers || []).map(c => this.addContainer(c));
        const model = new aws_sagemaker_1.CfnModel(this, 'Model', {
            executionRoleArn: this.role.roleArn,
            modelName: this.physicalName,
            primaryContainer: cdk.Lazy.any({ produce: () => this.renderPrimaryContainer() }),
            vpcConfig: cdk.Lazy.any({ produce: () => this.renderVpcConfig() }),
            containers: cdk.Lazy.any({ produce: () => this.renderContainers() }),
        });
        this.modelName = this.getResourceNameAttribute(model.attrModelName);
        this.modelArn = this.getResourceArnAttribute(model.ref, {
            service: 'sagemaker',
            resource: 'model',
            resourceName: this.physicalName,
        });
        /*
         * SageMaker model creation will fail if the model's execution role does not have read access to
         * its model data in S3. Since the CDK uses a separate AWS::IAM::Policy CloudFormation resource
         * to attach inline policies to IAM roles, the following line ensures that the role and its
         * AWS::IAM::Policy resource are deployed prior to model creation.
         */
        model.node.addDependency(this.role);
    }
    /**
     * Add containers to the model.
     *
     * @param container The container definition to add.
     */
    addContainer(container) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_sagemaker_alpha_ContainerDefinition(container);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addContainer);
            }
            throw error;
        }
        this.containers.push(this.renderContainer(container));
    }
    validateContainers() {
        // validate number of containers
        if (this.containers.length < 1) {
            throw new Error('Must configure at least 1 container for model');
        }
        else if (this.containers.length > 15) {
            throw new Error('Cannot have more than 15 containers in inference pipeline');
        }
    }
    renderPrimaryContainer() {
        return (this.containers.length === 1) ? this.containers[0] : undefined;
    }
    renderContainers() {
        this.validateContainers();
        return (this.containers.length === 1) ? undefined : this.containers;
    }
    renderContainer(container) {
        return {
            image: container.image.bind(this, this).imageName,
            containerHostname: container.containerHostname,
            environment: container.environment,
            modelDataUrl: container.modelData ? container.modelData.bind(this, this).uri : undefined,
        };
    }
    configureNetworking(props) {
        if ((props.securityGroups || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error('Cannot configure \'securityGroups\' or \'allowAllOutbound\' without configuring a VPC');
        }
        if (!props.vpc) {
            return undefined;
        }
        if ((props.securityGroups && props.securityGroups.length > 0) && props.allowAllOutbound !== undefined) {
            throw new Error('Configure \'allowAllOutbound\' directly on the supplied SecurityGroups');
        }
        let securityGroups;
        if (props.securityGroups && props.securityGroups.length > 0) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                allowAllOutbound: props.allowAllOutbound,
            });
            securityGroups = [securityGroup];
        }
        return new ec2.Connections({ securityGroups });
    }
    renderVpcConfig() {
        if (!this._connections) {
            return undefined;
        }
        return {
            subnets: this.subnets.subnetIds,
            securityGroupIds: this.connections.securityGroups.map(s => s.securityGroupId),
        };
    }
    createSageMakerRole() {
        const sagemakerRole = new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
        });
        // Grant SageMaker FullAccess
        sagemakerRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess'));
        return sagemakerRole;
    }
}
_a = JSII_RTTI_SYMBOL_1;
Model[_a] = { fqn: "@aws-cdk/aws-sagemaker-alpha.Model", version: "2.79.1-alpha.0" };
exports.Model = Model;
//# sourceMappingURL=data:application/json;base64,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