"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ModelData = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const assets = require("aws-cdk-lib/aws-s3-assets");
const util_1 = require("./private/util");
// The only supported extension for local asset model data
// https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-sagemaker-model-containerdefinition.html#cfn-sagemaker-model-containerdefinition-modeldataurl
const ARTIFACT_EXTENSION = '.tar.gz';
/**
 * Model data represents the source of model artifacts, which will ultimately be loaded from an S3
 * location.
 */
class ModelData {
    /**
     * Constructs model data which is already available within S3.
     * @param bucket The S3 bucket within which the model artifacts are stored
     * @param objectKey The S3 object key at which the model artifacts are stored
     */
    static fromBucket(bucket, objectKey) {
        return new S3ModelData(bucket, objectKey);
    }
    /**
     * Constructs model data that will be uploaded to S3 as part of the CDK app deployment.
     * @param path The local path to a model artifact file as a gzipped tar file
     * @param options The options to further configure the selected asset
     */
    static fromAsset(path, options = {}) {
        return new AssetModelData(path, options);
    }
}
_a = JSII_RTTI_SYMBOL_1;
ModelData[_a] = { fqn: "@aws-cdk/aws-sagemaker-alpha.ModelData", version: "2.79.1-alpha.0" };
exports.ModelData = ModelData;
class S3ModelData extends ModelData {
    constructor(bucket, objectKey) {
        super();
        this.bucket = bucket;
        this.objectKey = objectKey;
    }
    bind(_scope, model) {
        this.bucket.grantRead(model);
        return {
            uri: this.bucket.urlForObject(this.objectKey),
        };
    }
}
class AssetModelData extends ModelData {
    constructor(path, options) {
        super();
        this.path = path;
        this.options = options;
        if (!path.toLowerCase().endsWith(ARTIFACT_EXTENSION)) {
            throw new Error(`Asset must be a gzipped tar file with extension ${ARTIFACT_EXTENSION} (${this.path})`);
        }
    }
    bind(scope, model) {
        // Retain the first instantiation of this asset
        if (!this.asset) {
            this.asset = new assets.Asset(scope, `ModelData${(0, util_1.hashcode)(this.path)}`, {
                path: this.path,
                ...this.options,
            });
        }
        this.asset.grantRead(model);
        return {
            uri: this.asset.httpUrl,
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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