"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EndpointConfig = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const os_1 = require("os");
const cdk = require("aws-cdk-lib");
const instance_type_1 = require("./instance-type");
const util_1 = require("./private/util");
const aws_sagemaker_1 = require("aws-cdk-lib/aws-sagemaker");
/**
 * Defines a SageMaker EndpointConfig.
 */
class EndpointConfig extends cdk.Resource {
    /**
     * Imports an EndpointConfig defined either outside the CDK or in a different CDK stack.
     * @param scope the Construct scope.
     * @param id the resource id.
     * @param endpointConfigArn the ARN of the endpoint configuration.
     */
    static fromEndpointConfigArn(scope, id, endpointConfigArn) {
        const endpointConfigName = cdk.Stack.of(scope).splitArn(endpointConfigArn, cdk.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.endpointConfigArn = endpointConfigArn;
                this.endpointConfigName = endpointConfigName;
            }
        }
        return new Import(scope, id, {
            environmentFromArn: endpointConfigArn,
        });
    }
    /**
     * Imports an EndpointConfig defined either outside the CDK or in a different CDK stack.
     * @param scope the Construct scope.
     * @param id the resource id.
     * @param endpointConfigName the name of the endpoint configuration.
     */
    static fromEndpointConfigName(scope, id, endpointConfigName) {
        const endpointConfigArn = cdk.Stack.of(scope).formatArn({
            service: 'sagemaker',
            resource: 'endpoint-config',
            resourceName: endpointConfigName,
        });
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.endpointConfigArn = endpointConfigArn;
                this.endpointConfigName = endpointConfigName;
            }
        }
        return new Import(scope, id, {
            environmentFromArn: endpointConfigArn,
        });
    }
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.endpointConfigName,
        });
        this.instanceProductionVariantsByName = {};
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_sagemaker_alpha_EndpointConfigProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, EndpointConfig);
            }
            throw error;
        }
        (props.instanceProductionVariants || []).map(p => this.addInstanceProductionVariant(p));
        // create the endpoint configuration resource
        const endpointConfig = new aws_sagemaker_1.CfnEndpointConfig(this, 'EndpointConfig', {
            kmsKeyId: (props.encryptionKey) ? props.encryptionKey.keyArn : undefined,
            endpointConfigName: this.physicalName,
            productionVariants: cdk.Lazy.any({ produce: () => this.renderInstanceProductionVariants() }),
        });
        this.endpointConfigName = this.getResourceNameAttribute(endpointConfig.attrEndpointConfigName);
        this.endpointConfigArn = this.getResourceArnAttribute(endpointConfig.ref, {
            service: 'sagemaker',
            resource: 'endpoint-config',
            resourceName: this.physicalName,
        });
    }
    /**
     * Add production variant to the endpoint configuration.
     *
     * @param props The properties of a production variant to add.
     */
    addInstanceProductionVariant(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_sagemaker_alpha_InstanceProductionVariantProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addInstanceProductionVariant);
            }
            throw error;
        }
        if (props.variantName in this.instanceProductionVariantsByName) {
            throw new Error(`There is already a Production Variant with name '${props.variantName}'`);
        }
        this.validateInstanceProductionVariantProps(props);
        this.instanceProductionVariantsByName[props.variantName] = {
            acceleratorType: props.acceleratorType,
            initialInstanceCount: props.initialInstanceCount || 1,
            initialVariantWeight: props.initialVariantWeight || 1.0,
            instanceType: props.instanceType || instance_type_1.InstanceType.T2_MEDIUM,
            modelName: props.model.modelName,
            variantName: props.variantName,
        };
    }
    /**
     * Get instance production variants associated with endpoint configuration.
     *
     * @internal
     */
    get _instanceProductionVariants() {
        return Object.values(this.instanceProductionVariantsByName);
    }
    /**
     * Find instance production variant based on variant name
     * @param name Variant name from production variant
     *
     * @internal
     */
    _findInstanceProductionVariant(name) {
        const ret = this.instanceProductionVariantsByName[name];
        if (!ret) {
            throw new Error(`No variant with name: '${name}'`);
        }
        return ret;
    }
    validateProductionVariants() {
        // validate number of production variants
        if (this._instanceProductionVariants.length < 1) {
            throw new Error('Must configure at least 1 production variant');
        }
        else if (this._instanceProductionVariants.length > 10) {
            throw new Error('Can\'t have more than 10 production variants');
        }
    }
    validateInstanceProductionVariantProps(props) {
        const errors = [];
        // check instance count is greater than zero
        if (props.initialInstanceCount !== undefined && props.initialInstanceCount < 1) {
            errors.push('Must have at least one instance');
        }
        // check variant weight is not negative
        if (props.initialVariantWeight && props.initialVariantWeight < 0) {
            errors.push('Cannot have negative variant weight');
        }
        // check environment compatibility with model
        const model = props.model;
        if (!(0, util_1.sameEnv)(model.env.account, this.env.account)) {
            errors.push(`Cannot use model in account ${model.env.account} for endpoint configuration in account ${this.env.account}`);
        }
        else if (!(0, util_1.sameEnv)(model.env.region, this.env.region)) {
            errors.push(`Cannot use model in region ${model.env.region} for endpoint configuration in region ${this.env.region}`);
        }
        if (errors.length > 0) {
            throw new Error(`Invalid Production Variant Props: ${errors.join(os_1.EOL)}`);
        }
    }
    /**
     * Render the list of instance production variants.
     */
    renderInstanceProductionVariants() {
        this.validateProductionVariants();
        return this._instanceProductionVariants.map(v => ({
            acceleratorType: v.acceleratorType?.toString(),
            initialInstanceCount: v.initialInstanceCount,
            initialVariantWeight: v.initialVariantWeight,
            instanceType: v.instanceType.toString(),
            modelName: v.modelName,
            variantName: v.variantName,
        }));
    }
}
_a = JSII_RTTI_SYMBOL_1;
EndpointConfig[_a] = { fqn: "@aws-cdk/aws-sagemaker-alpha.EndpointConfig", version: "2.79.1-alpha.0" };
exports.EndpointConfig = EndpointConfig;
//# sourceMappingURL=data:application/json;base64,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