"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContainerImage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ecr = require("aws-cdk-lib/aws-ecr");
const assets = require("aws-cdk-lib/aws-ecr-assets");
const util_1 = require("./private/util");
const region_info_1 = require("aws-cdk-lib/region-info");
const core_1 = require("aws-cdk-lib/core");
/**
 * Constructs for types of container images
 */
class ContainerImage {
    /**
     * Reference an image in an ECR repository
     */
    static fromEcrRepository(repository, tag = 'latest') {
        return new EcrImage(repository, tag);
    }
    /**
     * Reference an image that's constructed directly from sources on disk
     * @param directory The directory where the Dockerfile is stored
     * @param options The options to further configure the selected image
     */
    static fromAsset(directory, options = {}) {
        return new AssetImage(directory, options);
    }
    /**
     * Reference an AWS Deep Learning Container image
     */
    static fromDlc(repositoryName, tag, accountId) {
        return new DlcEcrImage(repositoryName, tag, accountId);
    }
}
_a = JSII_RTTI_SYMBOL_1;
ContainerImage[_a] = { fqn: "@aws-cdk/aws-sagemaker-alpha.ContainerImage", version: "2.79.1-alpha.0" };
exports.ContainerImage = ContainerImage;
class EcrImage extends ContainerImage {
    constructor(repository, tag) {
        super();
        this.repository = repository;
        this.tag = tag;
    }
    bind(_scope, model) {
        this.repository.grantPull(model);
        return {
            imageName: this.repository.repositoryUriForTag(this.tag),
        };
    }
}
class AssetImage extends ContainerImage {
    constructor(directory, options = {}) {
        super();
        this.directory = directory;
        this.options = options;
    }
    bind(scope, model) {
        // Retain the first instantiation of this asset
        if (!this.asset) {
            this.asset = new assets.DockerImageAsset(scope, `ModelImage${(0, util_1.hashcode)(this.directory)}`, {
                directory: this.directory,
                ...this.options,
            });
        }
        this.asset.repository.grantPull(model);
        return {
            imageName: this.asset.imageUri,
        };
    }
}
class DlcEcrImage extends ContainerImage {
    constructor(repositoryName, tag, accountId) {
        super();
        this.repositoryName = repositoryName;
        this.tag = tag;
        this.accountId = accountId;
    }
    bind(scope, model) {
        const accountId = this.accountId ?? core_1.Stack.of(scope).regionalFact(region_info_1.FactName.DLC_REPOSITORY_ACCOUNT);
        const repository = ecr.Repository.fromRepositoryAttributes(scope, 'DlcRepository', {
            repositoryName: this.repositoryName,
            repositoryArn: ecr.Repository.arnForLocalRepository(this.repositoryName, scope, accountId),
        });
        repository.grantPull(model);
        return { imageName: `${repository.repositoryUri}:${this.tag}` };
    }
}
//# sourceMappingURL=data:application/json;base64,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