import * as ecr from 'aws-cdk-lib/aws-ecr';
import * as assets from 'aws-cdk-lib/aws-ecr-assets';
import { Construct } from 'constructs';
import { Model } from './model';
/**
 * The configuration for creating a container image.
 */
export interface ContainerImageConfig {
    /**
     * The image name. Images in Amazon ECR repositories can be specified by either using the full registry/repository:tag or
     * registry/repository@digest.
     *
     * For example, `012345678910.dkr.ecr.<region-name>.amazonaws.com/<repository-name>:latest` or
     * `012345678910.dkr.ecr.<region-name>.amazonaws.com/<repository-name>@sha256:94afd1f2e64d908bc90dbca0035a5b567EXAMPLE`.
     */
    readonly imageName: string;
}
/**
 * Constructs for types of container images
 */
export declare abstract class ContainerImage {
    /**
     * Reference an image in an ECR repository
     */
    static fromEcrRepository(repository: ecr.IRepository, tag?: string): ContainerImage;
    /**
     * Reference an image that's constructed directly from sources on disk
     * @param directory The directory where the Dockerfile is stored
     * @param options The options to further configure the selected image
     */
    static fromAsset(directory: string, options?: assets.DockerImageAssetOptions): ContainerImage;
    /**
     * Reference an AWS Deep Learning Container image
     */
    static fromDlc(repositoryName: string, tag: string, accountId?: string): ContainerImage;
    /**
     * Called when the image is used by a Model
     */
    abstract bind(scope: Construct, model: Model): ContainerImageConfig;
}
