"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ensureLambda = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const path = require("path");
const lambdaTimeout = aws_cdk_lib_1.Duration.seconds(10);
function ensureLambda(scope) {
    const stack = aws_cdk_lib_1.Stack.of(scope);
    const lambdaName = 'AthenaManager';
    const ID = 'CFN::Resource::Custom::Athena';
    const existing = stack.node.tryFindChild(lambdaName);
    if (existing) {
        return existing;
    }
    const policy = new aws_cdk_lib_1.aws_iam.ManagedPolicy(stack, 'Athena-Manager-Policy', {
        managedPolicyName: `${stack.stackName}-${lambdaName}`,
        description: `Used by Lambda ${lambdaName}, which is a custom CFN resource, managing Athena resources`,
        statements: [
            new aws_cdk_lib_1.aws_iam.PolicyStatement({
                actions: ['athena:GetWorkGroup'],
                resources: ['*'],
            }),
            new aws_cdk_lib_1.aws_iam.PolicyStatement({
                actions: ['athena:CreateWorkGroup', 'athena:TagResource'],
                resources: ['*'],
                conditions: {
                    StringLike: {
                        'aws:RequestTag/CreatedByCfnCustomResource': `${ID}-WorkGroup`,
                    },
                },
            }),
            new aws_cdk_lib_1.aws_iam.PolicyStatement({
                actions: [
                    'athena:DeleteWorkGroup',
                    'athena:TagResource',
                    'athena:UntagResource',
                    'athena:UpdateWorkGroup',
                ],
                resources: ['*'],
                conditions: {
                    StringLike: {
                        'aws:ResourceTag/CreatedByCfnCustomResource': `${ID}-WorkGroup`,
                    },
                },
            }),
            new aws_cdk_lib_1.aws_iam.PolicyStatement({
                actions: [
                    'athena:CreateNamedQuery',
                    'athena:DeleteNamedQuery',
                    'athena:GetNamedQuery',
                    'athena:ListNamedQueries',
                ],
                resources: ['*'],
            }),
        ],
    });
    const role = new aws_cdk_lib_1.aws_iam.Role(stack, 'Athena-Manager-Role', {
        roleName: `${stack.stackName}-${lambdaName}`,
        description: `Used by Lambda ${lambdaName}, which is a custom CFN resource, managing Athena resources`,
        assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('lambda.amazonaws.com'),
        managedPolicies: [
            policy,
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
        ],
    });
    const fn = new aws_cdk_lib_1.aws_lambda.Function(stack, lambdaName, {
        functionName: `${stack.stackName}-${lambdaName}`,
        role: role,
        description: 'Custom CFN resource: Manage Athena resources',
        runtime: aws_cdk_lib_1.aws_lambda.Runtime.NODEJS_14_X,
        handler: 'index.handler',
        code: aws_cdk_lib_1.aws_lambda.Code.fromAsset(path.join(__dirname, '../lambda/code.zip')),
        timeout: lambdaTimeout,
    });
    return fn;
}
exports.ensureLambda = ensureLambda;
//# sourceMappingURL=data:application/json;base64,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