"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardLogger = exports.CustomResource = void 0;
const https = require("https");
const URL = require("url");
/**
 * Custom CloudFormation resource helper
 */
class CustomResource {
    constructor(context, callback, logger) {
        /**
         * Stores functions executed when resource creation is requested
         */
        this.createFunctions = [];
        /**
         * Stores functions executed when resource update is requested
         */
        this.updateFunctions = [];
        /**
         * Stores functions executed when resource deletion is requested
         */
        this.deleteFunctions = [];
        /**
         * Stores values returned to CloudFormation
         */
        this.ResponseData = {};
        this.context = context;
        this.callback = callback;
        this.logger = logger || new StandardLogger();
    }
    /**
     * Adds a function to the CREATE queue
     */
    onCreate(func) {
        this.createFunctions.push(func);
        return this;
    }
    /**
     * Adds a function to the UPDATE queue
     */
    onUpdate(func) {
        this.updateFunctions.push(func);
        return this;
    }
    /**
     * Adds a function to the DELETE queue
     */
    onDelete(func) {
        this.deleteFunctions.push(func);
        return this;
    }
    /**
     * Handles the Lambda event
     */
    handle(event) {
        const lambdaEvent = event;
        const self = this;
        if (typeof lambdaEvent.ResponseURL === 'undefined') {
            throw new Error('ResponseURL missing');
        }
        this.logger.debug(`REQUEST RECEIVED:\n${JSON.stringify(lambdaEvent)}`);
        this.timeout(lambdaEvent);
        event.addResponseValue = (key, value) => {
            self.ResponseData[key] = value;
        };
        event.setPhysicalResourceId = (value) => {
            self.PhysicalResourceId = value;
        };
        try {
            let queue;
            if (lambdaEvent.RequestType == 'Create')
                queue = this.createFunctions;
            else if (lambdaEvent.RequestType == 'Update')
                queue = this.updateFunctions;
            else if (lambdaEvent.RequestType == 'Delete')
                queue = this.deleteFunctions;
            else {
                this.sendResponse(lambdaEvent, 'FAILED', `Unexpected request type: ${lambdaEvent.RequestType}`);
                return this;
            }
            let result = queue.reduce((current, next) => {
                return current.then((value) => {
                    return next(value);
                });
            }, Promise.resolve(event));
            result
                .then(function (event) {
                self.logger.debug(event);
                self.sendResponse(lambdaEvent, 'SUCCESS', `${lambdaEvent.RequestType} completed successfully`);
            })
                .catch(function (err) {
                self.logger.error(err, err.stack);
                self.sendResponse(lambdaEvent, 'FAILED', err.message || err.code);
            });
        }
        catch (err) {
            this.sendResponse(lambdaEvent, 'FAILED', err.message || err.code);
        }
        return this;
    }
    /**
     * Sends CloudFormation response just before the Lambda times out
     */
    timeout(event) {
        const self = this;
        const handler = () => {
            self.logger.error('Timeout FAILURE!');
            new Promise(() => self.sendResponse(event, 'FAILED', 'Function timed out')).then(() => self.callback(new Error('Function timed out')));
        };
        setTimeout(handler, this.context.getRemainingTimeInMillis() - 1000);
    }
    /**
     * Sends CloudFormation response
     */
    sendResponse(event, responseStatus, responseData) {
        const self = this;
        this.logger.debug(`Sending response ${responseStatus}:\n${JSON.stringify(responseData)}`);
        const data = this.ResponseData;
        data['Message'] = responseData;
        const body = JSON.stringify({
            Status: responseStatus,
            Reason: `${responseData} | Full error in CloudWatch ${this.context.logStreamName}`,
            PhysicalResourceId: self.PhysicalResourceId ||
                event.PhysicalResourceId ||
                event.ResourceProperties.Name,
            StackId: event.StackId,
            RequestId: event.RequestId,
            LogicalResourceId: event.LogicalResourceId,
            Data: data,
        });
        this.logger.debug('RESPONSE BODY:\n', body);
        const url = URL.parse(event.ResponseURL);
        const options = {
            hostname: url.hostname,
            port: 443,
            path: url.path,
            method: 'PUT',
            headers: {
                'content-type': '',
                'content-length': body.length,
            },
        };
        this.logger.info('SENDING RESPONSE...\n');
        const request = https.request(options, function (response) {
            self.logger.debug(`STATUS: ${response.statusCode}`);
            self.logger.debug(`HEADERS: ${JSON.stringify(response.headers)}`);
            self.context.done();
        });
        request.on('error', function (error) {
            self.logger.error(`sendResponse Error: ${error}`);
            self.context.done();
        });
        request.write(body);
        request.end();
    }
}
exports.CustomResource = CustomResource;
/**
 * Standard logger class
 */
class StandardLogger {
    constructor(level) {
        this.level = level || 1 /* WARN */;
    }
    /**
     * Logs message with level ERROR
     */
    error(message, ...optionalParams) {
        if (this.level < 0 /* ERROR */)
            return;
        console.error(message, ...optionalParams);
    }
    /**
     * Logs message with level WARN
     */
    warn(message, ...optionalParams) {
        if (this.level < 1 /* WARN */)
            return;
        console.warn(message, ...optionalParams);
    }
    /**
     * Logs message with level INFO
     */
    info(message, ...optionalParams) {
        if (this.level < 2 /* INFO */)
            return;
        console.info(message, ...optionalParams);
    }
    /**
     * Logs message with level DEBUG
     */
    debug(message, ...optionalParams) {
        if (this.level < 3 /* DEBUG */)
            return;
        console.debug(message, ...optionalParams);
    }
    /**
     * Alias for info
     */
    log(message, ...optionalParams) {
        this.info(message, ...optionalParams);
    }
}
exports.StandardLogger = StandardLogger;
//# sourceMappingURL=data:application/json;base64,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