from smoothcrawler import URL
from datetime import datetime, date, timedelta
import pytest
import re


TEST_TARGET_URL = "https://www.google.com"
TEST_TARGET_URL_WITH_INDEX = "https://www.test.com?index={index}"
TEST_TARGET_URL_WITH_DATE = "https://www.google.com?date={date}"
TEST_TARGET_URL_WITH_DATETIME = "https://www.google.com?datetime={datetime}"
TEST_TARGET_URL_WITH_ITERATOR = "https://www.google.com?option={iterator}"
TEST_TARGET_URL_WITH_DICT_ITERATOR = "https://www.google.com?{iterator}"
TEST_TARGET_URL_WITH_INVALID_OPTION = "https://www.google.com?test={test}"

START_INDEX = 1
END_INDEX = 10

START_DATE = "20211001"
END_DATE = "20211020"

START_DATETIME = "2021/10/01 00:00:00"
END_DATETIME = "2021/10/03 00:00:00"

TEST_ITERATOR_LIST = [i for i in range(1, 10)]
TEST_ITERATOR_DICT = {"index_1": 1, "index_2": 2, "index_3": 3}


class TestURLs:

    def test_generate(self):
        url = URL(TEST_TARGET_URL)
        assert url.base_url == TEST_TARGET_URL, "The URL value and target url is not the same."


    def test_generate_with_index_rule(self):
        urls = URL(base=TEST_TARGET_URL_WITH_INDEX, start=START_INDEX, end=END_INDEX)
        index_urls = urls.generate()
        # Check for the amounts of setting and the return value of URL.
        assert len(index_urls) == len(range(START_INDEX, END_INDEX + 1)), "The amounts of index range and URL should be the same."
        # Check the format of URLs which be return from URL object.
        index_options = [f"index={i}" for i in range(START_INDEX, END_INDEX)]
        index_urls_options = []
        for _url in index_urls:
            search_result = TestURLs._get_url_option(url=_url, option_format=r"index=[0-9]")
            index_urls_options.append(search_result.group(0))
            assert search_result.group(0) in index_options, "The URL value should only be generated by the passing option."
        assert set(index_urls_options) == set(index_options), "Index options list and URL list should be the same."


    def test_generate_with_date_rule(self):
        urls = URL(base=TEST_TARGET_URL_WITH_DATE, start=START_DATE, end=END_DATE, formatter="yyyymmdd")
        date_urls = urls.generate()
        d1, d2 = TestURLs._get_yyyymmdd_date_objs(START_DATE, END_DATE)
        diff_days = (d2 - d1).days
        assert len(date_urls) == (diff_days + 1), "The return value amount of URL should be the same with period days between target 2 dates."
        all_days = [(d1 + timedelta(days=i)).isoformat() for i in range(0, diff_days + 1)]
        all_days_arguments = [f"date={_day.replace('-', '')}" for _day in all_days]
        date_urls_options = []
        for _url in date_urls:
            search_result = TestURLs._get_url_option(url=_url, option_format=r"date=[0-9]{8}")
            date_urls_options.append(search_result.group(0))
            assert search_result.group(0) in all_days_arguments, "The URL value should only be generated by the passing option."
        assert set(date_urls_options) == set(all_days_arguments), "Date options list and URL list should be the same."


    def test_generate_with_datetime_rule(self):
        urls = URL(base=TEST_TARGET_URL_WITH_DATETIME, start=START_DATETIME, end=END_DATETIME, formatter="yyyy/mm/dd HH:MM:SS")
        urls.set_period(days=0, hours=0, minutes=20, seconds=0)
        datetime_urls = urls.generate()
        d1, d2 = TestURLs._get_yyyymmdd_datetime_objs(START_DATETIME, END_DATETIME, "%Y/%m/%d %H:%M:%S")
        diff_period_minutes = int((d2 - d1).days * 24 * 60 / 20)
        assert len(datetime_urls) == (diff_period_minutes + 1), "The return value amount of URL should be the same with period days between target 2 dates."
        all_periods = [(d1 + timedelta(minutes=20*i)).isoformat() for i in range(0, diff_period_minutes + 1)]
        all_periods_arguments = [f"datetime={_day.replace('-', '').replace('T', '').replace(':', '')}" for _day in all_periods]
        date_urls_options = []
        for _url in datetime_urls:
            search_result = TestURLs._get_url_option(url=_url, option_format=r"datetime=[0-9]{14}")
            date_urls_options.append(search_result.group(0))
            assert search_result.group(0) in all_periods_arguments, "The URL value should only be generated by the passing option."
        assert set(date_urls_options) == set(all_periods_arguments), "Date options list and URL list should be the same."


    def test_generate_with_iterator_rule(self):
        urls = URL(base=TEST_TARGET_URL_WITH_ITERATOR, iter=TEST_ITERATOR_LIST)
        iterator_urls = urls.generate()
        assert len(iterator_urls) == len(TEST_ITERATOR_LIST), "The length of these lists should be the same."
        all_iter_options = [f"option={i}" for i in TEST_ITERATOR_LIST]
        iter_urls_options = []
        for _url in iterator_urls:
            search_result = TestURLs._get_url_option(url=_url, option_format=r"option=[0-9]{1,8}")
            iter_urls_options.append(search_result.group(0))
            assert search_result.group(
                0) in all_iter_options, "The URL value should only be generated by the passing option."
        assert set(iter_urls_options) == set(all_iter_options), "Date options list and URL list should be the same."

        urls = URL(base=TEST_TARGET_URL_WITH_DICT_ITERATOR, iter=TEST_ITERATOR_DICT)
        iterator_urls = urls.generate()
        assert len(iterator_urls) == len(TEST_ITERATOR_DICT), "The length of these lists should be the same."
        all_dict_options = [f"{_key}={_value}" for _key, _value in TEST_ITERATOR_DICT.items()]
        iter_urls_options = []
        for _url in iterator_urls:
            for _key, _value in TEST_ITERATOR_DICT.items():
                search_result = TestURLs._get_url_option(url=_url, option_format=re.escape(_key) + r"=" + re.escape(str(_value)))
                if search_result is not None:
                    iter_urls_options.append(search_result.group(0))
                    assert search_result.group(
                        0) in all_dict_options, "The URL value should only be generated by the passing option."
                    break
        assert set(iter_urls_options) == set(all_dict_options), "Date options list and URL list should be the same."


    @staticmethod
    def _get_url_options(urls, option_format):
        urls_options = []
        for _url in urls:
            parameters = str(_url).split("?")
            argument = parameters[-1]
            search_result = re.search(option_format, argument)
            urls_options.append(search_result.group(0))


    @staticmethod
    def _get_url_option(url, option_format):
        parameters = str(url).split("?")
        argument = parameters[-1]
        search_result = re.search(option_format, argument)
        return search_result


    @staticmethod
    def _get_yyyymmdd_datetime_objs(date1, date2, formatter):
        d1 = datetime.strptime(date1, formatter)
        d2 = datetime.strptime(date2, formatter)
        return d1, d2


    @staticmethod
    def _get_yyyymmdd_date_objs(date1, date2):
        s_year = int(date1[:4])
        s_month = int(date1[4:6])
        s_day = int(date1[6:])
        e_year = int(date2[:4])
        e_month = int(date2[4:6])
        e_day = int(date2[6:])
        d1 = date(s_year, s_month, s_day)
        d2 = date(e_year, e_month, e_day)
        return d1, d2

