/*
 * Decompiled with CFR 0.152.
 */
package nl.esciencecenter.xenon.adaptors.schedulers.slurm;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import nl.esciencecenter.xenon.XenonException;
import nl.esciencecenter.xenon.adaptors.schedulers.CommandLineUtils;
import nl.esciencecenter.xenon.adaptors.schedulers.RemoteCommandRunner;
import nl.esciencecenter.xenon.adaptors.schedulers.ScriptingParser;
import nl.esciencecenter.xenon.adaptors.schedulers.ScriptingScheduler;
import nl.esciencecenter.xenon.adaptors.schedulers.StreamsImplementation;
import nl.esciencecenter.xenon.adaptors.schedulers.slurm.SlurmSchedulerAdaptor;
import nl.esciencecenter.xenon.adaptors.schedulers.slurm.SlurmSetup;
import nl.esciencecenter.xenon.adaptors.schedulers.slurm.SlurmUtils;
import nl.esciencecenter.xenon.credentials.Credential;
import nl.esciencecenter.xenon.filesystems.Path;
import nl.esciencecenter.xenon.schedulers.JobDescription;
import nl.esciencecenter.xenon.schedulers.JobStatus;
import nl.esciencecenter.xenon.schedulers.NoSuchJobException;
import nl.esciencecenter.xenon.schedulers.NoSuchQueueException;
import nl.esciencecenter.xenon.schedulers.QueueStatus;
import nl.esciencecenter.xenon.schedulers.Streams;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SlurmScheduler
extends ScriptingScheduler {
    private static final Logger LOGGER = LoggerFactory.getLogger(SlurmScheduler.class);
    private final String[] queueNames;
    private final String defaultQueueName;
    private final SlurmSetup setup;

    protected SlurmScheduler(String uniqueID, String location, Credential credential, Map<String, String> prop) throws XenonException {
        super(uniqueID, "slurm", location, credential, prop, SlurmSchedulerAdaptor.VALID_PROPERTIES, "xenon.adaptors.schedulers.slurm.poll.delay");
        boolean disableAccounting = this.properties.getBooleanProperty("xenon.adaptors.schedulers.slurm.disable.accounting.usage");
        String output = this.runCheckedCommand(null, "scontrol", "show", "config");
        Map<String, String> info = ScriptingParser.parseKeyValueLines(output, ScriptingParser.EQUALS_REGEX, "slurm", "Configuration data as of", "Slurmctld(primary/backup) at", "Account Gather");
        this.setup = new SlurmSetup(info, disableAccounting);
        output = this.runCheckedCommand(null, "sinfo", "--noheader", "--format=%120P");
        String[] foundQueueNames = ScriptingParser.parseList(output);
        String foundDefaultQueueName = null;
        for (int i = 0; i < foundQueueNames.length; ++i) {
            String queueName = foundQueueNames[i];
            if (!queueName.endsWith("*")) continue;
            foundQueueNames[i] = queueName.substring(0, queueName.length() - 1);
            foundDefaultQueueName = foundQueueNames[i];
        }
        this.queueNames = foundQueueNames;
        this.defaultQueueName = foundDefaultQueueName;
        LOGGER.debug("Created new SlurmConfig. version = \"{}\", accounting available: {}", (Object)this.setup.version(), (Object)this.setup.accountingAvailable());
    }

    @Override
    public String[] getQueueNames() {
        return (String[])this.queueNames.clone();
    }

    @Override
    public String getDefaultQueueName() {
        return this.defaultQueueName;
    }

    @Override
    protected void translateError(RemoteCommandRunner runner, String stdin, String executable, String ... arguments) throws XenonException {
        String error = runner.getStderr();
        if (error.contains("Invalid job id")) {
            throw new NoSuchJobException("slurm", "Invalid job ID");
        }
        throw new XenonException("slurm", "Could not run command \"" + executable + "\" with stdin \"" + stdin + "\" arguments \"" + Arrays.toString(arguments) + "\" at \"" + this.subScheduler + "\". Exit code = " + runner.getExitCode() + " Output: " + runner.getStdout() + " Error output: " + runner.getStderr());
    }

    @Override
    public String submitBatchJob(JobDescription description) throws XenonException {
        String output;
        Path fsEntryPath = this.getWorkingDirectory();
        SlurmUtils.verifyJobDescription(description, false);
        this.checkQueue(this.queueNames, description.getQueueName());
        String customScriptFile = description.getJobOptions().get("job.script");
        if (customScriptFile == null) {
            this.checkWorkingDirectory(description.getWorkingDirectory());
            String jobScript = SlurmUtils.generate(description, fsEntryPath);
            output = this.runCheckedCommand(jobScript, "sbatch", new String[0]);
        } else {
            if (!customScriptFile.startsWith("/")) {
                Path scriptFile = fsEntryPath.resolve(customScriptFile);
                customScriptFile = scriptFile.toString();
            }
            output = this.runCheckedCommand(null, "sbatch", customScriptFile);
        }
        return ScriptingParser.parseJobIDFromLine(output, "slurm", "Submitted batch job", "Granted job allocation");
    }

    private String findInteractiveJobInMap(Map<String, Map<String, String>> queueInfo, String tag) {
        for (Map.Entry<String, Map<String, String>> entry : queueInfo.entrySet()) {
            if ((!entry.getValue().containsKey("NAME") || !entry.getValue().get("NAME").equals(tag)) && (!entry.getValue().containsKey("JobName") || !entry.getValue().get("JobName").equals(tag))) continue;
            String jobID = entry.getKey();
            LOGGER.debug("Found interactive job ID: %s", (Object)jobID);
            return jobID;
        }
        return null;
    }

    private String findInteractiveJob(String tag) throws XenonException {
        String result = this.findInteractiveJobInMap(this.getSqueueInfo(new String[0]), tag);
        if (result != null) {
            return result;
        }
        return this.findInteractiveJobInMap(this.getSacctInfo(new String[0]), tag);
    }

    @Override
    public Streams submitInteractiveJob(JobDescription description) throws XenonException {
        JobStatus status;
        Path fsEntryPath = this.getWorkingDirectory();
        SlurmUtils.verifyJobDescription(description, true);
        this.checkWorkingDirectory(description.getWorkingDirectory());
        this.checkQueue(this.queueNames, description.getQueueName());
        UUID tag = UUID.randomUUID();
        String[] arguments = SlurmUtils.generateInteractiveArguments(description, fsEntryPath, tag);
        Streams interactiveJob = this.startInteractiveCommand("salloc", arguments);
        String result = this.findInteractiveJob(tag.toString());
        long end = System.currentTimeMillis() + 60000L;
        while (result == null && System.currentTimeMillis() < end) {
            try {
                Thread.sleep(1000L);
            }
            catch (InterruptedException e) {
                LOGGER.warn("Interrupted!", (Throwable)e);
                Thread.currentThread().interrupt();
            }
            result = this.findInteractiveJob(tag.toString());
        }
        if (result != null) {
            return new StreamsImplementation(result, interactiveJob.getStdout(), interactiveJob.getStdin(), interactiveJob.getStderr());
        }
        try {
            status = this.subScheduler.getJobStatus(interactiveJob.getJobIdentifier());
        }
        catch (XenonException e) {
            throw new XenonException("slurm", "Failed to submit interactive job");
        }
        if (status.isDone() && status.hasException()) {
            throw new XenonException("slurm", "Failed to submit interactive job", status.getException());
        }
        if (status.getExitCode() != null && status.getExitCode().equals(1)) {
            throw new XenonException("slurm", "Failed to submit interactive job, perhaps some job options are invalid? (e.g. too many nodes, or invalid partition name)");
        }
        throw new XenonException("slurm", "Failed to submit interactive job. Interactive job status is " + status.getState() + " exit code = " + status.getExitCode() + " tag=" + tag.toString(), status.getException());
    }

    @Override
    public JobStatus cancelJob(String jobIdentifier) throws XenonException {
        block3: {
            this.assertNonNullOrEmpty(jobIdentifier, "Job identifier cannot be null or empty");
            try {
                String output = this.runCheckedCommand(null, "scancel", jobIdentifier);
                if (!output.isEmpty()) {
                    throw new XenonException("slurm", "Got unexpected output on cancelling job: " + output);
                }
            }
            catch (XenonException e) {
                if (e.getMessage().contains("Job/step already completing or completed")) break block3;
                throw e;
            }
        }
        return this.getJobStatus(jobIdentifier);
    }

    @Override
    public String[] getJobs(String ... queueNames) throws XenonException {
        String output;
        if (queueNames == null || queueNames.length == 0) {
            output = this.runCheckedCommand(null, "squeue", "--noheader", "--format=%i");
        } else {
            this.checkQueueNames(queueNames);
            output = this.runCheckedCommand(null, "squeue", "--noheader", "--format=%i", "--partitions=" + CommandLineUtils.asCSList(queueNames));
        }
        return ScriptingParser.parseList(output);
    }

    private Map<String, String> getSControlInfo(String jobIdentifier) throws XenonException {
        RemoteCommandRunner runner = this.runCommand(null, "scontrol", "show", "job", jobIdentifier);
        if (!runner.success()) {
            LOGGER.debug("failed to get job status {}", (Object)runner);
            return null;
        }
        return ScriptingParser.parseKeyValuePairs(runner.getStdout(), "slurm", "WorkDir=", "Command=");
    }

    private Map<String, Map<String, String>> getSqueueInfo(String ... jobs) throws XenonException {
        String squeueOutput = "";
        squeueOutput = jobs == null || jobs.length == 0 ? this.runCheckedCommand(null, "squeue", "--format=%i %P %j %u %T %M %l %D %R %k") : this.runCheckedCommand(null, "squeue", "--format=%i %P %j %u %T %M %l %D %R %k", "--jobs=" + SlurmUtils.identifiersAsCSList(jobs));
        return ScriptingParser.parseTable(squeueOutput, "JOBID", ScriptingParser.WHITESPACE_REGEX, "slurm", "*", "~");
    }

    private Map<String, Map<String, String>> getSinfoInfo(String ... partitions) throws XenonException {
        String output = this.runCheckedCommand(null, "sinfo", "--format=%P %a %l %F %N %C %D", "--partition=" + CommandLineUtils.asCSList(partitions));
        return ScriptingParser.parseTable(output, "PARTITION", ScriptingParser.WHITESPACE_REGEX, "slurm", "*", "~");
    }

    private Map<String, Map<String, String>> getSacctInfo(String ... jobs) throws XenonException {
        if (!this.setup.accountingAvailable()) {
            return new HashMap<String, Map<String, String>>();
        }
        RemoteCommandRunner runner = jobs == null || jobs.length == 0 ? this.runCommand(null, "sacct", "-X", "-p", "--format=JobID,JobName,Partition,NTasks,Elapsed,State,ExitCode,AllocCPUS,DerivedExitCode,Submit,Suspended,Start,User,End,NNodes,Timelimit,Comment,Priority") : this.runCommand(null, "sacct", "-X", "-p", "--format=JobID,JobName,Partition,NTasks,Elapsed,State,ExitCode,AllocCPUS,DerivedExitCode,Submit,Suspended,Start,User,End,NNodes,Timelimit,Comment,Priority", "--jobs=" + SlurmUtils.identifiersAsCSList(jobs));
        if (runner.getExitCode() != 0) {
            throw new XenonException("slurm", "Error in getting sacct job status: " + runner);
        }
        if (!runner.getStderr().isEmpty()) {
            LOGGER.warn("Sacct produced error output: " + runner.getStderr());
        }
        return ScriptingParser.parseTable(runner.getStdout(), "JobID", ScriptingParser.BAR_REGEX, "slurm", "*", "~");
    }

    @Override
    public JobStatus getJobStatus(String jobIdentifier) throws XenonException {
        this.assertNonNullOrEmpty(jobIdentifier, "Job identifier cannot be null or empty");
        Map<String, Map<String, String>> sQueueInfo = this.getSqueueInfo(jobIdentifier);
        JobStatus result = SlurmUtils.getJobStatusFromSqueueInfo(sQueueInfo, jobIdentifier);
        if (result == null) {
            Map<String, Map<String, String>> sacctInfo = this.getSacctInfo(jobIdentifier);
            result = SlurmUtils.getJobStatusFromSacctInfo(sacctInfo, jobIdentifier);
        }
        if (result == null) {
            Map<String, String> scontrolInfo = this.getSControlInfo(jobIdentifier);
            result = SlurmUtils.getJobStatusFromScontrolInfo(scontrolInfo, jobIdentifier);
        }
        if (result == null) {
            throw new NoSuchJobException("slurm", "Unknown Job: " + jobIdentifier);
        }
        return result;
    }

    @Override
    public QueueStatus getQueueStatus(String queueName) throws XenonException {
        this.assertNonNullOrEmpty(queueName, "Queue name cannot be null or empty");
        Map<String, Map<String, String>> info = this.getSinfoInfo(queueName);
        QueueStatus result = SlurmUtils.getQueueStatusFromSInfo(info, queueName, this);
        if (result == null) {
            throw new NoSuchQueueException("slurm", "cannot get status of queue \"" + queueName + "\" from server");
        }
        return result;
    }

    @Override
    public QueueStatus[] getQueueStatuses(String ... requestedQueueNames) throws XenonException {
        if (requestedQueueNames == null) {
            throw new IllegalArgumentException("list of queue names cannot be null");
        }
        String[] targetQueueNames = requestedQueueNames.length == 0 ? this.getQueueNames() : requestedQueueNames;
        Map<String, Map<String, String>> info = this.getSinfoInfo(targetQueueNames);
        return this.getQueueStatuses(info, targetQueueNames);
    }
}

