"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseQuery = void 0;
const path = require("path");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const cdk = require("aws-cdk-lib");
const customresources = require("aws-cdk-lib/custom-resources");
const cluster_1 = require("../cluster");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const constructs_1 = require("constructs");
class DatabaseQuery extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const adminUser = this.getAdminUser(props);
        const handler = new lambda.SingletonFunction(this, 'Handler', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'database-query-provider')),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            timeout: cdk.Duration.minutes(1),
            uuid: '3de5bea7-27da-4796-8662-5efb56431b5f',
            lambdaPurpose: 'Query Redshift Database',
        });
        handler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['redshift-data:DescribeStatement', 'redshift-data:ExecuteStatement'],
            resources: ['*'],
        }));
        adminUser.grantRead(handler);
        const provider = new customresources.Provider(this, 'Provider', {
            onEventHandler: handler,
        });
        const queryHandlerProps = {
            handler: props.handler,
            clusterName: props.cluster.clusterName,
            adminUserArn: adminUser.secretArn,
            databaseName: props.databaseName,
            ...props.properties,
        };
        this.resource = new cdk.CustomResource(this, 'Resource', {
            resourceType: 'Custom::RedshiftDatabaseQuery',
            serviceToken: provider.serviceToken,
            removalPolicy: props.removalPolicy,
            properties: queryHandlerProps,
        });
        this.grantPrincipal = handler.grantPrincipal;
        this.ref = this.resource.ref;
    }
    applyRemovalPolicy(policy) {
        this.resource.applyRemovalPolicy(policy);
    }
    getAtt(attributeName) {
        return this.resource.getAtt(attributeName);
    }
    getAttString(attributeName) {
        return this.resource.getAttString(attributeName);
    }
    getAdminUser(props) {
        const cluster = props.cluster;
        let adminUser = props.adminUser;
        if (!adminUser) {
            if (cluster instanceof cluster_1.Cluster) {
                if (cluster.secret) {
                    adminUser = cluster.secret;
                }
                else {
                    throw new Error('Administrative access to the Redshift cluster is required but an admin user secret was not provided and the cluster did not generate admin user credentials (they were provided explicitly)');
                }
            }
            else {
                throw new Error('Administrative access to the Redshift cluster is required but an admin user secret was not provided and the cluster was imported');
            }
        }
        return adminUser;
    }
}
exports.DatabaseQuery = DatabaseQuery;
//# sourceMappingURL=data:application/json;base64,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