import { aws_ec2 as ec2, aws_iam as iam, aws_imagebuilder as imagebuilder, aws_logs as logs, aws_s3_assets as s3_assets } from 'aws-cdk-lib';
import { Construct } from 'constructs';
import { ImageBuilderObjectBase } from './common';
import { RunnerAmi, RunnerImage } from '../../common';
import { RunnerImageBuilderBase, RunnerImageBuilderProps } from '../common';
export interface AwsImageBuilderRunnerImageBuilderProps {
    /**
     * The instance type used to build the image.
     *
     * @default m5.large
     */
    readonly instanceType?: ec2.InstanceType;
}
/**
 * An asset including file or directory to place inside the built image.
 */
export interface ImageBuilderAsset {
    /**
     * Path to place asset in the image.
     */
    readonly path: string;
    /**
     * Asset to place in the image.
     */
    readonly asset: s3_assets.Asset;
}
/**
 * Properties for ImageBuilderComponent construct.
 */
export interface ImageBuilderComponentProperties {
    /**
     * Component platform. Must match the builder platform.
     */
    readonly platform: 'Linux' | 'Windows';
    /**
     * Component display name.
     */
    readonly displayName: string;
    /**
     * Component description.
     */
    readonly description: string;
    /**
     * Shell commands to run when adding this component to the image.
     *
     * On Linux, these are bash commands. On Windows, there are PowerShell commands.
     */
    readonly commands: string[];
    /**
     * Optional assets to add to the built image.
     */
    readonly assets?: ImageBuilderAsset[];
}
/**
 * Components are a set of commands to run and optional files to add to an image. Components are the building blocks of images built by Image Builder.
 *
 * Example:
 *
 * ```
 * new ImageBuilderComponent(this, 'AWS CLI', {
 *   platform: 'Windows',
 *   displayName: 'AWS CLI',
 *   description: 'Install latest version of AWS CLI',
 *   commands: [
 *     '$ErrorActionPreference = \'Stop\'',
 *     'Start-Process msiexec.exe -Wait -ArgumentList \'/i https://awscli.amazonaws.com/AWSCLIV2.msi /qn\'',
 *   ],
 * }
 * ```
 *
 * @deprecated Use `RunnerImageComponent` instead as this be internal soon.
 */
export declare class ImageBuilderComponent extends ImageBuilderObjectBase {
    /**
     * Component ARN.
     */
    readonly arn: string;
    /**
     * Supported platform for the component.
     */
    readonly platform: 'Windows' | 'Linux';
    private readonly assets;
    constructor(scope: Construct, id: string, props: ImageBuilderComponentProperties);
    /**
     * Grants read permissions to the principal on the assets buckets.
     *
     * @param grantee
     */
    grantAssetsRead(grantee: iam.IGrantable): void;
    prefixCommandsWithErrorHandling(platform: 'Windows' | 'Linux', commands: string[]): string[];
}
/**
 * @internal
 */
export declare class AwsImageBuilderRunnerImageBuilder extends RunnerImageBuilderBase {
    private boundDockerImage?;
    private boundAmi?;
    private readonly os;
    private readonly architecture;
    private readonly baseImage;
    private readonly baseAmi;
    private readonly logRetention;
    private readonly logRemovalPolicy;
    private readonly vpc;
    private readonly securityGroups;
    private readonly repository;
    private readonly subnetSelection;
    private readonly rebuildInterval;
    private readonly boundComponents;
    private readonly instanceType;
    private infrastructure;
    private readonly role;
    constructor(scope: Construct, id: string, props?: RunnerImageBuilderProps);
    private platform;
    /**
     * Called by IRunnerProvider to finalize settings and create the image builder.
     */
    bindDockerImage(): RunnerImage;
    private imageCleaner;
    protected createLog(id: string, recipeName: string): logs.LogGroup;
    protected createInfrastructure(managedPolicies: iam.IManagedPolicy[]): imagebuilder.CfnInfrastructureConfiguration;
    protected createImage(infra: imagebuilder.CfnInfrastructureConfiguration, dist: imagebuilder.CfnDistributionConfiguration, log: logs.LogGroup, imageRecipeArn?: string, containerRecipeArn?: string): imagebuilder.CfnImage;
    private amiOrContainerId;
    protected createPipeline(infra: imagebuilder.CfnInfrastructureConfiguration, dist: imagebuilder.CfnDistributionConfiguration, log: logs.LogGroup, imageRecipeArn?: string, containerRecipeArn?: string): imagebuilder.CfnImagePipeline;
    /**
     * The network connections associated with this resource.
     */
    get connections(): ec2.Connections;
    get grantPrincipal(): iam.IPrincipal;
    bindAmi(): RunnerAmi;
    private amiCleaner;
    private bindComponents;
}
