"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.defaultBaseAmi = exports.AmiRecipe = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const common_1 = require("./common");
const common_2 = require("../../common");
const common_3 = require("../common");
/**
 * Image builder recipe for Amazon Machine Image (AMI).
 *
 * @internal
 */
class AmiRecipe extends common_1.ImageBuilderObjectBase {
    constructor(scope, id, props) {
        super(scope, id);
        const name = (0, common_3.uniqueImageBuilderName)(this);
        let components = props.components.map(component => {
            return {
                componentArn: component.arn,
            };
        });
        let workingDirectory;
        if (props.platform == 'Linux') {
            workingDirectory = '/home/runner';
        }
        else if (props.platform == 'Windows') {
            workingDirectory = 'C:/'; // must exist or Image Builder fails and must not be empty or git will stall installing from the default windows\system32
        }
        else {
            throw new Error(`Unsupported AMI recipe platform: ${props.platform}`);
        }
        const recipe = new aws_cdk_lib_1.aws_imagebuilder.CfnImageRecipe(this, 'Recipe', {
            name: name,
            version: this.version('ImageRecipe', name, {
                platform: props.platform,
                components,
                parentAmi: props.baseAmi,
            }),
            parentImage: props.baseAmi,
            components,
            workingDirectory,
        });
        this.arn = recipe.attrArn;
        this.name = name;
    }
}
exports.AmiRecipe = AmiRecipe;
/**
 * Default base AMI for given OS and architecture.
 *
 * @internal
 */
function defaultBaseAmi(os, architecture) {
    let archUrl;
    let cpuType;
    if (architecture.is(common_2.Architecture.X86_64)) {
        archUrl = 'amd64';
        cpuType = aws_cdk_lib_1.aws_ec2.AmazonLinuxCpuType.X86_64;
    }
    else if (architecture.is(common_2.Architecture.ARM64)) {
        archUrl = 'arm64';
        cpuType = aws_cdk_lib_1.aws_ec2.AmazonLinuxCpuType.ARM_64;
    }
    else {
        throw new Error(`Unsupported architecture for base AMI: ${architecture.name}`);
    }
    if (os.is(common_2.Os.LINUX_UBUNTU) || os.is(common_2.Os.LINUX)) {
        return aws_cdk_lib_1.aws_ec2.MachineImage.fromSsmParameter(`/aws/service/canonical/ubuntu/server/focal/stable/current/${archUrl}/hvm/ebs-gp2/ami-id`, {
            os: aws_cdk_lib_1.aws_ec2.OperatingSystemType.LINUX,
        });
    }
    if (os.is(common_2.Os.LINUX_AMAZON_2)) {
        return aws_cdk_lib_1.aws_ec2.MachineImage.latestAmazonLinux({ cpuType });
    }
    if (os.is(common_2.Os.WINDOWS)) {
        return aws_cdk_lib_1.aws_ec2.MachineImage.latestWindows(aws_cdk_lib_1.aws_ec2.WindowsVersion.WINDOWS_SERVER_2022_ENGLISH_FULL_CONTAINERSLATEST);
    }
    throw new Error(`OS ${os.name} not supported for AMI runner image`);
}
exports.defaultBaseAmi = defaultBaseAmi;
//# sourceMappingURL=data:application/json;base64,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