"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable import/no-extraneous-dependencies */
const crypto = require("crypto");
const AWS = require("aws-sdk");
const helpers_1 = require("./helpers");
const sf = new AWS.StepFunctions();
// TODO use @octokit/webhooks?
function getHeader(event, header) {
    // API Gateway doesn't lowercase headers (V1 event) but Lambda URLs do (V2 event) :(
    for (const headerName of Object.keys(event.headers)) {
        if (headerName.toLowerCase() === header.toLowerCase()) {
            return event.headers[headerName];
        }
    }
    return undefined;
}
function verifyBody(event, secret) {
    const sig = Buffer.from(getHeader(event, 'x-hub-signature-256') || '', 'utf8');
    if (!event.body) {
        throw new Error('No body');
    }
    let body;
    if (event.isBase64Encoded) {
        body = Buffer.from(event.body, 'base64');
    }
    else {
        body = Buffer.from(event.body || '', 'utf8');
    }
    const hmac = crypto.createHmac('sha256', secret);
    hmac.update(body);
    const expectedSig = Buffer.from(`sha256=${hmac.digest('hex')}`, 'utf8');
    console.log('Calculated signature: ', expectedSig.toString());
    if (sig.length !== expectedSig.length || !crypto.timingSafeEqual(sig, expectedSig)) {
        throw new Error(`Signature mismatch. Expected ${expectedSig.toString()} but got ${sig.toString()}`);
    }
    return body.toString();
}
exports.handler = async function (event) {
    if (!process.env.WEBHOOK_SECRET_ARN || !process.env.STEP_FUNCTION_ARN) {
        throw new Error('Missing environment variables');
    }
    const webhookSecret = (await (0, helpers_1.getSecretJsonValue)(process.env.WEBHOOK_SECRET_ARN)).webhookSecret;
    let body;
    try {
        body = verifyBody(event, webhookSecret);
    }
    catch (e) {
        console.error(e);
        return {
            statusCode: 403,
            body: 'Bad signature',
        };
    }
    if (getHeader(event, 'content-type') !== 'application/json') {
        console.error(`This webhook only accepts JSON payloads, got ${getHeader(event, 'content-type')}`);
        return {
            statusCode: 400,
            body: 'Expecting JSON payload',
        };
    }
    if (getHeader(event, 'x-github-event') === 'ping') {
        return {
            statusCode: 200,
            body: 'Pong',
        };
    }
    // if (getHeader(event, 'x-github-event') !== 'workflow_job' && getHeader(event, 'x-github-event') !== 'workflow_run') {
    //     console.error(`This webhook only accepts workflow_job and workflow_run, got ${getHeader(event, 'x-github-event')}`);
    if (getHeader(event, 'x-github-event') !== 'workflow_job') {
        console.error(`This webhook only accepts workflow_job, got ${getHeader(event, 'x-github-event')}`);
        return {
            statusCode: 400,
            body: 'Expecting workflow_job',
        };
    }
    const payload = JSON.parse(body);
    if (payload.action !== 'queued') {
        console.log(`Ignoring action "${payload.action}", expecting "queued"`);
        return {
            statusCode: 200,
            body: 'OK. No runner started.',
        };
    }
    if (!payload.workflow_job.labels.includes('self-hosted')) {
        console.log(`Ignoring labels "${payload.workflow_job.labels}", expecting "self-hosted"`);
        return {
            statusCode: 200,
            body: 'OK. No runner started.',
        };
    }
    // it's easier to deal with maps in step functions
    let labels = {};
    payload.workflow_job.labels.forEach((l) => labels[l.toLowerCase()] = true);
    // set execution name which is also used as runner name which are limited to 64 characters
    let executionName = `${payload.repository.full_name.replace('/', '-')}-${getHeader(event, 'x-github-delivery')}`.slice(0, 64);
    // start execution
    const execution = await sf.startExecution({
        stateMachineArn: process.env.STEP_FUNCTION_ARN,
        input: JSON.stringify({
            owner: payload.repository.owner.login,
            repo: payload.repository.name,
            runId: payload.workflow_job.run_id,
            installationId: payload.installation?.id,
            labels: labels,
        }),
        // name is not random so multiple execution of this webhook won't cause multiple builders to start
        name: executionName,
    }).promise();
    console.log(`Started ${execution.executionArn}`);
    return {
        statusCode: 202,
        body: executionName,
    };
};
//# sourceMappingURL=data:application/json;base64,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