"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable import/no-extraneous-dependencies */
const crypto = require("crypto");
const fs = require("fs");
const rest_1 = require("@octokit/rest");
const github_1 = require("./github");
const helpers_1 = require("./helpers");
const nonce = crypto.randomBytes(64).toString('hex');
function getHtml(baseUrl, token, domain) {
    return fs.readFileSync('index.html', 'utf-8')
        .replace(/INSERT_WEBHOOK_URL_HERE/g, process.env.WEBHOOK_URL)
        .replace(/INSERT_BASE_URL_HERE/g, baseUrl)
        .replace(/INSERT_TOKEN_HERE/g, token)
        .replace(/INSERT_SECRET_ARN_HERE/g, process.env.SETUP_SECRET_ARN)
        .replace(/INSERT_DOMAIN_HERE/g, domain)
        .replace(/<script/g, `<script nonce="${nonce}"`)
        .replace(/<style/g, `<style nonce="${nonce}"`);
}
function response(code, body) {
    return {
        statusCode: code,
        headers: {
            'Content-Type': 'text/html',
            'Content-Security-Policy': `default-src 'unsafe-inline' 'nonce-${nonce}'; img-src data:; connect-src 'self'; form-action https:; frame-ancestors 'none'; object-src 'none'; base-uri 'self'`,
        },
        body: body,
    };
}
async function handleRoot(event, setupToken) {
    const stage = event.requestContext.stage == '$default' ? '' : `/${event.requestContext.stage}`;
    const setupBaseUrl = `https://${event.requestContext.domainName}${stage}`;
    const githubSecrets = await (0, helpers_1.getSecretJsonValue)(process.env.GITHUB_SECRET_ARN);
    return response(200, getHtml(setupBaseUrl, setupToken, githubSecrets.domain));
}
function decodeBody(event) {
    let body = event.body;
    if (!body) {
        throw new Error('No body found');
    }
    if (event.isBase64Encoded) {
        body = Buffer.from(body, 'base64').toString('utf-8');
    }
    return JSON.parse(body);
}
async function handleDomain(event) {
    const body = decodeBody(event);
    if (!body.domain) {
        return response(400, 'Invalid domain');
    }
    const githubSecrets = await (0, helpers_1.getSecretJsonValue)(process.env.GITHUB_SECRET_ARN);
    githubSecrets.domain = body.domain;
    await (0, helpers_1.updateSecretValue)(process.env.GITHUB_SECRET_ARN, JSON.stringify(githubSecrets));
    return response(200, 'Domain set');
}
async function handlePat(event) {
    const body = decodeBody(event);
    if (!body.pat || !body.domain) {
        return response(400, 'Invalid personal access token');
    }
    await (0, helpers_1.updateSecretValue)(process.env.GITHUB_SECRET_ARN, JSON.stringify({
        domain: body.domain,
        appId: '',
        personalAuthToken: body.pat,
    }));
    await (0, helpers_1.updateSecretValue)(process.env.SETUP_SECRET_ARN, JSON.stringify({ token: '' }));
    return response(200, 'Personal access token set');
}
async function handleNewApp(event) {
    if (!event.queryStringParameters) {
        return response(400, 'Invalid code');
    }
    const code = event.queryStringParameters.code;
    if (!code) {
        return response(400, 'Invalid code');
    }
    const githubSecrets = await (0, helpers_1.getSecretJsonValue)(process.env.GITHUB_SECRET_ARN);
    const baseUrl = (0, github_1.baseUrlFromDomain)(githubSecrets.domain);
    const newApp = await new rest_1.Octokit({ baseUrl }).rest.apps.createFromManifest({ code });
    await (0, helpers_1.updateSecretValue)(process.env.GITHUB_SECRET_ARN, JSON.stringify({
        domain: new URL(newApp.data.html_url).host,
        appId: newApp.data.id,
        personalAuthToken: '',
    }));
    await (0, helpers_1.updateSecretValue)(process.env.GITHUB_PRIVATE_KEY_SECRET_ARN, newApp.data.pem);
    await (0, helpers_1.updateSecretValue)(process.env.WEBHOOK_SECRET_ARN, JSON.stringify({
        webhookSecret: newApp.data.webhook_secret,
    }));
    await (0, helpers_1.updateSecretValue)(process.env.SETUP_SECRET_ARN, JSON.stringify({ token: '' }));
    return response(200, `New app set. <a href="${newApp.data.html_url}/installations/new">Install it</a> for your repositories.`);
}
async function handleExistingApp(event) {
    const body = decodeBody(event);
    if (!body.appid || !body.pk || !body.domain) {
        return response(400, 'Missing fields');
    }
    await (0, helpers_1.updateSecretValue)(process.env.GITHUB_SECRET_ARN, JSON.stringify({
        domain: body.domain,
        appId: body.appid,
        personalAuthToken: '',
    }));
    await (0, helpers_1.updateSecretValue)(process.env.GITHUB_PRIVATE_KEY_SECRET_ARN, body.pk);
    await (0, helpers_1.updateSecretValue)(process.env.SETUP_SECRET_ARN, JSON.stringify({ token: '' }));
    return response(200, 'Existing app set. Don\'t forget to set up the webhook.');
}
exports.handler = async function (event) {
    // confirm required environment variables
    if (!process.env.WEBHOOK_URL) {
        throw new Error('Missing environment variables');
    }
    const setupToken = (await (0, helpers_1.getSecretJsonValue)(process.env.SETUP_SECRET_ARN)).token;
    // bail out if setup was already completed
    if (!setupToken) {
        return response(200, 'Setup already complete. Put a new token in the setup secret if you want to redo it.');
    }
    if (!event.queryStringParameters) {
        return response(403, 'Wrong setup token.');
    }
    // safely confirm url token matches our secret
    const urlToken = event.queryStringParameters.token || event.queryStringParameters.state || '';
    if (urlToken.length != setupToken.length || !crypto.timingSafeEqual(Buffer.from(urlToken, 'utf-8'), Buffer.from(setupToken, 'utf-8'))) {
        return response(403, 'Wrong setup token.');
    }
    // handle requests
    try {
        const path = event.path ?? event.rawPath;
        const method = event.httpMethod ?? event.requestContext.http.method;
        if (path == '/') {
            return await handleRoot(event, setupToken);
        }
        else if (path == '/domain' && method == 'POST') {
            return await handleDomain(event);
        }
        else if (path == '/pat' && method == 'POST') {
            return await handlePat(event);
        }
        else if (path == '/complete-new-app' && method == 'GET') {
            return await handleNewApp(event);
        }
        else if (path == '/app' && method == 'POST') {
            return await handleExistingApp(event);
        }
        else {
            return response(404, 'Not found');
        }
    }
    catch (e) {
        console.error(e);
        return response(500, `<b>Error:</b> ${e}`);
    }
};
//# sourceMappingURL=data:application/json;base64,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