"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
const shared_interfaces_1 = require("./shared-interfaces");
class VirtualNodeBase extends cdk.Resource {
    constructor() {
        super(...arguments);
        this.backends = new Array();
        this.listeners = new Array();
    }
    /**
     * Add a Virtual Services that this node is expected to send outbound traffic to
     */
    addBackends(...props) {
        for (const s of props) {
            this.backends.push({
                virtualService: {
                    virtualServiceName: s.virtualServiceName,
                },
            });
        }
    }
    /**
     * Utility method to add an inbound listener for this virtual node
     */
    addListeners(...listeners) {
        if (this.listeners.length + listeners.length > 1) {
            throw new Error('VirtualNode may have at most one listener');
        }
        for (const listener of listeners) {
            const portMapping = listener.portMapping || { port: 8080, protocol: shared_interfaces_1.Protocol.HTTP };
            this.listeners.push({
                portMapping,
                healthCheck: renderHealthCheck(listener.healthCheck, portMapping),
            });
        }
    }
}
/**
 * Minimum and maximum thresholds for HeathCheck numeric properties
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_HealthCheckPolicy.html
 */
const HEALTH_CHECK_PROPERTY_THRESHOLDS = {
    healthyThreshold: [2, 10],
    intervalMillis: [5000, 300000],
    port: [1, 65535],
    timeoutMillis: [2000, 60000],
    unhealthyThreshold: [2, 10],
};
function renderHealthCheck(hc, pm) {
    if (hc === undefined) {
        return undefined;
    }
    if (hc.protocol === shared_interfaces_1.Protocol.TCP && hc.path) {
        throw new Error('The path property cannot be set with Protocol.TCP');
    }
    const healthCheck = {
        healthyThreshold: hc.healthyThreshold || 2,
        intervalMillis: (hc.interval || cdk.Duration.seconds(5)).toMilliseconds(),
        path: hc.path || (hc.protocol === shared_interfaces_1.Protocol.HTTP ? '/' : undefined),
        port: hc.port || pm.port,
        protocol: hc.protocol || pm.protocol,
        timeoutMillis: (hc.timeout || cdk.Duration.seconds(2)).toMilliseconds(),
        unhealthyThreshold: hc.unhealthyThreshold || 2,
    };
    Object.keys(healthCheck)
        .filter((key) => HEALTH_CHECK_PROPERTY_THRESHOLDS[key] &&
        typeof healthCheck[key] === 'number' &&
        !cdk.Token.isUnresolved(healthCheck[key])).map((key) => {
        const [min, max] = HEALTH_CHECK_PROPERTY_THRESHOLDS[key];
        const value = healthCheck[key];
        if (value < min) {
            throw new Error(`The value of '${key}' is below the minimum threshold (expected >=${min}, got ${value})`);
        }
        if (value > max) {
            throw new Error(`The value of '${key}' is above the maximum threshold (expected <=${max}, got ${value})`);
        }
    });
    return healthCheck;
}
/**
 * VirtualNode represents a newly defined AppMesh VirtualNode
 *
 * Any inbound traffic that your virtual node expects should be specified as a
 * listener. Any outbound traffic that your virtual node expects to reach
 * should be specified as a backend.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_nodes.html
 */
class VirtualNode extends VirtualNodeBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualNodeName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        this.mesh = props.mesh;
        this.addBackends(...props.backends || []);
        this.addListeners(...props.listener ? [props.listener] : []);
        const node = new appmesh_generated_1.CfnVirtualNode(this, 'Resource', {
            virtualNodeName: this.physicalName,
            meshName: this.mesh.meshName,
            spec: {
                backends: cdk.Lazy.anyValue({ produce: () => this.backends }, { omitEmptyArray: true }),
                listeners: cdk.Lazy.anyValue({ produce: () => this.listeners }, { omitEmptyArray: true }),
                serviceDiscovery: {
                    dns: props.dnsHostName !== undefined ? { hostname: props.dnsHostName } : undefined,
                    awsCloudMap: props.cloudMapService !== undefined ? {
                        serviceName: props.cloudMapService.serviceName,
                        namespaceName: props.cloudMapService.namespace.namespaceName,
                        attributes: renderAttributes(props.cloudMapServiceInstanceAttributes),
                    } : undefined,
                },
                logging: {
                    accessLog: {
                        file: {
                            path: '/dev/stdout',
                        },
                    },
                },
            },
        });
        this.virtualNodeName = this.getResourceNameAttribute(node.attrVirtualNodeName);
        this.virtualNodeArn = this.getResourceArnAttribute(node.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualNode`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualNode given an ARN
     */
    static fromVirtualNodeArn(scope, id, virtualNodeArn) {
        return new ImportedVirtualNode(scope, id, { virtualNodeArn });
    }
    /**
     * Import an existing VirtualNode given its name
     */
    static fromVirtualNodeName(scope, id, meshName, virtualNodeName) {
        return new ImportedVirtualNode(scope, id, {
            meshName,
            virtualNodeName,
        });
    }
}
exports.VirtualNode = VirtualNode;
function renderAttributes(attrs) {
    if (attrs === undefined) {
        return undefined;
    }
    return Object.entries(attrs).map(([key, value]) => ({ key, value }));
}
/**
 * Used to import a VirtualNode and read it's properties
 */
class ImportedVirtualNode extends VirtualNodeBase {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.virtualNodeArn) {
            this.virtualNodeArn = props.virtualNodeArn;
            this.virtualNodeName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualNodeArn).resourceName));
        }
        else if (props.virtualNodeName && props.meshName) {
            this.virtualNodeName = props.virtualNodeName;
            this.virtualNodeArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualNode`,
                resourceName: this.virtualNodeName,
            });
        }
        else {
            throw new Error('Need either arn or both names');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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