'''
Datasets

This file contains several datasets that are useful for testing and experimenting
To import you can use the following format:
    from reliability.Datasets import automotive
    failures = automotive().failures
    right_censored = automotive().right_censored

If you would like more information on a dataset type:
    from reliability.Datasets import automotive
    print(automotive().info)
    print(help(automotive))
'''

import pandas as pd
import numpy as np


class automotive:
    '''
    This dataset is relatively small and a challenging task to fit with any distribution due to its size and shape
    It also includes mostly right censored data which makes fitting more difficult.
    Sourced (with permission) from: V.V. Krivtsov and J. W. Case (1999), Peculiarities of Censored Data Analysis in Automotive Industry Applications - SAE Technical Paper Series, # 1999-01-3220
    '''

    def __init__(self):
        self.failures = [5248, 7454, 16890, 17200, 38700, 45000, 49390, 69040, 72280, 131900]
        self.right_censored = [3961, 4007, 4734, 6054, 7298, 10190, 23060, 27160, 28690, 37100, 40060, 45670, 53000, 67000, 69630, 77350, 78470, 91680, 105700, 106300, 150400]
        rc = len(self.right_censored)
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored'], 'Value': ['automotive', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)')]}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df


class mileage:
    '''
    This dataset is simple to fit. It contains 100 values with no right censoring. The data appears to be from a Normal Distribution.
    Sourced from Example 2.31 (page 63) of Reliability Engineering and Risk analysis 3rd Edition by M. Modarres, M. Kaminskiy, and V.V. Krivtsov
    '''

    def __init__(self):
        self.failures = [32797, 47119, 33532, 55627, 11538, 34107, 26704, 9979, 10014, 22159, 37603, 26830, 25210, 16946, 30368, 24571, 43665, 55269, 29760, 17438, 16768, 43154, 22418, 14525, 52448,
                         28968, 32628, 31388, 26252, 31565, 15261, 20269, 14318, 19938, 32231, 44404, 32176, 42911, 18318, 37623, 38071, 35589, 44264, 20588, 25746, 28109, 29807, 16735, 28688, 22532,
                         39485, 25056, 28127, 29015, 10539, 41821, 49436, 37752, 43939, 27322, 26713, 35390, 40902, 22456, 35138, 27837, 28219, 21293, 31084, 27037, 45283, 16651, 27300, 36837, 21075,
                         27836, 47590, 21248, 21757, 17861, 25754, 32677, 29180, 28185, 22374, 41267, 33703, 36258, 30935, 49432, 41064, 27812, 28433, 36531, 45554, 8734, 32914, 28172, 26208, 24993]
        self.right_censored = []
        rc = len(self.right_censored)
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored'], 'Value': ['mileage', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)')]}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df


class defective_sample:
    '''
    This dataset is heavily right censored with intermixed censoring (not all censored values are greater than the largest failure)
    It exhibits the behavior of a defective sample (aka. Limited fraction defective)
    Thanks to Alexander Davis for providing this dataset.
    '''

    def __init__(self):
        self.failures = [81, 163, 56, 86, 144, 47, 121, 56, 43, 106, 54, 50, 43, 107, 107, 22, 106, 106, 106, 71, 71, 68, 65, 65, 99, 70, 23, 34, 217, 81, 95, 78, 4, 183, 117, 36, 22, 5, 106, 36, 96, 6, 32, 239, 31, 232, 91, 182, 126, 55, 43, 40, 354, 170, 91, 51, 85, 58, 20, 238, 21, 21, 14, 12, 47, 32, 252, 144, 57, 51, 91, 90, 42, 90, 56, 56, 55, 55, 55, 55, 54, 54, 30, 28, 246, 89, 83, 22, 78, 88, 87, 84, 49, 44, 22, 89, 147, 25, 82, 56, 38, 34, 115, 86, 20, 43, 179, 116, 42, 85, 36, 196,
                         146, 84, 37, 32, 3, 279, 120, 90, 34, 301, 202, 27, 150, 89, 33, 169, 91, 50, 46, 45, 44, 44, 44, 43, 247, 114, 96, 45, 45, 45, 45, 45, 44, 44, 43, 43, 43, 26, 118, 25, 14, 189, 126, 125, 122, 122, 119, 117, 115, 111, 110, 109, 108, 101, 87, 53, 53, 43, 42, 35, 25, 25, 10, 26, 18, 5, 177, 113, 87, 91, 3, 90, 88, 35, 220, 15, 242, 91, 74, 9, 5, 47, 8, 67, 58, 8, 58, 42, 3, 237, 152, 109, 91, 2, 174, 48, 56, 19, 187, 91, 178, 99, 93, 70, 61, 55, 38, 98, 63, 81, 34, 74, 74,
                         33, 6, 18, 125, 76, 211, 167, 131, 42, 4, 253, 142, 38, 72, 27, 146, 202, 27, 70, 77, 83, 15, 18, 29, 97, 25, 210, 74, 36, 70, 76, 98, 38, 65, 159, 109, 86, 71, 71, 63, 63, 97, 70, 5, 67, 16, 39, 29, 29, 106, 24, 75, 77, 734, 712, 700, 651, 645, 633, 628, 606, 584, 561, 552, 517, 503, 500, 498, 485, 478, 467, 463, 459, 456, 433, 427, 424, 423, 420, 419, 413, 412, 402, 401, 387, 385, 378, 377, 375, 370, 370, 369, 368, 364, 363, 363, 362, 362, 362, 361, 361, 359, 359, 358,
                         350, 347, 347, 346, 343, 340, 339, 336, 334, 333, 333, 332, 331, 330, 328, 325, 323, 322, 320, 319, 317, 315, 315, 312, 310, 310, 309, 306, 306, 306, 304, 303, 303, 301, 300, 300, 299, 299, 298, 298, 297, 295, 295, 294, 294, 290, 288, 288, 287, 287, 287, 287, 287, 286, 286, 283, 281, 280, 280, 279, 278, 278, 278, 277, 277, 276, 276, 275, 273, 272, 272, 272, 271, 270, 270, 270, 269, 266, 265, 264, 263, 262, 260, 260, 259, 259, 259, 259, 259, 258, 254, 253, 252, 252, 252,
                         252, 252, 252, 251, 251, 251, 251, 249, 247, 247, 247, 246, 246, 246, 246, 246, 245, 245, 242, 242, 241, 240, 240, 238, 237, 236, 235, 234, 234, 233, 233, 232, 232, 232, 231, 231, 229, 229, 229, 229, 227, 227, 226, 226, 226, 225, 224, 224, 224, 224, 223, 223, 223, 223, 223, 221, 221, 221, 221, 220, 220, 220, 220, 219, 218, 218, 218, 218, 218, 217, 216, 215, 214, 214, 214, 212, 212, 212, 212, 212, 212, 211, 211, 210, 210, 210, 210, 209, 209, 209, 209, 208, 208, 208, 207,
                         207, 207, 207, 206, 205, 205, 205, 205, 204, 203, 203, 203, 203, 203, 202, 202, 201, 201, 201, 200, 200, 199, 199, 199, 198, 197, 197, 197, 197, 197, 197, 197, 196, 196, 196, 196, 196, 196, 196, 196, 195, 194, 194, 194, 194, 194, 193, 193, 193, 192, 192, 192, 192, 192, 191, 191, 190, 190, 189, 187, 187, 186, 185, 184, 184, 184, 184, 184, 184, 183, 183, 183, 183, 183, 182, 182, 182, 181, 180, 180, 180, 180, 179, 179, 179, 179, 179, 178, 178, 177, 177, 177, 176, 176, 175,
                         175, 175, 175, 175, 175, 175, 175, 174, 174, 174, 174, 174, 174, 173, 173, 173, 173, 172, 172, 172, 172, 172, 171, 171, 171, 171, 171, 171, 170, 170, 170, 170, 170, 170, 169, 168, 168, 168, 168, 168, 168, 168, 168, 167, 167, 167, 167, 167, 166, 166, 166, 166, 166, 165, 165, 165, 164, 164, 164, 164, 163, 163, 163, 162, 162, 162, 161, 161, 161, 161, 161, 160, 160, 159, 159, 159, 158, 158, 158, 158, 157, 157, 155, 154, 154, 154, 153, 153, 153, 153, 152, 152, 151, 151, 151,
                         150, 150, 149, 148, 148, 147, 147, 147, 147, 146, 146, 144, 143, 143, 143, 143, 143, 142, 142, 142, 141, 141, 141, 140, 140, 140, 140, 140, 139, 139, 139, 138, 138, 138, 137, 137, 137, 137, 137, 136, 136, 136, 136, 136, 134, 134, 134, 134, 134, 134, 134, 133, 133, 133, 133, 133, 133, 133, 133, 133, 133, 132, 132, 132, 132, 131, 131, 130, 130, 130, 129, 129, 129, 129, 129, 129, 129, 129, 129, 129, 129, 128, 128, 128, 128, 127, 127, 126, 126, 126, 126, 125, 124, 124, 124,
                         124, 123, 123, 123, 122, 122, 122, 122, 121, 121, 121, 120, 119, 119, 119, 118, 117, 117, 116, 115, 115, 115, 115, 115, 115, 114, 114, 114, 114, 114, 113, 113, 113, 113, 113, 113, 113, 113, 112, 112, 112, 112, 112, 112, 111, 111, 111, 110, 110, 110, 110, 110, 110, 109, 109, 109, 109, 109, 109, 108, 108, 108, 108, 108, 108, 107, 107, 107, 107, 106, 106, 105, 105, 105, 105, 104, 103, 102, 102, 102, 102, 101, 101, 101, 101, 101, 101, 100, 100, 100, 100, 99, 99, 99, 99, 99,
                         98, 98, 98, 98, 98, 98, 97, 97, 97, 97, 97, 97, 97, 96, 95, 95, 95, 95, 94, 94, 94, 94, 93, 93, 93, 93, 93, 93, 93, 93, 93, 93, 92, 92, 92, 92, 92, 92, 92, 92, 92, 91, 91, 91, 91, 90, 90, 90, 89, 89, 88, 88, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 86, 86, 86, 86, 86, 86, 86, 85, 85, 85, 85, 85, 85, 85, 85, 85, 84, 84, 84, 84, 84, 84, 84, 84, 84, 83, 83, 83, 83, 83, 83, 82, 82, 82, 82, 82, 81, 81, 81, 80, 80, 80, 80, 80, 80, 80, 79, 79, 79, 79, 79, 78,
                         78, 78, 78, 78, 78, 77, 77, 77, 77, 77, 77, 77, 76, 76, 76, 76, 75, 75, 74, 74, 74, 74, 73, 72, 72, 72, 72, 71, 71, 71, 71, 71, 71, 70, 70, 70, 70, 70, 70, 70, 69, 69, 69, 69, 69, 69, 68, 68, 68, 68, 68, 68, 68, 68, 68, 68, 66, 66, 66, 66, 65, 65, 65, 65, 65, 64, 64, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 61, 61, 61, 60, 60, 60, 60, 59, 59, 59, 59, 58, 58, 58, 58, 57, 57, 57, 57, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 55, 55, 55, 55, 55, 55, 54, 54, 54, 54, 54, 53,
                         53, 53, 53, 53, 53, 52, 52, 52, 52, 51, 51, 50, 50, 50, 50, 49, 49, 49, 49, 48, 48, 48, 48, 48, 48, 48, 48, 47, 47, 47, 47, 47, 47, 46, 46, 46, 46, 45, 45, 45, 44, 44, 44, 43, 43, 43, 43, 43, 43, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 41, 41, 41, 41, 41, 41, 40, 39, 39, 39, 39, 38, 38, 38, 38, 37, 37, 37, 37, 37, 37, 36, 36, 36, 36, 36, 36, 36, 36, 36, 36, 36, 35, 35, 35, 35, 34, 34, 33, 33, 33, 32, 32, 32, 32, 32, 32, 32, 32, 31, 31, 30, 30, 30, 30, 30,
                         30, 30, 29, 29, 29, 28, 28, 28, 27, 27, 27, 27, 26, 26, 26, 26, 25, 25, 25, 25, 24, 24, 24, 24, 24, 24, 23, 23, 23, 23, 23, 23, 23, 23, 23, 23, 22, 22, 22, 22, 21, 21, 21, 20, 20, 20, 20, 20, 19, 19, 19, 19, 19, 19, 18, 17, 17, 17, 17, 16, 15, 15, 15, 14, 14, 14, 14, 14, 13, 13, 12, 12, 12, 11, 10, 10, 9, 9, 8, 7, 7, 6, 6, 6, 5, 5, 5, 3, 3, 3, 2, 2, 2]
        self.right_censored = [922, 672, 580, 643, 425, 302, 38, 412, 195, 867, 61, 854, 176, 391, 364, 608, 646, 708, 432, 534, 171, 707, 377, 774, 455, 603, 209, 267, 227, 26, 253, 95, 90, 159, 350, 117, 176, 54, 88, 339, 203, 280, 335, 289, 125, 367, 43, 208, 291, 308, 323, 302, 311, 244, 94, 191, 205, 305, 240, 343, 70, 181, 193, 208, 333, 119, 238, 339, 43, 189, 176, 314, 16, 98, 13, 75, 246, 48, 152, 220, 240, 286, 6, 226, 240, 314, 125, 300, 133, 53, 150, 178, 221, 252, 329, 15, 46, 77,
                               196, 63, 127, 154, 184, 187, 166, 168, 195, 226, 265, 308, 333, 31, 152, 317, 106, 179, 228, 118, 182, 232, 301, 5, 36, 151, 207, 292, 333, 228, 260, 170, 248, 50, 169, 211, 36, 95, 102, 208, 288, 38, 44, 83, 154, 233, 253, 255, 270, 304, 92, 118, 183, 202, 204, 216, 287, 61, 271, 296, 33, 70, 270, 284, 173, 187, 248, 9, 19, 238, 20, 27, 127, 169, 191, 225, 247, 261, 16, 41, 105, 149, 237, 47, 47, 228, 265, 130, 192, 14, 59, 51, 63, 151, 147, 163, 123, 76, 160,
                               218, 162, 204, 224, 275, 21, 164, 207, 213, 244, 270, 175, 151, 199, 200, 226, 233, 249, 59, 129, 135, 204, 239, 41, 55, 115, 141, 162, 176, 198, 205, 216, 260, 100, 133, 163, 186, 233, 267, 84, 125, 199, 8, 86, 106, 130, 168, 212, 127, 127, 50, 154, 165, 209, 280, 70, 48, 63, 90, 118, 101, 185, 218, 240, 29, 114, 187, 191, 7, 7, 123, 5, 11, 49, 101, 120, 147, 186, 191, 244, 167, 86, 95, 168, 93, 143, 162, 21, 271, 6, 84, 179, 221, 133, 143, 183, 190, 211, 224,
                               225, 150, 168, 88, 245, 92, 97, 248, 86, 114, 60, 82, 85, 113, 200, 180, 61, 146, 87, 161, 204, 135, 153, 182, 75, 46, 151, 197, 11, 130, 18, 147, 112, 184, 71, 85, 150, 205, 179, 86, 78, 98, 111, 111, 129, 180, 117, 20, 87, 4, 146, 79, 198, 44, 64, 181, 25, 81, 162, 34, 134, 145, 110, 55, 45, 156, 210, 112, 179, 180, 6, 93, 9, 42, 179, 53, 56, 87, 112, 125, 77, 11, 159, 164, 94, 63, 146, 151, 163, 43, 6, 10, 31, 57, 66, 50, 95, 42, 62, 71, 114, 42, 46, 5, 12, 50,
                               63, 33, 58, 34, 56, 50, 46, 27, 22, 69, 84, 42, 83, 29, 47, 52, 65, 105, 19, 53, 4, 72, 52, 113, 49, 41, 46, 11, 30, 6, 15, 10, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 7, 7, 7, 7, 7, 7, 7, 7,
                               7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 14, 14, 14,
                               14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 18, 18, 18, 18, 18, 18, 18, 18, 18, 18, 18, 18, 18, 18, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 23, 23, 23,
                               23, 23, 23, 23, 23, 23, 23, 23, 23, 23, 23, 23, 24, 24, 24, 24, 24, 24, 24, 24, 24, 24, 24, 24, 24, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 26, 26, 26, 26, 26, 26, 26, 26, 26, 26, 26, 26, 26, 26, 26, 26, 26, 27, 27, 27, 27, 27, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 29, 29, 29, 29, 29, 29, 29, 29, 29, 29, 29, 29, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 31, 31, 31, 31, 31, 31, 31, 31, 31, 31, 31, 31, 31, 31, 31,
                               32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 34, 34, 34, 34, 34, 34, 34, 34, 34, 34, 34, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 36, 36, 36, 36, 36, 36, 36, 36, 36, 36, 36, 36, 36, 36, 37, 37, 37, 37, 37, 37, 37, 37, 37, 37, 37, 37, 37, 38, 38, 38, 38, 38, 38, 38, 38, 38, 39, 39, 39, 39, 39, 39, 39, 39, 39, 39, 39, 39, 40, 40, 40, 40,
                               40, 40, 40, 40, 40, 40, 41, 41, 41, 41, 41, 41, 41, 41, 41, 41, 41, 41, 41, 41, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 46, 46, 46, 46, 46, 46, 46, 46, 46, 46, 46, 47, 47, 47, 47, 47, 47, 47, 47, 47, 47, 47, 47, 48, 48, 48, 48,
                               48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 51, 51, 51, 51, 51, 51, 51, 51, 52, 52, 52, 52, 52, 52, 52, 52, 53, 53, 53, 53, 53, 53, 53, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56,
                               56, 56, 56, 56, 56, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
                               63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 67, 67, 67, 67, 67, 67, 67, 67, 67, 67, 68, 68, 68, 68, 68, 68, 68, 68, 68, 68, 68, 68, 68, 69, 69, 69, 69, 69, 69, 69, 69, 69, 69, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 71, 71, 71, 71, 71, 71, 71, 71,
                               71, 71, 71, 71, 71, 71, 71, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 74, 74, 74, 74, 74, 75, 75, 75, 75, 75, 75, 75, 75, 76, 76, 76, 76, 76, 76, 76, 76, 76, 76, 76, 76, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80,
                               80, 80, 80, 80, 81, 81, 81, 81, 81, 82, 82, 82, 82, 82, 82, 82, 82, 82, 82, 82, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 87, 87, 87, 87, 87, 87, 87, 87, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 89, 89, 89, 89,
                               89, 89, 89, 89, 89, 89, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 93, 93, 93, 93, 93, 93, 93, 93, 93, 93, 93, 93, 93, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 95, 95, 95, 95, 95, 95, 95, 95, 95, 95, 95, 95, 95, 95, 95, 95, 95, 96, 96, 96, 96, 96, 96, 96, 96, 96, 96, 96, 97, 97, 97, 97, 97, 97, 97, 97, 97,
                               97, 97, 97, 98, 98, 98, 98, 98, 98, 98, 98, 98, 98, 98, 98, 98, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 100, 100, 100, 100, 100, 100, 100, 100, 100, 100, 101, 101, 101, 101, 101, 101, 101, 101, 102, 102, 102, 102, 102, 102, 102, 102, 102, 102, 102, 102, 102, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 104, 104, 104, 104, 104, 104, 104, 104, 104, 104, 104, 104, 104, 104, 104, 104, 104, 104, 105, 105, 105, 105, 105, 105, 105, 105, 105, 105,
                               105, 105, 105, 105, 105, 105, 105, 105, 105, 105, 105, 105, 106, 106, 106, 106, 106, 106, 106, 106, 106, 106, 106, 106, 106, 106, 106, 107, 107, 107, 107, 107, 107, 107, 107, 107, 107, 107, 107, 107, 107, 107, 108, 108, 108, 108, 108, 108, 108, 108, 108, 108, 108, 108, 108, 108, 109, 109, 109, 109, 109, 109, 109, 109, 109, 109, 109, 109, 109, 109, 109, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 111, 111, 111, 111, 111, 111, 111, 111, 111, 111, 111,
                               111, 111, 111, 111, 111, 111, 111, 111, 111, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 113, 113, 113, 113, 113, 113, 113, 113, 113, 113, 113, 113, 113, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 115, 115, 115, 115, 115, 115, 115, 115, 115, 115, 115, 115, 115, 115, 115, 115, 116, 116, 116, 116, 116, 116, 116, 116, 116, 116, 116, 116, 116, 116, 117, 117, 117, 117, 117, 117, 117,
                               117, 117, 117, 117, 117, 117, 117, 118, 118, 118, 118, 118, 118, 118, 118, 118, 118, 118, 118, 118, 118, 118, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 120, 120, 120, 120, 120, 120, 120, 120, 120, 120, 120, 121, 121, 121, 121, 121, 121, 121, 121, 121, 122, 122, 122, 122, 122, 122, 122, 122, 122, 122, 122, 123, 123, 123, 123, 123, 123, 123, 123, 123, 123, 123, 123, 123, 124, 124, 124,
                               124, 124, 124, 124, 124, 124, 124, 124, 124, 124, 124, 124, 124, 124, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 126, 126, 126, 126, 126, 126, 126, 126, 126, 126, 126, 126, 126, 126, 126, 126, 126, 126, 126, 126, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 128, 128, 128, 128, 128, 128, 128, 128, 128, 128, 128, 128, 128, 128, 128, 128, 129, 129, 129, 129, 129, 129, 129, 129, 129, 130, 130,
                               130, 130, 130, 130, 130, 130, 130, 130, 130, 130, 130, 130, 130, 130, 130, 130, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 132, 132, 132, 132, 132, 132, 132, 132, 132, 132, 132, 132, 132, 132, 132, 132, 133, 133, 133, 133, 133, 133, 133, 133, 133, 133, 133, 133, 133, 133, 133, 134, 134, 134, 134, 134, 134, 134, 134, 134, 134, 134, 134, 134, 134, 134, 135, 135, 135, 135, 135, 135, 135, 135, 135, 135,
                               135, 135, 135, 135, 135, 135, 135, 136, 136, 136, 136, 136, 136, 136, 136, 136, 136, 137, 137, 137, 137, 137, 137, 137, 137, 137, 137, 138, 138, 138, 138, 138, 138, 138, 138, 138, 138, 138, 138, 138, 138, 138, 138, 138, 138, 139, 139, 139, 139, 139, 139, 139, 139, 139, 139, 139, 139, 139, 139, 139, 139, 139, 139, 140, 140, 140, 140, 140, 140, 140, 140, 140, 140, 140, 140, 140, 140, 140, 140, 140, 140, 140, 140, 140, 141, 141, 141, 141, 141, 141, 141, 141, 141, 141,
                               141, 141, 141, 141, 141, 141, 141, 142, 142, 142, 142, 142, 142, 142, 142, 142, 143, 143, 143, 143, 143, 143, 143, 143, 143, 143, 144, 144, 144, 144, 144, 144, 144, 144, 144, 144, 144, 144, 144, 144, 145, 145, 145, 145, 145, 145, 145, 145, 145, 145, 145, 145, 145, 145, 145, 145, 145, 145, 146, 146, 146, 146, 146, 146, 146, 146, 146, 146, 146, 146, 146, 146, 146, 146, 146, 147, 147, 147, 147, 147, 147, 147, 147, 147, 147, 147, 147, 147, 147, 147, 147, 147, 147, 147,
                               147, 147, 147, 147, 148, 148, 148, 148, 148, 148, 148, 148, 148, 148, 148, 148, 148, 148, 148, 148, 148, 148, 148, 149, 149, 149, 149, 149, 149, 149, 149, 150, 150, 150, 150, 150, 150, 150, 150, 150, 150, 150, 150, 150, 150, 151, 151, 151, 151, 151, 151, 151, 151, 151, 151, 152, 152, 152, 152, 152, 152, 152, 152, 152, 152, 152, 152, 152, 152, 152, 152, 152, 152, 153, 153, 153, 153, 153, 153, 153, 153, 153, 153, 153, 153, 153, 154, 154, 154, 154, 154, 154, 154, 154,
                               154, 154, 154, 154, 154, 154, 154, 154, 154, 154, 154, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 156, 156, 156, 156, 156, 156, 156, 156, 156, 156, 156, 156, 156, 156, 156, 157, 157, 157, 157, 157, 157, 157, 157, 157, 157, 157, 157, 157, 158, 158, 158, 158, 158, 158, 158, 158, 158, 158, 158, 158, 158, 158, 158, 158, 158, 158, 158, 158, 159, 159, 159, 159, 159, 159, 159, 159, 159, 159, 160, 160, 160, 160,
                               160, 160, 160, 160, 160, 160, 160, 160, 160, 160, 160, 160, 160, 160, 160, 160, 160, 160, 160, 160, 161, 161, 161, 161, 161, 161, 161, 161, 161, 161, 161, 161, 161, 161, 161, 161, 161, 161, 161, 161, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 163, 163, 163, 163, 163, 163, 163, 163, 163, 163, 163, 163, 163, 163, 163, 163, 163, 163, 164, 164, 164, 164, 164, 164, 164, 164, 164, 164, 164, 164, 164,
                               164, 164, 164, 165, 165, 165, 165, 165, 165, 165, 165, 165, 165, 165, 165, 165, 165, 165, 165, 165, 165, 165, 165, 165, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168,
                               168, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 170, 170, 170, 170, 170, 170, 170, 170, 170, 170, 170, 170, 170, 170, 170, 170, 171, 171, 171, 171, 171, 171, 171, 171, 171, 171, 172, 172, 172, 172, 172, 172, 172, 172, 172, 172, 172, 172, 172, 173, 173, 173, 173, 173, 173, 173, 173, 173, 173, 173, 173, 173, 173, 174, 174, 174, 174, 174, 174, 174, 174, 174, 174, 174, 174, 174,
                               174, 174, 175, 175, 175, 175, 175, 175, 175, 175, 175, 175, 175, 175, 175, 175, 175, 175, 175, 175, 175, 175, 176, 176, 176, 176, 176, 176, 176, 176, 176, 176, 176, 176, 176, 176, 176, 176, 176, 176, 176, 176, 177, 177, 177, 177, 177, 177, 177, 177, 177, 177, 177, 177, 177, 177, 177, 177, 177, 178, 178, 178, 178, 178, 178, 178, 178, 178, 178, 178, 178, 178, 178, 179, 179, 179, 179, 179, 179, 179, 179, 179, 179, 179, 179, 179, 179, 179, 179, 179, 179, 179, 180, 180,
                               180, 180, 180, 180, 180, 180, 180, 180, 180, 180, 180, 180, 181, 181, 181, 181, 181, 181, 181, 181, 181, 181, 181, 181, 181, 181, 181, 181, 181, 181, 181, 181, 182, 182, 182, 182, 182, 182, 182, 182, 182, 182, 182, 182, 182, 182, 182, 182, 182, 182, 182, 183, 183, 183, 183, 183, 183, 183, 183, 183, 183, 183, 183, 183, 183, 183, 183, 184, 184, 184, 184, 184, 184, 184, 184, 184, 184, 184, 184, 184, 184, 184, 184, 184, 184, 184, 185, 185, 185, 185, 185, 185, 185, 185,
                               185, 185, 185, 185, 185, 185, 185, 185, 185, 185, 185, 185, 185, 186, 186, 186, 186, 186, 186, 186, 186, 186, 186, 187, 187, 187, 187, 187, 187, 187, 187, 187, 187, 187, 187, 187, 187, 187, 187, 187, 188, 188, 188, 188, 188, 188, 188, 188, 188, 188, 188, 188, 189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 190, 190, 190, 190, 190, 190, 190, 190, 190, 190, 190, 190, 190, 190, 190, 190, 190, 191, 191, 191, 191, 191,
                               191, 191, 191, 191, 191, 191, 191, 191, 191, 191, 191, 191, 191, 191, 192, 192, 192, 192, 192, 192, 192, 192, 192, 192, 192, 192, 192, 192, 192, 192, 192, 193, 193, 193, 193, 193, 193, 193, 193, 193, 193, 193, 193, 193, 193, 194, 194, 194, 194, 194, 194, 194, 194, 194, 194, 194, 194, 194, 194, 194, 194, 195, 195, 195, 195, 195, 195, 195, 195, 195, 195, 195, 195, 195, 195, 195, 195, 196, 196, 196, 196, 196, 196, 196, 196, 196, 196, 196, 196, 196, 196, 196, 196, 196,
                               196, 196, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 198, 198, 198, 198, 198, 198, 198, 198, 198, 198, 198, 198, 198, 198, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199, 200, 200, 200, 200, 200, 200, 201, 201, 201, 201, 201, 201, 201, 201, 201, 201, 201, 201, 201, 201, 201, 201, 201, 202, 202, 202, 202, 202, 202, 202, 202, 202, 202, 202, 202, 202, 202, 203, 203, 203, 203, 203,
                               203, 203, 203, 203, 203, 203, 203, 203, 203, 203, 203, 204, 204, 204, 204, 204, 204, 204, 204, 204, 204, 204, 204, 204, 204, 204, 205, 205, 205, 205, 205, 205, 205, 205, 205, 206, 206, 206, 206, 206, 206, 206, 207, 207, 207, 207, 207, 207, 207, 207, 207, 207, 207, 207, 207, 207, 207, 207, 207, 207, 208, 208, 208, 208, 208, 208, 208, 208, 208, 208, 208, 208, 208, 208, 209, 209, 209, 209, 209, 209, 209, 209, 209, 209, 209, 209, 210, 210, 210, 210, 210, 210, 210, 210,
                               210, 210, 210, 210, 210, 210, 210, 210, 210, 210, 210, 210, 210, 210, 210, 210, 210, 210, 211, 211, 211, 211, 211, 211, 211, 211, 211, 211, 211, 211, 211, 211, 211, 211, 211, 211, 211, 212, 212, 212, 212, 212, 212, 212, 212, 212, 212, 212, 212, 212, 212, 212, 212, 212, 212, 213, 213, 213, 213, 213, 213, 213, 213, 213, 213, 213, 213, 213, 213, 213, 214, 214, 214, 214, 214, 214, 214, 214, 214, 214, 214, 215, 215, 215, 215, 215, 215, 215, 215, 215, 215, 215, 215, 215,
                               215, 215, 215, 215, 216, 216, 216, 216, 216, 216, 216, 216, 216, 216, 216, 216, 216, 216, 216, 216, 216, 216, 216, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 218, 218, 218, 218, 218, 218, 218, 218, 218, 218, 218, 218, 218, 218, 218, 219, 219, 219, 219, 219, 219, 219, 219, 219, 219, 219, 219, 219, 219, 219, 219, 219, 219, 219, 219, 219, 220, 220, 220, 220, 220,
                               220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 221, 221, 221, 221, 221, 221, 221, 221, 221, 221, 221, 221, 221, 221, 221, 221, 221, 222, 222, 222, 222, 222, 222, 222, 222, 222, 222, 222, 222, 222, 222, 222, 223, 223, 223, 223, 223, 223, 223, 223, 223, 223, 223, 223, 223, 223, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 225, 225, 225, 225, 225, 225, 225, 225, 225, 225, 225, 225, 225,
                               225, 225, 225, 226, 226, 226, 226, 226, 226, 226, 226, 226, 227, 227, 227, 227, 227, 227, 227, 227, 227, 227, 227, 227, 227, 227, 227, 228, 228, 228, 228, 228, 228, 228, 228, 228, 228, 228, 228, 228, 228, 228, 228, 228, 228, 229, 229, 229, 229, 229, 229, 229, 229, 229, 230, 230, 230, 230, 230, 230, 230, 230, 230, 230, 230, 230, 230, 230, 230, 230, 230, 230, 230, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231,
                               231, 231, 231, 231, 232, 232, 232, 232, 232, 232, 232, 232, 232, 232, 232, 232, 232, 232, 232, 232, 232, 233, 233, 233, 233, 233, 233, 233, 233, 233, 233, 233, 233, 233, 234, 234, 234, 234, 234, 234, 234, 234, 234, 234, 234, 234, 234, 234, 234, 234, 234, 234, 234, 235, 235, 235, 235, 235, 235, 235, 235, 235, 235, 235, 235, 235, 235, 235, 235, 235, 235, 235, 235, 236, 236, 236, 236, 236, 236, 236, 236, 236, 237, 237, 237, 237, 237, 237, 237, 237, 237, 237, 237, 237,
                               237, 237, 237, 237, 237, 237, 237, 237, 237, 237, 237, 237, 237, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 239, 239, 239, 239, 239, 239, 239, 239, 239, 239, 239, 239, 240, 240, 240, 240, 240, 240, 240, 240, 240, 240, 240, 240, 240, 241, 241, 241, 241, 241, 241, 241, 241, 241, 241, 241, 241, 241, 241, 241, 241, 242, 242, 242, 242, 242, 242, 243, 243,
                               243, 243, 243, 243, 243, 243, 243, 243, 243, 243, 243, 243, 244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 246, 246, 246, 246, 246, 246, 246, 246, 246, 246, 246, 246, 246, 246, 246, 246, 246, 247, 247, 247, 247, 247, 247, 247, 247, 247, 247, 247, 247, 247, 247, 247, 248, 248, 248, 248, 248, 248, 248, 248,
                               248, 248, 248, 248, 248, 248, 249, 249, 249, 249, 249, 249, 249, 249, 249, 249, 249, 249, 249, 249, 250, 250, 250, 250, 250, 250, 250, 250, 250, 250, 250, 250, 250, 250, 250, 250, 250, 250, 250, 250, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 253, 253, 253, 253,
                               253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 257, 257, 257, 257, 257, 257, 257, 257, 257, 257, 257, 257, 257, 257, 257, 257, 257, 257, 258, 258, 258,
                               258, 258, 258, 258, 258, 258, 258, 258, 258, 258, 258, 258, 258, 258, 258, 259, 259, 259, 259, 259, 259, 259, 259, 259, 259, 259, 259, 259, 259, 259, 259, 259, 259, 259, 259, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 261, 261, 261, 261, 261, 261, 261, 261, 261, 261, 261, 262, 262, 262, 262, 262, 262, 262, 262, 262, 262, 262, 263, 263, 263, 263, 263, 263, 263, 263, 263, 263, 263, 263, 263, 264, 264,
                               264, 264, 264, 264, 264, 264, 264, 264, 264, 264, 264, 264, 264, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 268, 268, 268, 268, 268, 268, 268, 268, 268, 268, 269, 269, 269, 269,
                               269, 269, 269, 269, 269, 269, 269, 269, 269, 269, 269, 269, 269, 269, 269, 270, 270, 270, 270, 270, 270, 270, 270, 270, 270, 270, 270, 270, 271, 271, 271, 271, 271, 271, 271, 271, 271, 271, 271, 271, 271, 271, 271, 271, 271, 271, 271, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 273, 273, 273, 273, 273, 273, 273, 273, 273, 273, 273, 273, 273, 273, 273, 273, 273, 273, 273, 274, 274, 274, 274, 274, 274,
                               274, 274, 274, 274, 274, 274, 274, 274, 274, 274, 274, 275, 275, 275, 275, 275, 275, 275, 275, 275, 275, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 277, 277, 277, 277, 277, 277, 277, 277, 277, 277, 277, 277, 277, 277, 277, 277, 277, 277, 277, 278, 278, 278, 278, 278, 278, 278, 278, 278, 278, 278, 278, 278, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279,
                               279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 283, 283, 283, 283, 283, 283, 283, 283, 283, 283,
                               283, 283, 283, 283, 283, 283, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 285, 285, 285, 285, 285, 285, 285, 285, 285, 285, 285, 285, 285, 285, 286, 286, 286, 286, 286, 286, 286, 286, 286, 286, 286, 286, 286, 286, 286, 286, 286, 286, 286, 286, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 288, 288, 288, 288, 288, 288, 288,
                               288, 288, 288, 288, 288, 288, 288, 288, 288, 288, 288, 288, 288, 288, 288, 289, 289, 289, 289, 289, 289, 289, 289, 289, 289, 289, 289, 289, 289, 289, 290, 290, 290, 290, 290, 290, 290, 290, 290, 290, 290, 290, 291, 291, 291, 291, 291, 291, 291, 291, 291, 291, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 293, 293, 293, 293, 293, 293, 293, 293, 293, 293, 293, 293, 293, 293, 293, 293, 293, 293, 293, 293, 293,
                               294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 295, 295, 295, 295, 295, 295, 295, 295, 295, 295, 295, 295, 295, 295, 295, 295, 295, 295, 295, 296, 296, 296, 296, 296, 296, 296, 296, 296, 296, 296, 296, 296, 296, 297, 297, 297, 297, 297, 297, 297, 297, 297, 298, 298, 298, 298, 298, 298, 298, 298, 298, 298, 298, 298, 298, 298, 298, 298, 298, 298, 298, 298, 298, 299, 299, 299, 299, 299, 299,
                               299, 299, 299, 299, 299, 299, 299, 299, 299, 299, 299, 299, 299, 299, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 303, 303, 303,
                               303, 303, 303, 303, 303, 303, 303, 303, 303, 303, 303, 303, 303, 304, 304, 304, 304, 304, 304, 304, 304, 304, 304, 304, 304, 304, 304, 304, 305, 305, 305, 305, 305, 305, 305, 305, 305, 305, 305, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 308, 308, 308, 308,
                               308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 310, 310, 310, 310, 310, 310, 310, 310, 310, 310, 310, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 312, 312, 312, 312, 312, 312, 312, 312, 312, 312, 312, 312, 312, 312,
                               313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 317, 317, 317, 317,
                               317, 317, 317, 317, 317, 317, 317, 317, 317, 317, 317, 317, 317, 317, 317, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 320, 320, 320, 320, 320, 320, 320, 320, 320, 320, 320, 320, 320, 320, 320, 320, 320, 320, 320, 320, 321, 321, 321, 321, 321, 321, 321, 321, 321, 321, 321, 321, 321, 321, 321, 321,
                               321, 321, 321, 321, 321, 321, 321, 321, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 324, 324, 324, 324, 324, 324, 324, 324, 324, 324, 324, 324, 324, 324, 325, 325, 325, 325, 325, 325, 325, 325, 325, 325, 325, 325, 326, 326, 326, 326, 326, 326,
                               326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 327, 327, 327, 327, 327, 327, 327, 327, 327, 327, 327, 327, 327, 327, 327, 327, 327, 327, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329,
                               330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 331, 331, 331, 331, 331, 331, 331, 331, 331, 331, 331, 331, 331, 331, 331, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 334,
                               334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336,
                               336, 336, 336, 336, 336, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 339, 339, 339, 339, 339, 339, 339, 339, 339, 339, 339, 339, 339, 340, 340, 340, 340, 340, 340, 340, 340, 340, 340, 340, 340, 340, 340, 340, 340, 340, 340,
                               340, 340, 340, 340, 340, 340, 340, 340, 340, 340, 340, 340, 341, 341, 341, 341, 341, 341, 341, 341, 341, 341, 341, 341, 341, 341, 341, 341, 341, 341, 341, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343,
                               343, 343, 343, 343, 343, 343, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 347, 347, 347, 347,
                               347, 347, 347, 347, 347, 347, 347, 347, 347, 347, 347, 347, 347, 347, 347, 347, 347, 347, 347, 347, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350,
                               350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353,
                               353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 356,
                               356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358,
                               358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361,
                               361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363,
                               363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364,
                               364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 365, 365, 365, 365, 365, 365, 365, 365, 365, 365, 365, 365, 366, 366, 366, 366, 366, 366, 366, 366, 366, 366, 367, 367, 367, 367, 367, 367, 367, 367, 367, 368, 368, 368, 368, 368, 368, 368, 368, 368, 369, 369, 369, 369, 369, 369, 369, 369, 369, 369, 369, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 371, 371, 371, 371, 371,
                               371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 374, 374, 374, 374, 374, 374, 374, 374, 374, 375, 375, 375, 375, 375, 375, 375, 375, 375, 375, 376, 376, 376, 376, 376, 376, 377, 377, 377, 377, 377, 377, 377, 377, 377, 378, 378, 378, 378, 378, 378, 378, 378, 378, 378, 378, 378, 378, 378, 379, 379, 379, 379, 379, 379, 379, 379, 379, 380, 380,
                               380, 380, 380, 380, 380, 380, 380, 380, 380, 380, 381, 381, 381, 381, 381, 382, 382, 382, 382, 382, 382, 382, 382, 382, 382, 382, 382, 382, 383, 383, 383, 383, 383, 383, 383, 383, 383, 383, 383, 383, 383, 384, 384, 384, 384, 384, 384, 384, 384, 384, 384, 384, 384, 384, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 386, 386, 386, 386, 386, 386, 386, 386, 386, 387, 387, 387, 387, 387, 387, 388, 388, 388, 388, 388,
                               388, 388, 388, 388, 388, 388, 388, 388, 388, 388, 388, 389, 389, 389, 389, 389, 389, 389, 389, 389, 389, 389, 389, 390, 390, 390, 390, 390, 390, 390, 390, 390, 391, 391, 391, 391, 391, 391, 391, 391, 391, 392, 392, 392, 392, 392, 392, 392, 392, 392, 392, 392, 392, 392, 392, 392, 392, 392, 392, 393, 393, 393, 393, 393, 393, 393, 393, 393, 393, 393, 394, 394, 394, 394, 394, 394, 394, 394, 394, 394, 394, 394, 395, 395, 395, 395, 395, 395, 395, 395, 396, 396, 396, 397,
                               397, 397, 398, 398, 398, 398, 398, 398, 398, 399, 399, 399, 399, 399, 399, 399, 399, 399, 399, 399, 400, 400, 400, 400, 400, 400, 401, 401, 401, 401, 401, 401, 402, 402, 402, 402, 402, 402, 403, 403, 403, 403, 403, 403, 403, 403, 403, 404, 404, 404, 404, 404, 404, 404, 404, 404, 404, 404, 405, 405, 405, 405, 405, 405, 405, 405, 405, 405, 405, 406, 406, 406, 406, 406, 406, 406, 406, 406, 406, 406, 406, 406, 406, 407, 407, 407, 407, 407, 407, 407, 407, 408, 408, 408,
                               408, 408, 408, 408, 408, 408, 408, 408, 408, 408, 408, 409, 409, 409, 409, 409, 410, 410, 410, 410, 410, 410, 410, 410, 410, 411, 411, 411, 412, 412, 412, 412, 412, 412, 412, 413, 413, 413, 413, 413, 414, 414, 414, 414, 414, 415, 415, 415, 415, 415, 416, 416, 416, 416, 416, 417, 417, 417, 417, 417, 418, 418, 418, 418, 418, 418, 418, 418, 418, 418, 418, 418, 418, 418, 418, 419, 419, 419, 420, 420, 420, 420, 420, 420, 420, 420, 420, 420, 420, 420, 420, 421, 421, 421,
                               421, 421, 421, 421, 421, 421, 421, 421, 421, 421, 421, 421, 422, 422, 422, 422, 422, 422, 422, 422, 422, 422, 422, 422, 422, 422, 422, 423, 423, 423, 423, 423, 423, 423, 423, 423, 423, 423, 424, 424, 424, 424, 424, 424, 424, 424, 424, 424, 424, 424, 425, 425, 425, 425, 425, 425, 425, 425, 426, 426, 426, 426, 426, 426, 426, 426, 426, 426, 426, 427, 427, 427, 427, 427, 427, 427, 427, 427, 427, 428, 428, 428, 428, 428, 428, 428, 428, 428, 429, 429, 429, 429, 430, 430,
                               430, 430, 430, 430, 430, 430, 431, 431, 431, 431, 431, 431, 431, 431, 431, 432, 432, 432, 433, 433, 433, 433, 433, 433, 433, 433, 433, 433, 433, 433, 434, 434, 434, 434, 434, 434, 434, 434, 434, 434, 434, 434, 434, 434, 434, 434, 435, 435, 435, 435, 435, 435, 435, 435, 435, 435, 435, 435, 435, 435, 435, 435, 435, 436, 436, 436, 436, 436, 437, 437, 437, 437, 437, 437, 437, 438, 438, 438, 438, 438, 438, 438, 438, 438, 438, 438, 438, 439, 439, 439, 439, 439, 439, 439,
                               440, 440, 440, 440, 440, 440, 440, 440, 440, 441, 441, 441, 441, 441, 441, 441, 441, 441, 441, 441, 441, 441, 442, 442, 442, 442, 442, 442, 442, 442, 442, 442, 442, 442, 443, 443, 444, 444, 444, 444, 444, 444, 445, 445, 445, 445, 445, 445, 445, 445, 445, 445, 445, 445, 445, 445, 446, 446, 446, 446, 446, 446, 446, 446, 447, 447, 447, 447, 447, 447, 447, 447, 447, 447, 447, 447, 447, 447, 447, 447, 447, 448, 448, 448, 448, 448, 448, 448, 448, 448, 448, 448, 448, 448,
                               448, 449, 449, 449, 449, 449, 449, 449, 449, 449, 449, 450, 450, 450, 450, 450, 450, 451, 451, 451, 451, 451, 451, 451, 451, 451, 452, 452, 452, 452, 452, 453, 453, 453, 453, 453, 453, 453, 454, 454, 454, 454, 454, 454, 454, 454, 454, 454, 454, 454, 455, 455, 455, 455, 455, 455, 455, 455, 455, 455, 455, 455, 455, 456, 456, 456, 456, 456, 456, 456, 456, 456, 456, 456, 456, 457, 457, 457, 458, 458, 458, 458, 458, 459, 459, 459, 459, 460, 460, 460, 460, 460, 460, 460,
                               460, 461, 461, 461, 461, 461, 461, 461, 461, 461, 461, 461, 461, 461, 462, 462, 462, 462, 462, 462, 462, 462, 462, 462, 462, 462, 462, 463, 463, 463, 463, 463, 463, 463, 463, 463, 463, 463, 463, 464, 464, 464, 464, 464, 464, 465, 465, 465, 465, 465, 466, 466, 466, 466, 466, 466, 467, 467, 467, 467, 467, 467, 467, 467, 467, 467, 468, 468, 468, 468, 468, 468, 468, 468, 469, 469, 469, 469, 469, 469, 469, 469, 469, 469, 469, 469, 469, 469, 469, 469, 469, 469, 469, 469,
                               469, 469, 470, 470, 470, 470, 470, 470, 470, 470, 470, 470, 470, 470, 470, 471, 471, 471, 471, 471, 472, 472, 472, 472, 472, 472, 472, 472, 472, 472, 472, 473, 473, 473, 473, 473, 473, 473, 473, 474, 474, 474, 474, 475, 475, 475, 475, 475, 475, 475, 475, 475, 475, 475, 475, 475, 475, 475, 475, 476, 476, 476, 476, 476, 476, 476, 476, 476, 476, 476, 476, 477, 477, 477, 477, 477, 477, 477, 477, 477, 477, 477, 477, 478, 478, 478, 478, 478, 478, 478, 478, 478, 479, 479,
                               479, 479, 479, 479, 479, 479, 479, 480, 480, 480, 480, 480, 481, 481, 481, 481, 481, 481, 481, 482, 482, 482, 482, 482, 482, 482, 482, 482, 482, 483, 483, 483, 483, 483, 483, 483, 484, 484, 484, 484, 484, 484, 484, 484, 484, 485, 485, 485, 485, 485, 485, 485, 485, 486, 486, 486, 486, 486, 486, 486, 486, 486, 486, 486, 486, 487, 487, 487, 487, 487, 487, 487, 487, 487, 487, 487, 487, 487, 488, 488, 488, 488, 488, 488, 488, 488, 488, 488, 488, 488, 488, 488, 488, 489,
                               489, 489, 489, 489, 489, 489, 489, 489, 489, 489, 489, 489, 489, 489, 490, 490, 490, 490, 490, 490, 490, 490, 490, 490, 490, 490, 490, 490, 491, 491, 491, 491, 491, 491, 491, 491, 491, 491, 491, 491, 491, 491, 492, 492, 492, 492, 492, 492, 493, 493, 493, 493, 493, 493, 493, 493, 493, 493, 493, 494, 494, 494, 494, 494, 494, 494, 494, 494, 494, 494, 494, 495, 495, 495, 495, 495, 495, 495, 495, 495, 495, 495, 495, 496, 496, 496, 496, 496, 496, 496, 496, 496, 496, 496,
                               496, 496, 497, 497, 497, 497, 497, 497, 497, 497, 497, 497, 497, 497, 497, 497, 497, 497, 497, 498, 498, 498, 498, 498, 498, 498, 498, 498, 498, 498, 498, 499, 499, 499, 499, 499, 499, 499, 499, 499, 499, 499, 499, 499, 500, 500, 500, 500, 500, 500, 500, 500, 500, 501, 501, 501, 501, 501, 501, 502, 502, 502, 502, 502, 502, 502, 502, 502, 503, 503, 503, 503, 503, 503, 503, 503, 503, 503, 503, 503, 503, 503, 503, 503, 503, 504, 504, 504, 504, 504, 504, 504, 504, 504,
                               504, 504, 504, 504, 504, 505, 505, 505, 505, 505, 505, 505, 506, 506, 506, 506, 506, 506, 506, 506, 506, 506, 506, 506, 506, 507, 507, 507, 507, 507, 507, 507, 507, 507, 507, 507, 508, 508, 508, 508, 508, 508, 509, 509, 509, 509, 509, 509, 509, 509, 509, 510, 510, 510, 510, 510, 510, 510, 510, 510, 510, 510, 510, 510, 510, 511, 511, 511, 511, 511, 511, 511, 511, 511, 511, 511, 511, 511, 511, 511, 511, 511, 511, 512, 512, 512, 512, 512, 512, 512, 512, 512, 512, 512,
                               512, 513, 513, 513, 513, 513, 513, 513, 513, 514, 514, 514, 514, 514, 514, 514, 514, 514, 514, 514, 514, 514, 514, 514, 515, 515, 515, 515, 515, 515, 515, 515, 515, 515, 515, 515, 515, 515, 515, 515, 515, 516, 516, 516, 516, 516, 516, 516, 516, 516, 516, 516, 516, 517, 517, 517, 517, 517, 517, 517, 517, 517, 517, 517, 517, 517, 517, 517, 517, 518, 518, 518, 518, 518, 518, 518, 518, 518, 518, 518, 519, 519, 519, 519, 519, 519, 519, 519, 519, 519, 519, 520, 520, 520,
                               521, 521, 521, 521, 521, 521, 521, 522, 522, 522, 522, 522, 522, 522, 523, 523, 523, 523, 523, 523, 523, 523, 523, 524, 524, 524, 524, 524, 524, 524, 524, 524, 524, 524, 524, 524, 524, 525, 525, 525, 525, 525, 525, 525, 525, 525, 525, 525, 525, 525, 525, 525, 525, 526, 526, 526, 526, 526, 526, 526, 526, 526, 526, 526, 526, 527, 527, 527, 527, 527, 528, 528, 528, 528, 528, 528, 528, 528, 529, 529, 529, 529, 529, 529, 530, 530, 530, 530, 530, 530, 530, 530, 530, 530,
                               530, 531, 531, 531, 531, 531, 531, 531, 532, 532, 532, 532, 532, 532, 532, 532, 532, 532, 532, 532, 532, 532, 532, 532, 532, 532, 533, 533, 533, 533, 533, 533, 533, 533, 533, 533, 533, 533, 533, 533, 533, 534, 534, 534, 534, 534, 534, 534, 534, 535, 535, 535, 535, 535, 535, 535, 535, 535, 536, 536, 536, 536, 536, 536, 536, 537, 537, 537, 537, 537, 537, 537, 537, 537, 537, 537, 538, 538, 538, 538, 538, 538, 538, 538, 538, 538, 538, 538, 538, 539, 539, 539, 539, 539,
                               539, 539, 539, 539, 539, 539, 539, 539, 539, 539, 539, 540, 540, 540, 540, 540, 540, 540, 540, 540, 541, 541, 541, 541, 541, 541, 541, 541, 541, 541, 542, 542, 542, 542, 542, 542, 542, 542, 542, 542, 542, 542, 542, 542, 542, 542, 543, 543, 543, 543, 543, 543, 543, 543, 543, 543, 544, 544, 544, 544, 544, 544, 545, 545, 545, 545, 545, 545, 545, 545, 545, 545, 546, 546, 546, 546, 546, 546, 546, 546, 546, 546, 546, 546, 546, 546, 546, 546, 546, 546, 547, 547, 547, 547,
                               547, 547, 547, 547, 547, 547, 548, 548, 548, 548, 548, 548, 548, 548, 548, 549, 549, 549, 549, 549, 549, 549, 549, 550, 550, 550, 550, 550, 550, 550, 550, 550, 550, 550, 550, 551, 551, 551, 551, 551, 551, 551, 551, 551, 551, 551, 551, 551, 551, 551, 552, 552, 552, 552, 552, 552, 552, 552, 552, 552, 552, 552, 552, 553, 553, 553, 553, 553, 553, 553, 553, 553, 553, 553, 553, 553, 553, 553, 553, 553, 553, 554, 554, 554, 554, 554, 554, 554, 554, 554, 554, 554, 554, 555,
                               555, 555, 555, 555, 555, 555, 555, 555, 555, 556, 556, 556, 556, 556, 556, 556, 557, 557, 557, 557, 557, 557, 558, 558, 558, 558, 558, 558, 558, 558, 558, 558, 558, 558, 559, 559, 559, 559, 559, 559, 559, 559, 559, 559, 560, 560, 560, 560, 560, 560, 560, 560, 560, 560, 560, 560, 560, 560, 561, 561, 561, 561, 561, 561, 561, 561, 561, 561, 561, 561, 561, 561, 562, 562, 562, 562, 562, 562, 562, 562, 562, 563, 563, 563, 563, 563, 563, 563, 563, 563, 564, 564, 564, 564,
                               564, 564, 564, 564, 564, 565, 565, 565, 565, 565, 565, 565, 565, 565, 565, 566, 566, 566, 566, 566, 566, 566, 566, 566, 566, 566, 566, 566, 567, 567, 567, 567, 567, 567, 567, 567, 567, 567, 567, 567, 567, 567, 567, 567, 567, 567, 567, 568, 568, 568, 568, 568, 568, 568, 568, 568, 569, 569, 569, 569, 569, 569, 569, 570, 570, 570, 570, 570, 570, 570, 570, 570, 570, 571, 571, 571, 571, 571, 571, 571, 571, 571, 572, 572, 572, 572, 572, 572, 572, 572, 573, 573, 573, 573,
                               573, 573, 573, 573, 573, 573, 573, 574, 574, 574, 574, 574, 574, 574, 574, 574, 575, 575, 575, 575, 575, 575, 575, 575, 576, 576, 576, 576, 576, 576, 576, 576, 576, 576, 576, 576, 576, 576, 576, 577, 577, 577, 577, 577, 577, 577, 578, 578, 578, 578, 578, 578, 578, 579, 579, 579, 579, 579, 579, 579, 579, 579, 579, 580, 580, 580, 580, 580, 580, 580, 580, 580, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581,
                               582, 582, 582, 582, 582, 582, 582, 582, 582, 583, 583, 583, 583, 583, 583, 583, 583, 583, 583, 584, 584, 584, 584, 584, 584, 584, 584, 584, 584, 585, 585, 585, 585, 585, 585, 585, 585, 585, 585, 585, 586, 586, 586, 586, 586, 586, 586, 586, 586, 586, 586, 587, 587, 587, 587, 587, 587, 587, 588, 588, 588, 588, 588, 588, 588, 588, 588, 588, 588, 588, 588, 588, 588, 588, 588, 589, 589, 589, 589, 589, 589, 589, 589, 589, 589, 589, 589, 589, 589, 589, 589, 589, 589, 589,
                               590, 590, 590, 590, 590, 590, 590, 590, 590, 590, 590, 590, 590, 590, 590, 591, 591, 591, 591, 591, 591, 591, 591, 591, 591, 591, 591, 592, 592, 592, 592, 592, 592, 592, 592, 593, 593, 593, 593, 593, 594, 594, 594, 594, 594, 594, 594, 594, 594, 594, 594, 595, 595, 595, 595, 595, 595, 595, 595, 595, 595, 595, 595, 595, 595, 595, 595, 595, 595, 595, 595, 596, 596, 596, 596, 596, 596, 596, 596, 596, 596, 596, 597, 597, 597, 597, 597, 597, 597, 597, 597, 597, 597, 597,
                               598, 598, 598, 598, 598, 598, 598, 598, 598, 598, 598, 598, 598, 598, 599, 599, 599, 599, 599, 599, 599, 599, 599, 599, 599, 599, 599, 599, 599, 599, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 601, 601, 601, 601, 601, 601, 601, 601, 601, 601, 601, 602, 602, 602, 602, 602, 602, 602, 602, 602, 602, 602, 602, 602, 602, 602, 602, 602, 602, 602, 603, 603, 603, 603, 603, 603, 603, 603, 603, 603, 603, 604, 604, 604, 604, 604, 604, 604, 604,
                               605, 605, 605, 605, 605, 605, 605, 605, 605, 605, 605, 605, 605, 606, 606, 606, 606, 606, 606, 606, 606, 606, 606, 607, 607, 607, 607, 607, 607, 608, 608, 608, 608, 608, 608, 608, 608, 608, 608, 609, 609, 609, 609, 609, 609, 609, 609, 609, 609, 609, 609, 610, 610, 610, 610, 610, 610, 610, 610, 611, 611, 611, 611, 611, 611, 611, 611, 611, 612, 612, 612, 612, 612, 612, 612, 612, 613, 613, 613, 613, 613, 613, 613, 613, 613, 613, 613, 613, 613, 613, 613, 613, 613, 614,
                               614, 614, 614, 614, 614, 614, 614, 614, 614, 614, 614, 615, 615, 615, 615, 615, 615, 615, 615, 616, 616, 616, 616, 616, 616, 616, 616, 616, 616, 616, 616, 616, 616, 616, 616, 616, 616, 617, 617, 617, 617, 617, 617, 617, 617, 617, 617, 618, 618, 618, 618, 618, 618, 618, 618, 618, 618, 619, 619, 619, 619, 619, 619, 619, 619, 620, 620, 620, 620, 620, 620, 620, 620, 621, 621, 621, 621, 621, 621, 621, 621, 621, 621, 622, 622, 622, 622, 622, 622, 622, 622, 622, 622, 622,
                               622, 623, 623, 623, 623, 623, 623, 623, 623, 623, 623, 623, 623, 623, 623, 624, 624, 624, 624, 624, 624, 624, 624, 624, 624, 624, 625, 625, 625, 625, 625, 625, 625, 625, 625, 625, 625, 626, 626, 626, 626, 626, 626, 627, 627, 627, 627, 627, 627, 627, 627, 627, 627, 627, 627, 628, 628, 628, 629, 629, 629, 629, 629, 629, 629, 629, 629, 629, 629, 629, 629, 629, 629, 629, 629, 630, 630, 630, 630, 630, 630, 630, 630, 630, 630, 630, 630, 630, 630, 630, 630, 630, 630, 631,
                               631, 631, 631, 631, 631, 631, 631, 631, 631, 631, 631, 631, 631, 631, 632, 632, 632, 632, 632, 632, 632, 632, 632, 632, 632, 632, 632, 632, 632, 633, 633, 633, 634, 634, 634, 634, 634, 635, 635, 635, 635, 635, 635, 636, 636, 636, 636, 636, 636, 636, 636, 637, 637, 637, 637, 637, 637, 637, 637, 637, 637, 637, 637, 637, 637, 637, 637, 637, 637, 637, 638, 638, 638, 638, 638, 638, 638, 638, 638, 639, 639, 639, 639, 639, 639, 639, 640, 640, 640, 640, 640, 640, 640, 641,
                               641, 641, 641, 641, 641, 641, 641, 641, 641, 642, 642, 642, 642, 642, 642, 642, 642, 642, 642, 643, 643, 643, 643, 643, 643, 643, 643, 643, 643, 643, 644, 644, 644, 644, 644, 644, 644, 644, 644, 644, 644, 644, 645, 645, 645, 645, 645, 645, 645, 645, 645, 645, 645, 645, 645, 645, 646, 646, 646, 646, 646, 646, 646, 646, 646, 646, 646, 646, 647, 647, 647, 647, 647, 647, 647, 648, 648, 648, 648, 648, 648, 648, 649, 649, 649, 649, 649, 649, 649, 649, 649, 649, 649, 649,
                               649, 649, 650, 650, 650, 650, 650, 650, 650, 650, 650, 650, 650, 650, 650, 651, 651, 651, 651, 651, 651, 651, 651, 651, 651, 651, 651, 651, 651, 651, 651, 651, 651, 652, 652, 652, 652, 652, 652, 652, 652, 653, 653, 653, 653, 653, 653, 653, 653, 653, 653, 653, 654, 654, 654, 654, 654, 654, 654, 655, 655, 655, 655, 655, 655, 655, 656, 656, 656, 656, 656, 656, 656, 656, 656, 656, 657, 657, 657, 657, 657, 657, 657, 657, 657, 657, 657, 657, 657, 657, 657, 658, 658, 658,
                               658, 658, 658, 658, 658, 658, 659, 659, 659, 659, 659, 659, 659, 659, 660, 660, 660, 660, 660, 660, 660, 660, 660, 660, 661, 661, 661, 661, 661, 661, 661, 662, 662, 662, 662, 662, 662, 662, 662, 662, 662, 662, 663, 663, 663, 663, 663, 663, 663, 663, 663, 663, 663, 663, 663, 663, 664, 664, 664, 664, 664, 664, 664, 664, 664, 664, 665, 665, 665, 665, 665, 665, 665, 665, 665, 665, 665, 665, 665, 665, 665, 665, 665, 665, 665, 666, 666, 666, 666, 666, 666, 666, 666, 666,
                               666, 666, 667, 667, 667, 667, 667, 667, 667, 667, 667, 667, 668, 668, 668, 668, 668, 669, 669, 669, 669, 669, 669, 669, 669, 669, 669, 669, 670, 670, 670, 670, 670, 670, 670, 670, 670, 670, 670, 670, 670, 670, 671, 671, 671, 671, 671, 671, 671, 671, 671, 671, 671, 672, 672, 672, 672, 672, 672, 672, 672, 672, 672, 672, 672, 672, 672, 673, 673, 673, 673, 673, 673, 673, 673, 673, 673, 673, 673, 673, 673, 673, 673, 674, 674, 674, 674, 674, 674, 674, 674, 674, 674, 674,
                               674, 675, 675, 675, 675, 675, 675, 675, 675, 675, 676, 676, 676, 676, 676, 676, 676, 676, 676, 676, 676, 677, 677, 677, 677, 677, 677, 677, 677, 678, 678, 678, 678, 678, 678, 678, 678, 678, 678, 679, 679, 679, 679, 679, 679, 679, 679, 679, 679, 679, 680, 680, 680, 680, 680, 680, 680, 680, 680, 680, 680, 680, 680, 680, 680, 680, 680, 681, 681, 681, 681, 681, 681, 681, 681, 681, 682, 682, 682, 682, 682, 682, 682, 682, 682, 683, 683, 683, 683, 683, 683, 683, 683, 683,
                               683, 683, 683, 684, 684, 684, 684, 684, 684, 684, 684, 684, 684, 685, 685, 685, 685, 685, 685, 685, 685, 685, 686, 686, 686, 686, 686, 686, 686, 686, 686, 686, 686, 686, 686, 686, 686, 686, 686, 686, 686, 687, 687, 687, 687, 687, 687, 687, 687, 687, 687, 687, 688, 688, 688, 688, 688, 688, 688, 688, 689, 689, 689, 689, 689, 689, 689, 689, 689, 689, 689, 689, 689, 689, 689, 690, 690, 690, 690, 690, 690, 690, 690, 690, 691, 691, 691, 691, 691, 691, 691, 691, 692, 692,
                               692, 692, 692, 692, 692, 692, 692, 692, 693, 693, 693, 693, 693, 693, 693, 693, 693, 693, 693, 693, 693, 693, 694, 694, 694, 694, 694, 694, 694, 695, 695, 695, 695, 695, 695, 695, 695, 695, 696, 696, 696, 696, 696, 696, 696, 696, 696, 696, 697, 697, 697, 697, 697, 697, 697, 697, 697, 698, 698, 698, 698, 698, 698, 699, 699, 699, 699, 699, 699, 699, 699, 699, 699, 699, 700, 700, 700, 700, 700, 700, 700, 700, 700, 700, 700, 700, 701, 701, 701, 701, 701, 701, 701, 701,
                               701, 701, 701, 701, 701, 701, 701, 701, 701, 702, 702, 702, 702, 702, 702, 702, 702, 702, 702, 702, 703, 703, 703, 703, 703, 703, 703, 704, 704, 704, 704, 704, 704, 704, 704, 704, 704, 704, 705, 705, 705, 705, 705, 705, 705, 705, 706, 706, 706, 706, 706, 706, 706, 706, 706, 706, 706, 706, 706, 706, 706, 706, 706, 706, 706, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 708, 708, 708, 708, 708,
                               708, 708, 708, 708, 709, 709, 709, 709, 709, 709, 709, 709, 709, 709, 709, 709, 709, 709, 710, 710, 710, 710, 710, 710, 710, 710, 710, 710, 711, 711, 711, 711, 711, 711, 711, 711, 711, 711, 711, 711, 711, 711, 712, 712, 712, 712, 712, 712, 712, 712, 712, 712, 712, 712, 712, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 714, 714, 714, 714, 714, 714, 714, 714, 714, 714, 714, 714, 714, 714, 714, 714, 714,
                               714, 714, 714, 714, 715, 715, 715, 715, 715, 715, 715, 715, 715, 715, 715, 715, 715, 715, 715, 716, 716, 716, 716, 716, 716, 716, 716, 716, 716, 716, 716, 716, 717, 717, 717, 717, 717, 717, 717, 717, 717, 717, 717, 718, 718, 718, 718, 718, 718, 718, 718, 718, 718, 718, 718, 718, 718, 718, 718, 718, 718, 718, 718, 718, 719, 719, 719, 719, 719, 719, 719, 719, 719, 719, 719, 719, 719, 719, 720, 720, 720, 720, 720, 720, 720, 720, 720, 720, 720, 720, 720, 720, 720, 721,
                               721, 721, 721, 721, 721, 721, 721, 721, 721, 721, 721, 721, 721, 721, 721, 721, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 723, 723, 723, 723, 723, 723, 723, 723, 723, 723, 723, 723, 723, 723, 723, 723, 724, 724, 724, 724, 724, 724, 724, 724, 724, 724, 724, 724, 724, 724, 725, 725, 725, 725, 725, 725, 725, 725, 725, 725, 725, 725, 725, 725, 725, 726, 726, 726, 726, 726, 726, 726, 726, 726, 727,
                               727, 727, 727, 727, 727, 727, 727, 727, 727, 727, 727, 727, 727, 727, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 730, 730, 730, 730, 730, 730, 730, 730, 731, 731, 731, 731, 731, 731, 731, 731, 731, 731, 731, 732, 732, 732, 732, 733, 733, 733, 733, 734, 734, 734, 734, 734, 734,
                               734, 734, 734, 734, 735, 735, 735, 735, 735, 735, 735, 736, 736, 736, 736, 736, 736, 736, 736, 736, 736, 737, 737, 737, 737, 737, 737, 738, 738, 738, 738, 738, 738, 738, 738, 738, 738, 739, 739, 739, 739, 739, 739, 740, 740, 740, 740, 740, 740, 740, 740, 740, 740, 741, 741, 742, 742, 742, 742, 742, 742, 742, 742, 743, 743, 743, 743, 743, 743, 744, 744, 744, 744, 744, 744, 744, 744, 745, 745, 745, 745, 746, 746, 746, 746, 747, 747, 747, 747, 747, 747, 748, 748, 748,
                               748, 748, 748, 748, 749, 749, 749, 749, 749, 750, 750, 750, 750, 751, 751, 751, 751, 752, 752, 752, 753, 753, 753, 753, 753, 753, 754, 754, 754, 755, 755, 755, 755, 755, 755, 755, 755, 755, 756, 756, 756, 756, 756, 756, 757, 757, 757, 757, 757, 758, 758, 758, 758, 758, 758, 759, 759, 759, 759, 760, 760, 760, 760, 760, 760, 760, 761, 761, 761, 761, 762, 762, 763, 763, 763, 763, 764, 764, 764, 764, 764, 765, 765, 765, 765, 765, 765, 765, 765, 765, 765, 766, 766, 766,
                               766, 766, 766, 767, 767, 767, 767, 767, 767, 768, 768, 768, 769, 769, 769, 769, 769, 769, 769, 769, 769, 769, 770, 770, 770, 770, 770, 770, 770, 770, 771, 771, 771, 771, 772, 772, 772, 772, 773, 773, 773, 773, 773, 773, 773, 773, 773, 773, 773, 773, 774, 774, 774, 774, 775, 775, 775, 775, 775, 775, 776, 776, 776, 777, 777, 777, 777, 778, 778, 778, 778, 779, 779, 779, 779, 779, 779, 780, 780, 780, 780, 780, 781, 781, 781, 781, 782, 782, 782, 782, 782, 782, 782, 782,
                               782, 782, 782, 783, 783, 783, 783, 783, 784, 784, 784, 784, 784, 784, 785, 785, 785, 785, 785, 785, 785, 786, 786, 786, 786, 786, 786, 787, 787, 787, 787, 787, 788, 788, 788, 788, 789, 789, 790, 790, 790, 790, 790, 790, 791, 791, 791, 791, 791, 791, 791, 792, 792, 792, 792, 793, 793, 793, 793, 794, 794, 794, 794, 796, 796, 796, 796, 796, 797, 797, 797, 797, 797, 797, 798, 798, 798, 798, 798, 798, 798, 798, 799, 799, 799, 799, 799, 800, 800, 800, 800, 800, 801, 801,
                               801, 801, 802, 802, 802, 803, 803, 803, 803, 803, 803, 803, 804, 804, 805, 805, 805, 806, 806, 806, 807, 807, 808, 808, 809, 809, 810, 810, 810, 810, 811, 811, 811, 811, 811, 812, 812, 812, 812, 813, 813, 813, 813, 814, 814, 814, 814, 814, 814, 814, 815, 815, 815, 815, 816, 816, 816, 817, 817, 818, 818, 819, 819, 819, 819, 819, 819, 820, 820, 820, 820, 820, 822, 822, 822, 822, 822, 823, 823, 824, 824, 824, 824, 825, 825, 825, 825, 825, 825, 825, 826, 826, 826, 826,
                               826, 826, 826, 826, 826, 826, 826, 826, 827, 827, 827, 828, 828, 828, 828, 830, 830, 830, 830, 831, 831, 831, 831, 831, 831, 831, 832, 832, 832, 833, 833, 833, 833, 833, 833, 834, 834, 834, 834, 834, 834, 834, 834, 835, 835, 835, 835, 835, 835, 835, 836, 836, 836, 837, 837, 837, 837, 837, 837, 838, 839, 839, 839, 839, 840, 840, 840, 840, 841, 841, 841, 841, 841, 842, 842, 842, 842, 842, 843, 843, 844, 844, 844, 844, 845, 845, 846, 846, 846, 847, 847, 847, 847, 847,
                               848, 848, 848, 849, 849, 849, 849, 849, 849, 849, 850, 850, 850, 850, 851, 851, 851, 851, 851, 852, 852, 853, 853, 853, 853, 854, 854, 854, 854, 854, 854, 854, 854, 854, 855, 855, 855, 855, 855, 855, 856, 856, 857, 857, 857, 858, 858, 858, 858, 859, 859, 859, 860, 860, 860, 860, 860, 861, 861, 861, 862, 862, 862, 862, 862, 863, 863, 863, 863, 863, 863, 864, 864, 864, 864, 864, 865, 865, 866, 866, 866, 867, 867, 867, 867, 867, 867, 867, 868, 868, 868, 868, 868, 869,
                               869, 869, 869, 870, 870, 870, 870, 871, 871, 871, 872, 872, 872, 872, 872, 872, 873, 873, 873, 874, 874, 874, 874, 875, 875, 875, 875, 875, 875, 875, 876, 876, 877, 877, 877, 878, 878, 878, 878, 879, 879, 880, 880, 880, 881, 881, 881, 881, 882, 882, 882, 882, 883, 883, 883, 883, 883, 883, 883, 883, 884, 884, 884, 885, 885, 886, 887, 887, 887, 887, 888, 888, 888, 888, 889, 889, 889, 889, 889, 890, 890, 890, 890, 890, 891, 891, 891, 892, 892, 892, 892, 892, 892, 892,
                               893, 893, 893, 893, 894, 894, 894, 895, 895, 895, 896, 897, 897, 897, 897, 897, 897, 898, 898, 898, 898, 899, 899, 899, 899, 900, 900, 901, 901, 901, 901, 901, 902, 902, 902, 902, 902, 902, 902, 903, 903, 903, 903, 903, 903, 904, 904, 904, 904, 904, 904, 904, 905, 905, 905, 905, 906, 906, 906, 906, 906, 906, 906, 908, 908, 908, 908, 909, 909, 909, 909, 910, 910, 910, 911, 911, 912, 912, 913, 913, 913, 914, 914, 915, 915, 915, 916, 916, 916, 916, 916, 916, 916, 917,
                               917, 917, 918, 918, 919, 919, 919, 920, 920, 920, 921, 921, 921, 921, 921, 922, 922, 923, 923, 923, 923, 923, 924, 924, 924, 924, 924, 924, 924, 924, 925, 925, 925, 927, 927, 927, 928, 929, 929, 930, 930, 930, 930, 930, 931, 931, 931, 931, 931, 931, 931, 932, 932, 932, 933, 933, 933, 933, 933, 933, 934, 934, 934, 935, 935, 935, 935, 935, 935, 936, 936, 936, 937, 937, 937, 937, 937, 938, 938, 938, 938, 938, 938, 939, 939, 939, 939, 939, 940, 940, 941, 941, 942, 942,
                               942, 942, 942, 943, 944, 945, 945, 945, 945, 945, 946, 946, 948, 949, 949, 950, 951, 951, 952, 952, 952, 952, 952, 952, 952, 953, 953, 953, 953, 954, 954, 954, 955, 955, 956, 956, 957, 958, 959, 959, 960, 960, 960, 960, 961, 961, 961, 962, 962, 962, 962, 962, 962, 962, 962, 962, 963, 963, 963, 963, 964, 965, 965, 966, 966, 966, 966, 967, 968, 968, 970, 970, 970, 971, 971, 971, 971, 972, 973, 973, 973, 974, 974, 975, 975, 976, 976, 976, 976, 977, 977, 977, 978, 978,
                               978, 979, 979, 979, 980, 980, 980, 980, 980, 980, 980, 981, 981, 981, 982, 982, 982, 982, 982, 983, 984, 986, 986, 986, 987, 987, 987, 988, 988, 988, 988, 988, 988, 990, 990, 990, 991, 991, 991, 992, 992, 993, 993, 994, 994, 995, 996, 996, 997, 997, 998, 998, 999, 1000, 1000, 1001, 1001, 1001, 1001, 1001, 1001, 1002, 1002, 1002, 1004, 1004, 1005, 1005, 1007, 1007, 1007, 1007, 1008, 1009, 1009, 1012, 1013, 1013, 1013, 1014, 1014, 1015, 1015, 1015, 1016, 1016, 1016,
                               1016, 1016, 1016, 1017, 1018, 1018, 1018, 1019, 1019, 1020, 1021, 1022, 1022, 1022, 1023, 1023, 1023, 1024, 1024, 1026, 1026, 1027, 1028, 1028, 1028, 1029, 1029, 1029, 1029, 1030, 1031, 1031, 1033, 1034, 1036, 1038, 1038, 1039, 1040, 1040, 1040, 1041, 1041, 1042, 1042, 1043, 1044, 1045, 1046, 1046, 1046, 1049, 1049, 1049, 1050, 1051, 1051, 1052, 1052, 1053, 1053, 1056, 1057, 1058, 1058, 1059, 1061, 1062, 1063, 1064, 1064, 1065, 1065, 1067, 1067, 1068, 1068, 1070,
                               1071, 1071, 1071, 1074, 1075, 1075, 1075, 1076, 1076, 1077, 1078, 1079, 1080, 1081, 1081, 1081, 1083, 1083, 1084, 1086, 1086, 1090, 1090, 1091, 1092, 1092, 1092, 1094, 1094, 1120, 1139]
        rc = len(self.right_censored)
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored'], 'Value': ['defective_sample', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)')]}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df


class mixture:
    '''
    This dataset is a mixture model with heavy censoring (97.90622%)
    It is best modelled using a Weibull Mixture Model.
    '''

    def __init__(self):
        self.failures = [9907, 11573, 7612, 2461, 2213, 11817, 10810, 21775, 13309, 1450, 10596, 12477, 22572, 8329, 6858, 14260, 15460, 6024, 6016, 17405, 5466, 5403, 5308, 5201, 5151, 12624, 10500, 5454, 14467, 14460, 3884, 13753, 14193, 3516, 18303, 3460, 10715, 7086, 9091, 4738, 15882, 13627, 7114, 15721, 2948, 7132, 12398, 21207, 9437, 1621, 9288, 4357, 20216, 6439, 873, 5589, 16449, 763, 7549, 19787, 12672, 12000, 425, 425, 13168, 11701, 9824, 114, 5080, 5684, 9]
        self.right_censored = [720, 1044, 1062, 1098, 1044, 1170, 1170, 1254, 666, 1254, 1302, 1302, 1356, 1356, 1386, 1386, 1398, 1398, 1422, 1422, 1464, 1476, 1506, 1506, 1518, 1518, 1476, 1530, 1530, 1566, 1584, 1584, 1590, 1584, 1608, 1608, 1644, 1644, 1680, 1668, 1680, 1758, 1758, 1758, 1776, 1776, 1794, 1794, 1764, 1818, 1818, 1944, 1944, 1944, 1944, 1956, 1956, 1992, 1992, 1992, 2034, 1956, 2034, 2076, 2076, 2094, 2094, 2046, 2130, 2148, 2148, 2178, 2178, 2130, 2226, 2226, 2244, 2244,
                               2190, 2190, 2268, 2274, 2304, 2304, 2322, 2322, 2364, 2364, 2400, 2400, 2430, 2268, 2430, 2454, 2454, 2496, 2496, 2496, 2496, 2526, 2526, 2568, 2568, 2496, 2676, 2742, 2742, 2856, 2820, 2898, 2898, 2916, 2946, 2952, 2982, 2982, 3048, 3048, 3102, 3150, 3150, 3228, 3228, 3288, 3288, 3408, 3408, 3594, 3594, 3594, 3672, 3708, 3708, 3648, 3822, 3822, 3828, 3828, 3780, 3966, 3966, 3948, 4158, 4176, 4200, 4206, 4194, 4134, 4194, 4320, 4500, 4500, 4134, 4536, 4620, 4626,
                               4686, 4572, 5082, 5244, 6138, 6318, 7806, 7860, 8700, 7698, 7698, 7698, 7698, 7698, 7698, 9066, 9066, 7020, 10392, 9618, 10728, 11466, 11478, 11844, 702, 11844, 11844, 9618, 12696, 15270, 17460, 16632, 16752, 16752, 16632, 19104, 22392, 22764, 15270, 10890, 4902, 4902, 4902, 7980, 7980, 7980, 8526, 8526, 8526, 8706, 8706, 8568, 10722, 10722, 10722, 10722, 10722, 10722, 10722, 10722, 10998, 10998, 10722, 10722, 11262, 11262, 11460, 11628, 11460, 11628, 12984, 12984,
                               13092, 13092, 13410, 13410, 13452, 13452, 13452, 13452, 17274, 18378, 19470, 19656, 19656, 19836, 19836, 19836, 21474, 21660, 21660, 21660, 21846, 21846, 22578, 22764, 22944, 22944, 8370, 8370, 22944, 22944, 22392, 15300, 24402, 24582, 24954, 24954, 15402, 25134, 25134, 25320, 27462, 372, 372, 372, 372, 372, 372, 474, 474, 840, 858, 972, 876, 1056, 1128, 1230, 1230, 1446, 1446, 1608, 1632, 1686, 1722, 1944, 1944, 1944, 2010, 2052, 2076, 2160, 2160, 2364, 2388,
                               2616, 2616, 2676, 2676, 2862, 2862, 3048, 3048, 3228, 3234, 3252, 3276, 3612, 3630, 4134, 4134, 4152, 4152, 4326, 4428, 4722, 4722, 5250, 5250, 8166, 8532, 8532, 8532, 8532, 9024, 7494, 7494, 9432, 9432, 7494, 7494, 7368, 10350, 10302, 10302, 11094, 7368, 7368, 9966, 7368, 7368, 7368, 10632, 10632, 7494, 7494, 7368, 7368, 18744, 11934, 876, 1050, 1206, 1590, 1650, 1782, 1968, 3714, 4212, 5244, 5352, 5790, 6342, 9246, 10014, 30504, 6906, 8562, 10896, 11442, 15816,
                               300, 984, 1032, 3246, 3246, 3456, 3606, 3606, 3948, 3948, 4212, 5244, 18924, 19284, 25536, 25536, 25536, 25536, 504, 684, 852, 858, 894, 1398, 4134, 5088, 16920, 17460, 1976, 4377, 4377, 1948, 1678, 1678, 7480, 7480, 8782, 8782, 11350, 11350, 13018, 24988, 1949, 1949, 3359, 3359, 3317, 3317, 3575, 4157, 4517, 4517, 5039, 5039, 5039, 5039, 1764, 1764, 2070, 2070, 1794, 1794, 4224, 4704, 4704, 5040, 5040, 5076, 6342, 6342, 6342, 6342, 4716, 4716, 6912, 6912, 6912,
                               21852, 21852, 6492, 1861, 1861, 1729, 2023, 2023, 4159, 4159, 3535, 3535, 4255, 5041, 5041, 23251, 23251, 4172, 4172, 4640, 5042, 5042, 22610, 3237, 3237, 4509, 4509, 21135, 32685, 32859, 32685, 5092, 4264, 4264, 10750, 10750, 2027, 2027, 3323, 3323, 18233, 21732, 1332, 1332, 18427, 18427, 20971, 20971, 20756, 1622, 1622, 6062, 6062, 1527, 4517, 4517, 20759, 20759, 6258, 2233, 2239, 4681, 4681, 21157, 5647, 5647, 1425, 3053, 14904, 2903, 2903, 16836, 16836, 6271,
                               2312, 2312, 6273, 340, 2081, 21151, 6740, 1881, 6184, 23267, 23267, 10919, 10919, 2312, 2312, 1748, 2000, 2000, 669, 669, 2515, 2515, 16496, 1250, 1951, 1951, 9045, 9045, 22934, 21498, 2092, 2092, 2092, 2807, 2807, 2814, 2814, 2164, 4432, 1351, 1351, 1536, 442, 2568, 6725, 6725, 5100, 6508, 6508, 2683, 488, 7336, 2599, 2599, 4023, 712, 958, 21651, 18155, 22648, 24274, 761, 3536, 3326, 6891, 1297, 1300, 1491, 10628, 1697, 1697, 3582, 2248, 2248, 21714, 2679, 25817,
                               6888, 6888, 1724, 26307, 26307, 23299, 2375, 1555, 1555, 19626, 5799, 5799, 14703, 6910, 6910, 2388, 4483, 2342, 6924, 6924, 6879, 6419, 1225, 1225, 1972, 1972, 930, 1764, 681, 681, 1235, 4333, 4146, 2886, 2886, 22793, 22793, 1621, 1621, 706, 19396, 715, 10428, 10428, 2785, 2481, 2392, 8438, 8438, 5676, 16637, 2504, 732, 1460, 1460, 3975, 5853, 5819, 990, 2109, 2109, 14533, 8444, 8444, 1574, 1574, 3659, 2497, 2497, 13251, 861, 6284, 4996, 7109, 7109, 898, 1463,
                               1463, 4393, 4393, 1250, 2738, 910, 839, 4681, 4419, 4419, 2154, 1859, 1859, 2774, 2229, 2229, 2827, 2827, 2784, 8149, 8149, 5162, 4760, 1888, 1888, 976, 10332, 10332, 2168, 4843, 4843, 6757, 6757, 4124, 4662, 29181, 6807, 6807, 5244, 2941, 2941, 9133, 2196, 2196, 20684, 5450, 2743, 2743, 6068, 1114, 1114, 6869, 2436, 2768, 2768, 1746, 2784, 2784, 2785, 2785, 5942, 2548, 9346, 1152, 1153, 4988, 1603, 1603, 2804, 23082, 1358, 2099, 2099, 20747, 20747, 7545, 7264,
                               2489, 1125, 6945, 6945, 1385, 2506, 2049, 2049, 6970, 6970, 6646, 3276, 3276, 2875, 2875, 1670, 6824, 6824, 1237, 1237, 1798, 1798, 8178, 3130, 3130, 1258, 1258, 5098, 2908, 1368, 3406, 3466, 3466, 7777, 7777, 3074, 3074, 1376, 7023, 7023, 2925, 2925, 3057, 3057, 7048, 7048, 1301, 7058, 4802, 4802, 7058, 1329, 2603, 2603, 5228, 3512, 8417, 1938, 2431, 2620, 1519, 1532, 1537, 2276, 2276, 3153, 3196, 9171, 1554, 6536, 2381, 1391, 8521, 8521, 8339, 8339, 8343, 8343,
                               3408, 3408, 2046, 5790, 1663, 3614, 1431, 3076, 3076, 1365, 1365, 9965, 1398, 1398, 1472, 1475, 10056, 1345, 7247, 3159, 3159, 6130, 1704, 3721, 2138, 2132, 1555, 3212, 3212, 11353, 1577, 2681, 3783, 3783, 2688, 7125, 3244, 3549, 2902, 2902, 6048, 6048, 5106, 5106, 7947, 3172, 3166, 15678, 15678, 2913, 2913, 15682, 20080, 9847, 7381, 7381, 3173, 3167, 5774, 2934, 2934, 1656, 1656, 2754, 2947, 1970, 3311, 3311, 1928, 1928, 3442, 1862, 1683, 8803, 8803, 6130, 6130,
                               2341, 2335, 2274, 2274, 2376, 2376, 3373, 7480, 3503, 3503, 1740, 1740, 1742, 5067, 3687, 3687, 1931, 5246, 5246, 3542, 3956, 2682, 3707, 9986, 7337, 3810, 3445, 12156, 2729, 2729, 2546, 3471, 3471, 2796, 7597, 7598, 7598, 8208, 3525, 7772, 26274, 1892, 1892, 6130, 5980, 9010, 9010, 4900, 7690, 2669, 2276, 3035, 3035, 3561, 3561, 4361, 4361, 7151, 7693, 2269, 5782, 21269, 2539, 2539, 3754, 2005, 2190, 3655, 3655, 2512, 2027, 2027, 3674, 3674, 7779, 7779, 3125,
                               2243, 4085, 16145, 2827, 2827, 10312, 5031, 2584, 2115, 3401, 3401, 6919, 6919, 2049, 2049, 5618, 6834, 8650, 8650, 2153, 2153, 7907, 7907, 2161, 8410, 8410, 2364, 2370, 5684, 5684, 4370, 8855, 16270, 3836, 3848, 4059, 3876, 3876, 2968, 2241, 2185, 2185, 7199, 6433, 6433, 8480, 7914, 3189, 3189, 8031, 8031, 8033, 8033, 8033, 8035, 8035, 2344, 2352, 2300, 8091, 8063, 8063, 6738, 21094, 8056, 8056, 4139, 4139, 4156, 4156, 6757, 7898, 2334, 2334, 2336, 6774, 3074,
                               3074, 5658, 4011, 3825, 3538, 5011, 2379, 24480, 4029, 2386, 2386, 18410, 4042, 4042, 7451, 3153, 3153, 7079, 2353, 2353, 4073, 4073, 2440, 15362, 28405, 3867, 2461, 4265, 4265, 8213, 8213, 5959, 5959, 10184, 4306, 3583, 2495, 20419, 2436, 2436, 4152, 9090, 6685, 6685, 6229, 4123, 4123, 6999, 5853, 6502, 8333, 2587, 3277, 4627, 8377, 8377, 4278, 2643, 8386, 8417, 330, 2664, 4289, 6500, 12690, 7146, 3812, 5155, 5155, 2652, 2652, 2724, 2731, 3652, 2675, 3117, 2756,
                               4960, 5827, 2768, 4612, 4430, 11556, 6479, 4470, 7221, 23944, 5841, 2817, 20533, 7305, 4534, 20789, 4554, 8450, 4757, 2926, 2926, 2939, 3156, 11899, 11899, 451, 2960, 4630, 2963, 4627, 7924, 2993, 2995, 4938, 8858, 7458, 4674, 3781, 3781, 4342, 4342, 11283, 3073, 3073, 3268, 3096, 3096, 3113, 3113, 3116, 3117, 3122, 3304, 3304, 5420, 4958, 7571, 7571, 11837, 11747, 3140, 4880, 8626, 8626, 3142, 3144, 5154, 6387, 4953, 24273, 25599, 5062, 12337, 7616, 4951, 4951,
                               3191, 9449, 4106, 3383, 3383, 27467, 3204, 9542, 9542, 3946, 3946, 3395, 3955, 3955, 6614, 4859, 4298, 4298, 4349, 8630, 3243, 3243, 3249, 3256, 5765, 4724, 4724, 10977, 4666, 5957, 4367, 7612, 8162, 5319, 11417, 10226, 19408, 11866, 3860, 5651, 4027, 4027, 4027, 4027, 4027, 4027, 7004, 7004, 5710, 4032, 4032, 5658, 3305, 9701, 4957, 4957, 3816, 12062, 3323, 12066, 6975, 7566, 8605, 12317, 14686, 3297, 3297, 20249, 11586, 8858, 3380, 3821, 4032, 4453, 4453, 7689,
                               7097, 3397, 3397, 19101, 3426, 3434, 3440, 9282, 6086, 3375, 1054, 7659, 5178, 3946, 3993, 5122, 5313, 6603, 5504, 5148, 7356, 7797, 19594, 7561, 3544, 5284, 3728, 7484, 7207, 6887, 3572, 3575, 7231, 20027, 3592, 3592, 5342, 5264, 5264, 5609, 8568, 3631, 5467, 5798, 18449, 8261, 3877, 4898, 9896, 9257, 3698, 4074, 6999, 10689, 20979, 5357, 5357, 3858, 3938, 7686, 3752, 7499, 6503, 19927, 3775, 3775, 5766, 4383, 3789, 15608, 12374, 9575, 9635, 15614, 5769, 7511,
                               5026, 19680, 12343, 30356, 5949, 5764, 4154, 3882, 15173, 3847, 4946, 5138, 3856, 3858, 3858, 6607, 12229, 3806, 3806, 15944, 4065, 4065, 6951, 4478, 19991, 3938, 6802, 5601, 5601, 3958, 11269, 11269, 3963, 4246, 3969, 3969, 6249, 16051, 7637, 12002, 5463, 4033, 2715, 18659, 20268, 4836, 8045, 8679, 4736, 4690, 4140, 5541, 6682, 6682, 4087, 4087, 4341, 4163, 4170, 4170, 4814, 4186, 4174, 6025, 8042, 6114, 7531, 7531, 7452, 4245, 4552, 7578, 7578, 4654, 8375, 8389,
                               7575, 6339, 8595, 7966, 8589, 19767, 8986, 4992, 19240, 4303, 10390, 8260, 9166, 5745, 5973, 6382, 5985, 8050, 8050, 5986, 7094, 4348, 4841, 9698, 4355, 6366, 5717, 8082, 5498, 4385, 5154, 5158, 8331, 4396, 5501, 23515, 12620, 8056, 8056, 8120, 8120, 4981, 17994, 5498, 5498, 13028, 11758, 11758, 16578, 9500, 4487, 5667, 5554, 4493, 4860, 4500, 6438, 12715, 4519, 4524, 10712, 4528, 4528, 6178, 35078, 4547, 4547, 5099, 4549, 12765, 5478, 4571, 4581, 4581, 9423, 6412,
                               6466, 4605, 5159, 9556, 4621, 4621, 4639, 6372, 8656, 7750, 5734, 5739, 8313, 6737, 4691, 6349, 4707, 6537, 6537, 4713, 4713, 8422, 8422, 4722, 5111, 4747, 5120, 6401, 8495, 6592, 4959, 3633, 7873, 23608, 4985, 2646, 6159, 8019, 4813, 4814, 5934, 6653, 7946, 5906, 7302, 12764, 9766, 5452, 4918, 4925, 6571, 6571, 4945, 4945, 6988, 9391, 10261, 7012, 6070, 7349, 7349, 8625, 26090, 4985, 5896, 5896, 5896, 4992, 4992, 4925, 4925, 8465, 6356, 5020, 5432, 10623, 6861,
                               506, 7381, 7909, 5076, 5009, 5009, 6738, 10809, 10443, 6280, 6973, 6973, 6838, 9701, 5112, 8968, 6317, 13367, 5156, 5387, 5387, 9515, 5194, 5194, 5379, 13415, 4247, 5205, 19170, 24078, 6867, 6674, 11427, 5246, 7086, 22400, 13476, 7167, 5819, 5456, 5645, 9222, 9674, 7908, 7841, 7687, 8606, 9686, 13522, 6528, 6528, 7631, 7631, 5508, 6429, 5702, 5702, 6775, 13984, 14020, 6439, 7004, 5366, 5297, 5549, 5551, 7846, 12898, 7214, 10976, 7038, 7038, 5416, 13632, 9806, 6638,
                               5795, 7602, 6161, 6758, 6668, 5630, 7838, 7838, 13678, 7849, 7849, 5481, 5851, 5851, 5417, 16338, 9214, 9214, 5884, 11725, 2654, 6558, 6158, 6401, 5541, 7160, 6585, 6108, 6722, 5647, 9708, 8233, 2694, 5687, 9350, 9350, 6814, 7735, 9386, 10132, 9795, 6484, 5770, 7431, 6339, 5723, 5723, 5800, 25806, 6371, 6002, 5760, 6012, 8683, 5847, 3154, 2624, 9037, 3564, 5837, 5837, 6794, 6795, 5917, 5917, 8235, 6894, 6894, 6700, 6337, 7880, 6548, 7280, 7123, 13105, 2611, 8172,
                               7836, 7063, 8342, 7803, 8015, 7202, 8148, 9186, 6478, 12298, 9821, 9821, 6339, 6339, 8034, 6675, 6123, 7239, 6148, 6834, 7348, 6544, 6734, 6734, 9876, 10583, 10583, 6584, 10498, 3303, 10556, 7530, 8557, 8557, 8104, 6255, 7714, 15112, 8650, 8650, 8106, 7922, 8523, 9023, 6296, 13142, 11037, 11805, 9927, 7871, 9371, 9041, 9948, 12330, 7220, 7758, 9961, 6852, 6500, 7065, 9992, 7988, 6333, 6333, 7366, 4466, 7194, 8395, 8041, 1238, 12596, 3106, 11095, 10077, 10077, 8815,
                               8815, 6453, 8504, 13044, 6474, 6474, 6484, 7943, 9183, 10174, 10174, 7978, 6464, 6464, 9289, 7641, 7097, 8054, 25367, 6925, 6567, 6567, 10226, 14608, 10198, 7699, 6602, 7948, 6611, 7532, 8445, 7536, 7536, 7926, 9366, 8455, 11759, 14989, 7830, 6572, 7161, 6581, 6581, 8659, 9041, 9041, 10314, 7033, 4962, 10323, 10323, 7845, 6605, 6607, 10392, 10392, 2534, 10631, 7073, 6695, 12379, 6714, 10366, 23137, 6737, 7843, 15601, 7683, 7683, 7871, 7832, 9939, 7899, 6804, 7774,
                               6819, 12399, 9217, 8661, 7690, 10939, 7981, 5966, 11628, 7466, 13126, 6942, 10404, 12486, 10699, 14715, 10656, 3576, 10730, 10731, 11408, 9042, 11418, 6962, 9018, 8553, 11162, 7566, 7958, 7958, 8566, 8542, 7800, 7283, 8937, 7288, 7861, 7674, 7353, 7353, 18826, 10788, 10788, 7520, 9086, 7083, 8256, 8077, 8077, 2394, 8594, 7359, 7397, 7678, 20470, 12455, 13528, 7132, 8310, 8310, 9612, 9273, 14394, 10295, 3849, 7728, 15883, 15883, 16004, 8187, 7642, 9720, 402, 7626,
                               7666, 7804, 13903, 9915, 7309, 8375, 1005, 20583, 7259, 7807, 11389, 11389, 11625, 16768, 8412, 4464, 9008, 8826, 1585, 25251, 10505, 10370, 7403, 10370, 7648, 14513, 8340, 11072, 7793, 7793, 10476, 7431, 10180, 11436, 9639, 8541, 8562, 1645, 11119, 11119, 9080, 13716, 7930, 7514, 8020, 7555, 7549, 8470, 11212, 11165, 8689, 8700, 11310, 8521, 8570, 8570, 27303, 8726, 11281, 11281, 10601, 4481, 7657, 3294, 7948, 11376, 11396, 8245, 15248, 7695, 14821, 11446, 7741,
                               16512, 7928, 7930, 11399, 7755, 11415, 11415, 11228, 14131, 8097, 8100, 9738, 9084, 8909, 7997, 7998, 9056, 11252, 10038, 8845, 16630, 18663, 10822, 9683, 10008, 11943, 8976, 8246, 11955, 9177, 8131, 8792, 8822, 9009, 9557, 9011, 12737, 7943, 11592, 8519, 9328, 8225, 15486, 27859, 8099, 8100, 13890, 8389, 13808, 8993, 14863, 8964, 8229, 10107, 11775, 26233, 11054, 11740, 23237, 9540, 8488, 11839, 12393, 8148, 12831, 10075, 8714, 9797, 8171, 9534, 11826, 13525,
                               8570, 9604, 8206, 8429, 8210, 11933, 8226, 12358, 11247, 8256, 10063, 11649, 10504, 13410, 17120, 10990, 11831, 10483, 9882, 10238, 15407, 6352, 10270, 10270, 16669, 10538, 17096, 10329, 13592, 9819, 2704, 10548, 8357, 9276, 10209, 8929, 11159, 12728, 15690, 12049, 15514, 10373, 9356, 18737, 12809, 9056, 3302, 10521, 10583, 348, 9852, 9479, 13158, 9762, 8919, 9707, 9205, 9739, 9824, 12140, 9222, 9591, 9851, 8859, 8911, 18807, 10327, 8739, 804, 8747, 9332, 9120,
                               8937, 10587, 9130, 10122, 10669, 19732, 1678, 9528, 10159, 16478, 8829, 8829, 9011, 8843, 8843, 16403, 9953, 10657, 11314, 9076, 15736, 10011, 7787, 9188, 21397, 8966, 9701, 29541, 9156, 8985, 3909, 10860, 10507, 2696, 9605, 11588, 10908, 9264, 9091, 9091, 9647, 11038, 9467, 10942, 10019, 12066, 9354, 9524, 9321, 9332, 11054, 12906, 11480, 14848, 11785, 12877, 5227, 12680, 11084, 10583, 9428, 3282, 16058, 11046, 9837, 9334, 9469, 11815, 11238, 22862, 9302, 770,
                               9493, 11006, 13028, 3416, 9892, 11711, 6542, 9547, 10939, 13000, 9395, 10501, 13075, 9626, 10425, 12946, 10429, 4967, 11321, 9655, 10031, 9482, 13136, 9674, 17819, 10059, 9512, 9519, 9519, 9526, 10109, 12662, 10660, 9596, 4247, 13329, 11213, 12203, 11613, 10053, 10033, 11095, 688, 2443, 12382, 29646, 13351, 10268, 9676, 9676, 3562, 10824, 10289, 11035, 9749, 11034, 9935, 12726, 9372, 9968, 9789, 10927, 10002, 21738, 9881, 10297, 11205, 10228, 10781, 10235, 10428,
                               10059, 10444, 10716, 10074, 10282, 11018, 11165, 11244, 9933, 10521, 4426, 11159, 10358, 12517, 10107, 13769, 10034, 11689, 11000, 11147, 11045, 10419, 2370, 10425, 11145, 10066, 12445, 3147, 14669, 13797, 748, 10098, 10473, 4173, 10293, 10301, 22579, 10092, 10092, 10158, 12032, 14166, 10817, 11289, 1609, 10205, 2334, 11729, 11134, 11470, 17976, 11317, 10499, 16252, 19847, 13886, 27227, 14107, 29941, 11023, 12262, 7458, 6431, 2528, 12318, 12387, 12933, 11858,
                               19318, 11297, 10382, 11270, 10953, 10584, 10409, 18662, 5394, 10611, 10800, 11520, 12553, 11365, 3540, 10449, 10449, 11373, 17400, 19862, 12510, 3788, 11998, 11057, 22467, 1841, 10906, 2938, 11678, 9760, 3071, 1632, 10609, 19766, 12118, 11712, 5897, 12169, 14281, 7106, 11734, 12237, 11010, 14293, 17637, 6228, 12180, 12180, 14324, 4214, 13650, 10901, 11094, 17243, 14983, 11425, 14380, 11583, 10766, 14258, 11149, 15255, 11721, 10368, 14459, 12079, 13861, 11773,
                               17041, 11778, 17814, 13091, 11454, 17645, 4991, 8358, 13665, 3070, 11313, 11334, 24000, 13029, 9284, 11530, 10997, 11000, 13617, 15724, 12117, 12460, 11034, 13394, 29499, 2280, 11607, 14844, 13349, 14404, 11278, 13038, 18671, 12645, 14200, 14734, 14038, 11299, 24493, 11510, 11511, 18097, 14278, 15314, 12615, 11550, 17921, 1740, 16589, 12665, 13380, 12324, 11928, 12563, 15229, 11851, 13238, 13206, 11514, 22462, 13329, 14473, 12072, 11744, 11384, 11388, 15039, 18974,
                               12375, 13102, 11825, 11467, 12025, 19060, 11499, 11499, 12052, 15151, 11892, 15194, 12647, 11563, 11571, 10166, 15303, 12274, 11585, 22240, 11974, 11799, 11630, 12735, 12960, 13650, 15302, 26112, 11675, 13321, 14054, 12784, 1283, 11885, 18297, 11733, 11733, 13959, 353, 14357, 12498, 12049, 18781, 13738, 13732, 12275, 12786, 780, 12382, 10799, 8247, 13804, 13005, 13980, 18606, 17746, 12838, 12111, 14296, 14377, 13040, 11943, 12324, 11961, 11966, 15920, 11940, 12629,
                               1682, 2836, 12051, 19361, 13493, 17193, 18917, 13392, 14924, 15740, 6636, 18943, 26445, 13829, 13564, 15475, 12257, 13175, 18769, 13703, 8749, 16825, 17926, 14283, 14213, 15321, 13256, 10197, 18219, 19225, 17222, 31374, 12941, 15433, 15897, 18989, 12805, 12832, 18574, 14451, 4909, 18062, 12514, 19068, 16212, 19743, 18117, 19551, 31049, 13840, 14545, 13741, 14357, 13035, 14312, 6780, 24549, 17284, 5001, 12766, 12801, 17299, 18038, 2082, 12820, 13930, 17841, 12856,
                               13996, 19532, 13084, 17010, 12901, 14868, 17484, 12912, 13094, 16016, 2650, 12950, 18591, 10549, 14077, 16617, 14938, 19549, 13196, 13021, 18239, 14158, 5136, 13065, 13585, 15437, 13476, 15544, 33197, 17072, 16596, 18136, 13511, 13457, 1327, 32365, 13191, 9001, 13251, 13624, 13625, 13454, 16844, 15173, 13350, 13351, 13724, 13719, 14468, 17211, 1657, 13414, 13794, 21826, 18977, 26664, 13823, 14742, 13833, 13841, 13478, 17141, 17702, 14038, 9602, 13528, 3850, 15369,
                               13535, 27201, 19067, 11986, 15674, 15926, 13645, 17999, 14308, 13822, 13829, 9359, 13859, 5851, 16687, 14261, 20905, 10277, 33454, 16339, 9479, 15179, 13936, 33482, 15081, 15762, 15894, 14804, 3887, 14013, 15812, 19531, 18648, 17566, 15019, 1478, 15756, 20577, 16719, 16036, 13998, 14694, 29514, 16628, 14075, 16270, 6126, 14336, 17318, 19651, 14611, 18811, 20507, 33615, 19008, 19883, 14449, 14272, 12077, 6896, 20009, 14307, 14504, 15104, 17999, 17498, 14966, 10355,
                               15927, 14604, 15525, 5132, 15184, 7163, 1916, 16174, 16073, 15588, 11090, 4125, 14686, 18192, 15619, 14739, 22302, 4380, 14598, 17942, 15143, 21606, 19761, 15175, 15017, 16372, 14666, 18993, 17261, 16638, 17457, 17012, 16814, 14904, 17022, 16018, 27981, 18136, 17774, 14869, 20064, 26505, 15349, 16894, 18452, 14807, 19390, 18499, 15035, 16144, 17545, 16698, 8699, 16173, 1286, 18396, 5978, 19469, 17070, 18285, 17774, 15561, 18095, 19731, 19607, 16161, 4026, 16801,
                               19969, 18975, 8457, 15338, 15353, 15181, 2985, 15199, 17767, 17776, 27245, 15457, 17196, 3997, 15874, 15374, 19512, 3582, 17494, 17502, 17278, 15803, 16037, 15459, 18082, 14210, 15466, 8910, 15469, 16933, 17374, 16584, 16590, 21837, 17022, 18086, 15524, 17648, 8732, 18323, 15316, 15775, 15776, 16289, 17607, 17463, 16021, 17973, 20117, 17000, 19933, 7492, 16910, 34036, 20105, 19673, 17382, 18588, 3834, 15463, 16099, 18307, 16225, 17191, 15992, 2268, 16213, 17535,
                               19706, 18409, 17825, 19473, 3840, 17273, 1918, 17473, 8158, 8638, 16763, 16776, 17311, 10292, 16218, 16590, 16226, 14115, 8992, 16320, 3924, 11943, 21697, 17505, 3264, 16477, 18062, 21147, 1295, 16989, 17993, 18595, 17613, 16542, 17837, 22059, 16562, 18943, 18881, 17966, 21081, 18477, 19430, 18190, 17057, 20317, 19555, 20918, 17829, 20814, 18269, 20358, 21079, 16801, 17408, 15190, 3468, 16848, 22798, 17605, 20210, 17275, 17924, 19689, 19667, 19240, 17354, 13497,
                               4978, 8925, 22507, 17991, 17991, 18940, 27742, 17884, 18968, 3227, 25548, 22295, 16898, 9836, 17606, 17639, 18269, 17239, 19860, 18954, 18922, 18995, 19849, 17719, 21892, 24818, 17878, 1662, 19897, 19261, 18022, 20932, 20762, 11986, 20542, 18364, 21064, 17771, 174, 18196, 26204, 17785, 20255, 1776, 20520, 18404, 25364, 20924, 22839, 23189, 22718, 10745, 19999, 19689, 23030, 19017, 12365, 20183, 22548, 20420, 5142, 1860, 21542, 22142, 20251, 18869, 20052, 5183,
                               20338, 16926, 20408, 4735, 22718, 26251, 9369, 11704, 15624, 20432, 2796, 24199, 7964, 19966, 19424, 7907, 19817, 19535, 24551, 20380, 4099, 20104, 16937, 24003, 21786, 20069, 19886, 21859, 19842, 22681, 26259, 12543, 19965, 12577, 20462, 20331, 11344, 20417, 21072, 21464, 11509, 24730, 24555, 24616, 24425, 19204, 11181, 20896, 20519, 6893, 6800, 21845, 20600, 22504, 22746, 18981, 4248, 20692, 20718, 21278, 15575, 25133, 22258, 21170, 23975, 22378, 25287, 24575,
                               20929, 23682, 20996, 21407, 24702, 1334, 21113, 24763, 4672, 5397, 13369, 21168, 26238, 21235, 5238, 21310, 4645, 24205, 21908, 25868, 21501, 22033, 6647, 954, 24038, 25932, 21684, 21773, 5640, 20101, 4548, 29139, 15200, 7746, 22575, 21978, 22034, 3168, 25575, 23080, 23959, 14214, 23995, 24819, 27719, 3294, 27686, 22505, 13204, 25797, 22491, 22499, 3900, 22573, 22593, 22598, 1782, 26401, 23416, 1986, 25360, 22789, 7639, 23927, 25066, 29352, 14028, 23060, 17189,
                               27541, 19661, 19016, 24844, 17656, 25830, 24878, 23643, 23251, 23371, 19933, 26156, 23476, 24629, 26739, 21079, 23656, 3097, 25277, 23847, 25344, 23817, 24457, 26334, 4323, 6990, 4309, 28465, 25665, 5168, 10800, 26451, 6129, 21346, 6229, 24777, 24784, 26482, 22744, 24712, 24926, 24782, 26451, 25906, 29324, 25210, 28214, 26570, 27142, 29348, 27180, 29447, 29228, 28936, 28949, 27483, 25307, 25057, 25716, 28518, 26305, 20952, 33549, 30344, 3162, 5990, 26576, 25158,
                               31073, 2544, 3965, 26749, 26796, 26823, 11329, 4424, 24574, 7157, 14972, 27579, 22932, 27416, 20829, 30738, 27820, 3733, 30622, 7871, 28227, 28963, 29797, 30514, 29186, 32676, 31632]
        rc = len(self.right_censored)
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored'], 'Value': ['mixture', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)')]}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df


class electronics:
    '''
    This dataset is heavily right censored without intermixed censoring (all censored values are greater than the largest failure)
    It is very difficult to fit and requires a specific combination of initial guess (least squares) and optimizer (TNC) to achieve the lowest log-likelihood.
    Thanks to Jiwon Cha for providing this dataset.
    '''

    def __init__(self):
        time = [220, 179, 123, 146, 199, 181, 191, 216, 1, 73, 44798, 62715, 81474, 80632, 62716]
        quantity = [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 817, 823, 815, 813, 804]
        category = ['F', 'F', 'F', 'F', 'F', 'F', 'F', 'F', 'F', 'F', 'C', 'C', 'C', 'C', 'C']
        data = {'time': time, 'quantity': quantity, 'category': category}
        self.dataframe = pd.DataFrame(data, columns=['time', 'quantity', 'category'])
        self.failures = [220, 179, 123, 146, 199, 181, 191, 216, 1, 73]
        self.right_censored = np.hstack([np.ones(817) * 44798, np.ones(823) * 62715, np.ones(815) * 81474, np.ones(813) * 80632, np.ones(804) * 62716])
        rc = len(self.right_censored)
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored'], 'Value': ['defective_sample2', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)')]}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df


class MCF_1:
    '''
    This dataset is formatted for use with the Mean Cumulative Function (MCF_parametric or MCF_nonparametric)
    It consists of failure times for 5 systems.
    It exhibits a fairly constant failure rate, appearing to be slightly increasing (beta > 1)
    '''

    def __init__(self):
        times = [
            [5, 10, 15, 17],
            [6, 13, 17, 19],
            [12, 20, 25, 26],
            [13, 15, 24],
            [16, 22, 25, 28]]
        self.times = times
        self.number_of_systems = len(times)


class MCF_2:
    '''
    This dataset is formatted for use with the Mean Cumulative Function (MCF_parametric or MCF_nonparametric)
    It consists of failure times for 56 systems.
    It exhibits an increasing failure rate at the start and a decreasing failure rate near the end.
    Due to this shape it is not fitted well by the power law model used in MCF parametric.
    '''

    def __init__(self):
        times = [
            [1276, 1317, 1449, 1475, 1475, 1476, 1489, 1492, 1511, 1568, 1575, 1581, 1581, 1582, 1583, 1585, 1588, 1600, 1607, 1613, 1617, 1708, 1716, 1718, 1733, 1737, 1746, 1756, 1763, 1827, 1829, 1935],
            [1054, 1087, 1147, 1156, 1199, 1232, 1276, 1278, 1283, 1320, 1491, 1620, 1660, 1662, 1681, 1702, 1811, 1815, 1818, 1820, 1838, 1945],
            [1803, 2024, 2032, 2033, 2039, 2304, 2418],
            [1436, 2070, 2216, 2222, 2240, 2539],
            [1441, 1799, 1960, 1961, 1964, 2064],
            [776, 830, 836, 966, 1265, 1641],
            [1845, 2132, 2436],
            [2809, 2812, 3116],
            [2102, 2886, 3495],
            [1793, 1800, 3150],
            [1895, 2217, 2301],
            [796, 1499, 2147],
            [1288, 2763],
            [2552, 2896],
            [2731, 2781],
            [1459, 2967],
            [3289],
            [3401],
            [3492],
            [2984],
            [2532],
            [2715],
            [2778],
            [2408],
            [2349],
            [2460],
            [2451],
            [2449],
            [2450],
            [2450],
            [3131],
            [3789],
            [3837],
            [3351],
            [2862],
            [2972],
            [3489],
            [1118],
            [1496],
            [711],
            [277],
            [299],
            [622],
            [253],
            [255],
            [255],
            [255],
            [255],
            [255],
            [255],
            [255],
            [255],
            [255],
            [255],
            [255],
            [151]]
        self.times = times
        self.number_of_systems = len(times)


class ALT_temperature:
    '''
    This is an accelerated life test (ALT) dataset conducted at 3 temperatures
    It should be used with an ALT probability plot
    The dataset contains mostly censored data but is easily fitted by Weibull_2P, Lognormal_2P, and Gamma_2P distributions.
    Normal_2P will fit but the ALT probability plot will not show Normal_2P is a good fit for this dataset
    Sourced from Dr. Mohammad Modarres, University of Maryland
    '''

    def __init__(self):
        self.failures = [1298, 1390, 3187, 3241, 3261, 3313, 4501, 4568, 4841, 4982, 581, 925, 1432, 1586, 2452, 2734, 2772, 4106, 4674, 283, 361, 515, 638, 854, 1024, 1030, 1045, 1767, 1777, 1856, 1951, 1964, 1951, 1964, 2884]
        self.failure_stresses = [40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 60, 60, 60, 60, 60, 60, 60, 60, 60, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80]
        self.right_censored = [5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000,
                               5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000]
        self.right_censored_stresses = [40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 80]

        rc = len(self.right_censored)
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored', 'Number of stresses'], 'Value': ['ALT_temperature', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)'), 3]}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df


class ALT_temperature2:
    '''
    This is an accelerated life test (ALT) dataset conducted at 4 temperatures
    It should be used with an ALT probability plot
    This is a relatively small dataset with just 40 values, 20 of which are censored.
    Sourced from Dr. Mohammad Modarres, University of Maryland
    '''

    def __init__(self):
        self.failures = [29.1, 80.7, 47.5, 71.8, 73.7, 86.2, 29.5, 52, 56.3, 63.5, 92.5, 99.5, 26.1, 47.5, 53.4, 56.1, 61.8, 76.6, 77.6, 80.9]
        self.failure_stresses = [300, 300, 350, 350, 350, 350, 400, 400, 400, 400, 400, 400, 500, 500, 500, 500, 500, 500, 500, 500]
        self.right_censored = [100, 100, 100, 100, 100, 100, 100, 100, 100, 100, 100, 100, 100, 100, 100, 100, 100, 100, 100, 100]
        self.right_censored_stresses = [300, 300, 300, 300, 300, 300, 300, 300, 350, 350, 350, 350, 350, 350, 400, 400, 400, 400, 500, 500]

        rc = len(self.right_censored)
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored', 'Number of stresses'], 'Value': ['ALT_temperature2', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)'), 4]}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df


class ALT_temperature3:
    '''
    This is an accelerated life test (ALT) dataset conducted at 3 temperatures
    It should be used with an ALT probability plot
    This is a relatively small dataset with just 30 values, none of which are censored.
    '''

    def __init__(self):
        self.failures = [3850, 4340, 4760, 5320, 5740, 6160, 6580, 7140, 7980, 8960, 3300, 3720, 4080, 4560, 4920, 5280, 5640, 6120, 6840, 7680, 2750, 3100, 3400, 3800, 4100, 4400, 4700, 5100, 5700, 6400]
        self.failure_stresses = [393, 393, 393, 393, 393, 393, 393, 393, 393, 393, 408, 408, 408, 408, 408, 408, 408, 408, 408, 408, 423, 423, 423, 423, 423, 423, 423, 423, 423, 423]

        rc = 0
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored', 'Number of stresses'], 'Value': ['ALT_temperature3', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)'), 3]}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df


class ALT_temperature4:
    '''
    This is an accelerated life test (ALT) dataset conducted at 3 temperatures
    It should be used with an ALT probability plot
    This is a relatively small dataset with just 20 values, none of which are censored.
    '''

    def __init__(self):
        self.failures = [267, 443, 521, 570, 713, 721, 802, 808, 173, 174, 290, 321, 334, 502, 102, 104, 158, 197, 224, 243]
        self.failure_stresses = [413, 413, 413, 413, 413, 413, 413, 413, 433, 433, 433, 433, 433, 433, 453, 453, 453, 453, 453, 453]
        rc = 0
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored', 'Number of stresses'], 'Value': ['ALT_temperature4', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)'), 3]}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df


class ALT_load:
    '''
    This is an accelerated life test (ALT) dataset conducted at 3 loads
    It should be used with an ALT probability plot
    This is a relatively small dataset with just 20 and no censoring
    Sourced from Dr. Mohammad Modarres, University of Maryland
    '''

    def __init__(self):
        self.failures = [250, 460, 530, 730, 820, 970, 970, 1530, 160, 180, 290, 320, 390, 460, 90, 100, 150, 180, 220, 230]
        self.failure_stresses = [200, 200, 200, 200, 200, 200, 200, 200, 300, 300, 300, 300, 300, 300, 466, 466, 466, 466, 466, 466]
        rc = 0
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored', 'Number of stresses'], 'Value': ['ALT_load', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)'), 3]}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df


class ALT_load2:
    '''
    This is an accelerated life test (ALT) dataset conducted at 3 loads
    It should be used with an ALT probability plot
    This is a relatively small dataset with just 18 values, 5 of which are censored.
    Sourced from Dr. Mohammad Modarres, University of Maryland
    '''

    def __init__(self):
        self.failures = [245, 312, 409, 110, 180, 200, 222, 50, 70, 88, 112, 140, 160]
        self.failure_stresses = [100, 100, 100, 200, 200, 200, 200, 300, 300, 300, 300, 300, 300]
        self.right_censored = [500, 500, 500, 250, 250]
        self.right_censored_stresses = [100, 100, 100, 200, 200]

        rc = len(self.right_censored)
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored', 'Number of stresses'], 'Value': ['ALT_load2', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)'), 3]}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df


class ALT_temperature_voltage:
    '''
    This is an accelerated life test (ALT) dataset conducted at 2 different temperatures and 2 different voltages
    The dataset is small but contains no censored values.
    It is recommended to use a dual-stress model such as the Power-Exponential model.
    '''

    def __init__(self):
        self.failures = [620, 632, 685, 822, 380, 416, 460, 596, 216, 146, 332, 400]
        self.failure_stress_temp = [348, 348, 348, 348, 348, 348, 348, 348, 378, 378, 378, 378]
        self.failure_stress_voltage = [3, 3, 3, 3, 5, 5, 5, 5, 3, 3, 3, 3]

        rc = 0
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored', 'Number of stresses'], 'Value': ['ALT_temperature_voltage2', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)'), '2 temperatures, 2 voltages']}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df


class ALT_temperature_voltage2:
    '''
    This is an accelerated life test (ALT) dataset conducted at 3 different temperatures and 2 different voltages
    The dataset is fairly small and the pattern of stresses make it extremely difficult to fit.
    Note that there is stress-pair that contains only a single failure.
    It is recommended to use a dual-stress model.
    '''

    def __init__(self):
        self.failures = [1200, 1205, 1509, 1560, 1780, 2390, 2490, 2690, 1466, 1536, 1698, 1784, 2689, 222, 250, 297, 354, 368]
        self.failure_stress_temp = [350, 350, 350, 350, 350, 350, 350, 350, 378, 378, 378, 378, 378, 398, 398, 398, 398, 398]
        self.failure_stress_voltage = [10, 10, 10, 10, 10, 12, 12, 12, 10, 10, 10, 10, 12, 10, 10, 10, 10, 10]

        self.right_censored = [2500, 2500, 2500, 2500, 2500, 2500, 2500, 2500]
        self.right_censored_stress_temp = [350, 350, 350, 378, 378, 378, 378, 378]
        self.right_censored_stress_voltage = [12, 12, 12, 12, 12, 12, 12, 12]

        rc = len(self.right_censored)
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored', 'Number of stresses'], 'Value': ['ALT_temperature_voltage', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)'), '3 temperatures, 2 voltages']}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df


class ALT_temperature_humidity:
    '''
    This is an accelerated life test (ALT) dataset conducted at 2 different temperatures and 2 different humidities
    The dataset is fairly small but has no censored data
    It is recommended to use a dual-stress model such as Dual-Exponential model
    '''

    def __init__(self):
        self.failures = [310, 316, 329, 411, 190, 208, 230, 298, 108, 123, 166, 200]
        self.failure_stress_temp = [378, 378, 378, 378, 378, 378, 378, 378, 398, 398, 398, 398]
        self.failure_stress_humidity = [0.4, 0.4, 0.4, 0.4, 0.8, 0.8, 0.8, 0.8, 0.4, 0.4, 0.4, 0.4]

        rc = 0
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored', 'Number of stresses'], 'Value': ['ALT_temperature_humidity', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)'), '2 temperatures, 2 humidities']}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df
