# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['barim']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'barim',
    'version': '0.1.0',
    'description': 'Barim is a Python library that helps building CLI applications',
    'long_description': '# Barim\n\n[![python](https://img.shields.io/badge/python->=3.7-blue?logo=python&logoColor=yellow)](https://python.org "Go to Python homepage")\n[![pipeline - passed](https://img.shields.io/badge/pipeline-passed-brightgreen?logo=gitlab)](https://)\n[![version alpha - 0.1.0](https://img.shields.io/badge/version-0.1.0-brightgreen)](https://)\n[![license - MIT](https://img.shields.io/badge/license-MIT-blue)](https://)\n[![code style - black](https://img.shields.io/badge/code_style-black-blue)](https://black.readthedocs.io/ "Go to Black homepage")\n[![coverage - unknown](https://img.shields.io/badge/coverage-87%25-green)](https://coverage.readthedocs.io/ "Go to Coverage homepage")\n\nBarim is a Python library that helps building CLI applications.\n\nThe [Barim API]() makes it easy to build terminal app that have one or more commands.\n\n\n## Requirements\n\n- Python ([3.7.X or later](https://www.python.org/))\n\n## Dev requirements\n\n- Git ([latest version](https://git-scm.com/))\n- Python ([3.7.X or later](https://www.python.org/))\n- Poetry ([latest version](https://python-poetry.org))\n\n## Installing\n\nInstall with pip.\n\n```bash\npython -m pip install barim\n```\n\nInstall with poetry.\n\n```bash\npoetry add barim\n```\n\n## Quickstart\n\nThis following section is dedicated to explain how barim can be used to \ncreate a simple CLI application. We will see how to make a terminal application \nwith a default command and how we can update it to allow multiple subcommand.\n\n### Create your first command\n\nThe first step will be to create a class that inherit from [barim.Command]() and\nprovide some values to the class attributes.\n\n```python\n# File name: example.py\nfrom types import SimpleNamespace\n\nfrom barim import Command\n\n\nclass MyCommand(Command):\n    \n    name = "example.py"\n    description = "Print \'Hello, World!\'"\n    version = "0.1.0"\n\n    def handle(self, argv: SimpleNamespace, opts: SimpleNamespace) -> None:\n        print("Hello, World!")\n```\n\nTake note that the class attributes \'description\' and \'version\' aren\'t mandatory.\nOnly \'name\' is.\n\n> The **handle()** method **must be override** has it will be the entry point for your\nsoftware. The argv and opts parameters are two SimpleNamespace that contain\nall the arguments (argv) and all the options (opts) declared in the\ncommand class. To see how to use them check out the [Barim API]() or just continue\nthe quickstart guide. \n\nThe next logical step now is just to tell barim to run the command when the\nscript is run. This can be done like the following.\n\n```python\nfrom types import SimpleNamespace\n\nfrom barim import Application, Command\n\n\nclass MyCommand(Command):\n    \n    name = "my_command"\n    description = "Print \'Hello, World!\'"\n    version = "0.1.0"\n\n    def handle(self, argv: SimpleNamespace, opts: SimpleNamespace) -> None:\n        print("Hello, World!")\n\n        \n# Notice we add a main method to be called once the script is runned        \ndef main() -> None:\n    application = Application(name="example.py")\n    application.register(MyCommand)\n    application.run(default=True)\n\n\nif __name__ == "__main__":\n    main()\n```\n\nThat should be all ! You can now test your CLI application by running the\nfollowing command.\n\n```bash\npython example.py --help\n```\n\nThe output of the command should be:\n\n```bash\nexample.py version 0.1.0\n\nDESCRIPTION\n    Print \'Hello, World!\'\n\nUSAGE\n    example.py  [-h] [-V] [-v]\n\nGLOBAL OPTIONS\n    -h (--help)             Display help message\n    -V (--version)          Display version number\n    -v (--verbose)          Display more log message\n```\n\n> Something to notice here is that the output doesn\'t show any colors (in this README) \nwhich will not be the case on the terminal.\n\n### Add arguments and options to the command\n\nLet\'s say now that we want to take a string as an argument to print \'Hello, {input}!\'.\nFor this we need to declare another class variable named \'arguments\' that\nis a list of [barim.Argument]().\n\n```python\nfrom types import SimpleNamespace\n\nfrom barim import Application, Argument, Command\n\n\nclass MyCommand(Command):\n    name = "my_command"\n    description = "Print \'Hello, {input}!\'"\n    version = "0.1.0"\n\n    arguments = [\n        Argument(\n            name="input",\n            description="Use input string in print statement"\n        ),\n    ]\n\n    def handle(self, argv: SimpleNamespace, opts: SimpleNamespace) -> None:\n        # Make use of the newly added argument\n        print(f"Hello, {argv.input}")\n\n\ndef main() -> None:\n    application = Application(name="example.py")\n    application.register(MyCommand)\n    application.run(default=True)\n\n\nif __name__ == "__main__":\n    main()\n```\n\n> Note that you can declare a [barim.Argument]() or by providing the needed data\nduring initialization like in the example above or by subclassing it and declaring the data as class variable.\n\nYou should now be able to run your script like the following.\n\n```sh\npython example.py Demo\n```\n\nExpected output:\n\n```bash\nHello, Demo!\n```\n\nNow when it comes to options, it doesn\'t change that much. Instead of declaring\narguments you declare options and provide a list of [barim.Option]().\nIn our example let\'s say we want to turn all the letter uppercase.\n\n```python\nfrom types import SimpleNamespace\n\nfrom barim import Application, Argument, Command, Option\n\n\nclass MyCommand(Command):\n    name = "my_command"\n    description = "Print \'Hello, {input}!\'"\n    version = "0.1.0"\n\n    arguments = [\n        Argument(\n            name="input",\n            description="Use input string in print statement"\n        ),\n    ]\n    options = [\n        Option(\n            short="u",\n            long="upper",\n            description="Turn all the letter uppercase",\n        ),\n    ]\n\n    def handle(self, argv: SimpleNamespace, opts: SimpleNamespace) -> None:\n        # Make use of newly added option\n        input_ = argv.input\n        if opts.upper:\n            input_ = input_.upper()\n\n        print(f"Hello, {input_}")\n\n\ndef main() -> None:\n    application = Application(name="example.py")\n    application.register(MyCommand)\n    application.run(default=True)\n\n\nif __name__ == "__main__":\n    main()\n```\n\nNow run your script as below.\n\n```bash\npython example.py demo --upper\n```\n\nExpected output:\n\n```bash\nHello, DEMO!\n```\n\n### Create subcommands\n\nThe default parameter declare earlier in `application.run(default=True)` indicate that we only have one registered\ncommand and that we want to use it as the default one. By removing this parameter we can now register multiple command \nto act as multiple sub command.\n\n```python\nfrom types import SimpleNamespace\n\nfrom barim import Application, Argument, Command, Option\n\n\nclass MyCommand(Command):\n    name = "my_command"\n    description = "Print \'Hello, {input}!\'"\n    version = "0.1.0"\n\n    arguments = [\n        Argument(\n            name="input",\n            description="Use input string in print statement"\n        ),\n    ]\n    options = [\n        Option(\n            short="u",\n            long="upper",\n            description="Turn all the letter uppercase",\n        ),\n    ]\n\n    def handle(self, argv: SimpleNamespace, opts: SimpleNamespace) -> None:\n        input_ = argv.input\n        if opts.upper:\n            input_ = input_.upper()\n\n        print(f"Hello, {input_}")\n\n\nclass MyOtherCommand(Command):\n    name = "my_other_command"\n    description = "Print \'Hello, World!\'"\n    version = "0.1.0"\n    \n    def handle(self, argv: SimpleNamespace, opts: SimpleNamespace) -> None:\n        print("Hello, World")\n        \n        \ndef main() -> None:\n    application = Application(name="example.py")\n    application.register(MyCommand)\n    application.register(MyOtherCommand)\n    application.run()\n\n\nif __name__ == "__main__":\n    main()\n```\n\n### Create command dynamically\n\nAs seen before, to create a command, we have to subclass [barim.Command]().\nBut this is not the only way we can create them. In any case you need to create them,\nfor example on the fly, you could do it like in the following example.\n\n```python\nfrom types import SimpleNamespace\n\nfrom barim import Application, Command\n\n\ndef my_command_handle(argv: SimpleNamespace, opts: SimpleNamespace) -> None:\n    print("Hello, World")\n\n    \ndef main() -> None:\n    my_command = Command(name="my_command", description="Print \'Hello, World!\'", handle=my_command_handle)\n    \n    application = Application(name="example.py")\n    application.register(my_command)\n    application.run(default=True)\n\n    \nif __name__ == "__main__":\n    main()\n```\n\nThe output of the command should look like the following:\n\n```bash\nexample.py version 0.1.0\n\nDESCRIPTION\n    Print \'Hello, World!\'\n\nUSAGE\n    example.py  [-h] [-V] [-v]\n\nGLOBAL OPTIONS\n    -h (--help)             Display help message\n    -V (--version)          Display version number\n    -v (--verbose)          Display more log message\n```',
    'author': 'Candotti Luca',
    'author_email': 'candotti.luca@outlook.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
