"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InterfaceVpcEndpoint = exports.InterfaceVpcEndpointAwsService = exports.InterfaceVpcEndpointService = exports.GatewayVpcEndpoint = exports.GatewayVpcEndpointAwsService = exports.VpcEndpointType = exports.VpcEndpoint = void 0;
const iam = require("@aws-cdk/aws-iam");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const peer_1 = require("./peer");
const port_1 = require("./port");
const security_group_1 = require("./security-group");
const util_1 = require("./util");
/**
 *
 */
class VpcEndpoint extends core_1.Resource {
    /**
     * Adds a statement to the policy document of the VPC endpoint. The statement must have a Principal.
     *
     * Not all interface VPC endpoints support policy. For more information
     * see https://docs.aws.amazon.com/vpc/latest/userguide/vpce-interface.html
     *
     * @param statement the IAM statement to add.
     */
    addToPolicy(statement) {
        if (!statement.hasPrincipal) {
            throw new Error('Statement must have a `Principal`.');
        }
        if (!this.policyDocument) {
            this.policyDocument = new iam.PolicyDocument();
        }
        this.policyDocument.addStatements(statement);
    }
}
exports.VpcEndpoint = VpcEndpoint;
/**
 * The type of VPC endpoint.
 */
var VpcEndpointType;
(function (VpcEndpointType) {
    VpcEndpointType["INTERFACE"] = "Interface";
    VpcEndpointType["GATEWAY"] = "Gateway";
})(VpcEndpointType = exports.VpcEndpointType || (exports.VpcEndpointType = {}));
/**
 * An AWS service for a gateway VPC endpoint.
 */
class GatewayVpcEndpointAwsService {
    /**
     *
     */
    constructor(name, prefix) {
        this.name = `${prefix || 'com.amazonaws'}.${core_1.Aws.REGION}.${name}`;
    }
}
exports.GatewayVpcEndpointAwsService = GatewayVpcEndpointAwsService;
/**
 *
 */
GatewayVpcEndpointAwsService.DYNAMODB = new GatewayVpcEndpointAwsService('dynamodb');
/**
 *
 */
GatewayVpcEndpointAwsService.S3 = new GatewayVpcEndpointAwsService('s3');
/**
 * A gateway VPC endpoint.
 *
 * @resource AWS::EC2::VPCEndpoint
 */
class GatewayVpcEndpoint extends VpcEndpoint {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        const subnets = props.subnets
            ? util_1.flatten(props.subnets.map(s => props.vpc.selectSubnets(s).subnets))
            : [...props.vpc.privateSubnets, ...props.vpc.publicSubnets, ...props.vpc.isolatedSubnets];
        const routeTableIds = util_1.allRouteTableIds(subnets);
        if (routeTableIds.length === 0) {
            throw new Error('Can\'t add a gateway endpoint to VPC; route table IDs are not available');
        }
        const endpoint = new ec2_generated_1.CfnVPCEndpoint(this, 'Resource', {
            policyDocument: core_1.Lazy.any({ produce: () => this.policyDocument }),
            routeTableIds,
            serviceName: props.service.name,
            vpcEndpointType: VpcEndpointType.GATEWAY,
            vpcId: props.vpc.vpcId,
        });
        this.vpcEndpointId = endpoint.ref;
        this.vpcEndpointCreationTimestamp = endpoint.attrCreationTimestamp;
        this.vpcEndpointDnsEntries = endpoint.attrDnsEntries;
        this.vpcEndpointNetworkInterfaceIds = endpoint.attrNetworkInterfaceIds;
    }
    /**
     *
     */
    static fromGatewayVpcEndpointId(scope, id, gatewayVpcEndpointId) {
        class Import extends VpcEndpoint {
            constructor() {
                super(...arguments);
                this.vpcEndpointId = gatewayVpcEndpointId;
            }
        }
        return new Import(scope, id);
    }
}
exports.GatewayVpcEndpoint = GatewayVpcEndpoint;
/**
 * A custom-hosted service for an interface VPC endpoint.
 */
class InterfaceVpcEndpointService {
    /**
     *
     */
    constructor(name, port) {
        /**
         * Whether Private DNS is supported by default.
         */
        this.privateDnsDefault = false;
        this.name = name;
        this.port = port || 443;
    }
}
exports.InterfaceVpcEndpointService = InterfaceVpcEndpointService;
/**
 * An AWS service for an interface VPC endpoint.
 */
class InterfaceVpcEndpointAwsService {
    /**
     *
     */
    constructor(name, prefix, port) {
        /**
         * Whether Private DNS is supported by default.
         */
        this.privateDnsDefault = true;
        const region = core_1.Lazy.uncachedString({
            produce: (context) => core_1.Stack.of(context.scope).region,
        });
        this.name = `${prefix || 'com.amazonaws'}.${region}.${name}`;
        this.port = port || 443;
    }
}
exports.InterfaceVpcEndpointAwsService = InterfaceVpcEndpointAwsService;
/**
 *
 */
InterfaceVpcEndpointAwsService.SAGEMAKER_NOTEBOOK = new InterfaceVpcEndpointAwsService('notebook', 'aws.sagemaker');
/**
 *
 */
InterfaceVpcEndpointAwsService.CLOUDFORMATION = new InterfaceVpcEndpointAwsService('cloudformation');
/**
 *
 */
InterfaceVpcEndpointAwsService.CLOUDTRAIL = new InterfaceVpcEndpointAwsService('cloudtrail');
/**
 *
 */
InterfaceVpcEndpointAwsService.CODEBUILD = new InterfaceVpcEndpointAwsService('codebuild');
/**
 *
 */
InterfaceVpcEndpointAwsService.CODEBUILD_FIPS = new InterfaceVpcEndpointAwsService('codebuild-fips');
/**
 *
 */
InterfaceVpcEndpointAwsService.CODECOMMIT = new InterfaceVpcEndpointAwsService('codecommit');
/**
 *
 */
InterfaceVpcEndpointAwsService.CODECOMMIT_FIPS = new InterfaceVpcEndpointAwsService('codecommit-fips');
/**
 *
 */
InterfaceVpcEndpointAwsService.CODEPIPELINE = new InterfaceVpcEndpointAwsService('codepipeline');
/**
 *
 */
InterfaceVpcEndpointAwsService.CONFIG = new InterfaceVpcEndpointAwsService('config');
/**
 *
 */
InterfaceVpcEndpointAwsService.EC2 = new InterfaceVpcEndpointAwsService('ec2');
/**
 *
 */
InterfaceVpcEndpointAwsService.EC2_MESSAGES = new InterfaceVpcEndpointAwsService('ec2messages');
/**
 *
 */
InterfaceVpcEndpointAwsService.ECR = new InterfaceVpcEndpointAwsService('ecr.api');
/**
 *
 */
InterfaceVpcEndpointAwsService.ECR_DOCKER = new InterfaceVpcEndpointAwsService('ecr.dkr');
/**
 *
 */
InterfaceVpcEndpointAwsService.ECS = new InterfaceVpcEndpointAwsService('ecs');
/**
 *
 */
InterfaceVpcEndpointAwsService.ECS_AGENT = new InterfaceVpcEndpointAwsService('ecs-agent');
/**
 *
 */
InterfaceVpcEndpointAwsService.ECS_TELEMETRY = new InterfaceVpcEndpointAwsService('ecs-telemetry');
/**
 *
 */
InterfaceVpcEndpointAwsService.ELASTIC_FILESYSTEM = new InterfaceVpcEndpointAwsService('elasticfilesystem');
/**
 *
 */
InterfaceVpcEndpointAwsService.ELASTIC_FILESYSTEM_FIPS = new InterfaceVpcEndpointAwsService('elasticfilesystem-fips');
/**
 *
 */
InterfaceVpcEndpointAwsService.ELASTIC_INFERENCE_RUNTIME = new InterfaceVpcEndpointAwsService('elastic-inference.runtime');
/**
 *
 */
InterfaceVpcEndpointAwsService.ELASTIC_LOAD_BALANCING = new InterfaceVpcEndpointAwsService('elasticloadbalancing');
/**
 *
 */
InterfaceVpcEndpointAwsService.CLOUDWATCH_EVENTS = new InterfaceVpcEndpointAwsService('events');
/**
 *
 */
InterfaceVpcEndpointAwsService.APIGATEWAY = new InterfaceVpcEndpointAwsService('execute-api');
/**
 *
 */
InterfaceVpcEndpointAwsService.CODECOMMIT_GIT = new InterfaceVpcEndpointAwsService('git-codecommit');
/**
 *
 */
InterfaceVpcEndpointAwsService.CODECOMMIT_GIT_FIPS = new InterfaceVpcEndpointAwsService('git-codecommit-fips');
/**
 *
 */
InterfaceVpcEndpointAwsService.KINESIS_STREAMS = new InterfaceVpcEndpointAwsService('kinesis-streams');
/**
 *
 */
InterfaceVpcEndpointAwsService.KINESIS_FIREHOSE = new InterfaceVpcEndpointAwsService('kinesis-firehose');
/**
 *
 */
InterfaceVpcEndpointAwsService.KMS = new InterfaceVpcEndpointAwsService('kms');
/**
 *
 */
InterfaceVpcEndpointAwsService.CLOUDWATCH_LOGS = new InterfaceVpcEndpointAwsService('logs');
/**
 *
 */
InterfaceVpcEndpointAwsService.CLOUDWATCH = new InterfaceVpcEndpointAwsService('monitoring');
/**
 *
 */
InterfaceVpcEndpointAwsService.SAGEMAKER_API = new InterfaceVpcEndpointAwsService('sagemaker.api');
/**
 *
 */
InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME = new InterfaceVpcEndpointAwsService('sagemaker.runtime');
/**
 *
 */
InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME_FIPS = new InterfaceVpcEndpointAwsService('sagemaker.runtime-fips');
/**
 *
 */
InterfaceVpcEndpointAwsService.SECRETS_MANAGER = new InterfaceVpcEndpointAwsService('secretsmanager');
/**
 *
 */
InterfaceVpcEndpointAwsService.SERVICE_CATALOG = new InterfaceVpcEndpointAwsService('servicecatalog');
/**
 *
 */
InterfaceVpcEndpointAwsService.SNS = new InterfaceVpcEndpointAwsService('sns');
/**
 *
 */
InterfaceVpcEndpointAwsService.SQS = new InterfaceVpcEndpointAwsService('sqs');
/**
 *
 */
InterfaceVpcEndpointAwsService.SSM = new InterfaceVpcEndpointAwsService('ssm');
/**
 *
 */
InterfaceVpcEndpointAwsService.SSM_MESSAGES = new InterfaceVpcEndpointAwsService('ssmmessages');
/**
 *
 */
InterfaceVpcEndpointAwsService.STS = new InterfaceVpcEndpointAwsService('sts');
/**
 *
 */
InterfaceVpcEndpointAwsService.TRANSFER = new InterfaceVpcEndpointAwsService('transfer.server');
/**
 *
 */
InterfaceVpcEndpointAwsService.STORAGE_GATEWAY = new InterfaceVpcEndpointAwsService('storagegateway');
/**
 *
 */
InterfaceVpcEndpointAwsService.REKOGNITION = new InterfaceVpcEndpointAwsService('rekognition');
/**
 *
 */
InterfaceVpcEndpointAwsService.REKOGNITION_FIPS = new InterfaceVpcEndpointAwsService('rekognition-fips');
/**
 *
 */
InterfaceVpcEndpointAwsService.STEP_FUNCTIONS = new InterfaceVpcEndpointAwsService('states');
/**
 *
 */
InterfaceVpcEndpointAwsService.LAMBDA = new InterfaceVpcEndpointAwsService('lambda');
/**
 * A interface VPC endpoint.
 *
 * @resource AWS::EC2::VPCEndpoint
 */
class InterfaceVpcEndpoint extends VpcEndpoint {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const securityGroups = props.securityGroups || [new security_group_1.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
            })];
        this.securityGroupId = securityGroups[0].securityGroupId;
        this.connections = new connections_1.Connections({
            defaultPort: port_1.Port.tcp(props.service.port),
            securityGroups,
        });
        if (props.open !== false) {
            this.connections.allowDefaultPortFrom(peer_1.Peer.ipv4(props.vpc.vpcCidrBlock));
        }
        // Determine which subnets to place the endpoint in
        const subnetIds = this.endpointSubnets(props);
        const endpoint = new ec2_generated_1.CfnVPCEndpoint(this, 'Resource', {
            privateDnsEnabled: (_b = (_a = props.privateDnsEnabled) !== null && _a !== void 0 ? _a : props.service.privateDnsDefault) !== null && _b !== void 0 ? _b : true,
            policyDocument: core_1.Lazy.any({ produce: () => this.policyDocument }),
            securityGroupIds: securityGroups.map(s => s.securityGroupId),
            serviceName: props.service.name,
            vpcEndpointType: VpcEndpointType.INTERFACE,
            subnetIds,
            vpcId: props.vpc.vpcId,
        });
        this.vpcEndpointId = endpoint.ref;
        this.vpcEndpointCreationTimestamp = endpoint.attrCreationTimestamp;
        this.vpcEndpointDnsEntries = endpoint.attrDnsEntries;
        this.vpcEndpointNetworkInterfaceIds = endpoint.attrNetworkInterfaceIds;
    }
    /**
     * Imports an existing interface VPC endpoint.
     */
    static fromInterfaceVpcEndpointAttributes(scope, id, attrs) {
        if (!attrs.securityGroups && !attrs.securityGroupId) {
            throw new Error('Either `securityGroups` or `securityGroupId` must be specified.');
        }
        const securityGroups = attrs.securityGroupId
            ? [security_group_1.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', attrs.securityGroupId)]
            : attrs.securityGroups;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.vpcEndpointId = attrs.vpcEndpointId;
                this.connections = new connections_1.Connections({
                    defaultPort: port_1.Port.tcp(attrs.port),
                    securityGroups,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Determine which subnets to place the endpoint in. This is in its own function
     * because there's a lot of code.
     */
    endpointSubnets(props) {
        var _a;
        const lookupSupportedAzs = (_a = props.lookupSupportedAzs) !== null && _a !== void 0 ? _a : false;
        const subnetSelection = props.vpc.selectSubnets({ ...props.subnets, onePerAz: true });
        // If we don't have an account/region, we will not be able to do filtering on AZs since
        // they will be undefined
        const agnosticAcct = core_1.Token.isUnresolved(this.stack.account);
        const agnosticRegion = core_1.Token.isUnresolved(this.stack.region);
        // Some service names, such as AWS service name references, use Tokens to automatically
        // fill in the region
        // If it is an InterfaceVpcEndpointAwsService, then the reference will be resolvable since
        // only references the region
        const isAwsService = core_1.Token.isUnresolved(props.service.name) && props.service instanceof InterfaceVpcEndpointAwsService;
        // Determine what name we pass to the context provider, either the verbatim name
        // or a resolved version if it is an AWS service reference
        let lookupServiceName = props.service.name;
        if (isAwsService && !agnosticRegion) {
            lookupServiceName = core_1.Stack.of(this).resolve(props.service.name);
        }
        else {
            // It's an agnostic service and we don't know how to resolve it.
            // This is ok if the stack is region agnostic and we're not looking up
            // AZs
            lookupServiceName = props.service.name;
        }
        // Check if lookup is impossible and throw an appropriate error
        // Context provider cannot make an AWS call without an account/region
        if ((agnosticAcct || agnosticRegion) && lookupSupportedAzs) {
            throw new Error('Cannot look up VPC endpoint availability zones if account/region are not specified');
        }
        // Context provider doesn't know the name of the service if there is a Token
        // in the name
        const agnosticService = core_1.Token.isUnresolved(lookupServiceName);
        if (agnosticService && lookupSupportedAzs) {
            throw new Error(`Cannot lookup AZs for a service name with a Token: ${props.service.name}`);
        }
        // Here we do the actual lookup for AZs, if told to do so
        let subnets;
        if (lookupSupportedAzs) {
            const availableAZs = this.availableAvailabilityZones(lookupServiceName);
            subnets = subnetSelection.subnets.filter(s => availableAZs.includes(s.availabilityZone));
        }
        else {
            subnets = subnetSelection.subnets;
        }
        const subnetIds = subnets.map(s => s.subnetId);
        return subnetIds;
    }
    availableAvailabilityZones(serviceName) {
        // Here we check what AZs the endpoint service is available in
        // If for whatever reason we can't retrieve the AZs, and no context is set,
        // we will fall back to all AZs
        const availableAZs = core_1.ContextProvider.getValue(this, {
            provider: cxschema.ContextProvider.ENDPOINT_SERVICE_AVAILABILITY_ZONE_PROVIDER,
            dummyValue: this.stack.availabilityZones,
            props: { serviceName },
        }).value;
        if (!Array.isArray(availableAZs)) {
            throw new Error(`Discovered AZs for endpoint service ${serviceName} must be an array`);
        }
        return availableAZs;
    }
}
exports.InterfaceVpcEndpoint = InterfaceVpcEndpoint;
//# sourceMappingURL=data:application/json;base64,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