// *WARNING* This file has been automatically generated by TPP do not edit directly.
/*
 * This file is part of TiPi (a Toolkit for Inverse Problems and Imaging)
 * developed by the MitiV project.
 *
 * Copyright (c) 2014 the MiTiV project, http://mitiv.univ-lyon1.fr/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

package org.mitiv.TiPi.io;

import static java.lang.Math.max;
import static java.lang.Math.min;
import static java.lang.Math.round;

import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.awt.image.DataBuffer;
import java.awt.image.Raster;
import java.awt.image.WritableRaster;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintStream;

import javax.imageio.ImageIO;

import org.mitiv.TiPi.array.ArrayFactory;
import org.mitiv.TiPi.array.Byte2D;
import org.mitiv.TiPi.array.Byte3D;
import org.mitiv.TiPi.array.Double2D;
import org.mitiv.TiPi.array.Double3D;
import org.mitiv.TiPi.array.Float2D;
import org.mitiv.TiPi.array.Float3D;
import org.mitiv.TiPi.array.Int2D;
import org.mitiv.TiPi.array.Int3D;
import org.mitiv.TiPi.array.Long2D;
import org.mitiv.TiPi.array.Long3D;
import org.mitiv.TiPi.array.ShapedArray;
import org.mitiv.TiPi.array.Short2D;
import org.mitiv.TiPi.array.Short3D;
import org.mitiv.TiPi.base.Traits;
import org.mitiv.TiPi.exception.IllegalTypeException;
import org.mitiv.TiPi.linalg.Vector;
import org.mitiv.TiPi.linalg.shaped.ShapedVector;

/**
 * Deals with identifying, reading and writing various data format.
 *
 * @author Éric Thiébaut.
 */
public enum DataFormat {

    PNM("PNM", "Portable anymap (PBM/PGM/PPM/PNM) image.", new String[]{"pnm", "ppm", "pgm", "pbm"}),
    JPEG("JPEG", "JPEG image.", new String[]{"jpg", "jpeg"}),
    PNG("PNG", "Portable Network Graphic (PNG) image.", new String[]{"png"}),
    GIF("GIF", "GIF image.", new String[]{"gif"}),
    BMP("BMP", "BMP image.", new String[]{"bmp"}),
    WBMP("WBMP", "Wireless Bitmap (WBMP) image format.", new String[]{"wbmp"}),
    TIFF("TIFF", "TIFF image format.", new String[]{"tiff", "tif"}),
    FITS("FITS", "Flexible Image Transport System (FITS) format.", new String[]{"fits", "fts", "fit"}),
    MDA("MDA", "Multi-dimensional array (MDA) format.", new String[]{"mda"});

    private final String identifier;

    private final String description;

    private final String[] extensions;

    private DataFormat(String ident, String descr, String[] extensions) {
        this.identifier = ident;
        this.description = descr;
        this.extensions = extensions;
    }

    /**
     * Get the name of the data file format.
     *
     * <p> For image format supported by Java, the format name is suitable for
     * {@link ImageIO#write}. </p>
     *
     * @return The name of the format.
     */
    public String identifier() {
        return identifier;
    }

    /** Get the description of the file format. */
    public String description() {
        return description;
    }

    /** Get the recognized file name extensions for the format. */
    public String[] extensions() {
        return extensions;
    }

    /** Get a string representation of the color model. */
    @Override
    public String toString() {
        return description;
    }

    /**
     * Check whether a suffix matches recognized file name extensions.
     *
     * @param suffix
     *        The suffix to check.
     *
     * @return A boolean value.
     */
    public boolean match(String suffix) {
        if (suffix != null && suffix.length() > 0) {
            int n = (extensions == null ? 0 : extensions.length);
            for (int k = 0; k < n; ++k) {
                if (suffix.equalsIgnoreCase(extensions[k])) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Guess data format from the extension of the file name.
     *
     * <p> This function examines the extension of the file name to determine
     * the data format.  </p>
     *
     * @param name
     *        The name of the destination file.
     *
     * @return A file format identifier: {@code null} if the
     *         format is not recognized; otherwise {@link #PNM},
     *         {@link #JPEG}, {@link #PNG}, {@link #TIFF},
     *         {@link #FITS}, {@link #GIF}, or {@link #MDA}.
     */
    public static final DataFormat guessFormat(String name) {
        int index = name.lastIndexOf('.');
        if (index < 0) {
            return null;
        }
        String suffix = name.substring(index + 1);
        if (MDA.match(suffix)) {
            return MDA;
        }
        if (PNG.match(suffix)) {
            return PNG;
        }
        if (JPEG.match(suffix)) {
            return JPEG;
        }
        if (PNM.match(suffix)) {
            return PNM;
        }
        if (TIFF.match(suffix)) {
            return TIFF;
        }
        if (FITS.match(suffix)) {
            return FITS;
        }
        if (BMP.match(suffix)) {
            return BMP;
        }
        if (WBMP.match(suffix)) {
            return WBMP;
        }
        if (FITS.match(suffix)) {
            return FITS;
        }
        if (GIF.match(suffix)) {
            return GIF;
        }
        return null;
    }


    /**
     * Guess data format from the given options or from the extension of
     * the file name.
     *
     * <p> It a preferred data format is specified in the options, this format
     * is returned; otherwise, the extension of the file name is examined to
     * determine the data format.  </p>
     *
     * @param name
     *        The name of the destination file.
     *
     * @return A file format identifier: {@code null} if the
     *         format is not recognized; otherwise {@link #PNM},
     *         {@link #JPEG}, {@link #PNG}, {@link #TIFF},
     *         {@link #FITS}, {@link #GIF}, or {@link #MDA}.
     */
    public static final DataFormat guessFormat(String name,
            FormatOptions opts) {
        DataFormat format = (opts == null ? null : opts.getDataFormat());
        if (format != null) {
            return format;
        } else {
            return guessFormat(name);
        }
    }

    /**
     * Guess data format from a few magic bytes.
     *
     * <p> This function examines the few next bytes available from the data
     * stream to determine the data format.  In any case, the stream position
     * is left where it was prior to calling the function (i.e. there is no
     * data consumption). </p>
     *
     * @param stream
     *        The input data stream.
     *
     * @return A file format identifier (see {@link #guessFormat(String)}),
     *         or {@code null} if the format is not recognized,.
     *
     * @throws IOException
     */
    public static final DataFormat guessFormat(BufferedInputDataStream stream)
            throws IOException {
        int preserved = stream.insure(80);
        if (preserved < 2) {
            return null;
        }
        DataFormat format = null;
        stream.mark();
        try {
            int length = min(preserved, 80);
            byte[] magic = new byte[length];
            length = stream.read(magic, 0, length);
            if (length < 2) {
                return null;
            }
            if (length >= 2 && matchMagic(magic, '\377', '\330')) {
                format = JPEG;
            } else if (length >= 4 && matchMagic(magic, '\211', 'P', 'N', 'G')) {
                format = PNG;
            } else if (length >= 4 && matchMagic(magic, 'M', 'M', '\000', '\052')) {
                format = TIFF;
            } else if (length >= 4 && matchMagic(magic, 'I', 'I', '\052', '\000')) {
                format = TIFF;
            } else if (length >= 4 && matchMagic(magic, 'G', 'I', 'F', '8')) {
                format = GIF;
            } else if (length >= 3 && magic[0] == (byte)'P' && isSpace(magic[2])
                    && (byte)'1' <= magic[1] && magic[1] <= (byte)'6') {
                /* "P1" space = ascii PBM (portable bitmap)
                 * "P2" space = ascii PGM (portable graymap)
                 * "P3" space = ascii PPM (portable pixmap)
                 * "P4" space = raw PBM
                 * "P5" space = raw PGM
                 * "P6" space = raw PPM
                 */
                format = PNM;
            } else if (length >= 9
                    && matchMagic(magic, 'S', 'I', 'M', 'P', 'L', 'E', ' ', ' ', '=')) {
                format = FITS;
            }
        } catch (IOException ex) {
            throw ex;
        } finally{
            stream.reset();
        }
        return format;
    }

    private static final boolean matchMagic(byte[] b,
            char c0, char c1) {
        return ((b[0] == (byte)c0) && (b[1] == (byte)c1));
    }

    private static final boolean matchMagic(byte[] b,
            char c0, char c1, char c2, char c3) {
        return ((b[0] == (byte)c0) && (b[1] == (byte)c1) &&
                (b[2] == (byte)c2) && (b[3] == (byte)c3));
    }

    private static final boolean matchMagic(byte[] b,
            char c0, char c1, char c2, char c3,
            char c4, char c5, char c6, char c7, char c8) {
        return ((b[0] == (byte)c0) && (b[1] == (byte)c1) &&
                (b[2] == (byte)c2) && (b[3] == (byte)c3) &&
                (b[4] == (byte)c4) && (b[5] == (byte)c5) &&
                (b[6] == (byte)c6) && (b[7] == (byte)c7) &&
                (b[8] == (byte)c8));
    }

    private static final boolean isSpace(byte s) {
        return (s == (byte)' ' || s == (byte)'\n' || s == (byte)'\r' ||
                s == (byte)'\t');
    }

    private static void fatal(String reason) {
        throw new IllegalArgumentException(reason);
    }

    /**
     * Load formatted data from a file.
     *
     * <p> This method attempts to give the most appropriate representation of
     * the data stored in the file as a shaped array.  For instance, it relies
     * on {@link #imageToShapedArray} to convert an image in a shaped array.
     * See {@link ColorModel} for the assumed conventions about the
     * representation of an image as a shaped array.  You may use {@link
     * ColorModel#guessColorModel} to determine the color model of an image
     * loaded by this method and {@link ColorModel#filterImageAsFloat} or
     * {@link ColorModel#filterImageAsDouble} to convert such an image
     * according to your needs.  </p>
     *
     * @param name
     *        The name of the source file.
     *
     * @return A shaped array.
     */
    public static ShapedArray load(String name) {
        ShapedArray arr = null;
        DataFormat format = guessFormat(name);
        try {
            if (format == MDA) {
                arr = MdaFormat.load(name);
            } else if (format == FITS) {
                arr = FitsFormat.load(name);
            } else {
                BufferedImage img = ImageIO.read(new File(name));
                arr = imageToShapedArray(img);
            }
        } catch (Exception e) {
            fatal("Error while reading " + name + "(" + e.getMessage() +")");
        }
        return arr;
    }

    /**
     * Save a shaped array to a file.
     *
     * <p> The file format is guessed from the extension of the file name and
     * default options are used to encode the file.  </p>
     *
     * @param arr
     *        The shaped array to save.
     *
     * @param name
     *        The name of the destination file.
     *
     * @throws IOException
     * @throws FileNotFoundException
     */
    public static void save(ShapedArray arr, String name)
            throws FileNotFoundException, IOException {
        save(arr, name, new FormatOptions());
    }

    /**
     * Save a shaped array to a file with given options.
     *
     * <p> The file format is guessed from the extension of the file name and
     * default options are used to encode the file.  </p>
     *
     * @param arr
     *        The shaped array to save.
     *
     * @param name
     *        The name of the destination file.
     *
     * @param opts
     *        Options for encoding the data.
     *
     * @throws IOException
     * @throws FileNotFoundException
     */
    public static void save(ShapedArray arr, String name,
            FormatOptions opts) throws FileNotFoundException, IOException {
        DataFormat format = DataFormat.guessFormat(name, opts);
        String identifier = null;
        switch (format) {
            //case PNM:
            //case TIFF:
            //case FITS:
            case JPEG:
            case PNG:
            case GIF:
            case BMP:
            case WBMP:
                identifier = format.identifier();
                break;
            case MDA:
                MdaFormat.save(arr, name);
                return;
            case FITS:
                FitsFormat.save(arr, name);
                return;
            default:
                identifier = null;
        }
        if (identifier == null) {
            fatal("Unknown/unsupported format name");
        }
        BufferedImage image = makeBufferedImage(arr, opts);
        ImageIO.write(image, identifier, new File(name));
    }

    /**
     * Save a shaped vector to a file.
     *
     * <p> The file format is guessed from the extension of the file name and
     * default options are used to encode the file.  </p>
     *
     * @param vec
     *        The shaped vector to save.
     *
     * @param name
     *        The name of the destination file.
     *
     * @throws IOException
     * @throws FileNotFoundException
     */
    public static void save(ShapedVector vec, String name)
            throws FileNotFoundException, IOException {
        save(vec, name, new FormatOptions());
    }

    /**
     * Save a shaped vector to a file with options.
     *
     * <p> The file format is guessed from the extension of the file name and
     * default options are used to encode the file.  </p>
     *
     * @param vec
     *        The shaped vector to save.
     *
     * @param name
     *        The name of the destination file.
     *
     * @param opts
     *        Options for encoding the data.
     *
     * @throws IOException
     * @throws FileNotFoundException
     */
    public static void save(ShapedVector vec, String name,
            FormatOptions opts)
                    throws FileNotFoundException, IOException {
        save(ArrayFactory.wrap(vec), name, opts);
    }

    /**
     * Save a vector to a file.
     *
     * <p> The file format is guessed from the extension of the file name and
     * default options are used to encode the file.  </p>
     *
     * @param vec
     *        The vector to save (only shaped vectors are supported for now).
     *
     * @param name
     *        The name of the destination file.
     *
     * @throws IOException
     * @throws FileNotFoundException
     */
    public static void save(Vector vec, String name)
            throws FileNotFoundException, IOException {
        save(vec, name, new FormatOptions());
    }

    /**
     * Save a vector to a file with options.
     *
     * <p> The file format is guessed from the extension of the file name and
     * default options are used to encode the file.  </p>
     *
     * @param vec
     *        The vector to save (only shaped vectors are supported for now).
     *
     * @param name
     *        The name of the destination file.
     *
     * @param opts
     *        Options for encoding the data.
     *
     * @throws IOException
     * @throws FileNotFoundException
     */
    public static void save(Vector vec, String name, FormatOptions opts)
            throws FileNotFoundException, IOException {
        if (! (vec instanceof ShapedVector)) {
            throw new IllegalArgumentException("Unsupported non-shaped vector");
        }
        save(ArrayFactory.wrap((ShapedVector)vec), name, opts);
    }

    /*=======================================================================*/
    /* MAKE BUFFERED IMAGES */

    /*
     * See: {@link Color#getRGB()} for the packing of colors in an int (Bits
     * 24-31 are alpha, 16-23 are red, 8-15 are green, 0-7 are blue)
     */

    /**
     * Make a buffered image from a shaped array with default options.
     *
     * @param arr
     *        The array to convert into an image.
     *
     * @see #makeBufferedImage(ShapedArray,FormatOptions) for more
     *      details about how is interpreted the array.
     */
    public static BufferedImage makeBufferedImage(ShapedArray arr) {
        return makeBufferedImage(arr, new FormatOptions());
    }

    /**
     * Make a buffered image from a shaped array with given options.
     *
     * <p> Different kind of shaped array can be interpreted as an image: 2D
     * shaped array are interpreted as grey-scaled images, 3D shaped arrays are
     * interpreted as RGB array if their first dimension is 3 and as RGBA array
     * if their first dimension is 4.  All other shapes result in throwing an
     * {@link IllegalArgumentException}.  The penultimate dimension of the
     * shaped array is the width of the image and the last dimension of the
     * shaped array is the height of the image.  Currently alpha channel (RGBA
     * images) are only implemented for byte shaped arrays.  </p>
     *
     * <p> If a color model of the format options is specified, then the source
     * image (that is the input shaped array interpreted as explained above) is
     * automatically converted according to this setting.  Otherwise, the
     * returned buffer image will have the same color model as the input one.
     * </p>
     *
     * @param arr
     *        The array to convert into an image.
     *
     * @param opts
     *        Options for conversion.
     */
    public static BufferedImage makeBufferedImage(ShapedArray arr,
            FormatOptions opts) {
        final int rank =  arr.getRank();
        final int arrayType = arr.getType();
        int depth = -1;
        int imageType = -1;
        if (rank == 2) {
            depth = 1;
            if (arrayType == Traits.BYTE) {
                imageType = BufferedImage.TYPE_BYTE_GRAY;
            } else {
                imageType = BufferedImage.TYPE_USHORT_GRAY;
            }
        } else if (rank == 3) {
            depth = arr.getDimension(0);
            if (depth == 3) {
                imageType = BufferedImage.TYPE_3BYTE_BGR;
            } else if (depth == 4 && arrayType == Traits.BYTE) {
                imageType = BufferedImage.TYPE_4BYTE_ABGR;
            }
        }
        if (imageType < 0) {
            throw new IllegalArgumentException("Conversion to image is only allowed for WIDTH x HEIGHT arrays, 3 x WIDTH x HEIGHT arrays or 4 x WIDTH x HEIGHT byte arrays");
        }
        final int width = arr.getDimension(rank - 2);
        final int height = arr.getDimension(rank - 1);
        final BufferedImage image = new BufferedImage(width, height, imageType);
        final WritableRaster raster = image.getRaster();
        final int minX = raster.getMinX();
        final int minY = raster.getMinY();
        if (imageType == BufferedImage.TYPE_BYTE_GRAY) {
            /* Input is a 2-D byte array.  No digitization is needed, the
             * array values are just copied as pixel values. */
            Byte2D src = (Byte2D)arr;
            if (raster.getNumBands() != 1 || raster.getNumDataElements() != 1 ||
                    raster.getTransferType() != DataBuffer.TYPE_BYTE) {
                throw new IllegalArgumentException("Assertion failed for TYPE_BYTE_GRAY");
            }
            if (src.isFlat()) {
                raster.setDataElements(minX, minY, width, height, src.flatten());
            } else {
                byte[] data = new byte[1];
                for (int y = 0; y < height; ++y) {
                    for (int x = 0; x < width; ++x) {
                        data[0] = src.get(x, y);
                        raster.setDataElements(minX + x, minY + y, data);
                    }
                }
            }
        } else if (imageType == BufferedImage.TYPE_USHORT_GRAY) {
            /* Input is 2-D array of any type but byte.  (FIXME: for small
             * integer types we can use a look-up table to speed up
             * computations.) */
            if (raster.getNumBands() != 1 || raster.getNumDataElements() != 1 ||
                    raster.getTransferType() != DataBuffer.TYPE_USHORT) {
                throw new IllegalArgumentException("Assertion failed for TYPE_USHORT_GRAY");
            }
            short[] data = new short[1];
            if (arrayType == Traits.SHORT) {
                Short2D src = (Short2D)arr;
                double[] sf = opts.getScaling(arr, 0, 0xFFFF);
                final float scale = (float)sf[0];
                final float bias = (float)sf[1];
                if (scale == 0) {
                    data[0] = 0;
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            raster.setDataElements(minX + x, minY + y, data);
                        }
                    }
                } else {
                    final float factor = 1/scale;
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            data[0] = toUInt16((src.get(x,y) - bias)*factor);
                            raster.setDataElements(minX + x, minY + y, data);
                        }
                    }
                }
            } else if (arrayType == Traits.INT) {
                Int2D src = (Int2D)arr;
                double[] sf = opts.getScaling(arr, 0, 0xFFFF);
                final float scale = (float)sf[0];
                final float bias = (float)sf[1];
                if (scale == 0) {
                    data[0] = 0;
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            raster.setDataElements(minX + x, minY + y, data);
                        }
                    }
                } else {
                    final float factor = 1/scale;
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            data[0] = toUInt16((src.get(x,y) - bias)*factor);
                            raster.setDataElements(minX + x, minY + y, data);
                        }
                    }
                }
            } else if (arrayType == Traits.LONG) {
                Long2D src = (Long2D)arr;
                double[] sf = opts.getScaling(arr, 0, 0xFFFF);
                final double scale = sf[0];
                final double bias = sf[1];
                if (scale == 0) {
                    data[0] = 0;
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            raster.setDataElements(minX + x, minY + y, data);
                        }
                    }
                } else {
                    final double factor = 1/scale;
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            data[0] = toUInt16((src.get(x,y) - bias)*factor);
                            raster.setDataElements(minX + x, minY + y, data);
                        }
                    }
                }
            } else if (arrayType == Traits.FLOAT) {
                Float2D src = (Float2D)arr;
                double[] sf = opts.getScaling(arr, 0, 0xFFFF);
                final float scale = (float)sf[0];
                final float bias = (float)sf[1];
                if (scale == 0) {
                    data[0] = 0;
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            raster.setDataElements(minX + x, minY + y, data);
                        }
                    }
                } else {
                    final float factor = 1/scale;
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            data[0] = toUInt16((src.get(x,y) - bias)*factor);
                            raster.setDataElements(minX + x, minY + y, data);
                        }
                    }
                }
            } else if (arrayType == Traits.DOUBLE) {
                Double2D src = (Double2D)arr;
                double[] sf = opts.getScaling(arr, 0, 0xFFFF);
                final double scale = sf[0];
                final double bias = sf[1];
                if (scale == 0) {
                    data[0] = 0;
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            raster.setDataElements(minX + x, minY + y, data);
                        }
                    }
                } else {
                    final double factor = 1/scale;
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            data[0] = toUInt16((src.get(x,y) - bias)*factor);
                            raster.setDataElements(minX + x, minY + y, data);
                        }
                    }
                }
            } else {
                throw new IllegalTypeException();
            }
        } else if (imageType == BufferedImage.TYPE_3BYTE_BGR) {
            /* Input is 3-D array of any type. */
            if (raster.getNumBands() != 3 || raster.getNumDataElements() != 3 ||
                    raster.getTransferType() != DataBuffer.TYPE_BYTE) {
                throw new IllegalArgumentException("Assertion failed for TYPE_3BYTE_BGR");
            }
            byte[] data = new byte[3];
            if (arrayType == Traits.BYTE) {
                Byte3D src = (Byte3D)arr;
                if (src.isFlat()) {
                    raster.setDataElements(minX, minY, width, height, src.flatten());
                } else {
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            data[0] = src.get(0, x, y);
                            data[1] = src.get(1, x, y);
                            data[2] = src.get(2, x, y);
                            raster.setDataElements(minX + x, minY + y, data);
                        }
                    }
                }
            } else if (arrayType == Traits.SHORT) {
                Short3D src = (Short3D)arr;
                double[] sf = opts.getScaling(arr, 0, 0xFF);
                final float scale = (float)sf[0];
                final float bias = (float)sf[1];
                final float factor = 1/scale;
                if (scale == 0) {
                    data[0] = 0;
                    data[1] = 0;
                    data[2] = 0;
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            raster.setDataElements(minX + x, minY + y, data);
                        }
                    }
                } else {
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            data[0] = toUInt8((src.get(0,x,y) - bias)*factor);
                            data[1] = toUInt8((src.get(1,x,y) - bias)*factor);
                            data[2] = toUInt8((src.get(2,x,y) - bias)*factor);
                            raster.setDataElements(minX + x, minY + y, data);
                        }
                    }
                }
            } else if (arrayType == Traits.INT) {
                Int3D src = (Int3D)arr;
                double[] sf = opts.getScaling(arr, 0, 0xFF);
                final float scale = (float)sf[0];
                final float bias = (float)sf[1];
                final float factor = 1/scale;
                if (scale == 0) {
                    data[0] = 0;
                    data[1] = 0;
                    data[2] = 0;
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            raster.setDataElements(minX + x, minY + y, data);
                        }
                    }
                } else {
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            data[0] = toUInt8((src.get(0,x,y) - bias)*factor);
                            data[1] = toUInt8((src.get(1,x,y) - bias)*factor);
                            data[2] = toUInt8((src.get(2,x,y) - bias)*factor);
                            raster.setDataElements(minX + x, minY + y, data);
                        }
                    }
                }
            } else if (arrayType == Traits.LONG) {
                Long3D src = (Long3D)arr;
                double[] sf = opts.getScaling(arr, 0, 0xFF);
                final double scale = sf[0];
                final double bias = sf[1];
                final double factor = 1/scale;
                if (scale == 0) {
                    data[0] = 0;
                    data[1] = 0;
                    data[2] = 0;
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            raster.setDataElements(minX + x, minY + y, data);
                        }
                    }
                } else {
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            data[0] = toUInt8((src.get(0,x,y) - bias)*factor);
                            data[1] = toUInt8((src.get(1,x,y) - bias)*factor);
                            data[2] = toUInt8((src.get(2,x,y) - bias)*factor);
                            raster.setDataElements(minX + x, minY + y, data);
                        }
                    }
                }
            } else if (arrayType == Traits.FLOAT) {
                Float3D src = (Float3D)arr;
                double[] sf = opts.getScaling(arr, 0, 0xFF);
                final float scale = (float)sf[0];
                final float bias = (float)sf[1];
                final float factor = 1/scale;
                if (scale == 0) {
                    data[0] = 0;
                    data[1] = 0;
                    data[2] = 0;
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            raster.setDataElements(minX + x, minY + y, data);
                        }
                    }
                } else {
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            data[0] = toUInt8((src.get(0,x,y) - bias)*factor);
                            data[1] = toUInt8((src.get(1,x,y) - bias)*factor);
                            data[2] = toUInt8((src.get(2,x,y) - bias)*factor);
                            raster.setDataElements(minX + x, minY + y, data);
                        }
                    }
                }
            } else if (arrayType == Traits.DOUBLE) {
                Double3D src = (Double3D)arr;
                double[] sf = opts.getScaling(arr, 0, 0xFF);
                final double scale = sf[0];
                final double bias = sf[1];
                final double factor = 1/scale;
                if (scale == 0) {
                    data[0] = 0;
                    data[1] = 0;
                    data[2] = 0;
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            raster.setDataElements(minX + x, minY + y, data);
                        }
                    }
                } else {
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            data[0] = toUInt8((src.get(0,x,y) - bias)*factor);
                            data[1] = toUInt8((src.get(1,x,y) - bias)*factor);
                            data[2] = toUInt8((src.get(2,x,y) - bias)*factor);
                            raster.setDataElements(minX + x, minY + y, data);
                        }
                    }
                }
            } else {
                throw new IllegalTypeException();
            }
        } else /* imageType == BufferedImage.TYPE_4BYTE_ABGR */ {
            /* Input is 3-D byte array. */
            Byte3D src = (Byte3D)arr;
            if (raster.getNumBands() != 4 || raster.getNumDataElements() != 4 ||
                    raster.getTransferType() != DataBuffer.TYPE_BYTE) {
                throw new IllegalArgumentException("Assertion failed for TYPE_4BYTE_ABGR");
            }
            if (src.isFlat()) {
                raster.setDataElements(minX, minY, width, height, src.flatten());
            } else {
                byte[] data = new byte[4];
                for (int y = 0; y < height; ++y) {
                    for (int x = 0; x < width; ++x) {
                        data[0] = src.get(0, x, y);
                        data[1] = src.get(1, x, y);
                        data[2] = src.get(2, x, y);
                        data[3] = src.get(3, x, y);
                        raster.setDataElements(minX + x, minY + y, data);
                    }
                }
            }
        }
        return image;
    }

    /**
     * Make a buffered image from a shaped vector with default options.
     *
     * @param vec
     *        The shaped vector.
     */
    public static BufferedImage makeBufferedImage(ShapedVector vec) {
        return makeBufferedImage(vec.asShapedArray());
    }

    /**
     * Make a buffered image from a shaped vector with given options.
     *
     * @param vec
     *        The shaped vector.
     *
     * @param opts
     *        The options for conversion.
     */
    public static BufferedImage makeBufferedImage(ShapedVector vec,
            FormatOptions opts) {
        return makeBufferedImage(vec.asShapedArray(), opts);
    }


    /*=======================================================================*/
    /* BUFFERED IMAGES TO ARRAYS */

    /**
     * Get the type name of a buffered image.
     *
     * @param image
     *        The buffered image.
     */
    public static String getImageTypeName(BufferedImage image) {
        return getImageTypeName(image.getType());
    }

    /**
     * Get the type name of a given buffered image type number.
     *
     * @param type
     *        The buffered image type number.
     */
    public static String getImageTypeName(int type) {
        switch (type) {
            case BufferedImage.TYPE_INT_RGB:
                return "TYPE_INT_RGB";
            case BufferedImage.TYPE_INT_BGR:
                return "TYPE_INT_BGR";
            case BufferedImage.TYPE_3BYTE_BGR:
                return "TYPE_3BYTE_BGR";
            case BufferedImage.TYPE_USHORT_565_RGB:
                return "TYPE_USHORT_565_RGB";
            case BufferedImage.TYPE_USHORT_555_RGB:
                return "TYPE_USHORT_555_RGB";
            case BufferedImage.TYPE_INT_ARGB:
                return "TYPE_INT_ARGB";
            case BufferedImage.TYPE_INT_ARGB_PRE:
                return "TYPE_INT_ARGB_PRE";
            case BufferedImage.TYPE_4BYTE_ABGR:
                return "TYPE_4BYTE_ABGR";
            case BufferedImage.TYPE_4BYTE_ABGR_PRE:
                return "TYPE_4BYTE_ABGR_PRE";
            case BufferedImage.TYPE_BYTE_GRAY:
                return "TYPE_BYTE_GRAY";
            case BufferedImage.TYPE_USHORT_GRAY:
                return "TYPE_USHORT_GRAY";
            case BufferedImage.TYPE_BYTE_BINARY:
                return "TYPE_BYTE_BINARY";
            case BufferedImage.TYPE_BYTE_INDEXED:
                return "TYPE_BYTE_INDEXED";
            case BufferedImage.TYPE_CUSTOM:
                return "TYPE_CUSTOM";
            default:
                return "UNKOWN";
        }
    }

    /** Get the name of the data type of a DataBuffer object. */
    public static String getDataTypeName(DataBuffer buffer) {
        return getDataTypeName(buffer.getDataType());
    }

    /** Get the name of the data type of a DataBuffer object. */
    public static String getDataTypeName(int type) {
        switch (type) {
            case DataBuffer.TYPE_BYTE:
                /* Unsigned byte data. */
                return "TYPE_BYTE";
            case DataBuffer.TYPE_SHORT:
                return "TYPE_SHORT";
            case DataBuffer.TYPE_USHORT:
                return "TYPE_USHORT";
            case DataBuffer.TYPE_INT:
                return "TYPE_INT";
            case DataBuffer.TYPE_FLOAT:
                return "TYPE_FLOAT";
            case DataBuffer.TYPE_DOUBLE:
                return "TYPE_DOUBLE";
            case DataBuffer.TYPE_UNDEFINED:
                return "TYPE_UNDEFINED";
            default:
                return "UNKOWN";
        }
    }

    public static void printImageInfo(PrintStream output, BufferedImage image, String name) {
        Raster raster = image.getRaster();
        final int height = raster.getHeight();
        final int width = raster.getWidth();
        final int minX = raster.getMinX();
        final int minY = raster.getMinY();
        final int numBands = raster.getNumBands();
        final int transferType = raster.getTransferType();
        final int numElements = raster.getNumDataElements();
        if (name != null) {
            output.format("image name       %s\n", name);
        }
        output.format("image type:      %s\n", getImageTypeName(image));
        output.format("image size:      %d x %d\n", width, height);
        output.format("image origin:    (%d,%d)\n", minX, minY);
        output.format("number of bands: %d\n", numBands);
        output.format("transfer data:   %s x %d\n", getDataTypeName(transferType), numElements);
    }

    /**
     * Convert a BufferedImage to a given image type.
     *
     * <p> This method converts an image of any type to the given type.  The
     * operation is lazy: if the image type of the original image is already
     * the correct one, it is returned as the result.  </p>>
     *
     * @param img
     *        The input image.
     *
     * @param type
     *        The image type of the result.
     *
     * @return A {@link BufferedImage} of the requested type.
     *
     * @see BufferedImage#TYPE_3BYTE_BGR
     * @see BufferedImage#TYPE_4BYTE_ABGR
     * @see BufferedImage#TYPE_4BYTE_ABGR_PRE
     * @see BufferedImage#TYPE_BYTE_BINARY
     * @see BufferedImage#TYPE_BYTE_INDEXED
     * @see BufferedImage#TYPE_BYTE_GRAY
     * @see BufferedImage#TYPE_USHORT_555_RGB
     * @see BufferedImage#TYPE_USHORT_565_RGB
     * @see BufferedImage#TYPE_USHORT_GRAY
     * @see BufferedImage#TYPE_INT_RGB
     * @see BufferedImage#TYPE_INT_BGR
     * @see BufferedImage#TYPE_INT_ARGB
     * @see BufferedImage#TYPE_INT_ARGB_PRE
     */
    public static BufferedImage convertImage(BufferedImage img, int type) {
        if (img.getType() == type) {
            return img;
        }
        BufferedImage result = new BufferedImage(img.getWidth(), img.getHeight(), type);
        Graphics2D g2 = result.createGraphics();
        g2.drawImage(img, null, null);
        g2.dispose();
        return result;
    }

    /**
     * Convert an image into a shaped array.
     *
     * <p> This method attempts to convert the image data with no loss of
     * information.  The data type and rank of the result depend on the image
     * type. </p>
     *
     * <p> Gray-scaled images yield 2D array while colored (RGB) and
     * translucent (RGBA) images yields 3D images.  The penultimate dimension
     * is the width of the image and the last dimension is the height of the
     * image.  For RGB and RGBA images, the first dimension is respectively 3
     * and 4 with the red red, green, blue, and alpha channels (if any) given
     * by: </p>
     *
     * <pre>
     *    Array3D arr   = imageToShapedArray(image);
     *    Array2D red   = arr.slice(0, 0);
     *    Array2D green = arr.slice(1, 0);
     *    Array2D blue  = arr.slice(2, 0);
     *    Array2D alpha = arr.slice(3, 0);
     * </pre>
     *
     * <p> If the result is a {@link mitiv.array.ByteArray}, the bytes are to
     * be interpreted as being unsigned.  For {@link
     * BufferedImage#TYPE_USHORT_GRAY} images, the result is a {@link Int2D}
     * array as TiPi does not implement {@code unsigned short} data. </p>
     *
     * <p> For translucent images, the color levels in the result are not
     * pre-multiplied by the alpha value (for {@link
     * BufferedImage#TYPE_INT_ARGB_PRE} and {@link
     * BufferedImage#TYPE_4BYTE_ABGR_PRE}, the pre-multiplication is reversed).
     * </p>
     *
     * @return A shaped array.
     */
    public static ShapedArray imageToShapedArray(BufferedImage image) {
        WritableRaster raster = image.getRaster();
        final int height = raster.getHeight();
        final int width = raster.getWidth();
        final int minX = raster.getMinX();
        final int minY = raster.getMinY();
        final int numBands = raster.getNumBands();
        final int transferType = raster.getTransferType();
        final int numDataElements = raster.getNumDataElements();
        final int type = image.getType();
        if (type == BufferedImage.TYPE_INT_RGB) {
            if (numBands != 3 || numDataElements != 1 || transferType != DataBuffer.TYPE_INT) {
                throw new IllegalArgumentException("Assertion failed for TYPE_INT_RGB");
            }
            Byte3D arr = Byte3D.wrap(new byte[3*width*height], 3, width, height);
            int[] data = new int[1];
            for (int y = 0; y < height; ++y) {
                for (int x = 0; x < width; ++x) {
                    raster.getDataElements(x + minX, y + minY, data);
                    int pixel = data[0];
                    byte red   = (byte)((pixel >> 16) & 0xFF);
                    byte green = (byte)((pixel >>  8) & 0xFF);
                    byte blue  = (byte)((pixel      ) & 0xFF);
                    arr.set(0, x, y, red);
                    arr.set(1, x, y, green);
                    arr.set(2, x, y, blue);
                }
            }
            return arr;
        } else if (type == BufferedImage.TYPE_INT_ARGB) {
            if (numBands != 4 || numDataElements != 1 || transferType != DataBuffer.TYPE_INT) {
                throw new IllegalArgumentException("Assertion failed for TYPE_INT_ARGB");
            }
            Byte3D arr = Byte3D.wrap(new byte[4*width*height], 4, width, height);
            int[] data = new int[1];
            for (int y = 0; y < height; ++y) {
                for (int x = 0; x < width; ++x) {
                    raster.getDataElements(x + minX, y + minY, data);
                    int pixel = data[0];
                    byte alpha = (byte)((pixel >> 24) & 0xFF);
                    byte red   = (byte)((pixel >> 16) & 0xFF);
                    byte green = (byte)((pixel >>  8) & 0xFF);
                    byte blue  = (byte)((pixel      ) & 0xFF);
                    arr.set(0, x, y, red);
                    arr.set(1, x, y, green);
                    arr.set(2, x, y, blue);
                    arr.set(3, x, y, alpha);
                }
            }
            return arr;
        } else if (type == BufferedImage.TYPE_INT_ARGB_PRE) {
            /* Assume colors have been premultiplied by alpha/255. */
            final int b = 255;
            final int a = 2*b;
            if (numBands != 4 || numDataElements != 1 || transferType != DataBuffer.TYPE_INT) {
                throw new IllegalArgumentException("Assertion failed for TYPE_INT_ARGB_PRE");
            }
            Byte3D arr = Byte3D.wrap(new byte[4*width*height], 4, width, height);
            int[] data = new int[1];
            for (int y = 0; y < height; ++y) {
                for (int x = 0; x < width; ++x) {
                    raster.getDataElements(x + minX, y + minY, data);
                    int pixel = data[0];
                    int alpha = (pixel >> 24) & 0xFF;
                    int c = 2*alpha;
                    int red   = (((pixel >> 16) & 0xFF)*c + b)/a;
                    int green = (((pixel >>  8) & 0xFF)*c + b)/a;
                    int blue  = (((pixel      ) & 0xFF)*c + b)/a;
                    arr.set(0, x, y, (byte)red);
                    arr.set(1, x, y, (byte)green);
                    arr.set(2, x, y, (byte)blue);
                    arr.set(3, x, y, (byte)alpha);
                }
            }
            return arr;
        } else if (type == BufferedImage.TYPE_INT_BGR) {
            if (numBands != 3 || numDataElements != 1 || transferType != DataBuffer.TYPE_INT) {
                throw new IllegalArgumentException("Assertion failed for TYPE_INT_BGR");
            }
            Byte3D arr = Byte3D.wrap(new byte[3*width*height], 3, width, height);
            int[] data = new int[1];
            for (int y = 0; y < height; ++y) {
                for (int x = 0; x < width; ++x) {
                    raster.getDataElements(x + minX, y + minY, data);
                    int pixel = data[0];
                    int red   = (pixel >>  8) & 0xFF;
                    int green = (pixel >> 16) & 0xFF;
                    int blue  = (pixel >> 24) & 0xFF;
                    arr.set(0, x, y, (byte)red);
                    arr.set(1, x, y, (byte)green);
                    arr.set(2, x, y, (byte)blue);
                }
            }
            return arr;
        } else if (type == BufferedImage.TYPE_USHORT_565_RGB) {
            if (numBands != 3 || numDataElements != 1 || transferType != DataBuffer.TYPE_USHORT) {
                throw new IllegalArgumentException("Assertion failed for TYPE_USHORT_565_RGB");
            }
            Byte3D arr = Byte3D.wrap(new byte[3*width*height], 3, width, height);
            short[] data = new short[1];
            for (int y = 0; y < height; ++y) {
                for (int x = 0; x < width; ++x) {
                    raster.getDataElements(x + minX, y + minY, data);
                    int pixel = data[0];
                    int red   = (pixel >> 11) & 0x1F;
                    int green = (pixel >>  5) & 0x3F;
                    int blue  = (pixel      ) & 0x1F;
                    arr.set(0, x, y, (byte)red);
                    arr.set(1, x, y, (byte)green);
                    arr.set(2, x, y, (byte)blue);
                }
            }
            return arr;
        } else if (type == BufferedImage.TYPE_USHORT_555_RGB) {
            if (numBands != 3 || numDataElements != 1 || transferType != DataBuffer.TYPE_USHORT) {
                throw new IllegalArgumentException("Assertion failed for TYPE_USHORT_555_RGB");
            }
            Byte3D arr = Byte3D.wrap(new byte[3*width*height], 3, width, height);
            short[] data = new short[1];
            for (int y = 0; y < height; ++y) {
                for (int x = 0; x < width; ++x) {
                    raster.getDataElements(x + minX, y + minY, data);
                    int pixel = data[0];
                    int red   = (pixel >> 10) & 0x1F;
                    int green = (pixel >>  5) & 0x1F;
                    int blue  = (pixel      ) & 0x1F;
                    arr.set(0, x, y, (byte)red);
                    arr.set(1, x, y, (byte)green);
                    arr.set(2, x, y, (byte)blue);
                }
            }
            return arr;
        } else if (type == BufferedImage.TYPE_3BYTE_BGR) {
            if (numBands != 3 || numDataElements != 3 || transferType != DataBuffer.TYPE_BYTE) {
                throw new IllegalArgumentException("Assertion failed for TYPE_3BYTE_BGR");
            }
            Byte3D arr = Byte3D.wrap(new byte[3*width*height], 3, width, height);
            byte[] data = new byte[3];
            for (int y = 0; y < height; ++y) {
                for (int x = 0; x < width; ++x) {
                    raster.getDataElements(x + minX, y + minY, data);
                    arr.set(0, x, y, data[0]); /* red*/
                    arr.set(1, x, y, data[1]); /* green */
                    arr.set(2, x, y, data[2]); /* blue */
                }
            }
            return arr;
        } else if (type == BufferedImage.TYPE_4BYTE_ABGR) {
            if (numBands != 4 || numDataElements != 4 || transferType != DataBuffer.TYPE_BYTE) {
                throw new IllegalArgumentException("Assertion failed for TYPE_4BYTE_ABGR");
            }
            Byte3D arr = Byte3D.wrap(new byte[4*width*height], 4, width, height);
            byte[] data = new byte[4];
            for (int y = 0; y < height; ++y) {
                for (int x = 0; x < width; ++x) {
                    raster.getDataElements(x + minX, y + minY, data);
                    arr.set(0, x, y, data[0]); /* red*/
                    arr.set(1, x, y, data[1]); /* green */
                    arr.set(2, x, y, data[2]); /* blue */
                    arr.set(3, x, y, data[3]); /* alpha */
                }
            }
            return arr;
        } else if (type == BufferedImage.TYPE_4BYTE_ABGR_PRE) {
            /* Assume colors have been premultiplied by alpha/255. */
            final int b = 255;
            final int a = 2*b;
            if (numBands != 4 || numDataElements != 4 || transferType != DataBuffer.TYPE_BYTE) {
                throw new IllegalArgumentException("Assertion failed for TYPE_4BYTE_ABGR_PRE");
            }
            Byte3D arr = Byte3D.wrap(new byte[4*width*height], 4, width, height);
            byte[] data = new byte[4];
            for (int y = 0; y < height; ++y) {
                for (int x = 0; x < width; ++x) {
                    raster.getDataElements(x + minX, y + minY, data);
                    int alpha = data[3];
                    int c = 2*alpha;
                    int red   = (data[0]*c + b)/a;
                    int green = (data[1]*c + b)/a;
                    int blue  = (data[2]*c + b)/a;
                    arr.set(0, x, y, (byte)red);
                    arr.set(1, x, y, (byte)green);
                    arr.set(2, x, y, (byte)blue);
                    arr.set(3, x, y, (byte)alpha);
                }
            }
            return arr;
        } else if (type == BufferedImage.TYPE_BYTE_GRAY) {
            if (numBands != 1 || numDataElements != 1 || transferType != DataBuffer.TYPE_BYTE) {
                throw new IllegalArgumentException("Assertion failed for TYPE_BYTE_GRAY");
            }
            Byte2D arr = Byte2D.wrap(new byte[width*height], width, height);
            byte[] data = new byte[1];
            for (int y = 0; y < height; ++y) {
                for (int x = 0; x < width; ++x) {
                    raster.getDataElements(x + minX, y + minY, data);
                    arr.set(x, y, data[0]);
                }
            }
            return arr;
        } else if (type == BufferedImage.TYPE_USHORT_GRAY) {
            if (numBands != 1 || numDataElements != 1 || transferType != DataBuffer.TYPE_USHORT) {
                throw new IllegalArgumentException("Assertion failed for TYPE_USHORT_GRAY");
            }
            Int2D arr = Int2D.wrap(new int[width*height], width, height);
            short[] data = new short[1];
            for (int y = 0; y < height; ++y) {
                for (int x = 0; x < width; ++x) {
                    raster.getDataElements(x + minX, y + minY, data);
                    arr.set(x, y, data[0]);
                }
            }
            return arr;
        } else if (type == BufferedImage.TYPE_BYTE_BINARY) {
            if (numBands != 1 || numDataElements != 1 || transferType != DataBuffer.TYPE_BYTE) {
                throw new IllegalArgumentException("Assertion failed for TYPE_BYTE_BINARY");
            }
            Byte2D arr = Byte2D.wrap(new byte[width*height], width, height);
            // FIXME:
            byte[] data = new byte[1];
            for (int y = 0; y < height; ++y) {
                for (int x = 0; x < width; ++x) {
                    raster.getDataElements(x + minX, y + minY, data);
                    arr.set(x, y, data[0]);
                }
            }
            return arr;
            //} else if (type == BufferedImage.TYPE_BYTE_INDEXED) {
            //} else if (type == BufferedImage.TYPE_CUSTOM) {
        }

        /* For all other image types, we use a fallback method. */
        if (transferType == DataBuffer.TYPE_FLOAT) {
            if (numBands == 1) {
                Float2D arr = Float2D.wrap(new float[width*height], width, height);
                for (int y = 0; y < height; ++y) {
                    for (int x = 0; x < width; ++x) {
                        arr.set(x, y, raster.getSampleFloat(x + minX, y + minY, 0));
                    }
                }
                return arr;
            } else {
                Float3D arr = Float3D.wrap(new float[numBands*width*height], numBands, width, height);
                if (numBands == 3) {
                    /* unroll the innermost loop for the most common case */
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            arr.set(0, x, y, raster.getSampleFloat(x + minX, y + minY, 0));
                            arr.set(1, x, y, raster.getSampleFloat(x + minX, y + minY, 1));
                            arr.set(2, x, y, raster.getSampleFloat(x + minX, y + minY, 2));
                        }
                    }
                } else {
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            for (int b = 0; b < numBands; ++b) {
                                arr.set(b, x, y, raster.getSampleFloat(x + minX, y + minY, b));
                            }
                        }
                    }
                }
                return arr;
            }
        } else if (transferType == DataBuffer.TYPE_DOUBLE) {
            if (numBands == 1) {
                Double2D arr = Double2D.wrap(new double[width*height], width, height);
                for (int y = 0; y < height; ++y) {
                    for (int x = 0; x < width; ++x) {
                        arr.set(x, y, raster.getSampleDouble(x + minX, y + minY, 0));
                    }
                }
                return arr;
            } else {
                Double3D arr = Double3D.wrap(new double[numBands*width*height], numBands, width, height);
                if (numBands == 3) {
                    /* unroll the innermost loop for the most common case */
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            arr.set(0, x, y, raster.getSampleDouble(x + minX, y + minY, 0));
                            arr.set(1, x, y, raster.getSampleDouble(x + minX, y + minY, 1));
                            arr.set(2, x, y, raster.getSampleDouble(x + minX, y + minY, 2));
                        }
                    }
                } else {
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            for (int b = 0; b < numBands; ++b) {
                                arr.set(b, x, y, raster.getSampleDouble(x + minX, y + minY, b));
                            }
                        }
                    }
                }
                return arr;
            }
        } else {
            if (numBands == 1) {
                Int2D arr = Int2D.wrap(new int[width*height], width, height);
                for (int y = 0; y < height; ++y) {
                    for (int x = 0; x < width; ++x) {
                        arr.set(x, y, raster.getSample(x + minX, y + minY, 0));
                    }
                }
                return arr;
            } else {
                Int3D arr = Int3D.wrap(new int[numBands*width*height], numBands, width, height);
                if (numBands == 3) {
                    /* unroll the innermost loop for the most common case */
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            arr.set(0, x, y, raster.getSample(x + minX, y + minY, 0));
                            arr.set(1, x, y, raster.getSample(x + minX, y + minY, 1));
                            arr.set(2, x, y, raster.getSample(x + minX, y + minY, 2));
                        }
                    }
                } else {
                    for (int y = 0; y < height; ++y) {
                        for (int x = 0; x < width; ++x) {
                            for (int b = 0; b < numBands; ++b) {
                                arr.set(b, x, y, raster.getSample(x + minX, y + minY, b));
                            }
                        }
                    }
                }
                return arr;
            }
        }
    }

    /**
     * Convert a floating point value in an 8-bit unsigned integer.
     *
     * <p> This static method, round the floating point value and clamp it to
     * fit in an 8-bit unsigned integer. </p>
     *
     * @param val
     *        A single precision floating point value.
     *
     * @return An 8-bit unsigned integer value stored into a {@code byte}.
     */
    private static final byte toUInt8(float val) {
        return (byte)max(0, min(round(val), 0xFF));
    }

    /**
     * Convert a floating point value in a 16-bit unsigned integer.
     *
     * <p> This static method, round the floating point value and clamp it to
     * fit in an 16-bit unsigned integer. </p>
     *
     * @param val
     *        A single precision floating point value.
     *
     * @return A 16-bit unsigned integer value stored into a {@code short}.
     */
    private static final short toUInt16(float val) {
        return (short)max(0, min(round(val), 0xFFFF));
    }

    /**
     * Convert a floating point value in an 8-bit unsigned integer.
     *
     * <p> This static method, round the floating point value and clamp it to
     * fit in an 8-bit unsigned integer. </p>
     *
     * @param val
     *        A double precision floating point value.
     *
     * @return An 8-bit unsigned integer value stored into a {@code byte}.
     */
    private static final byte toUInt8(double val) {
        return (byte)max(0, min(round(val), 0xFF));
    }

    /**
     * Convert a floating point value in a 16-bit unsigned integer.
     *
     * <p> This static method, round the floating point value and clamp it to
     * fit in an 16-bit unsigned integer. </p>
     *
     * @param val
     *        A double precision floating point value.
     *
     * @return A 16-bit unsigned integer value stored into a {@code short}.
     */
    private static final short toUInt16(double val) {
        return (short)max(0, min(round(val), 0xFFFF));
    }


    /*=======================================================================*/
    /* TESTS */

    public static void main(String[] args) {
        String[] str;
        str = ImageIO.getReaderFormatNames();
        System.out.format("Format names understood by registered readers:\n");
        for (int i = 0; i < str.length; ++i) {
            System.out.format("  - %s\n", str[i]);
        }

        str = ImageIO.getReaderFileSuffixes();
        System.out.format("\nImage suffixes understood by registered readers:\n");
        for (int i = 0; i < str.length; ++i) {
            System.out.format("  - %s\n", str[i]);
        }

        str = ImageIO.getWriterFormatNames();
        System.out.format("\nFormat names understood by registered writers:\n");
        for (int i = 0; i < str.length; ++i) {
            System.out.format("  - %s\n", str[i]);
        }

        str = ImageIO.getWriterFileSuffixes();
        System.out.format("\nImage suffixes understood by registered writers:\n");
        for (int i = 0; i < str.length; ++i) {
            System.out.format("  - %s\n", str[i]);
        }

        if (args == null || args.length == 0) {
            args = new String[]{"/tmp/test-image.jpg"};
        }
        BufferedImage image;
        int[] imageTypes = new int[]{
                BufferedImage.TYPE_INT_RGB,
                BufferedImage.TYPE_INT_BGR,
                BufferedImage.TYPE_3BYTE_BGR,
                BufferedImage.TYPE_USHORT_565_RGB,
                BufferedImage.TYPE_USHORT_555_RGB,
                BufferedImage.TYPE_INT_ARGB,
                BufferedImage.TYPE_INT_ARGB_PRE,
                BufferedImage.TYPE_4BYTE_ABGR,
                BufferedImage.TYPE_4BYTE_ABGR_PRE,
                BufferedImage.TYPE_BYTE_GRAY,
                BufferedImage.TYPE_USHORT_GRAY,
                BufferedImage.TYPE_BYTE_BINARY,
                BufferedImage.TYPE_BYTE_INDEXED
        };
        for (int k = 0; k < args.length; ++k) {
            String name = args[k];
            try {
                image = ImageIO.read(new File(name));
                printImageInfo(System.out, image, name);
                System.out.format("\n");
                for (int j = 0; j < imageTypes.length; ++j) {
                    BufferedImage converted = convertImage(image, imageTypes[j]);
                    printImageInfo(System.out, converted, name);
                }
            } catch (IOException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
        }
    }

}
