"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TextractAsyncToJSON = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
/**
 * combines the potentially paginated response from async Textract calls and stores as one combines JSON.
 *
 * This construct is not memory optimzed (yet) and will combine all JSON by loading them to memory.
 * Large responses could exceed the memory potentially, the memory size is set to Lambda max.
 *
 * Reduce the memory size to your needs if your processing does not yield large responses to save Lamda cost.
 *
 *
 * Input: "textract_result"."TextractTempOutputJsonPath"
 * Output: "TextractOutputJsonPath"
 *
 * Example (Python)
 * ```python
    textract_async_to_json = tcdk.TextractAsyncToJSON(
        self,
        "TextractAsyncToJSON2",
        s3_output_prefix=s3_output_prefix,
        s3_output_bucket=s3_output_bucket)
  ```
 */
class TextractAsyncToJSON extends sfn.StateMachineFragment {
    constructor(parent, id, props) {
        super(parent, id);
        var lambdaLogLevel = props.lambdaLogLevel === undefined ? 'DEBUG' : props.lambdaLogLevel;
        var lambdaTimeout = props.lambdaTimeout === undefined ? 900 : props.lambdaTimeout;
        var lambdaMemoryMB = props.lambdaMemoryMB === undefined ? 10240 : props.lambdaMemoryMB;
        var textractAPI = props.textractAPI === undefined ? 'GENERIC' : props.textractAPI;
        var s3OutputPrefix = props.s3OutputPrefix === undefined ? '' : props.s3OutputPrefix;
        var s3InputPrefix = props.s3InputPrefix === undefined ? '' : props.s3InputPrefix;
        this.asyncToJSONFunction = new lambda.DockerImageFunction(this, 'TextractAsyncToJSON', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/async_to_json/')),
            memorySize: lambdaMemoryMB,
            architecture: lambda.Architecture.X86_64,
            environment: {
                S3_OUTPUT_BUCKET: props.s3OutputBucket,
                S3_OUTPUT_PREFIX: props.s3OutputPrefix,
                LOG_LEVEL: lambdaLogLevel,
                TEXTRACT_API: textractAPI,
            },
            timeout: aws_cdk_lib_1.Duration.seconds(lambdaTimeout),
        });
        /** ################ INPUT BUCKET POLICIES */
        if (props.inputPolicyStatements === undefined) {
            if (props.s3InputBucket === undefined) {
                this.asyncToJSONFunction.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['s3:GetObject', 's3:ListBucket'],
                    resources: ['*'],
                }));
            }
            else {
                this.asyncToJSONFunction.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['s3:GetObject'],
                    resources: [
                        path.join(`arn:aws:s3:::${props.s3InputBucket}`, s3InputPrefix, '/*'),
                    ],
                }));
                this.asyncToJSONFunction.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['s3:ListBucket'],
                    resources: [
                        path.join(`arn:aws:s3:::${props.s3InputBucket}`),
                    ],
                }));
            }
        }
        else {
            for (var policyStatement of props.inputPolicyStatements) {
                this.asyncToJSONFunction.addToRolePolicy(policyStatement);
            }
        }
        /** ##################### OUTPUT BUCKET POLICIES */
        if (props.outputPolicyStatements === undefined) {
            if (props.s3OutputBucket === undefined) {
                this.asyncToJSONFunction.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['s3:PutObject'],
                    resources: ['*'],
                }));
            }
            else {
                this.asyncToJSONFunction.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['s3:PutObject'],
                    resources: [
                        path.join(`arn:aws:s3:::${props.s3OutputBucket}`, s3OutputPrefix, '/'),
                        path.join(`arn:aws:s3:::${props.s3OutputBucket}`, s3OutputPrefix, '/*'),
                    ],
                }));
            }
        }
        else {
            for (var policyStatement of props.outputPolicyStatements) {
                this.asyncToJSONFunction.addToRolePolicy(policyStatement);
            }
        }
        const asyncToJSON = new tasks.LambdaInvoke(this, id, {
            lambdaFunction: this.asyncToJSONFunction,
            timeout: aws_cdk_lib_1.Duration.seconds(900),
            outputPath: '$.Payload',
        });
        this.startState = asyncToJSON;
        this.endStates = [asyncToJSON];
    }
}
exports.TextractAsyncToJSON = TextractAsyncToJSON;
_a = JSII_RTTI_SYMBOL_1;
TextractAsyncToJSON[_a] = { fqn: "amazon-textract-idp-cdk-constructs.TextractAsyncToJSON", version: "0.0.16" };
//# sourceMappingURL=data:application/json;base64,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