# Copyright 2021 Sean Robertson

# Licensed under the Apache License,  Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This code was adapted from the sphere2pipe_2.5 tool. The original license
# is below

# Various portions of source code from Tony Robinson's "shorten-2.0"
# package are used here by permission of Tony Robinson and SoftSound,
# Inc. <http://www.softsound.com> -- these portions are found in the file
# "shorten_x.c"; please note the copyright information in that file.  By
# agreement with Tony Robinson and SoftSound, Inc, the Linguistic Data
# Consortium (LDC) grants permission to copy and use this software for the
# purpose of reading "shorten"-compressed speech data provided in NIST
# SPHERE file format by the LDC or others.  SoftSound provides useful
# tools for audio compression and other signal processing tasks.
#
# Other portions of source code (in particular the "writeRIFFHeader" and
# "writeAIFFHeader" functions in "file_headers.c", and the "alaw2pcm"
# conversion function) were adapted from the "SoX" package, a valuable
# open-source tool maintained primarily by Chris Bagwell, with assistance
# from many others (http://sox.sourceforge.net/).  We gratefully
# acknowledge the value provided by all contributors to SoX; sph2pipe
# would have been much harder to write without this resource.  We
# recommend that you use SoX if you need to do sample-rate conversion on
# audio data.

# we're not doing any encoding into shortened, so it looks like we're okay

# There be dragons here. God save us all


import struct
import warnings

import numpy as np


def c99_div(a, b):
    # c99 truncates towrads zero, whereas python floors. In python:
    # -1 // 2 == -1
    # in c:
    # -1 // 2 == 0
    # you only need to worry for potentially negative values
    return int(float(a) / b)


def read_header(file_, error):
    inpbuf = file_.read(1024)
    if len(inpbuf) != 1024 or inpbuf[:7] != b"NIST_1A":
        raise error
    hdrsize = int(inpbuf.split(b"\n")[1])
    if hdrsize < 1024:
        raise error
    inpbuf += file_.read(hdrsize - len(inpbuf))
    field = None
    samptype = sampsize = sampcount = samprate = chancount = inporder = None
    for field in inpbuf.split(b"\n")[2:]:
        if field == b"end_head":
            break
        field = field.decode().split()
        key, fmt = field[:2]
        value = " ".join(field[2:])
        if fmt == "-i":
            value = int(value)
        if key == "channel_count":
            chancount = value
        elif key == "sample_count":
            sampcount = value
        elif key == "sample_rate":
            samprate = value
        elif key == "sample_n_bytes":
            sampsize = value
        elif key == "sample_byte_format":
            inporder = value
        elif key == "sample_coding":
            for prefix in {"alaw", "ulaw", "pcm"}:
                if value.startswith(prefix):
                    samptype = prefix
    if field != b"end_head":
        raise error
    if not samptype and (sampsize == 2 or (inporder and len(inporder) == 2)):
        samptype = "pcm"
    if (
        not samptype
        or not sampcount
        or not samprate
        or not chancount
        or (samptype == "pcm" and not inporder)
    ):
        raise error
    if not sampsize:
        sampsize = samptype & 3
    return samptype, sampsize, sampcount, samprate, chancount, inporder


def fix_bitshift(buffer, nitem, bitshift, ftype):
    if ftype == TYPE_AU1:
        buffer[:nitem] = ULAW_OUTWARD[bitshift, buffer[:nitem] + 128]
    elif ftype == TYPE_AU2:
        for i in range(nitem):
            if buffer[i] >= 0:
                buffer[i] = ULAW_OUTWARD[bitshift, buffer[i] + 128]
            elif buffer[i] == -1:
                buffer[i] = NEGATIVE_ULAW_ZERO
            else:
                buffer[i] = ULAW_OUTWARD[bitshift, buffer[i] + 129]
    elif bitshift:
        buffer <<= bitshift


def copy_shortened_samples(inpbuf, file_, data, error):
    inpbuf = memoryview(inpbuf)
    maxnlpc = lpcqoffset = cmd = bitshift = 0
    sampsdone = 0

    assert inpbuf[:4] == MAGIC
    (version,) = struct.unpack("b", inpbuf[4:5].tobytes())
    # what's the point of nmean here if we just read it later?
    if version == 1:
        nmean = DEFAULT_V0NMEAN
    elif MIN_SUPPORTED_VERSION <= version <= MAX_SUPPORTED_VERSION:
        nmean = DEFAULT_V2NMEAN
    else:
        raise error

    def word_get():
        inpbuf = word_get.inpbuf
        if len(inpbuf) < 4:
            inpbuf = inpbuf.tobytes() + file_.read(BUFSIZ - len(inpbuf))
            inpbuf = memoryview(inpbuf)
            if len(inpbuf) < 4:
                raise error
        (buffer,) = struct.unpack(">l", inpbuf[:4].tobytes())
        word_get.inpbuf = inpbuf[4:]
        return buffer

    word_get.inpbuf = inpbuf[5:]

    masktab = np.empty(MASKTABSIZE, dtype=np.uint32)
    val = 0
    masktab[0] = val
    for i in range(1, MASKTABSIZE):
        val <<= 1
        val |= 1
        masktab[i] = val

    def uvar_get(nbin):
        gbuffer = uvar_get.gbuffer
        nbitget = uvar_get.nbitget
        if not nbitget:
            gbuffer = word_get()
            nbitget = NBITPERLONG
        result = 0
        while True:
            nbitget -= 1
            if gbuffer & (1 << nbitget):
                break
            if not nbitget:
                gbuffer = word_get()
                nbitget = NBITPERLONG
            result += 1
        while nbin:
            if nbitget >= nbin:
                result <<= nbin
                result |= (gbuffer >> (nbitget - nbin)) & masktab[nbin]
                nbitget -= nbin
                nbin = 0
            else:
                result = (result << nbitget) | (gbuffer & masktab[nbitget])
                gbuffer = word_get()
                nbin -= nbitget
                nbitget = NBITPERLONG
        uvar_get.gbuffer = gbuffer
        uvar_get.nbitget = nbitget
        return result

    uvar_get.gbuffer = uvar_get.nbitget = 0

    def ulong_get():
        nbit = uvar_get(ULONGSIZE)
        return uvar_get(nbit)

    def var_get(nbin):
        uvar = uvar_get(nbin + 1)
        if uvar & 1:
            return ~(uvar >> 1)
        else:
            return uvar >> 1

    ftype = ulong_get()
    if ftype >= 9:
        raise error
    convert = data.dtype.itemsize > 1 and ftype in {TYPE_AU1, TYPE_AU2}
    nchan = ulong_get()
    blocksize = ulong_get()
    maxnlpc = ulong_get()
    nmean = ulong_get()
    nskip = ulong_get()

    for _ in range(nskip):
        # shorten_x.c writes this directly out to fpout, but I don't think
        # it's part of the buffer. let's just see if this ever triggers
        if not nskip:
            warnings.warn("skipping {} bytes due to nskip".format(nskip))
        uvar_get(XBITESIZE)

    nwrap = max(maxnlpc, NWRAP)
    # careful! shorten_x.c shifts index i to index i - nwrap. Thus, whenever
    # i = 0 in shorten_x.c, we should be at nwrap
    buffer = np.zeros((nchan, blocksize + nwrap), dtype=np.int32)

    if maxnlpc:
        qlpc = np.empty(maxnlpc, dtype=np.int32)

    if version > 1:
        lpcqoffset = V2LPCQOFFSET

    nblock = max(1, nmean)
    if ftype in {TYPE_AU1, TYPE_S8, TYPE_S16HL, TYPE_S16LH, TYPE_ULAW, TYPE_AU2}:
        mean = 0
    elif ftype == TYPE_U8:
        mean = 0x8
    elif ftype in {TYPE_U16HL, TYPE_U16LH}:
        mean = 0x8000
    else:
        raise error
    offset = np.full((nchan, nblock), mean, dtype=np.int32)

    chan = 0
    while True:
        cmd = uvar_get(FNSIZE)
        if cmd == FN_QUIT:
            break
        if cmd in {FN_ZERO, FN_DIFF0, FN_DIFF1, FN_DIFF2, FN_DIFF3, FN_QLPC}:
            cbuffer = buffer[chan]

            if cmd != FN_ZERO:
                resn = uvar_get(ENERGYSIZE)

            if nmean:
                sum = 0 if version < 2 else nmean // 2
                sum += offset[chan, :nmean].sum()
                if version < 2:
                    coffset = c99_div(sum, nmean)
                else:
                    coffset = c99_div(sum, nmean) >> bitshift
            else:
                coffset = offset[chan, 0]

            if cmd == FN_ZERO:
                cbuffer[nwrap : blocksize + nwrap] = 0
            elif cmd == FN_DIFF0:
                for i in range(nwrap, blocksize + nwrap):
                    cbuffer[i] = var_get(resn) + coffset
            elif cmd == FN_DIFF1:
                for i in range(nwrap, blocksize + nwrap):
                    cbuffer[i] = var_get(resn) + cbuffer[i - 1]
            elif cmd == FN_DIFF2:
                for i in range(nwrap, blocksize + nwrap):
                    cbuffer[i] = var_get(resn) + 2 * cbuffer[i - 1]
                    cbuffer[i] -= cbuffer[i - 2]
            elif cmd == FN_DIFF3:
                for i in range(nwrap, blocksize + nwrap):
                    cbuffer[i] = var_get(resn)
                    cbuffer[i] += 3 * (cbuffer[i - 1] - cbuffer[i - 2])
                    cbuffer[i] += cbuffer[i - 3]
            else:  # FN_QLPC
                nlpc = uvar_get(LPCQSIZE)
                for i in range(nlpc):
                    qlpc[i] = var_get(LPCQUANT)
                cbuffer[nwrap - nlpc : nwrap] -= coffset
                for i in range(nwrap, nwrap + blocksize):
                    sum = lpcqoffset
                    for j in range(nlpc):
                        sum += qlpc[j] * cbuffer[i - j - 1]
                    cbuffer[i] = var_get(resn) + (sum >> LPCQUANT)
                if coffset:
                    cbuffer[nwrap : blocksize + nwrap] += coffset
            if nmean > 0:
                sum = 0 if version < 2 else blocksize // 2
                sum += cbuffer[nwrap : nwrap + blocksize].sum()
                offset[chan, : nmean - 1] = offset[chan, 1:nmean]
                offset[chan, nmean - 1] = c99_div(sum, blocksize)
                if version >= 2:
                    offset[chan, nmean - 1] <<= bitshift

            cbuffer[:nwrap] = cbuffer[blocksize : blocksize + nwrap]  # wrap

            fix_bitshift(cbuffer[nwrap:], blocksize, bitshift, ftype)

            if chan == nchan - 1:
                nitem = blocksize * nchan
                # channel is last index of reshaped data, hence the transpose
                data[:nitem] = buffer[:, nwrap : blocksize + nwrap].T.flat
                if convert:
                    data[:nitem] = ULAW2PCM[data[:nitem]]
                data = data[nitem:]
                sampsdone += blocksize
            chan = (chan + 1) % nchan
        elif cmd == FN_BLOCKSIZE:
            blocksize = ulong_get()
        elif cmd == FN_BITSHIFT:
            bitshift = uvar_get(BITSHIFTSIZE)
        else:
            raise error
    return sampsdone


def copy_samples(file_, header, dtype, error):
    samptype, sampsize, sampcount, samprate, chancount, inporder = header
    buf_size = 16384
    sampsdone = 0
    convert = False
    if sampsize == 1:
        in_type = np.dtype(np.uint8)
    elif sampsize == 2:
        in_type = np.dtype(np.int16)
    elif sampsize == 4:
        in_type = np.dtype(np.int32)
    else:
        raise error
    if dtype is None:
        dtype = in_type
    else:
        dtype = np.dtype(dtype)
        if sampsize < dtype.itemsize and samptype in {"alaw", "ulaw"}:
            # we'll first convert to pcm
            convert = True
    data = np.empty(sampcount * chancount, dtype=dtype)
    in_type = in_type.newbyteorder(">" if (inporder == "10") else "<")
    while sampsdone < sampcount:
        inpbuf = file_.read(buf_size)
        nb = len(inpbuf)
        if not nb:
            break
        if not sampsdone and inpbuf[:4] == b"ajkg":
            sampsdone = copy_shortened_samples(inpbuf, file_, data, error)
            break
        ns = nb // (chancount * sampsize)
        if (sampsdone + ns) > sampcount:
            ns = sampcount - sampsdone
            nb = ns * chancount * sampsize
        inpbuf = np.frombuffer(inpbuf, dtype=in_type, count=ns * chancount)
        assert len(inpbuf) == ns * chancount
        if convert and samptype == "alaw":
            inpbuf = ALAW2PCM[inpbuf]
        elif convert:
            inpbuf = ULAW2PCM[inpbuf]
        data[sampsdone * chancount : (sampsdone + ns) * chancount] = inpbuf
        sampsdone += ns
    if sampsdone != sampcount:
        warnings.warn(
            "{} samples read, {} samples expected".format(sampsdone, sampcount)
        )
    if chancount > 1:
        data = data[: sampsdone * chancount].reshape((sampsdone, chancount), order="C")
    return data


def sphere_read_signal(rfilename, dtype, key):
    file_ = open(rfilename, "rb")
    try:
        header = read_header(
            file_, IOError("{} header could not be read as sphere".format(rfilename))
        )
        data = copy_samples(
            file_,
            header,
            dtype,
            IOError(
                "{} data could not be read as sphere despite good header"
                "".format(rfilename)
            ),
        )
    finally:
        file_.close()
    if dtype:
        data = data.astype(dtype)
    return data


MAGIC, TYPE_EOF, UNDEFINED_UINT, NBITPERLONG = b"ajkg", 9, -1, 32
DEFAULT_MAXRESNSTR, MINBITRATE, BUFSIZ = "32.0", 2.5, 1024
MAX_LPC_ORDER, POSITIVE_ULAW_ZERO, NEGATIVE_ULAW_ZERO = 64, 0xFF, 0x7F
TYPE_GENERIC_ULAW, MASKTABSIZE = 128, 33
DEFAULT_V0NMEAN = DEFAULT_MAXNLPC = DEFAULT_NSKIP = DEFAULT_NDISCARD = 0
DEFAULT_QUANTERROR = CHANSIZE = FN_DIFF0 = TYPE_AU1 = 0
MIN_SUPPORTED_VERSION = DEFAULT_NCHAN = FN_DIFF1 = TYPE_S8 = NSKIPSIZE = 1
FORMAT_VERSION = MAX_SUPPORTED_VERSION = BITSHIFTSIZE = FNSIZE = 2
FN_DIFF2 = TYPE_U8 = ULONGSIZE = LPCQSIZE = 2
ENERGYSIZE = FN_DIFF3 = TYPE_S16HL = NWRAP = 3
DEFAULT_V2NMEAN = FN_QUIT = TYPESIZE = TYPE_U16HL = 4
FN_BLOCKSIZE = TYPE_S16LH = LPCQUANT = 5
FN_BITSHIFT = TYPE_U16LH = 6
MAX_VERSION = XBITESIZE = FN_QLPC = TYPE_ULAW = 7
FN_ZERO = TYPE_AU2 = 8
DEFAULT_BLOCK_SIZE = DEFAULT_MINSNR = 256
V2LPCQOFFSET = 1 << LPCQUANT


ULAW2PCM = np.array(
    [
        -32124,
        -31100,
        -30076,
        -29052,
        -28028,
        -27004,
        -25980,
        -24956,
        -23932,
        -22908,
        -21884,
        -20860,
        -19836,
        -18812,
        -17788,
        -16764,
        -15996,
        -15484,
        -14972,
        -14460,
        -13948,
        -13436,
        -12924,
        -12412,
        -11900,
        -11388,
        -10876,
        -10364,
        -9852,
        -9340,
        -8828,
        -8316,
        -7932,
        -7676,
        -7420,
        -7164,
        -6908,
        -6652,
        -6396,
        -6140,
        -5884,
        -5628,
        -5372,
        -5116,
        -4860,
        -4604,
        -4348,
        -4092,
        -3900,
        -3772,
        -3644,
        -3516,
        -3388,
        -3260,
        -3132,
        -3004,
        -2876,
        -2748,
        -2620,
        -2492,
        -2364,
        -2236,
        -2108,
        -1980,
        -1884,
        -1820,
        -1756,
        -1692,
        -1628,
        -1564,
        -1500,
        -1436,
        -1372,
        -1308,
        -1244,
        -1180,
        -1116,
        -1052,
        -988,
        -924,
        -876,
        -844,
        -812,
        -780,
        -748,
        -716,
        -684,
        -652,
        -620,
        -588,
        -556,
        -524,
        -492,
        -460,
        -428,
        -396,
        -372,
        -356,
        -340,
        -324,
        -308,
        -292,
        -276,
        -260,
        -244,
        -228,
        -212,
        -196,
        -180,
        -164,
        -148,
        -132,
        -120,
        -112,
        -104,
        -96,
        -88,
        -80,
        -72,
        -64,
        -56,
        -48,
        -40,
        -32,
        -24,
        -16,
        -8,
        0,
        32124,
        31100,
        30076,
        29052,
        28028,
        27004,
        25980,
        24956,
        23932,
        22908,
        21884,
        20860,
        19836,
        18812,
        17788,
        16764,
        15996,
        15484,
        14972,
        14460,
        13948,
        13436,
        12924,
        12412,
        11900,
        11388,
        10876,
        10364,
        9852,
        9340,
        8828,
        8316,
        7932,
        7676,
        7420,
        7164,
        6908,
        6652,
        6396,
        6140,
        5884,
        5628,
        5372,
        5116,
        4860,
        4604,
        4348,
        4092,
        3900,
        3772,
        3644,
        3516,
        3388,
        3260,
        3132,
        3004,
        2876,
        2748,
        2620,
        2492,
        2364,
        2236,
        2108,
        1980,
        1884,
        1820,
        1756,
        1692,
        1628,
        1564,
        1500,
        1436,
        1372,
        1308,
        1244,
        1180,
        1116,
        1052,
        988,
        924,
        876,
        844,
        812,
        780,
        748,
        716,
        684,
        652,
        620,
        588,
        556,
        524,
        492,
        460,
        428,
        396,
        372,
        356,
        340,
        324,
        308,
        292,
        276,
        260,
        244,
        228,
        212,
        196,
        180,
        164,
        148,
        132,
        120,
        112,
        104,
        96,
        88,
        80,
        72,
        64,
        56,
        48,
        40,
        32,
        24,
        16,
        8,
        0,
    ],
    dtype=np.int16,
)


ALAW2PCM = np.array(
    [
        -5504,
        -5248,
        -6016,
        -5760,
        -4480,
        -4224,
        -4992,
        -4736,
        -7552,
        -7296,
        -8064,
        -7808,
        -6528,
        -6272,
        -7040,
        -6784,
        -2752,
        -2624,
        -3008,
        -2880,
        -2240,
        -2112,
        -2496,
        -2368,
        -3776,
        -3648,
        -4032,
        -3904,
        -3264,
        -3136,
        -3520,
        -3392,
        -22016,
        -20992,
        -24064,
        -23040,
        -17920,
        -16896,
        -19968,
        -18944,
        -30208,
        -29184,
        -32256,
        -31232,
        -26112,
        -25088,
        -28160,
        -27136,
        -11008,
        -10496,
        -12032,
        -11520,
        -8960,
        -8448,
        -9984,
        -9472,
        -15104,
        -14592,
        -16128,
        -15616,
        -13056,
        -12544,
        -14080,
        -13568,
        -344,
        -328,
        -376,
        -360,
        -280,
        -264,
        -312,
        -296,
        -472,
        -456,
        -504,
        -488,
        -408,
        -392,
        -440,
        -424,
        -88,
        -72,
        -120,
        -104,
        -24,
        -8,
        -56,
        -40,
        -216,
        -200,
        -248,
        -232,
        -152,
        -136,
        -184,
        -168,
        -1376,
        -1312,
        -1504,
        -1440,
        -1120,
        -1056,
        -1248,
        -1184,
        -1888,
        -1824,
        -2016,
        -1952,
        -1632,
        -1568,
        -1760,
        -1696,
        -688,
        -656,
        -752,
        -720,
        -560,
        -528,
        -624,
        -592,
        -944,
        -912,
        -1008,
        -976,
        -816,
        -784,
        -880,
        -848,
        5504,
        5248,
        6016,
        5760,
        4480,
        4224,
        4992,
        4736,
        7552,
        7296,
        8064,
        7808,
        6528,
        6272,
        7040,
        6784,
        2752,
        2624,
        3008,
        2880,
        2240,
        2112,
        2496,
        2368,
        3776,
        3648,
        4032,
        3904,
        3264,
        3136,
        3520,
        3392,
        22016,
        20992,
        24064,
        23040,
        17920,
        16896,
        19968,
        18944,
        30208,
        29184,
        32256,
        31232,
        26112,
        25088,
        28160,
        27136,
        11008,
        10496,
        12032,
        11520,
        8960,
        8448,
        9984,
        9472,
        15104,
        14592,
        16128,
        15616,
        13056,
        12544,
        14080,
        13568,
        344,
        328,
        376,
        360,
        280,
        264,
        312,
        296,
        472,
        456,
        504,
        488,
        408,
        392,
        440,
        424,
        88,
        72,
        120,
        104,
        24,
        8,
        56,
        40,
        216,
        200,
        248,
        232,
        152,
        136,
        184,
        168,
        1376,
        1312,
        1504,
        1440,
        1120,
        1056,
        1248,
        1184,
        1888,
        1824,
        2016,
        1952,
        1632,
        1568,
        1760,
        1696,
        688,
        656,
        752,
        720,
        560,
        528,
        624,
        592,
        944,
        912,
        1008,
        976,
        816,
        784,
        880,
        848,
    ],
    dtype=np.int16,
)


_ULAW_MAXSHIFT = np.array(
    [
        12,
        8,
        7,
        9,
        7,
        8,
        7,
        10,
        7,
        8,
        7,
        9,
        7,
        8,
        7,
        11,
        6,
        7,
        6,
        8,
        6,
        7,
        6,
        9,
        6,
        7,
        6,
        8,
        6,
        7,
        6,
        10,
        5,
        6,
        5,
        7,
        5,
        6,
        5,
        8,
        5,
        6,
        5,
        7,
        5,
        6,
        5,
        9,
        5,
        4,
        6,
        4,
        5,
        4,
        7,
        4,
        5,
        4,
        6,
        4,
        5,
        4,
        8,
        4,
        3,
        5,
        3,
        4,
        3,
        6,
        3,
        4,
        3,
        5,
        3,
        4,
        3,
        7,
        3,
        4,
        2,
        3,
        2,
        5,
        2,
        3,
        2,
        4,
        2,
        3,
        2,
        6,
        2,
        3,
        2,
        4,
        1,
        2,
        1,
        3,
        1,
        2,
        1,
        5,
        1,
        2,
        1,
        3,
        1,
        2,
        1,
        4,
        0,
        1,
        0,
        2,
        0,
        1,
        0,
        3,
        0,
        1,
        0,
        2,
        0,
        1,
        0,
        0,
        7,
        8,
        7,
        9,
        7,
        8,
        7,
        10,
        7,
        8,
        7,
        9,
        7,
        8,
        7,
        11,
        6,
        7,
        6,
        8,
        6,
        7,
        6,
        9,
        6,
        7,
        6,
        8,
        6,
        7,
        6,
        10,
        5,
        6,
        5,
        7,
        5,
        6,
        5,
        8,
        5,
        6,
        5,
        7,
        5,
        6,
        5,
        9,
        5,
        4,
        6,
        4,
        5,
        4,
        7,
        4,
        5,
        4,
        6,
        4,
        5,
        4,
        8,
        4,
        3,
        5,
        3,
        4,
        3,
        6,
        3,
        4,
        3,
        5,
        3,
        4,
        3,
        7,
        3,
        4,
        2,
        3,
        2,
        5,
        2,
        3,
        2,
        4,
        2,
        3,
        2,
        6,
        2,
        3,
        2,
        4,
        1,
        2,
        1,
        3,
        1,
        2,
        1,
        5,
        1,
        2,
        1,
        3,
        1,
        2,
        1,
        4,
        0,
        1,
        0,
        2,
        0,
        1,
        0,
        3,
        0,
        1,
        0,
        2,
        0,
        1,
        0,
        12,
    ],
    dtype=np.uint8,
)


ULAW_INWARD = np.array(
    [
        [
            -127,
            -126,
            -125,
            -124,
            -123,
            -122,
            -121,
            -120,
            -119,
            -118,
            -117,
            -116,
            115,
            114,
            -113,
            -112,
            -111,
            -110,
            -109,
            -108,
            -107,
            -106,
            -105,
            -104,
            103,
            -102,
            101,
            -100,
            -99,
            -98,
            -97,
            -96,
            -95,
            -94,
            -93,
            -92,
            -91,
            -90,
            89,
            -88,
            -87,
            86,
            -85,
            -84,
            -83,
            -82,
            -81,
            -80,
            -79,
            -78,
            -77,
            -76,
            -75,
            74,
            -73,
            -72,
            71,
            -70,
            -69,
            -68,
            -67,
            -66,
            -65,
            -64,
            -63,
            -62,
            -61,
            -60,
            59,
            -58,
            -57,
            56,
            -55,
            -54,
            -53,
            -52,
            -51,
            -50,
            -49,
            -48,
            -47,
            -46,
            -45,
            44,
            -43,
            -42,
            41,
            -40,
            -39,
            -38,
            -37,
            -36,
            -35,
            -34,
            -33,
            -32,
            -31,
            -30,
            29,
            -28,
            -27,
            26,
            -25,
            -24,
            -23,
            -22,
            -21,
            -20,
            -19,
            -18,
            -17,
            -16,
            -15,
            14,
            -13,
            -12,
            11,
            -10,
            -9,
            -8,
            -7,
            -6,
            -5,
            -4,
            -3,
            -2,
            -1,
            -128,
            127,
            126,
            125,
            124,
            123,
            122,
            121,
            120,
            119,
            118,
            117,
            116,
            115,
            114,
            113,
            112,
            111,
            110,
            109,
            108,
            107,
            106,
            105,
            104,
            103,
            102,
            101,
            100,
            99,
            98,
            97,
            96,
            95,
            94,
            93,
            92,
            91,
            90,
            89,
            88,
            87,
            86,
            85,
            84,
            83,
            82,
            81,
            80,
            79,
            78,
            77,
            76,
            75,
            74,
            73,
            72,
            71,
            70,
            69,
            68,
            67,
            66,
            65,
            64,
            63,
            62,
            61,
            60,
            59,
            58,
            57,
            56,
            55,
            54,
            53,
            52,
            51,
            50,
            49,
            48,
            47,
            46,
            45,
            44,
            43,
            42,
            41,
            40,
            39,
            38,
            37,
            36,
            35,
            34,
            33,
            32,
            31,
            30,
            29,
            28,
            27,
            26,
            25,
            24,
            23,
            22,
            21,
            20,
            19,
            18,
            17,
            16,
            15,
            14,
            13,
            12,
            11,
            10,
            9,
            8,
            7,
            6,
            5,
            4,
            3,
            2,
            1,
            0,
        ],
        [
            -119,
            -118,
            -117,
            -116,
            -115,
            -114,
            -113,
            -112,
            -111,
            -110,
            -109,
            -108,
            -107,
            -106,
            -105,
            -104,
            -103,
            -102,
            -101,
            -100,
            -99,
            -98,
            -97,
            -96,
            -95,
            -94,
            -93,
            -92,
            -91,
            -90,
            -89,
            -88,
            -87,
            -86,
            -85,
            -84,
            -83,
            -82,
            -81,
            -80,
            -79,
            -78,
            -77,
            -76,
            -75,
            -74,
            -73,
            -72,
            -71,
            -70,
            -69,
            -68,
            -67,
            -66,
            -65,
            -64,
            -63,
            -62,
            -61,
            -60,
            -59,
            -58,
            -57,
            -56,
            -55,
            -54,
            -53,
            -52,
            -51,
            -50,
            -49,
            -48,
            -47,
            -46,
            -45,
            -44,
            -43,
            -42,
            -41,
            -40,
            -39,
            -38,
            -37,
            -36,
            -35,
            -34,
            -33,
            -32,
            -31,
            -30,
            -29,
            -28,
            -27,
            -26,
            -25,
            -24,
            -23,
            -22,
            -21,
            -20,
            -19,
            -18,
            -17,
            -16,
            -15,
            -14,
            -13,
            -12,
            -11,
            -10,
            -9,
            -8,
            -128,
            -7,
            -127,
            -6,
            -126,
            -5,
            -125,
            -4,
            -124,
            -3,
            -123,
            -2,
            -122,
            -1,
            -121,
            -120,
            119,
            118,
            117,
            116,
            115,
            114,
            113,
            112,
            111,
            110,
            109,
            108,
            107,
            106,
            105,
            104,
            103,
            102,
            101,
            100,
            99,
            98,
            97,
            96,
            95,
            94,
            93,
            92,
            91,
            90,
            89,
            88,
            87,
            86,
            85,
            84,
            83,
            82,
            81,
            80,
            79,
            78,
            77,
            76,
            75,
            74,
            73,
            72,
            71,
            70,
            69,
            68,
            67,
            66,
            65,
            64,
            63,
            62,
            61,
            60,
            59,
            58,
            57,
            56,
            55,
            54,
            53,
            52,
            51,
            50,
            49,
            48,
            47,
            46,
            45,
            44,
            43,
            42,
            41,
            40,
            39,
            38,
            37,
            36,
            35,
            34,
            33,
            32,
            31,
            30,
            29,
            28,
            27,
            26,
            25,
            24,
            23,
            22,
            21,
            20,
            19,
            18,
            17,
            16,
            15,
            14,
            13,
            12,
            11,
            10,
            9,
            8,
            127,
            7,
            126,
            6,
            125,
            5,
            124,
            4,
            123,
            3,
            122,
            2,
            121,
            1,
            120,
            0,
        ],
        [
            -107,
            -106,
            -105,
            -104,
            -103,
            -102,
            -101,
            -100,
            -99,
            -98,
            -97,
            -96,
            -95,
            -94,
            -93,
            -92,
            -91,
            -90,
            -89,
            -88,
            -87,
            -86,
            -85,
            -84,
            -83,
            -82,
            -81,
            -80,
            -79,
            -78,
            -77,
            -76,
            -75,
            -74,
            -73,
            -72,
            -71,
            -70,
            -69,
            -68,
            -67,
            -66,
            -65,
            -64,
            -63,
            -62,
            -61,
            -60,
            -59,
            -58,
            -57,
            -56,
            -55,
            -54,
            -53,
            -52,
            -51,
            -50,
            -49,
            -48,
            -47,
            -46,
            -45,
            -44,
            -43,
            -42,
            -41,
            -40,
            -39,
            -38,
            -37,
            -36,
            -35,
            -34,
            -33,
            -32,
            -31,
            -30,
            -29,
            -28,
            -27,
            -26,
            -25,
            -24,
            -23,
            -22,
            -21,
            -20,
            -19,
            -18,
            -17,
            -16,
            -15,
            -14,
            -13,
            -12,
            -128,
            -11,
            -127,
            -10,
            -126,
            -9,
            -125,
            -8,
            -124,
            -7,
            -123,
            -6,
            -122,
            -5,
            -121,
            -4,
            -120,
            -119,
            -118,
            -3,
            -117,
            -116,
            -115,
            -2,
            -114,
            -113,
            -112,
            -1,
            -111,
            -110,
            -109,
            -108,
            107,
            106,
            105,
            104,
            103,
            102,
            101,
            100,
            99,
            98,
            97,
            96,
            95,
            94,
            93,
            92,
            91,
            90,
            89,
            88,
            87,
            86,
            85,
            84,
            83,
            82,
            81,
            80,
            79,
            78,
            77,
            76,
            75,
            74,
            73,
            72,
            71,
            70,
            69,
            68,
            67,
            66,
            65,
            64,
            63,
            62,
            61,
            60,
            59,
            58,
            57,
            56,
            55,
            54,
            53,
            52,
            51,
            50,
            49,
            48,
            47,
            46,
            45,
            44,
            43,
            42,
            41,
            40,
            39,
            38,
            37,
            36,
            35,
            34,
            33,
            32,
            31,
            30,
            29,
            28,
            27,
            26,
            25,
            24,
            23,
            22,
            21,
            20,
            19,
            18,
            17,
            16,
            15,
            14,
            13,
            12,
            127,
            11,
            126,
            10,
            125,
            9,
            124,
            8,
            123,
            7,
            122,
            6,
            121,
            5,
            120,
            4,
            119,
            118,
            117,
            3,
            116,
            115,
            114,
            2,
            113,
            112,
            111,
            1,
            110,
            109,
            108,
            0,
        ],
        [
            -93,
            -92,
            -91,
            -90,
            -89,
            -88,
            -87,
            -86,
            -85,
            -84,
            -83,
            -82,
            -81,
            -80,
            -79,
            -78,
            -77,
            -76,
            -75,
            -74,
            -73,
            -72,
            -71,
            -70,
            -69,
            -68,
            -67,
            -66,
            -65,
            -64,
            -63,
            -62,
            -61,
            -60,
            -59,
            -58,
            -57,
            -56,
            -55,
            -54,
            -53,
            -52,
            -51,
            -50,
            -49,
            -48,
            -47,
            -46,
            -45,
            -44,
            -43,
            -42,
            -41,
            -40,
            -39,
            -38,
            -37,
            -36,
            -35,
            -34,
            -33,
            -32,
            -31,
            -30,
            -29,
            -28,
            -27,
            -26,
            -25,
            -24,
            -23,
            -22,
            -21,
            -20,
            -19,
            -18,
            -17,
            -16,
            -15,
            -14,
            -128,
            -13,
            -127,
            -12,
            -126,
            -11,
            -125,
            -10,
            -124,
            -9,
            -123,
            -8,
            -122,
            -7,
            -121,
            -6,
            -120,
            -119,
            -118,
            -5,
            -117,
            -116,
            -115,
            -4,
            -114,
            -113,
            -112,
            -3,
            -111,
            -110,
            -109,
            -2,
            -108,
            -107,
            -106,
            -105,
            -104,
            -103,
            -102,
            -1,
            -101,
            -100,
            -99,
            -98,
            -97,
            -96,
            -95,
            -94,
            93,
            92,
            91,
            90,
            89,
            88,
            87,
            86,
            85,
            84,
            83,
            82,
            81,
            80,
            79,
            78,
            77,
            76,
            75,
            74,
            73,
            72,
            71,
            70,
            69,
            68,
            67,
            66,
            65,
            64,
            63,
            62,
            61,
            60,
            59,
            58,
            57,
            56,
            55,
            54,
            53,
            52,
            51,
            50,
            49,
            48,
            47,
            46,
            45,
            44,
            43,
            42,
            41,
            40,
            39,
            38,
            37,
            36,
            35,
            34,
            33,
            32,
            31,
            30,
            29,
            28,
            27,
            26,
            25,
            24,
            23,
            22,
            21,
            20,
            19,
            18,
            17,
            16,
            15,
            14,
            127,
            13,
            126,
            12,
            125,
            11,
            124,
            10,
            123,
            9,
            122,
            8,
            121,
            7,
            120,
            6,
            119,
            118,
            117,
            5,
            116,
            115,
            114,
            4,
            113,
            112,
            111,
            3,
            110,
            109,
            108,
            2,
            107,
            106,
            105,
            104,
            103,
            102,
            101,
            1,
            100,
            99,
            98,
            97,
            96,
            95,
            94,
            0,
        ],
        [
            -78,
            -77,
            -76,
            -75,
            -74,
            -73,
            -72,
            -71,
            -70,
            -69,
            -68,
            -67,
            -66,
            -65,
            -64,
            -63,
            -62,
            -61,
            -60,
            -59,
            -58,
            -57,
            -56,
            -55,
            -54,
            -53,
            -52,
            -51,
            -50,
            -49,
            -48,
            -47,
            -46,
            -45,
            -44,
            -43,
            -42,
            -41,
            -40,
            -39,
            -38,
            -37,
            -36,
            -35,
            -34,
            -33,
            -32,
            -31,
            -30,
            -29,
            -28,
            -27,
            -26,
            -25,
            -24,
            -23,
            -22,
            -21,
            -20,
            -19,
            -18,
            -17,
            -16,
            -15,
            -128,
            -14,
            -127,
            -13,
            -126,
            -12,
            -125,
            -11,
            -124,
            -10,
            -123,
            -9,
            -122,
            -8,
            -121,
            -7,
            -120,
            -119,
            -118,
            -6,
            -117,
            -116,
            -115,
            -5,
            -114,
            -113,
            -112,
            -4,
            -111,
            -110,
            -109,
            -3,
            -108,
            -107,
            -106,
            -105,
            -104,
            -103,
            -102,
            -2,
            -101,
            -100,
            -99,
            -98,
            -97,
            -96,
            -95,
            -1,
            -94,
            -93,
            -92,
            -91,
            -90,
            -89,
            -88,
            -87,
            -86,
            -85,
            -84,
            -83,
            -82,
            -81,
            -80,
            -79,
            78,
            77,
            76,
            75,
            74,
            73,
            72,
            71,
            70,
            69,
            68,
            67,
            66,
            65,
            64,
            63,
            62,
            61,
            60,
            59,
            58,
            57,
            56,
            55,
            54,
            53,
            52,
            51,
            50,
            49,
            48,
            47,
            46,
            45,
            44,
            43,
            42,
            41,
            40,
            39,
            38,
            37,
            36,
            35,
            34,
            33,
            32,
            31,
            30,
            29,
            28,
            27,
            26,
            25,
            24,
            23,
            22,
            21,
            20,
            19,
            18,
            17,
            16,
            15,
            127,
            14,
            126,
            13,
            125,
            12,
            124,
            11,
            123,
            10,
            122,
            9,
            121,
            8,
            120,
            7,
            119,
            118,
            117,
            6,
            116,
            115,
            114,
            5,
            113,
            112,
            111,
            4,
            110,
            109,
            108,
            3,
            107,
            106,
            105,
            104,
            103,
            102,
            101,
            2,
            100,
            99,
            98,
            97,
            96,
            95,
            94,
            1,
            93,
            92,
            91,
            90,
            89,
            88,
            87,
            86,
            85,
            84,
            83,
            82,
            81,
            80,
            79,
            0,
        ],
        [
            -63,
            -62,
            -61,
            -60,
            -59,
            -58,
            -57,
            -56,
            -55,
            -54,
            -53,
            -52,
            -51,
            -50,
            -49,
            -48,
            -47,
            -46,
            -45,
            -44,
            -43,
            -42,
            -41,
            -40,
            -39,
            -38,
            -37,
            -36,
            -35,
            -34,
            -33,
            -32,
            -31,
            -30,
            -29,
            -28,
            -27,
            -26,
            -25,
            -24,
            -23,
            -22,
            -21,
            -20,
            -19,
            -18,
            -17,
            -16,
            -15,
            -128,
            -14,
            -127,
            -13,
            -126,
            -12,
            -125,
            -11,
            -124,
            -10,
            -123,
            -9,
            -122,
            -8,
            -121,
            -120,
            -7,
            -119,
            -118,
            -117,
            -6,
            -116,
            -115,
            -114,
            -5,
            -113,
            -112,
            -111,
            -4,
            -110,
            -109,
            -108,
            -107,
            -106,
            -3,
            -105,
            -104,
            -103,
            -102,
            -101,
            -100,
            -99,
            -2,
            -98,
            -97,
            -96,
            -95,
            -94,
            -93,
            -92,
            -91,
            -90,
            -89,
            -88,
            -1,
            -87,
            -86,
            -85,
            -84,
            -83,
            -82,
            -81,
            -80,
            -79,
            -78,
            -77,
            -76,
            -75,
            -74,
            -73,
            -72,
            -71,
            -70,
            -69,
            -68,
            -67,
            -66,
            -65,
            -64,
            63,
            62,
            61,
            60,
            59,
            58,
            57,
            56,
            55,
            54,
            53,
            52,
            51,
            50,
            49,
            48,
            47,
            46,
            45,
            44,
            43,
            42,
            41,
            40,
            39,
            38,
            37,
            36,
            35,
            34,
            33,
            32,
            31,
            30,
            29,
            28,
            27,
            26,
            25,
            24,
            23,
            22,
            21,
            20,
            19,
            18,
            17,
            16,
            15,
            127,
            14,
            126,
            13,
            125,
            12,
            124,
            11,
            123,
            10,
            122,
            9,
            121,
            8,
            120,
            119,
            7,
            118,
            117,
            116,
            6,
            115,
            114,
            113,
            5,
            112,
            111,
            110,
            4,
            109,
            108,
            107,
            106,
            105,
            3,
            104,
            103,
            102,
            101,
            100,
            99,
            98,
            2,
            97,
            96,
            95,
            94,
            93,
            92,
            91,
            90,
            89,
            88,
            87,
            1,
            86,
            85,
            84,
            83,
            82,
            81,
            80,
            79,
            78,
            77,
            76,
            75,
            74,
            73,
            72,
            71,
            70,
            69,
            68,
            67,
            66,
            65,
            64,
            0,
        ],
        [
            -47,
            -46,
            -45,
            -44,
            -43,
            -42,
            -41,
            -40,
            -39,
            -38,
            -37,
            -36,
            -35,
            -34,
            -33,
            -32,
            -31,
            -30,
            -29,
            -28,
            -27,
            -26,
            -25,
            -24,
            -23,
            -22,
            -21,
            -20,
            -19,
            -18,
            -17,
            -16,
            -128,
            -15,
            -127,
            -14,
            -126,
            -13,
            -125,
            -12,
            -124,
            -11,
            -123,
            -10,
            -122,
            -9,
            -121,
            -8,
            -120,
            -119,
            -7,
            -118,
            -117,
            -116,
            -6,
            -115,
            -114,
            -113,
            -5,
            -112,
            -111,
            -110,
            -4,
            -109,
            -108,
            -107,
            -106,
            -105,
            -104,
            -3,
            -103,
            -102,
            -101,
            -100,
            -99,
            -98,
            -97,
            -2,
            -96,
            -95,
            -94,
            -93,
            -92,
            -91,
            -90,
            -89,
            -88,
            -87,
            -86,
            -85,
            -84,
            -1,
            -83,
            -82,
            -81,
            -80,
            -79,
            -78,
            -77,
            -76,
            -75,
            -74,
            -73,
            -72,
            -71,
            -70,
            -69,
            -68,
            -67,
            -66,
            -65,
            -64,
            -63,
            -62,
            -61,
            -60,
            -59,
            -58,
            -57,
            -56,
            -55,
            -54,
            -53,
            -52,
            -51,
            -50,
            -49,
            -48,
            47,
            46,
            45,
            44,
            43,
            42,
            41,
            40,
            39,
            38,
            37,
            36,
            35,
            34,
            33,
            32,
            31,
            30,
            29,
            28,
            27,
            26,
            25,
            24,
            23,
            22,
            21,
            20,
            19,
            18,
            17,
            16,
            127,
            15,
            126,
            14,
            125,
            13,
            124,
            12,
            123,
            11,
            122,
            10,
            121,
            9,
            120,
            8,
            119,
            118,
            7,
            117,
            116,
            115,
            6,
            114,
            113,
            112,
            5,
            111,
            110,
            109,
            4,
            108,
            107,
            106,
            105,
            104,
            103,
            3,
            102,
            101,
            100,
            99,
            98,
            97,
            96,
            2,
            95,
            94,
            93,
            92,
            91,
            90,
            89,
            88,
            87,
            86,
            85,
            84,
            83,
            1,
            82,
            81,
            80,
            79,
            78,
            77,
            76,
            75,
            74,
            73,
            72,
            71,
            70,
            69,
            68,
            67,
            66,
            65,
            64,
            63,
            62,
            61,
            60,
            59,
            58,
            57,
            56,
            55,
            54,
            53,
            52,
            51,
            50,
            49,
            48,
            0,
        ],
        [
            -31,
            -30,
            -29,
            -28,
            -27,
            -26,
            -25,
            -24,
            -23,
            -22,
            -21,
            -20,
            -19,
            -18,
            -17,
            -16,
            -128,
            -15,
            -127,
            -14,
            -126,
            -13,
            -125,
            -12,
            -124,
            -11,
            -123,
            -10,
            -122,
            -9,
            -121,
            -8,
            -120,
            -119,
            -118,
            -7,
            -117,
            -116,
            -115,
            -6,
            -114,
            -113,
            -112,
            -5,
            -111,
            -110,
            -109,
            -4,
            -108,
            -107,
            -106,
            -105,
            -104,
            -103,
            -3,
            -102,
            -101,
            -100,
            -99,
            -98,
            -97,
            -96,
            -2,
            -95,
            -94,
            -93,
            -92,
            -91,
            -90,
            -89,
            -88,
            -87,
            -86,
            -85,
            -84,
            -83,
            -82,
            -1,
            -81,
            -80,
            -79,
            -78,
            -77,
            -76,
            -75,
            -74,
            -73,
            -72,
            -71,
            -70,
            -69,
            -68,
            -67,
            -66,
            -65,
            -64,
            -63,
            -62,
            -61,
            -60,
            -59,
            -58,
            -57,
            -56,
            -55,
            -54,
            -53,
            -52,
            -51,
            -50,
            -49,
            -48,
            -47,
            -46,
            -45,
            -44,
            -43,
            -42,
            -41,
            -40,
            -39,
            -38,
            -37,
            -36,
            -35,
            -34,
            -33,
            -32,
            31,
            30,
            29,
            28,
            27,
            26,
            25,
            24,
            23,
            22,
            21,
            20,
            19,
            18,
            17,
            16,
            127,
            15,
            126,
            14,
            125,
            13,
            124,
            12,
            123,
            11,
            122,
            10,
            121,
            9,
            120,
            8,
            119,
            118,
            117,
            7,
            116,
            115,
            114,
            6,
            113,
            112,
            111,
            5,
            110,
            109,
            108,
            4,
            107,
            106,
            105,
            104,
            103,
            102,
            3,
            101,
            100,
            99,
            98,
            97,
            96,
            95,
            2,
            94,
            93,
            92,
            91,
            90,
            89,
            88,
            87,
            86,
            85,
            84,
            83,
            82,
            81,
            1,
            80,
            79,
            78,
            77,
            76,
            75,
            74,
            73,
            72,
            71,
            70,
            69,
            68,
            67,
            66,
            65,
            64,
            63,
            62,
            61,
            60,
            59,
            58,
            57,
            56,
            55,
            54,
            53,
            52,
            51,
            50,
            49,
            48,
            47,
            46,
            45,
            44,
            43,
            42,
            41,
            40,
            39,
            38,
            37,
            36,
            35,
            34,
            33,
            32,
            0,
        ],
        [
            -16,
            -15,
            -128,
            -14,
            -127,
            -13,
            -126,
            -12,
            -125,
            -11,
            -124,
            -10,
            -123,
            -9,
            -122,
            -8,
            -121,
            -120,
            -119,
            -7,
            -118,
            -117,
            -116,
            -6,
            -115,
            -114,
            -113,
            -5,
            -112,
            -111,
            -110,
            -4,
            -109,
            -108,
            -107,
            -106,
            -105,
            -104,
            -103,
            -3,
            -102,
            -101,
            -100,
            -99,
            -98,
            -97,
            -96,
            -2,
            -95,
            -94,
            -93,
            -92,
            -91,
            -90,
            -89,
            -88,
            -87,
            -86,
            -85,
            -84,
            -83,
            -82,
            -1,
            -81,
            -80,
            -79,
            -78,
            -77,
            -76,
            -75,
            -74,
            -73,
            -72,
            -71,
            -70,
            -69,
            -68,
            -67,
            -66,
            -65,
            -64,
            -63,
            -62,
            -61,
            -60,
            -59,
            -58,
            -57,
            -56,
            -55,
            -54,
            -53,
            -52,
            -51,
            -50,
            -49,
            -48,
            -47,
            -46,
            -45,
            -44,
            -43,
            -42,
            -41,
            -40,
            -39,
            -38,
            -37,
            -36,
            -35,
            -34,
            -33,
            -32,
            -31,
            -30,
            -29,
            -28,
            -27,
            -26,
            -25,
            -24,
            -23,
            -22,
            -21,
            -20,
            -19,
            -18,
            -17,
            127,
            15,
            126,
            14,
            125,
            13,
            124,
            12,
            123,
            11,
            122,
            10,
            121,
            9,
            120,
            8,
            119,
            118,
            117,
            7,
            116,
            115,
            114,
            6,
            113,
            112,
            111,
            5,
            110,
            109,
            108,
            4,
            107,
            106,
            105,
            104,
            103,
            102,
            101,
            3,
            100,
            99,
            98,
            97,
            96,
            95,
            94,
            2,
            93,
            92,
            91,
            90,
            89,
            88,
            87,
            86,
            85,
            84,
            83,
            82,
            81,
            80,
            1,
            79,
            78,
            77,
            76,
            75,
            74,
            73,
            72,
            71,
            70,
            69,
            68,
            67,
            66,
            65,
            64,
            63,
            62,
            61,
            60,
            59,
            58,
            57,
            56,
            55,
            54,
            53,
            52,
            51,
            50,
            49,
            48,
            47,
            46,
            45,
            44,
            43,
            42,
            41,
            40,
            39,
            38,
            37,
            36,
            35,
            34,
            33,
            32,
            31,
            30,
            29,
            28,
            27,
            26,
            25,
            24,
            23,
            22,
            21,
            20,
            19,
            18,
            17,
            16,
            0,
        ],
        [
            -8,
            -128,
            -127,
            -7,
            -126,
            -125,
            -124,
            -6,
            -123,
            -122,
            -121,
            -5,
            -120,
            -119,
            -118,
            -4,
            -117,
            -116,
            -115,
            -114,
            -113,
            -112,
            -111,
            -3,
            -110,
            -109,
            -108,
            -107,
            -106,
            -105,
            -104,
            -2,
            -103,
            -102,
            -101,
            -100,
            -99,
            -98,
            -97,
            -96,
            -95,
            -94,
            -93,
            -92,
            -91,
            -90,
            -89,
            -1,
            -88,
            -87,
            -86,
            -85,
            -84,
            -83,
            -82,
            -81,
            -80,
            -79,
            -78,
            -77,
            -76,
            -75,
            -74,
            -73,
            -72,
            -71,
            -70,
            -69,
            -68,
            -67,
            -66,
            -65,
            -64,
            -63,
            -62,
            -61,
            -60,
            -59,
            -58,
            -57,
            -56,
            -55,
            -54,
            -53,
            -52,
            -51,
            -50,
            -49,
            -48,
            -47,
            -46,
            -45,
            -44,
            -43,
            -42,
            -41,
            -40,
            -39,
            -38,
            -37,
            -36,
            -35,
            -34,
            -33,
            -32,
            -31,
            -30,
            -29,
            -28,
            -27,
            -26,
            -25,
            -24,
            -23,
            -22,
            -21,
            -20,
            -19,
            -18,
            -17,
            -16,
            -15,
            -14,
            -13,
            -12,
            -11,
            -10,
            -9,
            127,
            126,
            125,
            7,
            124,
            123,
            122,
            6,
            121,
            120,
            119,
            5,
            118,
            117,
            116,
            4,
            115,
            114,
            113,
            112,
            111,
            110,
            109,
            3,
            108,
            107,
            106,
            105,
            104,
            103,
            102,
            2,
            101,
            100,
            99,
            98,
            97,
            96,
            95,
            94,
            93,
            92,
            91,
            90,
            89,
            88,
            87,
            1,
            86,
            85,
            84,
            83,
            82,
            81,
            80,
            79,
            78,
            77,
            76,
            75,
            74,
            73,
            72,
            71,
            70,
            69,
            68,
            67,
            66,
            65,
            64,
            63,
            62,
            61,
            60,
            59,
            58,
            57,
            56,
            55,
            54,
            53,
            52,
            51,
            50,
            49,
            48,
            47,
            46,
            45,
            44,
            43,
            42,
            41,
            40,
            39,
            38,
            37,
            36,
            35,
            34,
            33,
            32,
            31,
            30,
            29,
            28,
            27,
            26,
            25,
            24,
            23,
            22,
            21,
            20,
            19,
            18,
            17,
            16,
            15,
            14,
            13,
            12,
            11,
            10,
            9,
            8,
            0,
        ],
        [
            -4,
            -128,
            -127,
            -126,
            -125,
            -124,
            -123,
            -3,
            -122,
            -121,
            -120,
            -119,
            -118,
            -117,
            -116,
            -2,
            -115,
            -114,
            -113,
            -112,
            -111,
            -110,
            -109,
            -108,
            -107,
            -106,
            -105,
            -104,
            -103,
            -102,
            -101,
            -1,
            -100,
            -99,
            -98,
            -97,
            -96,
            -95,
            -94,
            -93,
            -92,
            -91,
            -90,
            -89,
            -88,
            -87,
            -86,
            -85,
            -84,
            -83,
            -82,
            -81,
            -80,
            -79,
            -78,
            -77,
            -76,
            -75,
            -74,
            -73,
            -72,
            -71,
            -70,
            -69,
            -68,
            -67,
            -66,
            -65,
            -64,
            -63,
            -62,
            -61,
            -60,
            -59,
            -58,
            -57,
            -56,
            -55,
            -54,
            -53,
            -52,
            -51,
            -50,
            -49,
            -48,
            -47,
            -46,
            -45,
            -44,
            -43,
            -42,
            -41,
            -40,
            -39,
            -38,
            -37,
            -36,
            -35,
            -34,
            -33,
            -32,
            -31,
            -30,
            -29,
            -28,
            -27,
            -26,
            -25,
            -24,
            -23,
            -22,
            -21,
            -20,
            -19,
            -18,
            -17,
            -16,
            -15,
            -14,
            -13,
            -12,
            -11,
            -10,
            -9,
            -8,
            -7,
            -6,
            -5,
            127,
            126,
            125,
            124,
            123,
            122,
            121,
            3,
            120,
            119,
            118,
            117,
            116,
            115,
            114,
            2,
            113,
            112,
            111,
            110,
            109,
            108,
            107,
            106,
            105,
            104,
            103,
            102,
            101,
            100,
            99,
            1,
            98,
            97,
            96,
            95,
            94,
            93,
            92,
            91,
            90,
            89,
            88,
            87,
            86,
            85,
            84,
            83,
            82,
            81,
            80,
            79,
            78,
            77,
            76,
            75,
            74,
            73,
            72,
            71,
            70,
            69,
            68,
            67,
            66,
            65,
            64,
            63,
            62,
            61,
            60,
            59,
            58,
            57,
            56,
            55,
            54,
            53,
            52,
            51,
            50,
            49,
            48,
            47,
            46,
            45,
            44,
            43,
            42,
            41,
            40,
            39,
            38,
            37,
            36,
            35,
            34,
            33,
            32,
            31,
            30,
            29,
            28,
            27,
            26,
            25,
            24,
            23,
            22,
            21,
            20,
            19,
            18,
            17,
            16,
            15,
            14,
            13,
            12,
            11,
            10,
            9,
            8,
            7,
            6,
            5,
            4,
            0,
        ],
        [
            -2,
            -128,
            -127,
            -126,
            -125,
            -124,
            -123,
            -122,
            -121,
            -120,
            -119,
            -118,
            -117,
            -116,
            -115,
            -1,
            -114,
            -113,
            -112,
            -111,
            -110,
            -109,
            -108,
            -107,
            -106,
            -105,
            -104,
            -103,
            -102,
            -101,
            -100,
            -99,
            -98,
            -97,
            -96,
            -95,
            -94,
            -93,
            -92,
            -91,
            -90,
            -89,
            -88,
            -87,
            -86,
            -85,
            -84,
            -83,
            -82,
            -81,
            -80,
            -79,
            -78,
            -77,
            -76,
            -75,
            -74,
            -73,
            -72,
            -71,
            -70,
            -69,
            -68,
            -67,
            -66,
            -65,
            -64,
            -63,
            -62,
            -61,
            -60,
            -59,
            -58,
            -57,
            -56,
            -55,
            -54,
            -53,
            -52,
            -51,
            -50,
            -49,
            -48,
            -47,
            -46,
            -45,
            -44,
            -43,
            -42,
            -41,
            -40,
            -39,
            -38,
            -37,
            -36,
            -35,
            -34,
            -33,
            -32,
            -31,
            -30,
            -29,
            -28,
            -27,
            -26,
            -25,
            -24,
            -23,
            -22,
            -21,
            -20,
            -19,
            -18,
            -17,
            -16,
            -15,
            -14,
            -13,
            -12,
            -11,
            -10,
            -9,
            -8,
            -7,
            -6,
            -5,
            -4,
            -3,
            127,
            126,
            125,
            124,
            123,
            122,
            121,
            120,
            119,
            118,
            117,
            116,
            115,
            114,
            113,
            1,
            112,
            111,
            110,
            109,
            108,
            107,
            106,
            105,
            104,
            103,
            102,
            101,
            100,
            99,
            98,
            97,
            96,
            95,
            94,
            93,
            92,
            91,
            90,
            89,
            88,
            87,
            86,
            85,
            84,
            83,
            82,
            81,
            80,
            79,
            78,
            77,
            76,
            75,
            74,
            73,
            72,
            71,
            70,
            69,
            68,
            67,
            66,
            65,
            64,
            63,
            62,
            61,
            60,
            59,
            58,
            57,
            56,
            55,
            54,
            53,
            52,
            51,
            50,
            49,
            48,
            47,
            46,
            45,
            44,
            43,
            42,
            41,
            40,
            39,
            38,
            37,
            36,
            35,
            34,
            33,
            32,
            31,
            30,
            29,
            28,
            27,
            26,
            25,
            24,
            23,
            22,
            21,
            20,
            19,
            18,
            17,
            16,
            15,
            14,
            13,
            12,
            11,
            10,
            9,
            8,
            7,
            6,
            5,
            4,
            3,
            2,
            0,
        ],
        [
            -1,
            -128,
            -127,
            -126,
            -125,
            -124,
            -123,
            -122,
            -121,
            -120,
            -119,
            -118,
            -117,
            -116,
            -115,
            -114,
            -113,
            -112,
            -111,
            -110,
            -109,
            -108,
            -107,
            -106,
            -105,
            -104,
            -103,
            -102,
            -101,
            -100,
            -99,
            -98,
            -97,
            -96,
            -95,
            -94,
            -93,
            -92,
            -91,
            -90,
            -89,
            -88,
            -87,
            -86,
            -85,
            -84,
            -83,
            -82,
            -81,
            -80,
            -79,
            -78,
            -77,
            -76,
            -75,
            -74,
            -73,
            -72,
            -71,
            -70,
            -69,
            -68,
            -67,
            -66,
            -65,
            -64,
            -63,
            -62,
            -61,
            -60,
            -59,
            -58,
            -57,
            -56,
            -55,
            -54,
            -53,
            -52,
            -51,
            -50,
            -49,
            -48,
            -47,
            -46,
            -45,
            -44,
            -43,
            -42,
            -41,
            -40,
            -39,
            -38,
            -37,
            -36,
            -35,
            -34,
            -33,
            -32,
            -31,
            -30,
            -29,
            -28,
            -27,
            -26,
            -25,
            -24,
            -23,
            -22,
            -21,
            -20,
            -19,
            -18,
            -17,
            -16,
            -15,
            -14,
            -13,
            -12,
            -11,
            -10,
            -9,
            -8,
            -7,
            -6,
            -5,
            -4,
            -3,
            -2,
            127,
            126,
            125,
            124,
            123,
            122,
            121,
            120,
            119,
            118,
            117,
            116,
            115,
            114,
            113,
            112,
            111,
            110,
            109,
            108,
            107,
            106,
            105,
            104,
            103,
            102,
            101,
            100,
            99,
            98,
            97,
            96,
            95,
            94,
            93,
            92,
            91,
            90,
            89,
            88,
            87,
            86,
            85,
            84,
            83,
            82,
            81,
            80,
            79,
            78,
            77,
            76,
            75,
            74,
            73,
            72,
            71,
            70,
            69,
            68,
            67,
            66,
            65,
            64,
            63,
            62,
            61,
            60,
            59,
            58,
            57,
            56,
            55,
            54,
            53,
            52,
            51,
            50,
            49,
            48,
            47,
            46,
            45,
            44,
            43,
            42,
            41,
            40,
            39,
            38,
            37,
            36,
            35,
            34,
            33,
            32,
            31,
            30,
            29,
            28,
            27,
            26,
            25,
            24,
            23,
            22,
            21,
            20,
            19,
            18,
            17,
            16,
            15,
            14,
            13,
            12,
            11,
            10,
            9,
            8,
            7,
            6,
            5,
            4,
            3,
            2,
            1,
            0,
        ],
    ],
    dtype=np.int8,
)


ULAW_OUTWARD = np.array(
    [
        [
            127,
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
            13,
            14,
            15,
            16,
            17,
            18,
            19,
            20,
            21,
            22,
            23,
            24,
            25,
            26,
            27,
            28,
            29,
            30,
            31,
            32,
            33,
            34,
            35,
            36,
            37,
            38,
            39,
            40,
            41,
            42,
            43,
            44,
            45,
            46,
            47,
            48,
            49,
            50,
            51,
            52,
            53,
            54,
            55,
            56,
            57,
            58,
            59,
            60,
            61,
            62,
            63,
            64,
            65,
            66,
            67,
            68,
            69,
            70,
            71,
            72,
            73,
            74,
            75,
            76,
            77,
            78,
            79,
            80,
            81,
            82,
            83,
            84,
            85,
            86,
            87,
            88,
            89,
            90,
            91,
            92,
            93,
            94,
            95,
            96,
            97,
            98,
            99,
            100,
            101,
            102,
            103,
            104,
            105,
            106,
            107,
            108,
            109,
            110,
            111,
            112,
            113,
            114,
            115,
            116,
            117,
            118,
            119,
            120,
            121,
            122,
            123,
            124,
            125,
            126,
            255,
            254,
            253,
            252,
            251,
            250,
            249,
            248,
            247,
            246,
            245,
            244,
            243,
            242,
            241,
            240,
            239,
            238,
            237,
            236,
            235,
            234,
            233,
            232,
            231,
            230,
            229,
            228,
            227,
            226,
            225,
            224,
            223,
            222,
            221,
            220,
            219,
            218,
            217,
            216,
            215,
            214,
            213,
            212,
            211,
            210,
            209,
            208,
            207,
            206,
            205,
            204,
            203,
            202,
            201,
            200,
            199,
            198,
            197,
            196,
            195,
            194,
            193,
            192,
            191,
            190,
            189,
            188,
            187,
            186,
            185,
            184,
            183,
            182,
            181,
            180,
            179,
            178,
            177,
            176,
            175,
            174,
            173,
            172,
            171,
            170,
            169,
            168,
            167,
            166,
            165,
            164,
            163,
            162,
            161,
            160,
            159,
            158,
            157,
            156,
            155,
            154,
            153,
            152,
            151,
            150,
            149,
            148,
            147,
            146,
            145,
            144,
            143,
            142,
            141,
            140,
            139,
            138,
            137,
            136,
            135,
            134,
            133,
            132,
            131,
            130,
            129,
            128,
        ],
        [
            112,
            114,
            116,
            118,
            120,
            122,
            124,
            126,
            127,
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
            13,
            14,
            15,
            16,
            17,
            18,
            19,
            20,
            21,
            22,
            23,
            24,
            25,
            26,
            27,
            28,
            29,
            30,
            31,
            32,
            33,
            34,
            35,
            36,
            37,
            38,
            39,
            40,
            41,
            42,
            43,
            44,
            45,
            46,
            47,
            48,
            49,
            50,
            51,
            52,
            53,
            54,
            55,
            56,
            57,
            58,
            59,
            60,
            61,
            62,
            63,
            64,
            65,
            66,
            67,
            68,
            69,
            70,
            71,
            72,
            73,
            74,
            75,
            76,
            77,
            78,
            79,
            80,
            81,
            82,
            83,
            84,
            85,
            86,
            87,
            88,
            89,
            90,
            91,
            92,
            93,
            94,
            95,
            96,
            97,
            98,
            99,
            100,
            101,
            102,
            103,
            104,
            105,
            106,
            107,
            108,
            109,
            110,
            111,
            113,
            115,
            117,
            119,
            121,
            123,
            125,
            255,
            253,
            251,
            249,
            247,
            245,
            243,
            241,
            239,
            238,
            237,
            236,
            235,
            234,
            233,
            232,
            231,
            230,
            229,
            228,
            227,
            226,
            225,
            224,
            223,
            222,
            221,
            220,
            219,
            218,
            217,
            216,
            215,
            214,
            213,
            212,
            211,
            210,
            209,
            208,
            207,
            206,
            205,
            204,
            203,
            202,
            201,
            200,
            199,
            198,
            197,
            196,
            195,
            194,
            193,
            192,
            191,
            190,
            189,
            188,
            187,
            186,
            185,
            184,
            183,
            182,
            181,
            180,
            179,
            178,
            177,
            176,
            175,
            174,
            173,
            172,
            171,
            170,
            169,
            168,
            167,
            166,
            165,
            164,
            163,
            162,
            161,
            160,
            159,
            158,
            157,
            156,
            155,
            154,
            153,
            152,
            151,
            150,
            149,
            148,
            147,
            146,
            145,
            144,
            143,
            142,
            141,
            140,
            139,
            138,
            137,
            136,
            135,
            134,
            133,
            132,
            131,
            130,
            129,
            128,
            254,
            252,
            250,
            248,
            246,
            244,
            242,
            240,
        ],
        [
            96,
            98,
            100,
            102,
            104,
            106,
            108,
            110,
            112,
            113,
            114,
            116,
            117,
            118,
            120,
            121,
            122,
            124,
            125,
            126,
            127,
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
            13,
            14,
            15,
            16,
            17,
            18,
            19,
            20,
            21,
            22,
            23,
            24,
            25,
            26,
            27,
            28,
            29,
            30,
            31,
            32,
            33,
            34,
            35,
            36,
            37,
            38,
            39,
            40,
            41,
            42,
            43,
            44,
            45,
            46,
            47,
            48,
            49,
            50,
            51,
            52,
            53,
            54,
            55,
            56,
            57,
            58,
            59,
            60,
            61,
            62,
            63,
            64,
            65,
            66,
            67,
            68,
            69,
            70,
            71,
            72,
            73,
            74,
            75,
            76,
            77,
            78,
            79,
            80,
            81,
            82,
            83,
            84,
            85,
            86,
            87,
            88,
            89,
            90,
            91,
            92,
            93,
            94,
            95,
            97,
            99,
            101,
            103,
            105,
            107,
            109,
            111,
            115,
            119,
            123,
            255,
            251,
            247,
            243,
            239,
            237,
            235,
            233,
            231,
            229,
            227,
            225,
            223,
            222,
            221,
            220,
            219,
            218,
            217,
            216,
            215,
            214,
            213,
            212,
            211,
            210,
            209,
            208,
            207,
            206,
            205,
            204,
            203,
            202,
            201,
            200,
            199,
            198,
            197,
            196,
            195,
            194,
            193,
            192,
            191,
            190,
            189,
            188,
            187,
            186,
            185,
            184,
            183,
            182,
            181,
            180,
            179,
            178,
            177,
            176,
            175,
            174,
            173,
            172,
            171,
            170,
            169,
            168,
            167,
            166,
            165,
            164,
            163,
            162,
            161,
            160,
            159,
            158,
            157,
            156,
            155,
            154,
            153,
            152,
            151,
            150,
            149,
            148,
            147,
            146,
            145,
            144,
            143,
            142,
            141,
            140,
            139,
            138,
            137,
            136,
            135,
            134,
            133,
            132,
            131,
            130,
            129,
            128,
            254,
            253,
            252,
            250,
            249,
            248,
            246,
            245,
            244,
            242,
            241,
            240,
            238,
            236,
            234,
            232,
            230,
            228,
            226,
            224,
        ],
        [
            80,
            82,
            84,
            86,
            88,
            90,
            92,
            94,
            96,
            97,
            98,
            100,
            101,
            102,
            104,
            105,
            106,
            108,
            109,
            110,
            112,
            113,
            114,
            115,
            116,
            117,
            118,
            120,
            121,
            122,
            123,
            124,
            125,
            126,
            127,
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
            13,
            14,
            15,
            16,
            17,
            18,
            19,
            20,
            21,
            22,
            23,
            24,
            25,
            26,
            27,
            28,
            29,
            30,
            31,
            32,
            33,
            34,
            35,
            36,
            37,
            38,
            39,
            40,
            41,
            42,
            43,
            44,
            45,
            46,
            47,
            48,
            49,
            50,
            51,
            52,
            53,
            54,
            55,
            56,
            57,
            58,
            59,
            60,
            61,
            62,
            63,
            64,
            65,
            66,
            67,
            68,
            69,
            70,
            71,
            72,
            73,
            74,
            75,
            76,
            77,
            78,
            79,
            81,
            83,
            85,
            87,
            89,
            91,
            93,
            95,
            99,
            103,
            107,
            111,
            119,
            255,
            247,
            239,
            235,
            231,
            227,
            223,
            221,
            219,
            217,
            215,
            213,
            211,
            209,
            207,
            206,
            205,
            204,
            203,
            202,
            201,
            200,
            199,
            198,
            197,
            196,
            195,
            194,
            193,
            192,
            191,
            190,
            189,
            188,
            187,
            186,
            185,
            184,
            183,
            182,
            181,
            180,
            179,
            178,
            177,
            176,
            175,
            174,
            173,
            172,
            171,
            170,
            169,
            168,
            167,
            166,
            165,
            164,
            163,
            162,
            161,
            160,
            159,
            158,
            157,
            156,
            155,
            154,
            153,
            152,
            151,
            150,
            149,
            148,
            147,
            146,
            145,
            144,
            143,
            142,
            141,
            140,
            139,
            138,
            137,
            136,
            135,
            134,
            133,
            132,
            131,
            130,
            129,
            128,
            254,
            253,
            252,
            251,
            250,
            249,
            248,
            246,
            245,
            244,
            243,
            242,
            241,
            240,
            238,
            237,
            236,
            234,
            233,
            232,
            230,
            229,
            228,
            226,
            225,
            224,
            222,
            220,
            218,
            216,
            214,
            212,
            210,
            208,
        ],
        [
            64,
            66,
            68,
            70,
            72,
            74,
            76,
            78,
            80,
            81,
            82,
            84,
            85,
            86,
            88,
            89,
            90,
            92,
            93,
            94,
            96,
            97,
            98,
            99,
            100,
            101,
            102,
            104,
            105,
            106,
            107,
            108,
            109,
            110,
            112,
            113,
            114,
            115,
            116,
            117,
            118,
            119,
            120,
            121,
            122,
            123,
            124,
            125,
            126,
            127,
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
            13,
            14,
            15,
            16,
            17,
            18,
            19,
            20,
            21,
            22,
            23,
            24,
            25,
            26,
            27,
            28,
            29,
            30,
            31,
            32,
            33,
            34,
            35,
            36,
            37,
            38,
            39,
            40,
            41,
            42,
            43,
            44,
            45,
            46,
            47,
            48,
            49,
            50,
            51,
            52,
            53,
            54,
            55,
            56,
            57,
            58,
            59,
            60,
            61,
            62,
            63,
            65,
            67,
            69,
            71,
            73,
            75,
            77,
            79,
            83,
            87,
            91,
            95,
            103,
            111,
            255,
            239,
            231,
            223,
            219,
            215,
            211,
            207,
            205,
            203,
            201,
            199,
            197,
            195,
            193,
            191,
            190,
            189,
            188,
            187,
            186,
            185,
            184,
            183,
            182,
            181,
            180,
            179,
            178,
            177,
            176,
            175,
            174,
            173,
            172,
            171,
            170,
            169,
            168,
            167,
            166,
            165,
            164,
            163,
            162,
            161,
            160,
            159,
            158,
            157,
            156,
            155,
            154,
            153,
            152,
            151,
            150,
            149,
            148,
            147,
            146,
            145,
            144,
            143,
            142,
            141,
            140,
            139,
            138,
            137,
            136,
            135,
            134,
            133,
            132,
            131,
            130,
            129,
            128,
            254,
            253,
            252,
            251,
            250,
            249,
            248,
            247,
            246,
            245,
            244,
            243,
            242,
            241,
            240,
            238,
            237,
            236,
            235,
            234,
            233,
            232,
            230,
            229,
            228,
            227,
            226,
            225,
            224,
            222,
            221,
            220,
            218,
            217,
            216,
            214,
            213,
            212,
            210,
            209,
            208,
            206,
            204,
            202,
            200,
            198,
            196,
            194,
            192,
        ],
        [
            49,
            51,
            53,
            55,
            57,
            59,
            61,
            63,
            64,
            66,
            67,
            68,
            70,
            71,
            72,
            74,
            75,
            76,
            78,
            79,
            80,
            81,
            82,
            84,
            85,
            86,
            87,
            88,
            89,
            90,
            92,
            93,
            94,
            95,
            96,
            97,
            98,
            99,
            100,
            101,
            102,
            104,
            105,
            106,
            107,
            108,
            109,
            110,
            111,
            112,
            113,
            114,
            115,
            116,
            117,
            118,
            119,
            120,
            121,
            122,
            123,
            124,
            125,
            126,
            127,
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
            13,
            14,
            15,
            16,
            17,
            18,
            19,
            20,
            21,
            22,
            23,
            24,
            25,
            26,
            27,
            28,
            29,
            30,
            31,
            32,
            33,
            34,
            35,
            36,
            37,
            38,
            39,
            40,
            41,
            42,
            43,
            44,
            45,
            46,
            47,
            48,
            50,
            52,
            54,
            56,
            58,
            60,
            62,
            65,
            69,
            73,
            77,
            83,
            91,
            103,
            255,
            231,
            219,
            211,
            205,
            201,
            197,
            193,
            190,
            188,
            186,
            184,
            182,
            180,
            178,
            176,
            175,
            174,
            173,
            172,
            171,
            170,
            169,
            168,
            167,
            166,
            165,
            164,
            163,
            162,
            161,
            160,
            159,
            158,
            157,
            156,
            155,
            154,
            153,
            152,
            151,
            150,
            149,
            148,
            147,
            146,
            145,
            144,
            143,
            142,
            141,
            140,
            139,
            138,
            137,
            136,
            135,
            134,
            133,
            132,
            131,
            130,
            129,
            128,
            254,
            253,
            252,
            251,
            250,
            249,
            248,
            247,
            246,
            245,
            244,
            243,
            242,
            241,
            240,
            239,
            238,
            237,
            236,
            235,
            234,
            233,
            232,
            230,
            229,
            228,
            227,
            226,
            225,
            224,
            223,
            222,
            221,
            220,
            218,
            217,
            216,
            215,
            214,
            213,
            212,
            210,
            209,
            208,
            207,
            206,
            204,
            203,
            202,
            200,
            199,
            198,
            196,
            195,
            194,
            192,
            191,
            189,
            187,
            185,
            183,
            181,
            179,
            177,
        ],
        [
            32,
            34,
            36,
            38,
            40,
            42,
            44,
            46,
            48,
            49,
            51,
            52,
            53,
            55,
            56,
            57,
            59,
            60,
            61,
            63,
            64,
            65,
            66,
            67,
            68,
            70,
            71,
            72,
            73,
            74,
            75,
            76,
            78,
            79,
            80,
            81,
            82,
            83,
            84,
            85,
            86,
            87,
            88,
            89,
            90,
            92,
            93,
            94,
            95,
            96,
            97,
            98,
            99,
            100,
            101,
            102,
            103,
            104,
            105,
            106,
            107,
            108,
            109,
            110,
            111,
            112,
            113,
            114,
            115,
            116,
            117,
            118,
            119,
            120,
            121,
            122,
            123,
            124,
            125,
            126,
            127,
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
            13,
            14,
            15,
            16,
            17,
            18,
            19,
            20,
            21,
            22,
            23,
            24,
            25,
            26,
            27,
            28,
            29,
            30,
            31,
            33,
            35,
            37,
            39,
            41,
            43,
            45,
            47,
            50,
            54,
            58,
            62,
            69,
            77,
            91,
            255,
            219,
            205,
            197,
            190,
            186,
            182,
            178,
            175,
            173,
            171,
            169,
            167,
            165,
            163,
            161,
            159,
            158,
            157,
            156,
            155,
            154,
            153,
            152,
            151,
            150,
            149,
            148,
            147,
            146,
            145,
            144,
            143,
            142,
            141,
            140,
            139,
            138,
            137,
            136,
            135,
            134,
            133,
            132,
            131,
            130,
            129,
            128,
            254,
            253,
            252,
            251,
            250,
            249,
            248,
            247,
            246,
            245,
            244,
            243,
            242,
            241,
            240,
            239,
            238,
            237,
            236,
            235,
            234,
            233,
            232,
            231,
            230,
            229,
            228,
            227,
            226,
            225,
            224,
            223,
            222,
            221,
            220,
            218,
            217,
            216,
            215,
            214,
            213,
            212,
            211,
            210,
            209,
            208,
            207,
            206,
            204,
            203,
            202,
            201,
            200,
            199,
            198,
            196,
            195,
            194,
            193,
            192,
            191,
            189,
            188,
            187,
            185,
            184,
            183,
            181,
            180,
            179,
            177,
            176,
            174,
            172,
            170,
            168,
            166,
            164,
            162,
            160,
        ],
        [
            16,
            18,
            20,
            22,
            24,
            26,
            28,
            30,
            32,
            33,
            34,
            36,
            37,
            38,
            40,
            41,
            42,
            44,
            45,
            46,
            48,
            49,
            50,
            51,
            52,
            53,
            55,
            56,
            57,
            58,
            59,
            60,
            61,
            63,
            64,
            65,
            66,
            67,
            68,
            69,
            70,
            71,
            72,
            73,
            74,
            75,
            76,
            78,
            79,
            80,
            81,
            82,
            83,
            84,
            85,
            86,
            87,
            88,
            89,
            90,
            91,
            92,
            93,
            94,
            95,
            96,
            97,
            98,
            99,
            100,
            101,
            102,
            103,
            104,
            105,
            106,
            107,
            108,
            109,
            110,
            111,
            112,
            113,
            114,
            115,
            116,
            117,
            118,
            119,
            120,
            121,
            122,
            123,
            124,
            125,
            126,
            127,
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
            13,
            14,
            15,
            17,
            19,
            21,
            23,
            25,
            27,
            29,
            31,
            35,
            39,
            43,
            47,
            54,
            62,
            77,
            255,
            205,
            190,
            182,
            175,
            171,
            167,
            163,
            159,
            157,
            155,
            153,
            151,
            149,
            147,
            145,
            143,
            142,
            141,
            140,
            139,
            138,
            137,
            136,
            135,
            134,
            133,
            132,
            131,
            130,
            129,
            128,
            254,
            253,
            252,
            251,
            250,
            249,
            248,
            247,
            246,
            245,
            244,
            243,
            242,
            241,
            240,
            239,
            238,
            237,
            236,
            235,
            234,
            233,
            232,
            231,
            230,
            229,
            228,
            227,
            226,
            225,
            224,
            223,
            222,
            221,
            220,
            219,
            218,
            217,
            216,
            215,
            214,
            213,
            212,
            211,
            210,
            209,
            208,
            207,
            206,
            204,
            203,
            202,
            201,
            200,
            199,
            198,
            197,
            196,
            195,
            194,
            193,
            192,
            191,
            189,
            188,
            187,
            186,
            185,
            184,
            183,
            181,
            180,
            179,
            178,
            177,
            176,
            174,
            173,
            172,
            170,
            169,
            168,
            166,
            165,
            164,
            162,
            161,
            160,
            158,
            156,
            154,
            152,
            150,
            148,
            146,
            144,
        ],
        [
            2,
            4,
            6,
            8,
            10,
            12,
            14,
            16,
            17,
            18,
            20,
            21,
            22,
            24,
            25,
            26,
            28,
            29,
            30,
            32,
            33,
            34,
            35,
            36,
            37,
            38,
            40,
            41,
            42,
            43,
            44,
            45,
            46,
            48,
            49,
            50,
            51,
            52,
            53,
            54,
            55,
            56,
            57,
            58,
            59,
            60,
            61,
            63,
            64,
            65,
            66,
            67,
            68,
            69,
            70,
            71,
            72,
            73,
            74,
            75,
            76,
            77,
            78,
            79,
            80,
            81,
            82,
            83,
            84,
            85,
            86,
            87,
            88,
            89,
            90,
            91,
            92,
            93,
            94,
            95,
            96,
            97,
            98,
            99,
            100,
            101,
            102,
            103,
            104,
            105,
            106,
            107,
            108,
            109,
            110,
            111,
            112,
            113,
            114,
            115,
            116,
            117,
            118,
            119,
            120,
            121,
            122,
            123,
            124,
            125,
            126,
            127,
            0,
            1,
            3,
            5,
            7,
            9,
            11,
            13,
            15,
            19,
            23,
            27,
            31,
            39,
            47,
            62,
            255,
            190,
            175,
            167,
            159,
            155,
            151,
            147,
            143,
            141,
            139,
            137,
            135,
            133,
            131,
            129,
            254,
            253,
            252,
            251,
            250,
            249,
            248,
            247,
            246,
            245,
            244,
            243,
            242,
            241,
            240,
            239,
            238,
            237,
            236,
            235,
            234,
            233,
            232,
            231,
            230,
            229,
            228,
            227,
            226,
            225,
            224,
            223,
            222,
            221,
            220,
            219,
            218,
            217,
            216,
            215,
            214,
            213,
            212,
            211,
            210,
            209,
            208,
            207,
            206,
            205,
            204,
            203,
            202,
            201,
            200,
            199,
            198,
            197,
            196,
            195,
            194,
            193,
            192,
            191,
            189,
            188,
            187,
            186,
            185,
            184,
            183,
            182,
            181,
            180,
            179,
            178,
            177,
            176,
            174,
            173,
            172,
            171,
            170,
            169,
            168,
            166,
            165,
            164,
            163,
            162,
            161,
            160,
            158,
            157,
            156,
            154,
            153,
            152,
            150,
            149,
            148,
            146,
            145,
            144,
            142,
            140,
            138,
            136,
            134,
            132,
            130,
            128,
        ],
        [
            1,
            2,
            4,
            5,
            6,
            8,
            9,
            10,
            12,
            13,
            14,
            16,
            17,
            18,
            19,
            20,
            21,
            22,
            24,
            25,
            26,
            27,
            28,
            29,
            30,
            32,
            33,
            34,
            35,
            36,
            37,
            38,
            39,
            40,
            41,
            42,
            43,
            44,
            45,
            46,
            48,
            49,
            50,
            51,
            52,
            53,
            54,
            55,
            56,
            57,
            58,
            59,
            60,
            61,
            62,
            63,
            64,
            65,
            66,
            67,
            68,
            69,
            70,
            71,
            72,
            73,
            74,
            75,
            76,
            77,
            78,
            79,
            80,
            81,
            82,
            83,
            84,
            85,
            86,
            87,
            88,
            89,
            90,
            91,
            92,
            93,
            94,
            95,
            96,
            97,
            98,
            99,
            100,
            101,
            102,
            103,
            104,
            105,
            106,
            107,
            108,
            109,
            110,
            111,
            112,
            113,
            114,
            115,
            116,
            117,
            118,
            119,
            120,
            121,
            122,
            123,
            124,
            125,
            126,
            127,
            0,
            3,
            7,
            11,
            15,
            23,
            31,
            47,
            255,
            175,
            159,
            151,
            143,
            139,
            135,
            131,
            254,
            253,
            252,
            251,
            250,
            249,
            248,
            247,
            246,
            245,
            244,
            243,
            242,
            241,
            240,
            239,
            238,
            237,
            236,
            235,
            234,
            233,
            232,
            231,
            230,
            229,
            228,
            227,
            226,
            225,
            224,
            223,
            222,
            221,
            220,
            219,
            218,
            217,
            216,
            215,
            214,
            213,
            212,
            211,
            210,
            209,
            208,
            207,
            206,
            205,
            204,
            203,
            202,
            201,
            200,
            199,
            198,
            197,
            196,
            195,
            194,
            193,
            192,
            191,
            190,
            189,
            188,
            187,
            186,
            185,
            184,
            183,
            182,
            181,
            180,
            179,
            178,
            177,
            176,
            174,
            173,
            172,
            171,
            170,
            169,
            168,
            167,
            166,
            165,
            164,
            163,
            162,
            161,
            160,
            158,
            157,
            156,
            155,
            154,
            153,
            152,
            150,
            149,
            148,
            147,
            146,
            145,
            144,
            142,
            141,
            140,
            138,
            137,
            136,
            134,
            133,
            132,
            130,
            129,
            128,
        ],
        [
            1,
            2,
            3,
            4,
            5,
            6,
            8,
            9,
            10,
            11,
            12,
            13,
            14,
            16,
            17,
            18,
            19,
            20,
            21,
            22,
            23,
            24,
            25,
            26,
            27,
            28,
            29,
            30,
            32,
            33,
            34,
            35,
            36,
            37,
            38,
            39,
            40,
            41,
            42,
            43,
            44,
            45,
            46,
            47,
            48,
            49,
            50,
            51,
            52,
            53,
            54,
            55,
            56,
            57,
            58,
            59,
            60,
            61,
            62,
            63,
            64,
            65,
            66,
            67,
            68,
            69,
            70,
            71,
            72,
            73,
            74,
            75,
            76,
            77,
            78,
            79,
            80,
            81,
            82,
            83,
            84,
            85,
            86,
            87,
            88,
            89,
            90,
            91,
            92,
            93,
            94,
            95,
            96,
            97,
            98,
            99,
            100,
            101,
            102,
            103,
            104,
            105,
            106,
            107,
            108,
            109,
            110,
            111,
            112,
            113,
            114,
            115,
            116,
            117,
            118,
            119,
            120,
            121,
            122,
            123,
            124,
            125,
            126,
            127,
            0,
            7,
            15,
            31,
            255,
            159,
            143,
            135,
            254,
            253,
            252,
            251,
            250,
            249,
            248,
            247,
            246,
            245,
            244,
            243,
            242,
            241,
            240,
            239,
            238,
            237,
            236,
            235,
            234,
            233,
            232,
            231,
            230,
            229,
            228,
            227,
            226,
            225,
            224,
            223,
            222,
            221,
            220,
            219,
            218,
            217,
            216,
            215,
            214,
            213,
            212,
            211,
            210,
            209,
            208,
            207,
            206,
            205,
            204,
            203,
            202,
            201,
            200,
            199,
            198,
            197,
            196,
            195,
            194,
            193,
            192,
            191,
            190,
            189,
            188,
            187,
            186,
            185,
            184,
            183,
            182,
            181,
            180,
            179,
            178,
            177,
            176,
            175,
            174,
            173,
            172,
            171,
            170,
            169,
            168,
            167,
            166,
            165,
            164,
            163,
            162,
            161,
            160,
            158,
            157,
            156,
            155,
            154,
            153,
            152,
            151,
            150,
            149,
            148,
            147,
            146,
            145,
            144,
            142,
            141,
            140,
            139,
            138,
            137,
            136,
            134,
            133,
            132,
            131,
            130,
            129,
            128,
        ],
        [
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
            13,
            14,
            16,
            17,
            18,
            19,
            20,
            21,
            22,
            23,
            24,
            25,
            26,
            27,
            28,
            29,
            30,
            31,
            32,
            33,
            34,
            35,
            36,
            37,
            38,
            39,
            40,
            41,
            42,
            43,
            44,
            45,
            46,
            47,
            48,
            49,
            50,
            51,
            52,
            53,
            54,
            55,
            56,
            57,
            58,
            59,
            60,
            61,
            62,
            63,
            64,
            65,
            66,
            67,
            68,
            69,
            70,
            71,
            72,
            73,
            74,
            75,
            76,
            77,
            78,
            79,
            80,
            81,
            82,
            83,
            84,
            85,
            86,
            87,
            88,
            89,
            90,
            91,
            92,
            93,
            94,
            95,
            96,
            97,
            98,
            99,
            100,
            101,
            102,
            103,
            104,
            105,
            106,
            107,
            108,
            109,
            110,
            111,
            112,
            113,
            114,
            115,
            116,
            117,
            118,
            119,
            120,
            121,
            122,
            123,
            124,
            125,
            126,
            127,
            0,
            15,
            255,
            143,
            254,
            253,
            252,
            251,
            250,
            249,
            248,
            247,
            246,
            245,
            244,
            243,
            242,
            241,
            240,
            239,
            238,
            237,
            236,
            235,
            234,
            233,
            232,
            231,
            230,
            229,
            228,
            227,
            226,
            225,
            224,
            223,
            222,
            221,
            220,
            219,
            218,
            217,
            216,
            215,
            214,
            213,
            212,
            211,
            210,
            209,
            208,
            207,
            206,
            205,
            204,
            203,
            202,
            201,
            200,
            199,
            198,
            197,
            196,
            195,
            194,
            193,
            192,
            191,
            190,
            189,
            188,
            187,
            186,
            185,
            184,
            183,
            182,
            181,
            180,
            179,
            178,
            177,
            176,
            175,
            174,
            173,
            172,
            171,
            170,
            169,
            168,
            167,
            166,
            165,
            164,
            163,
            162,
            161,
            160,
            159,
            158,
            157,
            156,
            155,
            154,
            153,
            152,
            151,
            150,
            149,
            148,
            147,
            146,
            145,
            144,
            142,
            141,
            140,
            139,
            138,
            137,
            136,
            135,
            134,
            133,
            132,
            131,
            130,
            129,
            128,
        ],
        [
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
            13,
            14,
            15,
            16,
            17,
            18,
            19,
            20,
            21,
            22,
            23,
            24,
            25,
            26,
            27,
            28,
            29,
            30,
            31,
            32,
            33,
            34,
            35,
            36,
            37,
            38,
            39,
            40,
            41,
            42,
            43,
            44,
            45,
            46,
            47,
            48,
            49,
            50,
            51,
            52,
            53,
            54,
            55,
            56,
            57,
            58,
            59,
            60,
            61,
            62,
            63,
            64,
            65,
            66,
            67,
            68,
            69,
            70,
            71,
            72,
            73,
            74,
            75,
            76,
            77,
            78,
            79,
            80,
            81,
            82,
            83,
            84,
            85,
            86,
            87,
            88,
            89,
            90,
            91,
            92,
            93,
            94,
            95,
            96,
            97,
            98,
            99,
            100,
            101,
            102,
            103,
            104,
            105,
            106,
            107,
            108,
            109,
            110,
            111,
            112,
            113,
            114,
            115,
            116,
            117,
            118,
            119,
            120,
            121,
            122,
            123,
            124,
            125,
            126,
            127,
            0,
            255,
            254,
            253,
            252,
            251,
            250,
            249,
            248,
            247,
            246,
            245,
            244,
            243,
            242,
            241,
            240,
            239,
            238,
            237,
            236,
            235,
            234,
            233,
            232,
            231,
            230,
            229,
            228,
            227,
            226,
            225,
            224,
            223,
            222,
            221,
            220,
            219,
            218,
            217,
            216,
            215,
            214,
            213,
            212,
            211,
            210,
            209,
            208,
            207,
            206,
            205,
            204,
            203,
            202,
            201,
            200,
            199,
            198,
            197,
            196,
            195,
            194,
            193,
            192,
            191,
            190,
            189,
            188,
            187,
            186,
            185,
            184,
            183,
            182,
            181,
            180,
            179,
            178,
            177,
            176,
            175,
            174,
            173,
            172,
            171,
            170,
            169,
            168,
            167,
            166,
            165,
            164,
            163,
            162,
            161,
            160,
            159,
            158,
            157,
            156,
            155,
            154,
            153,
            152,
            151,
            150,
            149,
            148,
            147,
            146,
            145,
            144,
            143,
            142,
            141,
            140,
            139,
            138,
            137,
            136,
            135,
            134,
            133,
            132,
            131,
            130,
            129,
            128,
        ],
    ],
    dtype=np.uint8,
)
