"""
Shows all defined flow classes.
"""

import argparse
import os
import tabulate
from trustgraph.api import Api
import json

default_url = os.getenv("TRUSTGRAPH_URL", 'http://localhost:8088/')

def show_flow_classes(url):

    api = Api(url).flow()

    class_names = api.list_classes()

    if len(class_names) == 0:
        print("No flows.")
        return

    classes = []

    for class_name in class_names:
        cls = api.get_class(class_name)
        classes.append((
            class_name,
            cls.get("description", ""),
            ", ".join(cls.get("tags", [])),
        ))

    print(tabulate.tabulate(
        classes,
        tablefmt="pretty",
        maxcolwidths=[None, 40, 20],
        stralign="left",
        headers = ["flow class", "description", "tags"],
    ))

def main():

    parser = argparse.ArgumentParser(
        prog='tg-show-flow-classes',
        description=__doc__,
    )

    parser.add_argument(
        '-u', '--api-url',
        default=default_url,
        help=f'API URL (default: {default_url})',
    )

    args = parser.parse_args()

    try:

        show_flow_classes(
            url=args.api_url,
        )

    except Exception as e:

        print("Exception:", e, flush=True)

if __name__ == "__main__":
    main()