# SpectroscPy 0.2.0
# SpectroscPy is a script package developed by and containing contributions from

    # Karen Oda Hjorth Minde Dundas
    # Magnus Ringholm
    # Yann Cornation
    # Benedicte Ofstad

# The package is released under a LGPL licence.
# For questions, please contact on karen.o.dundas@uit.no

from spectroscpy import get_average_beta_aaa2, get_average_beta_baa2, get_hyperraman_SI_scs, \
                        get_hyperraman_intensities, anharmonicProperty
import numpy as np

harmonic_beta_transition_moment = \
[[[[ 4.99428731e-53,  8.62230599e-55, -1.73620632e-68],
   [ 8.62230599e-55,  1.39523984e-53,  7.00738036e-70],
   [-1.73620632e-68,  7.00738036e-70,  1.39400627e-53]],
  [[ 8.62230599e-55,  1.39523984e-53,  7.00738036e-70],
   [ 1.39523984e-53,  1.63797818e-54, -1.12787012e-69],
   [ 7.00738036e-70, -1.12787012e-69,  5.45731510e-55]],
  [[-1.73620632e-68,  7.00738036e-70,  1.39400627e-53],
   [ 7.00738036e-70, -1.12787012e-69,  5.45731510e-55],
   [ 1.39400627e-53,  5.45731510e-55, -3.69015263e-69]]],
 [[[ 2.96766131e-59,  2.26255881e-58,  1.85012367e-68],
   [ 2.26255881e-58,  4.52329861e-59,  4.72297147e-69],
   [ 1.85012367e-68,  4.72297147e-69,  1.67629410e-59]],
  [[ 2.26255881e-58,  4.52329861e-59,  4.72297147e-69],
   [ 4.52329861e-59,  1.03090831e-58,  1.02910924e-69],
   [ 4.72297147e-69,  1.02910924e-69,  3.35165572e-59]],
  [[ 1.85012367e-68,  4.72297147e-69,  1.67629410e-59],
   [ 4.72297147e-69,  1.02910924e-69,  3.35165572e-59],
   [ 1.67629410e-59,  3.35165572e-59,  2.10909129e-69]]],
 [[[ 3.03215531e-65, -2.57877584e-64,  8.56945304e-53],
   [-2.57877584e-64, -1.38525446e-65,  2.65356980e-54],
   [ 8.56945304e-53,  2.65356980e-54,  2.09896986e-66]],
  [[-2.57877584e-64, -1.38525446e-65,  2.65356980e-54],
   [-1.38525446e-65, -1.62927325e-64,  1.80011061e-53],
   [ 2.65356980e-54,  1.80011061e-53, -5.39859849e-65]],
  [[ 8.56945304e-53,  2.65356980e-54,  2.09896986e-66],
   [ 2.65356980e-54,  1.80011061e-53, -5.39859849e-65],
   [ 2.09896986e-66, -5.39859849e-65,  5.36919699e-53]]],
 [[[ 1.00548338e-53, -8.54329699e-53, -1.56238589e-64],
   [-8.54329699e-53, -4.59937874e-54, -4.81631732e-66],
   [-1.56238589e-64, -4.81631732e-66,  7.00017778e-55]],
  [[-8.54329699e-53, -4.59937874e-54, -4.81631732e-66],
   [-4.59937874e-54, -5.39663710e-53, -3.28282168e-65],
   [-4.81631732e-66, -3.28282168e-65, -1.78851228e-53]],
  [[-1.56238589e-64, -4.81631732e-66,  7.00017778e-55],
   [-4.81631732e-66, -3.28282168e-65, -1.78851228e-53],
   [ 7.00017778e-55, -1.78851228e-53, -9.78862731e-65]]]]

fundamental_beta_transition_moment = \
[[[[ 4.97268930e-53,  6.65873346e-55,  8.01583452e-68],
   [ 6.65873346e-55,  1.63374643e-53, -1.25834746e-66],
   [ 8.01583452e-68, -1.25834746e-66,  1.63362292e-53]],
  [[ 6.65873346e-55,  1.63374643e-53, -1.25834746e-66],
   [ 1.63374643e-53,  1.91880188e-54, -1.00259234e-67],
   [-1.25834746e-66, -1.00259234e-67,  6.39489976e-55]],
  [[ 8.01583452e-68, -1.25834746e-66,  1.63362292e-53],
   [-1.25834746e-66, -1.00259234e-67,  6.39489976e-55],
   [ 1.63362292e-53,  6.39489976e-55, -4.32510446e-69]]],
 [[[ 1.55741744e-58, -5.43806885e-59, -2.63524626e-69],
   [-5.43806885e-59,  3.81209615e-59,  4.53472154e-69],
   [-2.63524626e-69,  4.53472154e-69, -1.46522270e-59]],
  [[-5.43806885e-59,  3.81209615e-59,  4.53472154e-69],
   [ 3.81209615e-59,  9.38524594e-61, -9.54981748e-70],
   [ 4.53472154e-69, -9.54981748e-70, -5.74245086e-60]],
  [[-2.63524626e-69,  4.53472154e-69, -1.46522270e-59],
   [ 4.53472154e-69, -9.54981748e-70, -5.74245086e-60],
   [-1.46522270e-59, -5.74245086e-60, -4.69920113e-69]]],
 [[[ 1.37508585e-65, -1.17087935e-64,  5.52925035e-53],
   [-1.17087935e-64, -6.42457807e-66,  1.72792224e-54],
   [ 5.52925035e-53,  1.72792224e-54,  8.86841416e-67]],
  [[-1.17087935e-64, -6.42457807e-66,  1.72792224e-54],
   [-6.42457807e-66, -7.02935573e-65,  1.12142452e-53],
   [ 1.72792224e-54,  1.12142452e-53, -2.28508052e-65]],
  [[ 5.52925035e-53,  1.72792224e-54,  8.86841416e-67],
   [ 1.72792224e-54,  1.12142452e-53, -2.28508052e-65],
   [ 8.86841416e-67, -2.28508052e-65,  3.34402029e-53]]],
 [[[ 6.48594793e-54, -5.51038760e-53, -1.50221666e-64],
   [-5.51038760e-53, -3.01371279e-54, -4.64677610e-66],
   [-1.50221666e-64, -4.64677610e-66,  4.35419551e-55]],
  [[-5.51038760e-53, -3.01371279e-54, -4.64677610e-66],
   [-3.01371279e-54, -3.36076047e-53, -3.12815760e-65],
   [-4.64677610e-66, -3.12815760e-65, -1.11351386e-53]],
  [[-1.50221666e-64, -4.64677610e-66,  4.35419551e-55],
   [-4.64677610e-66, -3.12815760e-65, -1.11351386e-53],
   [ 4.35419551e-55, -1.11351386e-53, -9.19873611e-65]]]]

overtone_beta_transition_moment = \
[[[[-2.67032534e-62,  4.55000407e-60, -1.43693571e-69],
   [ 4.55000407e-60,  4.84176804e-61,  7.68072666e-70],
   [-1.43693571e-69,  7.68072666e-70, -7.18269647e-61]],
  [[ 4.55000407e-60,  4.84176804e-61,  7.68072666e-70],
   [ 4.84176804e-61,  4.55101629e-60,  1.68667504e-70],
   [ 7.68072666e-70,  1.68667504e-70, -4.46839074e-62]],
  [[-1.43693571e-69,  7.68072666e-70, -7.18269647e-61],
   [ 7.68072666e-70,  1.68667504e-70, -4.46839074e-62],
   [-7.18269647e-61, -4.46839074e-62,  5.16918773e-71]]],
 [[[-3.86127277e-59,  2.83036284e-59, -9.72312310e-69],
   [ 2.83036284e-59, -5.59685326e-61, -3.62062979e-70],
   [-9.72312310e-69, -3.62062979e-70, -1.98698095e-60]],
  [[ 2.83036284e-59, -5.59685326e-61, -3.62062979e-70],
   [-5.59685326e-61, -2.81754752e-59,  7.26339422e-72],
   [-3.62062979e-70,  7.26339422e-72,  1.19073571e-60]],
  [[-9.72312310e-69, -3.62062979e-70, -1.98698095e-60],
   [-3.62062979e-70,  7.26339422e-72,  1.19073571e-60],
   [-1.98698095e-60,  1.19073571e-60,  2.92110006e-70]]],
 [[[ 9.76505489e-58,  4.43171816e-58,  1.20803544e-67],
   [ 4.43171816e-58,  1.00330773e-58, -6.00777671e-69],
   [ 1.20803544e-67, -6.00777671e-69, -4.49190430e-58]],
  [[ 4.43171816e-58,  1.00330773e-58, -6.00777671e-69],
   [ 1.00330773e-58,  2.14874404e-58,  2.10530171e-68],
   [-6.00777671e-69,  2.10530171e-68,  4.88660427e-59]],
  [[ 1.20803544e-67, -6.00777671e-69, -4.49190430e-58],
   [-6.00777671e-69,  2.10530171e-68,  4.88660427e-59],
   [-4.49190430e-58,  4.88660427e-59,  6.32040613e-68]]],
 [[[-9.66751364e-59,  1.11093528e-57,  2.52705952e-68],
   [ 1.11093528e-57,  5.86984244e-59, -1.47548200e-68],
   [ 2.52705952e-68, -1.47548200e-68,  1.05229983e-59]],
  [[ 1.11093528e-57,  5.86984244e-59, -1.47548200e-68],
   [ 5.86984244e-59,  6.28999934e-58,  9.48749777e-69],
   [-1.47548200e-68,  9.48749777e-69,  2.12712001e-58]],
  [[ 2.52705952e-68, -1.47548200e-68,  1.05229983e-59],
   [-1.47548200e-68,  9.48749777e-69,  2.12712001e-58],
   [ 1.05229983e-59,  2.12712001e-58,  2.09284958e-68]]]]

combotone_beta_transition_moment = \
[[[[[ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
    [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
    [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]],
   [[ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
    [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
    [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]],
   [[ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
    [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
    [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]]],
  [[[-2.19479302e-54, -6.21372914e-56, -4.62190338e-69],
    [-6.21372914e-56, -3.05665468e-55, -3.07023523e-69],
    [-4.62190338e-69, -3.07023523e-69, -3.03716479e-55]],
   [[-6.21372914e-56, -3.05665468e-55, -3.07023523e-69],
    [-3.05665468e-55, -3.56561126e-56, -4.32595802e-70],
    [-3.07023523e-69, -4.32595802e-70, -1.18908552e-56]],
   [[-4.62190338e-69, -3.07023523e-69, -3.03716479e-55],
    [-3.07023523e-69, -4.32595802e-70, -1.18908552e-56],
    [-3.03716479e-55, -1.18908552e-56, -5.44686790e-70]]],
  [[[-2.58755430e-68,  6.50417364e-69, -1.17831580e-58],
    [ 6.50417364e-69, -7.32794666e-69,  4.47524762e-59],
    [-1.17831580e-58,  4.47524762e-59, -7.05813098e-69]],
   [[ 6.50417364e-69, -7.32794666e-69,  4.47524762e-59],
    [-7.32794666e-69,  3.32716848e-70, -4.68658671e-59],
    [ 4.47524762e-59, -4.68658671e-59,  3.33191053e-69]],
   [[-1.17831580e-58,  4.47524762e-59, -7.05813098e-69],
    [ 4.47524762e-59, -4.68658671e-59,  3.33191053e-69],
    [-7.05813098e-69,  3.33191053e-69, -1.49106082e-58]]],
  [[[ 6.28956438e-58,  4.16311682e-59, -2.65184678e-69],
    [ 4.16311682e-59,  1.63936341e-59, -1.05484883e-69],
    [-2.65184678e-69, -1.05484883e-69,  1.29085172e-58]],
   [[ 4.16311682e-59,  1.63936341e-59, -1.05484883e-69],
    [ 1.63936341e-59,  1.72452998e-59, -3.29678755e-69],
    [-1.05484883e-69, -3.29678755e-69,  9.00056198e-60]],
   [[-2.65184678e-69, -1.05484883e-69,  1.29085172e-58],
    [-1.05484883e-69, -3.29678755e-69,  9.00056198e-60],
    [ 1.29085172e-58,  9.00056198e-60, -1.33198347e-70]]]],
 [[[[-2.19479302e-54, -6.21372914e-56, -4.62190338e-69],
    [-6.21372914e-56, -3.05665468e-55, -3.07023523e-69],
    [-4.62190338e-69, -3.07023523e-69, -3.03716479e-55]],
   [[-6.21372914e-56, -3.05665468e-55, -3.07023523e-69],
    [-3.05665468e-55, -3.56561126e-56, -4.32595802e-70],
    [-3.07023523e-69, -4.32595802e-70, -1.18908552e-56]],
   [[-4.62190338e-69, -3.07023523e-69, -3.03716479e-55],
    [-3.07023523e-69, -4.32595802e-70, -1.18908552e-56],
    [-3.03716479e-55, -1.18908552e-56, -5.44686790e-70]]],
  [[[ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
    [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
    [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]],
   [[ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
    [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
    [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]],
   [[ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
    [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
    [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]]],
  [[[ 2.46288432e-66, -2.14615390e-65,  5.88721764e-54],
    [-2.14615390e-65, -1.39741614e-66,  2.09649805e-55],
    [ 5.88721764e-54,  2.09649805e-55,  8.81575276e-68]],
   [[-2.14615390e-65, -1.39741614e-66,  2.09649805e-55],
    [-1.39741614e-66, -7.21641267e-66,  5.36248197e-55],
    [ 2.09649805e-55,  5.36248197e-55, -2.36847196e-66]],
   [[ 5.88721764e-54,  2.09649805e-55,  8.81575276e-68],
    [ 2.09649805e-55,  5.36248197e-55, -2.36847196e-66],
    [ 8.81575276e-68, -2.36847196e-66,  1.58405330e-54]]],
  [[[ 6.90552182e-55, -5.86713172e-54, -6.96090089e-66],
    [-5.86713172e-54, -3.98729217e-55, -2.60493291e-67],
    [-6.96090089e-66, -2.60493291e-67,  2.04105101e-56]],
   [[-5.86713172e-54, -3.98729217e-55, -2.60493291e-67],
    [-3.98729217e-55, -1.60759695e-54, -1.96459801e-67],
    [-2.60493291e-67, -1.96459801e-67, -5.27642781e-55]],
   [[-6.96090089e-66, -2.60493291e-67,  2.04105101e-56],
    [-2.60493291e-67, -1.96459801e-67, -5.27642781e-55],
    [ 2.04105101e-56, -5.27642781e-55, -5.34919132e-67]]]],
 [[[[-2.58755430e-68,  6.50417364e-69, -1.17831580e-58],
    [ 6.50417364e-69, -7.32794666e-69,  4.47524762e-59],
    [-1.17831580e-58,  4.47524762e-59, -7.05813098e-69]],
   [[ 6.50417364e-69, -7.32794666e-69,  4.47524762e-59],
    [-7.32794666e-69,  3.32716848e-70, -4.68658671e-59],
    [ 4.47524762e-59, -4.68658671e-59,  3.33191053e-69]],
   [[-1.17831580e-58,  4.47524762e-59, -7.05813098e-69],
    [ 4.47524762e-59, -4.68658671e-59,  3.33191053e-69],
    [-7.05813098e-69,  3.33191053e-69, -1.49106082e-58]]],
  [[[ 2.46288432e-66, -2.14615390e-65,  5.88721764e-54],
    [-2.14615390e-65, -1.39741614e-66,  2.09649805e-55],
    [ 5.88721764e-54,  2.09649805e-55,  8.81575276e-68]],
   [[-2.14615390e-65, -1.39741614e-66,  2.09649805e-55],
    [-1.39741614e-66, -7.21641267e-66,  5.36248197e-55],
    [ 2.09649805e-55,  5.36248197e-55, -2.36847196e-66]],
   [[ 5.88721764e-54,  2.09649805e-55,  8.81575276e-68],
    [ 2.09649805e-55,  5.36248197e-55, -2.36847196e-66],
    [ 8.81575276e-68, -2.36847196e-66,  1.58405330e-54]]],
  [[[ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
    [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
    [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]],
   [[ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
    [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
    [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]],
   [[ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
    [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
    [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]]],
  [[[ 5.87887041e-69, -5.51850194e-68, -5.35128533e-58],
    [-5.51850194e-68,  3.76523830e-69,  1.75740226e-58],
    [-5.35128533e-58,  1.75740226e-58,  1.34357543e-68]],
   [[-5.51850194e-68,  3.76523830e-69,  1.75740226e-58],
    [ 3.76523830e-69, -3.31744635e-68, -9.07091513e-59],
    [ 1.75740226e-58, -9.07091513e-59, -5.44467643e-69]],
   [[-5.35128533e-58,  1.75740226e-58,  1.34357543e-68],
    [ 1.75740226e-58, -9.07091513e-59, -5.44467643e-69],
    [ 1.34357543e-68, -5.44467643e-69, -3.08379597e-58]]]],
 [[[[ 6.28956438e-58,  4.16311682e-59, -2.65184678e-69],
    [ 4.16311682e-59,  1.63936341e-59, -1.05484883e-69],
    [-2.65184678e-69, -1.05484883e-69,  1.29085172e-58]],
   [[ 4.16311682e-59,  1.63936341e-59, -1.05484883e-69],
    [ 1.63936341e-59,  1.72452998e-59, -3.29678755e-69],
    [-1.05484883e-69, -3.29678755e-69,  9.00056198e-60]],
   [[-2.65184678e-69, -1.05484883e-69,  1.29085172e-58],
    [-1.05484883e-69, -3.29678755e-69,  9.00056198e-60],
    [ 1.29085172e-58,  9.00056198e-60, -1.33198347e-70]]],
  [[[ 6.90552182e-55, -5.86713172e-54, -6.96090089e-66],
    [-5.86713172e-54, -3.98729217e-55, -2.60493291e-67],
    [-6.96090089e-66, -2.60493291e-67,  2.04105101e-56]],
   [[-5.86713172e-54, -3.98729217e-55, -2.60493291e-67],
    [-3.98729217e-55, -1.60759695e-54, -1.96459801e-67],
    [-2.60493291e-67, -1.96459801e-67, -5.27642781e-55]],
   [[-6.96090089e-66, -2.60493291e-67,  2.04105101e-56],
    [-2.60493291e-67, -1.96459801e-67, -5.27642781e-55],
    [ 2.04105101e-56, -5.27642781e-55, -5.34919132e-67]]],
  [[[ 5.87887041e-69, -5.51850194e-68, -5.35128533e-58],
    [-5.51850194e-68,  3.76523830e-69,  1.75740226e-58],
    [-5.35128533e-58,  1.75740226e-58,  1.34357543e-68]],
   [[-5.51850194e-68,  3.76523830e-69,  1.75740226e-58],
    [ 3.76523830e-69, -3.31744635e-68, -9.07091513e-59],
    [ 1.75740226e-58, -9.07091513e-59, -5.44467643e-69]],
   [[-5.35128533e-58,  1.75740226e-58,  1.34357543e-68],
    [ 1.75740226e-58, -9.07091513e-59, -5.44467643e-69],
    [ 1.34357543e-68, -5.44467643e-69, -3.08379597e-58]]],
  [[[ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
    [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
    [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]],
   [[ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
    [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
    [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]],
   [[ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
    [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
    [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]]]]]

def test_get_average_beta_aaa2():

    ref_beta_aaa2 = [7.29628980e-106, 2.20311508e-116, 3.60480807e-105, 3.60564859e-105]

    beta_aaa2 = get_average_beta_aaa2(harmonic_beta_transition_moment)

    assert np.allclose(np.multiply(ref_beta_aaa2, 1.0e107), np.multiply(beta_aaa2, 1.0e107))


def test_get_average_beta_baa2():

    ref_beta_baa2 = [8.19115199e-107, 5.34919340e-117, 7.51814806e-106, 7.52012899e-106]

    beta_baa2 = get_average_beta_baa2(harmonic_beta_transition_moment)

    assert np.allclose(np.multiply(ref_beta_baa2, 1.0e108), np.multiply(beta_baa2, 1.0e108))


def test_get_hyperraman_SI_scs():

    beta_baa2 = [8.64830523e-167, 1.03070433e-166, 6.20946154e-167, \
                 1.08293729e-166, 3.76986837e-168, 7.72341092e-159]

    incident_wn = 4389492.078840123
    wavenumbers = [4.52164948e+05, 4.48694183e+05, 1.70673951e+05, \
                   1.50648557e+05, 1.28613587e+05, 2.13542539e-03]

    T = 298

    ref_sigma = [5.23419424e-102, 6.24851564e-102, 4.29380624e-102, 7.56162397e-102, \
                 2.66281592e-103, 5.60178933e-087]

    sigma = get_hyperraman_SI_scs(incident_wn, wavenumbers, beta_baa2, T)

    assert np.allclose(np.multiply(ref_sigma, 1.0e100), np.multiply(sigma, 1.0e100))

    # Check for combotone case
    combotones = [[     0.        , 466883.50538247, 317954.17644414, 317953.1650671 ],
                  [466883.50538247,      0.        , 198506.6568093 , 198507.79874123],
                  [317954.17644414, 198506.6568093 ,      0.        ,  59021.90967648],
                  [317953.1650671 , 198507.79874123,  59021.90967648,      0.        ]]
    beta_aaa2 = [9.82889264e-109, 0.00000000e+000, 1.16453810e-107, 1.16484296e-107]

    ref_sigma = [5.90676987e-44, 0.00000000e+00, 7.94752130e-43, 7.94959759e-43]

    sigma = get_hyperraman_SI_scs(incident_wn, combotones[1], beta_aaa2, T)

    assert np.allclose(np.multiply(ref_sigma, 1.0e43), np.multiply(sigma, 1.0e43))


def test_get_hyperraman_intensities():

    incident_wn = 4389492.078840123
    harmonic_wavenumbers = [308803.62879803, 169409.60714536,  15856.18156937,  15852.54619014]
    fundamental = [298437.35928427, 170192.2063791,   26246.64170339,  26247.44867762]
    overtones = [594112.21291436, 339972.06592672,  61950.8418014,   61957.63980011]
    combotones = [[     0.        , 466883.50538247, 317954.17644414, 317953.1650671 ],
                  [466883.50538247,      0.        , 198506.6568093 , 198507.79874123],
                  [317954.17644414, 198506.6568093 ,      0.        ,  59021.90967648],
                  [317953.1650671 , 198507.79874123,  59021.90967648,      0.        ]]
    T = 298
    print_level = 0

    # Harmonic
    beta_transition_moment = anharmonicProperty(harmonic_beta_transition_moment, \
                                                np.zeros((4, 3, 3, 3)), \
                                                np.zeros((4, 3, 3, 3)), \
                                                np.zeros((4, 4, 3, 3, 3)))

    wavenumbers = anharmonicProperty(harmonic_wavenumbers, fundamental, overtones, combotones)

    # Full SI
    specifications = ['Hyper-Raman: SI complete']

    ref_harmonic_vv = [4.72797502e-41, 1.52436169e-51, 5.00298333e-40, 5.00492198e-40]
    ref_fundamental_vv = [0., 0., 0., 0.]
    ref_overtones_vv = [0., 0., 0., 0.]
    ref_combotones_vv = [[0., 0., 0., 0.],
                         [0., 0., 0., 0.],
                         [0., 0., 0., 0.],
                         [0., 0., 0., 0.]]

    ref_harmonic_hv = [5.30784316e-42, 3.70117093e-52, 1.04341670e-40, 1.04385267e-40]
    ref_fundamental_hv = [0., 0., 0., 0.]
    ref_overtones_hv = [0., 0., 0., 0.]
    ref_combotones_hv = [[0., 0., 0., 0.],
                         [0., 0., 0., 0.],
                         [0., 0., 0., 0.],
                         [0., 0., 0., 0.]]

    sigma_vv, sigma_hv = get_hyperraman_intensities(specifications, incident_wn, wavenumbers, \
                                                    beta_transition_moment, T, False, print_level)

    assert np.allclose(np.multiply(ref_harmonic_vv, 1.0e42), np.multiply(sigma_vv.harmonic, 1.0e42))
    assert np.allclose(np.multiply(ref_harmonic_hv, 1.0e42), np.multiply(sigma_hv.harmonic, 1.0e42))
    assert np.allclose(np.multiply(ref_fundamental_vv, 1.0e42), \
                       np.multiply(sigma_vv.fundamental, 1.0e42))
    assert np.allclose(np.multiply(ref_fundamental_hv, 1.0e42), \
                       np.multiply(sigma_hv.fundamental, 1.0e42))
    assert np.allclose(np.multiply(ref_overtones_vv, 1.0e42), np.multiply(sigma_vv.overtones, 1.0e42))
    assert np.allclose(np.multiply(ref_overtones_hv, 1.0e42), np.multiply(sigma_hv.overtones, 1.0e42))
    assert np.allclose(np.multiply(ref_combotones_vv, 1.0e42), np.multiply(sigma_vv.combotones, 1.0e42))
    assert np.allclose(np.multiply(ref_combotones_hv, 1.0e42), np.multiply(sigma_hv.combotones, 1.0e42))

    # Relative
    specifications = ['Hyper-Raman: Relative']
    ref_harmonic_vv = [4.51112402e-02, 1.45444606e-12, 4.77351893e-01, 4.77536866e-01]
    ref_harmonic_hv = [2.47989751e-02, 1.72923809e-12, 4.87498668e-01, 4.87702357e-01]

    sigma_vv, sigma_hv = get_hyperraman_intensities(specifications, incident_wn, wavenumbers, \
                                                    beta_transition_moment, T, False, print_level)

    assert np.allclose(ref_harmonic_vv, sigma_vv.harmonic)
    assert np.allclose(ref_harmonic_hv, sigma_hv.harmonic)
    assert np.allclose(ref_fundamental_vv, sigma_vv.fundamental)
    assert np.allclose(ref_fundamental_hv, sigma_hv.fundamental)
    assert np.allclose(ref_overtones_vv, sigma_vv.overtones)
    assert np.allclose(ref_overtones_hv, sigma_hv.overtones)
    assert np.allclose(ref_combotones_vv, sigma_vv.combotones)
    assert np.allclose(ref_combotones_hv, sigma_hv.combotones)

    # Anharmonic
    beta_transition_moment = anharmonicProperty(harmonic_beta_transition_moment, \
                                                fundamental_beta_transition_moment, \
                                                overtone_beta_transition_moment, \
                                                combotone_beta_transition_moment)

    # Full SI
    ref_harmonic_vv = [4.72797502e-41, 1.52436169e-51, 5.00298333e-40, 5.00492198e-40]
    ref_harmonic_hv = [5.30784316e-42, 3.70117093e-52, 1.04341670e-40, 1.04385267e-40]
    ref_fundamental_vv = [5.31386461e-41, 3.62266560e-52, 1.50867724e-40, 1.50796330e-40]
    ref_fundamental_hv = [5.90472177e-42, 7.72700753e-53, 3.20820416e-41, 3.20682649e-41]
    ref_overtones_vv = [6.72206696e-55, 2.64485224e-53, 1.56224002e-50, 4.36210689e-50]
    ref_overtones_hv = [1.38949051e-55, 1.04136127e-53, 5.85515466e-51, 9.62290448e-51]
    ref_combotones_vv = [[0.00000000e+00, 5.90676987e-44, 8.48717794e-52, 5.00918824e-51],
                         [5.90676987e-44, 0.00000000e+00, 7.94752133e-43, 7.94959757e-43],
                         [8.48717794e-52, 7.94752133e-43, 0.00000000e+00, 1.09191773e-50],
                         [5.00918824e-51, 7.94959757e-43, 1.09191773e-50, 0.00000000e+00]]
    ref_combotones_hv = [[0.00000000e+00, 7.82635928e-45, 1.42247218e-52, 7.39909927e-52],
                         [7.82635928e-45, 0.00000000e+00, 2.38103668e-43, 2.38173649e-43],
                         [1.42247218e-52, 2.38103668e-43, 0.00000000e+00, 2.78422001e-51],
                         [7.39909927e-52, 2.38173649e-43, 2.78422001e-51, 0.00000000e+00]]

    specifications = ['Hyper-Raman: SI complete']
    sigma_vv, sigma_hv = get_hyperraman_intensities(specifications, incident_wn, wavenumbers, \
                                                    beta_transition_moment, T, True, print_level)

    assert np.allclose(np.multiply(ref_harmonic_vv, 1.0e42), np.multiply(sigma_vv.harmonic, 1.0e42))
    assert np.allclose(np.multiply(ref_harmonic_hv, 1.0e42), np.multiply(sigma_hv.harmonic, 1.0e42))
    assert np.allclose(np.multiply(ref_fundamental_vv, 1.0e42), \
                       np.multiply(sigma_vv.fundamental, 1.0e42))
    assert np.allclose(np.multiply(ref_fundamental_hv, 1.0e42), \
                       np.multiply(sigma_hv.fundamental, 1.0e42))
    assert np.allclose(np.multiply(ref_overtones_vv, 1.0e42), np.multiply(sigma_vv.overtones, 1.0e42))
    assert np.allclose(np.multiply(ref_overtones_hv, 1.0e42), np.multiply(sigma_hv.overtones, 1.0e42))
    assert np.allclose(np.multiply(ref_combotones_vv, 1.0e42), np.multiply(sigma_vv.combotones, 1.0e42))
    assert np.allclose(np.multiply(ref_combotones_hv, 1.0e42), np.multiply(sigma_hv.combotones, 1.0e42))

    # Relative
    ref_harmonic_vv = [4.51112402e-02, 1.45444606e-12, 4.77351893e-01, 4.77536866e-01]
    ref_harmonic_hv = [2.47989751e-02, 1.72923809e-12, 4.87498668e-01, 4.87702357e-01]
    ref_fundamental_vv = [1.49769565e-01, 1.02103665e-12, 4.25215828e-01, 4.25014606e-01]
    ref_fundamental_hv = [8.42869087e-02, 1.10299114e-12, 4.57954873e-01, 4.57758218e-01]
    ref_overtones_vv = [1.13413195e-05, 4.46233494e-04, 2.63577606e-01, 7.35964819e-01]
    ref_overtones_hv = [8.97104616e-06, 6.72339965e-04, 3.78029663e-01, 6.21289026e-01]
    ref_combotones_vv = [[0.00000000e+00, 3.58251031e-02, 1.06790249e-09, 6.30118455e-09],
                         [9.99999901e-01, 0.00000000e+00, 9.99999985e-01, 9.99999980e-01],
                         [1.43685589e-08, 4.82024486e-01, 0.00000000e+00, 1.37355092e-08],
                         [8.48041799e-08, 4.82150411e-01, 1.37390976e-08, 0.00000000e+00]]
    ref_combotones_hv = [[0.00000000e+00, 1.61667008e-02, 5.97417155e-10, 3.10659856e-09],
                         [9.99999887e-01, 0.00000000e+00, 9.99999988e-01, 9.99999985e-01],
                         [1.81753989e-08, 4.91844370e-01, 0.00000000e+00, 1.16898741e-08],
                         [9.45407458e-08, 4.91988929e-01, 1.16933099e-08, 0.00000000e+00]]

    specifications = ['Hyper-Raman: Relative']
    sigma_vv, sigma_hv = get_hyperraman_intensities(specifications, incident_wn, wavenumbers, \
                                                    beta_transition_moment, T, True, print_level)

    assert np.allclose(ref_harmonic_vv, sigma_vv.harmonic)
    assert np.allclose(ref_harmonic_hv, sigma_hv.harmonic)
    assert np.allclose(ref_fundamental_vv, sigma_vv.fundamental)
    assert np.allclose(ref_fundamental_hv, sigma_hv.fundamental)
    assert np.allclose(ref_overtones_vv, sigma_vv.overtones)
    assert np.allclose(ref_overtones_hv, sigma_hv.overtones)
    assert np.allclose(ref_combotones_vv, sigma_vv.combotones)
    assert np.allclose(ref_combotones_hv, sigma_hv.combotones)

    # Different temperature

    ref_harmonic_vv = [3.69690338e-02, 1.19427475e-12, 4.81418002e-01, 4.81612965e-01]
    ref_fundamental_vv = [1.30294249e-01, 8.89960439e-13, 4.34956489e-01, 4.34749262e-01]
    ref_overtones_vv = [1.06554138e-05, 4.19247991e-04, 2.63587262e-01, 7.35982835e-01]
    ref_combotones_vv = [[0.00000000e+00, 3.58001016e-02, 1.06714077e-09, 6.29669020e-09],
                         [9.99999901e-01, 0.00000000e+00, 9.99999984e-01, 9.99999979e-01],
                         [1.43687102e-08, 4.82036992e-01, 0.00000000e+00, 1.46892592e-08],
                         [8.48050726e-08, 4.82162907e-01, 1.46930964e-08, 0.00000000e+00]]
    ref_harmonic_hv = [2.02452354e-02, 1.41448122e-12, 4.89770783e-01, 4.89983982e-01]
    ref_fundamental_hv = [7.25995594e-02, 9.51861087e-13, 4.63800547e-01, 4.63599894e-01]
    ref_overtones_hv = [8.42859333e-06, 6.31688581e-04, 3.78048103e-01, 6.21311780e-01]
    ref_combotones_hv = [[0.00000000e+00, 1.61551885e-02, 5.96991027e-10, 3.10438276e-09],
                         [9.99999887e-01, 0.00000000e+00, 9.99999987e-01, 9.99999984e-01],
                         [1.81755902e-08, 4.91850133e-01, 0.00000000e+00, 1.25015817e-08],
                         [9.45417410e-08, 4.91994679e-01, 1.25052557e-08, 0.00000000e+00]]

    T = 400
    sigma_vv, sigma_hv = get_hyperraman_intensities(specifications, incident_wn, wavenumbers, \
                                                    beta_transition_moment, T, True, print_level)

    assert np.allclose(ref_harmonic_vv, sigma_vv.harmonic)
    assert np.allclose(ref_harmonic_hv, sigma_hv.harmonic)
    assert np.allclose(ref_fundamental_vv, sigma_vv.fundamental)
    assert np.allclose(ref_fundamental_hv, sigma_hv.fundamental)
    assert np.allclose(ref_overtones_vv, sigma_vv.overtones)
    assert np.allclose(ref_overtones_hv, sigma_hv.overtones)
    assert np.allclose(ref_combotones_vv, sigma_vv.combotones)
    assert np.allclose(ref_combotones_hv, sigma_hv.combotones)
