# SpectroscPy 0.2.0
# SpectroscPy is a script package developed by and containing contributions from

    # Karen Oda Hjorth Minde Dundas
    # Magnus Ringholm
    # Yann Cornation
    # Benedicte Ofstad

# The package is released under a LGPL licence.
# For questions, please contact on karen.o.dundas@uit.no

# Test get_spectroscopy

from spectroscpy import which_spectroscopies_to_be_calculated, get_spectroscopy_indices, \
                        get_energy_derivatives, get_vibrational_frequencies_and_intensities, \
                        read_openrsp_tensor_file, rspProperty, reduced_dims, getting_the_subblocks
from spectroscpy.parameters import*
import os
import numpy as np

num_coordinates = 12
vib_degrees_of_freedom = 6

transformation_matrix = [[-1.53222366e-02, -1.53669898e-02,  3.22981029e-03,  4.17339750e-03, \
                          -4.03628935e-04, \
                          -8.54362584e-04, -1.61101304e-03,  7.11046382e-04,  7.11046382e-04, -2.69284905e-03, \
                           3.66800068e-03, -3.29058085e-03], \
                         [-4.88454619e-04, -6.26091008e-04, -2.86856327e-04,  9.19391511e-04,  5.97284973e-04, \
                           1.58823725e-02,  7.38452498e-04, -1.27867893e-03, -1.27867893e-03,  5.88922522e-03, \
                          -9.41504666e-03,  1.13685751e-02], \
                         [-4.54192168e-03, -4.36068046e-03, -1.50488550e-02, -1.51582564e-02, -3.00715954e-03, \
                           3.65915247e-04,  1.59820255e-05, -9.02891936e-04, -9.02891936e-04, -2.12606041e-03, \
                          -9.12185034e-04,  7.08979866e-04], \
                         [-8.12088770e-03,  8.23711336e-03, -1.38024626e-03,  2.89722027e-03, -3.15677778e-04, \
                          -1.33095836e-02,  5.49364785e-03,  1.32649051e-03,  1.32649051e-03, -6.70530881e-04, \
                          -4.08694673e-03,  6.11322887e-03], \
                         [-1.30330778e-02,  1.29622284e-02, -2.92412439e-03,  3.13500942e-03,  6.51580289e-04, \
                           8.70769925e-03,  2.84197888e-03, -1.02092330e-03, -1.02092330e-03, -2.09321016e-03, \
                           4.32044798e-03, -5.86075083e-03], \
                         [-4.54112268e-03,  4.34618821e-03,  1.50507111e-02, -1.51597014e-02,  3.00626318e-03, \
                          -3.80756323e-04, -5.68500175e-03, -1.42660735e-03, -1.42660735e-03, -3.56032980e-04, \
                          -1.02787417e-03,  1.07596757e-03], \
                         [ 5.11949910e-04, -5.26040912e-04,  1.11679672e-03,  1.44609648e-04, -1.70604131e-04, \
                           9.86304207e-04,  1.21121509e-03, -1.52042468e-03, -1.52042468e-03, -4.17812568e-03, \
                           3.63473835e-03, -3.45181715e-03], \
                         [ 8.20763158e-04, -8.20829942e-04,  7.77635542e-04, -7.65522454e-04, -1.48224540e-04, \
                          -4.63682087e-04,  3.98948691e-03,  1.72597918e-04,  1.72597918e-04,  4.97545185e-04, \
                          -6.13731827e-04,  2.95535526e-04], \
                         [ 2.86187731e-04, -3.45501477e-04, -6.03883325e-04,  9.55520115e-04,  4.06140688e-03, \
                          -1.10939379e-04, -1.30743314e-03,  2.44103733e-04,  2.44103733e-04, -1.52164634e-03, \
                          -6.75471940e-04,  5.12251411e-04], \
                         [ 9.65179963e-04,  9.75287920e-04, -1.23333606e-03, -5.90122822e-04,  2.15926991e-04, \
                          -9.38468797e-05, -9.57684851e-04,  4.80562923e-04,  4.80562923e-04, -2.77223894e-03, \
                           3.30819645e-03, -2.87764630e-03], \
                         [ 3.12162608e-05,  4.35411808e-05, -5.75314576e-04,  5.10058350e-04,  6.95346793e-05, \
                          -1.08571590e-03, -2.49791511e-03, -4.24312198e-03, -4.24312198e-03,  1.23211573e-03, \
                          -2.40047149e-04, -5.97994640e-05], \
                         [ 2.86126606e-04,  3.46414621e-04,  6.03766372e-04,  9.54786739e-04, -4.06135040e-03, \
                           1.11874502e-04, -1.84475586e-03,  1.97047095e-04,  1.97047095e-04, -1.35446823e-03, \
                          -6.85897044e-04,  5.46328295e-04]]

ref_dipole_gradient = \
      [[[-3.55354959e-03, -2.04896144e-03,  4.65337226e-04],
       [-1.33397761e-03,  2.30954119e-03, -1.03887733e-06],
       [ 5.38119487e-04, -9.34229819e-04,  5.72701937e-07],
       [ 2.05564456e-04,  1.21485597e-04,  5.08962420e-03],
       [ 4.87101890e-05, -8.45631462e-05,  7.73668483e-08],
       [ 4.34177613e-03, -7.53776352e-03,  4.54332299e-06]]]

ref_dipole_hessian = [[[[-2.87325048e-06,  4.86055726e-06, -2.18635873e-07],
        [ 3.54598778e-05,  2.05226436e-05,  1.38808614e-04],
        [-1.93796303e-06, -1.08971386e-06,  5.21175788e-05],
        [ 4.76752172e-06, -8.29841928e-06, -5.32517396e-08],
        [-5.30195375e-06, -3.07727026e-06, -4.77222182e-05],
        [-2.81316128e-05, -1.60741472e-05,  1.94858922e-05]],
       [[ 3.54598778e-05,  2.05226436e-05,  1.38808614e-04],
        [ 5.06742791e-06, -8.67026197e-06,  2.20460611e-07],
        [-2.66791390e-06,  4.62838975e-06,  3.77966883e-08],
        [ 1.20287990e-05,  6.96541146e-06,  7.50387425e-05],
        [ 3.01907735e-06, -5.25093099e-06, -3.35027200e-08],
        [ 6.56752802e-05, -1.14069203e-04,  8.38984044e-08]],
       [[-1.93796303e-06, -1.08971386e-06,  5.21175788e-05],
        [-2.66791390e-06,  4.62838975e-06,  3.77966883e-08],
        [-5.90093550e-08,  1.02687348e-07, -3.14928257e-11],
        [ 3.00629902e-05,  1.73214421e-05,  8.56122172e-06],
        [ 2.80813310e-06, -4.87487362e-06,  2.97279751e-09],
        [-1.32582224e-05,  2.30177049e-05, -1.37646936e-08]],
       [[ 4.76752172e-06, -8.29841928e-06, -5.32517396e-08],
        [ 1.20287990e-05,  6.96541146e-06,  7.50387425e-05],
        [ 3.00629902e-05,  1.73214421e-05,  8.56122172e-06],
        [ 1.70194309e-05, -2.95479517e-05,  1.75838613e-08],
        [-6.15872619e-07, -3.46792726e-07,  1.34587800e-05],
        [ 1.97337023e-07,  1.25457944e-07,  1.95867530e-05]],
       [[-5.30195375e-06, -3.07727026e-06, -4.77222182e-05],
        [ 3.01907735e-06, -5.25093099e-06, -3.35027200e-08],
        [ 2.80813310e-06, -4.87487362e-06,  2.97279751e-09],
        [-6.15872619e-07, -3.46792726e-07,  1.34587800e-05],
        [ 1.54687741e-06, -2.68548911e-06,  1.78832522e-09],
        [ 1.83609050e-06, -3.18768575e-06,  2.00381981e-09]],
       [[-2.81316128e-05, -1.60741472e-05,  1.94858922e-05],
        [ 6.56752802e-05, -1.14069203e-04,  8.38984044e-08],
        [-1.32582224e-05,  2.30177049e-05, -1.37646936e-08],
        [ 1.97337023e-07,  1.25457944e-07,  1.95867530e-05],
        [ 1.83609050e-06, -3.18768575e-06,  2.00381981e-09],
        [ 4.36167172e-05, -7.57235133e-05,  4.57200150e-08]]]]


ref_polarizability_gradient = \
     [[[[ 6.04610202e-02, -3.51511055e-02,  2.05867010e-02],
        [-3.51511055e-02, -6.06465271e-02, -3.58100073e-02],
        [ 2.05867010e-02, -3.58100073e-02, -8.24129166e-06]],
       [[ 1.20853575e-01,  4.99878490e-02,  7.95877729e-02],
        [ 4.99878490e-02,  6.28477170e-02,  4.58251237e-02],
        [ 7.95877729e-02,  4.58251237e-02,  6.62798317e-02]],
       [[-2.34431047e-02, -7.82917443e-03,  4.02920687e-02],
        [-7.82917443e-03, -1.43042013e-02,  2.32435281e-02],
        [ 4.02920687e-02,  2.32435281e-02,  4.87524759e-02]],
       [[-1.55313827e-02,  9.01591539e-03,  1.14149851e-02],
        [ 9.01591539e-03,  1.55074857e-02, -1.98366345e-02],
        [ 1.14149851e-02, -1.98366345e-02,  2.34202541e-05]],
       [[ 4.39674438e-03,  4.92827640e-04,  7.19780552e-03],
        [ 4.92827640e-04,  3.83525060e-03,  4.19756005e-03],
        [ 7.19780552e-03,  4.19756005e-03,  8.94657717e-02]],
       [[-2.41613385e-02, -4.31408089e-02, -1.19041040e-02],
        [-4.31408089e-02,  2.58697434e-02, -6.88886076e-03],
        [-1.19041040e-02, -6.88886076e-03, -2.42915073e-03]]], 
       [[[ 6.04610200e-02, -3.51511054e-02,  2.05867014e-02],
        [-3.51511054e-02, -6.06465273e-02, -3.58100074e-02],
        [ 2.05867014e-02, -3.58100074e-02, -8.24116238e-06]],
       [[ 1.20853575e-01,  4.99878487e-02,  7.95877733e-02],
        [ 4.99878487e-02,  6.28477171e-02,  4.58251237e-02],
        [ 7.95877733e-02,  4.58251237e-02,  6.62798311e-02]],
       [[-2.34431046e-02, -7.82917446e-03,  4.02920685e-02],
        [-7.82917446e-03, -1.43042012e-02,  2.32435281e-02],
        [ 4.02920685e-02,  2.32435281e-02,  4.87524752e-02]],
       [[-1.55313822e-02,  9.01591527e-03,  1.14149849e-02],
        [ 9.01591527e-03,  1.55074855e-02, -1.98366344e-02],
        [ 1.14149849e-02, -1.98366344e-02,  2.34202945e-05]],
       [[ 4.39674437e-03,  4.92827626e-04,  7.19780546e-03],
        [ 4.92827626e-04,  3.83525056e-03,  4.19755994e-03],
        [ 7.19780546e-03,  4.19755994e-03,  8.94657709e-02]],
       [[-2.41613390e-02, -4.31408092e-02, -1.19041042e-02],
        [-4.31408092e-02,  2.58697431e-02, -6.88886095e-03],
        [-1.19041042e-02, -6.88886095e-03, -2.42915100e-03]]], 
       [[[ 6.24435726e-02, -3.63042282e-02,  2.14401945e-02],
        [-3.63042282e-02, -6.26361513e-02, -3.72962094e-02],
        [ 2.14401945e-02, -3.72962094e-02, -9.06377656e-06]],
       [[ 1.25458452e-01,  5.20423354e-02,  8.34157406e-02],
        [ 5.20423354e-02,  6.50703814e-02,  4.80299099e-02],
        [ 8.34157406e-02,  4.80299099e-02,  6.96537637e-02]],
       [[-2.37970681e-02, -7.93763434e-03,  4.05413248e-02],
        [-7.93763434e-03, -1.45319890e-02,  2.33876188e-02],
        [ 4.05413248e-02,  2.33876188e-02,  4.94476269e-02]],
       [[-1.59236762e-02,  9.24345865e-03,  1.14034738e-02],
        [ 9.24345865e-03,  1.58998009e-02, -1.98171324e-02],
        [ 1.14034738e-02, -1.98171324e-02,  2.33748102e-05]],
       [[ 4.60303582e-03,  4.14937046e-04,  7.97344582e-03],
        [ 4.14937046e-04,  4.13298957e-03,  4.64696407e-03],
        [ 7.97344582e-03,  4.64696407e-03,  9.41809158e-02]],
       [[-2.45703252e-02, -4.36328511e-02, -1.21236513e-02],
        [-4.36328511e-02,  2.60312669e-02, -7.01564868e-03],
        [-1.21236513e-02, -7.01564868e-03, -2.52655828e-03]]]]

ref_cubic_ff = [[[[-4.17562466e-08,  1.79199293e-05,  3.45610313e-07,
         -1.74229028e-10,  8.53739971e-08,  2.22782075e-07],
        [ 1.79199293e-05,  1.38001081e-08,  6.46829447e-11,
          1.10267284e-07,  2.98126755e-11,  4.53555070e-11],
        [ 3.45610313e-07,  6.46829447e-11,  1.48190489e-09,
         -2.10866845e-06, -2.06531812e-10, -6.92117146e-11],
        [-1.74229028e-10,  1.10267284e-07, -2.10866845e-06,
          1.82444529e-09,  2.80644013e-07,  6.02636471e-08],
        [ 8.53739971e-08,  2.98126755e-11, -2.06531812e-10,
          2.80644013e-07,  3.14391979e-11,  1.06398258e-10],
        [ 2.22782075e-07,  4.53555070e-11, -6.92117146e-11,
          6.02636471e-08,  1.06398258e-10,  2.45815419e-09]],
       [[ 1.79199293e-05,  1.38001081e-08,  6.46829447e-11,
          1.10267284e-07,  2.98126755e-11,  4.53555070e-11],
        [ 1.38001081e-08,  1.79944519e-05,  2.62465812e-07,
          1.67922719e-10,  4.74057679e-08,  1.65548654e-07],
        [ 6.46829447e-11,  2.62465812e-07, -1.91109118e-06,
         -1.62785955e-09,  2.53042689e-07,  8.63395059e-08],
        [ 1.10267284e-07,  1.67922719e-10, -1.62785955e-09,
         -2.32671361e-06,  2.05487033e-10,  4.79392309e-11],
        [ 2.98126755e-11,  4.74057679e-08,  2.53042689e-07,
          2.05487033e-10, -3.69281591e-08, -1.35875708e-07],
        [ 4.53555070e-11,  1.65548654e-07,  8.63395059e-08,
          4.79392309e-11, -1.35875708e-07, -3.14911849e-06]],
       [[ 3.45610313e-07,  6.46829447e-11,  1.48190489e-09,
         -2.10866845e-06, -2.06531812e-10, -6.92117146e-11],
        [ 6.46829447e-11,  2.62465812e-07, -1.91109118e-06,
         -1.62785955e-09,  2.53042689e-07,  8.63395059e-08],
        [ 1.48190489e-09, -1.91109118e-06,  2.10057360e-07,
         -8.31036724e-13,  3.38527318e-07,  2.42349317e-07],
        [-2.10866845e-06, -1.62785955e-09, -8.31036724e-13,
          2.96758939e-07,  1.01371354e-12, -8.82819137e-13],
        [-2.06531812e-10,  2.53042689e-07,  3.38527318e-07,
          1.01371354e-12,  5.23794536e-10, -4.69691037e-08],
        [-6.92117146e-11,  8.63395059e-08,  2.42349317e-07,
         -8.82819137e-13, -4.69691037e-08,  1.02063700e-07]],
       [[-1.74229028e-10,  1.10267284e-07, -2.10866845e-06,
          1.82444529e-09,  2.80644013e-07,  6.02636471e-08],
        [ 1.10267284e-07,  1.67922719e-10, -1.62785955e-09,
         -2.32671361e-06,  2.05487033e-10,  4.79392309e-11],
        [-2.10866845e-06, -1.62785955e-09, -8.31036724e-13,
          2.96758939e-07,  1.01371354e-12, -8.82819137e-13],
        [ 1.82444529e-09, -2.32671361e-06,  2.96758939e-07,
         -5.28645907e-12,  2.91781140e-07, -2.23621585e-07],
        [ 2.80644013e-07,  2.05487033e-10,  1.01371354e-12,
          2.91781140e-07, -1.94563755e-12, -6.29754547e-13],
        [ 6.02636471e-08,  4.79392309e-11, -8.82819137e-13,
         -2.23621585e-07, -6.29754547e-13, -3.93079786e-13]],
       [[ 8.53739971e-08,  2.98126755e-11, -2.06531812e-10,
          2.80644013e-07,  3.14391979e-11,  1.06398258e-10],
        [ 2.98126755e-11,  4.74057679e-08,  2.53042689e-07,
          2.05487033e-10, -3.69281591e-08, -1.35875708e-07],
        [-2.06531812e-10,  2.53042689e-07,  3.38527318e-07,
          1.01371354e-12,  5.23794536e-10, -4.69691037e-08],
        [ 2.80644013e-07,  2.05487033e-10,  1.01371354e-12,
          2.91781140e-07, -1.94563755e-12, -6.29754547e-13],
        [ 3.14391979e-11, -3.69281591e-08,  5.23794536e-10,
         -1.94563755e-12,  9.61199706e-07,  1.67458955e-08],
        [ 1.06398258e-10, -1.35875708e-07, -4.69691037e-08,
         -6.29754547e-13,  1.67458955e-08, -2.03938528e-08]],
       [[ 2.22782075e-07,  4.53555070e-11, -6.92117146e-11,
          6.02636471e-08,  1.06398258e-10,  2.45815419e-09],
        [ 4.53555070e-11,  1.65548654e-07,  8.63395059e-08,
          4.79392309e-11, -1.35875708e-07, -3.14911849e-06],
        [-6.92117146e-11,  8.63395059e-08,  2.42349317e-07,
         -8.82819137e-13, -4.69691037e-08,  1.02063700e-07],
        [ 6.02636471e-08,  4.79392309e-11, -8.82819137e-13,
         -2.23621585e-07, -6.29754547e-13, -3.93079786e-13],
        [ 1.06398258e-10, -1.35875708e-07, -4.69691037e-08,
         -6.29754547e-13,  1.67458955e-08, -2.03938528e-08],
        [ 2.45815419e-09, -3.14911849e-06,  1.02063700e-07,
         -3.93079786e-13, -2.03938528e-08, -7.35868401e-08]]]]

ref_hyper_polarizability_gradient = \
   [[[[[ 0.17587778,  0.1214237 ,  0.08764675],
         [ 0.1214237 ,  0.05715179,  0.05813472],
         [ 0.08764675,  0.05813472,  0.11869238]],
        [[ 0.12073663,  0.05836342,  0.05814728],
         [ 0.05836342,  0.01349165,  0.02034769],
         [ 0.05814728,  0.02034769,  0.0683274 ]],
        [[ 0.08949794,  0.05549355,  0.12084899],
         [ 0.05549355,  0.02521763,  0.06959147],
         [ 0.12084899,  0.06959147,  0.09086193]]],
       [[[-0.04149689, -0.0020507664841713656, -0.01116578],
         [-0.0020507664841713656,  0.04832673,  0.004006105523379081],
         [-0.01116578,  0.004006105523379081,  0.020458075246416445]],
        [[-0.004209771173609765,  0.04709664481317306,  0.00873144669066752],
         [ 0.04709664481317306, -0.007923226252559392,  0.01092292449656617],
         [ 0.00873144669066752,  0.01092292449656617, -0.0350074 ]],
        [[-0.021367294719130257,  0.012261967604661726,  0.030426775150505916],
         [ 0.012261967604661726,  0.02108574280390427, -0.05242135],
         [ 0.030426775150505916, -0.05242135,  0.0012440929778656231]]],
       [[[ 0.03412447, -0.010090768115349858, -0.013081175990389961],
         [-0.010090768115349858, -0.0165402 ,  0.002712436142715228],
         [-0.013081175990389961,  0.002712436142715228, -0.0241171 ]],
        [[-0.00512432, -0.01367083,  0.01252866],
         [-0.01367083, -0.0255157 ,  0.01323414],
         [ 0.01252866,  0.01323414,  0.04212922]],
        [[-0.02267988,  0.0133447 ,  0.00397849],
         [ 0.0133447 ,  0.02291822, -0.00672098],
         [ 0.00397849, -0.00672098,  0.00244773]]],
       [[[-0.05680783, -0.03500606,  0.04772037],
         [-0.03500606, -0.01960747,  0.01488809],
         [ 0.04772037,  0.01488809,  0.12578315]],
        [[-0.0357841 , -0.01825808,  0.01488111],
         [-0.01825808, -0.00815919,  0.03052813],
         [ 0.01488111,  0.03052813,  0.07259695]],
        [[ 0.04357009,  0.01111845,  0.12415317],
         [ 0.01111845,  0.0307047 ,  0.07167396],
         [ 0.12415317,  0.07167396,  0.10158605]]],
       [[[ 0.00998428, -0.00389458,  0.01114878],
         [-0.00389458, -0.00297943, -0.00816592],
         [ 0.01114878, -0.00816592,  0.01372287]],
        [[-0.00227564, -0.0020425 , -0.00469732],
         [-0.0020425 , -0.00997192, -0.0111154 ],
         [-0.00469732, -0.0111154 , -0.02371718]],
        [[ 0.00842858, -0.00482389,  0.02074203],
         [-0.00482389, -0.00838164, -0.03594238],
         [ 0.02074203, -0.03594238,  0.00066928]]],
       [[[ 0.20734257, -0.06173008,  0.07437265],
         [-0.06173008, -0.06981042, -0.04407297],
         [ 0.07437265, -0.04407297,  0.0354464 ]],
        [[-0.05814701, -0.06776477, -0.04182602],
         [-0.06776477, -0.17969909, -0.07398054],
         [-0.04182602, -0.07398054, -0.06144632]],
        [[ 0.08397181, -0.04848373,  0.03428225],
         [-0.04848373, -0.08345295, -0.05937212],
         [ 0.03428225, -0.05937212, -0.00022920269966851883 ]]]], 
        [[[[ 0.15032497,  0.10886933,  0.07699703],
         [ 0.10886933,  0.04959019,  0.05005866],
         [ 0.07699703,  0.05005866,  0.10123878]],
        [[ 0.10886933,  0.04959019,  0.05005866],
         [ 0.04959019,  0.00639102,  0.01911525],
         [ 0.05005866,  0.01911525,  0.05830806]],
        [[ 0.07699703,  0.05005866,  0.10123878],
         [ 0.05005866,  0.01911525,  0.05830806],
         [ 0.10123878,  0.05830806,  0.07297953]]],
       [[[-0.02841494, -0.00516954, -0.01017341],
         [-0.00516954,  0.04020646,  0.00586081],
         [-0.01017341,  0.00586081,  0.02213489]],
        [[-0.00516954,  0.04020646,  0.00586081],
         [ 0.04020646, -0.01521534,  0.01004565],
         [ 0.00586081,  0.01004565, -0.03804587]],
        [[-0.01017341,  0.00586081,  0.02213489],
         [ 0.00586081,  0.01004565, -0.03804587],
         [ 0.02213489, -0.03804587,  0.0008717338493781023]]],
       [[[ 0.03993635, -0.00820047, -0.0179262 ],
         [-0.00820047, -0.02095479,  0.01048564],
         [-0.0179262 ,  0.01048564, -0.01141521]],
        [[-0.00820047, -0.02095479,  0.01048564],
         [-0.02095479, -0.0245979 ,  0.01810307],
         [ 0.01048564,  0.01810307,  0.01989428]],
        [[-0.0179262 ,  0.01048564, -0.01141521],
         [ 0.01048564,  0.01810307,  0.01989428],
         [-0.01141521,  0.01989428,  0.0014796063067753448]]],
       [[[-0.05381275, -0.03338372,  0.04564154],
         [-0.03338372, -0.01772252,  0.01130176],
         [ 0.04564154,  0.01130176,  0.11214544]],
        [[-0.03338372, -0.01772252,  0.01130176],
         [-0.01772252, -0.0077297 ,  0.03259076],
         [ 0.01130176,  0.03259076,  0.06473155]],
        [[ 0.04564154,  0.01130176,  0.11214544],
         [ 0.01130176,  0.03259076,  0.06473155],
         [ 0.11214544,  0.06473155,  0.09810841]]],
       [[[ 0.01011243, -0.0031187149510231295,  0.00933912],
         [-0.0031187149510231295, -0.00301255, -0.00537947],
         [ 0.00933912, -0.00537947,  0.01489873]],
        [[-0.0031187149510231295, -0.00301255, -0.00537947],
         [-0.00301255, -0.00925884, -0.00931122],
         [-0.00537947, -0.00931122, -0.02583531]],
        [[ 0.00933912, -0.00537947,  0.01489873],
         [-0.00537947, -0.00931122, -0.02583531],
         [ 0.01489873, -0.02583531,  0.0004224694615169236]]],
       [[[ 0.17437904, -0.05244226,  0.06356234],
         [-0.05244226, -0.05429012, -0.03677508],
         [ 0.06356234, -0.03677508,  0.02778907]],
        [[-0.05244226, -0.05429012, -0.03677508],
         [-0.05429012, -0.15579895, -0.06334062],
         [-0.03677508, -0.06334062, -0.04824893]],
        [[ 0.06356234, -0.03677508,  0.02778907],
         [-0.03677508, -0.06334062, -0.04824893],
         [ 0.02778907, -0.04824893, -0.00020926632642968992]]]], 
        [[[[ 0.17587778,  0.1214237 ,  0.08764675],
         [ 0.1214237 ,  0.05715179,  0.05813472],
         [ 0.08764675,  0.05813472,  0.11869238]],
        [[ 0.12073663,  0.05836342,  0.05814728],
         [ 0.05836342,  0.01349165,  0.02034769],
         [ 0.05814728,  0.02034769,  0.0683274 ]],
        [[ 0.08949794,  0.05549355,  0.12084899],
         [ 0.05549355,  0.02521763,  0.06959147],
         [ 0.12084899,  0.06959147,  0.09086193]]],
       [[[-0.04149689, -0.0020507664841713656, -0.01116578],
         [-0.0020507664841713656,  0.04832673,  0.004006105523379081],
         [-0.01116578,  0.004006105523379081,  0.020458075246416445]],
        [[-0.004209771173609765,  0.04709664481317306,  0.00873144669066752],
         [ 0.04709664481317306, -0.007923226252559392,  0.01092292449656617],
         [ 0.00873144669066752,  0.01092292449656617, -0.0350074 ]],
        [[-0.021367294719130257,  0.012261967604661726,  0.030426775150505916],
         [ 0.012261967604661726,  0.02108574280390427, -0.05242135],
         [ 0.030426775150505916, -0.05242135,  0.0012440929778656231]]],
       [[[ 0.03412447, -0.010090768115349858, -0.013081175990389961],
         [-0.010090768115349858, -0.0165402 ,  0.002712436142715228],
         [-0.013081175990389961,  0.002712436142715228, -0.0241171 ]],
        [[-0.00512432, -0.01367083,  0.01252866],
         [-0.01367083, -0.0255157 ,  0.01323414],
         [ 0.01252866,  0.01323414,  0.04212922]],
        [[-0.02267988,  0.0133447 ,  0.00397849],
         [ 0.0133447 ,  0.02291822, -0.00672098],
         [ 0.00397849, -0.00672098,  0.00244773]]],
       [[[-0.05680783, -0.03500606,  0.04772037],
         [-0.03500606, -0.01960747,  0.01488809],
         [ 0.04772037,  0.01488809,  0.12578315]],
        [[-0.0357841 , -0.01825808,  0.01488111],
         [-0.01825808, -0.00815919,  0.03052813],
         [ 0.01488111,  0.03052813,  0.07259695]],
        [[ 0.04357009,  0.01111845,  0.12415317],
         [ 0.01111845,  0.0307047 ,  0.07167396],
         [ 0.12415317,  0.07167396,  0.10158605]]],
       [[[ 0.00998428, -0.00389458,  0.01114878],
         [-0.00389458, -0.00297943, -0.00816592],
         [ 0.01114878, -0.00816592,  0.01372287]],
        [[-0.00227564, -0.0020425 , -0.00469732],
         [-0.0020425 , -0.00997192, -0.0111154 ],
         [-0.00469732, -0.0111154 , -0.02371718]],
        [[ 0.00842858, -0.00482389,  0.02074203],
         [-0.00482389, -0.00838164, -0.03594238],
         [ 0.02074203, -0.03594238,  0.00066928]]],
       [[[ 0.20734257, -0.06173008,  0.07437265],
         [-0.06173008, -0.06981042, -0.04407297],
         [ 0.07437265, -0.04407297,  0.0354464 ]],
        [[-0.05814701, -0.06776477, -0.04182602],
         [-0.06776477, -0.17969909, -0.07398054],
         [-0.04182602, -0.07398054, -0.06144632]],
        [[ 0.08397181, -0.04848373,  0.03428225],
         [-0.04848373, -0.08345295, -0.05937212],
         [ 0.03428225, -0.05937212, -0.00022920269966851883 ]]]]]

def test_get_vibrational_frequencies_and_intensities():

    data_dir = '{0}/'.format(os.path.dirname(__file__))
    tensor_file = data_dir + 'hf_H2O2.rsp_tensor'
    mol_file = data_dir + 'H2O2.mol'

    # IR SSDG a.u. and Raman SCS 45 + 7 SI
    spectroscopy_type = ['IR', 'Raman']
    specifications = ['Vib modes: 1/m', 'IR: SSDG, a.u.', 'Raman: SCS 45+7, SI units']

    ref_wavenumbers = [414847.6830235662, 414104.2828525629, 178148.02217701354, 158913.5038025385, \
                       148697.90566093015, 18299.94565115991]
    ref_fundamental = [0., 0., 0., 0., 0., 0.]
    ref_overtones = [0., 0., 0., 0., 0., 0.]
    ref_combotones = [[0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0.]]

    ref_input_wavenumers = [0.0, 2194746.0394200613]
    ref_hyperraman_freq = ['You did not ask for Hyper-Raman intensities']

    ref_harmonic_ir = [1.70424946e-05, 7.11347711e-06, 1.16235812e-06, 2.59612870e-05, \
                                   9.52361364e-09, 7.56689106e-05]
    ref_fundamental_ir = np.zeros((len(ref_harmonic_ir)))
    ref_overtone_ir = np.zeros((len(ref_harmonic_ir)))
    ref_combotone_ir = np.zeros((len(ref_harmonic_ir)))

    ref_harmonic_raman = [[2.95575992e-59, 1.19538260e-58, 1.34538908e-60,
                              2.12576034e-61, 9.81855394e-61, 6.03603025e-64],
                             [1.07304739e-56, 4.44425097e-56, 2.25236566e-56,
                              5.83041269e-57, 3.90731305e-56, 1.23657925e-55]]
    ref_hyperraman = ['You did not ask for Hyper-Raman intensities']

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, False)

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert np.allclose(ref_input_wavenumers, properties.input_raman_frequencies)
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert np.allclose(np.multiply(ref_harmonic_raman, 1.0e57), \
           np.multiply(properties.raman.harmonic, 1.0e57))
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # No outproj
    spectroscopy_type = ['IR']
    specifications = ['Vib modes: 1/m', 'IR: SSDG, a.u.']

    ref_wavenumbers = [(4.14848132e+05   +0.j), (4.14105192e+05   +0.j),
                       (1.78147036e+05   +0.j),         (1.58912905e+05   +0.j),
                       (1.48698110e+05   +0.j),         (1.83061036e+04   +0.j),
                       (4.66792909e+02   +0.j),         (2.28608186e+02   +0.j),
                       (3.76592066e+00   +0.j),         (1.78308846e+00   +0.j),
                       (2.00676340e-16   +3.27729334j), (6.82319818e-14+1114.31282585j)]
    ref_fundamental = [0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0.]
    ref_overtones = [0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0.]
    ref_combotones = [[0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0.]]

    ref_input_wavenumers = ['You did not ask for Raman intensities']

    ref_harmonic_ir = [(1.70425815e-05+0.00000000e+00j), (7.11360916e-06+0.00000000e+00j),
                          (1.16243696e-06+0.00000000e+00j), (2.59611627e-05+0.00000000e+00j),
                          (9.52928458e-09+0.00000000e+00j), (7.56686604e-05+0.00000000e+00j),
                          (4.15214594e-06+0.00000000e+00j), (1.41336245e-09+0.00000000e+00j),
                          (1.03022827e-14+0.00000000e+00j), (6.28763989e-14+0.00000000e+00j),
                          (2.10884638e-30+3.44400750e-14j), (6.41679479e-22+1.04794212e-05j)]

    ref_fundamental_ir = np.zeros((len(ref_harmonic_ir)))
    ref_overtone_ir = np.zeros((len(ref_harmonic_ir)))
    ref_combotone_ir = np.zeros((len(ref_harmonic_ir)))

    ref_harmonic_raman = ['You did not ask for Raman intensities']


    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, False, False)

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    ###################################################
    # Anharmonic, check the different frequency units
    ###################################################
    # 'Vib modes: 1/m'
    data_dir = '{0}/'.format(os.path.dirname(__file__))
    tensor_file = data_dir + 'hf_anharm_H2O2.rsp_tensor'

    spectroscopy_type = ['IR']
    specifications = ['Vib modes: 1/m', 'IR: MDAC, m**2/(s*mol)']
    ref_input_wavenumers = ['You did not ask for Raman intensities']
    ref_harmonic_raman = ['You did not ask for Raman intensities']

    # Freq GVPT2, intensities DVPT2, w/ 1-1 checks. This is default
    ref_wavenumbers = [414847.6830235662, 414104.2828525629, 178148.02217701354, 158913.5038025385, \
                       148697.90566093015, 18299.94565115991]
    ref_fundamental = [396057.43586589, 404798.68025287, 181622.38677956,
                    145526.50449582, 147425.41271963, -35751.88302422]
    ref_overtones = [ 785763.7904302,   803683.37589429,  348760.7362605,
                      280130.8044133,   293772.40096096, -133966.9388231 ]
    ref_combotones = [[     0.        , 789532.03985542, 571337.7795644,  542945.85920111,
                       543137.76510721, 345048.23894016],
                      [789532.03985542,      0.        , 581047.24654331, 550884.16599507,
                       551929.17084098, 371669.13263512],
                      [571337.7795644 , 581047.24654331,      0.        , 328411.46648376,
                       324992.05211235, 156838.92199932],
                      [542945.85920111, 550884.16599507, 328411.46648376,      0.,
                       291836.56645761,  83339.75275562],
                      [543137.76510721, 551929.17084098, 324992.05211235, 291836.56645761,
                            0.        , 110068.23014444],
                      [345048.23894016, 371669.13263512, 156838.92199932,  83339.75275562,
                       110068.23014444,      0.        ]]

    ref_harmonic_ir = [2.47758950e+13, 1.03413711e+13, 1.68980326e+12, \
                                   3.77417826e+13, 1.38451583e+10, 1.10005317e+14]
    ref_fundamental_ir = [ 2.30619415e+13,  3.26466951e+12,  3.61011365e+11,
                           4.08975052e+13,  3.44932309e+09, -7.63359156e+14]
    ref_overtone_ir = [ 8.58392415e+10,  4.23865536e+10,  1.84918690e+11,
                                    2.05189755e+10,  1.70427797e+09, -1.48001366e+14],
    ref_combotone_ir = [[0.00000000e+00, 1.64118417e+12, 1.71857707e+11 ,
                                     4.42260746e+09, 2.94685593e+11, 3.07793407e+12],
                                    [1.64118417e+12, 0.00000000e+00, 1.07872034e+10,
                                     4.42508342e+11, 2.63620441e+09, 1.76135047e+12],
                                    [1.71857707e+11, 1.07872034e+10, 0.00000000e+00,
                                     3.56424083e+11, 2.43212365e+10, 1.87416123e+11],
                                    [4.42260746e+09, 4.42508342e+11, 3.56424083e+11,
                                     0.00000000e+00, 1.13326043e+11, 2.77976060e+11],
                                    [2.94685593e+11, 2.63620441e+09, 2.43212365e+10,
                                     1.13326043e+11, 0.00000000e+00, 2.19788277e+10],
                                    [3.07793407e+12, 1.76135047e+12, 1.87416123e+11,
                                     2.77976060e+11, 2.19788277e+10, 0.00000000e+00]]

    properties = \
        get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                    specifications, 0, 298, True, \
                                                    'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # 'Vib modes: 1/cm'
    specifications = ['Vib modes: 1/cm', 'IR: MDAC, m**2/(s*mol)']

    ref_wavenumbers = [4148.47682994, 4141.04282864, 1781.48021991, 1589.13504011, \
                       1486.97905735, 182.99945262]
    ref_fundamental = [3960.57435866, 4047.98680253, 1816.2238678,
                    1455.26504496, 1474.2541272, -357.51883024]
    ref_overtones = [ 7857.6379043,   8036.83375894,  3487.6073626,
                      2801.30804413,  2937.72400961, -1339.66938823]
    ref_combotones = [[   0.        , 7895.32039855, 5713.37779564, 5429.45859201, 5431.37765107,
                       3450.4823894 ],
                      [7895.32039855,    0.        , 5810.47246543, 5508.84165995, 5519.29170841,
                       3716.69132635],
                      [5713.37779564, 5810.47246543,    0.        , 3284.11466484, 3249.92052112,
                       1568.38921999],
                      [5429.45859201, 5508.84165995, 3284.11466484,    0.        , 2918.36566458,
                       833.39752756],
                      [5431.37765107, 5519.29170841, 3249.92052112, 2918.36566458,    0.,
                       1100.68230144],
                      [3450.4823894,  3716.69132635, 1568.38921999,  833.39752756, 1100.68230144,
                       0.        ]]

    properties = \
        get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                    specifications, 0, 298, True, \
                                                    'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # 'Vib modes: 1/s'
    specifications = ['Vib modes: 1/s', 'IR: MDAC, m**2/(s*mol)']

    ref_wavenumbers = [1.24368207e+14, 1.24145341e+14, 5.34074334e+13, 4.76410700e+13, \
                       4.45785107e+13, 5.48618557e+12]
    ref_fundamental = [ 1.18735032e+14,  1.21355591e+14,  5.44490218e+13,
                     4.36277485e+13,  4.41970269e+13, -1.07181449e+13]
    ref_overtones = [ 2.35566058e+14,  2.40938215e+14,  1.04555838e+14,
                      8.39811024e+13,  8.80707502e+13, -4.01622779e+13]
    ref_combotones = [[0.00000000e+00, 2.36695751e+14, 1.71282757e+14, 1.62771074e+14,
                       1.62828606e+14, 1.03442860e+14],
                      [2.36695751e+14, 0.00000000e+00, 1.74193582e+14, 1.65150918e+14,
                       1.65464203e+14, 1.11423603e+14],
                      [1.71282757e+14, 1.74193582e+14, 0.00000000e+00, 9.84552808e+13,
                       9.74301661e+13, 4.70191259e+13],
                      [1.62771074e+14, 1.65150918e+14, 9.84552808e+13, 0.00000000e+00,
                       8.74904016e+13, 2.49846293e+13],
                      [1.62828606e+14, 1.65464203e+14, 9.74301661e+13, 8.74904016e+13,
                       0.00000000e+00, 3.29976253e+13],
                      [1.03442860e+14, 1.11423603e+14, 4.70191259e+13, 2.49846293e+13,
                       3.29976253e+13, 0.00000000e+00]]

    properties = \
        get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                    specifications, 0, 298, True, \
                                                    'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # 'Vib modes: ang. 1/s'
    specifications = ['Vib modes: ang. 1/s', 'IR: MDAC, m**2/(s*mol)']

    ref_wavenumbers = [7.81428488e+14, 7.80028181e+14, 3.35568801e+14, 2.99337671e+14, \
                       2.80095043e+14, 3.44707206e+13]
    ref_fundamental = [ 7.46034210e+14,  7.62499669e+14,  3.42113294e+14,
                     2.74121228e+14,  2.77698110e+14, -6.73440905e+13]
    ref_overtones = [ 1.48010520e+15,  1.51385945e+15,  6.56943707e+14,
                      5.27668829e+14,  5.53364844e+14, -2.52347034e+14]
    ref_combotones = [[0.00000000e+00, 1.48720326e+15, 1.07620130e+15, 1.02272082e+15,
                       1.02308230e+15, 6.49950656e+14],
                      [1.48720326e+15, 0.00000000e+00, 1.09449056e+15, 1.03767382e+15,
                       1.03964225e+15, 7.00095144e+14],
                      [1.07620130e+15, 1.09449056e+15, 0.00000000e+00, 6.18612774e+14,
                       6.12171788e+14, 2.95429881e+14],
                      [1.02272082e+15, 1.03767382e+15, 6.18612774e+14, 0.00000000e+00,
                       5.49718406e+14, 1.56983056e+14],
                      [1.02308230e+15, 1.03964225e+15, 6.12171788e+14, 5.49718406e+14,
                       0.00000000e+00, 2.07330194e+14],
                      [6.49950656e+14, 7.00095144e+14, 2.95429881e+14, 1.56983056e+14,
                       2.07330194e+14, 0.00000000e+00]]

    properties = \
        get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                    specifications, 0, 298, True, \
                                                    'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # 'Vib modes: Eh'
    specifications = ['Vib modes: Eh', 'IR: MDAC, m**2/(s*mol)']

    ref_wavenumbers = [0.01890185, 0.01886798, 0.00811702, 0.00724063, 0.00677518, 0.00083381]
    ref_fundamental = [ 0.01804571,  0.01844399,  0.00827533,  0.00663068,  0.0067172,  -0.00162898]
    ref_overtones = [ 0.03580204,  0.03661851,  0.01589071,  0.0127637,   0.01338526, -0.00610398]
    ref_combotones = [[0.        , 0.03597373, 0.02603207, 0.02473844, 0.02474718, 0.01572156],
                      [0.03597373, 0.        , 0.02647446, 0.02510013, 0.02514775, 0.01693449],
                      [0.02603207, 0.02647446, 0.        , 0.01496353, 0.01480773, 0.00714611],
                      [0.02473844, 0.02510013, 0.01496353, 0.        , 0.01329705, 0.00379724],
                      [0.02474718, 0.02514775, 0.01480773, 0.01329705, 0.        , 0.00501508],
                      [0.01572156, 0.01693449, 0.00714611, 0.00379724, 0.00501508, 0.        ]]

    properties = \
        get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                    specifications, 0, 298, True, \
                                                    'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # New molecule, CO2
    # GVPT2 frequencies, DVPT2 intensities, w/ 1-1 checks.  This is default.
    data_dir = '{0}/'.format(os.path.dirname(__file__))
    tensor_file = data_dir + 'anharm_ir_opt_CO2.rsp_tensor'
    mol_file = data_dir + 'opt_CO2.mol'

    specifications = ['Vib modes: 1/m', 'IR: MDAC, m**2/(s*mol)']

    ref_wavenumbers = [256437.00659636, 151124.33230516,  77296.77672684,  77296.64946092]
    ref_fundamental = [251541.01307331, 145222.64954562,  76854.85245673,  76854.70570159]
    ref_overtones = [500606.12530794, 297503.0186331,  154143.18970647, 157941.75784598]
    ref_combotones = [[     0.        , 398506.11685921, 327003.83987687, 327003.67295073],
                      [398506.11685921,      0.        , 225340.43060746, 225340.28098378],
                      [327003.83987687, 225340.43060746,      0.        , 153886.10372417],
                      [327003.67295073, 225340.28098378, 153886.10372417,      0.        ]]

    ref_harmonic_ir = [8.54625832e+14, 3.17635369e+03, 5.19429527e+13, 5.19426496e+13]
    ref_fundamental_ir = [8.19474208e+14, 2.84260765e+03, 5.11382110e+13, 5.11379038e+13]
    ref_overtone_ir = [  25.40036256,  218.31910831, 2636.89009248,   58.22707756]
    ref_combotone_ir = [[0.00000000e+00, 1.66393085e+13, 1.85592250e+03, 4.67037802e+02],
                                    [1.66393085e+13, 0.00000000e+00, 1.40798509e+10, 1.40806092e+10],
                                    [1.85592250e+03, 1.40798509e+10, 0.00000000e+00, 5.32281245e+00],
                                    [4.67037802e+02, 1.40806092e+10, 5.32281245e+00, 0.00000000e+00]]

    properties = \
        get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                    specifications, 0, 298, True, \
                                                    'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # VPT2
    specifications = ['Vib modes: 1/m', 'IR: NIMAC, km/mol']

    ref_fundamental = [251541.01307331, 139253.45676694,  76854.85245673,  76854.70570159]
    ref_overtones = [500606.12530794, 277967.49681247, 159027.0578499,  159027.08248123]
    ref_combotones = [[     0.        , 388738.35594889, 327003.83987687, 327003.67295073],
                      [388738.35594889,      0.        , 205805.23489789, 205804.43305208],
                      [327003.83987687, 205805.23489789,      0.        , 153886.10372417],
                      [327003.67295073, 205804.43305208, 153886.10372417,      0.        ]]
    ref_harmonic_ir = [1.04469889e+03, 3.88279063e-09, 6.34953251e+01, 6.34949545e+01]
    ref_fundamental_ir = [1.00172937e+03, 6.67988353e-09, 6.25116048e+01, 6.25112293e+01]
    ref_overtone_ir = [3.10495302e-11, 2.49350076e-10, 1.33029629e-08, 5.29699937e-09]
    ref_combotone_ir = [[0.00000000e+00, 1.98414220e+01, 2.26868894e-09, 5.70909344e-10],
                                    [1.98414220e+01, 0.00000000e+00, 1.57192012e-02, 1.57199969e-02],
                                    [2.26868894e-09, 1.57192012e-02, 0.00000000e+00, 6.50663255e-12],
                                    [5.70909344e-10, 1.57199969e-02, 6.50663255e-12, 0.00000000e+00]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, \
                                                             'Anharmonic: VPT2')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # DVPT frequencies, VPT2 intensities
    ref_fundamental = [251541.01307331, 149021.21767726,  76854.85245673,  76854.70570159]
    ref_overtones = [500606.12530794, 297503.0186331,  154143.34045028, 154143.03897053]
    ref_combotones = [[     0.        , 398506.11685921, 327003.83987687, 327003.67295073],
                      [398506.11685921,      0.        , 225340.43060746, 225340.28098378],
                      [327003.83987687, 225340.43060746,      0.        , 153886.10372417],
                      [327003.67295073, 225340.28098378, 153886.10372417,      0.        ]]
    ref_fundamental_ir = [1.00172937e+03, 7.14843567e-09, 6.25116048e+01, 6.25112293e+01]
    ref_overtone_ir = [3.10495302e-11, 2.66874369e-10, 1.28944292e-08, 5.13431780e-09]
    ref_combotone_ir = [[0.00000000e+00, 2.03399740e+01, 2.26868894e-09, 5.70909344e-10],
                                    [2.03399740e+01, 0.00000000e+00, 1.72112802e-02, 1.72122071e-02],
                                    [2.26868894e-09, 1.72112802e-02, 0.00000000e+00, 6.50663255e-12],
                                    [5.70909344e-10, 1.72122071e-02, 6.50663255e-12, 0.00000000e+00]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, \
                                                             'Anharmonic: Freq DVPT2, Int VPT2')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # DVPT2
    ref_overtone_ir = [3.10495302e-11, 2.66874369e-10, 3.22335078e-09, 6.94651632e-11]
    ref_combotone_ir = [[0.00000000e+00, 2.03399740e+01, 2.26868894e-09, 5.70909344e-10],
                                    [2.03399740e+01, 0.00000000e+00, 1.72112802e-02, 1.72122071e-02],
                                    [2.26868894e-09, 1.72112802e-02, 0.00000000e+00, 6.50663255e-12],
                                    [5.70909344e-10, 1.72122071e-02, 6.50663255e-12, 0.00000000e+00]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, \
                                                             'Anharmonic: DVPT2')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # DVPT2 w/ 1-1 checks
    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, \
                                                             'Anharmonic: DVPT2, w/ 1-1 checks')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # GVPT2 frequencies, DVPT2 intensities
    ref_fundamental = [251541.01307331, 145222.64954562,  76854.85245673,  76854.70570159]
    ref_overtones = [500606.12530794, 297503.0186331,  154143.18970647, 157941.75784598]
    ref_combotones = [[     0.        , 398506.11685921, 327003.83987687, 327003.67295073],
                      [398506.11685921,      0.        , 225340.43060746, 225340.28098378],
                      [327003.83987687, 225340.43060746,      0.        , 153886.10372417],
                      [327003.67295073, 225340.28098378, 153886.10372417,      0.        ]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, \
                                                             'Anharmonic: Freq GVPT2, Int DVPT2')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # GVPT2 frequencies, DVPT2 intensities, w/ 1-1 checks and forced removal of i,j
    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, \
                                    'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks and forced removal')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # Acetonitrile in order to cover 1-1 resonances
    tensor_file = data_dir + 'anharm_ir_opt_acetonitrile.rsp_tensor'
    mol_file = data_dir + 'opt_acetonitrile.mol'

    specifications = ['Vib modes: 1/m', 'IR: NIMAC, km/mol']

    ref_wavenumbers = [374350.8202307 , 374350.28018724, 356828.8502615 , 276617.89109862,
                       178355.46672788, 178354.36884708, 169774.22872026, 125681.69077727,
                       125681.23222229, 103782.00124461,  43811.12776865,  43810.48927511]

    ref_harmonic_ir = [3.58744632, 3.5875805,  0.44445775, 6.45233264, 6.25004736, 
                                   6.25021258, 0.21279382, 1.98399231, 1.98310685, 5.85495755,
                                   1.39590972, 1.39581924]

    # VPT2
    ref_fundamental = [364493.3616846 , 364492.81891045, 563795.81813929, 273696.22741809,
                    176054.23506227, 176052.66959639, 168033.58935985, 124250.92670532,
                    124250.636213  , 103593.91506512,  43279.22242258,  43278.90038942]
    ref_overtones = [ 724946.04530362,  724945.00144859, 1125387.2224792 ,  545504.15213915,
                      244105.88443736 , 246080.32293124,  333799.70598635,  248410.81893035,
                      248410.29815381 , 206711.42004983,   86190.31265091,   86189.84621582]
    ref_combotones = \
[[     0.        , 726519.94666005, 923895.47389995, 638100.1327461,
  539952.18392763, 538314.41981108, 531888.50284765, 488277.36450132,
  488079.51797684, 468107.58427863, 407758.58732991, 407678.66287225],
 [726519.94666005,      0.        , 923894.96864798, 638099.5949555,
  538315.72861382, 539950.06646422, 531888.02446863, 488079.00239111,
  488276.61752907, 468106.97503835, 407678.26379667, 407757.67169647],
 [923895.47389995, 923894.96864798,      0.        , 837402.01265195,
  954152.01407854, 950194.88500481, 735144.42612188, 687729.79973034,
  687729.51816804, 667344.11013336, 607043.14586921, 607042.91140754],
 [638100.1327461 , 638099.5949555 , 837402.01265195,      0.,
  449689.81109915, 449688.23337771, 441925.14087941, 397700.18171437,
  397699.94809464, 377475.31719096, 316193.93643268, 316193.56403786],
 [539952.18392763, 538315.72861382, 954152.01407854, 449689.81109915,
       0.        , 351891.81723687, 344093.61283963, 300311.94407939,
  300253.98304269, 279542.01122237, 219519.8548285 , 219494.41866362],
 [538314.41981108, 539950.06646422, 950194.88500481, 449688.23337771,
  351891.81723687,      0.        , 344092.18438366, 300252.68857401,
  300309.71129636, 279540.49793879, 219493.16746258, 219517.97940182],
 [531888.50284765, 531888.02446863, 735144.42612188, 441925.14087941,
  344093.61283963, 344092.18438366,      0.        , 291980.91981143,
  291980.54261488, 271019.56857578, 211324.71349891, 211324.24526886],
 [488277.36450132, 488079.00239111, 687729.79973034, 397700.18171437,
  300311.94407939, 300252.68857401, 291980.91981143,      0.,
  248724.837551  , 227637.60660408, 167216.64150473, 167191.14237446],
 [488079.51797684, 488276.61752907, 687729.51816804, 397699.94809464,
  300253.98304269, 300309.71129636, 291980.54261488, 248724.837551,
       0.        , 227637.28346477, 167191.1491904 , 167216.33069344],
 [468107.58427863, 468106.97503835, 667344.11013336, 377475.31719096,
  279542.01122237, 279540.49793879, 271019.56857578, 227637.60660408,
  227637.28346477,      0.        , 147688.7870742 , 147688.37789822],
 [407758.58732991, 407678.26379667, 607043.14586921, 316193.93643268,
  219519.8548285 , 219493.16746258, 211324.71349891, 167216.64150473,
  167191.1491904 , 147688.7870742 ,      0.        ,  86628.75190077],
 [407678.66287225, 407757.67169647, 607042.91140754, 316193.56403786,
  219494.41866362, 219517.97940182, 211324.24526886, 167191.14237446,
  167216.33069344, 147688.37789822,  86628.75190077,      0.        ]]

    ref_fundamental_ir = \
        [3.30840196e+00, 3.30843679e+00, 4.74706189e+05, 4.78857242e+00,
         6.22145661e+00, 6.22148085e+00, 4.64557259e-03, 1.95260620e+00,
         1.95158922e+00, 5.29491529e+00, 1.22825746e+00, 1.22821070e+00]
    ref_overtone_ir = [3.07218104e-01, 3.07262594e-01, 7.57508412e-01, 3.59841170e-01,
                                   2.55656801e+02, 2.47974750e+02, 1.63219794e-02, 6.83668917e-02,
                                   6.83030276e-02, 6.13968054e-02, 4.04276381e-01, 4.04271788e-01]
    ref_combotone_ir = \
[[0.00000000e+00, 3.44107725e-01, 1.60500155e+00, 1.67621171e-03,
  1.61317334e-01, 1.60136983e-01, 1.82111757e-01, 5.73671208e-02,
  4.31266175e-02, 1.86346920e-02, 1.62291234e-03, 3.34200686e-04],
 [3.44107725e-01, 0.00000000e+00, 1.60499173e+00, 1.67577349e-03,
  1.60142981e-01, 1.61269596e-01, 1.82134188e-01, 4.31372637e-02,
  5.73810672e-02, 1.86325206e-02, 3.35184151e-04, 1.62442525e-03],
 [1.60500155e+00, 1.60499173e+00, 0.00000000e+00, 1.82500112e-06,
  7.26813597e-01, 7.23745948e-01, 8.33242663e-01, 8.23470777e-01,
  8.23459209e-01, 1.48592379e-03, 4.39783185e-03, 4.39352254e-03],
 [1.67621171e-03, 1.67577349e-03, 1.82500112e-06, 0.00000000e+00,
  4.15685096e-03, 4.15923543e-03, 3.38492930e-02, 4.04314188e-04,
  4.04243401e-04, 7.46734610e-02, 1.20578989e-01, 1.20598393e-01],
 [1.61317334e-01, 1.60142981e-01, 7.26813597e-01, 4.15685096e-03,
  0.00000000e+00, 7.34603033e-04, 3.03870717e-01, 1.70806030e-01,
  1.17212204e-02, 1.75957332e-04, 4.18063895e-02, 8.48046996e-05],
 [1.60136983e-01, 1.61269596e-01, 7.23745948e-01, 4.15923543e-03,
  7.34603033e-04, 0.00000000e+00, 3.03865366e-01, 1.17092069e-02,
  1.70694309e-01, 1.76811931e-04, 8.42569315e-05, 4.18051600e-02],
 [1.82111757e-01, 1.82134188e-01, 8.33242663e-01, 3.38492930e-02,
  3.03870717e-01, 3.03865366e-01, 0.00000000e+00, 6.76154526e-02,
  6.76231851e-02, 1.87607809e+00, 3.78908792e-03, 3.79278306e-03],
 [5.73671208e-02, 4.31372637e-02, 8.23470777e-01, 4.04314188e-04,
  1.70806030e-01, 1.17092069e-02, 6.76154526e-02, 0.00000000e+00,
  7.13278074e-02, 4.89012150e-02, 3.93827043e-01, 1.40264883e-01],
 [4.31266175e-02, 5.73810672e-02, 8.23459209e-01, 4.04243401e-04,
  1.17212204e-02, 1.70694309e-01, 6.76231851e-02, 7.13278074e-02,
  0.00000000e+00, 4.88972196e-02, 1.40146537e-01, 3.93536866e-01],
 [1.86346920e-02, 1.86325206e-02, 1.48592379e-03, 7.46734610e-02,
  1.75957332e-04, 1.76811931e-04, 1.87607809e+00, 4.89012150e-02,
  4.88972196e-02, 0.00000000e+00, 7.62894260e-04, 7.60915779e-04],
 [1.62291234e-03, 3.35184151e-04, 4.39783185e-03, 1.20578989e-01,
  4.18063895e-02, 8.42569315e-05, 3.78908792e-03, 3.93827043e-01,
  1.40146537e-01, 7.62894260e-04, 0.00000000e+00, 6.84290332e-04],
 [3.34200686e-04, 1.62442525e-03, 4.39352254e-03, 1.20598393e-01,
  8.48046996e-05, 4.18051600e-02, 3.79278306e-03, 1.40264883e-01,
  3.93536866e-01, 7.60915779e-04, 6.84290332e-04, 0.00000000e+00]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, \
                                                             'Anharmonic: VPT2')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # DVPT2 frequencies, VPT2 intensities
    ref_fundamental = [364493.3616846 , 364492.81891045, 346962.51315305, 273696.22741809,
                    176054.23506227, 176052.66959639, 168033.58935985, 124250.92670532,
                    124250.636213  , 103593.91506512,  43279.22242258,  43278.90038942]
    ref_overtones = [724946.04530362, 724945.00144859, 691720.61250672, 545504.15213915,
                     352160.50149705, 352157.19964037, 336501.51720377, 248410.81893035,
                     248410.29815381, 206711.42004983,  86190.31265091,  86189.84621582]
    ref_combotones = \
[[     0.        , 726519.94666005, 707062.16891371, 638100.1327461,
  539952.18392763, 538314.41981108, 531888.50284765, 488277.36450132,
  488079.51797684, 468107.58427863, 407758.58732991, 407678.66287225],
 [726519.94666005,      0.        , 707061.66366174, 638099.5949555,
  538315.72861382, 539950.06646422, 531888.02446863, 488079.00239111,
  488276.61752907, 468106.97503835, 407678.26379667, 407757.67169647],
 [707062.16891371, 707061.66366174,      0.        , 620568.70766571,
  521209.47497292, 521207.8266003 , 512907.49870081, 470896.4947441,
  470896.2131818 , 450510.80514713, 390209.84088297, 390209.6064213 ],
 [638100.1327461 , 638099.5949555 , 620568.70766571,      0.,
  449689.81109915, 449688.23337771, 441925.14087941, 397700.18171437,
  397699.94809464, 377475.31719096, 316193.93643268, 316193.56403786],
 [539952.18392763, 538315.72861382, 521209.47497292, 449689.81109915,
       0.        , 351891.81723687, 344093.61283963, 300311.94407939,
  300253.98304269, 279542.01122237, 219519.8548285 , 219494.41866362],
 [538314.41981108, 539950.06646422, 521207.8266003 , 449688.23337771,
  351891.81723687,      0.        , 344092.18438366, 300252.68857401,
  300309.71129636, 279540.49793879, 219493.16746258, 219517.97940182],
 [531888.50284765, 531888.02446863, 512907.49870081, 441925.14087941,
  344093.61283963, 344092.18438366,      0.        , 291980.91981143,
  291980.54261488, 271019.56857578, 211324.71349891, 211324.24526886],
 [488277.36450132, 488079.00239111, 470896.4947441 , 397700.18171437,
  300311.94407939, 300252.68857401, 291980.91981143,      0.,
  248724.837551  , 227637.60660408, 167216.64150473, 167191.14237446],
 [488079.51797684, 488276.61752907, 470896.2131818 , 397699.94809464,
  300253.98304269, 300309.71129636, 291980.54261488, 248724.837551,
       0.        , 227637.28346477, 167191.1491904 , 167216.33069344],
 [468107.58427863, 468106.97503835, 450510.80514713, 377475.31719096,
  279542.01122237, 279540.49793879, 271019.56857578, 227637.60660408,
  227637.28346477,      0.        , 147688.7870742 , 147688.37789822],
 [407758.58732991, 407678.26379667, 390209.84088297, 316193.93643268,
  219519.8548285 , 219493.16746258, 211324.71349891, 167216.64150473,
  167191.1491904 , 147688.7870742 ,      0.        ,  86628.75190077],
 [407678.66287225, 407757.67169647, 390209.6064213 , 316193.56403786,
  219494.41866362, 219517.97940182, 211324.24526886, 167191.14237446,
  167216.33069344, 147688.37789822,  86628.75190077,      0.        ]]

    ref_fundamental_ir = [3.30840196e+00, 3.30843679e+00, 2.92136350e+05, 4.78857242e+00,
                                      6.22145661e+00, 6.22148085e+00, 4.64557259e-03, 1.95260620e+00,
                                      1.95158922e+00, 5.29491529e+00, 1.22825746e+00, 1.22821070e+00]
    ref_overtone_ir = [3.07218104e-01, 3.07262594e-01, 4.65603459e-01, 3.59841170e-01,
                                   3.68824486e+02, 3.54868249e+02, 1.64540913e-02, 6.83668917e-02,
                                   6.83030276e-02, 6.13968054e-02, 4.04276381e-01, 4.04271788e-01]
    ref_combotone_ir = \
[[0.00000000e+00, 3.44107725e-01, 1.22831631e+00, 1.67621171e-03,
  1.61317334e-01, 1.60136983e-01, 1.82111757e-01, 5.73671208e-02,
  4.31266175e-02, 1.86346920e-02, 1.62291234e-03, 3.34200686e-04],
 [3.44107725e-01, 0.00000000e+00, 1.22830858e+00, 1.67577349e-03,
  1.60142981e-01, 1.61269596e-01, 1.82134188e-01, 4.31372637e-02,
  5.73810672e-02, 1.86325206e-02, 3.35184151e-04, 1.62442525e-03],
 [1.22831631e+00, 1.22830858e+00, 0.00000000e+00, 1.35244311e-06,
  3.97024926e-01, 3.96994404e-01, 5.81350269e-01, 5.63839901e-01,
  5.63831874e-01, 1.00311775e-03, 2.82694447e-03, 2.82417382e-03],
 [1.67621171e-03, 1.67577349e-03, 1.35244311e-06, 0.00000000e+00,
  4.15685096e-03, 4.15923543e-03, 3.38492930e-02, 4.04314188e-04,
  4.04243401e-04, 7.46734610e-02, 1.20578989e-01, 1.20598393e-01],
 [1.61317334e-01, 1.60142981e-01, 3.97024926e-01, 4.15685096e-03,
  0.00000000e+00, 7.34603033e-04, 3.03870717e-01, 1.70806030e-01,
  1.17212204e-02, 1.75957332e-04, 4.18063895e-02, 8.48046996e-05],
 [1.60136983e-01, 1.61269596e-01, 3.96994404e-01, 4.15923543e-03,
  7.34603033e-04, 0.00000000e+00, 3.03865366e-01, 1.17092069e-02,
  1.70694309e-01, 1.76811931e-04, 8.42569315e-05, 4.18051600e-02],
 [1.82111757e-01, 1.82134188e-01, 5.81350269e-01, 3.38492930e-02,
  3.03870717e-01, 3.03865366e-01, 0.00000000e+00, 6.76154526e-02,
  6.76231851e-02, 1.87607809e+00, 3.78908792e-03, 3.79278306e-03],
 [5.73671208e-02, 4.31372637e-02, 5.63839901e-01, 4.04314188e-04,
  1.70806030e-01, 1.17092069e-02, 6.76154526e-02, 0.00000000e+00,
  7.13278074e-02, 4.89012150e-02, 3.93827043e-01, 1.40264883e-01],
 [4.31266175e-02, 5.73810672e-02, 5.63831874e-01, 4.04243401e-04,
  1.17212204e-02, 1.70694309e-01, 6.76231851e-02, 7.13278074e-02,
  0.00000000e+00, 4.88972196e-02, 1.40146537e-01, 3.93536866e-01],
 [1.86346920e-02, 1.86325206e-02, 1.00311775e-03, 7.46734610e-02,
  1.75957332e-04, 1.76811931e-04, 1.87607809e+00, 4.89012150e-02,
  4.88972196e-02, 0.00000000e+00, 7.62894260e-04, 7.60915779e-04],
 [1.62291234e-03, 3.35184151e-04, 2.82694447e-03, 1.20578989e-01,
  4.18063895e-02, 8.42569315e-05, 3.78908792e-03, 3.93827043e-01,
  1.40146537e-01, 7.62894260e-04, 0.00000000e+00, 6.84290332e-04],
 [3.34200686e-04, 1.62442525e-03, 2.82417382e-03, 1.20598393e-01,
  8.48046996e-05, 4.18051600e-02, 3.79278306e-03, 1.40264883e-01,
  3.93536866e-01, 7.60915779e-04, 6.84290332e-04, 0.00000000e+00]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, \
                                                             'Anharmonic: Freq DVPT2, Int VPT2')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # DVPT2
    ref_fundamental_ir = [3.30840196e+00, 3.30843679e+00, 3.99401207e-01, 4.78857242e+00,
                                      6.22145661e+00, 6.22148085e+00, 4.64557259e-03, 1.95260620e+00,
                                      1.95158922e+00, 5.29491529e+00, 1.22825746e+00, 1.22821070e+00]
    ref_overtone_ir = [0.3072181 , 0.30726259, 0.46560346, 0.35984117, 0.7127933 ,
                                   0.71284074, 0.14766137, 0.06836689, 0.06830303, 0.06139681,
                                   0.40427638, 0.40427179]
    ref_combotone_ir = \
[[0.00000000e+00, 3.44107725e-01, 1.22831631e+00, 1.67621171e-03,
  1.61317334e-01, 1.60136983e-01, 1.82111757e-01, 5.73671208e-02,
  4.31266175e-02, 1.86346920e-02, 1.62291234e-03, 3.34200686e-04],
 [3.44107725e-01, 0.00000000e+00, 1.22830858e+00, 1.67577349e-03,
  1.60142981e-01, 1.61269596e-01, 1.82134188e-01, 4.31372637e-02,
  5.73810672e-02, 1.86325206e-02, 3.35184151e-04, 1.62442525e-03],
 [1.22831631e+00, 1.22830858e+00, 0.00000000e+00, 1.35244311e-06,
  3.97024926e-01, 3.96994404e-01, 5.81350269e-01, 5.63839901e-01,
  5.63831874e-01, 1.00311775e-03, 2.82694447e-03, 2.82417382e-03],
 [1.67621171e-03, 1.67577349e-03, 1.35244311e-06, 0.00000000e+00,
  4.15685096e-03, 4.15923543e-03, 3.38492930e-02, 4.04314188e-04,
  4.04243401e-04, 7.46734610e-02, 1.20578989e-01, 1.20598393e-01],
 [1.61317334e-01, 1.60142981e-01, 3.97024926e-01, 4.15685096e-03,
  0.00000000e+00, 7.34603033e-04, 3.03870717e-01, 1.70806030e-01,
  1.17212204e-02, 1.75957332e-04, 4.18063895e-02, 8.48046996e-05],
 [1.60136983e-01, 1.61269596e-01, 3.96994404e-01, 4.15923543e-03,
  7.34603033e-04, 0.00000000e+00, 3.03865366e-01, 1.17092069e-02,
  1.70694309e-01, 1.76811931e-04, 8.42569315e-05, 4.18051600e-02],
 [1.82111757e-01, 1.82134188e-01, 5.81350269e-01, 3.38492930e-02,
  3.03870717e-01, 3.03865366e-01, 0.00000000e+00, 6.76154526e-02,
  6.76231851e-02, 1.87607809e+00, 3.78908792e-03, 3.79278306e-03],
 [5.73671208e-02, 4.31372637e-02, 5.63839901e-01, 4.04314188e-04,
  1.70806030e-01, 1.17092069e-02, 6.76154526e-02, 0.00000000e+00,
  7.13278074e-02, 4.89012150e-02, 3.93827043e-01, 1.40264883e-01],
 [4.31266175e-02, 5.73810672e-02, 5.63831874e-01, 4.04243401e-04,
  1.17212204e-02, 1.70694309e-01, 6.76231851e-02, 7.13278074e-02,
  0.00000000e+00, 4.88972196e-02, 1.40146537e-01, 3.93536866e-01],
 [1.86346920e-02, 1.86325206e-02, 1.00311775e-03, 7.46734610e-02,
  1.75957332e-04, 1.76811931e-04, 1.87607809e+00, 4.89012150e-02,
  4.88972196e-02, 0.00000000e+00, 7.62894260e-04, 7.60915779e-04],
 [1.62291234e-03, 3.35184151e-04, 2.82694447e-03, 1.20578989e-01,
  4.18063895e-02, 8.42569315e-05, 3.78908792e-03, 3.93827043e-01,
  1.40146537e-01, 7.62894260e-04, 0.00000000e+00, 6.84290332e-04],
 [3.34200686e-04, 1.62442525e-03, 2.82417382e-03, 1.20598393e-01,
  8.48046996e-05, 4.18051600e-02, 3.79278306e-03, 1.40264883e-01,
  3.93536866e-01, 7.60915779e-04, 6.84290332e-04, 0.00000000e+00]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, \
                                                             'Anharmonic: DVPT2')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # DVPT2 with 1-1 checks
    ref_fundamental_ir = [3.29893424e+00, 3.29896246e+00, 3.99401207e-01, 4.78857242e+00,
                          6.21480325e+00, 6.21483035e+00, 4.64530538e-03, 1.93644177e+00,
                          1.93543268e+00, 5.29491529e+00, 1.22473880e+00, 1.22469268e+00]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, \
                                                             'Anharmonic: DVPT2, w/ 1-1 checks')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # GVPT2 frequencies, DVPT2 intensities
    ref_fundamental = [364493.3616846 , 364492.81891045, 346774.5278423 , 273696.22741809,
                    176054.23506227, 176052.66959639, 168033.58935985, 124250.92670532,
                    124250.636213  , 103593.91506512,  43279.22242258,  43278.90038942]
    ref_overtones = [724946.04530362, 724945.00144859, 691720.61250672, 545504.15213915,
                     352158.85031311, 355996.87482312, 332851.4785157 , 248410.81893035,
                     248410.29815381, 206711.42004983,  86190.31265091,  86189.84621582]
    ref_combotones = \
[[     0.        , 726519.94666005, 707062.16891371, 638100.1327461,
  539952.18392763, 538314.41981108, 531888.50284765, 488277.36450132,
  488079.51797684, 468107.58427863, 407758.58732991, 407678.66287225],
 [726519.94666005,      0.        , 707061.66366174, 638099.5949555,
  538315.72861382, 539950.06646422, 531888.02446863, 488079.00239111,
  488276.61752907, 468106.97503835, 407678.26379667, 407757.67169647],
 [707062.16891371, 707061.66366174,      0.        , 620568.70766571,
  521209.47497292, 521207.8266003 , 512907.49870081, 470896.4947441,
  470896.2131818 , 450510.80514713, 390209.84088297, 390209.6064213 ],
 [638100.1327461 , 638099.5949555 , 620568.70766571,      0.,
  449689.81109915, 449688.23337771, 441925.14087941, 397700.18171437,
  397699.94809464, 377475.31719096, 316193.93643268, 316193.56403786],
 [539952.18392763, 538315.72861382, 521209.47497292, 449689.81109915,
       0.        , 351891.81723687, 344093.61283963, 300311.94407939,
  300253.98304269, 279542.01122237, 219519.8548285 , 219494.41866362],
 [538314.41981108, 539950.06646422, 521207.8266003 , 449688.23337771,
  351891.81723687,      0.        , 344092.18438366, 300252.68857401,
  300309.71129636, 279540.49793879, 219493.16746258, 219517.97940182],
 [531888.50284765, 531888.02446863, 512907.49870081, 441925.14087941,
  344093.61283963, 344092.18438366,      0.        , 291980.91981143,
  291980.54261488, 271019.56857578, 211324.71349891, 211324.24526886],
 [488277.36450132, 488079.00239111, 470896.4947441 , 397700.18171437,
  300311.94407939, 300252.68857401, 291980.91981143,      0.,
  248724.837551  , 227637.60660408, 167216.64150473, 167191.14237446],
 [488079.51797684, 488276.61752907, 470896.2131818 , 397699.94809464,
  300253.98304269, 300309.71129636, 291980.54261488, 248724.837551,
       0.        , 227637.28346477, 167191.1491904 , 167216.33069344],
 [468107.58427863, 468106.97503835, 450510.80514713, 377475.31719096,
  279542.01122237, 279540.49793879, 271019.56857578, 227637.60660408,
  227637.28346477,      0.        , 147688.7870742 , 147688.37789822],
 [407758.58732991, 407678.26379667, 390209.84088297, 316193.93643268,
  219519.8548285 , 219493.16746258, 211324.71349891, 167216.64150473,
  167191.1491904 , 147688.7870742 ,      0.        ,  86628.75190077],
 [407678.66287225, 407757.67169647, 390209.6064213 , 316193.56403786,
  219494.41866362, 219517.97940182, 211324.24526886, 167191.14237446,
  167216.33069344, 147688.37789822,  86628.75190077,      0.        ]]
    ref_fundamental_ir = [3.30840196e+00, 3.30843679e+00, 3.99184810e-01, 4.78857242e+00,
                                      6.22145661e+00, 6.22148085e+00, 4.64557259e-03, 1.95260620e+00,
                                      1.95158922e+00, 5.29491529e+00, 1.22825746e+00, 1.22821070e+00]
    ref_overtone_ir = [0.3072181 , 0.30726259, 0.46560346, 0.35984117, 0.71278995, 
                                   0.72061306, 0.14605968, 0.06836689, 0.06830303, 0.06139681,
                                   0.40427638, 0.40427179]
    ref_combotone_ir = \
[[0.00000000e+00, 3.44107725e-01, 1.22831631e+00, 1.67621171e-03,
  1.61317334e-01, 1.60136983e-01, 1.82111757e-01, 5.73671208e-02,
  4.31266175e-02, 1.86346920e-02, 1.62291234e-03, 3.34200686e-04],
 [3.44107725e-01, 0.00000000e+00, 1.22830858e+00, 1.67577349e-03,
  1.60142981e-01, 1.61269596e-01, 1.82134188e-01, 4.31372637e-02,
  5.73810672e-02, 1.86325206e-02, 3.35184151e-04, 1.62442525e-03],
 [1.22831631e+00, 1.22830858e+00, 0.00000000e+00, 1.35244311e-06,
  3.97024926e-01, 3.96994404e-01, 5.81350269e-01, 5.63839901e-01,
  5.63831874e-01, 1.00311775e-03, 2.82694447e-03, 2.82417382e-03],
 [1.67621171e-03, 1.67577349e-03, 1.35244311e-06, 0.00000000e+00,
  4.15685096e-03, 4.15923543e-03, 3.38492930e-02, 4.04314188e-04,
  4.04243401e-04, 7.46734610e-02, 1.20578989e-01, 1.20598393e-01],
 [1.61317334e-01, 1.60142981e-01, 3.97024926e-01, 4.15685096e-03,
  0.00000000e+00, 7.34603033e-04, 3.03870717e-01, 1.70806030e-01,
  1.17212204e-02, 1.75957332e-04, 4.18063895e-02, 8.48046996e-05],
 [1.60136983e-01, 1.61269596e-01, 3.96994404e-01, 4.15923543e-03,
  7.34603033e-04, 0.00000000e+00, 3.03865366e-01, 1.17092069e-02,
  1.70694309e-01, 1.76811931e-04, 8.42569315e-05, 4.18051600e-02],
 [1.82111757e-01, 1.82134188e-01, 5.81350269e-01, 3.38492930e-02,
  3.03870717e-01, 3.03865366e-01, 0.00000000e+00, 6.76154526e-02,
  6.76231851e-02, 1.87607809e+00, 3.78908792e-03, 3.79278306e-03],
 [5.73671208e-02, 4.31372637e-02, 5.63839901e-01, 4.04314188e-04,
  1.70806030e-01, 1.17092069e-02, 6.76154526e-02, 0.00000000e+00,
  7.13278074e-02, 4.89012150e-02, 3.93827043e-01, 1.40264883e-01],
 [4.31266175e-02, 5.73810672e-02, 5.63831874e-01, 4.04243401e-04,
  1.17212204e-02, 1.70694309e-01, 6.76231851e-02, 7.13278074e-02,
  0.00000000e+00, 4.88972196e-02, 1.40146537e-01, 3.93536866e-01],
 [1.86346920e-02, 1.86325206e-02, 1.00311775e-03, 7.46734610e-02,
  1.75957332e-04, 1.76811931e-04, 1.87607809e+00, 4.89012150e-02,
  4.88972196e-02, 0.00000000e+00, 7.62894260e-04, 7.60915779e-04],
 [1.62291234e-03, 3.35184151e-04, 2.82694447e-03, 1.20578989e-01,
  4.18063895e-02, 8.42569315e-05, 3.78908792e-03, 3.93827043e-01,
  1.40146537e-01, 7.62894260e-04, 0.00000000e+00, 6.84290332e-04],
 [3.34200686e-04, 1.62442525e-03, 2.82417382e-03, 1.20598393e-01,
  8.48046996e-05, 4.18051600e-02, 3.79278306e-03, 1.40264883e-01,
  3.93536866e-01, 7.60915779e-04, 6.84290332e-04, 0.00000000e+00]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, \
                                                             'Anharmonic: Freq GVPT2, Int DVPT2')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # GVPT2 frequencies, DVPT2 intensities, w/ 1-1 checks
    ref_fundamental = [364493.3616846 , 364492.81891045, 346774.5278423 , 273696.22741809,
                    176054.23506227, 176052.66959639, 168033.58935985, 124250.92670532,
                    124250.636213  , 103593.91506512,  43279.22242258,  43278.90038942]
    ref_overtones = [724946.04530362, 724945.00144859, 691720.61250672, 545504.15213915,
                     352158.85031311, 355996.87482312, 332851.4785157 , 248410.81893035,
                     248410.29815381, 206711.42004983,  86190.31265091,  86189.84621582]
    ref_combotones = [[     0.        , 726519.94666005, 707062.16891371, 638100.1327461,
                       539952.18392763, 538314.41981108, 531888.50284765, 488277.36450132,
                       488079.51797684, 468107.58427863, 407758.58732991, 407678.66287225],
                      [726519.94666005,      0.        , 707061.66366174, 638099.5949555,
                       538315.72861382, 539950.06646422, 531888.02446863, 488079.00239111,
                       488276.61752907, 468106.97503835, 407678.26379667, 407757.67169647],
                      [707062.16891371, 707061.66366174,      0.        , 620568.70766571,
                       521209.47497292, 521207.8266003 , 512907.49870081, 470896.4947441,
                       470896.2131818 , 450510.80514713, 390209.84088297, 390209.6064213 ],
                      [638100.1327461 , 638099.5949555 , 620568.70766571,      0.,
                       449689.81109915, 449688.23337771, 441925.14087941, 397700.18171437,
                       397699.94809464, 377475.31719096, 316193.93643268, 316193.56403786],
                      [539952.18392763, 538315.72861382, 521209.47497292, 449689.81109915,
                            0.        , 351891.81723687, 344093.61283963, 300311.94407939,
                       300253.98304269, 279542.01122237, 219519.8548285 , 219494.41866362],
                      [538314.41981108, 539950.06646422, 521207.8266003 , 449688.23337771,
                       351891.81723687,      0.        , 344092.18438366, 300252.68857401,
                       300309.71129636, 279540.49793879, 219493.16746258, 219517.97940182],
                      [531888.50284765, 531888.02446863, 512907.49870081, 441925.14087941,
                       344093.61283963, 344092.18438366,      0.        , 291980.91981143,
                       291980.54261488, 271019.56857578, 211324.71349891, 211324.24526886],
                      [488277.36450132, 488079.00239111, 470896.4947441 , 397700.18171437,
                       300311.94407939, 300252.68857401, 291980.91981143,      0.,
                       248724.837551  , 227637.60660408, 167216.64150473, 167191.14237446],
                      [488079.51797684, 488276.61752907, 470896.2131818 , 397699.94809464,
                       300253.98304269, 300309.71129636, 291980.54261488, 248724.837551,
                            0.        , 227637.28346477, 167191.1491904 , 167216.33069344],
                      [468107.58427863, 468106.97503835, 450510.80514713, 377475.31719096,
                       279542.01122237, 279540.49793879, 271019.56857578, 227637.60660408,
                       227637.28346477,      0.        , 147688.7870742 , 147688.37789822],
                      [407758.58732991, 407678.26379667, 390209.84088297, 316193.93643268,
                       219519.8548285 , 219493.16746258, 211324.71349891, 167216.64150473,
                       167191.1491904 , 147688.7870742 ,      0.        ,  86628.75190077],
                      [407678.66287225, 407757.67169647, 390209.6064213 , 316193.56403786,
                       219494.41866362, 219517.97940182, 211324.24526886, 167191.14237446,
                       167216.33069344, 147688.37789822,  86628.75190077,      0.        ]]

    ref_harmonic_ir = [3.58744632, 3.5875805,  0.44445775, 6.45233264, 6.25004736, 
                                   6.25021258, 0.21279382, 1.98399231, 1.98310685, 5.85495755,
                                   1.39590972, 1.39581924]
    ref_fundamental_ir = [3.29893424e+00, 3.29896246e+00, 3.99184810e-01, 4.78857242e+00,
                          6.21480325e+00, 6.21483035e+00, 4.64530538e-03, 1.93644177e+00,
                          1.93543268e+00, 5.29491529e+00, 1.22473880e+00, 1.22469268e+00]

    ref_overtone_ir = [0.3072181 , 0.30726259, 0.46560346, 0.35984117, 0.71278995, 
                                   0.72061306, 0.14605968, 0.06836689, 0.06830303, 0.06139681,
                                   0.40427638, 0.40427179]
    ref_combotone_ir = [[0.00000000e+00, 3.44107725e-01, 1.22831631e+00, 1.67621171e-03,
                                     1.61317334e-01, 1.60136983e-01, 1.82111757e-01, 5.73671208e-02,
                                     4.31266175e-02, 1.86346920e-02, 1.62291234e-03, 3.34200686e-04],
                                    [3.44107725e-01, 0.00000000e+00, 1.22830858e+00, 1.67577349e-03,
                                     1.60142981e-01, 1.61269596e-01, 1.82134188e-01, 4.31372637e-02,
                                     5.73810672e-02, 1.86325206e-02, 3.35184151e-04, 1.62442525e-03],
                                    [1.22831631e+00, 1.22830858e+00, 0.00000000e+00, 1.35244311e-06,
                                     3.97024926e-01, 3.96994404e-01, 5.81350269e-01, 5.63839901e-01,
                                     5.63831874e-01, 1.00311775e-03, 2.82694447e-03, 2.82417382e-03],
                                    [1.67621171e-03, 1.67577349e-03, 1.35244311e-06, 0.00000000e+00,
                                     4.15685096e-03, 4.15923543e-03, 3.38492930e-02, 4.04314188e-04,
                                     4.04243401e-04, 7.46734610e-02, 1.20578989e-01, 1.20598393e-01],
                                    [1.61317334e-01, 1.60142981e-01, 3.97024926e-01, 4.15685096e-03,
                                     0.00000000e+00, 7.34603033e-04, 3.03870717e-01, 1.70806030e-01,
                                     1.17212204e-02, 1.75957332e-04, 4.18063895e-02, 8.48046996e-05],
                                    [1.60136983e-01, 1.61269596e-01, 3.96994404e-01, 4.15923543e-03,
                                     7.34603033e-04, 0.00000000e+00, 3.03865366e-01, 1.17092069e-02,
                                     1.70694309e-01, 1.76811931e-04, 8.42569315e-05, 4.18051600e-02],
                                    [1.82111757e-01, 1.82134188e-01, 5.81350269e-01, 3.38492930e-02,
                                     3.03870717e-01, 3.03865366e-01, 0.00000000e+00, 6.76154526e-02,
                                     6.76231851e-02, 1.87607809e+00, 3.78908792e-03, 3.79278306e-03],
                                    [5.73671208e-02, 4.31372637e-02, 5.63839901e-01, 4.04314188e-04,
                                     1.70806030e-01, 1.17092069e-02, 6.76154526e-02, 0.00000000e+00,
                                     7.13278074e-02, 4.89012150e-02, 3.93827043e-01, 1.40264883e-01],
                                    [4.31266175e-02, 5.73810672e-02, 5.63831874e-01, 4.04243401e-04,
                                     1.17212204e-02, 1.70694309e-01, 6.76231851e-02, 7.13278074e-02,
                                     0.00000000e+00, 4.88972196e-02, 1.40146537e-01, 3.93536866e-01],
                                    [1.86346920e-02, 1.86325206e-02, 1.00311775e-03, 7.46734610e-02,
                                     1.75957332e-04, 1.76811931e-04, 1.87607809e+00, 4.89012150e-02,
                                     4.88972196e-02, 0.00000000e+00, 7.62894260e-04, 7.60915779e-04],
                                    [1.62291234e-03, 3.35184151e-04, 2.82694447e-03, 1.20578989e-01,
                                     4.18063895e-02, 8.42569315e-05, 3.78908792e-03, 3.93827043e-01,
                                     1.40146537e-01, 7.62894260e-04, 0.00000000e+00, 6.84290332e-04],
                                    [3.34200686e-04, 1.62442525e-03, 2.82417382e-03, 1.20598393e-01,
                                     8.48046996e-05, 4.18051600e-02, 3.79278306e-03, 1.40264883e-01,
                                     3.93536866e-01, 7.60915779e-04, 6.84290332e-04, 0.00000000e+00]]

    properties = \
        get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                    specifications, 0, 298, True, \
                                                    'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # GVPT2 frequencies, DVPT2 intensities, w/ 1-1 checks and forced removal of i,j
    properties = \
        get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                    specifications, 0, 298, True, \
                                'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks and forced removal')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    #####################################
    # Back to harmonic
    #####################################
    # No IR and Raman SCS 45 + 4 SI

    data_dir = '{0}/'.format(os.path.dirname(__file__))
    tensor_file = data_dir + 'hf_H2O2.rsp_tensor'
    mol_file = data_dir + 'H2O2.mol'

    spectroscopy_type = ['Raman']
    specifications = ['Vib modes: 1/m', 'Raman: SCS 45+4, SI units']

    ref_wavenumbers = [414847.6830235662, 414104.2828525629, 178148.02217701354,
                       158913.5038025385, 148697.90566093015, 18299.94565115991]
    ref_fundamental = [0., 0., 0., 0., 0., 0.]
    ref_overtones = [0., 0., 0., 0., 0., 0.]
    ref_combotones = [[0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0.]]

    ref_input_wavenumers = [0.0, 2194746.0394200613]
    ref_harmonic_ir = ['You did not ask for IR intensities']
    ref_fundamental_ir = ['You did not ask for IR intensities']
    ref_overtone_ir = ['You did not ask for IR intensities']
    ref_combotone_ir = ['You did not ask for IR intensities']
    ref_harmonic_raman = [[1.68900738e-59, 9.66746914e-59, 7.73170273e-61,
                              1.21472019e-61, 7.61518614e-61, 3.44927954e-64],
                             [6.13170564e-57, 3.59014087e-56, 1.29440085e-56,
                              3.33166439e-57, 3.03014380e-56, 7.06668933e-56]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, False)

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert np.allclose(ref_input_wavenumers, properties.input_raman_frequencies)
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert np.allclose(np.multiply(ref_harmonic_raman, 1.0e57), \
                       np.multiply(properties.raman.harmonic, 1.0e57))
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # No IR, Raman CPG 45+4, a.u.
    specifications = ['Vib modes: 1/m', 'Raman: CPG 45+4, a.u.']
    ref_harmonic_raman = [[0.07930255, 0.45635755, 0.0458494 , 0.01015121, 0.07769959, \
                              0.03215856], [0.08495927, 0.49572019, 0.04674917, 0.0103366 , \
                              0.08624967, 0.03293001]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, False)

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert np.allclose(ref_input_wavenumers, properties.input_raman_frequencies)
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert np.allclose(ref_harmonic_raman, properties.raman.harmonic)
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # No IR, Raman CPG 45+7, a.u.
    specifications = ['Vib modes: 1/m', 'Raman: CPG 45+7, a.u.']
    ref_harmonic_raman = [[0.13877932, 0.56428613, 0.07978227, 
                                       0.01776461, 0.10018108, 0.05627554],
                                      [0.14867857, 0.61365418, 0.08134747,
                                       0.01808905, 0.11121732, 0.05762325]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, False)

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert np.allclose(ref_input_wavenumers, properties.input_raman_frequencies)
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert np.allclose(ref_harmonic_raman, properties.raman.harmonic)
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # No IR, Raman PCPG 45+4, Å^4/amu
    specifications = ['Vib modes: 1/m', 'Raman: PCPG 45+4, Å^4/amu']
    ref_harmonic_raman = [[11.33579065, 65.23338327,  6.55387684,  1.45104975, 11.10665772, \
                              4.59686046], [12.14438241, 70.86002056,  6.68249426,  1.47755042, \
                              12.32883741, 4.70713297]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, False)

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert np.allclose(ref_input_wavenumers, properties.input_raman_frequencies)
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert np.allclose(ref_harmonic_raman, properties.raman.harmonic)
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # No IR, Raman PCPG 45+7, Å^4/amu
    specifications = ['Vib modes: 1/m', 'Raman: PCPG 45+7, Å^4/amu']
    ref_harmonic_raman = [[19.83761352, 80.66108141, 11.40436285,  2.53933706, 14.32024325, \
                              8.04422734], [21.25264742, 87.71792697, 11.6280985 ,  2.58571323, \
                              15.89780238, 8.236874  ]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, False)

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert np.allclose(ref_input_wavenumers, properties.input_raman_frequencies)
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert np.allclose(ref_harmonic_raman, properties.raman.harmonic)
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    #############################
    # Anharmonic Raman
    #############################
    data_dir = '{0}/'.format(os.path.dirname(__file__))
    # Raman CPG 45+4, a.u.
    specifications = ['Vib modes: 1/m', 'Raman: CPG 45+4, a.u.']

    # Acetonitrile
    # VPT2
    tensor_file = data_dir + 'anharm_raman_opt_cc-pvtz_acetonitrile.rsp_tensor'
    mol_file = data_dir + 'opt_cc-pvtz_acetonitrile.mol'

    ref_wavenumbers = [327001.3366191 , 327000.54349758, 319461.22821959, 259463.36970985,
                       159332.36953063, 159331.99994985, 153430.47673732, 115907.85002183,
                       115907.38886913,  95963.53572688,  42450.2083468 ,  42450.01611678]
    ref_fundamental = [314012.4329859 , 314011.5397931 , 331018.0440387 , 256465.5267761,
                       158393.22664415, 158393.99198592, 150041.13717988, 113543.31473953,
                       113542.65562776,  95712.27990253,  42042.98978772,  42042.96229373]
    ref_overtones = [622382.43837642, 622381.06278153, 658986.5113287 , 510942.8909668,
                     305232.26472312, 305244.65783224, 297348.07029445, 226811.20363388,
                     226809.79092346, 190623.68917456,  83664.32840274,  83664.35714601]
    ref_combotones = \
[[     0.        , 624705.06507988, 638494.57613781, 570460.42997652,
  471955.03714625, 470045.53246076, 463345.23626724, 426758.64359272,
  426878.70637728, 409933.29258168, 356095.00690145, 355979.95301673],
 [624705.06507988,      0.        , 638492.79207298, 570459.54133875,
  470043.90031179, 471954.84878795, 463344.37693128, 426878.52019748,
  426756.96398992, 409932.37194507, 355979.07078397, 356094.0744474 ],
 [638494.57613781, 638492.79207298,      0.        , 587433.96854641,
  509465.22280931, 509444.10224315, 484269.28799931, 444046.51491985,
  444045.88620074, 426811.17142134, 373054.9108516 , 373054.89407508],
 [570460.42997652, 570459.54133875, 587433.96854641,      0.,
  414762.66479031, 414763.43364647, 406517.23180261, 369746.20969462,
  369745.52050236, 352217.88793035, 297874.57456661, 297874.54602436],
 [471955.03714625, 470043.90031179, 509465.22280931, 414762.66479031,
       0.        , 316045.53423445, 307820.85839257, 273061.08956485,
  273038.84027123, 254026.77798938, 202082.23776885, 202080.29627501],
 [470045.53246076, 471954.84878795, 509444.10224315, 414763.43364647,
  316045.53423445,      0.        , 307821.65476166, 273041.73378068,
  273060.95780071, 254027.54944197, 202082.31494164, 202082.99586616],
 [463345.23626724, 463344.37693128, 484269.28799931, 406517.23180261,
  307820.85839257, 307821.65476166,      0.        , 263064.20459843,
  263063.48796079, 245037.34667449, 192018.76030357, 192018.7568776 ],
 [426758.64359272, 426878.52019748, 444046.51491985, 369746.20969462,
  273061.08956485, 273041.73378068, 263064.20459843,      0.,
  227174.58107739, 209062.63359026, 153788.44573249, 153852.45352726],
 [426878.70637728, 426756.96398992, 444045.88620074, 369745.52050236,
  273038.84027123, 273060.95780071, 263063.48796079, 227174.58107739,
       0.        , 209062.0012221 , 153851.99860226, 153789.2882092 ],
 [409933.29258168, 409932.37194507, 426811.17142134, 352217.88793035,
  254026.77798938, 254027.54944197, 245037.34667449, 209062.63359026,
  209062.0012221 ,      0.        , 138665.57419633, 138665.52339957],
 [356095.00690145, 355979.07078397, 373054.9108516 , 297874.57456661,
  202082.23776885, 202082.31494164, 192018.76030357, 153788.44573249,
  153851.99860226, 138665.57419633,      0.        ,  84185.97021075],
 [355979.95301673, 356094.0744474 , 373054.89407508, 297874.54602436,
  202080.29627501, 202082.99586616, 192018.7568776 , 153852.45352726,
  153789.2882092 , 138665.52339957,  84185.97021075,      0.        ]]

    ref_input_wavenumers = [219474.60394200613]
    ref_harmonic_raman = [[2.30417822e-01, 2.30424787e-01, 1.11543705e+00, 3.11979396e-01,
                           2.60290356e-02, 2.60291741e-02, 1.29769247e-02, 1.59864844e-04,
                           1.59940236e-04, 3.08172078e-02, 1.27028956e-02, 1.27029059e-02]]
    ref_fundamental_raman = [[2.44403343e-01, 2.44411088e-01, 1.69166014e+02, 3.25830352e-01,
                              1.25204678e-02, 1.25675459e-02, 1.66988271e-02, 1.78550708e-04,
                              1.78626633e-04, 2.96550645e-02, 1.27752268e-02, 1.27752155e-02]]
    ref_overtone_raman = [[6.89075239e-04, 6.89080146e-04, 2.53358864e-04, 7.56238745e-05,
                           1.39951504e+01, 1.39681558e+01, 1.31971593e-01, 4.09371175e-04,
                           4.09296479e-04, 1.72278578e-04, 1.10275848e-03, 1.10273169e-03]]
    ref_combotone_raman = \
     [[[0.00000000e+00, 4.84152902e-04, 6.03699844e-04, 2.58177749e-04,
        2.26186379e-03, 6.45491854e-04, 1.58606940e-03, 1.12092039e-03,
        2.90377109e-03, 2.43488264e-05, 1.35744125e-03, 5.32724707e-04],
       [4.84152902e-04, 0.00000000e+00, 6.03768684e-04, 2.58171461e-04,
        6.45512106e-04, 2.26178742e-03, 1.58594127e-03, 2.90389577e-03,
        1.12097521e-03, 2.43335373e-05, 5.32914571e-04, 1.35772436e-03],
       [6.03699844e-04, 6.03768684e-04, 0.00000000e+00, 1.51655243e-03,
        2.15268171e-03, 2.15236938e-03, 5.23057107e-03, 4.26917031e-04,
        4.26854071e-04, 2.50660445e-04, 2.69810211e-04, 2.69882869e-04],
       [2.58177749e-04, 2.58171461e-04, 1.51655243e-03, 0.00000000e+00,
        3.56360990e-05, 3.56407917e-05, 2.41143336e-03, 6.43771513e-05,
        6.43976379e-05, 4.65774521e-03, 4.29317651e-04, 4.29300284e-04],
       [2.26186379e-03, 6.45512106e-04, 2.15268171e-03, 3.56360990e-05,
        0.00000000e+00, 1.26551677e-02, 2.38543622e-03, 1.80984566e-03,
        3.31735003e-03, 9.39886472e-06, 5.62916702e-04, 1.74937503e-04],
       [6.45491854e-04, 2.26178742e-03, 2.15236938e-03, 3.56407917e-05,
        1.26551677e-02, 0.00000000e+00, 2.38500034e-03, 3.31744653e-03,
        1.80968580e-03, 9.38820037e-06, 1.74856602e-04, 5.62869298e-04],
       [1.58606940e-03, 1.58594127e-03, 5.23057107e-03, 2.41143336e-03,
        2.38543622e-03, 2.38500034e-03, 0.00000000e+00, 2.30244910e-03,
        2.30238125e-03, 3.32745744e-03, 4.49855475e-05, 4.49787861e-05],
       [1.12092039e-03, 2.90389577e-03, 4.26917031e-04, 6.43771513e-05,
        1.80984566e-03, 3.31744653e-03, 2.30244910e-03, 0.00000000e+00,
        1.93047159e-04, 2.35563143e-05, 4.50087106e-02, 4.55710910e-02],
       [2.90377109e-03, 1.12097521e-03, 4.26854071e-04, 6.43976379e-05,
        3.31735003e-03, 1.80968580e-03, 2.30238125e-03, 1.93047159e-04,
        0.00000000e+00, 2.35587309e-05, 4.55494274e-02, 4.49403539e-02],
       [2.43488264e-05, 2.43335373e-05, 2.50660445e-04, 4.65774521e-03,
        9.39886472e-06, 9.38820037e-06, 3.32745744e-03, 2.35563143e-05,
        2.35587309e-05, 0.00000000e+00, 1.18207962e-05, 1.18216259e-05],
       [1.35744125e-03, 5.32914571e-04, 2.69810211e-04, 4.29317651e-04,
        5.62916702e-04, 1.74856602e-04, 4.49855475e-05, 4.50087106e-02,
        4.55494274e-02, 1.18207962e-05, 0.00000000e+00, 1.42942522e-04],
       [5.32724707e-04, 1.35772436e-03, 2.69882869e-04, 4.29300284e-04,
        1.74937503e-04, 5.62869298e-04, 4.49787861e-05, 4.55710910e-02,
        4.49403539e-02, 1.18216259e-05, 1.42942522e-04, 0.00000000e+00]]]

    properties = \
        get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                    specifications, 0, 298, True, \
                                                    'Anharmonic: VPT2')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert np.allclose(ref_input_wavenumers, properties.input_raman_frequencies)
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert np.allclose(ref_harmonic_raman, properties.raman.harmonic)
    assert np.allclose(ref_fundamental_raman, properties.raman.fundamental)
    assert np.allclose(ref_overtone_raman, properties.raman.overtones)
    assert np.allclose(ref_combotone_raman, properties.raman.combotones)
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # DVPT frequencies, VPT2 intensities
    ref_fundamental = [314012.4329859 , 314011.5397931 , 306448.9812023 , 256465.5267761,
                       155036.0957592 , 155035.65184517, 150041.13717988, 113543.31473953,
                       113542.65562776,  95712.27990253,  42042.98978772,  42042.96229373]
    ref_overtones = [622382.43837642, 622381.06278153, 609848.3856559 , 510942.8909668,
                     309486.29978915, 309485.28644159, 299991.52740406, 226811.20363388,
                     226809.79092346, 190623.68917456,  83664.32840274,  83664.35714601]
    ref_combotones = \
[[     0.        , 624705.06507988, 613925.51330141, 570460.42997652,
  468597.9062613 , 466687.19232001, 463345.23626724, 426758.64359272,
  426878.70637728, 409933.29258168, 356095.00690145, 355979.95301673],
 [624705.06507988,      0.        , 613923.72923658, 570459.54133875,
  466686.76942684, 468596.5086472 , 463344.37693128, 426878.52019748,
  426756.96398992, 409932.37194507, 355979.07078397, 356094.0744474 ],
 [613925.51330141, 613923.72923658,      0.        , 562864.90571001,
  459602.43541608, 459602.08148429, 454413.31094369, 419477.45208345,
  419476.82336434, 402242.10858494, 348485.8480152 , 348485.83123869],
 [570460.42997652, 570459.54133875, 562864.90571001,      0.,
  411405.53390535, 411405.09350572, 406517.23180261, 369746.20969462,
  369745.52050236, 352217.88793035, 297874.57456661, 297874.54602436],
 [468597.9062613 , 466686.76942684, 459602.43541608, 411405.53390535,
       0.        , 309330.06320874, 304463.72750761, 268025.23186363,
  268003.30531758, 250669.64710442, 197046.70281521, 197044.4385738 ],
 [466687.19232001, 468596.5086472 , 459602.08148429, 411405.09350572,
  309330.06320874,      0.        , 304463.31462091, 268003.49154734,
  268024.17961178, 250669.20930122, 197044.0727083 , 197046.21767723],
 [463345.23626724, 463344.37693128, 454413.31094369, 406517.23180261,
  304463.72750761, 304463.31462091,      0.        , 263064.20459843,
  263063.48796079, 245037.34667449, 192018.76030357, 192018.7568776 ],
 [426758.64359272, 426878.52019748, 419477.45208345, 369746.20969462,
  268025.23186363, 268003.49154734, 263064.20459843,      0.,
  227174.58107739, 209062.63359026, 155468.34782507, 155531.18034352],
 [426878.70637728, 426756.96398992, 419476.82336434, 369745.52050236,
  268003.30531758, 268024.17961178, 263063.48796079, 227174.58107739,
       0.        , 209062.0012221 , 155530.40267096, 155467.72625738],
 [409933.29258168, 409932.37194507, 402242.10858494, 352217.88793035,
  250669.64710442, 250669.20930122, 245037.34667449, 209062.63359026,
  209062.0012221 ,      0.        , 138665.57419633, 138665.52339957],
 [356095.00690145, 355979.07078397, 348485.8480152 , 297874.57456661,
  197046.70281521, 197044.0727083 , 192018.76030357, 155468.34782507,
  155530.40267096, 138665.57419633,      0.        ,  84185.97021075],
 [355979.95301673, 356094.0744474 , 348485.83123869, 297874.54602436,
  197044.4385738 , 197046.21767723, 192018.7568776 , 155531.18034352,
  155467.72625738, 138665.52339957,  84185.97021075,      0.        ]]

    ref_fundamental_raman = [[2.44403343e-01, 2.44411088e-01, 1.56610051e+02, 3.25830352e-01,
                              1.22550975e-02, 1.23010832e-02, 1.66988271e-02, 1.78550708e-04,
                              1.78626633e-04, 2.96550645e-02, 1.27752268e-02, 1.27752155e-02]]
    ref_overtone_raman = [[6.89075239e-04, 6.89080146e-04, 2.34466854e-04, 7.56238745e-05,
                           1.41902015e+01, 1.41622092e+01, 1.33144835e-01, 4.09371175e-04,
                           4.09296479e-04, 1.72278578e-04, 1.10275848e-03, 1.10273169e-03]]
    ref_combotone_raman = \
     [[[0.00000000e+00, 4.84152902e-04, 5.80469671e-04, 2.58177749e-04,
        2.24577460e-03, 6.40880000e-04, 1.58606940e-03, 1.12092039e-03,
        2.90377109e-03, 2.43488264e-05, 1.35744125e-03, 5.32724707e-04],
       [4.84152902e-04, 0.00000000e+00, 5.80535798e-04, 2.58171461e-04,
        6.40901752e-04, 2.24569298e-03, 1.58594127e-03, 2.90389577e-03,
        1.12097521e-03, 2.43335373e-05, 5.32914571e-04, 1.35772436e-03],
       [5.80469671e-04, 5.80535798e-04, 0.00000000e+00, 1.45312356e-03,
        1.94199273e-03, 1.94178997e-03, 4.90809799e-03, 4.03295742e-04,
        4.03236231e-04, 2.36231366e-04, 2.52040752e-04, 2.52108623e-04],
       [2.58177749e-04, 2.58171461e-04, 1.45312356e-03, 0.00000000e+00,
        3.53476568e-05, 3.53522082e-05, 2.41143336e-03, 6.43771513e-05,
        6.43976379e-05, 4.65774521e-03, 4.29317651e-04, 4.29300284e-04],
       [2.24577460e-03, 6.40901752e-04, 1.94199273e-03, 3.53476568e-05,
        0.00000000e+00, 1.23862653e-02, 2.35942037e-03, 1.77646805e-03,
        3.25616961e-03, 9.27465254e-06, 5.48889805e-04, 1.70578046e-04],
       [6.40880000e-04, 2.24569298e-03, 1.94178997e-03, 3.53522082e-05,
        1.23862653e-02, 0.00000000e+00, 2.35897994e-03, 3.25623208e-03,
        1.77630502e-03, 9.26408482e-06, 1.70497141e-04, 5.48840172e-04],
       [1.58606940e-03, 1.58594127e-03, 4.90809799e-03, 2.41143336e-03,
        2.35942037e-03, 2.35897994e-03, 0.00000000e+00, 2.30244910e-03,
        2.30238125e-03, 3.32745744e-03, 4.49855475e-05, 4.49787861e-05],
       [1.12092039e-03, 2.90389577e-03, 4.03295742e-04, 6.43771513e-05,
        1.77646805e-03, 3.25623208e-03, 2.30244910e-03, 0.00000000e+00,
        1.93047159e-04, 2.35563143e-05, 4.55003615e-02, 4.60683299e-02],
       [2.90377109e-03, 1.12097521e-03, 4.03236231e-04, 6.43976379e-05,
        3.25616961e-03, 1.77630502e-03, 2.30238125e-03, 1.93047159e-04,
        0.00000000e+00, 2.35587309e-05, 4.60463357e-02, 4.54308276e-02],
       [2.43488264e-05, 2.43335373e-05, 2.36231366e-04, 4.65774521e-03,
        9.27465254e-06, 9.26408482e-06, 3.32745744e-03, 2.35563143e-05,
        2.35587309e-05, 0.00000000e+00, 1.18207962e-05, 1.18216259e-05],
       [1.35744125e-03, 5.32914571e-04, 2.52040752e-04, 4.29317651e-04,
        5.48889805e-04, 1.70497141e-04, 4.49855475e-05, 4.55003615e-02,
        4.60463357e-02, 1.18207962e-05, 0.00000000e+00, 1.42942522e-04],
       [5.32724707e-04, 1.35772436e-03, 2.52108623e-04, 4.29300284e-04,
        1.70578046e-04, 5.48840172e-04, 4.49787861e-05, 4.60683299e-02,
        4.54308276e-02, 1.18216259e-05, 1.42942522e-04, 0.00000000e+00]]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, \
                                                             'Anharmonic: Freq DVPT2, Int VPT2')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert np.allclose(ref_input_wavenumers, properties.input_raman_frequencies)
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert np.allclose(ref_harmonic_raman, properties.raman.harmonic)
    assert np.allclose(ref_fundamental_raman, properties.raman.fundamental)
    assert np.allclose(ref_overtone_raman, properties.raman.overtones)
    assert np.allclose(ref_combotone_raman, properties.raman.combotones)
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # DVPT2
    ref_fundamental_raman = [[2.44403343e-01, 2.44411088e-01, 1.21265336e+00, 3.25830352e-01,
                              5.82250798e+04, 5.82752047e+04, 1.66988271e-02, 1.78550708e-04,
                              1.78626633e-04, 2.96550645e-02, 1.27752268e-02, 1.27752155e-02]]
    ref_overtone_raman = [[6.89075239e-04, 6.89080146e-04, 2.34466854e-04, 7.56238745e-05,
                           1.46759084e-02, 1.46765793e-02, 1.39355209e-02, 4.09371175e-04,
                           4.09296479e-04, 1.72278578e-04, 1.10275848e-03, 1.10273169e-03]]
    ref_combotone_raman = \
     [[[0.00000000e+00, 4.84152902e-04, 5.80469671e-04, 2.58177749e-04,
        2.24577460e-03, 6.40880000e-04, 1.58606940e-03, 1.12092039e-03,
        2.90377109e-03, 2.43488264e-05, 1.35744125e-03, 5.32724707e-04],
       [4.84152902e-04, 0.00000000e+00, 5.80535798e-04, 2.58171461e-04,
        6.40901752e-04, 2.24569298e-03, 1.58594127e-03, 2.90389577e-03,
        1.12097521e-03, 2.43335373e-05, 5.32914571e-04, 1.35772436e-03],
       [5.80469671e-04, 5.80535798e-04, 0.00000000e+00, 1.45312356e-03,
        1.94199273e-03, 1.94178997e-03, 4.90809799e-03, 4.03295742e-04,
        4.03236231e-04, 2.36231366e-04, 2.52040752e-04, 2.52108623e-04],
       [2.58177749e-04, 2.58171461e-04, 1.45312356e-03, 0.00000000e+00,
        3.53476568e-05, 3.53522082e-05, 2.41143336e-03, 6.43771513e-05,
        6.43976379e-05, 4.65774521e-03, 4.29317651e-04, 4.29300284e-04],
       [2.24577460e-03, 6.40901752e-04, 1.94199273e-03, 3.53476568e-05,
        0.00000000e+00, 1.23862653e-02, 2.35942037e-03, 1.77646805e-03,
        3.25616961e-03, 9.27465254e-06, 5.48889805e-04, 1.70578046e-04],
       [6.40880000e-04, 2.24569298e-03, 1.94178997e-03, 3.53522082e-05,
        1.23862653e-02, 0.00000000e+00, 2.35897994e-03, 3.25623208e-03,
        1.77630502e-03, 9.26408482e-06, 1.70497141e-04, 5.48840172e-04],
       [1.58606940e-03, 1.58594127e-03, 4.90809799e-03, 2.41143336e-03,
        2.35942037e-03, 2.35897994e-03, 0.00000000e+00, 2.30244910e-03,
        2.30238125e-03, 3.32745744e-03, 4.49855475e-05, 4.49787861e-05],
       [1.12092039e-03, 2.90389577e-03, 4.03295742e-04, 6.43771513e-05,
        1.77646805e-03, 3.25623208e-03, 2.30244910e-03, 0.00000000e+00,
        1.93047159e-04, 2.35563143e-05, 1.33687041e-03, 1.94270423e-03],
       [2.90377109e-03, 1.12097521e-03, 4.03236231e-04, 6.43976379e-05,
        3.25616961e-03, 1.77630502e-03, 2.30238125e-03, 1.93047159e-04,
        0.00000000e+00, 2.35587309e-05, 1.94313277e-03, 1.33545234e-03],
       [2.43488264e-05, 2.43335373e-05, 2.36231366e-04, 4.65774521e-03,
        9.27465254e-06, 9.26408482e-06, 3.32745744e-03, 2.35563143e-05,
        2.35587309e-05, 0.00000000e+00, 1.18207962e-05, 1.18216259e-05],
       [1.35744125e-03, 5.32914571e-04, 2.52040752e-04, 4.29317651e-04,
        5.48889805e-04, 1.70497141e-04, 4.49855475e-05, 1.33687041e-03,
        1.94313277e-03, 1.18207962e-05, 0.00000000e+00, 1.42942522e-04],
       [5.32724707e-04, 1.35772436e-03, 2.52108623e-04, 4.29300284e-04,
        1.70578046e-04, 5.48840172e-04, 4.49787861e-05, 1.94270423e-03,
        1.33545234e-03, 1.18216259e-05, 1.42942522e-04, 0.00000000e+00]]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, \
                                                             'Anharmonic: DVPT2')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert np.allclose(ref_input_wavenumers, properties.input_raman_frequencies)
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert np.allclose(ref_harmonic_raman, properties.raman.harmonic)
    assert np.allclose(ref_fundamental_raman, properties.raman.fundamental)
    assert np.allclose(ref_overtone_raman, properties.raman.overtones)
    assert np.allclose(ref_combotone_raman, properties.raman.combotones)
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # DVPT2 w/ 1-1 checks
    ref_fundamental_raman = [[2.44346107e-01, 2.44353836e-01, 1.21265336e+00, 3.25830352e-01,
                              2.83898742e-02, 2.83876638e-02, 1.66988271e-02, 1.78405618e-04,
                              1.78481842e-04, 2.96550645e-02, 1.24172761e-02, 1.24172725e-02]]
    ref_overtone_raman = [[6.89075239e-04, 6.89080146e-04, 2.34466854e-04, 7.56238745e-05,
                           1.46759084e-02, 1.46765793e-02, 1.39355209e-02, 4.09371175e-04,
                           4.09296479e-04, 1.72278578e-04, 1.10275848e-03, 1.10273169e-03]]
    ref_combotone_raman = \
     [[[0.00000000e+00, 4.84152902e-04, 5.80469671e-04, 2.58177749e-04,
        2.24577460e-03, 6.40880000e-04, 1.58606940e-03, 1.12092039e-03,
        2.90377109e-03, 2.43488264e-05, 1.35744125e-03, 5.32724707e-04],
       [4.84152902e-04, 0.00000000e+00, 5.80535798e-04, 2.58171461e-04,
        6.40901752e-04, 2.24569298e-03, 1.58594127e-03, 2.90389577e-03,
        1.12097521e-03, 2.43335373e-05, 5.32914571e-04, 1.35772436e-03],
       [5.80469671e-04, 5.80535798e-04, 0.00000000e+00, 1.45312356e-03,
        1.94199273e-03, 1.94178997e-03, 4.90809799e-03, 4.03295742e-04,
        4.03236231e-04, 2.36231366e-04, 2.52040752e-04, 2.52108623e-04],
       [2.58177749e-04, 2.58171461e-04, 1.45312356e-03, 0.00000000e+00,
        3.53476568e-05, 3.53522082e-05, 2.41143336e-03, 6.43771513e-05,
        6.43976379e-05, 4.65774521e-03, 4.29317651e-04, 4.29300284e-04],
       [2.24577460e-03, 6.40901752e-04, 1.94199273e-03, 3.53476568e-05,
        0.00000000e+00, 1.23862653e-02, 2.35942037e-03, 1.77646805e-03,
        3.25616961e-03, 9.27465254e-06, 5.48889805e-04, 1.70578046e-04],
       [6.40880000e-04, 2.24569298e-03, 1.94178997e-03, 3.53522082e-05,
        1.23862653e-02, 0.00000000e+00, 2.35897994e-03, 3.25623208e-03,
        1.77630502e-03, 9.26408482e-06, 1.70497141e-04, 5.48840172e-04],
       [1.58606940e-03, 1.58594127e-03, 4.90809799e-03, 2.41143336e-03,
        2.35942037e-03, 2.35897994e-03, 0.00000000e+00, 2.30244910e-03,
        2.30238125e-03, 3.32745744e-03, 4.49855475e-05, 4.49787861e-05],
       [1.12092039e-03, 2.90389577e-03, 4.03295742e-04, 6.43771513e-05,
        1.77646805e-03, 3.25623208e-03, 2.30244910e-03, 0.00000000e+00,
        1.93047159e-04, 2.35563143e-05, 1.33687041e-03, 1.94270423e-03],
       [2.90377109e-03, 1.12097521e-03, 4.03236231e-04, 6.43976379e-05,
        3.25616961e-03, 1.77630502e-03, 2.30238125e-03, 1.93047159e-04,
        0.00000000e+00, 2.35587309e-05, 1.94313277e-03, 1.33545234e-03],
       [2.43488264e-05, 2.43335373e-05, 2.36231366e-04, 4.65774521e-03,
        9.27465254e-06, 9.26408482e-06, 3.32745744e-03, 2.35563143e-05,
        2.35587309e-05, 0.00000000e+00, 1.18207962e-05, 1.18216259e-05],
       [1.35744125e-03, 5.32914571e-04, 2.52040752e-04, 4.29317651e-04,
        5.48889805e-04, 1.70497141e-04, 4.49855475e-05, 1.33687041e-03,
        1.94313277e-03, 1.18207962e-05, 0.00000000e+00, 1.42942522e-04],
       [5.32724707e-04, 1.35772436e-03, 2.52108623e-04, 4.29300284e-04,
        1.70578046e-04, 5.48840172e-04, 4.49787861e-05, 1.94270423e-03,
        1.33545234e-03, 1.18216259e-05, 1.42942522e-04, 0.00000000e+00]]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, \
                                                             'Anharmonic: DVPT2, w/ 1-1 checks')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert np.allclose(ref_input_wavenumers, properties.input_raman_frequencies)
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert np.allclose(ref_harmonic_raman, properties.raman.harmonic)
    assert np.allclose(ref_fundamental_raman, properties.raman.fundamental)
    assert np.allclose(ref_overtone_raman, properties.raman.overtones)
    assert np.allclose(ref_combotone_raman, properties.raman.combotones)
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # GVPT2 frequencies, DVPT2 intensities
    ref_fundamental = [314012.4329859 , 314011.5397931 , 313398.59960535, 256465.5267761,
                       153457.74707921, 153430.12464183, 150041.13717988, 113543.31473953,
                       113542.65562776,  95712.27990253,  42042.98978772,  42042.96229373]
    ref_overtones = [622382.43837642, 622381.06278153, 609848.3856559 , 510942.8909668,
                     309485.79310337, 306284.42754267, 296243.2745857 , 226811.20363388,
                     226809.79092346, 190623.68917456,  83664.32840274,  83664.35714601]
    ref_combotones = \
[[     0.       ,  624705.06507988, 613925.51330141, 570460.42997652,
  468597.9062613,  466687.19232001, 463345.23626724, 426758.64359272,
  426878.70637728, 409933.29258168, 356095.00690145, 355979.95301673],
 [624705.06507988,      0.        , 613923.72923658, 570459.54133875,
  466686.76942684, 468596.5086472 , 463344.37693128, 426878.52019748,
  426756.96398992, 409932.37194507, 355979.07078397, 356094.0744474 ],
 [613925.51330141, 613923.72923658,      0.        , 562864.90571001,
  459602.43541608, 459602.08148429, 454413.31094369, 419477.45208345,
  419476.82336434, 402242.10858494, 348485.8480152 , 348485.83123869],
 [570460.42997652, 570459.54133875, 562864.90571001,      0.,
  411405.53390535, 411405.09350572, 406517.23180261, 369746.20969462,
  369745.52050236, 352217.88793035, 297874.57456661, 297874.54602436],
 [468597.9062613 , 466686.76942684, 459602.43541608, 411405.53390535,
       0.        , 309330.06320874, 304463.72750761, 268025.23186363,
  268003.30531758, 250669.64710442, 197046.70281521, 197044.4385738 ],
 [466687.19232001, 468596.5086472 , 459602.08148429, 411405.09350572,
  309330.06320874,      0.        , 304463.31462091, 268003.49154734,
  268024.17961178, 250669.20930122, 197044.0727083 , 197046.21767723],
 [463345.23626724, 463344.37693128, 454413.31094369, 406517.23180261,
  304463.72750761, 304463.31462091,      0.        , 263064.20459843,
  263063.48796079, 245037.34667449, 192018.76030357, 192018.7568776 ],
 [426758.64359272, 426878.52019748, 419477.45208345, 369746.20969462,
  268025.23186363, 268003.49154734, 263064.20459843,      0.,
  227174.58107739, 209062.63359026, 155468.03699719, 157109.14019383],
 [426878.70637728, 426756.96398992, 419476.82336434, 369745.52050236,
  268003.30531758, 268024.17961178, 263063.48796079, 227174.58107739,
       0.        , 209062.0012221 , 155530.79150064, 157073.5642886 ],
 [409933.29258168, 409932.37194507, 402242.10858494, 352217.88793035,
  250669.64710442, 250669.20930122, 245037.34667449, 209062.63359026,
  209062.0012221 ,      0.        , 138665.57419633, 138665.52339957],
 [356095.00690145, 355979.07078397, 348485.8480152 , 297874.57456661,
  197046.70281521, 197044.0727083 , 192018.76030357, 155468.03699719,
  155530.79150064, 138665.57419633,      0.        ,  84185.97021075],
 [355979.95301673, 356094.0744474 , 348485.83123869, 297874.54602436,
  197044.4385738 , 197046.21767723, 192018.7568776 , 157109.14019383,
  157073.5642886 , 138665.52339957,  84185.97021075,      0.        ]]

    ref_fundamental_raman = [[2.44403343e-01, 2.44411088e-01, 1.24015379e+00, 3.25830352e-01,
                              5.76323180e+04, 5.76717150e+04, 1.66988271e-02, 1.78550708e-04,
                              1.78626633e-04, 2.96550645e-02, 1.27752268e-02, 1.27752155e-02]]
    ref_overtone_raman = [[6.89075239e-04, 6.89080146e-04, 2.34466854e-04, 7.56238745e-05,
                           1.46758844e-02, 1.45247864e-02, 1.37614031e-02, 4.09371175e-04,
                           4.09296479e-04, 1.72278578e-04, 1.10275848e-03, 1.10273169e-03]]
    ref_combotone_raman = \
     [[[0.00000000e+00, 4.84152902e-04, 5.80469671e-04, 2.58177749e-04,
        2.24577460e-03, 6.40880000e-04, 1.58606940e-03, 1.12092039e-03,
        2.90377109e-03, 2.43488264e-05, 1.35744125e-03, 5.32724707e-04],
       [4.84152902e-04, 0.00000000e+00, 5.80535798e-04, 2.58171461e-04,
        6.40901752e-04, 2.24569298e-03, 1.58594127e-03, 2.90389577e-03,
        1.12097521e-03, 2.43335373e-05, 5.32914571e-04, 1.35772436e-03],
       [5.80469671e-04, 5.80535798e-04, 0.00000000e+00, 1.45312356e-03,
        1.94199273e-03, 1.94178997e-03, 4.90809799e-03, 4.03295742e-04,
        4.03236231e-04, 2.36231366e-04, 2.52040752e-04, 2.52108623e-04],
       [2.58177749e-04, 2.58171461e-04, 1.45312356e-03, 0.00000000e+00,
        3.53476568e-05, 3.53522082e-05, 2.41143336e-03, 6.43771513e-05,
        6.43976379e-05, 4.65774521e-03, 4.29317651e-04, 4.29300284e-04],
       [2.24577460e-03, 6.40901752e-04, 1.94199273e-03, 3.53476568e-05,
        0.00000000e+00, 1.23862653e-02, 2.35942037e-03, 1.77646805e-03,
        3.25616961e-03, 9.27465254e-06, 5.48889805e-04, 1.70578046e-04],
       [6.40880000e-04, 2.24569298e-03, 1.94178997e-03, 3.53522082e-05,
        1.23862653e-02, 0.00000000e+00, 2.35897994e-03, 3.25623208e-03,
        1.77630502e-03, 9.26408482e-06, 1.70497141e-04, 5.48840172e-04],
       [1.58606940e-03, 1.58594127e-03, 4.90809799e-03, 2.41143336e-03,
        2.35942037e-03, 2.35897994e-03, 0.00000000e+00, 2.30244910e-03,
        2.30238125e-03, 3.32745744e-03, 4.49855475e-05, 4.49787861e-05],
       [1.12092039e-03, 2.90389577e-03, 4.03295742e-04, 6.43771513e-05,
        1.77646805e-03, 3.25623208e-03, 2.30244910e-03, 0.00000000e+00,
        1.93047159e-04, 2.35563143e-05, 1.33686774e-03, 1.96241416e-03],
       [2.90377109e-03, 1.12097521e-03, 4.03236231e-04, 6.43976379e-05,
        3.25616961e-03, 1.77630502e-03, 2.30238125e-03, 1.93047159e-04,
        0.00000000e+00, 2.35587309e-05, 1.94313763e-03, 1.34924633e-03],
       [2.43488264e-05, 2.43335373e-05, 2.36231366e-04, 4.65774521e-03,
        9.27465254e-06, 9.26408482e-06, 3.32745744e-03, 2.35563143e-05,
        2.35587309e-05, 0.00000000e+00, 1.18207962e-05, 1.18216259e-05],
       [1.35744125e-03, 5.32914571e-04, 2.52040752e-04, 4.29317651e-04,
        5.48889805e-04, 1.70497141e-04, 4.49855475e-05, 1.33686774e-03,
        1.94313763e-03, 1.18207962e-05, 0.00000000e+00, 1.42942522e-04],
       [5.32724707e-04, 1.35772436e-03, 2.52108623e-04, 4.29300284e-04,
        1.70578046e-04, 5.48840172e-04, 4.49787861e-05, 1.96241416e-03,
        1.34924633e-03, 1.18216259e-05, 1.42942522e-04, 0.00000000e+00]]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, \
                                                             'Anharmonic: Freq GVPT2, Int DVPT2')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert np.allclose(ref_input_wavenumers, properties.input_raman_frequencies)
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert np.allclose(ref_harmonic_raman, properties.raman.harmonic)
    assert np.allclose(ref_fundamental_raman, properties.raman.fundamental)
    assert np.allclose(ref_overtone_raman, properties.raman.overtones)
    assert np.allclose(ref_combotone_raman, properties.raman.combotones)
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # GVPT2 frequencies, DVPT2 intensities, w/ 1-1 checks
    ref_fundamental_raman = [[2.44346107e-01, 2.44353836e-01, 1.24015379e+00, 3.25830352e-01,
                              2.81008504e-02, 2.80936852e-02, 1.66988271e-02, 1.78405618e-04,
                              1.78481842e-04, 2.96550645e-02, 1.24172761e-02, 1.24172725e-02]]
    ref_overtone_raman = [[6.89075239e-04, 6.89080146e-04, 2.34466854e-04, 7.56238745e-05,
                           1.46758844e-02, 1.45247864e-02, 1.37614031e-02, 4.09371175e-04,
                           4.09296479e-04, 1.72278578e-04, 1.10275848e-03, 1.10273169e-03]]
    ref_combotone_raman = \
     [[[0.00000000e+00, 4.84152902e-04, 5.80469671e-04, 2.58177749e-04,
        2.24577460e-03, 6.40880000e-04, 1.58606940e-03, 1.12092039e-03,
        2.90377109e-03, 2.43488264e-05, 1.35744125e-03, 5.32724707e-04],
       [4.84152902e-04, 0.00000000e+00, 5.80535798e-04, 2.58171461e-04,
        6.40901752e-04, 2.24569298e-03, 1.58594127e-03, 2.90389577e-03,
        1.12097521e-03, 2.43335373e-05, 5.32914571e-04, 1.35772436e-03],
       [5.80469671e-04, 5.80535798e-04, 0.00000000e+00, 1.45312356e-03,
        1.94199273e-03, 1.94178997e-03, 4.90809799e-03, 4.03295742e-04,
        4.03236231e-04, 2.36231366e-04, 2.52040752e-04, 2.52108623e-04],
       [2.58177749e-04, 2.58171461e-04, 1.45312356e-03, 0.00000000e+00,
        3.53476568e-05, 3.53522082e-05, 2.41143336e-03, 6.43771513e-05,
        6.43976379e-05, 4.65774521e-03, 4.29317651e-04, 4.29300284e-04],
       [2.24577460e-03, 6.40901752e-04, 1.94199273e-03, 3.53476568e-05,
        0.00000000e+00, 1.23862653e-02, 2.35942037e-03, 1.77646805e-03,
        3.25616961e-03, 9.27465254e-06, 5.48889805e-04, 1.70578046e-04],
       [6.40880000e-04, 2.24569298e-03, 1.94178997e-03, 3.53522082e-05,
        1.23862653e-02, 0.00000000e+00, 2.35897994e-03, 3.25623208e-03,
        1.77630502e-03, 9.26408482e-06, 1.70497141e-04, 5.48840172e-04],
       [1.58606940e-03, 1.58594127e-03, 4.90809799e-03, 2.41143336e-03,
        2.35942037e-03, 2.35897994e-03, 0.00000000e+00, 2.30244910e-03,
        2.30238125e-03, 3.32745744e-03, 4.49855475e-05, 4.49787861e-05],
       [1.12092039e-03, 2.90389577e-03, 4.03295742e-04, 6.43771513e-05,
        1.77646805e-03, 3.25623208e-03, 2.30244910e-03, 0.00000000e+00,
        1.93047159e-04, 2.35563143e-05, 1.33686774e-03, 1.96241416e-03],
       [2.90377109e-03, 1.12097521e-03, 4.03236231e-04, 6.43976379e-05,
        3.25616961e-03, 1.77630502e-03, 2.30238125e-03, 1.93047159e-04,
        0.00000000e+00, 2.35587309e-05, 1.94313763e-03, 1.34924633e-03],
       [2.43488264e-05, 2.43335373e-05, 2.36231366e-04, 4.65774521e-03,
        9.27465254e-06, 9.26408482e-06, 3.32745744e-03, 2.35563143e-05,
        2.35587309e-05, 0.00000000e+00, 1.18207962e-05, 1.18216259e-05],
       [1.35744125e-03, 5.32914571e-04, 2.52040752e-04, 4.29317651e-04,
        5.48889805e-04, 1.70497141e-04, 4.49855475e-05, 1.33686774e-03,
        1.94313763e-03, 1.18207962e-05, 0.00000000e+00, 1.42942522e-04],
       [5.32724707e-04, 1.35772436e-03, 2.52108623e-04, 4.29300284e-04,
        1.70578046e-04, 5.48840172e-04, 4.49787861e-05, 1.96241416e-03,
        1.34924633e-03, 1.18216259e-05, 1.42942522e-04, 0.00000000e+00]]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, \
                                                    'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert np.allclose(ref_input_wavenumers, properties.input_raman_frequencies)
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert np.allclose(ref_harmonic_raman, properties.raman.harmonic)
    assert np.allclose(ref_fundamental_raman, properties.raman.fundamental)
    assert np.allclose(ref_overtone_raman, properties.raman.overtones)
    assert np.allclose(ref_combotone_raman, properties.raman.combotones)
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # GVPT2 frequencies, DVPT2 intensities, w/ 1-1 checks and forced removal of i,j
    ref_fundamental_raman = [[2.44346107e-01, 2.44353836e-01, 1.24015379e+00, 3.25830352e-01,
                              2.81008504e-02, 2.80936852e-02, 1.66988271e-02, 1.78405618e-04,
                              1.78481842e-04, 2.96550645e-02, 1.24172761e-02, 1.24172725e-02]]
    ref_overtone_raman = [[6.89075239e-04, 6.89080146e-04, 2.34466854e-04, 7.56238745e-05,
                           1.46758844e-02, 1.45247864e-02, 1.37614031e-02, 4.09371175e-04,
                           4.09296479e-04, 1.72278578e-04, 1.10275848e-03, 1.10273169e-03]]
    ref_combotone_raman = \
     [[[0.00000000e+00, 4.84152902e-04, 5.80469671e-04, 2.58177749e-04,
        2.24577460e-03, 6.40880000e-04, 1.58606940e-03, 1.12092039e-03,
        2.90377109e-03, 2.43488264e-05, 1.35744125e-03, 5.32724707e-04],
       [4.84152902e-04, 0.00000000e+00, 5.80535798e-04, 2.58171461e-04,
        6.40901752e-04, 2.24569298e-03, 1.58594127e-03, 2.90389577e-03,
        1.12097521e-03, 2.43335373e-05, 5.32914571e-04, 1.35772436e-03],
       [5.80469671e-04, 5.80535798e-04, 0.00000000e+00, 1.45312356e-03,
        1.94199273e-03, 1.94178997e-03, 4.90809799e-03, 4.03295742e-04,
        4.03236231e-04, 2.36231366e-04, 2.52040752e-04, 2.52108623e-04],
       [2.58177749e-04, 2.58171461e-04, 1.45312356e-03, 0.00000000e+00,
        3.53476568e-05, 3.53522082e-05, 2.41143336e-03, 6.43771513e-05,
        6.43976379e-05, 4.65774521e-03, 4.29317651e-04, 4.29300284e-04],
       [2.24577460e-03, 6.40901752e-04, 1.94199273e-03, 3.53476568e-05,
        0.00000000e+00, 1.23862653e-02, 2.35942037e-03, 1.77646805e-03,
        3.25616961e-03, 9.27465254e-06, 5.48889805e-04, 1.70578046e-04],
       [6.40880000e-04, 2.24569298e-03, 1.94178997e-03, 3.53522082e-05,
        1.23862653e-02, 0.00000000e+00, 2.35897994e-03, 3.25623208e-03,
        1.77630502e-03, 9.26408482e-06, 1.70497141e-04, 5.48840172e-04],
       [1.58606940e-03, 1.58594127e-03, 4.90809799e-03, 2.41143336e-03,
        2.35942037e-03, 2.35897994e-03, 0.00000000e+00, 2.30244910e-03,
        2.30238125e-03, 3.32745744e-03, 4.49855475e-05, 4.49787861e-05],
       [1.12092039e-03, 2.90389577e-03, 4.03295742e-04, 6.43771513e-05,
        1.77646805e-03, 3.25623208e-03, 2.30244910e-03, 0.00000000e+00,
        1.93047159e-04, 2.35563143e-05, 1.33686774e-03, 1.96241416e-03],
       [2.90377109e-03, 1.12097521e-03, 4.03236231e-04, 6.43976379e-05,
        3.25616961e-03, 1.77630502e-03, 2.30238125e-03, 1.93047159e-04,
        0.00000000e+00, 2.35587309e-05, 1.94313763e-03, 1.34924633e-03],
       [2.43488264e-05, 2.43335373e-05, 2.36231366e-04, 4.65774521e-03,
        9.27465254e-06, 9.26408482e-06, 3.32745744e-03, 2.35563143e-05,
        2.35587309e-05, 0.00000000e+00, 1.18207962e-05, 1.18216259e-05],
       [1.35744125e-03, 5.32914571e-04, 2.52040752e-04, 4.29317651e-04,
        5.48889805e-04, 1.70497141e-04, 4.49855475e-05, 1.33686774e-03,
        1.94313763e-03, 1.18207962e-05, 0.00000000e+00, 1.42942522e-04],
       [5.32724707e-04, 1.35772436e-03, 2.52108623e-04, 4.29300284e-04,
        1.70578046e-04, 5.48840172e-04, 4.49787861e-05, 1.96241416e-03,
        1.34924633e-03, 1.18216259e-05, 1.42942522e-04, 0.00000000e+00]]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, \
                                'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks and forced removal')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert np.allclose(ref_input_wavenumers, properties.input_raman_frequencies)
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert np.allclose(ref_harmonic_raman, properties.raman.harmonic)
    assert np.allclose(ref_fundamental_raman, properties.raman.fundamental)
    assert np.allclose(ref_overtone_raman, properties.raman.overtones)
    assert np.allclose(ref_combotone_raman, properties.raman.combotones)
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    ###############################################################################################
    # Different units anharmonic Raman.  All default anharmonic: GVPT2 frequencies, DVPT2 intensities, 
    # w/ 1-1 checks
    ###############################################################################################
    # Raman CPG 45+7, a.u.
    tensor_file = data_dir + 'anharm_raman_opt_CO2.rsp_tensor'
    mol_file = data_dir + 'opt_CO2.mol'
    specifications = ['Vib modes: 1/m', 'Raman: CPG 45+7, a.u.']

    ref_wavenumbers = [256437.00659636, 151124.33230516,  77296.77672684,  77296.64946092]
    ref_fundamental = [251541.01307331, 145222.64954562,  76854.85245673,  76854.70570159]
    ref_overtones = [500606.12530794, 297503.0186331,  154143.18970647, 157941.75784598]
    ref_combotones = [[     0.        , 398506.11685921, 327003.83987687, 327003.67295073],
                      [398506.11685921,      0.        , 225340.43060746, 225340.28098378],
                      [327003.83987687, 225340.43060746,      0.        , 153886.10372417],
                      [327003.67295073, 225340.28098378, 153886.10372417,      0.        ]]

    ref_input_wavenumers = [0.0, 219474.60394200613]
    ref_harmonic_raman = [[4.53263491e-12, 9.26005295e-02, 3.17855256e-11, 3.24475018e-12],
                          [4.53936045e-12, 9.26966048e-02, 3.18050950e-11, 3.24599800e-12]]
    ref_fundamental_raman = [[4.80799128e-12, 8.88403378e-02, 3.16489362e-11, 3.26558293e-12],
                             [4.87633186e-12, 8.90586079e-02, 3.17364560e-11, 3.26775105e-12]]
    ref_overtone_raman = [[1.82913516e-03, 6.82920887e-05, 9.39258683e-04, 9.62426439e-04],
                          [1.97633183e-03, 5.49728045e-05, 1.12084414e-03, 1.14848731e-03]]
    ref_combotone_raman = [[[0.00000000e+00, 6.68051330e-15, 5.36442562e-04, 5.36458689e-04],
                            [6.68051330e-15, 0.00000000e+00, 1.93304594e-13, 2.36882808e-13],
                            [5.36442562e-04, 1.93304594e-13, 0.00000000e+00, 2.72619337e-04],
                            [5.36458689e-04, 2.36882808e-13, 2.72619337e-04, 0.00000000e+00]],
                           [[0.00000000e+00, 4.78453649e-15, 6.39682834e-04, 6.39713789e-04],
                            [4.78453649e-15, 0.00000000e+00, 2.07142407e-13, 2.45106722e-13],
                            [6.39682834e-04, 2.07142407e-13, 0.00000000e+00, 2.93236803e-04],
                            [6.39713789e-04, 2.45106722e-13, 2.93236803e-04, 0.00000000e+00]]]

    properties = \
        get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                    specifications, 0, 298, True, \
                                                    'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert np.allclose(ref_input_wavenumers, properties.input_raman_frequencies)
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert np.allclose(ref_harmonic_raman, properties.raman.harmonic)
    assert np.allclose(ref_fundamental_raman, properties.raman.fundamental)
    assert np.allclose(ref_overtone_raman, properties.raman.overtones)
    assert np.allclose(ref_combotone_raman, properties.raman.combotones)
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # Raman PCPG 45+4, Å^4/amu
    specifications = ['Vib modes: 1/m', 'Raman: PCPG 45+4, Å^4/amu']

    ref_harmonic_raman = [[3.81083081e-10, 1.15697070e+01, 2.59630681e-09, 3.12307805e-10],
                          [3.81640459e-10, 1.15815282e+01, 2.59790529e-09, 3.12553320e-10]]
    ref_fundamental_raman = [[4.09240458e-10, 1.10840895e+01, 2.58514991e-09, 3.15822921e-10],
                             [4.15253816e-10, 1.11117286e+01, 2.59229871e-09, 3.17705303e-10]]
    ref_overtone_raman = [[0.20280919, 0.00950771, 0.12581642, 0.12891942],
                          [0.21569893, 0.00774044, 0.151204  , 0.15493273]]
    ref_combotone_raman = [[[0.00000000e+00, 6.05468958e-13, 4.38177269e-02, 4.38190441e-02],
                            [6.05468958e-13, 0.00000000e+00, 1.57895150e-11, 3.20581884e-11],
                            [4.38177269e-02, 1.57895150e-11, 0.00000000e+00, 2.22681056e-02],
                            [4.38190441e-02, 3.20581884e-11, 2.22681056e-02, 0.00000000e+00]],
                           [[0.00000000e+00, 4.61122896e-13, 5.22506037e-02, 5.22531321e-02],
                            [4.61122896e-13, 0.00000000e+00, 1.69198159e-11, 3.32305790e-11],
                            [5.22506037e-02, 1.69198159e-11, 0.00000000e+00, 2.39521825e-02],
                            [5.22531321e-02, 3.32305790e-11, 2.39521825e-02, 0.00000000e+00]]]

    properties = \
        get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                    specifications, 0, 298, True, \
                                                    'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert np.allclose(ref_input_wavenumers, properties.input_raman_frequencies)
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert np.allclose(ref_harmonic_raman, properties.raman.harmonic)
    assert np.allclose(ref_fundamental_raman, properties.raman.fundamental)
    assert np.allclose(ref_overtone_raman, properties.raman.overtones)
    assert np.allclose(ref_combotone_raman, properties.raman.combotones)
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # Raman PCPG 45+7, Å^4/amu
    specifications = ['Vib modes: 1/m', 'Raman: PCPG 45+7, Å^4/amu']

    ref_harmonic_raman = [[6.47911075e-10, 1.32366515e+01, 4.54353692e-09, 4.63816217e-10],
                          [6.48872448e-10, 1.32503848e+01, 4.54633425e-09, 4.63994585e-10]]
    ref_fundamental_raman = [[6.87271501e-10, 1.26991562e+01, 4.52401235e-09, 4.66794126e-10],
                             [6.97040348e-10, 1.27303565e+01, 4.53652273e-09, 4.67104045e-10]]
    ref_overtone_raman = [[0.26146313, 0.00976192, 0.134261  , 0.13757268],
                          [0.28250395, 0.00785801, 0.16021748, 0.16416889]]
    ref_combotone_raman = [[[0.00000000e+00, 9.54936509e-13, 7.66810220e-02, 7.66833272e-02],
                            [9.54936509e-13, 0.00000000e+00, 2.76316513e-11, 3.38608774e-11],
                            [7.66810220e-02, 2.76316513e-11, 0.00000000e+00, 3.89691848e-02],
                            [7.66833272e-02, 3.38608774e-11, 3.89691848e-02, 0.00000000e+00]],
                           [[0.00000000e+00, 6.83918789e-13, 9.14385564e-02, 9.14429812e-02],
                            [6.83918789e-13, 0.00000000e+00, 2.96096778e-11, 3.50364332e-11],
                            [9.14385564e-02, 2.96096778e-11, 0.00000000e+00, 4.19163194e-02],
                            [9.14429812e-02, 3.50364332e-11, 4.19163194e-02, 0.00000000e+00]]]

    properties = \
        get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                    specifications, 0, 298, True, \
                                                    'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert np.allclose(ref_input_wavenumers, properties.input_raman_frequencies)
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert np.allclose(ref_harmonic_raman, properties.raman.harmonic)
    assert np.allclose(ref_fundamental_raman, properties.raman.fundamental)
    assert np.allclose(ref_overtone_raman, properties.raman.overtones)
    assert np.allclose(ref_combotone_raman, properties.raman.combotones)
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # Raman SCS 45+4 SI
    specifications = ['Vib modes: 1/m', 'Raman: SCS 45+4, SI units']

    ref_harmonic_raman = [[1.34113861e-70, 8.32807729e-61, 2.44246281e-71, 2.93800530e-72],
                          [5.79730818e-74, 3.48828254e-62, 2.79754587e-70, 3.36573736e-71]]
    ref_fundamental_raman = [[1.35930365e-70, 7.07823200e-61, 2.38922848e-71, 2.91885789e-72],
                             [3.64265844e-74, 4.84955546e-62, 2.84112474e-70, 3.48202866e-71]]
    ref_overtone_raman = [[5.30994415e-61, 5.22472447e-63, 9.61102192e-63, 1.05952554e-62],
                          [5.61699418e-62, 2.01279150e-65, 3.72722940e-64, 2.93342319e-64]]
    ref_combotone_raman = [[[0.00000000e+00, 7.99669659e-73, 3.19758578e-62, 3.19767657e-62],
                            [7.99669659e-73, 0.00000000e+00, 3.77043736e-72, 7.65527897e-72],
                            [3.19758578e-62, 3.77043736e-72, 0.00000000e+00, 1.69253455e-63],
                            [3.19767657e-62, 7.65527897e-72, 1.69253455e-63, 0.00000000e+00]],
                           [[0.00000000e+00, 2.48092390e-74, 4.45819312e-64, 4.45838117e-64],
                            [2.48092390e-74, 0.00000000e+00, 1.85515728e-78, 3.64315365e-78],
                            [4.45819312e-64, 1.85515728e-78, 0.00000000e+00, 6.00775460e-65],
                            [4.45838117e-64, 3.64315365e-78, 6.00775460e-65, 0.00000000e+00]]]

    properties = \
        get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                    specifications, 0, 298, True, \
                                                    'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert np.allclose(ref_input_wavenumers, properties.input_raman_frequencies)
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert np.allclose(np.multiply(ref_harmonic_raman, 1.0e70), \
                       np.multiply(properties.raman.harmonic, 1.0e70))
    assert np.allclose(np.multiply(ref_fundamental_raman, 1.0e70), \
                       np.multiply(properties.raman.fundamental, 1.0e70))
    assert np.allclose(np.multiply(ref_overtone_raman, 1.0e70), \
                       np.multiply(properties.raman.overtones, 1.0e70))
    assert np.allclose(np.multiply(ref_combotone_raman, 1.0e70), \
                       np.multiply(properties.raman.combotones, 1.0e70))
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # Raman SCS 45+7 SI
    specifications = ['Vib modes: 1/m', 'Raman: SCS 45+7, SI units']

    ref_harmonic_raman = [[2.28018142e-70, 9.52797304e-61, 4.27430992e-71, 4.36330595e-72],
                          [9.85669487e-74, 3.99093152e-62, 4.89570527e-70, 4.99653598e-71]]
    ref_fundamental_raman = [[2.28279155e-70, 8.10960374e-61, 4.18114984e-71, 4.31414450e-72],
                             [6.11452517e-74, 5.55598251e-62, 4.97196829e-70, 5.11942877e-71]]
    ref_overtone_raman = [[6.84561995e-61, 5.36441798e-63, 1.02560971e-62, 1.13064246e-62],
                          [7.35665707e-62, 2.04336302e-65, 3.94941476e-64, 3.10829626e-64]]
    ref_combotone_raman = [[[0.00000000e+00, 1.26122693e-72, 5.59577512e-62, 5.59593400e-62],
                            [1.26122693e-72, 0.00000000e+00, 6.59826537e-72, 8.08574894e-72],
                            [5.59577512e-62, 6.59826537e-72, 0.00000000e+00, 2.96193546e-63],
                            [5.59593400e-62, 8.08574894e-72, 2.96193546e-63, 0.00000000e+00]],
                           [[0.00000000e+00, 3.67960577e-74, 7.80183797e-64, 7.80216705e-64],
                            [3.67960577e-74, 0.00000000e+00, 3.24652525e-78, 3.84113408e-78],
                            [7.80183797e-64, 3.24652525e-78, 0.00000000e+00, 1.05135706e-64],
                            [7.80216705e-64, 3.84113408e-78, 1.05135706e-64, 0.00000000e+00]]]

    properties = \
        get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                    specifications, 0, 298, True, \
                                                    'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert np.allclose(ref_input_wavenumers, properties.input_raman_frequencies)
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert np.allclose(np.multiply(ref_harmonic_raman, 1.0e70), \
                       np.multiply(properties.raman.harmonic, 1.0e70))
    assert np.allclose(np.multiply(ref_fundamental_raman, 1.0e70), \
                       np.multiply(properties.raman.fundamental, 1.0e70))
    assert np.allclose(np.multiply(ref_overtone_raman, 1.0e70), \
                       np.multiply(properties.raman.overtones, 1.0e70))
    assert np.allclose(np.multiply(ref_combotone_raman, 1.0e70), \
                       np.multiply(properties.raman.combotones, 1.0e70))
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # Different temperature, T = 400K
    ref_harmonic_raman = [[2.27996604e-70, 9.49289051e-61, 4.10758139e-71, 4.19310530e-72],
                          [9.85576384e-74, 3.97623669e-62, 4.70473789e-70, 4.80163476e-71]]
    ref_fundamental_raman = [[2.28253512e-70, 8.07318648e-61, 4.01592319e-71, 4.14366154e-72],
                             [6.11383829e-74, 5.53103263e-62, 4.77549084e-70, 4.91712322e-71]]
    ref_overtone_raman = [[6.84561984e-61, 5.36430026e-63, 1.02219948e-62, 1.12733702e-62],
                          [7.35665696e-62, 2.04331818e-65, 3.93628266e-64, 3.09920912e-64]]
    ref_combotone_raman = [[[0.00000000e+00, 1.26122618e-72, 5.59573228e-62, 5.59589116e-62],
                            [1.26122618e-72, 0.00000000e+00, 6.59639752e-72, 8.08345999e-72],
                            [5.59573228e-62, 6.59639752e-72, 0.00000000e+00, 2.95200080e-63],
                            [5.59589116e-62, 8.08345999e-72, 2.95200080e-63, 0.00000000e+00]],
                           [[0.00000000e+00, 3.67960359e-74, 7.80177824e-64, 7.80210732e-64],
                            [3.67960359e-74, 0.00000000e+00, 3.24560621e-78, 3.84004672e-78],
                            [7.80177824e-64, 3.24560621e-78, 0.00000000e+00, 1.04783068e-64],
                            [7.80210732e-64, 3.84004672e-78, 1.04783068e-64, 0.00000000e+00]]]

    properties = \
        get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                    specifications, 0, 400, True, \
                                                    'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert np.allclose(ref_input_wavenumers, properties.input_raman_frequencies)
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert np.allclose(np.multiply(ref_harmonic_raman, 1.0e70), \
                       np.multiply(properties.raman.harmonic, 1.0e70))
    assert np.allclose(np.multiply(ref_fundamental_raman, 1.0e70), \
                       np.multiply(properties.raman.fundamental, 1.0e70))
    assert np.allclose(np.multiply(ref_overtone_raman, 1.0e70), \
                       np.multiply(properties.raman.overtones, 1.0e70))
    assert np.allclose(np.multiply(ref_combotone_raman, 1.0e70), \
                       np.multiply(properties.raman.combotones, 1.0e70))
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    ##############################
    # Back to harmonic IR
    ##############################

    # IR SSDG C²/kg, No Raman
    data_dir = '{0}/'.format(os.path.dirname(__file__))
    tensor_file = data_dir + 'hf_H2O2.rsp_tensor'
    mol_file = data_dir + 'H2O2.mol'
    specifications = ['Vib modes: 1/m', 'IR: SSDG, C**2/kg']
    spectroscopy_type = ['IR']

    ref_wavenumbers = [414847.6830235662, 414104.2828525629, 178148.02217701354,
                       158913.5038025385, 148697.90566093015, 18299.94565115991]
    ref_fundamental = [0., 0., 0., 0., 0., 0.]
    ref_overtones = [0., 0., 0., 0., 0., 0.]
    ref_combotones = [[0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0.],
                      [0., 0., 0., 0., 0., 0.]]
    ref_harmonic_ir = [4.80247329e-13, 2.00453540e-13, 3.27545582e-14, \
                                   7.31573576e-13, 2.68369749e-16, 2.13230475e-12]
    ref_fundamental_ir = np.zeros((len(ref_harmonic_ir)))
    ref_overtone_ir = np.zeros((len(ref_harmonic_ir)))
    ref_combotone_ir = np.zeros((len(ref_harmonic_ir)))

    ref_input_wavenumers = ['You did not ask for Raman intensities']
    ref_harmonic_raman = ['You did not ask for Raman intensities']

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, False)

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # IR SSDG (D/Å)²/amu
    specifications = ['Vib modes: 1/m', 'IR: SSDG, D2A2/amu']
    spectroscopy_type = ['IR']

    ref_harmonic_ir = [7.16729788e-01, 2.99160484e-01, 4.88834942e-02, 1.09181362e+00, \
                          4.00519861e-04, 3.18229012e+00]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, False)

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # IR MDAC m²/(s*mol)
    specifications = ['Vib modes: 1/m', 'IR: MDAC, m**2/(s*mol)']
    spectroscopy_type = ['IR']

    ref_harmonic_ir = [2.47758955e+13, 1.03413714e+13, 1.68980328e+12, 3.77417830e+13, \
                          1.38451595e+10, 1.10005317e+14]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, False)

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # IR MDAC L/(cm*s*mol)
    specifications = ['Vib modes: 1/m', 'IR: MDAC, L/(cm*s*mol)']
    spectroscopy_type = ['IR']

    ref_harmonic_ir = [2.47758955e+14, 1.03413714e+14, 1.68980328e+13, 3.77417830e+14, \
                          1.38451595e+11, 1.10005317e+15]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, False)

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # IR NIMAC m/mol
    specifications = ['Vib modes: 1/m', 'IR: NIMAC, m/mol']
    spectroscopy_type = ['IR']

    ref_harmonic_ir = [3.02861786e+04, 1.26413440e+04, 2.06562397e+03, 4.61357442e+04, \
                          1.69243922e+01, 1.34471049e+05]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, False)

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    # IR NIMAC km/mol
    specifications = ['Vib modes: 1/m', 'IR: NIMAC, km/mol']

    ref_harmonic_ir = [3.02861786e+01, 1.26413440e+01, 2.06562397e+00, 4.61357442e+01, \
                          1.69243922e-02, 1.34471049e+02]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, False)

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert ref_hyperraman_freq == properties.input_hyperraman_frequencies
    assert np.allclose(ref_harmonic_ir, properties.ir.harmonic)
    assert np.allclose(ref_fundamental_ir, properties.ir.fundamental)
    assert np.allclose(ref_overtone_ir, properties.ir.overtones)
    assert np.allclose(ref_combotone_ir, properties.ir.combotones)
    assert ref_harmonic_raman == properties.raman.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.harmonic
    assert ref_hyperraman == properties.hyper_raman_hv.harmonic
    assert ref_hyperraman == properties.hyper_raman_vv.fundamental
    assert ref_hyperraman == properties.hyper_raman_hv.fundamental
    assert ref_hyperraman == properties.hyper_raman_vv.overtones
    assert ref_hyperraman == properties.hyper_raman_hv.overtones
    assert ref_hyperraman == properties.hyper_raman_vv.combotones
    assert ref_hyperraman == properties.hyper_raman_hv.combotones

    ########################################
    # Hyper-Raman
    ########################################
    spectroscopy_type = ['Hyper-Raman']
    specifications = ['Vib modes: 1/m', 'Hyper-Raman: SI complete']

    ref_ih = [2194746.0394200613, 0.0]
    ref_harmonic_ir = ['You did not ask for IR intensities']
    ref_fundamental_ir = ['You did not ask for IR intensities']
    ref_overtone_ir = ['You did not ask for IR intensities']
    ref_combotone_ir = ['You did not ask for IR intensities']
    ref_vv_harmonic = [[5.31275323e-42, 1.63731429e-42, 4.38462803e-43,
                        2.74009360e-42, 6.76023854e-44, 2.97702629e-41],
                       [4.04902604e-46, 1.31612249e-46, 1.26646968e-48,
                        4.77616251e-48, 8.16225856e-50, 7.27144463e-51]]
    ref_vv_fundamental = np.zeros((2, 6))
    ref_vv_overtones = np.zeros((2, 6))
    ref_vv_combotones = np.zeros((2, 6, 6))
    ref_hv_harmonic = [[7.03927756e-43, 5.22978431e-43, 2.73830629e-43,
                        1.05652253e-42, 1.15579372e-44, 6.26610333e-42],
                       [5.13529085e-47, 4.26925953e-47, 7.97713412e-49,
                        1.79640060e-48, 1.95696552e-50, 1.48825197e-51]]
    ref_hv_fundamental = np.zeros((2, 6))
    ref_hv_overtones = np.zeros((2, 6))
    ref_hv_combotones = np.zeros((2, 6, 6))

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, False)

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert np.allclose(ref_ih, properties.input_hyperraman_frequencies)
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert ref_harmonic_raman == properties.raman.harmonic
    assert np.allclose(np.multiply(ref_vv_harmonic, 1.0e51), \
           np.multiply(properties.hyper_raman_vv.harmonic, 1.0e51))
    assert np.allclose(np.multiply(ref_hv_harmonic, 1.0e51), \
           np.multiply(properties.hyper_raman_hv.harmonic, 1.0e51))
    assert np.allclose(ref_vv_fundamental, properties.hyper_raman_vv.fundamental)
    assert np.allclose(ref_hv_fundamental, properties.hyper_raman_hv.fundamental)
    assert np.allclose(ref_vv_overtones, properties.hyper_raman_vv.overtones)
    assert np.allclose(ref_hv_overtones, properties.hyper_raman_hv.overtones)
    assert np.allclose(ref_vv_combotones, properties.hyper_raman_vv.combotones)
    assert np.allclose(ref_hv_combotones, properties.hyper_raman_hv.combotones)

    specifications = ['Vib modes: 1/m', 'Hyper-Raman: Relative']

    ref_vv_harmonic = [[0.1329302 , 0.04096718, 0.01097076, 
                        0.06855978, 0.00169148, 0.74488061], 
                       [7.46162915e-01, 2.42537782e-01, 2.33387660e-03, 
                        8.80161132e-03,1.50415793e-04, 1.33999690e-05]]
    ref_hv_harmonic = [[0.07967562, 0.05919447, 0.03099412, 
                        0.11958484, 0.00130821, 0.70924274], 
                       [5.31269910e-01, 4.41674910e-01, 8.25271918e-03,
                        1.85846063e-02, 2.02457256e-04, 1.53966642e-05]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, False)

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert np.allclose(ref_ih, properties.input_hyperraman_frequencies)
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert ref_harmonic_raman == properties.raman.harmonic
    assert np.allclose(ref_vv_harmonic, properties.hyper_raman_vv.harmonic)
    assert np.allclose(ref_hv_harmonic, properties.hyper_raman_hv.harmonic)
    assert np.allclose(ref_vv_fundamental, properties.hyper_raman_vv.fundamental)
    assert np.allclose(ref_hv_fundamental, properties.hyper_raman_hv.fundamental)
    assert np.allclose(ref_vv_overtones, properties.hyper_raman_vv.overtones)
    assert np.allclose(ref_hv_overtones, properties.hyper_raman_hv.overtones)
    assert np.allclose(ref_vv_combotones, properties.hyper_raman_vv.combotones)
    assert np.allclose(ref_hv_combotones, properties.hyper_raman_hv.combotones)

    # Anharmonic
    # Acetonitrile
    tensor_file = data_dir + 'anharm_hR_opt_cc-pvtz_acetonitrile.rsp_tensor'
    mol_file = data_dir + 'opt_cc-pvtz_acetonitrile.mol'

    # VPT2
    ref_wavenumbers = [327001.33661905, 327000.54349763, 319461.22821944, 259463.36970978,
                       159332.36953073, 159331.9999498 , 153430.47673707, 115907.85002149,
                       115907.38886888,  95963.53572707,  42450.20834683,  42450.01611672]
    ref_fundamental = [314012.43298487, 314011.53979323, 331018.04404073, 256465.52677474,
                       158393.22664253, 158393.99198678, 150041.13718027, 113543.31474027,
                       113542.65561761,  95712.27989893,  42042.98978884,  42042.96229487]
    ref_overtones = [622382.43837506, 622381.06278219, 658986.51133344, 510942.8909639,
                     305232.26471604, 305244.65783572, 297348.07029565, 226811.20363659,
                     226809.79089787, 190623.68916583,  83664.32840753,  83664.35715109]
    ref_combotones = \
[[     0.        , 624705.06507938, 638494.57613904, 570460.42997406,
  471955.03713515, 470045.53246848, 463345.23626701, 426758.64359099,
  426878.70636511, 409933.29257683, 356095.0069167 , 355979.95300053],
 [624705.06507938,      0.        , 638492.79207507, 570459.54133761,
  470043.90031885, 471954.8487804 , 463344.37693254, 426878.52019834,
  426756.96397891, 409932.37194193, 355979.07076914, 356094.0744635 ],
 [638494.57613904, 638492.79207507,      0.        , 587433.96854705,
  509465.22282014, 509444.10224351, 484269.28799995, 444046.51492181,
  444045.88619207, 426811.17141962, 373054.91085386, 373054.89407738],
 [570460.42997406, 570459.54133761, 587433.96854705,      0.,
  414762.66478743, 414763.43364599, 406517.23180163, 369746.20969394,
  369745.52049082, 352217.88792428, 297874.57456537, 297874.54602396],
 [471955.03713515, 470043.90031885, 509465.22282014, 414762.66478743,
       0.        , 316045.53423348, 307820.8583927 , 273061.08955946,
  273038.84025806, 254026.77798466, 202082.23777047, 202080.29627025],
 [470045.53246848, 471954.8487804 , 509444.10224351, 414763.43364599,
  316045.53423348,      0.        , 307821.65476245, 273041.73378183,
  273060.95779239, 254027.5494395 , 202082.31494437, 202082.99586869],
 [463345.23626701, 463344.37693254, 484269.28799995, 406517.23180163,
  307820.8583927 , 307821.65476245,      0.        , 263064.20459946,
  263063.48794891, 245037.34667131, 192018.76030679, 192018.75687982],
 [426758.64359099, 426878.52019834, 444046.51492181, 369746.20969394,
  273061.08955946, 273041.73378183, 263064.20459946,      0.,
  227174.58106829, 209062.63358751, 153788.44572047, 153852.45354977],
 [426878.70636511, 426756.96397891, 444045.88619207, 369745.52049082,
  273038.84025806, 273060.95779239, 263063.48794891, 227174.58106829,
       0.        , 209062.00120743, 153851.99860502, 153789.28818488],
 [409933.29257683, 409932.37194193, 426811.17141962, 352217.88792428,
  254026.77798466, 254027.5494395 , 245037.34667131, 209062.63358751,
  209062.00120743,      0.        , 138665.5741924 , 138665.52339525],
 [356095.0069167 , 355979.07076914, 373054.91085386, 297874.57456537,
  202082.23777047, 202082.31494437, 192018.76030679, 153788.44572047,
  153851.99860502, 138665.5741924 ,      0.        ,  84185.97021186],
 [355979.95300053, 356094.0744635 , 373054.89407738, 297874.54602396,
  202080.29627025, 202082.99586869, 192018.75687982, 153852.45354977,
  153789.28818488, 138665.52339525,  84185.97021186,      0.        ]]

    ref_ih = [0.0, 2194746.0394200613]
    ref_vv_harmonic = [[3.57079000e-01, 3.57103322e-01, 2.74969083e-01, 4.19021588e-03,
                        1.47246010e-03, 1.47260098e-03, 9.48076274e-04, 1.19795248e-03,
                        1.19793434e-03, 2.53678638e-04, 5.78389703e-05, 5.78372922e-05], 
                       [0.21228464, 0.21230267, 0.18457996, 0.00934262, 0.0214306 ,
                        0.02143232, 0.01339279, 0.04156793, 0.04156889, 0.01189928,
                        0.11509793, 0.11510038]]
    ref_vv_fundamental = [[7.97245031e-03, 7.97304298e-03, 9.83833033e-01, 1.17710722e-04,
                           8.47025450e-06, 8.50911503e-06, 1.95536993e-05, 2.99188398e-05,
                           2.99183312e-05, 4.84191971e-06, 1.27524517e-06, 1.27523008e-06],
                          [1.11669891e-02, 1.11679849e-02, 9.61857481e-01, 6.44824297e-04,
                           1.60623769e-04, 1.61490384e-04, 7.89757641e-04, 2.10748143e-03,
                           2.10754951e-03, 3.53961681e-04, 4.74088679e-03, 4.74096975e-03]]
    ref_vv_overtones = [[4.47761055e-03, 4.47792377e-03, 3.02884380e-03, 1.36786482e-05,
                         4.92311842e-01, 4.91413789e-01, 4.21242617e-03, 2.06049276e-05,
                         2.05997448e-05, 1.98711125e-05, 1.40543823e-06, 1.40553758e-06],
                        [4.43307719e-04, 4.43347531e-04, 2.11481101e-04, 1.38659364e-05,
                         4.97465272e-01, 4.96465824e-01, 4.40473335e-03, 5.04323034e-05,
                         5.04178543e-05, 1.68268602e-04, 1.41521912e-04, 1.41528492e-04]]
    ref_vv_combotones = \
     [[[0.00000000e+00, 1.55515506e-01, 1.66065382e-01, 1.58667258e-01,
        2.07710780e-01, 2.67667087e-01, 2.45619172e-01, 7.61556342e-02,
        4.72080232e-01, 1.15464498e-01, 3.99994443e-01, 9.20976395e-02],
       [1.55524472e-01, 0.00000000e+00, 1.66080257e-01, 1.58710408e-01,
        2.67673613e-01, 2.07716102e-01, 2.45599718e-01, 4.72036689e-01,
        7.61881034e-02, 1.15464397e-01, 9.20990604e-02, 4.00073643e-01],
       [1.63652283e-01, 1.63657506e-01, 0.00000000e+00, 3.24568064e-01,
        1.53664636e-01, 1.53634531e-01, 3.89674451e-01, 2.11153420e-01,
        2.11179999e-01, 4.94563457e-01, 1.97729219e-01, 1.97776498e-01],
       [2.39668267e-02, 2.39719625e-02, 4.97491936e-02, 0.00000000e+00,
        5.39466008e-03, 5.39427492e-03, 2.24618690e-02, 1.85935479e-02,
        1.85946356e-02, 2.30503691e-01, 3.49069008e-02, 3.49068367e-02],
       [1.12615000e-01, 1.45116812e-01, 8.45411551e-02, 1.93632419e-02,
        0.00000000e+00, 3.05141543e-01, 3.27796577e-02, 1.37640270e-02,
        3.16296158e-02, 2.90322222e-03, 8.70109965e-04, 5.51565895e-04],
       [1.45107820e-01, 1.12600669e-01, 8.45165435e-02, 1.93600157e-02,
        3.05112485e-01, 0.00000000e+00, 3.27755041e-02, 3.16241943e-02,
        1.37668792e-02, 2.90244937e-03, 5.51604121e-04, 8.70366601e-04],
       [1.36014640e-01, 1.35996027e-01, 2.18968865e-01, 8.23466746e-02,
        3.34803972e-02, 3.34793430e-02, 0.00000000e+00, 2.41282990e-02,
        2.41307292e-02, 3.43914076e-02, 8.68310167e-04, 8.68206597e-04],
       [2.38617308e-02, 1.47894023e-01, 6.71359385e-02, 3.85690509e-02,
        7.95441443e-03, 1.82777834e-02, 1.36522180e-02, 0.00000000e+00,
        1.28056038e-03, 1.66213879e-03, 1.35966080e-01, 1.36806964e-01],
       [1.47879014e-01, 2.38645285e-02, 6.71275130e-02, 3.85616125e-02,
        1.82745816e-02, 7.95482057e-03, 1.36501614e-02, 1.28023852e-03,
        0.00000000e+00, 1.66230669e-03, 1.36731549e-01, 1.35765518e-01],
       [6.03003597e-03, 6.02968309e-03, 2.62090026e-02, 7.96940863e-02,
        2.79649794e-04, 2.79601976e-04, 3.24338115e-03, 2.77037662e-04,
        2.77135303e-04, 0.00000000e+00, 7.26232214e-05, 7.26447382e-05],
       [6.93757754e-02, 1.59729104e-02, 3.48002036e-02, 4.00813154e-02,
        2.78349746e-04, 1.76475958e-04, 2.71960221e-04, 7.52635270e-02,
        7.57062779e-02, 2.41189264e-04, 0.00000000e+00, 2.10116538e-04],
       [1.59724015e-02, 6.93803717e-02, 3.48059460e-02, 4.00782725e-02,
        1.76433852e-04, 2.78437733e-04, 2.71907637e-04, 7.57233852e-02,
        7.51658317e-02, 2.41242850e-04, 2.10100972e-04, 0.00000000e+00]],
       [[0.00000000e+00, 8.36930291e-02, 1.24258762e-01, 6.64408401e-02,
        9.61464787e-02, 8.56918713e-02, 1.65556753e-01, 4.81604588e-03,
        3.14510095e-02, 1.10379146e-01, 2.89813458e-02, 7.59340815e-03],
       [8.37001396e-02, 0.00000000e+00, 1.24277138e-01, 6.64582886e-02,
        8.57036239e-02, 9.61437740e-02, 1.65547959e-01, 3.14268458e-02,
        4.82144777e-03, 1.10382464e-01, 7.59033101e-03, 2.89999087e-02],
       [1.04715935e-01, 1.04722524e-01, 0.00000000e+00, 1.10825028e-01,
        5.36064282e-02, 5.36057479e-02, 2.19923065e-01, 1.04105896e-02,
        1.04187845e-02, 2.45127700e-01, 1.15042355e-02, 1.15121540e-02],
       [1.98236026e-02, 1.98271241e-02, 3.92373406e-02, 0.00000000e+00,
        6.09741007e-03, 6.09726709e-03, 4.35786253e-02, 2.77008250e-03,
        2.77208013e-03, 3.34220840e-01, 7.80403685e-03, 7.80821282e-03],
       [1.21263692e-01, 1.08083573e-01, 8.02284720e-02, 2.57748021e-02,
        0.00000000e+00, 5.96527732e-01, 9.20132513e-02, 6.28223938e-03,
        1.51016902e-02, 6.48480389e-03, 7.62275110e-04, 5.50110549e-04],
       [1.08074009e-01, 1.21245577e-01, 8.02245409e-02, 2.57732619e-02,
        5.96506073e-01, 0.00000000e+00, 9.20011106e-02, 1.50866948e-02,
        6.28760535e-03, 6.48442639e-03, 5.49969316e-04, 7.62923185e-04],
       [1.32543698e-01, 1.32525398e-01, 2.08928270e-01, 1.16933310e-01,
        5.84070344e-02, 5.84014483e-02, 0.00000000e+00, 1.32060491e-02,
        1.32161651e-02, 8.84726172e-02, 1.01015396e-03, 1.01056778e-03],
       [2.75310466e-02, 1.79637087e-01, 7.06190188e-02, 5.30734528e-02,
        2.84740461e-02, 6.83824372e-02, 9.42959049e-02, 0.00000000e+00,
        7.90743765e-04, 2.08354342e-02, 4.68355762e-01, 4.69058552e-01],
       [1.79622212e-01, 2.75337916e-02, 7.06084623e-02, 5.30620181e-02,
        6.83838586e-02, 2.84727287e-02, 9.42798150e-02, 7.90003692e-04,
        0.00000000e+00, 2.08322224e-02, 4.68601918e-01, 4.67861504e-01],
       [1.87041294e-02, 1.87031026e-02, 4.92898053e-02, 1.89817621e-01,
        8.71264280e-04, 8.71245196e-04, 1.87260923e-02, 6.17619380e-04,
        6.18102670e-04, 0.00000000e+00, 8.62247511e-04, 8.62873865e-04],
       [1.61681307e-01, 4.23413871e-02, 7.61576039e-02, 1.45919424e-01,
        3.37175180e-03, 2.43275340e-03, 7.03909215e-03, 4.57072428e-01,
        4.57741063e-01, 2.83872123e-02, 0.00000000e+00, 3.97978487e-03],
       [4.23402292e-02, 1.61687407e-01, 7.61705864e-02, 1.45921954e-01,
        2.43203057e-03, 3.37299455e-03, 7.03833161e-03, 4.57521401e-01,
        4.56781308e-01, 2.83931326e-02, 3.97772537e-03, 0.00000000e+00]]]

    ref_hv_harmonic = [[2.96286781e-01, 2.96305756e-01, 3.91407701e-01, 6.33936630e-03,
                        1.63763931e-03, 1.63757968e-03, 2.07821813e-03, 2.05128771e-03,
                        2.05110142e-03, 1.07999225e-04, 4.82862875e-05, 4.82834920e-05],
                       [0.17701748, 0.17703244, 0.24198763, 0.01017655, 0.01889756,
                        0.01889695, 0.02585506, 0.06846015, 0.06845571, 0.00472868,
                        0.09424632, 0.09424546]]
    ref_hv_fundamental = [[4.51831926e-03, 4.51863453e-03, 9.90739315e-01, 1.01277687e-04,
                           9.77373117e-06, 9.81022926e-06, 2.84591507e-05, 3.55887437e-05,
                           3.55854099e-05, 1.60913335e-06, 8.13373405e-07, 8.13344229e-07],
                          [6.89108562e-03, 6.89169898e-03, 9.72726380e-01, 4.12269769e-04,
                           1.63740157e-04, 1.64434287e-04, 8.38836333e-04, 2.59703351e-03,
                           2.59687551e-03, 1.21277436e-04, 3.29820118e-03, 3.29816718e-03]]
    ref_hv_overtones = [[1.77252852e-03, 1.77269632e-03, 2.76560486e-03, 1.63031336e-05,
                         4.94642690e-01, 4.93746287e-01, 5.21685650e-03, 2.95291520e-05,
                         2.95205546e-05, 5.95722443e-06, 1.01334585e-06, 1.01348485e-06],
                        [1.64208520e-04, 1.64225582e-04, 1.85764556e-04, 7.89315101e-06,
                         4.97276353e-01, 4.96282471e-01, 5.47392488e-03, 8.40497150e-05,
                         8.40233314e-05, 4.61113630e-05, 1.15481473e-04, 1.15493147e-04]]
    ref_hv_combotones = \
     [[[0.00000000e+00, 9.46488377e-02, 1.55279671e-01, 2.74898477e-01,
        1.93152927e-01, 4.29180022e-01, 3.70296882e-01, 1.23042640e-01,
        2.97141364e-01, 1.88288412e-01, 2.43765812e-01, 1.31034109e-01],
       [9.46532330e-02, 0.00000000e+00, 1.55286929e-01, 2.74949174e-01,
        4.29197409e-01, 1.93170398e-01, 3.70259365e-01, 2.97127679e-01,
        1.23092688e-01, 1.88264932e-01, 1.31037315e-01, 2.43803824e-01],
       [1.20296000e-01, 1.20296036e-01, 0.00000000e+00, 1.39341777e-01,
        6.91332403e-02, 6.91242259e-02, 1.33786453e-01, 3.63352081e-01,
        3.63383077e-01, 3.45022864e-01, 3.53638620e-01, 3.53720829e-01],
       [3.15408098e-02, 3.15451617e-02, 2.06369383e-02, 0.00000000e+00,
        7.51602418e-03, 7.51523213e-03, 8.81831392e-03, 2.13654269e-02,
        2.13658165e-02, 1.52934409e-01, 1.38425536e-02, 1.38428495e-02],
       [1.11349841e-01, 2.47414546e-01, 5.14444624e-02, 3.77638242e-02,
        0.00000000e+00, 2.40421087e-01, 4.32050841e-02, 1.61003788e-02,
        2.21743344e-02, 5.30688751e-03, 1.56654587e-03, 9.92877955e-04],
       [2.47406456e-01, 1.11350440e-01, 5.14357678e-02, 3.77583862e-02,
        2.40411801e-01, 0.00000000e+00, 4.31998913e-02, 2.21727490e-02,
        1.61050644e-02, 5.30754796e-03, 9.92926616e-04, 1.56700545e-03],
       [1.86465043e-01, 1.86437493e-01, 8.69606800e-02, 3.87019119e-02,
        3.77393070e-02, 3.77362286e-02, 0.00000000e+00, 1.93002884e-02,
        1.93030272e-02, 1.02084567e-01, 5.16053069e-04, 5.16019809e-04],
       [4.05048318e-02, 9.78079454e-02, 1.54398210e-01, 6.13002521e-02,
        9.19388322e-03, 1.26619094e-02, 1.26173346e-02, 0.00000000e+00,
        2.13771549e-03, 6.00702708e-03, 1.26751141e-01, 1.27526831e-01],
       [9.77973878e-02, 4.05113047e-02, 1.54380434e-01, 6.12890840e-02,
        1.26597879e-02, 9.19507080e-03, 1.26165959e-02, 2.13728706e-03,
        0.00000000e+00, 6.00849367e-03, 1.27461538e-01, 1.26568095e-01],
       [4.49709529e-03, 4.49632569e-03, 1.06370154e-02, 3.18356119e-02,
        2.19867012e-04, 2.19902868e-04, 4.84196021e-03, 4.35829773e-04,
        4.36023566e-04, 0.00000000e+00, 5.29412249e-05, 5.29758917e-05],
       [4.25947092e-02, 2.28958979e-02, 7.97638237e-02, 2.10813574e-02,
        4.74829479e-04, 3.00973656e-04, 1.79072896e-04, 6.72796227e-02,
        6.76702645e-02, 3.87318597e-04, 0.00000000e+00, 3.74583619e-04],
       [2.28945940e-02, 4.25960111e-02, 7.97760694e-02, 2.10801443e-02,
        3.00923531e-04, 4.74949640e-04, 1.79047223e-04, 6.76860171e-02,
        6.71906259e-02, 3.87541632e-04, 3.74554056e-04, 0.00000000e+00]],
      [[0.00000000e+00, 5.09283065e-02, 9.53002432e-02, 1.20856534e-01,
        8.73558220e-02, 1.61007099e-01, 2.38697952e-01, 1.13480166e-02,
        3.13604175e-02, 1.40608550e-01, 2.44805266e-02, 1.54731601e-02],
       [5.09320176e-02, 0.00000000e+00, 9.53105647e-02, 1.20876784e-01,
        1.61023944e-01, 8.73622408e-02, 2.38682384e-01, 3.13367693e-02,
        1.13604186e-02, 1.40601760e-01, 1.54669300e-02, 2.44947767e-02],
       [7.30892585e-02, 7.30918482e-02, 0.00000000e+00, 7.31508773e-02,
        2.21352911e-02, 2.21374019e-02, 7.29588119e-02, 2.60910378e-02,
        2.61106250e-02, 1.31358251e-01, 2.91043176e-02, 2.91240113e-02],
       [2.52275455e-02, 2.52299341e-02, 1.99097064e-02, 0.00000000e+00,
        1.06267436e-02, 1.06261011e-02, 2.16787686e-02, 5.17943633e-03,
        5.18311685e-03, 2.27465160e-01, 3.60078135e-03, 3.60224436e-03],
       [1.04033805e-01, 1.91752679e-01, 3.43722781e-02, 6.06286594e-02,
        0.00000000e+00, 5.43411172e-01, 1.17283935e-01, 9.45192897e-03,
        1.43001322e-02, 1.85974314e-02, 1.90171863e-03, 1.39552247e-03],
       [1.91743922e-01, 1.04032420e-01, 3.43750774e-02, 6.06241501e-02,
        5.43403608e-01, 0.00000000e+00, 1.17270899e-01, 1.42875452e-02,
        9.46117262e-03, 1.85994495e-02, 1.39509152e-03, 1.90330874e-03],
       [1.76671862e-01, 1.76647467e-01, 7.04104141e-02, 7.68685058e-02,
        7.28911155e-02, 7.28840276e-02, 0.00000000e+00, 1.31197553e-02,
        1.31298855e-02, 1.93022717e-01, 6.38267180e-04, 6.38460591e-04],
       [5.06181021e-02, 1.39768232e-01, 1.51746157e-01, 1.10678568e-01,
        3.54016675e-02, 5.35139407e-02, 7.90665773e-02, 0.00000000e+00,
        2.01640686e-03, 4.53343618e-02, 4.55953315e-01, 4.57344251e-01],
       [1.39759480e-01, 5.06246611e-02, 1.51725005e-01, 1.10658703e-01,
        5.35127002e-02, 3.54052630e-02, 7.90572469e-02, 2.01461336e-03,
        0.00000000e+00, 4.53371191e-02, 4.56914041e-01, 4.55473563e-01],
       [1.50095206e-02, 1.50077022e-02, 1.82832290e-02, 1.16322871e-01,
        1.66696096e-03, 1.66716506e-03, 2.78384353e-02, 1.08491746e-03,
        1.08594935e-03, 0.00000000e+00, 4.81635956e-04, 4.82208025e-04],
       [1.05969476e-01, 6.69472135e-02, 1.64269867e-01, 7.46709916e-02,
        6.91231729e-03, 5.07091287e-03, 3.73287849e-03, 4.42480884e-01,
        4.43807969e-01, 1.95309713e-02, 0.00000000e+00, 1.00684931e-02],
       [6.69450110e-02, 1.05969537e-01, 1.64297457e-01, 7.46633554e-02,
        5.06983010e-03, 6.91467635e-03, 3.73211141e-03, 4.43605095e-01,
        4.42183906e-01, 1.95442289e-02, 1.00633747e-02, 0.00000000e+00]]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, \
                                                             'Anharmonic: VPT2')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert np.allclose(ref_ih, properties.input_hyperraman_frequencies)
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert ref_harmonic_raman == properties.raman.harmonic
    assert np.allclose(ref_vv_harmonic, properties.hyper_raman_vv.harmonic)
    assert np.allclose(ref_hv_harmonic, properties.hyper_raman_hv.harmonic)
    assert np.allclose(ref_vv_fundamental, properties.hyper_raman_vv.fundamental)
    assert np.allclose(ref_hv_fundamental, properties.hyper_raman_hv.fundamental)
    assert np.allclose(ref_vv_overtones, properties.hyper_raman_vv.overtones)
    assert np.allclose(ref_hv_overtones, properties.hyper_raman_hv.overtones)
    assert np.allclose(ref_vv_combotones, properties.hyper_raman_vv.combotones)
    assert np.allclose(ref_hv_combotones, properties.hyper_raman_hv.combotones)

    # DVPT frequencies, VPT2 intensities
    ref_fundamental = [314012.43298487, 314011.53979323, 306448.98120075, 256465.52677474,
                       155036.09521096, 155035.65129466, 150041.13718027, 113543.31474027,
                       113542.65561761,  95712.27989893,  42042.98978884,  42042.96229487]
    ref_overtones = [622382.43837506, 622381.06278219, 609848.3856535 , 510942.8909639,
                     309486.29869424, 309485.28534106, 299991.52740468, 226811.20363659,
                     226809.79089787, 190623.68916583,  83664.32840753,  83664.35715109]
    ref_combotones = \
[[     0.        , 624705.06507938, 613925.51329906, 570460.42997406,
  468597.90570357, 466687.19177635, 463345.23626701, 426758.64359099,
  426878.70636511, 409933.29257683, 356095.0069167 , 355979.95300053],
 [624705.06507938,      0.        , 613923.7292351 , 570459.54133761,
  466686.76888728, 468596.50808828, 463344.37693254, 426878.52019834,
  426756.96397891, 409932.37194193, 355979.07076914, 356094.0744635 ],
 [613925.51329906, 613923.7292351 ,      0.        , 562864.90570708,
  459602.43486588, 459602.08093223, 454413.31094192, 419477.45208183,
  419476.8233521 , 402242.10857965, 348485.84801389, 348485.83123741],
 [570460.42997406, 570459.54133761, 562864.90570708,      0.,
  411405.53335585, 411405.09295387, 406517.23180163, 369746.20969394,
  369745.52049082, 352217.88792428, 297874.57456537, 297874.54602396],
 [468597.90570357, 466686.76888728, 459602.43486588, 411405.53335585,
       0.        , 309330.06210978, 304463.72696112, 268025.2310408,
  268003.30448199, 250669.64655308, 197046.7019944 , 197044.43775159],
 [466687.19177635, 468596.50808828, 459602.08093223, 411405.09295387,
  309330.06210978,      0.        , 304463.31407033, 268003.49072155,
  268024.1787763 , 250669.20874738, 197044.07188409, 197046.2168526 ],
 [463345.23626701, 463344.37693254, 454413.31094192, 406517.23180163,
  304463.72696112, 304463.31407033,      0.        , 263064.20459946,
  263063.48794891, 245037.34667131, 192018.76030679, 192018.75687982],
 [426758.64359099, 426878.52019834, 419477.45208183, 369746.20969394,
  268025.2310408 , 268003.49072155, 263064.20459946,      0.,
  227174.58106829, 209062.63358751, 155468.34808863, 155531.18063685],
 [426878.70636511, 426756.96397891, 419476.8233521 , 369745.52049082,
  268003.30448199, 268024.1787763 , 263063.48794891, 227174.58106829,
       0.        , 209062.00120743, 155530.40294951, 155467.72650885],
 [409933.29257683, 409932.37194193, 402242.10857965, 352217.88792428,
  250669.64655308, 250669.20874738, 245037.34667131, 209062.63358751,
  209062.00120743,      0.        , 138665.5741924 , 138665.52339525],
 [356095.0069167 , 355979.07076914, 348485.84801389, 297874.57456537,
  197046.7019944 , 197044.07188409, 192018.76030679, 155468.34808863,
  155530.40294951, 138665.5741924 ,      0.        ,  84185.97021186],
 [355979.95300053, 356094.0744635 , 348485.83123741, 297874.54602396,
  197044.43775159, 197046.2168526 , 192018.75687982, 155531.18063685,
  155467.72650885, 138665.52339525,  84185.97021186,      0.        ]]

    ref_vv_fundamental = [[1.07903777e-02, 1.07911798e-02, 9.78120573e-01, 1.59316533e-04,
                           1.05235590e-05, 1.05715147e-05, 2.64651133e-05, 4.04938970e-05,
                           4.04932086e-05, 6.55333560e-06, 1.72599094e-06, 1.72597052e-06],
                          [1.09105357e-02, 1.09115086e-02, 9.62732409e-01, 6.30015706e-04,
                           1.57446891e-04, 1.58296552e-04, 7.71620610e-04, 2.05908246e-03,
                           2.05914897e-03, 3.45832841e-04, 4.63201084e-03, 4.63209190e-03]]
    ref_vv_overtones = [[4.24316967e-03, 4.24346649e-03, 2.10524367e-03, 1.29624549e-05,
                         4.93092370e-01, 4.92106520e-01, 4.13572651e-03, 1.95260849e-05,
                         1.95211735e-05, 1.88306912e-05, 1.33185162e-06, 1.33194577e-06],
                        [4.45145328e-04, 4.45185304e-04, 2.23769433e-04, 1.39234138e-05,
                         4.97449421e-01, 4.96456524e-01, 4.41157399e-03, 5.06413564e-05,
                         5.06268475e-05, 1.68966113e-04, 1.42108552e-04, 1.42115159e-04]]
    ref_vv_combotones = \
     [[[0.00000000e+00, 1.60498259e-01, 1.80089946e-01, 1.67410376e-01,
        2.22799773e-01, 2.87069446e-01, 2.70024816e-01, 7.92922719e-02,
        4.91530952e-01, 1.28969787e-01, 4.14592824e-01, 9.54609607e-02],
       [1.60507453e-01, 0.00000000e+00, 1.80105996e-01, 1.67455904e-01,
        2.87085109e-01, 2.22798742e-01, 2.70003429e-01, 4.91478560e-01,
        7.93272169e-02, 1.28969674e-01, 9.54603501e-02, 4.14683965e-01],
       [1.44361841e-01, 1.44366438e-01, 0.00000000e+00, 2.88656475e-01,
        1.12331583e-01, 1.12325619e-01, 3.32123662e-01, 1.75084445e-01,
        1.75108940e-01, 4.35781261e-01, 1.56058787e-01, 1.56099507e-01],
       [2.47347202e-02, 2.47400296e-02, 5.32038410e-02, 0.00000000e+00,
        5.76371918e-03, 5.76312548e-03, 2.46937648e-02, 1.93593642e-02,
        1.93607746e-02, 2.57464522e-01, 3.61808791e-02, 3.61816023e-02],
       [1.12951390e-01, 1.45533383e-01, 7.10420962e-02, 1.97767543e-02,
        0.00000000e+00, 3.09057615e-01, 3.44902160e-02, 1.33026451e-02,
        3.05697629e-02, 3.07474529e-03, 8.15291658e-04, 5.16823756e-04],
       [1.45522818e-01, 1.12935891e-01, 7.10329740e-02, 1.97732279e-02,
        3.09034338e-01, 0.00000000e+00, 3.44853017e-02, 3.05628738e-02,
        1.33054094e-02, 3.07386796e-03, 5.16823808e-04, 8.15529658e-04],
       [1.40372529e-01, 1.40353370e-01, 2.15384932e-01, 8.68842631e-02,
        3.53669431e-02, 3.53645674e-02, 0.00000000e+00, 2.51220762e-02,
        2.51249671e-02, 3.84139937e-02, 9.00000415e-04, 8.99912704e-04],
       [2.46262571e-02, 1.52632581e-01, 6.78347404e-02, 4.06943399e-02,
        8.14945786e-03, 1.87248183e-02, 1.50087537e-02, 0.00000000e+00,
        1.33332222e-03, 1.85655062e-03, 1.47180856e-01, 1.48087217e-01],
       [1.52617036e-01, 2.46291534e-02, 6.78262048e-02, 4.06864916e-02,
        1.87226526e-02, 8.14959940e-03, 1.50064927e-02, 1.33296796e-03,
        0.00000000e+00, 1.85673816e-03, 1.48001145e-01, 1.46961394e-01],
       [6.22323741e-03, 6.22287551e-03, 2.62321565e-02, 8.40855080e-02,
        2.92658200e-04, 2.92596732e-04, 3.56565569e-03, 2.88448069e-04,
        2.88553873e-04, 0.00000000e+00, 7.52737119e-05, 7.52976572e-05],
       [7.15985648e-02, 1.64846861e-02, 3.36207826e-02, 4.22899355e-02,
        2.77727382e-04, 1.76068200e-04, 2.98983211e-04, 8.18401305e-02,
        8.23181247e-02, 2.69399932e-04, 0.00000000e+00, 2.17789801e-04],
       [1.64841548e-02, 7.16033346e-02, 3.36263301e-02, 4.22867249e-02,
        1.76038037e-04, 2.77802736e-04, 2.98925402e-04, 8.23362166e-02,
        8.17319768e-02, 2.69459786e-04, 2.17768914e-04, 0.00000000e+00]],
      [[0.00000000e+00, 8.33963149e-02, 1.23709550e-01, 6.62376946e-02,
        9.57017331e-02, 8.52955038e-02, 1.64339139e-01, 4.82147182e-03,
        3.14864062e-02, 1.09701386e-01, 2.90172641e-02, 7.60281318e-03],
       [8.34034135e-02, 0.00000000e+00, 1.23727843e-01, 6.62550897e-02,
        8.53070414e-02, 9.56992210e-02, 1.64330409e-01, 3.14622525e-02,
        4.82687409e-03, 1.09704683e-01, 7.59973816e-03, 2.90358273e-02],
       [1.07105518e-01, 1.07112238e-01, 0.00000000e+00, 1.13369844e-01,
        5.59625825e-02, 5.59607103e-02, 2.25058461e-01, 1.06843609e-02,
        1.06927586e-02, 2.49720915e-01, 1.18029308e-02, 1.18110458e-02],
       [1.97533257e-02, 1.97568316e-02, 3.90502615e-02, 0.00000000e+00,
        6.06890614e-03, 6.06877515e-03, 4.32581192e-02, 2.77320338e-03,
        2.77519998e-03, 3.32168625e-01, 7.81370885e-03, 7.81788390e-03],
       [1.21248526e-01, 1.08069857e-01, 8.18928137e-02, 2.57829180e-02,
        0.00000000e+00, 5.95649650e-01, 9.16373946e-02, 6.32015327e-03,
        1.51928071e-02, 6.46594405e-03, 7.66909908e-04, 5.53455079e-04],
       [1.08060445e-01, 1.21230543e-01, 8.18870475e-02, 2.57814086e-02,
        5.95627637e-01, 0.00000000e+00, 9.16254120e-02, 1.51777792e-02,
        6.32554979e-03, 6.46557521e-03, 5.53314683e-04, 7.67562239e-04],
       [1.32073816e-01, 1.32055560e-01, 2.08911547e-01, 1.16575781e-01,
        5.81288577e-02, 5.81234048e-02, 0.00000000e+00, 1.32209276e-02,
        1.32310393e-02, 8.79293692e-02, 1.01140590e-03, 1.01181944e-03],
       [2.74334460e-02, 1.79000225e-01, 7.02167400e-02, 5.29111786e-02,
        2.83838745e-02, 6.81660844e-02, 9.36023901e-02, 0.00000000e+00,
        7.91633712e-04, 2.07074984e-02, 4.68170980e-01, 4.68873718e-01],
       [1.78985431e-01, 2.74361769e-02, 7.02062433e-02, 5.28997788e-02,
        6.81672777e-02, 2.83826051e-02, 9.35864185e-02, 7.90893740e-04,
        0.00000000e+00, 2.07043064e-02, 4.68417776e-01, 4.67677218e-01],
       [1.86378212e-02, 1.86367951e-02, 4.90037512e-02, 1.89237246e-01,
        8.67078276e-04, 8.67060859e-04, 1.85883681e-02, 6.18315213e-04,
        6.18798316e-04, 0.00000000e+00, 8.63316146e-04, 8.63942602e-04],
       [1.61108129e-01, 4.21912756e-02, 7.56906501e-02, 1.45473269e-01,
        3.36084650e-03, 2.42489297e-03, 6.98732198e-03, 4.56840660e-01,
        4.57509135e-01, 2.82129064e-02, 0.00000000e+00, 3.98471415e-03],
       [4.21901285e-02, 1.61114182e-01, 7.57035530e-02, 1.45475791e-01,
        2.42416537e-03, 3.36209141e-03, 6.98656703e-03, 4.57289982e-01,
        4.56549798e-01, 2.82187903e-02, 3.98265520e-03, 0.00000000e+00]]]

    ref_hv_fundamental = [[6.13056876e-03, 6.13099653e-03, 9.87437052e-01, 1.37416103e-04,
                           1.21732236e-05, 1.22183064e-05, 3.86140886e-05, 4.82876990e-05,
                           4.82831756e-05, 2.18331245e-06, 1.10360541e-06, 1.10356583e-06],
                          [6.73108288e-03, 6.73168199e-03, 9.73358595e-01, 4.02697360e-04,
                           1.60460001e-04, 1.61140415e-04, 8.19359559e-04, 2.53673351e-03,
                           2.53657918e-03, 1.18461520e-04, 3.22162091e-03, 3.22158769e-03]]
    ref_hv_overtones = [[1.67913067e-03, 1.67928962e-03, 1.92159918e-03, 1.54440909e-05,
                         4.95252615e-01, 4.94268353e-01, 5.12006626e-03, 2.79732056e-05,
                         2.79650612e-05, 5.64332710e-06, 9.59950753e-07, 9.60082434e-07],
                        [1.64889630e-04, 1.64906763e-04, 1.96559114e-04, 7.92589053e-06,
                         4.97261804e-01, 4.96274469e-01, 5.48244028e-03, 8.43983397e-05,
                         8.43718466e-05, 4.63026255e-05, 1.15960472e-04, 1.15972194e-04]]
    ref_hv_combotones = \
     [[[0.00000000e+00, 9.73355868e-02, 1.67117634e-01, 2.81746449e-01,
        2.00597523e-01, 4.45664163e-01, 3.83240365e-01, 1.32410420e-01,
        3.19768910e-01, 2.03202206e-01, 2.63055724e-01, 1.41407620e-01],
       [9.73402121e-02, 0.00000000e+00, 1.67125370e-01, 2.81798408e-01,
        4.45687814e-01, 2.00613133e-01, 3.83201536e-01, 3.19749322e-01,
        1.32466292e-01, 2.03176866e-01, 1.41406686e-01, 2.63104918e-01],
       [1.05740632e-01, 1.05740502e-01, 0.00000000e+00, 1.20378221e-01,
        4.89309246e-02, 4.89323953e-02, 1.07346972e-01, 3.11397266e-01,
        3.11428462e-01, 2.93737720e-01, 2.90592366e-01, 2.90668946e-01],
       [3.24361781e-02, 3.24406183e-02, 2.19027943e-02, 0.00000000e+00,
        7.77491159e-03, 7.77398349e-03, 9.12655226e-03, 2.29920712e-02,
        2.29928400e-02, 1.65047912e-01, 1.49379560e-02, 1.49387394e-02],
       [1.11287223e-01, 2.47246335e-01, 4.29025928e-02, 3.74665733e-02,
        0.00000000e+00, 2.35769098e-01, 4.27962908e-02, 1.60829711e-02,
        2.21506673e-02, 5.43042805e-03, 1.52823136e-03, 9.68616993e-04],
       [2.47235980e-01, 1.11286552e-01, 4.29023236e-02, 3.74607399e-02,
        2.35760533e-01, 0.00000000e+00, 4.27904721e-02, 2.21478696e-02,
        1.60876746e-02, 5.43099995e-03, 9.68588478e-04, 1.52868922e-03],
       [1.91758340e-01, 1.91729801e-01, 8.48893470e-02, 3.96660118e-02,
        3.85984092e-02, 3.85945634e-02, 0.00000000e+00, 2.07697046e-02,
        2.07729677e-02, 1.10170397e-01, 5.56889879e-04, 5.56871288e-04],
       [4.16546672e-02, 1.00584370e-01, 1.54823428e-01, 6.28272972e-02,
        9.11985743e-03, 1.25594176e-02, 1.30583652e-02, 0.00000000e+00,
        2.30050419e-03, 6.48282674e-03, 1.42849841e-01, 1.43721613e-01],
       [1.00573622e-01, 4.16612788e-02, 1.54805552e-01, 6.28158509e-02,
        1.25578395e-02, 9.12088890e-03, 1.30576007e-02, 2.30000816e-03,
        0.00000000e+00, 6.48440949e-03, 1.43642393e-01, 1.42642578e-01],
       [4.62475708e-03, 4.62396063e-03, 1.05657567e-02, 3.26286660e-02,
        2.22779298e-04, 2.22810855e-04, 5.01120774e-03, 4.69011418e-04,
        4.69227100e-04, 0.00000000e+00, 5.71306210e-05, 5.71698073e-05],
       [4.38038713e-02, 2.35458323e-02, 7.64767310e-02, 2.16065132e-02,
        4.58705668e-04, 2.90736846e-04, 1.85332271e-04, 7.56141359e-02,
        7.60500522e-02, 4.17997010e-04, 0.00000000e+00, 4.04238092e-04],
       [2.35445168e-02, 4.38051627e-02, 7.64884711e-02, 2.16052698e-02,
        2.90702865e-04, 4.58809296e-04, 1.85305700e-04, 7.60672200e-02,
        7.55124032e-02, 4.18237711e-04, 4.04193630e-04, 0.00000000e+00]],
      [[0.00000000e+00, 5.07785651e-02, 9.51995653e-02, 1.20576860e-01,
        8.70999590e-02, 1.60535404e-01, 2.37976972e-01, 1.13556589e-02,
        3.13814905e-02, 1.40130823e-01, 2.44990067e-02, 1.54848239e-02],
       [5.07822646e-02, 0.00000000e+00, 9.52098746e-02, 1.20597064e-01,
        1.60552041e-01, 8.71064453e-02, 2.37961451e-01, 3.13578732e-02,
        1.13680524e-02, 1.40124056e-01, 1.54786058e-02, 2.45132410e-02],
       [7.48025091e-02, 7.48051597e-02, 0.00000000e+00, 7.48864041e-02,
        2.31475905e-02, 2.31492966e-02, 7.49884633e-02, 2.67650425e-02,
        2.67850957e-02, 1.34188947e-01, 2.98455321e-02, 2.98656951e-02],
       [2.51533701e-02, 2.51557521e-02, 1.98817216e-02, 0.00000000e+00,
        1.05950960e-02, 1.05944658e-02, 2.16132886e-02, 5.18292444e-03,
        5.18659970e-03, 2.26692332e-01, 3.60349953e-03, 3.60495975e-03],
       [1.04083935e-01, 1.91844760e-01, 3.52038645e-02, 6.06929762e-02,
        0.00000000e+00, 5.43535738e-01, 1.17314859e-01, 9.50466776e-03,
        1.43798961e-02, 1.85945178e-02, 1.91235570e-03, 1.40332711e-03],
       [1.91836232e-01, 1.04082679e-01, 3.52059777e-02, 6.06885359e-02,
        5.43528304e-01, 0.00000000e+00, 1.17301957e-01, 1.43672982e-02,
        9.51395729e-03, 1.85965574e-02, 1.40289846e-03, 1.91395492e-03],
       [1.76152402e-01, 1.76128081e-01, 7.06426346e-02, 7.66906245e-02,
        7.26676149e-02, 7.26606173e-02, 0.00000000e+00, 1.31285909e-02,
        1.31387083e-02, 1.92366909e-01, 6.38749000e-04, 6.38941865e-04],
       [5.04692720e-02, 1.39357280e-01, 1.51391481e-01, 1.10422447e-01,
        3.53497129e-02, 5.34355145e-02, 7.88277592e-02, 0.00000000e+00,
        2.01776180e-03, 4.51803352e-02, 4.55552891e-01, 4.56942687e-01],
       [1.39348552e-01, 5.04758126e-02, 1.51370378e-01, 1.10402628e-01,
        5.34341481e-02, 3.53533261e-02, 7.88184569e-02, 2.01597011e-03,
        0.00000000e+00, 4.51830831e-02, 4.56513491e-01, 4.55073715e-01],
       [1.49653888e-02, 1.49635760e-02, 1.82385318e-02, 1.16053689e-01,
        1.66177990e-03, 1.66198491e-03, 2.77543502e-02, 1.08564811e-03,
        1.08667907e-03, 0.00000000e+00, 4.81999537e-04, 4.82571516e-04],
       [1.05657899e-01, 6.67503728e-02, 1.63814229e-01, 7.44981955e-02,
        6.90170545e-03, 5.06313986e-03, 3.72160344e-03, 4.42056318e-01,
        4.43382165e-01, 1.94646135e-02, 0.00000000e+00, 1.00760828e-02],
       [6.67481757e-02, 1.05657961e-01, 1.63841742e-01, 7.44905771e-02,
        5.06204842e-03, 6.90406743e-03, 3.72083869e-03, 4.43180008e-01,
        4.41759594e-01, 1.94778261e-02, 1.00709714e-02, 0.00000000e+00]]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, \
                                                             'Anharmonic: Freq DVPT2, Int VPT2')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert np.allclose(ref_ih, properties.input_hyperraman_frequencies)
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert ref_harmonic_raman == properties.raman.harmonic
    assert np.allclose(ref_vv_harmonic, properties.hyper_raman_vv.harmonic)
    assert np.allclose(ref_hv_harmonic, properties.hyper_raman_hv.harmonic)
    assert np.allclose(ref_vv_fundamental, properties.hyper_raman_vv.fundamental)
    assert np.allclose(ref_hv_fundamental, properties.hyper_raman_hv.fundamental)
    assert np.allclose(ref_vv_overtones, properties.hyper_raman_vv.overtones)
    assert np.allclose(ref_hv_overtones, properties.hyper_raman_hv.overtones)
    assert np.allclose(ref_vv_combotones, properties.hyper_raman_vv.combotones)
    assert np.allclose(ref_hv_combotones, properties.hyper_raman_hv.combotones)

    # DVPT2
    ref_vv_fundamental = [[6.04977001e-05, 6.05021974e-05, 4.70230386e-05, 8.93229516e-07,
                           4.99735175e-01, 5.00095250e-01, 1.48380208e-07, 2.27034466e-07,
                           2.27030606e-07, 3.67421552e-08, 9.67699977e-09, 9.67688524e-09],
                          [2.96454107e-06, 2.96480542e-06, 2.62963151e-06, 1.71183844e-07,
                           4.99805411e-01, 5.00181919e-01, 2.09659824e-07, 5.59480735e-07,
                           5.59498807e-07, 9.39674906e-08, 1.25858040e-06, 1.25860243e-06]]
    ref_vv_overtones = [[2.98193580e-01, 2.98214439e-01, 1.47948396e-01, 9.10951281e-04,
                         1.04763315e-01, 1.04810079e-01, 4.09045996e-02, 1.37221785e-03,
                         1.37187269e-03, 1.32334827e-03, 9.35973887e-05, 9.36040054e-05],
                        [0.07890154, 0.07890862, 0.03966289, 0.00246791, 0.28605115,
                         0.28617112, 0.12955967, 0.00897613, 0.00897355, 0.02994906,
                         0.02518859, 0.02518976]]
    ref_vv_combotones = \
     [[[0.00000000e+00, 1.60498259e-01, 1.80089946e-01, 1.67410376e-01,
        2.22799773e-01, 2.87069446e-01, 2.70024816e-01, 9.41849812e-02,
        5.83763221e-01, 1.28969787e-01, 5.79276920e-01, 1.33355920e-01],
       [1.60507453e-01, 0.00000000e+00, 1.80105996e-01, 1.67455904e-01,
        2.87085109e-01, 2.22798742e-01, 2.70003429e-01, 5.83788280e-01,
        9.42124020e-02, 1.28969674e-01, 1.33379003e-01, 5.79300285e-01],
       [1.44361841e-01, 1.44366438e-01, 0.00000000e+00, 2.88656475e-01,
        1.12331583e-01, 1.12325619e-01, 3.32123662e-01, 2.07968883e-01,
        2.07966880e-01, 4.35781261e-01, 2.18048283e-01, 2.18066038e-01],
       [2.47347202e-02, 2.47400296e-02, 5.32038410e-02, 0.00000000e+00,
        5.76371918e-03, 5.76312548e-03, 2.46937648e-02, 2.29954485e-02,
        2.29936855e-02, 2.57464522e-01, 5.05526073e-02, 5.05445455e-02],
       [1.12951390e-01, 1.45533383e-01, 7.10420962e-02, 1.97767543e-02,
        0.00000000e+00, 3.09057615e-01, 3.44902160e-02, 1.58011538e-02,
        3.63059603e-02, 3.07474529e-03, 1.13914090e-03, 7.21986319e-04],
       [1.45522818e-01, 1.12935891e-01, 7.10329740e-02, 1.97732279e-02,
        3.09034338e-01, 0.00000000e+00, 3.44853017e-02, 3.63032062e-02,
        1.58020743e-02, 3.07386796e-03, 7.22115981e-04, 1.13926895e-03],
       [1.40372529e-01, 1.40353370e-01, 2.15384932e-01, 8.68842631e-02,
        3.53669431e-02, 3.53645674e-02, 0.00000000e+00, 2.98405155e-02,
        2.98394876e-02, 3.84139937e-02, 1.25749757e-03, 1.25714937e-03],
       [2.46262571e-02, 1.52632581e-01, 6.78347404e-02, 4.06943399e-02,
        8.14945786e-03, 1.87248183e-02, 1.50087537e-02, 0.00000000e+00,
        1.58351060e-03, 1.85655062e-03, 7.00081537e-03, 8.21178505e-03],
       [1.52617036e-01, 2.46291534e-02, 6.78262048e-02, 4.06864916e-02,
        1.87226526e-02, 8.14959940e-03, 1.50064927e-02, 1.58332659e-03,
        0.00000000e+00, 1.85673816e-03, 8.21417161e-03, 6.99358764e-03],
       [6.22323741e-03, 6.22287551e-03, 2.62321565e-02, 8.40855080e-02,
        2.92658200e-04, 2.92596732e-04, 3.56565569e-03, 3.42624512e-04,
        3.42698944e-04, 0.00000000e+00, 1.05173851e-04, 1.05188428e-04],
       [7.15985648e-02, 1.64846861e-02, 3.36207826e-02, 4.22899355e-02,
        2.77727382e-04, 1.76068200e-04, 2.98983211e-04, 3.30940533e-03,
        3.88343087e-03, 2.69399932e-04, 0.00000000e+00, 3.04245412e-04],
       [1.64841548e-02, 7.16033346e-02, 3.36263301e-02, 4.22867249e-02,
        1.76038037e-04, 2.77802736e-04, 2.98925402e-04, 3.88217584e-03,
        3.30664918e-03, 2.69459786e-04, 3.04270837e-04, 0.00000000e+00]],
      [[0.        , 0.08339631, 0.12370955, 0.06623769, 0.09570173,
        0.0852955 , 0.16433914, 0.04149506, 0.27077094, 0.10970139,
        0.30708604, 0.08042974],
       [0.08340341, 0.        , 0.12372784, 0.06625509, 0.08530704,
        0.09569922, 0.16433041, 0.27077379, 0.04150925, 0.10970468,
        0.08042707, 0.30716841],
       [0.10710552, 0.10711224, 0.        , 0.11336984, 0.05596258,
        0.05596071, 0.22505846, 0.09195288, 0.0919536 , 0.24972092,
        0.12490893, 0.1249484 ],
       [0.01975333, 0.01975683, 0.03905026, 0.        , 0.00606891,
        0.00606878, 0.04325812, 0.02386704, 0.02386565, 0.33216863,
        0.0826915 , 0.08270496],
       [0.12124853, 0.10806986, 0.08189281, 0.02578292, 0.        ,
        0.59564965, 0.09163739, 0.05439318, 0.13065228, 0.00646594,
        0.00811611, 0.00585497],
       [0.10806045, 0.12123054, 0.08188705, 0.02578141, 0.59562764,
        0.        , 0.09162541, 0.13062462, 0.05439729, 0.00646558,
        0.00585566, 0.00812   ],
       [0.13207382, 0.13205556, 0.20891155, 0.11657578, 0.05812886,
        0.0581234 , 0.        , 0.11378335, 0.11378183, 0.08792937,
        0.01070358, 0.01070398],
       [0.02743345, 0.17900023, 0.07021674, 0.05291118, 0.02838387,
        0.06816608, 0.09360239, 0.        , 0.00680774, 0.0207075 ,
        0.1584736 , 0.17043093],
       [0.17898543, 0.02743618, 0.07020624, 0.05289978, 0.06816728,
        0.02838261, 0.09358642, 0.00680667, 0.        , 0.02070431,
        0.17045321, 0.15834493],
       [0.01863782, 0.0186368 , 0.04900375, 0.18923725, 0.00086708,
        0.00086706, 0.01858837, 0.00532141, 0.00532143, 0.        ,
        0.00913637, 0.0091396 ],
       [0.16110813, 0.04219128, 0.07569065, 0.14547327, 0.00336085,
        0.00242489, 0.00698732, 0.12575648, 0.13528415, 0.02821291,
        0.        , 0.04215407],
       [0.04219013, 0.16111418, 0.07570355, 0.14547579, 0.00242417,
        0.00336209, 0.00698657, 0.13522552, 0.12565585, 0.02821879,
        0.04214794, 0.        ]]]

    ref_hv_fundamental = [[4.40800845e-05, 4.40831603e-05, 5.80632509e-05, 9.88050812e-07,
                           4.99709734e-01, 5.00142048e-01, 2.77643454e-07, 3.47198757e-07,
                           3.47166233e-07, 1.56984777e-08, 7.93515607e-09, 7.93487143e-09],
                          [2.72532175e-06, 2.72556432e-06, 3.68801701e-06, 1.63046555e-07,
                           4.99777627e-01, 5.00208029e-01, 3.31747279e-07, 1.02708808e-06,
                           1.02702559e-06, 4.79634201e-08, 1.30438946e-06, 1.30437601e-06]]
    ref_hv_overtones = [[2.12849937e-01, 2.12870087e-01, 2.43585727e-01, 1.95772362e-03,
                         1.45700376e-01, 1.45735435e-01, 2.92511218e-02, 3.54593908e-03,
                         3.54490668e-03, 7.15359347e-04, 1.21685263e-04, 1.21701955e-04],
                        [0.04161802, 0.04162235, 0.04961138, 0.00200049, 0.33530567,
                         0.33538943, 0.08162881, 0.02130208, 0.02129539, 0.01168675,
                         0.02926834, 0.0292713 ]]
    ref_hv_combotones = \
     [[[0.00000000e+00, 9.73355868e-02, 1.67117634e-01, 2.81746449e-01,
        2.00597523e-01, 4.45664163e-01, 3.83240365e-01, 1.54917541e-01,
        3.74071384e-01, 2.03202206e-01, 3.62538485e-01, 1.94852227e-01],
       [9.73402121e-02, 0.00000000e+00, 1.67125370e-01, 2.81798408e-01,
        4.45687814e-01, 2.00613133e-01, 3.83201536e-01, 3.74100307e-01,
        1.54961435e-01, 2.03176866e-01, 1.94884054e-01, 3.62544671e-01],
       [1.05740632e-01, 1.05740502e-01, 0.00000000e+00, 1.20378221e-01,
        4.89309246e-02, 4.89323953e-02, 1.07346972e-01, 3.64328569e-01,
        3.64314579e-01, 2.93737720e-01, 4.00488970e-01, 4.00526444e-01],
       [3.24361781e-02, 3.24406183e-02, 2.19027943e-02, 0.00000000e+00,
        7.77491159e-03, 7.77398349e-03, 9.12655226e-03, 2.69002631e-02,
        2.68974351e-02, 1.65047912e-01, 2.05872119e-02, 2.05847933e-02],
       [1.11287223e-01, 2.47246335e-01, 4.29025928e-02, 3.74665733e-02,
        0.00000000e+00, 2.35769098e-01, 4.27962908e-02, 1.88167543e-02,
        2.59122464e-02, 5.43042805e-03, 2.10617992e-03, 1.33470302e-03],
       [2.47235980e-01, 1.11286552e-01, 4.29023236e-02, 3.74607399e-02,
        2.35760533e-01, 0.00000000e+00, 4.27904721e-02, 2.59125641e-02,
        1.88196492e-02, 5.43099995e-03, 1.33489055e-03, 2.10645294e-03],
       [1.91758340e-01, 1.91729801e-01, 8.48893470e-02, 3.96660118e-02,
        3.85984092e-02, 3.85945634e-02, 0.00000000e+00, 2.43001387e-02,
        2.43005888e-02, 1.10170397e-01, 7.67495229e-04, 7.67339201e-04],
       [4.16546672e-02, 1.00584370e-01, 1.54823428e-01, 6.28272972e-02,
        9.11985743e-03, 1.25594176e-02, 1.30583652e-02, 0.00000000e+00,
        2.69117091e-03, 6.48282674e-03, 7.76782782e-03, 8.88677139e-03],
       [1.00573622e-01, 4.16612788e-02, 1.54805552e-01, 6.28158509e-02,
        1.25578395e-02, 9.12088890e-03, 1.30576007e-02, 2.69096351e-03,
        0.00000000e+00, 6.48440949e-03, 8.88909766e-03, 7.76080348e-03],
       [4.62475708e-03, 4.62396063e-03, 1.05657567e-02, 3.26286660e-02,
        2.22779298e-04, 2.22810855e-04, 5.01120774e-03, 5.48733972e-04,
        5.48910245e-04, 0.00000000e+00, 7.87363548e-05, 7.87769727e-05],
       [4.38038713e-02, 2.35458323e-02, 7.64767310e-02, 2.16065132e-02,
        4.58705668e-04, 2.90736846e-04, 1.85332271e-04, 3.49055829e-03,
        3.99472100e-03, 4.17997010e-04, 0.00000000e+00, 5.57018724e-04],
       [2.35445168e-02, 4.38051627e-02, 7.64884711e-02, 2.16052698e-02,
        2.90702865e-04, 4.58809296e-04, 1.85305700e-04, 3.99360716e-03,
        3.48787965e-03, 4.18237711e-04, 5.57052111e-04, 0.00000000e+00]],
      [[0.        , 0.05077857, 0.09519957, 0.12057686, 0.08709996,
        0.1605354 , 0.23797697, 0.07515284, 0.20752518, 0.14013082,
        0.19539046, 0.12344968],
       [0.05078226, 0.        , 0.09520987, 0.12059706, 0.16055204,
        0.08710645, 0.23796145, 0.20752943, 0.07517671, 0.14012406,
        0.12344876, 0.19542695],
       [0.07480251, 0.07480516, 0.        , 0.0748864 , 0.02314759,
        0.0231493 , 0.07498846, 0.17713363, 0.17712931, 0.13418895,
        0.23803137, 0.23809833],
       [0.02515337, 0.02515575, 0.01988172, 0.        , 0.0105951 ,
        0.01059447, 0.02161329, 0.03430109, 0.03429888, 0.22669233,
        0.02873951, 0.02873983],
       [0.10408393, 0.19184476, 0.03520386, 0.06069298, 0.        ,
        0.54353574, 0.11731486, 0.06290281, 0.09509397, 0.01859452,
        0.01525189, 0.01118775],
       [0.19183623, 0.10408268, 0.03520598, 0.06068854, 0.5435283 ,
        0.        , 0.11730196, 0.09508416, 0.06291561, 0.01859656,
        0.01118874, 0.01525863],
       [0.1761524 , 0.17612808, 0.07064263, 0.07669062, 0.07266761,
        0.07266062, 0.        , 0.08688628, 0.08688602, 0.19236691,
        0.00509431, 0.00509384],
       [0.05046927, 0.13935728, 0.15139148, 0.11042245, 0.03534971,
        0.05343551, 0.07882776, 0.        , 0.01334342, 0.04518034,
        0.14232329, 0.1563477 ],
       [0.13934855, 0.05047581, 0.15137038, 0.11040263, 0.05343415,
        0.03535333, 0.07881846, 0.01334188, 0.        , 0.04518308,
        0.15636705, 0.14222053],
       [0.01496539, 0.01496358, 0.01823853, 0.11605369, 0.00166178,
        0.00166198, 0.02775435, 0.00718492, 0.00718619, 0.        ,
        0.00384416, 0.00384721],
       [0.1056579 , 0.06675037, 0.16381423, 0.0744982 , 0.00690171,
        0.00506314, 0.0037216 , 0.11460221, 0.12592507, 0.01946461,
        0.        , 0.08032957],
       [0.06674818, 0.10565796, 0.16384174, 0.07449058, 0.00506205,
        0.00690407, 0.00372084, 0.12588075, 0.11451966, 0.01947783,
        0.08032047, 0.        ]]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, \
                                                             'Anharmonic: DVPT2')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert np.allclose(ref_ih, properties.input_hyperraman_frequencies)
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert ref_harmonic_raman == properties.raman.harmonic
    assert np.allclose(ref_vv_harmonic, properties.hyper_raman_vv.harmonic)
    assert np.allclose(ref_hv_harmonic, properties.hyper_raman_hv.harmonic)
    assert np.allclose(ref_vv_fundamental, properties.hyper_raman_vv.fundamental)
    assert np.allclose(ref_hv_fundamental, properties.hyper_raman_hv.fundamental)
    assert np.allclose(ref_vv_overtones, properties.hyper_raman_vv.overtones)
    assert np.allclose(ref_hv_overtones, properties.hyper_raman_hv.overtones)
    assert np.allclose(ref_vv_combotones, properties.hyper_raman_vv.combotones)
    assert np.allclose(ref_hv_combotones, properties.hyper_raman_hv.combotones)

    # DVPT2 w/ 1-1 checks
    ref_vv_fundamental = [[3.55422815e-01, 3.55449206e-01, 2.76318522e-01, 5.24882837e-03,
                           1.84791935e-03, 1.84801318e-03, 8.71917276e-04, 1.33311999e-03,
                           1.33309786e-03, 2.15905613e-04, 5.53281452e-05, 5.53269760e-05],
                          [0.22311458, 0.22313446, 0.19794645, 0.01288592, 0.02570203,
                           0.02570299, 0.01578222, 0.04208577, 0.04208716, 0.00707344,
                           0.09224186, 0.09224312]]

    ref_hv_fundamental = [[2.96263827e-01, 2.96284467e-01, 3.90330645e-01, 6.64217908e-03,
                           1.86929128e-03, 1.86915666e-03, 1.86646023e-03, 2.33230671e-03,
                           2.33208627e-03, 1.05533136e-04, 5.20249572e-05, 5.20224120e-05],
                          [0.18320081, 0.18321709, 0.24796318, 0.01096241, 0.01993129,
                           0.01992999, 0.02230497, 0.06900719, 0.06900294, 0.00322481,
                           0.08562846, 0.08562685]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, \
                                                             'Anharmonic: DVPT2, w/ 1-1 checks')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert np.allclose(ref_ih, properties.input_hyperraman_frequencies)
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert ref_harmonic_raman == properties.raman.harmonic
    assert np.allclose(ref_vv_harmonic, properties.hyper_raman_vv.harmonic)
    assert np.allclose(ref_hv_harmonic, properties.hyper_raman_hv.harmonic)
    assert np.allclose(ref_vv_fundamental, properties.hyper_raman_vv.fundamental)
    assert np.allclose(ref_hv_fundamental, properties.hyper_raman_hv.fundamental)
    assert np.allclose(ref_vv_overtones, properties.hyper_raman_vv.overtones)
    assert np.allclose(ref_hv_overtones, properties.hyper_raman_hv.overtones)
    assert np.allclose(ref_vv_combotones, properties.hyper_raman_vv.combotones)
    assert np.allclose(ref_hv_combotones, properties.hyper_raman_hv.combotones)

    # GVPT2 frequencies, DVPT2 intensities
    ref_fundamental = [314012.43298487, 314011.53979323, 313398.59906594, 256465.52677474,
                       153457.74674526, 153430.1242997 , 150041.13718027, 113543.31474027,
                       113542.65561761,  95712.27989893,  42042.98978884,  42042.96229487]
    ref_overtones = [622382.43837506, 622381.06278219, 609848.3856535 , 510942.8909639,
                     309485.79200565, 306284.42701446, 296243.27455469, 226811.20363659,
                     226809.79089787, 190623.68916583,  83664.32840753,  83664.35715109]
    ref_combotones = \
[[     0.        , 624705.06507938, 613925.51329906, 570460.42997406,
  468597.90570357, 466687.19177635, 463345.23626701, 426758.64359099,
  426878.70636511, 409933.29257683, 356095.0069167 , 355979.95300053],
 [624705.06507938,      0.        , 613923.7292351 , 570459.54133761,
  466686.76888728, 468596.50808828, 463344.37693254, 426878.52019834,
  426756.96397891, 409932.37194193, 355979.07076914, 356094.0744635 ],
 [613925.51329906, 613923.7292351 ,      0.        , 562864.90570708,
  459602.43486588, 459602.08093223, 454413.31094192, 419477.45208183,
  419476.8233521 , 402242.10857965, 348485.84801389, 348485.83123741],
 [570460.42997406, 570459.54133761, 562864.90570708,      0.,
  411405.53335585, 411405.09295387, 406517.23180163, 369746.20969394,
  369745.52049082, 352217.88792428, 297874.57456537, 297874.54602396],
 [468597.90570357, 466686.76888728, 459602.43486588, 411405.53335585,
       0.        , 309330.06210978, 304463.72696112, 268025.2310408,
  268003.30448199, 250669.64655308, 197046.7019944 , 197044.43775159],
 [466687.19177635, 468596.50808828, 459602.08093223, 411405.09295387,
  309330.06210978,      0.        , 304463.31407033, 268003.49072155,
  268024.1787763 , 250669.20874738, 197044.07188409, 197046.2168526 ],
 [463345.23626701, 463344.37693254, 454413.31094192, 406517.23180163,
  304463.72696112, 304463.31407033,      0.        , 263064.20459946,
  263063.48794891, 245037.34667131, 192018.76030679, 192018.75687982],
 [426758.64359099, 426878.52019834, 419477.45208183, 369746.20969394,
  268025.2310408 , 268003.49072155, 263064.20459946,      0.,
  227174.58106829, 209062.63358751, 155468.0372547 , 157109.14026547],
 [426878.70636511, 426756.96397891, 419476.8233521 , 369745.52049082,
  268003.30448199, 268024.1787763 , 263063.48794891, 227174.58106829,
       0.        , 209062.00120743, 155530.79178659, 157073.56433773],
 [409933.29257683, 409932.37194193, 402242.10857965, 352217.88792428,
  250669.64655308, 250669.20874738, 245037.34667131, 209062.63358751,
  209062.00120743,      0.        , 138665.5741924 , 138665.52339525],
 [356095.0069167 , 355979.07076914, 348485.84801389, 297874.57456537,
  197046.7019944 , 197044.07188409, 192018.76030679, 155468.0372547,
  155530.79178659, 138665.5741924 ,      0.        ,  84185.97021186],
 [355979.95300053, 356094.0744635 , 348485.83123741, 297874.54602396,
  197044.43775159, 197046.2168526 , 192018.75687982, 157109.14026547,
  157073.56433773, 138665.52339525,  84185.97021186,      0.        ]]

    ref_vv_fundamental = [[6.30440639e-05, 6.30487506e-05, 5.36008227e-05, 9.30825777e-07,
                           4.99906231e-01, 4.99912458e-01, 1.54625569e-07, 2.36590405e-07,
                           2.36586383e-07, 3.82886420e-08, 1.00843072e-08, 1.00841878e-08],
                          [2.95995449e-06, 2.96021843e-06, 2.60773284e-06, 1.70918997e-07,
                           4.99798814e-01, 5.00188553e-01, 2.09335449e-07, 5.58615136e-07,
                           5.58633180e-07, 9.38221089e-08, 1.25663319e-06, 1.25665518e-06]]
    ref_vv_overtones = [[3.00076926e-01, 3.00097917e-01, 1.48882816e-01, 9.16704714e-04,
                         1.05424294e-01, 1.01175889e-01, 3.91439407e-02, 1.38088457e-03,
                         1.38053724e-03, 1.33170634e-03, 9.41885360e-05, 9.41951945e-05],
                        [0.0787932 , 0.07880028, 0.03960844, 0.00246452, 0.28565854,
                         0.28667607, 0.12985679, 0.0089638 , 0.00896123, 0.02990794,
                         0.02515401, 0.02515517]]
    ref_vv_combotones = \
     [[[0.00000000e+00, 1.60498259e-01, 1.80089946e-01, 1.67410376e-01,
        2.22799773e-01, 2.87069446e-01, 2.70024816e-01, 9.41699354e-02,
        5.83682295e-01, 1.28969787e-01, 5.79276905e-01, 1.33271802e-01],
       [1.60507453e-01, 0.00000000e+00, 1.80105996e-01, 1.67455904e-01,
        2.87085109e-01, 2.22798742e-01, 2.70003429e-01, 5.83695021e-01,
        9.41993415e-02, 1.28969674e-01, 1.33379000e-01, 5.78934876e-01],
       [1.44361841e-01, 1.44366438e-01, 0.00000000e+00, 2.88656475e-01,
        1.12331583e-01, 1.12325619e-01, 3.32123662e-01, 2.07935660e-01,
        2.07938050e-01, 4.35781261e-01, 2.18048278e-01, 2.17928487e-01],
       [2.47347202e-02, 2.47400296e-02, 5.32038410e-02, 0.00000000e+00,
        5.76371918e-03, 5.76312548e-03, 2.46937648e-02, 2.29917750e-02,
        2.29904979e-02, 2.57464522e-01, 5.05526060e-02, 5.05126632e-02],
       [1.12951390e-01, 1.45533383e-01, 7.10420962e-02, 1.97767543e-02,
        0.00000000e+00, 3.09057615e-01, 3.44902160e-02, 1.57986296e-02,
        3.63009273e-02, 3.07474529e-03, 1.13914087e-03, 7.21530907e-04],
       [1.45522818e-01, 1.12935891e-01, 7.10329740e-02, 1.97732279e-02,
        3.09034338e-01, 0.00000000e+00, 3.44853017e-02, 3.62974068e-02,
        1.57998837e-02, 3.07386796e-03, 7.22115962e-04, 1.13855032e-03],
       [1.40372529e-01, 1.40353370e-01, 2.15384932e-01, 8.68842631e-02,
        3.53669431e-02, 3.53645674e-02, 0.00000000e+00, 2.98357486e-02,
        2.98353510e-02, 3.84139937e-02, 1.25749754e-03, 1.25635639e-03],
       [2.46262571e-02, 1.52632581e-01, 6.78347404e-02, 4.06943399e-02,
        8.14945786e-03, 1.87248183e-02, 1.50087537e-02, 0.00000000e+00,
        1.58329108e-03, 1.85655062e-03, 7.00075926e-03, 8.54440862e-03],
       [1.52617036e-01, 2.46291534e-02, 6.78262048e-02, 4.06864916e-02,
        1.87226526e-02, 8.14959940e-03, 1.50064927e-02, 1.58307366e-03,
        0.00000000e+00, 1.85673816e-03, 8.21425346e-03, 7.28214930e-03],
       [6.22323741e-03, 6.22287551e-03, 2.62321565e-02, 8.40855080e-02,
        2.92658200e-04, 2.92596732e-04, 3.56565569e-03, 3.42569778e-04,
        3.42651436e-04, 0.00000000e+00, 1.05173849e-04, 1.05122077e-04],
       [7.15985648e-02, 1.64846861e-02, 3.36207826e-02, 4.22899355e-02,
        2.77727382e-04, 1.76068200e-04, 2.98983211e-04, 3.30885023e-03,
        3.88293131e-03, 2.69399932e-04, 0.00000000e+00, 3.04053501e-04],
       [1.64841548e-02, 7.16033346e-02, 3.36263301e-02, 4.22867249e-02,
        1.76038037e-04, 2.77802736e-04, 2.98925402e-04, 4.04132973e-03,
        3.44478005e-03, 2.69459786e-04, 3.04270830e-04, 0.00000000e+00]],
      [[0.        , 0.08339631, 0.12370955, 0.06623769, 0.09570173,
        0.0852955 , 0.16433914, 0.04150365, 0.27082394, 0.10970139,
        0.30708605, 0.08047057],
       [0.08340341, 0.        , 0.12372784, 0.06625509, 0.08530704,
        0.09569922, 0.16433041, 0.27082981, 0.04151738, 0.10970468,
        0.08042707, 0.30732433],
       [0.10710552, 0.10711224, 0.        , 0.11336984, 0.05596258,
        0.05596071, 0.22505846, 0.09197191, 0.0919716 , 0.24972092,
        0.12490893, 0.12501183],
       [0.01975333, 0.01975683, 0.03905026, 0.        , 0.00606891,
        0.00606878, 0.04325812, 0.02387198, 0.02387032, 0.33216863,
        0.0826915 , 0.08274694],
       [0.12124853, 0.10806986, 0.08189281, 0.02578292, 0.        ,
        0.59564965, 0.09163739, 0.05440443, 0.13067785, 0.00646594,
        0.00811611, 0.00585794],
       [0.10806045, 0.12123054, 0.08188705, 0.02578141, 0.59562764,
        0.        , 0.09162541, 0.13065165, 0.05440793, 0.00646558,
        0.00585566, 0.00812412],
       [0.13207382, 0.13205556, 0.20891155, 0.11657578, 0.05812886,
        0.0581234 , 0.        , 0.1138069 , 0.1138041 , 0.08792937,
        0.01070358, 0.01070942],
       [0.02743345, 0.17900023, 0.07021674, 0.05291118, 0.02838387,
        0.06816608, 0.09360239, 0.        , 0.00680908, 0.0207075 ,
        0.15847365, 0.17025653],
       [0.17898543, 0.02743618, 0.07020624, 0.05289978, 0.06816728,
        0.02838261, 0.09358642, 0.00680808, 0.        , 0.02070431,
        0.17045315, 0.15817861],
       [0.01863782, 0.0186368 , 0.04900375, 0.18923725, 0.00086708,
        0.00086706, 0.01858837, 0.00532251, 0.00532247, 0.        ,
        0.00913637, 0.00914424],
       [0.16110813, 0.04219128, 0.07569065, 0.14547327, 0.00336085,
        0.00242489, 0.00698732, 0.12578254, 0.13531058, 0.02821291,
        0.        , 0.04217547],
       [0.04219013, 0.16111418, 0.07570355, 0.14547579, 0.00242417,
        0.00336209, 0.00698657, 0.13504655, 0.12548474, 0.02821879,
        0.04214794, 0.        ]]]

    ref_hv_fundamental = [[4.59354196e-05, 4.59386248e-05, 6.61853769e-05, 1.02963797e-06,
                           4.99880704e-01, 4.99959161e-01, 2.89329494e-07, 3.61812386e-07,
                           3.61778493e-07, 1.63592282e-08, 8.26914756e-09, 8.26885094e-09],
                          [2.72110530e-06, 2.72134749e-06, 3.65730450e-06, 1.62794299e-07,
                           4.99771035e-01, 5.00214668e-01, 3.31234019e-07, 1.02549903e-06,
                           1.02543664e-06, 4.78892140e-08, 1.30237138e-06, 1.30235795e-06]]
    ref_hv_overtones = [[2.14430695e-01, 2.14450995e-01, 2.45394749e-01, 1.97226292e-03,
                         1.46781478e-01, 1.40837483e-01, 2.80229622e-02, 3.57227347e-03,
                         3.57123340e-03, 7.20672059e-04, 1.22588975e-04, 1.22605791e-04],
                        [0.04156176, 0.04156608, 0.04954431, 0.00199778, 0.33485258,
                         0.33598837, 0.08181775, 0.02127328, 0.02126661, 0.01167095,
                         0.02922877, 0.02923173]]
    ref_hv_combotones = \
     [[[0.00000000e+00, 9.73355868e-02, 1.67117634e-01, 2.81746449e-01,
        2.00597523e-01, 4.45664163e-01, 3.83240365e-01, 1.54892083e-01,
        3.74016686e-01, 2.03202206e-01, 3.62538475e-01, 1.94717654e-01],
       [9.73402121e-02, 0.00000000e+00, 1.67125370e-01, 2.81798408e-01,
        4.45687814e-01, 2.00613133e-01, 3.83201536e-01, 3.74038830e-01,
        1.54938776e-01, 2.03176866e-01, 1.94884048e-01, 3.62294282e-01],
       [1.05740632e-01, 1.05740502e-01, 0.00000000e+00, 1.20378221e-01,
        4.89309246e-02, 4.89323953e-02, 1.07346972e-01, 3.64268698e-01,
        3.64261308e-01, 2.93737720e-01, 4.00488960e-01, 4.00249823e-01],
       [3.24361781e-02, 3.24406183e-02, 2.19027943e-02, 0.00000000e+00,
        7.77491159e-03, 7.77398349e-03, 9.12655226e-03, 2.68958425e-02,
        2.68935020e-02, 1.65047912e-01, 2.05872113e-02, 2.05705766e-02],
       [1.11287223e-01, 2.47246335e-01, 4.29025928e-02, 3.74665733e-02,
        0.00000000e+00, 2.35769098e-01, 4.27962908e-02, 1.88136621e-02,
        2.59084575e-02, 5.43042805e-03, 2.10617986e-03, 1.33378122e-03],
       [2.47235980e-01, 1.11286552e-01, 4.29023236e-02, 3.74607399e-02,
        2.35760533e-01, 0.00000000e+00, 4.27904721e-02, 2.59083059e-02,
        1.88168973e-02, 5.43099995e-03, 1.33489051e-03, 2.10499814e-03],
       [1.91758340e-01, 1.91729801e-01, 8.48893470e-02, 3.96660118e-02,
        3.85984092e-02, 3.85945634e-02, 0.00000000e+00, 2.42961454e-02,
        2.42970355e-02, 1.10170397e-01, 7.67495208e-04, 7.66809245e-04],
       [4.16546672e-02, 1.00584370e-01, 1.54823428e-01, 6.28272972e-02,
        9.11985743e-03, 1.25594176e-02, 1.30583652e-02, 0.00000000e+00,
        2.69077740e-03, 6.48282674e-03, 7.76776555e-03, 9.24618180e-03],
       [1.00573622e-01, 4.16612788e-02, 1.54805552e-01, 6.28158509e-02,
        1.25578395e-02, 9.12088890e-03, 1.30576007e-02, 2.69052130e-03,
        0.00000000e+00, 6.48440949e-03, 8.88918622e-03, 8.08053707e-03],
       [4.62475708e-03, 4.62396063e-03, 1.05657567e-02, 3.26286660e-02,
        2.22779298e-04, 2.22810855e-04, 5.01120774e-03, 5.48643798e-04,
        5.48829981e-04, 0.00000000e+00, 7.87363527e-05, 7.87225661e-05],
       [4.38038713e-02, 2.35458323e-02, 7.64767310e-02, 2.16065132e-02,
        4.58705668e-04, 2.90736846e-04, 1.85332271e-04, 3.48995680e-03,
        3.99417678e-03, 4.17997010e-04, 0.00000000e+00, 5.56634023e-04],
       [2.35445168e-02, 4.38051627e-02, 7.64884711e-02, 2.16052698e-02,
        2.90702865e-04, 4.58809296e-04, 1.85305700e-04, 4.15731023e-03,
        3.63355358e-03, 4.18237711e-04, 5.57052096e-04, 0.00000000e+00]],
      [[0.        , 0.05077857, 0.09519957, 0.12057686, 0.08709996,
        0.1605354 , 0.23797697, 0.07516732, 0.2075622 , 0.14013082,
        0.19539046, 0.12350658],
       [0.05078226, 0.        , 0.09520987, 0.12059706, 0.16055204,
        0.08710645, 0.23796145, 0.2075694 , 0.07519012, 0.14012406,
        0.12344876, 0.19551702],
       [0.07480251, 0.07480516, 0.        , 0.0748864 , 0.02314759,
        0.0231493 , 0.07498846, 0.17716775, 0.17716091, 0.13418895,
        0.23803138, 0.23820807],
       [0.02515337, 0.02515575, 0.01988172, 0.        , 0.0105951 ,
        0.01059447, 0.02161329, 0.0343077 , 0.034305  , 0.22669233,
        0.02873951, 0.02875307],
       [0.10408393, 0.19184476, 0.03520386, 0.06069298, 0.        ,
        0.54353574, 0.11731486, 0.06291492, 0.09511093, 0.01859452,
        0.01525189, 0.0111929 ],
       [0.19183623, 0.10408268, 0.03520598, 0.06068854, 0.5435283 ,
        0.        , 0.11730196, 0.09510248, 0.06292684, 0.01859656,
        0.01118874, 0.01526566],
       [0.1761524 , 0.17612808, 0.07064263, 0.07669062, 0.07266761,
        0.07266062, 0.        , 0.08690301, 0.08690152, 0.19236691,
        0.00509431, 0.00509618],
       [0.05046927, 0.13935728, 0.15139148, 0.11042245, 0.03534971,
        0.05343551, 0.07882776, 0.        , 0.0133458 , 0.04518034,
        0.14232334, 0.15618043],
       [0.13934855, 0.05047581, 0.15137038, 0.11040263, 0.05343415,
        0.03535333, 0.07881846, 0.01334445, 0.        , 0.04518308,
        0.15636699, 0.14206451],
       [0.01496539, 0.01496358, 0.01823853, 0.11605369, 0.00166178,
        0.00166198, 0.02775435, 0.00718631, 0.00718747, 0.        ,
        0.00384416, 0.00384898],
       [0.1056579 , 0.06675037, 0.16381423, 0.0744982 , 0.00690171,
        0.00506314, 0.0037216 , 0.11462431, 0.12594749, 0.01946461,
        0.        , 0.08036659],
       [0.06674818, 0.10565796, 0.16384174, 0.07449058, 0.00506205,
        0.00690407, 0.00372084, 0.12571235, 0.11436173, 0.01947783,
        0.08032047, 0.        ]]]
    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, \
                                                             'Anharmonic: Freq GVPT2, Int DVPT2')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert np.allclose(ref_ih, properties.input_hyperraman_frequencies)
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert ref_harmonic_raman == properties.raman.harmonic
    assert np.allclose(ref_vv_harmonic, properties.hyper_raman_vv.harmonic)
    assert np.allclose(ref_hv_harmonic, properties.hyper_raman_hv.harmonic)
    assert np.allclose(ref_vv_fundamental, properties.hyper_raman_vv.fundamental)
    assert np.allclose(ref_hv_fundamental, properties.hyper_raman_hv.fundamental)
    assert np.allclose(ref_vv_overtones, properties.hyper_raman_vv.overtones)
    assert np.allclose(ref_hv_overtones, properties.hyper_raman_hv.overtones)
    assert np.allclose(ref_vv_combotones, properties.hyper_raman_vv.combotones)
    assert np.allclose(ref_hv_combotones, properties.hyper_raman_hv.combotones)

    # GVPT2 frequencies, DVPT2 intensities w/ 1-1 checks
    ref_vv_fundamental = [[3.46489774e-01, 3.46515502e-01, 2.94652782e-01, 5.11690663e-03,
                           1.72930432e-03, 1.72816847e-03, 8.50002890e-04, 1.29961394e-03,
                           1.29959236e-03, 2.10479137e-04, 5.39375519e-05, 5.39364121e-05],
                          [0.22339709, 0.22341699, 0.19685113, 0.01290224, 0.02577411,
                           0.02577576, 0.0158022 , 0.04213906, 0.04214045, 0.00708239,
                           0.09235866, 0.09235992]]

    ref_hv_fundamental = [[2.85836735e-01, 2.85856648e-01, 4.11933860e-01, 6.40840563e-03,
                           1.73124961e-03, 1.72990004e-03, 1.80076961e-03, 2.25022049e-03,
                           2.25000781e-03, 1.01818866e-04, 5.01939235e-05, 5.01914679e-05],
                          [0.18349847, 0.18351478, 0.24667941, 0.01098022, 0.01999434,
                           0.01999357, 0.02234121, 0.06911931, 0.06911505, 0.00323005,
                           0.08576759, 0.08576598]]

    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, \
                                                    'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert np.allclose(ref_ih, properties.input_hyperraman_frequencies)
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert ref_harmonic_raman == properties.raman.harmonic
    assert np.allclose(ref_vv_harmonic, properties.hyper_raman_vv.harmonic)
    assert np.allclose(ref_hv_harmonic, properties.hyper_raman_hv.harmonic)
    assert np.allclose(ref_vv_fundamental, properties.hyper_raman_vv.fundamental)
    assert np.allclose(ref_hv_fundamental, properties.hyper_raman_hv.fundamental)
    assert np.allclose(ref_vv_overtones, properties.hyper_raman_vv.overtones)
    assert np.allclose(ref_hv_overtones, properties.hyper_raman_hv.overtones)
    assert np.allclose(ref_vv_combotones, properties.hyper_raman_vv.combotones)
    assert np.allclose(ref_hv_combotones, properties.hyper_raman_hv.combotones)

    # GVPT2 frequencies, DVPT2 intensities w/ 1-1 checks and forced i,j removal
    properties = get_vibrational_frequencies_and_intensities(tensor_file, mol_file, spectroscopy_type, \
                                                             specifications, 0, 298, True, \
                                'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks and forced removal')

    assert np.allclose(ref_wavenumbers, properties.frequencies.harmonic)
    assert np.allclose(ref_fundamental, properties.frequencies.fundamental)
    assert np.allclose(ref_overtones, properties.frequencies.overtones)
    assert np.allclose(ref_combotones, properties.frequencies.combotones)
    assert ref_input_wavenumers == properties.input_raman_frequencies
    assert np.allclose(ref_ih, properties.input_hyperraman_frequencies)
    assert ref_harmonic_ir == properties.ir.harmonic
    assert ref_fundamental_ir == properties.ir.fundamental
    assert ref_overtone_ir == properties.ir.overtones
    assert ref_combotone_ir == properties.ir.combotones
    assert ref_harmonic_raman == properties.raman.harmonic
    assert np.allclose(ref_vv_harmonic, properties.hyper_raman_vv.harmonic)
    assert np.allclose(ref_hv_harmonic, properties.hyper_raman_hv.harmonic)
    assert np.allclose(ref_vv_fundamental, properties.hyper_raman_vv.fundamental)
    assert np.allclose(ref_hv_fundamental, properties.hyper_raman_hv.fundamental)
    assert np.allclose(ref_vv_overtones, properties.hyper_raman_vv.overtones)
    assert np.allclose(ref_hv_overtones, properties.hyper_raman_hv.overtones)
    assert np.allclose(ref_vv_combotones, properties.hyper_raman_vv.combotones)
    assert np.allclose(ref_hv_combotones, properties.hyper_raman_hv.combotones)


def test_which_spectroscopies_to_be_calculated():

    spectroscopy_types = ['IR', 'Raman', 'Hyper-Raman']

    IR, Raman, hyper_raman = which_spectroscopies_to_be_calculated(spectroscopy_types)

    assert IR == True
    assert Raman == True
    assert hyper_raman == True

    spectroscopy_types = ['IR', 'Raman']

    IR, Raman, hyper_raman = which_spectroscopies_to_be_calculated(spectroscopy_types)

    assert IR == True
    assert Raman == True
    assert hyper_raman == False

    spectroscopy_types = ['Raman', 'Hyper-Raman']

    IR, Raman, hyper_raman = which_spectroscopies_to_be_calculated(spectroscopy_types)

    assert IR == False
    assert Raman == True
    assert hyper_raman == True

    spectroscopy_types = ['IR', 'Hyper-Raman']

    IR, Raman, hyper_raman = which_spectroscopies_to_be_calculated(spectroscopy_types)

    assert IR == True
    assert Raman == False
    assert hyper_raman == True


def test_get_spectroscopy_indices():

    data_dir = '{0}/'.format(os.path.dirname(__file__))
    tensor_file = data_dir + 'hf_H2O2.rsp_tensor'

    redundant_properties, tensors = read_openrsp_tensor_file(tensor_file)

    ref_spectroscopy_indices = [9]

    spectroscopy_indices = get_spectroscopy_indices(redundant_properties, hess_operator)

    assert np.allclose(ref_spectroscopy_indices, spectroscopy_indices)

    ref_spectroscopy_indices = [3]

    spectroscopy_indices = get_spectroscopy_indices(redundant_properties, dip_grad_operator)

    assert np.allclose(ref_spectroscopy_indices, spectroscopy_indices)

    ref_spectroscopy_indices = [4, 5, 6]

    spectroscopy_indices = get_spectroscopy_indices(redundant_properties, polariz_grad_operator)

    assert np.allclose(ref_spectroscopy_indices, spectroscopy_indices)

    data_dir = '{0}/'.format(os.path.dirname(__file__))
    tensor_file = data_dir + 'multifreq_H2O2.rsp_tensor'

    redundant_properties, tensors = read_openrsp_tensor_file(tensor_file)

    ref_spectroscopy_indices = [0, 1, 2]

    spectroscopy_indices = get_spectroscopy_indices(redundant_properties, hyper_polariz_grad_operator)

    assert np.allclose(ref_spectroscopy_indices, spectroscopy_indices)


def test_reduced_dims():

    mat_dims = (12, 3)
    remove_elms = 6
    rank_el = 1
    ref_end_dims = (6, 3)

    end_dims = reduced_dims(mat_dims, remove_elms, rank_el)

    assert ref_end_dims == end_dims

    mat_dims = (12, 3, 3)
    rank_el = 2
    ref_end_dims = (6, 3, 3)

    end_dims = reduced_dims(mat_dims, remove_elms, rank_el)

    assert ref_end_dims == end_dims

    mat_dims = (12, 3)
    remove_elms = 5
    rank_el = 1
    ref_end_dims = (7, 3)

    end_dims = reduced_dims(mat_dims, remove_elms, rank_el)

    assert ref_end_dims == end_dims

    mat_dims = (12, 12, 3)
    remove_elms = 6
    ref_end_dims = (6, 6, 3)

    end_dims = reduced_dims(mat_dims, remove_elms, rank_el)

    assert ref_end_dims == end_dims

    mat_dims = (18, 18, 3)
    remove_elms = 6
    ref_end_dims = (12, 12, 3)

    end_dims = reduced_dims(mat_dims, remove_elms, rank_el)

    assert ref_end_dims == end_dims

    remove_elms = 0
    ref_end_dims = (18, 18, 3)

    end_dims = reduced_dims(mat_dims, remove_elms, rank_el)

    assert ref_end_dims == end_dims


def test_getting_the_subblocks():

    input_mat = \
        [[-3.55354959e-03, -2.04896144e-03,  4.65337226e-04], \
         [-1.33397761e-03,  2.30954119e-03, -1.03887733e-06], \
         [ 5.38119487e-04, -9.34229819e-04,  5.72701937e-07], \
         [ 2.05564456e-04,  1.21485597e-04,  5.08962420e-03], \
         [ 4.87101890e-05, -8.45631462e-05,  7.73668483e-08], \
         [ 4.34177613e-03, -7.53776352e-03,  4.54332299e-06], \
         [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00], \
         [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00], \
         [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00], \
         [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00], \
         [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00], \
         [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]]

    remove_elms = 6
    out_mat = np.zeros(18)
    offset = 0
    rank_el = 1

    ref_out_mat = np.reshape(ref_dipole_gradient, (18))

    getting_the_subblocks(input_mat, remove_elms, out_mat, offset, rank_el)

    assert np.allclose(ref_out_mat, out_mat)

    input_mat = \
        [[[ 6.04610202e-02, -3.51511055e-02,  2.05867010e-02], \
          [-3.51511055e-02, -6.06465271e-02, -3.58100073e-02], \
          [ 2.05867010e-02, -3.58100073e-02, -8.24129166e-06]], \
         [[ 1.20853575e-01,  4.99878490e-02,  7.95877729e-02], \
          [ 4.99878490e-02,  6.28477170e-02,  4.58251237e-02], \
          [ 7.95877729e-02,  4.58251237e-02,  6.62798317e-02]], \
         [[-2.34431047e-02, -7.82917443e-03,  4.02920687e-02], \
          [-7.82917443e-03, -1.43042013e-02,  2.32435281e-02], \
          [ 4.02920687e-02,  2.32435281e-02,  4.87524759e-02]], \
         [[-1.55313827e-02,  9.01591539e-03,  1.14149851e-02], \
          [ 9.01591539e-03,  1.55074857e-02, -1.98366345e-02], \
          [ 1.14149851e-02, -1.98366345e-02,  2.34202541e-05]], \
         [[ 4.39674438e-03,  4.92827640e-04,  7.19780552e-03], \
          [ 4.92827640e-04,  3.83525060e-03,  4.19756005e-03], \
          [ 7.19780552e-03,  4.19756005e-03,  8.94657717e-02]], \
         [[-2.41613385e-02, -4.31408089e-02, -1.19041040e-02], \
          [-4.31408089e-02,  2.58697434e-02, -6.88886076e-03], \
          [-1.19041040e-02, -6.88886076e-03, -2.42915073e-03]], \
         [[ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00], \
          [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00], \
          [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]], \
         [[ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00], \
          [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00], \
          [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]], \
         [[ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00], \
          [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00], \
          [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]], \
         [[ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00], \
          [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00], \
          [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]], \
         [[ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00], \
          [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00], \
          [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]], \
         [[ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00], \
          [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00], \
          [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]]]

    out_mat = np.zeros(54)
    rank_el = 2

    ref_out_mat = np.reshape(ref_polarizability_gradient[0], (54))

    getting_the_subblocks(input_mat, remove_elms, out_mat, offset, rank_el)

    assert np.allclose(ref_out_mat, out_mat)


def test_get_energy_derivatives():

    data_dir = '{0}/'.format(os.path.dirname(__file__))
    tensor_file = data_dir + 'hf_anharm_H2O2.rsp_tensor'

    redundant_properties, tensors = read_openrsp_tensor_file(tensor_file)

    cubic_index = [7]

    cubic_forcefield = get_energy_derivatives(cubic_index, tensors, redundant_properties, 3, \
                                              0, num_coordinates, vib_degrees_of_freedom, \
                                              transformation_matrix)

    assert np.allclose(ref_cubic_ff, cubic_forcefield)

    dip_grad_index = [0]

    dipole_gradient = get_energy_derivatives(dip_grad_index, tensors, redundant_properties, 1, \
                                             1, num_coordinates, vib_degrees_of_freedom, \
                                             transformation_matrix)

    assert np.allclose(ref_dipole_gradient, dipole_gradient)

    dip_hess_index = [1]
    dipole_hessian = get_energy_derivatives(dip_hess_index, tensors, redundant_properties, 2, 1, \
                                            num_coordinates, vib_degrees_of_freedom, \
                                            transformation_matrix)

    assert np.allclose(ref_dipole_hessian, dipole_hessian)

    polariz_grad_index = [3, 4, 5]

    polarizability_gradient = get_energy_derivatives(polariz_grad_index, tensors, redundant_properties, \
                                                     1, 2, num_coordinates, vib_degrees_of_freedom, \
                                                     transformation_matrix)

    assert np.allclose(ref_polarizability_gradient, polarizability_gradient)

    data_dir = '{0}/'.format(os.path.dirname(__file__))
    tensor_file = data_dir + 'multifreq_H2O2.rsp_tensor'

    redundant_properties, tensors = read_openrsp_tensor_file(tensor_file)

    hyper_polariz_grad_index = [0, 1, 2]

    hyper_polarizability_gradients = get_energy_derivatives(hyper_polariz_grad_index, tensors, \
                                                            redundant_properties, 1, 3, \
                                                            num_coordinates, \
                                                            vib_degrees_of_freedom, \
                                                            transformation_matrix)

    assert np.allclose(ref_hyper_polarizability_gradient, hyper_polarizability_gradients)
