import * as cloudwatch from '@aws-cdk/aws-cloudwatch';
import * as iam from '@aws-cdk/aws-iam';
import * as kms from '@aws-cdk/aws-kms';
import { Duration, IResource, RemovalPolicy, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { EnableScalingProps, IScalableTableAttribute } from './scalable-attribute-api';
/**
 * Options for configuring a system errors metric that considers multiple operations.
 *
 * @stability stable
 */
export interface SystemErrorsForOperationsMetricOptions extends cloudwatch.MetricOptions {
    /**
     * The operations to apply the metric to.
     *
     * @default - All operations available by DynamoDB tables will be considered.
     * @stability stable
     */
    readonly operations?: Operation[];
}
/**
 * Supported DynamoDB table operations.
 *
 * @stability stable
 */
export declare enum Operation {
    /**
     * GetItem.
     *
     * @stability stable
     */
    GET_ITEM = "GetItem",
    /**
     * BatchGetItem.
     *
     * @stability stable
     */
    BATCH_GET_ITEM = "BatchGetItem",
    /**
     * Scan.
     *
     * @stability stable
     */
    SCAN = "Scan",
    /**
     * Query.
     *
     * @stability stable
     */
    QUERY = "Query",
    /**
     * GetRecords.
     *
     * @stability stable
     */
    GET_RECORDS = "GetRecords",
    /**
     * PutItem.
     *
     * @stability stable
     */
    PUT_ITEM = "PutItem",
    /**
     * DeleteItem.
     *
     * @stability stable
     */
    DELETE_ITEM = "DeleteItem",
    /**
     * UpdateItem.
     *
     * @stability stable
     */
    UPDATE_ITEM = "UpdateItem",
    /**
     * BatchWriteItem.
     *
     * @stability stable
     */
    BATCH_WRITE_ITEM = "BatchWriteItem"
}
/**
 * Represents an attribute for describing the key schema for the table and indexes.
 *
 * @stability stable
 */
export interface Attribute {
    /**
     * The name of an attribute.
     *
     * @stability stable
     */
    readonly name: string;
    /**
     * The data type of an attribute.
     *
     * @stability stable
     */
    readonly type: AttributeType;
}
/**
 * What kind of server-side encryption to apply to this table.
 *
 * @stability stable
 */
export declare enum TableEncryption {
    /**
     * Server-side KMS encryption with a master key owned by AWS.
     *
     * @stability stable
     */
    DEFAULT = "AWS_OWNED",
    /**
     * Server-side KMS encryption with a customer master key managed by customer.
     *
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     *
     * @stability stable
     */
    CUSTOMER_MANAGED = "CUSTOMER_MANAGED",
    /**
     * Server-side KMS encryption with a master key managed by AWS.
     *
     * @stability stable
     */
    AWS_MANAGED = "AWS_MANAGED"
}
/**
 * Properties of a DynamoDB Table.
 *
 * Use {@link TableProps} for all table properties
 *
 * @stability stable
 */
export interface TableOptions {
    /**
     * Partition key attribute definition.
     *
     * @stability stable
     */
    readonly partitionKey: Attribute;
    /**
     * Table sort key attribute definition.
     *
     * @default no sort key
     * @stability stable
     */
    readonly sortKey?: Attribute;
    /**
     * The read capacity for the table.
     *
     * Careful if you add Global Secondary Indexes, as
     * those will share the table's provisioned throughput.
     *
     * Can only be provided if billingMode is Provisioned.
     *
     * @default 5
     * @stability stable
     */
    readonly readCapacity?: number;
    /**
     * The write capacity for the table.
     *
     * Careful if you add Global Secondary Indexes, as
     * those will share the table's provisioned throughput.
     *
     * Can only be provided if billingMode is Provisioned.
     *
     * @default 5
     * @stability stable
     */
    readonly writeCapacity?: number;
    /**
     * Specify how you are charged for read and write throughput and how you manage capacity.
     *
     * @default PROVISIONED if `replicationRegions` is not specified, PAY_PER_REQUEST otherwise
     * @stability stable
     */
    readonly billingMode?: BillingMode;
    /**
     * Whether point-in-time recovery is enabled.
     *
     * @default - point-in-time recovery is disabled
     * @stability stable
     */
    readonly pointInTimeRecovery?: boolean;
    /**
     * (deprecated) Whether server-side encryption with an AWS managed customer master key is enabled.
     *
     * This property cannot be set if `encryption` and/or `encryptionKey` is set.
     *
     * @default - server-side encryption is enabled with an AWS owned customer master key
     * @deprecated This property is deprecated. In order to obtain the same behavior as
     * enabling this, set the `encryption` property to `TableEncryption.AWS_MANAGED` instead.
     */
    readonly serverSideEncryption?: boolean;
    /**
     * Whether server-side encryption with an AWS managed customer master key is enabled.
     *
     * This property cannot be set if `serverSideEncryption` is set.
     *
     * @default - server-side encryption is enabled with an AWS owned customer master key
     * @stability stable
     */
    readonly encryption?: TableEncryption;
    /**
     * External KMS key to use for table encryption.
     *
     * This property can only be set if `encryption` is set to `TableEncryption.CUSTOMER_MANAGED`.
     *
     * @default - If `encryption` is set to `TableEncryption.CUSTOMER_MANAGED` and this
     * property is undefined, a new KMS key will be created and associated with this table.
     * @stability stable
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * The name of TTL attribute.
     *
     * @default - TTL is disabled
     * @stability stable
     */
    readonly timeToLiveAttribute?: string;
    /**
     * When an item in the table is modified, StreamViewType determines what information is written to the stream for this table.
     *
     * @default - streams are disabled unless `replicationRegions` is specified
     * @stability stable
     */
    readonly stream?: StreamViewType;
    /**
     * The removal policy to apply to the DynamoDB Table.
     *
     * @default RemovalPolicy.RETAIN
     * @stability stable
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * (experimental) Regions where replica tables will be created.
     *
     * @default - no replica tables are created
     * @experimental
     */
    readonly replicationRegions?: string[];
    /**
     * The timeout for a table replication operation in a single region.
     *
     * @default Duration.minutes(30)
     * @stability stable
     */
    readonly replicationTimeout?: Duration;
}
/**
 * Properties for a DynamoDB Table.
 *
 * @stability stable
 */
export interface TableProps extends TableOptions {
    /**
     * Enforces a particular physical table name.
     *
     * @default <generated>
     * @stability stable
     */
    readonly tableName?: string;
}
/**
 * Properties for a secondary index.
 *
 * @stability stable
 */
export interface SecondaryIndexProps {
    /**
     * The name of the secondary index.
     *
     * @stability stable
     */
    readonly indexName: string;
    /**
     * The set of attributes that are projected into the secondary index.
     *
     * @default ALL
     * @stability stable
     */
    readonly projectionType?: ProjectionType;
    /**
     * The non-key attributes that are projected into the secondary index.
     *
     * @default - No additional attributes
     * @stability stable
     */
    readonly nonKeyAttributes?: string[];
}
/**
 * Properties for a global secondary index.
 *
 * @stability stable
 */
export interface GlobalSecondaryIndexProps extends SecondaryIndexProps {
    /**
     * The attribute of a partition key for the global secondary index.
     *
     * @stability stable
     */
    readonly partitionKey: Attribute;
    /**
     * The attribute of a sort key for the global secondary index.
     *
     * @default - No sort key
     * @stability stable
     */
    readonly sortKey?: Attribute;
    /**
     * The read capacity for the global secondary index.
     *
     * Can only be provided if table billingMode is Provisioned or undefined.
     *
     * @default 5
     * @stability stable
     */
    readonly readCapacity?: number;
    /**
     * The write capacity for the global secondary index.
     *
     * Can only be provided if table billingMode is Provisioned or undefined.
     *
     * @default 5
     * @stability stable
     */
    readonly writeCapacity?: number;
}
/**
 * Properties for a local secondary index.
 *
 * @stability stable
 */
export interface LocalSecondaryIndexProps extends SecondaryIndexProps {
    /**
     * The attribute of a sort key for the local secondary index.
     *
     * @stability stable
     */
    readonly sortKey: Attribute;
}
/**
 * An interface that represents a DynamoDB Table - either created with the CDK, or an existing one.
 *
 * @stability stable
 */
export interface ITable extends IResource {
    /**
     * Arn of the dynamodb table.
     *
     * @stability stable
     * @attribute true
     */
    readonly tableArn: string;
    /**
     * Table name of the dynamodb table.
     *
     * @stability stable
     * @attribute true
     */
    readonly tableName: string;
    /**
     * ARN of the table's stream, if there is one.
     *
     * @stability stable
     * @attribute true
     */
    readonly tableStreamArn?: string;
    /**
     * Optional KMS encryption key associated with this table.
     *
     * @stability stable
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * Adds an IAM policy statement associated with this table to an IAM principal's policy.
     *
     * If `encryptionKey` is present, appropriate grants to the key needs to be added
     * separately using the `table.encryptionKey.grant*` methods.
     *
     * @param grantee The principal (no-op if undefined).
     * @param actions The set of actions to allow (i.e. "dynamodb:PutItem", "dynamodb:GetItem", ...).
     * @stability stable
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Adds an IAM policy statement associated with this table's stream to an IAM principal's policy.
     *
     * If `encryptionKey` is present, appropriate grants to the key needs to be added
     * separately using the `table.encryptionKey.grant*` methods.
     *
     * @param grantee The principal (no-op if undefined).
     * @param actions The set of actions to allow (i.e. "dynamodb:DescribeStream", "dynamodb:GetRecords", ...).
     * @stability stable
     */
    grantStream(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Permits an IAM principal all data read operations from this table: BatchGetItem, GetRecords, GetShardIterator, Query, GetItem, Scan.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantReadData(grantee: iam.IGrantable): iam.Grant;
    /**
     * Permits an IAM Principal to list streams attached to current dynamodb table.
     *
     * @param grantee The principal (no-op if undefined).
     * @stability stable
     */
    grantTableListStreams(grantee: iam.IGrantable): iam.Grant;
    /**
     * Permits an IAM principal all stream data read operations for this table's stream: DescribeStream, GetRecords, GetShardIterator, ListStreams.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantStreamRead(grantee: iam.IGrantable): iam.Grant;
    /**
     * Permits an IAM principal all data write operations to this table: BatchWriteItem, PutItem, UpdateItem, DeleteItem.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantWriteData(grantee: iam.IGrantable): iam.Grant;
    /**
     * Permits an IAM principal to all data read/write operations to this table.
     *
     * BatchGetItem, GetRecords, GetShardIterator, Query, GetItem, Scan,
     * BatchWriteItem, PutItem, UpdateItem, DeleteItem
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantReadWriteData(grantee: iam.IGrantable): iam.Grant;
    /**
     * Permits all DynamoDB operations ("dynamodb:*") to an IAM principal.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantFullAccess(grantee: iam.IGrantable): iam.Grant;
    /**
     * Metric for the number of Errors executing all Lambdas.
     *
     * @stability stable
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the consumed read capacity units.
     *
     * @param props properties of a metric.
     * @stability stable
     */
    metricConsumedReadCapacityUnits(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the consumed write capacity units.
     *
     * @param props properties of a metric.
     * @stability stable
     */
    metricConsumedWriteCapacityUnits(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * (deprecated) Metric for the system errors.
     *
     * @param props properties of a metric.
     * @deprecated use `metricSystemErrorsForOperations`
     */
    metricSystemErrors(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the system errors this table.
     *
     * @param props properties of a metric.
     * @stability stable
     */
    metricSystemErrorsForOperations(props?: SystemErrorsForOperationsMetricOptions): cloudwatch.IMetric;
    /**
     * Metric for the user errors.
     *
     * @param props properties of a metric.
     * @stability stable
     */
    metricUserErrors(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the conditional check failed requests.
     *
     * @param props properties of a metric.
     * @stability stable
     */
    metricConditionalCheckFailedRequests(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for throttled requests.
     *
     * @param props properties of a metric.
     * @stability stable
     */
    metricThrottledRequests(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the successful request latency.
     *
     * @param props properties of a metric.
     * @stability stable
     */
    metricSuccessfulRequestLatency(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
}
/**
 * Reference to a dynamodb table.
 *
 * @stability stable
 */
export interface TableAttributes {
    /**
     * The ARN of the dynamodb table.
     *
     * One of this, or {@link tableName}, is required.
     *
     * @default - no table arn
     * @stability stable
     */
    readonly tableArn?: string;
    /**
     * The table name of the dynamodb table.
     *
     * One of this, or {@link tableArn}, is required.
     *
     * @default - no table name
     * @stability stable
     */
    readonly tableName?: string;
    /**
     * The ARN of the table's stream.
     *
     * @default - no table stream
     * @stability stable
     */
    readonly tableStreamArn?: string;
    /**
     * KMS encryption key, if this table uses a customer-managed encryption key.
     *
     * @default - no key
     * @stability stable
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * The name of the global indexes set for this Table.
     *
     * Note that you need to set either this property,
     * or {@link localIndexes},
     * if you want methods like grantReadData()
     * to grant permissions for indexes as well as the table itself.
     *
     * @default - no global indexes
     * @stability stable
     */
    readonly globalIndexes?: string[];
    /**
     * The name of the local indexes set for this Table.
     *
     * Note that you need to set either this property,
     * or {@link globalIndexes},
     * if you want methods like grantReadData()
     * to grant permissions for indexes as well as the table itself.
     *
     * @default - no local indexes
     * @stability stable
     */
    readonly localIndexes?: string[];
}
declare abstract class TableBase extends Resource implements ITable {
    /**
     * @attribute
     */
    abstract readonly tableArn: string;
    /**
     * @attribute
     */
    abstract readonly tableName: string;
    /**
     * @attribute
     */
    abstract readonly tableStreamArn?: string;
    /**
     * KMS encryption key, if this table uses a customer-managed encryption key.
     */
    abstract readonly encryptionKey?: kms.IKey;
    /**
     * @stability stable
     */
    protected readonly regionalArns: string[];
    /**
     * Adds an IAM policy statement associated with this table to an IAM principal's policy.
     *
     * If `encryptionKey` is present, appropriate grants to the key needs to be added
     * separately using the `table.encryptionKey.grant*` methods.
     *
     * @param grantee The principal (no-op if undefined).
     * @param actions The set of actions to allow (i.e. "dynamodb:PutItem", "dynamodb:GetItem", ...).
     * @stability stable
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Adds an IAM policy statement associated with this table's stream to an IAM principal's policy.
     *
     * If `encryptionKey` is present, appropriate grants to the key needs to be added
     * separately using the `table.encryptionKey.grant*` methods.
     *
     * @param grantee The principal (no-op if undefined).
     * @param actions The set of actions to allow (i.e. "dynamodb:DescribeStream", "dynamodb:GetRecords", ...).
     * @stability stable
     */
    grantStream(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Permits an IAM principal all data read operations from this table: BatchGetItem, GetRecords, GetShardIterator, Query, GetItem, Scan.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantReadData(grantee: iam.IGrantable): iam.Grant;
    /**
     * Permits an IAM Principal to list streams attached to current dynamodb table.
     *
     * @param grantee The principal (no-op if undefined).
     * @stability stable
     */
    grantTableListStreams(grantee: iam.IGrantable): iam.Grant;
    /**
     * Permits an IAM principal all stream data read operations for this table's stream: DescribeStream, GetRecords, GetShardIterator, ListStreams.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantStreamRead(grantee: iam.IGrantable): iam.Grant;
    /**
     * Permits an IAM principal all data write operations to this table: BatchWriteItem, PutItem, UpdateItem, DeleteItem.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantWriteData(grantee: iam.IGrantable): iam.Grant;
    /**
     * Permits an IAM principal to all data read/write operations to this table.
     *
     * BatchGetItem, GetRecords, GetShardIterator, Query, GetItem, Scan,
     * BatchWriteItem, PutItem, UpdateItem, DeleteItem
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantReadWriteData(grantee: iam.IGrantable): iam.Grant;
    /**
     * Permits all DynamoDB operations ("dynamodb:*") to an IAM principal.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantFullAccess(grantee: iam.IGrantable): iam.Grant;
    /**
     * Return the given named metric for this Table.
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     *
     * @stability stable
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the consumed read capacity units this table.
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     *
     * @stability stable
     */
    metricConsumedReadCapacityUnits(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the consumed write capacity units this table.
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     *
     * @stability stable
     */
    metricConsumedWriteCapacityUnits(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * (deprecated) Metric for the system errors this table.
     *
     * @deprecated use `metricSystemErrorsForOperations`.
     */
    metricSystemErrors(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the user errors.
     *
     * Note that this metric reports user errors across all
     * the tables in the account and region the table resides in.
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     *
     * @stability stable
     */
    metricUserErrors(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the conditional check failed requests this table.
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     *
     * @stability stable
     */
    metricConditionalCheckFailedRequests(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * How many requests are throttled on this table.
     *
     * Default: sum over 5 minutes
     *
     * @stability stable
     */
    metricThrottledRequests(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the successful request latency this table.
     *
     * By default, the metric will be calculated as an average over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     *
     * @stability stable
     */
    metricSuccessfulRequestLatency(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the system errors this table.
     *
     * This will sum errors across all possible operations.
     * Note that by default, each individual metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     *
     * @stability stable
     */
    metricSystemErrorsForOperations(props?: SystemErrorsForOperationsMetricOptions): cloudwatch.IMetric;
    /**
     * Create a map of metrics that can be used in a math expression.
     *
     * Using the return value of this function as the `usingMetrics` property in `cloudwatch.MathExpression` allows you to
     * use the keys of this map as metric names inside you expression.
     *
     * @param metricName The metric name.
     * @param operations The list of operations to create metrics for.
     * @param props Properties for the individual metrics.
     * @param metricNameMapper Mapper function to allow controlling the individual metric name per operation.
     */
    private createMetricsForOperations;
    protected abstract get hasIndex(): boolean;
    /**
     * Adds an IAM policy statement associated with this table to an IAM
     * principal's policy.
     * @param grantee The principal (no-op if undefined)
     * @param opts Options for keyActions, tableActions and streamActions
     */
    private combinedGrant;
    private cannedMetric;
}
/**
 * Provides a DynamoDB table.
 *
 * @stability stable
 */
export declare class Table extends TableBase {
    /**
     * (deprecated) Permits an IAM Principal to list all DynamoDB Streams.
     *
     * @param grantee The principal (no-op if undefined).
     * @deprecated Use {@link #grantTableListStreams} for more granular permission
     */
    static grantListStreams(grantee: iam.IGrantable): iam.Grant;
    /**
     * Creates a Table construct that represents an external table via table name.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param tableName The table's name.
     * @stability stable
     */
    static fromTableName(scope: Construct, id: string, tableName: string): ITable;
    /**
     * Creates a Table construct that represents an external table via table arn.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param tableArn The table's ARN.
     * @stability stable
     */
    static fromTableArn(scope: Construct, id: string, tableArn: string): ITable;
    /**
     * Creates a Table construct that represents an external table.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `TableAttributes` object.
     * @stability stable
     */
    static fromTableAttributes(scope: Construct, id: string, attrs: TableAttributes): ITable;
    /**
     * KMS encryption key, if this table uses a customer-managed encryption key.
     *
     * @stability stable
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * Arn of the dynamodb table.
     *
     * @stability stable
     * @attribute true
     */
    readonly tableArn: string;
    /**
     * Table name of the dynamodb table.
     *
     * @stability stable
     * @attribute true
     */
    readonly tableName: string;
    /**
     * ARN of the table's stream, if there is one.
     *
     * @stability stable
     * @attribute true
     */
    readonly tableStreamArn: string | undefined;
    private readonly table;
    private readonly keySchema;
    private readonly attributeDefinitions;
    private readonly globalSecondaryIndexes;
    private readonly localSecondaryIndexes;
    private readonly secondaryIndexNames;
    private readonly nonKeyAttributes;
    private readonly tablePartitionKey;
    private readonly tableSortKey?;
    private readonly billingMode;
    private readonly tableScaling;
    private readonly indexScaling;
    private readonly scalingRole;
    private readonly globalReplicaCustomResources;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: TableProps);
    /**
     * Add a global secondary index of table.
     *
     * @param props the property of global secondary index.
     * @stability stable
     */
    addGlobalSecondaryIndex(props: GlobalSecondaryIndexProps): void;
    /**
     * Add a local secondary index of table.
     *
     * @param props the property of local secondary index.
     * @stability stable
     */
    addLocalSecondaryIndex(props: LocalSecondaryIndexProps): void;
    /**
     * Enable read capacity scaling for this table.
     *
     * @returns An object to configure additional AutoScaling settings
     * @stability stable
     */
    autoScaleReadCapacity(props: EnableScalingProps): IScalableTableAttribute;
    /**
     * Enable write capacity scaling for this table.
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     * @stability stable
     */
    autoScaleWriteCapacity(props: EnableScalingProps): IScalableTableAttribute;
    /**
     * Enable read capacity scaling for the given GSI.
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     * @stability stable
     */
    autoScaleGlobalSecondaryIndexReadCapacity(indexName: string, props: EnableScalingProps): IScalableTableAttribute;
    /**
     * Enable write capacity scaling for the given GSI.
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     * @stability stable
     */
    autoScaleGlobalSecondaryIndexWriteCapacity(indexName: string, props: EnableScalingProps): IScalableTableAttribute;
    /**
     * Validate the table construct.
     *
     * @returns an array of validation error message
     * @stability stable
     */
    protected validate(): string[];
    /**
     * Validate read and write capacity are not specified for on-demand tables (billing mode PAY_PER_REQUEST).
     *
     * @param props read and write capacity properties
     */
    private validateProvisioning;
    /**
     * Validate index name to check if a duplicate name already exists.
     *
     * @param indexName a name of global or local secondary index
     */
    private validateIndexName;
    /**
     * Validate non-key attributes by checking limits within secondary index, which may vary in future.
     *
     * @param nonKeyAttributes a list of non-key attribute names
     */
    private validateNonKeyAttributes;
    private buildIndexKeySchema;
    private buildIndexProjection;
    private findKey;
    private addKey;
    /**
     * Register the key attribute of table or secondary index to assemble attribute definitions of TableResourceProps.
     *
     * @param attribute the key attribute of table or secondary index
     */
    private registerAttribute;
    /**
     * Return the role that will be used for AutoScaling
     */
    private makeScalingRole;
    /**
     * Creates replica tables
     *
     * @param regions regions where to create tables
     */
    private createReplicaTables;
    /**
     * Whether this table has indexes.
     *
     * @stability stable
     */
    protected get hasIndex(): boolean;
    /**
     * Set up key properties and return the Table encryption property from the
     * user's configuration.
     */
    private parseEncryption;
}
/**
 * Data types for attributes within a table.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes
 * @stability stable
 */
export declare enum AttributeType {
    /**
     * Up to 400KiB of binary data (which must be encoded as base64 before sending to DynamoDB).
     *
     * @stability stable
     */
    BINARY = "B",
    /**
     * Numeric values made of up to 38 digits (positive, negative or zero).
     *
     * @stability stable
     */
    NUMBER = "N",
    /**
     * Up to 400KiB of UTF-8 encoded text.
     *
     * @stability stable
     */
    STRING = "S"
}
/**
 * DyanmoDB's Read/Write capacity modes.
 *
 * @stability stable
 */
export declare enum BillingMode {
    /**
     * Pay only for what you use.
     *
     * You don't configure Read/Write capacity units.
     *
     * @stability stable
     */
    PAY_PER_REQUEST = "PAY_PER_REQUEST",
    /**
     * Explicitly specified Read/Write capacity units.
     *
     * @stability stable
     */
    PROVISIONED = "PROVISIONED"
}
/**
 * The set of attributes that are projected into the index.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Projection.html
 * @stability stable
 */
export declare enum ProjectionType {
    /**
     * Only the index and primary keys are projected into the index.
     *
     * @stability stable
     */
    KEYS_ONLY = "KEYS_ONLY",
    /**
     * Only the specified table attributes are projected into the index.
     *
     * The list of projected attributes is in `nonKeyAttributes`.
     *
     * @stability stable
     */
    INCLUDE = "INCLUDE",
    /**
     * All of the table attributes are projected into the index.
     *
     * @stability stable
     */
    ALL = "ALL"
}
/**
 * When an item in the table is modified, StreamViewType determines what information is written to the stream for this table.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_StreamSpecification.html
 * @stability stable
 */
export declare enum StreamViewType {
    /**
     * The entire item, as it appears after it was modified, is written to the stream.
     *
     * @stability stable
     */
    NEW_IMAGE = "NEW_IMAGE",
    /**
     * The entire item, as it appeared before it was modified, is written to the stream.
     *
     * @stability stable
     */
    OLD_IMAGE = "OLD_IMAGE",
    /**
     * Both the new and the old item images of the item are written to the stream.
     *
     * @stability stable
     */
    NEW_AND_OLD_IMAGES = "NEW_AND_OLD_IMAGES",
    /**
     * Only the key attributes of the modified item are written to the stream.
     *
     * @stability stable
     */
    KEYS_ONLY = "KEYS_ONLY"
}
export {};
