# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/window_generator.ipynb.

# %% auto 0
__all__ = ['WindowGenerator']

# %% ../nbs/window_generator.ipynb 3
from pathlib import Path
import pandas as pd
import numpy as np
import joblib
import tensorflow as tf
import matplotlib.pyplot as plt
import matplotlib as mpl
from typing import Dict, Optional, Union, Tuple, Sequence

from . import defaults as dflt

# %% ../nbs/window_generator.ipynb 5
class WindowGenerator():
    r"""Window generator for time-series forecasting.
    
    This class efficiently generate batches of windows from the training, 
    validation, and test data, using `tf.data.Datasets` (see `make_dataset`
    method). Each window is further split into ((input_features, 
    target_time_features, input_dates, target_dates), targets) tuples,
    see `split_window` method.
    
    Parameters
    ----------
    input_width: int, default 28 (4 weeks)
        Width (number of time steps) of the input.
    target_width: int, default 28 (4 weeks)
        Width (number of time steps) of the target.
    shift: int or None, default None
        Time offset between the end of the input window and the beginning of 
        the next input window. If None is given, `target_width` is used as
        shift.
    target_columns: list-like, default ['TotalPrice']
        Name of the features used as target.
    time_ft_columns: list-like, default: columns generated by `PeriodicTimeTransform`.
        Name of time features.
    batch_size: int, default 32
        Size of batches of windows generated.
    """
    def __init__(
        self, 
        input_width: int=dflt.input_width, 
        target_width: int=dflt.target_width, 
        shift: Optional[int]=None,
        target_columns: Sequence=dflt.target_columns, 
        time_ft_columns: Sequence=dflt.time_ft_columns, 
        batch_size: int=dflt.batch_size
    ):
        shift = target_width if shift is None else shift
        # Work out the target column indices.
        self.target_columns = target_columns
        if target_columns is not None:
            self.target_columns_indices = {name: i for i, name in enumerate(target_columns)}
        self.time_ft_columns = time_ft_columns
        if time_ft_columns is not None:
            self.time_ft_columns_indices = {name: i for i, name in enumerate(time_ft_columns)}
        
        # Work out the window parameters.
        self.input_width = input_width
        self.target_width = target_width
        self.shift = shift

        self.total_window_size = input_width + shift

        self.input_slice = slice(0, input_width)
        self.input_indices = np.arange(self.total_window_size)[self.input_slice]

        self.target_start = self.total_window_size - self.target_width
        self.targets_slice = slice(self.target_start, None)
        self.target_indices = np.arange(self.total_window_size)[self.targets_slice]
        
        # other hyper-parameters
        self.batch_size = batch_size
        
    def setup (self, df_dict: Dict[str, pd.DataFrame]):
        r"""Stores the training, validation, and test DataFrames.

        These will be converted to tf.data.Datasets of windows later.

        Parameters
        ----------
        df_dict: dict
            Dictionary of dataframes, one per subset (training, validation,
            and test).
        """
        # Store the raw data.
        self.train_df = df_dict['training'] if 'training' in df_dict else None
        self.val_df = df_dict['validation'] if 'validation' in df_dict else None
        self.test_df = df_dict['test'] if 'test' in df_dict else None
        self.main_df = (self.train_df if self.train_df is not None 
                        else self.test_df if self.test_df is not None 
                        else self.val_df if self.val_df is not None 
                        else None)
        
        # Calculate the last period in the training+validation data. 
        # To be used as prior data when predicting targets over a given
        # period of time.
        if 'validation' in df_dict or 'training' in df_dict:
            df_last = (df_dict['validation'] if ('validation' in df_dict and 'test' in df_dict) 
                       else df_dict['training'])
            self.last_period_df = df_last[-self.input_width:]
            self.last_time_step_plus_1 = self.last_period_df.index[-1] + pd.offsets.Day()
        else:
            self.last_period_df = None
            self.last_time_step_plus_1 = None
        
        assert self.main_df is not None, 'there must be at least one split given in df_dict'

        # Create target columns if not present in test data-frame, and fill them with zeros.
        if self.target_columns is not None:
            if self.test_df is not None and not set(self.target_columns).issubset (self.test_df.columns):
                self.test_df [self.target_columns] = 0
            if not set(self.target_columns).issubset (self.main_df.columns):
                self.main_df [self.target_columns] = 0
            self._target_columns_in_input_indices = [i for i, x in enumerate (self.main_df.columns) if x in self.target_columns]
        else:
            self._target_columns_in_input_indices = range (len(self.main_df.columns))
        if len(self._target_columns_in_input_indices) != (max(self._target_columns_in_input_indices) - min(self._target_columns_in_input_indices) +1):
            raise ValueError ('target columns must be consecutive in original dataframe')
        self._target_columns_in_input_indices = slice (min(self._target_columns_in_input_indices), max(self._target_columns_in_input_indices)+1)
        
        self.column_indices = {name: i for i, name in enumerate(self.main_df.columns)}
        
    def __repr__(self):
        return '\n'.join([
            f'Total window size: {self.total_window_size}',
            f'Input indices: {self.input_indices}',
            f'Target indices: {self.target_indices}',
            f'Target column name(s): {self.target_columns}',
            f'Time features column name(s): {self.time_ft_columns}'])
    
    def split_window(self, features: pd.DataFrame) -> Tuple:
        r"""Splits the data into tuples with features and labels.
        
        Splits the data into tuples containing the features, labels, and 
        dates for the input and target windows.
                
        Parameters
        ----------
        features: DataFrame
            DataFrame to be split into tuples of input and target 
            features and labels.
        
        Returns
        -------
        tuple
            Tuple of the form: 
            ((input_features, target_time_features, input_dates, target_dates), 
             targets)
            where:
            - input_features: features for the input window.
            - target_time_features: time-based features for the target window, since 
                those only depend on the date and are known beforehand.
            - input_dates: dates associated with each time-step of the input window, 
                mainly used for visualization and debugging purposes.
            - target_dates: dates for the target window, mainly used for visualization 
                and debugging purposes.
            - targets: ground-truth values or labels for each time-step of the target
                window, used for training and evaluation.
        """
        # extract time index
        input_time_index = features[:, self.input_slice, 0]
        target_time_index = features[:, self.targets_slice, 0]
        features = features[:, :, 1:]
        
        inputs = features[:, self.input_slice, :]
        targets = features[:, self.targets_slice, :]
        if self.target_columns is not None:
            targets = tf.stack(
                [targets[:, :, self.column_indices[name]] for name in self.target_columns],
                axis=-1)
        time_ft = features[:, self.targets_slice, :]
        if self.time_ft_columns is not None:
            time_ft = tf.stack(
                [time_ft[:, :, self.column_indices[name]] for name in self.time_ft_columns],
                axis=-1)

        # Slicing doesn't preserve static shape information, so set the shapes
        # manually. This way the `tf.data.Datasets` are easier to inspect.
        inputs.set_shape([None, self.input_width, None])
        targets.set_shape([None, self.target_width, None])
        time_ft.set_shape([None, self.target_width, None])
        
        return (inputs, time_ft, input_time_index, target_time_index), targets
    
    def plot(
        self, 
        model: Optional['Model']=None, 
        plot_col: str='TotalPrice', 
        max_subplots: int=3
    ):
        r"""Plots examples of input and target windows, overlaid with predictions.
        
        Parameters
        ----------
        model: Model or None, default None
            Model used for visualizing predictions.
        plot_col: str, default 'TotalPrice'
            Name of column with values to visualize.
        max_subplots: Model or None, default None
            Maximum number of example windows to visualize, one in each subplot.
        """
        # unpack
        input_model, targets = self.example
        inputs, time_ft, input_time_index, target_time_index = input_model
        input_x = input_time_index.numpy().astype('datetime64[ns]')
        target_x = target_time_index.numpy().astype('datetime64[ns]')
            
        plt.figure(figsize=(12, 8))
        plot_col_index = self.column_indices[plot_col]
        max_n = min(max_subplots, len(inputs))
        for n in range(max_n):
            plt.subplot(max_n, 1, n+1)
            plt.ylabel(f'{plot_col} [normed]')
            
            plt.plot(input_x[n,:], inputs[n, :, plot_col_index],
                     label='Inputs', marker='.', zorder=-10)

            if self.target_columns:
                target_col_index = self.target_columns_indices.get(plot_col, None)
            else:
                target_col_index = plot_col_index

            if target_col_index is None:
                continue

            plt.scatter(target_x[n,:], targets[n, :, target_col_index],
                        edgecolors='k', label='Targets', c='#2ca02c', s=64)
            if model is not None:
                predictions = model(input_model)
                plt.scatter(target_x[n,:], predictions[n, :, target_col_index],
                          marker='X', edgecolors='k', label='Predictions',
                          c='#ff7f0e', s=64)

            if n == 0:
                plt.legend()

        plt.xlabel('Time [days]')
    
    def make_dataset (
        self, 
        data: pd.DataFrame, 
        shuffle: bool=True, 
        batch_size: Optional[int]=None
    ) -> tf.data.Dataset:
        r"""Generate batches of windows from input data using `tf.data.Datasets`.
        
        Each window is further split into ((input_features, 
        target_time_features, input_dates, target_dates), targets) tuples,
        see `split_window` method.
        
        Parameters
        ----------
        data: DataFrame
            DataFrame from which the batches of windows are generated.
        shuffle: bool, default True
            If True, generated windows are shuffled.
        batch_size: int or None, default None
            Number of windows per batch. If None is given, the `batch_size`
            indicated in the object constructor is used instead.
        
        Returns
        -------
        tf.data.Dataset
            Batches of windows generated from input data.
        """
        data = np.concatenate ([np.array(data.index, dtype=int).reshape(-1,1), np.array(data, dtype=np.float32)], axis=1)
        
        ds = tf.keras.utils.timeseries_dataset_from_array(
                  data=data,
                  targets=None,
                  sequence_length=self.total_window_size,
                  sequence_stride=1,
                  shuffle=shuffle,
                  batch_size=self.batch_size if batch_size is None else batch_size)

        ds = ds.map(self.split_window)
        
        return ds
        
    @property
    def train(self) -> tf.data.Dataset:
        return self.make_dataset(self.train_df, shuffle=True)

    @property
    def val(self) -> tf.data.Dataset:
        return self.make_dataset(self.val_df, shuffle=True)

    @property
    def test(self) -> tf.data.Dataset:
        return self.make_dataset(self.test_df, shuffle=False)
    
    @property
    def main(self) -> tf.data.Dataset:
        return self.make_dataset(self.main_df, shuffle=False)
    
    @property
    def example(self) -> Tuple:
        r"""Get and cache an example batch of `inputs, targets` for plotting.
        
        """
        result = getattr(self, '_example', None)
        if result is None:
            # No example batch was found, so get one from the `.train` dataset
            result = next(iter(self.train))
            # And cache it for next time
            self._example = result
        return result
    
    def save (self, path: Union[str, Path]) -> None:
        r"""Saves data subsets stored in window_generator object.
                
        Parameters
        ----------
        path: str or Path
            Path to folder where window_generator object are saved.
        """
        path = Path (path) / 'subsets'
        path.mkdir (parents=True, exist_ok=True)
        if self.train_df is not None:
            self.train_df.to_parquet (path / 'train_df.parquet')
            #self.train_df.to_csv (path / 'train_df.csv')
        if self.val_df is not None:
            self.val_df.to_parquet (path / 'val_df.parquet')
            #self.val_df.to_csv (path / 'val_df.csv')
        if self.test_df is not None:
            self.test_df.to_parquet (path / 'test_df.parquet')
            #self.test_df.to_csv (path / 'test_df.csv')
        
        # extract dataframes
        train_df, val_df, test_df = self.train_df, self.val_df, self.test_df
        self.train_df, self.val_df, self.test_df = None, None, None
        
        # store remaining fields
        joblib.dump (self, path.parent / 'window_generator.pk')
        
        # restore dataframes
        self.train_df, self.val_df, self.test_df = train_df, val_df, test_df
    
    def load (self, path: Union[str, Path]) -> None:
        r"""Loads data subsets into window_generator fields.
                
        Parameters
        ----------
        path: str or Path
            Path to folder where window_generator object are saved.
        """
        path = Path (path) / 'subsets'
        if (path / 'train_df.parquet').exists ():
            self.train_df = pd.read_parquet (path / 'train_df.parquet')
        if (path / 'val_df.parquet').exists ():
            self.val_df = pd.read_parquet (path / 'val_df.parquet')
        if (path / 'test_df.parquet').exists ():
            self.test_df = pd.read_parquet (path / 'test_df.parquet')
