# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/model.ipynb.

# %% auto 0
__all__ = ['MultiStepLastBaseline', 'RepeatBaseline', 'FeedBackWithTimeFeatures', 'TimeIndexer', 'Model']

# %% ../nbs/model.ipynb 3
from pathlib import Path
import tensorflow as tf
import pandas as pd
import numpy as np
from typing import Dict, Optional, Union, Tuple

from . import defaults as dflt

# %% ../nbs/model.ipynb 7
# Baselines
class MultiStepLastBaseline(tf.keras.Model):
    def call(self, inputs):
        return tf.tile(inputs[0][:, -1:, self._target_indices], [1, OUT_STEPS, 1])

# %% ../nbs/model.ipynb 9
class RepeatBaseline(tf.keras.Model):
    def call(self, inputs):
        return inputs[0][:,:,self._target_indices]

# %% ../nbs/model.ipynb 11
class FeedBackWithTimeFeatures(tf.keras.Model):
    def __init__(self, 
                 units=dflt.units, 
                 dropout=dflt.dropout, 
                 recurrent_dropout=dflt.recurrent_dropout, 
                 lstm_regularizer_type=dflt.lstm_regularizer_type,
                 lstm_regularizer_magnitude=dflt.lstm_regularizer_magnitude,
                 dense_regularizer_type=dflt.dense_regularizer_type,
                 dense_regularizer_magnitude=dflt.dense_regularizer_magnitude,
                 out_steps=dflt.out_steps):
        super().__init__()
        self.out_steps = out_steps
        self.units = units
        
        # TODO: add GaussianNoise as first layer
            
        self.lstm_cell = tf.keras.layers.LSTMCell(units, 
                                                  dropout=dropout, 
                                                  recurrent_dropout=recurrent_dropout,
                                                  **self._get_regularizer (lstm_regularizer_type, lstm_regularizer_magnitude))
        # Also wrap the LSTMCell in an RNN to simplify the `warmup` method.
        self.lstm_rnn = tf.keras.layers.RNN(self.lstm_cell, return_state=True)
        self.dense = tf.keras.layers.Dense(1, **self._get_regularizer (dense_regularizer_type, dense_regularizer_magnitude))

    def _get_regularizer (self, regularizer_type, regularizer_magnitude):
        if regularizer_type is None:
            regularizer = dict ()
        elif lstm_regularizer_type=='L1':
            regularizer = dict (kernel_regularizer=tf.keras.regularizers.L1 (regularizer_magnitude))
        elif regularizer_type=='L2':
            regularizer = dict (kernel_regularizer=tf.keras.regularizers.L2 (regularizer_magnitude))
        else:
            raise ValueError (f'{regularizer_type} not recognized')
        return regularizer
        
    def warmup(self, inputs):
        """
        Initializes the model internal state based on the inputs. 
        
        Once trained, this state will capture the relevant parts 
        of the input history. This is equivalent to the single-step 
        `LSTM` model.
        """
        # inputs[0].shape => (batch, time, 1)
        # x.shape => (batch, lstm_units)
        x, *state = self.lstm_rnn(inputs[0])

        # predictions.shape => (batch, 1)
        prediction = self.dense(x)
        return prediction, state
    
    def predict_from_numpy_array (self, inputs, out_steps):
        # Use a TensorArray to capture dynamically unrolled outputs.
        predictions = []
        # Initialize the LSTM state.
        prediction, state = self.warmup(inputs)

        # Insert the first prediction.
        predictions.append(prediction)

        # Run the rest of the prediction steps.
        for n in range(1, out_steps):
            # Use the last prediction as input, stacking time-features to it
            # prediction.shape => (batch, 1)
            # x.shape => (batch, features)
            x = tf.concat([prediction, inputs[1][:,n,:]], axis=1)

            # Execute one lstm step.
            x, state = self.lstm_cell(x, states=state,
                                      training=False)
            # Convert the lstm output to a prediction.
            prediction = self.dense(x)
            # Add the prediction to the output.
            predictions.append(prediction)

        # predictions.shape => (time, batch, features)
        predictions = tf.stack(predictions)
        # predictions.shape => (batch, time, features)
        predictions = tf.transpose(predictions, [1, 0, 2])
        # predictions.shape => (batch, time, 1)
        predictions = predictions[:,:,0]
        predictions = predictions[:,:, tf.newaxis]

        return predictions
    
    def call(self, inputs, training=None):
        """
        Prediction step.
        
        Iterates the model feeding the predictions at each step back as the input.
        """
        out_steps = self.out_steps
        # Use a TensorArray to capture dynamically unrolled outputs.
        predictions = []
        # Initialize the LSTM state.
        prediction, state = self.warmup(inputs)

        # Insert the first prediction.
        predictions.append(prediction)

        # Run the rest of the prediction steps.
        for n in range(1, out_steps):
            # Use the last prediction as input, stacking time-features to it
            # prediction.shape => (batch, 1)
            # x.shape => (batch, features)
            x = tf.concat([prediction, inputs[1][:,n,:]], axis=1)

            # Execute one lstm step.
            x, state = self.lstm_cell(x, states=state,
                                      training=training)
            # Convert the lstm output to a prediction.
            prediction = self.dense(x)
            # Add the prediction to the output.
            predictions.append(prediction)

        # predictions.shape => (time, batch, features)
        predictions = tf.stack(predictions)
        # predictions.shape => (batch, time, features)
        predictions = tf.transpose(predictions, [1, 0, 2])
        # predictions.shape => (batch, time, 1)
        predictions = predictions[:,:,0]
        predictions = predictions[:,:, tf.newaxis]

        return predictions

# %% ../nbs/model.ipynb 16
class TimeIndexer(tf.keras.Model):
    def __init__ (self, time_index=3):
        super().__init__()
        self.time_index = time_index
    def call(self, inputs):
        return inputs[self.time_index]
    def get_time_index (self, windows):
        time_index = self.predict (windows)
        return time_index.astype('datetime64[ns]')

# %% ../nbs/model.ipynb 18
class Model ():
    r"""Wrapper around forecasting model.
    
    This class wraps the selected forecasting model and presents
    a uniform API. 
    
    Parameters
    ----------
    model_class: str, default 'FeedBackWithTimeFeatures'
        Name of model class to be used for forecasting.
    patience: int, default 2
        Patience used in early stop callback.
    epochs: int, default 20
        Epochs used for fitting the model.
    learning_rate: float, default 0.001 (default for Adam optimizer)
        Learning rate used for fitting the model.
    round_period: str, default 'D'
        Dates are rounded to this unit. Useful for reporting the 
        dates associated to each time-step in the predictions, for 
        visualization and debugging.
    out_steps: int, default 28 (4 weeks)
        Number of time-steps (days) to predict in the future.
    verbose: int, default 0
        Verbosity level used during fitting.
    """
    def __init__ (
        self, 
        model_class: str=dflt.model_class,
        patience: int=dflt.patience, 
        epochs: int=dflt.epochs,
        learning_rate: float=dflt.learning_rate,
        round_period: str='D',
        out_steps: int=dflt.out_steps,
        verbose: int=dflt.verbose,
        **kwargs
    ):
        Model = eval (model_class)
        model_parameters = dict (out_steps=out_steps, **kwargs) if model_class == 'FeedBackWithTimeFeatures' else kwargs
        self._model = Model (**model_parameters)
        self.patience = patience
        self.epochs = epochs
        self.learning_rate = learning_rate
        self._time_indexer = TimeIndexer ()
        self.round_period = round_period
        self.verbose = verbose
    
    def setup (self):
        r"""Sets up the NN model, creating callbacks and compiling it.
        
        """
        self.early_stopping = tf.keras.callbacks.EarlyStopping(monitor='val_loss',
                                                            patience=self.patience,
                                                            mode='min')

        self._model.compile(loss=tf.keras.losses.MeanSquaredError(),
                            optimizer=tf.keras.optimizers.Adam(learning_rate=self.learning_rate),
                            metrics=[tf.keras.metrics.MeanAbsoluteError()])
        self._time_indexer.compile ()

    def fit (self, windows: 'WindowGenerator') -> 'History':
        r"""Fits the NN model to the training data.
        
        Parameters
        ----------
        windows: WindowGenerator
            Training and validation windows used for fitting and monitoring
            the model performance across epochs.

        Returns
        -------
        History:
            History keras object storing monitored metrics across training epochs.
        """
        # only used by baseline models
        self._model._target_indices = windows._target_columns_in_input_indices
        history = self._model.fit(windows.train, epochs=self.epochs,
                                  validation_data=windows.val if windows.val_df is not None else None,
                                  callbacks=[self.early_stopping] if windows.val_df is not None else None,
                                  verbose=self.verbose)
        return history
    
    
    def evaluate (self, windows: 'WindowGenerator') -> Dict[str, dict]:
        r"""Evaluates the model performance. 

        Parameters
        ----------
        windows: WindowGenerator
            Validation and test windows used for evaluating the model
            performance. Either validation, test, or both subsets can
            be included in the windows object.
        
        Returns
        -------
        dict
            Dictionary providing evaluation metrics for each subset. 
            It contains one key per subset (validation and test) used 
            for the evaluation. For each subset, a dictionary is provided
            with the different metric scores used. Current metrics are the
            loss and the mean absolute error.
        """
        performance = {}
        if windows.val_df is not None:
            performance['validation'] = self._model.evaluate(windows.val)
        if windows.test_df is not None:
            performance['test'] = self._model.evaluate(windows.test)
        return performance
    
    def predict (
        self, 
        windows: 'WindowGenerator', 
        subset: str='main', 
        include_time_index: bool=True, 
        single_period: bool=False
    ) -> pd.DataFrame:
        r"""Predicts targets for the requested subset of windows.
        
        Parameters
        ----------
        windows: WindowGenerator
            Windows on which predictions are performed.
        subset: str, default 'main'
            Substet, in the windows object, used for performing the predictions.
        include_time_index: bool, default True
            If True, the result contains, as index of the dataframe, the dates 
            associated with each predicted time-step. If `single_period`=False,
            there is one window with multiple time-steps per row, and the index
            provides the start and end date for the entire window. If 
            `single_period`=False, each row contains a single time-step prediction 
            and the index of the row is the date associated with this time-step.
        single_period: bool, default False
            If `single_period`=False, there is one window with multiple time-steps 
            per row. If `single_period`=False, each row contains a single time-step 
            prediction.
            
        Returns
        -------
        DataFrame
            DataFrame with predictions. 
        """
        subset = getattr(windows, subset)
        if single_period:
            subset_np = subset.as_numpy_iterator ()
            subset_np = list(subset_np)[0][0]
            pred = self._model.predict_from_numpy_array (subset_np, out_steps=subset_np[1].shape[1])
        else:
            pred = self._model.predict (subset)
        pred = np.squeeze(pred)
        if include_time_index:
            time_index = self._time_indexer.get_time_index (subset)
            time_index = np.squeeze(time_index)
            if single_period and len(pred.shape)==1:
                time_index = pd.Series(time_index.astype('datetime64[ns]'))
                time_index = time_index.round(self.round_period) if self.round_period is not None else time_index
                return pd.DataFrame (pred, index=time_index)
            else:
                min_t, max_t = time_index.min(axis=1), time_index.max(axis=1)
                min_t = pd.Series(min_t).round(self.round_period) if self.round_period is not None else min_t
                max_t = pd.Series(max_t).round(self.round_period) if self.round_period is not None else max_t
                df_index = pd.MultiIndex.from_tuples (zip(min_t, max_t))
                return pd.DataFrame (pred, index=df_index)
        else:
            return pd.DataFrame(pred)
    
    def load_model (self, path: Union[str, Path]) -> None:
        r"""Loads a model from disk.

        Parameters
        ----------
        path: str or Path
            Path to folder where model is saved.
        """
        path = Path(path)
        self._model.load_weights (path / 'model')
    
    def save_model (self, path: Union[str, Path]) -> None:
        r"""Saves the model to disk.

        Parameters
        ----------
        path: str or Path
            Path to folder where model is saved.
        """
        path = Path (path)
        self._model.save_weights (path / 'model')
    
    def __repr__ (self):
        return (self._model.__class__.__name__)
    
    def get_model (self) -> 'Model':
        return self._model
    
    def set_model (self, model: 'Model') -> None:
        self._model = model        
