# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/hyper_parameters.ipynb.

# %% auto 0
__all__ = ['hp_search', 'plot_selected', 'analysis', 'retrain_and_save_selected_experiment']

# %% ../nbs/hyper_parameters.ipynb 4
from sklearn.model_selection import ParameterGrid
import numpy as np
import pandas as pd
from pathlib import Path
import joblib
import matplotlib.pyplot as plt
from IPython.display import display
import os

from .forecasting_pipeline import ForecastingPipeline, SplitError
import tsforecast.defaults as dflt

# %% ../nbs/hyper_parameters.ipynb 6
def hp_search (include_baseline=False, 
               model_parameters_grid={}, 
               window_parameters_grid={},
               time_transform_parameters_grid={},
               pipeline_parameters={},
               debug=False,
               path_experiments=dflt.path_experiments,
               splits=None,
               metric_names = ['loss', 'mae'],
               performance_df=None,
               load_previous=True):
    
    if splits is None:
        if 'split_ratios' in pipeline_parameters:
            splits = pipeline_parameters['split_ratios'].keys()
        else:
            splits = ['validation', 'test']
    parameters_grid = dict (
        **model_parameters_grid,
        **window_parameters_grid,
        **time_transform_parameters_grid
    )
    parameters_list = list(ParameterGrid (parameters_grid))
    if include_baseline:
        baseline_parameters_grid = dict (
            model_class = ['MultiStepLastBaseline', 'RepeatBaseline'],
            input_width = window_parameters_grid['input_width']
        )
        parameters_list = list (ParameterGrid (baseline_parameters_grid)) + parameters_list
    
    performance_names = [f'{s}_{m}' for s in splits for m in metric_names]
    other_info = [f'windows_{split}' for split in splits] + [f'steps_{split}' for split in splits] + ['total_time_steps', 'total_window_size']
    columns = pd.MultiIndex.from_tuples (
        pd.MultiIndex.from_product([['performance'], performance_names]).values.tolist() + 
        pd.MultiIndex.from_product([['parameters'], parameters_grid.keys()]).values.tolist()+
        pd.MultiIndex.from_product([['other_info'], other_info]).values.tolist())

    if debug:
        parameters_list = parameters_list[:2]

    path_experiments = Path(path_experiments)
    path_experiments.mkdir (parents=True, exist_ok=True)

    if performance_df is None:
        if load_previous and (path_experiments  / 'performance.pk').exists():
            performance_df = pd.read_pickle (path_experiments  / 'performance.pk')
            next_experiment = performance_df.index.max()+1
        else:
            performance_df = pd.DataFrame (columns=columns)
            next_experiment = 0
    else:
        next_experiment = performance_df.index.max()+1
        
    for experiment_number, parameters in enumerate(parameters_list, next_experiment):
        
        print (f'\n{"*"*50}')
        print (f'experiment: {experiment_number}')

        # Add new experiment if not found before
        for k in parameters:
            performance_df.loc[experiment_number, ('parameters',k)] = parameters[k]
        duplicate =  performance_df.duplicated (subset = [('parameters', k) for k in parameters], keep='first')
        if duplicate.loc[experiment_number]:
            performance_df = performance_df.drop (experiment_number)
            print (f'previously used parameters: {parameters}\nskipping...')
            continue
            
        model_parameters = {k:parameters[k] for k in model_parameters_grid}
        window_parameters = {k:parameters[k] for k in window_parameters_grid}
        time_transform_parameters = {k:parameters[k] for k in time_transform_parameters_grid}

        # run experiment
        # TODO: this can be done more efficiently by 
        # reusing previously calculcated windows if the input_width and target_width don't change
        pipeline = ForecastingPipeline (window_parameters=window_parameters, 
                                        time_transform_parameters=time_transform_parameters, 
                                        model_parameters=model_parameters, 
                                        **pipeline_parameters)
        try:
            performance = pipeline.run_all ()
            print (f'performance: {performance}')
        except SplitError as e:
            print (e)
            performance_df = performance_df.drop (experiment_number)
            continue            
               
        for k in performance:
            for idx_score, score in enumerate(performance[k]):
                metric = metric_names[idx_score]
                performance_df.loc[experiment_number, ('performance', f'{k}_{metric}')] = score
        # other info
        other_info = pipeline.get_debugging_info ()
        for k in other_info:
            performance_df.loc[experiment_number, ('other_info',k)] = other_info[k]
        
        # save checkpoint
        performance_df.to_pickle (path_experiments  / 'performance.pk')
        joblib.dump ([experiment_number, parameters_list], path_experiments  / 'experiment_iteration.pk')

        # save history from this experiment
        history = pipeline.get_history ()
        path_experiment_data = path_experiments / f'experiments/{experiment_number:05d}'
        path_experiment_data.mkdir (parents=True, exist_ok=True)
        joblib.dump (history, path_experiment_data  / 'history.pk')
        # save config parameters
        pipeline.save_config (path_experiment_data)
    
    if debug:
        return performance_df, pipeline
    else:
        return performance_df

# %% ../nbs/hyper_parameters.ipynb 14
def plot_selected (selection, path_experiments=dflt.path_experiments):
    experiment_data = pd.read_pickle (f'{path_experiments}/performance.pk')

    last_experiment, parameters_list = joblib.load (f'{path_experiments}/experiment_iteration.pk')

    for experiment_number in range (last_experiment):
        path_history = Path (f'{path_experiments}/experiments/{experiment_number:05d}/history.pk')
        if path_history.exists():
            parameters = experiment_data.loc[experiment_number]['parameters']
            history = joblib.load (path_history)
            if selection (parameters):
                plt.figure ()
                plt.plot (history['mean_absolute_error'], label='train')
                plt.plot (history['val_mean_absolute_error'],'m', label='val')
                plt.legend (['train','val'])
                plt.title (f'lr: {parameters["learning_rate"]}')

def analysis (path_experiments=dflt.path_experiments):
    plot_selected (lambda parameters: parameters['epochs']>16, path_experiments=path_experiments)
    experiment_data = pd.read_pickle (f'{path_experiments}/performance.pk')
    metric_cols =  [('performance', k) for k in ['validation_mae', 'test_mae']]
    parameters_cols = [('parameters', k) for k in ['epochs','learning_rate','input_width','units']]
    experiment_data = experiment_data[metric_cols + parameters_cols].sort_values (by=metric_cols, ascending=True)
    display (experiment_data)
    return experiment_data

# %% ../nbs/hyper_parameters.ipynb 41
def retrain_and_save_selected_experiment (experiment_number, 
                                          path_experiments=dflt.path_experiments):
    path_experiment_data = Path(path_experiments) / f'experiments/{experiment_number:05d}'
    stored_args = joblib.load (path_experiment_data / 'pipeline_parameters.pk')
    pipeline = ForecastingPipeline (**stored_args)
    pipeline.run_all ()
    
    pipeline.save ('pipeline_object')
