# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/forecasting_pipeline.ipynb.

# %% auto 0
__all__ = ['ForecastingPipeline', 'SplitError', 'time_steps_per_split', 'load_pipeline', 'run_prediction_pipeline']

# %% ../nbs/forecasting_pipeline.ipynb 3
from pathlib import Path
import numpy as np
import pandas as pd
import joblib
from sklearn.preprocessing import StandardScaler
from typing import Dict, Optional, Union, Tuple

from .dataset import UCIRetailDataSet
from .time_transform import make_time_transform, get_time_ft_columns
from .model import Model
from .window_generator import WindowGenerator
from . import defaults as dflt

# %% ../nbs/forecasting_pipeline.ipynb 5
class ForecastingPipeline ():
    r"""Forecasting pipeline with all the steps up to evaluation.
    
    The main method of this class is `run_all`, which loads the data, 
    preprocesses and splits it, fits the model to the training set 
    and evaluates the performance on the other subsets.
    
    Parameters
    ----------
    dataset: str, default 'UCIRetailDataSet'
        Name of the dataset to use. It needs to match one of the dataset 
        classes implemented in the module `datasets`.
    split_ratios: dict, default  {'training':0.7, 'validation':0.15, 'test':0.15}
        Proportion of samples for training, validation, and test. It may 
        contain one, two or three splits. The sum needs to add up to 1.
    time_transform_parameters: dict, default {}
        Parameters used by the TimeTransform component. One of the possible 
        parameters is the type of time transform to use, which can be 
        'CalendarTimeTransform' or 'PeriodicTimeTransform'. Additional 
        parameters depend on the type of TimeTransform used, see the 
        mentioned two classes for reference, in the module `time_transform`.
    window_parameters: dict, default {}
        Parameters used by the `WindowGenerator` component. One of the 
        possible parameters is the type of time transform to use, which can 
        be 'CalendarTimeTransform' or 'PeriodicTimeTransform'. Other 
        parameters to be used depend on the type of TimeTransform used, see 
        the mentioned two classes for reference, in the module 
        `time_transform`.
    model_parameters: dict, default {}
        Parameters used by the `Model` component. One of the possible 
        parameters is the type of model to use, which needs to match one of 
        the model classes implemented in the module `models`. By default, 
        the class of model is `FeedBackWithTimeFeatures`. Additional 
        parameters depend on the type of model used, see the classes in the 
        module `models` for reference.
        
    Other Parameters
    ----------------
    freq_parameters: dict, default {}
        Parameters indicating the time granularity at which the inference 
        is done. This parameter is currently not used, and the granularity 
        is always days ('D').
    
    Raises
    ------
    SplitError
        Raised when the number of windows in a given split (training, 
        validation, or test) is lower than a specified minimum, which is 
        2 by default.
    """
    def __init__ (
        self, 
        dataset: str=dflt.dataset,
        split_ratios: Dict=dflt.split_ratios,
        freq_parameters: Dict={}, # currently not used 
        time_transform_parameters: Dict={},
        window_parameters: Dict={},
        model_parameters: Dict={}
    ):
        # save config for later pickling 
        self._stored_args = dict (dataset=dataset,
                                 split_ratios=split_ratios.copy(),
                                 freq_parameters=freq_parameters.copy(),
                                 time_transform_parameters=time_transform_parameters.copy(),
                                 window_parameters=window_parameters.copy(),
                                 model_parameters=model_parameters.copy())
        
        DataSet = eval (dataset)
        self.dataset = DataSet ()
        if sum(split_ratios.values())!=1:
            raise ValueError ('split_ratios values need to add up to 1')
        if not set(split_ratios.keys()).issubset ({'training', 'validation', 'test'}):
            raise ValueError ('split_ratios keys need to be subset of {"training", "validation", "test"}')
        self.split_ratios = split_ratios
        self._is_fit = False
        # TODO: use freq_parameters to allow granularity different than 'day'
        self.freq_parameters = freq_parameters
        
        # pipeline components
        self.time_transform = make_time_transform (**time_transform_parameters)
        time_ft_columns = get_time_ft_columns (**time_transform_parameters)
        time_ft_columns = window_parameters.pop ('time_ft_columns', time_ft_columns)
        self.window_generator = WindowGenerator (time_ft_columns=time_ft_columns, **window_parameters)
        self.model = Model (out_steps=window_parameters.get ('target_width', dflt.target_width),
                            **model_parameters)
        self.preprocessor = StandardScaler ()
        
    
    def setup (self) -> None:
        r"""Sets up components of the pipeline.
        
        """
        self.model.setup ()
        
    def split (self, df: pd.DataFrame) -> Dict[str, pd.DataFrame]:
        r"""Splits the data in subsets.
        
        The proportion of windows per subset is indicated by the
        `split_ratios` parameter in the pipeline constructor, please refer 
        to it.
        
        
        Parameters
        ----------
        df: DataFrame
            DataFrame to be split into subsets.
        
        Returns
        -------
        dict
            Dictionary of dataframes, one per split.
        """
        time_steps, number_of_windows = time_steps_per_split (df.shape[0], self.window_generator.total_window_size, self.split_ratios)
        df_dict = {}
        last_time_step = 0
        for split in time_steps:
            df_dict[split] = df[last_time_step:last_time_step+time_steps[split]]
            last_time_step += time_steps[split]

        # stored for post-analysis / debugging
        self._time_steps = time_steps
        self._number_of_windows = number_of_windows 
        return df_dict
        
    def prepare_data (
        self, 
        df: pd.DataFrame, 
        split: bool=False, 
        window_generator: Optional[WindowGenerator]=None
    ) -> WindowGenerator:
        r"""Preprocesses and splits the data into subsets. 
        
        Parameters
        ----------
        df: DataFrame
            Input dataFrame to be preprocessed and split.
        split: bool, default False
            If True, the data is split into subsets. If False, a single 
            set named `test` containing all the data is included in the 
            output dictionary.
        window_generator: WindowGenerator or None, default None
            Window generator used to split the data of each subset into 
            windows. This object contains the desired parameters for 
            generating the windows, such as the input width, the target 
            width, and so on, see `WindowGenerator` class. If None is given, 
            the pipeline's window generator is used.
        Returns
        -------
        WindowGenerator
            Window generator set up to generate windows from the computed subsets.
        """
        window_generator = self.window_generator if window_generator is None else window_generator
        df = self.time_transform.transform (df)
        if split:
            df_dict = self.split (df)
        else:
            df_dict = {'test': df}
            
        for k in df_dict:
            if k=='training':
                preprocessed = self.preprocessor.fit_transform (df_dict[k])
            else:
                preprocessed = self.preprocessor.transform (df_dict[k])
            df_dict[k] = pd.DataFrame (preprocessed, columns=df_dict[k].columns, index=df_dict[k].index)
        
        window_generator.setup (df_dict)
        
        return window_generator
    
    def load_data_and_get_windows (self, split: bool=True) -> WindowGenerator:
        r"""Performs all steps from data loading to window generation.
        
        Parameters
        ----------
        split: bool, default False
            If True, the data is split into subsets. If False, a single 
            set named `test` containing all the data is included in the 
            output dictionary.
        
        Returns
        -------
        WindowGenerator
            Window generator set up to generate windows from the computed subsets.
        """
        df = self.dataset.load ()
        df = self.dataset.clean_and_format (df)
        self._total_time_steps = df.shape[0]  # stored for post-analysis / debugging
        windows = self.prepare_data (df, split=split)
        return windows
    
    def fit (self, windows: WindowGenerator) -> None:
        r"""Fits the model to the training set.
        
        Parameters
        ----------
        windows: WindowGenerator
            Training and validation windows used for fitting and monitoring
            the model performance across epochs.
        """
        self._history = self.model.fit (windows)
        self._is_fit = True
    
    def evaluate (self, windows: WindowGenerator) -> Dict[str, dict]:
        r"""Evaluates the model performance. 

        Parameters
        ----------
        windows: WindowGenerator
            Validation and test windows used for evaluating the model
            performance. Either validation, test, or both subsets can
            be included in the windows object.
        
        Returns
        -------
        dict
            Dictionary providing evaluation metrics for each subset. 
            It contains one key per subset (validation and test) used 
            for the evaluation. For each subset, a dictionary is provided
            with the different metric scores used. Current metrics are the
            loss and the mean absolute error.
        """
        return self.model.evaluate (windows)
            
    def run_all (self) -> Dict[str, dict]:
        r"""Runs all the steps from data loading to evaluation.
        
        Loads the data, preprocesses and splits it, fits the model to the 
        training set and evaluates the performance on the other subsets.
            
        Returns
        -------
        dict
            Dictionary providing evaluation metrics for each subset. 
            It contains one key per subset (validation and test) used 
            for the evaluation. For each subset, a dictionary is provided
            with the different metric scores used. Current metrics are the
            loss and the mean absolute error.
        """
        self.setup ()
        windows = self.load_data_and_get_windows (split=True)
        self.fit (windows)
        performance = self.evaluate (windows)
        return performance
        
    def predict_period (
        self, 
        start: Union[str, pd.Timestamp], 
        end: Union[str, pd.Timestamp], 
        prior_data: Optional[pd.DataFrame]=None
    ) -> pd.DataFrame:
        r"""Predicts targets for the given period.

        Parameters
        ----------
        start: string or pd.Timestamp
            Starting date of the period for which we want to do inference.
        end: string or pd.Timestamp
            End date of the period for which we want to do inference.
        prior_data: pd.DataFrame or None, default None
            Data prior to the first time-step (day) to be predicted, used 
            as lagged data by the model. If not given, the last window of data 
            in the training+validation set is used, of size input_size (see 
            WindowGenerator). This is useful for requesting the prediction of 
            any given period, not necessarily the one that is consecutive to the
            training+validation data. If not given, the predictions will start
            from the first time-step right after the one in training+validation, 
            and continue until the end of the requested period, in an 
            auto-regressive fashion. Then, only the part from start to end will
            be returned.
            
        Returns
        -------
        DataFrame
            DataFrame of predictions indexed by date.
        """
        if not self._is_fit:
            raise RuntimeError ('You need to fit the model before calling predict_period')
        if self.window_generator.last_time_step_plus_1 is None:
            raise RuntimeError ('You need to call prepare_data on the training set before calling predict_period')
        windows = self._make_windows_from_period (start, end, prior_data=prior_data)
        result = self.model.predict (windows, single_period=True)
        result = result.loc[start:end]
        
        # invert scaling
        X = np.concatenate([result,np.zeros((result.shape[0],len(self.window_generator.column_indices)-1))], axis=1)
        X = self.preprocessor.inverse_transform (X)
        result = pd.DataFrame (X, 
                               index=result.index,
                               columns=self.window_generator.column_indices.keys())
        result = result[self.window_generator.target_columns]
        return result
        
    
    def predict (self, df: pd.DataFrame) -> pd.DataFrame:
        r"""Predicts targets for the input data frame.
        
        Mostly used for debugging purposes, to compare the result against
        the one obtained by the `predict_period` method.
        
        Parameters
        ----------
        df: DataFrame
            Input data, prior to cleaning and formating.
            
        Returns
        -------
        DataFrame
            DataFrame with prediction windows, each window being of 
            target_width size.
        """
        if not self._is_fit:
            return RuntimeError ('You need to fit the model before calling predict_period')
        df = self.dataset.clean_and_format (df)
        window_generator = WindowGenerator (self.window_generator.input_width, 
                                            self.window_generator.target_width, 
                                            self.window_generator.shift, 
                                            target_columns=self.window_generator.target_columns, 
                                            time_ft_columns=self.window_generator.time_ft_columns, 
                                            batch_size=self.window_generator.batch_size)
        windows = self.prepare_data (df, split=False, window_generator=window_generator)
        df = self.model.predict (windows)
        
        return df
    
    def _get_last_period (
        self, 
        start:pd.Timestamp, 
        prior_data: pd.DataFrame=None
    ) -> pd.DataFrame:
        if prior_data is None:
            return self.window_generator.last_period_df
        if start > prior_data.index[-1]:
            return prior_data.iloc[-self.window_generator.input_width:]
        else:
            prior_data_before_start = prior_data.loc[:pd.Timestamp(start)-pd.offsets.Day()]
            if prior_data_before_start.shape[0] < self.window_generator.input_width:
                raise ValueError (f'prior_data used for picking the last period should have at least {self.window_generator.input_width} time-steps')
            return prior_data_before_start.iloc[-self.window_generator.input_width:]
    
    def _make_windows_from_period (
        self, 
        start: Union[str, pd.Timestamp], 
        end: Union[str, pd.Timestamp], 
        prior_data: Optional[pd.DataFrame]=None
    ) -> WindowGenerator:
        start = pd.Timestamp(start) 
        end = pd.Timestamp(end) 
        if start > end:
            raise ValueError ('start must be lower or equal than end')
        if start < self.window_generator.last_time_step_plus_1:
            raise ValueError (f'start shouuld not be earlier than {self.window_generator.last_time_step_plus_1}')
        last_period_df = self._get_last_period (start, prior_data=prior_data)
        step_after_last_period = last_period_df.index[-1] + pd.offsets.Day()
        index = pd.date_range (step_after_last_period, end=end, freq='D')
        df = pd.DataFrame ({dflt.target_col: np.zeros ((len(index),))}, index=index)
        window_generator = WindowGenerator (self.window_generator.input_width,
                                            df.shape[0], 
                                            df.shape[0], 
                                            target_columns=self.window_generator.target_columns, 
                                            time_ft_columns=self.window_generator.time_ft_columns, 
                                            batch_size=1)
        windows = self.prepare_data (df, split=False, window_generator=window_generator)
        df_plus_warmup = pd.concat ([last_period_df, windows.main_df], axis=0)
        window_generator.setup ({'training': df_plus_warmup})
        return window_generator
    
    def plot (self, windows: WindowGenerator) -> None:
        r"""Plots predictions obtained for example windows.

        Parameters
        ----------
        windows: WindowGenerator
            WindowGenerator containing example windows used for visualization.
        """
        windows.plot (self.model._model)
        
    def get_history (self) -> Dict:
        r"""Gets history of monitored metrics across training epochs.
        
        """
        return self._history.history
    
    def get_debugging_info (self) -> Dict:    
        r"""Gets internally calculated values for debugging purposes.
        
        Returns
        ----------
        dict: 
            Dictionary of containing pipeline internal values of 
            interest.
        """
        return dict (**{f'windows_{split}':self._number_of_windows[split] for split in self._number_of_windows},
                     **{f'steps_{split}':self._time_steps[split] for split in self._time_steps},
                     total_time_steps=self._total_time_steps,
                     total_window_size=self.window_generator.total_window_size)
    
    def save (self, path: Union[str, Path], save_model: bool=True, save_data: bool=True) -> None:
        r"""Saves the pipeline and optionally the model and data subsets, separately.

        Parameters
        ----------
        path: str or Path
            Path to folder where pipeline object, model, and window_geneator are saved.
        save_model: bool, default True
            If True, the model is saved separately in same folder.
        save_data: bool, default True
            If True, the data subsets stored in the window_generator 
            are saved separately in same folder.
        """
        # extract model and window_generator components
        path = Path (path)
        path.mkdir (parents=True, exist_ok=True)
        self.save_config (path)
        if save_model:
            self.model.save_model (path)
            joblib.dump (self.preprocessor, path / 'scaler.pk')
        if save_data:
            self.window_generator.save (path)
        
    def save_config (self, path: Union[str, Path]) -> None:
        r"""Saves the pipeline config parameters.

        Parameters
        ----------
        path: str or Path
            Path to folder where pipeline config parameters are saved.
        """
        path = Path (path)
        path.mkdir (parents=True, exist_ok=True)
        joblib.dump (self._stored_args, path / 'pipeline_parameters.pk')
        
    def load (self, path: Union[str, Path]) -> None:
        r"""Loads the model and data subsets, stored separately.

        Parameters
        ----------
        path: str or Path
            Path to folder where pipeline object, model, and window_geneator are saved.
        """
        path = Path (path)
        self.model.load_model (path)
        self.preprocessor = joblib.load (path / 'scaler.pk')
        self._is_fit = True
        self.window_generator = joblib.load (path / 'window_generator.pk')
        self.window_generator.load (path)

# %% ../nbs/forecasting_pipeline.ipynb 7
class SplitError (Exception):
    r"""Error when splitting data into subsets. 
    
    Raised when the number of windows in a given split (training, 
    validation, or test) is lower than a specified minimum, which is 
    2 by default.
    """
    pass

def time_steps_per_split (
    total_time_steps: int,
    total_window_size: int, 
    split_ratios: Dict[str, float], 
    min_number_windows: int=2, 
    show: bool=False
) -> Tuple:
    r"""Calculates time-steps per split based on ratios in `split_ratios`.
    
    Ensures that the proportion of windows in each split is close to the 
    value indicated in `split_ratios`. Note that the 
    number of time-steps per split does not need to follow the proportions 
    indicated in `split_ratios`, but the number of windows, since it is 
    the latter which determines how many samples the model is trained,
    validated, and tested on.
    
    Parameters
    ----------
    total_time_steps: int
        Total number of time-steps in the time-series dataset.
    total_window_size: int
        Size of the windows to be generated from the time-series.
        See `WindowGenerator` class in `window_generator` module.
    split_ratios: dict
        Proportion of samples for training, validation, and test. It may 
        contain one, two or three splits. The sum needs to add up to 1.
    min_number_windows: int, default 2
        Minimum allowable number of windows in each split. When the 
        number is lower than this minimum, a SplitError exception is 
        raised.        
    show: bool, default False
        If True, the actual proportions of windows per split are 
        compared against the ones indicated in `split_ratios`, and 
        the difference is shown on console.
        
    Raises
    ------
    SplitError
        Raised when the number of windows in a given split (training, 
        validation, or test) is lower than a specified minimum, which is 
        2 by default.
    """

    number_splits = len(split_ratios)
    time_steps = {}
    number_of_windows = {}
    for split in split_ratios:
        ratio = split_ratios[split]
        # the following ensures that the number of windows in this split 
        # over the total number of windows is close to ratio:
        time_steps[split] = ratio * (total_time_steps-number_splits*total_window_size+number_splits)+total_window_size-1
        number_of_windows[split] = time_steps[split]-total_window_size+1
        if number_of_windows[split] < min_number_windows:
            raise SplitError (f'number of windows {number_of_windows[split]} for split {split} is too small')
    
    if show:
        total_number_of_windows = sum (number_of_windows.values())
        for split in split_ratios:
            actual_ratio = number_of_windows[split] / total_number_of_windows
            desired_ratio = split_ratios[split]
            print (abs(actual_ratio-desired_ratio))
        
        calculated_total_time_steps = total_number_of_windows+number_splits*(total_window_size-1)
        print (abs(calculated_total_time_steps-total_time_steps))
    
    # round numbers 
    cumulative_time_steps = 0
    for i, split in enumerate(split_ratios):
        if i < len(split_ratios)-1:
            time_steps[split] = int(round(time_steps[split]))
            cumulative_time_steps += time_steps[split]
        else:
            # last split receives remaining time steps
            time_steps[split] = total_time_steps - cumulative_time_steps
        
    return time_steps, number_of_windows

# %% ../nbs/forecasting_pipeline.ipynb 9
def load_pipeline (path: Union[str, Path]) -> ForecastingPipeline:
    r"""Loads a forecasting pipeline from disk.

    Parameters
    ----------
    path: str or Path
        Path to folder where pipeline object is saved.
    
    Returns
    -------
    ForecastingPipeline
    """
    path = Path (path)
    # pipeline = joblib.load (path / 'pipeline.pk') # doesnt' work
    stored_args = joblib.load (path / 'pipeline_parameters.pk')
    pipeline = ForecastingPipeline (**stored_args)
    pipeline.load (path)
    return pipeline

# %% ../nbs/forecasting_pipeline.ipynb 11
def run_prediction_pipeline (
    pipeline_path: str='pipeline_object',
    start: str='2012-01-01', 
    end: str='2012-12-31',
    result_file='predictions.parquet'
) -> None:
    r"""Load previously stored pipeline and performs prediction.

    Parameters
    ----------
    path: str, default 'pipeline_object'
        Path to folder where pipeline object is saved.
    start: str, default '2012-01-01'
        Starting date of the period for which we want to do inference.
    end: str, default '2012-12-12'
        End date of the period for which we want to do inference.
    result_file: str, default 'predictions.parquet'
        File where result is stored.
    """
    pipeline = load_pipeline (pipeline_path)
    predictions = pipeline.predict_period (start=start, end=end)
    predictions.to_parquet (result_file)
