# AUTOGENERATED! DO NOT EDIT! File to edit: 00_core.ipynb (unless otherwise specified).

__all__ = ['symbol_map', 'get_ohlcv_from_db', 'from_db_load_df_for', 'load_binance_data', 'load_both_datasources',
           'compare_timeseries', 'previous_local_data_exists', 'load_data_keep_best', 'add_BTC_price_as_column',
           'build_segmented_parts', 'check_for_RSI_buys']

# Cell
import os
import psycopg2
import ujson
import numpy as np
import pandas as pd

# Cell
def symbol_map():
    """
    Obtains the dictionary {symbol : id} to perform queries
    directly on symbols.

    returns:: mapping, a dict containing {'symbol' : id }
    """
    # Prepare query
    symbols_ids_query = """/* script to retrieve OCHLV candles from Binance and whitelist */
    select "id", "name"
    from symbols"""

    # Try querying
    try:
        connection = psycopg2.connect(user = user,
                                  password = password,
                                  host = host,
                                  port = "5432",
                                  database = db_name)
        cursor = connection.cursor()
        print(f"Retrieving symbol_id map from DB...")
        cursor.execute(symbols_ids_query)
        records = cursor.fetchall()
        print("Done")

        # Converting to a usable dict
        mapping = dict(records)
        mapping = {v: k for k, v in mapping.items()}

    except (Exception, psycopg2.Error)  as error:
        print("Error while fetching data from PostgreSQl", error)

    finally:
        # Close database connection (let's keep things tidy)
        if(connection):
            cursor.close()
            connection.close()
            print("PostgreSQl connection is closed")


    return mapping

# Cell
def get_ohlcv_from_db(target_coin = '*'):
    """
    Returns the "raw" list of tuples (O,H,L,C,V) from the PostgreSQL database.

    :target_coin: either str of the symbol of a coin, or an int with a symbol_id from db
    :return: list of tuples ("timestamp", "open", "high", "low", "close", "volume")
    """
    # Prepare query
    binance_coin_query = """/* script to retrieve OCHLV candles from Binance and whitelist */
    select "timestamp" ,"open" , "close" , high , low , volume
    from historical_ochlv
    where platform_id =1
    and symbol_id = """

    symbol_mapy = symbol_map()

    # Fine-tune query for target_coin
    if isinstance(target_coin, str):
        if '/' not in target_coin:
            print(f"Quote currency not specified, assuming BTC: {target_coin}/BTC")
            target_coin += "/BTC"
        if not symbol_mapy:
            symbol_mapy = symbol_map()
        coin_id = symbol_mapy[f"{target_coin}"]
        binance_coin_query += str(coin_id)
    elif isinstance(target_coin, int):
        binance_coin_query += str(target_coin)

    # Try querying
    try:
        connection = psycopg2.connect(user = user,
                                  password = password,
                                  host = host,
                                  port = "5432",
                                  database = db_name)
        cursor = connection.cursor()
        print(f"Retrieving {target_coin} with ID {coin_id } from DB... Could take 2 minutes")
        cursor.execute(binance_coin_query)
        records = cursor.fetchall()
        print("Query done")

    except (Exception, psycopg2.Error)  as error:
        print("Error while fetching data from PostgreSQl", error)

    finally:
        # Close database connection (let's keep things tidy)
        if(connection):
            cursor.close()
            connection.close()
            print("PostgreSQl connection is closed")

    return records

# Cell
def from_db_load_df_for(target_coin):
    """
    Fancy parser for the SQL records to dataframe,
    with columns ("timestamp", "open", "high", "low", "close", "volume")

    :target_coin: either symbol of a known coin as str ("ETH"), or an int with a symbol_id from db (308)
    :return: dataframe with columns ["timestamp", "open", "high", "low", "close", "volume"]
    """
    # Calls db
    records = get_ohlcv_from_db(target_coin)

    # To DF
    df_labels = ["timestamp", "open", "high", "low", "close", "volume"]
    df = pd.DataFrame.from_records(records, columns = df_labels)
    df.astype('float64').dtypes
    df.timestamp = pd.to_datetime(df['timestamp'],
                                       unit='ms',
                                       utc=True,
                                       infer_datetime_format=True)
    # Sorting by timestamp and fixing index
    df = df.sort_values(by ='timestamp', ascending = True)
    df = df.reset_index(drop=True)

    return df

# Cell
def load_binance_data(target_coin: str ):
    """
    Internal method used to load data for one pair from disk.
    Returns a Pandas dataframe.
    :target_coin: symbol of the coin. Must be among those on the data folder

    :return: DataFrame with ohlcv data, or empty DataFrame
    """
    filename = './gdrive/My Drive/binance_longer_2000_days/'+ target_coin + '_BTC-1m.json'
    filename_local = './binance_longer_2000_days/'+ target_coin + '_BTC-1m.json'
    USDT_path = './binance_USDT/BTC_USDT-1d.json'

    print('  searching in '+ filename)

    if target_coin == 'USDT':
        # BTC price
        filename = USDT_path
        print("  USDT/BTC path")
    else:
        if os.path.isfile(filename):
            print('  data file exists')
        elif os.path.isfile(filename_local):
            filename = filename_local
            print('  local path used instead')
        else:
            print(f'  data file does not exist. Locate it or try with the DB instead, with one-liner:\ndf = from_db_load_df_for({target_coin})')

    print(filename)
    with open(filename) as f:
        pairdata = ujson.load(f)
    print("  loaded")

    df_labels = ["timestamp", "open", "high", "low", "close", "volume"]
    df = pd.DataFrame.from_records(pairdata, columns = df_labels)

    df.astype('float64').dtypes
    df.timestamp = pd.to_datetime(df['timestamp'],
                                   unit='ms',
                                   utc=True,
                                   infer_datetime_format=True)
    print("    data len for "+str(target_coin)+ ": " + str(len(df)))

    return df

# Cell
def load_both_datasources(target_coin):
    """
    One-liner to load data from two data sources

    :return: two OHCLV dataframes (from binance and db) with columns
          ["timestamp", "open", "high", "low", "close", "volume"]
    """

    # Binance local data (repo included)
    try:
        df_from_binance = load_binance_data(target_coin)
        print("Binance data source succesfully loaded.\n")
    except:
        print("Couldn't load data from local Binance data.")
        df_from_binance = None

    # Database data
    try:
        df_from_db = from_db_load_df_for(target_coin)
        print("Database data source succesfully loaded.\n")
    except:
        print(f"Couldn't load data from {db_name} database")
        df_from_db = None

    return df_from_binance, df_from_db

# Cell
def compare_timeseries(binance, database):
    """
    Plots two timeseries from different datasources
    to explore the difference in length and their similarity,
    scaling the X-axis to the longest one.
    """
    try:
        # We get the longest timeframe to plot on the same scale
        if binance.timestamp.iloc[0] < database.timestamp.iloc[0]:
            earlier = binance.timestamp.iloc[0]
        else:
            earlier = database.timestamp.iloc[0]
        if binance.timestamp.iloc[-1] < database.timestamp.iloc[-1]:
            later = database.timestamp.iloc[-1]
        else:
            later = binance.timestamp.iloc[-1]

        # Plot to compare both timeseries
        fig, axes = plt.subplots(1, 1, figsize=(10,5))
        axes.plot(binance.timestamp, binance.open, label = "From Binance / ft")
        axes.plot(database.timestamp, database.open, label = "From our DB")
        # axes.grid()
        # axes.legend()
        # axes.set_xlim(left = earlier, right = later)
        # axes.set_title("Comparison of data sources")
        # plt.show()
    except:
        print("An error occured. Make sure right arguments were provided, or make sure that arrays 'df_from_binance' and 'df_from_db' exist. Check the Jupyter notebook for guidance.")

# Cell
def previous_local_data_exists(target_coin):
    """
    Checks whether previously exported data exists and returns it.

    Data is exported only by load_data_keep_best() for now!! Work in progress...
    """
    path = f"./exported_best_binance_coin_{target_coin}.csv"
    if os.path.isfile(path):
        print("Previous datafile exist")
        previous_exists = True
    else:
        path = f"./exported_best_database_coin_{target_coin}.csv"
        print ("Checking from database source")
        if os.path.isfile(path):
            print("Previous datafile file exist")
            previous_exists = True
        else:
            print("Previous datafile does not exist")
            previous_exists = False

    return previous_exists, path

# Cell
def load_data_keep_best(target_coin):
    """
    One-liner to load data from two data sources,
    compare them, plot a comparison,
    and return longest of them

    :return: best OHCLV dataframe (the with the most info) with columns
          ["timestamp", "open", "high", "low", "close", "volume"]
    """

    previous_data_exists, path = previous_local_data_exists(target_coin)
    if previous_data_exists:
        print(f"Loading past local data from {path}")
        df = pd.read_csv(path, index_col=0)
        best_data_source = path.split('_')[-3]
    else:
        # Binance local data (repo included)
        try:
            df_from_binance = load_binance_data(target_coin)
            print("Binance data source succesfully loaded.\n")
        except:
            print("Couldn't load data from local Binance data.")
            df_from_binance = None

        # Database data
        try:
            df_from_db = from_db_load_df_for(target_coin)
            print("Database data source succesfully loaded.\n")
        except:
            print(f"Couldn't load data from {db_name} database")
            df_from_db = None

        # keeps and returns best (longest)
        if df_from_binance is not None:
            if len(df_from_db) < len(df_from_binance):
                print("Keeping longest DataFrame: from binance")
                best_data_source = "binance"
                df = df_from_binance
            else:
                print("Keeping longest DataFrame: from database")
                best_data_source = "database"
                df = df_from_db
        else:
            print("Keeping longest DataFrame: from database")
            best_data_source = "database"
            df = df_from_db

        # export to file
        df.to_csv(f"./exported_best_{best_data_source}_coin_{target_coin}.csv")

        # plot comparison
       # try:
        #    compare_timeseries(df_from_binance, df_from_db)
       # except:
        #    print("Could not plot two data sources")

    return df, best_data_source

# Cell
def add_BTC_price_as_column(df, best_data_source):
    """
    This one should be rewritten from sracth:
    - Using the method commented below (df.between_time)
    - Ensuring better handling of local data absence (coherence with export/import)
    """
    # BTC price data from chosen data_source (USDT/BTC, 1d candle)
    if best_data_source == "binance":
        USDT = load_binance_data('USDT')
    else:
        USDT = from_db_load_df_for("BTC/USDT")

    # We ensure USDT and target_coin data
    #   comprise the same periods and trim
    #   the non-common dates

    # NOTE: we probably should use pd.Dataframe.between_time:
    #   https://pandas.pydata.org/pandas-docs/stable/reference/api/pandas.DataFrame.between_time.html

    # trim the past
    if pd.to_datetime(USDT.timestamp.iloc[0]) < pd.to_datetime(df.timestamp.iloc[0]):
        # USDT is earlier
        print('USDT reaches earlier, trimming USDT')
        earl_common = pd.to_datetime(df.timestamp.iloc[0])
        USDT = USDT.where(pd.to_datetime(USDT.timestamp) >= earl_common)
        USDT = USDT.dropna()
    else:
        # target_coin is earlier
        print('target_coin reaches earlier, trimming target_coin')
        earl_common = pd.to_datetime(USDT.timestamp.iloc[0])
        df = df.where(pd.to_datetime(df.timestamp) >= earl_common)
        df = df.dropna()

    # trim the future
    if pd.to_datetime(USDT.timestamp.iloc[-1]) < pd.to_datetime(df.timestamp.iloc[-1]):
        # USDT is earlier
        print('target_coin reaches later, trimming target_coin')
        earl_common = pd.to_datetime(USDT.timestamp.iloc[-1])
        df = df.where(pd.to_datetime(df.timestamp) <= earl_common)
        df = df.dropna()
    else:
        # target_coin is earlier
        print('USDT reaches later, trimming USDT')
        earl_common = pd.to_datetime(df.timestamp.iloc[-1])
        USDT = USDT.where(pd.to_datetime(USDT.timestamp) <= earl_common)
        USDT = USDT.dropna()

    # Add BTC price into single dataframe
    mapy = dict(zip(pd.to_datetime(USDT.timestamp).dt.date, USDT.open))
    # We add column price, mapping the dates from both DataFrames
    df['BTC_price'] = pd.to_datetime(df.timestamp).dt.date.map(mapy)
    df = df.dropna()

    return df

# Cell
def build_segmented_parts(i, rsis, iteration_step, indicator):
    """
    Prepares array segments to plot
    in every frame of the animation.

    It gets tricky because we have to split the entire
    dataset into parts to see intervals. So for every
    quantity of points to be plotted, we need to check
    how many "full segments" are part of the next plot,
    and how many points of a "partial segment" need
    to be plotted

    i :: implicit matplotlib's frame number (undeclared here)
    rsis :: dataset split into a list of dataframes

    returns :: 3 arrays to be plot and 1 array for timeseries axis
    """

    quantity_of_points_to_load = i * iteration_step + 1 # (1440 min per day)

    # Target segment
    n = 0
    len_loaded = len(rsis[n])
    while quantity_of_points_to_load > len_loaded:
        len_loaded += len(rsis[n])
        n += 1
    n_segment = n

    # Initialize the lists of lists
    array = []
    timestamps_array = []
    prices_array = []
    asset_prices = []

    # We loop over the needed data (1440 min = 1 day)
    n = 0
    len_array = 0
    while n < n_segment:
        part = rsis[n]

        # we add the previous whole segments
        array.append(part[indicator])
        timestamps_array.append(part.timestamp)
        prices_array.append(part.BTC_price)
        asset_prices.append(part.open)

        len_array += len(part)
        n += 1
    else:
        part = rsis[n]
        # we add remaining elements for current segment
        elements_yet_to_add = quantity_of_points_to_load - len_array

        # we add parts
        array.append(part[:elements_yet_to_add][indicator])
        timestamps_array.append(part[:elements_yet_to_add].timestamp)
        prices_array.append(part[:elements_yet_to_add].BTC_price)
        asset_prices.append(part[:elements_yet_to_add].open)

        len_array += elements_yet_to_add

    return array, prices_array, timestamps_array, asset_prices

# Cell
def check_for_RSI_buys(df, rsi_buy):
    """
    Uses a temporal, 1-timestamp-shifted RSI column
    that allows comparing two rows using the very performant
    same-row-logic, and then removes it.
    """
    df['RSI_shifted_one_day'] = df.RSI_indicator.shift(1)
    def search_buy_signals(row):
        """
        Map function
        """
        if (row.RSI_indicator > rsi_buy and  row.RSI_shifted_one_day < rsi_buy):
            return 1
        else:
            return 0
    # map function and using .apply is vector friendly
    df[f'buy_trigger_on_{rsi_buy}'] = df.apply(search_buy_signals, axis = 1)
    del df['RSI_shifted_one_day']

    return df