"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const secretsmanager = require("../lib");
module.exports = {
    'secret rotation single user'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const secret = new secretsmanager.Secret(stack, 'Secret');
        const target = new ec2.Connections({
            defaultPort: ec2.Port.tcp(3306),
            securityGroups: [new ec2.SecurityGroup(stack, 'SecurityGroup', { vpc })]
        });
        // WHEN
        new secretsmanager.SecretRotation(stack, 'SecretRotation', {
            application: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            secret,
            target,
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            Description: 'from SecretRotationSecurityGroupAEC520AB:3306',
            FromPort: 3306,
            GroupId: {
                'Fn::GetAtt': [
                    'SecurityGroupDD263621',
                    'GroupId'
                ]
            },
            SourceSecurityGroupId: {
                'Fn::GetAtt': [
                    'SecretRotationSecurityGroup9985012B',
                    'GroupId'
                ]
            },
            ToPort: 3306
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::RotationSchedule', {
            SecretId: {
                Ref: 'SecretA720EF05'
            },
            RotationLambdaARN: {
                'Fn::GetAtt': [
                    'SecretRotationA9FFCFA9',
                    'Outputs.RotationLambdaARN'
                ]
            },
            RotationRules: {
                AutomaticallyAfterDays: 30
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            GroupDescription: 'SecretRotation/SecurityGroup'
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Serverless::Application', {
            Location: {
                ApplicationId: 'arn:aws:serverlessrepo:us-east-1:297356227824:applications/SecretsManagerRDSMySQLRotationSingleUser',
                SemanticVersion: '1.1.3'
            },
            Parameters: {
                endpoint: {
                    'Fn::Join': [
                        '',
                        [
                            'https://secretsmanager.',
                            {
                                Ref: 'AWS::Region'
                            },
                            '.',
                            {
                                Ref: 'AWS::URLSuffix'
                            }
                        ]
                    ]
                },
                functionName: 'SecretRotation',
                vpcSecurityGroupIds: {
                    'Fn::GetAtt': [
                        'SecretRotationSecurityGroup9985012B',
                        'GroupId'
                    ]
                },
                vpcSubnetIds: {
                    'Fn::Join': [
                        '',
                        [
                            {
                                Ref: 'VPCPrivateSubnet1Subnet8BCA10E0'
                            },
                            ',',
                            {
                                Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A'
                            },
                        ]
                    ]
                }
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::ResourcePolicy', {
            ResourcePolicy: {
                Statement: [
                    {
                        Action: 'secretsmanager:DeleteSecret',
                        Effect: 'Deny',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition'
                                        },
                                        ':iam::',
                                        {
                                            Ref: 'AWS::AccountId'
                                        },
                                        ':root'
                                    ]
                                ]
                            }
                        },
                        Resource: '*'
                    }
                ],
                Version: '2012-10-17'
            },
            SecretId: {
                Ref: 'SecretA720EF05'
            }
        }));
        test.done();
    },
    'secret rotation multi user'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const secret = new secretsmanager.Secret(stack, 'Secret');
        const masterSecret = new secretsmanager.Secret(stack, 'MasterSecret');
        const target = new ec2.Connections({
            defaultPort: ec2.Port.tcp(3306),
            securityGroups: [new ec2.SecurityGroup(stack, 'SecurityGroup', { vpc })]
        });
        // WHEN
        new secretsmanager.SecretRotation(stack, 'SecretRotation', {
            application: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER,
            secret,
            masterSecret,
            target,
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Serverless::Application', {
            Parameters: {
                endpoint: {
                    'Fn::Join': [
                        '',
                        [
                            'https://secretsmanager.',
                            {
                                Ref: 'AWS::Region'
                            },
                            '.',
                            {
                                Ref: 'AWS::URLSuffix'
                            }
                        ]
                    ]
                },
                functionName: 'SecretRotation',
                vpcSecurityGroupIds: {
                    'Fn::GetAtt': [
                        'SecretRotationSecurityGroup9985012B',
                        'GroupId'
                    ]
                },
                vpcSubnetIds: {
                    'Fn::Join': [
                        '',
                        [
                            {
                                Ref: 'VPCPrivateSubnet1Subnet8BCA10E0'
                            },
                            ',',
                            {
                                Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A'
                            },
                        ]
                    ]
                },
                masterSecretArn: {
                    Ref: 'MasterSecretA11BF785'
                }
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::ResourcePolicy', {
            ResourcePolicy: {
                Statement: [
                    {
                        Action: 'secretsmanager:DeleteSecret',
                        Effect: 'Deny',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition'
                                        },
                                        ':iam::',
                                        {
                                            Ref: 'AWS::AccountId'
                                        },
                                        ':root'
                                    ]
                                ]
                            }
                        },
                        Resource: '*'
                    }
                ],
                Version: '2012-10-17'
            },
            SecretId: {
                Ref: 'MasterSecretA11BF785'
            }
        }));
        test.done();
    },
    'throws when connections object has no default port range'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const secret = new secretsmanager.Secret(stack, 'Secret');
        const securityGroup = new ec2.SecurityGroup(stack, 'SecurityGroup', {
            vpc,
        });
        // WHEN
        const target = new ec2.Connections({
            securityGroups: [securityGroup]
        });
        // THEN
        test.throws(() => new secretsmanager.SecretRotation(stack, 'Rotation', {
            secret,
            application: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            vpc,
            target
        }), /`target`.+default port range/);
        test.done();
    },
    'throws when master secret is missing for a multi user application'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const secret = new secretsmanager.Secret(stack, 'Secret');
        const target = new ec2.Connections({
            defaultPort: ec2.Port.tcp(3306),
            securityGroups: [new ec2.SecurityGroup(stack, 'SecurityGroup', { vpc })]
        });
        // THEN
        test.throws(() => new secretsmanager.SecretRotation(stack, 'Rotation', {
            secret,
            application: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER,
            vpc,
            target
        }), /The `masterSecret` must be specified for application using the multi user scheme/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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