"""csv plugin for translating Financial Information eXchange (FIX) messages.

https://github.com/markuskimius/csvmagic
"""

from csvmagic.libcsv import Cell

__copyright__ = "Copyright 2018-2022 Mark Kim"
__license__ = "Apache 2.0"
__version__ = '2.0.0'
__author__ = "Mark Kim"


class Translator(object):
    def __init__(self):
        self.__fixversion = None
        self.__has_header = False

    def __call__(self, cell):
        tagnum = cell.colname()
        value = cell.value()

        # Are we expected to extract our own header or provide them?
        if tagnum is not None:
            self.__has_header = True

        # Extract the header from the value if required
        if not self.__has_header and value.find('=') >= 0:
            tagnum, value = value.split("=", 1)

        # Save the FIX version for more precise translation
        if tagnum == '8':
            self.__fixversion = value

        # Translate the tag name and value
        tagname = self.__get_tag(tagnum, value)
        value = self.__get_value(tagnum, value)

        # We don't want tagname to be None because if it gets set to None then the
        # read command displays the column number which looks confusing because it
        # looks like a tag number but they're not the same thing!
        if tagname is None: tagname = ''

        return Cell(cell.rownum(), tagname, cell.colnum(), value)

    def __get_tag(self, tagnum, value):
        tagname = Translator.__TAG_MAP.get((self.__fixversion, tagnum),
                  Translator.__TAG_MAP.get(("FIX.5.0", tagnum),
                  Translator.__TAG_MAP.get(("FIX.4.4", tagnum),
                  Translator.__TAG_MAP.get(("FIX.4.2", tagnum),
                  Translator.__TAG_MAP.get(("FIX.4.0", tagnum), tagnum)))))

        return tagname

    def __get_value(self, tagnum, value):
        if(tagnum in ("30", "100", "207")):
            value = Translator.__MARKET_MAP.get(("MIC", value),
                    Translator.__MARKET_MAP.get(("RIC", value), value))
        else:
            value = Translator.__VALUE_MAP.get((self.__fixversion, tagnum, value),
                    Translator.__VALUE_MAP.get(("FIX.5.0", tagnum, value),
                    Translator.__VALUE_MAP.get(("FIX.4.4", tagnum, value),
                    Translator.__VALUE_MAP.get(("FIX.4.2", tagnum, value),
                    Translator.__VALUE_MAP.get(("FIX.4.0", tagnum, value), value)))))

        return value

    __TAG_MAP = {
        ("FIX.4.0", "1"): "1    Account",
        ("FIX.4.0", "2"): "2    AdvId",
        ("FIX.4.0", "3"): "3    AdvRefID",
        ("FIX.4.0", "4"): "4    AdvSide",
        ("FIX.4.0", "5"): "5    AdvTransType",
        ("FIX.4.0", "6"): "6    AvgPx",
        ("FIX.4.0", "7"): "7    BeginSeqNo",
        ("FIX.4.0", "8"): "8    BeginString",
        ("FIX.4.0", "9"): "9    BodyLength",
        ("FIX.4.0", "10"): "10   CheckSum",
        ("FIX.4.0", "11"): "11   ClOrdID",
        ("FIX.4.0", "12"): "12   Commission",
        ("FIX.4.0", "13"): "13   CommType",
        ("FIX.4.0", "14"): "14   CumQty",
        ("FIX.4.0", "15"): "15   Currency",
        ("FIX.4.0", "16"): "16   EndSeqNo",
        ("FIX.4.0", "17"): "17   ExecID",
        ("FIX.4.0", "18"): "18   ExecInst",
        ("FIX.4.0", "19"): "19   ExecRefID",
        ("FIX.4.0", "20"): "20   ExecTransType",
        ("FIX.4.0", "21"): "21   HandlInst",
        ("FIX.4.0", "22"): "22   IDSource",
        ("FIX.4.0", "23"): "23   IOIid",
        ("FIX.4.0", "24"): "24   IOIOthSvc",
        ("FIX.4.0", "25"): "25   IOIQltyInd",
        ("FIX.4.0", "26"): "26   IOIRefID",
        ("FIX.4.0", "27"): "27   IOIShares",
        ("FIX.4.0", "28"): "28   IOITransType",
        ("FIX.4.0", "29"): "29   LastCapacity",
        ("FIX.4.0", "30"): "30   LastMkt",
        ("FIX.4.0", "31"): "31   LastPx",
        ("FIX.4.0", "32"): "32   LastShares",
        ("FIX.4.0", "33"): "33   LinesOfText",
        ("FIX.4.0", "34"): "34   MsgSeqNum",
        ("FIX.4.0", "35"): "35   MsgType",
        ("FIX.4.0", "36"): "36   NewSeqNo",
        ("FIX.4.0", "37"): "37   OrderID",
        ("FIX.4.0", "38"): "38   OrderQty",
        ("FIX.4.0", "39"): "39   OrdStatus",
        ("FIX.4.0", "40"): "40   OrdType",
        ("FIX.4.0", "41"): "41   OrigClOrdID",
        ("FIX.4.0", "42"): "42   OrigTime",
        ("FIX.4.0", "43"): "43   PossDupFlag",
        ("FIX.4.0", "44"): "44   Price",
        ("FIX.4.0", "45"): "45   RefSeqNum",
        ("FIX.4.0", "46"): "46   RelatdSym",
        ("FIX.4.0", "47"): "47   Rule80A",
        ("FIX.4.0", "48"): "48   SecurityID",
        ("FIX.4.0", "49"): "49   SenderCompID",
        ("FIX.4.0", "50"): "50   SenderSubID",
        ("FIX.4.0", "52"): "52   SendingTime",
        ("FIX.4.0", "53"): "53   Shares",
        ("FIX.4.0", "54"): "54   Side",
        ("FIX.4.0", "55"): "55   Symbol",
        ("FIX.4.0", "56"): "56   TargetCompID",
        ("FIX.4.0", "57"): "57   TargetSubID",
        ("FIX.4.0", "58"): "58   Text",
        ("FIX.4.0", "59"): "59   TimeInForce",
        ("FIX.4.0", "60"): "60   TransactTime",
        ("FIX.4.0", "61"): "61   Urgency",
        ("FIX.4.0", "62"): "62   ValidUntilTime",
        ("FIX.4.0", "63"): "63   SettlmntTyp",
        ("FIX.4.0", "64"): "64   FutSettDate",
        ("FIX.4.0", "65"): "65   SymbolSfx",
        ("FIX.4.0", "66"): "66   ListID",
        ("FIX.4.0", "67"): "67   ListSeqNo",
        ("FIX.4.0", "68"): "68   ListNoOrds",
        ("FIX.4.0", "69"): "69   ListExecInst",
        ("FIX.4.0", "70"): "70   AllocID",
        ("FIX.4.0", "71"): "71   AllocTransType",
        ("FIX.4.0", "72"): "72   RefAllocID",
        ("FIX.4.0", "73"): "73   NoOrders",
        ("FIX.4.0", "74"): "74   AvgPrxPrecision",
        ("FIX.4.0", "75"): "75   TradeDate",
        ("FIX.4.0", "76"): "76   ExecBroker",
        ("FIX.4.0", "77"): "77   OpenClose",
        ("FIX.4.0", "78"): "78   NoAllocs",
        ("FIX.4.0", "79"): "79   AllocAccount",
        ("FIX.4.0", "80"): "80   AllocShares",
        ("FIX.4.0", "81"): "81   ProcessCode",
        ("FIX.4.0", "82"): "82   NoRpts",
        ("FIX.4.0", "83"): "83   RptSeq",
        ("FIX.4.0", "84"): "84   CxlQty",
        ("FIX.4.0", "85"): "85   NoDlvyInst",
        ("FIX.4.0", "86"): "86   DlvyInst",
        ("FIX.4.0", "87"): "87   AllocStatus",
        ("FIX.4.0", "88"): "88   AllocRejCode",
        ("FIX.4.0", "89"): "89   Signature",
        ("FIX.4.0", "90"): "90   SecureDataLen",
        ("FIX.4.0", "91"): "91   SecureData",
        ("FIX.4.0", "92"): "92   BrokerOfCredit",
        ("FIX.4.0", "93"): "93   SignatureLength",
        ("FIX.4.0", "94"): "94   EmailType",
        ("FIX.4.0", "95"): "95   RawDataLength",
        ("FIX.4.0", "96"): "96   RawData",
        ("FIX.4.0", "97"): "97   PossResend",
        ("FIX.4.0", "98"): "98   EncryptMethod",
        ("FIX.4.0", "99"): "99   StopPx",
        ("FIX.4.0", "100"): "100  ExDestination",
        ("FIX.4.0", "102"): "102  CxlRejReason",
        ("FIX.4.0", "103"): "103  OrdRejReason",
        ("FIX.4.0", "104"): "104  IOIQualifier",
        ("FIX.4.0", "105"): "105  WaveNo",
        ("FIX.4.0", "106"): "106  Issuer",
        ("FIX.4.0", "107"): "107  SecurityDesc",
        ("FIX.4.0", "108"): "108  HeartBtInt",
        ("FIX.4.0", "109"): "109  ClientID",
        ("FIX.4.0", "110"): "110  MinQty",
        ("FIX.4.0", "111"): "111  MaxFloor",
        ("FIX.4.0", "112"): "112  TestReqID",
        ("FIX.4.0", "113"): "113  ReportToExch",
        ("FIX.4.0", "114"): "114  LocateReqd",
        ("FIX.4.0", "115"): "115  OnBehalfOfCompID",
        ("FIX.4.0", "116"): "116  OnBehalfOfSubID",
        ("FIX.4.0", "117"): "117  QuoteID",
        ("FIX.4.0", "118"): "118  NetMoney",
        ("FIX.4.0", "119"): "119  SettlCurrAmt",
        ("FIX.4.0", "120"): "120  SettlCurrency",
        ("FIX.4.0", "121"): "121  ForexReq",
        ("FIX.4.0", "122"): "122  OrigSendingTime",
        ("FIX.4.0", "123"): "123  GapFillFlag",
        ("FIX.4.0", "124"): "124  NoExecs",
        ("FIX.4.0", "125"): "125  CxlType",
        ("FIX.4.0", "126"): "126  ExpireTime",
        ("FIX.4.0", "127"): "127  DKReason",
        ("FIX.4.0", "128"): "128  DeliverToCompID",
        ("FIX.4.0", "129"): "129  DeliverToSubID",
        ("FIX.4.0", "130"): "130  IOINaturalFlag",
        ("FIX.4.0", "131"): "131  QuoteReqID",
        ("FIX.4.0", "132"): "132  BidPx",
        ("FIX.4.0", "133"): "133  OfferPx",
        ("FIX.4.0", "134"): "134  BidSize",
        ("FIX.4.0", "135"): "135  OfferSize",
        ("FIX.4.0", "136"): "136  NoMiscFees",
        ("FIX.4.0", "137"): "137  MiscFeeAmt",
        ("FIX.4.0", "138"): "138  MiscFeeCurr",
        ("FIX.4.0", "139"): "139  MiscFeeType",
        ("FIX.4.0", "140"): "140  PrevClosePx",
        ("FIX.4.2", "1"): "1    Account",
        ("FIX.4.2", "2"): "2    AdvId",
        ("FIX.4.2", "3"): "3    AdvRefID",
        ("FIX.4.2", "4"): "4    AdvSide",
        ("FIX.4.2", "5"): "5    AdvTransType",
        ("FIX.4.2", "6"): "6    AvgPx",
        ("FIX.4.2", "7"): "7    BeginSeqNo",
        ("FIX.4.2", "8"): "8    BeginString",
        ("FIX.4.2", "9"): "9    BodyLength",
        ("FIX.4.2", "10"): "10   CheckSum",
        ("FIX.4.2", "11"): "11   ClOrdID",
        ("FIX.4.2", "12"): "12   Commission",
        ("FIX.4.2", "13"): "13   CommType",
        ("FIX.4.2", "14"): "14   CumQty",
        ("FIX.4.2", "15"): "15   Currency",
        ("FIX.4.2", "16"): "16   EndSeqNo",
        ("FIX.4.2", "17"): "17   ExecID",
        ("FIX.4.2", "18"): "18   ExecInst",
        ("FIX.4.2", "19"): "19   ExecRefID",
        ("FIX.4.2", "20"): "20   ExecTransType",
        ("FIX.4.2", "21"): "21   HandlInst",
        ("FIX.4.2", "22"): "22   IDSource",
        ("FIX.4.2", "23"): "23   IOIid",
        ("FIX.4.2", "24"): "24   IOIOthSvc",
        ("FIX.4.2", "25"): "25   IOIQltyInd",
        ("FIX.4.2", "26"): "26   IOIRefID",
        ("FIX.4.2", "27"): "27   IOIShares",
        ("FIX.4.2", "28"): "28   IOITransType",
        ("FIX.4.2", "29"): "29   LastCapacity",
        ("FIX.4.2", "30"): "30   LastMkt",
        ("FIX.4.2", "31"): "31   LastPx",
        ("FIX.4.2", "32"): "32   LastShares",
        ("FIX.4.2", "33"): "33   LinesOfText",
        ("FIX.4.2", "34"): "34   MsgSeqNum",
        ("FIX.4.2", "35"): "35   MsgType",
        ("FIX.4.2", "36"): "36   NewSeqNo",
        ("FIX.4.2", "37"): "37   OrderID",
        ("FIX.4.2", "38"): "38   OrderQty",
        ("FIX.4.2", "39"): "39   OrdStatus",
        ("FIX.4.2", "40"): "40   OrdType",
        ("FIX.4.2", "41"): "41   OrigClOrdID",
        ("FIX.4.2", "42"): "42   OrigTime",
        ("FIX.4.2", "43"): "43   PossDupFlag",
        ("FIX.4.2", "44"): "44   Price",
        ("FIX.4.2", "45"): "45   RefSeqNum",
        ("FIX.4.2", "46"): "46   RelatdSym",
        ("FIX.4.2", "47"): "47   Rule80A",
        ("FIX.4.2", "48"): "48   SecurityID",
        ("FIX.4.2", "49"): "49   SenderCompID",
        ("FIX.4.2", "50"): "50   SenderSubID",
        ("FIX.4.2", "51"): "51   SendingDate",
        ("FIX.4.2", "52"): "52   SendingTime",
        ("FIX.4.2", "53"): "53   Shares",
        ("FIX.4.2", "54"): "54   Side",
        ("FIX.4.2", "55"): "55   Symbol",
        ("FIX.4.2", "56"): "56   TargetCompID",
        ("FIX.4.2", "57"): "57   TargetSubID",
        ("FIX.4.2", "58"): "58   Text",
        ("FIX.4.2", "59"): "59   TimeInForce",
        ("FIX.4.2", "60"): "60   TransactTime",
        ("FIX.4.2", "61"): "61   Urgency",
        ("FIX.4.2", "62"): "62   ValidUntilTime",
        ("FIX.4.2", "63"): "63   SettlmntTyp",
        ("FIX.4.2", "64"): "64   FutSettDate",
        ("FIX.4.2", "65"): "65   SymbolSfx",
        ("FIX.4.2", "66"): "66   ListID",
        ("FIX.4.2", "67"): "67   ListSeqNo",
        ("FIX.4.2", "68"): "68   TotNoOrders",
        ("FIX.4.2", "69"): "69   ListExecInst",
        ("FIX.4.2", "70"): "70   AllocID",
        ("FIX.4.2", "71"): "71   AllocTransType",
        ("FIX.4.2", "72"): "72   RefAllocID",
        ("FIX.4.2", "73"): "73   NoOrders",
        ("FIX.4.2", "74"): "74   AvgPrxPrecision",
        ("FIX.4.2", "75"): "75   TradeDate",
        ("FIX.4.2", "76"): "76   ExecBroker",
        ("FIX.4.2", "77"): "77   OpenClose",
        ("FIX.4.2", "78"): "78   NoAllocs",
        ("FIX.4.2", "79"): "79   AllocAccount",
        ("FIX.4.2", "80"): "80   AllocShares",
        ("FIX.4.2", "81"): "81   ProcessCode",
        ("FIX.4.2", "82"): "82   NoRpts",
        ("FIX.4.2", "83"): "83   RptSeq",
        ("FIX.4.2", "84"): "84   CxlQty",
        ("FIX.4.2", "85"): "85   NoDlvyInst",
        ("FIX.4.2", "86"): "86   DlvyInst",
        ("FIX.4.2", "87"): "87   AllocStatus",
        ("FIX.4.2", "88"): "88   AllocRejCode",
        ("FIX.4.2", "89"): "89   Signature",
        ("FIX.4.2", "90"): "90   SecureDataLen",
        ("FIX.4.2", "91"): "91   SecureData",
        ("FIX.4.2", "92"): "92   BrokerOfCredit",
        ("FIX.4.2", "93"): "93   SignatureLength",
        ("FIX.4.2", "94"): "94   EmailType",
        ("FIX.4.2", "95"): "95   RawDataLength",
        ("FIX.4.2", "96"): "96   RawData",
        ("FIX.4.2", "97"): "97   PossResend",
        ("FIX.4.2", "98"): "98   EncryptMethod",
        ("FIX.4.2", "99"): "99   StopPx",
        ("FIX.4.2", "100"): "100  ExDestination",
        ("FIX.4.2", "102"): "102  CxlRejReason",
        ("FIX.4.2", "103"): "103  OrdRejReason",
        ("FIX.4.2", "104"): "104  IOIQualifier",
        ("FIX.4.2", "105"): "105  WaveNo",
        ("FIX.4.2", "106"): "106  Issuer",
        ("FIX.4.2", "107"): "107  SecurityDesc",
        ("FIX.4.2", "108"): "108  HeartBtInt",
        ("FIX.4.2", "109"): "109  ClientID",
        ("FIX.4.2", "110"): "110  MinQty",
        ("FIX.4.2", "111"): "111  MaxFloor",
        ("FIX.4.2", "112"): "112  TestReqID",
        ("FIX.4.2", "113"): "113  ReportToExch",
        ("FIX.4.2", "114"): "114  LocateReqd",
        ("FIX.4.2", "115"): "115  OnBehalfOfCompID",
        ("FIX.4.2", "116"): "116  OnBehalfOfSubID",
        ("FIX.4.2", "117"): "117  QuoteID",
        ("FIX.4.2", "118"): "118  NetMoney",
        ("FIX.4.2", "119"): "119  SettlCurrAmt",
        ("FIX.4.2", "120"): "120  SettlCurrency",
        ("FIX.4.2", "121"): "121  ForexReq",
        ("FIX.4.2", "122"): "122  OrigSendingTime",
        ("FIX.4.2", "123"): "123  GapFillFlag",
        ("FIX.4.2", "124"): "124  NoExecs",
        ("FIX.4.2", "125"): "125  CxlType",
        ("FIX.4.2", "126"): "126  ExpireTime",
        ("FIX.4.2", "127"): "127  DKReason",
        ("FIX.4.2", "128"): "128  DeliverToCompID",
        ("FIX.4.2", "129"): "129  DeliverToSubID",
        ("FIX.4.2", "130"): "130  IOINaturalFlag",
        ("FIX.4.2", "131"): "131  QuoteReqID",
        ("FIX.4.2", "132"): "132  BidPx",
        ("FIX.4.2", "133"): "133  OfferPx",
        ("FIX.4.2", "134"): "134  BidSize",
        ("FIX.4.2", "135"): "135  OfferSize",
        ("FIX.4.2", "136"): "136  NoMiscFees",
        ("FIX.4.2", "137"): "137  MiscFeeAmt",
        ("FIX.4.2", "138"): "138  MiscFeeCurr",
        ("FIX.4.2", "139"): "139  MiscFeeType",
        ("FIX.4.2", "140"): "140  PrevClosePx",
        ("FIX.4.2", "141"): "141  ResetSeqNumFlag",
        ("FIX.4.2", "142"): "142  SenderLocationID",
        ("FIX.4.2", "143"): "143  TargetLocationID",
        ("FIX.4.2", "144"): "144  OnBehalfOfLocationID",
        ("FIX.4.2", "145"): "145  DeliverToLocationID",
        ("FIX.4.2", "146"): "146  NoRelatedSym",
        ("FIX.4.2", "147"): "147  Subject",
        ("FIX.4.2", "148"): "148  Headline",
        ("FIX.4.2", "149"): "149  URLLink",
        ("FIX.4.2", "150"): "150  ExecType",
        ("FIX.4.2", "151"): "151  LeavesQty",
        ("FIX.4.2", "152"): "152  CashOrderQty",
        ("FIX.4.2", "153"): "153  AllocAvgPx",
        ("FIX.4.2", "154"): "154  AllocNetMoney",
        ("FIX.4.2", "155"): "155  SettlCurrFxRate",
        ("FIX.4.2", "156"): "156  SettlCurrFxRateCalc",
        ("FIX.4.2", "157"): "157  NumDaysInterest",
        ("FIX.4.2", "158"): "158  AccruedInterestRate",
        ("FIX.4.2", "159"): "159  AccruedInterestAmt",
        ("FIX.4.2", "160"): "160  SettlInstMode",
        ("FIX.4.2", "161"): "161  AllocText",
        ("FIX.4.2", "162"): "162  SettlInstID",
        ("FIX.4.2", "163"): "163  SettlInstTransType",
        ("FIX.4.2", "164"): "164  EmailThreadID",
        ("FIX.4.2", "165"): "165  SettlInstSource",
        ("FIX.4.2", "166"): "166  SettlLocation",
        ("FIX.4.2", "167"): "167  SecurityType",
        ("FIX.4.2", "168"): "168  EffectiveTime",
        ("FIX.4.2", "169"): "169  StandInstDbType",
        ("FIX.4.2", "170"): "170  StandInstDbName",
        ("FIX.4.2", "171"): "171  StandInstDbID",
        ("FIX.4.2", "172"): "172  SettlDeliveryType",
        ("FIX.4.2", "173"): "173  SettlDepositoryCode",
        ("FIX.4.2", "174"): "174  SettlBrkrCode",
        ("FIX.4.2", "175"): "175  SettlInstCode",
        ("FIX.4.2", "176"): "176  SecuritySettlAgentName",
        ("FIX.4.2", "177"): "177  SecuritySettlAgentCode",
        ("FIX.4.2", "178"): "178  SecuritySettlAgentAcctNum",
        ("FIX.4.2", "179"): "179  SecuritySettlAgentAcctName",
        ("FIX.4.2", "180"): "180  SecuritySettlAgentContactName",
        ("FIX.4.2", "181"): "181  SecuritySettlAgentContactPhone",
        ("FIX.4.2", "182"): "182  CashSettlAgentName",
        ("FIX.4.2", "183"): "183  CashSettlAgentCode",
        ("FIX.4.2", "184"): "184  CashSettlAgentAcctNum",
        ("FIX.4.2", "185"): "185  CashSettlAgentAcctName",
        ("FIX.4.2", "186"): "186  CashSettlAgentContactName",
        ("FIX.4.2", "187"): "187  CashSettlAgentContactPhone",
        ("FIX.4.2", "188"): "188  BidSpotRate",
        ("FIX.4.2", "189"): "189  BidForwardPoints",
        ("FIX.4.2", "190"): "190  OfferSpotRate",
        ("FIX.4.2", "191"): "191  OfferForwardPoints",
        ("FIX.4.2", "192"): "192  OrderQty2",
        ("FIX.4.2", "193"): "193  FutSettDate2",
        ("FIX.4.2", "194"): "194  LastSpotRate",
        ("FIX.4.2", "195"): "195  LastForwardPoints",
        ("FIX.4.2", "196"): "196  AllocLinkID",
        ("FIX.4.2", "197"): "197  AllocLinkType",
        ("FIX.4.2", "198"): "198  SecondaryOrderID",
        ("FIX.4.2", "199"): "199  NoIOIQualifiers",
        ("FIX.4.2", "200"): "200  MaturityMonthYear",
        ("FIX.4.2", "201"): "201  PutOrCall",
        ("FIX.4.2", "202"): "202  StrikePrice",
        ("FIX.4.2", "203"): "203  CoveredOrUncovered",
        ("FIX.4.2", "204"): "204  CustomerOrFirm",
        ("FIX.4.2", "205"): "205  MaturityDay",
        ("FIX.4.2", "206"): "206  OptAttribute",
        ("FIX.4.2", "207"): "207  SecurityExchange",
        ("FIX.4.2", "208"): "208  NotifyBrokerOfCredit",
        ("FIX.4.2", "209"): "209  AllocHandlInst",
        ("FIX.4.2", "210"): "210  MaxShow",
        ("FIX.4.2", "211"): "211  PegDifference",
        ("FIX.4.2", "212"): "212  XmlDataLen",
        ("FIX.4.2", "213"): "213  XmlData",
        ("FIX.4.2", "214"): "214  SettlInstRefID",
        ("FIX.4.2", "215"): "215  NoRoutingIDs",
        ("FIX.4.2", "216"): "216  RoutingType",
        ("FIX.4.2", "217"): "217  RoutingID",
        ("FIX.4.2", "218"): "218  SpreadToBenchmark",
        ("FIX.4.2", "219"): "219  Benchmark",
        ("FIX.4.2", "223"): "223  CouponRate",
        ("FIX.4.2", "231"): "231  ContractMultiplier",
        ("FIX.4.2", "262"): "262  MDReqID",
        ("FIX.4.2", "263"): "263  SubscriptionRequestType",
        ("FIX.4.2", "264"): "264  MarketDepth",
        ("FIX.4.2", "265"): "265  MDUpdateType",
        ("FIX.4.2", "266"): "266  AggregatedBook",
        ("FIX.4.2", "267"): "267  NoMDEntryTypes",
        ("FIX.4.2", "268"): "268  NoMDEntries",
        ("FIX.4.2", "269"): "269  MDEntryType",
        ("FIX.4.2", "270"): "270  MDEntryPx",
        ("FIX.4.2", "271"): "271  MDEntrySize",
        ("FIX.4.2", "272"): "272  MDEntryDate",
        ("FIX.4.2", "273"): "273  MDEntryTime",
        ("FIX.4.2", "274"): "274  TickDirection",
        ("FIX.4.2", "275"): "275  MDMkt",
        ("FIX.4.2", "276"): "276  QuoteCondition",
        ("FIX.4.2", "277"): "277  TradeCondition",
        ("FIX.4.2", "278"): "278  MDEntryID",
        ("FIX.4.2", "279"): "279  MDUpdateAction",
        ("FIX.4.2", "280"): "280  MDEntryRefID",
        ("FIX.4.2", "281"): "281  MDReqRejReason",
        ("FIX.4.2", "282"): "282  MDEntryOriginator",
        ("FIX.4.2", "283"): "283  LocationID",
        ("FIX.4.2", "284"): "284  DeskID",
        ("FIX.4.2", "285"): "285  DeleteReason",
        ("FIX.4.2", "286"): "286  OpenCloseSettleFlag",
        ("FIX.4.2", "287"): "287  SellerDays",
        ("FIX.4.2", "288"): "288  MDEntryBuyer",
        ("FIX.4.2", "289"): "289  MDEntrySeller",
        ("FIX.4.2", "290"): "290  MDEntryPositionNo",
        ("FIX.4.2", "291"): "291  FinancialStatus",
        ("FIX.4.2", "292"): "292  CorporateAction",
        ("FIX.4.2", "293"): "293  DefBidSize",
        ("FIX.4.2", "294"): "294  DefOfferSize",
        ("FIX.4.2", "295"): "295  NoQuoteEntries",
        ("FIX.4.2", "296"): "296  NoQuoteSets",
        ("FIX.4.2", "297"): "297  QuoteAckStatus",
        ("FIX.4.2", "298"): "298  QuoteCancelType",
        ("FIX.4.2", "299"): "299  QuoteEntryID",
        ("FIX.4.2", "300"): "300  QuoteRejectReason",
        ("FIX.4.2", "301"): "301  QuoteResponseLevel",
        ("FIX.4.2", "302"): "302  QuoteSetID",
        ("FIX.4.2", "303"): "303  QuoteRequestType",
        ("FIX.4.2", "304"): "304  TotQuoteEntries",
        ("FIX.4.2", "305"): "305  UnderlyingIDSource",
        ("FIX.4.2", "306"): "306  UnderlyingIssuer",
        ("FIX.4.2", "307"): "307  UnderlyingSecurityDesc",
        ("FIX.4.2", "308"): "308  UnderlyingSecurityExchange",
        ("FIX.4.2", "309"): "309  UnderlyingSecurityID",
        ("FIX.4.2", "310"): "310  UnderlyingSecurityType",
        ("FIX.4.2", "311"): "311  UnderlyingSymbol",
        ("FIX.4.2", "312"): "312  UnderlyingSymbolSfx",
        ("FIX.4.2", "313"): "313  UnderlyingMaturityMonthYear",
        ("FIX.4.2", "314"): "314  UnderlyingMaturityDay",
        ("FIX.4.2", "315"): "315  UnderlyingPutOrCall",
        ("FIX.4.2", "316"): "316  UnderlyingStrikePrice",
        ("FIX.4.2", "317"): "317  UnderlyingOptAttribute",
        ("FIX.4.2", "318"): "318  UnderlyingCurrency",
        ("FIX.4.2", "319"): "319  RatioQty",
        ("FIX.4.2", "320"): "320  SecurityReqID",
        ("FIX.4.2", "321"): "321  SecurityRequestType",
        ("FIX.4.2", "322"): "322  SecurityResponseID",
        ("FIX.4.2", "323"): "323  SecurityResponseType",
        ("FIX.4.2", "324"): "324  SecurityStatusReqID",
        ("FIX.4.2", "325"): "325  UnsolicitedIndicator",
        ("FIX.4.2", "326"): "326  SecurityTradingStatus",
        ("FIX.4.2", "327"): "327  HaltReason",
        ("FIX.4.2", "328"): "328  InViewOfCommon",
        ("FIX.4.2", "329"): "329  DueToRelated",
        ("FIX.4.2", "330"): "330  BuyVolume",
        ("FIX.4.2", "331"): "331  SellVolume",
        ("FIX.4.2", "332"): "332  HighPx",
        ("FIX.4.2", "333"): "333  LowPx",
        ("FIX.4.2", "334"): "334  Adjustment",
        ("FIX.4.2", "335"): "335  TradSesReqID",
        ("FIX.4.2", "336"): "336  TradingSessionID",
        ("FIX.4.2", "337"): "337  ContraTrader",
        ("FIX.4.2", "338"): "338  TradSesMethod",
        ("FIX.4.2", "339"): "339  TradSesMode",
        ("FIX.4.2", "340"): "340  TradSesStatus",
        ("FIX.4.2", "341"): "341  TradSesStartTime",
        ("FIX.4.2", "342"): "342  TradSesOpenTime",
        ("FIX.4.2", "343"): "343  TradSesPreCloseTime",
        ("FIX.4.2", "344"): "344  TradSesCloseTime",
        ("FIX.4.2", "345"): "345  TradSesEndTime",
        ("FIX.4.2", "346"): "346  NumberOfOrders",
        ("FIX.4.2", "347"): "347  MessageEncoding",
        ("FIX.4.2", "348"): "348  EncodedIssuerLen",
        ("FIX.4.2", "349"): "349  EncodedIssuer",
        ("FIX.4.2", "350"): "350  EncodedSecurityDescLen",
        ("FIX.4.2", "351"): "351  EncodedSecurityDesc",
        ("FIX.4.2", "352"): "352  EncodedListExecInstLen",
        ("FIX.4.2", "353"): "353  EncodedListExecInst",
        ("FIX.4.2", "354"): "354  EncodedTextLen",
        ("FIX.4.2", "355"): "355  EncodedText",
        ("FIX.4.2", "356"): "356  EncodedSubjectLen",
        ("FIX.4.2", "357"): "357  EncodedSubject",
        ("FIX.4.2", "358"): "358  EncodedHeadlineLen",
        ("FIX.4.2", "359"): "359  EncodedHeadline",
        ("FIX.4.2", "360"): "360  EncodedAllocTextLen",
        ("FIX.4.2", "361"): "361  EncodedAllocText",
        ("FIX.4.2", "362"): "362  EncodedUnderlyingIssuerLen",
        ("FIX.4.2", "363"): "363  EncodedUnderlyingIssuer",
        ("FIX.4.2", "364"): "364  EncodedUnderlyingSecurityDescLen",
        ("FIX.4.2", "365"): "365  EncodedUnderlyingSecurityDesc",
        ("FIX.4.2", "366"): "366  AllocPrice",
        ("FIX.4.2", "367"): "367  QuoteSetValidUntilTime",
        ("FIX.4.2", "368"): "368  QuoteEntryRejectReason",
        ("FIX.4.2", "369"): "369  LastMsgSeqNumProcessed",
        ("FIX.4.2", "370"): "370  OnBehalfOfSendingTime",
        ("FIX.4.2", "371"): "371  RefTagID",
        ("FIX.4.2", "372"): "372  RefMsgType",
        ("FIX.4.2", "373"): "373  SessionRejectReason",
        ("FIX.4.2", "374"): "374  BidRequestTransType",
        ("FIX.4.2", "375"): "375  ContraBroker",
        ("FIX.4.2", "376"): "376  ComplianceID",
        ("FIX.4.2", "377"): "377  SolicitedFlag",
        ("FIX.4.2", "378"): "378  ExecRestatementReason",
        ("FIX.4.2", "379"): "379  BusinessRejectRefID",
        ("FIX.4.2", "380"): "380  BusinessRejectReason",
        ("FIX.4.2", "381"): "381  GrossTradeAmt",
        ("FIX.4.2", "382"): "382  NoContraBrokers",
        ("FIX.4.2", "383"): "383  MaxMessageSize",
        ("FIX.4.2", "384"): "384  NoMsgTypes",
        ("FIX.4.2", "385"): "385  MsgDirection",
        ("FIX.4.2", "386"): "386  NoTradingSessions",
        ("FIX.4.2", "387"): "387  TotalVolumeTraded",
        ("FIX.4.2", "388"): "388  DiscretionInst",
        ("FIX.4.2", "389"): "389  DiscretionOffset",
        ("FIX.4.2", "390"): "390  BidID",
        ("FIX.4.2", "391"): "391  ClientBidID",
        ("FIX.4.2", "392"): "392  ListName",
        ("FIX.4.2", "393"): "393  TotalNumSecurities",
        ("FIX.4.2", "394"): "394  BidType",
        ("FIX.4.2", "395"): "395  NumTickets",
        ("FIX.4.2", "396"): "396  SideValue1",
        ("FIX.4.2", "397"): "397  SideValue2",
        ("FIX.4.2", "398"): "398  NoBidDescriptors",
        ("FIX.4.2", "399"): "399  BidDescriptorType",
        ("FIX.4.2", "400"): "400  BidDescriptor",
        ("FIX.4.2", "401"): "401  SideValueInd",
        ("FIX.4.2", "402"): "402  LiquidityPctLow",
        ("FIX.4.2", "403"): "403  LiquidityPctHigh",
        ("FIX.4.2", "404"): "404  LiquidityValue",
        ("FIX.4.2", "405"): "405  EFPTrackingError",
        ("FIX.4.2", "406"): "406  FairValue",
        ("FIX.4.2", "407"): "407  OutsideIndexPct",
        ("FIX.4.2", "408"): "408  ValueOfFutures",
        ("FIX.4.2", "409"): "409  LiquidityIndType",
        ("FIX.4.2", "410"): "410  WtAverageLiquidity",
        ("FIX.4.2", "411"): "411  ExchangeForPhysical",
        ("FIX.4.2", "412"): "412  OutMainCntryUIndex",
        ("FIX.4.2", "413"): "413  CrossPercent",
        ("FIX.4.2", "414"): "414  ProgRptReqs",
        ("FIX.4.2", "415"): "415  ProgPeriodInterval",
        ("FIX.4.2", "416"): "416  IncTaxInd",
        ("FIX.4.2", "417"): "417  NumBidders",
        ("FIX.4.2", "418"): "418  TradeType",
        ("FIX.4.2", "419"): "419  BasisPxType",
        ("FIX.4.2", "420"): "420  NoBidComponents",
        ("FIX.4.2", "421"): "421  Country",
        ("FIX.4.2", "422"): "422  TotNoStrikes",
        ("FIX.4.2", "423"): "423  PriceType",
        ("FIX.4.2", "424"): "424  DayOrderQty",
        ("FIX.4.2", "425"): "425  DayCumQty",
        ("FIX.4.2", "426"): "426  DayAvgPx",
        ("FIX.4.2", "427"): "427  GTBookingInst",
        ("FIX.4.2", "428"): "428  NoStrikes",
        ("FIX.4.2", "429"): "429  ListStatusType",
        ("FIX.4.2", "430"): "430  NetGrossInd",
        ("FIX.4.2", "431"): "431  ListOrderStatus",
        ("FIX.4.2", "432"): "432  ExpireDate",
        ("FIX.4.2", "433"): "433  ListExecInstType",
        ("FIX.4.2", "434"): "434  CxlRejResponseTo",
        ("FIX.4.2", "435"): "435  UnderlyingCouponRate",
        ("FIX.4.2", "436"): "436  UnderlyingContractMultiplier",
        ("FIX.4.2", "437"): "437  ContraTradeQty",
        ("FIX.4.2", "438"): "438  ContraTradeTime",
        ("FIX.4.2", "439"): "439  ClearingFirm",
        ("FIX.4.2", "440"): "440  ClearingAccount",
        ("FIX.4.2", "441"): "441  LiquidityNumSecurities",
        ("FIX.4.2", "442"): "442  MultiLegReportingType",
        ("FIX.4.2", "443"): "443  StrikeTime",
        ("FIX.4.2", "444"): "444  ListStatusText",
        ("FIX.4.2", "445"): "445  EncodedListStatusTextLen",
        ("FIX.4.2", "446"): "446  EncodedListStatusText",
        ("FIX.4.4", "1"): "1    Account",
        ("FIX.4.4", "2"): "2    AdvId",
        ("FIX.4.4", "3"): "3    AdvRefID",
        ("FIX.4.4", "4"): "4    AdvSide",
        ("FIX.4.4", "5"): "5    AdvTransType",
        ("FIX.4.4", "6"): "6    AvgPx",
        ("FIX.4.4", "7"): "7    BeginSeqNo",
        ("FIX.4.4", "8"): "8    BeginString",
        ("FIX.4.4", "9"): "9    BodyLength",
        ("FIX.4.4", "10"): "10   CheckSum",
        ("FIX.4.4", "11"): "11   ClOrdID",
        ("FIX.4.4", "12"): "12   Commission",
        ("FIX.4.4", "13"): "13   CommType",
        ("FIX.4.4", "14"): "14   CumQty",
        ("FIX.4.4", "15"): "15   Currency",
        ("FIX.4.4", "16"): "16   EndSeqNo",
        ("FIX.4.4", "17"): "17   ExecID",
        ("FIX.4.4", "18"): "18   ExecInst",
        ("FIX.4.4", "19"): "19   ExecRefID",
        ("FIX.4.4", "21"): "21   HandlInst",
        ("FIX.4.4", "22"): "22   SecurityIDSource",
        ("FIX.4.4", "23"): "23   IOIID",
        ("FIX.4.4", "25"): "25   IOIQltyInd",
        ("FIX.4.4", "26"): "26   IOIRefID",
        ("FIX.4.4", "27"): "27   IOIQty",
        ("FIX.4.4", "28"): "28   IOITransType",
        ("FIX.4.4", "29"): "29   LastCapacity",
        ("FIX.4.4", "30"): "30   LastMkt",
        ("FIX.4.4", "31"): "31   LastPx",
        ("FIX.4.4", "32"): "32   LastQty",
        ("FIX.4.4", "33"): "33   NoLinesOfText",
        ("FIX.4.4", "34"): "34   MsgSeqNum",
        ("FIX.4.4", "35"): "35   MsgType",
        ("FIX.4.4", "36"): "36   NewSeqNo",
        ("FIX.4.4", "37"): "37   OrderID",
        ("FIX.4.4", "38"): "38   OrderQty",
        ("FIX.4.4", "39"): "39   OrdStatus",
        ("FIX.4.4", "40"): "40   OrdType",
        ("FIX.4.4", "41"): "41   OrigClOrdID",
        ("FIX.4.4", "42"): "42   OrigTime",
        ("FIX.4.4", "43"): "43   PossDupFlag",
        ("FIX.4.4", "44"): "44   Price",
        ("FIX.4.4", "45"): "45   RefSeqNum",
        ("FIX.4.4", "48"): "48   SecurityID",
        ("FIX.4.4", "49"): "49   SenderCompID",
        ("FIX.4.4", "50"): "50   SenderSubID",
        ("FIX.4.4", "52"): "52   SendingTime",
        ("FIX.4.4", "53"): "53   Quantity",
        ("FIX.4.4", "54"): "54   Side",
        ("FIX.4.4", "55"): "55   Symbol",
        ("FIX.4.4", "56"): "56   TargetCompID",
        ("FIX.4.4", "57"): "57   TargetSubID",
        ("FIX.4.4", "58"): "58   Text",
        ("FIX.4.4", "59"): "59   TimeInForce",
        ("FIX.4.4", "60"): "60   TransactTime",
        ("FIX.4.4", "61"): "61   Urgency",
        ("FIX.4.4", "62"): "62   ValidUntilTime",
        ("FIX.4.4", "63"): "63   SettlType",
        ("FIX.4.4", "64"): "64   SettlDate",
        ("FIX.4.4", "65"): "65   SymbolSfx",
        ("FIX.4.4", "66"): "66   ListID",
        ("FIX.4.4", "67"): "67   ListSeqNo",
        ("FIX.4.4", "68"): "68   TotNoOrders",
        ("FIX.4.4", "69"): "69   ListExecInst",
        ("FIX.4.4", "70"): "70   AllocID",
        ("FIX.4.4", "71"): "71   AllocTransType",
        ("FIX.4.4", "72"): "72   RefAllocID",
        ("FIX.4.4", "73"): "73   NoOrders",
        ("FIX.4.4", "74"): "74   AvgPxPrecision",
        ("FIX.4.4", "75"): "75   TradeDate",
        ("FIX.4.4", "77"): "77   PositionEffect",
        ("FIX.4.4", "78"): "78   NoAllocs",
        ("FIX.4.4", "79"): "79   AllocAccount",
        ("FIX.4.4", "80"): "80   AllocQty",
        ("FIX.4.4", "81"): "81   ProcessCode",
        ("FIX.4.4", "82"): "82   NoRpts",
        ("FIX.4.4", "83"): "83   RptSeq",
        ("FIX.4.4", "84"): "84   CxlQty",
        ("FIX.4.4", "85"): "85   NoDlvyInst",
        ("FIX.4.4", "87"): "87   AllocStatus",
        ("FIX.4.4", "88"): "88   AllocRejCode",
        ("FIX.4.4", "89"): "89   Signature",
        ("FIX.4.4", "90"): "90   SecureDataLen",
        ("FIX.4.4", "91"): "91   SecureData",
        ("FIX.4.4", "93"): "93   SignatureLength",
        ("FIX.4.4", "94"): "94   EmailType",
        ("FIX.4.4", "95"): "95   RawDataLength",
        ("FIX.4.4", "96"): "96   RawData",
        ("FIX.4.4", "97"): "97   PossResend",
        ("FIX.4.4", "98"): "98   EncryptMethod",
        ("FIX.4.4", "99"): "99   StopPx",
        ("FIX.4.4", "100"): "100  ExDestination",
        ("FIX.4.4", "102"): "102  CxlRejReason",
        ("FIX.4.4", "103"): "103  OrdRejReason",
        ("FIX.4.4", "104"): "104  IOIQualifier",
        ("FIX.4.4", "106"): "106  Issuer",
        ("FIX.4.4", "107"): "107  SecurityDesc",
        ("FIX.4.4", "108"): "108  HeartBtInt",
        ("FIX.4.4", "110"): "110  MinQty",
        ("FIX.4.4", "111"): "111  MaxFloor",
        ("FIX.4.4", "112"): "112  TestReqID",
        ("FIX.4.4", "113"): "113  ReportToExch",
        ("FIX.4.4", "114"): "114  LocateReqd",
        ("FIX.4.4", "115"): "115  OnBehalfOfCompID",
        ("FIX.4.4", "116"): "116  OnBehalfOfSubID",
        ("FIX.4.4", "117"): "117  QuoteID",
        ("FIX.4.4", "118"): "118  NetMoney",
        ("FIX.4.4", "119"): "119  SettlCurrAmt",
        ("FIX.4.4", "120"): "120  SettlCurrency",
        ("FIX.4.4", "121"): "121  ForexReq",
        ("FIX.4.4", "122"): "122  OrigSendingTime",
        ("FIX.4.4", "123"): "123  GapFillFlag",
        ("FIX.4.4", "124"): "124  NoExecs",
        ("FIX.4.4", "126"): "126  ExpireTime",
        ("FIX.4.4", "127"): "127  DKReason",
        ("FIX.4.4", "128"): "128  DeliverToCompID",
        ("FIX.4.4", "129"): "129  DeliverToSubID",
        ("FIX.4.4", "130"): "130  IOINaturalFlag",
        ("FIX.4.4", "131"): "131  QuoteReqID",
        ("FIX.4.4", "132"): "132  BidPx",
        ("FIX.4.4", "133"): "133  OfferPx",
        ("FIX.4.4", "134"): "134  BidSize",
        ("FIX.4.4", "135"): "135  OfferSize",
        ("FIX.4.4", "136"): "136  NoMiscFees",
        ("FIX.4.4", "137"): "137  MiscFeeAmt",
        ("FIX.4.4", "138"): "138  MiscFeeCurr",
        ("FIX.4.4", "139"): "139  MiscFeeType",
        ("FIX.4.4", "140"): "140  PrevClosePx",
        ("FIX.4.4", "141"): "141  ResetSeqNumFlag",
        ("FIX.4.4", "142"): "142  SenderLocationID",
        ("FIX.4.4", "143"): "143  TargetLocationID",
        ("FIX.4.4", "144"): "144  OnBehalfOfLocationID",
        ("FIX.4.4", "145"): "145  DeliverToLocationID",
        ("FIX.4.4", "146"): "146  NoRelatedSym",
        ("FIX.4.4", "147"): "147  Subject",
        ("FIX.4.4", "148"): "148  Headline",
        ("FIX.4.4", "149"): "149  URLLink",
        ("FIX.4.4", "150"): "150  ExecType",
        ("FIX.4.4", "151"): "151  LeavesQty",
        ("FIX.4.4", "152"): "152  CashOrderQty",
        ("FIX.4.4", "153"): "153  AllocAvgPx",
        ("FIX.4.4", "154"): "154  AllocNetMoney",
        ("FIX.4.4", "155"): "155  SettlCurrFxRate",
        ("FIX.4.4", "156"): "156  SettlCurrFxRateCalc",
        ("FIX.4.4", "157"): "157  NumDaysInterest",
        ("FIX.4.4", "158"): "158  AccruedInterestRate",
        ("FIX.4.4", "159"): "159  AccruedInterestAmt",
        ("FIX.4.4", "160"): "160  SettlInstMode",
        ("FIX.4.4", "161"): "161  AllocText",
        ("FIX.4.4", "162"): "162  SettlInstID",
        ("FIX.4.4", "163"): "163  SettlInstTransType",
        ("FIX.4.4", "164"): "164  EmailThreadID",
        ("FIX.4.4", "165"): "165  SettlInstSource",
        ("FIX.4.4", "167"): "167  SecurityType",
        ("FIX.4.4", "168"): "168  EffectiveTime",
        ("FIX.4.4", "169"): "169  StandInstDbType",
        ("FIX.4.4", "170"): "170  StandInstDbName",
        ("FIX.4.4", "171"): "171  StandInstDbID",
        ("FIX.4.4", "172"): "172  SettlDeliveryType",
        ("FIX.4.4", "188"): "188  BidSpotRate",
        ("FIX.4.4", "189"): "189  BidForwardPoints",
        ("FIX.4.4", "190"): "190  OfferSpotRate",
        ("FIX.4.4", "191"): "191  OfferForwardPoints",
        ("FIX.4.4", "192"): "192  OrderQty2",
        ("FIX.4.4", "193"): "193  SettlDate2",
        ("FIX.4.4", "194"): "194  LastSpotRate",
        ("FIX.4.4", "195"): "195  LastForwardPoints",
        ("FIX.4.4", "196"): "196  AllocLinkID",
        ("FIX.4.4", "197"): "197  AllocLinkType",
        ("FIX.4.4", "198"): "198  SecondaryOrderID",
        ("FIX.4.4", "199"): "199  NoIOIQualifiers",
        ("FIX.4.4", "200"): "200  MaturityMonthYear",
        ("FIX.4.4", "201"): "201  PutOrCall",
        ("FIX.4.4", "202"): "202  StrikePrice",
        ("FIX.4.4", "203"): "203  CoveredOrUncovered",
        ("FIX.4.4", "206"): "206  OptAttribute",
        ("FIX.4.4", "207"): "207  SecurityExchange",
        ("FIX.4.4", "208"): "208  NotifyBrokerOfCredit",
        ("FIX.4.4", "209"): "209  AllocHandlInst",
        ("FIX.4.4", "210"): "210  MaxShow",
        ("FIX.4.4", "211"): "211  PegOffsetValue",
        ("FIX.4.4", "212"): "212  XmlDataLen",
        ("FIX.4.4", "213"): "213  XmlData",
        ("FIX.4.4", "214"): "214  SettlInstRefID",
        ("FIX.4.4", "215"): "215  NoRoutingIDs",
        ("FIX.4.4", "216"): "216  RoutingType",
        ("FIX.4.4", "217"): "217  RoutingID",
        ("FIX.4.4", "218"): "218  Spread",
        ("FIX.4.4", "220"): "220  BenchmarkCurveCurrency",
        ("FIX.4.4", "221"): "221  BenchmarkCurveName",
        ("FIX.4.4", "222"): "222  BenchmarkCurvePoint",
        ("FIX.4.4", "223"): "223  CouponRate",
        ("FIX.4.4", "224"): "224  CouponPaymentDate",
        ("FIX.4.4", "225"): "225  IssueDate",
        ("FIX.4.4", "226"): "226  RepurchaseTerm",
        ("FIX.4.4", "227"): "227  RepurchaseRate",
        ("FIX.4.4", "228"): "228  Factor",
        ("FIX.4.4", "229"): "229  TradeOriginationDate",
        ("FIX.4.4", "230"): "230  ExDate",
        ("FIX.4.4", "231"): "231  ContractMultiplier",
        ("FIX.4.4", "232"): "232  NoStipulations",
        ("FIX.4.4", "233"): "233  StipulationType",
        ("FIX.4.4", "234"): "234  StipulationValue",
        ("FIX.4.4", "235"): "235  YieldType",
        ("FIX.4.4", "236"): "236  Yield",
        ("FIX.4.4", "237"): "237  TotalTakedown",
        ("FIX.4.4", "238"): "238  Concession",
        ("FIX.4.4", "239"): "239  RepoCollateralSecurityType",
        ("FIX.4.4", "240"): "240  RedemptionDate",
        ("FIX.4.4", "241"): "241  UnderlyingCouponPaymentDate",
        ("FIX.4.4", "242"): "242  UnderlyingIssueDate",
        ("FIX.4.4", "243"): "243  UnderlyingRepoCollateralSecurityType",
        ("FIX.4.4", "244"): "244  UnderlyingRepurchaseTerm",
        ("FIX.4.4", "245"): "245  UnderlyingRepurchaseRate",
        ("FIX.4.4", "246"): "246  UnderlyingFactor",
        ("FIX.4.4", "247"): "247  UnderlyingRedemptionDate",
        ("FIX.4.4", "248"): "248  LegCouponPaymentDate",
        ("FIX.4.4", "249"): "249  LegIssueDate",
        ("FIX.4.4", "250"): "250  LegRepoCollateralSecurityType",
        ("FIX.4.4", "251"): "251  LegRepurchaseTerm",
        ("FIX.4.4", "252"): "252  LegRepurchaseRate",
        ("FIX.4.4", "253"): "253  LegFactor",
        ("FIX.4.4", "254"): "254  LegRedemptionDate",
        ("FIX.4.4", "255"): "255  CreditRating",
        ("FIX.4.4", "256"): "256  UnderlyingCreditRating",
        ("FIX.4.4", "257"): "257  LegCreditRating",
        ("FIX.4.4", "258"): "258  TradedFlatSwitch",
        ("FIX.4.4", "259"): "259  BasisFeatureDate",
        ("FIX.4.4", "260"): "260  BasisFeaturePrice",
        ("FIX.4.4", "262"): "262  MDReqID",
        ("FIX.4.4", "263"): "263  SubscriptionRequestType",
        ("FIX.4.4", "264"): "264  MarketDepth",
        ("FIX.4.4", "265"): "265  MDUpdateType",
        ("FIX.4.4", "266"): "266  AggregatedBook",
        ("FIX.4.4", "267"): "267  NoMDEntryTypes",
        ("FIX.4.4", "268"): "268  NoMDEntries",
        ("FIX.4.4", "269"): "269  MDEntryType",
        ("FIX.4.4", "270"): "270  MDEntryPx",
        ("FIX.4.4", "271"): "271  MDEntrySize",
        ("FIX.4.4", "272"): "272  MDEntryDate",
        ("FIX.4.4", "273"): "273  MDEntryTime",
        ("FIX.4.4", "274"): "274  TickDirection",
        ("FIX.4.4", "275"): "275  MDMkt",
        ("FIX.4.4", "276"): "276  QuoteCondition",
        ("FIX.4.4", "277"): "277  TradeCondition",
        ("FIX.4.4", "278"): "278  MDEntryID",
        ("FIX.4.4", "279"): "279  MDUpdateAction",
        ("FIX.4.4", "280"): "280  MDEntryRefID",
        ("FIX.4.4", "281"): "281  MDReqRejReason",
        ("FIX.4.4", "282"): "282  MDEntryOriginator",
        ("FIX.4.4", "283"): "283  LocationID",
        ("FIX.4.4", "284"): "284  DeskID",
        ("FIX.4.4", "285"): "285  DeleteReason",
        ("FIX.4.4", "286"): "286  OpenCloseSettlFlag",
        ("FIX.4.4", "287"): "287  SellerDays",
        ("FIX.4.4", "288"): "288  MDEntryBuyer",
        ("FIX.4.4", "289"): "289  MDEntrySeller",
        ("FIX.4.4", "290"): "290  MDEntryPositionNo",
        ("FIX.4.4", "291"): "291  FinancialStatus",
        ("FIX.4.4", "292"): "292  CorporateAction",
        ("FIX.4.4", "293"): "293  DefBidSize",
        ("FIX.4.4", "294"): "294  DefOfferSize",
        ("FIX.4.4", "295"): "295  NoQuoteEntries",
        ("FIX.4.4", "296"): "296  NoQuoteSets",
        ("FIX.4.4", "297"): "297  QuoteStatus",
        ("FIX.4.4", "298"): "298  QuoteCancelType",
        ("FIX.4.4", "299"): "299  QuoteEntryID",
        ("FIX.4.4", "300"): "300  QuoteRejectReason",
        ("FIX.4.4", "301"): "301  QuoteResponseLevel",
        ("FIX.4.4", "302"): "302  QuoteSetID",
        ("FIX.4.4", "303"): "303  QuoteRequestType",
        ("FIX.4.4", "304"): "304  TotNoQuoteEntries",
        ("FIX.4.4", "305"): "305  UnderlyingSecurityIDSource",
        ("FIX.4.4", "306"): "306  UnderlyingIssuer",
        ("FIX.4.4", "307"): "307  UnderlyingSecurityDesc",
        ("FIX.4.4", "308"): "308  UnderlyingSecurityExchange",
        ("FIX.4.4", "309"): "309  UnderlyingSecurityID",
        ("FIX.4.4", "310"): "310  UnderlyingSecurityType",
        ("FIX.4.4", "311"): "311  UnderlyingSymbol",
        ("FIX.4.4", "312"): "312  UnderlyingSymbolSfx",
        ("FIX.4.4", "313"): "313  UnderlyingMaturityMonthYear",
        ("FIX.4.4", "315"): "315  UnderlyingPutOrCall",
        ("FIX.4.4", "316"): "316  UnderlyingStrikePrice",
        ("FIX.4.4", "317"): "317  UnderlyingOptAttribute",
        ("FIX.4.4", "318"): "318  UnderlyingCurrency",
        ("FIX.4.4", "320"): "320  SecurityReqID",
        ("FIX.4.4", "321"): "321  SecurityRequestType",
        ("FIX.4.4", "322"): "322  SecurityResponseID",
        ("FIX.4.4", "323"): "323  SecurityResponseType",
        ("FIX.4.4", "324"): "324  SecurityStatusReqID",
        ("FIX.4.4", "325"): "325  UnsolicitedIndicator",
        ("FIX.4.4", "326"): "326  SecurityTradingStatus",
        ("FIX.4.4", "327"): "327  HaltReason",
        ("FIX.4.4", "328"): "328  InViewOfCommon",
        ("FIX.4.4", "329"): "329  DueToRelated",
        ("FIX.4.4", "330"): "330  BuyVolume",
        ("FIX.4.4", "331"): "331  SellVolume",
        ("FIX.4.4", "332"): "332  HighPx",
        ("FIX.4.4", "333"): "333  LowPx",
        ("FIX.4.4", "334"): "334  Adjustment",
        ("FIX.4.4", "335"): "335  TradSesReqID",
        ("FIX.4.4", "336"): "336  TradingSessionID",
        ("FIX.4.4", "337"): "337  ContraTrader",
        ("FIX.4.4", "338"): "338  TradSesMethod",
        ("FIX.4.4", "339"): "339  TradSesMode",
        ("FIX.4.4", "340"): "340  TradSesStatus",
        ("FIX.4.4", "341"): "341  TradSesStartTime",
        ("FIX.4.4", "342"): "342  TradSesOpenTime",
        ("FIX.4.4", "343"): "343  TradSesPreCloseTime",
        ("FIX.4.4", "344"): "344  TradSesCloseTime",
        ("FIX.4.4", "345"): "345  TradSesEndTime",
        ("FIX.4.4", "346"): "346  NumberOfOrders",
        ("FIX.4.4", "347"): "347  MessageEncoding",
        ("FIX.4.4", "348"): "348  EncodedIssuerLen",
        ("FIX.4.4", "349"): "349  EncodedIssuer",
        ("FIX.4.4", "350"): "350  EncodedSecurityDescLen",
        ("FIX.4.4", "351"): "351  EncodedSecurityDesc",
        ("FIX.4.4", "352"): "352  EncodedListExecInstLen",
        ("FIX.4.4", "353"): "353  EncodedListExecInst",
        ("FIX.4.4", "354"): "354  EncodedTextLen",
        ("FIX.4.4", "355"): "355  EncodedText",
        ("FIX.4.4", "356"): "356  EncodedSubjectLen",
        ("FIX.4.4", "357"): "357  EncodedSubject",
        ("FIX.4.4", "358"): "358  EncodedHeadlineLen",
        ("FIX.4.4", "359"): "359  EncodedHeadline",
        ("FIX.4.4", "360"): "360  EncodedAllocTextLen",
        ("FIX.4.4", "361"): "361  EncodedAllocText",
        ("FIX.4.4", "362"): "362  EncodedUnderlyingIssuerLen",
        ("FIX.4.4", "363"): "363  EncodedUnderlyingIssuer",
        ("FIX.4.4", "364"): "364  EncodedUnderlyingSecurityDescLen",
        ("FIX.4.4", "365"): "365  EncodedUnderlyingSecurityDesc",
        ("FIX.4.4", "366"): "366  AllocPrice",
        ("FIX.4.4", "367"): "367  QuoteSetValidUntilTime",
        ("FIX.4.4", "368"): "368  QuoteEntryRejectReason",
        ("FIX.4.4", "369"): "369  LastMsgSeqNumProcessed",
        ("FIX.4.4", "371"): "371  RefTagID",
        ("FIX.4.4", "372"): "372  RefMsgType",
        ("FIX.4.4", "373"): "373  SessionRejectReason",
        ("FIX.4.4", "374"): "374  BidRequestTransType",
        ("FIX.4.4", "375"): "375  ContraBroker",
        ("FIX.4.4", "376"): "376  ComplianceID",
        ("FIX.4.4", "377"): "377  SolicitedFlag",
        ("FIX.4.4", "378"): "378  ExecRestatementReason",
        ("FIX.4.4", "379"): "379  BusinessRejectRefID",
        ("FIX.4.4", "380"): "380  BusinessRejectReason",
        ("FIX.4.4", "381"): "381  GrossTradeAmt",
        ("FIX.4.4", "382"): "382  NoContraBrokers",
        ("FIX.4.4", "383"): "383  MaxMessageSize",
        ("FIX.4.4", "384"): "384  NoMsgTypes",
        ("FIX.4.4", "385"): "385  MsgDirection",
        ("FIX.4.4", "386"): "386  NoTradingSessions",
        ("FIX.4.4", "387"): "387  TotalVolumeTraded",
        ("FIX.4.4", "388"): "388  DiscretionInst",
        ("FIX.4.4", "389"): "389  DiscretionOffsetValue",
        ("FIX.4.4", "390"): "390  BidID",
        ("FIX.4.4", "391"): "391  ClientBidID",
        ("FIX.4.4", "392"): "392  ListName",
        ("FIX.4.4", "393"): "393  TotNoRelatedSym",
        ("FIX.4.4", "394"): "394  BidType",
        ("FIX.4.4", "395"): "395  NumTickets",
        ("FIX.4.4", "396"): "396  SideValue1",
        ("FIX.4.4", "397"): "397  SideValue2",
        ("FIX.4.4", "398"): "398  NoBidDescriptors",
        ("FIX.4.4", "399"): "399  BidDescriptorType",
        ("FIX.4.4", "400"): "400  BidDescriptor",
        ("FIX.4.4", "401"): "401  SideValueInd",
        ("FIX.4.4", "402"): "402  LiquidityPctLow",
        ("FIX.4.4", "403"): "403  LiquidityPctHigh",
        ("FIX.4.4", "404"): "404  LiquidityValue",
        ("FIX.4.4", "405"): "405  EFPTrackingError",
        ("FIX.4.4", "406"): "406  FairValue",
        ("FIX.4.4", "407"): "407  OutsideIndexPct",
        ("FIX.4.4", "408"): "408  ValueOfFutures",
        ("FIX.4.4", "409"): "409  LiquidityIndType",
        ("FIX.4.4", "410"): "410  WtAverageLiquidity",
        ("FIX.4.4", "411"): "411  ExchangeForPhysical",
        ("FIX.4.4", "412"): "412  OutMainCntryUIndex",
        ("FIX.4.4", "413"): "413  CrossPercent",
        ("FIX.4.4", "414"): "414  ProgRptReqs",
        ("FIX.4.4", "415"): "415  ProgPeriodInterval",
        ("FIX.4.4", "416"): "416  IncTaxInd",
        ("FIX.4.4", "417"): "417  NumBidders",
        ("FIX.4.4", "418"): "418  BidTradeType",
        ("FIX.4.4", "419"): "419  BasisPxType",
        ("FIX.4.4", "420"): "420  NoBidComponents",
        ("FIX.4.4", "421"): "421  Country",
        ("FIX.4.4", "422"): "422  TotNoStrikes",
        ("FIX.4.4", "423"): "423  PriceType",
        ("FIX.4.4", "424"): "424  DayOrderQty",
        ("FIX.4.4", "425"): "425  DayCumQty",
        ("FIX.4.4", "426"): "426  DayAvgPx",
        ("FIX.4.4", "427"): "427  GTBookingInst",
        ("FIX.4.4", "428"): "428  NoStrikes",
        ("FIX.4.4", "429"): "429  ListStatusType",
        ("FIX.4.4", "430"): "430  NetGrossInd",
        ("FIX.4.4", "431"): "431  ListOrderStatus",
        ("FIX.4.4", "432"): "432  ExpireDate",
        ("FIX.4.4", "433"): "433  ListExecInstType",
        ("FIX.4.4", "434"): "434  CxlRejResponseTo",
        ("FIX.4.4", "435"): "435  UnderlyingCouponRate",
        ("FIX.4.4", "436"): "436  UnderlyingContractMultiplier",
        ("FIX.4.4", "437"): "437  ContraTradeQty",
        ("FIX.4.4", "438"): "438  ContraTradeTime",
        ("FIX.4.4", "441"): "441  LiquidityNumSecurities",
        ("FIX.4.4", "442"): "442  MultiLegReportingType",
        ("FIX.4.4", "443"): "443  StrikeTime",
        ("FIX.4.4", "444"): "444  ListStatusText",
        ("FIX.4.4", "445"): "445  EncodedListStatusTextLen",
        ("FIX.4.4", "446"): "446  EncodedListStatusText",
        ("FIX.4.4", "447"): "447  PartyIDSource",
        ("FIX.4.4", "448"): "448  PartyID",
        ("FIX.4.4", "451"): "451  NetChgPrevDay",
        ("FIX.4.4", "452"): "452  PartyRole",
        ("FIX.4.4", "453"): "453  NoPartyIDs",
        ("FIX.4.4", "454"): "454  NoSecurityAltID",
        ("FIX.4.4", "455"): "455  SecurityAltID",
        ("FIX.4.4", "456"): "456  SecurityAltIDSource",
        ("FIX.4.4", "457"): "457  NoUnderlyingSecurityAltID",
        ("FIX.4.4", "458"): "458  UnderlyingSecurityAltID",
        ("FIX.4.4", "459"): "459  UnderlyingSecurityAltIDSource",
        ("FIX.4.4", "460"): "460  Product",
        ("FIX.4.4", "461"): "461  CFICode",
        ("FIX.4.4", "462"): "462  UnderlyingProduct",
        ("FIX.4.4", "463"): "463  UnderlyingCFICode",
        ("FIX.4.4", "464"): "464  TestMessageIndicator",
        ("FIX.4.4", "466"): "466  BookingRefID",
        ("FIX.4.4", "467"): "467  IndividualAllocID",
        ("FIX.4.4", "468"): "468  RoundingDirection",
        ("FIX.4.4", "469"): "469  RoundingModulus",
        ("FIX.4.4", "470"): "470  CountryOfIssue",
        ("FIX.4.4", "471"): "471  StateOrProvinceOfIssue",
        ("FIX.4.4", "472"): "472  LocaleOfIssue",
        ("FIX.4.4", "473"): "473  NoRegistDtls",
        ("FIX.4.4", "474"): "474  MailingDtls",
        ("FIX.4.4", "475"): "475  InvestorCountryOfResidence",
        ("FIX.4.4", "476"): "476  PaymentRef",
        ("FIX.4.4", "477"): "477  DistribPaymentMethod",
        ("FIX.4.4", "478"): "478  CashDistribCurr",
        ("FIX.4.4", "479"): "479  CommCurrency",
        ("FIX.4.4", "480"): "480  CancellationRights",
        ("FIX.4.4", "481"): "481  MoneyLaunderingStatus",
        ("FIX.4.4", "482"): "482  MailingInst",
        ("FIX.4.4", "483"): "483  TransBkdTime",
        ("FIX.4.4", "484"): "484  ExecPriceType",
        ("FIX.4.4", "485"): "485  ExecPriceAdjustment",
        ("FIX.4.4", "486"): "486  DateOfBirth",
        ("FIX.4.4", "487"): "487  TradeReportTransType",
        ("FIX.4.4", "488"): "488  CardHolderName",
        ("FIX.4.4", "489"): "489  CardNumber",
        ("FIX.4.4", "490"): "490  CardExpDate",
        ("FIX.4.4", "491"): "491  CardIssNum",
        ("FIX.4.4", "492"): "492  PaymentMethod",
        ("FIX.4.4", "493"): "493  RegistAcctType",
        ("FIX.4.4", "494"): "494  Designation",
        ("FIX.4.4", "495"): "495  TaxAdvantageType",
        ("FIX.4.4", "496"): "496  RegistRejReasonText",
        ("FIX.4.4", "497"): "497  FundRenewWaiv",
        ("FIX.4.4", "498"): "498  CashDistribAgentName",
        ("FIX.4.4", "499"): "499  CashDistribAgentCode",
        ("FIX.4.4", "500"): "500  CashDistribAgentAcctNumber",
        ("FIX.4.4", "501"): "501  CashDistribPayRef",
        ("FIX.4.4", "502"): "502  CashDistribAgentAcctName",
        ("FIX.4.4", "503"): "503  CardStartDate",
        ("FIX.4.4", "504"): "504  PaymentDate",
        ("FIX.4.4", "505"): "505  PaymentRemitterID",
        ("FIX.4.4", "506"): "506  RegistStatus",
        ("FIX.4.4", "507"): "507  RegistRejReasonCode",
        ("FIX.4.4", "508"): "508  RegistRefID",
        ("FIX.4.4", "509"): "509  RegistDtls",
        ("FIX.4.4", "510"): "510  NoDistribInsts",
        ("FIX.4.4", "511"): "511  RegistEmail",
        ("FIX.4.4", "512"): "512  DistribPercentage",
        ("FIX.4.4", "513"): "513  RegistID",
        ("FIX.4.4", "514"): "514  RegistTransType",
        ("FIX.4.4", "515"): "515  ExecValuationPoint",
        ("FIX.4.4", "516"): "516  OrderPercent",
        ("FIX.4.4", "517"): "517  OwnershipType",
        ("FIX.4.4", "518"): "518  NoContAmts",
        ("FIX.4.4", "519"): "519  ContAmtType",
        ("FIX.4.4", "520"): "520  ContAmtValue",
        ("FIX.4.4", "521"): "521  ContAmtCurr",
        ("FIX.4.4", "522"): "522  OwnerType",
        ("FIX.4.4", "523"): "523  PartySubID",
        ("FIX.4.4", "524"): "524  NestedPartyID",
        ("FIX.4.4", "525"): "525  NestedPartyIDSource",
        ("FIX.4.4", "526"): "526  SecondaryClOrdID",
        ("FIX.4.4", "527"): "527  SecondaryExecID",
        ("FIX.4.4", "528"): "528  OrderCapacity",
        ("FIX.4.4", "529"): "529  OrderRestrictions",
        ("FIX.4.4", "530"): "530  MassCancelRequestType",
        ("FIX.4.4", "531"): "531  MassCancelResponse",
        ("FIX.4.4", "532"): "532  MassCancelRejectReason",
        ("FIX.4.4", "533"): "533  TotalAffectedOrders",
        ("FIX.4.4", "534"): "534  NoAffectedOrders",
        ("FIX.4.4", "535"): "535  AffectedOrderID",
        ("FIX.4.4", "536"): "536  AffectedSecondaryOrderID",
        ("FIX.4.4", "537"): "537  QuoteType",
        ("FIX.4.4", "538"): "538  NestedPartyRole",
        ("FIX.4.4", "539"): "539  NoNestedPartyIDs",
        ("FIX.4.4", "540"): "540  TotalAccruedInterestAmt",
        ("FIX.4.4", "541"): "541  MaturityDate",
        ("FIX.4.4", "542"): "542  UnderlyingMaturityDate",
        ("FIX.4.4", "543"): "543  InstrRegistry",
        ("FIX.4.4", "544"): "544  CashMargin",
        ("FIX.4.4", "545"): "545  NestedPartySubID",
        ("FIX.4.4", "546"): "546  Scope",
        ("FIX.4.4", "547"): "547  MDImplicitDelete",
        ("FIX.4.4", "548"): "548  CrossID",
        ("FIX.4.4", "549"): "549  CrossType",
        ("FIX.4.4", "550"): "550  CrossPrioritization",
        ("FIX.4.4", "551"): "551  OrigCrossID",
        ("FIX.4.4", "552"): "552  NoSides",
        ("FIX.4.4", "553"): "553  Username",
        ("FIX.4.4", "554"): "554  Password",
        ("FIX.4.4", "555"): "555  NoLegs",
        ("FIX.4.4", "556"): "556  LegCurrency",
        ("FIX.4.4", "557"): "557  TotNoSecurityTypes",
        ("FIX.4.4", "558"): "558  NoSecurityTypes",
        ("FIX.4.4", "559"): "559  SecurityListRequestType",
        ("FIX.4.4", "560"): "560  SecurityRequestResult",
        ("FIX.4.4", "561"): "561  RoundLot",
        ("FIX.4.4", "562"): "562  MinTradeVol",
        ("FIX.4.4", "563"): "563  MultiLegRptTypeReq",
        ("FIX.4.4", "564"): "564  LegPositionEffect",
        ("FIX.4.4", "565"): "565  LegCoveredOrUncovered",
        ("FIX.4.4", "566"): "566  LegPrice",
        ("FIX.4.4", "567"): "567  TradSesStatusRejReason",
        ("FIX.4.4", "568"): "568  TradeRequestID",
        ("FIX.4.4", "569"): "569  TradeRequestType",
        ("FIX.4.4", "570"): "570  PreviouslyReported",
        ("FIX.4.4", "571"): "571  TradeReportID",
        ("FIX.4.4", "572"): "572  TradeReportRefID",
        ("FIX.4.4", "573"): "573  MatchStatus",
        ("FIX.4.4", "574"): "574  MatchType",
        ("FIX.4.4", "575"): "575  OddLot",
        ("FIX.4.4", "576"): "576  NoClearingInstructions",
        ("FIX.4.4", "577"): "577  ClearingInstruction",
        ("FIX.4.4", "578"): "578  TradeInputSource",
        ("FIX.4.4", "579"): "579  TradeInputDevice",
        ("FIX.4.4", "580"): "580  NoDates",
        ("FIX.4.4", "581"): "581  AccountType",
        ("FIX.4.4", "582"): "582  CustOrderCapacity",
        ("FIX.4.4", "583"): "583  ClOrdLinkID",
        ("FIX.4.4", "584"): "584  MassStatusReqID",
        ("FIX.4.4", "585"): "585  MassStatusReqType",
        ("FIX.4.4", "586"): "586  OrigOrdModTime",
        ("FIX.4.4", "587"): "587  LegSettlType",
        ("FIX.4.4", "588"): "588  LegSettlDate",
        ("FIX.4.4", "589"): "589  DayBookingInst",
        ("FIX.4.4", "590"): "590  BookingUnit",
        ("FIX.4.4", "591"): "591  PreallocMethod",
        ("FIX.4.4", "592"): "592  UnderlyingCountryOfIssue",
        ("FIX.4.4", "593"): "593  UnderlyingStateOrProvinceOfIssue",
        ("FIX.4.4", "594"): "594  UnderlyingLocaleOfIssue",
        ("FIX.4.4", "595"): "595  UnderlyingInstrRegistry",
        ("FIX.4.4", "596"): "596  LegCountryOfIssue",
        ("FIX.4.4", "597"): "597  LegStateOrProvinceOfIssue",
        ("FIX.4.4", "598"): "598  LegLocaleOfIssue",
        ("FIX.4.4", "599"): "599  LegInstrRegistry",
        ("FIX.4.4", "600"): "600  LegSymbol",
        ("FIX.4.4", "601"): "601  LegSymbolSfx",
        ("FIX.4.4", "602"): "602  LegSecurityID",
        ("FIX.4.4", "603"): "603  LegSecurityIDSource",
        ("FIX.4.4", "604"): "604  NoLegSecurityAltID",
        ("FIX.4.4", "605"): "605  LegSecurityAltID",
        ("FIX.4.4", "606"): "606  LegSecurityAltIDSource",
        ("FIX.4.4", "607"): "607  LegProduct",
        ("FIX.4.4", "608"): "608  LegCFICode",
        ("FIX.4.4", "609"): "609  LegSecurityType",
        ("FIX.4.4", "610"): "610  LegMaturityMonthYear",
        ("FIX.4.4", "611"): "611  LegMaturityDate",
        ("FIX.4.4", "612"): "612  LegStrikePrice",
        ("FIX.4.4", "613"): "613  LegOptAttribute",
        ("FIX.4.4", "614"): "614  LegContractMultiplier",
        ("FIX.4.4", "615"): "615  LegCouponRate",
        ("FIX.4.4", "616"): "616  LegSecurityExchange",
        ("FIX.4.4", "617"): "617  LegIssuer",
        ("FIX.4.4", "618"): "618  EncodedLegIssuerLen",
        ("FIX.4.4", "619"): "619  EncodedLegIssuer",
        ("FIX.4.4", "620"): "620  LegSecurityDesc",
        ("FIX.4.4", "621"): "621  EncodedLegSecurityDescLen",
        ("FIX.4.4", "622"): "622  EncodedLegSecurityDesc",
        ("FIX.4.4", "623"): "623  LegRatioQty",
        ("FIX.4.4", "624"): "624  LegSide",
        ("FIX.4.4", "625"): "625  TradingSessionSubID",
        ("FIX.4.4", "626"): "626  AllocType",
        ("FIX.4.4", "627"): "627  NoHops",
        ("FIX.4.4", "628"): "628  HopCompID",
        ("FIX.4.4", "629"): "629  HopSendingTime",
        ("FIX.4.4", "630"): "630  HopRefID",
        ("FIX.4.4", "631"): "631  MidPx",
        ("FIX.4.4", "632"): "632  BidYield",
        ("FIX.4.4", "633"): "633  MidYield",
        ("FIX.4.4", "634"): "634  OfferYield",
        ("FIX.4.4", "635"): "635  ClearingFeeIndicator",
        ("FIX.4.4", "636"): "636  WorkingIndicator",
        ("FIX.4.4", "637"): "637  LegLastPx",
        ("FIX.4.4", "638"): "638  PriorityIndicator",
        ("FIX.4.4", "639"): "639  PriceImprovement",
        ("FIX.4.4", "640"): "640  Price2",
        ("FIX.4.4", "641"): "641  LastForwardPoints2",
        ("FIX.4.4", "642"): "642  BidForwardPoints2",
        ("FIX.4.4", "643"): "643  OfferForwardPoints2",
        ("FIX.4.4", "644"): "644  RFQReqID",
        ("FIX.4.4", "645"): "645  MktBidPx",
        ("FIX.4.4", "646"): "646  MktOfferPx",
        ("FIX.4.4", "647"): "647  MinBidSize",
        ("FIX.4.4", "648"): "648  MinOfferSize",
        ("FIX.4.4", "649"): "649  QuoteStatusReqID",
        ("FIX.4.4", "650"): "650  LegalConfirm",
        ("FIX.4.4", "651"): "651  UnderlyingLastPx",
        ("FIX.4.4", "652"): "652  UnderlyingLastQty",
        ("FIX.4.4", "654"): "654  LegRefID",
        ("FIX.4.4", "655"): "655  ContraLegRefID",
        ("FIX.4.4", "656"): "656  SettlCurrBidFxRate",
        ("FIX.4.4", "657"): "657  SettlCurrOfferFxRate",
        ("FIX.4.4", "658"): "658  QuoteRequestRejectReason",
        ("FIX.4.4", "659"): "659  SideComplianceID",
        ("FIX.4.4", "660"): "660  AcctIDSource",
        ("FIX.4.4", "661"): "661  AllocAcctIDSource",
        ("FIX.4.4", "662"): "662  BenchmarkPrice",
        ("FIX.4.4", "663"): "663  BenchmarkPriceType",
        ("FIX.4.4", "664"): "664  ConfirmID",
        ("FIX.4.4", "665"): "665  ConfirmStatus",
        ("FIX.4.4", "666"): "666  ConfirmTransType",
        ("FIX.4.4", "667"): "667  ContractSettlMonth",
        ("FIX.4.4", "668"): "668  DeliveryForm",
        ("FIX.4.4", "669"): "669  LastParPx",
        ("FIX.4.4", "670"): "670  NoLegAllocs",
        ("FIX.4.4", "671"): "671  LegAllocAccount",
        ("FIX.4.4", "672"): "672  LegIndividualAllocID",
        ("FIX.4.4", "673"): "673  LegAllocQty",
        ("FIX.4.4", "674"): "674  LegAllocAcctIDSource",
        ("FIX.4.4", "675"): "675  LegSettlCurrency",
        ("FIX.4.4", "676"): "676  LegBenchmarkCurveCurrency",
        ("FIX.4.4", "677"): "677  LegBenchmarkCurveName",
        ("FIX.4.4", "678"): "678  LegBenchmarkCurvePoint",
        ("FIX.4.4", "679"): "679  LegBenchmarkPrice",
        ("FIX.4.4", "680"): "680  LegBenchmarkPriceType",
        ("FIX.4.4", "681"): "681  LegBidPx",
        ("FIX.4.4", "682"): "682  LegIOIQty",
        ("FIX.4.4", "683"): "683  NoLegStipulations",
        ("FIX.4.4", "684"): "684  LegOfferPx",
        ("FIX.4.4", "686"): "686  LegPriceType",
        ("FIX.4.4", "687"): "687  LegQty",
        ("FIX.4.4", "688"): "688  LegStipulationType",
        ("FIX.4.4", "689"): "689  LegStipulationValue",
        ("FIX.4.4", "690"): "690  LegSwapType",
        ("FIX.4.4", "691"): "691  Pool",
        ("FIX.4.4", "692"): "692  QuotePriceType",
        ("FIX.4.4", "693"): "693  QuoteRespID",
        ("FIX.4.4", "694"): "694  QuoteRespType",
        ("FIX.4.4", "695"): "695  QuoteQualifier",
        ("FIX.4.4", "696"): "696  YieldRedemptionDate",
        ("FIX.4.4", "697"): "697  YieldRedemptionPrice",
        ("FIX.4.4", "698"): "698  YieldRedemptionPriceType",
        ("FIX.4.4", "699"): "699  BenchmarkSecurityID",
        ("FIX.4.4", "700"): "700  ReversalIndicator",
        ("FIX.4.4", "701"): "701  YieldCalcDate",
        ("FIX.4.4", "702"): "702  NoPositions",
        ("FIX.4.4", "703"): "703  PosType",
        ("FIX.4.4", "704"): "704  LongQty",
        ("FIX.4.4", "705"): "705  ShortQty",
        ("FIX.4.4", "706"): "706  PosQtyStatus",
        ("FIX.4.4", "707"): "707  PosAmtType",
        ("FIX.4.4", "708"): "708  PosAmt",
        ("FIX.4.4", "709"): "709  PosTransType",
        ("FIX.4.4", "710"): "710  PosReqID",
        ("FIX.4.4", "711"): "711  NoUnderlyings",
        ("FIX.4.4", "712"): "712  PosMaintAction",
        ("FIX.4.4", "713"): "713  OrigPosReqRefID",
        ("FIX.4.4", "714"): "714  PosMaintRptRefID",
        ("FIX.4.4", "715"): "715  ClearingBusinessDate",
        ("FIX.4.4", "716"): "716  SettlSessID",
        ("FIX.4.4", "717"): "717  SettlSessSubID",
        ("FIX.4.4", "718"): "718  AdjustmentType",
        ("FIX.4.4", "719"): "719  ContraryInstructionIndicator",
        ("FIX.4.4", "720"): "720  PriorSpreadIndicator",
        ("FIX.4.4", "721"): "721  PosMaintRptID",
        ("FIX.4.4", "722"): "722  PosMaintStatus",
        ("FIX.4.4", "723"): "723  PosMaintResult",
        ("FIX.4.4", "724"): "724  PosReqType",
        ("FIX.4.4", "725"): "725  ResponseTransportType",
        ("FIX.4.4", "726"): "726  ResponseDestination",
        ("FIX.4.4", "727"): "727  TotalNumPosReports",
        ("FIX.4.4", "728"): "728  PosReqResult",
        ("FIX.4.4", "729"): "729  PosReqStatus",
        ("FIX.4.4", "730"): "730  SettlPrice",
        ("FIX.4.4", "731"): "731  SettlPriceType",
        ("FIX.4.4", "732"): "732  UnderlyingSettlPrice",
        ("FIX.4.4", "733"): "733  UnderlyingSettlPriceType",
        ("FIX.4.4", "734"): "734  PriorSettlPrice",
        ("FIX.4.4", "735"): "735  NoQuoteQualifiers",
        ("FIX.4.4", "736"): "736  AllocSettlCurrency",
        ("FIX.4.4", "737"): "737  AllocSettlCurrAmt",
        ("FIX.4.4", "738"): "738  InterestAtMaturity",
        ("FIX.4.4", "739"): "739  LegDatedDate",
        ("FIX.4.4", "740"): "740  LegPool",
        ("FIX.4.4", "741"): "741  AllocInterestAtMaturity",
        ("FIX.4.4", "742"): "742  AllocAccruedInterestAmt",
        ("FIX.4.4", "743"): "743  DeliveryDate",
        ("FIX.4.4", "744"): "744  AssignmentMethod",
        ("FIX.4.4", "745"): "745  AssignmentUnit",
        ("FIX.4.4", "746"): "746  OpenInterest",
        ("FIX.4.4", "747"): "747  ExerciseMethod",
        ("FIX.4.4", "748"): "748  TotNumTradeReports",
        ("FIX.4.4", "749"): "749  TradeRequestResult",
        ("FIX.4.4", "750"): "750  TradeRequestStatus",
        ("FIX.4.4", "751"): "751  TradeReportRejectReason",
        ("FIX.4.4", "752"): "752  SideMultiLegReportingType",
        ("FIX.4.4", "753"): "753  NoPosAmt",
        ("FIX.4.4", "754"): "754  AutoAcceptIndicator",
        ("FIX.4.4", "755"): "755  AllocReportID",
        ("FIX.4.4", "756"): "756  NoNested2PartyIDs",
        ("FIX.4.4", "757"): "757  Nested2PartyID",
        ("FIX.4.4", "758"): "758  Nested2PartyIDSource",
        ("FIX.4.4", "759"): "759  Nested2PartyRole",
        ("FIX.4.4", "760"): "760  Nested2PartySubID",
        ("FIX.4.4", "761"): "761  BenchmarkSecurityIDSource",
        ("FIX.4.4", "762"): "762  SecuritySubType",
        ("FIX.4.4", "763"): "763  UnderlyingSecuritySubType",
        ("FIX.4.4", "764"): "764  LegSecuritySubType",
        ("FIX.4.4", "765"): "765  AllowableOneSidednessPct",
        ("FIX.4.4", "766"): "766  AllowableOneSidednessValue",
        ("FIX.4.4", "767"): "767  AllowableOneSidednessCurr",
        ("FIX.4.4", "768"): "768  NoTrdRegTimestamps",
        ("FIX.4.4", "769"): "769  TrdRegTimestamp",
        ("FIX.4.4", "770"): "770  TrdRegTimestampType",
        ("FIX.4.4", "771"): "771  TrdRegTimestampOrigin",
        ("FIX.4.4", "772"): "772  ConfirmRefID",
        ("FIX.4.4", "773"): "773  ConfirmType",
        ("FIX.4.4", "774"): "774  ConfirmRejReason",
        ("FIX.4.4", "775"): "775  BookingType",
        ("FIX.4.4", "776"): "776  IndividualAllocRejCode",
        ("FIX.4.4", "777"): "777  SettlInstMsgID",
        ("FIX.4.4", "778"): "778  NoSettlInst",
        ("FIX.4.4", "779"): "779  LastUpdateTime",
        ("FIX.4.4", "780"): "780  AllocSettlInstType",
        ("FIX.4.4", "781"): "781  NoSettlPartyIDs",
        ("FIX.4.4", "782"): "782  SettlPartyID",
        ("FIX.4.4", "783"): "783  SettlPartyIDSource",
        ("FIX.4.4", "784"): "784  SettlPartyRole",
        ("FIX.4.4", "785"): "785  SettlPartySubID",
        ("FIX.4.4", "786"): "786  SettlPartySubIDType",
        ("FIX.4.4", "787"): "787  DlvyInstType",
        ("FIX.4.4", "788"): "788  TerminationType",
        ("FIX.4.4", "789"): "789  NextExpectedMsgSeqNum",
        ("FIX.4.4", "790"): "790  OrdStatusReqID",
        ("FIX.4.4", "791"): "791  SettlInstReqID",
        ("FIX.4.4", "792"): "792  SettlInstReqRejCode",
        ("FIX.4.4", "793"): "793  SecondaryAllocID",
        ("FIX.4.4", "794"): "794  AllocReportType",
        ("FIX.4.4", "795"): "795  AllocReportRefID",
        ("FIX.4.4", "796"): "796  AllocCancReplaceReason",
        ("FIX.4.4", "797"): "797  CopyMsgIndicator",
        ("FIX.4.4", "798"): "798  AllocAccountType",
        ("FIX.4.4", "799"): "799  OrderAvgPx",
        ("FIX.4.4", "800"): "800  OrderBookingQty",
        ("FIX.4.4", "801"): "801  NoSettlPartySubIDs",
        ("FIX.4.4", "802"): "802  NoPartySubIDs",
        ("FIX.4.4", "803"): "803  PartySubIDType",
        ("FIX.4.4", "804"): "804  NoNestedPartySubIDs",
        ("FIX.4.4", "805"): "805  NestedPartySubIDType",
        ("FIX.4.4", "806"): "806  NoNested2PartySubIDs",
        ("FIX.4.4", "807"): "807  Nested2PartySubIDType",
        ("FIX.4.4", "808"): "808  AllocIntermedReqType",
        ("FIX.4.4", "810"): "810  UnderlyingPx",
        ("FIX.4.4", "811"): "811  PriceDelta",
        ("FIX.4.4", "812"): "812  ApplQueueMax",
        ("FIX.4.4", "813"): "813  ApplQueueDepth",
        ("FIX.4.4", "814"): "814  ApplQueueResolution",
        ("FIX.4.4", "815"): "815  ApplQueueAction",
        ("FIX.4.4", "816"): "816  NoAltMDSource",
        ("FIX.4.4", "817"): "817  AltMDSourceID",
        ("FIX.4.4", "818"): "818  SecondaryTradeReportID",
        ("FIX.4.4", "819"): "819  AvgPxIndicator",
        ("FIX.4.4", "820"): "820  TradeLinkID",
        ("FIX.4.4", "821"): "821  OrderInputDevice",
        ("FIX.4.4", "822"): "822  UnderlyingTradingSessionID",
        ("FIX.4.4", "823"): "823  UnderlyingTradingSessionSubID",
        ("FIX.4.4", "824"): "824  TradeLegRefID",
        ("FIX.4.4", "825"): "825  ExchangeRule",
        ("FIX.4.4", "826"): "826  TradeAllocIndicator",
        ("FIX.4.4", "827"): "827  ExpirationCycle",
        ("FIX.4.4", "828"): "828  TrdType",
        ("FIX.4.4", "829"): "829  TrdSubType",
        ("FIX.4.4", "830"): "830  TransferReason",
        ("FIX.4.4", "832"): "832  TotNumAssignmentReports",
        ("FIX.4.4", "833"): "833  AsgnRptID",
        ("FIX.4.4", "834"): "834  ThresholdAmount",
        ("FIX.4.4", "835"): "835  PegMoveType",
        ("FIX.4.4", "836"): "836  PegOffsetType",
        ("FIX.4.4", "837"): "837  PegLimitType",
        ("FIX.4.4", "838"): "838  PegRoundDirection",
        ("FIX.4.4", "839"): "839  PeggedPrice",
        ("FIX.4.4", "840"): "840  PegScope",
        ("FIX.4.4", "841"): "841  DiscretionMoveType",
        ("FIX.4.4", "842"): "842  DiscretionOffsetType",
        ("FIX.4.4", "843"): "843  DiscretionLimitType",
        ("FIX.4.4", "844"): "844  DiscretionRoundDirection",
        ("FIX.4.4", "845"): "845  DiscretionPrice",
        ("FIX.4.4", "846"): "846  DiscretionScope",
        ("FIX.4.4", "847"): "847  TargetStrategy",
        ("FIX.4.4", "848"): "848  TargetStrategyParameters",
        ("FIX.4.4", "849"): "849  ParticipationRate",
        ("FIX.4.4", "850"): "850  TargetStrategyPerformance",
        ("FIX.4.4", "851"): "851  LastLiquidityInd",
        ("FIX.4.4", "852"): "852  PublishTrdIndicator",
        ("FIX.4.4", "853"): "853  ShortSaleReason",
        ("FIX.4.4", "854"): "854  QtyType",
        ("FIX.4.4", "855"): "855  SecondaryTrdType",
        ("FIX.4.4", "856"): "856  TradeReportType",
        ("FIX.4.4", "857"): "857  AllocNoOrdersType",
        ("FIX.4.4", "858"): "858  SharedCommission",
        ("FIX.4.4", "859"): "859  ConfirmReqID",
        ("FIX.4.4", "860"): "860  AvgParPx",
        ("FIX.4.4", "861"): "861  ReportedPx",
        ("FIX.4.4", "862"): "862  NoCapacities",
        ("FIX.4.4", "863"): "863  OrderCapacityQty",
        ("FIX.4.4", "864"): "864  NoEvents",
        ("FIX.4.4", "865"): "865  EventType",
        ("FIX.4.4", "866"): "866  EventDate",
        ("FIX.4.4", "867"): "867  EventPx",
        ("FIX.4.4", "868"): "868  EventText",
        ("FIX.4.4", "869"): "869  PctAtRisk",
        ("FIX.4.4", "870"): "870  NoInstrAttrib",
        ("FIX.4.4", "871"): "871  InstrAttribType",
        ("FIX.4.4", "872"): "872  InstrAttribValue",
        ("FIX.4.4", "873"): "873  DatedDate",
        ("FIX.4.4", "874"): "874  InterestAccrualDate",
        ("FIX.4.4", "875"): "875  CPProgram",
        ("FIX.4.4", "876"): "876  CPRegType",
        ("FIX.4.4", "877"): "877  UnderlyingCPProgram",
        ("FIX.4.4", "878"): "878  UnderlyingCPRegType",
        ("FIX.4.4", "879"): "879  UnderlyingQty",
        ("FIX.4.4", "880"): "880  TrdMatchID",
        ("FIX.4.4", "881"): "881  SecondaryTradeReportRefID",
        ("FIX.4.4", "882"): "882  UnderlyingDirtyPrice",
        ("FIX.4.4", "883"): "883  UnderlyingEndPrice",
        ("FIX.4.4", "884"): "884  UnderlyingStartValue",
        ("FIX.4.4", "885"): "885  UnderlyingCurrentValue",
        ("FIX.4.4", "886"): "886  UnderlyingEndValue",
        ("FIX.4.4", "887"): "887  NoUnderlyingStips",
        ("FIX.4.4", "888"): "888  UnderlyingStipType",
        ("FIX.4.4", "889"): "889  UnderlyingStipValue",
        ("FIX.4.4", "890"): "890  MaturityNetMoney",
        ("FIX.4.4", "891"): "891  MiscFeeBasis",
        ("FIX.4.4", "892"): "892  TotNoAllocs",
        ("FIX.4.4", "893"): "893  LastFragment",
        ("FIX.4.4", "894"): "894  CollReqID",
        ("FIX.4.4", "895"): "895  CollAsgnReason",
        ("FIX.4.4", "896"): "896  CollInquiryQualifier",
        ("FIX.4.4", "897"): "897  NoTrades",
        ("FIX.4.4", "898"): "898  MarginRatio",
        ("FIX.4.4", "899"): "899  MarginExcess",
        ("FIX.4.4", "900"): "900  TotalNetValue",
        ("FIX.4.4", "901"): "901  CashOutstanding",
        ("FIX.4.4", "902"): "902  CollAsgnID",
        ("FIX.4.4", "903"): "903  CollAsgnTransType",
        ("FIX.4.4", "904"): "904  CollRespID",
        ("FIX.4.4", "905"): "905  CollAsgnRespType",
        ("FIX.4.4", "906"): "906  CollAsgnRejectReason",
        ("FIX.4.4", "907"): "907  CollAsgnRefID",
        ("FIX.4.4", "908"): "908  CollRptID",
        ("FIX.4.4", "909"): "909  CollInquiryID",
        ("FIX.4.4", "910"): "910  CollStatus",
        ("FIX.4.4", "911"): "911  TotNumReports",
        ("FIX.4.4", "912"): "912  LastRptRequested",
        ("FIX.4.4", "913"): "913  AgreementDesc",
        ("FIX.4.4", "914"): "914  AgreementID",
        ("FIX.4.4", "915"): "915  AgreementDate",
        ("FIX.4.4", "916"): "916  StartDate",
        ("FIX.4.4", "917"): "917  EndDate",
        ("FIX.4.4", "918"): "918  AgreementCurrency",
        ("FIX.4.4", "919"): "919  DeliveryType",
        ("FIX.4.4", "920"): "920  EndAccruedInterestAmt",
        ("FIX.4.4", "921"): "921  StartCash",
        ("FIX.4.4", "922"): "922  EndCash",
        ("FIX.4.4", "923"): "923  UserRequestID",
        ("FIX.4.4", "924"): "924  UserRequestType",
        ("FIX.4.4", "925"): "925  NewPassword",
        ("FIX.4.4", "926"): "926  UserStatus",
        ("FIX.4.4", "927"): "927  UserStatusText",
        ("FIX.4.4", "928"): "928  StatusValue",
        ("FIX.4.4", "929"): "929  StatusText",
        ("FIX.4.4", "930"): "930  RefCompID",
        ("FIX.4.4", "931"): "931  RefSubID",
        ("FIX.4.4", "932"): "932  NetworkResponseID",
        ("FIX.4.4", "933"): "933  NetworkRequestID",
        ("FIX.4.4", "934"): "934  LastNetworkResponseID",
        ("FIX.4.4", "935"): "935  NetworkRequestType",
        ("FIX.4.4", "936"): "936  NoCompIDs",
        ("FIX.4.4", "937"): "937  NetworkStatusResponseType",
        ("FIX.4.4", "938"): "938  NoCollInquiryQualifier",
        ("FIX.4.4", "939"): "939  TrdRptStatus",
        ("FIX.4.4", "940"): "940  AffirmStatus",
        ("FIX.4.4", "941"): "941  UnderlyingStrikeCurrency",
        ("FIX.4.4", "942"): "942  LegStrikeCurrency",
        ("FIX.4.4", "943"): "943  TimeBracket",
        ("FIX.4.4", "944"): "944  CollAction",
        ("FIX.4.4", "945"): "945  CollInquiryStatus",
        ("FIX.4.4", "946"): "946  CollInquiryResult",
        ("FIX.4.4", "947"): "947  StrikeCurrency",
        ("FIX.4.4", "948"): "948  NoNested3PartyIDs",
        ("FIX.4.4", "949"): "949  Nested3PartyID",
        ("FIX.4.4", "950"): "950  Nested3PartyIDSource",
        ("FIX.4.4", "951"): "951  Nested3PartyRole",
        ("FIX.4.4", "952"): "952  NoNested3PartySubIDs",
        ("FIX.4.4", "953"): "953  Nested3PartySubID",
        ("FIX.4.4", "954"): "954  Nested3PartySubIDType",
        ("FIX.4.4", "955"): "955  LegContractSettlMonth",
        ("FIX.4.4", "956"): "956  LegInterestAccrualDate",
        ("FIX.5.0", "1"): "1    Account",
        ("FIX.5.0", "2"): "2    AdvId",
        ("FIX.5.0", "3"): "3    AdvRefID",
        ("FIX.5.0", "4"): "4    AdvSide",
        ("FIX.5.0", "5"): "5    AdvTransType",
        ("FIX.5.0", "6"): "6    AvgPx",
        ("FIX.5.0", "7"): "7    BeginSeqNo",
        ("FIX.5.0", "8"): "8    BeginString",
        ("FIX.5.0", "9"): "9    BodyLength",
        ("FIX.5.0", "10"): "10   CheckSum",
        ("FIX.5.0", "11"): "11   ClOrdID",
        ("FIX.5.0", "12"): "12   Commission",
        ("FIX.5.0", "13"): "13   CommType",
        ("FIX.5.0", "14"): "14   CumQty",
        ("FIX.5.0", "15"): "15   Currency",
        ("FIX.5.0", "16"): "16   EndSeqNo",
        ("FIX.5.0", "17"): "17   ExecID",
        ("FIX.5.0", "18"): "18   ExecInst",
        ("FIX.5.0", "19"): "19   ExecRefID",
        ("FIX.5.0", "21"): "21   HandlInst",
        ("FIX.5.0", "22"): "22   SecurityIDSource",
        ("FIX.5.0", "23"): "23   IOIID",
        ("FIX.5.0", "25"): "25   IOIQltyInd",
        ("FIX.5.0", "26"): "26   IOIRefID",
        ("FIX.5.0", "27"): "27   IOIQty",
        ("FIX.5.0", "28"): "28   IOITransType",
        ("FIX.5.0", "29"): "29   LastCapacity",
        ("FIX.5.0", "30"): "30   LastMkt",
        ("FIX.5.0", "31"): "31   LastPx",
        ("FIX.5.0", "32"): "32   LastQty",
        ("FIX.5.0", "33"): "33   NoLinesOfText",
        ("FIX.5.0", "34"): "34   MsgSeqNum",
        ("FIX.5.0", "35"): "35   MsgType",
        ("FIX.5.0", "36"): "36   NewSeqNo",
        ("FIX.5.0", "37"): "37   OrderID",
        ("FIX.5.0", "38"): "38   OrderQty",
        ("FIX.5.0", "39"): "39   OrdStatus",
        ("FIX.5.0", "40"): "40   OrdType",
        ("FIX.5.0", "41"): "41   OrigClOrdID",
        ("FIX.5.0", "42"): "42   OrigTime",
        ("FIX.5.0", "43"): "43   PossDupFlag",
        ("FIX.5.0", "44"): "44   Price",
        ("FIX.5.0", "45"): "45   RefSeqNum",
        ("FIX.5.0", "48"): "48   SecurityID",
        ("FIX.5.0", "49"): "49   SenderCompID",
        ("FIX.5.0", "50"): "50   SenderSubID",
        ("FIX.5.0", "52"): "52   SendingTime",
        ("FIX.5.0", "53"): "53   Quantity",
        ("FIX.5.0", "54"): "54   Side",
        ("FIX.5.0", "55"): "55   Symbol",
        ("FIX.5.0", "56"): "56   TargetCompID",
        ("FIX.5.0", "57"): "57   TargetSubID",
        ("FIX.5.0", "58"): "58   Text",
        ("FIX.5.0", "59"): "59   TimeInForce",
        ("FIX.5.0", "60"): "60   TransactTime",
        ("FIX.5.0", "61"): "61   Urgency",
        ("FIX.5.0", "62"): "62   ValidUntilTime",
        ("FIX.5.0", "63"): "63   SettlType",
        ("FIX.5.0", "64"): "64   SettlDate",
        ("FIX.5.0", "65"): "65   SymbolSfx",
        ("FIX.5.0", "66"): "66   ListID",
        ("FIX.5.0", "67"): "67   ListSeqNo",
        ("FIX.5.0", "68"): "68   TotNoOrders",
        ("FIX.5.0", "69"): "69   ListExecInst",
        ("FIX.5.0", "70"): "70   AllocID",
        ("FIX.5.0", "71"): "71   AllocTransType",
        ("FIX.5.0", "72"): "72   RefAllocID",
        ("FIX.5.0", "73"): "73   NoOrders",
        ("FIX.5.0", "74"): "74   AvgPxPrecision",
        ("FIX.5.0", "75"): "75   TradeDate",
        ("FIX.5.0", "77"): "77   PositionEffect",
        ("FIX.5.0", "78"): "78   NoAllocs",
        ("FIX.5.0", "79"): "79   AllocAccount",
        ("FIX.5.0", "80"): "80   AllocQty",
        ("FIX.5.0", "81"): "81   ProcessCode",
        ("FIX.5.0", "82"): "82   NoRpts",
        ("FIX.5.0", "83"): "83   RptSeq",
        ("FIX.5.0", "84"): "84   CxlQty",
        ("FIX.5.0", "85"): "85   NoDlvyInst",
        ("FIX.5.0", "87"): "87   AllocStatus",
        ("FIX.5.0", "88"): "88   AllocRejCode",
        ("FIX.5.0", "89"): "89   Signature",
        ("FIX.5.0", "90"): "90   SecureDataLen",
        ("FIX.5.0", "91"): "91   SecureData",
        ("FIX.5.0", "93"): "93   SignatureLength",
        ("FIX.5.0", "94"): "94   EmailType",
        ("FIX.5.0", "95"): "95   RawDataLength",
        ("FIX.5.0", "96"): "96   RawData",
        ("FIX.5.0", "97"): "97   PossResend",
        ("FIX.5.0", "98"): "98   EncryptMethod",
        ("FIX.5.0", "99"): "99   StopPx",
        ("FIX.5.0", "100"): "100  ExDestination",
        ("FIX.5.0", "102"): "102  CxlRejReason",
        ("FIX.5.0", "103"): "103  OrdRejReason",
        ("FIX.5.0", "104"): "104  IOIQualifier",
        ("FIX.5.0", "106"): "106  Issuer",
        ("FIX.5.0", "107"): "107  SecurityDesc",
        ("FIX.5.0", "108"): "108  HeartBtInt",
        ("FIX.5.0", "110"): "110  MinQty",
        ("FIX.5.0", "111"): "111  MaxFloor",
        ("FIX.5.0", "112"): "112  TestReqID",
        ("FIX.5.0", "113"): "113  ReportToExch",
        ("FIX.5.0", "114"): "114  LocateReqd",
        ("FIX.5.0", "115"): "115  OnBehalfOfCompID",
        ("FIX.5.0", "116"): "116  OnBehalfOfSubID",
        ("FIX.5.0", "117"): "117  QuoteID",
        ("FIX.5.0", "118"): "118  NetMoney",
        ("FIX.5.0", "119"): "119  SettlCurrAmt",
        ("FIX.5.0", "120"): "120  SettlCurrency",
        ("FIX.5.0", "121"): "121  ForexReq",
        ("FIX.5.0", "122"): "122  OrigSendingTime",
        ("FIX.5.0", "123"): "123  GapFillFlag",
        ("FIX.5.0", "124"): "124  NoExecs",
        ("FIX.5.0", "126"): "126  ExpireTime",
        ("FIX.5.0", "127"): "127  DKReason",
        ("FIX.5.0", "128"): "128  DeliverToCompID",
        ("FIX.5.0", "129"): "129  DeliverToSubID",
        ("FIX.5.0", "130"): "130  IOINaturalFlag",
        ("FIX.5.0", "131"): "131  QuoteReqID",
        ("FIX.5.0", "132"): "132  BidPx",
        ("FIX.5.0", "133"): "133  OfferPx",
        ("FIX.5.0", "134"): "134  BidSize",
        ("FIX.5.0", "135"): "135  OfferSize",
        ("FIX.5.0", "136"): "136  NoMiscFees",
        ("FIX.5.0", "137"): "137  MiscFeeAmt",
        ("FIX.5.0", "138"): "138  MiscFeeCurr",
        ("FIX.5.0", "139"): "139  MiscFeeType",
        ("FIX.5.0", "140"): "140  PrevClosePx",
        ("FIX.5.0", "141"): "141  ResetSeqNumFlag",
        ("FIX.5.0", "142"): "142  SenderLocationID",
        ("FIX.5.0", "143"): "143  TargetLocationID",
        ("FIX.5.0", "144"): "144  OnBehalfOfLocationID",
        ("FIX.5.0", "145"): "145  DeliverToLocationID",
        ("FIX.5.0", "146"): "146  NoRelatedSym",
        ("FIX.5.0", "147"): "147  Subject",
        ("FIX.5.0", "148"): "148  Headline",
        ("FIX.5.0", "149"): "149  URLLink",
        ("FIX.5.0", "150"): "150  ExecType",
        ("FIX.5.0", "151"): "151  LeavesQty",
        ("FIX.5.0", "152"): "152  CashOrderQty",
        ("FIX.5.0", "153"): "153  AllocAvgPx",
        ("FIX.5.0", "154"): "154  AllocNetMoney",
        ("FIX.5.0", "155"): "155  SettlCurrFxRate",
        ("FIX.5.0", "156"): "156  SettlCurrFxRateCalc",
        ("FIX.5.0", "157"): "157  NumDaysInterest",
        ("FIX.5.0", "158"): "158  AccruedInterestRate",
        ("FIX.5.0", "159"): "159  AccruedInterestAmt",
        ("FIX.5.0", "160"): "160  SettlInstMode",
        ("FIX.5.0", "161"): "161  AllocText",
        ("FIX.5.0", "162"): "162  SettlInstID",
        ("FIX.5.0", "163"): "163  SettlInstTransType",
        ("FIX.5.0", "164"): "164  EmailThreadID",
        ("FIX.5.0", "165"): "165  SettlInstSource",
        ("FIX.5.0", "167"): "167  SecurityType",
        ("FIX.5.0", "168"): "168  EffectiveTime",
        ("FIX.5.0", "169"): "169  StandInstDbType",
        ("FIX.5.0", "170"): "170  StandInstDbName",
        ("FIX.5.0", "171"): "171  StandInstDbID",
        ("FIX.5.0", "172"): "172  SettlDeliveryType",
        ("FIX.5.0", "188"): "188  BidSpotRate",
        ("FIX.5.0", "189"): "189  BidForwardPoints",
        ("FIX.5.0", "190"): "190  OfferSpotRate",
        ("FIX.5.0", "191"): "191  OfferForwardPoints",
        ("FIX.5.0", "192"): "192  OrderQty2",
        ("FIX.5.0", "193"): "193  SettlDate2",
        ("FIX.5.0", "194"): "194  LastSpotRate",
        ("FIX.5.0", "195"): "195  LastForwardPoints",
        ("FIX.5.0", "196"): "196  AllocLinkID",
        ("FIX.5.0", "197"): "197  AllocLinkType",
        ("FIX.5.0", "198"): "198  SecondaryOrderID",
        ("FIX.5.0", "199"): "199  NoIOIQualifiers",
        ("FIX.5.0", "200"): "200  MaturityMonthYear",
        ("FIX.5.0", "201"): "201  PutOrCall",
        ("FIX.5.0", "202"): "202  StrikePrice",
        ("FIX.5.0", "203"): "203  CoveredOrUncovered",
        ("FIX.5.0", "206"): "206  OptAttribute",
        ("FIX.5.0", "207"): "207  SecurityExchange",
        ("FIX.5.0", "208"): "208  NotifyBrokerOfCredit",
        ("FIX.5.0", "209"): "209  AllocHandlInst",
        ("FIX.5.0", "210"): "210  MaxShow",
        ("FIX.5.0", "211"): "211  PegOffsetValue",
        ("FIX.5.0", "212"): "212  XmlDataLen",
        ("FIX.5.0", "213"): "213  XmlData",
        ("FIX.5.0", "214"): "214  SettlInstRefID",
        ("FIX.5.0", "215"): "215  NoRoutingIDs",
        ("FIX.5.0", "216"): "216  RoutingType",
        ("FIX.5.0", "217"): "217  RoutingID",
        ("FIX.5.0", "218"): "218  Spread",
        ("FIX.5.0", "220"): "220  BenchmarkCurveCurrency",
        ("FIX.5.0", "221"): "221  BenchmarkCurveName",
        ("FIX.5.0", "222"): "222  BenchmarkCurvePoint",
        ("FIX.5.0", "223"): "223  CouponRate",
        ("FIX.5.0", "224"): "224  CouponPaymentDate",
        ("FIX.5.0", "225"): "225  IssueDate",
        ("FIX.5.0", "226"): "226  RepurchaseTerm",
        ("FIX.5.0", "227"): "227  RepurchaseRate",
        ("FIX.5.0", "228"): "228  Factor",
        ("FIX.5.0", "229"): "229  TradeOriginationDate",
        ("FIX.5.0", "230"): "230  ExDate",
        ("FIX.5.0", "231"): "231  ContractMultiplier",
        ("FIX.5.0", "232"): "232  NoStipulations",
        ("FIX.5.0", "233"): "233  StipulationType",
        ("FIX.5.0", "234"): "234  StipulationValue",
        ("FIX.5.0", "235"): "235  YieldType",
        ("FIX.5.0", "236"): "236  Yield",
        ("FIX.5.0", "237"): "237  TotalTakedown",
        ("FIX.5.0", "238"): "238  Concession",
        ("FIX.5.0", "239"): "239  RepoCollateralSecurityType",
        ("FIX.5.0", "240"): "240  RedemptionDate",
        ("FIX.5.0", "241"): "241  UnderlyingCouponPaymentDate",
        ("FIX.5.0", "242"): "242  UnderlyingIssueDate",
        ("FIX.5.0", "243"): "243  UnderlyingRepoCollateralSecurityType",
        ("FIX.5.0", "244"): "244  UnderlyingRepurchaseTerm",
        ("FIX.5.0", "245"): "245  UnderlyingRepurchaseRate",
        ("FIX.5.0", "246"): "246  UnderlyingFactor",
        ("FIX.5.0", "247"): "247  UnderlyingRedemptionDate",
        ("FIX.5.0", "248"): "248  LegCouponPaymentDate",
        ("FIX.5.0", "249"): "249  LegIssueDate",
        ("FIX.5.0", "250"): "250  LegRepoCollateralSecurityType",
        ("FIX.5.0", "251"): "251  LegRepurchaseTerm",
        ("FIX.5.0", "252"): "252  LegRepurchaseRate",
        ("FIX.5.0", "253"): "253  LegFactor",
        ("FIX.5.0", "254"): "254  LegRedemptionDate",
        ("FIX.5.0", "255"): "255  CreditRating",
        ("FIX.5.0", "256"): "256  UnderlyingCreditRating",
        ("FIX.5.0", "257"): "257  LegCreditRating",
        ("FIX.5.0", "258"): "258  TradedFlatSwitch",
        ("FIX.5.0", "259"): "259  BasisFeatureDate",
        ("FIX.5.0", "260"): "260  BasisFeaturePrice",
        ("FIX.5.0", "262"): "262  MDReqID",
        ("FIX.5.0", "263"): "263  SubscriptionRequestType",
        ("FIX.5.0", "264"): "264  MarketDepth",
        ("FIX.5.0", "265"): "265  MDUpdateType",
        ("FIX.5.0", "266"): "266  AggregatedBook",
        ("FIX.5.0", "267"): "267  NoMDEntryTypes",
        ("FIX.5.0", "268"): "268  NoMDEntries",
        ("FIX.5.0", "269"): "269  MDEntryType",
        ("FIX.5.0", "270"): "270  MDEntryPx",
        ("FIX.5.0", "271"): "271  MDEntrySize",
        ("FIX.5.0", "272"): "272  MDEntryDate",
        ("FIX.5.0", "273"): "273  MDEntryTime",
        ("FIX.5.0", "274"): "274  TickDirection",
        ("FIX.5.0", "275"): "275  MDMkt",
        ("FIX.5.0", "276"): "276  QuoteCondition",
        ("FIX.5.0", "277"): "277  TradeCondition",
        ("FIX.5.0", "278"): "278  MDEntryID",
        ("FIX.5.0", "279"): "279  MDUpdateAction",
        ("FIX.5.0", "280"): "280  MDEntryRefID",
        ("FIX.5.0", "281"): "281  MDReqRejReason",
        ("FIX.5.0", "282"): "282  MDEntryOriginator",
        ("FIX.5.0", "283"): "283  LocationID",
        ("FIX.5.0", "284"): "284  DeskID",
        ("FIX.5.0", "285"): "285  DeleteReason",
        ("FIX.5.0", "286"): "286  OpenCloseSettlFlag",
        ("FIX.5.0", "287"): "287  SellerDays",
        ("FIX.5.0", "288"): "288  MDEntryBuyer",
        ("FIX.5.0", "289"): "289  MDEntrySeller",
        ("FIX.5.0", "290"): "290  MDEntryPositionNo",
        ("FIX.5.0", "291"): "291  FinancialStatus",
        ("FIX.5.0", "292"): "292  CorporateAction",
        ("FIX.5.0", "293"): "293  DefBidSize",
        ("FIX.5.0", "294"): "294  DefOfferSize",
        ("FIX.5.0", "295"): "295  NoQuoteEntries",
        ("FIX.5.0", "296"): "296  NoQuoteSets",
        ("FIX.5.0", "297"): "297  QuoteStatus",
        ("FIX.5.0", "298"): "298  QuoteCancelType",
        ("FIX.5.0", "299"): "299  QuoteEntryID",
        ("FIX.5.0", "300"): "300  QuoteRejectReason",
        ("FIX.5.0", "301"): "301  QuoteResponseLevel",
        ("FIX.5.0", "302"): "302  QuoteSetID",
        ("FIX.5.0", "303"): "303  QuoteRequestType",
        ("FIX.5.0", "304"): "304  TotNoQuoteEntries",
        ("FIX.5.0", "305"): "305  UnderlyingSecurityIDSource",
        ("FIX.5.0", "306"): "306  UnderlyingIssuer",
        ("FIX.5.0", "307"): "307  UnderlyingSecurityDesc",
        ("FIX.5.0", "308"): "308  UnderlyingSecurityExchange",
        ("FIX.5.0", "309"): "309  UnderlyingSecurityID",
        ("FIX.5.0", "310"): "310  UnderlyingSecurityType",
        ("FIX.5.0", "311"): "311  UnderlyingSymbol",
        ("FIX.5.0", "312"): "312  UnderlyingSymbolSfx",
        ("FIX.5.0", "313"): "313  UnderlyingMaturityMonthYear",
        ("FIX.5.0", "315"): "315  UnderlyingPutOrCall",
        ("FIX.5.0", "316"): "316  UnderlyingStrikePrice",
        ("FIX.5.0", "317"): "317  UnderlyingOptAttribute",
        ("FIX.5.0", "318"): "318  UnderlyingCurrency",
        ("FIX.5.0", "320"): "320  SecurityReqID",
        ("FIX.5.0", "321"): "321  SecurityRequestType",
        ("FIX.5.0", "322"): "322  SecurityResponseID",
        ("FIX.5.0", "323"): "323  SecurityResponseType",
        ("FIX.5.0", "324"): "324  SecurityStatusReqID",
        ("FIX.5.0", "325"): "325  UnsolicitedIndicator",
        ("FIX.5.0", "326"): "326  SecurityTradingStatus",
        ("FIX.5.0", "327"): "327  HaltReason",
        ("FIX.5.0", "328"): "328  InViewOfCommon",
        ("FIX.5.0", "329"): "329  DueToRelated",
        ("FIX.5.0", "330"): "330  BuyVolume",
        ("FIX.5.0", "331"): "331  SellVolume",
        ("FIX.5.0", "332"): "332  HighPx",
        ("FIX.5.0", "333"): "333  LowPx",
        ("FIX.5.0", "334"): "334  Adjustment",
        ("FIX.5.0", "335"): "335  TradSesReqID",
        ("FIX.5.0", "336"): "336  TradingSessionID",
        ("FIX.5.0", "337"): "337  ContraTrader",
        ("FIX.5.0", "338"): "338  TradSesMethod",
        ("FIX.5.0", "339"): "339  TradSesMode",
        ("FIX.5.0", "340"): "340  TradSesStatus",
        ("FIX.5.0", "341"): "341  TradSesStartTime",
        ("FIX.5.0", "342"): "342  TradSesOpenTime",
        ("FIX.5.0", "343"): "343  TradSesPreCloseTime",
        ("FIX.5.0", "344"): "344  TradSesCloseTime",
        ("FIX.5.0", "345"): "345  TradSesEndTime",
        ("FIX.5.0", "346"): "346  NumberOfOrders",
        ("FIX.5.0", "347"): "347  MessageEncoding",
        ("FIX.5.0", "348"): "348  EncodedIssuerLen",
        ("FIX.5.0", "349"): "349  EncodedIssuer",
        ("FIX.5.0", "350"): "350  EncodedSecurityDescLen",
        ("FIX.5.0", "351"): "351  EncodedSecurityDesc",
        ("FIX.5.0", "352"): "352  EncodedListExecInstLen",
        ("FIX.5.0", "353"): "353  EncodedListExecInst",
        ("FIX.5.0", "354"): "354  EncodedTextLen",
        ("FIX.5.0", "355"): "355  EncodedText",
        ("FIX.5.0", "356"): "356  EncodedSubjectLen",
        ("FIX.5.0", "357"): "357  EncodedSubject",
        ("FIX.5.0", "358"): "358  EncodedHeadlineLen",
        ("FIX.5.0", "359"): "359  EncodedHeadline",
        ("FIX.5.0", "360"): "360  EncodedAllocTextLen",
        ("FIX.5.0", "361"): "361  EncodedAllocText",
        ("FIX.5.0", "362"): "362  EncodedUnderlyingIssuerLen",
        ("FIX.5.0", "363"): "363  EncodedUnderlyingIssuer",
        ("FIX.5.0", "364"): "364  EncodedUnderlyingSecurityDescLen",
        ("FIX.5.0", "365"): "365  EncodedUnderlyingSecurityDesc",
        ("FIX.5.0", "366"): "366  AllocPrice",
        ("FIX.5.0", "367"): "367  QuoteSetValidUntilTime",
        ("FIX.5.0", "368"): "368  QuoteEntryRejectReason",
        ("FIX.5.0", "369"): "369  LastMsgSeqNumProcessed",
        ("FIX.5.0", "371"): "371  RefTagID",
        ("FIX.5.0", "372"): "372  RefMsgType",
        ("FIX.5.0", "373"): "373  SessionRejectReason",
        ("FIX.5.0", "374"): "374  BidRequestTransType",
        ("FIX.5.0", "375"): "375  ContraBroker",
        ("FIX.5.0", "376"): "376  ComplianceID",
        ("FIX.5.0", "377"): "377  SolicitedFlag",
        ("FIX.5.0", "378"): "378  ExecRestatementReason",
        ("FIX.5.0", "379"): "379  BusinessRejectRefID",
        ("FIX.5.0", "380"): "380  BusinessRejectReason",
        ("FIX.5.0", "381"): "381  GrossTradeAmt",
        ("FIX.5.0", "382"): "382  NoContraBrokers",
        ("FIX.5.0", "383"): "383  MaxMessageSize",
        ("FIX.5.0", "384"): "384  NoMsgTypes",
        ("FIX.5.0", "385"): "385  MsgDirection",
        ("FIX.5.0", "386"): "386  NoTradingSessions",
        ("FIX.5.0", "387"): "387  TotalVolumeTraded",
        ("FIX.5.0", "388"): "388  DiscretionInst",
        ("FIX.5.0", "389"): "389  DiscretionOffsetValue",
        ("FIX.5.0", "390"): "390  BidID",
        ("FIX.5.0", "391"): "391  ClientBidID",
        ("FIX.5.0", "392"): "392  ListName",
        ("FIX.5.0", "393"): "393  TotNoRelatedSym",
        ("FIX.5.0", "394"): "394  BidType",
        ("FIX.5.0", "395"): "395  NumTickets",
        ("FIX.5.0", "396"): "396  SideValue1",
        ("FIX.5.0", "397"): "397  SideValue2",
        ("FIX.5.0", "398"): "398  NoBidDescriptors",
        ("FIX.5.0", "399"): "399  BidDescriptorType",
        ("FIX.5.0", "400"): "400  BidDescriptor",
        ("FIX.5.0", "401"): "401  SideValueInd",
        ("FIX.5.0", "402"): "402  LiquidityPctLow",
        ("FIX.5.0", "403"): "403  LiquidityPctHigh",
        ("FIX.5.0", "404"): "404  LiquidityValue",
        ("FIX.5.0", "405"): "405  EFPTrackingError",
        ("FIX.5.0", "406"): "406  FairValue",
        ("FIX.5.0", "407"): "407  OutsideIndexPct",
        ("FIX.5.0", "408"): "408  ValueOfFutures",
        ("FIX.5.0", "409"): "409  LiquidityIndType",
        ("FIX.5.0", "410"): "410  WtAverageLiquidity",
        ("FIX.5.0", "411"): "411  ExchangeForPhysical",
        ("FIX.5.0", "412"): "412  OutMainCntryUIndex",
        ("FIX.5.0", "413"): "413  CrossPercent",
        ("FIX.5.0", "414"): "414  ProgRptReqs",
        ("FIX.5.0", "415"): "415  ProgPeriodInterval",
        ("FIX.5.0", "416"): "416  IncTaxInd",
        ("FIX.5.0", "417"): "417  NumBidders",
        ("FIX.5.0", "418"): "418  BidTradeType",
        ("FIX.5.0", "419"): "419  BasisPxType",
        ("FIX.5.0", "420"): "420  NoBidComponents",
        ("FIX.5.0", "421"): "421  Country",
        ("FIX.5.0", "422"): "422  TotNoStrikes",
        ("FIX.5.0", "423"): "423  PriceType",
        ("FIX.5.0", "424"): "424  DayOrderQty",
        ("FIX.5.0", "425"): "425  DayCumQty",
        ("FIX.5.0", "426"): "426  DayAvgPx",
        ("FIX.5.0", "427"): "427  GTBookingInst",
        ("FIX.5.0", "428"): "428  NoStrikes",
        ("FIX.5.0", "429"): "429  ListStatusType",
        ("FIX.5.0", "430"): "430  NetGrossInd",
        ("FIX.5.0", "431"): "431  ListOrderStatus",
        ("FIX.5.0", "432"): "432  ExpireDate",
        ("FIX.5.0", "433"): "433  ListExecInstType",
        ("FIX.5.0", "434"): "434  CxlRejResponseTo",
        ("FIX.5.0", "435"): "435  UnderlyingCouponRate",
        ("FIX.5.0", "436"): "436  UnderlyingContractMultiplier",
        ("FIX.5.0", "437"): "437  ContraTradeQty",
        ("FIX.5.0", "438"): "438  ContraTradeTime",
        ("FIX.5.0", "441"): "441  LiquidityNumSecurities",
        ("FIX.5.0", "442"): "442  MultiLegReportingType",
        ("FIX.5.0", "443"): "443  StrikeTime",
        ("FIX.5.0", "444"): "444  ListStatusText",
        ("FIX.5.0", "445"): "445  EncodedListStatusTextLen",
        ("FIX.5.0", "446"): "446  EncodedListStatusText",
        ("FIX.5.0", "447"): "447  PartyIDSource",
        ("FIX.5.0", "448"): "448  PartyID",
        ("FIX.5.0", "451"): "451  NetChgPrevDay",
        ("FIX.5.0", "452"): "452  PartyRole",
        ("FIX.5.0", "453"): "453  NoPartyIDs",
        ("FIX.5.0", "454"): "454  NoSecurityAltID",
        ("FIX.5.0", "455"): "455  SecurityAltID",
        ("FIX.5.0", "456"): "456  SecurityAltIDSource",
        ("FIX.5.0", "457"): "457  NoUnderlyingSecurityAltID",
        ("FIX.5.0", "458"): "458  UnderlyingSecurityAltID",
        ("FIX.5.0", "459"): "459  UnderlyingSecurityAltIDSource",
        ("FIX.5.0", "460"): "460  Product",
        ("FIX.5.0", "461"): "461  CFICode",
        ("FIX.5.0", "462"): "462  UnderlyingProduct",
        ("FIX.5.0", "463"): "463  UnderlyingCFICode",
        ("FIX.5.0", "464"): "464  TestMessageIndicator",
        ("FIX.5.0", "466"): "466  BookingRefID",
        ("FIX.5.0", "467"): "467  IndividualAllocID",
        ("FIX.5.0", "468"): "468  RoundingDirection",
        ("FIX.5.0", "469"): "469  RoundingModulus",
        ("FIX.5.0", "470"): "470  CountryOfIssue",
        ("FIX.5.0", "471"): "471  StateOrProvinceOfIssue",
        ("FIX.5.0", "472"): "472  LocaleOfIssue",
        ("FIX.5.0", "473"): "473  NoRegistDtls",
        ("FIX.5.0", "474"): "474  MailingDtls",
        ("FIX.5.0", "475"): "475  InvestorCountryOfResidence",
        ("FIX.5.0", "476"): "476  PaymentRef",
        ("FIX.5.0", "477"): "477  DistribPaymentMethod",
        ("FIX.5.0", "478"): "478  CashDistribCurr",
        ("FIX.5.0", "479"): "479  CommCurrency",
        ("FIX.5.0", "480"): "480  CancellationRights",
        ("FIX.5.0", "481"): "481  MoneyLaunderingStatus",
        ("FIX.5.0", "482"): "482  MailingInst",
        ("FIX.5.0", "483"): "483  TransBkdTime",
        ("FIX.5.0", "484"): "484  ExecPriceType",
        ("FIX.5.0", "485"): "485  ExecPriceAdjustment",
        ("FIX.5.0", "486"): "486  DateOfBirth",
        ("FIX.5.0", "487"): "487  TradeReportTransType",
        ("FIX.5.0", "488"): "488  CardHolderName",
        ("FIX.5.0", "489"): "489  CardNumber",
        ("FIX.5.0", "490"): "490  CardExpDate",
        ("FIX.5.0", "491"): "491  CardIssNum",
        ("FIX.5.0", "492"): "492  PaymentMethod",
        ("FIX.5.0", "493"): "493  RegistAcctType",
        ("FIX.5.0", "494"): "494  Designation",
        ("FIX.5.0", "495"): "495  TaxAdvantageType",
        ("FIX.5.0", "496"): "496  RegistRejReasonText",
        ("FIX.5.0", "497"): "497  FundRenewWaiv",
        ("FIX.5.0", "498"): "498  CashDistribAgentName",
        ("FIX.5.0", "499"): "499  CashDistribAgentCode",
        ("FIX.5.0", "500"): "500  CashDistribAgentAcctNumber",
        ("FIX.5.0", "501"): "501  CashDistribPayRef",
        ("FIX.5.0", "502"): "502  CashDistribAgentAcctName",
        ("FIX.5.0", "503"): "503  CardStartDate",
        ("FIX.5.0", "504"): "504  PaymentDate",
        ("FIX.5.0", "505"): "505  PaymentRemitterID",
        ("FIX.5.0", "506"): "506  RegistStatus",
        ("FIX.5.0", "507"): "507  RegistRejReasonCode",
        ("FIX.5.0", "508"): "508  RegistRefID",
        ("FIX.5.0", "509"): "509  RegistDtls",
        ("FIX.5.0", "510"): "510  NoDistribInsts",
        ("FIX.5.0", "511"): "511  RegistEmail",
        ("FIX.5.0", "512"): "512  DistribPercentage",
        ("FIX.5.0", "513"): "513  RegistID",
        ("FIX.5.0", "514"): "514  RegistTransType",
        ("FIX.5.0", "515"): "515  ExecValuationPoint",
        ("FIX.5.0", "516"): "516  OrderPercent",
        ("FIX.5.0", "517"): "517  OwnershipType",
        ("FIX.5.0", "518"): "518  NoContAmts",
        ("FIX.5.0", "519"): "519  ContAmtType",
        ("FIX.5.0", "520"): "520  ContAmtValue",
        ("FIX.5.0", "521"): "521  ContAmtCurr",
        ("FIX.5.0", "522"): "522  OwnerType",
        ("FIX.5.0", "523"): "523  PartySubID",
        ("FIX.5.0", "524"): "524  NestedPartyID",
        ("FIX.5.0", "525"): "525  NestedPartyIDSource",
        ("FIX.5.0", "526"): "526  SecondaryClOrdID",
        ("FIX.5.0", "527"): "527  SecondaryExecID",
        ("FIX.5.0", "528"): "528  OrderCapacity",
        ("FIX.5.0", "529"): "529  OrderRestrictions",
        ("FIX.5.0", "530"): "530  MassCancelRequestType",
        ("FIX.5.0", "531"): "531  MassCancelResponse",
        ("FIX.5.0", "532"): "532  MassCancelRejectReason",
        ("FIX.5.0", "533"): "533  TotalAffectedOrders",
        ("FIX.5.0", "534"): "534  NoAffectedOrders",
        ("FIX.5.0", "535"): "535  AffectedOrderID",
        ("FIX.5.0", "536"): "536  AffectedSecondaryOrderID",
        ("FIX.5.0", "537"): "537  QuoteType",
        ("FIX.5.0", "538"): "538  NestedPartyRole",
        ("FIX.5.0", "539"): "539  NoNestedPartyIDs",
        ("FIX.5.0", "540"): "540  TotalAccruedInterestAmt",
        ("FIX.5.0", "541"): "541  MaturityDate",
        ("FIX.5.0", "542"): "542  UnderlyingMaturityDate",
        ("FIX.5.0", "543"): "543  InstrRegistry",
        ("FIX.5.0", "544"): "544  CashMargin",
        ("FIX.5.0", "545"): "545  NestedPartySubID",
        ("FIX.5.0", "546"): "546  Scope",
        ("FIX.5.0", "547"): "547  MDImplicitDelete",
        ("FIX.5.0", "548"): "548  CrossID",
        ("FIX.5.0", "549"): "549  CrossType",
        ("FIX.5.0", "550"): "550  CrossPrioritization",
        ("FIX.5.0", "551"): "551  OrigCrossID",
        ("FIX.5.0", "552"): "552  NoSides",
        ("FIX.5.0", "553"): "553  Username",
        ("FIX.5.0", "554"): "554  Password",
        ("FIX.5.0", "555"): "555  NoLegs",
        ("FIX.5.0", "556"): "556  LegCurrency",
        ("FIX.5.0", "557"): "557  TotNoSecurityTypes",
        ("FIX.5.0", "558"): "558  NoSecurityTypes",
        ("FIX.5.0", "559"): "559  SecurityListRequestType",
        ("FIX.5.0", "560"): "560  SecurityRequestResult",
        ("FIX.5.0", "561"): "561  RoundLot",
        ("FIX.5.0", "562"): "562  MinTradeVol",
        ("FIX.5.0", "563"): "563  MultiLegRptTypeReq",
        ("FIX.5.0", "564"): "564  LegPositionEffect",
        ("FIX.5.0", "565"): "565  LegCoveredOrUncovered",
        ("FIX.5.0", "566"): "566  LegPrice",
        ("FIX.5.0", "567"): "567  TradSesStatusRejReason",
        ("FIX.5.0", "568"): "568  TradeRequestID",
        ("FIX.5.0", "569"): "569  TradeRequestType",
        ("FIX.5.0", "570"): "570  PreviouslyReported",
        ("FIX.5.0", "571"): "571  TradeReportID",
        ("FIX.5.0", "572"): "572  TradeReportRefID",
        ("FIX.5.0", "573"): "573  MatchStatus",
        ("FIX.5.0", "574"): "574  MatchType",
        ("FIX.5.0", "575"): "575  OddLot",
        ("FIX.5.0", "576"): "576  NoClearingInstructions",
        ("FIX.5.0", "577"): "577  ClearingInstruction",
        ("FIX.5.0", "578"): "578  TradeInputSource",
        ("FIX.5.0", "579"): "579  TradeInputDevice",
        ("FIX.5.0", "580"): "580  NoDates",
        ("FIX.5.0", "581"): "581  AccountType",
        ("FIX.5.0", "582"): "582  CustOrderCapacity",
        ("FIX.5.0", "583"): "583  ClOrdLinkID",
        ("FIX.5.0", "584"): "584  MassStatusReqID",
        ("FIX.5.0", "585"): "585  MassStatusReqType",
        ("FIX.5.0", "586"): "586  OrigOrdModTime",
        ("FIX.5.0", "587"): "587  LegSettlType",
        ("FIX.5.0", "588"): "588  LegSettlDate",
        ("FIX.5.0", "589"): "589  DayBookingInst",
        ("FIX.5.0", "590"): "590  BookingUnit",
        ("FIX.5.0", "591"): "591  PreallocMethod",
        ("FIX.5.0", "592"): "592  UnderlyingCountryOfIssue",
        ("FIX.5.0", "593"): "593  UnderlyingStateOrProvinceOfIssue",
        ("FIX.5.0", "594"): "594  UnderlyingLocaleOfIssue",
        ("FIX.5.0", "595"): "595  UnderlyingInstrRegistry",
        ("FIX.5.0", "596"): "596  LegCountryOfIssue",
        ("FIX.5.0", "597"): "597  LegStateOrProvinceOfIssue",
        ("FIX.5.0", "598"): "598  LegLocaleOfIssue",
        ("FIX.5.0", "599"): "599  LegInstrRegistry",
        ("FIX.5.0", "600"): "600  LegSymbol",
        ("FIX.5.0", "601"): "601  LegSymbolSfx",
        ("FIX.5.0", "602"): "602  LegSecurityID",
        ("FIX.5.0", "603"): "603  LegSecurityIDSource",
        ("FIX.5.0", "604"): "604  NoLegSecurityAltID",
        ("FIX.5.0", "605"): "605  LegSecurityAltID",
        ("FIX.5.0", "606"): "606  LegSecurityAltIDSource",
        ("FIX.5.0", "607"): "607  LegProduct",
        ("FIX.5.0", "608"): "608  LegCFICode",
        ("FIX.5.0", "609"): "609  LegSecurityType",
        ("FIX.5.0", "610"): "610  LegMaturityMonthYear",
        ("FIX.5.0", "611"): "611  LegMaturityDate",
        ("FIX.5.0", "612"): "612  LegStrikePrice",
        ("FIX.5.0", "613"): "613  LegOptAttribute",
        ("FIX.5.0", "614"): "614  LegContractMultiplier",
        ("FIX.5.0", "615"): "615  LegCouponRate",
        ("FIX.5.0", "616"): "616  LegSecurityExchange",
        ("FIX.5.0", "617"): "617  LegIssuer",
        ("FIX.5.0", "618"): "618  EncodedLegIssuerLen",
        ("FIX.5.0", "619"): "619  EncodedLegIssuer",
        ("FIX.5.0", "620"): "620  LegSecurityDesc",
        ("FIX.5.0", "621"): "621  EncodedLegSecurityDescLen",
        ("FIX.5.0", "622"): "622  EncodedLegSecurityDesc",
        ("FIX.5.0", "623"): "623  LegRatioQty",
        ("FIX.5.0", "624"): "624  LegSide",
        ("FIX.5.0", "625"): "625  TradingSessionSubID",
        ("FIX.5.0", "626"): "626  AllocType",
        ("FIX.5.0", "627"): "627  NoHops",
        ("FIX.5.0", "628"): "628  HopCompID",
        ("FIX.5.0", "629"): "629  HopSendingTime",
        ("FIX.5.0", "630"): "630  HopRefID",
        ("FIX.5.0", "631"): "631  MidPx",
        ("FIX.5.0", "632"): "632  BidYield",
        ("FIX.5.0", "633"): "633  MidYield",
        ("FIX.5.0", "634"): "634  OfferYield",
        ("FIX.5.0", "635"): "635  ClearingFeeIndicator",
        ("FIX.5.0", "636"): "636  WorkingIndicator",
        ("FIX.5.0", "637"): "637  LegLastPx",
        ("FIX.5.0", "638"): "638  PriorityIndicator",
        ("FIX.5.0", "639"): "639  PriceImprovement",
        ("FIX.5.0", "640"): "640  Price2",
        ("FIX.5.0", "641"): "641  LastForwardPoints2",
        ("FIX.5.0", "642"): "642  BidForwardPoints2",
        ("FIX.5.0", "643"): "643  OfferForwardPoints2",
        ("FIX.5.0", "644"): "644  RFQReqID",
        ("FIX.5.0", "645"): "645  MktBidPx",
        ("FIX.5.0", "646"): "646  MktOfferPx",
        ("FIX.5.0", "647"): "647  MinBidSize",
        ("FIX.5.0", "648"): "648  MinOfferSize",
        ("FIX.5.0", "649"): "649  QuoteStatusReqID",
        ("FIX.5.0", "650"): "650  LegalConfirm",
        ("FIX.5.0", "651"): "651  UnderlyingLastPx",
        ("FIX.5.0", "652"): "652  UnderlyingLastQty",
        ("FIX.5.0", "654"): "654  LegRefID",
        ("FIX.5.0", "655"): "655  ContraLegRefID",
        ("FIX.5.0", "656"): "656  SettlCurrBidFxRate",
        ("FIX.5.0", "657"): "657  SettlCurrOfferFxRate",
        ("FIX.5.0", "658"): "658  QuoteRequestRejectReason",
        ("FIX.5.0", "659"): "659  SideComplianceID",
        ("FIX.5.0", "660"): "660  AcctIDSource",
        ("FIX.5.0", "661"): "661  AllocAcctIDSource",
        ("FIX.5.0", "662"): "662  BenchmarkPrice",
        ("FIX.5.0", "663"): "663  BenchmarkPriceType",
        ("FIX.5.0", "664"): "664  ConfirmID",
        ("FIX.5.0", "665"): "665  ConfirmStatus",
        ("FIX.5.0", "666"): "666  ConfirmTransType",
        ("FIX.5.0", "667"): "667  ContractSettlMonth",
        ("FIX.5.0", "668"): "668  DeliveryForm",
        ("FIX.5.0", "669"): "669  LastParPx",
        ("FIX.5.0", "670"): "670  NoLegAllocs",
        ("FIX.5.0", "671"): "671  LegAllocAccount",
        ("FIX.5.0", "672"): "672  LegIndividualAllocID",
        ("FIX.5.0", "673"): "673  LegAllocQty",
        ("FIX.5.0", "674"): "674  LegAllocAcctIDSource",
        ("FIX.5.0", "675"): "675  LegSettlCurrency",
        ("FIX.5.0", "676"): "676  LegBenchmarkCurveCurrency",
        ("FIX.5.0", "677"): "677  LegBenchmarkCurveName",
        ("FIX.5.0", "678"): "678  LegBenchmarkCurvePoint",
        ("FIX.5.0", "679"): "679  LegBenchmarkPrice",
        ("FIX.5.0", "680"): "680  LegBenchmarkPriceType",
        ("FIX.5.0", "681"): "681  LegBidPx",
        ("FIX.5.0", "682"): "682  LegIOIQty",
        ("FIX.5.0", "683"): "683  NoLegStipulations",
        ("FIX.5.0", "684"): "684  LegOfferPx",
        ("FIX.5.0", "685"): "685  LegOrderQty",
        ("FIX.5.0", "686"): "686  LegPriceType",
        ("FIX.5.0", "687"): "687  LegQty",
        ("FIX.5.0", "688"): "688  LegStipulationType",
        ("FIX.5.0", "689"): "689  LegStipulationValue",
        ("FIX.5.0", "690"): "690  LegSwapType",
        ("FIX.5.0", "691"): "691  Pool",
        ("FIX.5.0", "692"): "692  QuotePriceType",
        ("FIX.5.0", "693"): "693  QuoteRespID",
        ("FIX.5.0", "694"): "694  QuoteRespType",
        ("FIX.5.0", "695"): "695  QuoteQualifier",
        ("FIX.5.0", "696"): "696  YieldRedemptionDate",
        ("FIX.5.0", "697"): "697  YieldRedemptionPrice",
        ("FIX.5.0", "698"): "698  YieldRedemptionPriceType",
        ("FIX.5.0", "699"): "699  BenchmarkSecurityID",
        ("FIX.5.0", "700"): "700  ReversalIndicator",
        ("FIX.5.0", "701"): "701  YieldCalcDate",
        ("FIX.5.0", "702"): "702  NoPositions",
        ("FIX.5.0", "703"): "703  PosType",
        ("FIX.5.0", "704"): "704  LongQty",
        ("FIX.5.0", "705"): "705  ShortQty",
        ("FIX.5.0", "706"): "706  PosQtyStatus",
        ("FIX.5.0", "707"): "707  PosAmtType",
        ("FIX.5.0", "708"): "708  PosAmt",
        ("FIX.5.0", "709"): "709  PosTransType",
        ("FIX.5.0", "710"): "710  PosReqID",
        ("FIX.5.0", "711"): "711  NoUnderlyings",
        ("FIX.5.0", "712"): "712  PosMaintAction",
        ("FIX.5.0", "713"): "713  OrigPosReqRefID",
        ("FIX.5.0", "714"): "714  PosMaintRptRefID",
        ("FIX.5.0", "715"): "715  ClearingBusinessDate",
        ("FIX.5.0", "716"): "716  SettlSessID",
        ("FIX.5.0", "717"): "717  SettlSessSubID",
        ("FIX.5.0", "718"): "718  AdjustmentType",
        ("FIX.5.0", "719"): "719  ContraryInstructionIndicator",
        ("FIX.5.0", "720"): "720  PriorSpreadIndicator",
        ("FIX.5.0", "721"): "721  PosMaintRptID",
        ("FIX.5.0", "722"): "722  PosMaintStatus",
        ("FIX.5.0", "723"): "723  PosMaintResult",
        ("FIX.5.0", "724"): "724  PosReqType",
        ("FIX.5.0", "725"): "725  ResponseTransportType",
        ("FIX.5.0", "726"): "726  ResponseDestination",
        ("FIX.5.0", "727"): "727  TotalNumPosReports",
        ("FIX.5.0", "728"): "728  PosReqResult",
        ("FIX.5.0", "729"): "729  PosReqStatus",
        ("FIX.5.0", "730"): "730  SettlPrice",
        ("FIX.5.0", "731"): "731  SettlPriceType",
        ("FIX.5.0", "732"): "732  UnderlyingSettlPrice",
        ("FIX.5.0", "733"): "733  UnderlyingSettlPriceType",
        ("FIX.5.0", "734"): "734  PriorSettlPrice",
        ("FIX.5.0", "735"): "735  NoQuoteQualifiers",
        ("FIX.5.0", "736"): "736  AllocSettlCurrency",
        ("FIX.5.0", "737"): "737  AllocSettlCurrAmt",
        ("FIX.5.0", "738"): "738  InterestAtMaturity",
        ("FIX.5.0", "739"): "739  LegDatedDate",
        ("FIX.5.0", "740"): "740  LegPool",
        ("FIX.5.0", "741"): "741  AllocInterestAtMaturity",
        ("FIX.5.0", "742"): "742  AllocAccruedInterestAmt",
        ("FIX.5.0", "743"): "743  DeliveryDate",
        ("FIX.5.0", "744"): "744  AssignmentMethod",
        ("FIX.5.0", "745"): "745  AssignmentUnit",
        ("FIX.5.0", "746"): "746  OpenInterest",
        ("FIX.5.0", "747"): "747  ExerciseMethod",
        ("FIX.5.0", "748"): "748  TotNumTradeReports",
        ("FIX.5.0", "749"): "749  TradeRequestResult",
        ("FIX.5.0", "750"): "750  TradeRequestStatus",
        ("FIX.5.0", "751"): "751  TradeReportRejectReason",
        ("FIX.5.0", "752"): "752  SideMultiLegReportingType",
        ("FIX.5.0", "753"): "753  NoPosAmt",
        ("FIX.5.0", "754"): "754  AutoAcceptIndicator",
        ("FIX.5.0", "755"): "755  AllocReportID",
        ("FIX.5.0", "756"): "756  NoNested2PartyIDs",
        ("FIX.5.0", "757"): "757  Nested2PartyID",
        ("FIX.5.0", "758"): "758  Nested2PartyIDSource",
        ("FIX.5.0", "759"): "759  Nested2PartyRole",
        ("FIX.5.0", "760"): "760  Nested2PartySubID",
        ("FIX.5.0", "761"): "761  BenchmarkSecurityIDSource",
        ("FIX.5.0", "762"): "762  SecuritySubType",
        ("FIX.5.0", "763"): "763  UnderlyingSecuritySubType",
        ("FIX.5.0", "764"): "764  LegSecuritySubType",
        ("FIX.5.0", "765"): "765  AllowableOneSidednessPct",
        ("FIX.5.0", "766"): "766  AllowableOneSidednessValue",
        ("FIX.5.0", "767"): "767  AllowableOneSidednessCurr",
        ("FIX.5.0", "768"): "768  NoTrdRegTimestamps",
        ("FIX.5.0", "769"): "769  TrdRegTimestamp",
        ("FIX.5.0", "770"): "770  TrdRegTimestampType",
        ("FIX.5.0", "771"): "771  TrdRegTimestampOrigin",
        ("FIX.5.0", "772"): "772  ConfirmRefID",
        ("FIX.5.0", "773"): "773  ConfirmType",
        ("FIX.5.0", "774"): "774  ConfirmRejReason",
        ("FIX.5.0", "775"): "775  BookingType",
        ("FIX.5.0", "776"): "776  IndividualAllocRejCode",
        ("FIX.5.0", "777"): "777  SettlInstMsgID",
        ("FIX.5.0", "778"): "778  NoSettlInst",
        ("FIX.5.0", "779"): "779  LastUpdateTime",
        ("FIX.5.0", "780"): "780  AllocSettlInstType",
        ("FIX.5.0", "781"): "781  NoSettlPartyIDs",
        ("FIX.5.0", "782"): "782  SettlPartyID",
        ("FIX.5.0", "783"): "783  SettlPartyIDSource",
        ("FIX.5.0", "784"): "784  SettlPartyRole",
        ("FIX.5.0", "785"): "785  SettlPartySubID",
        ("FIX.5.0", "786"): "786  SettlPartySubIDType",
        ("FIX.5.0", "787"): "787  DlvyInstType",
        ("FIX.5.0", "788"): "788  TerminationType",
        ("FIX.5.0", "789"): "789  NextExpectedMsgSeqNum",
        ("FIX.5.0", "790"): "790  OrdStatusReqID",
        ("FIX.5.0", "791"): "791  SettlInstReqID",
        ("FIX.5.0", "792"): "792  SettlInstReqRejCode",
        ("FIX.5.0", "793"): "793  SecondaryAllocID",
        ("FIX.5.0", "794"): "794  AllocReportType",
        ("FIX.5.0", "795"): "795  AllocReportRefID",
        ("FIX.5.0", "796"): "796  AllocCancReplaceReason",
        ("FIX.5.0", "797"): "797  CopyMsgIndicator",
        ("FIX.5.0", "798"): "798  AllocAccountType",
        ("FIX.5.0", "799"): "799  OrderAvgPx",
        ("FIX.5.0", "800"): "800  OrderBookingQty",
        ("FIX.5.0", "801"): "801  NoSettlPartySubIDs",
        ("FIX.5.0", "802"): "802  NoPartySubIDs",
        ("FIX.5.0", "803"): "803  PartySubIDType",
        ("FIX.5.0", "804"): "804  NoNestedPartySubIDs",
        ("FIX.5.0", "805"): "805  NestedPartySubIDType",
        ("FIX.5.0", "806"): "806  NoNested2PartySubIDs",
        ("FIX.5.0", "807"): "807  Nested2PartySubIDType",
        ("FIX.5.0", "808"): "808  AllocIntermedReqType",
        ("FIX.5.0", "809"): "809  NoUsernames",
        ("FIX.5.0", "810"): "810  UnderlyingPx",
        ("FIX.5.0", "811"): "811  PriceDelta",
        ("FIX.5.0", "812"): "812  ApplQueueMax",
        ("FIX.5.0", "813"): "813  ApplQueueDepth",
        ("FIX.5.0", "814"): "814  ApplQueueResolution",
        ("FIX.5.0", "815"): "815  ApplQueueAction",
        ("FIX.5.0", "816"): "816  NoAltMDSource",
        ("FIX.5.0", "817"): "817  AltMDSourceID",
        ("FIX.5.0", "818"): "818  SecondaryTradeReportID",
        ("FIX.5.0", "819"): "819  AvgPxIndicator",
        ("FIX.5.0", "820"): "820  TradeLinkID",
        ("FIX.5.0", "821"): "821  OrderInputDevice",
        ("FIX.5.0", "822"): "822  UnderlyingTradingSessionID",
        ("FIX.5.0", "823"): "823  UnderlyingTradingSessionSubID",
        ("FIX.5.0", "824"): "824  TradeLegRefID",
        ("FIX.5.0", "825"): "825  ExchangeRule",
        ("FIX.5.0", "826"): "826  TradeAllocIndicator",
        ("FIX.5.0", "827"): "827  ExpirationCycle",
        ("FIX.5.0", "828"): "828  TrdType",
        ("FIX.5.0", "829"): "829  TrdSubType",
        ("FIX.5.0", "830"): "830  TransferReason",
        ("FIX.5.0", "832"): "832  TotNumAssignmentReports",
        ("FIX.5.0", "833"): "833  AsgnRptID",
        ("FIX.5.0", "834"): "834  ThresholdAmount",
        ("FIX.5.0", "835"): "835  PegMoveType",
        ("FIX.5.0", "836"): "836  PegOffsetType",
        ("FIX.5.0", "837"): "837  PegLimitType",
        ("FIX.5.0", "838"): "838  PegRoundDirection",
        ("FIX.5.0", "839"): "839  PeggedPrice",
        ("FIX.5.0", "840"): "840  PegScope",
        ("FIX.5.0", "841"): "841  DiscretionMoveType",
        ("FIX.5.0", "842"): "842  DiscretionOffsetType",
        ("FIX.5.0", "843"): "843  DiscretionLimitType",
        ("FIX.5.0", "844"): "844  DiscretionRoundDirection",
        ("FIX.5.0", "845"): "845  DiscretionPrice",
        ("FIX.5.0", "846"): "846  DiscretionScope",
        ("FIX.5.0", "847"): "847  TargetStrategy",
        ("FIX.5.0", "848"): "848  TargetStrategyParameters",
        ("FIX.5.0", "849"): "849  ParticipationRate",
        ("FIX.5.0", "850"): "850  TargetStrategyPerformance",
        ("FIX.5.0", "851"): "851  LastLiquidityInd",
        ("FIX.5.0", "852"): "852  PublishTrdIndicator",
        ("FIX.5.0", "853"): "853  ShortSaleReason",
        ("FIX.5.0", "854"): "854  QtyType",
        ("FIX.5.0", "855"): "855  SecondaryTrdType",
        ("FIX.5.0", "856"): "856  TradeReportType",
        ("FIX.5.0", "857"): "857  AllocNoOrdersType",
        ("FIX.5.0", "858"): "858  SharedCommission",
        ("FIX.5.0", "859"): "859  ConfirmReqID",
        ("FIX.5.0", "860"): "860  AvgParPx",
        ("FIX.5.0", "861"): "861  ReportedPx",
        ("FIX.5.0", "862"): "862  NoCapacities",
        ("FIX.5.0", "863"): "863  OrderCapacityQty",
        ("FIX.5.0", "864"): "864  NoEvents",
        ("FIX.5.0", "865"): "865  EventType",
        ("FIX.5.0", "866"): "866  EventDate",
        ("FIX.5.0", "867"): "867  EventPx",
        ("FIX.5.0", "868"): "868  EventText",
        ("FIX.5.0", "869"): "869  PctAtRisk",
        ("FIX.5.0", "870"): "870  NoInstrAttrib",
        ("FIX.5.0", "871"): "871  InstrAttribType",
        ("FIX.5.0", "872"): "872  InstrAttribValue",
        ("FIX.5.0", "873"): "873  DatedDate",
        ("FIX.5.0", "874"): "874  InterestAccrualDate",
        ("FIX.5.0", "875"): "875  CPProgram",
        ("FIX.5.0", "876"): "876  CPRegType",
        ("FIX.5.0", "877"): "877  UnderlyingCPProgram",
        ("FIX.5.0", "878"): "878  UnderlyingCPRegType",
        ("FIX.5.0", "879"): "879  UnderlyingQty",
        ("FIX.5.0", "880"): "880  TrdMatchID",
        ("FIX.5.0", "881"): "881  SecondaryTradeReportRefID",
        ("FIX.5.0", "882"): "882  UnderlyingDirtyPrice",
        ("FIX.5.0", "883"): "883  UnderlyingEndPrice",
        ("FIX.5.0", "884"): "884  UnderlyingStartValue",
        ("FIX.5.0", "885"): "885  UnderlyingCurrentValue",
        ("FIX.5.0", "886"): "886  UnderlyingEndValue",
        ("FIX.5.0", "887"): "887  NoUnderlyingStips",
        ("FIX.5.0", "888"): "888  UnderlyingStipType",
        ("FIX.5.0", "889"): "889  UnderlyingStipValue",
        ("FIX.5.0", "890"): "890  MaturityNetMoney",
        ("FIX.5.0", "891"): "891  MiscFeeBasis",
        ("FIX.5.0", "892"): "892  TotNoAllocs",
        ("FIX.5.0", "893"): "893  LastFragment",
        ("FIX.5.0", "894"): "894  CollReqID",
        ("FIX.5.0", "895"): "895  CollAsgnReason",
        ("FIX.5.0", "896"): "896  CollInquiryQualifier",
        ("FIX.5.0", "897"): "897  NoTrades",
        ("FIX.5.0", "898"): "898  MarginRatio",
        ("FIX.5.0", "899"): "899  MarginExcess",
        ("FIX.5.0", "900"): "900  TotalNetValue",
        ("FIX.5.0", "901"): "901  CashOutstanding",
        ("FIX.5.0", "902"): "902  CollAsgnID",
        ("FIX.5.0", "903"): "903  CollAsgnTransType",
        ("FIX.5.0", "904"): "904  CollRespID",
        ("FIX.5.0", "905"): "905  CollAsgnRespType",
        ("FIX.5.0", "906"): "906  CollAsgnRejectReason",
        ("FIX.5.0", "907"): "907  CollAsgnRefID",
        ("FIX.5.0", "908"): "908  CollRptID",
        ("FIX.5.0", "909"): "909  CollInquiryID",
        ("FIX.5.0", "910"): "910  CollStatus",
        ("FIX.5.0", "911"): "911  TotNumReports",
        ("FIX.5.0", "912"): "912  LastRptRequested",
        ("FIX.5.0", "913"): "913  AgreementDesc",
        ("FIX.5.0", "914"): "914  AgreementID",
        ("FIX.5.0", "915"): "915  AgreementDate",
        ("FIX.5.0", "916"): "916  StartDate",
        ("FIX.5.0", "917"): "917  EndDate",
        ("FIX.5.0", "918"): "918  AgreementCurrency",
        ("FIX.5.0", "919"): "919  DeliveryType",
        ("FIX.5.0", "920"): "920  EndAccruedInterestAmt",
        ("FIX.5.0", "921"): "921  StartCash",
        ("FIX.5.0", "922"): "922  EndCash",
        ("FIX.5.0", "923"): "923  UserRequestID",
        ("FIX.5.0", "924"): "924  UserRequestType",
        ("FIX.5.0", "925"): "925  NewPassword",
        ("FIX.5.0", "926"): "926  UserStatus",
        ("FIX.5.0", "927"): "927  UserStatusText",
        ("FIX.5.0", "928"): "928  StatusValue",
        ("FIX.5.0", "929"): "929  StatusText",
        ("FIX.5.0", "930"): "930  RefCompID",
        ("FIX.5.0", "931"): "931  RefSubID",
        ("FIX.5.0", "932"): "932  NetworkResponseID",
        ("FIX.5.0", "933"): "933  NetworkRequestID",
        ("FIX.5.0", "934"): "934  LastNetworkResponseID",
        ("FIX.5.0", "935"): "935  NetworkRequestType",
        ("FIX.5.0", "936"): "936  NoCompIDs",
        ("FIX.5.0", "937"): "937  NetworkStatusResponseType",
        ("FIX.5.0", "938"): "938  NoCollInquiryQualifier",
        ("FIX.5.0", "939"): "939  TrdRptStatus",
        ("FIX.5.0", "940"): "940  AffirmStatus",
        ("FIX.5.0", "941"): "941  UnderlyingStrikeCurrency",
        ("FIX.5.0", "942"): "942  LegStrikeCurrency",
        ("FIX.5.0", "943"): "943  TimeBracket",
        ("FIX.5.0", "944"): "944  CollAction",
        ("FIX.5.0", "945"): "945  CollInquiryStatus",
        ("FIX.5.0", "946"): "946  CollInquiryResult",
        ("FIX.5.0", "947"): "947  StrikeCurrency",
        ("FIX.5.0", "948"): "948  NoNested3PartyIDs",
        ("FIX.5.0", "949"): "949  Nested3PartyID",
        ("FIX.5.0", "950"): "950  Nested3PartyIDSource",
        ("FIX.5.0", "951"): "951  Nested3PartyRole",
        ("FIX.5.0", "952"): "952  NoNested3PartySubIDs",
        ("FIX.5.0", "953"): "953  Nested3PartySubID",
        ("FIX.5.0", "954"): "954  Nested3PartySubIDType",
        ("FIX.5.0", "955"): "955  LegContractSettlMonth",
        ("FIX.5.0", "956"): "956  LegInterestAccrualDate",
        ("FIX.5.0", "957"): "957  NoStrategyParameters",
        ("FIX.5.0", "958"): "958  StrategyParameterName",
        ("FIX.5.0", "959"): "959  StrategyParameterType",
        ("FIX.5.0", "960"): "960  StrategyParameterValue",
        ("FIX.5.0", "961"): "961  HostCrossID",
        ("FIX.5.0", "962"): "962  SideTimeInForce",
        ("FIX.5.0", "963"): "963  MDReportID",
        ("FIX.5.0", "964"): "964  SecurityReportID",
        ("FIX.5.0", "965"): "965  SecurityStatus",
        ("FIX.5.0", "966"): "966  SettleOnOpenFlag",
        ("FIX.5.0", "967"): "967  StrikeMultiplier",
        ("FIX.5.0", "968"): "968  StrikeValue",
        ("FIX.5.0", "969"): "969  MinPriceIncrement",
        ("FIX.5.0", "970"): "970  PositionLimit",
        ("FIX.5.0", "971"): "971  NTPositionLimit",
        ("FIX.5.0", "972"): "972  UnderlyingAllocationPercent",
        ("FIX.5.0", "973"): "973  UnderlyingCashAmount",
        ("FIX.5.0", "974"): "974  UnderlyingCashType",
        ("FIX.5.0", "975"): "975  UnderlyingSettlementType",
        ("FIX.5.0", "976"): "976  QuantityDate",
        ("FIX.5.0", "977"): "977  ContIntRptID",
        ("FIX.5.0", "978"): "978  LateIndicator",
        ("FIX.5.0", "979"): "979  InputSource",
        ("FIX.5.0", "980"): "980  SecurityUpdateAction",
        ("FIX.5.0", "981"): "981  NoExpiration",
        ("FIX.5.0", "982"): "982  ExpirationQtyType",
        ("FIX.5.0", "983"): "983  ExpQty",
        ("FIX.5.0", "984"): "984  NoUnderlyingAmounts",
        ("FIX.5.0", "985"): "985  UnderlyingPayAmount",
        ("FIX.5.0", "986"): "986  UnderlyingCollectAmount",
        ("FIX.5.0", "987"): "987  UnderlyingSettlementDate",
        ("FIX.5.0", "988"): "988  UnderlyingSettlementStatus",
        ("FIX.5.0", "989"): "989  SecondaryIndividualAllocID",
        ("FIX.5.0", "990"): "990  LegReportID",
        ("FIX.5.0", "991"): "991  RndPx",
        ("FIX.5.0", "992"): "992  IndividualAllocType",
        ("FIX.5.0", "993"): "993  AllocCustomerCapacity",
        ("FIX.5.0", "994"): "994  TierCode",
        ("FIX.5.0", "996"): "996  UnitOfMeasure",
        ("FIX.5.0", "997"): "997  TimeUnit",
        ("FIX.5.0", "998"): "998  UnderlyingUnitOfMeasure",
        ("FIX.5.0", "999"): "999  LegUnitOfMeasure",
        ("FIX.5.0", "1000"): "1000 UnderlyingTimeUnit",
        ("FIX.5.0", "1001"): "1001 LegTimeUnit",
        ("FIX.5.0", "1002"): "1002 AllocMethod",
        ("FIX.5.0", "1003"): "1003 TradeID",
        ("FIX.5.0", "1005"): "1005 SideTradeReportID",
        ("FIX.5.0", "1006"): "1006 SideFillStationCd",
        ("FIX.5.0", "1007"): "1007 SideReasonCd",
        ("FIX.5.0", "1008"): "1008 SideTrdSubTyp",
        ("FIX.5.0", "1009"): "1009 SideLastQty",
        ("FIX.5.0", "1011"): "1011 MessageEventSource",
        ("FIX.5.0", "1012"): "1012 SideTrdRegTimestamp",
        ("FIX.5.0", "1013"): "1013 SideTrdRegTimestampType",
        ("FIX.5.0", "1014"): "1014 SideTrdRegTimestampSrc",
        ("FIX.5.0", "1015"): "1015 AsOfIndicator",
        ("FIX.5.0", "1016"): "1016 NoSideTrdRegTS",
        ("FIX.5.0", "1017"): "1017 LegOptionRatio",
        ("FIX.5.0", "1018"): "1018 NoInstrumentParties",
        ("FIX.5.0", "1019"): "1019 InstrumentPartyID",
        ("FIX.5.0", "1020"): "1020 TradeVolume",
        ("FIX.5.0", "1021"): "1021 MDBookType",
        ("FIX.5.0", "1022"): "1022 MDFeedType",
        ("FIX.5.0", "1023"): "1023 MDPriceLevel",
        ("FIX.5.0", "1024"): "1024 MDOriginType",
        ("FIX.5.0", "1025"): "1025 FirstPx",
        ("FIX.5.0", "1026"): "1026 MDEntrySpotRate",
        ("FIX.5.0", "1027"): "1027 MDEntryForwardPoints",
        ("FIX.5.0", "1028"): "1028 ManualOrderIndicator",
        ("FIX.5.0", "1029"): "1029 CustDirectedOrder",
        ("FIX.5.0", "1030"): "1030 ReceivedDeptID",
        ("FIX.5.0", "1031"): "1031 CustOrderHandlingInst",
        ("FIX.5.0", "1032"): "1032 OrderHandlingInstSource",
        ("FIX.5.0", "1033"): "1033 DeskType",
        ("FIX.5.0", "1034"): "1034 DeskTypeSource",
        ("FIX.5.0", "1035"): "1035 DeskOrderHandlingInst",
        ("FIX.5.0", "1036"): "1036 ExecAckStatus",
        ("FIX.5.0", "1037"): "1037 UnderlyingDeliveryAmount",
        ("FIX.5.0", "1038"): "1038 UnderlyingCapValue",
        ("FIX.5.0", "1039"): "1039 UnderlyingSettlMethod",
        ("FIX.5.0", "1040"): "1040 SecondaryTradeID",
        ("FIX.5.0", "1041"): "1041 FirmTradeID",
        ("FIX.5.0", "1042"): "1042 SecondaryFirmTradeID",
        ("FIX.5.0", "1043"): "1043 CollApplType",
        ("FIX.5.0", "1044"): "1044 UnderlyingAdjustedQuantity",
        ("FIX.5.0", "1045"): "1045 UnderlyingFXRate",
        ("FIX.5.0", "1046"): "1046 UnderlyingFXRateCalc",
        ("FIX.5.0", "1047"): "1047 AllocPositionEffect",
        ("FIX.5.0", "1048"): "1048 DealingCapacity",
        ("FIX.5.0", "1049"): "1049 InstrmtAssignmentMethod",
        ("FIX.5.0", "1050"): "1050 InstrumentPartyIDSource",
        ("FIX.5.0", "1051"): "1051 InstrumentPartyRole",
        ("FIX.5.0", "1052"): "1052 NoInstrumentPartySubIDs",
        ("FIX.5.0", "1053"): "1053 InstrumentPartySubID",
        ("FIX.5.0", "1054"): "1054 InstrumentPartySubIDType",
        ("FIX.5.0", "1055"): "1055 PositionCurrency",
        ("FIX.5.0", "1056"): "1056 CalculatedCcyLastQty",
        ("FIX.5.0", "1057"): "1057 AggressorIndicator",
        ("FIX.5.0", "1058"): "1058 NoUndlyInstrumentParties",
        ("FIX.5.0", "1059"): "1059 UnderlyingInstrumentPartyID",
        ("FIX.5.0", "1060"): "1060 UnderlyingInstrumentPartyIDSource",
        ("FIX.5.0", "1061"): "1061 UnderlyingInstrumentPartyRole",
        ("FIX.5.0", "1062"): "1062 NoUndlyInstrumentPartySubIDs",
        ("FIX.5.0", "1063"): "1063 UnderlyingInstrumentPartySubID",
        ("FIX.5.0", "1064"): "1064 UnderlyingInstrumentPartySubIDType",
        ("FIX.5.0", "1065"): "1065 BidSwapPoints",
        ("FIX.5.0", "1066"): "1066 OfferSwapPoints",
        ("FIX.5.0", "1067"): "1067 LegBidForwardPoints",
        ("FIX.5.0", "1068"): "1068 LegOfferForwardPoints",
        ("FIX.5.0", "1069"): "1069 SwapPoints",
        ("FIX.5.0", "1070"): "1070 MDQuoteType",
        ("FIX.5.0", "1071"): "1071 LastSwapPoints",
        ("FIX.5.0", "1072"): "1072 SideGrossTradeAmt",
        ("FIX.5.0", "1073"): "1073 LegLastForwardPoints",
        ("FIX.5.0", "1074"): "1074 LegCalculatedCcyLastQty",
        ("FIX.5.0", "1075"): "1075 LegGrossTradeAmt",
        ("FIX.5.0", "1079"): "1079 MaturityTime",
        ("FIX.5.0", "1080"): "1080 RefOrderID",
        ("FIX.5.0", "1081"): "1081 RefOrderIDSource",
        ("FIX.5.0", "1082"): "1082 SecondaryDisplayQty",
        ("FIX.5.0", "1083"): "1083 DisplayWhen",
        ("FIX.5.0", "1084"): "1084 DisplayMethod",
        ("FIX.5.0", "1085"): "1085 DisplayLowQty",
        ("FIX.5.0", "1086"): "1086 DisplayHighQty",
        ("FIX.5.0", "1087"): "1087 DisplayMinIncr",
        ("FIX.5.0", "1088"): "1088 RefreshQty",
        ("FIX.5.0", "1089"): "1089 MatchIncrement",
        ("FIX.5.0", "1090"): "1090 MaxPriceLevels",
        ("FIX.5.0", "1091"): "1091 PreTradeAnonymity",
        ("FIX.5.0", "1092"): "1092 PriceProtectionScope",
        ("FIX.5.0", "1093"): "1093 LotType",
        ("FIX.5.0", "1094"): "1094 PegPriceType",
        ("FIX.5.0", "1095"): "1095 PeggedRefPrice",
        ("FIX.5.0", "1096"): "1096 PegSecurityIDSource",
        ("FIX.5.0", "1097"): "1097 PegSecurityID",
        ("FIX.5.0", "1098"): "1098 PegSymbol",
        ("FIX.5.0", "1099"): "1099 PegSecurityDesc",
        ("FIX.5.0", "1100"): "1100 TriggerType",
        ("FIX.5.0", "1101"): "1101 TriggerAction",
        ("FIX.5.0", "1102"): "1102 TriggerPrice",
        ("FIX.5.0", "1103"): "1103 TriggerSymbol",
        ("FIX.5.0", "1104"): "1104 TriggerSecurityID",
        ("FIX.5.0", "1105"): "1105 TriggerSecurityIDSource",
        ("FIX.5.0", "1106"): "1106 TriggerSecurityDesc",
        ("FIX.5.0", "1107"): "1107 TriggerPriceType",
        ("FIX.5.0", "1108"): "1108 TriggerPriceTypeScope",
        ("FIX.5.0", "1109"): "1109 TriggerPriceDirection",
        ("FIX.5.0", "1110"): "1110 TriggerNewPrice",
        ("FIX.5.0", "1111"): "1111 TriggerOrderType",
        ("FIX.5.0", "1112"): "1112 TriggerNewQty",
        ("FIX.5.0", "1113"): "1113 TriggerTradingSessionID",
        ("FIX.5.0", "1114"): "1114 TriggerTradingSessionSubID",
        ("FIX.5.0", "1115"): "1115 OrderCategory",
        ("FIX.5.0", "1116"): "1116 NoRootPartyIDs",
        ("FIX.5.0", "1117"): "1117 RootPartyID",
        ("FIX.5.0", "1118"): "1118 RootPartyIDSource",
        ("FIX.5.0", "1119"): "1119 RootPartyRole",
        ("FIX.5.0", "1120"): "1120 NoRootPartySubIDs",
        ("FIX.5.0", "1121"): "1121 RootPartySubID",
        ("FIX.5.0", "1122"): "1122 RootPartySubIDType",
        ("FIX.5.0", "1123"): "1123 TradeHandlingInstr",
        ("FIX.5.0", "1124"): "1124 OrigTradeHandlingInstr",
        ("FIX.5.0", "1125"): "1125 OrigTradeDate",
        ("FIX.5.0", "1126"): "1126 OrigTradeID",
        ("FIX.5.0", "1127"): "1127 OrigSecondaryTradeID",
        ("FIX.5.0", "1128"): "1128 ApplVerID",
        ("FIX.5.0", "1129"): "1129 CstmApplVerID",
        ("FIX.5.0", "1130"): "1130 RefApplVerID",
        ("FIX.5.0", "1131"): "1131 RefCstmApplVerID",
        ("FIX.5.0", "1132"): "1132 TZTransactTime",
        ("FIX.5.0", "1133"): "1133 ExDestinationIDSource",
        ("FIX.5.0", "1134"): "1134 ReportedPxDiff",
        ("FIX.5.0", "1135"): "1135 RptSys",
        ("FIX.5.0", "1136"): "1136 AllocClearingFeeIndicator",
        ("FIX.5.0", "1137"): "1137 DefaultApplVerID",
        ("FIX.5.0", "1138"): "1138 DisplayQty",
        ("FIX.5.0", "1139"): "1139 ExchangeSpecialInstructions",
        ("FIX.5.0", "1140"): "1140 MaxTradeVol",
        ("FIX.5.0", "1141"): "1141 NoMDFeedTypes",
        ("FIX.5.0", "1142"): "1142 MatchAlgorithm",
        ("FIX.5.0", "1143"): "1143 MaxPriceVariation",
        ("FIX.5.0", "1144"): "1144 ImpliedMarketIndicator",
        ("FIX.5.0", "1145"): "1145 EventTime",
        ("FIX.5.0", "1146"): "1146 MinPriceIncrementAmount",
        ("FIX.5.0", "1147"): "1147 UnitOfMeasureQty",
        ("FIX.5.0", "1148"): "1148 LowLimitPrice",
        ("FIX.5.0", "1149"): "1149 HighLimitPrice",
        ("FIX.5.0", "1150"): "1150 TradingReferencePrice",
        ("FIX.5.0", "1151"): "1151 SecurityGroup",
        ("FIX.5.0", "1152"): "1152 LegNumber",
        ("FIX.5.0", "1153"): "1153 SettlementCycleNo",
        ("FIX.5.0", "1154"): "1154 SideCurrency",
        ("FIX.5.0", "1155"): "1155 SideSettlCurrency",
        ("FIX.5.0", "1156"): "1156 ApplExtID",
        ("FIX.5.0", "1157"): "1157 CcyAmt",
        ("FIX.5.0", "1158"): "1158 NoSettlDetails",
        ("FIX.5.0", "1159"): "1159 SettlObligMode",
        ("FIX.5.0", "1160"): "1160 SettlObligMsgID",
        ("FIX.5.0", "1161"): "1161 SettlObligID",
        ("FIX.5.0", "1162"): "1162 SettlObligTransType",
        ("FIX.5.0", "1163"): "1163 SettlObligRefID",
        ("FIX.5.0", "1164"): "1164 SettlObligSource",
        ("FIX.5.0", "1165"): "1165 NoSettlOblig",
        ("FIX.5.0", "1166"): "1166 QuoteMsgID",
        ("FIX.5.0", "1167"): "1167 QuoteEntryStatus",
        ("FIX.5.0", "1168"): "1168 TotNoCxldQuotes",
        ("FIX.5.0", "1169"): "1169 TotNoAccQuotes",
        ("FIX.5.0", "1170"): "1170 TotNoRejQuotes",
        ("FIX.5.0", "1171"): "1171 PrivateQuote",
        ("FIX.5.0", "1172"): "1172 RespondentType",
        ("FIX.5.0", "1173"): "1173 MDSubBookType",
        ("FIX.5.0", "1174"): "1174 SecurityTradingEvent",
        ("FIX.5.0", "1175"): "1175 NoStatsIndicators",
        ("FIX.5.0", "1176"): "1176 StatsType",
        ("FIX.5.0", "1177"): "1177 NoOfSecSizes",
        ("FIX.5.0", "1178"): "1178 MDSecSizeType",
        ("FIX.5.0", "1179"): "1179 MDSecSize",
        ("FIX.5.0", "1180"): "1180 ApplID",
        ("FIX.5.0", "1181"): "1181 ApplSeqNum",
        ("FIX.5.0", "1182"): "1182 ApplBegSeqNum",
        ("FIX.5.0", "1183"): "1183 ApplEndSeqNum",
        ("FIX.5.0", "1184"): "1184 SecurityXMLLen",
        ("FIX.5.0", "1185"): "1185 SecurityXML",
        ("FIX.5.0", "1186"): "1186 SecurityXMLSchema",
        ("FIX.5.0", "1187"): "1187 RefreshIndicator",
        ("FIX.5.0", "1188"): "1188 Volatility",
        ("FIX.5.0", "1189"): "1189 TimeToExpiration",
        ("FIX.5.0", "1190"): "1190 RiskFreeRate",
        ("FIX.5.0", "1191"): "1191 PriceUnitOfMeasure",
        ("FIX.5.0", "1192"): "1192 PriceUnitOfMeasureQty",
        ("FIX.5.0", "1193"): "1193 SettlMethod",
        ("FIX.5.0", "1194"): "1194 ExerciseStyle",
        ("FIX.5.0", "1195"): "1195 OptPayoutAmount",
        ("FIX.5.0", "1196"): "1196 PriceQuoteMethod",
        ("FIX.5.0", "1197"): "1197 ValuationMethod",
        ("FIX.5.0", "1198"): "1198 ListMethod",
        ("FIX.5.0", "1199"): "1199 CapPrice",
        ("FIX.5.0", "1200"): "1200 FloorPrice",
        ("FIX.5.0", "1201"): "1201 NoStrikeRules",
        ("FIX.5.0", "1202"): "1202 StartStrikePxRange",
        ("FIX.5.0", "1203"): "1203 EndStrikePxRange",
        ("FIX.5.0", "1204"): "1204 StrikeIncrement",
        ("FIX.5.0", "1205"): "1205 NoTickRules",
        ("FIX.5.0", "1206"): "1206 StartTickPriceRange",
        ("FIX.5.0", "1207"): "1207 EndTickPriceRange",
        ("FIX.5.0", "1208"): "1208 TickIncrement",
        ("FIX.5.0", "1209"): "1209 TickRuleType",
        ("FIX.5.0", "1210"): "1210 NestedInstrAttribType",
        ("FIX.5.0", "1211"): "1211 NestedInstrAttribValue",
        ("FIX.5.0", "1212"): "1212 LegMaturityTime",
        ("FIX.5.0", "1213"): "1213 UnderlyingMaturityTime",
        ("FIX.5.0", "1214"): "1214 DerivativeSymbol",
        ("FIX.5.0", "1215"): "1215 DerivativeSymbolSfx",
        ("FIX.5.0", "1216"): "1216 DerivativeSecurityID",
        ("FIX.5.0", "1217"): "1217 DerivativeSecurityIDSource",
        ("FIX.5.0", "1218"): "1218 NoDerivativeSecurityAltID",
        ("FIX.5.0", "1219"): "1219 DerivativeSecurityAltID",
        ("FIX.5.0", "1220"): "1220 DerivativeSecurityAltIDSource",
        ("FIX.5.0", "1221"): "1221 SecondaryLowLimitPrice",
        ("FIX.5.0", "1222"): "1222 MaturityRuleID",
        ("FIX.5.0", "1223"): "1223 StrikeRuleID",
        ("FIX.5.0", "1224"): "1224 LegUnitOfMeasureQty",
        ("FIX.5.0", "1225"): "1225 DerivativeOptPayAmount",
        ("FIX.5.0", "1226"): "1226 EndMaturityMonthYear",
        ("FIX.5.0", "1227"): "1227 ProductComplex",
        ("FIX.5.0", "1228"): "1228 DerivativeProductComplex",
        ("FIX.5.0", "1229"): "1229 MaturityMonthYearIncrement",
        ("FIX.5.0", "1230"): "1230 SecondaryHighLimitPrice",
        ("FIX.5.0", "1231"): "1231 MinLotSize",
        ("FIX.5.0", "1232"): "1232 NoExecInstRules",
        ("FIX.5.0", "1233"): "1233 CommRate",
        ("FIX.5.0", "1234"): "1234 NoLotTypeRules",
        ("FIX.5.0", "1235"): "1235 NoMatchRules",
        ("FIX.5.0", "1236"): "1236 NoMaturityRules",
        ("FIX.5.0", "1237"): "1237 NoOrdTypeRules",
        ("FIX.5.0", "1238"): "1238 CommUnitOfMeasure",
        ("FIX.5.0", "1239"): "1239 NoTimeInForceRules",
        ("FIX.5.0", "1240"): "1240 SecondaryTradingReferencePrice",
        ("FIX.5.0", "1241"): "1241 StartMaturityMonthYear",
        ("FIX.5.0", "1242"): "1242 FlexProductEligibilityIndicator",
        ("FIX.5.0", "1243"): "1243 DerivFlexProductEligibilityIndicator",
        ("FIX.5.0", "1244"): "1244 FlexibleIndicator",
        ("FIX.5.0", "1245"): "1245 TradingCurrency",
        ("FIX.5.0", "1246"): "1246 DerivativeProduct",
        ("FIX.5.0", "1247"): "1247 DerivativeSecurityGroup",
        ("FIX.5.0", "1248"): "1248 DerivativeCFICode",
        ("FIX.5.0", "1249"): "1249 DerivativeSecurityType",
        ("FIX.5.0", "1250"): "1250 DerivativeSecuritySubType",
        ("FIX.5.0", "1251"): "1251 DerivativeMaturityMonthYear",
        ("FIX.5.0", "1252"): "1252 DerivativeMaturityDate",
        ("FIX.5.0", "1253"): "1253 DerivativeMaturityTime",
        ("FIX.5.0", "1254"): "1254 DerivativeSettleOnOpenFlag",
        ("FIX.5.0", "1255"): "1255 DerivativeInstrmtAssignmentMethod",
        ("FIX.5.0", "1256"): "1256 DerivativeSecurityStatus",
        ("FIX.5.0", "1257"): "1257 DerivativeInstrRegistry",
        ("FIX.5.0", "1258"): "1258 DerivativeCountryOfIssue",
        ("FIX.5.0", "1259"): "1259 DerivativeStateOrProvinceOfIssue",
        ("FIX.5.0", "1260"): "1260 DerivativeLocaleOfIssue",
        ("FIX.5.0", "1261"): "1261 DerivativeStrikePrice",
        ("FIX.5.0", "1262"): "1262 DerivativeStrikeCurrency",
        ("FIX.5.0", "1263"): "1263 DerivativeStrikeMultiplier",
        ("FIX.5.0", "1264"): "1264 DerivativeStrikeValue",
        ("FIX.5.0", "1265"): "1265 DerivativeOptAttribute",
        ("FIX.5.0", "1266"): "1266 DerivativeContractMultiplier",
        ("FIX.5.0", "1267"): "1267 DerivativeMinPriceIncrement",
        ("FIX.5.0", "1268"): "1268 DerivativeMinPriceIncrementAmount",
        ("FIX.5.0", "1269"): "1269 DerivativeUnitOfMeasure",
        ("FIX.5.0", "1270"): "1270 DerivativeUnitOfMeasureQty",
        ("FIX.5.0", "1271"): "1271 DerivativeTimeUnit",
        ("FIX.5.0", "1272"): "1272 DerivativeSecurityExchange",
        ("FIX.5.0", "1273"): "1273 DerivativePositionLimit",
        ("FIX.5.0", "1274"): "1274 DerivativeNTPositionLimit",
        ("FIX.5.0", "1275"): "1275 DerivativeIssuer",
        ("FIX.5.0", "1276"): "1276 DerivativeIssueDate",
        ("FIX.5.0", "1277"): "1277 DerivativeEncodedIssuerLen",
        ("FIX.5.0", "1278"): "1278 DerivativeEncodedIssuer",
        ("FIX.5.0", "1279"): "1279 DerivativeSecurityDesc",
        ("FIX.5.0", "1280"): "1280 DerivativeEncodedSecurityDescLen",
        ("FIX.5.0", "1281"): "1281 DerivativeEncodedSecurityDesc",
        ("FIX.5.0", "1282"): "1282 DerivativeSecurityXMLLen",
        ("FIX.5.0", "1283"): "1283 DerivativeSecurityXML",
        ("FIX.5.0", "1284"): "1284 DerivativeSecurityXMLSchema",
        ("FIX.5.0", "1285"): "1285 DerivativeContractSettlMonth",
        ("FIX.5.0", "1286"): "1286 NoDerivativeEvents",
        ("FIX.5.0", "1287"): "1287 DerivativeEventType",
        ("FIX.5.0", "1288"): "1288 DerivativeEventDate",
        ("FIX.5.0", "1289"): "1289 DerivativeEventTime",
        ("FIX.5.0", "1290"): "1290 DerivativeEventPx",
        ("FIX.5.0", "1291"): "1291 DerivativeEventText",
        ("FIX.5.0", "1292"): "1292 NoDerivativeInstrumentParties",
        ("FIX.5.0", "1293"): "1293 DerivativeInstrumentPartyID",
        ("FIX.5.0", "1294"): "1294 DerivativeInstrumentPartyIDSource",
        ("FIX.5.0", "1295"): "1295 DerivativeInstrumentPartyRole",
        ("FIX.5.0", "1296"): "1296 NoDerivativeInstrumentPartySubIDs",
        ("FIX.5.0", "1297"): "1297 DerivativeInstrumentPartySubID",
        ("FIX.5.0", "1298"): "1298 DerivativeInstrumentPartySubIDType",
        ("FIX.5.0", "1299"): "1299 DerivativeExerciseStyle",
        ("FIX.5.0", "1300"): "1300 MarketSegmentID",
        ("FIX.5.0", "1301"): "1301 MarketID",
        ("FIX.5.0", "1302"): "1302 MaturityMonthYearIncrementUnits",
        ("FIX.5.0", "1303"): "1303 MaturityMonthYearFormat",
        ("FIX.5.0", "1304"): "1304 StrikeExerciseStyle",
        ("FIX.5.0", "1305"): "1305 SecondaryPriceLimitType",
        ("FIX.5.0", "1306"): "1306 PriceLimitType",
        ("FIX.5.0", "1308"): "1308 ExecInstValue",
        ("FIX.5.0", "1309"): "1309 NoTradingSessionRules",
        ("FIX.5.0", "1310"): "1310 NoMarketSegments",
        ("FIX.5.0", "1311"): "1311 NoDerivativeInstrAttrib",
        ("FIX.5.0", "1312"): "1312 NoNestedInstrAttrib",
        ("FIX.5.0", "1313"): "1313 DerivativeInstrAttribType",
        ("FIX.5.0", "1314"): "1314 DerivativeInstrAttribValue",
        ("FIX.5.0", "1315"): "1315 DerivativePriceUnitOfMeasure",
        ("FIX.5.0", "1316"): "1316 DerivativePriceUnitOfMeasureQty",
        ("FIX.5.0", "1317"): "1317 DerivativeSettlMethod",
        ("FIX.5.0", "1318"): "1318 DerivativePriceQuoteMethod",
        ("FIX.5.0", "1319"): "1319 DerivativeValuationMethod",
        ("FIX.5.0", "1320"): "1320 DerivativeListMethod",
        ("FIX.5.0", "1321"): "1321 DerivativeCapPrice",
        ("FIX.5.0", "1322"): "1322 DerivativeFloorPrice",
        ("FIX.5.0", "1323"): "1323 DerivativePutOrCall",
        ("FIX.5.0", "1324"): "1324 ListUpdateAction",
        ("FIX.5.0", "1325"): "1325 ParentMktSegmID",
        ("FIX.5.0", "1326"): "1326 TradingSessionDesc",
        ("FIX.5.0", "1327"): "1327 TradSesUpdateAction",
        ("FIX.5.0", "1328"): "1328 RejectText",
        ("FIX.5.0", "1329"): "1329 FeeMultiplier",
        ("FIX.5.0", "1330"): "1330 UnderlyingLegSymbol",
        ("FIX.5.0", "1331"): "1331 UnderlyingLegSymbolSfx",
        ("FIX.5.0", "1332"): "1332 UnderlyingLegSecurityID",
        ("FIX.5.0", "1333"): "1333 UnderlyingLegSecurityIDSource",
        ("FIX.5.0", "1334"): "1334 NoUnderlyingLegSecurityAltID",
        ("FIX.5.0", "1335"): "1335 UnderlyingLegSecurityAltID",
        ("FIX.5.0", "1336"): "1336 UnderlyingLegSecurityAltIDSource",
        ("FIX.5.0", "1337"): "1337 UnderlyingLegSecurityType",
        ("FIX.5.0", "1338"): "1338 UnderlyingLegSecuritySubType",
        ("FIX.5.0", "1339"): "1339 UnderlyingLegMaturityMonthYear",
        ("FIX.5.0", "1340"): "1340 UnderlyingLegStrikePrice",
        ("FIX.5.0", "1341"): "1341 UnderlyingLegSecurityExchange",
        ("FIX.5.0", "1342"): "1342 NoOfLegUnderlyings",
        ("FIX.5.0", "1343"): "1343 UnderlyingLegPutOrCall",
        ("FIX.5.0", "1344"): "1344 UnderlyingLegCFICode",
        ("FIX.5.0", "1345"): "1345 UnderlyingLegMaturityDate",
        ("FIX.5.0", "1346"): "1346 ApplReqID",
        ("FIX.5.0", "1347"): "1347 ApplReqType",
        ("FIX.5.0", "1348"): "1348 ApplResponseType",
        ("FIX.5.0", "1349"): "1349 ApplTotalMessageCount",
        ("FIX.5.0", "1350"): "1350 ApplLastSeqNum",
        ("FIX.5.0", "1351"): "1351 NoApplIDs",
        ("FIX.5.0", "1352"): "1352 ApplResendFlag",
        ("FIX.5.0", "1353"): "1353 ApplResponseID",
        ("FIX.5.0", "1354"): "1354 ApplResponseError",
        ("FIX.5.0", "1355"): "1355 RefApplID",
        ("FIX.5.0", "1356"): "1356 ApplReportID",
        ("FIX.5.0", "1357"): "1357 RefApplLastSeqNum",
        ("FIX.5.0", "1358"): "1358 LegPutOrCall",
        ("FIX.5.0", "1361"): "1361 TotNoFills",
        ("FIX.5.0", "1362"): "1362 NoFills",
        ("FIX.5.0", "1363"): "1363 FillExecID",
        ("FIX.5.0", "1364"): "1364 FillPx",
        ("FIX.5.0", "1365"): "1365 FillQty",
        ("FIX.5.0", "1366"): "1366 LegAllocID",
        ("FIX.5.0", "1367"): "1367 LegAllocSettlCurrency",
        ("FIX.5.0", "1368"): "1368 TradSesEvent",
        ("FIX.5.0", "1369"): "1369 MassActionReportID",
        ("FIX.5.0", "1370"): "1370 NoNotAffectedOrders",
        ("FIX.5.0", "1371"): "1371 NotAffectedOrderID",
        ("FIX.5.0", "1372"): "1372 NotAffOrigClOrdID",
        ("FIX.5.0", "1373"): "1373 MassActionType",
        ("FIX.5.0", "1374"): "1374 MassActionScope",
        ("FIX.5.0", "1375"): "1375 MassActionResponse",
        ("FIX.5.0", "1376"): "1376 MassActionRejectReason",
        ("FIX.5.0", "1377"): "1377 MultilegModel",
        ("FIX.5.0", "1378"): "1378 MultilegPriceMethod",
        ("FIX.5.0", "1379"): "1379 LegVolatility",
        ("FIX.5.0", "1380"): "1380 DividendYield",
        ("FIX.5.0", "1381"): "1381 LegDividendYield",
        ("FIX.5.0", "1382"): "1382 CurrencyRatio",
        ("FIX.5.0", "1383"): "1383 LegCurrencyRatio",
        ("FIX.5.0", "1384"): "1384 LegExecInst",
        ("FIX.5.0", "1385"): "1385 ContingencyType",
        ("FIX.5.0", "1386"): "1386 ListRejectReason",
        ("FIX.5.0", "1387"): "1387 NoTrdRepIndicators",
        ("FIX.5.0", "1388"): "1388 TrdRepPartyRole",
        ("FIX.5.0", "1389"): "1389 TrdRepIndicator",
        ("FIX.5.0", "1390"): "1390 TradePublishIndicator",
        ("FIX.5.0", "1391"): "1391 UnderlyingLegOptAttribute",
        ("FIX.5.0", "1392"): "1392 UnderlyingLegSecurityDesc",
        ("FIX.5.0", "1393"): "1393 MarketReqID",
        ("FIX.5.0", "1394"): "1394 MarketReportID",
        ("FIX.5.0", "1395"): "1395 MarketUpdateAction",
        ("FIX.5.0", "1396"): "1396 MarketSegmentDesc",
        ("FIX.5.0", "1397"): "1397 EncodedMktSegmDescLen",
        ("FIX.5.0", "1398"): "1398 EncodedMktSegmDesc",
        ("FIX.5.0", "1399"): "1399 ApplNewSeqNum",
        ("FIX.5.0", "1400"): "1400 EncryptedPasswordMethod",
        ("FIX.5.0", "1401"): "1401 EncryptedPasswordLen",
        ("FIX.5.0", "1402"): "1402 EncryptedPassword",
        ("FIX.5.0", "1403"): "1403 EncryptedNewPasswordLen",
        ("FIX.5.0", "1404"): "1404 EncryptedNewPassword",
        ("FIX.5.0", "1405"): "1405 UnderlyingLegMaturityTime",
        ("FIX.5.0", "1406"): "1406 RefApplExtID",
        ("FIX.5.0", "1407"): "1407 DefaultApplExtID",
        ("FIX.5.0", "1408"): "1408 DefaultCstmApplVerID",
        ("FIX.5.0", "1409"): "1409 SessionStatus",
        ("FIX.5.0", "1410"): "1410 DefaultVerIndicator",
        ("FIX.5.0", "1411"): "1411 Nested4PartySubIDType",
        ("FIX.5.0", "1412"): "1412 Nested4PartySubID",
        ("FIX.5.0", "1413"): "1413 NoNested4PartySubIDs",
        ("FIX.5.0", "1414"): "1414 NoNested4PartyIDs",
        ("FIX.5.0", "1415"): "1415 Nested4PartyID",
        ("FIX.5.0", "1416"): "1416 Nested4PartyIDSource",
        ("FIX.5.0", "1417"): "1417 Nested4PartyRole",
        ("FIX.5.0", "1418"): "1418 LegLastQty",
        ("FIX.5.0", "1419"): "1419 UnderlyingExerciseStyle",
        ("FIX.5.0", "1420"): "1420 LegExerciseStyle",
        ("FIX.5.0", "1421"): "1421 LegPriceUnitOfMeasure",
        ("FIX.5.0", "1422"): "1422 LegPriceUnitOfMeasureQty",
        ("FIX.5.0", "1423"): "1423 UnderlyingUnitOfMeasureQty",
        ("FIX.5.0", "1424"): "1424 UnderlyingPriceUnitOfMeasure",
        ("FIX.5.0", "1425"): "1425 UnderlyingPriceUnitOfMeasureQty",
        ("FIX.5.0", "1426"): "1426 ApplReportType",
        ("FIX.5.0", "1427"): "1427 SideExecID",
        ("FIX.5.0", "1428"): "1428 OrderDelay",
        ("FIX.5.0", "1429"): "1429 OrderDelayUnit",
        ("FIX.5.0", "1430"): "1430 VenueType",
        ("FIX.5.0", "1431"): "1431 RefOrdIDReason",
        ("FIX.5.0", "1432"): "1432 OrigCustOrderCapacity",
        ("FIX.5.0", "1433"): "1433 RefApplReqID",
        ("FIX.5.0", "1434"): "1434 ModelType",
        ("FIX.5.0", "1435"): "1435 ContractMultiplierUnit",
        ("FIX.5.0", "1436"): "1436 LegContractMultiplierUnit",
        ("FIX.5.0", "1437"): "1437 UnderlyingContractMultiplierUnit",
        ("FIX.5.0", "1438"): "1438 DerivativeContractMultiplierUnit",
        ("FIX.5.0", "1439"): "1439 FlowScheduleType",
        ("FIX.5.0", "1440"): "1440 LegFlowScheduleType",
        ("FIX.5.0", "1441"): "1441 UnderlyingFlowScheduleType",
        ("FIX.5.0", "1442"): "1442 DerivativeFlowScheduleType",
        ("FIX.5.0", "1443"): "1443 FillLiquidityInd",
        ("FIX.5.0", "1444"): "1444 SideLiquidityInd",
        ("FIX.5.0", "1445"): "1445 NoRateSources",
        ("FIX.5.0", "1446"): "1446 RateSource",
        ("FIX.5.0", "1447"): "1447 RateSourceType",
        ("FIX.5.0", "1448"): "1448 ReferencePage",
        ("FIX.5.0", "1449"): "1449 RestructuringType",
        ("FIX.5.0", "1450"): "1450 Seniority",
        ("FIX.5.0", "1451"): "1451 NotionalPercentageOutstanding",
        ("FIX.5.0", "1452"): "1452 OriginalNotionalPercentageOutstanding",
        ("FIX.5.0", "1453"): "1453 UnderlyingRestructuringType",
        ("FIX.5.0", "1454"): "1454 UnderlyingSeniority",
        ("FIX.5.0", "1455"): "1455 UnderlyingNotionalPercentageOutstanding",
        ("FIX.5.0", "1456"): "1456 UnderlyingOriginalNotionalPercentageOutstanding",
        ("FIX.5.0", "1457"): "1457 AttachmentPoint",
        ("FIX.5.0", "1458"): "1458 DetachmentPoint",
        ("FIX.5.0", "1459"): "1459 UnderlyingAttachmentPoint",
        ("FIX.5.0", "1460"): "1460 UnderlyingDetachmentPoint",
        ("FIX.5.0", "1461"): "1461 NoTargetPartyIDs",
        ("FIX.5.0", "1462"): "1462 TargetPartyID",
        ("FIX.5.0", "1463"): "1463 TargetPartyIDSource",
        ("FIX.5.0", "1464"): "1464 TargetPartyRole",
        ("FIX.5.0", "1465"): "1465 SecurityListID",
        ("FIX.5.0", "1466"): "1466 SecurityListRefID",
        ("FIX.5.0", "1467"): "1467 SecurityListDesc",
        ("FIX.5.0", "1468"): "1468 EncodedSecurityListDescLen",
        ("FIX.5.0", "1469"): "1469 EncodedSecurityListDesc",
        ("FIX.5.0", "1470"): "1470 SecurityListType",
        ("FIX.5.0", "1471"): "1471 SecurityListTypeSource",
        ("FIX.5.0", "1472"): "1472 NewsID",
        ("FIX.5.0", "1473"): "1473 NewsCategory",
        ("FIX.5.0", "1474"): "1474 LanguageCode",
        ("FIX.5.0", "1475"): "1475 NoNewsRefIDs",
        ("FIX.5.0", "1476"): "1476 NewsRefID",
        ("FIX.5.0", "1477"): "1477 NewsRefType",
        ("FIX.5.0", "1478"): "1478 StrikePriceDeterminationMethod",
        ("FIX.5.0", "1479"): "1479 StrikePriceBoundaryMethod",
        ("FIX.5.0", "1480"): "1480 StrikePriceBoundaryPrecision",
        ("FIX.5.0", "1481"): "1481 UnderlyingPriceDeterminationMethod",
        ("FIX.5.0", "1482"): "1482 OptPayoutType",
        ("FIX.5.0", "1483"): "1483 NoComplexEvents",
        ("FIX.5.0", "1484"): "1484 ComplexEventType",
        ("FIX.5.0", "1485"): "1485 ComplexOptPayoutAmount",
        ("FIX.5.0", "1486"): "1486 ComplexEventPrice",
        ("FIX.5.0", "1487"): "1487 ComplexEventPriceBoundaryMethod",
        ("FIX.5.0", "1488"): "1488 ComplexEventPriceBoundaryPrecision",
        ("FIX.5.0", "1489"): "1489 ComplexEventPriceTimeType",
        ("FIX.5.0", "1490"): "1490 ComplexEventCondition",
        ("FIX.5.0", "1491"): "1491 NoComplexEventDates",
        ("FIX.5.0", "1492"): "1492 ComplexEventStartDate",
        ("FIX.5.0", "1493"): "1493 ComplexEventEndDate",
        ("FIX.5.0", "1494"): "1494 NoComplexEventTimes",
        ("FIX.5.0", "1495"): "1495 ComplexEventStartTime",
        ("FIX.5.0", "1496"): "1496 ComplexEventEndTime",
        ("FIX.5.0", "1497"): "1497 StreamAsgnReqID",
        ("FIX.5.0", "1498"): "1498 StreamAsgnReqType",
        ("FIX.5.0", "1499"): "1499 NoAsgnReqs",
        ("FIX.5.0", "1500"): "1500 MDStreamID",
        ("FIX.5.0", "1501"): "1501 StreamAsgnRptID",
        ("FIX.5.0", "1502"): "1502 StreamAsgnRejReason",
        ("FIX.5.0", "1503"): "1503 StreamAsgnAckType",
        ("FIX.5.0", "1504"): "1504 RelSymTransactTime",
        ("FIX.5.0", "1505"): "1505 PartyDetailsListRequestID",
        ("FIX.5.0", "1506"): "1506 SideTradeID",
        ("FIX.5.0", "1507"): "1507 SideOrigTradeID",
        ("FIX.5.0", "1508"): "1508 NoRequestedPartyRoles",
        ("FIX.5.0", "1509"): "1509 RequestedPartyRole",
        ("FIX.5.0", "1510"): "1510 PartyDetailsListReportID",
        ("FIX.5.0", "1511"): "1511 RequestResult",
        ("FIX.5.0", "1512"): "1512 TotNoParties",
        ("FIX.5.0", "1513"): "1513 DocumentationText",
        ("FIX.5.0", "1514"): "1514 NoPartyRelationships",
        ("FIX.5.0", "1515"): "1515 PartyRelationship",
        ("FIX.5.0", "1516"): "1516 NoPartyDetailAltID",
        ("FIX.5.0", "1517"): "1517 PartyDetailAltID",
        ("FIX.5.0", "1518"): "1518 PartyDetailAltIDSource",
        ("FIX.5.0", "1519"): "1519 NoPartyDetailAltSubIDs",
        ("FIX.5.0", "1520"): "1520 PartyDetailAltSubID",
        ("FIX.5.0", "1521"): "1521 PartyDetailAltSubIDType",
        ("FIX.5.0", "1522"): "1522 DifferentialPrice",
        ("FIX.5.0", "1523"): "1523 TrdAckStatus",
        ("FIX.5.0", "1524"): "1524 PriceQuoteCurrency",
        ("FIX.5.0", "1525"): "1525 EncodedDocumentationTextLen",
        ("FIX.5.0", "1526"): "1526 UnderlyingPriceQuoteCurrency",
        ("FIX.5.0", "1527"): "1527 EncodedDocumentationText",
        ("FIX.5.0", "1528"): "1528 LegPriceQuoteCurrency",
        ("FIX.5.0", "1529"): "1529 NoRiskLimitTypes",
        ("FIX.5.0", "1530"): "1530 RiskLimitType",
        ("FIX.5.0", "1531"): "1531 RiskLimitAmount",
        ("FIX.5.0", "1532"): "1532 RiskLimitCurrency",
        ("FIX.5.0", "1533"): "1533 RiskLimitPlatform",
        ("FIX.5.0", "1534"): "1534 NoRiskInstrumentScopes",
        ("FIX.5.0", "1535"): "1535 InstrumentScopeOperator",
        ("FIX.5.0", "1536"): "1536 InstrumentScopeSymbol",
        ("FIX.5.0", "1537"): "1537 InstrumentScopeSymbolSfx",
        ("FIX.5.0", "1538"): "1538 InstrumentScopeSecurityID",
        ("FIX.5.0", "1539"): "1539 InstrumentScopeSecurityIDSource",
        ("FIX.5.0", "1540"): "1540 NoInstrumentScopeSecurityAltID",
        ("FIX.5.0", "1541"): "1541 InstrumentScopeSecurityAltID",
        ("FIX.5.0", "1542"): "1542 InstrumentScopeSecurityAltIDSource",
        ("FIX.5.0", "1543"): "1543 InstrumentScopeProduct",
        ("FIX.5.0", "1544"): "1544 InstrumentScopeProductComplex",
        ("FIX.5.0", "1545"): "1545 InstrumentScopeSecurityGroup",
        ("FIX.5.0", "1546"): "1546 InstrumentScopeCFICode",
        ("FIX.5.0", "1547"): "1547 InstrumentScopeSecurityType",
        ("FIX.5.0", "1548"): "1548 InstrumentScopeSecuritySubType",
        ("FIX.5.0", "1549"): "1549 InstrumentScopeMaturityMonthYear",
        ("FIX.5.0", "1550"): "1550 InstrumentScopeMaturityTime",
        ("FIX.5.0", "1551"): "1551 InstrumentScopeRestructuringType",
        ("FIX.5.0", "1552"): "1552 InstrumentScopeSeniority",
        ("FIX.5.0", "1553"): "1553 InstrumentScopePutOrCall",
        ("FIX.5.0", "1554"): "1554 InstrumentScopeFlexibleIndicator",
        ("FIX.5.0", "1555"): "1555 InstrumentScopeCouponRate",
        ("FIX.5.0", "1556"): "1556 InstrumentScopeSecurityDesc",
        ("FIX.5.0", "1557"): "1557 InstrumentScopeSettlType",
        ("FIX.5.0", "1558"): "1558 RiskInstrumentMultiplier",
        ("FIX.5.0", "1559"): "1559 NoRiskWarningLevels",
        ("FIX.5.0", "1560"): "1560 RiskWarningLevelPercent",
        ("FIX.5.0", "1561"): "1561 RiskWarningLevelName",
        ("FIX.5.0", "1562"): "1562 NoRelatedPartyDetailID",
        ("FIX.5.0", "1563"): "1563 RelatedPartyDetailID",
        ("FIX.5.0", "1564"): "1564 RelatedPartyDetailIDSource",
        ("FIX.5.0", "1565"): "1565 RelatedPartyDetailRole",
        ("FIX.5.0", "1566"): "1566 NoRelatedPartyDetailSubIDs",
        ("FIX.5.0", "1567"): "1567 RelatedPartyDetailSubID",
        ("FIX.5.0", "1568"): "1568 RelatedPartyDetailSubIDType",
        ("FIX.5.0", "1569"): "1569 NoRelatedPartyDetailAltID",
        ("FIX.5.0", "1570"): "1570 RelatedPartyDetailAltID",
        ("FIX.5.0", "1571"): "1571 RelatedPartyDetailAltIDSource",
        ("FIX.5.0", "1572"): "1572 NoRelatedPartyDetailAltSubIDs",
        ("FIX.5.0", "1573"): "1573 RelatedPartyDetailAltSubID",
        ("FIX.5.0", "1574"): "1574 RelatedPartyDetailAltSubIDType",
        ("FIX.5.0", "1575"): "1575 SwapSubClass",
        ("FIX.5.0", "1576"): "1576 DerivativePriceQuoteCurrency",
        ("FIX.5.0", "1577"): "1577 SettlRateIndex",
        ("FIX.5.0", "1578"): "1578 EncodedEventTextLen",
        ("FIX.5.0", "1579"): "1579 EncodedEventText",
        ("FIX.5.0", "1580"): "1580 SettlRateIndexLocation",
        ("FIX.5.0", "1581"): "1581 OptionExpirationDesc",
        ("FIX.5.0", "1582"): "1582 NoSecurityClassifications",
        ("FIX.5.0", "1583"): "1583 SecurityClassificationReason",
        ("FIX.5.0", "1584"): "1584 SecurityClassificationValue",
        ("FIX.5.0", "1585"): "1585 PosAmtReason",
        ("FIX.5.0", "1586"): "1586 NoLegPosAmt",
        ("FIX.5.0", "1587"): "1587 LegPosAmt",
        ("FIX.5.0", "1588"): "1588 LegPosAmtType",
        ("FIX.5.0", "1589"): "1589 LegPosCurrency",
        ("FIX.5.0", "1590"): "1590 LegPosAmtReason",
        ("FIX.5.0", "1591"): "1591 LegQtyType",
        ("FIX.5.0", "1592"): "1592 DiscountFactor",
        ("FIX.5.0", "1593"): "1593 ParentAllocID",
        ("FIX.5.0", "1594"): "1594 LegSecurityGroup",
        ("FIX.5.0", "1595"): "1595 PositionContingentPrice",
        ("FIX.5.0", "1596"): "1596 ClearingTradePrice",
        ("FIX.5.0", "1597"): "1597 SideClearingTradePrice",
        ("FIX.5.0", "1598"): "1598 SideClearingTradePriceType",
        ("FIX.5.0", "1599"): "1599 SidePriceDifferential",
        ("FIX.5.0", "1600"): "1600 FIXEngineName",
        ("FIX.5.0", "1601"): "1601 FIXEngineVersion",
        ("FIX.5.0", "1602"): "1602 FIXEngineVendor",
        ("FIX.5.0", "1603"): "1603 ApplicationSystemName",
        ("FIX.5.0", "1604"): "1604 ApplicationSystemVersion",
        ("FIX.5.0", "1605"): "1605 ApplicationSystemVendor",
        ("FIX.5.0", "1606"): "1606 NumOfSimpleInstruments",
        ("FIX.5.0", "1607"): "1607 SecurityRejectReason",
        ("FIX.5.0", "1608"): "1608 InitialDisplayQty",
        ("FIX.5.0", "1609"): "1609 ThrottleStatus",
        ("FIX.5.0", "1610"): "1610 NoThrottles",
        ("FIX.5.0", "1611"): "1611 ThrottleAction",
        ("FIX.5.0", "1612"): "1612 ThrottleType",
        ("FIX.5.0", "1613"): "1613 ThrottleNoMsgs",
        ("FIX.5.0", "1614"): "1614 ThrottleTimeInterval",
        ("FIX.5.0", "1615"): "1615 ThrottleTimeUnit",
        ("FIX.5.0", "1616"): "1616 InstrumentScopeSecurityExchange",
        ("FIX.5.0", "1617"): "1617 StreamAsgnType",
        ("FIX.5.0", "1618"): "1618 NoThrottleMsgType",
        ("FIX.5.0", "1619"): "1619 ThrottleMsgType",
        ("FIX.5.0", "1620"): "1620 InstrumentScopeEncodedSecurityDescLen",
        ("FIX.5.0", "1621"): "1621 InstrumentScopeEncodedSecurityDesc",
        ("FIX.5.0", "1622"): "1622 FillYieldType",
        ("FIX.5.0", "1623"): "1623 FillYield",
        ("FIX.5.0", "1624"): "1624 NoMatchInst",
        ("FIX.5.0", "1625"): "1625 MatchInst",
        ("FIX.5.0", "1626"): "1626 MatchAttribTagID",
        ("FIX.5.0", "1627"): "1627 MatchAttribValue",
        ("FIX.5.0", "1628"): "1628 TriggerScope",
        ("FIX.5.0", "1629"): "1629 ExposureDuration",
        ("FIX.5.0", "1630"): "1630 NoLimitAmts",
        ("FIX.5.0", "1631"): "1631 LimitAmtType",
        ("FIX.5.0", "1632"): "1632 LastLimitAmt",
        ("FIX.5.0", "1633"): "1633 LimitAmtRemaining",
        ("FIX.5.0", "1634"): "1634 LimitAmtCurrency",
        ("FIX.5.0", "1635"): "1635 MarginReqmtInqID",
        ("FIX.5.0", "1636"): "1636 NoMarginReqmtInqQualifier",
        ("FIX.5.0", "1637"): "1637 MarginReqmtInqQualifier",
        ("FIX.5.0", "1638"): "1638 MarginReqmtRptType",
        ("FIX.5.0", "1639"): "1639 MarginClass",
        ("FIX.5.0", "1640"): "1640 MarginReqmtInqStatus",
        ("FIX.5.0", "1641"): "1641 MarginReqmtInqResult",
        ("FIX.5.0", "1642"): "1642 MarginReqmtRptID",
        ("FIX.5.0", "1643"): "1643 NoMarginAmt",
        ("FIX.5.0", "1644"): "1644 MarginAmtType",
        ("FIX.5.0", "1645"): "1645 MarginAmt",
        ("FIX.5.0", "1646"): "1646 MarginAmtCcy",
        ("FIX.5.0", "1647"): "1647 NoRelatedInstruments",
        ("FIX.5.0", "1648"): "1648 RelatedInstrumentType",
        ("FIX.5.0", "1649"): "1649 RelatedSymbol",
        ("FIX.5.0", "1650"): "1650 RelatedSecurityID",
        ("FIX.5.0", "1651"): "1651 RelatedSecurityIDSource",
        ("FIX.5.0", "1652"): "1652 RelatedSecurityType",
        ("FIX.5.0", "1653"): "1653 RelatedMaturityMonthYear",
        ("FIX.5.0", "1654"): "1654 CoveredQty",
        ("FIX.5.0", "1655"): "1655 MarketMakerActivity",
        ("FIX.5.0", "1656"): "1656 NoInstrumentScopes",
        ("FIX.5.0", "1657"): "1657 NoRequestingPartyIDs",
        ("FIX.5.0", "1658"): "1658 RequestingPartyID",
        ("FIX.5.0", "1659"): "1659 RequestingPartyIDSource",
        ("FIX.5.0", "1660"): "1660 RequestingPartyRole",
        ("FIX.5.0", "1661"): "1661 NoRequestingPartySubIDs",
        ("FIX.5.0", "1662"): "1662 RequestingPartySubID",
        ("FIX.5.0", "1663"): "1663 RequestingPartySubIDType",
        ("FIX.5.0", "1664"): "1664 EncodedRejectTextLen",
        ("FIX.5.0", "1665"): "1665 EncodedRejectText",
        ("FIX.5.0", "1666"): "1666 RiskLimitRequestID",
        ("FIX.5.0", "1667"): "1667 RiskLimitReportID",
        ("FIX.5.0", "1668"): "1668 NoRequestedRiskLimitType",
        ("FIX.5.0", "1669"): "1669 NoRiskLimits",
        ("FIX.5.0", "1670"): "1670 RiskLimitID",
        ("FIX.5.0", "1671"): "1671 NoPartyDetails",
        ("FIX.5.0", "1672"): "1672 PartyDetailStatus",
        ("FIX.5.0", "1673"): "1673 MatchInstMarketID",
        ("FIX.5.0", "1674"): "1674 PartyDetailRoleQualifier",
        ("FIX.5.0", "1675"): "1675 RelatedPartyDetailRoleQualifier",
        ("FIX.5.0", "1676"): "1676 NoPartyUpdates",
        ("FIX.5.0", "1677"): "1677 NoPartyRiskLimits",
        ("FIX.5.0", "1678"): "1678 EncodedOptionExpirationDescLen",
        ("FIX.5.0", "1679"): "1679 SecurityMassTradingStatus",
        ("FIX.5.0", "1680"): "1680 SecurityMassTradingEvent",
        ("FIX.5.0", "1681"): "1681 MassHaltReason",
        ("FIX.5.0", "1682"): "1682 MDSecurityTradingStatus",
        ("FIX.5.0", "1683"): "1683 MDSubFeedType",
        ("FIX.5.0", "1684"): "1684 MDHaltReason",
        ("FIX.5.0", "1685"): "1685 ThrottleInst",
        ("FIX.5.0", "1686"): "1686 ThrottleCountIndicator",
        ("FIX.5.0", "1687"): "1687 ShortSaleRestriction",
        ("FIX.5.0", "1688"): "1688 ShortSaleExemptionReason",
        ("FIX.5.0", "1689"): "1689 LegShortSaleExemptionReason",
        ("FIX.5.0", "1690"): "1690 SideShortSaleExemptionReason",
        ("FIX.5.0", "1691"): "1691 PartyDetailID",
        ("FIX.5.0", "1692"): "1692 PartyDetailIDSource",
        ("FIX.5.0", "1693"): "1693 PartyDetailRole",
        ("FIX.5.0", "1694"): "1694 NoPartyDetailSubIDs",
        ("FIX.5.0", "1695"): "1695 PartyDetailSubID",
        ("FIX.5.0", "1696"): "1696 PartyDetailSubIDType",
        ("FIX.5.0", "1697"): "1697 EncodedOptionExpirationDesc",
        ("FIX.5.0", "1698"): "1698 StrikeUnitOfMeasure",
        ("FIX.5.0", "1699"): "1699 AccountSummaryReportID",
        ("FIX.5.0", "1700"): "1700 NoSettlementAmounts",
        ("FIX.5.0", "1701"): "1701 SettlementAmount",
        ("FIX.5.0", "1702"): "1702 SettlementAmountCurrency",
        ("FIX.5.0", "1703"): "1703 NoCollateralAmounts",
        ("FIX.5.0", "1704"): "1704 CurrentCollateralAmount",
        ("FIX.5.0", "1705"): "1705 CollateralCurrency",
        ("FIX.5.0", "1706"): "1706 CollateralType",
        ("FIX.5.0", "1707"): "1707 NoPayCollects",
        ("FIX.5.0", "1708"): "1708 PayCollectType",
        ("FIX.5.0", "1709"): "1709 PayCollectCurrency",
        ("FIX.5.0", "1710"): "1710 PayAmount",
        ("FIX.5.0", "1711"): "1711 CollectAmount",
        ("FIX.5.0", "1712"): "1712 PayCollectMarketSegmentID",
        ("FIX.5.0", "1713"): "1713 PayCollectMarketID",
        ("FIX.5.0", "1714"): "1714 MarginAmountMarketSegmentID",
        ("FIX.5.0", "1715"): "1715 MarginAmountMarketID",
        ("FIX.5.0", "1716"): "1716 UnitOfMeasureCurrency",
        ("FIX.5.0", "1717"): "1717 PriceUnitOfMeasureCurrency",
        ("FIX.5.0", "1718"): "1718 UnderlyingUnitOfMeasureCurrency",
        ("FIX.5.0", "1719"): "1719 UnderlyingPriceUnitOfMeasureCurrency",
        ("FIX.5.0", "1720"): "1720 LegUnitOfMeasureCurrency",
        ("FIX.5.0", "1721"): "1721 LegPriceUnitOfMeasureCurrency",
        ("FIX.5.0", "1722"): "1722 DerivativeUnitOfMeasureCurrency",
        ("FIX.5.0", "1723"): "1723 DerivativePriceUnitOfMeasureCurrency",
        ("FIX.5.0", "1724"): "1724 OrderOrigination",
        ("FIX.5.0", "1725"): "1725 OriginatingDeptID",
        ("FIX.5.0", "1726"): "1726 ReceivingDeptID",
        ("FIX.5.0", "1727"): "1727 InformationBarrierID",
        ("FIX.5.0", "1728"): "1728 FirmGroupID",
        ("FIX.5.0", "1729"): "1729 FirmMnemonic",
        ("FIX.5.0", "1730"): "1730 AllocGroupID",
        ("FIX.5.0", "1731"): "1731 AvgPxGroupID",
        ("FIX.5.0", "1732"): "1732 FirmAllocText",
        ("FIX.5.0", "1733"): "1733 EncodedFirmAllocTextLen",
        ("FIX.5.0", "1734"): "1734 EncodedFirmAllocText",
        ("FIX.5.0", "1735"): "1735 AllocationRollupInstruction",
        ("FIX.5.0", "1736"): "1736 AllocGroupQuantity",
        ("FIX.5.0", "1737"): "1737 AllocGroupRemainingQuantity",
        ("FIX.5.0", "1738"): "1738 AllocReversalStatus",
        ("FIX.5.0", "1739"): "1739 ObligationType",
        ("FIX.5.0", "1740"): "1740 TradePriceNegotiationMethod",
        ("FIX.5.0", "1741"): "1741 UpfrontPriceType",
        ("FIX.5.0", "1742"): "1742 UpfrontPrice",
        ("FIX.5.0", "1743"): "1743 LastUpfrontPrice",
        ("FIX.5.0", "1744"): "1744 ApplLevelRecoveryIndicator",
        ("FIX.5.0", "1745"): "1745 BidMDEntryID",
        ("FIX.5.0", "1746"): "1746 OfferMDEntryID",
        ("FIX.5.0", "1747"): "1747 BidQuoteID",
        ("FIX.5.0", "1748"): "1748 OfferQuoteID",
        ("FIX.5.0", "1749"): "1749 TotalBidSize",
        ("FIX.5.0", "1750"): "1750 TotalOfferSize",
        ("FIX.5.0", "1751"): "1751 SecondaryQuoteID",
        ("FIX.5.0", "1752"): "1752 CustodialLotID",
        ("FIX.5.0", "1753"): "1753 VersusPurchaseDate",
        ("FIX.5.0", "1754"): "1754 VersusPurchasePrice",
        ("FIX.5.0", "1755"): "1755 CurrentCostBasis",
        ("FIX.5.0", "1756"): "1756 LegCustodialLotID",
        ("FIX.5.0", "1757"): "1757 LegVersusPurchaseDate",
        ("FIX.5.0", "1758"): "1758 LegVersusPurchasePrice",
        ("FIX.5.0", "1759"): "1759 LegCurrentCostBasis",
        ("FIX.5.0", "1760"): "1760 RiskLimitRequestType",
        ("FIX.5.0", "1761"): "1761 RiskLimitRequestResult",
        ("FIX.5.0", "1762"): "1762 RiskLimitRequestStatus",
        ("FIX.5.0", "1763"): "1763 RiskLimitStatus",
        ("FIX.5.0", "1764"): "1764 RiskLimitResult",
        ("FIX.5.0", "1765"): "1765 RiskLimitUtilizationPercent",
        ("FIX.5.0", "1766"): "1766 RiskLimitUtilizationAmount",
        ("FIX.5.0", "1767"): "1767 RiskLimitAction",
        ("FIX.5.0", "1768"): "1768 RiskWarningLevelAmount",
        ("FIX.5.0", "1769"): "1769 RiskWarningLevelAction",
        ("FIX.5.0", "1770"): "1770 EntitlementRequestID",
        ("FIX.5.0", "1771"): "1771 EntitlementReportID",
        ("FIX.5.0", "1772"): "1772 NoPartyEntitlements",
        ("FIX.5.0", "1773"): "1773 NoEntitlements",
        ("FIX.5.0", "1774"): "1774 EntitlementIndicator",
        ("FIX.5.0", "1775"): "1775 EntitlementType",
        ("FIX.5.0", "1776"): "1776 EntitlementID",
        ("FIX.5.0", "1777"): "1777 NoEntitlementAttrib",
        ("FIX.5.0", "1778"): "1778 EntitlementAttribType",
        ("FIX.5.0", "1779"): "1779 EntitlementAttribDatatype",
        ("FIX.5.0", "1780"): "1780 EntitlementAttribValue",
        ("FIX.5.0", "1781"): "1781 EntitlementAttribCurrency",
        ("FIX.5.0", "1782"): "1782 EntitlementStartDate",
        ("FIX.5.0", "1783"): "1783 EntitlementEndDate",
        ("FIX.5.0", "1784"): "1784 EntitlementPlatform",
        ("FIX.5.0", "1785"): "1785 TradSesControl",
        ("FIX.5.0", "1786"): "1786 TradeVolType",
        ("FIX.5.0", "1787"): "1787 RefTickTableID",
        ("FIX.5.0", "1788"): "1788 LegID",
        ("FIX.5.0", "1789"): "1789 NoTargetMarketSegments",
        ("FIX.5.0", "1790"): "1790 TargetMarketSegmentID",
        ("FIX.5.0", "1791"): "1791 NoAffectedMarketSegments",
        ("FIX.5.0", "1792"): "1792 AffectedMarketSegmentID",
        ("FIX.5.0", "1793"): "1793 NoNotAffectedMarketSegments",
        ("FIX.5.0", "1794"): "1794 NotAffectedMarketSegmentID",
        ("FIX.5.0", "1795"): "1795 NoOrderEvents",
        ("FIX.5.0", "1796"): "1796 OrderEventType",
        ("FIX.5.0", "1797"): "1797 OrderEventExecID",
        ("FIX.5.0", "1798"): "1798 OrderEventReason",
        ("FIX.5.0", "1799"): "1799 OrderEventPx",
        ("FIX.5.0", "1800"): "1800 OrderEventQty",
        ("FIX.5.0", "1801"): "1801 OrderEventLiquidityIndicator",
        ("FIX.5.0", "1802"): "1802 OrderEventText",
        ("FIX.5.0", "1803"): "1803 AuctionType",
        ("FIX.5.0", "1804"): "1804 AuctionAllocationPct",
        ("FIX.5.0", "1805"): "1805 AuctionInstruction",
        ("FIX.5.0", "1806"): "1806 RefClOrdID",
        ("FIX.5.0", "1807"): "1807 LockType",
        ("FIX.5.0", "1808"): "1808 LockedQty",
        ("FIX.5.0", "1809"): "1809 SecondaryLockedQty",
        ("FIX.5.0", "1810"): "1810 ReleaseInstruction",
        ("FIX.5.0", "1811"): "1811 ReleaseQty",
        ("FIX.5.0", "1812"): "1812 NoDisclosureInstructions",
        ("FIX.5.0", "1813"): "1813 DisclosureType",
        ("FIX.5.0", "1814"): "1814 DisclosureInstruction",
        ("FIX.5.0", "1815"): "1815 TradingCapacity",
        ("FIX.5.0", "1816"): "1816 ClearingAccountType",
        ("FIX.5.0", "1817"): "1817 LegClearingAccountType",
        ("FIX.5.0", "1818"): "1818 TargetPartyRoleQualifier",
        ("FIX.5.0", "1819"): "1819 RelatedHighPrice",
        ("FIX.5.0", "1820"): "1820 RelatedLowPrice",
        ("FIX.5.0", "1821"): "1821 RelatedPriceSource",
        ("FIX.5.0", "1822"): "1822 MinQtyMethod",
        ("FIX.5.0", "1823"): "1823 Triggered",
        ("FIX.5.0", "1824"): "1824 AffectedOrigClOrdID",
        ("FIX.5.0", "1825"): "1825 NotAffSecondaryOrderID",
        ("FIX.5.0", "1826"): "1826 EventTimePeriod",
        ("FIX.5.0", "1827"): "1827 EventTimeUnit",
        ("FIX.5.0", "1828"): "1828 LastQtyVariance",
        ("FIX.5.0", "1829"): "1829 NoCrossLegs",
        ("FIX.5.0", "1830"): "1830 SettlPriceIncrement",
        ("FIX.5.0", "1831"): "1831 SettlPriceSecondaryIncrement",
        ("FIX.5.0", "1832"): "1832 ClearedIndicator",
        ("FIX.5.0", "1833"): "1833 ContractRefPosType",
        ("FIX.5.0", "1834"): "1834 PositionCapacity",
        ("FIX.5.0", "1835"): "1835 PosQtyUnitOfMeasureCurrency",
        ("FIX.5.0", "1836"): "1836 PosQtyUnitOfMeasure",
        ("FIX.5.0", "1837"): "1837 UnderlyingContractPriceRefMonth",
        ("FIX.5.0", "1838"): "1838 NoTradePriceConditions",
        ("FIX.5.0", "1839"): "1839 TradePriceCondition",
        ("FIX.5.0", "1840"): "1840 TradeAllocStatus",
        ("FIX.5.0", "1841"): "1841 NoTradeQtys",
        ("FIX.5.0", "1842"): "1842 TradeQtyType",
        ("FIX.5.0", "1843"): "1843 TradeQty",
        ("FIX.5.0", "1844"): "1844 NoTradeAllocAmts",
        ("FIX.5.0", "1845"): "1845 TradeAllocAmtType",
        ("FIX.5.0", "1846"): "1846 TradeAllocAmt",
        ("FIX.5.0", "1847"): "1847 TradeAllocCurrency",
        ("FIX.5.0", "1848"): "1848 TradeAllocGroupInstruction",
        ("FIX.5.0", "1849"): "1849 OffsetInstruction",
        ("FIX.5.0", "1850"): "1850 TradeAllocAmtReason",
        ("FIX.5.0", "1851"): "1851 StrategyLinkID",
        ("FIX.5.0", "1852"): "1852 SideAvgPx",
        ("FIX.5.0", "1853"): "1853 SideAvgPxIndicator",
        ("FIX.5.0", "1854"): "1854 SideAvgPxGroupID",
        ("FIX.5.0", "1855"): "1855 NoRelatedTrades",
        ("FIX.5.0", "1856"): "1856 RelatedTradeID",
        ("FIX.5.0", "1857"): "1857 RelatedTradeIDSource",
        ("FIX.5.0", "1858"): "1858 RelatedTradeDate",
        ("FIX.5.0", "1859"): "1859 RelatedTradeMarketID",
        ("FIX.5.0", "1860"): "1860 RelatedTradeQuantity",
        ("FIX.5.0", "1861"): "1861 NoRelatedPositions",
        ("FIX.5.0", "1862"): "1862 RelatedPositionID",
        ("FIX.5.0", "1863"): "1863 RelatedPositionIDSource",
        ("FIX.5.0", "1864"): "1864 RelatedPositionDate",
        ("FIX.5.0", "1865"): "1865 QuoteAckStatus",
        ("FIX.5.0", "1866"): "1866 StrikeIndex",
        ("FIX.5.0", "1867"): "1867 OfferID",
        ("FIX.5.0", "1868"): "1868 NoValueChecks",
        ("FIX.5.0", "1869"): "1869 ValueCheckType",
        ("FIX.5.0", "1870"): "1870 ValueCheckAction",
        ("FIX.5.0", "1871"): "1871 LegSecurityXMLLen",
        ("FIX.5.0", "1872"): "1872 LegSecurityXML",
        ("FIX.5.0", "1873"): "1873 LegSecurityXMLSchema",
        ("FIX.5.0", "1874"): "1874 UnderlyingSecurityXMLLen",
        ("FIX.5.0", "1875"): "1875 UnderlyingSecurityXML",
        ("FIX.5.0", "1876"): "1876 UnderlyingSecurityXMLSchema",
        ("FIX.5.0", "1877"): "1877 PartyDetailRequestResult",
        ("FIX.5.0", "1878"): "1878 PartyDetailRequestStatus",
        ("FIX.5.0", "1879"): "1879 PartyDetailDefinitionStatus",
        ("FIX.5.0", "1880"): "1880 PartyDetailDefinitionResult",
        ("FIX.5.0", "1881"): "1881 EntitlementRequestResult",
        ("FIX.5.0", "1882"): "1882 EntitlementRequestStatus",
        ("FIX.5.0", "1883"): "1883 EntitlementStatus",
        ("FIX.5.0", "1884"): "1884 EntitlementResult",
        ("FIX.5.0", "1885"): "1885 EntitlementRefID",
        ("FIX.5.0", "1886"): "1886 SettlPriceUnitOfMeasure",
        ("FIX.5.0", "1887"): "1887 SettlPriceUnitOfMeasureCurrency",
        ("FIX.5.0", "1888"): "1888 TradeMatchTimestamp",
        ("FIX.5.0", "1889"): "1889 NoInstrmtMatchSides",
        ("FIX.5.0", "1890"): "1890 NoTrdMatchSides",
        ("FIX.5.0", "1891"): "1891 TrdMatchSubID",
        ("FIX.5.0", "1892"): "1892 NoLegExecs",
        ("FIX.5.0", "1893"): "1893 LegExecID",
        ("FIX.5.0", "1894"): "1894 LegTradeID",
        ("FIX.5.0", "1895"): "1895 LegTradeReportID",
        ("FIX.5.0", "1896"): "1896 TradeMatchAckStatus",
        ("FIX.5.0", "1897"): "1897 TradeMatchRejectReason",
        ("FIX.5.0", "1898"): "1898 SideMarketSegmentID",
        ("FIX.5.0", "1899"): "1899 SideVenueType",
        ("FIX.5.0", "1900"): "1900 SideExecRefID",
        ("FIX.5.0", "1901"): "1901 LegExecRefID",
        ("FIX.5.0", "1902"): "1902 HaircutIndicator",
        ("FIX.5.0", "1903"): "1903 RegulatoryTradeID",
        ("FIX.5.0", "1904"): "1904 RegulatoryTradeIDEvent",
        ("FIX.5.0", "1905"): "1905 RegulatoryTradeIDSource",
        ("FIX.5.0", "1906"): "1906 RegulatoryTradeIDType",
        ("FIX.5.0", "1907"): "1907 NoRegulatoryTradeIDs",
        ("FIX.5.0", "1908"): "1908 NoAllocRegulatoryTradeIDs",
        ("FIX.5.0", "1909"): "1909 AllocRegulatoryTradeID",
        ("FIX.5.0", "1910"): "1910 AllocRegulatoryTradeIDSource",
        ("FIX.5.0", "1911"): "1911 AllocRegulatoryTradeIDEvent",
        ("FIX.5.0", "1912"): "1912 AllocRegulatoryTradeIDType",
        ("FIX.5.0", "1913"): "1913 NumOfCompetitors",
        ("FIX.5.0", "1914"): "1914 ResponseTime",
        ("FIX.5.0", "1915"): "1915 QuoteDisplayTime",
        ("FIX.5.0", "1916"): "1916 ExposureDurationUnit",
        ("FIX.5.0", "1917"): "1917 CoverPrice",
        ("FIX.5.0", "1918"): "1918 NoClearingAccountTypes",
        ("FIX.5.0", "1919"): "1919 NoPriceMovements",
        ("FIX.5.0", "1920"): "1920 NoPriceMovementValues",
        ("FIX.5.0", "1921"): "1921 PriceMovementValue",
        ("FIX.5.0", "1922"): "1922 PriceMovementPoint",
        ("FIX.5.0", "1923"): "1923 PriceMovementType",
        ("FIX.5.0", "1924"): "1924 ClearingIntention",
        ("FIX.5.0", "1925"): "1925 TradeClearingInstruction",
        ("FIX.5.0", "1926"): "1926 BackloadedTradeIndicator",
        ("FIX.5.0", "1927"): "1927 ConfirmationMethod",
        ("FIX.5.0", "1928"): "1928 MandatoryClearingIndicator",
        ("FIX.5.0", "1929"): "1929 MixedSwapIndicator",
        ("FIX.5.0", "1930"): "1930 OffMarketPriceIndicator",
        ("FIX.5.0", "1931"): "1931 VerificationMethod",
        ("FIX.5.0", "1932"): "1932 ClearingRequirementException",
        ("FIX.5.0", "1933"): "1933 IRSDirection",
        ("FIX.5.0", "1934"): "1934 RegulatoryReportType",
        ("FIX.5.0", "1935"): "1935 VoluntaryRegulatoryReport",
        ("FIX.5.0", "1936"): "1936 TradeCollateralization",
        ("FIX.5.0", "1937"): "1937 TradeContinuation",
        ("FIX.5.0", "1938"): "1938 AssetClass",
        ("FIX.5.0", "1939"): "1939 AssetSubClass",
        ("FIX.5.0", "1940"): "1940 AssetType",
        ("FIX.5.0", "1941"): "1941 SwapClass",
        ("FIX.5.0", "1942"): "1942 NthToDefault",
        ("FIX.5.0", "1943"): "1943 MthToDefault",
        ("FIX.5.0", "1944"): "1944 SettledEntityMatrixSource",
        ("FIX.5.0", "1945"): "1945 SettledEntityMatrixPublicationDate",
        ("FIX.5.0", "1946"): "1946 CouponType",
        ("FIX.5.0", "1947"): "1947 TotalIssuedAmount",
        ("FIX.5.0", "1948"): "1948 CouponFrequencyPeriod",
        ("FIX.5.0", "1949"): "1949 CouponFrequencyUnit",
        ("FIX.5.0", "1950"): "1950 CouponDayCount",
        ("FIX.5.0", "1951"): "1951 ConvertibleBondEquityID",
        ("FIX.5.0", "1952"): "1952 ConvertibleBondEquityIDSource",
        ("FIX.5.0", "1953"): "1953 ContractPriceRefMonth",
        ("FIX.5.0", "1954"): "1954 LienSeniority",
        ("FIX.5.0", "1955"): "1955 LoanFacility",
        ("FIX.5.0", "1956"): "1956 ReferenceEntityType",
        ("FIX.5.0", "1957"): "1957 IndexSeries",
        ("FIX.5.0", "1958"): "1958 IndexAnnexVersion",
        ("FIX.5.0", "1959"): "1959 IndexAnnexDate",
        ("FIX.5.0", "1960"): "1960 IndexAnnexSource",
        ("FIX.5.0", "1961"): "1961 AgreementVersion",
        ("FIX.5.0", "1962"): "1962 MasterConfirmationDesc",
        ("FIX.5.0", "1963"): "1963 MasterConfirmationDate",
        ("FIX.5.0", "1964"): "1964 MasterConfirmationAnnexDesc",
        ("FIX.5.0", "1965"): "1965 MasterConfirmationAnnexDate",
        ("FIX.5.0", "1966"): "1966 BrokerConfirmationDesc",
        ("FIX.5.0", "1967"): "1967 CreditSupportAgreementDesc",
        ("FIX.5.0", "1968"): "1968 CreditSupportAgreementDate",
        ("FIX.5.0", "1969"): "1969 CreditSupportAgreementID",
        ("FIX.5.0", "1970"): "1970 GoverningLaw",
        ("FIX.5.0", "1971"): "1971 NoSideRegulatoryTradeIDs",
        ("FIX.5.0", "1972"): "1972 SideRegulatoryTradeID",
        ("FIX.5.0", "1973"): "1973 SideRegulatoryTradeIDSource",
        ("FIX.5.0", "1974"): "1974 SideRegulatoryTradeIDEvent",
        ("FIX.5.0", "1975"): "1975 SideRegulatoryTradeIDType",
        ("FIX.5.0", "1976"): "1976 NoSecondaryAssetClasses",
        ("FIX.5.0", "1977"): "1977 SecondaryAssetClass",
        ("FIX.5.0", "1978"): "1978 SecondaryAssetSubClass",
        ("FIX.5.0", "1979"): "1979 SecondaryAssetType",
        ("FIX.5.0", "1980"): "1980 BlockTrdAllocIndicator",
        ("FIX.5.0", "1981"): "1981 NoUnderlyingEvents",
        ("FIX.5.0", "1982"): "1982 UnderlyingEventType",
        ("FIX.5.0", "1983"): "1983 UnderlyingEventDate",
        ("FIX.5.0", "1984"): "1984 UnderlyingEventTime",
        ("FIX.5.0", "1985"): "1985 UnderlyingEventTimeUnit",
        ("FIX.5.0", "1986"): "1986 UnderlyingEventTimePeriod",
        ("FIX.5.0", "1987"): "1987 UnderlyingEventPx",
        ("FIX.5.0", "1988"): "1988 UnderlyingConstituentWeight",
        ("FIX.5.0", "1989"): "1989 UnderlyingCouponType",
        ("FIX.5.0", "1990"): "1990 UnderlyingTotalIssuedAmount",
        ("FIX.5.0", "1991"): "1991 UnderlyingCouponFrequencyPeriod",
        ("FIX.5.0", "1992"): "1992 UnderlyingCouponFrequencyUnit",
        ("FIX.5.0", "1993"): "1993 UnderlyingCouponDayCount",
        ("FIX.5.0", "1994"): "1994 UnderlyingObligationID",
        ("FIX.5.0", "1995"): "1995 UnderlyingObligationIDSource",
        ("FIX.5.0", "1996"): "1996 UnderlyingEquityID",
        ("FIX.5.0", "1997"): "1997 UnderlyingEquityIDSource",
        ("FIX.5.0", "1998"): "1998 UnderlyingLienSeniority",
        ("FIX.5.0", "1999"): "1999 UnderlyingLoanFacility",
        ("FIX.5.0", "2000"): "2000 UnderlyingReferenceEntityType",
        ("FIX.5.0", "2001"): "2001 StrikeIndexSpread",
        ("FIX.5.0", "2002"): "2002 ValuationSource",
        ("FIX.5.0", "2003"): "2003 UnderlyingIndexSeries",
        ("FIX.5.0", "2004"): "2004 UnderlyingIndexAnnexVersion",
        ("FIX.5.0", "2005"): "2005 UnderlyingIndexAnnexDate",
        ("FIX.5.0", "2006"): "2006 UnderlyingIndexAnnexSource",
        ("FIX.5.0", "2007"): "2007 UnderlyingProductComplex",
        ("FIX.5.0", "2008"): "2008 UnderlyingSecurityGroup",
        ("FIX.5.0", "2009"): "2009 UnderlyingSettleOnOpenFlag",
        ("FIX.5.0", "2010"): "2010 UnderlyingAssignmentMethod",
        ("FIX.5.0", "2011"): "2011 UnderlyingSecurityStatus",
        ("FIX.5.0", "2012"): "2012 UnderlyingObligationType",
        ("FIX.5.0", "2013"): "2013 UnderlyingAssetClass",
        ("FIX.5.0", "2014"): "2014 UnderlyingAssetSubClass",
        ("FIX.5.0", "2015"): "2015 UnderlyingAssetType",
        ("FIX.5.0", "2016"): "2016 UnderlyingSwapClass",
        ("FIX.5.0", "2017"): "2017 UnderlyingNthToDefault",
        ("FIX.5.0", "2018"): "2018 UnderlyingMthToDefault",
        ("FIX.5.0", "2019"): "2019 UnderlyingSettledEntityMatrixSource",
        ("FIX.5.0", "2020"): "2020 UnderlyingSettledEntityMatrixPublicationDate",
        ("FIX.5.0", "2021"): "2021 UnderlyingStrikeMultiplier",
        ("FIX.5.0", "2022"): "2022 UnderlyingStrikeValue",
        ("FIX.5.0", "2023"): "2023 UnderlyingStrikePriceDeterminationMethod",
        ("FIX.5.0", "2024"): "2024 UnderlyingStrikePriceBoundaryMethod",
        ("FIX.5.0", "2025"): "2025 UnderlyingStrikePriceBoundaryPrecision",
        ("FIX.5.0", "2026"): "2026 UnderlyingMinPriceIncrement",
        ("FIX.5.0", "2027"): "2027 UnderlyingMinPriceIncrementAmount",
        ("FIX.5.0", "2028"): "2028 UnderlyingOptPayoutType",
        ("FIX.5.0", "2029"): "2029 UnderlyingOptPayoutAmount",
        ("FIX.5.0", "2030"): "2030 UnderlyingPriceQuoteMethod",
        ("FIX.5.0", "2031"): "2031 UnderlyingValuationMethod",
        ("FIX.5.0", "2032"): "2032 UnderlyingListMethod",
        ("FIX.5.0", "2033"): "2033 UnderlyingCapPrice",
        ("FIX.5.0", "2034"): "2034 UnderlyingFloorPrice",
        ("FIX.5.0", "2035"): "2035 UnderlyingFlexibleIndicator",
        ("FIX.5.0", "2036"): "2036 UnderlyingFlexProductEligibilityIndicator",
        ("FIX.5.0", "2037"): "2037 UnderlyingPositionLimit",
        ("FIX.5.0", "2038"): "2038 UnderlyingNTPositionLimit",
        ("FIX.5.0", "2039"): "2039 UnderlyingPool",
        ("FIX.5.0", "2040"): "2040 UnderlyingContractSettlMonth",
        ("FIX.5.0", "2041"): "2041 UnderlyingDatedDate",
        ("FIX.5.0", "2042"): "2042 UnderlyingInterestAccrualDate",
        ("FIX.5.0", "2043"): "2043 UnderlyingShortSaleRestriction",
        ("FIX.5.0", "2044"): "2044 UnderlyingRefTickTableID",
        ("FIX.5.0", "2045"): "2045 NoUnderlyingComplexEvents",
        ("FIX.5.0", "2046"): "2046 UnderlyingComplexEventType",
        ("FIX.5.0", "2047"): "2047 UnderlyingComplexOptPayoutAmount",
        ("FIX.5.0", "2048"): "2048 UnderlyingComplexEventPrice",
        ("FIX.5.0", "2049"): "2049 UnderlyingComplexEventPriceBoundaryMethod",
        ("FIX.5.0", "2050"): "2050 UnderlyingComplexEventPriceBoundaryPrecision",
        ("FIX.5.0", "2051"): "2051 UnderlyingComplexEventPriceTimeType",
        ("FIX.5.0", "2052"): "2052 UnderlyingComplexEventCondition",
        ("FIX.5.0", "2053"): "2053 NoUnderlyingComplexEventDates",
        ("FIX.5.0", "2054"): "2054 UnderlyingComplexEventStartDate",
        ("FIX.5.0", "2055"): "2055 UnderlyingComplexEventEndDate",
        ("FIX.5.0", "2056"): "2056 NoUnderlyingComplexEventTimes",
        ("FIX.5.0", "2057"): "2057 UnderlyingComplexEventStartTime",
        ("FIX.5.0", "2058"): "2058 UnderlyingComplexEventEndTime",
        ("FIX.5.0", "2059"): "2059 NoLegEvents",
        ("FIX.5.0", "2060"): "2060 LegEventType",
        ("FIX.5.0", "2061"): "2061 LegEventDate",
        ("FIX.5.0", "2062"): "2062 LegEventTime",
        ("FIX.5.0", "2063"): "2063 LegEventTimeUnit",
        ("FIX.5.0", "2064"): "2064 LegEventTimePeriod",
        ("FIX.5.0", "2065"): "2065 LegEventPx",
        ("FIX.5.0", "2066"): "2066 LegEventText",
        ("FIX.5.0", "2067"): "2067 LegAssetClass",
        ("FIX.5.0", "2068"): "2068 LegAssetSubClass",
        ("FIX.5.0", "2069"): "2069 LegAssetType",
        ("FIX.5.0", "2070"): "2070 LegSwapClass",
        ("FIX.5.0", "2071"): "2071 UnderlyingEventText",
        ("FIX.5.0", "2072"): "2072 EncodedUnderlyingEventTextLen",
        ("FIX.5.0", "2073"): "2073 EncodedUnderlyingEventText",
        ("FIX.5.0", "2074"): "2074 EncodedLegEventTextLen",
        ("FIX.5.0", "2075"): "2075 EncodedLegEventText",
        ("FIX.5.0", "2076"): "2076 NoLegSecondaryAssetClasses",
        ("FIX.5.0", "2077"): "2077 LegSecondaryAssetClass",
        ("FIX.5.0", "2078"): "2078 LegSecondaryAssetSubClass",
        ("FIX.5.0", "2079"): "2079 LegSecondaryAssetType",
        ("FIX.5.0", "2080"): "2080 NoUnderlyingSecondaryAssetClasses",
        ("FIX.5.0", "2081"): "2081 UnderlyingSecondaryAssetClass",
        ("FIX.5.0", "2082"): "2082 UnderlyingSecondaryAssetSubClass",
        ("FIX.5.0", "2083"): "2083 UnderlyingSecondaryAssetType",
        ("FIX.5.0", "2084"): "2084 PreviousClearingBusinessDate",
        ("FIX.5.0", "2085"): "2085 ValuationDate",
        ("FIX.5.0", "2086"): "2086 ValuationTime",
        ("FIX.5.0", "2087"): "2087 ValuationBusinessCenter",
        ("FIX.5.0", "2088"): "2088 MarginAmtFXRate",
        ("FIX.5.0", "2089"): "2089 MarginAmtFXRateCalc",
        ("FIX.5.0", "2090"): "2090 CollateralFXRate",
        ("FIX.5.0", "2091"): "2091 CollateralFXRateCalc",
        ("FIX.5.0", "2092"): "2092 CollateralAmountMarketSegmentID",
        ("FIX.5.0", "2093"): "2093 CollateralAmountMarketID",
        ("FIX.5.0", "2094"): "2094 PayCollectFXRate",
        ("FIX.5.0", "2095"): "2095 PayCollectFXRateCalc",
        ("FIX.5.0", "2096"): "2096 PosAmtStreamDesc",
        ("FIX.5.0", "2097"): "2097 PositionFXRate",
        ("FIX.5.0", "2098"): "2098 PositionFXRateCalc",
        ("FIX.5.0", "2099"): "2099 PosAmtMarketSegmentID",
        ("FIX.5.0", "2100"): "2100 PosAmtMarketID",
        ("FIX.5.0", "2101"): "2101 TerminatedIndicator",
        ("FIX.5.0", "2102"): "2102 ShortMarkingExemptIndicator",
        ("FIX.5.0", "2103"): "2103 RelatedRegulatoryTradeIDSource",
        ("FIX.5.0", "2104"): "2104 NoAttachments",
        ("FIX.5.0", "2105"): "2105 AttachmentName",
        ("FIX.5.0", "2106"): "2106 AttachmentMediaType",
        ("FIX.5.0", "2107"): "2107 AttachmentClassification",
        ("FIX.5.0", "2108"): "2108 AttachmentExternalURL",
        ("FIX.5.0", "2109"): "2109 AttachmentEncodingType",
        ("FIX.5.0", "2110"): "2110 UnencodedAttachmentLen",
        ("FIX.5.0", "2111"): "2111 EncodedAttachmentLen",
        ("FIX.5.0", "2112"): "2112 EncodedAttachment",
        ("FIX.5.0", "2113"): "2113 NoAttachmentKeywords",
        ("FIX.5.0", "2114"): "2114 AttachmentKeyword",
        ("FIX.5.0", "2115"): "2115 NegotiationMethod",
        ("FIX.5.0", "2116"): "2116 NextAuctionTime",
        ("FIX.5.0", "2117"): "2117 ComplexOptPayoutPaySide",
        ("FIX.5.0", "2118"): "2118 ComplexOptPayoutReceiveSide",
        ("FIX.5.0", "2119"): "2119 ComplexOptPayoutUnderlier",
        ("FIX.5.0", "2120"): "2120 ComplexOptPayoutPercentage",
        ("FIX.5.0", "2121"): "2121 ComplexOptPayoutTime",
        ("FIX.5.0", "2122"): "2122 ComplexOptPayoutCurrency",
        ("FIX.5.0", "2123"): "2123 ComplexEventPricePercentage",
        ("FIX.5.0", "2124"): "2124 ComplexEventCurrencyOne",
        ("FIX.5.0", "2125"): "2125 ComplexEventCurrencyTwo",
        ("FIX.5.0", "2126"): "2126 ComplexEventQuoteBasis",
        ("FIX.5.0", "2127"): "2127 ComplexEventFixedFXRate",
        ("FIX.5.0", "2128"): "2128 ComplexEventDeterminationMethod",
        ("FIX.5.0", "2129"): "2129 ComplexEventCalculationAgent",
        ("FIX.5.0", "2130"): "2130 ComplexEventStrikePrice",
        ("FIX.5.0", "2131"): "2131 ComplexEventStrikeFactor",
        ("FIX.5.0", "2132"): "2132 ComplexEventStrikeNumberOfOptions",
        ("FIX.5.0", "2133"): "2133 ComplexEventCreditEventsXIDRef",
        ("FIX.5.0", "2134"): "2134 ComplexEventCreditEventNotifyingParty",
        ("FIX.5.0", "2135"): "2135 ComplexEventCreditEventBusinessCenter",
        ("FIX.5.0", "2136"): "2136 ComplexEventCreditEventStandardSources",
        ("FIX.5.0", "2137"): "2137 ComplexEventCreditEventMinimumSources",
        ("FIX.5.0", "2138"): "2138 ComplexEventXID",
        ("FIX.5.0", "2139"): "2139 ComplexEventXIDRef",
        ("FIX.5.0", "2140"): "2140 ValuationReferenceModel",
        ("FIX.5.0", "2141"): "2141 StrategyType",
        ("FIX.5.0", "2142"): "2142 CommonPricingIndicator",
        ("FIX.5.0", "2143"): "2143 SettlDisruptionProvision",
        ("FIX.5.0", "2144"): "2144 InstrumentRoundingDirection",
        ("FIX.5.0", "2145"): "2145 InstrumentRoundingPrecision",
        ("FIX.5.0", "2146"): "2146 LegSettleOnOpenFlag",
        ("FIX.5.0", "2147"): "2147 LegInstrmtAssignmentMethod",
        ("FIX.5.0", "2148"): "2148 LegSecurityStatus",
        ("FIX.5.0", "2149"): "2149 LegRestructuringType",
        ("FIX.5.0", "2150"): "2150 LegSeniority",
        ("FIX.5.0", "2151"): "2151 LegNotionalPercentageOutstanding",
        ("FIX.5.0", "2152"): "2152 LegOriginalNotionalPercentageOutstanding",
        ("FIX.5.0", "2153"): "2153 LegAttachmentPoint",
        ("FIX.5.0", "2154"): "2154 LegDetachmentPoint",
        ("FIX.5.0", "2155"): "2155 LegObligationType",
        ("FIX.5.0", "2156"): "2156 LegSwapSubClass",
        ("FIX.5.0", "2157"): "2157 LegNthToDefault",
        ("FIX.5.0", "2158"): "2158 LegMthToDefault",
        ("FIX.5.0", "2159"): "2159 LegSettledEntityMatrixSource",
        ("FIX.5.0", "2160"): "2160 LegSettledEntityMatrixPublicationDate",
        ("FIX.5.0", "2161"): "2161 LegCouponType",
        ("FIX.5.0", "2162"): "2162 LegTotalIssuedAmount",
        ("FIX.5.0", "2163"): "2163 LegCouponFrequencyPeriod",
        ("FIX.5.0", "2164"): "2164 LegCouponFrequencyUnit",
        ("FIX.5.0", "2165"): "2165 LegCouponDayCount",
        ("FIX.5.0", "2166"): "2166 LegConvertibleBondEquityID",
        ("FIX.5.0", "2167"): "2167 LegConvertibleBondEquityIDSource",
        ("FIX.5.0", "2168"): "2168 LegContractPriceRefMonth",
        ("FIX.5.0", "2169"): "2169 LegLienSeniority",
        ("FIX.5.0", "2170"): "2170 LegLoanFacility",
        ("FIX.5.0", "2171"): "2171 LegReferenceEntityType",
        ("FIX.5.0", "2172"): "2172 LegIndexSeries",
        ("FIX.5.0", "2173"): "2173 LegIndexAnnexVersion",
        ("FIX.5.0", "2174"): "2174 LegIndexAnnexDate",
        ("FIX.5.0", "2175"): "2175 LegIndexAnnexSource",
        ("FIX.5.0", "2176"): "2176 LegSettlRateIndex",
        ("FIX.5.0", "2177"): "2177 LegSettlRateIndexLocation",
        ("FIX.5.0", "2178"): "2178 LegOptionExpirationDesc",
        ("FIX.5.0", "2179"): "2179 EncodedLegOptionExpirationDescLen",
        ("FIX.5.0", "2180"): "2180 EncodedLegOptionExpirationDesc",
        ("FIX.5.0", "2181"): "2181 LegStrikeMultiplier",
        ("FIX.5.0", "2182"): "2182 LegStrikeValue",
        ("FIX.5.0", "2183"): "2183 LegStrikeUnitOfMeasure",
        ("FIX.5.0", "2184"): "2184 LegStrikeIndex",
        ("FIX.5.0", "2185"): "2185 LegStrikeIndexSpread",
        ("FIX.5.0", "2186"): "2186 LegStrikePriceDeterminationMethod",
        ("FIX.5.0", "2187"): "2187 LegStrikePriceBoundaryMethod",
        ("FIX.5.0", "2188"): "2188 LegStrikePriceBoundaryPrecision",
        ("FIX.5.0", "2189"): "2189 LegUnderlyingPriceDeterminationMethod",
        ("FIX.5.0", "2190"): "2190 LegMinPriceIncrement",
        ("FIX.5.0", "2191"): "2191 LegMinPriceIncrementAmount",
        ("FIX.5.0", "2192"): "2192 LegSettlMethod",
        ("FIX.5.0", "2193"): "2193 LegOptPayoutType",
        ("FIX.5.0", "2194"): "2194 LegOptPayoutAmount",
        ("FIX.5.0", "2195"): "2195 LegPriceQuoteMethod",
        ("FIX.5.0", "2196"): "2196 LegValuationMethod",
        ("FIX.5.0", "2197"): "2197 LegValuationSource",
        ("FIX.5.0", "2198"): "2198 LegValuationReferenceModel",
        ("FIX.5.0", "2199"): "2199 LegListMethod",
        ("FIX.5.0", "2200"): "2200 LegCapPrice",
        ("FIX.5.0", "2201"): "2201 LegFloorPrice",
        ("FIX.5.0", "2202"): "2202 LegFlexibleIndicator",
        ("FIX.5.0", "2203"): "2203 LegFlexProductEligibilityIndicator",
        ("FIX.5.0", "2204"): "2204 LegComplexEventStartTime",
        ("FIX.5.0", "2205"): "2205 LegPositionLimit",
        ("FIX.5.0", "2206"): "2206 LegNTPositionLimit",
        ("FIX.5.0", "2207"): "2207 LegCPProgram",
        ("FIX.5.0", "2208"): "2208 LegCPRegType",
        ("FIX.5.0", "2209"): "2209 LegShortSaleRestriction",
        ("FIX.5.0", "2210"): "2210 AssetGroup",
        ("FIX.5.0", "2211"): "2211 LegStrategyType",
        ("FIX.5.0", "2212"): "2212 LegCommonPricingIndicator",
        ("FIX.5.0", "2213"): "2213 LegSettlDisruptionProvision",
        ("FIX.5.0", "2214"): "2214 LegInstrumentRoundingDirection",
        ("FIX.5.0", "2215"): "2215 LegInstrumentRoundingPrecision",
        ("FIX.5.0", "2216"): "2216 MiscFeeRate",
        ("FIX.5.0", "2217"): "2217 MiscFeeAmountDue",
        ("FIX.5.0", "2218"): "2218 NoLegComplexEvents",
        ("FIX.5.0", "2219"): "2219 LegComplexEventType",
        ("FIX.5.0", "2220"): "2220 LegComplexOptPayoutPaySide",
        ("FIX.5.0", "2221"): "2221 LegComplexOptPayoutReceiveSide",
        ("FIX.5.0", "2222"): "2222 LegComplexOptPayoutUnderlier",
        ("FIX.5.0", "2223"): "2223 LegComplexOptPayoutAmount",
        ("FIX.5.0", "2224"): "2224 LegComplexOptPayoutPercentage",
        ("FIX.5.0", "2225"): "2225 LegComplexOptPayoutTime",
        ("FIX.5.0", "2226"): "2226 LegComplexOptPayoutCurrency",
        ("FIX.5.0", "2227"): "2227 LegComplexEventPrice",
        ("FIX.5.0", "2228"): "2228 LegComplexEventPricePercentage",
        ("FIX.5.0", "2229"): "2229 LegComplexEventPriceBoundaryMethod",
        ("FIX.5.0", "2230"): "2230 LegComplexEventPriceBoundaryPrecision",
        ("FIX.5.0", "2231"): "2231 LegComplexEventPriceTimeType",
        ("FIX.5.0", "2232"): "2232 LegComplexEventCondition",
        ("FIX.5.0", "2233"): "2233 LegComplexEventCurrencyOne",
        ("FIX.5.0", "2234"): "2234 LegComplexEventCurrencyTwo",
        ("FIX.5.0", "2235"): "2235 LegComplexEventQuoteBasis",
        ("FIX.5.0", "2236"): "2236 LegComplexEventFixedFXRate",
        ("FIX.5.0", "2237"): "2237 LegComplexEventDeterminationMethod",
        ("FIX.5.0", "2238"): "2238 LegComplexEventCalculationAgent",
        ("FIX.5.0", "2239"): "2239 LegComplexEventStrikePrice",
        ("FIX.5.0", "2240"): "2240 LegComplexEventStrikeFactor",
        ("FIX.5.0", "2241"): "2241 LegComplexEventStrikeNumberOfOptions",
        ("FIX.5.0", "2242"): "2242 LegComplexEventCreditEventsXIDRef",
        ("FIX.5.0", "2243"): "2243 LegComplexEventCreditEventNotifyingParty",
        ("FIX.5.0", "2244"): "2244 LegComplexEventCreditEventBusinessCenter",
        ("FIX.5.0", "2245"): "2245 LegComplexEventCreditEventStandardSources",
        ("FIX.5.0", "2246"): "2246 LegComplexEventCreditEventMinimumSources",
        ("FIX.5.0", "2247"): "2247 LegComplexEventEndTime",
        ("FIX.5.0", "2248"): "2248 LegComplexEventXID",
        ("FIX.5.0", "2249"): "2249 LegComplexEventXIDRef",
        ("FIX.5.0", "2250"): "2250 NoLegComplexEventDates",
        ("FIX.5.0", "2251"): "2251 LegComplexEventStartDate",
        ("FIX.5.0", "2252"): "2252 LegComplexEventEndDate",
        ("FIX.5.0", "2253"): "2253 NoLegComplexEventTimes",
        ("FIX.5.0", "2254"): "2254 NoLegInstrumentParties",
        ("FIX.5.0", "2255"): "2255 LegInstrumentPartyID",
        ("FIX.5.0", "2256"): "2256 LegInstrumentPartyIDSource",
        ("FIX.5.0", "2257"): "2257 LegInstrumentPartyRole",
        ("FIX.5.0", "2258"): "2258 NoLegInstrumentPartySubIDs",
        ("FIX.5.0", "2259"): "2259 LegInstrumentPartySubID",
        ("FIX.5.0", "2260"): "2260 LegInstrumentPartySubIDType",
        ("FIX.5.0", "2261"): "2261 UnderlyingComplexOptPayoutPaySide",
        ("FIX.5.0", "2262"): "2262 UnderlyingComplexOptPayoutReceiveSide",
        ("FIX.5.0", "2263"): "2263 UnderlyingComplexOptPayoutUnderlier",
        ("FIX.5.0", "2264"): "2264 UnderlyingComplexOptPayoutPercentage",
        ("FIX.5.0", "2265"): "2265 UnderlyingComplexOptPayoutTime",
        ("FIX.5.0", "2266"): "2266 UnderlyingComplexOptPayoutCurrency",
        ("FIX.5.0", "2267"): "2267 UnderlyingComplexEventPricePercentage",
        ("FIX.5.0", "2268"): "2268 UnderlyingComplexEventCurrencyOne",
        ("FIX.5.0", "2269"): "2269 UnderlyingComplexEventCurrencyTwo",
        ("FIX.5.0", "2270"): "2270 UnderlyingComplexEventQuoteBasis",
        ("FIX.5.0", "2271"): "2271 UnderlyingComplexEventFixedFXRate",
        ("FIX.5.0", "2272"): "2272 UnderlyingComplexEventDeterminationMethod",
        ("FIX.5.0", "2273"): "2273 UnderlyingComplexEventCalculationAgent",
        ("FIX.5.0", "2274"): "2274 UnderlyingComplexEventStrikePrice",
        ("FIX.5.0", "2275"): "2275 UnderlyingComplexEventStrikeFactor",
        ("FIX.5.0", "2276"): "2276 UnderlyingComplexEventStrikeNumberOfOptions",
        ("FIX.5.0", "2277"): "2277 UnderlyingComplexEventCreditEventsXIDRef",
        ("FIX.5.0", "2278"): "2278 UnderlyingComplexEventCreditEventNotifyingParty",
        ("FIX.5.0", "2279"): "2279 UnderlyingComplexEventCreditEventBusinessCenter",
        ("FIX.5.0", "2280"): "2280 UnderlyingComplexEventCreditEventStandardSources",
        ("FIX.5.0", "2281"): "2281 UnderlyingComplexEventCreditEventMinimumSources",
        ("FIX.5.0", "2282"): "2282 UnderlyingComplexEventXID",
        ("FIX.5.0", "2283"): "2283 UnderlyingComplexEventXIDRef",
        ("FIX.5.0", "2284"): "2284 UnderlyingSettlRateIndex",
        ("FIX.5.0", "2285"): "2285 UnderlyingSettlRateIndexLocation",
        ("FIX.5.0", "2286"): "2286 UnderlyingOptionExpirationDesc",
        ("FIX.5.0", "2287"): "2287 EncodedUnderlyingOptionExpirationDescLen",
        ("FIX.5.0", "2288"): "2288 EncodedUnderlyingOptionExpirationDesc",
        ("FIX.5.0", "2289"): "2289 UnderlyingSwapSubClass",
        ("FIX.5.0", "2290"): "2290 UnderlyingStrikeUnitOfMeasure",
        ("FIX.5.0", "2291"): "2291 UnderlyingStrikeIndex",
        ("FIX.5.0", "2292"): "2292 UnderlyingStrikeIndexSpread",
        ("FIX.5.0", "2293"): "2293 UnderlyingValuationSource",
        ("FIX.5.0", "2294"): "2294 UnderlyingValuationReferenceModel",
        ("FIX.5.0", "2295"): "2295 UnderlyingStrategyType",
        ("FIX.5.0", "2296"): "2296 UnderlyingCommonPricingIndicator",
        ("FIX.5.0", "2297"): "2297 UnderlyingSettlDisruptionProvision",
        ("FIX.5.0", "2298"): "2298 UnderlyingInstrumentRoundingDirection",
        ("FIX.5.0", "2299"): "2299 UnderlyingInstrumentRoundingPrecision",
        ("FIX.5.0", "2300"): "2300 AllocGrossTradeAmt",
        ("FIX.5.0", "2301"): "2301 LastQtyChanged",
        ("FIX.5.0", "2302"): "2302 TradeVersion",
        ("FIX.5.0", "2303"): "2303 HistoricalReportIndicator",
        ("FIX.5.0", "2304"): "2304 NoAssetAttributes",
        ("FIX.5.0", "2305"): "2305 AssetAttributeType",
        ("FIX.5.0", "2306"): "2306 AssetAttributeValue",
        ("FIX.5.0", "2307"): "2307 AssetAttributeLimit",
        ("FIX.5.0", "2308"): "2308 NoLegAssetAttributes",
        ("FIX.5.0", "2309"): "2309 LegAssetAttributeType",
        ("FIX.5.0", "2310"): "2310 LegAssetAttributeValue",
        ("FIX.5.0", "2311"): "2311 LegAssetAttributeLimit",
        ("FIX.5.0", "2312"): "2312 NoUnderlyingAssetAttributes",
        ("FIX.5.0", "2313"): "2313 UnderlyingAssetAttributeType",
        ("FIX.5.0", "2314"): "2314 UnderlyingAssetAttributeValue",
        ("FIX.5.0", "2315"): "2315 UnderlyingAssetAttributeLimit",
        ("FIX.5.0", "2316"): "2316 RiskLimitReportStatus",
        ("FIX.5.0", "2317"): "2317 RiskLimitReportRejectReason",
        ("FIX.5.0", "2318"): "2318 RiskLimitCheckRequestID",
        ("FIX.5.0", "2319"): "2319 RiskLimitCheckID",
        ("FIX.5.0", "2320"): "2320 RiskLimitCheckTransType",
        ("FIX.5.0", "2321"): "2321 RiskLimitCheckType",
        ("FIX.5.0", "2322"): "2322 RiskLimitCheckRequestRefID",
        ("FIX.5.0", "2323"): "2323 RiskLimitCheckRequestType",
        ("FIX.5.0", "2324"): "2324 RiskLimitCheckAmount",
        ("FIX.5.0", "2325"): "2325 RiskLimitCheckRequestStatus",
        ("FIX.5.0", "2326"): "2326 RiskLimitCheckRequestResult",
        ("FIX.5.0", "2327"): "2327 RiskLimitApprovedAmount",
        ("FIX.5.0", "2328"): "2328 PartyActionRequestID",
        ("FIX.5.0", "2329"): "2329 PartyActionType",
        ("FIX.5.0", "2330"): "2330 ApplTestMessageIndicator",
        ("FIX.5.0", "2331"): "2331 PartyActionReportID",
        ("FIX.5.0", "2332"): "2332 PartyActionResponse",
        ("FIX.5.0", "2333"): "2333 PartyActionRejectReason",
        ("FIX.5.0", "2334"): "2334 RefRiskLimitCheckID",
        ("FIX.5.0", "2335"): "2335 RefRiskLimitCheckIDType",
        ("FIX.5.0", "2336"): "2336 RiskLimitVelocityPeriod",
        ("FIX.5.0", "2337"): "2337 RiskLimitVelocityUnit",
        ("FIX.5.0", "2338"): "2338 RequestingPartyRoleQualifier",
        ("FIX.5.0", "2339"): "2339 RiskLimitCheckModelType",
        ("FIX.5.0", "2340"): "2340 EventMonthYear",
        ("FIX.5.0", "2341"): "2341 LegEventMonthYear",
        ("FIX.5.0", "2342"): "2342 UnderlyingEventMonthYear",
        ("FIX.5.0", "2343"): "2343 RiskLimitCheckStatus",
        ("FIX.5.0", "2344"): "2344 SideRiskLimitCheckStatus",
        ("FIX.5.0", "2345"): "2345 NoEntitlementTypes",
        ("FIX.5.0", "2346"): "2346 LegMidPx",
        ("FIX.5.0", "2347"): "2347 RegulatoryTransactionType",
        ("FIX.5.0", "2348"): "2348 LegAssetGroup",
        ("FIX.5.0", "2349"): "2349 PricePrecision",
        ("FIX.5.0", "2350"): "2350 CollateralPortfolioID",
        ("FIX.5.0", "2351"): "2351 EncodedComplianceTextLen",
        ("FIX.5.0", "2352"): "2352 EncodedComplianceText",
        ("FIX.5.0", "2353"): "2353 TradingUnitPeriodMultiplier",
        ("FIX.5.0", "2354"): "2354 LegTradingUnitPeriodMultiplier",
        ("FIX.5.0", "2355"): "2355 PartyRiskLimitStatus",
        ("FIX.5.0", "2356"): "2356 RemunerationIndicator",
        ("FIX.5.0", "2357"): "2357 LegTotalTradeQty",
        ("FIX.5.0", "2358"): "2358 LegLastMultipliedQty",
        ("FIX.5.0", "2359"): "2359 LegTotalGrossTradeAmt",
        ("FIX.5.0", "2360"): "2360 LegTotalTradeMultipliedQty",
        ("FIX.5.0", "2361"): "2361 CompressionGroupID",
        ("FIX.5.0", "2362"): "2362 SelfMatchPreventionID",
        ("FIX.5.0", "2363"): "2363 UnderlyingTradingUnitPeriodMultiplier",
        ("FIX.5.0", "2364"): "2364 PosReportAction",
        ("FIX.5.0", "2365"): "2365 SettlForwardPoints",
        ("FIX.5.0", "2366"): "2366 SettlPriceFxRateCalc",
        ("FIX.5.0", "2367"): "2367 TotalTradeQty",
        ("FIX.5.0", "2368"): "2368 LastMultipliedQty",
        ("FIX.5.0", "2369"): "2369 TotalGrossTradeAmt",
        ("FIX.5.0", "2370"): "2370 TotalTradeMultipliedQty",
        ("FIX.5.0", "2371"): "2371 EncodedTradeContinuationText",
        ("FIX.5.0", "2372"): "2372 EncodedTradeContinuationTextLen",
        ("FIX.5.0", "2373"): "2373 IntraFirmTradeIndicator",
        ("FIX.5.0", "2374"): "2374 TradeContinuationText",
        ("FIX.5.0", "2375"): "2375 TaxonomyType",
        ("FIX.5.0", "2376"): "2376 PartyRoleQualifier",
        ("FIX.5.0", "2377"): "2377 DerivativeInstrumentPartyRoleQualifier",
        ("FIX.5.0", "2378"): "2378 InstrumentPartyRoleQualifier",
        ("FIX.5.0", "2379"): "2379 LegInstrumentPartyRoleQualifier",
        ("FIX.5.0", "2380"): "2380 LegProvisionPartyRoleQualifier",
        ("FIX.5.0", "2381"): "2381 Nested2PartyRoleQualifier",
        ("FIX.5.0", "2382"): "2382 Nested3PartyRoleQualifier",
        ("FIX.5.0", "2383"): "2383 Nested4PartyRoleQualifier",
        ("FIX.5.0", "2384"): "2384 NestedPartyRoleQualifier",
        ("FIX.5.0", "2385"): "2385 ProvisionPartyRoleQualifier",
        ("FIX.5.0", "2386"): "2386 RequestedPartyRoleQualifier",
        ("FIX.5.0", "2387"): "2387 TradeContingency",
        ("FIX.5.0", "2388"): "2388 RootPartyRoleQualifier",
        ("FIX.5.0", "2389"): "2389 SettlPartyRoleQualifier",
        ("FIX.5.0", "2390"): "2390 TradeConfirmationReferenceID",
        ("FIX.5.0", "2391"): "2391 UnderlyingInstrumentPartyRoleQualifier",
        ("FIX.5.0", "2392"): "2392 AllocRefRiskLimitCheckID",
        ("FIX.5.0", "2393"): "2393 AllocRefRiskLimitCheckIDType",
        ("FIX.5.0", "2394"): "2394 LimitUtilizationAmt",
        ("FIX.5.0", "2395"): "2395 LimitAmt",
        ("FIX.5.0", "2396"): "2396 LimitRole",
        ("FIX.5.0", "2397"): "2397 RegulatoryTradeIDScope",
        ("FIX.5.0", "2398"): "2398 SideRegulatoryTradeIDScope",
        ("FIX.5.0", "2399"): "2399 AllocRegulatoryTradeIDScope",
        ("FIX.5.0", "2400"): "2400 EffectiveBusinessDate",
        ("FIX.5.0", "2401"): "2401 ListManualOrderIndicator",
        ("FIX.5.0", "2402"): "2402 EntitlementSubType",
        ("FIX.5.0", "2403"): "2403 QuoteModelType",
        ("FIX.5.0", "2404"): "2404 ComplianceText",
        ("FIX.5.0", "2405"): "2405 ExecMethod",
        ("FIX.5.0", "2406"): "2406 AllocRegulatoryLegRefID",
        ("FIX.5.0", "2407"): "2407 ComplexEventSpotRate",
        ("FIX.5.0", "2408"): "2408 ComplexEventForwardPoints",
        ("FIX.5.0", "2409"): "2409 LegComplexEventSpotRate",
        ("FIX.5.0", "2410"): "2410 LegComplexEventForwardPoints",
        ("FIX.5.0", "2411"): "2411 RegulatoryLegRefID",
        ("FIX.5.0", "2412"): "2412 RateSourceReferemcePageHeading",
        ("FIX.5.0", "2413"): "2413 RelatedToSecurityID",
        ("FIX.5.0", "2414"): "2414 RelatedToSecurityIDSource",
        ("FIX.5.0", "2415"): "2415 RelatedToStreamXIDRef",
        ("FIX.5.0", "2416"): "2416 SideRegulatoryLegRefID",
        ("FIX.5.0", "2417"): "2417 RelatedToDividendPeriodXIDRef",
        ("FIX.5.0", "2418"): "2418 FirmTradeEventID",
        ("FIX.5.0", "2419"): "2419 UnderlyingComplexEventSpotRate",
        ("FIX.5.0", "2420"): "2420 UnderlyingComplexEventForwardPoints",
        ("FIX.5.0", "2421"): "2421 FillRefID",
        ("FIX.5.0", "2422"): "2422 OrderRequestID",
        ("FIX.5.0", "2423"): "2423 MassOrderRequestID",
        ("FIX.5.0", "2424"): "2424 MassOrderReportID",
        ("FIX.5.0", "2425"): "2425 MassOrderRequestStatus",
        ("FIX.5.0", "2426"): "2426 MassOrderRequestResult",
        ("FIX.5.0", "2427"): "2427 OrderResponseLevel",
        ("FIX.5.0", "2428"): "2428 NoOrderEntries",
        ("FIX.5.0", "2429"): "2429 OrderEntryAction",
        ("FIX.5.0", "2430"): "2430 OrderEntryID",
        ("FIX.5.0", "2431"): "2431 ExecTypeReason",
        ("FIX.5.0", "2432"): "2432 TotNoOrderEntries",
        ("FIX.5.0", "2433"): "2433 NoTargetPartySubIDs",
        ("FIX.5.0", "2434"): "2434 TargetPartySubID",
        ("FIX.5.0", "2435"): "2435 TargetPartySubIDType",
        ("FIX.5.0", "2436"): "2436 TransferInstructionID",
        ("FIX.5.0", "2437"): "2437 TransferID",
        ("FIX.5.0", "2438"): "2438 TransferReportID",
        ("FIX.5.0", "2439"): "2439 TransferTransType",
        ("FIX.5.0", "2440"): "2440 TransferType",
        ("FIX.5.0", "2441"): "2441 TransferScope",
        ("FIX.5.0", "2442"): "2442 TransferStatus",
        ("FIX.5.0", "2443"): "2443 TransferRejectReason",
        ("FIX.5.0", "2444"): "2444 TransferReportType",
        ("FIX.5.0", "2445"): "2445 AggressorTime",
        ("FIX.5.0", "2446"): "2446 AggressorSide",
        ("FIX.5.0", "2447"): "2447 FastMarketIndicator",
        ("FIX.5.0", "2448"): "2448 LinkageHandlingIndicator",
        ("FIX.5.0", "2449"): "2449 NumberOfBuyOrders",
        ("FIX.5.0", "2450"): "2450 NumberOfSellOrders",
        ("FIX.5.0", "2451"): "2451 SettlPriceDeterminationMethod",
        ("FIX.5.0", "2452"): "2452 MDStatisticReqID",
        ("FIX.5.0", "2453"): "2453 MDStatisticRptID",
        ("FIX.5.0", "2454"): "2454 MDStatisticName",
        ("FIX.5.0", "2455"): "2455 MDStatisticDesc",
        ("FIX.5.0", "2456"): "2456 MDStatisticType",
        ("FIX.5.0", "2457"): "2457 MDStatisticScope",
        ("FIX.5.0", "2458"): "2458 MDStatisticSubScope",
        ("FIX.5.0", "2459"): "2459 MDStatisticScopeType",
        ("FIX.5.0", "2460"): "2460 MDStatisticFrequencyPeriod",
        ("FIX.5.0", "2461"): "2461 MDStatisticFrequencyUnit",
        ("FIX.5.0", "2462"): "2462 MDStatisticDelayPeriod",
        ("FIX.5.0", "2463"): "2463 MDStatisticDelayUnit",
        ("FIX.5.0", "2464"): "2464 MDStatisticIntervalType",
        ("FIX.5.0", "2465"): "2465 MDStatisticIntervalTypeUnit",
        ("FIX.5.0", "2466"): "2466 MDStatisticIntervalPeriod",
        ("FIX.5.0", "2467"): "2467 MDStatisticIntervalUnit",
        ("FIX.5.0", "2468"): "2468 MDStatisticStartDate",
        ("FIX.5.0", "2469"): "2469 MDStatisticEndDate",
        ("FIX.5.0", "2470"): "2470 MDStatisticStartTime",
        ("FIX.5.0", "2471"): "2471 MDStatisticEndTime",
        ("FIX.5.0", "2472"): "2472 MDStatisticRatioType",
        ("FIX.5.0", "2473"): "2473 MDStatisticRequestResult",
        ("FIX.5.0", "2474"): "2474 NoMDStatistics",
        ("FIX.5.0", "2475"): "2475 MDStatisticID",
        ("FIX.5.0", "2476"): "2476 MDStatisticTime",
        ("FIX.5.0", "2477"): "2477 MDStatisticStatus",
        ("FIX.5.0", "2478"): "2478 MDStatisticValue",
        ("FIX.5.0", "2479"): "2479 MDStatisticValueType",
        ("FIX.5.0", "2480"): "2480 MDStatisticValueUnit",
        ("FIX.5.0", "2481"): "2481 EncodedMDStatisticDescLen",
        ("FIX.5.0", "2482"): "2482 EncodedMDStatisticDesc",
        ("FIX.5.0", "2483"): "2483 AllocRiskLimitCheckStatus",
        ("FIX.5.0", "2484"): "2484 FirmTransactionID",
        ("FIX.5.0", "2485"): "2485 TransactionID",
        ("FIX.5.0", "2486"): "2486 WireReference",
        ("FIX.5.0", "2487"): "2487 CollRptRejectReason",
        ("FIX.5.0", "2488"): "2488 CollRptStatus",
        ("FIX.5.0", "2489"): "2489 PackageID",
        ("FIX.5.0", "2490"): "2490 TradeNumber",
        ("FIX.5.0", "2491"): "2491 UnderlyingAssetGroup",
        ("FIX.5.0", "2492"): "2492 LegDifferentialPrice",
        ("FIX.5.0", "2493"): "2493 EncodedLegDocumentationText",
        ("FIX.5.0", "2494"): "2494 EncodedLegDocumentationTextLen",
        ("FIX.5.0", "2495"): "2495 LegAgreementCurrency",
        ("FIX.5.0", "2496"): "2496 LegAgreementDate",
        ("FIX.5.0", "2497"): "2497 LegAgreementDesc",
        ("FIX.5.0", "2498"): "2498 LegAgreementID",
        ("FIX.5.0", "2499"): "2499 LegAgreementVersion",
        ("FIX.5.0", "2500"): "2500 LegBrokerConfirmationDesc",
        ("FIX.5.0", "2501"): "2501 LegCreditSupportAgreementDate",
        ("FIX.5.0", "2502"): "2502 LegCreditSupportAgreementDesc",
        ("FIX.5.0", "2503"): "2503 LegCreditSupportAgreementID",
        ("FIX.5.0", "2504"): "2504 LegDeliveryType",
        ("FIX.5.0", "2505"): "2505 LegDocumentationText",
        ("FIX.5.0", "2506"): "2506 LegEndDate",
        ("FIX.5.0", "2507"): "2507 LegGoverningLaw",
        ("FIX.5.0", "2508"): "2508 LegMarginRatio",
        ("FIX.5.0", "2509"): "2509 LegMasterConfirmationAnnexDate",
        ("FIX.5.0", "2510"): "2510 LegMasterConfirmationDate",
        ("FIX.5.0", "2511"): "2511 LegMasterConfirmationDesc",
        ("FIX.5.0", "2512"): "2512 LegMasterConfirmationAnnexDesc",
        ("FIX.5.0", "2513"): "2513 LegStartDate",
        ("FIX.5.0", "2514"): "2514 LegTerminationType",
        ("FIX.5.0", "2515"): "2515 AllocCalculatedCcyQty",
        ("FIX.5.0", "2516"): "2516 CollateralRequestInstruction",
        ("FIX.5.0", "2517"): "2517 CollateralRequestLinkID",
        ("FIX.5.0", "2518"): "2518 CollateralRequestNumber",
        ("FIX.5.0", "2519"): "2519 TotNumCollateralRequests",
        ("FIX.5.0", "2520"): "2520 WarningText",
        ("FIX.5.0", "2521"): "2521 EncodedWarningText",
        ("FIX.5.0", "2522"): "2522 EncodedWarningTextLen",
        ("FIX.5.0", "2523"): "2523 CrossedIndicator",
        ("FIX.5.0", "2524"): "2524 TradeReportingIndicator",
        ("FIX.5.0", "2525"): "2525 AffiliatedFirmsTradeIndicator",
        ("FIX.5.0", "2526"): "2526 InternationalSwapIndicator",
        ("FIX.5.0", "2527"): "2527 MultiAssetSwapIndicator",
        ("FIX.5.0", "2528"): "2528 ClearingSettlPrice",
        ("FIX.5.0", "2529"): "2529 NoRelativeValues",
        ("FIX.5.0", "2530"): "2530 RelativeValueType",
        ("FIX.5.0", "2531"): "2531 RelativeValue",
        ("FIX.5.0", "2532"): "2532 RelativeValueSide",
        ("FIX.5.0", "2533"): "2533 BidSpread",
        ("FIX.5.0", "2534"): "2534 OfferSpread",
        ("FIX.5.0", "2535"): "2535 MDReportEvent",
        ("FIX.5.0", "2536"): "2536 MDReportCount",
        ("FIX.5.0", "2537"): "2537 TotNoMarketSegmentReports",
        ("FIX.5.0", "2538"): "2538 TotNoInstrumentReports",
        ("FIX.5.0", "2539"): "2539 TotNoPartyDetailReports",
        ("FIX.5.0", "2540"): "2540 TotNoEntitlementReports",
        ("FIX.5.0", "2541"): "2541 TotNoRiskLimitReports",
        ("FIX.5.0", "2542"): "2542 MarketSegmentStatus",
        ("FIX.5.0", "2543"): "2543 MarketSegmentType",
        ("FIX.5.0", "2544"): "2544 MarketSegmentSubType",
        ("FIX.5.0", "2545"): "2545 NoRelatedMarketSegments",
        ("FIX.5.0", "2546"): "2546 RelatedMarketSegmentID",
        ("FIX.5.0", "2547"): "2547 MarketSegmentRelationship",
        ("FIX.5.0", "2548"): "2548 NoAuctionTypeRules",
        ("FIX.5.0", "2549"): "2549 AuctionTypeProductComplex",
        ("FIX.5.0", "2550"): "2550 NoPriceRangeRules",
        ("FIX.5.0", "2551"): "2551 StartPriceRange",
        ("FIX.5.0", "2552"): "2552 EndPriceRange",
        ("FIX.5.0", "2553"): "2553 PriceRangeValue",
        ("FIX.5.0", "2554"): "2554 PriceRangePercentage",
        ("FIX.5.0", "2555"): "2555 PriceRangeProductComplex",
        ("FIX.5.0", "2556"): "2556 PriceRangeRuleID",
        ("FIX.5.0", "2557"): "2557 FastMarketPercentage",
        ("FIX.5.0", "2558"): "2558 NoQuoteSizeRules",
        ("FIX.5.0", "2559"): "2559 QuoteSideIndicator",
        ("FIX.5.0", "2560"): "2560 NoFlexProductEligibilities",
        ("FIX.5.0", "2561"): "2561 FlexProductEligibilityComplex",
        ("FIX.5.0", "2562"): "2562 NumOfComplexInstruments",
        ("FIX.5.0", "2563"): "2563 MarketDepthTimeInterval",
        ("FIX.5.0", "2564"): "2564 MarketDepthTimeIntervalUnit",
        ("FIX.5.0", "2565"): "2565 MDRecoveryTimeInterval",
        ("FIX.5.0", "2566"): "2566 MDRecoveryTimeIntervalUnit",
        ("FIX.5.0", "2567"): "2567 PrimaryServiceLocationID",
        ("FIX.5.0", "2568"): "2568 SecondaryServiceLocationID",
        ("FIX.5.0", "2569"): "2569 MatchRuleProductComplex",
        ("FIX.5.0", "2570"): "2570 CustomerPriority",
        ("FIX.5.0", "2571"): "2571 TickRuleProductComplex",
        ("FIX.5.0", "2572"): "2572 PreviousAdjustedOpenInterest",
        ("FIX.5.0", "2573"): "2573 PreviousUnadjustedOpenInterest",
        ("FIX.5.0", "2574"): "2574 LowExercisePriceOptionIndicator",
        ("FIX.5.0", "2575"): "2575 BlockTradeEligibilityIndicator",
        ("FIX.5.0", "2576"): "2576 InstrumentPricePrecision",
        ("FIX.5.0", "2577"): "2577 StrikePricePrecision",
        ("FIX.5.0", "2578"): "2578 OrigStrikePrice",
        ("FIX.5.0", "2579"): "2579 SettlSubMethod",
        ("FIX.5.0", "2580"): "2580 NoClearingPriceParameters",
        ("FIX.5.0", "2581"): "2581 BusinessDayType",
        ("FIX.5.0", "2582"): "2582 ClearingPriceOffset",
        ("FIX.5.0", "2583"): "2583 VegaMultiplier",
        ("FIX.5.0", "2584"): "2584 AnnualTradingBusinessDays",
        ("FIX.5.0", "2585"): "2585 TotalTradingBusinessDays",
        ("FIX.5.0", "2586"): "2586 TradingBusinessDays",
        ("FIX.5.0", "2587"): "2587 RealizedVariance",
        ("FIX.5.0", "2588"): "2588 StandardVariance",
        ("FIX.5.0", "2589"): "2589 RelatedClosePrice",
        ("FIX.5.0", "2590"): "2590 OvernightInterestRate",
        ("FIX.5.0", "2591"): "2591 AccumulatedReturnModifiedVariationMargin",
        ("FIX.5.0", "2592"): "2592 CalculationMethod",
        ("FIX.5.0", "2593"): "2593 NoOrderAttributes",
        ("FIX.5.0", "2594"): "2594 OrderAttributeType",
        ("FIX.5.0", "2595"): "2595 OrderAttributeValue",
        ("FIX.5.0", "2596"): "2596 DeltaCrossed",
        ("FIX.5.0", "2597"): "2597 ComplexEventFuturesPriceValuation",
        ("FIX.5.0", "2598"): "2598 ComplexEventOptionsPriceValuation",
        ("FIX.5.0", "2599"): "2599 ComplexEventPVFinalPriceElectionFallback",
        ("FIX.5.0", "2600"): "2600 StrikeIndexCurvePoint",
        ("FIX.5.0", "2601"): "2601 StrikeIndexQuote",
        ("FIX.5.0", "2602"): "2602 ExtraordinaryEventAdjustmentMethod",
        ("FIX.5.0", "2603"): "2603 ExchangeLookAlike",
        ("FIX.5.0", "2604"): "2604 LegStrikeIndexCurvePoint",
        ("FIX.5.0", "2605"): "2605 LegStrikeIndexQuote",
        ("FIX.5.0", "2606"): "2606 LegExtraordinaryEventAdjustmentMethod",
        ("FIX.5.0", "2607"): "2607 LegExchangeLookAlike",
        ("FIX.5.0", "2608"): "2608 LegComplexEventFuturesPriceValuation",
        ("FIX.5.0", "2609"): "2609 LegComplexEventOptionsPriceValuation",
        ("FIX.5.0", "2610"): "2610 LegComplexEventPVFinalPriceElectionFallback",
        ("FIX.5.0", "2611"): "2611 UnderlyingComplexEventFuturesPriceValuation",
        ("FIX.5.0", "2612"): "2612 UnderlyingComplexEventOptionsPriceValuation",
        ("FIX.5.0", "2613"): "2613 UnderlyingComplexEventPVFinalPriceElectionFallback",
        ("FIX.5.0", "2614"): "2614 UnderlyingNotional",
        ("FIX.5.0", "2615"): "2615 UnderlyingNotionalCurrency",
        ("FIX.5.0", "2616"): "2616 UnderlyingNotionalDeterminationMethod",
        ("FIX.5.0", "2617"): "2617 UnderlyingNotionalAdjustments",
        ("FIX.5.0", "2618"): "2618 PositionID",
        ("FIX.5.0", "2619"): "2619 UnderlyingNotionalXIDRef",
        ("FIX.5.0", "2620"): "2620 UnderlyingFutureID",
        ("FIX.5.0", "2621"): "2621 UnderlyingFutureIDSource",
        ("FIX.5.0", "2622"): "2622 UnderlyingStrikeIndexCurvePoint",
        ("FIX.5.0", "2623"): "2623 UnderlyingStrikeIndexQuote",
        ("FIX.5.0", "2624"): "2624 UnderlyingExtraordinaryEventAdjustmentMethod",
        ("FIX.5.0", "2625"): "2625 UnderlyingExchangeLookAlike",
        ("FIX.5.0", "2626"): "2626 UnderlyingAverageVolumeLimitationPercentage",
        ("FIX.5.0", "2627"): "2627 UnderlyingAverageVolumeLimitationPeriodDays",
        ("FIX.5.0", "2628"): "2628 UnderlyingDepositoryReceiptIndicator",
        ("FIX.5.0", "2629"): "2629 UnderlyingOpenUnits",
        ("FIX.5.0", "2630"): "2630 UnderlyingBasketDivisor",
        ("FIX.5.0", "2631"): "2631 UnderlyingInstrumentXID",
        ("FIX.5.0", "2632"): "2632 CollateralAmountType",
        ("FIX.5.0", "2633"): "2633 NoMiscFeeSubTypes",
        ("FIX.5.0", "2634"): "2634 MiscFeeSubType",
        ("FIX.5.0", "2635"): "2635 MiscFeeSubTypeAmt",
        ("FIX.5.0", "2636"): "2636 MiscFeeSubTypeDesc",
        ("FIX.5.0", "2637"): "2637 EncodedMiscFeeSubTypeDescLen",
        ("FIX.5.0", "2638"): "2638 EncodedMiscFeeSubTypeDesc",
        ("FIX.5.0", "2639"): "2639 NoCommissions",
        ("FIX.5.0", "2640"): "2640 CommissionAmount",
        ("FIX.5.0", "2641"): "2641 CommissionAmountType",
        ("FIX.5.0", "2642"): "2642 CommissionBasis",
        ("FIX.5.0", "2643"): "2643 CommissionCurrency",
        ("FIX.5.0", "2644"): "2644 CommissionUnitOfMeasure",
        ("FIX.5.0", "2645"): "2645 CommissionUnitOfMeasureCurrency",
        ("FIX.5.0", "2646"): "2646 CommissionRate",
        ("FIX.5.0", "2647"): "2647 CommissionSharedIndicator",
        ("FIX.5.0", "2648"): "2648 CommissionAmountShared",
        ("FIX.5.0", "2649"): "2649 CommissionLegRefID",
        ("FIX.5.0", "2650"): "2650 CommissionDesc",
        ("FIX.5.0", "2651"): "2651 EncodedCommissionDescLen",
        ("FIX.5.0", "2652"): "2652 EncodedCommissionDesc",
        ("FIX.5.0", "2653"): "2653 NoAllocCommissions",
        ("FIX.5.0", "2654"): "2654 AllocCommissionAmount",
        ("FIX.5.0", "2655"): "2655 AllocCommissionAmountType",
        ("FIX.5.0", "2656"): "2656 AllocCommissionBasis",
        ("FIX.5.0", "2657"): "2657 AllocCommissionCurrency",
        ("FIX.5.0", "2658"): "2658 AllocCommissionUnitOfMeasure",
        ("FIX.5.0", "2659"): "2659 AllocCommissionUnitOfMeasureCurrency",
        ("FIX.5.0", "2660"): "2660 AllocCommissionRate",
        ("FIX.5.0", "2661"): "2661 AllocCommissionSharedIndicator",
        ("FIX.5.0", "2662"): "2662 AllocCommissionAmountShared",
        ("FIX.5.0", "2663"): "2663 AllocCommissionLegRefID",
        ("FIX.5.0", "2664"): "2664 AllocCommissionDesc",
        ("FIX.5.0", "2665"): "2665 EncodedAllocCommissionDescLen",
        ("FIX.5.0", "2666"): "2666 EncodedAllocCommissionDesc",
        ("FIX.5.0", "2667"): "2667 AlgorithmicTradeIndicator",
        ("FIX.5.0", "2668"): "2668 NoTrdRegPublications",
        ("FIX.5.0", "2669"): "2669 TrdRegPublicationType",
        ("FIX.5.0", "2670"): "2670 TrdRegPublicationReason",
        ("FIX.5.0", "2671"): "2671 SideTradeReportingIndicator",
        ("FIX.5.0", "2672"): "2672 CrossRequestID",
        ("FIX.5.0", "2673"): "2673 FillMatchID",
        ("FIX.5.0", "2674"): "2674 FillMatchSubID",
        ("FIX.5.0", "2675"): "2675 MassActionReason",
        ("FIX.5.0", "2676"): "2676 MaximumPricePercentage",
        ("FIX.5.0", "2677"): "2677 NotAffectedReason",
        ("FIX.5.0", "2678"): "2678 TotalNotAffectedOrders",
        ("FIX.5.0", "2679"): "2679 OrderOwnershipIndicator",
        ("FIX.5.0", "2680"): "2680 LegAccount",
        ("FIX.5.0", "2681"): "2681 InTheMoneyCondition",
        ("FIX.5.0", "2682"): "2682 LegInTheMoneyCondition",
        ("FIX.5.0", "2683"): "2683 UnderlyingInTheMoneyCondition",
        ("FIX.5.0", "2684"): "2684 DerivativeInTheMoneyCondition",
        ("FIX.5.0", "2685"): "2685 ContraryInstructionEligibilityIndicator",
        ("FIX.5.0", "2686"): "2686 LegContraryInstructionEligibilityIndicator",
        ("FIX.5.0", "2687"): "2687 UnderlyingContraryInstructionEligibilityIndicator",
        ("FIX.5.0", "2688"): "2688 DerivativeContraryInstructionEligibilityIndicator",
        ("FIX.5.0", "2689"): "2689 CollateralMarketPrice",
        ("FIX.5.0", "2690"): "2690 CollateralPercentOverage",
        ("FIX.5.0", "2691"): "2691 NoSideCollateralAmounts",
        ("FIX.5.0", "2692"): "2692 SideCollateralAmountMarketID",
        ("FIX.5.0", "2693"): "2693 SideCollateralAmountMarketSegmentID",
        ("FIX.5.0", "2694"): "2694 SideCollateralAmountType",
        ("FIX.5.0", "2695"): "2695 SideCollateralCurrency",
        ("FIX.5.0", "2696"): "2696 SideCollateralFXRate",
        ("FIX.5.0", "2697"): "2697 SideCollateralFXRateCalc",
        ("FIX.5.0", "2698"): "2698 SideCollateralMarketPrice",
        ("FIX.5.0", "2699"): "2699 SideCollateralPercentOverage",
        ("FIX.5.0", "2700"): "2700 SideCollateralPortfolioID",
        ("FIX.5.0", "2701"): "2701 SideCollateralType",
        ("FIX.5.0", "2702"): "2702 SideCurrentCollateralAmount",
        ("FIX.5.0", "2703"): "2703 SideHaircutIndicator",
        ("FIX.5.0", "2704"): "2704 ExDestinationType",
        ("FIX.5.0", "2705"): "2705 MarketCondition",
        ("FIX.5.0", "2706"): "2706 NoQuoteAttributes",
        ("FIX.5.0", "2707"): "2707 QuoteAttributeType",
        ("FIX.5.0", "2708"): "2708 QuoteAttributeValue",
        ("FIX.5.0", "2709"): "2709 NoPriceQualifiers",
        ("FIX.5.0", "2710"): "2710 PriceQualifier",
        ("FIX.5.0", "2711"): "2711 MDValueTier",
        ("FIX.5.0", "2712"): "2712 MiscFeeQualifier",
        ("FIX.5.0", "2713"): "2713 MiscFeeDesc",
        ("FIX.5.0", "2714"): "2714 FinancialInstrumentFullName",
        ("FIX.5.0", "2715"): "2715 EncodedFinancialInstrumentFullNameLen",
        ("FIX.5.0", "2716"): "2716 EncodedFinancialInstrumentFullName",
        ("FIX.5.0", "2717"): "2717 LegFinancialInstrumentFullName",
        ("FIX.5.0", "2718"): "2718 EncodedLegFinancialInstrumentFullNameLen",
        ("FIX.5.0", "2719"): "2719 EncodedLegFinancialInstrumentFullName",
        ("FIX.5.0", "2720"): "2720 UnderlyingFinancialInstrumentFullName",
        ("FIX.5.0", "2721"): "2721 EncodedUnderlyingFinancialInstrumentFullNameLen",
        ("FIX.5.0", "2722"): "2722 EncodedUnderlyingFinancialInstrumentFullName",
        ("FIX.5.0", "2723"): "2723 UnderlyingIndexCurveUnit",
        ("FIX.5.0", "2724"): "2724 UnderlyingIndexCurvePeriod",
        ("FIX.5.0", "2725"): "2725 CommissionAmountSubType",
        ("FIX.5.0", "2726"): "2726 AllocCommissionAmountSubType",
        ("FIX.5.0", "2727"): "2727 AllocLegRefID",
        ("FIX.5.0", "2728"): "2728 FloatingRateIndexCurvePeriod",
        ("FIX.5.0", "2729"): "2729 FloatingRateIndexCurveSpread",
        ("FIX.5.0", "2730"): "2730 FloatingRateIndexCurveUnit",
        ("FIX.5.0", "2731"): "2731 FloatingRateIndexID",
        ("FIX.5.0", "2732"): "2732 FloatingRateIndexIDSource",
        ("FIX.5.0", "2733"): "2733 IndexRollMonth",
        ("FIX.5.0", "2734"): "2734 NoIndexRollMonths",
        ("FIX.5.0", "2735"): "2735 AssetSubType",
        ("FIX.5.0", "2736"): "2736 CommodityFinalPriceType",
        ("FIX.5.0", "2737"): "2737 FinancialInstrumentShortName",
        ("FIX.5.0", "2738"): "2738 NextIndexRollDate",
        ("FIX.5.0", "2739"): "2739 LegAssetSubType",
        ("FIX.5.0", "2740"): "2740 LegFinancialInstrumentShortName",
        ("FIX.5.0", "2741"): "2741 SecondaryAssetSubType",
        ("FIX.5.0", "2742"): "2742 UnderlyingFinancialInstrumentShortName",
        ("FIX.5.0", "2743"): "2743 LegSecondaryAssetSubType",
        ("FIX.5.0", "2744"): "2744 UnderlyingAssetSubType",
        ("FIX.5.0", "2745"): "2745 UnderlyingSecondaryAssetSubType",
        ("FIX.5.0", "2746"): "2746 NoReferenceDataDates",
        ("FIX.5.0", "2747"): "2747 ReferenceDataDate",
        ("FIX.5.0", "2748"): "2748 ReferenceDataDateType",
        ("FIX.5.0", "2749"): "2749 ExecutionTimestamp",
        ("FIX.5.0", "2750"): "2750 ReportingPx",
        ("FIX.5.0", "2751"): "2751 ReportingQty",
        ("FIX.5.0", "2752"): "2752 DeliveryRouteOrCharter",
        ("FIX.5.0", "2753"): "2753 ReturnTrigger",
        ("FIX.5.0", "2754"): "2754 LegDeliveryRouteOrCharter",
        ("FIX.5.0", "2755"): "2755 LegReturnTrigger",
        ("FIX.5.0", "2756"): "2756 UnderlyingDeliveryRouteOrCharter",
        ("FIX.5.0", "2757"): "2757 UnderlyingReturnTrigger",
        ("FIX.5.0", "2758"): "2758 AllocRequestID",
        ("FIX.5.0", "2759"): "2759 GroupAmount",
        ("FIX.5.0", "2760"): "2760 GroupRemainingAmount",
        ("FIX.5.0", "2761"): "2761 AllocGroupAmount",
        ("FIX.5.0", "2762"): "2762 PriceMarkup",
        ("FIX.5.0", "2763"): "2763 AveragePriceType",
        ("FIX.5.0", "2764"): "2764 AveragePriceStartTime",
        ("FIX.5.0", "2765"): "2765 AveragePriceEndTime",
        ("FIX.5.0", "2766"): "2766 OrderPercentOfTotalVolume",
        ("FIX.5.0", "2767"): "2767 AllocGroupStatus",
        ("FIX.5.0", "40000"): "40000 NoAdditionalTermBondRefs",
        ("FIX.5.0", "40001"): "40001 AdditionalTermBondSecurityID",
        ("FIX.5.0", "40002"): "40002 AdditionalTermBondSecurityIDSource",
        ("FIX.5.0", "40003"): "40003 AdditionalTermBondDesc",
        ("FIX.5.0", "40004"): "40004 EncodedAdditionalTermBondDescLen",
        ("FIX.5.0", "40005"): "40005 EncodedAdditionalTermBondDesc",
        ("FIX.5.0", "40006"): "40006 AdditionalTermBondCurrency",
        ("FIX.5.0", "40007"): "40007 AdditionalTermBondIssuer",
        ("FIX.5.0", "40008"): "40008 EncodedAdditionalTermBondIssuerLen",
        ("FIX.5.0", "40009"): "40009 EncodedAdditionalTermBondIssuer",
        ("FIX.5.0", "40010"): "40010 AdditionalTermBondSeniority",
        ("FIX.5.0", "40011"): "40011 AdditionalTermBondCouponType",
        ("FIX.5.0", "40012"): "40012 AdditionalTermBondCouponRate",
        ("FIX.5.0", "40013"): "40013 AdditionalTermBondMaturityDate",
        ("FIX.5.0", "40014"): "40014 AdditionalTermBondParValue",
        ("FIX.5.0", "40015"): "40015 AdditionalTermBondCurrentTotalIssuedAmount",
        ("FIX.5.0", "40016"): "40016 AdditionalTermBondCouponFrequencyPeriod",
        ("FIX.5.0", "40017"): "40017 AdditionalTermBondCouponFrequencyUnit",
        ("FIX.5.0", "40018"): "40018 AdditionalTermBondDayCount",
        ("FIX.5.0", "40019"): "40019 NoAdditionalTerms",
        ("FIX.5.0", "40020"): "40020 AdditionalTermConditionPrecedentBondIndicator",
        ("FIX.5.0", "40021"): "40021 AdditionalTermDiscrepancyClauseIndicator",
        ("FIX.5.0", "40022"): "40022 NoCashSettlTerms",
        ("FIX.5.0", "40023"): "40023 CashSettlCurrency",
        ("FIX.5.0", "40024"): "40024 CashSettlValuationFirstBusinessDayOffset",
        ("FIX.5.0", "40025"): "40025 CashSettlValuationTime",
        ("FIX.5.0", "40026"): "40026 CashSettlBusinessCenter",
        ("FIX.5.0", "40027"): "40027 CashSettlQuoteMethod",
        ("FIX.5.0", "40028"): "40028 CashSettlQuoteAmount",
        ("FIX.5.0", "40029"): "40029 CashSettlQuoteCurrency",
        ("FIX.5.0", "40030"): "40030 CashSettlMinimumQuoteAmount",
        ("FIX.5.0", "40031"): "40031 CashSettlMinimumQuoteCurrency",
        ("FIX.5.0", "40032"): "40032 CashSettlDealer",
        ("FIX.5.0", "40033"): "40033 CashSettlBusinessDays",
        ("FIX.5.0", "40034"): "40034 CashSettlAmount",
        ("FIX.5.0", "40035"): "40035 CashSettlRecoveryFactor",
        ("FIX.5.0", "40036"): "40036 CashSettlFixedTermIndicator",
        ("FIX.5.0", "40037"): "40037 CashSettlAccruedInterestIndicator",
        ("FIX.5.0", "40038"): "40038 CashSettlValuationMethod",
        ("FIX.5.0", "40039"): "40039 CashSettlTermXID",
        ("FIX.5.0", "40040"): "40040 NoContractualDefinitions",
        ("FIX.5.0", "40041"): "40041 ContractualDefinition",
        ("FIX.5.0", "40042"): "40042 NoContractualMatrices",
        ("FIX.5.0", "40043"): "40043 ContractualMatrixSource",
        ("FIX.5.0", "40044"): "40044 ContractualMatrixDate",
        ("FIX.5.0", "40045"): "40045 ContractualMatrixTerm",
        ("FIX.5.0", "40046"): "40046 NoFinancingTermSupplements",
        ("FIX.5.0", "40047"): "40047 FinancingTermSupplementDesc",
        ("FIX.5.0", "40048"): "40048 FinancingTermSupplementDate",
        ("FIX.5.0", "40049"): "40049 NoStreams",
        ("FIX.5.0", "40050"): "40050 StreamType",
        ("FIX.5.0", "40051"): "40051 StreamDesc",
        ("FIX.5.0", "40052"): "40052 StreamPaySide",
        ("FIX.5.0", "40053"): "40053 StreamReceiveSide",
        ("FIX.5.0", "40054"): "40054 StreamNotional",
        ("FIX.5.0", "40055"): "40055 StreamCurrency",
        ("FIX.5.0", "40056"): "40056 StreamText",
        ("FIX.5.0", "40057"): "40057 UnderlyingStreamEffectiveDateUnadjusted",
        ("FIX.5.0", "40058"): "40058 UnderlyingStreamEffectiveDateBusinessDayConvention",
        ("FIX.5.0", "40059"): "40059 UnderlyingStreamEffectiveDateBusinessCenter",
        ("FIX.5.0", "40060"): "40060 UnderlyingStreamEffectiveDateRelativeTo",
        ("FIX.5.0", "40061"): "40061 UnderlyingStreamEffectiveDateOffsetPeriod",
        ("FIX.5.0", "40062"): "40062 UnderlyingStreamEffectiveDateOffsetUnit",
        ("FIX.5.0", "40063"): "40063 UnderlyingStreamEffectiveDateOffsetDayType",
        ("FIX.5.0", "40064"): "40064 UnderlyingStreamEffectiveDateAdjusted",
        ("FIX.5.0", "40065"): "40065 StreamTerminationDateUnadjusted",
        ("FIX.5.0", "40066"): "40066 StreamTerminationDateBusinessDayConvention",
        ("FIX.5.0", "40067"): "40067 StreamTerminationDateBusinessCenter",
        ("FIX.5.0", "40068"): "40068 StreamTerminationDateRelativeTo",
        ("FIX.5.0", "40069"): "40069 StreamTerminationDateOffsetPeriod",
        ("FIX.5.0", "40070"): "40070 StreamTerminationDateOffsetUnit",
        ("FIX.5.0", "40071"): "40071 StreamTerminationDateOffsetDayType",
        ("FIX.5.0", "40072"): "40072 StreamTerminationDateAdjusted",
        ("FIX.5.0", "40073"): "40073 StreamCalculationPeriodBusinessDayConvention",
        ("FIX.5.0", "40074"): "40074 StreamCalculationPeriodBusinessCenter",
        ("FIX.5.0", "40075"): "40075 StreamFirstPeriodStartDateUnadjusted",
        ("FIX.5.0", "40076"): "40076 StreamFirstPeriodStartDateBusinessDayConvention",
        ("FIX.5.0", "40077"): "40077 StreamFirstPeriodStartDateBusinessCenter",
        ("FIX.5.0", "40078"): "40078 StreamFirstPeriodStartDateAdjusted",
        ("FIX.5.0", "40079"): "40079 StreamFirstRegularPeriodStartDateUnadjusted",
        ("FIX.5.0", "40080"): "40080 StreamFirstCompoundingPeriodEndDateUnadjusted",
        ("FIX.5.0", "40081"): "40081 StreamLastRegularPeriodEndDateUnadjusted",
        ("FIX.5.0", "40082"): "40082 StreamCalculationFrequencyPeriod",
        ("FIX.5.0", "40083"): "40083 StreamCalculationFrequencyUnit",
        ("FIX.5.0", "40084"): "40084 StreamCalculationRollConvention",
        ("FIX.5.0", "40085"): "40085 NoSettlRateFallbacks",
        ("FIX.5.0", "40086"): "40086 SettlRatePostponementMaximumDays",
        ("FIX.5.0", "40087"): "40087 LegPaymentStreamNonDeliverableSettlRateSource",
        ("FIX.5.0", "40088"): "40088 SettlRatePostponementSurvey",
        ("FIX.5.0", "40089"): "40089 SettlRatePostponementCalculationAgent",
        ("FIX.5.0", "40090"): "40090 NoProvisions",
        ("FIX.5.0", "40091"): "40091 ProvisionType",
        ("FIX.5.0", "40092"): "40092 ProvisionDateUnadjusted",
        ("FIX.5.0", "40093"): "40093 ProvisionDateBusinessDayConvention",
        ("FIX.5.0", "40094"): "40094 ProvisionDateBusinessCenter",
        ("FIX.5.0", "40095"): "40095 ProvisionDateAdjusted",
        ("FIX.5.0", "40096"): "40096 ProvisionDateTenorPeriod",
        ("FIX.5.0", "40097"): "40097 ProvisionDateTenorUnit",
        ("FIX.5.0", "40098"): "40098 ProvisionCalculationAgent",
        ("FIX.5.0", "40099"): "40099 ProvisionOptionSinglePartyBuyerSide",
        ("FIX.5.0", "40100"): "40100 ProvisionOptionSinglePartySellerSide",
        ("FIX.5.0", "40101"): "40101 ProvisionOptionExerciseStyle",
        ("FIX.5.0", "40102"): "40102 ProvisionOptionExerciseMultipleNotional",
        ("FIX.5.0", "40103"): "40103 ProvisionOptionExerciseMinimumNotional",
        ("FIX.5.0", "40104"): "40104 ProvisionOptionExerciseMaximumNotional",
        ("FIX.5.0", "40105"): "40105 ProvisionOptionMinimumNumber",
        ("FIX.5.0", "40106"): "40106 ProvisionOptionMaximumNumber",
        ("FIX.5.0", "40107"): "40107 ProvisionOptionExerciseConfirmation",
        ("FIX.5.0", "40108"): "40108 ProvisionCashSettlMethod",
        ("FIX.5.0", "40109"): "40109 ProvisionCashSettlCurrency",
        ("FIX.5.0", "40110"): "40110 ProvisionCashSettlCurrency2",
        ("FIX.5.0", "40111"): "40111 ProvisionCashSettlQuoteType",
        ("FIX.5.0", "40112"): "40112 ProvisionCashSettlQuoteSource",
        ("FIX.5.0", "40113"): "40113 ProvisionText",
        ("FIX.5.0", "40114"): "40114 ProvisionCashSettlValueTime",
        ("FIX.5.0", "40115"): "40115 ProvisionCashSettlValueTimeBusinessCenter",
        ("FIX.5.0", "40116"): "40116 ProvisionCashSettlValueDateBusinessDayConvention",
        ("FIX.5.0", "40117"): "40117 ProvisionCashSettlValueDateBusinessCenter",
        ("FIX.5.0", "40118"): "40118 ProvisionCashSettlValueDateRelativeTo",
        ("FIX.5.0", "40119"): "40119 ProvisionCashSettlValueDateOffsetPeriod",
        ("FIX.5.0", "40120"): "40120 ProvisionCashSettlValueDateOffsetUnit",
        ("FIX.5.0", "40121"): "40121 ProvisionCashSettlValueDateOffsetDayType",
        ("FIX.5.0", "40122"): "40122 ProvisionCashSettlValueDateAdjusted",
        ("FIX.5.0", "40123"): "40123 ProvisionOptionExerciseBusinessDayConvention",
        ("FIX.5.0", "40124"): "40124 ProvisionOptionExerciseBusinessCenter",
        ("FIX.5.0", "40125"): "40125 ProvisionOptionExerciseEarliestDateOffsetPeriod",
        ("FIX.5.0", "40126"): "40126 ProvisionOptionExerciseEarliestDateOffsetUnit",
        ("FIX.5.0", "40127"): "40127 ProvisionOptionExerciseFrequencyPeriod",
        ("FIX.5.0", "40128"): "40128 ProvisionOptionExerciseFrequencyUnit",
        ("FIX.5.0", "40129"): "40129 ProvisionOptionExerciseStartDateUnadjusted",
        ("FIX.5.0", "40130"): "40130 ProvisionOptionExerciseStartDateRelativeTo",
        ("FIX.5.0", "40131"): "40131 ProvisionOptionExerciseStartDateOffsetPeriod",
        ("FIX.5.0", "40132"): "40132 ProvisionOptionExerciseStartDateOffsetUnit",
        ("FIX.5.0", "40133"): "40133 ProvisionOptionExerciseStartDateOffsetDayType",
        ("FIX.5.0", "40134"): "40134 ProvisionOptionExerciseStartDateAdjusted",
        ("FIX.5.0", "40135"): "40135 ProvisionOptionExercisePeriodSkip",
        ("FIX.5.0", "40136"): "40136 ProvisionOptionExerciseBoundsFirstDateUnadjusted",
        ("FIX.5.0", "40137"): "40137 ProvisionOptionExerciseBoundsLastDateUnadjusted",
        ("FIX.5.0", "40138"): "40138 ProvisionOptionExerciseEarliestTime",
        ("FIX.5.0", "40139"): "40139 ProvisionOptionExerciseEarliestTimeBusinessCenter",
        ("FIX.5.0", "40140"): "40140 ProvisionOptionExerciseLatestTime",
        ("FIX.5.0", "40141"): "40141 ProvisionOptionExerciseLatestTimeBusinessCenter",
        ("FIX.5.0", "40142"): "40142 NoProvisionOptionExerciseFixedDates",
        ("FIX.5.0", "40143"): "40143 ProvisionOptionExerciseFixedDate",
        ("FIX.5.0", "40144"): "40144 ProvisionOptionExerciseFixedDateType",
        ("FIX.5.0", "40145"): "40145 ProvisionOptionExpirationDateUnadjusted",
        ("FIX.5.0", "40146"): "40146 ProvisionOptionExpirationDateBusinessDayConvention",
        ("FIX.5.0", "40147"): "40147 ProvisionOptionExpirationDateBusinessCenter",
        ("FIX.5.0", "40148"): "40148 ProvisionOptionExpirationDateRelativeTo",
        ("FIX.5.0", "40149"): "40149 ProvisionOptionExpirationDateOffsetPeriod",
        ("FIX.5.0", "40150"): "40150 ProvisionOptionExpirationDateOffsetUnit",
        ("FIX.5.0", "40151"): "40151 ProvisionOptionExpirationDateOffsetDayType",
        ("FIX.5.0", "40152"): "40152 ProvisionOptionExpirationDateAdjusted",
        ("FIX.5.0", "40153"): "40153 ProvisionOptionExpirationTime",
        ("FIX.5.0", "40154"): "40154 ProvisionOptionExpirationTimeBusinessCenter",
        ("FIX.5.0", "40155"): "40155 ProvisionOptionRelevantUnderlyingDateUnadjusted",
        ("FIX.5.0", "40156"): "40156 ProvisionOptionRelevantUnderlyingDateBusinessDayConvention",
        ("FIX.5.0", "40157"): "40157 ProvisionOptionRelevantUnderlyingDateBusinessCenter",
        ("FIX.5.0", "40158"): "40158 ProvisionOptionRelevantUnderlyingDateRelativeTo",
        ("FIX.5.0", "40159"): "40159 ProvisionOptionRelevantUnderlyingDateOffsetPeriod",
        ("FIX.5.0", "40160"): "40160 ProvisionOptionRelevantUnderlyingDateOffsetUnit",
        ("FIX.5.0", "40161"): "40161 ProvisionOptionRelevantUnderlyingDateOffsetDayType",
        ("FIX.5.0", "40162"): "40162 ProvisionOptionRelevantUnderlyingDateAdjusted",
        ("FIX.5.0", "40163"): "40163 ProvisionCashSettlPaymentDateBusinessDayConvention",
        ("FIX.5.0", "40164"): "40164 ProvisionCashSettlPaymentDateBusinessCenter",
        ("FIX.5.0", "40165"): "40165 ProvisionCashSettlPaymentDateRelativeTo",
        ("FIX.5.0", "40166"): "40166 ProvisionCashSettlPaymentDateOffsetPeriod",
        ("FIX.5.0", "40167"): "40167 ProvisionCashSettlPaymentDateOffsetUnit",
        ("FIX.5.0", "40168"): "40168 ProvisionCashSettlPaymentDateOffsetDayType",
        ("FIX.5.0", "40169"): "40169 ProvisionCashSettlPaymentDateRangeFirst",
        ("FIX.5.0", "40170"): "40170 ProvisionCashSettlPaymentDateRangeLast",
        ("FIX.5.0", "40171"): "40171 NoProvisionCashSettlPaymentDates",
        ("FIX.5.0", "40172"): "40172 ProvisionCashSettlPaymentDate",
        ("FIX.5.0", "40173"): "40173 ProvisionCashSettlPaymentDateType",
        ("FIX.5.0", "40174"): "40174 NoProvisionPartyIDs",
        ("FIX.5.0", "40175"): "40175 ProvisionPartyID",
        ("FIX.5.0", "40176"): "40176 ProvisionPartyIDSource",
        ("FIX.5.0", "40177"): "40177 ProvisionPartyRole",
        ("FIX.5.0", "40178"): "40178 NoProvisionPartySubIDs",
        ("FIX.5.0", "40179"): "40179 ProvisionPartySubID",
        ("FIX.5.0", "40180"): "40180 ProvisionPartySubIDType",
        ("FIX.5.0", "40181"): "40181 NoProtectionTerms",
        ("FIX.5.0", "40182"): "40182 ProtectionTermNotional",
        ("FIX.5.0", "40183"): "40183 ProtectionTermCurrency",
        ("FIX.5.0", "40184"): "40184 ProtectionTermSellerNotifies",
        ("FIX.5.0", "40185"): "40185 ProtectionTermBuyerNotifies",
        ("FIX.5.0", "40186"): "40186 ProtectionTermEventBusinessCenter",
        ("FIX.5.0", "40187"): "40187 ProtectionTermStandardSources",
        ("FIX.5.0", "40188"): "40188 ProtectionTermEventMinimumSources",
        ("FIX.5.0", "40189"): "40189 ProtectionTermEventNewsSource",
        ("FIX.5.0", "40190"): "40190 ProtectionTermXID",
        ("FIX.5.0", "40191"): "40191 NoProtectionTermEvents",
        ("FIX.5.0", "40192"): "40192 ProtectionTermEventType",
        ("FIX.5.0", "40193"): "40193 ProtectionTermEventValue",
        ("FIX.5.0", "40194"): "40194 ProtectionTermEventCurrency",
        ("FIX.5.0", "40195"): "40195 ProtectionTermEventPeriod",
        ("FIX.5.0", "40196"): "40196 ProtectionTermEventUnit",
        ("FIX.5.0", "40197"): "40197 ProtectionTermEventDayType",
        ("FIX.5.0", "40198"): "40198 ProtectionTermEventRateSource",
        ("FIX.5.0", "40199"): "40199 NoProtectionTermEventQualifiers",
        ("FIX.5.0", "40200"): "40200 ProtectionTermEventQualifier",
        ("FIX.5.0", "40201"): "40201 NoProtectionTermObligations",
        ("FIX.5.0", "40202"): "40202 ProtectionTermObligationType",
        ("FIX.5.0", "40203"): "40203 ProtectionTermObligationValue",
        ("FIX.5.0", "40204"): "40204 NoPhysicalSettlTerms",
        ("FIX.5.0", "40205"): "40205 PhysicalSettlCurrency",
        ("FIX.5.0", "40206"): "40206 PhysicalSettlBusinessDays",
        ("FIX.5.0", "40207"): "40207 PhysicalSettlMaximumBusinessDays",
        ("FIX.5.0", "40208"): "40208 PhysicalSettlTermXID",
        ("FIX.5.0", "40209"): "40209 NoPhysicalSettlDeliverableObligations",
        ("FIX.5.0", "40210"): "40210 PhysicalSettlDeliverableObligationType",
        ("FIX.5.0", "40211"): "40211 PhysicalSettlDeliverableObligationValue",
        ("FIX.5.0", "40212"): "40212 NoPayments",
        ("FIX.5.0", "40213"): "40213 PaymentType",
        ("FIX.5.0", "40214"): "40214 PaymentPaySide",
        ("FIX.5.0", "40215"): "40215 PaymentReceiveSide",
        ("FIX.5.0", "40216"): "40216 PaymentCurrency",
        ("FIX.5.0", "40217"): "40217 PaymentAmount",
        ("FIX.5.0", "40218"): "40218 PaymentPrice",
        ("FIX.5.0", "40219"): "40219 PaymentDateUnadjusted",
        ("FIX.5.0", "40220"): "40220 PaymentBusinessDayConvention",
        ("FIX.5.0", "40221"): "40221 PaymentBusinessCenter",
        ("FIX.5.0", "40222"): "40222 PaymentDateAdjusted",
        ("FIX.5.0", "40223"): "40223 LegMarketDisruptionValue",
        ("FIX.5.0", "40224"): "40224 PaymentDiscountFactor",
        ("FIX.5.0", "40225"): "40225 PaymentPresentValueAmount",
        ("FIX.5.0", "40226"): "40226 PaymentPresentValueCurrency",
        ("FIX.5.0", "40227"): "40227 PaymentSettlStyle",
        ("FIX.5.0", "40228"): "40228 LegPaymentStreamNonDeliverableSettlReferencePage",
        ("FIX.5.0", "40229"): "40229 PaymentText",
        ("FIX.5.0", "40230"): "40230 NoPaymentSettls",
        ("FIX.5.0", "40231"): "40231 PaymentSettlAmount",
        ("FIX.5.0", "40232"): "40232 PaymentSettlCurrency",
        ("FIX.5.0", "40233"): "40233 NoPaymentSettlPartyIDs",
        ("FIX.5.0", "40234"): "40234 PaymentSettlPartyID",
        ("FIX.5.0", "40235"): "40235 PaymentSettlPartyIDSource",
        ("FIX.5.0", "40236"): "40236 PaymentSettlPartyRole",
        ("FIX.5.0", "40237"): "40237 PaymentSettlPartyRoleQualifier",
        ("FIX.5.0", "40238"): "40238 NoPaymentSettlPartySubIDs",
        ("FIX.5.0", "40239"): "40239 PaymentSettlPartySubID",
        ("FIX.5.0", "40240"): "40240 PaymentSettlPartySubIDType",
        ("FIX.5.0", "40241"): "40241 NoLegStreams",
        ("FIX.5.0", "40242"): "40242 LegStreamType",
        ("FIX.5.0", "40243"): "40243 LegStreamDesc",
        ("FIX.5.0", "40244"): "40244 LegStreamPaySide",
        ("FIX.5.0", "40245"): "40245 LegStreamReceiveSide",
        ("FIX.5.0", "40246"): "40246 LegStreamNotional",
        ("FIX.5.0", "40247"): "40247 LegStreamCurrency",
        ("FIX.5.0", "40248"): "40248 LegStreamText",
        ("FIX.5.0", "40249"): "40249 LegStreamEffectiveDateUnadjusted",
        ("FIX.5.0", "40250"): "40250 LegStreamEffectiveDateBusinessDayConvention",
        ("FIX.5.0", "40251"): "40251 LegStreamEffectiveDateBusinessCenter",
        ("FIX.5.0", "40252"): "40252 LegStreamEffectiveDateRelativeTo",
        ("FIX.5.0", "40253"): "40253 LegStreamEffectiveDateOffsetPeriod",
        ("FIX.5.0", "40254"): "40254 LegStreamEffectiveDateOffsetUnit",
        ("FIX.5.0", "40255"): "40255 LegStreamEffectiveDateOffsetDayType",
        ("FIX.5.0", "40256"): "40256 LegStreamEffectiveDateAdjusted",
        ("FIX.5.0", "40257"): "40257 LegStreamTerminationDateUnadjusted",
        ("FIX.5.0", "40258"): "40258 LegStreamTerminationDateBusinessDayConvention",
        ("FIX.5.0", "40259"): "40259 LegStreamTerminationDateBusinessCenter",
        ("FIX.5.0", "40260"): "40260 LegStreamTerminationDateRelativeTo",
        ("FIX.5.0", "40261"): "40261 LegStreamTerminationDateOffsetPeriod",
        ("FIX.5.0", "40262"): "40262 LegStreamTerminationDateOffsetUnit",
        ("FIX.5.0", "40263"): "40263 LegStreamTerminationDateOffsetDayType",
        ("FIX.5.0", "40264"): "40264 LegStreamTerminationDateAdjusted",
        ("FIX.5.0", "40265"): "40265 LegStreamCalculationPeriodBusinessDayConvention",
        ("FIX.5.0", "40266"): "40266 LegStreamCalculationPeriodBusinessCenter",
        ("FIX.5.0", "40267"): "40267 LegStreamFirstPeriodStartDateUnadjusted",
        ("FIX.5.0", "40268"): "40268 LegStreamFirstPeriodStartDateBusinessDayConvention",
        ("FIX.5.0", "40269"): "40269 LegStreamFirstPeriodStartDateBusinessCenter",
        ("FIX.5.0", "40270"): "40270 LegStreamFirstPeriodStartDateAdjusted",
        ("FIX.5.0", "40271"): "40271 LegStreamFirstRegularPeriodStartDateUnadjusted",
        ("FIX.5.0", "40272"): "40272 LegStreamFirstCompoundingPeriodEndDateUnadjusted",
        ("FIX.5.0", "40273"): "40273 LegStreamLastRegularPeriodEndDateUnadjusted",
        ("FIX.5.0", "40274"): "40274 LegStreamCalculationFrequencyPeriod",
        ("FIX.5.0", "40275"): "40275 LegStreamCalculationFrequencyUnit",
        ("FIX.5.0", "40276"): "40276 LegStreamCalculationRollConvention",
        ("FIX.5.0", "40277"): "40277 NoCashSettlDealers",
        ("FIX.5.0", "40278"): "40278 NoBusinessCenters",
        ("FIX.5.0", "40279"): "40279 LegPaymentStreamType",
        ("FIX.5.0", "40280"): "40280 LegPaymentStreamMarketRate",
        ("FIX.5.0", "40281"): "40281 LegPaymentStreamDelayIndicator",
        ("FIX.5.0", "40282"): "40282 LegPaymentStreamSettlCurrency",
        ("FIX.5.0", "40283"): "40283 LegPaymentStreamDayCount",
        ("FIX.5.0", "40284"): "40284 LegPaymentStreamAccrualDays",
        ("FIX.5.0", "40285"): "40285 LegPaymentStreamDiscountType",
        ("FIX.5.0", "40286"): "40286 LegPaymentStreamDiscountRate",
        ("FIX.5.0", "40287"): "40287 LegPaymentStreamDiscountRateDayCount",
        ("FIX.5.0", "40288"): "40288 LegPaymentStreamCompoundingMethod",
        ("FIX.5.0", "40289"): "40289 LegPaymentStreamInitialPrincipalExchangeIndicator",
        ("FIX.5.0", "40290"): "40290 LegPaymentStreamInterimPrincipalExchangeIndicator",
        ("FIX.5.0", "40291"): "40291 LegPaymentStreamFinalPrincipalExchangeIndicator",
        ("FIX.5.0", "40292"): "40292 LegPaymentStreamPaymentDateBusinessDayConvention",
        ("FIX.5.0", "40293"): "40293 LegPaymentStreamPaymentDateBusinessCenter",
        ("FIX.5.0", "40294"): "40294 LegPaymentStreamPaymentFrequencyPeriod",
        ("FIX.5.0", "40295"): "40295 LegPaymentStreamPaymentFrequencyUnit",
        ("FIX.5.0", "40296"): "40296 LegPaymentStreamPaymentRollConvention",
        ("FIX.5.0", "40297"): "40297 LegPaymentStreamFirstPaymentDateUnadjusted",
        ("FIX.5.0", "40298"): "40298 LegPaymentStreamLastRegularPaymentDateUnadjusted",
        ("FIX.5.0", "40299"): "40299 LegPaymentStreamPaymentDateRelativeTo",
        ("FIX.5.0", "40300"): "40300 LegPaymentStreamPaymentDateOffsetPeriod",
        ("FIX.5.0", "40301"): "40301 LegPaymentStreamPaymentDateOffsetUnit",
        ("FIX.5.0", "40302"): "40302 LegPaymentStreamPaymentDateOffsetDayType",
        ("FIX.5.0", "40303"): "40303 LegPaymentStreamResetDateRelativeTo",
        ("FIX.5.0", "40304"): "40304 LegPaymentStreamResetDateBusinessDayConvention",
        ("FIX.5.0", "40305"): "40305 LegPaymentStreamResetDateBusinessCenter",
        ("FIX.5.0", "40306"): "40306 LegPaymentStreamResetFrequencyPeriod",
        ("FIX.5.0", "40307"): "40307 LegPaymentStreamResetFrequencyUnit",
        ("FIX.5.0", "40308"): "40308 LegPaymentStreamResetWeeklyRollConvention",
        ("FIX.5.0", "40309"): "40309 LegPaymentStreamInitialFixingDateRelativeTo",
        ("FIX.5.0", "40310"): "40310 LegPaymentStreamInitialFixingDateBusinessDayConvention",
        ("FIX.5.0", "40311"): "40311 LegPaymentStreamInitialFixingDateBusinessCenter",
        ("FIX.5.0", "40312"): "40312 LegPaymentStreamInitialFixingDateOffsetPeriod",
        ("FIX.5.0", "40313"): "40313 LegPaymentStreamInitialFixingDateOffsetUnit",
        ("FIX.5.0", "40314"): "40314 LegPaymentStreamInitialFixingDateOffsetDayType",
        ("FIX.5.0", "40315"): "40315 LegPaymentStreamInitialFixingDateAdjusted",
        ("FIX.5.0", "40316"): "40316 LegPaymentStreamFixingDateRelativeTo",
        ("FIX.5.0", "40317"): "40317 LegPaymentStreamFixingDateBusinessDayConvention",
        ("FIX.5.0", "40318"): "40318 LegPaymentStreamFixingDateBusinessCenter",
        ("FIX.5.0", "40319"): "40319 LegPaymentStreamFixingDateOffsetPeriod",
        ("FIX.5.0", "40320"): "40320 LegPaymentStreamFixingDateOffsetUnit",
        ("FIX.5.0", "40321"): "40321 LegPaymentStreamFixingDateOffsetDayType",
        ("FIX.5.0", "40322"): "40322 LegPaymentStreamFixingDateAdjusted",
        ("FIX.5.0", "40323"): "40323 LegPaymentStreamRateCutoffDateOffsetPeriod",
        ("FIX.5.0", "40324"): "40324 LegPaymentStreamRateCutoffDateOffsetUnit",
        ("FIX.5.0", "40325"): "40325 LegPaymentStreamRateCutoffDateOffsetDayType",
        ("FIX.5.0", "40326"): "40326 LegPaymentStreamRate",
        ("FIX.5.0", "40327"): "40327 LegPaymentStreamFixedAmount",
        ("FIX.5.0", "40328"): "40328 LegPaymentStreamRateOrAmountCurrency",
        ("FIX.5.0", "40329"): "40329 LegPaymentStreamFutureValueNotional",
        ("FIX.5.0", "40330"): "40330 LegPaymentStreamFutureValueDateAdjusted",
        ("FIX.5.0", "40331"): "40331 LegPaymentStreamRateIndex",
        ("FIX.5.0", "40332"): "40332 LegPaymentStreamRateIndexSource",
        ("FIX.5.0", "40333"): "40333 LegPaymentStreamRateIndexCurveUnit",
        ("FIX.5.0", "40334"): "40334 LegPaymentStreamRateIndexCurvePeriod",
        ("FIX.5.0", "40335"): "40335 LegPaymentStreamRateMultiplier",
        ("FIX.5.0", "40336"): "40336 LegPaymentStreamRateSpread",
        ("FIX.5.0", "40337"): "40337 LegPaymentStreamRateSpreadPositionType",
        ("FIX.5.0", "40338"): "40338 LegPaymentStreamRateTreatment",
        ("FIX.5.0", "40339"): "40339 LegPaymentStreamCapRate",
        ("FIX.5.0", "40340"): "40340 LegPaymentStreamCapRateBuySide",
        ("FIX.5.0", "40341"): "40341 LegPaymentStreamCapRateSellSide",
        ("FIX.5.0", "40342"): "40342 LegPaymentStreamFloorRate",
        ("FIX.5.0", "40343"): "40343 LegPaymentStreamFloorRateBuySide",
        ("FIX.5.0", "40344"): "40344 LegPaymentStreamFloorRateSellSide",
        ("FIX.5.0", "40345"): "40345 LegPaymentStreamInitialRate",
        ("FIX.5.0", "40346"): "40346 LegPaymentStreamFinalRateRoundingDirection",
        ("FIX.5.0", "40347"): "40347 LegPaymentStreamFinalRatePrecision",
        ("FIX.5.0", "40348"): "40348 LegPaymentStreamAveragingMethod",
        ("FIX.5.0", "40349"): "40349 LegPaymentStreamNegativeRateTreatment",
        ("FIX.5.0", "40350"): "40350 LegPaymentStreamInflationLagPeriod",
        ("FIX.5.0", "40351"): "40351 LegPaymentStreamInflationLagUnit",
        ("FIX.5.0", "40352"): "40352 LegPaymentStreamInflationLagDayType",
        ("FIX.5.0", "40353"): "40353 LegPaymentStreamInflationInterpolationMethod",
        ("FIX.5.0", "40354"): "40354 LegPaymentStreamInflationIndexSource",
        ("FIX.5.0", "40355"): "40355 LegPaymentStreamInflationPublicationSource",
        ("FIX.5.0", "40356"): "40356 LegPaymentStreamInflationInitialIndexLevel",
        ("FIX.5.0", "40357"): "40357 LegPaymentStreamInflationFallbackBondApplicable",
        ("FIX.5.0", "40358"): "40358 LegPaymentStreamFRADiscounting",
        ("FIX.5.0", "40359"): "40359 LegPaymentStreamNonDeliverableRefCurrency",
        ("FIX.5.0", "40360"): "40360 LegPaymentStreamNonDeliverableFixingDatesBusinessDayConvention",
        ("FIX.5.0", "40361"): "40361 LegPaymentStreamNonDeliverableFixingDatesBusinessCenter",
        ("FIX.5.0", "40362"): "40362 LegPaymentStreamNonDeliverableFixingDatesRelativeTo",
        ("FIX.5.0", "40363"): "40363 LegPaymentStreamNonDeliverableFixingDatesOffsetPeriod",
        ("FIX.5.0", "40364"): "40364 LegPaymentStreamNonDeliverableFixingDatesOffsetUnit",
        ("FIX.5.0", "40365"): "40365 LegPaymentStreamNonDeliverableFixingDatesOffsetDayType",
        ("FIX.5.0", "40366"): "40366 LegSettlRateFallbackRateSource",
        ("FIX.5.0", "40367"): "40367 NoLegNonDeliverableFixingDates",
        ("FIX.5.0", "40368"): "40368 LegNonDeliverableFixingDate",
        ("FIX.5.0", "40369"): "40369 LegNonDeliverableFixingDateType",
        ("FIX.5.0", "40370"): "40370 LegSettlRateFallbackReferencePage",
        ("FIX.5.0", "40371"): "40371 PaymentStreamNonDeliverableSettlRateSource",
        ("FIX.5.0", "40372"): "40372 PaymentStreamNonDeliverableSettlReferencePage",
        ("FIX.5.0", "40373"): "40373 SettlRateFallbackRateSource",
        ("FIX.5.0", "40374"): "40374 NoLegPaymentSchedules",
        ("FIX.5.0", "40375"): "40375 LegPaymentScheduleType",
        ("FIX.5.0", "40376"): "40376 LegPaymentScheduleStubType",
        ("FIX.5.0", "40377"): "40377 LegPaymentScheduleStartDateUnadjusted",
        ("FIX.5.0", "40378"): "40378 LegPaymentScheduleEndDateUnadjusted",
        ("FIX.5.0", "40379"): "40379 LegPaymentSchedulePaySide",
        ("FIX.5.0", "40380"): "40380 LegPaymentScheduleReceiveSide",
        ("FIX.5.0", "40381"): "40381 LegPaymentScheduleNotional",
        ("FIX.5.0", "40382"): "40382 LegPaymentScheduleCurrency",
        ("FIX.5.0", "40383"): "40383 LegPaymentScheduleRate",
        ("FIX.5.0", "40384"): "40384 LegPaymentScheduleRateMultiplier",
        ("FIX.5.0", "40385"): "40385 LegPaymentScheduleRateSpread",
        ("FIX.5.0", "40386"): "40386 LegPaymentScheduleRateSpreadPositionType",
        ("FIX.5.0", "40387"): "40387 LegPaymentScheduleRateTreatment",
        ("FIX.5.0", "40388"): "40388 LegPaymentScheduleFixedAmount",
        ("FIX.5.0", "40389"): "40389 LegPaymentScheduleFixedCurrency",
        ("FIX.5.0", "40390"): "40390 LegPaymentScheduleStepFrequencyPeriod",
        ("FIX.5.0", "40391"): "40391 LegPaymentScheduleStepFrequencyUnit",
        ("FIX.5.0", "40392"): "40392 LegPaymentScheduleStepOffsetValue",
        ("FIX.5.0", "40393"): "40393 LegPaymentScheduleStepRate",
        ("FIX.5.0", "40394"): "40394 LegPaymentScheduleStepOffsetRate",
        ("FIX.5.0", "40395"): "40395 LegPaymentScheduleStepRelativeTo",
        ("FIX.5.0", "40396"): "40396 LegPaymentScheduleFixingDateUnadjusted",
        ("FIX.5.0", "40397"): "40397 LegPaymentScheduleWeight",
        ("FIX.5.0", "40398"): "40398 LegPaymentScheduleFixingDateRelativeTo",
        ("FIX.5.0", "40399"): "40399 LegPaymentScheduleFixingDateBusinessDayConvention",
        ("FIX.5.0", "40400"): "40400 LegPaymentScheduleFixingDateBusinessCenter",
        ("FIX.5.0", "40401"): "40401 LegPaymentScheduleFixingDateOffsetPeriod",
        ("FIX.5.0", "40402"): "40402 LegPaymentScheduleFixingDateOffsetUnit",
        ("FIX.5.0", "40403"): "40403 LegPaymentScheduleFixingDateOffsetDayType",
        ("FIX.5.0", "40404"): "40404 LegPaymentScheduleFixingDateAdjusted",
        ("FIX.5.0", "40405"): "40405 LegPaymentScheduleFixingTime",
        ("FIX.5.0", "40406"): "40406 LegPaymentScheduleFixingTimeBusinessCenter",
        ("FIX.5.0", "40407"): "40407 LegPaymentScheduleInterimExchangePaymentDateRelativeTo",
        ("FIX.5.0", "40408"): "40408 LegPaymentScheduleInterimExchangeDatesBusinessDayConvention",
        ("FIX.5.0", "40409"): "40409 LegPaymentScheduleInterimExchangeDatesBusinessCenter",
        ("FIX.5.0", "40410"): "40410 LegPaymentScheduleInterimExchangeDatesOffsetPeriod",
        ("FIX.5.0", "40411"): "40411 LegPaymentScheduleInterimExchangeDatesOffsetUnit",
        ("FIX.5.0", "40412"): "40412 LegPaymentScheduleInterimExchangeDatesOffsetDayType",
        ("FIX.5.0", "40413"): "40413 LegPaymentScheduleInterimExchangeDateAdjusted",
        ("FIX.5.0", "40414"): "40414 NoLegPaymentScheduleRateSources",
        ("FIX.5.0", "40415"): "40415 LegPaymentScheduleRateSource",
        ("FIX.5.0", "40416"): "40416 LegPaymentScheduleRateSourceType",
        ("FIX.5.0", "40417"): "40417 LegPaymentScheduleReferencePage",
        ("FIX.5.0", "40418"): "40418 NoLegPaymentStubs",
        ("FIX.5.0", "40419"): "40419 LegPaymentStubType",
        ("FIX.5.0", "40420"): "40420 LegPaymentStubLength",
        ("FIX.5.0", "40421"): "40421 LegPaymentStubRate",
        ("FIX.5.0", "40422"): "40422 LegPaymentStubFixedAmount",
        ("FIX.5.0", "40423"): "40423 LegPaymentStubFixedCurrency",
        ("FIX.5.0", "40424"): "40424 LegPaymentStubIndex",
        ("FIX.5.0", "40425"): "40425 LegPaymentStubIndexSource",
        ("FIX.5.0", "40426"): "40426 LegPaymentStubIndexCurvePeriod",
        ("FIX.5.0", "40427"): "40427 LegPaymentStubIndexCurveUnit",
        ("FIX.5.0", "40428"): "40428 LegPaymentStubIndexRateMultiplier",
        ("FIX.5.0", "40429"): "40429 LegPaymentStubIndexRateSpread",
        ("FIX.5.0", "40430"): "40430 LegPaymentStubIndexRateSpreadPositionType",
        ("FIX.5.0", "40431"): "40431 LegPaymentStubIndexRateTreatment",
        ("FIX.5.0", "40432"): "40432 LegPaymentStubIndexCapRate",
        ("FIX.5.0", "40433"): "40433 LegPaymentStubIndexCapRateBuySide",
        ("FIX.5.0", "40434"): "40434 LegPaymentStubIndexCapRateSellSide",
        ("FIX.5.0", "40435"): "40435 LegPaymentStubIndexFloorRate",
        ("FIX.5.0", "40436"): "40436 LegPaymentStubIndexFloorRateBuySide",
        ("FIX.5.0", "40437"): "40437 LegPaymentStubIndexFloorRateSellSide",
        ("FIX.5.0", "40438"): "40438 LegPaymentStubIndex2",
        ("FIX.5.0", "40439"): "40439 LegPaymentStubIndex2Source",
        ("FIX.5.0", "40440"): "40440 LegPaymentStubIndex2CurvePeriod",
        ("FIX.5.0", "40441"): "40441 LegPaymentStubIndex2CurveUnit",
        ("FIX.5.0", "40442"): "40442 LegPaymentStubIndex2RateMultiplier",
        ("FIX.5.0", "40443"): "40443 LegPaymentStubIndex2RateSpread",
        ("FIX.5.0", "40444"): "40444 LegPaymentStubIndex2RateSpreadPositionType",
        ("FIX.5.0", "40445"): "40445 LegPaymentStubIndex2RateTreatment",
        ("FIX.5.0", "40446"): "40446 LegPaymentStubIndex2CapRate",
        ("FIX.5.0", "40447"): "40447 LegPaymentStubIndex2FloorRate",
        ("FIX.5.0", "40448"): "40448 NoLegProvisions",
        ("FIX.5.0", "40449"): "40449 LegProvisionType",
        ("FIX.5.0", "40450"): "40450 LegProvisionDateUnadjusted",
        ("FIX.5.0", "40451"): "40451 LegProvisionDateBusinessDayConvention",
        ("FIX.5.0", "40452"): "40452 LegProvisionDateBusinessCenter",
        ("FIX.5.0", "40453"): "40453 LegProvisionDateAdjusted",
        ("FIX.5.0", "40454"): "40454 LegProvisionDateTenorPeriod",
        ("FIX.5.0", "40455"): "40455 LegProvisionDateTenorUnit",
        ("FIX.5.0", "40456"): "40456 LegProvisionCalculationAgent",
        ("FIX.5.0", "40457"): "40457 LegProvisionOptionSinglePartyBuyerSide",
        ("FIX.5.0", "40458"): "40458 LegProvisionOptionSinglePartySellerSide",
        ("FIX.5.0", "40459"): "40459 LegProvisionOptionExerciseStyle",
        ("FIX.5.0", "40460"): "40460 LegProvisionOptionExerciseMultipleNotional",
        ("FIX.5.0", "40461"): "40461 LegProvisionOptionExerciseMinimumNotional",
        ("FIX.5.0", "40462"): "40462 LegProvisionOptionExerciseMaximumNotional",
        ("FIX.5.0", "40463"): "40463 LegProvisionOptionMinimumNumber",
        ("FIX.5.0", "40464"): "40464 LegProvisionOptionMaximumNumber",
        ("FIX.5.0", "40465"): "40465 LegProvisionOptionExerciseConfirmation",
        ("FIX.5.0", "40466"): "40466 LegProvisionCashSettlMethod",
        ("FIX.5.0", "40467"): "40467 LegProvisionCashSettlCurrency",
        ("FIX.5.0", "40468"): "40468 LegProvisionCashSettlCurrency2",
        ("FIX.5.0", "40469"): "40469 LegProvisionCashSettlQuoteType",
        ("FIX.5.0", "40470"): "40470 LegProvisionCashSettlQuoteSource",
        ("FIX.5.0", "40471"): "40471 BusinessCenter",
        ("FIX.5.0", "40472"): "40472 LegProvisionText",
        ("FIX.5.0", "40473"): "40473 NoLegProvisionCashSettlPaymentDates",
        ("FIX.5.0", "40474"): "40474 LegProvisionCashSettlPaymentDate",
        ("FIX.5.0", "40475"): "40475 LegProvisionCashSettlPaymentDateType",
        ("FIX.5.0", "40476"): "40476 LegProvisionOptionExerciseBusinessDayConvention",
        ("FIX.5.0", "40477"): "40477 LegProvisionOptionExerciseBusinessCenter",
        ("FIX.5.0", "40478"): "40478 LegProvisionOptionExerciseEarliestDateOffsetPeriod",
        ("FIX.5.0", "40479"): "40479 LegProvisionOptionExerciseEarliestDateOffsetUnit",
        ("FIX.5.0", "40480"): "40480 LegProvisionOptionExerciseFrequencyPeriod",
        ("FIX.5.0", "40481"): "40481 LegProvisionOptionExerciseFrequencyUnit",
        ("FIX.5.0", "40482"): "40482 LegProvisionOptionExerciseStartDateUnadjusted",
        ("FIX.5.0", "40483"): "40483 LegProvisionOptionExerciseStartDateRelativeTo",
        ("FIX.5.0", "40484"): "40484 LegProvisionOptionExerciseStartDateOffsetPeriod",
        ("FIX.5.0", "40485"): "40485 LegProvisionOptionExerciseStartDateOffsetUnit",
        ("FIX.5.0", "40486"): "40486 LegProvisionOptionExerciseStartDateOffsetDayType",
        ("FIX.5.0", "40487"): "40487 LegProvisionOptionExerciseStartDateAdjusted",
        ("FIX.5.0", "40488"): "40488 LegProvisionOptionExercisePeriodSkip",
        ("FIX.5.0", "40489"): "40489 LegProvisionOptionExerciseBoundsFirstDateUnadjusted",
        ("FIX.5.0", "40490"): "40490 LegProvisionOptionExerciseBoundsLastDateUnadjusted",
        ("FIX.5.0", "40491"): "40491 LegProvisionOptionExerciseEarliestTime",
        ("FIX.5.0", "40492"): "40492 LegProvisionOptionExerciseEarliestTimeBusinessCenter",
        ("FIX.5.0", "40493"): "40493 LegProvisionOptionExerciseLatestTime",
        ("FIX.5.0", "40494"): "40494 LegProvisionOptionExerciseLatestTimeBusinessCenter",
        ("FIX.5.0", "40495"): "40495 NoLegProvisionOptionExerciseFixedDates",
        ("FIX.5.0", "40496"): "40496 LegProvisionOptionExerciseFixedDate",
        ("FIX.5.0", "40497"): "40497 LegProvisionOptionExerciseFixedDateType",
        ("FIX.5.0", "40498"): "40498 LegProvisionOptionExpirationDateUnadjusted",
        ("FIX.5.0", "40499"): "40499 LegProvisionOptionExpirationDateBusinessDayConvention",
        ("FIX.5.0", "40500"): "40500 LegProvisionOptionExpirationDateBusinessCenter",
        ("FIX.5.0", "40501"): "40501 LegProvisionOptionExpirationDateRelativeTo",
        ("FIX.5.0", "40502"): "40502 LegProvisionOptionExpirationDateOffsetPeriod",
        ("FIX.5.0", "40503"): "40503 LegProvisionOptionExpirationDateOffsetUnit",
        ("FIX.5.0", "40504"): "40504 LegProvisionOptionExpirationDateOffsetDayType",
        ("FIX.5.0", "40505"): "40505 LegProvisionOptionExpirationDateAdjusted",
        ("FIX.5.0", "40506"): "40506 LegProvisionOptionExpirationTime",
        ("FIX.5.0", "40507"): "40507 LegProvisionOptionExpirationTimeBusinessCenter",
        ("FIX.5.0", "40508"): "40508 LegProvisionOptionRelevantUnderlyingDateUnadjusted",
        ("FIX.5.0", "40509"): "40509 LegProvisionOptionRelevantUnderlyingDateBusinessDayConvention",
        ("FIX.5.0", "40510"): "40510 LegProvisionOptionRelevantUnderlyingDateBusinessCenter",
        ("FIX.5.0", "40511"): "40511 LegProvisionOptionRelevantUnderlyingDateRelativeTo",
        ("FIX.5.0", "40512"): "40512 LegProvisionOptionRelevantUnderlyingDateOffsetPeriod",
        ("FIX.5.0", "40513"): "40513 LegProvisionOptionRelevantUnderlyingDateOffsetUnit",
        ("FIX.5.0", "40514"): "40514 LegProvisionOptionRelevantUnderlyingDateOffsetDayType",
        ("FIX.5.0", "40515"): "40515 LegProvisionOptionRelevantUnderlyingDateAdjusted",
        ("FIX.5.0", "40516"): "40516 LegProvisionCashSettlPaymentDateBusinessDayConvention",
        ("FIX.5.0", "40517"): "40517 LegProvisionCashSettlPaymentDateBusinessCenter",
        ("FIX.5.0", "40518"): "40518 LegProvisionCashSettlPaymentDateRelativeTo",
        ("FIX.5.0", "40519"): "40519 LegProvisionCashSettlPaymentDateOffsetPeriod",
        ("FIX.5.0", "40520"): "40520 LegProvisionCashSettlPaymentDateOffsetUnit",
        ("FIX.5.0", "40521"): "40521 LegProvisionCashSettlPaymentDateOffsetDayType",
        ("FIX.5.0", "40522"): "40522 LegProvisionCashSettlPaymentDateRangeFirst",
        ("FIX.5.0", "40523"): "40523 LegProvisionCashSettlPaymentDateRangeLast",
        ("FIX.5.0", "40524"): "40524 LegProvisionCashSettlValueTime",
        ("FIX.5.0", "40525"): "40525 LegProvisionCashSettlValueTimeBusinessCenter",
        ("FIX.5.0", "40526"): "40526 LegProvisionCashSettlValueDateBusinessDayConvention",
        ("FIX.5.0", "40527"): "40527 LegProvisionCashSettlValueDateBusinessCenter",
        ("FIX.5.0", "40528"): "40528 LegProvisionCashSettlValueDateRelativeTo",
        ("FIX.5.0", "40529"): "40529 LegProvisionCashSettlValueDateOffsetPeriod",
        ("FIX.5.0", "40530"): "40530 LegProvisionCashSettlValueDateOffsetUnit",
        ("FIX.5.0", "40531"): "40531 LegProvisionCashSettlValueDateOffsetDayType",
        ("FIX.5.0", "40532"): "40532 LegProvisionCashSettlValueDateAdjusted",
        ("FIX.5.0", "40533"): "40533 NoLegProvisionPartyIDs",
        ("FIX.5.0", "40534"): "40534 LegProvisionPartyID",
        ("FIX.5.0", "40535"): "40535 LegProvisionPartyIDSource",
        ("FIX.5.0", "40536"): "40536 LegProvisionPartyRole",
        ("FIX.5.0", "40537"): "40537 NoLegProvisionPartySubIDs",
        ("FIX.5.0", "40538"): "40538 LegProvisionPartySubID",
        ("FIX.5.0", "40539"): "40539 LegProvisionPartySubIDType",
        ("FIX.5.0", "40540"): "40540 NoUnderlyingStreams",
        ("FIX.5.0", "40541"): "40541 UnderlyingStreamType",
        ("FIX.5.0", "40542"): "40542 UnderlyingStreamDesc",
        ("FIX.5.0", "40543"): "40543 UnderlyingStreamPaySide",
        ("FIX.5.0", "40544"): "40544 UnderlyingStreamReceiveSide",
        ("FIX.5.0", "40545"): "40545 UnderlyingStreamNotional",
        ("FIX.5.0", "40546"): "40546 UnderlyingStreamCurrency",
        ("FIX.5.0", "40547"): "40547 UnderlyingStreamText",
        ("FIX.5.0", "40548"): "40548 UnderlyingStreamTerminationDateUnadjusted",
        ("FIX.5.0", "40549"): "40549 UnderlyingStreamTerminationDateBusinessDayConvention",
        ("FIX.5.0", "40550"): "40550 UnderlyingStreamTerminationDateBusinessCenter",
        ("FIX.5.0", "40551"): "40551 UnderlyingStreamTerminationDateRelativeTo",
        ("FIX.5.0", "40552"): "40552 UnderlyingStreamTerminationDateOffsetPeriod",
        ("FIX.5.0", "40553"): "40553 UnderlyingStreamTerminationDateOffsetUnit",
        ("FIX.5.0", "40554"): "40554 UnderlyingStreamTerminationDateOffsetDayType",
        ("FIX.5.0", "40555"): "40555 UnderlyingStreamTerminationDateAdjusted",
        ("FIX.5.0", "40556"): "40556 UnderlyingStreamCalculationPeriodBusinessDayConvention",
        ("FIX.5.0", "40557"): "40557 UnderlyingStreamCalculationPeriodBusinessCenter",
        ("FIX.5.0", "40558"): "40558 UnderlyingStreamFirstPeriodStartDateUnadjusted",
        ("FIX.5.0", "40559"): "40559 UnderlyingStreamFirstPeriodStartDateBusinessDayConvention",
        ("FIX.5.0", "40560"): "40560 UnderlyingStreamFirstPeriodStartDateBusinessCenter",
        ("FIX.5.0", "40561"): "40561 UnderlyingStreamFirstPeriodStartDateAdjusted",
        ("FIX.5.0", "40562"): "40562 UnderlyingStreamFirstRegularPeriodStartDateUnadjusted",
        ("FIX.5.0", "40563"): "40563 UnderlyingStreamFirstCompoundingPeriodEndDateUnadjusted",
        ("FIX.5.0", "40564"): "40564 UnderlyingStreamLastRegularPeriodEndDateUnadjusted",
        ("FIX.5.0", "40565"): "40565 UnderlyingStreamCalculationFrequencyPeriod",
        ("FIX.5.0", "40566"): "40566 UnderlyingStreamCalculationFrequencyUnit",
        ("FIX.5.0", "40567"): "40567 UnderlyingStreamCalculationRollConvention",
        ("FIX.5.0", "40568"): "40568 UnderlyingPaymentStreamType",
        ("FIX.5.0", "40569"): "40569 UnderlyingPaymentStreamMarketRate",
        ("FIX.5.0", "40570"): "40570 UnderlyingPaymentStreamDelayIndicator",
        ("FIX.5.0", "40571"): "40571 UnderlyingPaymentStreamSettlCurrency",
        ("FIX.5.0", "40572"): "40572 UnderlyingPaymentStreamDayCount",
        ("FIX.5.0", "40573"): "40573 UnderlyingPaymentStreamAccrualDays",
        ("FIX.5.0", "40574"): "40574 UnderlyingPaymentStreamDiscountType",
        ("FIX.5.0", "40575"): "40575 UnderlyingPaymentStreamDiscountRate",
        ("FIX.5.0", "40576"): "40576 UnderlyingPaymentStreamDiscountRateDayCount",
        ("FIX.5.0", "40577"): "40577 UnderlyingPaymentStreamCompoundingMethod",
        ("FIX.5.0", "40578"): "40578 UnderlyingPaymentStreamInitialPrincipalExchangeIndicator",
        ("FIX.5.0", "40579"): "40579 UnderlyingPaymentStreamInterimPrincipalExchangeIndicator",
        ("FIX.5.0", "40580"): "40580 UnderlyingPaymentStreamFinalPrincipalExchangeIndicator",
        ("FIX.5.0", "40581"): "40581 UnderlyingPaymentStreamPaymentDateBusinessDayConvention",
        ("FIX.5.0", "40582"): "40582 UnderlyingPaymentStreamPaymentDateBusinessCenter",
        ("FIX.5.0", "40583"): "40583 UnderlyingPaymentStreamPaymentFrequencyPeriod",
        ("FIX.5.0", "40584"): "40584 UnderlyingPaymentStreamPaymentFrequencyUnit",
        ("FIX.5.0", "40585"): "40585 UnderlyingPaymentStreamPaymentRollConvention",
        ("FIX.5.0", "40586"): "40586 UnderlyingPaymentStreamFirstPaymentDateUnadjusted",
        ("FIX.5.0", "40587"): "40587 UnderlyingPaymentStreamLastRegularPaymentDateUnadjusted",
        ("FIX.5.0", "40588"): "40588 UnderlyingPaymentStreamPaymentDateRelativeTo",
        ("FIX.5.0", "40589"): "40589 UnderlyingPaymentStreamPaymentDateOffsetPeriod",
        ("FIX.5.0", "40590"): "40590 UnderlyingPaymentStreamPaymentDateOffsetUnit",
        ("FIX.5.0", "40591"): "40591 UnderlyingPaymentStreamPaymentDateOffsetDayType",
        ("FIX.5.0", "40592"): "40592 UnderlyingPaymentStreamResetDateRelativeTo",
        ("FIX.5.0", "40593"): "40593 UnderlyingPaymentStreamResetDateBusinessDayConvention",
        ("FIX.5.0", "40594"): "40594 UnderlyingPaymentStreamResetDateBusinessCenter",
        ("FIX.5.0", "40595"): "40595 UnderlyingPaymentStreamResetFrequencyPeriod",
        ("FIX.5.0", "40596"): "40596 UnderlyingPaymentStreamResetFrequencyUnit",
        ("FIX.5.0", "40597"): "40597 UnderlyingPaymentStreamResetWeeklyRollConvention",
        ("FIX.5.0", "40598"): "40598 UnderlyingPaymentStreamInitialFixingDateRelativeTo",
        ("FIX.5.0", "40599"): "40599 UnderlyingPaymentStreamInitialFixingDateBusinessDayConvention",
        ("FIX.5.0", "40600"): "40600 UnderlyingPaymentStreamInitialFixingDateBusinessCenter",
        ("FIX.5.0", "40601"): "40601 UnderlyingPaymentStreamInitialFixingDateOffsetPeriod",
        ("FIX.5.0", "40602"): "40602 UnderlyingPaymentStreamInitialFixingDateOffsetUnit",
        ("FIX.5.0", "40603"): "40603 UnderlyingPaymentStreamInitialFixingDateOffsetDayType",
        ("FIX.5.0", "40604"): "40604 UnderlyingPaymentStreamInitialFixingDateAdjusted",
        ("FIX.5.0", "40605"): "40605 UnderlyingPaymentStreamFixingDateRelativeTo",
        ("FIX.5.0", "40606"): "40606 UnderlyingPaymentStreamFixingDateBusinessDayConvention",
        ("FIX.5.0", "40607"): "40607 UnderlyingPaymentStreamFixingDateBusinessCenter",
        ("FIX.5.0", "40608"): "40608 UnderlyingPaymentStreamFixingDateOffsetPeriod",
        ("FIX.5.0", "40609"): "40609 UnderlyingPaymentStreamFixingDateOffsetUnit",
        ("FIX.5.0", "40610"): "40610 UnderlyingPaymentStreamFixingDateOffsetDayType",
        ("FIX.5.0", "40611"): "40611 UnderlyingPaymentStreamFixingDateAdjusted",
        ("FIX.5.0", "40612"): "40612 UnderlyingPaymentStreamRateCutoffDateOffsetPeriod",
        ("FIX.5.0", "40613"): "40613 UnderlyingPaymentStreamRateCutoffDateOffsetUnit",
        ("FIX.5.0", "40614"): "40614 UnderlyingPaymentStreamRateCutoffDateOffsetDayType",
        ("FIX.5.0", "40615"): "40615 UnderlyingPaymentStreamRate",
        ("FIX.5.0", "40616"): "40616 UnderlyingPaymentStreamFixedAmount",
        ("FIX.5.0", "40617"): "40617 UnderlyingPaymentStreamRateOrAmountCurrency",
        ("FIX.5.0", "40618"): "40618 UnderlyingPaymentStreamFutureValueNotional",
        ("FIX.5.0", "40619"): "40619 UnderlyingPaymentStreamFutureValueDateAdjusted",
        ("FIX.5.0", "40620"): "40620 UnderlyingPaymentStreamRateIndex",
        ("FIX.5.0", "40621"): "40621 UnderlyingPaymentStreamRateIndexSource",
        ("FIX.5.0", "40622"): "40622 UnderlyingPaymentStreamRateIndexCurveUnit",
        ("FIX.5.0", "40623"): "40623 UnderlyingPaymentStreamRateIndexCurvePeriod",
        ("FIX.5.0", "40624"): "40624 UnderlyingPaymentStreamRateMultiplier",
        ("FIX.5.0", "40625"): "40625 UnderlyingPaymentStreamRateSpread",
        ("FIX.5.0", "40626"): "40626 UnderlyingPaymentStreamRateSpreadPositionType",
        ("FIX.5.0", "40627"): "40627 UnderlyingPaymentStreamRateTreatment",
        ("FIX.5.0", "40628"): "40628 UnderlyingPaymentStreamCapRate",
        ("FIX.5.0", "40629"): "40629 UnderlyingPaymentStreamCapRateBuySide",
        ("FIX.5.0", "40630"): "40630 UnderlyingPaymentStreamCapRateSellSide",
        ("FIX.5.0", "40631"): "40631 UnderlyingPaymentStreamFloorRate",
        ("FIX.5.0", "40632"): "40632 UnderlyingPaymentStreamFloorRateBuySide",
        ("FIX.5.0", "40633"): "40633 UnderlyingPaymentStreamFloorRateSellSide",
        ("FIX.5.0", "40634"): "40634 UnderlyingPaymentStreamInitialRate",
        ("FIX.5.0", "40635"): "40635 UnderlyingPaymentStreamFinalRateRoundingDirection",
        ("FIX.5.0", "40636"): "40636 UnderlyingPaymentStreamFinalRatePrecision",
        ("FIX.5.0", "40637"): "40637 UnderlyingPaymentStreamAveragingMethod",
        ("FIX.5.0", "40638"): "40638 UnderlyingPaymentStreamNegativeRateTreatment",
        ("FIX.5.0", "40639"): "40639 UnderlyingPaymentStreamInflationLagPeriod",
        ("FIX.5.0", "40640"): "40640 UnderlyingPaymentStreamInflationLagUnit",
        ("FIX.5.0", "40641"): "40641 UnderlyingPaymentStreamInflationLagDayType",
        ("FIX.5.0", "40642"): "40642 UnderlyingPaymentStreamInflationInterpolationMethod",
        ("FIX.5.0", "40643"): "40643 UnderlyingPaymentStreamInflationIndexSource",
        ("FIX.5.0", "40644"): "40644 UnderlyingPaymentStreamInflationPublicationSource",
        ("FIX.5.0", "40645"): "40645 UnderlyingPaymentStreamInflationInitialIndexLevel",
        ("FIX.5.0", "40646"): "40646 UnderlyingPaymentStreamInflationFallbackBondApplicable",
        ("FIX.5.0", "40647"): "40647 UnderlyingPaymentStreamFRADiscounting",
        ("FIX.5.0", "40648"): "40648 UnderlyingPaymentStreamNonDeliverableRefCurrency",
        ("FIX.5.0", "40649"): "40649 UnderlyingPaymentStreamNonDeliverableFixingDatesBusinessDayConvention",
        ("FIX.5.0", "40650"): "40650 UnderlyingPaymentStreamNonDeliverableFixingDatesBusinessCenter",
        ("FIX.5.0", "40651"): "40651 UnderlyingPaymentStreamNonDeliverableFixingDatesRelativeTo",
        ("FIX.5.0", "40652"): "40652 UnderlyingPaymentStreamNonDeliverableFixingDatesOffsetPeriod",
        ("FIX.5.0", "40653"): "40653 UnderlyingPaymentStreamNonDeliverableFixingDatesOffsetUnit",
        ("FIX.5.0", "40654"): "40654 UnderlyingPaymentStreamNonDeliverableFixingDatesOffsetDayType",
        ("FIX.5.0", "40655"): "40655 SettlRateFallbackReferencePage",
        ("FIX.5.0", "40656"): "40656 NoUnderlyingNonDeliverableFixingDates",
        ("FIX.5.0", "40657"): "40657 UnderlyingNonDeliverableFixingDate",
        ("FIX.5.0", "40658"): "40658 UnderlyingNonDeliverableFixingDateType",
        ("FIX.5.0", "40659"): "40659 NoUnderlyingSettlRateFallbacks",
        ("FIX.5.0", "40660"): "40660 UnderlyingSettlRatePostponementMaximumDays",
        ("FIX.5.0", "40661"): "40661 UnderlyingPaymentStreamNonDeliverableSettlRateSource",
        ("FIX.5.0", "40662"): "40662 UnderlyingSettlRatePostponementSurvey",
        ("FIX.5.0", "40663"): "40663 UnderlyingSettlRatePostponementCalculationAgent",
        ("FIX.5.0", "40664"): "40664 NoUnderlyingPaymentSchedules",
        ("FIX.5.0", "40665"): "40665 UnderlyingPaymentScheduleType",
        ("FIX.5.0", "40666"): "40666 UnderlyingPaymentScheduleStubType",
        ("FIX.5.0", "40667"): "40667 UnderlyingPaymentScheduleStartDateUnadjusted",
        ("FIX.5.0", "40668"): "40668 UnderlyingPaymentScheduleEndDateUnadjusted",
        ("FIX.5.0", "40669"): "40669 UnderlyingPaymentSchedulePaySide",
        ("FIX.5.0", "40670"): "40670 UnderlyingPaymentScheduleReceiveSide",
        ("FIX.5.0", "40671"): "40671 UnderlyingPaymentScheduleNotional",
        ("FIX.5.0", "40672"): "40672 UnderlyingPaymentScheduleCurrency",
        ("FIX.5.0", "40673"): "40673 UnderlyingPaymentScheduleRate",
        ("FIX.5.0", "40674"): "40674 UnderlyingPaymentScheduleRateMultiplier",
        ("FIX.5.0", "40675"): "40675 UnderlyingPaymentScheduleRateSpread",
        ("FIX.5.0", "40676"): "40676 UnderlyingPaymentScheduleRateSpreadPositionType",
        ("FIX.5.0", "40677"): "40677 UnderlyingPaymentScheduleRateTreatment",
        ("FIX.5.0", "40678"): "40678 UnderlyingPaymentScheduleFixedAmount",
        ("FIX.5.0", "40679"): "40679 UnderlyingPaymentScheduleFixedCurrency",
        ("FIX.5.0", "40680"): "40680 UnderlyingPaymentScheduleStepFrequencyPeriod",
        ("FIX.5.0", "40681"): "40681 UnderlyingPaymentScheduleStepFrequencyUnit",
        ("FIX.5.0", "40682"): "40682 UnderlyingPaymentScheduleStepOffsetValue",
        ("FIX.5.0", "40683"): "40683 UnderlyingPaymentScheduleStepRate",
        ("FIX.5.0", "40684"): "40684 UnderlyingPaymentScheduleStepOffsetRate",
        ("FIX.5.0", "40685"): "40685 UnderlyingPaymentScheduleStepRelativeTo",
        ("FIX.5.0", "40686"): "40686 UnderlyingPaymentScheduleFixingDateUnadjusted",
        ("FIX.5.0", "40687"): "40687 UnderlyingPaymentScheduleWeight",
        ("FIX.5.0", "40688"): "40688 UnderlyingPaymentScheduleFixingDateRelativeTo",
        ("FIX.5.0", "40689"): "40689 UnderlyingPaymentScheduleFixingDateBusinessDayCnvtn",
        ("FIX.5.0", "40690"): "40690 UnderlyingPaymentScheduleFixingDateBusinessCenter",
        ("FIX.5.0", "40691"): "40691 UnderlyingPaymentScheduleFixingDateOffsetPeriod",
        ("FIX.5.0", "40692"): "40692 UnderlyingPaymentScheduleFixingDateOffsetUnit",
        ("FIX.5.0", "40693"): "40693 UnderlyingPaymentScheduleFixingDateOffsetDayType",
        ("FIX.5.0", "40694"): "40694 UnderlyingPaymentScheduleFixingDateAdjusted",
        ("FIX.5.0", "40695"): "40695 UnderlyingPaymentScheduleFixingTime",
        ("FIX.5.0", "40696"): "40696 UnderlyingPaymentScheduleFixingTimeBusinessCenter",
        ("FIX.5.0", "40697"): "40697 UnderlyingPaymentScheduleInterimExchangePaymentDateRelativeTo",
        ("FIX.5.0", "40698"): "40698 UnderlyingPaymentScheduleInterimExchangeDatesBusinessDayConvention",
        ("FIX.5.0", "40699"): "40699 UnderlyingPaymentScheduleInterimExchangeDatesBusinessCenter",
        ("FIX.5.0", "40700"): "40700 UnderlyingPaymentScheduleInterimExchangeDatesOffsetPeriod",
        ("FIX.5.0", "40701"): "40701 UnderlyingPaymentScheduleInterimExchangeDatesOffsetUnit",
        ("FIX.5.0", "40702"): "40702 UnderlyingPaymentScheduleInterimExchangeDatesOffsetDayType",
        ("FIX.5.0", "40703"): "40703 UnderlyingPaymentScheduleInterimExchangeDateAdjusted",
        ("FIX.5.0", "40704"): "40704 NoUnderlyingPaymentScheduleRateSources",
        ("FIX.5.0", "40705"): "40705 UnderlyingPaymentScheduleRateSource",
        ("FIX.5.0", "40706"): "40706 UnderlyingPaymentScheduleRateSourceType",
        ("FIX.5.0", "40707"): "40707 UnderlyingPaymentScheduleReferencePage",
        ("FIX.5.0", "40708"): "40708 NoUnderlyingPaymentStubs",
        ("FIX.5.0", "40709"): "40709 UnderlyingPaymentStubType",
        ("FIX.5.0", "40710"): "40710 UnderlyingPaymentStubLength",
        ("FIX.5.0", "40711"): "40711 UnderlyingPaymentStubRate",
        ("FIX.5.0", "40712"): "40712 UnderlyingPaymentStubFixedAmount",
        ("FIX.5.0", "40713"): "40713 UnderlyingPaymentStubFixedCurrency",
        ("FIX.5.0", "40714"): "40714 UnderlyingPaymentStubIndex",
        ("FIX.5.0", "40715"): "40715 UnderlyingPaymentStubIndexSource",
        ("FIX.5.0", "40716"): "40716 UnderlyingPaymentStubIndexCurvePeriod",
        ("FIX.5.0", "40717"): "40717 UnderlyingPaymentStubIndexCurveUnit",
        ("FIX.5.0", "40718"): "40718 UnderlyingPaymentStubIndexRateMultiplier",
        ("FIX.5.0", "40719"): "40719 UnderlyingPaymentStubIndexRateSpread",
        ("FIX.5.0", "40720"): "40720 UnderlyingPaymentStubIndexRateSpreadPositionType",
        ("FIX.5.0", "40721"): "40721 UnderlyingPaymentStubIndexRateTreatment",
        ("FIX.5.0", "40722"): "40722 UnderlyingPaymentStubIndexCapRate",
        ("FIX.5.0", "40723"): "40723 UnderlyingPaymentStubIndexCapRateBuySide",
        ("FIX.5.0", "40724"): "40724 UnderlyingPaymentStubIndexCapRateSellSide",
        ("FIX.5.0", "40725"): "40725 UnderlyingPaymentStubIndexFloorRate",
        ("FIX.5.0", "40726"): "40726 UnderlyingPaymentStubIndexFloorRateBuySide",
        ("FIX.5.0", "40727"): "40727 UnderlyingPaymentStubIndexFloorRateSellSide",
        ("FIX.5.0", "40728"): "40728 UnderlyingPaymentStubIndex2",
        ("FIX.5.0", "40729"): "40729 UnderlyingPaymentStubIndex2Source",
        ("FIX.5.0", "40730"): "40730 UnderlyingPaymentStubIndex2CurvePeriod",
        ("FIX.5.0", "40731"): "40731 UnderlyingPaymentStubIndex2CurveUnit",
        ("FIX.5.0", "40732"): "40732 UnderlyingPaymentStubIndex2RateMultiplier",
        ("FIX.5.0", "40733"): "40733 UnderlyingPaymentStubIndex2RateSpread",
        ("FIX.5.0", "40734"): "40734 UnderlyingPaymentStubIndex2RateSpreadPositionType",
        ("FIX.5.0", "40735"): "40735 UnderlyingPaymentStubIndex2RateTreatment",
        ("FIX.5.0", "40736"): "40736 UnderlyingPaymentStubIndex2CapRate",
        ("FIX.5.0", "40737"): "40737 UnderlyingPaymentStubIndex2FloorRate",
        ("FIX.5.0", "40738"): "40738 PaymentStreamType",
        ("FIX.5.0", "40739"): "40739 PaymentStreamMarketRate",
        ("FIX.5.0", "40740"): "40740 PaymentStreamDelayIndicator",
        ("FIX.5.0", "40741"): "40741 PaymentStreamSettlCurrency",
        ("FIX.5.0", "40742"): "40742 PaymentStreamDayCount",
        ("FIX.5.0", "40743"): "40743 PaymentStreamAccrualDays",
        ("FIX.5.0", "40744"): "40744 PaymentStreamDiscountType",
        ("FIX.5.0", "40745"): "40745 PaymentStreamDiscountRate",
        ("FIX.5.0", "40746"): "40746 PaymentStreamDiscountRateDayCount",
        ("FIX.5.0", "40747"): "40747 PaymentStreamCompoundingMethod",
        ("FIX.5.0", "40748"): "40748 PaymentStreamInitialPrincipalExchangeIndicator",
        ("FIX.5.0", "40749"): "40749 PaymentStreamInterimPrincipalExchangeIndicator",
        ("FIX.5.0", "40750"): "40750 PaymentStreamFinalPrincipalExchangeIndicator",
        ("FIX.5.0", "40751"): "40751 PaymentStreamPaymentDateBusinessDayConvention",
        ("FIX.5.0", "40752"): "40752 PaymentStreamPaymentDateBusinessCenter",
        ("FIX.5.0", "40753"): "40753 PaymentStreamPaymentFrequencyPeriod",
        ("FIX.5.0", "40754"): "40754 PaymentStreamPaymentFrequencyUnit",
        ("FIX.5.0", "40755"): "40755 PaymentStreamPaymentRollConvention",
        ("FIX.5.0", "40756"): "40756 PaymentStreamFirstPaymentDateUnadjusted",
        ("FIX.5.0", "40757"): "40757 PaymentStreamLastRegularPaymentDateUnadjusted",
        ("FIX.5.0", "40758"): "40758 PaymentStreamPaymentDateRelativeTo",
        ("FIX.5.0", "40759"): "40759 PaymentStreamPaymentDateOffsetPeriod",
        ("FIX.5.0", "40760"): "40760 PaymentStreamPaymentDateOffsetUnit",
        ("FIX.5.0", "40761"): "40761 PaymentStreamResetDateRelativeTo",
        ("FIX.5.0", "40762"): "40762 PaymentStreamResetDateBusinessDayConvention",
        ("FIX.5.0", "40763"): "40763 PaymentStreamResetDateBusinessCenter",
        ("FIX.5.0", "40764"): "40764 PaymentStreamResetFrequencyPeriod",
        ("FIX.5.0", "40765"): "40765 PaymentStreamResetFrequencyUnit",
        ("FIX.5.0", "40766"): "40766 PaymentStreamResetWeeklyRollConvention",
        ("FIX.5.0", "40767"): "40767 PaymentStreamInitialFixingDateRelativeTo",
        ("FIX.5.0", "40768"): "40768 PaymentStreamInitialFixingDateBusinessDayConvention",
        ("FIX.5.0", "40769"): "40769 PaymentStreamInitialFixingDateBusinessCenter",
        ("FIX.5.0", "40770"): "40770 PaymentStreamInitialFixingDateOffsetPeriod",
        ("FIX.5.0", "40771"): "40771 PaymentStreamInitialFixingDateOffsetUnit",
        ("FIX.5.0", "40772"): "40772 PaymentStreamInitialFixingDateOffsetDayType",
        ("FIX.5.0", "40773"): "40773 PaymentStreamInitialFixingDateAdjusted",
        ("FIX.5.0", "40774"): "40774 PaymentStreamFixingDateRelativeTo",
        ("FIX.5.0", "40775"): "40775 PaymentStreamFixingDateBusinessDayConvention",
        ("FIX.5.0", "40776"): "40776 PaymentStreamFixingDateBusinessCenter",
        ("FIX.5.0", "40777"): "40777 PaymentStreamFixingDateOffsetPeriod",
        ("FIX.5.0", "40778"): "40778 PaymentStreamFixingDateOffsetUnit",
        ("FIX.5.0", "40779"): "40779 PaymentStreamFixingDateOffsetDayType",
        ("FIX.5.0", "40780"): "40780 PaymentStreamFixingDateAdjusted",
        ("FIX.5.0", "40781"): "40781 PaymentStreamRateCutoffDateOffsetPeriod",
        ("FIX.5.0", "40782"): "40782 PaymentStreamRateCutoffDateOffsetUnit",
        ("FIX.5.0", "40783"): "40783 PaymentStreamRateCutoffDateOffsetDayType",
        ("FIX.5.0", "40784"): "40784 PaymentStreamRate",
        ("FIX.5.0", "40785"): "40785 PaymentStreamFixedAmount",
        ("FIX.5.0", "40786"): "40786 PaymentStreamRateOrAmountCurrency",
        ("FIX.5.0", "40787"): "40787 PaymentStreamFutureValueNotional",
        ("FIX.5.0", "40788"): "40788 PaymentStreamFutureValueDateAdjusted",
        ("FIX.5.0", "40789"): "40789 PaymentStreamRateIndex",
        ("FIX.5.0", "40790"): "40790 PaymentStreamRateIndexSource",
        ("FIX.5.0", "40791"): "40791 PaymentStreamRateIndexCurveUnit",
        ("FIX.5.0", "40792"): "40792 PaymentStreamRateIndexCurvePeriod",
        ("FIX.5.0", "40793"): "40793 PaymentStreamRateMultiplier",
        ("FIX.5.0", "40794"): "40794 PaymentStreamRateSpread",
        ("FIX.5.0", "40795"): "40795 PaymentStreamRateSpreadPositionType",
        ("FIX.5.0", "40796"): "40796 PaymentStreamRateTreatment",
        ("FIX.5.0", "40797"): "40797 PaymentStreamCapRate",
        ("FIX.5.0", "40798"): "40798 PaymentStreamCapRateBuySide",
        ("FIX.5.0", "40799"): "40799 PaymentStreamCapRateSellSide",
        ("FIX.5.0", "40800"): "40800 PaymentStreamFloorRate",
        ("FIX.5.0", "40801"): "40801 PaymentStreamFloorRateBuySide",
        ("FIX.5.0", "40802"): "40802 PaymentStreamFloorRateSellSide",
        ("FIX.5.0", "40803"): "40803 PaymentStreamInitialRate",
        ("FIX.5.0", "40804"): "40804 PaymentStreamFinalRateRoundingDirection",
        ("FIX.5.0", "40805"): "40805 PaymentStreamFinalRatePrecision",
        ("FIX.5.0", "40806"): "40806 PaymentStreamAveragingMethod",
        ("FIX.5.0", "40807"): "40807 PaymentStreamNegativeRateTreatment",
        ("FIX.5.0", "40808"): "40808 PaymentStreamInflationLagPeriod",
        ("FIX.5.0", "40809"): "40809 PaymentStreamInflationLagUnit",
        ("FIX.5.0", "40810"): "40810 PaymentStreamInflationLagDayType",
        ("FIX.5.0", "40811"): "40811 PaymentStreamInflationInterpolationMethod",
        ("FIX.5.0", "40812"): "40812 PaymentStreamInflationIndexSource",
        ("FIX.5.0", "40813"): "40813 PaymentStreamInflationPublicationSource",
        ("FIX.5.0", "40814"): "40814 PaymentStreamInflationInitialIndexLevel",
        ("FIX.5.0", "40815"): "40815 PaymentStreamInflationFallbackBondApplicable",
        ("FIX.5.0", "40816"): "40816 PaymentStreamFRADiscounting",
        ("FIX.5.0", "40817"): "40817 PaymentStreamNonDeliverableRefCurrency",
        ("FIX.5.0", "40818"): "40818 PaymentStreamNonDeliverableFixingDatesBusinessDayConvention",
        ("FIX.5.0", "40819"): "40819 PaymentStreamNonDeliverableFixingDatesBusinessCenter",
        ("FIX.5.0", "40820"): "40820 PaymentStreamNonDeliverableFixingDatesRelativeTo",
        ("FIX.5.0", "40821"): "40821 PaymentStreamNonDeliverableFixingDatesOffsetPeriod",
        ("FIX.5.0", "40822"): "40822 PaymentStreamNonDeliverableFixingDatesOffsetUnit",
        ("FIX.5.0", "40823"): "40823 PaymentStreamNonDeliverableFixingDatesOffsetDayType",
        ("FIX.5.0", "40824"): "40824 UnderlyingPaymentStreamNonDeliverableSettlReferencePage",
        ("FIX.5.0", "40825"): "40825 NoNonDeliverableFixingDates",
        ("FIX.5.0", "40826"): "40826 NonDeliverableFixingDate",
        ("FIX.5.0", "40827"): "40827 NonDeliverableFixingDateType",
        ("FIX.5.0", "40828"): "40828 NoPaymentSchedules",
        ("FIX.5.0", "40829"): "40829 PaymentScheduleType",
        ("FIX.5.0", "40830"): "40830 PaymentScheduleStubType",
        ("FIX.5.0", "40831"): "40831 PaymentScheduleStartDateUnadjusted",
        ("FIX.5.0", "40832"): "40832 PaymentScheduleEndDateUnadjusted",
        ("FIX.5.0", "40833"): "40833 PaymentSchedulePaySide",
        ("FIX.5.0", "40834"): "40834 PaymentScheduleReceiveSide",
        ("FIX.5.0", "40835"): "40835 PaymentScheduleNotional",
        ("FIX.5.0", "40836"): "40836 PaymentScheduleCurrency",
        ("FIX.5.0", "40837"): "40837 PaymentScheduleRate",
        ("FIX.5.0", "40838"): "40838 PaymentScheduleRateMultiplier",
        ("FIX.5.0", "40839"): "40839 PaymentScheduleRateSpread",
        ("FIX.5.0", "40840"): "40840 PaymentScheduleRateSpreadPositionType",
        ("FIX.5.0", "40841"): "40841 PaymentScheduleRateTreatment",
        ("FIX.5.0", "40842"): "40842 PaymentScheduleFixedAmount",
        ("FIX.5.0", "40843"): "40843 PaymentScheduleFixedCurrency",
        ("FIX.5.0", "40844"): "40844 PaymentScheduleStepFrequencyPeriod",
        ("FIX.5.0", "40845"): "40845 PaymentScheduleStepFrequencyUnit",
        ("FIX.5.0", "40846"): "40846 PaymentScheduleStepOffsetValue",
        ("FIX.5.0", "40847"): "40847 PaymentScheduleStepRate",
        ("FIX.5.0", "40848"): "40848 PaymentScheduleStepOffsetRate",
        ("FIX.5.0", "40849"): "40849 PaymentScheduleStepRelativeTo",
        ("FIX.5.0", "40850"): "40850 PaymentScheduleFixingDateUnadjusted",
        ("FIX.5.0", "40851"): "40851 PaymentScheduleWeight",
        ("FIX.5.0", "40852"): "40852 PaymentScheduleFixingDateRelativeTo",
        ("FIX.5.0", "40853"): "40853 PaymentScheduleFixingDateBusinessDayConvention",
        ("FIX.5.0", "40854"): "40854 PaymentScheduleFixingDateBusinessCenter",
        ("FIX.5.0", "40855"): "40855 PaymentScheduleFixingDateOffsetPeriod",
        ("FIX.5.0", "40856"): "40856 PaymentScheduleFixingDateOffsetUnit",
        ("FIX.5.0", "40857"): "40857 PaymentScheduleFixingDateOffsetDayType",
        ("FIX.5.0", "40858"): "40858 PaymentScheduleFixingDateAdjusted",
        ("FIX.5.0", "40859"): "40859 PaymentScheduleFixingTime",
        ("FIX.5.0", "40860"): "40860 PaymentScheduleFixingTimeBusinessCenter",
        ("FIX.5.0", "40861"): "40861 PaymentScheduleInterimExchangePaymentDateRelativeTo",
        ("FIX.5.0", "40862"): "40862 PaymentScheduleInterimExchangeDatesBusinessDayConvention",
        ("FIX.5.0", "40863"): "40863 PaymentScheduleInterimExchangeDatesBusinessCenter",
        ("FIX.5.0", "40864"): "40864 PaymentScheduleInterimExchangeDatesOffsetPeriod",
        ("FIX.5.0", "40865"): "40865 PaymentScheduleInterimExchangeDatesOffsetUnit",
        ("FIX.5.0", "40866"): "40866 PaymentScheduleInterimExchangeDatesOffsetDayType",
        ("FIX.5.0", "40867"): "40867 PaymentScheduleInterimExchangeDateAdjusted",
        ("FIX.5.0", "40868"): "40868 NoPaymentScheduleRateSources",
        ("FIX.5.0", "40869"): "40869 PaymentScheduleRateSource",
        ("FIX.5.0", "40870"): "40870 PaymentScheduleRateSourceType",
        ("FIX.5.0", "40871"): "40871 PaymentScheduleReferencePage",
        ("FIX.5.0", "40872"): "40872 NoPaymentStubs",
        ("FIX.5.0", "40873"): "40873 PaymentStubType",
        ("FIX.5.0", "40874"): "40874 PaymentStubLength",
        ("FIX.5.0", "40875"): "40875 PaymentStubRate",
        ("FIX.5.0", "40876"): "40876 PaymentStubFixedAmount",
        ("FIX.5.0", "40877"): "40877 PaymentStubFixedCurrency",
        ("FIX.5.0", "40878"): "40878 PaymentStubIndex",
        ("FIX.5.0", "40879"): "40879 PaymentStubIndexSource",
        ("FIX.5.0", "40880"): "40880 PaymentStubIndexCurvePeriod",
        ("FIX.5.0", "40881"): "40881 PaymentStubIndexCurveUnit",
        ("FIX.5.0", "40882"): "40882 PaymentStubIndexRateMultiplier",
        ("FIX.5.0", "40883"): "40883 PaymentStubIndexRateSpread",
        ("FIX.5.0", "40884"): "40884 PaymentStubIndexRateSpreadPositionType",
        ("FIX.5.0", "40885"): "40885 PaymentStubIndexRateTreatment",
        ("FIX.5.0", "40886"): "40886 PaymentStubIndexCapRate",
        ("FIX.5.0", "40887"): "40887 PaymentStubIndexCapRateBuySide",
        ("FIX.5.0", "40888"): "40888 PaymentStubIndexCapRateSellSide",
        ("FIX.5.0", "40889"): "40889 PaymentStubIndexFloorRate",
        ("FIX.5.0", "40890"): "40890 PaymentStubIndexFloorRateBuySide",
        ("FIX.5.0", "40891"): "40891 PaymentStubIndexFloorRateSellSide",
        ("FIX.5.0", "40892"): "40892 PaymentStubIndex2",
        ("FIX.5.0", "40893"): "40893 PaymentStubIndex2Source",
        ("FIX.5.0", "40894"): "40894 PaymentStubIndex2CurvePeriod",
        ("FIX.5.0", "40895"): "40895 PaymentStubIndex2CurveUnit",
        ("FIX.5.0", "40896"): "40896 PaymentStubIndex2RateMultiplier",
        ("FIX.5.0", "40897"): "40897 PaymentStubIndex2RateSpread",
        ("FIX.5.0", "40898"): "40898 PaymentStubIndex2RateSpreadPositionType",
        ("FIX.5.0", "40899"): "40899 PaymentStubIndex2RateTreatment",
        ("FIX.5.0", "40900"): "40900 PaymentStubIndex2CapRate",
        ("FIX.5.0", "40901"): "40901 PaymentStubIndex2FloorRate",
        ("FIX.5.0", "40902"): "40902 NoLegSettlRateFallbacks",
        ("FIX.5.0", "40903"): "40903 LegSettlRatePostponementMaximumDays",
        ("FIX.5.0", "40904"): "40904 UnderlyingSettlRateFallbackRateSource",
        ("FIX.5.0", "40905"): "40905 LegSettlRatePostponementSurvey",
        ("FIX.5.0", "40906"): "40906 LegSettlRatePostponementCalculationAgent",
        ("FIX.5.0", "40907"): "40907 StreamEffectiveDateUnadjusted",
        ("FIX.5.0", "40908"): "40908 StreamEffectiveDateBusinessDayConvention",
        ("FIX.5.0", "40909"): "40909 StreamEffectiveDateBusinessCenter",
        ("FIX.5.0", "40910"): "40910 StreamEffectiveDateRelativeTo",
        ("FIX.5.0", "40911"): "40911 StreamEffectiveDateOffsetPeriod",
        ("FIX.5.0", "40912"): "40912 StreamEffectiveDateOffsetUnit",
        ("FIX.5.0", "40913"): "40913 StreamEffectiveDateOffsetDayType",
        ("FIX.5.0", "40914"): "40914 StreamEffectiveDateAdjusted",
        ("FIX.5.0", "40915"): "40915 UnderlyingSettlRateFallbackReferencePage",
        ("FIX.5.0", "40916"): "40916 CashSettlValuationSubsequentBusinessDaysOffset",
        ("FIX.5.0", "40917"): "40917 CashSettlNumOfValuationDates",
        ("FIX.5.0", "40918"): "40918 UnderlyingProvisionPartyRoleQualifier",
        ("FIX.5.0", "40919"): "40919 PaymentPriceType",
        ("FIX.5.0", "40920"): "40920 PaymentStreamPaymentDateOffsetDayType",
        ("FIX.5.0", "40921"): "40921 BusinessDayConvention",
        ("FIX.5.0", "40922"): "40922 DateRollConvention",
        ("FIX.5.0", "40923"): "40923 NoLegBusinessCenters",
        ("FIX.5.0", "40924"): "40924 LegBusinessCenter",
        ("FIX.5.0", "40925"): "40925 LegBusinessDayConvention",
        ("FIX.5.0", "40926"): "40926 LegDateRollConvention",
        ("FIX.5.0", "40927"): "40927 NoLegPaymentScheduleFixingDateBusinessCenters",
        ("FIX.5.0", "40928"): "40928 NoLegPaymentScheduleInterimExchangeDateBusinessCenters",
        ("FIX.5.0", "40929"): "40929 NoLegPaymentStreamNonDeliverableFixingDateBusinessCenters",
        ("FIX.5.0", "40930"): "40930 NoLegPaymentStreamPaymentDateBusinessCenters",
        ("FIX.5.0", "40931"): "40931 NoLegPaymentStreamResetDateBusinessCenters",
        ("FIX.5.0", "40932"): "40932 NoLegPaymentStreamInitialFixingDateBusinessCenters",
        ("FIX.5.0", "40933"): "40933 NoLegPaymentStreamFixingDateBusinessCenters",
        ("FIX.5.0", "40934"): "40934 NoLegProvisionCashSettlPaymentDateBusinessCenters",
        ("FIX.5.0", "40935"): "40935 NoLegProvisionCashSettlValueDateBusinessCenters",
        ("FIX.5.0", "40936"): "40936 NoLegProvisionOptionExerciseBusinessCenters",
        ("FIX.5.0", "40937"): "40937 NoLegProvisionOptionExpirationDateBusinessCenters",
        ("FIX.5.0", "40938"): "40938 NoLegProvisionOptionRelevantUnderlyingDateBusinessCenters",
        ("FIX.5.0", "40939"): "40939 NoLegProvisionDateBusinessCenters",
        ("FIX.5.0", "40940"): "40940 NoLegStreamCalculationPeriodBusinessCenters",
        ("FIX.5.0", "40941"): "40941 NoLegStreamFirstPeriodStartDateBusinessCenters",
        ("FIX.5.0", "40942"): "40942 NoLegStreamEffectiveDateBusinessCenters",
        ("FIX.5.0", "40943"): "40943 NoLegStreamTerminationDateBusinessCenters",
        ("FIX.5.0", "40944"): "40944 NoPaymentBusinessCenters",
        ("FIX.5.0", "40945"): "40945 NoPaymentScheduleInterimExchangeDateBusinessCenters",
        ("FIX.5.0", "40946"): "40946 NoPaymentStreamNonDeliverableFixingDatesBusinessCenters",
        ("FIX.5.0", "40947"): "40947 NoPaymentStreamPaymentDateBusinessCenters",
        ("FIX.5.0", "40948"): "40948 NoPaymentStreamResetDateBusinessCenters",
        ("FIX.5.0", "40949"): "40949 NoPaymentStreamInitialFixingDateBusinessCenters",
        ("FIX.5.0", "40950"): "40950 NoPaymentStreamFixingDateBusinessCenters",
        ("FIX.5.0", "40951"): "40951 NoProtectionTermEventNewsSources",
        ("FIX.5.0", "40952"): "40952 NoProvisionCashSettlPaymentDateBusinessCenters",
        ("FIX.5.0", "40953"): "40953 NoProvisionCashSettlValueDateBusinessCenters",
        ("FIX.5.0", "40954"): "40954 NoProvisionOptionExerciseBusinessCenters",
        ("FIX.5.0", "40955"): "40955 NoProvisionOptionExpirationDateBusinessCenters",
        ("FIX.5.0", "40956"): "40956 NoProvisionOptionRelevantUnderlyingDateBusinessCenters",
        ("FIX.5.0", "40957"): "40957 NoProvisionDateBusinessCenters",
        ("FIX.5.0", "40958"): "40958 NoStreamCalculationPeriodBusinessCenters",
        ("FIX.5.0", "40959"): "40959 NoStreamFirstPeriodStartDateBusinessCenters",
        ("FIX.5.0", "40960"): "40960 NoStreamEffectiveBusinessCenters",
        ("FIX.5.0", "40961"): "40961 NoStreamTerminationDateBusinessCenters",
        ("FIX.5.0", "40962"): "40962 NoUnderlyingBusinessCenters",
        ("FIX.5.0", "40963"): "40963 UnderlyingBusinessCenter",
        ("FIX.5.0", "40964"): "40964 UnderlyingBusinessDayConvention",
        ("FIX.5.0", "40965"): "40965 UnderlyingDateRollConvention",
        ("FIX.5.0", "40966"): "40966 NoUnderlyingPaymentScheduleFixingDateBusinessCenters",
        ("FIX.5.0", "40967"): "40967 NoUnderlyingPaymentScheduleInterimExchangeDateBusinessCenters",
        ("FIX.5.0", "40968"): "40968 NoUnderlyingPaymentStreamNonDeliverableFixingDatesBusinessCenters",
        ("FIX.5.0", "40969"): "40969 NoUnderlyingPaymentStreamPaymentDateBusinessCenters",
        ("FIX.5.0", "40970"): "40970 NoUnderlyingPaymentStreamResetDateBusinessCenters",
        ("FIX.5.0", "40971"): "40971 NoUnderlyingPaymentStreamInitialFixingDateBusinessCenters",
        ("FIX.5.0", "40972"): "40972 NoUnderlyingPaymentStreamFixingDateBusinessCenters",
        ("FIX.5.0", "40973"): "40973 NoUnderlyingStreamCalculationPeriodBusinessCenters",
        ("FIX.5.0", "40974"): "40974 NoUnderlyingStreamFirstPeriodStartDateBusinessCenters",
        ("FIX.5.0", "40975"): "40975 NoUnderlyingStreamEffectiveDateBusinessCenters",
        ("FIX.5.0", "40976"): "40976 NoUnderlyingStreamTerminationDateBusinessCenters",
        ("FIX.5.0", "40977"): "40977 NoPaymentScheduleFixingDateBusinessCenters",
        ("FIX.5.0", "40978"): "40978 EncodedLegStreamTextLen",
        ("FIX.5.0", "40979"): "40979 EncodedLegStreamText",
        ("FIX.5.0", "40980"): "40980 EncodedLegProvisionTextLen",
        ("FIX.5.0", "40981"): "40981 EncodedLegProvisionText",
        ("FIX.5.0", "40982"): "40982 EncodedStreamTextLen",
        ("FIX.5.0", "40983"): "40983 EncodedStreamText",
        ("FIX.5.0", "40984"): "40984 EncodedPaymentTextLen",
        ("FIX.5.0", "40985"): "40985 EncodedPaymentText",
        ("FIX.5.0", "40986"): "40986 EncodedProvisionTextLen",
        ("FIX.5.0", "40987"): "40987 EncodedProvisionText",
        ("FIX.5.0", "40988"): "40988 EncodedUnderlyingStreamTextLen",
        ("FIX.5.0", "40989"): "40989 EncodedUnderlyingStreamText",
        ("FIX.5.0", "40990"): "40990 LegMarketDisruptionFallbackValue",
        ("FIX.5.0", "40991"): "40991 MarketDisruptionValue",
        ("FIX.5.0", "40992"): "40992 MarketDisruptionFallbackValue",
        ("FIX.5.0", "40993"): "40993 PaymentSubType",
        ("FIX.5.0", "40994"): "40994 NoComplexEventAveragingObservations",
        ("FIX.5.0", "40995"): "40995 ComplexEventAveragingObservationNumber",
        ("FIX.5.0", "40996"): "40996 ComplexEventAveragingWeight",
        ("FIX.5.0", "40997"): "40997 NoComplexEventCreditEvents",
        ("FIX.5.0", "40998"): "40998 ComplexEventCreditEventType",
        ("FIX.5.0", "40999"): "40999 ComplexEventCreditEventValue",
        ("FIX.5.0", "41000"): "41000 ComplexEventCreditEventCurrency",
        ("FIX.5.0", "41001"): "41001 ComplexEventCreditEventPeriod",
        ("FIX.5.0", "41002"): "41002 ComplexEventCreditEventUnit",
        ("FIX.5.0", "41003"): "41003 ComplexEventCreditEventDayType",
        ("FIX.5.0", "41004"): "41004 ComplexEventCreditEventRateSource",
        ("FIX.5.0", "41005"): "41005 NoComplexEventCreditEventQualifiers",
        ("FIX.5.0", "41006"): "41006 ComplexEventCreditEventQualifier",
        ("FIX.5.0", "41007"): "41007 NoComplexEventPeriodDateTimes",
        ("FIX.5.0", "41008"): "41008 ComplexEventPeriodDate",
        ("FIX.5.0", "41009"): "41009 ComplexEventPeriodTime",
        ("FIX.5.0", "41010"): "41010 NoComplexEventPeriods",
        ("FIX.5.0", "41011"): "41011 ComplexEventPeriodType",
        ("FIX.5.0", "41012"): "41012 ComplexEventBusinessCenter",
        ("FIX.5.0", "41013"): "41013 NoComplexEventRateSources",
        ("FIX.5.0", "41014"): "41014 ComplexEventRateSource",
        ("FIX.5.0", "41015"): "41015 ComplexEventRateSourceType",
        ("FIX.5.0", "41016"): "41016 ComplexEventReferencePage",
        ("FIX.5.0", "41017"): "41017 ComplexEventReferencePageHeading",
        ("FIX.5.0", "41018"): "41018 NoComplexEventDateBusinessCenters",
        ("FIX.5.0", "41019"): "41019 ComplexEventDateBusinessCenter",
        ("FIX.5.0", "41020"): "41020 ComplexEventDateUnadjusted",
        ("FIX.5.0", "41021"): "41021 ComplexEventDateRelativeTo",
        ("FIX.5.0", "41022"): "41022 ComplexEventDateOffsetPeriod",
        ("FIX.5.0", "41023"): "41023 ComplexEventDateOffsetUnit",
        ("FIX.5.0", "41024"): "41024 ComplexEventDateOffsetDayType",
        ("FIX.5.0", "41025"): "41025 ComplexEventDateBusinessDayConvention",
        ("FIX.5.0", "41026"): "41026 ComplexEventDateAdjusted",
        ("FIX.5.0", "41027"): "41027 ComplexEventFixingTime",
        ("FIX.5.0", "41028"): "41028 ComplexEventFixingTimeBusinessCenter",
        ("FIX.5.0", "41029"): "41029 NoComplexEventCreditEventSources",
        ("FIX.5.0", "41030"): "41030 ComplexEventCreditEventSource",
        ("FIX.5.0", "41031"): "41031 NoComplexEventSchedules",
        ("FIX.5.0", "41032"): "41032 ComplexEventScheduleStartDate",
        ("FIX.5.0", "41033"): "41033 ComplexEventScheduleEndDate",
        ("FIX.5.0", "41034"): "41034 ComplexEventScheduleFrequencyPeriod",
        ("FIX.5.0", "41035"): "41035 ComplexEventScheduleFrequencyUnit",
        ("FIX.5.0", "41036"): "41036 ComplexEventScheduleRollConvention",
        ("FIX.5.0", "41037"): "41037 NoDeliverySchedules",
        ("FIX.5.0", "41038"): "41038 DeliveryScheduleType",
        ("FIX.5.0", "41039"): "41039 DeliveryScheduleXID",
        ("FIX.5.0", "41040"): "41040 DeliveryScheduleNotional",
        ("FIX.5.0", "41041"): "41041 DeliveryScheduleNotionalUnitOfMeasure",
        ("FIX.5.0", "41042"): "41042 DeliveryScheduleNotionalCommodityFrequency",
        ("FIX.5.0", "41043"): "41043 DeliveryScheduleNegativeTolerance",
        ("FIX.5.0", "41044"): "41044 DeliverySchedulePositiveTolerance",
        ("FIX.5.0", "41045"): "41045 DeliveryScheduleToleranceUnitOfMeasure",
        ("FIX.5.0", "41046"): "41046 DeliveryScheduleToleranceType",
        ("FIX.5.0", "41047"): "41047 DeliveryScheduleSettlCountry",
        ("FIX.5.0", "41048"): "41048 DeliveryScheduleSettlTimeZone",
        ("FIX.5.0", "41049"): "41049 DeliveryScheduleSettlFlowType",
        ("FIX.5.0", "41050"): "41050 DeliveryScheduleSettlHolidaysProcessingInstruction",
        ("FIX.5.0", "41051"): "41051 NoDeliveryScheduleSettlDays",
        ("FIX.5.0", "41052"): "41052 DeliveryScheduleSettlDay",
        ("FIX.5.0", "41053"): "41053 DeliveryScheduleSettlTotalHours",
        ("FIX.5.0", "41054"): "41054 NoDeliveryScheduleSettlTimes",
        ("FIX.5.0", "41055"): "41055 DeliveryScheduleSettlStart",
        ("FIX.5.0", "41056"): "41056 DeliveryScheduleSettlEnd",
        ("FIX.5.0", "41057"): "41057 DeliveryScheduleSettlTimeType",
        ("FIX.5.0", "41058"): "41058 DeliveryStreamType",
        ("FIX.5.0", "41059"): "41059 DeliveryStreamPipeline",
        ("FIX.5.0", "41060"): "41060 DeliveryStreamEntryPoint",
        ("FIX.5.0", "41061"): "41061 DeliveryStreamWithdrawalPoint",
        ("FIX.5.0", "41062"): "41062 DeliveryStreamDeliveryPoint",
        ("FIX.5.0", "41063"): "41063 DeliveryStreamDeliveryRestriction",
        ("FIX.5.0", "41064"): "41064 DeliveryStreamDeliveryContingency",
        ("FIX.5.0", "41065"): "41065 DeliveryStreamDeliveryContingentPartySide",
        ("FIX.5.0", "41066"): "41066 DeliveryStreamDeliverAtSourceIndicator",
        ("FIX.5.0", "41067"): "41067 DeliveryStreamRiskApportionment",
        ("FIX.5.0", "41068"): "41068 DeliveryStreamTitleTransferLocation",
        ("FIX.5.0", "41069"): "41069 DeliveryStreamTitleTransferCondition",
        ("FIX.5.0", "41070"): "41070 DeliveryStreamImporterOfRecord",
        ("FIX.5.0", "41071"): "41071 DeliveryStreamNegativeTolerance",
        ("FIX.5.0", "41072"): "41072 DeliveryStreamPositiveTolerance",
        ("FIX.5.0", "41073"): "41073 DeliveryStreamToleranceUnitOfMeasure",
        ("FIX.5.0", "41074"): "41074 DeliveryStreamToleranceType",
        ("FIX.5.0", "41075"): "41075 DeliveryStreamToleranceOptionSide",
        ("FIX.5.0", "41076"): "41076 DeliveryStreamTotalPositiveTolerance",
        ("FIX.5.0", "41077"): "41077 DeliveryStreamTotalNegativeTolerance",
        ("FIX.5.0", "41078"): "41078 DeliveryStreamNotionalConversionFactor",
        ("FIX.5.0", "41079"): "41079 DeliveryStreamTransportEquipment",
        ("FIX.5.0", "41080"): "41080 DeliveryStreamElectingPartySide",
        ("FIX.5.0", "41081"): "41081 NoDeliveryStreamCycles",
        ("FIX.5.0", "41082"): "41082 DeliveryStreamCycleDesc",
        ("FIX.5.0", "41083"): "41083 EncodedDeliveryStreamCycleDescLen",
        ("FIX.5.0", "41084"): "41084 EncodedDeliveryStreamCycleDesc",
        ("FIX.5.0", "41085"): "41085 NoDeliveryStreamCommoditySources",
        ("FIX.5.0", "41086"): "41086 DeliveryStreamCommoditySource",
        ("FIX.5.0", "41087"): "41087 MarketDisruptionProvision",
        ("FIX.5.0", "41088"): "41088 MarketDisruptionFallbackProvision",
        ("FIX.5.0", "41089"): "41089 MarketDisruptionMaximumDays",
        ("FIX.5.0", "41090"): "41090 MarketDisruptionMaterialityPercentage",
        ("FIX.5.0", "41091"): "41091 MarketDisruptionMinimumFuturesContracts",
        ("FIX.5.0", "41092"): "41092 NoMarketDisruptionEvents",
        ("FIX.5.0", "41093"): "41093 MarketDisruptionEvent",
        ("FIX.5.0", "41094"): "41094 NoMarketDisruptionFallbacks",
        ("FIX.5.0", "41095"): "41095 MarketDisruptionFallbackType",
        ("FIX.5.0", "41096"): "41096 NoMarketDisruptionFallbackReferencePrices",
        ("FIX.5.0", "41097"): "41097 MarketDisruptionFallbackUnderlierType",
        ("FIX.5.0", "41098"): "41098 MarketDisruptionFallbackUnderlierSecurityID",
        ("FIX.5.0", "41099"): "41099 MarketDisruptionFallbackUnderlierSecurityIDSource",
        ("FIX.5.0", "41100"): "41100 MarketDisruptionFallbackUnderlierSecurityDesc",
        ("FIX.5.0", "41101"): "41101 EncodedMarketDisruptionFallbackUnderlierSecurityDescLen",
        ("FIX.5.0", "41102"): "41102 EncodedMarketDisruptionFallbackUnderlierSecurityDesc",
        ("FIX.5.0", "41103"): "41103 MarketDisruptionFallbackOpenUnits",
        ("FIX.5.0", "41104"): "41104 MarketDisruptionFallbackBasketCurrency",
        ("FIX.5.0", "41105"): "41105 MarketDisruptionFallbackBasketDivisor",
        ("FIX.5.0", "41106"): "41106 ExerciseDesc",
        ("FIX.5.0", "41107"): "41107 EncodedExerciseDescLen",
        ("FIX.5.0", "41108"): "41108 EncodedExerciseDesc",
        ("FIX.5.0", "41109"): "41109 AutomaticExerciseIndicator",
        ("FIX.5.0", "41110"): "41110 AutomaticExerciseThresholdRate",
        ("FIX.5.0", "41111"): "41111 ExerciseConfirmationMethod",
        ("FIX.5.0", "41112"): "41112 ManualNoticeBusinessCenter",
        ("FIX.5.0", "41113"): "41113 FallbackExerciseIndicator",
        ("FIX.5.0", "41114"): "41114 LimitedRightToConfirmIndicator",
        ("FIX.5.0", "41115"): "41115 ExerciseSplitTicketIndicator",
        ("FIX.5.0", "41116"): "41116 NoOptionExerciseBusinessCenters",
        ("FIX.5.0", "41117"): "41117 OptionExerciseBusinessCenter",
        ("FIX.5.0", "41118"): "41118 OptionExerciseBusinessDayConvention",
        ("FIX.5.0", "41119"): "41119 OptionExerciseEarliestDateOffsetDayType",
        ("FIX.5.0", "41120"): "41120 OptionExerciseEarliestDateOffsetPeriod",
        ("FIX.5.0", "41121"): "41121 OptionExerciseEarliestDateOffsetUnit",
        ("FIX.5.0", "41122"): "41122 OptionExerciseFrequencyPeriod",
        ("FIX.5.0", "41123"): "41123 OptionExerciseFrequencyUnit",
        ("FIX.5.0", "41124"): "41124 OptionExerciseStartDateUnadjusted",
        ("FIX.5.0", "41125"): "41125 OptionExerciseStartDateRelativeTo",
        ("FIX.5.0", "41126"): "41126 OptionExerciseStartDateOffsetPeriod",
        ("FIX.5.0", "41127"): "41127 OptionExerciseStartDateOffsetUnit",
        ("FIX.5.0", "41128"): "41128 OptionExerciseStartDateOffsetDayType",
        ("FIX.5.0", "41129"): "41129 OptionExerciseStartDateAdjusted",
        ("FIX.5.0", "41130"): "41130 OptionExerciseSkip",
        ("FIX.5.0", "41131"): "41131 OptionExerciseNominationDeadline",
        ("FIX.5.0", "41132"): "41132 OptionExerciseFirstDateUnadjusted",
        ("FIX.5.0", "41133"): "41133 OptionExerciseLastDateUnadjusted",
        ("FIX.5.0", "41134"): "41134 OptionExerciseEarliestTime",
        ("FIX.5.0", "41135"): "41135 OptionExerciseLatestTime",
        ("FIX.5.0", "41136"): "41136 OptionExerciseTimeBusinessCenter",
        ("FIX.5.0", "41137"): "41137 NoOptionExerciseDates",
        ("FIX.5.0", "41138"): "41138 OptionExerciseDate",
        ("FIX.5.0", "41139"): "41139 OptionExerciseDateType",
        ("FIX.5.0", "41140"): "41140 NoOptionExerciseExpirationDateBusinessCenters",
        ("FIX.5.0", "41141"): "41141 OptionExerciseExpirationDateBusinessCenter",
        ("FIX.5.0", "41142"): "41142 OptionExerciseExpirationDateBusinessDayConvention",
        ("FIX.5.0", "41143"): "41143 OptionExerciseExpirationDateRelativeTo",
        ("FIX.5.0", "41144"): "41144 OptionExerciseExpirationDateOffsetPeriod",
        ("FIX.5.0", "41145"): "41145 OptionExerciseExpirationDateOffsetUnit",
        ("FIX.5.0", "41146"): "41146 OptionExerciseExpirationFrequencyPeriod",
        ("FIX.5.0", "41147"): "41147 OptionExerciseExpirationFrequencyUnit",
        ("FIX.5.0", "41148"): "41148 OptionExerciseExpirationRollConvention",
        ("FIX.5.0", "41149"): "41149 OptionExerciseExpirationDateOffsetDayType",
        ("FIX.5.0", "41150"): "41150 OptionExerciseExpirationTime",
        ("FIX.5.0", "41151"): "41151 OptionExerciseExpirationTimeBusinessCenter",
        ("FIX.5.0", "41152"): "41152 NoOptionExerciseExpirationDates",
        ("FIX.5.0", "41153"): "41153 OptionExerciseExpirationDate",
        ("FIX.5.0", "41154"): "41154 OptionExerciseExpirationDateType",
        ("FIX.5.0", "41155"): "41155 PaymentUnitOfMeasure",
        ("FIX.5.0", "41156"): "41156 PaymentDateRelativeTo",
        ("FIX.5.0", "41157"): "41157 PaymentDateOffsetPeriod",
        ("FIX.5.0", "41158"): "41158 PaymentDateOffsetUnit",
        ("FIX.5.0", "41159"): "41159 PaymentDateOffsetDayType",
        ("FIX.5.0", "41160"): "41160 PaymentForwardStartType",
        ("FIX.5.0", "41161"): "41161 NoPaymentScheduleFixingDays",
        ("FIX.5.0", "41162"): "41162 PaymentScheduleFixingDayOfWeek",
        ("FIX.5.0", "41163"): "41163 PaymentScheduleFixingDayNumber",
        ("FIX.5.0", "41164"): "41164 PaymentScheduleXID",
        ("FIX.5.0", "41165"): "41165 PaymentScheduleXIDRef",
        ("FIX.5.0", "41166"): "41166 PaymentScheduleRateCurrency",
        ("FIX.5.0", "41167"): "41167 PaymentScheduleRateUnitOfMeasure",
        ("FIX.5.0", "41168"): "41168 PaymentScheduleRateConversionFactor",
        ("FIX.5.0", "41169"): "41169 PaymentScheduleRateSpreadType",
        ("FIX.5.0", "41170"): "41170 PaymentScheduleSettlPeriodPrice",
        ("FIX.5.0", "41171"): "41171 PaymentScheduleSettlPeriodPriceCurrency",
        ("FIX.5.0", "41172"): "41172 PaymentScheduleSettlPeriodPriceUnitOfMeasure",
        ("FIX.5.0", "41173"): "41173 PaymentScheduleStepUnitOfMeasure",
        ("FIX.5.0", "41174"): "41174 PaymentScheduleFixingDayDistribution",
        ("FIX.5.0", "41175"): "41175 PaymentScheduleFixingDayCount",
        ("FIX.5.0", "41176"): "41176 PaymentScheduleFixingLagPeriod",
        ("FIX.5.0", "41177"): "41177 PaymentScheduleFixingLagUnit",
        ("FIX.5.0", "41178"): "41178 PaymentScheduleFixingFirstObservationDateOffsetPeriod",
        ("FIX.5.0", "41179"): "41179 PaymentScheduleFixingFirstObservationDateOffsetUnit",
        ("FIX.5.0", "41180"): "41180 PaymentStreamFlatRateIndicator",
        ("FIX.5.0", "41181"): "41181 PaymentStreamFlatRateAmount",
        ("FIX.5.0", "41182"): "41182 PaymentStreamFlatRateCurrency",
        ("FIX.5.0", "41183"): "41183 PaymentStreamMaximumPaymentAmount",
        ("FIX.5.0", "41184"): "41184 PaymentStreamMaximumPaymentCurrency",
        ("FIX.5.0", "41185"): "41185 PaymentStreamMaximumTransactionAmount",
        ("FIX.5.0", "41186"): "41186 PaymentStreamMaximumTransactionCurrency",
        ("FIX.5.0", "41187"): "41187 PaymentStreamFixedAmountUnitOfMeasure",
        ("FIX.5.0", "41188"): "41188 PaymentStreamTotalFixedAmount",
        ("FIX.5.0", "41189"): "41189 PaymentStreamWorldScaleRate",
        ("FIX.5.0", "41190"): "41190 PaymentStreamContractPrice",
        ("FIX.5.0", "41191"): "41191 PaymentStreamContractPriceCurrency",
        ("FIX.5.0", "41192"): "41192 NoPaymentStreamPricingBusinessCenters",
        ("FIX.5.0", "41193"): "41193 PaymentStreamPricingBusinessCenter",
        ("FIX.5.0", "41194"): "41194 PaymentStreamRateIndex2CurvePeriod",
        ("FIX.5.0", "41195"): "41195 PaymentStreamRateIndex2CurveUnit",
        ("FIX.5.0", "41196"): "41196 PaymentStreamRateIndexLocation",
        ("FIX.5.0", "41197"): "41197 PaymentStreamRateIndexLevel",
        ("FIX.5.0", "41198"): "41198 PaymentStreamRateIndexUnitOfMeasure",
        ("FIX.5.0", "41199"): "41199 PaymentStreamSettlLevel",
        ("FIX.5.0", "41200"): "41200 PaymentStreamReferenceLevel",
        ("FIX.5.0", "41201"): "41201 PaymentStreamReferenceLevelUnitOfMeasure",
        ("FIX.5.0", "41202"): "41202 PaymentStreamReferenceLevelEqualsZeroIndicator",
        ("FIX.5.0", "41203"): "41203 PaymentStreamRateSpreadCurrency",
        ("FIX.5.0", "41204"): "41204 PaymentStreamRateSpreadUnitOfMeasure",
        ("FIX.5.0", "41205"): "41205 PaymentStreamRateConversionFactor",
        ("FIX.5.0", "41206"): "41206 PaymentStreamRateSpreadType",
        ("FIX.5.0", "41207"): "41207 PaymentStreamLastResetRate",
        ("FIX.5.0", "41208"): "41208 PaymentStreamFinalRate",
        ("FIX.5.0", "41209"): "41209 PaymentStreamCalculationLagPeriod",
        ("FIX.5.0", "41210"): "41210 PaymentStreamCalculationLagUnit",
        ("FIX.5.0", "41211"): "41211 PaymentStreamFirstObservationDateOffsetPeriod",
        ("FIX.5.0", "41212"): "41212 PaymentStreamFirstObservationDateOffsetUnit",
        ("FIX.5.0", "41213"): "41213 PaymentStreamPricingDayType",
        ("FIX.5.0", "41214"): "41214 PaymentStreamPricingDayDistribution",
        ("FIX.5.0", "41215"): "41215 PaymentStreamPricingDayCount",
        ("FIX.5.0", "41216"): "41216 PaymentStreamPricingBusinessCalendar",
        ("FIX.5.0", "41217"): "41217 PaymentStreamPricingBusinessDayConvention",
        ("FIX.5.0", "41218"): "41218 DeliveryStreamRiskApportionmentSource",
        ("FIX.5.0", "41219"): "41219 LegDeliveryStreamRiskApportionmentSource",
        ("FIX.5.0", "41220"): "41220 NoPaymentStreamPaymentDates",
        ("FIX.5.0", "41221"): "41221 PaymentStreamPaymentDate",
        ("FIX.5.0", "41222"): "41222 PaymentStreamPaymentDateType",
        ("FIX.5.0", "41223"): "41223 PaymentStreamMasterAgreementPaymentDatesIndicator",
        ("FIX.5.0", "41224"): "41224 NoPaymentStreamPricingDates",
        ("FIX.5.0", "41225"): "41225 PaymentStreamPricingDate",
        ("FIX.5.0", "41226"): "41226 PaymentStreamPricingDateType",
        ("FIX.5.0", "41227"): "41227 NoPaymentStreamPricingDays",
        ("FIX.5.0", "41228"): "41228 PaymentStreamPricingDayOfWeek",
        ("FIX.5.0", "41229"): "41229 PaymentStreamPricingDayNumber",
        ("FIX.5.0", "41230"): "41230 NoPricingDateBusinessCenters",
        ("FIX.5.0", "41231"): "41231 PricingDateBusinessCenter",
        ("FIX.5.0", "41232"): "41232 PricingDateUnadjusted",
        ("FIX.5.0", "41233"): "41233 PricingDateBusinessDayConvention",
        ("FIX.5.0", "41234"): "41234 PricingDateAdjusted",
        ("FIX.5.0", "41235"): "41235 PricingTime",
        ("FIX.5.0", "41236"): "41236 PricingTimeBusinessCenter",
        ("FIX.5.0", "41237"): "41237 NoStreamAssetAttributes",
        ("FIX.5.0", "41238"): "41238 StreamAssetAttributeType",
        ("FIX.5.0", "41239"): "41239 StreamAssetAttributeValue",
        ("FIX.5.0", "41240"): "41240 StreamAssetAttributeLimit",
        ("FIX.5.0", "41241"): "41241 NoStreamCalculationPeriodDates",
        ("FIX.5.0", "41242"): "41242 StreamCalculationPeriodDate",
        ("FIX.5.0", "41243"): "41243 StreamCalculationPeriodDateType",
        ("FIX.5.0", "41244"): "41244 StreamCalculationPeriodDatesXID",
        ("FIX.5.0", "41245"): "41245 StreamCalculationPeriodDatesXIDRef",
        ("FIX.5.0", "41246"): "41246 StreamCalculationBalanceOfFirstPeriod",
        ("FIX.5.0", "41247"): "41247 StreamCalculationCorrectionPeriod",
        ("FIX.5.0", "41248"): "41248 StreamCalculationCorrectionUnit",
        ("FIX.5.0", "41249"): "41249 NoStreamCommoditySettlBusinessCenters",
        ("FIX.5.0", "41250"): "41250 StreamCommoditySettlBusinessCenter",
        ("FIX.5.0", "41251"): "41251 StreamCommodityBase",
        ("FIX.5.0", "41252"): "41252 StreamCommodityType",
        ("FIX.5.0", "41253"): "41253 StreamCommoditySecurityID",
        ("FIX.5.0", "41254"): "41254 StreamCommoditySecurityIDSource",
        ("FIX.5.0", "41255"): "41255 StreamCommodityDesc",
        ("FIX.5.0", "41256"): "41256 EncodedStreamCommodityDescLen",
        ("FIX.5.0", "41257"): "41257 EncodedStreamCommodityDesc",
        ("FIX.5.0", "41258"): "41258 StreamCommodityUnitOfMeasure",
        ("FIX.5.0", "41259"): "41259 StreamCommodityCurrency",
        ("FIX.5.0", "41260"): "41260 StreamCommodityExchange",
        ("FIX.5.0", "41261"): "41261 StreamCommodityRateSource",
        ("FIX.5.0", "41262"): "41262 StreamCommodityRateReferencePage",
        ("FIX.5.0", "41263"): "41263 StreamCommodityRateReferencePageHeading",
        ("FIX.5.0", "41264"): "41264 StreamDataProvider",
        ("FIX.5.0", "41265"): "41265 StreamCommodityPricingType",
        ("FIX.5.0", "41266"): "41266 StreamCommodityNearbySettlDayPeriod",
        ("FIX.5.0", "41267"): "41267 StreamCommodityNearbySettlDayUnit",
        ("FIX.5.0", "41268"): "41268 StreamCommoditySettlDateUnadjusted",
        ("FIX.5.0", "41269"): "41269 StreamCommoditySettlDateBusinessDayConvention",
        ("FIX.5.0", "41270"): "41270 StreamCommoditySettlDateAdjusted",
        ("FIX.5.0", "41271"): "41271 StreamCommoditySettlMonth",
        ("FIX.5.0", "41272"): "41272 StreamCommoditySettlDateRollPeriod",
        ("FIX.5.0", "41273"): "41273 StreamCommoditySettlDateRollUnit",
        ("FIX.5.0", "41274"): "41274 StreamCommoditySettlDayType",
        ("FIX.5.0", "41275"): "41275 StreamCommodityXID",
        ("FIX.5.0", "41276"): "41276 StreamCommodityXIDRef",
        ("FIX.5.0", "41277"): "41277 NoStreamCommodityAltIDs",
        ("FIX.5.0", "41278"): "41278 StreamCommodityAltID",
        ("FIX.5.0", "41279"): "41279 StreamCommodityAltIDSource",
        ("FIX.5.0", "41280"): "41280 NoStreamCommodityDataSources",
        ("FIX.5.0", "41281"): "41281 StreamCommodityDataSourceID",
        ("FIX.5.0", "41282"): "41282 StreamCommodityDataSourceIDType",
        ("FIX.5.0", "41283"): "41283 NoStreamCommoditySettlDays",
        ("FIX.5.0", "41284"): "41284 StreamCommoditySettlDay",
        ("FIX.5.0", "41285"): "41285 StreamCommoditySettlTotalHours",
        ("FIX.5.0", "41286"): "41286 NoStreamCommoditySettlTimes",
        ("FIX.5.0", "41287"): "41287 StreamCommoditySettlStart",
        ("FIX.5.0", "41288"): "41288 StreamCommoditySettlEnd",
        ("FIX.5.0", "41289"): "41289 NoStreamCommoditySettlPeriods",
        ("FIX.5.0", "41290"): "41290 StreamCommoditySettlCountry",
        ("FIX.5.0", "41291"): "41291 StreamCommoditySettlTimeZone",
        ("FIX.5.0", "41292"): "41292 StreamCommoditySettlFlowType",
        ("FIX.5.0", "41293"): "41293 StreamCommoditySettlPeriodNotional",
        ("FIX.5.0", "41294"): "41294 StreamCommoditySettlPeriodNotionalUnitOfMeasure",
        ("FIX.5.0", "41295"): "41295 StreamCommoditySettlPeriodFrequencyPeriod",
        ("FIX.5.0", "41296"): "41296 StreamCommoditySettlPeriodFrequencyUnit",
        ("FIX.5.0", "41297"): "41297 StreamCommoditySettlPeriodPrice",
        ("FIX.5.0", "41298"): "41298 StreamCommoditySettlPeriodPriceUnitOfMeasure",
        ("FIX.5.0", "41299"): "41299 StreamCommoditySettlPeriodPriceCurrency",
        ("FIX.5.0", "41300"): "41300 StreamCommoditySettlHolidaysProcessingInstruction",
        ("FIX.5.0", "41301"): "41301 StreamCommoditySettlPeriodXID",
        ("FIX.5.0", "41302"): "41302 StreamCommoditySettlPeriodXIDRef",
        ("FIX.5.0", "41303"): "41303 StreamXID",
        ("FIX.5.0", "41304"): "41304 PaymentLegRefID",
        ("FIX.5.0", "41305"): "41305 StreamNotionalXIDRef",
        ("FIX.5.0", "41306"): "41306 StreamNotionalFrequencyPeriod",
        ("FIX.5.0", "41307"): "41307 StreamNotionalFrequencyUnit",
        ("FIX.5.0", "41308"): "41308 StreamNotionalCommodityFrequency",
        ("FIX.5.0", "41309"): "41309 StreamNotionalUnitOfMeasure",
        ("FIX.5.0", "41310"): "41310 StreamTotalNotional",
        ("FIX.5.0", "41311"): "41311 StreamTotalNotionalUnitOfMeasure",
        ("FIX.5.0", "41312"): "41312 NoMandatoryClearingJurisdictions",
        ("FIX.5.0", "41313"): "41313 MandatoryClearingJurisdiction",
        ("FIX.5.0", "41314"): "41314 UnderlyingProtectionTermXIDRef",
        ("FIX.5.0", "41315"): "41315 UnderlyingSettlTermXIDRef",
        ("FIX.5.0", "41316"): "41316 NoLegAdditionalTermBondRefs",
        ("FIX.5.0", "41317"): "41317 LegAdditionalTermBondSecurityID",
        ("FIX.5.0", "41318"): "41318 LegAdditionalTermBondSecurityIDSource",
        ("FIX.5.0", "41319"): "41319 LegAdditionalTermBondDesc",
        ("FIX.5.0", "41320"): "41320 EncodedLegAdditionalTermBondDescLen",
        ("FIX.5.0", "41321"): "41321 EncodedLegAdditionalTermBondDesc",
        ("FIX.5.0", "41322"): "41322 LegAdditionalTermBondCurrency",
        ("FIX.5.0", "41323"): "41323 LegAdditionalTermBondIssuer",
        ("FIX.5.0", "41324"): "41324 EncodedLegAdditionalTermBondIssuerLen",
        ("FIX.5.0", "41325"): "41325 EncodedLegAdditionalTermBondIssuer",
        ("FIX.5.0", "41326"): "41326 LegAdditionalTermBondSeniority",
        ("FIX.5.0", "41327"): "41327 LegAdditionalTermBondCouponType",
        ("FIX.5.0", "41328"): "41328 LegAdditionalTermBondCouponRate",
        ("FIX.5.0", "41329"): "41329 LegAdditionalTermBondMaturityDate",
        ("FIX.5.0", "41330"): "41330 LegAdditionalTermBondParValue",
        ("FIX.5.0", "41331"): "41331 LegAdditionalTermBondCurrentTotalIssuedAmount",
        ("FIX.5.0", "41332"): "41332 LegAdditionalTermBondCouponFrequencyPeriod",
        ("FIX.5.0", "41333"): "41333 LegAdditionalTermBondCouponFrequencyUnit",
        ("FIX.5.0", "41334"): "41334 LegAdditionalTermBondDayCount",
        ("FIX.5.0", "41335"): "41335 NoLegAdditionalTerms",
        ("FIX.5.0", "41336"): "41336 LegAdditionalTermConditionPrecedentBondIndicator",
        ("FIX.5.0", "41337"): "41337 LegAdditionalTermDiscrepancyClauseIndicator",
        ("FIX.5.0", "41338"): "41338 UnderlyingMarketDisruptionValue",
        ("FIX.5.0", "41339"): "41339 UnderlyingMarketDisruptionFallbackValue",
        ("FIX.5.0", "41340"): "41340 NoUnderlyingAdditionalTermBondRefs",
        ("FIX.5.0", "41341"): "41341 UnderlyingAdditionalTermBondSecurityID",
        ("FIX.5.0", "41342"): "41342 NoLegCashSettlDealers",
        ("FIX.5.0", "41343"): "41343 LegCashSettlDealer",
        ("FIX.5.0", "41344"): "41344 NoLegCashSettlTerms",
        ("FIX.5.0", "41345"): "41345 LegCashSettlCurrency",
        ("FIX.5.0", "41346"): "41346 LegCasSettlValuationFirstBusinessDayOffset",
        ("FIX.5.0", "41347"): "41347 LegCashSettlValuationSubsequentBusinessDaysOffset",
        ("FIX.5.0", "41348"): "41348 LegCashSettlNumOfValuationDates",
        ("FIX.5.0", "41349"): "41349 LegCashSettlValuationTime",
        ("FIX.5.0", "41350"): "41350 LegCashSettlBusinessCenter",
        ("FIX.5.0", "41351"): "41351 LegCashSettlQuoteMethod",
        ("FIX.5.0", "41352"): "41352 LegCashSettlQuoteAmount",
        ("FIX.5.0", "41353"): "41353 LegCashSettlQuoteCurrency",
        ("FIX.5.0", "41354"): "41354 LegCashSettlMinimumQuoteAmount",
        ("FIX.5.0", "41355"): "41355 LegCashSettlMinimumQuoteCurrency",
        ("FIX.5.0", "41356"): "41356 LegCashSettlBusinessDays",
        ("FIX.5.0", "41357"): "41357 LegCashSettlAmount",
        ("FIX.5.0", "41358"): "41358 LegCashSettlRecoveryFactor",
        ("FIX.5.0", "41359"): "41359 LegCashSettlFixedTermIndicator",
        ("FIX.5.0", "41360"): "41360 LegCashSettlAccruedInterestIndicator",
        ("FIX.5.0", "41361"): "41361 LegCashSettlValuationMethod",
        ("FIX.5.0", "41362"): "41362 LegCashSettlTermXID",
        ("FIX.5.0", "41363"): "41363 NoLegComplexEventAveragingObservations",
        ("FIX.5.0", "41364"): "41364 LegComplexEventAveragingObservationNumber",
        ("FIX.5.0", "41365"): "41365 LegComplexEventAveragingWeight",
        ("FIX.5.0", "41366"): "41366 NoLegComplexEventCreditEvents",
        ("FIX.5.0", "41367"): "41367 LegComplexEventCreditEventType",
        ("FIX.5.0", "41368"): "41368 LegComplexEventCreditEventValue",
        ("FIX.5.0", "41369"): "41369 LegComplexEventCreditEventCurrency",
        ("FIX.5.0", "41370"): "41370 LegComplexEventCreditEventPeriod",
        ("FIX.5.0", "41371"): "41371 LegComplexEventCreditEventUnit",
        ("FIX.5.0", "41372"): "41372 LegComplexEventCreditEventDayType",
        ("FIX.5.0", "41373"): "41373 LegComplexEventCreditEventRateSource",
        ("FIX.5.0", "41374"): "41374 NoLegComplexEventCreditEventQualifiers",
        ("FIX.5.0", "41375"): "41375 LegComplexEventCreditEventQualifier",
        ("FIX.5.0", "41376"): "41376 NoLegComplexEventPeriodDateTimes",
        ("FIX.5.0", "41377"): "41377 LegComplexEventPeriodDate",
        ("FIX.5.0", "41378"): "41378 LegComplexEventPeriodTime",
        ("FIX.5.0", "41379"): "41379 NoLegComplexEventPeriods",
        ("FIX.5.0", "41380"): "41380 LegComplexEventPeriodType",
        ("FIX.5.0", "41381"): "41381 LegComplexEventBusinessCenter",
        ("FIX.5.0", "41382"): "41382 NoLegComplexEventRateSources",
        ("FIX.5.0", "41383"): "41383 LegComplexEventRateSource",
        ("FIX.5.0", "41384"): "41384 LegComplexEventRateSourceType",
        ("FIX.5.0", "41385"): "41385 LegComplexEventReferencePage",
        ("FIX.5.0", "41386"): "41386 LegComplexEvenReferencePageHeading",
        ("FIX.5.0", "41387"): "41387 NoLegComplexEventDateBusinessCenters",
        ("FIX.5.0", "41388"): "41388 LegComplexEventDateBusinessCenter",
        ("FIX.5.0", "41389"): "41389 LegComplexEventDateUnadjusted",
        ("FIX.5.0", "41390"): "41390 LegComplexEventDateRelativeTo",
        ("FIX.5.0", "41391"): "41391 LegComplexEventDateOffsetPeriod",
        ("FIX.5.0", "41392"): "41392 LegComplexEventDateOffsetUnit",
        ("FIX.5.0", "41393"): "41393 LegComplexEventDateOffsetDayType",
        ("FIX.5.0", "41394"): "41394 LegComplexEventDateBusinessDayConvention",
        ("FIX.5.0", "41395"): "41395 LegComplexEventDateAdjusted",
        ("FIX.5.0", "41396"): "41396 LegComplexEventFixingTime",
        ("FIX.5.0", "41397"): "41397 LegComplexEventFixingTimeBusinessCenter",
        ("FIX.5.0", "41398"): "41398 NoLegComplexEventCreditEventSources",
        ("FIX.5.0", "41399"): "41399 LegComplexEventCreditEventSource",
        ("FIX.5.0", "41400"): "41400 NoLegComplexEventSchedules",
        ("FIX.5.0", "41401"): "41401 LegComplexEventScheduleStartDate",
        ("FIX.5.0", "41402"): "41402 LegComplexEventScheduleEndDate",
        ("FIX.5.0", "41403"): "41403 LegComplexEventScheduleFrequencyPeriod",
        ("FIX.5.0", "41404"): "41404 LegComplexEventScheduleFrequencyUnit",
        ("FIX.5.0", "41405"): "41405 LegComplexEventScheduleRollConvention",
        ("FIX.5.0", "41406"): "41406 ProvisionCashSettlQuoteReferencePage",
        ("FIX.5.0", "41407"): "41407 LegProvisionCashSettlQuoteReferencePage",
        ("FIX.5.0", "41408"): "41408 NoLegDeliverySchedules",
        ("FIX.5.0", "41409"): "41409 LegDeliveryScheduleType",
        ("FIX.5.0", "41410"): "41410 LegDeliveryScheduleXID",
        ("FIX.5.0", "41411"): "41411 LegDeliveryScheduleNotional",
        ("FIX.5.0", "41412"): "41412 LegDeliveryScheduleNotionalUnitOfMeasure",
        ("FIX.5.0", "41413"): "41413 LegDeliveryScheduleNotionalCommodityFrequency",
        ("FIX.5.0", "41414"): "41414 LegDeliveryScheduleNegativeTolerance",
        ("FIX.5.0", "41415"): "41415 LegDeliverySchedulePositiveTolerance",
        ("FIX.5.0", "41416"): "41416 LegDeliveryScheduleToleranceUnitOfMeasure",
        ("FIX.5.0", "41417"): "41417 LegDeliveryScheduleToleranceType",
        ("FIX.5.0", "41418"): "41418 LegDeliveryScheduleSettlCountry",
        ("FIX.5.0", "41419"): "41419 LegDeliveryScheduleSettlTimeZone",
        ("FIX.5.0", "41420"): "41420 LegDeliveryScheduleSettlFlowType",
        ("FIX.5.0", "41421"): "41421 LegDeliveryScheduleSettlHolidaysProcessingInstruction",
        ("FIX.5.0", "41422"): "41422 NoLegDeliveryScheduleSettlDays",
        ("FIX.5.0", "41423"): "41423 LegDeliveryScheduleSettlDay",
        ("FIX.5.0", "41424"): "41424 LegDeliveryScheduleSettlTotalHours",
        ("FIX.5.0", "41425"): "41425 NoLegDeliveryScheduleSettlTimes",
        ("FIX.5.0", "41426"): "41426 LegDeliveryScheduleSettlStart",
        ("FIX.5.0", "41427"): "41427 LegDeliveryScheduleSettlEnd",
        ("FIX.5.0", "41428"): "41428 LegDeliveryScheduleSettlTimeType",
        ("FIX.5.0", "41429"): "41429 LegDeliveryStreamType",
        ("FIX.5.0", "41430"): "41430 LegDeliveryStreamPipeline",
        ("FIX.5.0", "41431"): "41431 LegDeliveryStreamEntryPoint",
        ("FIX.5.0", "41432"): "41432 LegDeliveryStreamWithdrawalPoint",
        ("FIX.5.0", "41433"): "41433 LegDeliveryStreamDeliveryPoint",
        ("FIX.5.0", "41434"): "41434 LegDeliveryStreamDeliveryRestriction",
        ("FIX.5.0", "41435"): "41435 LegDeliveryStreamDeliveryContingency",
        ("FIX.5.0", "41436"): "41436 LegDeliveryStreamDeliveryContingentPartySide",
        ("FIX.5.0", "41437"): "41437 LegDeliveryStreamDeliverAtSourceIndicator",
        ("FIX.5.0", "41438"): "41438 LegDeliveryStreamRiskApportionment",
        ("FIX.5.0", "41439"): "41439 LegDeliveryStreamTitleTransferLocation",
        ("FIX.5.0", "41440"): "41440 LegDeliveryStreamTitleTransferCondition",
        ("FIX.5.0", "41441"): "41441 LegDeliveryStreamImporterOfRecord",
        ("FIX.5.0", "41442"): "41442 LegDeliveryStreamNegativeTolerance",
        ("FIX.5.0", "41443"): "41443 LegDeliveryStreamPositiveTolerance",
        ("FIX.5.0", "41444"): "41444 LegDeliveryStreamToleranceUnitOfMeasure",
        ("FIX.5.0", "41445"): "41445 LegDeliveryStreamToleranceType",
        ("FIX.5.0", "41446"): "41446 LegDeliveryStreamToleranceOptionSide",
        ("FIX.5.0", "41447"): "41447 LegDeliveryStreamTotalPositiveTolerance",
        ("FIX.5.0", "41448"): "41448 LegDeliveryStreamTotalNegativeTolerance",
        ("FIX.5.0", "41449"): "41449 LegDeliveryStreamNotionalConversionFactor",
        ("FIX.5.0", "41450"): "41450 LegDeliveryStreamTransportEquipment",
        ("FIX.5.0", "41451"): "41451 LegDeliveryStreamElectingPartySide",
        ("FIX.5.0", "41452"): "41452 NoLegStreamAssetAttributes",
        ("FIX.5.0", "41453"): "41453 LegStreamAssetAttributeType",
        ("FIX.5.0", "41454"): "41454 LegStreamAssetAttributeValue",
        ("FIX.5.0", "41455"): "41455 LegStreamAssetAttributeLimit",
        ("FIX.5.0", "41456"): "41456 NoLegDeliveryStreamCycles",
        ("FIX.5.0", "41457"): "41457 LegDeliveryStreamCycleDesc",
        ("FIX.5.0", "41458"): "41458 EncodedLegDeliveryStreamCycleDescLen",
        ("FIX.5.0", "41459"): "41459 EncodedLegDeliveryStreamCycleDesc",
        ("FIX.5.0", "41460"): "41460 NoLegDeliveryStreamCommoditySources",
        ("FIX.5.0", "41461"): "41461 LegDeliveryStreamCommoditySource",
        ("FIX.5.0", "41462"): "41462 LegMarketDisruptionProvision",
        ("FIX.5.0", "41463"): "41463 LegMarketDisruptionFallbackProvision",
        ("FIX.5.0", "41464"): "41464 LegMarketDisruptionMaximumDays",
        ("FIX.5.0", "41465"): "41465 LegMarketDisruptionMaterialityPercentage",
        ("FIX.5.0", "41466"): "41466 LegMarketDisruptionMinimumFuturesContracts",
        ("FIX.5.0", "41467"): "41467 NoLegMarketDisruptionEvents",
        ("FIX.5.0", "41468"): "41468 LegMarketDisruptionEvent",
        ("FIX.5.0", "41469"): "41469 NoLegMarketDisruptionFallbacks",
        ("FIX.5.0", "41470"): "41470 LegMarketDisruptionFallbackType",
        ("FIX.5.0", "41471"): "41471 NoLegMarketDisruptionFallbackReferencePrices",
        ("FIX.5.0", "41472"): "41472 LegMarketDisruptionFallbackUnderlierType",
        ("FIX.5.0", "41473"): "41473 LegMarketDisruptionFallbackUnderlierSecurityID",
        ("FIX.5.0", "41474"): "41474 LegMarketDisruptionFallbackUnderlierSecurityIDSource",
        ("FIX.5.0", "41475"): "41475 LegMarketDisruptionFallbackUnderlierSecurityDesc",
        ("FIX.5.0", "41476"): "41476 EncodedLegMarketDisruptionFallbackUnderlierSecurityDescLen",
        ("FIX.5.0", "41477"): "41477 EncodedLegMarketDisruptionFallbackUnderlierSecurityDesc",
        ("FIX.5.0", "41478"): "41478 LegMarketDisruptionFallbackOpenUnits",
        ("FIX.5.0", "41479"): "41479 LegMarketDisruptionFallbackBasketCurrency",
        ("FIX.5.0", "41480"): "41480 LegMarketDisruptionFallbackBasketDivisor",
        ("FIX.5.0", "41481"): "41481 LegExerciseDesc",
        ("FIX.5.0", "41482"): "41482 EncodedLegExerciseDescLen",
        ("FIX.5.0", "41483"): "41483 EncodedLegExerciseDesc",
        ("FIX.5.0", "41484"): "41484 LegAutomaticExerciseIndicator",
        ("FIX.5.0", "41485"): "41485 LegAutomaticExerciseThresholdRate",
        ("FIX.5.0", "41486"): "41486 LegExerciseConfirmationMethod",
        ("FIX.5.0", "41487"): "41487 LegManualNoticeBusinessCenter",
        ("FIX.5.0", "41488"): "41488 LegFallbackExerciseIndicator",
        ("FIX.5.0", "41489"): "41489 LegLimitRightToConfirmIndicator",
        ("FIX.5.0", "41490"): "41490 LegExerciseSplitTicketIndicator",
        ("FIX.5.0", "41491"): "41491 NoLegOptionExerciseBusinessCenters",
        ("FIX.5.0", "41492"): "41492 LegOptionExerciseBusinessCenter",
        ("FIX.5.0", "41493"): "41493 LegOptionExerciseBusinessDayConvention",
        ("FIX.5.0", "41494"): "41494 LegOptionExerciseEarliestDateOffsetDayType",
        ("FIX.5.0", "41495"): "41495 LegOptionExerciseEarliestDateOffsetPeriod",
        ("FIX.5.0", "41496"): "41496 LegOptionExerciseEarliestDateOffsetUnit",
        ("FIX.5.0", "41497"): "41497 LegOptionExerciseFrequencyPeriod",
        ("FIX.5.0", "41498"): "41498 LegOptionExerciseFrequencyUnit",
        ("FIX.5.0", "41499"): "41499 LegOptionExerciseStartDateUnadjusted",
        ("FIX.5.0", "41500"): "41500 LegOptionExerciseStartDateRelativeTo",
        ("FIX.5.0", "41501"): "41501 LegOptionExerciseStartDateOffsetPeriod",
        ("FIX.5.0", "41502"): "41502 LegOptionExerciseStartDateOffsetUnit",
        ("FIX.5.0", "41503"): "41503 LegOptionExerciseStartDateOffsetDayType",
        ("FIX.5.0", "41504"): "41504 LegOptionExerciseStartDateAdjusted",
        ("FIX.5.0", "41505"): "41505 LegOptionExerciseSkip",
        ("FIX.5.0", "41506"): "41506 LegOptionExerciseNominationDeadline",
        ("FIX.5.0", "41507"): "41507 LegOptionExerciseFirstDateUnadjusted",
        ("FIX.5.0", "41508"): "41508 LegOptionExerciseLastDateUnadjusted",
        ("FIX.5.0", "41509"): "41509 LegOptionExerciseEarliestTime",
        ("FIX.5.0", "41510"): "41510 LegOptionExerciseLatestTime",
        ("FIX.5.0", "41511"): "41511 LegOptionExerciseTimeBusinessCenter",
        ("FIX.5.0", "41512"): "41512 NoLegOptionExerciseDates",
        ("FIX.5.0", "41513"): "41513 LegOptionExerciseDate",
        ("FIX.5.0", "41514"): "41514 LegOptionExerciseDateType",
        ("FIX.5.0", "41515"): "41515 NoLegOptionExerciseExpirationDateBusinessCenters",
        ("FIX.5.0", "41516"): "41516 LegOptionExerciseExpirationDateBusinessCenter",
        ("FIX.5.0", "41517"): "41517 LegOptionExerciseExpirationDateBusinessDayConvention",
        ("FIX.5.0", "41518"): "41518 LegOptionExerciseExpirationDateRelativeTo",
        ("FIX.5.0", "41519"): "41519 LegOptionExerciseExpirationDateOffsetPeriod",
        ("FIX.5.0", "41520"): "41520 LegOptionExerciseExpirationDateOffsetUnit",
        ("FIX.5.0", "41521"): "41521 LegOptionExerciseExpirationFrequencyPeriod",
        ("FIX.5.0", "41522"): "41522 LegOptionExerciseExpirationFrequencyUnit",
        ("FIX.5.0", "41523"): "41523 LegOptionExerciseExpirationRollConvention",
        ("FIX.5.0", "41524"): "41524 LegOptionExerciseExpirationDateOffsetDayType",
        ("FIX.5.0", "41525"): "41525 LegOptionExerciseExpirationTime",
        ("FIX.5.0", "41526"): "41526 LegOptionExerciseExpirationTimeBusinessCenter",
        ("FIX.5.0", "41527"): "41527 NoLegOptionExerciseExpirationDates",
        ("FIX.5.0", "41528"): "41528 LegOptionExerciseExpirationDate",
        ("FIX.5.0", "41529"): "41529 LegOptionExerciseExpirationDateType",
        ("FIX.5.0", "41530"): "41530 NoLegPaymentScheduleFixingDays",
        ("FIX.5.0", "41531"): "41531 LegPaymentScheduleFixingDayOfWeek",
        ("FIX.5.0", "41532"): "41532 LegPaymentScheduleFixingDayNumber",
        ("FIX.5.0", "41533"): "41533 LegPaymentScheduleXID",
        ("FIX.5.0", "41534"): "41534 LegPaymentScheduleXIDRef",
        ("FIX.5.0", "41535"): "41535 LegPaymentScheduleRateCurrency",
        ("FIX.5.0", "41536"): "41536 LegPaymentScheduleRateUnitOfMeasure",
        ("FIX.5.0", "41537"): "41537 LegPaymentScheduleRateConversionFactor",
        ("FIX.5.0", "41538"): "41538 LegPaymentScheduleRateSpreadType",
        ("FIX.5.0", "41539"): "41539 LegPaymentScheduleSettlPeriodPrice",
        ("FIX.5.0", "41540"): "41540 LegPaymentScheduleSettlPeriodPriceCurrency",
        ("FIX.5.0", "41541"): "41541 LegPaymentScheduleSettlPeriodPriceUnitOfMeasure",
        ("FIX.5.0", "41542"): "41542 LegPaymentScheduleStepUnitOfMeasure",
        ("FIX.5.0", "41543"): "41543 LegPaymentScheduleFixingDayDistribution",
        ("FIX.5.0", "41544"): "41544 LegPaymentScheduleFixingDayCount",
        ("FIX.5.0", "41545"): "41545 LegPaymentScheduleFixingLagPeriod",
        ("FIX.5.0", "41546"): "41546 LegPaymentScheduleFixingLagUnit",
        ("FIX.5.0", "41547"): "41547 LegPaymentScheduleFixingFirstObservationDateOffsetPeriod",
        ("FIX.5.0", "41548"): "41548 LegPaymentScheduleFixingFirstObservationDateOffsetUnit",
        ("FIX.5.0", "41549"): "41549 LegPaymentStreamFlatRateIndicator",
        ("FIX.5.0", "41550"): "41550 LegPaymentStreamFlatRateAmount",
        ("FIX.5.0", "41551"): "41551 LegPaymentStreamFlatRateCurrency",
        ("FIX.5.0", "41552"): "41552 LegStreamMaximumPaymentAmount",
        ("FIX.5.0", "41553"): "41553 LegStreamMaximumPaymentCurrency",
        ("FIX.5.0", "41554"): "41554 LegStreamMaximumTransactionAmount",
        ("FIX.5.0", "41555"): "41555 LegStreamMaximumTransactionCurrency",
        ("FIX.5.0", "41556"): "41556 LegPaymentStreamFixedAmountUnitOfMeasure",
        ("FIX.5.0", "41557"): "41557 LegPaymentStreamTotalFixedAmount",
        ("FIX.5.0", "41558"): "41558 LegPaymentStreamWorldScaleRate",
        ("FIX.5.0", "41559"): "41559 LegPaymentStreamContractPrice",
        ("FIX.5.0", "41560"): "41560 LegPaymentStreamContractPriceCurrency",
        ("FIX.5.0", "41561"): "41561 NoLegPaymentStreamPricingBusinessCenters",
        ("FIX.5.0", "41562"): "41562 LegPaymentStreamPricingBusinessCenter",
        ("FIX.5.0", "41563"): "41563 LegPaymentStreamRateIndex2CurveUnit",
        ("FIX.5.0", "41564"): "41564 LegPaymentStreamRateIndex2CurvePeriod",
        ("FIX.5.0", "41565"): "41565 LegPaymentStreamRateIndexLocation",
        ("FIX.5.0", "41566"): "41566 LegPaymentStreamRateIndexLevel",
        ("FIX.5.0", "41567"): "41567 LegPaymentStreamRateIndexUnitOfMeasure",
        ("FIX.5.0", "41568"): "41568 LegPaymentStreamSettlLevel",
        ("FIX.5.0", "41569"): "41569 LegPaymentStreamReferenceLevel",
        ("FIX.5.0", "41570"): "41570 LegPaymentStreamReferenceLevelUnitOfMeasure",
        ("FIX.5.0", "41571"): "41571 LegPaymentStreamReferenceLevelEqualsZeroIndicator",
        ("FIX.5.0", "41572"): "41572 LegPaymentStreamRateSpreadCurrency",
        ("FIX.5.0", "41573"): "41573 LegPaymentStreamRateSpreadUnitOfMeasure",
        ("FIX.5.0", "41574"): "41574 LegPaymentStreamRateConversionFactor",
        ("FIX.5.0", "41575"): "41575 LegPaymentStreamRateSpreadType",
        ("FIX.5.0", "41576"): "41576 LegPaymentStreamLastResetRate",
        ("FIX.5.0", "41577"): "41577 LegPaymentStreamFinalRate",
        ("FIX.5.0", "41578"): "41578 LegPaymentStreamCalculationLagPeriod",
        ("FIX.5.0", "41579"): "41579 LegPaymentStreamCalculationLagUnit",
        ("FIX.5.0", "41580"): "41580 LegPaymentStreamFirstObservationDateOffsetPeriod",
        ("FIX.5.0", "41581"): "41581 LegPaymentStreamFirstObservationDateOffsetUnit",
        ("FIX.5.0", "41582"): "41582 LegPaymentStreamPricingDayType",
        ("FIX.5.0", "41583"): "41583 LegPaymentStreamPricingDayDistribution",
        ("FIX.5.0", "41584"): "41584 LegPaymentStreamPricingDayCount",
        ("FIX.5.0", "41585"): "41585 LegPaymentStreamPricingBusinessCalendar",
        ("FIX.5.0", "41586"): "41586 LegPaymentStreamPricingBusinessDayConvention",
        ("FIX.5.0", "41587"): "41587 UnderlyingDeliveryStreamRiskApportionmentSource",
        ("FIX.5.0", "41588"): "41588 StreamCommoditySettlTimeType",
        ("FIX.5.0", "41589"): "41589 NoLegPaymentStreamPaymentDates",
        ("FIX.5.0", "41590"): "41590 LegPaymentStreamPaymentDate",
        ("FIX.5.0", "41591"): "41591 LegPaymentStreamPaymentDateType",
        ("FIX.5.0", "41592"): "41592 LegPaymentStreamMasterAgreementPaymentDatesIndicator",
        ("FIX.5.0", "41593"): "41593 NoLegPaymentStreamPricingDates",
        ("FIX.5.0", "41594"): "41594 LegPaymentStreamPricingDate",
        ("FIX.5.0", "41595"): "41595 LegPaymentStreamPricingDateType",
        ("FIX.5.0", "41596"): "41596 NoLegPaymentStreamPricingDays",
        ("FIX.5.0", "41597"): "41597 LegPaymentStreamPricingDayOfWeek",
        ("FIX.5.0", "41598"): "41598 LegPaymentStreamPricingDayNumber",
        ("FIX.5.0", "41599"): "41599 NoLegPhysicalSettlTerms",
        ("FIX.5.0", "41600"): "41600 LegPhysicalSettlTermXID",
        ("FIX.5.0", "41601"): "41601 LegPhysicalSettlCurency",
        ("FIX.5.0", "41602"): "41602 LegPhysicalSettlBusinessDays",
        ("FIX.5.0", "41603"): "41603 LegPhysicalSettlMaximumBusinessDays",
        ("FIX.5.0", "41604"): "41604 NoLegPhysicalSettlDeliverableObligations",
        ("FIX.5.0", "41605"): "41605 LegPhysicalSettlDeliverableObligationType",
        ("FIX.5.0", "41606"): "41606 LegPhysicalSettlDeliverableObligationValue",
        ("FIX.5.0", "41607"): "41607 NoLegPricingDateBusinessCenters",
        ("FIX.5.0", "41608"): "41608 LegPricingDateBusinessCenter",
        ("FIX.5.0", "41609"): "41609 LegPricingDateUnadjusted",
        ("FIX.5.0", "41610"): "41610 LegPricingDateBusinessDayConvention",
        ("FIX.5.0", "41611"): "41611 LegPricingDateAdjusted",
        ("FIX.5.0", "41612"): "41612 LegPricingTime",
        ("FIX.5.0", "41613"): "41613 LegPricingTimeBusinessCenter",
        ("FIX.5.0", "41614"): "41614 NoLegProtectionTermEventNewsSources",
        ("FIX.5.0", "41615"): "41615 LegProtectionTermEventNewsSource",
        ("FIX.5.0", "41616"): "41616 NoLegProtectionTerms",
        ("FIX.5.0", "41617"): "41617 LegProtectionTermXID",
        ("FIX.5.0", "41618"): "41618 LegProtectionTermNotional",
        ("FIX.5.0", "41619"): "41619 LegProtectionTermCurrency",
        ("FIX.5.0", "41620"): "41620 LegProtectionTermSellerNotifies",
        ("FIX.5.0", "41621"): "41621 LegProtectionTermBuyerNotifies",
        ("FIX.5.0", "41622"): "41622 LegProtectionTermEventBusinessCenter",
        ("FIX.5.0", "41623"): "41623 LegProtectionTermStandardSources",
        ("FIX.5.0", "41624"): "41624 LegProtectionTermEventMinimumSources",
        ("FIX.5.0", "41625"): "41625 NoLegProtectionTermEvents",
        ("FIX.5.0", "41626"): "41626 LegProtectionTermEventType",
        ("FIX.5.0", "41627"): "41627 LegProtectionTermEventValue",
        ("FIX.5.0", "41628"): "41628 LegProtectionTermEventCurrency",
        ("FIX.5.0", "41629"): "41629 LegProtectionTermEventPeriod",
        ("FIX.5.0", "41630"): "41630 LegProtectionTermEventUnit",
        ("FIX.5.0", "41631"): "41631 LegProtectionTermEventDayType",
        ("FIX.5.0", "41632"): "41632 LegProtectionTermEventRateSource",
        ("FIX.5.0", "41633"): "41633 NoLegProtectionTermEventQualifiers",
        ("FIX.5.0", "41634"): "41634 LegProtectionTermEventQualifier",
        ("FIX.5.0", "41635"): "41635 NoLegProtectionTermObligations",
        ("FIX.5.0", "41636"): "41636 LegProtectionTermObligationType",
        ("FIX.5.0", "41637"): "41637 LegProtectionTermObligationValue",
        ("FIX.5.0", "41638"): "41638 NoLegStreamCalculationPeriodDates",
        ("FIX.5.0", "41639"): "41639 LegStreamCalculationPeriodDate",
        ("FIX.5.0", "41640"): "41640 LegStreamCalculationPeriodDateType",
        ("FIX.5.0", "41641"): "41641 LegStreamCalculationPeriodDatesXID",
        ("FIX.5.0", "41642"): "41642 LegStreamCalculationPeriodDatesXIDRef",
        ("FIX.5.0", "41643"): "41643 LegStreamCalculationBalanceOfFirstPeriod",
        ("FIX.5.0", "41644"): "41644 LegStreamCalculationCorrectionPeriod",
        ("FIX.5.0", "41645"): "41645 LegStreamCalculationCorrectionUnit",
        ("FIX.5.0", "41646"): "41646 NoLegStreamCommoditySettlBusinessCenters",
        ("FIX.5.0", "41647"): "41647 LegStreamCommoditySettlBusinessCenter",
        ("FIX.5.0", "41648"): "41648 LegStreamCommodityBase",
        ("FIX.5.0", "41649"): "41649 LegStreamCommodityType",
        ("FIX.5.0", "41650"): "41650 LegStreamCommoditySecurityID",
        ("FIX.5.0", "41651"): "41651 LegStreamCommoditySecurityIDSource",
        ("FIX.5.0", "41652"): "41652 LegStreamCommodityDesc",
        ("FIX.5.0", "41653"): "41653 EncodedLegStreamCommodityDescLen",
        ("FIX.5.0", "41654"): "41654 EncodedLegStreamCommodityDesc",
        ("FIX.5.0", "41655"): "41655 LegStreamCommodityUnitOfMeasure",
        ("FIX.5.0", "41656"): "41656 LegStreamCommodityCurrency",
        ("FIX.5.0", "41657"): "41657 LegStreamCommodityExchange",
        ("FIX.5.0", "41658"): "41658 LegStreamCommodityRateSource",
        ("FIX.5.0", "41659"): "41659 LegStreamCommodityRateReferencePage",
        ("FIX.5.0", "41660"): "41660 LegStreamCommodityRateReferencePageHeading",
        ("FIX.5.0", "41661"): "41661 LegStreamDataProvider",
        ("FIX.5.0", "41662"): "41662 LegStreamCommodityPricingType",
        ("FIX.5.0", "41663"): "41663 LegStreamCommodityNearbySettlDayPeriod",
        ("FIX.5.0", "41664"): "41664 LegStreamCommodityNearbySettlDayUnit",
        ("FIX.5.0", "41665"): "41665 LegStreamCommoditySettlDateUnadjusted",
        ("FIX.5.0", "41666"): "41666 LegStreamCommoditySettlDateBusinessDayConvention",
        ("FIX.5.0", "41667"): "41667 LegStreamCommoditySettlDateAdjusted",
        ("FIX.5.0", "41668"): "41668 LegStreamCommoditySettlMonth",
        ("FIX.5.0", "41669"): "41669 LegStreamCommoditySettlDateRollPeriod",
        ("FIX.5.0", "41670"): "41670 LegStreamCommoditySettlDateRollUnit",
        ("FIX.5.0", "41671"): "41671 LegStreamCommoditySettlDayType",
        ("FIX.5.0", "41672"): "41672 LegStreamCommodityXID",
        ("FIX.5.0", "41673"): "41673 LegStreamCommodityXIDRef",
        ("FIX.5.0", "41674"): "41674 NoLegStreamCommodityAltIDs",
        ("FIX.5.0", "41675"): "41675 LegStreamCommodityAltID",
        ("FIX.5.0", "41676"): "41676 LegStreamCommodityAltIDSource",
        ("FIX.5.0", "41677"): "41677 NoLegStreamCommodityDataSources",
        ("FIX.5.0", "41678"): "41678 LegStreamCommodityDataSourceID",
        ("FIX.5.0", "41679"): "41679 LegStreamCommodityDataSourceIDType",
        ("FIX.5.0", "41680"): "41680 NoLegStreamCommoditySettlDays",
        ("FIX.5.0", "41681"): "41681 LegStreamCommoditySettlDay",
        ("FIX.5.0", "41682"): "41682 LegStreamCommoditySettlTotalHours",
        ("FIX.5.0", "41683"): "41683 NoLegStreamCommoditySettlTimes",
        ("FIX.5.0", "41684"): "41684 LegStreamCommoditySettlStart",
        ("FIX.5.0", "41685"): "41685 LegStreamCommoditySettlEnd",
        ("FIX.5.0", "41686"): "41686 NoLegStreamCommoditySettlPeriods",
        ("FIX.5.0", "41687"): "41687 LegStreamCommoditySettlCountry",
        ("FIX.5.0", "41688"): "41688 LegStreamCommoditySettlTimeZone",
        ("FIX.5.0", "41689"): "41689 LegStreamCommoditySettlFlowType",
        ("FIX.5.0", "41690"): "41690 LegStreamCommoditySettlPeriodNotional",
        ("FIX.5.0", "41691"): "41691 LegStreamCommoditySettlPeriodNotionalUnitOfMeasure",
        ("FIX.5.0", "41692"): "41692 LegStreamCommoditySettlPeriodFrequencyPeriod",
        ("FIX.5.0", "41693"): "41693 LegStreamCommoditySettlPeriodFrequencyUnit",
        ("FIX.5.0", "41694"): "41694 LegStreamCommoditySettlPeriodPrice",
        ("FIX.5.0", "41695"): "41695 LegStreamCommoditySettlPeriodPriceUnitOfMeasure",
        ("FIX.5.0", "41696"): "41696 LegStreamCommoditySettlPeriodPriceCurrency",
        ("FIX.5.0", "41697"): "41697 LegStreamCommoditySettlHolidaysProcessingInstruction",
        ("FIX.5.0", "41698"): "41698 LegStreamCommoditySettlPeriodXID",
        ("FIX.5.0", "41699"): "41699 LegStreamCommoditySettlPeriodXIDRef",
        ("FIX.5.0", "41700"): "41700 LegStreamXID",
        ("FIX.5.0", "41701"): "41701 UnderlyingAdditionalTermBondSecurityIDSource",
        ("FIX.5.0", "41702"): "41702 LegStreamNotionalXIDRef",
        ("FIX.5.0", "41703"): "41703 LegStreamNotionalFrequencyPeriod",
        ("FIX.5.0", "41704"): "41704 LegStreamNotionalFrequencyUnit",
        ("FIX.5.0", "41705"): "41705 LegStreamNotionalCommodityFrequency",
        ("FIX.5.0", "41706"): "41706 LegStreamNotionalUnitOfMeasure",
        ("FIX.5.0", "41707"): "41707 LegStreamTotalNotional",
        ("FIX.5.0", "41708"): "41708 LegStreamTotalNotionalUnitOfMeasure",
        ("FIX.5.0", "41709"): "41709 UnderlyingAdditionalTermBondDesc",
        ("FIX.5.0", "41710"): "41710 EncodedUnderlyingAdditionalTermBondDescLen",
        ("FIX.5.0", "41711"): "41711 EncodedUnderlyingAdditionalTermBondDesc",
        ("FIX.5.0", "41712"): "41712 UnderlyingAdditionalTermBondCurrency",
        ("FIX.5.0", "41713"): "41713 NoUnderlyingComplexEventAveragingObservations",
        ("FIX.5.0", "41714"): "41714 UnderlyingComplexEventAveragingObservationNumber",
        ("FIX.5.0", "41715"): "41715 UnderlyingComplexEventAveragingWeight",
        ("FIX.5.0", "41716"): "41716 NoUnderlyingComplexEventCreditEvents",
        ("FIX.5.0", "41717"): "41717 UnderlyingComplexEventCreditEventType",
        ("FIX.5.0", "41718"): "41718 UnderlyingComplexEventCreditEventValue",
        ("FIX.5.0", "41719"): "41719 UnderlyingComplexEventCreditEventCurrency",
        ("FIX.5.0", "41720"): "41720 UnderlyingComplexEventCreditEventPeriod",
        ("FIX.5.0", "41721"): "41721 UnderlyingComplexEventCreditEventUnit",
        ("FIX.5.0", "41722"): "41722 UnderlyingComplexEventCreditEventDayType",
        ("FIX.5.0", "41723"): "41723 UnderlyingComplexEventCreditEventRateSource",
        ("FIX.5.0", "41724"): "41724 NoUnderlyingComplexEventCreditEventQualifiers",
        ("FIX.5.0", "41725"): "41725 UnderlyingComplexEventCreditEventQualifier",
        ("FIX.5.0", "41726"): "41726 NoUnderlyingComplexEventPeriodDateTimes",
        ("FIX.5.0", "41727"): "41727 UnderlyingComplexEventPeriodDate",
        ("FIX.5.0", "41728"): "41728 UnderlyingComplexEventPeriodTime",
        ("FIX.5.0", "41729"): "41729 NoUnderlyingComplexEventPeriods",
        ("FIX.5.0", "41730"): "41730 UnderlyingComplexEventPeriodType",
        ("FIX.5.0", "41731"): "41731 UnderlyingComplexEventBusinessCenter",
        ("FIX.5.0", "41732"): "41732 NoUnderlyingComplexEventRateSources",
        ("FIX.5.0", "41733"): "41733 UnderlyingComplexEventRateSource",
        ("FIX.5.0", "41734"): "41734 UnderlyingComplexEventRateSourceType",
        ("FIX.5.0", "41735"): "41735 UnderlyingComplexEventReferencePage",
        ("FIX.5.0", "41736"): "41736 UnderlyingComplexEventReferencePageHeading",
        ("FIX.5.0", "41737"): "41737 NoUnderlyingComplexEventDateBusinessCenters",
        ("FIX.5.0", "41738"): "41738 UnderlyingComplexEventDateBusinessCenter",
        ("FIX.5.0", "41739"): "41739 UnderlyingComplexEventDateUnadjusted",
        ("FIX.5.0", "41740"): "41740 UnderlyingComplexEventDateRelativeTo",
        ("FIX.5.0", "41741"): "41741 UnderlyingComplexEventDateOffsetPeriod",
        ("FIX.5.0", "41742"): "41742 UnderlyingComplexEventDateOffsetUnit",
        ("FIX.5.0", "41743"): "41743 UnderlyingComplexEventDateOffsetDayType",
        ("FIX.5.0", "41744"): "41744 UnderlyingComplexEventDateBusinessDayConvention",
        ("FIX.5.0", "41745"): "41745 UnderlyingComplexEventDateAdjusted",
        ("FIX.5.0", "41746"): "41746 UnderlyingComplexEventFixingTime",
        ("FIX.5.0", "41747"): "41747 UnderlyingComplexEventFixingTimeBusinessCenter",
        ("FIX.5.0", "41748"): "41748 NoUnderlyingComplexEventCreditEventSources",
        ("FIX.5.0", "41749"): "41749 UnderlyingComplexEventCreditEventSource",
        ("FIX.5.0", "41750"): "41750 NoUnderlyingComplexEventSchedules",
        ("FIX.5.0", "41751"): "41751 UnderlyingComplexEventScheduleStartDate",
        ("FIX.5.0", "41752"): "41752 UnderlyingComplexEventScheduleEndDate",
        ("FIX.5.0", "41753"): "41753 UnderlyingComplexEventScheduleFrequencyPeriod",
        ("FIX.5.0", "41754"): "41754 UnderlyingComplexEventScheduleFrequencyUnit",
        ("FIX.5.0", "41755"): "41755 UnderlyingComplexEventScheduleRollConvention",
        ("FIX.5.0", "41756"): "41756 NoUnderlyingDeliverySchedules",
        ("FIX.5.0", "41757"): "41757 UnderlyingDeliveryScheduleType",
        ("FIX.5.0", "41758"): "41758 UnderlyingDeliveryScheduleXID",
        ("FIX.5.0", "41759"): "41759 UnderlyingDeliveryScheduleNotional",
        ("FIX.5.0", "41760"): "41760 UnderlyingDeliveryScheduleNotionalUnitOfMeasure",
        ("FIX.5.0", "41761"): "41761 UnderlyingDeliveryScheduleNotionalCommodityFrequency",
        ("FIX.5.0", "41762"): "41762 UnderlyingDeliveryScheduleNegativeTolerance",
        ("FIX.5.0", "41763"): "41763 UnderlyingDeliverySchedulePositiveTolerance",
        ("FIX.5.0", "41764"): "41764 UnderlyingDeliveryScheduleToleranceUnitOfMeasure",
        ("FIX.5.0", "41765"): "41765 UnderlyingDeliveryScheduleToleranceType",
        ("FIX.5.0", "41766"): "41766 UnderlyingDeliveryScheduleSettlCountry",
        ("FIX.5.0", "41767"): "41767 UnderlyingDeliveryScheduleSettlTimeZone",
        ("FIX.5.0", "41768"): "41768 UnderlyingDeliveryScheduleSettlFlowType",
        ("FIX.5.0", "41769"): "41769 UnderlyingDeliveryScheduleSettlHolidaysProcessingInstruction",
        ("FIX.5.0", "41770"): "41770 NoUnderlyingDeliveryScheduleSettlDays",
        ("FIX.5.0", "41771"): "41771 UnderlyingDeliveryScheduleSettlDay",
        ("FIX.5.0", "41772"): "41772 UnderlyingDeliveryScheduleSettlTotalHours",
        ("FIX.5.0", "41773"): "41773 NoUnderlyingDeliveryScheduleSettlTimes",
        ("FIX.5.0", "41774"): "41774 UnderlyingDeliveryScheduleSettlStart",
        ("FIX.5.0", "41775"): "41775 UnderlyingDeliveryScheduleSettlEnd",
        ("FIX.5.0", "41776"): "41776 UnderlyingDeliveryScheduleSettlTimeType",
        ("FIX.5.0", "41777"): "41777 UnderlyingDeliveryStreamType",
        ("FIX.5.0", "41778"): "41778 UnderlyingDeliveryStreamPipeline",
        ("FIX.5.0", "41779"): "41779 UnderlyingDeliveryStreamEntryPoint",
        ("FIX.5.0", "41780"): "41780 UnderlyingDeliveryStreamWithdrawalPoint",
        ("FIX.5.0", "41781"): "41781 UnderlyingDeliveryStreamDeliveryPoint",
        ("FIX.5.0", "41782"): "41782 UnderlyingDeliveryStreamDeliveryRestriction",
        ("FIX.5.0", "41783"): "41783 UnderlyingDeliveryStreamDeliveryContingency",
        ("FIX.5.0", "41784"): "41784 UnderlyingDeliveryStreamDeliveryContingentPartySide",
        ("FIX.5.0", "41785"): "41785 UnderlyingDeliveryStreamDeliverAtSourceIndicator",
        ("FIX.5.0", "41786"): "41786 UnderlyingDeliveryStreamRiskApportionment",
        ("FIX.5.0", "41787"): "41787 UnderlyingDeliveryStreamTitleTransferLocation",
        ("FIX.5.0", "41788"): "41788 UnderlyingDeliveryStreamTitleTransferCondition",
        ("FIX.5.0", "41789"): "41789 UnderlyingDeliveryStreamImporterOfRecord",
        ("FIX.5.0", "41790"): "41790 UnderlyingDeliveryStreamNegativeTolerance",
        ("FIX.5.0", "41791"): "41791 UnderlyingDeliveryStreamPositiveTolerance",
        ("FIX.5.0", "41792"): "41792 UnderlyingDeliveryStreamToleranceUnitOfMeasure",
        ("FIX.5.0", "41793"): "41793 UnderlyingDeliveryStreamToleranceType",
        ("FIX.5.0", "41794"): "41794 UnderlyingDeliveryStreamToleranceOptionSide",
        ("FIX.5.0", "41795"): "41795 UnderlyingDeliveryStreamTotalPositiveTolerance",
        ("FIX.5.0", "41796"): "41796 UnderlyingDeliveryStreamTotalNegativeTolerance",
        ("FIX.5.0", "41797"): "41797 UnderlyingDeliveryStreamNotionalConversionFactor",
        ("FIX.5.0", "41798"): "41798 UnderlyingDeliveryStreamTransportEquipment",
        ("FIX.5.0", "41799"): "41799 UnderlyingDeliveryStreamElectingPartySide",
        ("FIX.5.0", "41800"): "41800 NoUnderlyingStreamAssetAttributes",
        ("FIX.5.0", "41801"): "41801 UnderlyingStreamAssetAttributeType",
        ("FIX.5.0", "41802"): "41802 UnderlyingStreamAssetAttributeValue",
        ("FIX.5.0", "41803"): "41803 UnderlyingStreamAssetAttributeLimit",
        ("FIX.5.0", "41804"): "41804 NoUnderlyingDeliveryStreamCycles",
        ("FIX.5.0", "41805"): "41805 UnderlyingDeliveryStreamCycleDesc",
        ("FIX.5.0", "41806"): "41806 EncodedUnderlyingDeliveryStreamCycleDescLen",
        ("FIX.5.0", "41807"): "41807 EncodedUnderlyingDeliveryStreamCycleDesc",
        ("FIX.5.0", "41808"): "41808 NoUnderlyingDeliveryStreamCommoditySources",
        ("FIX.5.0", "41809"): "41809 UnderlyingDeliveryStreamCommoditySource",
        ("FIX.5.0", "41810"): "41810 UnderlyingExerciseDesc",
        ("FIX.5.0", "41811"): "41811 EncodedUnderlyingExerciseDescLen",
        ("FIX.5.0", "41812"): "41812 EncodedUnderlyingExerciseDesc",
        ("FIX.5.0", "41813"): "41813 UnderlyingAutomaticExerciseIndicator",
        ("FIX.5.0", "41814"): "41814 UnderlyingAutomaticExerciseThresholdRate",
        ("FIX.5.0", "41815"): "41815 UnderlyingExerciseConfirmationMethod",
        ("FIX.5.0", "41816"): "41816 UnderlyingManualNoticeBusinessCenter",
        ("FIX.5.0", "41817"): "41817 UnderlyingFallbackExerciseIndicator",
        ("FIX.5.0", "41818"): "41818 UnderlyingLimitedRightToConfirmIndicator",
        ("FIX.5.0", "41819"): "41819 UnderlyingExerciseSplitTicketIndicator",
        ("FIX.5.0", "41820"): "41820 NoUnderlyingOptionExerciseBusinessCenters",
        ("FIX.5.0", "41821"): "41821 UnderlyingOptionExerciseBusinessCenter",
        ("FIX.5.0", "41822"): "41822 UnderlyingOptionExerciseBusinessDayConvention",
        ("FIX.5.0", "41823"): "41823 UnderlyingOptionExerciseEarliestDateOffsetDayType",
        ("FIX.5.0", "41824"): "41824 UnderlyingOptionExerciseEarliestDateOffsetPeriod",
        ("FIX.5.0", "41825"): "41825 UnderlyingOptionExerciseEarliestDateOffsetUnit",
        ("FIX.5.0", "41826"): "41826 UnderlyingOptionExerciseFrequencyPeriod",
        ("FIX.5.0", "41827"): "41827 UnderlyingOptionExerciseFrequencyUnit",
        ("FIX.5.0", "41828"): "41828 UnderlyingOptionExerciseStartDateUnadjusted",
        ("FIX.5.0", "41829"): "41829 UnderlyingOptionExerciseStartDateRelativeTo",
        ("FIX.5.0", "41830"): "41830 UnderlyingOptionExerciseStartDateOffsetPeriod",
        ("FIX.5.0", "41831"): "41831 UnderlyingOptionExerciseStartDateOffsetUnit",
        ("FIX.5.0", "41832"): "41832 UnderlyingOptionExerciseStartDateOffsetDayType",
        ("FIX.5.0", "41833"): "41833 UnderlyingOptionExerciseStartDateAdjusted",
        ("FIX.5.0", "41834"): "41834 UnderlyingOptionExerciseSkip",
        ("FIX.5.0", "41835"): "41835 UnderlyingOptionExerciseNominationDeadline",
        ("FIX.5.0", "41836"): "41836 UnderlyingOptionExerciseFirstDateUnadjusted",
        ("FIX.5.0", "41837"): "41837 UnderlyingOptionExerciseLastDateUnadjusted",
        ("FIX.5.0", "41838"): "41838 UnderlyingOptionExerciseEarliestTime",
        ("FIX.5.0", "41839"): "41839 UnderlyingOptionExerciseLatestTime",
        ("FIX.5.0", "41840"): "41840 UnderlyingOptionExerciseTimeBusinessCenter",
        ("FIX.5.0", "41841"): "41841 NoUnderlyingOptionExerciseDates",
        ("FIX.5.0", "41842"): "41842 UnderlyingOptionExerciseDate",
        ("FIX.5.0", "41843"): "41843 UnderlyingOptionExerciseDateType",
        ("FIX.5.0", "41844"): "41844 NoUnderlyingOptionExerciseExpirationDateBusinessCenters",
        ("FIX.5.0", "41845"): "41845 UnderlyingOptionExerciseExpirationDateBusinessCenter",
        ("FIX.5.0", "41846"): "41846 UnderlyingOptionExerciseExpirationDateBusinessDayConvention",
        ("FIX.5.0", "41847"): "41847 UnderlyingOptionExerciseExpirationDateRelativeTo",
        ("FIX.5.0", "41848"): "41848 UnderlyingOptionExerciseExpirationDateOffsetPeriod",
        ("FIX.5.0", "41849"): "41849 UnderlyingOptionExerciseExpirationDateOffsetUnit",
        ("FIX.5.0", "41850"): "41850 UnderlyingOptionExerciseExpirationFrequencyPeriod",
        ("FIX.5.0", "41851"): "41851 UnderlyingOptionExerciseExpirationFrequencyUnit",
        ("FIX.5.0", "41852"): "41852 UnderlyingOptionExerciseExpirationRollConvention",
        ("FIX.5.0", "41853"): "41853 UnderlyingOptionExerciseExpirationDateOffsetDayType",
        ("FIX.5.0", "41854"): "41854 UnderlyingOptionExerciseExpirationTime",
        ("FIX.5.0", "41855"): "41855 UnderlyingOptionExerciseExpirationTimeBusinessCenter",
        ("FIX.5.0", "41856"): "41856 NoUnderlyingOptionExerciseExpirationDates",
        ("FIX.5.0", "41857"): "41857 UnderlyingOptionExerciseExpirationDate",
        ("FIX.5.0", "41858"): "41858 UnderlyingOptionExerciseExpirationDateType",
        ("FIX.5.0", "41859"): "41859 UnderlyingMarketDisruptionProvision",
        ("FIX.5.0", "41860"): "41860 UnderlyingMarketDisruptionFallbackProvision",
        ("FIX.5.0", "41861"): "41861 UnderlyingMarketDisruptionMaximumDays",
        ("FIX.5.0", "41862"): "41862 UnderlyingMarketDisruptionMaterialityPercentage",
        ("FIX.5.0", "41863"): "41863 UnderlyingMarketDisruptionMinimumFuturesContracts",
        ("FIX.5.0", "41864"): "41864 NoUnderlyingMarketDisruptionEvents",
        ("FIX.5.0", "41865"): "41865 UnderlyingMarketDisruptionEvent",
        ("FIX.5.0", "41866"): "41866 NoUnderlyingMarketDisruptionFallbacks",
        ("FIX.5.0", "41867"): "41867 UnderlyingMarketDisruptionFallbackType",
        ("FIX.5.0", "41868"): "41868 NoUnderlyingMarketDisruptionFallbackReferencePrices",
        ("FIX.5.0", "41869"): "41869 UnderlyingMarketDisruptionFallbackUnderlierType",
        ("FIX.5.0", "41870"): "41870 UnderlyingMarketDisruptionFallbackUnderlierSecurityID",
        ("FIX.5.0", "41871"): "41871 UnderlyingMarketDisruptionFallbackUnderlierSecurityIDSource",
        ("FIX.5.0", "41872"): "41872 UnderlyingMarketDisruptionFallbackUnderlierSecurityDesc",
        ("FIX.5.0", "41873"): "41873 EncodedUnderlyingMarketDisruptionFallbackUnderlierSecurityDescLen",
        ("FIX.5.0", "41874"): "41874 EncodedUnderlyingMarketDisruptionFallbackUnderlierSecurityDesc",
        ("FIX.5.0", "41875"): "41875 UnderlyingMarketDisruptionFallbackOpenUnits",
        ("FIX.5.0", "41876"): "41876 UnderlyingMarketDisruptionFallbackBasketCurrency",
        ("FIX.5.0", "41877"): "41877 UnderlyingMarketDisruptionFallbackBasketDivisor",
        ("FIX.5.0", "41878"): "41878 NoUnderlyingPaymentScheduleFixingDays",
        ("FIX.5.0", "41879"): "41879 UnderlyingPaymentScheduleFixingDayOfWeek",
        ("FIX.5.0", "41880"): "41880 UnderlyingPaymentScheduleFixingDayNumber",
        ("FIX.5.0", "41881"): "41881 UnderlyingPaymentScheduleXID",
        ("FIX.5.0", "41882"): "41882 UnderlyingPaymentScheduleXIDRef",
        ("FIX.5.0", "41883"): "41883 UnderlyingPaymentScheduleRateCurrency",
        ("FIX.5.0", "41884"): "41884 UnderlyingPaymentScheduleRateUnitOfMeasure",
        ("FIX.5.0", "41885"): "41885 UnderlyingPaymentScheduleRateConversionFactor",
        ("FIX.5.0", "41886"): "41886 UnderlyingPaymentScheduleRateSpreadType",
        ("FIX.5.0", "41887"): "41887 UnderlyingPaymentScheduleSettlPeriodPrice",
        ("FIX.5.0", "41888"): "41888 UnderlyingPaymentScheduleSettlPeriodPriceCurrency",
        ("FIX.5.0", "41889"): "41889 UnderlyingPaymentScheduleSettlPeriodPriceUnitOfMeasure",
        ("FIX.5.0", "41890"): "41890 UnderlyingPaymentScheduleStepUnitOfMeasure",
        ("FIX.5.0", "41891"): "41891 UnderlyingPaymentScheduleFixingDayDistribution",
        ("FIX.5.0", "41892"): "41892 UnderlyingPaymentScheduleFixingDayCount",
        ("FIX.5.0", "41893"): "41893 UnderlyingPaymentScheduleFixingLagPeriod",
        ("FIX.5.0", "41894"): "41894 UnderlyingPaymentScheduleFixingLagUnit",
        ("FIX.5.0", "41895"): "41895 UnderlyingPaymentScheduleFixingFirstObservationDateOffsetPeriod",
        ("FIX.5.0", "41896"): "41896 UnderlyingPaymentScheduleFixingFirstObservationDateOffsetUnit",
        ("FIX.5.0", "41897"): "41897 UnderlyingPaymentStreamFlatRateIndicator",
        ("FIX.5.0", "41898"): "41898 UnderlyingPaymentStreamFlatRateAmount",
        ("FIX.5.0", "41899"): "41899 UnderlyingPaymentStreamFlatRateCurrency",
        ("FIX.5.0", "41900"): "41900 UnderlyingPaymentStreamMaximumPaymentAmount",
        ("FIX.5.0", "41901"): "41901 UnderlyingPaymentStreamMaximumPaymentCurrency",
        ("FIX.5.0", "41902"): "41902 UnderlyingPaymentStreamMaximumTransactionAmount",
        ("FIX.5.0", "41903"): "41903 UnderlyingPaymentStreamMaximumTransactionCurrency",
        ("FIX.5.0", "41904"): "41904 UnderlyingPaymentStreamFixedAmountUnitOfMeasure",
        ("FIX.5.0", "41905"): "41905 UnderlyingPaymentStreamTotalFixedAmount",
        ("FIX.5.0", "41906"): "41906 UnderlyingPaymentStreamWorldScaleRate",
        ("FIX.5.0", "41907"): "41907 UnderlyingPaymentStreamContractPrice",
        ("FIX.5.0", "41908"): "41908 UnderlyingPaymentStreamContractPriceCurrency",
        ("FIX.5.0", "41909"): "41909 NoUnderlyingPaymentStreamPricingBusinessCenters",
        ("FIX.5.0", "41910"): "41910 UnderlyingPaymentStreamPricingBusinessCenter",
        ("FIX.5.0", "41911"): "41911 UnderlyingPaymentStreamRateIndex2CurveUnit",
        ("FIX.5.0", "41912"): "41912 UnderlyingPaymentStreamRateIndex2CurvePeriod",
        ("FIX.5.0", "41913"): "41913 UnderlyingPaymentStreamRateIndexLocation",
        ("FIX.5.0", "41914"): "41914 UnderlyingPaymentStreamRateIndexLevel",
        ("FIX.5.0", "41915"): "41915 UnderlyingPaymentStreamRateIndexUnitOfMeasure",
        ("FIX.5.0", "41916"): "41916 UnderlyingPaymentStreamSettlLevel",
        ("FIX.5.0", "41917"): "41917 UnderlyingPaymentStreamReferenceLevel",
        ("FIX.5.0", "41918"): "41918 UnderlyingPaymentStreamReferenceLevelUnitOfMeasure",
        ("FIX.5.0", "41919"): "41919 UnderlyingPaymentStreamReferenceLevelEqualsZeroIndicator",
        ("FIX.5.0", "41920"): "41920 UnderlyingPaymentStreamRateSpreadCurrency",
        ("FIX.5.0", "41921"): "41921 UnderlyingPaymentStreamRateSpreadUnitOfMeasure",
        ("FIX.5.0", "41922"): "41922 UnderlyingPaymentStreamRateConversionFactor",
        ("FIX.5.0", "41923"): "41923 UnderlyingPaymentStreamRateSpreadType",
        ("FIX.5.0", "41924"): "41924 UnderlyingPaymentStreamLastResetRate",
        ("FIX.5.0", "41925"): "41925 UnderlyingPaymentStreamFinalRate",
        ("FIX.5.0", "41926"): "41926 UnderlyingPaymentStreamCalculationLagPeriod",
        ("FIX.5.0", "41927"): "41927 UnderlyingPaymentStreamCalculationLagUnit",
        ("FIX.5.0", "41928"): "41928 UnderlyingPaymentStreamFirstObservationDateOffsetPeriod",
        ("FIX.5.0", "41929"): "41929 UnderlyingPaymentStreamFirstObservationDateOffsetUnit",
        ("FIX.5.0", "41930"): "41930 UnderlyingPaymentStreamPricingDayType",
        ("FIX.5.0", "41931"): "41931 UnderlyingPaymentStreamPricingDayDistribution",
        ("FIX.5.0", "41932"): "41932 UnderlyingPaymentStreamPricingDayCount",
        ("FIX.5.0", "41933"): "41933 UnderlyingPaymentStreamPricingBusinessCalendar",
        ("FIX.5.0", "41934"): "41934 UnderlyingPaymentStreamPricingBusinessDayConvention",
        ("FIX.5.0", "41935"): "41935 LegStreamCommoditySettlTimeType",
        ("FIX.5.0", "41936"): "41936 UnderlyingStreamCommoditySettlTimeType",
        ("FIX.5.0", "41937"): "41937 NoUnderlyingPaymentStreamPaymentDates",
        ("FIX.5.0", "41938"): "41938 UnderlyingPaymentStreamPaymentDate",
        ("FIX.5.0", "41939"): "41939 UnderlyingPaymentStreamPaymentDateType",
        ("FIX.5.0", "41940"): "41940 UnderlyingPaymentStreamMasterAgreementPaymentDatesIndicator",
        ("FIX.5.0", "41941"): "41941 NoUnderlyingPaymentStreamPricingDates",
        ("FIX.5.0", "41942"): "41942 UnderlyingPaymentStreamPricingDate",
        ("FIX.5.0", "41943"): "41943 UnderlyingPaymentStreamPricingDateType",
        ("FIX.5.0", "41944"): "41944 NoUnderlyingPaymentStreamPricingDays",
        ("FIX.5.0", "41945"): "41945 UnderlyingPaymentStreamPricingDayOfWeek",
        ("FIX.5.0", "41946"): "41946 UnderlyingPaymentStreamPricingDayNumber",
        ("FIX.5.0", "41947"): "41947 NoUnderlyingPricingDateBusinessCenters",
        ("FIX.5.0", "41948"): "41948 UnderlyingPricingDateBusinessCenter",
        ("FIX.5.0", "41949"): "41949 UnderlyingPricingDateUnadjusted",
        ("FIX.5.0", "41950"): "41950 UnderlyingPricingDateBusinessDayConvention",
        ("FIX.5.0", "41951"): "41951 UnderlyingPricingDateAdjusted",
        ("FIX.5.0", "41952"): "41952 UnderlyingPricingTime",
        ("FIX.5.0", "41953"): "41953 UnderlyingPricingTimeBusinessCenter",
        ("FIX.5.0", "41954"): "41954 NoUnderlyingStreamCalculationPeriodDates",
        ("FIX.5.0", "41955"): "41955 UnderlyingStreamCalculationPeriodDate",
        ("FIX.5.0", "41956"): "41956 UnderlyingStreamCalculationPeriodDateType",
        ("FIX.5.0", "41957"): "41957 UnderlyingStreamCalculationPeriodDatesXID",
        ("FIX.5.0", "41958"): "41958 UnderlyingStreamCalculationPeriodDatesXIDRef",
        ("FIX.5.0", "41959"): "41959 UnderlyingStreamCalculationBalanceOfFirstPeriod",
        ("FIX.5.0", "41960"): "41960 UnderlyingStreamCalculationCorrectionPeriod",
        ("FIX.5.0", "41961"): "41961 UnderlyingStreamCalculationCorrectionUnit",
        ("FIX.5.0", "41962"): "41962 NoUnderlyingStreamCommoditySettlBusinessCenters",
        ("FIX.5.0", "41963"): "41963 UnderlyingStreamCommoditySettlBusinessCenter",
        ("FIX.5.0", "41964"): "41964 UnderlyingStreamCommodityBase",
        ("FIX.5.0", "41965"): "41965 UnderlyingStreamCommodityType",
        ("FIX.5.0", "41966"): "41966 UnderlyingStreamCommoditySecurityID",
        ("FIX.5.0", "41967"): "41967 UnderlyingStreamCommoditySecurityIDSource",
        ("FIX.5.0", "41968"): "41968 UnderlyingStreamCommodityDesc",
        ("FIX.5.0", "41969"): "41969 EncodedUnderlyingStreamCommodityDescLen",
        ("FIX.5.0", "41970"): "41970 EncodedUnderlyingStreamCommodityDesc",
        ("FIX.5.0", "41971"): "41971 UnderlyingStreamCommodityUnitOfMeasure",
        ("FIX.5.0", "41972"): "41972 UnderlyingStreamCommodityCurrency",
        ("FIX.5.0", "41973"): "41973 UnderlyingStreamCommodityExchange",
        ("FIX.5.0", "41974"): "41974 UnderlyingStreamCommodityRateSource",
        ("FIX.5.0", "41975"): "41975 UnderlyingStreamCommodityRateReferencePage",
        ("FIX.5.0", "41976"): "41976 UnderlyingStreamCommodityRateReferencePageHeading",
        ("FIX.5.0", "41977"): "41977 UnderlyingStreamDataProvider",
        ("FIX.5.0", "41978"): "41978 UnderlyingStreamCommodityPricingType",
        ("FIX.5.0", "41979"): "41979 UnderlyingStreamCommodityNearbySettlDayPeriod",
        ("FIX.5.0", "41980"): "41980 UnderlyingStreamCommodityNearbySettlDayUnit",
        ("FIX.5.0", "41981"): "41981 UnderlyingStreamCommoditySettlDateUnadjusted",
        ("FIX.5.0", "41982"): "41982 UnderlyingStreamCommoditySettlDateBusinessDayConvention",
        ("FIX.5.0", "41983"): "41983 UnderlyingStreamCommoditySettlDateAdjusted",
        ("FIX.5.0", "41984"): "41984 UnderlyingStreamCommoditySettlMonth",
        ("FIX.5.0", "41985"): "41985 UnderlyingStreamCommoditySettlDateRollPeriod",
        ("FIX.5.0", "41986"): "41986 UnderlyingStreamCommoditySettlDateRollUnit",
        ("FIX.5.0", "41987"): "41987 UnderlyingStreamCommoditySettlDayType",
        ("FIX.5.0", "41988"): "41988 UnderlyingStreamCommodityXID",
        ("FIX.5.0", "41989"): "41989 UnderlyingStreamCommodityXIDRef",
        ("FIX.5.0", "41990"): "41990 NoUnderlyingStreamCommodityAltIDs",
        ("FIX.5.0", "41991"): "41991 UnderlyingStreamCommodityAltID",
        ("FIX.5.0", "41992"): "41992 UnderlyingStreamCommodityAltIDSource",
        ("FIX.5.0", "41993"): "41993 NoUnderlyingStreamCommodityDataSources",
        ("FIX.5.0", "41994"): "41994 UnderlyingStreamCommodityDataSourceID",
        ("FIX.5.0", "41995"): "41995 UnderlyingStreamCommodityDataSourceIDType",
        ("FIX.5.0", "41996"): "41996 NoUnderlyingStreamCommoditySettlDays",
        ("FIX.5.0", "41997"): "41997 UnderlyingStreamCommoditySettlDay",
        ("FIX.5.0", "41998"): "41998 UnderlyingStreamCommoditySettlTotalHours",
        ("FIX.5.0", "41999"): "41999 NoUnderlyingStreamCommoditySettlTimes",
        ("FIX.5.0", "42000"): "42000 UnderlyingStreamCommoditySettlStart",
        ("FIX.5.0", "42001"): "42001 UnderlyingStreamCommoditySettlEnd",
        ("FIX.5.0", "42002"): "42002 NoUnderlyingStreamCommoditySettlPeriods",
        ("FIX.5.0", "42003"): "42003 UnderlyingStreamCommoditySettlCountry",
        ("FIX.5.0", "42004"): "42004 UnderlyingStreamCommoditySettlTimeZone",
        ("FIX.5.0", "42005"): "42005 UnderlyingStreamCommoditySettlFlowType",
        ("FIX.5.0", "42006"): "42006 UnderlyingStreamCommoditySettlPeriodNotional",
        ("FIX.5.0", "42007"): "42007 UnderlyingStreamCommoditySettlPeriodNotionalUnitOfMeasure",
        ("FIX.5.0", "42008"): "42008 UnderlyingStreamCommoditySettlPeriodFrequencyPeriod",
        ("FIX.5.0", "42009"): "42009 UnderlyingStreamCommoditySettlPeriodFrequencyUnit",
        ("FIX.5.0", "42010"): "42010 UnderlyingStreamCommoditySettlPeriodPrice",
        ("FIX.5.0", "42011"): "42011 UnderlyingStreamCommoditySettlPeriodPriceUnitOfMeasure",
        ("FIX.5.0", "42012"): "42012 UnderlyingStreamCommoditySettlPeriodPriceCurrency",
        ("FIX.5.0", "42013"): "42013 UnderlyingStreamCommoditySettlHolidaysProcessingInstruction",
        ("FIX.5.0", "42014"): "42014 UnderlyingStreamCommoditySettlPeriodXID",
        ("FIX.5.0", "42015"): "42015 UnderlyingStreamCommoditySettlPeriodXIDRef",
        ("FIX.5.0", "42016"): "42016 UnderlyingStreamXID",
        ("FIX.5.0", "42017"): "42017 UnderlyingAdditionalTermBondIssuer",
        ("FIX.5.0", "42018"): "42018 UnderlyingStreamNotionalXIDRef",
        ("FIX.5.0", "42019"): "42019 UnderlyingStreamNotionalFrequencyPeriod",
        ("FIX.5.0", "42020"): "42020 UnderlyingStreamNotionalFrequencyUnit",
        ("FIX.5.0", "42021"): "42021 UnderlyingStreamNotionalCommodityFrequency",
        ("FIX.5.0", "42022"): "42022 UnderlyingStreamNotionalUnitOfMeasure",
        ("FIX.5.0", "42023"): "42023 UnderlyingStreamTotalNotional",
        ("FIX.5.0", "42024"): "42024 UnderlyingStreamTotalNotionalUnitOfMeasure",
        ("FIX.5.0", "42025"): "42025 EncodedUnderlyingAdditionalTermBondIssuerLen",
        ("FIX.5.0", "42026"): "42026 EncodedUnderlyingAdditionalTermBondIssuer",
        ("FIX.5.0", "42027"): "42027 UnderlyingAdditionalTermBondSeniority",
        ("FIX.5.0", "42028"): "42028 UnderlyingAdditionalTermBondCouponType",
        ("FIX.5.0", "42029"): "42029 UnderlyingAdditionalTermBondCouponRate",
        ("FIX.5.0", "42030"): "42030 UnderlyingAdditionalTermBondMaturityDate",
        ("FIX.5.0", "42031"): "42031 UnderlyingAdditionalTermBondParValue",
        ("FIX.5.0", "42032"): "42032 UnderlyingAdditionalTermBondCurrentTotalIssuedAmount",
        ("FIX.5.0", "42033"): "42033 UnderlyingAdditionalTermBondCouponFrequencyPeriod",
        ("FIX.5.0", "42034"): "42034 UnderlyingAdditionalTermBondCouponFrequencyUnit",
        ("FIX.5.0", "42035"): "42035 UnderlyingAdditionalTermBondDayCount",
        ("FIX.5.0", "42036"): "42036 NoUnderlyingAdditionalTerms",
        ("FIX.5.0", "42037"): "42037 UnderlyingAdditionalTermConditionPrecedentBondIndicator",
        ("FIX.5.0", "42038"): "42038 UnderlyingAdditionalTermDiscrepancyClauseIndicator",
        ("FIX.5.0", "42039"): "42039 NoUnderlyingCashSettlDealers",
        ("FIX.5.0", "42040"): "42040 UnderlyingCashSettlDealer",
        ("FIX.5.0", "42041"): "42041 NoUnderlyingCashSettlTerms",
        ("FIX.5.0", "42042"): "42042 UnderlyingCashSettlCurrency",
        ("FIX.5.0", "42043"): "42043 UnderlyingCashSettlValuationFirstBusinessDayOffset",
        ("FIX.5.0", "42044"): "42044 UnderlyingCashSettlValuationSubsequentBusinessDaysOffset",
        ("FIX.5.0", "42045"): "42045 UnderlyingCashSettlNumOfValuationDates",
        ("FIX.5.0", "42046"): "42046 UnderlyingCashSettlValuationTime",
        ("FIX.5.0", "42047"): "42047 UnderlyingCashSettlBusinessCenter",
        ("FIX.5.0", "42048"): "42048 UnderlyingCashSettlQuoteMethod",
        ("FIX.5.0", "42049"): "42049 UnderlyingCashSettlQuoteAmount",
        ("FIX.5.0", "42050"): "42050 UnderlyingCashSettlQuoteCurrency",
        ("FIX.5.0", "42051"): "42051 UnderlyingCashSettlMinimumQuoteAmount",
        ("FIX.5.0", "42052"): "42052 UnderlyingCashSettlMinimumQuoteCurrency",
        ("FIX.5.0", "42053"): "42053 UnderlyingCashSettlBusinessDays",
        ("FIX.5.0", "42054"): "42054 UnderlyingCashSettlAmount",
        ("FIX.5.0", "42055"): "42055 UnderlyingCashSettlRecoveryFactor",
        ("FIX.5.0", "42056"): "42056 UnderlyingCashSettlFixedTermIndicator",
        ("FIX.5.0", "42057"): "42057 UnderlyingCashSettlAccruedInterestIndicator",
        ("FIX.5.0", "42058"): "42058 UnderlyingCashSettlValuationMethod",
        ("FIX.5.0", "42059"): "42059 UnderlyingCashSettlTermXID",
        ("FIX.5.0", "42060"): "42060 NoUnderlyingPhysicalSettlTerms",
        ("FIX.5.0", "42061"): "42061 UnderlyingPhysicalSettlCurrency",
        ("FIX.5.0", "42062"): "42062 UnderlyingPhysicalSettlBusinessDays",
        ("FIX.5.0", "42063"): "42063 UnderlyingPhysicalSettlMaximumBusinessDays",
        ("FIX.5.0", "42064"): "42064 UnderlyingPhysicalSettlTermXID",
        ("FIX.5.0", "42065"): "42065 NoUnderlyingPhysicalSettlDeliverableObligations",
        ("FIX.5.0", "42066"): "42066 UnderlyingPhysicalSettlDeliverableObligationType",
        ("FIX.5.0", "42067"): "42067 UnderlyingPhysicalSettlDeliverableObligationValue",
        ("FIX.5.0", "42068"): "42068 NoUnderlyingProtectionTerms",
        ("FIX.5.0", "42069"): "42069 UnderlyingProtectionTermNotional",
        ("FIX.5.0", "42070"): "42070 UnderlyingProtectionTermCurrency",
        ("FIX.5.0", "42071"): "42071 UnderlyingProtectionTermSellerNotifies",
        ("FIX.5.0", "42072"): "42072 UnderlyingProtectionTermBuyerNotifies",
        ("FIX.5.0", "42073"): "42073 UnderlyingProtectionTermEventBusinessCenter",
        ("FIX.5.0", "42074"): "42074 UnderlyingProtectionTermStandardSources",
        ("FIX.5.0", "42075"): "42075 UnderlyingProtectionTermEventMinimumSources",
        ("FIX.5.0", "42076"): "42076 UnderlyingProtectionTermXID",
        ("FIX.5.0", "42077"): "42077 NoUnderlyingProtectionTermEvents",
        ("FIX.5.0", "42078"): "42078 UnderlyingProtectionTermEventType",
        ("FIX.5.0", "42079"): "42079 UnderlyingProtectionTermEventValue",
        ("FIX.5.0", "42080"): "42080 UnderlyingProtectionTermEventCurrency",
        ("FIX.5.0", "42081"): "42081 UnderlyingProtectionTermEventPeriod",
        ("FIX.5.0", "42082"): "42082 UnderlyingProtectionTermEventUnit",
        ("FIX.5.0", "42083"): "42083 UnderlyingProtectionTermEventDayType",
        ("FIX.5.0", "42084"): "42084 UnderlyingProtectionTermEventRateSource",
        ("FIX.5.0", "42085"): "42085 NoUnderlyingProtectionTermEventQualifiers",
        ("FIX.5.0", "42086"): "42086 UnderlyingProtectionTermEventQualifier",
        ("FIX.5.0", "42087"): "42087 NoUnderlyingProtectionTermObligations",
        ("FIX.5.0", "42088"): "42088 UnderlyingProtectionTermObligationType",
        ("FIX.5.0", "42089"): "42089 UnderlyingProtectionTermObligationValue",
        ("FIX.5.0", "42090"): "42090 NoUnderlyingProtectionTermEventNewsSources",
        ("FIX.5.0", "42091"): "42091 UnderlyingProtectionTermEventNewsSource",
        ("FIX.5.0", "42092"): "42092 UnderlyingProvisionCashSettlPaymentDateBusinessDayConvention",
        ("FIX.5.0", "42093"): "42093 UnderlyingProvisionCashSettlPaymentDateRelativeTo",
        ("FIX.5.0", "42094"): "42094 UnderlyingProvisionCashSettlPaymentDateOffsetPeriod",
        ("FIX.5.0", "42095"): "42095 UnderlyingProvisionCashSettlPaymentDateOffsetUnit",
        ("FIX.5.0", "42096"): "42096 UnderlyingProvisionCashSettlPaymentDateOffsetDayType",
        ("FIX.5.0", "42097"): "42097 UnderlyingProvisionCashSettlPaymentDateRangeFirst",
        ("FIX.5.0", "42098"): "42098 UnderlyingProvisionCashSettlPaymentDateRangeLast",
        ("FIX.5.0", "42099"): "42099 NoUnderlyingProvisionCashSettlPaymentDates",
        ("FIX.5.0", "42100"): "42100 UnderlyingProvisionCashSettlPaymentDate",
        ("FIX.5.0", "42101"): "42101 UnderlyingProvisionCashSettlPaymentDateType",
        ("FIX.5.0", "42102"): "42102 UnderlyingProvisionCashSettlQuoteSource",
        ("FIX.5.0", "42103"): "42103 UnderlyingProvisionCashSettlQuoteReferencePage",
        ("FIX.5.0", "42104"): "42104 UnderlyingProvisionCashSettlValueTime",
        ("FIX.5.0", "42105"): "42105 UnderlyingProvisionCashSettlValueTimeBusinessCenter",
        ("FIX.5.0", "42106"): "42106 UnderlyingProvisionCashSettlValueDateBusinessDayConvention",
        ("FIX.5.0", "42107"): "42107 UnderlyingProvisionCashSettlValueDateRelativeTo",
        ("FIX.5.0", "42108"): "42108 UnderlyingProvisionCashSettlValueDateOffsetPeriod",
        ("FIX.5.0", "42109"): "42109 UnderlyingProvisionCashSettlValueDateOffsetUnit",
        ("FIX.5.0", "42110"): "42110 UnderlyingProvisionCashSettlValueDateOffsetDayType",
        ("FIX.5.0", "42111"): "42111 UnderlyingProvisionCashSettlValueDateAdjusted",
        ("FIX.5.0", "42112"): "42112 NoUnderlyingProvisionOptionExerciseFixedDates",
        ("FIX.5.0", "42113"): "42113 UnderlyingProvisionOptionExerciseFixedDate",
        ("FIX.5.0", "42114"): "42114 UnderlyingProvisionOptionExerciseFixedDateType",
        ("FIX.5.0", "42115"): "42115 UnderlyingProvisionOptionExerciseBusinessDayConvention",
        ("FIX.5.0", "42116"): "42116 UnderlyingProvisionOptionExerciseEarliestDateOffsetPeriod",
        ("FIX.5.0", "42117"): "42117 UnderlyingProvisionOptionExerciseEarliestDateOffsetUnit",
        ("FIX.5.0", "42118"): "42118 UnderlyingProvisionOptionExerciseFrequencyPeriod",
        ("FIX.5.0", "42119"): "42119 UnderlyingProvisionOptionExerciseFrequencyUnit",
        ("FIX.5.0", "42120"): "42120 UnderlyingProvisionOptionExerciseStartDateUnadjusted",
        ("FIX.5.0", "42121"): "42121 UnderlyingProvisionOptionExerciseStartDateRelativeTo",
        ("FIX.5.0", "42122"): "42122 UnderlyingProvisionOptionExerciseStartDateOffsetPeriod",
        ("FIX.5.0", "42123"): "42123 UnderlyingProvisionOptionExerciseStartDateOffsetUnit",
        ("FIX.5.0", "42124"): "42124 UnderlyingProvisionOptionExerciseStartDateOffsetDayType",
        ("FIX.5.0", "42125"): "42125 UnderlyingProvisionOptionExerciseStartDateAdjusted",
        ("FIX.5.0", "42126"): "42126 UnderlyingProvisionOptionExercisePeriodSkip",
        ("FIX.5.0", "42127"): "42127 UnderlyingProvisionOptionExerciseBoundsFirstDateUnadjusted",
        ("FIX.5.0", "42128"): "42128 UnderlyingProvisionOptionExerciseBoundsLastDateUnadjusted",
        ("FIX.5.0", "42129"): "42129 UnderlyingProvisionOptionExerciseEarliestTime",
        ("FIX.5.0", "42130"): "42130 UnderlyingProvisionOptionExerciseEarliestTimeBusinessCenter",
        ("FIX.5.0", "42131"): "42131 UnderlyingProvisionOptionExerciseLatestTime",
        ("FIX.5.0", "42132"): "42132 UnderlyingProvisionOptionExerciseLatestTimeBusinessCenter",
        ("FIX.5.0", "42133"): "42133 UnderlyingProvisionOptionExpirationDateUnadjusted",
        ("FIX.5.0", "42134"): "42134 UnderlyingProvisionOptionExpirationDateBusinessDayConvention",
        ("FIX.5.0", "42135"): "42135 UnderlyingProvisionOptionExpirationDateRelativeTo",
        ("FIX.5.0", "42136"): "42136 UnderlyingProvisionOptionExpirationDateOffsetPeriod",
        ("FIX.5.0", "42137"): "42137 UnderlyingProvisionOptionExpirationDateOffsetUnit",
        ("FIX.5.0", "42138"): "42138 UnderlyingProvisionOptionExpirationDateOffsetDayType",
        ("FIX.5.0", "42139"): "42139 UnderlyingProvisionOptionExpirationDateAdjusted",
        ("FIX.5.0", "42140"): "42140 UnderlyingProvisionOptionExpirationTime",
        ("FIX.5.0", "42141"): "42141 UnderlyingProvisionOptionExpirationTimeBusinessCenter",
        ("FIX.5.0", "42142"): "42142 UnderlyingProvisionOptionRelevantUnderlyingDateUnadjusted",
        ("FIX.5.0", "42143"): "42143 UnderlyingProvisionOptionRelevantUnderlyingDateBusinessDayConvention",
        ("FIX.5.0", "42144"): "42144 UnderlyingProvisionOptionRelevantUnderlyingDateRelativeTo",
        ("FIX.5.0", "42145"): "42145 UnderlyingProvisionOptionRelevantUnderlyingDateOffsetPeriod",
        ("FIX.5.0", "42146"): "42146 UnderlyingProvisionOptionRelevantUnderlyingDateOffsetUnit",
        ("FIX.5.0", "42147"): "42147 UnderlyingProvisionOptionRelevantUnderlyingDateOffsetDayType",
        ("FIX.5.0", "42148"): "42148 UnderlyingProvisionOptionRelevantUnderlyingDateAdjusted",
        ("FIX.5.0", "42149"): "42149 NoUnderlyingProvisions",
        ("FIX.5.0", "42150"): "42150 UnderlyingProvisionType",
        ("FIX.5.0", "42151"): "42151 UnderlyingProvisionDateUnadjusted",
        ("FIX.5.0", "42152"): "42152 UnderlyingProvisionDateBusinessDayConvention",
        ("FIX.5.0", "42153"): "42153 UnderlyingProvisionDateAdjusted",
        ("FIX.5.0", "42154"): "42154 UnderlyingProvisionDateTenorPeriod",
        ("FIX.5.0", "42155"): "42155 UnderlyingProvisionDateTenorUnit",
        ("FIX.5.0", "42156"): "42156 UnderlyingProvisionCalculationAgent",
        ("FIX.5.0", "42157"): "42157 UnderlyingProvisionOptionSinglePartyBuyerSide",
        ("FIX.5.0", "42158"): "42158 UnderlyingProvisionOptionSinglePartySellerSide",
        ("FIX.5.0", "42159"): "42159 UnderlyingProvisionOptionExerciseStyle",
        ("FIX.5.0", "42160"): "42160 UnderlyingProvisionOptionExerciseMultipleNotional",
        ("FIX.5.0", "42161"): "42161 UnderlyingProvisionOptionExerciseMinimumNotional",
        ("FIX.5.0", "42162"): "42162 UnderlyingProvisionOptionExerciseMaximumNotional",
        ("FIX.5.0", "42163"): "42163 UnderlyingProvisionOptionMinimumNumber",
        ("FIX.5.0", "42164"): "42164 UnderlyingProvisionOptionMaximumNumber",
        ("FIX.5.0", "42165"): "42165 UnderlyingProvisionOptionExerciseConfirmation",
        ("FIX.5.0", "42166"): "42166 UnderlyingProvisionCashSettlMethod",
        ("FIX.5.0", "42167"): "42167 UnderlyingProvisionCashSettlCurrency",
        ("FIX.5.0", "42168"): "42168 UnderlyingProvisionCashSettlCurrency2",
        ("FIX.5.0", "42169"): "42169 UnderlyingProvisionCashSettlQuoteType",
        ("FIX.5.0", "42170"): "42170 UnderlyingProvisionText",
        ("FIX.5.0", "42171"): "42171 EncodedUnderlyingProvisionTextLen",
        ("FIX.5.0", "42172"): "42172 EncodedUnderlyingProvisionText",
        ("FIX.5.0", "42173"): "42173 NoUnderlyingProvisionPartyIDs",
        ("FIX.5.0", "42174"): "42174 UnderlyingProvisionPartyID",
        ("FIX.5.0", "42175"): "42175 UnderlyingProvisionPartyIDSource",
        ("FIX.5.0", "42176"): "42176 UnderlyingProvisionPartyRole",
        ("FIX.5.0", "42177"): "42177 NoUnderlyingProvisionPartySubIDs",
        ("FIX.5.0", "42178"): "42178 UnderlyingProvisionPartySubID",
        ("FIX.5.0", "42179"): "42179 UnderlyingProvisionPartySubIDType",
        ("FIX.5.0", "42180"): "42180 NoUnderlyingProvisionCashSettlPaymentDateBusinessCenters",
        ("FIX.5.0", "42181"): "42181 UnderlyingProvisionCashSettlPaymentDateBusinessCenter",
        ("FIX.5.0", "42182"): "42182 NoUnderlyingProvisionCashSettlValueDateBusinessCenters",
        ("FIX.5.0", "42183"): "42183 UnderlyingProvisionCashSettlValueDateBusinessCenter",
        ("FIX.5.0", "42184"): "42184 NoUnderlyingProvisionOptionExerciseBusinessCenters",
        ("FIX.5.0", "42185"): "42185 UnderlyingProvisionOptionExerciseBusinessCenter",
        ("FIX.5.0", "42186"): "42186 NoUnderlyingProvisionOptionExpirationDateBusinessCenters",
        ("FIX.5.0", "42187"): "42187 UnderlyingProvisionOptionExpirationDateBusinessCenter",
        ("FIX.5.0", "42188"): "42188 NoUnderlyingProvisionOptionRelevantUnderlyingDateBusinessCenters",
        ("FIX.5.0", "42189"): "42189 UnderlyingProvisionOptionRelevantUnderlyingDateBusinessCenter",
        ("FIX.5.0", "42190"): "42190 NoUnderlyingProvisionDateBusinessCenters",
        ("FIX.5.0", "42191"): "42191 UnderlyingProvisionDateBusinessCenter",
        ("FIX.5.0", "42192"): "42192 DeliveryStreamDeliveryPointSource",
        ("FIX.5.0", "42193"): "42193 DeliveryStreamDeliveryPointDesc",
        ("FIX.5.0", "42194"): "42194 LegDeliveryStreamDeliveryPointSource",
        ("FIX.5.0", "42195"): "42195 LegDeliveryStreamDeliveryPointDesc",
        ("FIX.5.0", "42196"): "42196 UnderlyingDeliveryStreamDeliveryPointSource",
        ("FIX.5.0", "42197"): "42197 UnderlyingDeliveryStreamDeliveryPointDesc",
        ("FIX.5.0", "42198"): "42198 NoLegContractualDefinitions",
        ("FIX.5.0", "42199"): "42199 LegContractualDefinition",
        ("FIX.5.0", "42200"): "42200 NoLegFinancingTermSupplements",
        ("FIX.5.0", "42201"): "42201 LegFinancingTermSupplementDesc",
        ("FIX.5.0", "42202"): "42202 LegFinancingTermSupplementDate",
        ("FIX.5.0", "42203"): "42203 NoLegContractualMatrices",
        ("FIX.5.0", "42204"): "42204 LegContractualMatrixSource",
        ("FIX.5.0", "42205"): "42205 LegContractualMatrixDate",
        ("FIX.5.0", "42206"): "42206 LegContractualMatrixTerm",
        ("FIX.5.0", "42207"): "42207 CashSettlDateUnadjusted",
        ("FIX.5.0", "42208"): "42208 CashSettlDateBusinessDayConvention",
        ("FIX.5.0", "42209"): "42209 CashSettlDateRelativeTo",
        ("FIX.5.0", "42210"): "42210 CashSettlDateOffsetPeriod",
        ("FIX.5.0", "42211"): "42211 CashSettlDateOffsetUnit",
        ("FIX.5.0", "42212"): "42212 CashSettlDateOffsetDayType",
        ("FIX.5.0", "42213"): "42213 CashSettlDateAdjusted",
        ("FIX.5.0", "42214"): "42214 NoCashSettlDateBusinessCenters",
        ("FIX.5.0", "42215"): "42215 CashSettlDateBusinessCenter",
        ("FIX.5.0", "42216"): "42216 CashSettlPriceSource",
        ("FIX.5.0", "42217"): "42217 CashSettlPriceDefault",
        ("FIX.5.0", "42218"): "42218 DividendFloatingRateIndex",
        ("FIX.5.0", "42219"): "42219 DividendFloatingRateIndexCurvePeriod",
        ("FIX.5.0", "42220"): "42220 DividendFloatingRateIndexCurveUnit",
        ("FIX.5.0", "42221"): "42221 DividendFloatingRateMultiplier",
        ("FIX.5.0", "42222"): "42222 DividendFloatingRateSpread",
        ("FIX.5.0", "42223"): "42223 DividendFloatingRateSpreadPositionType",
        ("FIX.5.0", "42224"): "42224 DividendFloatingRateTreatment",
        ("FIX.5.0", "42225"): "42225 DividendCapRate",
        ("FIX.5.0", "42226"): "42226 DividendCapRateBuySide",
        ("FIX.5.0", "42227"): "42227 DividendCapRateSellSide",
        ("FIX.5.0", "42228"): "42228 DividendFloorRate",
        ("FIX.5.0", "42229"): "42229 DividendFloorRateBuySide",
        ("FIX.5.0", "42230"): "42230 DividendFloorRateSellSide",
        ("FIX.5.0", "42231"): "42231 DividendInitialRate",
        ("FIX.5.0", "42232"): "42232 DividendFinalRateRoundingDirection",
        ("FIX.5.0", "42233"): "42233 DividendFinalRatePrecision",
        ("FIX.5.0", "42234"): "42234 DividendAveragingMethod",
        ("FIX.5.0", "42235"): "42235 DividendNegativeRateTreatment",
        ("FIX.5.0", "42236"): "42236 NoDividendAccrualPaymentDateBusinessCenters",
        ("FIX.5.0", "42237"): "42237 DividendAccrualPaymentDateBusinessCenter",
        ("FIX.5.0", "42238"): "42238 DividendAccrualPaymentDateRelativeTo",
        ("FIX.5.0", "42239"): "42239 DividendAccrualPaymentDateOffsetPeriod",
        ("FIX.5.0", "42240"): "42240 DividendAccrualPaymentDateOffsetUnit",
        ("FIX.5.0", "42241"): "42241 DividendAccrualPaymentDateOffsetDayType",
        ("FIX.5.0", "42242"): "42242 DividendAccrualPaymentDateUnadjusted",
        ("FIX.5.0", "42243"): "42243 DividendAccrualPaymeentDateBusinessDayConvention",
        ("FIX.5.0", "42244"): "42244 DividendAccrualPaymentDateAdjusted",
        ("FIX.5.0", "42245"): "42245 DividendReinvestmentIndicator",
        ("FIX.5.0", "42246"): "42246 DividendEntitlementEvent",
        ("FIX.5.0", "42247"): "42247 DividendAmountType",
        ("FIX.5.0", "42248"): "42248 DividendUnderlierRefID",
        ("FIX.5.0", "42249"): "42249 ExtraordinaryDividendPartySide",
        ("FIX.5.0", "42250"): "42250 ExtraordinaryDividendAmountType",
        ("FIX.5.0", "42251"): "42251 ExtraordinaryDividendCurrency",
        ("FIX.5.0", "42252"): "42252 ExtraordinaryDividendDeterminationMethod",
        ("FIX.5.0", "42253"): "42253 DividendAccrualFixedRate",
        ("FIX.5.0", "42254"): "42254 DividendCompoundingMethod",
        ("FIX.5.0", "42255"): "42255 DividendNumOfIndexUnits",
        ("FIX.5.0", "42256"): "42256 DividendCashPercentage",
        ("FIX.5.0", "42257"): "42257 DividendCashEquivalentPercentage",
        ("FIX.5.0", "42258"): "42258 NonCashDividendTreatment",
        ("FIX.5.0", "42259"): "42259 DividendComposition",
        ("FIX.5.0", "42260"): "42260 SpecialDividendsIndicator",
        ("FIX.5.0", "42261"): "42261 MaterialDividendsIndicator",
        ("FIX.5.0", "42262"): "42262 OptionsExchangeDividendsIndicator",
        ("FIX.5.0", "42263"): "42263 AdditionalDividendsIndicator",
        ("FIX.5.0", "42264"): "42264 AllDividendsIndicator",
        ("FIX.5.0", "42265"): "42265 DividendFXTriggerDateRelativeTo",
        ("FIX.5.0", "42266"): "42266 DividendFXTriggerDateOffsetPeriod",
        ("FIX.5.0", "42267"): "42267 DividendFXTriggerDateOffsetUnit",
        ("FIX.5.0", "42268"): "42268 DividendFXTriggerDateOffsetDayType",
        ("FIX.5.0", "42269"): "42269 DividendFXTriggerDateUnadjusted",
        ("FIX.5.0", "42270"): "42270 DividendFXTriggerDateBusinessDayConvention",
        ("FIX.5.0", "42271"): "42271 DividendFXTriggerDateAdjusted",
        ("FIX.5.0", "42272"): "42272 NoDividendFXTriggerDateBusinessCenters",
        ("FIX.5.0", "42273"): "42273 DividendFXTriggerDateBusinessCenter",
        ("FIX.5.0", "42274"): "42274 NoDividendPeriods",
        ("FIX.5.0", "42275"): "42275 DividendPeriodSequence",
        ("FIX.5.0", "42276"): "42276 DividendPeriodStartDateUnadjusted",
        ("FIX.5.0", "42277"): "42277 DividendPeriodEndDateUnadjusted",
        ("FIX.5.0", "42278"): "42278 DividendPeriodUnderlierRefID",
        ("FIX.5.0", "42279"): "42279 DividendPeriodStrikePrice",
        ("FIX.5.0", "42280"): "42280 DividendPeriodBusinessDayConvention",
        ("FIX.5.0", "42281"): "42281 DividendPeriodValuationDateUnadjusted",
        ("FIX.5.0", "42282"): "42282 DividendPeriodValuationDateRelativeTo",
        ("FIX.5.0", "42283"): "42283 DividendPeriodValuationDateOffsetPeriod",
        ("FIX.5.0", "42284"): "42284 DividendPeriodValuationDateOffsetUnit",
        ("FIX.5.0", "42285"): "42285 DividendPeriodValuationDateOffsetDayType",
        ("FIX.5.0", "42286"): "42286 DividendPeriodValuationDateAdjusted",
        ("FIX.5.0", "42287"): "42287 DividendPeriodPaymentDateUnadjusted",
        ("FIX.5.0", "42288"): "42288 DividendPeriodPaymentDateRelativeTo",
        ("FIX.5.0", "42289"): "42289 DividendPeriodPaymentDateOffsetPeriod",
        ("FIX.5.0", "42290"): "42290 DividendPeriodPaymentDateOffsetUnit",
        ("FIX.5.0", "42291"): "42291 DividendPeriodPaymentDateOffsetDayType",
        ("FIX.5.0", "42292"): "42292 DividendPeriodPaymentDateAdjusted",
        ("FIX.5.0", "42293"): "42293 DividendPeriodXID",
        ("FIX.5.0", "42294"): "42294 NoDividendPeriodBusinessCenters",
        ("FIX.5.0", "42295"): "42295 DividendPeriodBusinessCenter",
        ("FIX.5.0", "42296"): "42296 NoExtraordinaryEvents",
        ("FIX.5.0", "42297"): "42297 ExtraordinaryEventType",
        ("FIX.5.0", "42298"): "42298 ExtraordinaryEventValue",
        ("FIX.5.0", "42299"): "42299 LegCashSettlDateUnadjusted",
        ("FIX.5.0", "42300"): "42300 LegCashSettlDateBusinessDayConvention",
        ("FIX.5.0", "42301"): "42301 LegCashSettlDateRelativeTo",
        ("FIX.5.0", "42302"): "42302 LegCashSettlDateOffsetPeriod",
        ("FIX.5.0", "42303"): "42303 LegCashSettlDateOffsetUnit",
        ("FIX.5.0", "42304"): "42304 LegCashSettlDateOffsetDayType",
        ("FIX.5.0", "42305"): "42305 LegCashSettlDateAdjusted",
        ("FIX.5.0", "42306"): "42306 NoLegCashSettlDateBusinessCenters",
        ("FIX.5.0", "42307"): "42307 LegCashSettlDateBusinessCenter",
        ("FIX.5.0", "42308"): "42308 LegCashSettlPriceSource",
        ("FIX.5.0", "42309"): "42309 LegCashSettlPriceDefault",
        ("FIX.5.0", "42310"): "42310 NoLegDividendAccrualPaymentDateBusinessCenters",
        ("FIX.5.0", "42311"): "42311 LegDividendAccrualPaymentDateBusinessCenter",
        ("FIX.5.0", "42312"): "42312 LegDividendFloatingRateIndex",
        ("FIX.5.0", "42313"): "42313 LegDividendFloatingRateIndexCurvePeriod",
        ("FIX.5.0", "42314"): "42314 LegDividendFloatingRateIndexCurveUnit",
        ("FIX.5.0", "42315"): "42315 LegDividendFloatingRateMultiplier",
        ("FIX.5.0", "42316"): "42316 LegDividendFloatingRateSpread",
        ("FIX.5.0", "42317"): "42317 LegDividendFloatingRateSpreadPositionType",
        ("FIX.5.0", "42318"): "42318 LegDividendFloatingRateTreatment",
        ("FIX.5.0", "42319"): "42319 LegDividendCapRate",
        ("FIX.5.0", "42320"): "42320 LegDividendCapRateBuySide",
        ("FIX.5.0", "42321"): "42321 LegDividendCapRateSellSide",
        ("FIX.5.0", "42322"): "42322 LegDividendFloorRate",
        ("FIX.5.0", "42323"): "42323 LegDividendFloorRateBuySide",
        ("FIX.5.0", "42324"): "42324 LegDividendFloorRateSellSide",
        ("FIX.5.0", "42325"): "42325 LegDividendInitialRate",
        ("FIX.5.0", "42326"): "42326 LegDividendFinalRateRoundingDirection",
        ("FIX.5.0", "42327"): "42327 LegDividendFinalRatePrecision",
        ("FIX.5.0", "42328"): "42328 LegDividendAveragingMethod",
        ("FIX.5.0", "42329"): "42329 LegDividendNegativeRateTreatment",
        ("FIX.5.0", "42330"): "42330 LegDividendAccrualPaymentDateRelativeTo",
        ("FIX.5.0", "42331"): "42331 LegDividendAccrualPaymentDateOffsetPeriod",
        ("FIX.5.0", "42332"): "42332 LegDividendAccrualPaymentDateOffsetUnit",
        ("FIX.5.0", "42333"): "42333 LegDividendAccrualPaymentDateOffsetDayType",
        ("FIX.5.0", "42334"): "42334 LegDividendAccrualPaymentDateUnadjusted",
        ("FIX.5.0", "42335"): "42335 LegDividendAccrualPaymentDateBusinessDayConvention",
        ("FIX.5.0", "42336"): "42336 LegDividendAccrualPaymentDateAdjusted",
        ("FIX.5.0", "42337"): "42337 LegDividendReinvestmentIndicator",
        ("FIX.5.0", "42338"): "42338 LegDividendEntitlementEvent",
        ("FIX.5.0", "42339"): "42339 LegDividendAmountType",
        ("FIX.5.0", "42340"): "42340 LegDividendUnderlierRefID",
        ("FIX.5.0", "42341"): "42341 LegExtraordinaryDividendPartySide",
        ("FIX.5.0", "42342"): "42342 LegExtraordinaryDividendAmountType",
        ("FIX.5.0", "42343"): "42343 LegExtraordinaryDividendCurrency",
        ("FIX.5.0", "42344"): "42344 LegExtraordinaryDividendDeterminationMethod",
        ("FIX.5.0", "42345"): "42345 LegDividendAccrualFixedRate",
        ("FIX.5.0", "42346"): "42346 LegDividendCompoundingMethod",
        ("FIX.5.0", "42347"): "42347 LegDividendNumOfIndexUnits",
        ("FIX.5.0", "42348"): "42348 LegDividendCashPercentage",
        ("FIX.5.0", "42349"): "42349 LegDividendCashEquivalentPercentage",
        ("FIX.5.0", "42350"): "42350 LegNonCashDividendTreatment",
        ("FIX.5.0", "42351"): "42351 LegDividendComposition",
        ("FIX.5.0", "42352"): "42352 LegSpecialDividendsIndicator",
        ("FIX.5.0", "42353"): "42353 LegMaterialDividendsIndicator",
        ("FIX.5.0", "42354"): "42354 LegOptionsExchangeDividendsIndicator",
        ("FIX.5.0", "42355"): "42355 LegAdditionalDividendsIndicator",
        ("FIX.5.0", "42356"): "42356 LegAllDividendsIndicator",
        ("FIX.5.0", "42357"): "42357 LegDividendFXTriggerDateRelativeTo",
        ("FIX.5.0", "42358"): "42358 LegDividendFXTriggerDateOffsetPeriod",
        ("FIX.5.0", "42359"): "42359 LegDividendFXTriggerDateOffsetUnit",
        ("FIX.5.0", "42360"): "42360 LegDividendFXTriggerDateOffsetDayType",
        ("FIX.5.0", "42361"): "42361 LegDividendFXTriggerDateUnadjusted",
        ("FIX.5.0", "42362"): "42362 LegDividendFXTriggerDateBusinessDayConvention",
        ("FIX.5.0", "42363"): "42363 LegDividendFXTriggerDateAdjusted",
        ("FIX.5.0", "42364"): "42364 NoLegDividendFXTriggerDateBusinessCenters",
        ("FIX.5.0", "42365"): "42365 LegDividendFXTriggerDateBusinessCenter",
        ("FIX.5.0", "42366"): "42366 NoLegDividendPeriods",
        ("FIX.5.0", "42367"): "42367 LegDividendPeriodSequence",
        ("FIX.5.0", "42368"): "42368 LegDividendPeriodStartDateUnadjusted",
        ("FIX.5.0", "42369"): "42369 LegDividendPeriodEndDateUnadjusted",
        ("FIX.5.0", "42370"): "42370 LegDividendPeriodUnderlierRefID",
        ("FIX.5.0", "42371"): "42371 LegDividendPeriodStrikePrice",
        ("FIX.5.0", "42372"): "42372 LegDividendPeriodBusinessDayConvention",
        ("FIX.5.0", "42373"): "42373 LegDividendPeriodValuationDateUnadjusted",
        ("FIX.5.0", "42374"): "42374 LegDividendPeriodValuationDateRelativeTo",
        ("FIX.5.0", "42375"): "42375 LegDividendPeriodValuationDateOffsetPeriod",
        ("FIX.5.0", "42376"): "42376 LegDividendPeriodValuationDateOffsetUnit",
        ("FIX.5.0", "42377"): "42377 LegDividendPeriodValuationDateOffsetDayType",
        ("FIX.5.0", "42378"): "42378 LegDividendPeriodValuationDateAdjusted",
        ("FIX.5.0", "42379"): "42379 LegDividendPeriodPaymentDateUnadjusted",
        ("FIX.5.0", "42380"): "42380 LegDividendPeriodPaymentDateRelativeTo",
        ("FIX.5.0", "42381"): "42381 LegDividendPeriodPaymentDateOffsetPeriod",
        ("FIX.5.0", "42382"): "42382 LegDividendPeriodPaymentDateOffsetUnit",
        ("FIX.5.0", "42383"): "42383 LegDividendPeriodPaymentDateOffsetDayType",
        ("FIX.5.0", "42384"): "42384 LegDividendPeriodPaymentDateAdjusted",
        ("FIX.5.0", "42385"): "42385 LegDividendPeriodXID",
        ("FIX.5.0", "42386"): "42386 NoLegDividendPeriodBusinessCenters",
        ("FIX.5.0", "42387"): "42387 LegDividendPeriodBusinessCenter",
        ("FIX.5.0", "42388"): "42388 NoLegExtraordinaryEvents",
        ("FIX.5.0", "42389"): "42389 LegExtraordinaryEventType",
        ("FIX.5.0", "42390"): "42390 LegExtraordinaryEventValue",
        ("FIX.5.0", "42391"): "42391 LegSettlMethodElectingPartySide",
        ("FIX.5.0", "42392"): "42392 LegMakeWholeDate",
        ("FIX.5.0", "42393"): "42393 LegMakeWholeAmount",
        ("FIX.5.0", "42394"): "42394 LegMakeWholeBenchmarkCurveName",
        ("FIX.5.0", "42395"): "42395 LegMakeWholeBenchmarkCurvePoint",
        ("FIX.5.0", "42396"): "42396 LegMakeWholeRecallSpread",
        ("FIX.5.0", "42397"): "42397 LegMakeWholeBenchmarkQuote",
        ("FIX.5.0", "42398"): "42398 LegMakeWholeInterpolationMethod",
        ("FIX.5.0", "42399"): "42399 LegPaymentStreamCashSettlIndicator",
        ("FIX.5.0", "42400"): "42400 LegPaymentStreamCompoundingXIDRef",
        ("FIX.5.0", "42401"): "42401 LegPaymentStreamCompoundingSpread",
        ("FIX.5.0", "42402"): "42402 LegPaymentStreamInterpolationMethod",
        ("FIX.5.0", "42403"): "42403 LegPaymentStreamInterpolationPeriod",
        ("FIX.5.0", "42404"): "42404 LegPaymentStreamCompoundingFixedRate",
        ("FIX.5.0", "42405"): "42405 NoLegPaymentStreamCompoundingDates",
        ("FIX.5.0", "42406"): "42406 LegPaymentStreamCompoundingDate",
        ("FIX.5.0", "42407"): "42407 LegPaymentStreamCompoundingDateType",
        ("FIX.5.0", "42408"): "42408 LegPaymentStreamCompoundingDatesBusinessDayConvention",
        ("FIX.5.0", "42409"): "42409 LegPaymentStreamCompoundingDatesRelativeTo",
        ("FIX.5.0", "42410"): "42410 LegPaymentStreamCompoundingDatesOffsetPeriod",
        ("FIX.5.0", "42411"): "42411 LegPaymentStreamCompoundingDatesOffsetUnit",
        ("FIX.5.0", "42412"): "42412 LegPaymentStreamCompoundingDatesOffsetDayType",
        ("FIX.5.0", "42413"): "42413 LegPaymentStreamCompoundingPeriodSkip",
        ("FIX.5.0", "42414"): "42414 LegPaymentStreamCompoundingFrequencyPeriod",
        ("FIX.5.0", "42415"): "42415 LegPaymentStreamCompoundingFrequencyUnit",
        ("FIX.5.0", "42416"): "42416 LegPaymentStreamCompoundingRollConvention",
        ("FIX.5.0", "42417"): "42417 LegPaymentStreamBoundsFirstDateUnadjusted",
        ("FIX.5.0", "42418"): "42418 LegPaymentStreamBoundsLastDateUnadjusted",
        ("FIX.5.0", "42419"): "42419 NoLegPaymentStreamCompoundingDatesBusinessCenters",
        ("FIX.5.0", "42420"): "42420 LegPaymentStreamCompoundingDatesBusinessCenter",
        ("FIX.5.0", "42421"): "42421 LegPaymentStreamCompoundingEndDateUnadjusted",
        ("FIX.5.0", "42422"): "42422 LegPaymentStreamCompoundingEndDateRelativeTo",
        ("FIX.5.0", "42423"): "42423 LegPaymentStreamCompoundingEndDateOffsetPeriod",
        ("FIX.5.0", "42424"): "42424 LegPaymentStreamCompoundingEndDateOffsetUnit",
        ("FIX.5.0", "42425"): "42425 LegPaymentStreamCompoundingEndDateOffsetDayType",
        ("FIX.5.0", "42426"): "42426 LegPaymentStreamCompoundingEndDateAdjusted",
        ("FIX.5.0", "42427"): "42427 LegPaymentStreamCompoundingRateIndex",
        ("FIX.5.0", "42428"): "42428 LegPaymentStreamCompoundingRateIndexCurvePeriod",
        ("FIX.5.0", "42429"): "42429 LegPaymentStreamCompoundingRateIndexCurveUnit",
        ("FIX.5.0", "42430"): "42430 LegPaymentStreamCompoundingRateMultiplier",
        ("FIX.5.0", "42431"): "42431 LegPaymentStreamCompoundingRateSpread",
        ("FIX.5.0", "42432"): "42432 LegPaymentStreamCompoundingRateSpreadPositionType",
        ("FIX.5.0", "42433"): "42433 LegPaymentStreamCompoundingRateTreatment",
        ("FIX.5.0", "42434"): "42434 LegPaymentStreamCompoundingCapRate",
        ("FIX.5.0", "42435"): "42435 LegPaymentStreamCompoundingCapRateBuySide",
        ("FIX.5.0", "42436"): "42436 LegPaymentStreamCompoundingCapRateSellSide",
        ("FIX.5.0", "42437"): "42437 LegPaymentStreamCompoundingFloorRate",
        ("FIX.5.0", "42438"): "42438 LegPaymentStreamCompoundingFloorRateBuySide",
        ("FIX.5.0", "42439"): "42439 LegPaymentStreamCompoundingFloorRateSellSide",
        ("FIX.5.0", "42440"): "42440 LegPaymentStreamCompoundingInitialRate",
        ("FIX.5.0", "42441"): "42441 LegPaymentStreamCompoundingFinalRateRoundingDirection",
        ("FIX.5.0", "42442"): "42442 LegPaymentStreamCompoundingFinalRatePrecision",
        ("FIX.5.0", "42443"): "42443 LegPaymentStreamCompoundingAveragingMethod",
        ("FIX.5.0", "42444"): "42444 LegPaymentStreamCompoundingNegativeRateTreatment",
        ("FIX.5.0", "42445"): "42445 LegPaymentStreamCompoundingStartDateUnadjusted",
        ("FIX.5.0", "42446"): "42446 LegPaymentStreamCompoundingStartDateRelativeTo",
        ("FIX.5.0", "42447"): "42447 LegPaymentStreamCompoundingStartDateOffsetPeriod",
        ("FIX.5.0", "42448"): "42448 LegPaymentStreamCompoundingStartDateOffsetUnit",
        ("FIX.5.0", "42449"): "42449 LegPaymentStreamCompoundingStartDateOffsetDayType",
        ("FIX.5.0", "42450"): "42450 LegPaymentStreamCompoundingStartDateAdjusted",
        ("FIX.5.0", "42451"): "42451 LegPaymentStreamFormulaImageLength",
        ("FIX.5.0", "42452"): "42452 LegPaymentStreamFormulaImage",
        ("FIX.5.0", "42453"): "42453 LegPaymentStreamFinalPricePaymentDateUnadjusted",
        ("FIX.5.0", "42454"): "42454 LegPaymentStreamFinalPricePaymentDateRelativeTo",
        ("FIX.5.0", "42455"): "42455 LegPaymentStreamFinalPricePaymentDateOffsetPeriod",
        ("FIX.5.0", "42456"): "42456 LegPaymentStreamFinalPricePaymentDateOffsetUnit",
        ("FIX.5.0", "42457"): "42457 LegPaymentStreamFinalPricePaymentDateOffsetDayType",
        ("FIX.5.0", "42458"): "42458 LegPaymentStreamFinalPricePaymentDateAdjusted",
        ("FIX.5.0", "42459"): "42459 NoLegPaymentStreamFixingDates",
        ("FIX.5.0", "42460"): "42460 LegPaymentStreamFixingDate",
        ("FIX.5.0", "42461"): "42461 LegPaymentStreamFixingDateType",
        ("FIX.5.0", "42462"): "42462 LegPaymentStreamFirstObservationDateUnadjusted",
        ("FIX.5.0", "42463"): "42463 LegPaymentStreamFirstObservationDateRelativeTo",
        ("FIX.5.0", "42464"): "42464 LegPaymentStreamFirstObservationDateOffsetDayType",
        ("FIX.5.0", "42465"): "42465 LegPaymentStreamFirstObservationDateAdjusted",
        ("FIX.5.0", "42466"): "42466 LegPaymentStreamUnderlierRefID",
        ("FIX.5.0", "42467"): "42467 LegReturnRateNotionalReset",
        ("FIX.5.0", "42468"): "42468 LegPaymentStreamLinkInitialLevel",
        ("FIX.5.0", "42469"): "42469 LegPaymentStreamLinkClosingLevelIndicator",
        ("FIX.5.0", "42470"): "42470 LegPaymentStreamLinkExpiringLevelIndicator",
        ("FIX.5.0", "42471"): "42471 LegPaymentStreamLinkEstimatedTradingDays",
        ("FIX.5.0", "42472"): "42472 LegPaymentStreamLinkStrikePrice",
        ("FIX.5.0", "42473"): "42473 LegPaymentStreamLinkStrikePriceType",
        ("FIX.5.0", "42474"): "42474 LegPaymentStreamLinkMaximumBoundary",
        ("FIX.5.0", "42475"): "42475 LegPaymentStreamLinkMinimumBoundary",
        ("FIX.5.0", "42476"): "42476 LegPaymentStreamLinkNumberOfDataSeries",
        ("FIX.5.0", "42477"): "42477 LegPaymentStreamVarianceUnadjustedCap",
        ("FIX.5.0", "42478"): "42478 LegPaymentStreamRealizedVarianceMethod",
        ("FIX.5.0", "42479"): "42479 LegPaymentStreamDaysAdjustmentIndicator",
        ("FIX.5.0", "42480"): "42480 LegPaymentStreamNearestExchangeContractRefID",
        ("FIX.5.0", "42481"): "42481 LegPaymentStreamVegaNotionalAmount",
        ("FIX.5.0", "42482"): "42482 LegPaymentStreamFormulaCurrency",
        ("FIX.5.0", "42483"): "42483 LegPaymentStreamFormulaCurrencyDeterminationMethod",
        ("FIX.5.0", "42484"): "42484 LegPaymentStreamFormulaReferenceAmount",
        ("FIX.5.0", "42485"): "42485 NoLegPaymentStreamFormulas",
        ("FIX.5.0", "42486"): "42486 LegPaymentStreamFormula",
        ("FIX.5.0", "42487"): "42487 LegPaymentStreamFormulaDesc",
        ("FIX.5.0", "42488"): "42488 LegPaymentStubEndDateUnadjusted",
        ("FIX.5.0", "42489"): "42489 LegPaymentStubEndDateBusinessDayConvention",
        ("FIX.5.0", "42490"): "42490 LegPaymentStubEndDateRelativeTo",
        ("FIX.5.0", "42491"): "42491 LegPaymentStubEndDateOffsetPeriod",
        ("FIX.5.0", "42492"): "42492 LegPaymentStubEndDateOffsetUnit",
        ("FIX.5.0", "42493"): "42493 LegPaymentStubEndDateOffsetDayType",
        ("FIX.5.0", "42494"): "42494 LegPaymentStubEndDateAdjusted",
        ("FIX.5.0", "42495"): "42495 NoLegPaymentStubEndDateBusinessCenters",
        ("FIX.5.0", "42496"): "42496 LegPaymentStubEndDateBusinessCenter",
        ("FIX.5.0", "42497"): "42497 LegPaymentStubStartDateUnadjusted",
        ("FIX.5.0", "42498"): "42498 LegPaymentStubStartDateBusinessDayConvention",
        ("FIX.5.0", "42499"): "42499 LegPaymentStubStartDateRelativeTo",
        ("FIX.5.0", "42500"): "42500 LegPaymentStubStartDateOffsetPeriod",
        ("FIX.5.0", "42501"): "42501 LegPaymentStubStartDateOffsetUnit",
        ("FIX.5.0", "42502"): "42502 LegPaymentStubStartDateOffsetDayType",
        ("FIX.5.0", "42503"): "42503 LegPaymentStubStartDateAdjusted",
        ("FIX.5.0", "42504"): "42504 NoLegPaymentStubStartDateBusinessCenters",
        ("FIX.5.0", "42505"): "42505 LegPaymentStubStartDateBusinessCenter",
        ("FIX.5.0", "42506"): "42506 LegProvisionBreakFeeElection",
        ("FIX.5.0", "42507"): "42507 LegProvisionBreakFeeRate",
        ("FIX.5.0", "42508"): "42508 NoLegReturnRateDates",
        ("FIX.5.0", "42509"): "42509 LegReturnRateDateMode",
        ("FIX.5.0", "42510"): "42510 LegReturnRateValuationDateRelativeTo",
        ("FIX.5.0", "42511"): "42511 LegReturnRateValuationDateOffsetPeriod",
        ("FIX.5.0", "42512"): "42512 LegReturnRateValuationDateOffsetUnit",
        ("FIX.5.0", "42513"): "42513 LegReturnRateValuationDateOffsetDayType",
        ("FIX.5.0", "42514"): "42514 LegReturnRateValuationStartDateUnadjusted",
        ("FIX.5.0", "42515"): "42515 LegReturnRateValuationStartDateRelativeTo",
        ("FIX.5.0", "42516"): "42516 LegReturnRateValuationStartDateOffsetPeriod",
        ("FIX.5.0", "42517"): "42517 LegReturnRateValuationStartDateOffsetUnit",
        ("FIX.5.0", "42518"): "42518 LegReturnRateValuationStartDateOffsetDayType",
        ("FIX.5.0", "42519"): "42519 LegReturnRateValuationStartDateAdjusted",
        ("FIX.5.0", "42520"): "42520 LegReturnRateValuationEndDateUnadjusted",
        ("FIX.5.0", "42521"): "42521 LegReturnRateValuationEndDateRelativeTo",
        ("FIX.5.0", "42522"): "42522 LegReturnRateValuationEndDateOffsetPeriod",
        ("FIX.5.0", "42523"): "42523 LegReturnRateValuationEndDateOffsetUnit",
        ("FIX.5.0", "42524"): "42524 LegReturnRateValuationEndDateOffsetDayType",
        ("FIX.5.0", "42525"): "42525 LegReturnRateValuationEndDateAdjusted",
        ("FIX.5.0", "42526"): "42526 LegReturnRateValuationFrequencyPeriod",
        ("FIX.5.0", "42527"): "42527 LegReturnRateValuationFrequencyUnit",
        ("FIX.5.0", "42528"): "42528 LegReturnRateValuationFrequencyRollConvention",
        ("FIX.5.0", "42529"): "42529 LegReturnRateValuationDateBusinessDayConvention",
        ("FIX.5.0", "42530"): "42530 NoLegReturnRateFXConversions",
        ("FIX.5.0", "42531"): "42531 LegReturnRateFXCurrencySymbol",
        ("FIX.5.0", "42532"): "42532 LegReturnRateFXRate",
        ("FIX.5.0", "42533"): "42533 LegReturnRateFXRateCalc",
        ("FIX.5.0", "42534"): "42534 NoLegReturnRates",
        ("FIX.5.0", "42535"): "42535 LegReturnRatePriceSequence",
        ("FIX.5.0", "42536"): "42536 LegReturnRateCommissionBasis",
        ("FIX.5.0", "42537"): "42537 LegReturnRateCommissionAmount",
        ("FIX.5.0", "42538"): "42538 LegReturnRateCommissionCurrency",
        ("FIX.5.0", "42539"): "42539 LegReturnRateTotalCommissionPerTrade",
        ("FIX.5.0", "42540"): "42540 LegReturnRateDeterminationMethod",
        ("FIX.5.0", "42541"): "42541 LegReturnRateAmountRelativeTo",
        ("FIX.5.0", "42542"): "42542 LegReturnRateQuoteMeasureType",
        ("FIX.5.0", "42543"): "42543 LegReturnRateQuoteUnits",
        ("FIX.5.0", "42544"): "42544 LegReturnRateQuoteMethod",
        ("FIX.5.0", "42545"): "42545 LegReturnRateQuoteCurrency",
        ("FIX.5.0", "42546"): "42546 LegReturnRateQuoteCurrencyType",
        ("FIX.5.0", "42547"): "42547 LegReturnRateQuoteTimeType",
        ("FIX.5.0", "42548"): "42548 LegReturnRateQuoteTime",
        ("FIX.5.0", "42549"): "42549 LegReturnRateQuoteDate",
        ("FIX.5.0", "42550"): "42550 LegReturnRateQuoteExpirationTime",
        ("FIX.5.0", "42551"): "42551 LegReturnRateQuoteBusinessCenter",
        ("FIX.5.0", "42552"): "42552 LegReturnRateQuoteExchange",
        ("FIX.5.0", "42553"): "42553 LegReturnRateQuotePricingModel",
        ("FIX.5.0", "42554"): "42554 LegReturnRateCashFlowType",
        ("FIX.5.0", "42555"): "42555 LegReturnRateValuationTimeType",
        ("FIX.5.0", "42556"): "42556 LegReturnRateValuationTime",
        ("FIX.5.0", "42557"): "42557 LegReturnRateValuationTimeBusinessCenter",
        ("FIX.5.0", "42558"): "42558 LegReturnRateValuationPriceOption",
        ("FIX.5.0", "42559"): "42559 LegReturnRateFinalPriceFallback",
        ("FIX.5.0", "42560"): "42560 NoLegReturnRateInformationSources",
        ("FIX.5.0", "42561"): "42561 LegReturnRateInformationSource",
        ("FIX.5.0", "42562"): "42562 LegReturnRateReferencePage",
        ("FIX.5.0", "42563"): "42563 LegReturnRateReferencePageHeading",
        ("FIX.5.0", "42564"): "42564 NoLegReturnRatePrices",
        ("FIX.5.0", "42565"): "42565 LegReturnRatePriceBasis",
        ("FIX.5.0", "42566"): "42566 LegReturnRatePrice",
        ("FIX.5.0", "42567"): "42567 LegReturnRatePriceCurrency",
        ("FIX.5.0", "42568"): "42568 LegReturnRatePriceType",
        ("FIX.5.0", "42569"): "42569 NoLegReturnRateValuationDateBusinessCenters",
        ("FIX.5.0", "42570"): "42570 LegReturnRateValuationDateBusinessCenter",
        ("FIX.5.0", "42571"): "42571 NoLegReturnRateValuationDates",
        ("FIX.5.0", "42572"): "42572 LegReturnRateValuationDate",
        ("FIX.5.0", "42573"): "42573 LegReturnRateValuationDateType",
        ("FIX.5.0", "42574"): "42574 LegSettlMethodElectionDateUnadjusted",
        ("FIX.5.0", "42575"): "42575 LegSettlMethodElectionDateBusinessDayConvention",
        ("FIX.5.0", "42576"): "42576 LegSettlMethodElectionDateRelativeTo",
        ("FIX.5.0", "42577"): "42577 LegSettlMethodElectionDateOffsetPeriod",
        ("FIX.5.0", "42578"): "42578 LegSettlMethodElectionDateOffsetUnit",
        ("FIX.5.0", "42579"): "42579 LegSettlMethodElectionDateOffsetDayType",
        ("FIX.5.0", "42580"): "42580 LegSettlMethodElectionDateAdjusted",
        ("FIX.5.0", "42581"): "42581 NoLegSettlMethodElectionDateBusinessCenters",
        ("FIX.5.0", "42582"): "42582 LegSettlMethodElectionDateBusinessCenter",
        ("FIX.5.0", "42583"): "42583 LegStreamVersion",
        ("FIX.5.0", "42584"): "42584 LegStreamVersionEffectiveDate",
        ("FIX.5.0", "42585"): "42585 LegStreamNotionalDeterminationMethod",
        ("FIX.5.0", "42586"): "42586 LegStreamNotionalAdjustments",
        ("FIX.5.0", "42587"): "42587 StreamCommodityDeliveryPricingRegion",
        ("FIX.5.0", "42588"): "42588 LegStreamCommodityDeliveryPricingRegion",
        ("FIX.5.0", "42589"): "42589 UnderlyingStreamCommodityDeliveryPricingRegion",
        ("FIX.5.0", "42590"): "42590 SettlMethodElectingPartySide",
        ("FIX.5.0", "42591"): "42591 MakeWholeDate",
        ("FIX.5.0", "42592"): "42592 MakeWholeAmount",
        ("FIX.5.0", "42593"): "42593 MakeWholeBenchmarkCurveName",
        ("FIX.5.0", "42594"): "42594 MakeWholeBenchmarkCurvePoint",
        ("FIX.5.0", "42595"): "42595 MakeWholeRecallSpread",
        ("FIX.5.0", "42596"): "42596 MakeWholeBenchmarkQuote",
        ("FIX.5.0", "42597"): "42597 MakeWholeInterpolationMethod",
        ("FIX.5.0", "42598"): "42598 PaymentAmountRelativeTo",
        ("FIX.5.0", "42599"): "42599 PaymentAmountDeterminationMethod",
        ("FIX.5.0", "42600"): "42600 PaymentStreamCashSettlIndicator",
        ("FIX.5.0", "42601"): "42601 PaymentStreamCompoundingXIDRef",
        ("FIX.5.0", "42602"): "42602 PaymentStreamCompoundingSpread",
        ("FIX.5.0", "42603"): "42603 PaymentStreamInterpolationMethod",
        ("FIX.5.0", "42604"): "42604 PaymentStreamInterpolationPeriod",
        ("FIX.5.0", "42605"): "42605 PaymentStreamCompoundingFixedRate",
        ("FIX.5.0", "42606"): "42606 NoPaymentStreamCompoundingDates",
        ("FIX.5.0", "42607"): "42607 PaymentStreamCompoundingDate",
        ("FIX.5.0", "42608"): "42608 PaymentStreamCompoundingDateType",
        ("FIX.5.0", "42609"): "42609 PaymentStreamCompoundingDatesBusinessDayConvention",
        ("FIX.5.0", "42610"): "42610 PaymentStreamCompoundingDatesRelativeTo",
        ("FIX.5.0", "42611"): "42611 PaymentStreamCompoundingDatesOffsetPeriod",
        ("FIX.5.0", "42612"): "42612 PaymentStreamCompoundingDatesOffsetUnit",
        ("FIX.5.0", "42613"): "42613 PaymentStreamCompoundingDatesOffsetDayType",
        ("FIX.5.0", "42614"): "42614 PaymentStreamCompoundingPeriodSkip",
        ("FIX.5.0", "42615"): "42615 PaymentStreamCompoundingFrequencyPeriod",
        ("FIX.5.0", "42616"): "42616 PaymentStreamCompoundingFrequencyUnit",
        ("FIX.5.0", "42617"): "42617 PaymentStreamCompoundingRollConvention",
        ("FIX.5.0", "42618"): "42618 PaymentStreamBoundsFirstDateUnadjusted",
        ("FIX.5.0", "42619"): "42619 PaymentStreamBoundsLastDateUnadjusted",
        ("FIX.5.0", "42620"): "42620 NoPaymentStreamCompoundingDatesBusinessCenters",
        ("FIX.5.0", "42621"): "42621 PaymentStreamCompoundingDatesBusinessCenter",
        ("FIX.5.0", "42622"): "42622 PaymentStreamCompoundingEndDateUnadjusted",
        ("FIX.5.0", "42623"): "42623 PaymentStreamCompoundingEndDateRelativeTo",
        ("FIX.5.0", "42624"): "42624 PaymentStreamCompoundingEndDateOffsetPeriod",
        ("FIX.5.0", "42625"): "42625 PaymentStreamCompoundingEndDateOffsetUnit",
        ("FIX.5.0", "42626"): "42626 PaymentStreamCompoundingEndDateOffsetDayType",
        ("FIX.5.0", "42627"): "42627 PaymentStreamCompoundingEndDateAdjusted",
        ("FIX.5.0", "42628"): "42628 PaymentStreamCompoundingRateIndex",
        ("FIX.5.0", "42629"): "42629 PaymentStreamCompoundingRateIndexCurvePeriod",
        ("FIX.5.0", "42630"): "42630 PaymentStreamCompoundingRateIndexCurveUnit",
        ("FIX.5.0", "42631"): "42631 PaymentStreamCompoundingRateMultiplier",
        ("FIX.5.0", "42632"): "42632 PaymentStreamCompoundingRateSpread",
        ("FIX.5.0", "42633"): "42633 PaymentStreamCompoundingRateSpreadPositionType",
        ("FIX.5.0", "42634"): "42634 PaymentStreamCompoundingRateTreatment",
        ("FIX.5.0", "42635"): "42635 PaymentStreamCompoundingCapRate",
        ("FIX.5.0", "42636"): "42636 PaymentStreamCompoundingCapRateBuySide",
        ("FIX.5.0", "42637"): "42637 PaymentStreamCompoundingCapRateSellSide",
        ("FIX.5.0", "42638"): "42638 PaymentStreamCompoundingFloorRate",
        ("FIX.5.0", "42639"): "42639 PaymentStreamCompoundingFloorRateBuySide",
        ("FIX.5.0", "42640"): "42640 PaymentStreamCompoundingFloorRateSellSide",
        ("FIX.5.0", "42641"): "42641 PaymentStreamCompoundingInitialRate",
        ("FIX.5.0", "42642"): "42642 PaymentStreamCompoundingFinalRateRoundingDirection",
        ("FIX.5.0", "42643"): "42643 PaymentStreamCompoundingFinalRatePrecision",
        ("FIX.5.0", "42644"): "42644 PaymentStreamCompoundingAveragingMethod",
        ("FIX.5.0", "42645"): "42645 PaymentStreamCompoundingNegativeRateTreatment",
        ("FIX.5.0", "42646"): "42646 PaymentStreamCompoundingStartDateUnadjusted",
        ("FIX.5.0", "42647"): "42647 PaymentStreamCompoundingStartDateRelativeTo",
        ("FIX.5.0", "42648"): "42648 PaymentStreamCompoundingStartDateOffsetPeriod",
        ("FIX.5.0", "42649"): "42649 PaymentStreamCompoundingStartDateOffsetUnit",
        ("FIX.5.0", "42650"): "42650 PaymentStreamCompoundingStartDateOffsetDayType",
        ("FIX.5.0", "42651"): "42651 PaymentStreamCompoundingStartDateAdjusted",
        ("FIX.5.0", "42652"): "42652 PaymentStreamFormulaImageLength",
        ("FIX.5.0", "42653"): "42653 PaymentStreamFormulaImage",
        ("FIX.5.0", "42654"): "42654 PaymentStreamFinalPricePaymentDateUnadjusted",
        ("FIX.5.0", "42655"): "42655 PaymentStreamFinalPricePaymentDateRelativeTo",
        ("FIX.5.0", "42656"): "42656 PaymentStreamFinalPricePaymentDateOffsetfPeriod",
        ("FIX.5.0", "42657"): "42657 PaymentStreamFinalPricePaymentDateOffsetUnit",
        ("FIX.5.0", "42658"): "42658 PaymentStreamFinalPricePaymentDateOffsetDayType",
        ("FIX.5.0", "42659"): "42659 PaymentStreamFinalPricePaymentDateAdjusted",
        ("FIX.5.0", "42660"): "42660 NoPaymentStreamFixingDates",
        ("FIX.5.0", "42661"): "42661 PaymentStreamFixingDate",
        ("FIX.5.0", "42662"): "42662 PaymentStreamFixingDateType",
        ("FIX.5.0", "42663"): "42663 PaymentStreamFirstObservationDateUnadjusted",
        ("FIX.5.0", "42664"): "42664 PaymentStreamFirstObservationDateRelativeTo",
        ("FIX.5.0", "42665"): "42665 PaymentStreamFirstObservationDateOffsetDayType",
        ("FIX.5.0", "42666"): "42666 PaymentStreamFirstObservationDateAdjusted",
        ("FIX.5.0", "42667"): "42667 PaymentStreamUnderlierRefID",
        ("FIX.5.0", "42668"): "42668 ReturnRateNotionalReset",
        ("FIX.5.0", "42669"): "42669 PaymentStreamLinkInitialLevel",
        ("FIX.5.0", "42670"): "42670 PaymentStreamLinkClosingLevelIndicator",
        ("FIX.5.0", "42671"): "42671 PaymentStreamLinkExpiringLevelIndicator",
        ("FIX.5.0", "42672"): "42672 PaymentStreamLinkEstimatedTradingDays",
        ("FIX.5.0", "42673"): "42673 PaymentStreamLinkStrikePrice",
        ("FIX.5.0", "42674"): "42674 PaymentStreamLinkStrikePriceType",
        ("FIX.5.0", "42675"): "42675 PaymentStreamLinkMaximumBoundary",
        ("FIX.5.0", "42676"): "42676 PaymentStreamLinkMinimumBoundary",
        ("FIX.5.0", "42677"): "42677 PaymentStreamLinkNumberOfDataSeries",
        ("FIX.5.0", "42678"): "42678 PaymentStreamVarianceUnadjustedCap",
        ("FIX.5.0", "42679"): "42679 PaymentStreamRealizedVarianceMethod",
        ("FIX.5.0", "42680"): "42680 PaymentStreamDaysAdjustmentIndicator",
        ("FIX.5.0", "42681"): "42681 PaymentStreamNearestExchangeContractRefID",
        ("FIX.5.0", "42682"): "42682 PaymentStreamVegaNotionalAmount",
        ("FIX.5.0", "42683"): "42683 NoPaymentStreamFormulas",
        ("FIX.5.0", "42684"): "42684 PaymentStreamFormula",
        ("FIX.5.0", "42685"): "42685 PaymentStreamFormulaDesc",
        ("FIX.5.0", "42686"): "42686 PaymentStreamFormulaCurrency",
        ("FIX.5.0", "42687"): "42687 PaymentStreamFormulaCurrencyDeterminationMethod",
        ("FIX.5.0", "42688"): "42688 PaymentStreamFormulaReferenceAmount",
        ("FIX.5.0", "42689"): "42689 PaymentStubEndDateUnadjusted",
        ("FIX.5.0", "42690"): "42690 PaymentStubEndDateBusinessDayConvention",
        ("FIX.5.0", "42691"): "42691 PaymentStubEndDateRelativeTo",
        ("FIX.5.0", "42692"): "42692 PaymentStubEndDateOffsetPeriod",
        ("FIX.5.0", "42693"): "42693 PaymentStubEndDateOffsetUnit",
        ("FIX.5.0", "42694"): "42694 PaymentStubEndDateOffsetDayType",
        ("FIX.5.0", "42695"): "42695 PaymentStubEndDateAdjusted",
        ("FIX.5.0", "42696"): "42696 NoPaymentStubEndDateBusinessCenters",
        ("FIX.5.0", "42697"): "42697 PaymentStubEndDateBusinessCenter",
        ("FIX.5.0", "42698"): "42698 PaymentStubStartDateUnadjusted",
        ("FIX.5.0", "42699"): "42699 PaymentStubStartDateBusinessDayConvention",
        ("FIX.5.0", "42700"): "42700 PaymentStubStartDateRelativeTo",
        ("FIX.5.0", "42701"): "42701 PaymentStubStartDateOffsetPeriod",
        ("FIX.5.0", "42702"): "42702 PaymentStubStartDateOffsetUnit",
        ("FIX.5.0", "42703"): "42703 PaymentStubStartDateOffsetDayType",
        ("FIX.5.0", "42704"): "42704 PaymentStubStartDateAdjusted",
        ("FIX.5.0", "42705"): "42705 NoPaymentStubStartDateBusinessCenters",
        ("FIX.5.0", "42706"): "42706 PaymentStubStartDateBusinessCenter",
        ("FIX.5.0", "42707"): "42707 ProvisionBreakFeeElection",
        ("FIX.5.0", "42708"): "42708 ProvisionBreakFeeRate",
        ("FIX.5.0", "42709"): "42709 NoReturnRateDates",
        ("FIX.5.0", "42710"): "42710 ReturnRateDateMode",
        ("FIX.5.0", "42711"): "42711 ReturnRateValuationDateRelativeTo",
        ("FIX.5.0", "42712"): "42712 ReturnRateValuationDateOffsetPeriod",
        ("FIX.5.0", "42713"): "42713 ReturnRateValuationDateOffsetUnit",
        ("FIX.5.0", "42714"): "42714 ReturnRateValuationDateOffsetDayType",
        ("FIX.5.0", "42715"): "42715 ReturnRateValuationStartDateUnadjusted",
        ("FIX.5.0", "42716"): "42716 ReturnRateValuationStartDateRelativeTo",
        ("FIX.5.0", "42717"): "42717 ReturnRateValuationStartDateOffsetPeriod",
        ("FIX.5.0", "42718"): "42718 ReturnRateValuationStartDateOffsetUnit",
        ("FIX.5.0", "42719"): "42719 ReturnRateValuationStartDateOffsetDayType",
        ("FIX.5.0", "42720"): "42720 ReturnRateValuationStartDateAdjusted",
        ("FIX.5.0", "42721"): "42721 ReturnRateValuationEndDateUnadjusted",
        ("FIX.5.0", "42722"): "42722 ReturnRateValuationEndDateRelativeTo",
        ("FIX.5.0", "42723"): "42723 ReturnRateValuationEndDateOffsetPeriod",
        ("FIX.5.0", "42724"): "42724 ReturnRateValuationEndDateOffsetUnit",
        ("FIX.5.0", "42725"): "42725 ReturnRateValuationEndDateOffsetDayType",
        ("FIX.5.0", "42726"): "42726 ReturnRateValuationEndDateAdjusted",
        ("FIX.5.0", "42727"): "42727 ReturnRateValuationFrequencyPeriod",
        ("FIX.5.0", "42728"): "42728 ReturnRateValuationFrequencyUnit",
        ("FIX.5.0", "42729"): "42729 ReturnRateValuationFrequencyRollConvention",
        ("FIX.5.0", "42730"): "42730 ReturnRateValuationDateBusinessDayConvention",
        ("FIX.5.0", "42731"): "42731 NoReturnRateFXConversions",
        ("FIX.5.0", "42732"): "42732 ReturnRateFXCurrencySymbol",
        ("FIX.5.0", "42733"): "42733 ReturnRateFXRate",
        ("FIX.5.0", "42734"): "42734 ReturnRateFXRateCalc",
        ("FIX.5.0", "42735"): "42735 NoReturnRates",
        ("FIX.5.0", "42736"): "42736 ReturnRatePriceSequence",
        ("FIX.5.0", "42737"): "42737 ReturnRateCommissionBasis",
        ("FIX.5.0", "42738"): "42738 ReturnRateCommissionAmount",
        ("FIX.5.0", "42739"): "42739 ReturnRateCommissionCurrency",
        ("FIX.5.0", "42740"): "42740 ReturnRateTotalCommissionPerTrade",
        ("FIX.5.0", "42741"): "42741 ReturnRateDeterminationMethod",
        ("FIX.5.0", "42742"): "42742 ReturnRateAmountRelativeTo",
        ("FIX.5.0", "42743"): "42743 ReturnRateQuoteMeasureType",
        ("FIX.5.0", "42744"): "42744 ReturnRateQuoteUnits",
        ("FIX.5.0", "42745"): "42745 ReturnRateQuoteMethod",
        ("FIX.5.0", "42746"): "42746 ReturnRateQuoteCurrency",
        ("FIX.5.0", "42747"): "42747 ReturnRateQuoteCurrencyType",
        ("FIX.5.0", "42748"): "42748 ReturnRateQuoteTimeType",
        ("FIX.5.0", "42749"): "42749 ReturnRateQuoteTime",
        ("FIX.5.0", "42750"): "42750 ReturnRateQuoteDate",
        ("FIX.5.0", "42751"): "42751 ReturnRateQuoteExpirationTime",
        ("FIX.5.0", "42752"): "42752 ReturnRateQuoteBusinessCenter",
        ("FIX.5.0", "42753"): "42753 ReturnRateQuoteExchange",
        ("FIX.5.0", "42754"): "42754 ReturnRateQuotePricingModel",
        ("FIX.5.0", "42755"): "42755 ReturnRateCashFlowType",
        ("FIX.5.0", "42756"): "42756 ReturnRateValuationTimeType",
        ("FIX.5.0", "42757"): "42757 ReturnRateValuationTime",
        ("FIX.5.0", "42758"): "42758 ReturnRateValuationTimeBusinessCenter",
        ("FIX.5.0", "42759"): "42759 ReturnRateValuationPriceOption",
        ("FIX.5.0", "42760"): "42760 ReturnRateFinalPriceFallback",
        ("FIX.5.0", "42761"): "42761 NoReturnRateInformationSources",
        ("FIX.5.0", "42762"): "42762 ReturnRateInformationSource",
        ("FIX.5.0", "42763"): "42763 ReturnRateReferencePage",
        ("FIX.5.0", "42764"): "42764 ReturnRateReferencePageHeading",
        ("FIX.5.0", "42765"): "42765 NoReturnRatePrices",
        ("FIX.5.0", "42766"): "42766 ReturnRatePriceBasis",
        ("FIX.5.0", "42767"): "42767 ReturnRatePrice",
        ("FIX.5.0", "42768"): "42768 ReturnRatePriceCurrency",
        ("FIX.5.0", "42769"): "42769 ReturnRatePriceType",
        ("FIX.5.0", "42770"): "42770 NoReturnRateValuationDateBusinessCenters",
        ("FIX.5.0", "42771"): "42771 ReturnRateValuationDateBusinessCenter",
        ("FIX.5.0", "42772"): "42772 NoReturnRateValuationDates",
        ("FIX.5.0", "42773"): "42773 ReturnRateValuationDate",
        ("FIX.5.0", "42774"): "42774 ReturnRateValuationDateType",
        ("FIX.5.0", "42775"): "42775 NoSettlMethodElectionDateBusinessCenters",
        ("FIX.5.0", "42776"): "42776 SettlMethodElectionDateBusinessCenter",
        ("FIX.5.0", "42777"): "42777 SettlMethodElectionDateUnadjusted",
        ("FIX.5.0", "42778"): "42778 SettlMethodElectionDateBusinessDayConvention",
        ("FIX.5.0", "42779"): "42779 SettlMethodElectionDateRelativeTo",
        ("FIX.5.0", "42780"): "42780 SettlMethodElectionDateOffsetPeriod",
        ("FIX.5.0", "42781"): "42781 SettlMethodElectionDateOffsetUnit",
        ("FIX.5.0", "42782"): "42782 SettlMethodElectionDateOffsetDayType",
        ("FIX.5.0", "42783"): "42783 SettlMethodElectionDateAdjusted",
        ("FIX.5.0", "42784"): "42784 StreamVersion",
        ("FIX.5.0", "42785"): "42785 StreamVersionEffectiveDate",
        ("FIX.5.0", "42786"): "42786 StreamNotionalDeterminationMethod",
        ("FIX.5.0", "42787"): "42787 StreamNotionalAdjustments",
        ("FIX.5.0", "42788"): "42788 NoUnderlyingCashSettlDateBusinessCenters",
        ("FIX.5.0", "42789"): "42789 UnderlyingCashSettlDateBusinessCenter",
        ("FIX.5.0", "42790"): "42790 UnderlyingCashSettlDateUnadjusted",
        ("FIX.5.0", "42791"): "42791 UnderlyingCashSettlDateBusinessDayConvention",
        ("FIX.5.0", "42792"): "42792 UnderlyingCashSettlDateRelativeTo",
        ("FIX.5.0", "42793"): "42793 UnderlyingCashSettlDateOffsetPeriod",
        ("FIX.5.0", "42794"): "42794 UnderlyingCashSettlDateOffsetUnit",
        ("FIX.5.0", "42795"): "42795 UnderlyingCashSettlDateOffsetDayType",
        ("FIX.5.0", "42796"): "42796 UnderlyingCashSettlDateAdjusted",
        ("FIX.5.0", "42797"): "42797 UnderlyingCashSettlPriceSource",
        ("FIX.5.0", "42798"): "42798 UnderlyingCashSettlPriceDefault",
        ("FIX.5.0", "42799"): "42799 NoUnderlyingDividendAccrualPaymentDateBusinessCenters",
        ("FIX.5.0", "42800"): "42800 UnderlyingDividendAccrualPaymentDateBusinessCenter",
        ("FIX.5.0", "42801"): "42801 UnderlyingDividendFloatingRateIndex",
        ("FIX.5.0", "42802"): "42802 UnderlyingDividendFloatingRateIndexCurvePeriod",
        ("FIX.5.0", "42803"): "42803 UnderlyingDividendFloatingRateIndexCurveUnit",
        ("FIX.5.0", "42804"): "42804 UnderlyingDividendFloatingRateMultiplier",
        ("FIX.5.0", "42805"): "42805 UnderlyingDividendFloatingRateSpread",
        ("FIX.5.0", "42806"): "42806 UnderlyingDividendFloatingRateSpreadPositionType",
        ("FIX.5.0", "42807"): "42807 UnderlyingDividendFloatingRateTreatment",
        ("FIX.5.0", "42808"): "42808 UnderlyingDividendCapRate",
        ("FIX.5.0", "42809"): "42809 UnderlyingDividendCapRateBuySide",
        ("FIX.5.0", "42810"): "42810 UnderlyingDividendCapRateSellSide",
        ("FIX.5.0", "42811"): "42811 UnderlyingDividendFloorRate",
        ("FIX.5.0", "42812"): "42812 UnderlyingDividendFloorRateBuySide",
        ("FIX.5.0", "42813"): "42813 UnderlyingDividendFloorRateSellSide",
        ("FIX.5.0", "42814"): "42814 UnderlyingDividendInitialRate",
        ("FIX.5.0", "42815"): "42815 UnderlyingDividendFinalRateRoundingDirection",
        ("FIX.5.0", "42816"): "42816 UnderlyingDividendFinalRatePrecision",
        ("FIX.5.0", "42817"): "42817 UnderlyingDividendAveragingMethod",
        ("FIX.5.0", "42818"): "42818 UnderlyingDividendNegativeRateTreatment",
        ("FIX.5.0", "42819"): "42819 UnderlyingDividendAccrualPaymentDateRelativeTo",
        ("FIX.5.0", "42820"): "42820 UnderlyingDividendAccrualPaymentDateOffsetPeriod",
        ("FIX.5.0", "42821"): "42821 UnderlyingDividendAccrualPaymentDateOffsetUnit",
        ("FIX.5.0", "42822"): "42822 UnderlyingDividendAccrualPaymentDateOffsetDayType",
        ("FIX.5.0", "42823"): "42823 UnderlyingDividendAccrualPaymentDateUnadjusted",
        ("FIX.5.0", "42824"): "42824 UnderlyingDividendAccrualPaymentDateBusinessDayConvention",
        ("FIX.5.0", "42825"): "42825 UnderlyingDividendAccrualPaymentDateAdjusted",
        ("FIX.5.0", "42826"): "42826 UnderlyingDividendReinvestmentIndicator",
        ("FIX.5.0", "42827"): "42827 UnderlyingDividendEntitlementEvent",
        ("FIX.5.0", "42828"): "42828 UnderlyingDividendAmountType",
        ("FIX.5.0", "42829"): "42829 UnderlyingDividendUnderlierRefID",
        ("FIX.5.0", "42830"): "42830 UnderlyingExtraordinaryDividendPartySide",
        ("FIX.5.0", "42831"): "42831 UnderlyingExtraordinaryDividendAmountType",
        ("FIX.5.0", "42832"): "42832 UnderlyingExtraordinaryDividendCurrency",
        ("FIX.5.0", "42833"): "42833 UnderlyingExtraordinaryDividendDeterminationMethod",
        ("FIX.5.0", "42834"): "42834 UnderlyingDividendAccrualFixedRate",
        ("FIX.5.0", "42835"): "42835 UnderlyingDividendCompoundingMethod",
        ("FIX.5.0", "42836"): "42836 UnderlyingDividendNumOfIndexUnits",
        ("FIX.5.0", "42837"): "42837 UnderlyingDividendCashPercentage",
        ("FIX.5.0", "42838"): "42838 UnderlyingDividendCashEquivalentPercentage",
        ("FIX.5.0", "42839"): "42839 UnderlyingNonCashDividendTreatment",
        ("FIX.5.0", "42840"): "42840 UnderlyingDividendComposition",
        ("FIX.5.0", "42841"): "42841 UnderlyingSpecialDividendsIndicator",
        ("FIX.5.0", "42842"): "42842 UnderlyingMaterialDividendsIndicator",
        ("FIX.5.0", "42843"): "42843 UnderlyingOptionsExchangeDividendsIndicator",
        ("FIX.5.0", "42844"): "42844 UnderlyingAdditionalDividendsIndicator",
        ("FIX.5.0", "42845"): "42845 UnderlyingAllDividendsIndicator",
        ("FIX.5.0", "42846"): "42846 UnderlyingDividendFXTriggerDateRelativeTo",
        ("FIX.5.0", "42847"): "42847 UnderlyingDividendFXTriggerDateOffsetPeriod",
        ("FIX.5.0", "42848"): "42848 UnderlyingDividendFXTriggerDateOffsetUnit",
        ("FIX.5.0", "42849"): "42849 UnderlyingDividendFXTriggerDateOffsetDayType",
        ("FIX.5.0", "42850"): "42850 UnderlyingDividendFXTriggerDateUnadjusted",
        ("FIX.5.0", "42851"): "42851 UnderlyingDividendFXTriggerDateBusinessDayConvention",
        ("FIX.5.0", "42852"): "42852 UnderlyingDividendFXTriggerDateAdjusted",
        ("FIX.5.0", "42853"): "42853 NoUnderlyingDividendFXTriggerDateBusinessCenters",
        ("FIX.5.0", "42854"): "42854 UnderlyingDividendFXTriggerDateBusinessCenter",
        ("FIX.5.0", "42855"): "42855 NoUnderlyingDividendPayments",
        ("FIX.5.0", "42856"): "42856 UnderlyingDividendPaymentDate",
        ("FIX.5.0", "42857"): "42857 UnderlyingDividendPaymentAmount",
        ("FIX.5.0", "42858"): "42858 UnderlyingDividendPaymentCurrency",
        ("FIX.5.0", "42859"): "42859 UnderlyingDividendAccruedInterest",
        ("FIX.5.0", "42860"): "42860 UnderlyingDividendPayoutRatio",
        ("FIX.5.0", "42861"): "42861 UnderlyingDividendPayoutConditions",
        ("FIX.5.0", "42862"): "42862 NoUnderlyingDividendPeriods",
        ("FIX.5.0", "42863"): "42863 UnderlyingDividendPeriodSequence",
        ("FIX.5.0", "42864"): "42864 UnderlyingDividendPeriodStartDateUnadjusted",
        ("FIX.5.0", "42865"): "42865 UnderlyingDividendPeriodEndDateUnadjusted",
        ("FIX.5.0", "42866"): "42866 UnderlyingDividendPeriodUnderlierRefID",
        ("FIX.5.0", "42867"): "42867 UnderlyingDividendPeriodStrikePrice",
        ("FIX.5.0", "42868"): "42868 UnderlyingDividendPeriodBusinessDayConvention",
        ("FIX.5.0", "42869"): "42869 UnderlyingDividendPeriodValuationDateUnadjusted",
        ("FIX.5.0", "42870"): "42870 UnderlyingDividendPeriodValuationDateRelativeTo",
        ("FIX.5.0", "42871"): "42871 UnderlyingDividendPeriodValuationDateOffsetPeriod",
        ("FIX.5.0", "42872"): "42872 UnderlyingDividendPeriodValuationDateOffsetUnit",
        ("FIX.5.0", "42873"): "42873 UnderlyingDividendPeriodValuationDateOffsetDayType",
        ("FIX.5.0", "42874"): "42874 UnderlyingDividendPeriodValuationDateAdjusted",
        ("FIX.5.0", "42875"): "42875 UnderlyingDividendPeriodPaymentDateUnadjusted",
        ("FIX.5.0", "42876"): "42876 UnderlyingDividendPeriodPaymentDateRelativeTo",
        ("FIX.5.0", "42877"): "42877 UnderlyingDividendPeriodPaymentDateOffsetPeriod",
        ("FIX.5.0", "42878"): "42878 UnderlyingDividendPeriodPaymentDateOffsetUnit",
        ("FIX.5.0", "42879"): "42879 UnderlyingDividendPeriodPaymentDateOffsetDayType",
        ("FIX.5.0", "42880"): "42880 UnderlyingDividendPeriodPaymentDateAdjusted",
        ("FIX.5.0", "42881"): "42881 UnderlyingDividendPeriodXID",
        ("FIX.5.0", "42882"): "42882 NoUnderlyingDividendPeriodBusinessCenters",
        ("FIX.5.0", "42883"): "42883 UnderlyingDividendPeriodBusinessCenter",
        ("FIX.5.0", "42884"): "42884 NoUnderlyingExtraordinaryEvents",
        ("FIX.5.0", "42885"): "42885 UnderlyingExtraordinaryEventType",
        ("FIX.5.0", "42886"): "42886 UnderlyingExtraordinaryEventValue",
        ("FIX.5.0", "42887"): "42887 UnderlyingSettlMethodElectingPartySide",
        ("FIX.5.0", "42888"): "42888 UnderlyingMakeWholeDate",
        ("FIX.5.0", "42889"): "42889 UnderlyingMakeWholeAmount",
        ("FIX.5.0", "42890"): "42890 UnderlyingMakeWholeBenchmarkCurveName",
        ("FIX.5.0", "42891"): "42891 UnderlyingMakeWholeBenchmarkCurvePoint",
        ("FIX.5.0", "42892"): "42892 UnderlyingMakeWholeRecallSpread",
        ("FIX.5.0", "42893"): "42893 UnderlyingMakeWholeBenchmarkQuote",
        ("FIX.5.0", "42894"): "42894 UnderlyingMakeWholeInterpolationMethod",
        ("FIX.5.0", "42895"): "42895 UnderlyingPaymentStreamCashSettlIndicator",
        ("FIX.5.0", "42896"): "42896 UnderlyingPaymentStreamCompoundingXIDRef",
        ("FIX.5.0", "42897"): "42897 UnderlyingPaymentStreamCompoundingSpread",
        ("FIX.5.0", "42898"): "42898 UnderlyingPaymentStreamInterpolationMethod",
        ("FIX.5.0", "42899"): "42899 UnderlyingPaymentStreamInterpolationPeriod",
        ("FIX.5.0", "42900"): "42900 UnderlyingPaymentStreamCompoundingFixedRate",
        ("FIX.5.0", "42901"): "42901 NoUnderlyingPaymentStreamCompoundingDates",
        ("FIX.5.0", "42902"): "42902 UnderlyingPaymentStreamCompoundingDate",
        ("FIX.5.0", "42903"): "42903 UnderlyingPaymentStreamCompoundingDateType",
        ("FIX.5.0", "42904"): "42904 UnderlyingPaymentStreamCompoundingDatesBusinessDayConvention",
        ("FIX.5.0", "42905"): "42905 UnderlyingPaymentStreamCompoundingDatesRelativeTo",
        ("FIX.5.0", "42906"): "42906 UnderlyingPaymentStreamCompoundingDatesOffsetPeriod",
        ("FIX.5.0", "42907"): "42907 UnderlyingPaymentStreamCompoundingDatesOffsetUnit",
        ("FIX.5.0", "42908"): "42908 UnderlyingPaymentStreamCompoundingDatesOffsetDayType",
        ("FIX.5.0", "42909"): "42909 UnderlyingPaymentStreamCompoundingPeriodSkip",
        ("FIX.5.0", "42910"): "42910 UnderlyingPaymentStreamCompoundingFrequencyPeriod",
        ("FIX.5.0", "42911"): "42911 UnderlyingPaymentStreamCompoundingFrequencyUnit",
        ("FIX.5.0", "42912"): "42912 UnderlyingPaymentStreamCompoundingRollConvention",
        ("FIX.5.0", "42913"): "42913 UnderlyingPaymentStreamBoundsFirstDateUnadjusted",
        ("FIX.5.0", "42914"): "42914 UnderlyingPaymentStreamBoundsLastDateUnadjusted",
        ("FIX.5.0", "42915"): "42915 NoUnderlyingPaymentStreamCompoundingDatesBusinessCenters",
        ("FIX.5.0", "42916"): "42916 UnderlyingPaymentStreamCompoundingDatesBusinessCenter",
        ("FIX.5.0", "42917"): "42917 UnderlyingPaymentStreamCompoundingEndDateUnadjusted",
        ("FIX.5.0", "42918"): "42918 UnderlyingPaymentStreamCompoundingEndDateRelativeTo",
        ("FIX.5.0", "42919"): "42919 UnderlyingPaymentStreamCompoundingEndDateOffsetPeriod",
        ("FIX.5.0", "42920"): "42920 UnderlyingPaymentStreamCompoundingEndDateOffsetUnit",
        ("FIX.5.0", "42921"): "42921 UnderlyingPaymentStreamCompoundingEndDateOffsetDayType",
        ("FIX.5.0", "42922"): "42922 UnderlyingPaymentStreamCompoundingEndDateAdjusted",
        ("FIX.5.0", "42923"): "42923 UnderlyingPaymentStreamCompoundingRateIndex",
        ("FIX.5.0", "42924"): "42924 UnderlyingPaymentStreamCompoundingRateIndexCurvePeriod",
        ("FIX.5.0", "42925"): "42925 UnderlyingPaymentStreamCompoundingRateIndexCurveUnit",
        ("FIX.5.0", "42926"): "42926 UnderlyingPaymentStreamCompoundingRateMultiplier",
        ("FIX.5.0", "42927"): "42927 UnderlyingPaymentStreamCompoundingRateSpread",
        ("FIX.5.0", "42928"): "42928 UnderlyingPaymentStreamCompoundingRateSpreadPositionType",
        ("FIX.5.0", "42929"): "42929 UnderlyingPaymentStreamCompoundingRateTreatment",
        ("FIX.5.0", "42930"): "42930 UnderlyingPaymentStreamCompoundingCapRate",
        ("FIX.5.0", "42931"): "42931 UnderlyingPaymentStreamCompoundingCapRateBuySide",
        ("FIX.5.0", "42932"): "42932 UnderlyingPaymentStreamCompoundingCapRateSellSide",
        ("FIX.5.0", "42933"): "42933 UnderlyingPaymentStreamCompoundingFloorRate",
        ("FIX.5.0", "42934"): "42934 UnderlyingPaymentStreamCompoundingFloorRateBuySide",
        ("FIX.5.0", "42935"): "42935 UnderlyingPaymentStreamCompoundingFloorRateSellSide",
        ("FIX.5.0", "42936"): "42936 UnderlyingPaymentStreamCompoundingInitialRate",
        ("FIX.5.0", "42937"): "42937 UnderlyingPaymentStreamCompoundingFinalRateRoundingDirection",
        ("FIX.5.0", "42938"): "42938 UnderlyingPaymentStreamCompoundingFinalRatePrecision",
        ("FIX.5.0", "42939"): "42939 UnderlyingPaymentStreamCompoundingAveragingMethod",
        ("FIX.5.0", "42940"): "42940 UnderlyingPaymentStreamCompoundingNegativeRateTreatment",
        ("FIX.5.0", "42941"): "42941 UnderlyingPaymentStreamCompoundingStartDateUnadjusted",
        ("FIX.5.0", "42942"): "42942 UnderlyingPaymentStreamCompoundingStartDateRelativeTo",
        ("FIX.5.0", "42943"): "42943 UnderlyingPaymentStreamCompoundingStartDateOffsetPeriod",
        ("FIX.5.0", "42944"): "42944 UnderlyingPaymentStreamCompoundingStartDateOffsetUnit",
        ("FIX.5.0", "42945"): "42945 UnderlyingPaymentStreamCompoundingStartDateOffsetDayType",
        ("FIX.5.0", "42946"): "42946 UnderlyingPaymentStreamCompoundingStartDateAdjusted",
        ("FIX.5.0", "42947"): "42947 UnderlyingPaymentStreamFormulaImageLength",
        ("FIX.5.0", "42948"): "42948 UnderlyingPaymentStreamFormulaImage",
        ("FIX.5.0", "42949"): "42949 UnderlyingPaymentStreamFinalPricePaymentDateUnadjusted",
        ("FIX.5.0", "42950"): "42950 UnderlyingPaymentStreamFinalPricePaymentDateRelativeTo",
        ("FIX.5.0", "42951"): "42951 UnderlyingPaymentStreamFinalPricePaymentDateOffsetPeriod",
        ("FIX.5.0", "42952"): "42952 UnderlyingPaymentStreamFinalPricePaymentDateOffsetUnit",
        ("FIX.5.0", "42953"): "42953 UnderlyingPaymentStreamFinalPricePaymentDateOffsetDayType",
        ("FIX.5.0", "42954"): "42954 UnderlyingPaymentStreamFinalPricePaymentDateAdjusted",
        ("FIX.5.0", "42955"): "42955 NoUnderlyingPaymentStreamFixingDates",
        ("FIX.5.0", "42956"): "42956 UnderlyingPaymentStreamFixingDate",
        ("FIX.5.0", "42957"): "42957 UnderlyingPaymentStreamFixingDateType",
        ("FIX.5.0", "42958"): "42958 UnderlyingPaymentStreamFirstObservationDateUnadjusted",
        ("FIX.5.0", "42959"): "42959 UnderlyingPaymentStreamFirstObservationDateRelativeTo",
        ("FIX.5.0", "42960"): "42960 UnderlyingPaymentStreamFirstObservationDateOffsetDayType",
        ("FIX.5.0", "42961"): "42961 UnderlyingPaymentStreamFirstObservationDateAdjusted",
        ("FIX.5.0", "42962"): "42962 UnderlyingPaymentStreamUnderlierRefID",
        ("FIX.5.0", "42963"): "42963 UnderlyingReturnRateNotionalReset",
        ("FIX.5.0", "42964"): "42964 UnderlyingPaymentStreamLinkInitialLevel",
        ("FIX.5.0", "42965"): "42965 UnderlyingPaymentStreamLinkClosingLevelIndicator",
        ("FIX.5.0", "42966"): "42966 UnderlyingPaymentStreamLinkExpiringLevelIndicator",
        ("FIX.5.0", "42967"): "42967 UnderlyingPaymentStreamLinkEstimatedTradingDays",
        ("FIX.5.0", "42968"): "42968 UnderlyingPaymentStreamLinkStrikePrice",
        ("FIX.5.0", "42969"): "42969 UnderlyingPaymentStreamLinkStrikePriceType",
        ("FIX.5.0", "42970"): "42970 UnderlyingPaymentStreamLinkMaximumBoundary",
        ("FIX.5.0", "42971"): "42971 UnderlyingPaymentStreamLinkMinimumBoundary",
        ("FIX.5.0", "42972"): "42972 UnderlyingPaymentStreamLinkNumberOfDataSeries",
        ("FIX.5.0", "42973"): "42973 UnderlyingPaymentStreamVarianceUnadjustedCap",
        ("FIX.5.0", "42974"): "42974 UnderlyingPaymentStreamRealizedVarianceMethod",
        ("FIX.5.0", "42975"): "42975 UnderlyingPaymentStreamDaysAdjustmentIndicator",
        ("FIX.5.0", "42976"): "42976 UnderlyingPaymentStreamNearestExchangeContractRefID",
        ("FIX.5.0", "42977"): "42977 UnderlyingPaymentStreamVegaNotionalAmount",
        ("FIX.5.0", "42978"): "42978 UnderlyingPaymentStreamFormulaCurrency",
        ("FIX.5.0", "42979"): "42979 UnderlyingPaymentStreamFormulaCurrencyDeterminationMethod",
        ("FIX.5.0", "42980"): "42980 UnderlyingPaymentStreamFormulaReferenceAmount",
        ("FIX.5.0", "42981"): "42981 NoUnderlyingPaymentStreamFormulas",
        ("FIX.5.0", "42982"): "42982 UnderlyingPaymentStreamFormula",
        ("FIX.5.0", "42983"): "42983 UnderlyingPaymentStreamFormulaDesc",
        ("FIX.5.0", "42984"): "42984 UnderlyingPaymentStubEndDateUnadjusted",
        ("FIX.5.0", "42985"): "42985 UnderlyingPaymentStubEndDateBusinessDayConvention",
        ("FIX.5.0", "42986"): "42986 UnderlyingPaymentStubEndDateRelativeTo",
        ("FIX.5.0", "42987"): "42987 UnderlyingPaymentStubEndDateOffsetPeriod",
        ("FIX.5.0", "42988"): "42988 UnderlyingPaymentStubEndDateOffsetUnit",
        ("FIX.5.0", "42989"): "42989 UnderlyingPaymentStubEndDateOffsetDayType",
        ("FIX.5.0", "42990"): "42990 UnderlyingPaymentStubEndDateAdjusted",
        ("FIX.5.0", "42991"): "42991 NoUnderlyingPaymentStubEndDateBusinessCenters",
        ("FIX.5.0", "42992"): "42992 UnderlyingPaymentStubEndDateBusinessCenter",
        ("FIX.5.0", "42993"): "42993 UnderlyingPaymentStubStartDateUnadjusted",
        ("FIX.5.0", "42994"): "42994 UnderlyingPaymentStubStartDateBusinessDayConvention",
        ("FIX.5.0", "42995"): "42995 UnderlyingPaymentStubStartDateRelativeTo",
        ("FIX.5.0", "42996"): "42996 UnderlyingPaymentStubStartDateOffsetPeriod",
        ("FIX.5.0", "42997"): "42997 UnderlyingPaymentStubStartDateOffsetUnit",
        ("FIX.5.0", "42998"): "42998 UnderlyingPaymentStubStartDateOffsetDayType",
        ("FIX.5.0", "42999"): "42999 UnderlyingPaymentStubStartDateAdjusted",
        ("FIX.5.0", "43000"): "43000 NoUnderlyingPaymentStubStartDateBusinessCenters",
        ("FIX.5.0", "43001"): "43001 UnderlyingPaymentStubStartDateBusinessCenter",
        ("FIX.5.0", "43002"): "43002 UnderlyingProvisionBreakFeeElection",
        ("FIX.5.0", "43003"): "43003 UnderlyingProvisionBreakFeeRate",
        ("FIX.5.0", "43004"): "43004 UnderlyingRateSpreadInitialValue",
        ("FIX.5.0", "43005"): "43005 NoUnderlyingRateSpreadSteps",
        ("FIX.5.0", "43006"): "43006 UnderlyingRateSpreadStepDate",
        ("FIX.5.0", "43007"): "43007 UnderlyingRateSpreadStepValue",
        ("FIX.5.0", "43008"): "43008 NoUnderlyingReturnRateDates",
        ("FIX.5.0", "43009"): "43009 UnderlyingReturnRateDateMode",
        ("FIX.5.0", "43010"): "43010 UnderlyingReturnRateValuationDateRelativeTo",
        ("FIX.5.0", "43011"): "43011 UnderlyingReturnRateValuationDateOffsetPeriod",
        ("FIX.5.0", "43012"): "43012 UnderlyingReturnRateValuationDateOffsetUnit",
        ("FIX.5.0", "43013"): "43013 UnderlyingReturnRateValuationDateOffsetDayType",
        ("FIX.5.0", "43014"): "43014 UnderlyingReturnRateValuationStartDateUnadjusted",
        ("FIX.5.0", "43015"): "43015 UnderlyingReturnRateValuationStartDateRelativeTo",
        ("FIX.5.0", "43016"): "43016 UnderlyingReturnRateValuationStartDateOffsetPeriod",
        ("FIX.5.0", "43017"): "43017 UnderlyingReturnRateValuationStartDateOffsetUnit",
        ("FIX.5.0", "43018"): "43018 UnderlyingReturnRateValuationStartDateOffsetDayType",
        ("FIX.5.0", "43019"): "43019 UnderlyingReturnRateValuationStartDateAdjusted",
        ("FIX.5.0", "43020"): "43020 UnderlyingReturnRateValuationEndDateUnadjusted",
        ("FIX.5.0", "43021"): "43021 UnderlyingReturnRateValuationEndDateRelativeTo",
        ("FIX.5.0", "43022"): "43022 UnderlyingReturnRateValuationEndDateOffsetPeriod",
        ("FIX.5.0", "43023"): "43023 UnderlyingReturnRateValuationEndDateOffsetUnit",
        ("FIX.5.0", "43024"): "43024 UnderlyingReturnRateValuationEndDateOffsetDayType",
        ("FIX.5.0", "43025"): "43025 UnderlyingReturnRateValuationEndDateAdjusted",
        ("FIX.5.0", "43026"): "43026 UnderlyingReturnRateValuationFrequencyPeriod",
        ("FIX.5.0", "43027"): "43027 UnderlyingReturnRateValuationFrequencyUnit",
        ("FIX.5.0", "43028"): "43028 UnderlyingReturnRateValuationFrequencyRollConvention",
        ("FIX.5.0", "43029"): "43029 UnderlyingReturnRateValuationDateBusinessDayConvention",
        ("FIX.5.0", "43030"): "43030 NoUnderlyingReturnRateFXConversions",
        ("FIX.5.0", "43031"): "43031 UnderlyingReturnRateFXCurrencySymbol",
        ("FIX.5.0", "43032"): "43032 UnderlyingReturnRateFXRate",
        ("FIX.5.0", "43033"): "43033 UnderlyingReturnRateFXRateCalc",
        ("FIX.5.0", "43034"): "43034 NoUnderlyingReturnRates",
        ("FIX.5.0", "43035"): "43035 UnderlyingReturnRatePriceSequence",
        ("FIX.5.0", "43036"): "43036 UnderlyingReturnRateCommissionBasis",
        ("FIX.5.0", "43037"): "43037 UnderlyingReturnRateCommissionAmount",
        ("FIX.5.0", "43038"): "43038 UnderlyingReturnRateCommissionCurrency",
        ("FIX.5.0", "43039"): "43039 UnderlyingReturnRateTotalCommissionPerTrade",
        ("FIX.5.0", "43040"): "43040 UnderlyingReturnRateDeterminationMethod",
        ("FIX.5.0", "43041"): "43041 UnderlyingReturnRateAmountRelativeTo",
        ("FIX.5.0", "43042"): "43042 UnderlyingReturnRateQuoteMeasureType",
        ("FIX.5.0", "43043"): "43043 UnderlyingReturnRateQuoteUnits",
        ("FIX.5.0", "43044"): "43044 UnderlyingReturnRateQuoteMethod",
        ("FIX.5.0", "43045"): "43045 UnderlyingReturnRateQuoteCurrency",
        ("FIX.5.0", "43046"): "43046 UnderlyingReturnRateQuoteCurrencyType",
        ("FIX.5.0", "43047"): "43047 UnderlyingReturnRateQuoteTimeType",
        ("FIX.5.0", "43048"): "43048 UnderlyingReturnRateQuoteTime",
        ("FIX.5.0", "43049"): "43049 UnderlyingReturnRateQuoteDate",
        ("FIX.5.0", "43050"): "43050 UnderlyingReturnRateQuoteExpirationTime",
        ("FIX.5.0", "43051"): "43051 UnderlyingReturnRateQuoteBusinessCenter",
        ("FIX.5.0", "43052"): "43052 UnderlyingReturnRateQuoteExchange",
        ("FIX.5.0", "43053"): "43053 UnderlyingReturnRateQuotePricingModel",
        ("FIX.5.0", "43054"): "43054 UnderlyingReturnRateCashFlowType",
        ("FIX.5.0", "43055"): "43055 UnderlyingReturnRateValuationTimeType",
        ("FIX.5.0", "43056"): "43056 UnderlyingReturnRateValuationTime",
        ("FIX.5.0", "43057"): "43057 UnderlyingReturnRateValuationTimeBusinessCenter",
        ("FIX.5.0", "43058"): "43058 UnderlyingReturnRateValuationPriceOption",
        ("FIX.5.0", "43059"): "43059 UnderlyingReturnRateFinalPriceFallback",
        ("FIX.5.0", "43060"): "43060 NoUnderlyingReturnRateInformationSources",
        ("FIX.5.0", "43061"): "43061 UnderlyingReturnRateInformationSource",
        ("FIX.5.0", "43062"): "43062 UnderlyingReturnRateReferencePage",
        ("FIX.5.0", "43063"): "43063 UnderlyingReturnRateReferencePageHeading",
        ("FIX.5.0", "43064"): "43064 NoUnderlyingReturnRatePrices",
        ("FIX.5.0", "43065"): "43065 UnderlyingReturnRatePriceBasis",
        ("FIX.5.0", "43066"): "43066 UnderlyingReturnRatePrice",
        ("FIX.5.0", "43067"): "43067 UnderlyingReturnRatePriceCurrency",
        ("FIX.5.0", "43068"): "43068 UnderlyingReturnRatePriceType",
        ("FIX.5.0", "43069"): "43069 NoUnderlyingReturnRateValuationDateBusinessCenters",
        ("FIX.5.0", "43070"): "43070 UnderlyingReturnRateValuationDateBusinessCenter",
        ("FIX.5.0", "43071"): "43071 NoUnderlyingReturnRateValuationDates",
        ("FIX.5.0", "43072"): "43072 UnderlyingReturnRateValuationDate",
        ("FIX.5.0", "43073"): "43073 UnderlyingReturnRateValuationDateType",
        ("FIX.5.0", "43074"): "43074 NoUnderlyingSettlMethodElectionDateBusinessCenters",
        ("FIX.5.0", "43075"): "43075 UnderlyingSettlMethodElectionDateBusinessCenter",
        ("FIX.5.0", "43076"): "43076 UnderlyingSettlMethodElectionDateUnadjusted",
        ("FIX.5.0", "43077"): "43077 UnderlyingSettlMethodElectionDateBusinessDayConvention",
        ("FIX.5.0", "43078"): "43078 UnderlyingSettlMethodElectionDateRelativeTo",
        ("FIX.5.0", "43079"): "43079 UnderlyingSettlMethodElectionDateOffsetPeriod",
        ("FIX.5.0", "43080"): "43080 UnderlyingSettlMethodElectionDateOffsetUnit",
        ("FIX.5.0", "43081"): "43081 UnderlyingSettlMethodElectionDateOffsetDayType",
        ("FIX.5.0", "43082"): "43082 UnderlyingSettlMethodElectionDateAdjusted",
        ("FIX.5.0", "43083"): "43083 UnderlyingStreamVersion",
        ("FIX.5.0", "43084"): "43084 UnderlyingStreamVersionEffectiveDate",
        ("FIX.5.0", "43085"): "43085 UnderlyingStreamNotionalDeterminationMethod",
        ("FIX.5.0", "43086"): "43086 UnderlyingStreamNotionalAdjustments",
        ("FIX.5.0", "43087"): "43087 PaymentDesc",
        ("FIX.5.0", "43088"): "43088 LegPaymentStreamRateIndexID",
        ("FIX.5.0", "43089"): "43089 LegPaymentStreamRateIndexIDSource",
        ("FIX.5.0", "43090"): "43090 PaymentStreamRateIndexID",
        ("FIX.5.0", "43091"): "43091 PaymentStreamRateIndexIDSource",
        ("FIX.5.0", "43092"): "43092 UnderlyingPaymentStreamRateIndexID",
        ("FIX.5.0", "43093"): "43093 UnderlyingPaymentStreamRateIndexIDSource",
        ("FIX.5.0", "43094"): "43094 DeliveryStreamRouteOrCharter",
        ("FIX.5.0", "43095"): "43095 LegDeliveryStreamRouteOrCharter",
        ("FIX.5.0", "43096"): "43096 UnderlyingDeliveryStreamRouteOrCharter",
        ("FIX.5.0", "50000"): "50000 BatchID",
        ("FIX.5.0", "50001"): "50001 BatchTotalMessages",
        ("FIX.5.0", "50002"): "50002 BatchProcessMode",
    }

    __VALUE_MAP = {
        ("FIX.4.0", "4", "B"): "B .. Buy",
        ("FIX.4.0", "4", "S"): "S .. Sell",
        ("FIX.4.0", "4", "T"): "T .. Trade",
        ("FIX.4.0", "4", "X"): "X .. Cross",
        ("FIX.4.0", "5", "C"): "C .. Cancel",
        ("FIX.4.0", "5", "N"): "N .. New",
        ("FIX.4.0", "5", "R"): "R .. Replace",
        ("FIX.4.0", "13", "1"): "1 .. PerUnit",
        ("FIX.4.0", "13", "2"): "2 .. Percent",
        ("FIX.4.0", "13", "3"): "3 .. Absolute",
        ("FIX.4.0", "18", "0"): "0 .. StayOnOfferSide",
        ("FIX.4.0", "18", "1"): "1 .. NotHeld",
        ("FIX.4.0", "18", "2"): "2 .. Work",
        ("FIX.4.0", "18", "3"): "3 .. GoAlong",
        ("FIX.4.0", "18", "4"): "4 .. OverTheDay",
        ("FIX.4.0", "18", "5"): "5 .. Held",
        ("FIX.4.0", "18", "6"): "6 .. ParticipateDoNotInitiate",
        ("FIX.4.0", "18", "7"): "7 .. StrictScale",
        ("FIX.4.0", "18", "8"): "8 .. TryToScale",
        ("FIX.4.0", "18", "9"): "9 .. StayOnBidSide",
        ("FIX.4.0", "18", "A"): "A .. NoCross",
        ("FIX.4.0", "18", "B"): "B .. OKToCross",
        ("FIX.4.0", "18", "C"): "C .. CallFirst",
        ("FIX.4.0", "18", "D"): "D .. PercentOfVolume",
        ("FIX.4.0", "18", "E"): "E .. DoNotIncrease",
        ("FIX.4.0", "18", "F"): "F .. DoNotReduce",
        ("FIX.4.0", "18", "G"): "G .. AllOrNone",
        ("FIX.4.0", "18", "I"): "I .. InstitutionsOnly",
        ("FIX.4.0", "18", "L"): "L .. LastPeg",
        ("FIX.4.0", "18", "M"): "M .. MidPricePeg",
        ("FIX.4.0", "18", "N"): "N .. NonNegotiable",
        ("FIX.4.0", "18", "O"): "O .. OpeningPeg",
        ("FIX.4.0", "18", "P"): "P .. MarketPeg",
        ("FIX.4.0", "18", "R"): "R .. PrimaryPeg",
        ("FIX.4.0", "18", "S"): "S .. Suspend",
        ("FIX.4.0", "20", "0"): "0 .. New",
        ("FIX.4.0", "20", "1"): "1 .. Cancel",
        ("FIX.4.0", "20", "2"): "2 .. Correct",
        ("FIX.4.0", "20", "3"): "3 .. Status",
        ("FIX.4.0", "21", "1"): "1 .. AutomatedExecutionNoIntervention",
        ("FIX.4.0", "21", "2"): "2 .. AutomatedExecutionInterventionOK",
        ("FIX.4.0", "21", "3"): "3 .. ManualOrder",
        ("FIX.4.0", "22", "1"): "1 .. CUSIP",
        ("FIX.4.0", "22", "2"): "2 .. SEDOL",
        ("FIX.4.0", "22", "3"): "3 .. QUIK",
        ("FIX.4.0", "22", "4"): "4 .. ISINNumber",
        ("FIX.4.0", "22", "5"): "5 .. RICCode",
        ("FIX.4.0", "24", "A"): "A .. Autex",
        ("FIX.4.0", "24", "B"): "B .. Bridge",
        ("FIX.4.0", "25", "H"): "H .. High",
        ("FIX.4.0", "25", "L"): "L .. Low",
        ("FIX.4.0", "25", "M"): "M .. Medium",
        ("FIX.4.0", "27", "L"): "L .. Large",
        ("FIX.4.0", "27", "M"): "M .. Medium",
        ("FIX.4.0", "27", "S"): "S .. Small",
        ("FIX.4.0", "28", "C"): "C .. Cancel",
        ("FIX.4.0", "28", "N"): "N .. New",
        ("FIX.4.0", "28", "R"): "R .. Replace",
        ("FIX.4.0", "29", "1"): "1 .. Agent",
        ("FIX.4.0", "29", "2"): "2 .. CrossAsAgent",
        ("FIX.4.0", "29", "3"): "3 .. CrossAsPrincipal",
        ("FIX.4.0", "29", "4"): "4 .. Principal",
        ("FIX.4.0", "35", "0"): "0 .. Heartbeat",
        ("FIX.4.0", "35", "1"): "1 .. TestRequest",
        ("FIX.4.0", "35", "2"): "2 .. ResendRequest",
        ("FIX.4.0", "35", "3"): "3 .. Reject",
        ("FIX.4.0", "35", "4"): "4 .. SequenceReset",
        ("FIX.4.0", "35", "5"): "5 .. Logout",
        ("FIX.4.0", "35", "6"): "6 .. IOI",
        ("FIX.4.0", "35", "7"): "7 .. Advertisement",
        ("FIX.4.0", "35", "8"): "8 .. ExecutionReport",
        ("FIX.4.0", "35", "9"): "9 .. OrderCancelReject",
        ("FIX.4.0", "35", "A"): "A .. Logon",
        ("FIX.4.0", "35", "B"): "B .. News",
        ("FIX.4.0", "35", "C"): "C .. Email",
        ("FIX.4.0", "35", "D"): "D .. NewOrderSingle",
        ("FIX.4.0", "35", "E"): "E .. NewOrderList",
        ("FIX.4.0", "35", "F"): "F .. OrderCancelRequest",
        ("FIX.4.0", "35", "G"): "G .. OrderCancelReplaceRequest",
        ("FIX.4.0", "35", "H"): "H .. OrderStatusRequest",
        ("FIX.4.0", "35", "J"): "J .. AllocationInstruction",
        ("FIX.4.0", "35", "K"): "K .. ListCancelRequest",
        ("FIX.4.0", "35", "L"): "L .. ListExecute",
        ("FIX.4.0", "35", "M"): "M .. ListStatusRequest",
        ("FIX.4.0", "35", "N"): "N .. ListStatus",
        ("FIX.4.0", "35", "P"): "P .. AllocationInstructionAck",
        ("FIX.4.0", "35", "Q"): "Q .. DontKnowTrade",
        ("FIX.4.0", "35", "R"): "R .. QuoteRequest",
        ("FIX.4.0", "35", "S"): "S .. Quote",
        ("FIX.4.0", "39", "0"): "0 .. New",
        ("FIX.4.0", "39", "1"): "1 .. PartiallyFilled",
        ("FIX.4.0", "39", "2"): "2 .. Filled",
        ("FIX.4.0", "39", "3"): "3 .. DoneForDay",
        ("FIX.4.0", "39", "4"): "4 .. Canceled",
        ("FIX.4.0", "39", "5"): "5 .. Replaced",
        ("FIX.4.0", "39", "6"): "6 .. PendingCancel",
        ("FIX.4.0", "39", "7"): "7 .. Stopped",
        ("FIX.4.0", "39", "8"): "8 .. Rejected",
        ("FIX.4.0", "39", "9"): "9 .. Suspended",
        ("FIX.4.0", "39", "A"): "A .. PendingNew",
        ("FIX.4.0", "39", "B"): "B .. Calculated",
        ("FIX.4.0", "39", "C"): "C .. Expired",
        ("FIX.4.0", "40", "1"): "1 .. Market",
        ("FIX.4.0", "40", "2"): "2 .. Limit",
        ("FIX.4.0", "40", "3"): "3 .. Stop",
        ("FIX.4.0", "40", "4"): "4 .. StopLimit",
        ("FIX.4.0", "40", "5"): "5 .. MarketOnClose",
        ("FIX.4.0", "40", "6"): "6 .. WithOrWithout",
        ("FIX.4.0", "40", "7"): "7 .. LimitOrBetter",
        ("FIX.4.0", "40", "8"): "8 .. LimitWithOrWithout",
        ("FIX.4.0", "40", "9"): "9 .. OnBasis",
        ("FIX.4.0", "40", "A"): "A .. OnClose",
        ("FIX.4.0", "40", "B"): "B .. LimitOnClose",
        ("FIX.4.0", "40", "C"): "C .. ForexMarket",
        ("FIX.4.0", "40", "D"): "D .. PreviouslyQuoted",
        ("FIX.4.0", "40", "E"): "E .. PreviouslyIndicated",
        ("FIX.4.0", "40", "P"): "P .. Pegged",
        ("FIX.4.0", "43", "N"): "N .. OriginalTransmission",
        ("FIX.4.0", "43", "Y"): "Y .. PossibleDuplicate",
        ("FIX.4.0", "47", "A"): "A .. AgencySingleOrder",
        ("FIX.4.0", "47", "C"): "C .. ProprietaryNonAlgo",
        ("FIX.4.0", "47", "D"): "D .. ProgramOrderMember",
        ("FIX.4.0", "47", "I"): "I .. IndividualInvestor",
        ("FIX.4.0", "47", "J"): "J .. ProprietaryAlgo",
        ("FIX.4.0", "47", "K"): "K .. AgencyAlgo",
        ("FIX.4.0", "47", "M"): "M .. ProgramOrderOtherMember",
        ("FIX.4.0", "47", "N"): "N .. AgentForOtherMember",
        ("FIX.4.0", "47", "U"): "U .. AgencyIndexArb",
        ("FIX.4.0", "47", "W"): "W .. AllOtherOrdersAsAgentForOtherMember",
        ("FIX.4.0", "47", "Y"): "Y .. AgencyNonAlgo",
        ("FIX.4.0", "54", "1"): "1 .. Buy",
        ("FIX.4.0", "54", "2"): "2 .. Sell",
        ("FIX.4.0", "54", "3"): "3 .. BuyMinus",
        ("FIX.4.0", "54", "4"): "4 .. SellPlus",
        ("FIX.4.0", "54", "5"): "5 .. SellShort",
        ("FIX.4.0", "54", "6"): "6 .. SellShortExempt",
        ("FIX.4.0", "59", "0"): "0 .. Day",
        ("FIX.4.0", "59", "1"): "1 .. GoodTillCancel",
        ("FIX.4.0", "59", "2"): "2 .. AtTheOpening",
        ("FIX.4.0", "59", "3"): "3 .. ImmediateOrCancel",
        ("FIX.4.0", "59", "4"): "4 .. FillOrKill",
        ("FIX.4.0", "59", "5"): "5 .. GoodTillCrossing",
        ("FIX.4.0", "59", "6"): "6 .. GoodTillDate",
        ("FIX.4.0", "61", "0"): "0 .. Normal",
        ("FIX.4.0", "61", "1"): "1 .. Flash",
        ("FIX.4.0", "61", "2"): "2 .. Background",
        ("FIX.4.0", "63", "0"): "0 .. Regular",
        ("FIX.4.0", "63", "1"): "1 .. Cash",
        ("FIX.4.0", "63", "2"): "2 .. NextDay",
        ("FIX.4.0", "63", "3"): "3 .. TPlus2",
        ("FIX.4.0", "63", "4"): "4 .. TPlus3",
        ("FIX.4.0", "63", "5"): "5 .. TPlus4",
        ("FIX.4.0", "63", "6"): "6 .. Future",
        ("FIX.4.0", "63", "7"): "7 .. WhenAndIfIssued",
        ("FIX.4.0", "63", "8"): "8 .. SellersOption",
        ("FIX.4.0", "63", "9"): "9 .. TPlus5",
        ("FIX.4.0", "71", "0"): "0 .. New",
        ("FIX.4.0", "71", "1"): "1 .. Replace",
        ("FIX.4.0", "71", "2"): "2 .. Cancel",
        ("FIX.4.0", "81", "0"): "0 .. Regular",
        ("FIX.4.0", "81", "1"): "1 .. SoftDollar",
        ("FIX.4.0", "81", "2"): "2 .. StepIn",
        ("FIX.4.0", "81", "3"): "3 .. StepOut",
        ("FIX.4.0", "81", "4"): "4 .. SoftDollarStepIn",
        ("FIX.4.0", "81", "5"): "5 .. SoftDollarStepOut",
        ("FIX.4.0", "81", "6"): "6 .. PlanSponsor",
        ("FIX.4.0", "87", "0"): "0 .. Accepted",
        ("FIX.4.0", "87", "1"): "1 .. BlockLevelReject",
        ("FIX.4.0", "87", "2"): "2 .. AccountLevelReject",
        ("FIX.4.0", "87", "3"): "3 .. Received",
        ("FIX.4.0", "88", "0"): "0 .. UnknownAccount",
        ("FIX.4.0", "88", "1"): "1 .. IncorrectQuantity",
        ("FIX.4.0", "88", "2"): "2 .. IncorrectAveragegPrice",
        ("FIX.4.0", "88", "3"): "3 .. UnknownExecutingBrokerMnemonic",
        ("FIX.4.0", "88", "4"): "4 .. CommissionDifference",
        ("FIX.4.0", "88", "5"): "5 .. UnknownOrderID",
        ("FIX.4.0", "88", "6"): "6 .. UnknownListID",
        ("FIX.4.0", "88", "7"): "7 .. OtherSeeText",
        ("FIX.4.0", "94", "0"): "0 .. New",
        ("FIX.4.0", "94", "1"): "1 .. Reply",
        ("FIX.4.0", "94", "2"): "2 .. AdminReply",
        ("FIX.4.0", "98", "0"): "0 .. None",
        ("FIX.4.0", "98", "1"): "1 .. PKCS",
        ("FIX.4.0", "98", "2"): "2 .. DES",
        ("FIX.4.0", "98", "3"): "3 .. PKCSDES",
        ("FIX.4.0", "98", "4"): "4 .. PGPDES",
        ("FIX.4.0", "98", "5"): "5 .. PGPDESMD5",
        ("FIX.4.0", "98", "6"): "6 .. PEM",
        ("FIX.4.0", "100", "0"): "0 .. None",
        ("FIX.4.0", "100", "4"): "4 .. POSIT",
        ("FIX.4.0", "102", "0"): "0 .. TooLateToCancel",
        ("FIX.4.0", "102", "1"): "1 .. UnknownOrder",
        ("FIX.4.0", "103", "0"): "0 .. BrokerCredit",
        ("FIX.4.0", "103", "1"): "1 .. UnknownSymbol",
        ("FIX.4.0", "103", "2"): "2 .. ExchangeClosed",
        ("FIX.4.0", "103", "3"): "3 .. OrderExceedsLimit",
        ("FIX.4.0", "103", "4"): "4 .. TooLateToEnter",
        ("FIX.4.0", "104", "A"): "A .. AllOrNone",
        ("FIX.4.0", "104", "C"): "C .. AtTheClose",
        ("FIX.4.0", "104", "I"): "I .. InTouchWith",
        ("FIX.4.0", "104", "L"): "L .. Limit",
        ("FIX.4.0", "104", "M"): "M .. MoreBehind",
        ("FIX.4.0", "104", "O"): "O .. AtTheOpen",
        ("FIX.4.0", "104", "P"): "P .. TakingAPosition",
        ("FIX.4.0", "104", "Q"): "Q .. AtTheMarket",
        ("FIX.4.0", "104", "S"): "S .. PortfolioShown",
        ("FIX.4.0", "104", "T"): "T .. ThroughTheDay",
        ("FIX.4.0", "104", "V"): "V .. Versus",
        ("FIX.4.0", "104", "W"): "W .. Indication",
        ("FIX.4.0", "104", "X"): "X .. CrossingOpportunity",
        ("FIX.4.0", "113", "N"): "N .. SenderReports",
        ("FIX.4.0", "113", "Y"): "Y .. ReceiverReports",
        ("FIX.4.0", "114", "N"): "N .. No",
        ("FIX.4.0", "114", "Y"): "Y .. Yes",
        ("FIX.4.0", "121", "N"): "N .. DoNotExecuteForexAfterSecurityTrade",
        ("FIX.4.0", "121", "Y"): "Y .. ExecuteForexAfterSecurityTrade",
        ("FIX.4.0", "123", "N"): "N .. SequenceReset",
        ("FIX.4.0", "123", "Y"): "Y .. GapFillMessage",
        ("FIX.4.0", "125", "F"): "F .. FullRemainingQuantity",
        ("FIX.4.0", "125", "P"): "P .. PartialCancel",
        ("FIX.4.0", "127", "A"): "A .. UnknownSymbol",
        ("FIX.4.0", "127", "B"): "B .. WrongSide",
        ("FIX.4.0", "127", "C"): "C .. QuantityExceedsOrder",
        ("FIX.4.0", "127", "D"): "D .. NoMatchingOrder",
        ("FIX.4.0", "127", "E"): "E .. PriceExceedsLimit",
        ("FIX.4.0", "127", "Z"): "Z .. Other",
        ("FIX.4.0", "130", "N"): "N .. NotNatural",
        ("FIX.4.0", "130", "Y"): "Y .. Natural",
        ("FIX.4.0", "139", "1"): "1 .. Regulatory",
        ("FIX.4.0", "139", "2"): "2 .. Tax",
        ("FIX.4.0", "139", "3"): "3 .. LocalCommission",
        ("FIX.4.0", "139", "4"): "4 .. ExchangeFees",
        ("FIX.4.0", "139", "5"): "5 .. Stamp",
        ("FIX.4.0", "139", "6"): "6 .. Levy",
        ("FIX.4.0", "139", "7"): "7 .. Other",
        ("FIX.4.2", "4", "B"): "B .. Buy",
        ("FIX.4.2", "4", "S"): "S .. Sell",
        ("FIX.4.2", "4", "T"): "T .. Trade",
        ("FIX.4.2", "4", "X"): "X .. Cross",
        ("FIX.4.2", "5", "C"): "C .. Cancel",
        ("FIX.4.2", "5", "N"): "N .. New",
        ("FIX.4.2", "5", "R"): "R .. Replace",
        ("FIX.4.2", "13", "1"): "1 .. PerUnit",
        ("FIX.4.2", "13", "2"): "2 .. Percent",
        ("FIX.4.2", "13", "3"): "3 .. Absolute",
        ("FIX.4.2", "18", "0"): "0 .. StayOnOfferSide",
        ("FIX.4.2", "18", "1"): "1 .. NotHeld",
        ("FIX.4.2", "18", "2"): "2 .. Work",
        ("FIX.4.2", "18", "3"): "3 .. GoAlong",
        ("FIX.4.2", "18", "4"): "4 .. OverTheDay",
        ("FIX.4.2", "18", "5"): "5 .. Held",
        ("FIX.4.2", "18", "6"): "6 .. ParticipateDoNotInitiate",
        ("FIX.4.2", "18", "7"): "7 .. StrictScale",
        ("FIX.4.2", "18", "8"): "8 .. TryToScale",
        ("FIX.4.2", "18", "9"): "9 .. StayOnBidSide",
        ("FIX.4.2", "18", "A"): "A .. NoCross",
        ("FIX.4.2", "18", "B"): "B .. OKToCross",
        ("FIX.4.2", "18", "C"): "C .. CallFirst",
        ("FIX.4.2", "18", "D"): "D .. PercentOfVolume",
        ("FIX.4.2", "18", "E"): "E .. DoNotIncrease",
        ("FIX.4.2", "18", "F"): "F .. DoNotReduce",
        ("FIX.4.2", "18", "G"): "G .. AllOrNone",
        ("FIX.4.2", "18", "I"): "I .. InstitutionsOnly",
        ("FIX.4.2", "18", "L"): "L .. LastPeg",
        ("FIX.4.2", "18", "M"): "M .. MidPricePeg",
        ("FIX.4.2", "18", "N"): "N .. NonNegotiable",
        ("FIX.4.2", "18", "O"): "O .. OpeningPeg",
        ("FIX.4.2", "18", "P"): "P .. MarketPeg",
        ("FIX.4.2", "18", "R"): "R .. PrimaryPeg",
        ("FIX.4.2", "18", "S"): "S .. Suspend",
        ("FIX.4.2", "18", "T"): "T .. FixedPegToLocalBestBidOrOfferAtTimeOfOrder",
        ("FIX.4.2", "18", "U"): "U .. CustomerDisplayInstruction",
        ("FIX.4.2", "18", "V"): "V .. Netting",
        ("FIX.4.2", "18", "W"): "W .. PegToVWAP",
        ("FIX.4.2", "20", "0"): "0 .. New",
        ("FIX.4.2", "20", "1"): "1 .. Cancel",
        ("FIX.4.2", "20", "2"): "2 .. Correct",
        ("FIX.4.2", "20", "3"): "3 .. Status",
        ("FIX.4.2", "21", "1"): "1 .. AutomatedExecutionNoIntervention",
        ("FIX.4.2", "21", "2"): "2 .. AutomatedExecutionInterventionOK",
        ("FIX.4.2", "21", "3"): "3 .. ManualOrder",
        ("FIX.4.2", "22", "1"): "1 .. CUSIP",
        ("FIX.4.2", "22", "2"): "2 .. SEDOL",
        ("FIX.4.2", "22", "3"): "3 .. QUIK",
        ("FIX.4.2", "22", "4"): "4 .. ISINNumber",
        ("FIX.4.2", "22", "5"): "5 .. RICCode",
        ("FIX.4.2", "22", "6"): "6 .. ISOCurrencyCode",
        ("FIX.4.2", "22", "7"): "7 .. ISOCountryCode",
        ("FIX.4.2", "22", "8"): "8 .. ExchangeSymbol",
        ("FIX.4.2", "22", "9"): "9 .. ConsolidatedTapeAssociation",
        ("FIX.4.2", "25", "H"): "H .. High",
        ("FIX.4.2", "25", "L"): "L .. Low",
        ("FIX.4.2", "25", "M"): "M .. Medium",
        ("FIX.4.2", "27", "L"): "L .. Large",
        ("FIX.4.2", "27", "M"): "M .. Medium",
        ("FIX.4.2", "27", "S"): "S .. Small",
        ("FIX.4.2", "28", "C"): "C .. Cancel",
        ("FIX.4.2", "28", "N"): "N .. New",
        ("FIX.4.2", "28", "R"): "R .. Replace",
        ("FIX.4.2", "29", "1"): "1 .. Agent",
        ("FIX.4.2", "29", "2"): "2 .. CrossAsAgent",
        ("FIX.4.2", "29", "3"): "3 .. CrossAsPrincipal",
        ("FIX.4.2", "29", "4"): "4 .. Principal",
        ("FIX.4.2", "35", "0"): "0 .. Heartbeat",
        ("FIX.4.2", "35", "1"): "1 .. TestRequest",
        ("FIX.4.2", "35", "2"): "2 .. ResendRequest",
        ("FIX.4.2", "35", "3"): "3 .. Reject",
        ("FIX.4.2", "35", "4"): "4 .. SequenceReset",
        ("FIX.4.2", "35", "5"): "5 .. Logout",
        ("FIX.4.2", "35", "6"): "6 .. IOI",
        ("FIX.4.2", "35", "7"): "7 .. Advertisement",
        ("FIX.4.2", "35", "8"): "8 .. ExecutionReport",
        ("FIX.4.2", "35", "9"): "9 .. OrderCancelReject",
        ("FIX.4.2", "35", "A"): "A .. Logon",
        ("FIX.4.2", "35", "B"): "B .. News",
        ("FIX.4.2", "35", "C"): "C .. Email",
        ("FIX.4.2", "35", "D"): "D .. NewOrderSingle",
        ("FIX.4.2", "35", "E"): "E .. NewOrderList",
        ("FIX.4.2", "35", "F"): "F .. OrderCancelRequest",
        ("FIX.4.2", "35", "G"): "G .. OrderCancelReplaceRequest",
        ("FIX.4.2", "35", "H"): "H .. OrderStatusRequest",
        ("FIX.4.2", "35", "J"): "J .. AllocationInstruction",
        ("FIX.4.2", "35", "K"): "K .. ListCancelRequest",
        ("FIX.4.2", "35", "L"): "L .. ListExecute",
        ("FIX.4.2", "35", "M"): "M .. ListStatusRequest",
        ("FIX.4.2", "35", "N"): "N .. ListStatus",
        ("FIX.4.2", "35", "P"): "P .. AllocationInstructionAck",
        ("FIX.4.2", "35", "Q"): "Q .. DontKnowTrade",
        ("FIX.4.2", "35", "R"): "R .. QuoteRequest",
        ("FIX.4.2", "35", "S"): "S .. Quote",
        ("FIX.4.2", "35", "T"): "T .. SettlementInstructions",
        ("FIX.4.2", "35", "V"): "V .. MarketDataRequest",
        ("FIX.4.2", "35", "W"): "W .. MarketDataSnapshotFullRefresh",
        ("FIX.4.2", "35", "X"): "X .. MarketDataIncrementalRefresh",
        ("FIX.4.2", "35", "Y"): "Y .. MarketDataRequestReject",
        ("FIX.4.2", "35", "Z"): "Z .. QuoteCancel",
        ("FIX.4.2", "35", "a"): "a .. QuoteStatusRequest",
        ("FIX.4.2", "35", "b"): "b .. MassQuoteAcknowledgement",
        ("FIX.4.2", "35", "c"): "c .. SecurityDefinitionRequest",
        ("FIX.4.2", "35", "d"): "d .. SecurityDefinition",
        ("FIX.4.2", "35", "e"): "e .. SecurityStatusRequest",
        ("FIX.4.2", "35", "f"): "f .. SecurityStatus",
        ("FIX.4.2", "35", "g"): "g .. TradingSessionStatusRequest",
        ("FIX.4.2", "35", "h"): "h .. TradingSessionStatus",
        ("FIX.4.2", "35", "i"): "i .. MassQuote",
        ("FIX.4.2", "35", "j"): "j .. BusinessMessageReject",
        ("FIX.4.2", "35", "k"): "k .. BidRequest",
        ("FIX.4.2", "35", "l"): "l .. BidResponse",
        ("FIX.4.2", "35", "m"): "m .. ListStrikePrice",
        ("FIX.4.2", "39", "0"): "0 .. New",
        ("FIX.4.2", "39", "1"): "1 .. PartiallyFilled",
        ("FIX.4.2", "39", "2"): "2 .. Filled",
        ("FIX.4.2", "39", "3"): "3 .. DoneForDay",
        ("FIX.4.2", "39", "4"): "4 .. Canceled",
        ("FIX.4.2", "39", "5"): "5 .. Replaced",
        ("FIX.4.2", "39", "6"): "6 .. PendingCancel",
        ("FIX.4.2", "39", "7"): "7 .. Stopped",
        ("FIX.4.2", "39", "8"): "8 .. Rejected",
        ("FIX.4.2", "39", "9"): "9 .. Suspended",
        ("FIX.4.2", "39", "A"): "A .. PendingNew",
        ("FIX.4.2", "39", "B"): "B .. Calculated",
        ("FIX.4.2", "39", "C"): "C .. Expired",
        ("FIX.4.2", "39", "D"): "D .. AcceptedForBidding",
        ("FIX.4.2", "39", "E"): "E .. PendingReplace",
        ("FIX.4.2", "40", "1"): "1 .. Market",
        ("FIX.4.2", "40", "2"): "2 .. Limit",
        ("FIX.4.2", "40", "3"): "3 .. Stop",
        ("FIX.4.2", "40", "4"): "4 .. StopLimit",
        ("FIX.4.2", "40", "5"): "5 .. MarketOnClose",
        ("FIX.4.2", "40", "6"): "6 .. WithOrWithout",
        ("FIX.4.2", "40", "7"): "7 .. LimitOrBetter",
        ("FIX.4.2", "40", "8"): "8 .. LimitWithOrWithout",
        ("FIX.4.2", "40", "9"): "9 .. OnBasis",
        ("FIX.4.2", "40", "A"): "A .. OnClose",
        ("FIX.4.2", "40", "B"): "B .. LimitOnClose",
        ("FIX.4.2", "40", "C"): "C .. ForexMarket",
        ("FIX.4.2", "40", "D"): "D .. PreviouslyQuoted",
        ("FIX.4.2", "40", "E"): "E .. PreviouslyIndicated",
        ("FIX.4.2", "40", "F"): "F .. ForexLimit",
        ("FIX.4.2", "40", "G"): "G .. ForexSwap",
        ("FIX.4.2", "40", "H"): "H .. ForexPreviouslyQuoted",
        ("FIX.4.2", "40", "I"): "I .. Funari",
        ("FIX.4.2", "40", "P"): "P .. Pegged",
        ("FIX.4.2", "43", "N"): "N .. OriginalTransmission",
        ("FIX.4.2", "43", "Y"): "Y .. PossibleDuplicate",
        ("FIX.4.2", "47", "A"): "A .. AgencySingleOrder",
        ("FIX.4.2", "47", "B"): "B .. ShortExemptTransactionAType",
        ("FIX.4.2", "47", "C"): "C .. ProprietaryNonAlgo",
        ("FIX.4.2", "47", "D"): "D .. ProgramOrderMember",
        ("FIX.4.2", "47", "E"): "E .. ShortExemptTransactionForPrincipal",
        ("FIX.4.2", "47", "F"): "F .. ShortExemptTransactionWType",
        ("FIX.4.2", "47", "H"): "H .. ShortExemptTransactionIType",
        ("FIX.4.2", "47", "I"): "I .. IndividualInvestor",
        ("FIX.4.2", "47", "J"): "J .. ProprietaryAlgo",
        ("FIX.4.2", "47", "K"): "K .. AgencyAlgo",
        ("FIX.4.2", "47", "L"): "L .. ShortExemptTransactionMemberAffliated",
        ("FIX.4.2", "47", "M"): "M .. ProgramOrderOtherMember",
        ("FIX.4.2", "47", "N"): "N .. AgentForOtherMember",
        ("FIX.4.2", "47", "O"): "O .. ProprietaryTransactionAffiliated",
        ("FIX.4.2", "47", "P"): "P .. Principal",
        ("FIX.4.2", "47", "R"): "R .. TransactionNonMember",
        ("FIX.4.2", "47", "S"): "S .. SpecialistTrades",
        ("FIX.4.2", "47", "T"): "T .. TransactionUnaffiliatedMember",
        ("FIX.4.2", "47", "U"): "U .. AgencyIndexArb",
        ("FIX.4.2", "47", "W"): "W .. AllOtherOrdersAsAgentForOtherMember",
        ("FIX.4.2", "47", "X"): "X .. ShortExemptTransactionMemberNotAffliated",
        ("FIX.4.2", "47", "Y"): "Y .. AgencyNonAlgo",
        ("FIX.4.2", "47", "Z"): "Z .. ShortExemptTransactionNonMember",
        ("FIX.4.2", "54", "1"): "1 .. Buy",
        ("FIX.4.2", "54", "2"): "2 .. Sell",
        ("FIX.4.2", "54", "3"): "3 .. BuyMinus",
        ("FIX.4.2", "54", "4"): "4 .. SellPlus",
        ("FIX.4.2", "54", "5"): "5 .. SellShort",
        ("FIX.4.2", "54", "6"): "6 .. SellShortExempt",
        ("FIX.4.2", "54", "7"): "7 .. Undisclosed",
        ("FIX.4.2", "54", "8"): "8 .. Cross",
        ("FIX.4.2", "54", "9"): "9 .. CrossShort",
        ("FIX.4.2", "59", "0"): "0 .. Day",
        ("FIX.4.2", "59", "1"): "1 .. GoodTillCancel",
        ("FIX.4.2", "59", "2"): "2 .. AtTheOpening",
        ("FIX.4.2", "59", "3"): "3 .. ImmediateOrCancel",
        ("FIX.4.2", "59", "4"): "4 .. FillOrKill",
        ("FIX.4.2", "59", "5"): "5 .. GoodTillCrossing",
        ("FIX.4.2", "59", "6"): "6 .. GoodTillDate",
        ("FIX.4.2", "61", "0"): "0 .. Normal",
        ("FIX.4.2", "61", "1"): "1 .. Flash",
        ("FIX.4.2", "61", "2"): "2 .. Background",
        ("FIX.4.2", "63", "0"): "0 .. Regular",
        ("FIX.4.2", "63", "1"): "1 .. Cash",
        ("FIX.4.2", "63", "2"): "2 .. NextDay",
        ("FIX.4.2", "63", "3"): "3 .. TPlus2",
        ("FIX.4.2", "63", "4"): "4 .. TPlus3",
        ("FIX.4.2", "63", "5"): "5 .. TPlus4",
        ("FIX.4.2", "63", "6"): "6 .. Future",
        ("FIX.4.2", "63", "7"): "7 .. WhenAndIfIssued",
        ("FIX.4.2", "63", "8"): "8 .. SellersOption",
        ("FIX.4.2", "63", "9"): "9 .. TPlus5",
        ("FIX.4.2", "71", "0"): "0 .. New",
        ("FIX.4.2", "71", "1"): "1 .. Replace",
        ("FIX.4.2", "71", "2"): "2 .. Cancel",
        ("FIX.4.2", "71", "3"): "3 .. Preliminary",
        ("FIX.4.2", "71", "4"): "4 .. Calculated",
        ("FIX.4.2", "71", "5"): "5 .. CalculatedWithoutPreliminary",
        ("FIX.4.2", "77", "C"): "C .. Close",
        ("FIX.4.2", "77", "O"): "O .. Open",
        ("FIX.4.2", "81", "0"): "0 .. Regular",
        ("FIX.4.2", "81", "1"): "1 .. SoftDollar",
        ("FIX.4.2", "81", "2"): "2 .. StepIn",
        ("FIX.4.2", "81", "3"): "3 .. StepOut",
        ("FIX.4.2", "81", "4"): "4 .. SoftDollarStepIn",
        ("FIX.4.2", "81", "5"): "5 .. SoftDollarStepOut",
        ("FIX.4.2", "81", "6"): "6 .. PlanSponsor",
        ("FIX.4.2", "87", "0"): "0 .. Accepted",
        ("FIX.4.2", "87", "1"): "1 .. BlockLevelReject",
        ("FIX.4.2", "87", "2"): "2 .. AccountLevelReject",
        ("FIX.4.2", "87", "3"): "3 .. Received",
        ("FIX.4.2", "88", "0"): "0 .. UnknownAccount",
        ("FIX.4.2", "88", "1"): "1 .. IncorrectQuantity",
        ("FIX.4.2", "88", "2"): "2 .. IncorrectAveragegPrice",
        ("FIX.4.2", "88", "3"): "3 .. UnknownExecutingBrokerMnemonic",
        ("FIX.4.2", "88", "4"): "4 .. CommissionDifference",
        ("FIX.4.2", "88", "5"): "5 .. UnknownOrderID",
        ("FIX.4.2", "88", "6"): "6 .. UnknownListID",
        ("FIX.4.2", "88", "7"): "7 .. OtherSeeText",
        ("FIX.4.2", "94", "0"): "0 .. New",
        ("FIX.4.2", "94", "1"): "1 .. Reply",
        ("FIX.4.2", "94", "2"): "2 .. AdminReply",
        ("FIX.4.2", "97", "N"): "N .. OriginalTransmission",
        ("FIX.4.2", "97", "Y"): "Y .. PossibleResend",
        ("FIX.4.2", "98", "0"): "0 .. None",
        ("FIX.4.2", "98", "1"): "1 .. PKCS",
        ("FIX.4.2", "98", "2"): "2 .. DES",
        ("FIX.4.2", "98", "3"): "3 .. PKCSDES",
        ("FIX.4.2", "98", "4"): "4 .. PGPDES",
        ("FIX.4.2", "98", "5"): "5 .. PGPDESMD5",
        ("FIX.4.2", "98", "6"): "6 .. PEM",
        ("FIX.4.2", "102", "0"): "0 .. TooLateToCancel",
        ("FIX.4.2", "102", "1"): "1 .. UnknownOrder",
        ("FIX.4.2", "102", "2"): "2 .. BrokerCredit",
        ("FIX.4.2", "102", "3"): "3 .. OrderAlreadyInPendingStatus",
        ("FIX.4.2", "103", "0"): "0 .. BrokerCredit",
        ("FIX.4.2", "103", "1"): "1 .. UnknownSymbol",
        ("FIX.4.2", "103", "2"): "2 .. ExchangeClosed",
        ("FIX.4.2", "103", "3"): "3 .. OrderExceedsLimit",
        ("FIX.4.2", "103", "4"): "4 .. TooLateToEnter",
        ("FIX.4.2", "103", "5"): "5 .. UnknownOrder",
        ("FIX.4.2", "103", "6"): "6 .. DuplicateOrder",
        ("FIX.4.2", "103", "7"): "7 .. DuplicateOfAVerballyCommunicatedOrder",
        ("FIX.4.2", "103", "8"): "8 .. StaleOrder",
        ("FIX.4.2", "104", "A"): "A .. AllOrNone",
        ("FIX.4.2", "104", "C"): "C .. AtTheClose",
        ("FIX.4.2", "104", "I"): "I .. InTouchWith",
        ("FIX.4.2", "104", "L"): "L .. Limit",
        ("FIX.4.2", "104", "M"): "M .. MoreBehind",
        ("FIX.4.2", "104", "O"): "O .. AtTheOpen",
        ("FIX.4.2", "104", "P"): "P .. TakingAPosition",
        ("FIX.4.2", "104", "Q"): "Q .. AtTheMarket",
        ("FIX.4.2", "104", "R"): "R .. ReadyToTrade",
        ("FIX.4.2", "104", "S"): "S .. PortfolioShown",
        ("FIX.4.2", "104", "T"): "T .. ThroughTheDay",
        ("FIX.4.2", "104", "V"): "V .. Versus",
        ("FIX.4.2", "104", "W"): "W .. Indication",
        ("FIX.4.2", "104", "X"): "X .. CrossingOpportunity",
        ("FIX.4.2", "104", "Y"): "Y .. AtTheMidpoint",
        ("FIX.4.2", "104", "Z"): "Z .. PreOpen",
        ("FIX.4.2", "113", "N"): "N .. SenderReports",
        ("FIX.4.2", "113", "Y"): "Y .. ReceiverReports",
        ("FIX.4.2", "114", "N"): "N .. No",
        ("FIX.4.2", "114", "Y"): "Y .. Yes",
        ("FIX.4.2", "121", "N"): "N .. DoNotExecuteForexAfterSecurityTrade",
        ("FIX.4.2", "121", "Y"): "Y .. ExecuteForexAfterSecurityTrade",
        ("FIX.4.2", "123", "N"): "N .. SequenceReset",
        ("FIX.4.2", "123", "Y"): "Y .. GapFillMessage",
        ("FIX.4.2", "127", "A"): "A .. UnknownSymbol",
        ("FIX.4.2", "127", "B"): "B .. WrongSide",
        ("FIX.4.2", "127", "C"): "C .. QuantityExceedsOrder",
        ("FIX.4.2", "127", "D"): "D .. NoMatchingOrder",
        ("FIX.4.2", "127", "E"): "E .. PriceExceedsLimit",
        ("FIX.4.2", "127", "Z"): "Z .. Other",
        ("FIX.4.2", "130", "N"): "N .. NotNatural",
        ("FIX.4.2", "130", "Y"): "Y .. Natural",
        ("FIX.4.2", "139", "1"): "1 .. Regulatory",
        ("FIX.4.2", "139", "2"): "2 .. Tax",
        ("FIX.4.2", "139", "3"): "3 .. LocalCommission",
        ("FIX.4.2", "139", "4"): "4 .. ExchangeFees",
        ("FIX.4.2", "139", "5"): "5 .. Stamp",
        ("FIX.4.2", "139", "6"): "6 .. Levy",
        ("FIX.4.2", "139", "7"): "7 .. Other",
        ("FIX.4.2", "139", "8"): "8 .. Markup",
        ("FIX.4.2", "139", "9"): "9 .. ConsumptionTax",
        ("FIX.4.2", "141", "N"): "N .. No",
        ("FIX.4.2", "141", "Y"): "Y .. Yes",
        ("FIX.4.2", "150", "0"): "0 .. New",
        ("FIX.4.2", "150", "1"): "1 .. PartialFill",
        ("FIX.4.2", "150", "2"): "2 .. Fill",
        ("FIX.4.2", "150", "3"): "3 .. DoneForDay",
        ("FIX.4.2", "150", "4"): "4 .. Canceled",
        ("FIX.4.2", "150", "5"): "5 .. Replaced",
        ("FIX.4.2", "150", "6"): "6 .. PendingCancel",
        ("FIX.4.2", "150", "7"): "7 .. Stopped",
        ("FIX.4.2", "150", "8"): "8 .. Rejected",
        ("FIX.4.2", "150", "9"): "9 .. Suspended",
        ("FIX.4.2", "150", "A"): "A .. PendingNew",
        ("FIX.4.2", "150", "B"): "B .. Calculated",
        ("FIX.4.2", "150", "C"): "C .. Expired",
        ("FIX.4.2", "150", "D"): "D .. Restated",
        ("FIX.4.2", "150", "E"): "E .. PendingReplace",
        ("FIX.4.2", "160", "0"): "0 .. Default",
        ("FIX.4.2", "160", "1"): "1 .. StandingInstructionsProvided",
        ("FIX.4.2", "160", "2"): "2 .. SpecificAllocationAccountOverriding",
        ("FIX.4.2", "160", "3"): "3 .. SpecificAllocationAccountStanding",
        ("FIX.4.2", "163", "C"): "C .. Cancel",
        ("FIX.4.2", "163", "N"): "N .. New",
        ("FIX.4.2", "163", "R"): "R .. Replace",
        ("FIX.4.2", "165", "1"): "1 .. BrokerCredit",
        ("FIX.4.2", "165", "2"): "2 .. Institution",
        ("FIX.4.2", "166", "CED"): "CED .. CEDEL",
        ("FIX.4.2", "166", "DTC"): "DTC .. DepositoryTrustCompany",
        ("FIX.4.2", "166", "EUR"): "EUR .. EuroClear",
        ("FIX.4.2", "166", "FED"): "FED .. FederalBookEntry",
        ("FIX.4.2", "166", "ISO Country Code"): "ISO Country Code .. LocalMarketSettleLocation",
        ("FIX.4.2", "166", "PNY"): "PNY .. Physical",
        ("FIX.4.2", "166", "PTC"): "PTC .. ParticipantTrustCompany",
        ("FIX.4.2", "167", "?"): "? .. Wildcard",
        ("FIX.4.2", "167", "BA"): "BA .. BankersAcceptance",
        ("FIX.4.2", "167", "CB"): "CB .. ConvertibleBond",
        ("FIX.4.2", "167", "CD"): "CD .. CertificateOfDeposit",
        ("FIX.4.2", "167", "CMO"): "CMO .. CollateralizedMortgageObligation",
        ("FIX.4.2", "167", "CORP"): "CORP .. CorporateBond",
        ("FIX.4.2", "167", "CP"): "CP .. CommercialPaper",
        ("FIX.4.2", "167", "CPP"): "CPP .. CorporatePrivatePlacement",
        ("FIX.4.2", "167", "CS"): "CS .. CommonStock",
        ("FIX.4.2", "167", "FHA"): "FHA .. FederalHousingAuthority",
        ("FIX.4.2", "167", "FHL"): "FHL .. FederalHomeLoan",
        ("FIX.4.2", "167", "FN"): "FN .. FederalNationalMortgageAssociation",
        ("FIX.4.2", "167", "FOR"): "FOR .. ForeignExchangeContract",
        ("FIX.4.2", "167", "FUT"): "FUT .. Future",
        ("FIX.4.2", "167", "GN"): "GN .. GovernmentNationalMortgageAssociation",
        ("FIX.4.2", "167", "GOVT"): "GOVT .. TreasuriesAgencyDebenture",
        ("FIX.4.2", "167", "IET"): "IET .. IOETTEMortgage",
        ("FIX.4.2", "167", "MF"): "MF .. MutualFund",
        ("FIX.4.2", "167", "MIO"): "MIO .. MortgageInterestOnly",
        ("FIX.4.2", "167", "MPO"): "MPO .. MortgagePrincipalOnly",
        ("FIX.4.2", "167", "MPP"): "MPP .. MortgagePrivatePlacement",
        ("FIX.4.2", "167", "MPT"): "MPT .. MiscellaneousPassThrough",
        ("FIX.4.2", "167", "MUNI"): "MUNI .. MunicipalBond",
        ("FIX.4.2", "167", "NONE"): "NONE .. NoSecurityType",
        ("FIX.4.2", "167", "OPT"): "OPT .. Option",
        ("FIX.4.2", "167", "PS"): "PS .. PreferredStock",
        ("FIX.4.2", "167", "RP"): "RP .. RepurchaseAgreement",
        ("FIX.4.2", "167", "RVRP"): "RVRP .. ReverseRepurchaseAgreement",
        ("FIX.4.2", "167", "SL"): "SL .. StudentLoanMarketingAssociation",
        ("FIX.4.2", "167", "TD"): "TD .. TimeDeposit",
        ("FIX.4.2", "167", "USTB"): "USTB .. USTreasuryBillOld",
        ("FIX.4.2", "167", "WAR"): "WAR .. Warrant",
        ("FIX.4.2", "167", "ZOO"): "ZOO .. CatsTigersAndLions",
        ("FIX.4.2", "169", "0"): "0 .. Other",
        ("FIX.4.2", "169", "1"): "1 .. DTCSID",
        ("FIX.4.2", "169", "2"): "2 .. ThomsonALERT",
        ("FIX.4.2", "169", "3"): "3 .. AGlobalCustodian",
        ("FIX.4.2", "197", "0"): "0 .. FXNetting",
        ("FIX.4.2", "197", "1"): "1 .. FXSwap",
        ("FIX.4.2", "201", "0"): "0 .. Put",
        ("FIX.4.2", "201", "1"): "1 .. Call",
        ("FIX.4.2", "203", "0"): "0 .. Covered",
        ("FIX.4.2", "203", "1"): "1 .. Uncovered",
        ("FIX.4.2", "204", "0"): "0 .. Customer",
        ("FIX.4.2", "204", "1"): "1 .. Firm",
        ("FIX.4.2", "208", "N"): "N .. DetailsShouldNotBeCommunicated",
        ("FIX.4.2", "208", "Y"): "Y .. DetailsShouldBeCommunicated",
        ("FIX.4.2", "209", "1"): "1 .. Match",
        ("FIX.4.2", "209", "2"): "2 .. Forward",
        ("FIX.4.2", "209", "3"): "3 .. ForwardAndMatch",
        ("FIX.4.2", "216", "1"): "1 .. TargetFirm",
        ("FIX.4.2", "216", "2"): "2 .. TargetList",
        ("FIX.4.2", "216", "3"): "3 .. BlockFirm",
        ("FIX.4.2", "216", "4"): "4 .. BlockList",
        ("FIX.4.2", "219", "1"): "1 .. CURVE",
        ("FIX.4.2", "219", "2"): "2 .. FiveYR",
        ("FIX.4.2", "219", "3"): "3 .. OLD5",
        ("FIX.4.2", "219", "4"): "4 .. TenYR",
        ("FIX.4.2", "219", "5"): "5 .. OLD10",
        ("FIX.4.2", "219", "6"): "6 .. ThirtyYR",
        ("FIX.4.2", "219", "7"): "7 .. OLD30",
        ("FIX.4.2", "219", "8"): "8 .. ThreeMOLIBOR",
        ("FIX.4.2", "219", "9"): "9 .. SixMOLIBOR",
        ("FIX.4.2", "263", "0"): "0 .. Snapshot",
        ("FIX.4.2", "263", "1"): "1 .. SnapshotAndUpdates",
        ("FIX.4.2", "263", "2"): "2 .. DisablePreviousSnapshot",
        ("FIX.4.2", "265", "0"): "0 .. FullRefresh",
        ("FIX.4.2", "265", "1"): "1 .. IncrementalRefresh",
        ("FIX.4.2", "266", "N"): "N .. BookEntriesShouldNotBeAggregated",
        ("FIX.4.2", "266", "Y"): "Y .. BookEntriesToBeAggregated",
        ("FIX.4.2", "269", "0"): "0 .. Bid",
        ("FIX.4.2", "269", "1"): "1 .. Offer",
        ("FIX.4.2", "269", "2"): "2 .. Trade",
        ("FIX.4.2", "269", "3"): "3 .. IndexValue",
        ("FIX.4.2", "269", "4"): "4 .. OpeningPrice",
        ("FIX.4.2", "269", "5"): "5 .. ClosingPrice",
        ("FIX.4.2", "269", "6"): "6 .. SettlementPrice",
        ("FIX.4.2", "269", "7"): "7 .. TradingSessionHighPrice",
        ("FIX.4.2", "269", "8"): "8 .. TradingSessionLowPrice",
        ("FIX.4.2", "269", "9"): "9 .. TradingSessionVWAPPrice",
        ("FIX.4.2", "274", "0"): "0 .. PlusTick",
        ("FIX.4.2", "274", "1"): "1 .. ZeroPlusTick",
        ("FIX.4.2", "274", "2"): "2 .. MinusTick",
        ("FIX.4.2", "274", "3"): "3 .. ZeroMinusTick",
        ("FIX.4.2", "276", "A"): "A .. Open",
        ("FIX.4.2", "276", "B"): "B .. Closed",
        ("FIX.4.2", "276", "C"): "C .. ExchangeBest",
        ("FIX.4.2", "276", "D"): "D .. ConsolidatedBest",
        ("FIX.4.2", "276", "E"): "E .. Locked",
        ("FIX.4.2", "276", "F"): "F .. Crossed",
        ("FIX.4.2", "276", "G"): "G .. Depth",
        ("FIX.4.2", "276", "H"): "H .. FastTrading",
        ("FIX.4.2", "276", "I"): "I .. NonFirm",
        ("FIX.4.2", "277", "A"): "A .. Cash",
        ("FIX.4.2", "277", "B"): "B .. AveragePriceTrade",
        ("FIX.4.2", "277", "C"): "C .. CashTrade",
        ("FIX.4.2", "277", "D"): "D .. NextDay",
        ("FIX.4.2", "277", "E"): "E .. Opening",
        ("FIX.4.2", "277", "F"): "F .. IntradayTradeDetail",
        ("FIX.4.2", "277", "G"): "G .. Rule127Trade",
        ("FIX.4.2", "277", "H"): "H .. Rule155Trade",
        ("FIX.4.2", "277", "I"): "I .. SoldLast",
        ("FIX.4.2", "277", "J"): "J .. NextDayTrade",
        ("FIX.4.2", "277", "K"): "K .. Opened",
        ("FIX.4.2", "277", "L"): "L .. Seller",
        ("FIX.4.2", "277", "M"): "M .. Sold",
        ("FIX.4.2", "277", "N"): "N .. StoppedStock",
        ("FIX.4.2", "279", "0"): "0 .. New",
        ("FIX.4.2", "279", "1"): "1 .. Change",
        ("FIX.4.2", "279", "2"): "2 .. Delete",
        ("FIX.4.2", "281", "0"): "0 .. UnknownSymbol",
        ("FIX.4.2", "281", "1"): "1 .. DuplicateMDReqID",
        ("FIX.4.2", "281", "2"): "2 .. InsufficientBandwidth",
        ("FIX.4.2", "281", "3"): "3 .. InsufficientPermissions",
        ("FIX.4.2", "281", "4"): "4 .. UnsupportedSubscriptionRequestType",
        ("FIX.4.2", "281", "5"): "5 .. UnsupportedMarketDepth",
        ("FIX.4.2", "281", "6"): "6 .. UnsupportedMDUpdateType",
        ("FIX.4.2", "281", "7"): "7 .. UnsupportedAggregatedBook",
        ("FIX.4.2", "281", "8"): "8 .. UnsupportedMDEntryType",
        ("FIX.4.2", "285", "0"): "0 .. Cancellation",
        ("FIX.4.2", "285", "1"): "1 .. Error",
        ("FIX.4.2", "286", "0"): "0 .. DailyOpen",
        ("FIX.4.2", "286", "1"): "1 .. SessionOpen",
        ("FIX.4.2", "286", "2"): "2 .. DeliverySettlementEntry",
        ("FIX.4.2", "291", "1"): "1 .. Bankrupt",
        ("FIX.4.2", "292", "A"): "A .. ExDividend",
        ("FIX.4.2", "292", "B"): "B .. ExDistribution",
        ("FIX.4.2", "292", "C"): "C .. ExRights",
        ("FIX.4.2", "292", "D"): "D .. New",
        ("FIX.4.2", "292", "E"): "E .. ExInterest",
        ("FIX.4.2", "297", "0"): "0 .. Accepted",
        ("FIX.4.2", "297", "1"): "1 .. CancelForSymbol",
        ("FIX.4.2", "297", "2"): "2 .. CanceledForSecurityType",
        ("FIX.4.2", "297", "3"): "3 .. CanceledForUnderlying",
        ("FIX.4.2", "297", "4"): "4 .. CanceledAll",
        ("FIX.4.2", "297", "5"): "5 .. Rejected",
        ("FIX.4.2", "298", "1"): "1 .. CancelForOneOrMoreSecurities",
        ("FIX.4.2", "298", "2"): "2 .. CancelForSecurityType",
        ("FIX.4.2", "298", "3"): "3 .. CancelForUnderlyingSecurity",
        ("FIX.4.2", "298", "4"): "4 .. CancelAllQuotes",
        ("FIX.4.2", "300", "1"): "1 .. UnknownSymbol",
        ("FIX.4.2", "300", "2"): "2 .. Exchange",
        ("FIX.4.2", "300", "3"): "3 .. QuoteRequestExceedsLimit",
        ("FIX.4.2", "300", "4"): "4 .. TooLateToEnter",
        ("FIX.4.2", "300", "5"): "5 .. UnknownQuote",
        ("FIX.4.2", "300", "6"): "6 .. DuplicateQuote",
        ("FIX.4.2", "300", "7"): "7 .. InvalidBid",
        ("FIX.4.2", "300", "8"): "8 .. InvalidPrice",
        ("FIX.4.2", "300", "9"): "9 .. NotAuthorizedToQuoteSecurity",
        ("FIX.4.2", "301", "0"): "0 .. NoAcknowledgement",
        ("FIX.4.2", "301", "1"): "1 .. AcknowledgeOnlyNegativeOrErroneousQuotes",
        ("FIX.4.2", "301", "2"): "2 .. AcknowledgeEachQuoteMessage",
        ("FIX.4.2", "303", "1"): "1 .. Manual",
        ("FIX.4.2", "303", "2"): "2 .. Automatic",
        ("FIX.4.2", "321", "0"): "0 .. RequestSecurityIdentityAndSpecifications",
        ("FIX.4.2", "321", "1"): "1 .. RequestSecurityIdentityForSpecifications",
        ("FIX.4.2", "321", "2"): "2 .. RequestListSecurityTypes",
        ("FIX.4.2", "321", "3"): "3 .. RequestListSecurities",
        ("FIX.4.2", "323", "1"): "1 .. AcceptAsIs",
        ("FIX.4.2", "323", "2"): "2 .. AcceptWithRevisions",
        ("FIX.4.2", "323", "3"): "3 .. ListOfSecurityTypesReturnedPerRequest",
        ("FIX.4.2", "323", "4"): "4 .. ListOfSecuritiesReturnedPerRequest",
        ("FIX.4.2", "323", "5"): "5 .. RejectSecurityProposal",
        ("FIX.4.2", "323", "6"): "6 .. CannotMatchSelectionCriteria",
        ("FIX.4.2", "325", "N"): "N .. MessageIsBeingSentAsAResultOfAPriorRequest",
        ("FIX.4.2", "325", "Y"): "Y .. MessageIsBeingSentUnsolicited",
        ("FIX.4.2", "326", "1"): "1 .. OpeningDelay",
        ("FIX.4.2", "326", "10"): "10 .. MarketOnCloseImbalanceSell",
        ("FIX.4.2", "326", "12"): "12 .. NoMarketImbalance",
        ("FIX.4.2", "326", "13"): "13 .. NoMarketOnCloseImbalance",
        ("FIX.4.2", "326", "14"): "14 .. ITSPreOpening",
        ("FIX.4.2", "326", "15"): "15 .. NewPriceIndication",
        ("FIX.4.2", "326", "16"): "16 .. TradeDisseminationTime",
        ("FIX.4.2", "326", "17"): "17 .. ReadyToTrade",
        ("FIX.4.2", "326", "18"): "18 .. NotAvailableForTrading",
        ("FIX.4.2", "326", "19"): "19 .. NotTradedOnThisMarket",
        ("FIX.4.2", "326", "2"): "2 .. TradingHalt",
        ("FIX.4.2", "326", "20"): "20 .. UnknownOrInvalid",
        ("FIX.4.2", "326", "3"): "3 .. Resume",
        ("FIX.4.2", "326", "4"): "4 .. NoOpen",
        ("FIX.4.2", "326", "5"): "5 .. PriceIndication",
        ("FIX.4.2", "326", "6"): "6 .. TradingRangeIndication",
        ("FIX.4.2", "326", "7"): "7 .. MarketImbalanceBuy",
        ("FIX.4.2", "326", "8"): "8 .. MarketImbalanceSell",
        ("FIX.4.2", "326", "9"): "9 .. MarketOnCloseImbalanceBuy",
        ("FIX.4.2", "327", "D"): "D .. NewsDissemination",
        ("FIX.4.2", "327", "E"): "E .. OrderInflux",
        ("FIX.4.2", "327", "I"): "I .. OrderImbalance",
        ("FIX.4.2", "327", "M"): "M .. AdditionalInformation",
        ("FIX.4.2", "327", "P"): "P .. NewsPending",
        ("FIX.4.2", "327", "X"): "X .. EquipmentChangeover",
        ("FIX.4.2", "328", "N"): "N .. HaltWasNotRelatedToAHaltOfTheCommonStock",
        ("FIX.4.2", "328", "Y"): "Y .. HaltWasDueToCommonStockBeingHalted",
        ("FIX.4.2", "329", "N"): "N .. NotRelatedToSecurityHalt",
        ("FIX.4.2", "329", "Y"): "Y .. RelatedToSecurityHalt",
        ("FIX.4.2", "334", "1"): "1 .. Cancel",
        ("FIX.4.2", "334", "2"): "2 .. Error",
        ("FIX.4.2", "334", "3"): "3 .. Correction",
        ("FIX.4.2", "338", "1"): "1 .. Electronic",
        ("FIX.4.2", "338", "2"): "2 .. OpenOutcry",
        ("FIX.4.2", "338", "3"): "3 .. TwoParty",
        ("FIX.4.2", "339", "1"): "1 .. Testing",
        ("FIX.4.2", "339", "2"): "2 .. Simulated",
        ("FIX.4.2", "339", "3"): "3 .. Production",
        ("FIX.4.2", "340", "1"): "1 .. Halted",
        ("FIX.4.2", "340", "2"): "2 .. Open",
        ("FIX.4.2", "340", "3"): "3 .. Closed",
        ("FIX.4.2", "340", "4"): "4 .. PreOpen",
        ("FIX.4.2", "340", "5"): "5 .. PreClose",
        ("FIX.4.2", "347", "EUC-JP"): "EUC-JP .. EUCJP",
        ("FIX.4.2", "347", "ISO-2022-JP"): "ISO-2022-JP .. ISO2022JP",
        ("FIX.4.2", "347", "Shift_JIS"): "Shift_JIS .. ShiftJIS",
        ("FIX.4.2", "347", "UTF-8"): "UTF-8 .. UTF8",
        ("FIX.4.2", "368", "1"): "1 .. UnknownSymbol",
        ("FIX.4.2", "368", "2"): "2 .. Exchange",
        ("FIX.4.2", "368", "3"): "3 .. QuoteExceedsLimit",
        ("FIX.4.2", "368", "4"): "4 .. TooLateToEnter",
        ("FIX.4.2", "368", "5"): "5 .. UnknownQuote",
        ("FIX.4.2", "368", "6"): "6 .. DuplicateQuote",
        ("FIX.4.2", "368", "7"): "7 .. InvalidBidAskSpread",
        ("FIX.4.2", "368", "8"): "8 .. InvalidPrice",
        ("FIX.4.2", "368", "9"): "9 .. NotAuthorizedToQuoteSecurity",
        ("FIX.4.2", "373", "0"): "0 .. InvalidTagNumber",
        ("FIX.4.2", "373", "1"): "1 .. RequiredTagMissing",
        ("FIX.4.2", "373", "10"): "10 .. SendingTimeAccuracyProblem",
        ("FIX.4.2", "373", "11"): "11 .. InvalidMsgType",
        ("FIX.4.2", "373", "2"): "2 .. TagNotDefinedForThisMessageType",
        ("FIX.4.2", "373", "3"): "3 .. UndefinedTag",
        ("FIX.4.2", "373", "4"): "4 .. TagSpecifiedWithoutAValue",
        ("FIX.4.2", "373", "5"): "5 .. ValueIsIncorrect",
        ("FIX.4.2", "373", "6"): "6 .. IncorrectDataFormatForValue",
        ("FIX.4.2", "373", "7"): "7 .. DecryptionProblem",
        ("FIX.4.2", "373", "8"): "8 .. SignatureProblem",
        ("FIX.4.2", "373", "9"): "9 .. CompIDProblem",
        ("FIX.4.2", "374", "C"): "C .. Cancel",
        ("FIX.4.2", "374", "N"): "N .. New",
        ("FIX.4.2", "377", "N"): "N .. WasNotSolicited",
        ("FIX.4.2", "377", "Y"): "Y .. WasSolicited",
        ("FIX.4.2", "378", "0"): "0 .. GTCorporateAction",
        ("FIX.4.2", "378", "1"): "1 .. GTRenewal",
        ("FIX.4.2", "378", "2"): "2 .. VerbalChange",
        ("FIX.4.2", "378", "3"): "3 .. RepricingOfOrder",
        ("FIX.4.2", "378", "4"): "4 .. BrokerOption",
        ("FIX.4.2", "378", "5"): "5 .. PartialDeclineOfOrderQty",
        ("FIX.4.2", "380", "0"): "0 .. Other",
        ("FIX.4.2", "380", "1"): "1 .. UnknownID",
        ("FIX.4.2", "380", "2"): "2 .. UnknownSecurity",
        ("FIX.4.2", "380", "3"): "3 .. UnsupportedMessageType",
        ("FIX.4.2", "380", "4"): "4 .. ApplicationNotAvailable",
        ("FIX.4.2", "380", "5"): "5 .. ConditionallyRequiredFieldMissing",
        ("FIX.4.2", "385", "R"): "R .. Receive",
        ("FIX.4.2", "385", "S"): "S .. Send",
        ("FIX.4.2", "388", "0"): "0 .. RelatedToDisplayedPrice",
        ("FIX.4.2", "388", "1"): "1 .. RelatedToMarketPrice",
        ("FIX.4.2", "388", "2"): "2 .. RelatedToPrimaryPrice",
        ("FIX.4.2", "388", "3"): "3 .. RelatedToLocalPrimaryPrice",
        ("FIX.4.2", "388", "4"): "4 .. RelatedToMidpointPrice",
        ("FIX.4.2", "388", "5"): "5 .. RelatedToLastTradePrice",
        ("FIX.4.2", "409", "1"): "1 .. FiveDayMovingAverage",
        ("FIX.4.2", "409", "2"): "2 .. TwentyDayMovingAverage",
        ("FIX.4.2", "409", "3"): "3 .. NormalMarketSize",
        ("FIX.4.2", "409", "4"): "4 .. Other",
        ("FIX.4.2", "411", "N"): "N .. False",
        ("FIX.4.2", "411", "Y"): "Y .. True",
        ("FIX.4.2", "414", "1"): "1 .. BuySideRequests",
        ("FIX.4.2", "414", "2"): "2 .. SellSideSends",
        ("FIX.4.2", "414", "3"): "3 .. RealTimeExecutionReports",
        ("FIX.4.2", "416", "1"): "1 .. Net",
        ("FIX.4.2", "416", "2"): "2 .. Gross",
        ("FIX.4.2", "418", "A"): "A .. Agency",
        ("FIX.4.2", "418", "G"): "G .. VWAPGuarantee",
        ("FIX.4.2", "418", "J"): "J .. GuaranteedClose",
        ("FIX.4.2", "418", "R"): "R .. RiskTrade",
        ("FIX.4.2", "419", "2"): "2 .. ClosingPriceAtMorningSession",
        ("FIX.4.2", "419", "3"): "3 .. ClosingPrice",
        ("FIX.4.2", "419", "4"): "4 .. CurrentPrice",
        ("FIX.4.2", "419", "5"): "5 .. SQ",
        ("FIX.4.2", "419", "6"): "6 .. VWAPThroughADay",
        ("FIX.4.2", "419", "7"): "7 .. VWAPThroughAMorningSession",
        ("FIX.4.2", "419", "8"): "8 .. VWAPThroughAnAfternoonSession",
        ("FIX.4.2", "419", "9"): "9 .. VWAPThroughADayExcept",
        ("FIX.4.2", "419", "A"): "A .. VWAPThroughAMorningSessionExcept",
        ("FIX.4.2", "419", "B"): "B .. VWAPThroughAnAfternoonSessionExcept",
        ("FIX.4.2", "419", "C"): "C .. Strike",
        ("FIX.4.2", "419", "D"): "D .. Open",
        ("FIX.4.2", "419", "Z"): "Z .. Others",
        ("FIX.4.2", "423", "1"): "1 .. Percentage",
        ("FIX.4.2", "423", "2"): "2 .. PerUnit",
        ("FIX.4.2", "423", "3"): "3 .. FixedAmount",
        ("FIX.4.2", "427", "0"): "0 .. BookOutAllTradesOnDayOfExecution",
        ("FIX.4.2", "427", "1"): "1 .. AccumulateUntilFilledOrExpired",
        ("FIX.4.2", "427", "2"): "2 .. AccumulateUntilVerballlyNotifiedOtherwise",
        ("FIX.4.2", "430", "1"): "1 .. Net",
        ("FIX.4.2", "430", "2"): "2 .. Gross",
        ("FIX.4.2", "433", "1"): "1 .. Immediate",
        ("FIX.4.2", "433", "2"): "2 .. WaitForInstruction",
        ("FIX.4.2", "434", "1"): "1 .. OrderCancelRequest",
        ("FIX.4.2", "434", "2"): "2 .. OrderCancel",
        ("FIX.4.2", "442", "1"): "1 .. SingleSecurity",
        ("FIX.4.2", "442", "2"): "2 .. IndividualLegOfAMultiLegSecurity",
        ("FIX.4.2", "442", "3"): "3 .. MultiLegSecurity",
        ("FIX.4.4", "4", "B"): "B .. Buy",
        ("FIX.4.4", "4", "S"): "S .. Sell",
        ("FIX.4.4", "4", "X"): "X .. Cross",
        ("FIX.4.4", "4", "T"): "T .. Trade",
        ("FIX.4.4", "5", "N"): "N .. New",
        ("FIX.4.4", "5", "C"): "C .. Cancel",
        ("FIX.4.4", "5", "R"): "R .. Replace",
        ("FIX.4.4", "13", "1"): "1 .. PerUnit",
        ("FIX.4.4", "13", "2"): "2 .. Percent",
        ("FIX.4.4", "13", "3"): "3 .. Absolute",
        ("FIX.4.4", "13", "4"): "4 .. PercentageWaivedCashDiscount",
        ("FIX.4.4", "13", "5"): "5 .. PercentageWaivedEnhancedUnits",
        ("FIX.4.4", "13", "6"): "6 .. PointsPerBondOrContract",
        ("FIX.4.4", "18", "1"): "1 .. NotHeld",
        ("FIX.4.4", "18", "2"): "2 .. Work",
        ("FIX.4.4", "18", "3"): "3 .. GoAlong",
        ("FIX.4.4", "18", "4"): "4 .. OverTheDay",
        ("FIX.4.4", "18", "5"): "5 .. Held",
        ("FIX.4.4", "18", "6"): "6 .. ParticipateDoNotInitiate",
        ("FIX.4.4", "18", "7"): "7 .. StrictScale",
        ("FIX.4.4", "18", "8"): "8 .. TryToScale",
        ("FIX.4.4", "18", "9"): "9 .. StayOnBidSide",
        ("FIX.4.4", "18", "0"): "0 .. StayOnOfferSide",
        ("FIX.4.4", "18", "A"): "A .. NoCross",
        ("FIX.4.4", "18", "B"): "B .. OKToCross",
        ("FIX.4.4", "18", "C"): "C .. CallFirst",
        ("FIX.4.4", "18", "D"): "D .. PercentOfVolume",
        ("FIX.4.4", "18", "E"): "E .. DoNotIncrease",
        ("FIX.4.4", "18", "F"): "F .. DoNotReduce",
        ("FIX.4.4", "18", "G"): "G .. AllOrNone",
        ("FIX.4.4", "18", "H"): "H .. ReinstateOnSystemFailure",
        ("FIX.4.4", "18", "I"): "I .. InstitutionsOnly",
        ("FIX.4.4", "18", "J"): "J .. ReinstateOnTradingHalt",
        ("FIX.4.4", "18", "K"): "K .. CancelOnTradingHalt",
        ("FIX.4.4", "18", "L"): "L .. LastPeg",
        ("FIX.4.4", "18", "M"): "M .. MidPricePeg",
        ("FIX.4.4", "18", "N"): "N .. NonNegotiable",
        ("FIX.4.4", "18", "O"): "O .. OpeningPeg",
        ("FIX.4.4", "18", "P"): "P .. MarketPeg",
        ("FIX.4.4", "18", "Q"): "Q .. CancelOnSystemFailure",
        ("FIX.4.4", "18", "R"): "R .. PrimaryPeg",
        ("FIX.4.4", "18", "S"): "S .. Suspend",
        ("FIX.4.4", "18", "U"): "U .. CustomerDisplayInstruction",
        ("FIX.4.4", "18", "V"): "V .. Netting",
        ("FIX.4.4", "18", "W"): "W .. PegToVWAP",
        ("FIX.4.4", "18", "X"): "X .. TradeAlong",
        ("FIX.4.4", "18", "Y"): "Y .. TryToStop",
        ("FIX.4.4", "18", "Z"): "Z .. CancelIfNotBest",
        ("FIX.4.4", "18", "a"): "a .. TrailingStopPeg",
        ("FIX.4.4", "18", "b"): "b .. StrictLimit",
        ("FIX.4.4", "18", "c"): "c .. IgnorePriceValidityChecks",
        ("FIX.4.4", "18", "d"): "d .. PegToLimitPrice",
        ("FIX.4.4", "18", "e"): "e .. WorkToTargetStrategy",
        ("FIX.4.4", "21", "1"): "1 .. AutomatedExecutionNoIntervention",
        ("FIX.4.4", "21", "2"): "2 .. AutomatedExecutionInterventionOK",
        ("FIX.4.4", "21", "3"): "3 .. ManualOrder",
        ("FIX.4.4", "22", "1"): "1 .. CUSIP",
        ("FIX.4.4", "22", "2"): "2 .. SEDOL",
        ("FIX.4.4", "22", "3"): "3 .. QUIK",
        ("FIX.4.4", "22", "4"): "4 .. ISINNumber",
        ("FIX.4.4", "22", "5"): "5 .. RICCode",
        ("FIX.4.4", "22", "6"): "6 .. ISOCurrencyCode",
        ("FIX.4.4", "22", "7"): "7 .. ISOCountryCode",
        ("FIX.4.4", "22", "8"): "8 .. ExchangeSymbol",
        ("FIX.4.4", "22", "9"): "9 .. ConsolidatedTapeAssociation",
        ("FIX.4.4", "22", "A"): "A .. BloombergSymbol",
        ("FIX.4.4", "22", "B"): "B .. Wertpapier",
        ("FIX.4.4", "22", "C"): "C .. Dutch",
        ("FIX.4.4", "22", "D"): "D .. Valoren",
        ("FIX.4.4", "22", "E"): "E .. Sicovam",
        ("FIX.4.4", "22", "F"): "F .. Belgian",
        ("FIX.4.4", "22", "G"): "G .. Common",
        ("FIX.4.4", "22", "H"): "H .. ClearingHouse",
        ("FIX.4.4", "22", "I"): "I .. ISDAFpMLSpecification",
        ("FIX.4.4", "22", "J"): "J .. OptionPriceReportingAuthority",
        ("FIX.4.4", "25", "L"): "L .. Low",
        ("FIX.4.4", "25", "M"): "M .. Medium",
        ("FIX.4.4", "25", "H"): "H .. High",
        ("FIX.4.4", "27", "S"): "S .. Small",
        ("FIX.4.4", "27", "M"): "M .. Medium",
        ("FIX.4.4", "27", "L"): "L .. Large",
        ("FIX.4.4", "28", "N"): "N .. New",
        ("FIX.4.4", "28", "C"): "C .. Cancel",
        ("FIX.4.4", "28", "R"): "R .. Replace",
        ("FIX.4.4", "29", "1"): "1 .. Agent",
        ("FIX.4.4", "29", "2"): "2 .. CrossAsAgent",
        ("FIX.4.4", "29", "3"): "3 .. CrossAsPrincipal",
        ("FIX.4.4", "29", "4"): "4 .. Principal",
        ("FIX.4.4", "35", "0"): "0 .. Heartbeat",
        ("FIX.4.4", "35", "1"): "1 .. TestRequest",
        ("FIX.4.4", "35", "2"): "2 .. ResendRequest",
        ("FIX.4.4", "35", "3"): "3 .. Reject",
        ("FIX.4.4", "35", "4"): "4 .. SequenceReset",
        ("FIX.4.4", "35", "5"): "5 .. Logout",
        ("FIX.4.4", "35", "6"): "6 .. IOI",
        ("FIX.4.4", "35", "7"): "7 .. Advertisement",
        ("FIX.4.4", "35", "8"): "8 .. ExecutionReport",
        ("FIX.4.4", "35", "9"): "9 .. OrderCancelReject",
        ("FIX.4.4", "35", "A"): "A .. Logon",
        ("FIX.4.4", "35", "B"): "B .. News",
        ("FIX.4.4", "35", "C"): "C .. Email",
        ("FIX.4.4", "35", "D"): "D .. NewOrderSingle",
        ("FIX.4.4", "35", "E"): "E .. NewOrderList",
        ("FIX.4.4", "35", "F"): "F .. OrderCancelRequest",
        ("FIX.4.4", "35", "G"): "G .. OrderCancelReplaceRequest",
        ("FIX.4.4", "35", "H"): "H .. OrderStatusRequest",
        ("FIX.4.4", "35", "J"): "J .. AllocationInstruction",
        ("FIX.4.4", "35", "K"): "K .. ListCancelRequest",
        ("FIX.4.4", "35", "L"): "L .. ListExecute",
        ("FIX.4.4", "35", "M"): "M .. ListStatusRequest",
        ("FIX.4.4", "35", "N"): "N .. ListStatus",
        ("FIX.4.4", "35", "P"): "P .. AllocationInstructionAck",
        ("FIX.4.4", "35", "Q"): "Q .. DontKnowTrade",
        ("FIX.4.4", "35", "R"): "R .. QuoteRequest",
        ("FIX.4.4", "35", "S"): "S .. Quote",
        ("FIX.4.4", "35", "T"): "T .. SettlementInstructions",
        ("FIX.4.4", "35", "V"): "V .. MarketDataRequest",
        ("FIX.4.4", "35", "W"): "W .. MarketDataSnapshotFullRefresh",
        ("FIX.4.4", "35", "X"): "X .. MarketDataIncrementalRefresh",
        ("FIX.4.4", "35", "Y"): "Y .. MarketDataRequestReject",
        ("FIX.4.4", "35", "Z"): "Z .. QuoteCancel",
        ("FIX.4.4", "35", "a"): "a .. QuoteStatusRequest",
        ("FIX.4.4", "35", "b"): "b .. MassQuoteAcknowledgement",
        ("FIX.4.4", "35", "c"): "c .. SecurityDefinitionRequest",
        ("FIX.4.4", "35", "d"): "d .. SecurityDefinition",
        ("FIX.4.4", "35", "e"): "e .. SecurityStatusRequest",
        ("FIX.4.4", "35", "f"): "f .. SecurityStatus",
        ("FIX.4.4", "35", "g"): "g .. TradingSessionStatusRequest",
        ("FIX.4.4", "35", "h"): "h .. TradingSessionStatus",
        ("FIX.4.4", "35", "i"): "i .. MassQuote",
        ("FIX.4.4", "35", "j"): "j .. BusinessMessageReject",
        ("FIX.4.4", "35", "k"): "k .. BidRequest",
        ("FIX.4.4", "35", "l"): "l .. BidResponse",
        ("FIX.4.4", "35", "m"): "m .. ListStrikePrice",
        ("FIX.4.4", "35", "n"): "n .. XMLNonFIX",
        ("FIX.4.4", "35", "o"): "o .. RegistrationInstructions",
        ("FIX.4.4", "35", "p"): "p .. RegistrationInstructionsResponse",
        ("FIX.4.4", "35", "q"): "q .. OrderMassCancelRequest",
        ("FIX.4.4", "35", "r"): "r .. OrderMassCancelReport",
        ("FIX.4.4", "35", "s"): "s .. NewOrderCross",
        ("FIX.4.4", "35", "t"): "t .. CrossOrderCancelReplaceRequest",
        ("FIX.4.4", "35", "u"): "u .. CrossOrderCancelRequest",
        ("FIX.4.4", "35", "v"): "v .. SecurityTypeRequest",
        ("FIX.4.4", "35", "w"): "w .. SecurityTypes",
        ("FIX.4.4", "35", "x"): "x .. SecurityListRequest",
        ("FIX.4.4", "35", "y"): "y .. SecurityList",
        ("FIX.4.4", "35", "z"): "z .. DerivativeSecurityListRequest",
        ("FIX.4.4", "35", "AA"): "AA .. DerivativeSecurityList",
        ("FIX.4.4", "35", "AB"): "AB .. NewOrderMultileg",
        ("FIX.4.4", "35", "AC"): "AC .. MultilegOrderCancelReplace",
        ("FIX.4.4", "35", "AD"): "AD .. TradeCaptureReportRequest",
        ("FIX.4.4", "35", "AE"): "AE .. TradeCaptureReport",
        ("FIX.4.4", "35", "AF"): "AF .. OrderMassStatusRequest",
        ("FIX.4.4", "35", "AG"): "AG .. QuoteRequestReject",
        ("FIX.4.4", "35", "AH"): "AH .. RFQRequest",
        ("FIX.4.4", "35", "AI"): "AI .. QuoteStatusReport",
        ("FIX.4.4", "35", "AJ"): "AJ .. QuoteResponse",
        ("FIX.4.4", "35", "AK"): "AK .. Confirmation",
        ("FIX.4.4", "35", "AL"): "AL .. PositionMaintenanceRequest",
        ("FIX.4.4", "35", "AM"): "AM .. PositionMaintenanceReport",
        ("FIX.4.4", "35", "AN"): "AN .. RequestForPositions",
        ("FIX.4.4", "35", "AO"): "AO .. RequestForPositionsAck",
        ("FIX.4.4", "35", "AP"): "AP .. PositionReport",
        ("FIX.4.4", "35", "AQ"): "AQ .. TradeCaptureReportRequestAck",
        ("FIX.4.4", "35", "AR"): "AR .. TradeCaptureReportAck",
        ("FIX.4.4", "35", "AS"): "AS .. AllocationReport",
        ("FIX.4.4", "35", "AT"): "AT .. AllocationReportAck",
        ("FIX.4.4", "35", "AU"): "AU .. ConfirmationAck",
        ("FIX.4.4", "35", "AV"): "AV .. SettlementInstructionRequest",
        ("FIX.4.4", "35", "AW"): "AW .. AssignmentReport",
        ("FIX.4.4", "35", "AX"): "AX .. CollateralRequest",
        ("FIX.4.4", "35", "AY"): "AY .. CollateralAssignment",
        ("FIX.4.4", "35", "AZ"): "AZ .. CollateralResponse",
        ("FIX.4.4", "35", "BA"): "BA .. CollateralReport",
        ("FIX.4.4", "35", "BB"): "BB .. CollateralInquiry",
        ("FIX.4.4", "35", "BC"): "BC .. NetworkCounterpartySystemStatusRequest",
        ("FIX.4.4", "35", "BD"): "BD .. NetworkCounterpartySystemStatusResponse",
        ("FIX.4.4", "35", "BE"): "BE .. UserRequest",
        ("FIX.4.4", "35", "BF"): "BF .. UserResponse",
        ("FIX.4.4", "35", "BG"): "BG .. CollateralInquiryAck",
        ("FIX.4.4", "35", "BH"): "BH .. ConfirmationRequest",
        ("FIX.4.4", "39", "0"): "0 .. New",
        ("FIX.4.4", "39", "1"): "1 .. PartiallyFilled",
        ("FIX.4.4", "39", "2"): "2 .. Filled",
        ("FIX.4.4", "39", "3"): "3 .. DoneForDay",
        ("FIX.4.4", "39", "4"): "4 .. Canceled",
        ("FIX.4.4", "39", "6"): "6 .. PendingCancel",
        ("FIX.4.4", "39", "7"): "7 .. Stopped",
        ("FIX.4.4", "39", "8"): "8 .. Rejected",
        ("FIX.4.4", "39", "9"): "9 .. Suspended",
        ("FIX.4.4", "39", "A"): "A .. PendingNew",
        ("FIX.4.4", "39", "B"): "B .. Calculated",
        ("FIX.4.4", "39", "C"): "C .. Expired",
        ("FIX.4.4", "39", "D"): "D .. AcceptedForBidding",
        ("FIX.4.4", "39", "E"): "E .. PendingReplace",
        ("FIX.4.4", "40", "1"): "1 .. Market",
        ("FIX.4.4", "40", "2"): "2 .. Limit",
        ("FIX.4.4", "40", "3"): "3 .. Stop",
        ("FIX.4.4", "40", "4"): "4 .. StopLimit",
        ("FIX.4.4", "40", "6"): "6 .. WithOrWithout",
        ("FIX.4.4", "40", "7"): "7 .. LimitOrBetter",
        ("FIX.4.4", "40", "8"): "8 .. LimitWithOrWithout",
        ("FIX.4.4", "40", "9"): "9 .. OnBasis",
        ("FIX.4.4", "40", "D"): "D .. PreviouslyQuoted",
        ("FIX.4.4", "40", "E"): "E .. PreviouslyIndicated",
        ("FIX.4.4", "40", "G"): "G .. ForexSwap",
        ("FIX.4.4", "40", "I"): "I .. Funari",
        ("FIX.4.4", "40", "J"): "J .. MarketIfTouched",
        ("FIX.4.4", "40", "K"): "K .. MarketWithLeftOverAsLimit",
        ("FIX.4.4", "40", "L"): "L .. PreviousFundValuationPoint",
        ("FIX.4.4", "40", "M"): "M .. NextFundValuationPoint",
        ("FIX.4.4", "40", "P"): "P .. Pegged",
        ("FIX.4.4", "43", "Y"): "Y .. PossibleDuplicate",
        ("FIX.4.4", "43", "N"): "N .. OriginalTransmission",
        ("FIX.4.4", "54", "1"): "1 .. Buy",
        ("FIX.4.4", "54", "2"): "2 .. Sell",
        ("FIX.4.4", "54", "3"): "3 .. BuyMinus",
        ("FIX.4.4", "54", "4"): "4 .. SellPlus",
        ("FIX.4.4", "54", "5"): "5 .. SellShort",
        ("FIX.4.4", "54", "6"): "6 .. SellShortExempt",
        ("FIX.4.4", "54", "7"): "7 .. Undisclosed",
        ("FIX.4.4", "54", "8"): "8 .. Cross",
        ("FIX.4.4", "54", "9"): "9 .. CrossShort",
        ("FIX.4.4", "54", "A"): "A .. CrossShortExempt",
        ("FIX.4.4", "54", "B"): "B .. AsDefined",
        ("FIX.4.4", "54", "C"): "C .. Opposite",
        ("FIX.4.4", "54", "D"): "D .. Subscribe",
        ("FIX.4.4", "54", "E"): "E .. Redeem",
        ("FIX.4.4", "54", "F"): "F .. Lend",
        ("FIX.4.4", "54", "G"): "G .. Borrow",
        ("FIX.4.4", "59", "0"): "0 .. Day",
        ("FIX.4.4", "59", "1"): "1 .. GoodTillCancel",
        ("FIX.4.4", "59", "2"): "2 .. AtTheOpening",
        ("FIX.4.4", "59", "3"): "3 .. ImmediateOrCancel",
        ("FIX.4.4", "59", "4"): "4 .. FillOrKill",
        ("FIX.4.4", "59", "5"): "5 .. GoodTillCrossing",
        ("FIX.4.4", "59", "6"): "6 .. GoodTillDate",
        ("FIX.4.4", "59", "7"): "7 .. AtTheClose",
        ("FIX.4.4", "61", "0"): "0 .. Normal",
        ("FIX.4.4", "61", "1"): "1 .. Flash",
        ("FIX.4.4", "61", "2"): "2 .. Background",
        ("FIX.4.4", "63", "0"): "0 .. Regular",
        ("FIX.4.4", "63", "1"): "1 .. Cash",
        ("FIX.4.4", "63", "2"): "2 .. NextDay",
        ("FIX.4.4", "63", "3"): "3 .. TPlus2",
        ("FIX.4.4", "63", "4"): "4 .. TPlus3",
        ("FIX.4.4", "63", "5"): "5 .. TPlus4",
        ("FIX.4.4", "63", "6"): "6 .. Future",
        ("FIX.4.4", "63", "7"): "7 .. WhenAndIfIssued",
        ("FIX.4.4", "63", "8"): "8 .. SellersOption",
        ("FIX.4.4", "63", "9"): "9 .. TPlus5",
        ("FIX.4.4", "71", "0"): "0 .. New",
        ("FIX.4.4", "71", "1"): "1 .. Replace",
        ("FIX.4.4", "71", "2"): "2 .. Cancel",
        ("FIX.4.4", "77", "O"): "O .. Open",
        ("FIX.4.4", "77", "C"): "C .. Close",
        ("FIX.4.4", "77", "R"): "R .. Rolled",
        ("FIX.4.4", "77", "F"): "F .. FIFO",
        ("FIX.4.4", "81", "0"): "0 .. Regular",
        ("FIX.4.4", "81", "1"): "1 .. SoftDollar",
        ("FIX.4.4", "81", "2"): "2 .. StepIn",
        ("FIX.4.4", "81", "3"): "3 .. StepOut",
        ("FIX.4.4", "81", "4"): "4 .. SoftDollarStepIn",
        ("FIX.4.4", "81", "5"): "5 .. SoftDollarStepOut",
        ("FIX.4.4", "81", "6"): "6 .. PlanSponsor",
        ("FIX.4.4", "87", "0"): "0 .. Accepted",
        ("FIX.4.4", "87", "1"): "1 .. BlockLevelReject",
        ("FIX.4.4", "87", "2"): "2 .. AccountLevelReject",
        ("FIX.4.4", "87", "3"): "3 .. Received",
        ("FIX.4.4", "87", "4"): "4 .. Incomplete",
        ("FIX.4.4", "87", "5"): "5 .. RejectedByIntermediary",
        ("FIX.4.4", "88", "0"): "0 .. UnknownAccount",
        ("FIX.4.4", "88", "1"): "1 .. IncorrectQuantity",
        ("FIX.4.4", "88", "2"): "2 .. IncorrectAveragegPrice",
        ("FIX.4.4", "88", "3"): "3 .. UnknownExecutingBrokerMnemonic",
        ("FIX.4.4", "88", "4"): "4 .. CommissionDifference",
        ("FIX.4.4", "88", "5"): "5 .. UnknownOrderID",
        ("FIX.4.4", "88", "6"): "6 .. UnknownListID",
        ("FIX.4.4", "88", "7"): "7 .. OtherSeeText",
        ("FIX.4.4", "88", "8"): "8 .. IncorrectAllocatedQuantity",
        ("FIX.4.4", "88", "9"): "9 .. CalculationDifference",
        ("FIX.4.4", "88", "10"): "10 .. UnknownOrStaleExecID",
        ("FIX.4.4", "88", "11"): "11 .. MismatchedData",
        ("FIX.4.4", "88", "12"): "12 .. UnknownClOrdID",
        ("FIX.4.4", "88", "13"): "13 .. WarehouseRequestRejected",
        ("FIX.4.4", "94", "0"): "0 .. New",
        ("FIX.4.4", "94", "1"): "1 .. Reply",
        ("FIX.4.4", "94", "2"): "2 .. AdminReply",
        ("FIX.4.4", "97", "Y"): "Y .. PossibleResend",
        ("FIX.4.4", "97", "N"): "N .. OriginalTransmission",
        ("FIX.4.4", "98", "0"): "0 .. None",
        ("FIX.4.4", "98", "1"): "1 .. PKCS",
        ("FIX.4.4", "98", "2"): "2 .. DES",
        ("FIX.4.4", "98", "3"): "3 .. PKCSDES",
        ("FIX.4.4", "98", "4"): "4 .. PGPDES",
        ("FIX.4.4", "98", "5"): "5 .. PGPDESMD5",
        ("FIX.4.4", "98", "6"): "6 .. PEM",
        ("FIX.4.4", "102", "0"): "0 .. TooLateToCancel",
        ("FIX.4.4", "102", "1"): "1 .. UnknownOrder",
        ("FIX.4.4", "102", "2"): "2 .. BrokerCredit",
        ("FIX.4.4", "102", "3"): "3 .. OrderAlreadyInPendingStatus",
        ("FIX.4.4", "102", "4"): "4 .. UnableToProcessOrderMassCancelRequest",
        ("FIX.4.4", "102", "5"): "5 .. OrigOrdModTime",
        ("FIX.4.4", "102", "6"): "6 .. DuplicateClOrdID",
        ("FIX.4.4", "102", "99"): "99 .. Other",
        ("FIX.4.4", "103", "0"): "0 .. BrokerCredit",
        ("FIX.4.4", "103", "1"): "1 .. UnknownSymbol",
        ("FIX.4.4", "103", "2"): "2 .. ExchangeClosed",
        ("FIX.4.4", "103", "3"): "3 .. OrderExceedsLimit",
        ("FIX.4.4", "103", "4"): "4 .. TooLateToEnter",
        ("FIX.4.4", "103", "5"): "5 .. UnknownOrder",
        ("FIX.4.4", "103", "6"): "6 .. DuplicateOrder",
        ("FIX.4.4", "103", "7"): "7 .. DuplicateOfAVerballyCommunicatedOrder",
        ("FIX.4.4", "103", "8"): "8 .. StaleOrder",
        ("FIX.4.4", "103", "9"): "9 .. TradeAlongRequired",
        ("FIX.4.4", "103", "10"): "10 .. InvalidInvestorID",
        ("FIX.4.4", "103", "11"): "11 .. UnsupportedOrderCharacteristic",
        ("FIX.4.4", "103", "13"): "13 .. IncorrectQuantity",
        ("FIX.4.4", "103", "14"): "14 .. IncorrectAllocatedQuantity",
        ("FIX.4.4", "103", "15"): "15 .. UnknownAccount",
        ("FIX.4.4", "103", "99"): "99 .. Other",
        ("FIX.4.4", "104", "A"): "A .. AllOrNone",
        ("FIX.4.4", "104", "B"): "B .. MarketOnClose",
        ("FIX.4.4", "104", "C"): "C .. AtTheClose",
        ("FIX.4.4", "104", "D"): "D .. VWAP",
        ("FIX.4.4", "104", "I"): "I .. InTouchWith",
        ("FIX.4.4", "104", "L"): "L .. Limit",
        ("FIX.4.4", "104", "M"): "M .. MoreBehind",
        ("FIX.4.4", "104", "O"): "O .. AtTheOpen",
        ("FIX.4.4", "104", "P"): "P .. TakingAPosition",
        ("FIX.4.4", "104", "Q"): "Q .. AtTheMarket",
        ("FIX.4.4", "104", "R"): "R .. ReadyToTrade",
        ("FIX.4.4", "104", "S"): "S .. PortfolioShown",
        ("FIX.4.4", "104", "T"): "T .. ThroughTheDay",
        ("FIX.4.4", "104", "V"): "V .. Versus",
        ("FIX.4.4", "104", "W"): "W .. Indication",
        ("FIX.4.4", "104", "X"): "X .. CrossingOpportunity",
        ("FIX.4.4", "104", "Y"): "Y .. AtTheMidpoint",
        ("FIX.4.4", "104", "Z"): "Z .. PreOpen",
        ("FIX.4.4", "113", "Y"): "Y .. ReceiverReports",
        ("FIX.4.4", "113", "N"): "N .. SenderReports",
        ("FIX.4.4", "114", "Y"): "Y .. Yes",
        ("FIX.4.4", "114", "N"): "N .. No",
        ("FIX.4.4", "121", "Y"): "Y .. ExecuteForexAfterSecurityTrade",
        ("FIX.4.4", "121", "N"): "N .. DoNotExecuteForexAfterSecurityTrade",
        ("FIX.4.4", "123", "Y"): "Y .. GapFillMessage",
        ("FIX.4.4", "123", "N"): "N .. SequenceReset",
        ("FIX.4.4", "127", "A"): "A .. UnknownSymbol",
        ("FIX.4.4", "127", "B"): "B .. WrongSide",
        ("FIX.4.4", "127", "C"): "C .. QuantityExceedsOrder",
        ("FIX.4.4", "127", "D"): "D .. NoMatchingOrder",
        ("FIX.4.4", "127", "E"): "E .. PriceExceedsLimit",
        ("FIX.4.4", "127", "F"): "F .. CalculationDifference",
        ("FIX.4.4", "127", "Z"): "Z .. Other",
        ("FIX.4.4", "130", "Y"): "Y .. Natural",
        ("FIX.4.4", "130", "N"): "N .. NotNatural",
        ("FIX.4.4", "139", "1"): "1 .. Regulatory",
        ("FIX.4.4", "139", "2"): "2 .. Tax",
        ("FIX.4.4", "139", "3"): "3 .. LocalCommission",
        ("FIX.4.4", "139", "4"): "4 .. ExchangeFees",
        ("FIX.4.4", "139", "5"): "5 .. Stamp",
        ("FIX.4.4", "139", "6"): "6 .. Levy",
        ("FIX.4.4", "139", "7"): "7 .. Other",
        ("FIX.4.4", "139", "8"): "8 .. Markup",
        ("FIX.4.4", "139", "9"): "9 .. ConsumptionTax",
        ("FIX.4.4", "139", "10"): "10 .. PerTransaction",
        ("FIX.4.4", "139", "11"): "11 .. Conversion",
        ("FIX.4.4", "139", "12"): "12 .. Agent",
        ("FIX.4.4", "141", "Y"): "Y .. Yes",
        ("FIX.4.4", "141", "N"): "N .. No",
        ("FIX.4.4", "150", "0"): "0 .. New",
        ("FIX.4.4", "150", "3"): "3 .. DoneForDay",
        ("FIX.4.4", "150", "4"): "4 .. Canceled",
        ("FIX.4.4", "150", "5"): "5 .. Replaced",
        ("FIX.4.4", "150", "6"): "6 .. PendingCancel",
        ("FIX.4.4", "150", "7"): "7 .. Stopped",
        ("FIX.4.4", "150", "8"): "8 .. Rejected",
        ("FIX.4.4", "150", "9"): "9 .. Suspended",
        ("FIX.4.4", "150", "A"): "A .. PendingNew",
        ("FIX.4.4", "150", "B"): "B .. Calculated",
        ("FIX.4.4", "150", "C"): "C .. Expired",
        ("FIX.4.4", "150", "D"): "D .. Restated",
        ("FIX.4.4", "150", "E"): "E .. PendingReplace",
        ("FIX.4.4", "150", "F"): "F .. Trade",
        ("FIX.4.4", "150", "G"): "G .. TradeCorrect",
        ("FIX.4.4", "150", "H"): "H .. TradeCancel",
        ("FIX.4.4", "150", "I"): "I .. OrderStatus",
        ("FIX.4.4", "156", "M"): "M .. Multiply",
        ("FIX.4.4", "156", "D"): "D .. Divide",
        ("FIX.4.4", "160", "1"): "1 .. StandingInstructionsProvided",
        ("FIX.4.4", "160", "4"): "4 .. SpecificOrderForASingleAccount",
        ("FIX.4.4", "160", "5"): "5 .. RequestReject",
        ("FIX.4.4", "163", "N"): "N .. New",
        ("FIX.4.4", "163", "C"): "C .. Cancel",
        ("FIX.4.4", "163", "R"): "R .. Replace",
        ("FIX.4.4", "163", "T"): "T .. Restate",
        ("FIX.4.4", "165", "1"): "1 .. BrokerCredit",
        ("FIX.4.4", "165", "2"): "2 .. Institution",
        ("FIX.4.4", "165", "3"): "3 .. Investor",
        ("FIX.4.4", "167", "EUSUPRA"): "EUSUPRA .. EuroSupranationalCoupons",
        ("FIX.4.4", "167", "FAC"): "FAC .. FederalAgencyCoupon",
        ("FIX.4.4", "167", "FADN"): "FADN .. FederalAgencyDiscountNote",
        ("FIX.4.4", "167", "PEF"): "PEF .. PrivateExportFunding",
        ("FIX.4.4", "167", "SUPRA"): "SUPRA .. USDSupranationalCoupons",
        ("FIX.4.4", "167", "CORP"): "CORP .. CorporateBond",
        ("FIX.4.4", "167", "CPP"): "CPP .. CorporatePrivatePlacement",
        ("FIX.4.4", "167", "CB"): "CB .. ConvertibleBond",
        ("FIX.4.4", "167", "DUAL"): "DUAL .. DualCurrency",
        ("FIX.4.4", "167", "EUCORP"): "EUCORP .. EuroCorporateBond",
        ("FIX.4.4", "167", "XLINKD"): "XLINKD .. IndexedLinked",
        ("FIX.4.4", "167", "STRUCT"): "STRUCT .. StructuredNotes",
        ("FIX.4.4", "167", "YANK"): "YANK .. YankeeCorporateBond",
        ("FIX.4.4", "167", "FOR"): "FOR .. ForeignExchangeContract",
        ("FIX.4.4", "167", "CS"): "CS .. CommonStock",
        ("FIX.4.4", "167", "PS"): "PS .. PreferredStock",
        ("FIX.4.4", "167", "BRADY"): "BRADY .. BradyBond",
        ("FIX.4.4", "167", "EUSOV"): "EUSOV .. EuroSovereigns",
        ("FIX.4.4", "167", "TBOND"): "TBOND .. USTreasuryBond",
        ("FIX.4.4", "167", "TINT"): "TINT .. InterestStripFromAnyBondOrNote",
        ("FIX.4.4", "167", "TIPS"): "TIPS .. TreasuryInflationProtectedSecurities",
        ("FIX.4.4", "167", "TCAL"): "TCAL .. PrincipalStripOfACallableBondOrNote",
        ("FIX.4.4", "167", "TPRN"): "TPRN .. PrincipalStripFromANonCallableBondOrNote",
        ("FIX.4.4", "167", "UST"): "UST .. USTreasuryNoteOld",
        ("FIX.4.4", "167", "USTB"): "USTB .. USTreasuryBillOld",
        ("FIX.4.4", "167", "TNOTE"): "TNOTE .. USTreasuryNote",
        ("FIX.4.4", "167", "TBILL"): "TBILL .. USTreasuryBill",
        ("FIX.4.4", "167", "REPO"): "REPO .. Repurchase",
        ("FIX.4.4", "167", "FORWARD"): "FORWARD .. Forward",
        ("FIX.4.4", "167", "BUYSELL"): "BUYSELL .. BuySellback",
        ("FIX.4.4", "167", "SECLOAN"): "SECLOAN .. SecuritiesLoan",
        ("FIX.4.4", "167", "SECPLEDGE"): "SECPLEDGE .. SecuritiesPledge",
        ("FIX.4.4", "167", "TERM"): "TERM .. TermLoan",
        ("FIX.4.4", "167", "RVLV"): "RVLV .. RevolverLoan",
        ("FIX.4.4", "167", "RVLVTRM"): "RVLVTRM .. Revolver",
        ("FIX.4.4", "167", "BRIDGE"): "BRIDGE .. BridgeLoan",
        ("FIX.4.4", "167", "LOFC"): "LOFC .. LetterOfCredit",
        ("FIX.4.4", "167", "SWING"): "SWING .. SwingLineFacility",
        ("FIX.4.4", "167", "DINP"): "DINP .. DebtorInPossession",
        ("FIX.4.4", "167", "DEFLTED"): "DEFLTED .. Defaulted",
        ("FIX.4.4", "167", "WITHDRN"): "WITHDRN .. Withdrawn",
        ("FIX.4.4", "167", "REPLACD"): "REPLACD .. Replaced",
        ("FIX.4.4", "167", "MATURED"): "MATURED .. Matured",
        ("FIX.4.4", "167", "AMENDED"): "AMENDED .. Amended",
        ("FIX.4.4", "167", "RETIRED"): "RETIRED .. Retired",
        ("FIX.4.4", "167", "BA"): "BA .. BankersAcceptance",
        ("FIX.4.4", "167", "BN"): "BN .. BankNotes",
        ("FIX.4.4", "167", "BOX"): "BOX .. BillOfExchanges",
        ("FIX.4.4", "167", "CD"): "CD .. CertificateOfDeposit",
        ("FIX.4.4", "167", "CL"): "CL .. CallLoans",
        ("FIX.4.4", "167", "CP"): "CP .. CommercialPaper",
        ("FIX.4.4", "167", "DN"): "DN .. DepositNotes",
        ("FIX.4.4", "167", "EUCD"): "EUCD .. EuroCertificateOfDeposit",
        ("FIX.4.4", "167", "EUCP"): "EUCP .. EuroCommercialPaper",
        ("FIX.4.4", "167", "LQN"): "LQN .. LiquidityNote",
        ("FIX.4.4", "167", "MTN"): "MTN .. MediumTermNotes",
        ("FIX.4.4", "167", "ONITE"): "ONITE .. Overnight",
        ("FIX.4.4", "167", "PN"): "PN .. PromissoryNote",
        ("FIX.4.4", "167", "PZFJ"): "PZFJ .. PlazosFijos",
        ("FIX.4.4", "167", "STN"): "STN .. ShortTermLoanNote",
        ("FIX.4.4", "167", "TD"): "TD .. TimeDeposit",
        ("FIX.4.4", "167", "XCN"): "XCN .. ExtendedCommNote",
        ("FIX.4.4", "167", "YCD"): "YCD .. YankeeCertificateOfDeposit",
        ("FIX.4.4", "167", "ABS"): "ABS .. AssetBackedSecurities",
        ("FIX.4.4", "167", "CMBS"): "CMBS .. Corp",
        ("FIX.4.4", "167", "CMO"): "CMO .. CollateralizedMortgageObligation",
        ("FIX.4.4", "167", "IET"): "IET .. IOETTEMortgage",
        ("FIX.4.4", "167", "MBS"): "MBS .. MortgageBackedSecurities",
        ("FIX.4.4", "167", "MIO"): "MIO .. MortgageInterestOnly",
        ("FIX.4.4", "167", "MPO"): "MPO .. MortgagePrincipalOnly",
        ("FIX.4.4", "167", "MPP"): "MPP .. MortgagePrivatePlacement",
        ("FIX.4.4", "167", "MPT"): "MPT .. MiscellaneousPassThrough",
        ("FIX.4.4", "167", "PFAND"): "PFAND .. Pfandbriefe",
        ("FIX.4.4", "167", "TBA"): "TBA .. ToBeAnnounced",
        ("FIX.4.4", "167", "AN"): "AN .. OtherAnticipationNotes",
        ("FIX.4.4", "167", "COFO"): "COFO .. CertificateOfObligation",
        ("FIX.4.4", "167", "COFP"): "COFP .. CertificateOfParticipation",
        ("FIX.4.4", "167", "GO"): "GO .. GeneralObligationBonds",
        ("FIX.4.4", "167", "MT"): "MT .. MandatoryTender",
        ("FIX.4.4", "167", "RAN"): "RAN .. RevenueAnticipationNote",
        ("FIX.4.4", "167", "REV"): "REV .. RevenueBonds",
        ("FIX.4.4", "167", "SPCLA"): "SPCLA .. SpecialAssessment",
        ("FIX.4.4", "167", "SPCLO"): "SPCLO .. SpecialObligation",
        ("FIX.4.4", "167", "SPCLT"): "SPCLT .. SpecialTax",
        ("FIX.4.4", "167", "TAN"): "TAN .. TaxAnticipationNote",
        ("FIX.4.4", "167", "TAXA"): "TAXA .. TaxAllocation",
        ("FIX.4.4", "167", "TECP"): "TECP .. TaxExemptCommercialPaper",
        ("FIX.4.4", "167", "TRAN"): "TRAN .. TaxRevenueAnticipationNote",
        ("FIX.4.4", "167", "VRDN"): "VRDN .. VariableRateDemandNote",
        ("FIX.4.4", "167", "WAR"): "WAR .. Warrant",
        ("FIX.4.4", "167", "MF"): "MF .. MutualFund",
        ("FIX.4.4", "167", "MLEG"): "MLEG .. MultilegInstrument",
        ("FIX.4.4", "167", "NONE"): "NONE .. NoSecurityType",
        ("FIX.4.4", "167", "FUT"): "FUT .. Future",
        ("FIX.4.4", "167", "OPT"): "OPT .. Option",
        ("FIX.4.4", "169", "0"): "0 .. Other",
        ("FIX.4.4", "169", "1"): "1 .. DTCSID",
        ("FIX.4.4", "169", "2"): "2 .. ThomsonALERT",
        ("FIX.4.4", "169", "3"): "3 .. AGlobalCustodian",
        ("FIX.4.4", "169", "4"): "4 .. AccountNet",
        ("FIX.4.4", "172", "0"): "0 .. Versus",
        ("FIX.4.4", "172", "1"): "1 .. Free",
        ("FIX.4.4", "172", "2"): "2 .. TriParty",
        ("FIX.4.4", "172", "3"): "3 .. HoldInCustody",
        ("FIX.4.4", "197", "0"): "0 .. FXNetting",
        ("FIX.4.4", "197", "1"): "1 .. FXSwap",
        ("FIX.4.4", "201", "0"): "0 .. Put",
        ("FIX.4.4", "201", "1"): "1 .. Call",
        ("FIX.4.4", "203", "0"): "0 .. Covered",
        ("FIX.4.4", "203", "1"): "1 .. Uncovered",
        ("FIX.4.4", "208", "Y"): "Y .. DetailsShouldBeCommunicated",
        ("FIX.4.4", "208", "N"): "N .. DetailsShouldNotBeCommunicated",
        ("FIX.4.4", "209", "1"): "1 .. Match",
        ("FIX.4.4", "209", "2"): "2 .. Forward",
        ("FIX.4.4", "209", "3"): "3 .. ForwardAndMatch",
        ("FIX.4.4", "216", "1"): "1 .. TargetFirm",
        ("FIX.4.4", "216", "2"): "2 .. TargetList",
        ("FIX.4.4", "216", "3"): "3 .. BlockFirm",
        ("FIX.4.4", "216", "4"): "4 .. BlockList",
        ("FIX.4.4", "233", "AMT"): "AMT .. AlternativeMinimumTax",
        ("FIX.4.4", "233", "AUTOREINV"): "AUTOREINV .. AutoReinvestment",
        ("FIX.4.4", "233", "BANKQUAL"): "BANKQUAL .. BankQualified",
        ("FIX.4.4", "233", "BGNCON"): "BGNCON .. BargainConditions",
        ("FIX.4.4", "233", "COUPON"): "COUPON .. CouponRange",
        ("FIX.4.4", "233", "CURRENCY"): "CURRENCY .. ISOCurrencyCode",
        ("FIX.4.4", "233", "CUSTOMDATE"): "CUSTOMDATE .. CustomStart",
        ("FIX.4.4", "233", "GEOG"): "GEOG .. Geographics",
        ("FIX.4.4", "233", "HAIRCUT"): "HAIRCUT .. ValuationDiscount",
        ("FIX.4.4", "233", "INSURED"): "INSURED .. Insured",
        ("FIX.4.4", "233", "ISSUE"): "ISSUE .. IssueDate",
        ("FIX.4.4", "233", "ISSUER"): "ISSUER .. Issuer",
        ("FIX.4.4", "233", "ISSUESIZE"): "ISSUESIZE .. IssueSizeRange",
        ("FIX.4.4", "233", "LOOKBACK"): "LOOKBACK .. LookbackDays",
        ("FIX.4.4", "233", "LOT"): "LOT .. ExplicitLotIdentifier",
        ("FIX.4.4", "233", "LOTVAR"): "LOTVAR .. LotVariance",
        ("FIX.4.4", "233", "MAT"): "MAT .. MaturityYearAndMonth",
        ("FIX.4.4", "233", "MATURITY"): "MATURITY .. MaturityRange",
        ("FIX.4.4", "233", "MAXSUBS"): "MAXSUBS .. MaximumSubstitutions",
        ("FIX.4.4", "233", "MINQTY"): "MINQTY .. MinimumQuantity",
        ("FIX.4.4", "233", "MININCR"): "MININCR .. MinimumIncrement",
        ("FIX.4.4", "233", "MINDNOM"): "MINDNOM .. MinimumDenomination",
        ("FIX.4.4", "233", "PAYFREQ"): "PAYFREQ .. PaymentFrequency",
        ("FIX.4.4", "233", "PIECES"): "PIECES .. NumberOfPieces",
        ("FIX.4.4", "233", "PMAX"): "PMAX .. PoolsMaximum",
        ("FIX.4.4", "233", "PPM"): "PPM .. PoolsPerMillion",
        ("FIX.4.4", "233", "PPL"): "PPL .. PoolsPerLot",
        ("FIX.4.4", "233", "PPT"): "PPT .. PoolsPerTrade",
        ("FIX.4.4", "233", "PRICE"): "PRICE .. PriceRange",
        ("FIX.4.4", "233", "PRICEFREQ"): "PRICEFREQ .. PricingFrequency",
        ("FIX.4.4", "233", "PROD"): "PROD .. ProductionYear",
        ("FIX.4.4", "233", "PROTECT"): "PROTECT .. CallProtection",
        ("FIX.4.4", "233", "PURPOSE"): "PURPOSE .. Purpose",
        ("FIX.4.4", "233", "PXSOURCE"): "PXSOURCE .. BenchmarkPriceSource",
        ("FIX.4.4", "233", "RATING"): "RATING .. RatingSourceAndRange",
        ("FIX.4.4", "233", "REDEMPTION"): "REDEMPTION .. TypeOfRedemption",
        ("FIX.4.4", "233", "RESTRICTED"): "RESTRICTED .. Restricted",
        ("FIX.4.4", "233", "SECTOR"): "SECTOR .. MarketSector",
        ("FIX.4.4", "233", "SECTYPE"): "SECTYPE .. SecurityTypeIncludedOrExcluded",
        ("FIX.4.4", "233", "STRUCT"): "STRUCT .. Structure",
        ("FIX.4.4", "233", "SUBSFREQ"): "SUBSFREQ .. SubstitutionsFrequency",
        ("FIX.4.4", "233", "SUBSLEFT"): "SUBSLEFT .. SubstitutionsLeft",
        ("FIX.4.4", "233", "TEXT"): "TEXT .. FreeformText",
        ("FIX.4.4", "233", "TRDVAR"): "TRDVAR .. TradeVariance",
        ("FIX.4.4", "233", "WAC"): "WAC .. WeightedAverageCoupon",
        ("FIX.4.4", "233", "WAL"): "WAL .. WeightedAverageLifeCoupon",
        ("FIX.4.4", "233", "WALA"): "WALA .. WeightedAverageLoanAge",
        ("FIX.4.4", "233", "WAM"): "WAM .. WeightedAverageMaturity",
        ("FIX.4.4", "233", "WHOLE"): "WHOLE .. WholePool",
        ("FIX.4.4", "233", "YIELD"): "YIELD .. YieldRange",
        ("FIX.4.4", "235", "AFTERTAX"): "AFTERTAX .. AfterTaxYield",
        ("FIX.4.4", "235", "ANNUAL"): "ANNUAL .. AnnualYield",
        ("FIX.4.4", "235", "ATISSUE"): "ATISSUE .. YieldAtIssue",
        ("FIX.4.4", "235", "AVGMATURITY"): "AVGMATURITY .. YieldToAverageMaturity",
        ("FIX.4.4", "235", "BOOK"): "BOOK .. BookYield",
        ("FIX.4.4", "235", "CALL"): "CALL .. YieldToNextCall",
        ("FIX.4.4", "235", "CHANGE"): "CHANGE .. YieldChangeSinceClose",
        ("FIX.4.4", "235", "CLOSE"): "CLOSE .. ClosingYield",
        ("FIX.4.4", "235", "COMPOUND"): "COMPOUND .. CompoundYield",
        ("FIX.4.4", "235", "CURRENT"): "CURRENT .. CurrentYield",
        ("FIX.4.4", "235", "GROSS"): "GROSS .. TrueGrossYield",
        ("FIX.4.4", "235", "GOVTEQUIV"): "GOVTEQUIV .. GvntEquivalentYield",
        ("FIX.4.4", "235", "INFLATION"): "INFLATION .. YieldWithInflationAssumption",
        ("FIX.4.4", "235", "INVERSEFLOATER"): "INVERSEFLOATER .. InverseFloaterBondYield",
        ("FIX.4.4", "235", "LASTCLOSE"): "LASTCLOSE .. MostRecentClosingYield",
        ("FIX.4.4", "235", "LASTMONTH"): "LASTMONTH .. ClosingYieldMostRecentMonth",
        ("FIX.4.4", "235", "LASTQUARTER"): "LASTQUARTER .. ClosingYieldMostRecentQuarter",
        ("FIX.4.4", "235", "LASTYEAR"): "LASTYEAR .. ClosingYieldMostRecentYear",
        ("FIX.4.4", "235", "LONGAVGLIFE"): "LONGAVGLIFE .. YieldToLongestAverageLife",
        ("FIX.4.4", "235", "MARK"): "MARK .. MarkToMarketYield",
        ("FIX.4.4", "235", "MATURITY"): "MATURITY .. YieldToMaturity",
        ("FIX.4.4", "235", "NEXTREFUND"): "NEXTREFUND .. YieldToNextRefund",
        ("FIX.4.4", "235", "OPENAVG"): "OPENAVG .. OpenAverageYield",
        ("FIX.4.4", "235", "PUT"): "PUT .. YieldToNextPut",
        ("FIX.4.4", "235", "PREVCLOSE"): "PREVCLOSE .. PreviousCloseYield",
        ("FIX.4.4", "235", "PROCEEDS"): "PROCEEDS .. ProceedsYield",
        ("FIX.4.4", "235", "SEMIANNUAL"): "SEMIANNUAL .. SemiAnnualYield",
        ("FIX.4.4", "235", "SHORTAVGLIFE"): "SHORTAVGLIFE .. YieldToShortestAverageLife",
        ("FIX.4.4", "235", "SIMPLE"): "SIMPLE .. SimpleYield",
        ("FIX.4.4", "235", "TAXEQUIV"): "TAXEQUIV .. TaxEquivalentYield",
        ("FIX.4.4", "235", "TENDER"): "TENDER .. YieldToTenderDate",
        ("FIX.4.4", "235", "TRUE"): "TRUE .. TrueYield",
        ("FIX.4.4", "235", "VALUE1/32"): "VALUE1/32 .. YieldValueOf132",
        ("FIX.4.4", "235", "WORST"): "WORST .. YieldToWorst",
        ("FIX.4.4", "258", "Y"): "Y .. TradedFlat",
        ("FIX.4.4", "258", "N"): "N .. NotTradedFlat",
        ("FIX.4.4", "263", "0"): "0 .. Snapshot",
        ("FIX.4.4", "263", "1"): "1 .. SnapshotAndUpdates",
        ("FIX.4.4", "263", "2"): "2 .. DisablePreviousSnapshot",
        ("FIX.4.4", "265", "0"): "0 .. FullRefresh",
        ("FIX.4.4", "265", "1"): "1 .. IncrementalRefresh",
        ("FIX.4.4", "266", "Y"): "Y .. BookEntriesToBeAggregated",
        ("FIX.4.4", "266", "N"): "N .. BookEntriesShouldNotBeAggregated",
        ("FIX.4.4", "269", "0"): "0 .. Bid",
        ("FIX.4.4", "269", "1"): "1 .. Offer",
        ("FIX.4.4", "269", "2"): "2 .. Trade",
        ("FIX.4.4", "269", "3"): "3 .. IndexValue",
        ("FIX.4.4", "269", "4"): "4 .. OpeningPrice",
        ("FIX.4.4", "269", "5"): "5 .. ClosingPrice",
        ("FIX.4.4", "269", "6"): "6 .. SettlementPrice",
        ("FIX.4.4", "269", "7"): "7 .. TradingSessionHighPrice",
        ("FIX.4.4", "269", "8"): "8 .. TradingSessionLowPrice",
        ("FIX.4.4", "269", "9"): "9 .. TradingSessionVWAPPrice",
        ("FIX.4.4", "269", "A"): "A .. Imbalance",
        ("FIX.4.4", "269", "B"): "B .. TradeVolume",
        ("FIX.4.4", "269", "C"): "C .. OpenInterest",
        ("FIX.4.4", "274", "0"): "0 .. PlusTick",
        ("FIX.4.4", "274", "1"): "1 .. ZeroPlusTick",
        ("FIX.4.4", "274", "2"): "2 .. MinusTick",
        ("FIX.4.4", "274", "3"): "3 .. ZeroMinusTick",
        ("FIX.4.4", "276", "A"): "A .. Open",
        ("FIX.4.4", "276", "B"): "B .. Closed",
        ("FIX.4.4", "276", "C"): "C .. ExchangeBest",
        ("FIX.4.4", "276", "D"): "D .. ConsolidatedBest",
        ("FIX.4.4", "276", "E"): "E .. Locked",
        ("FIX.4.4", "276", "F"): "F .. Crossed",
        ("FIX.4.4", "276", "G"): "G .. Depth",
        ("FIX.4.4", "276", "H"): "H .. FastTrading",
        ("FIX.4.4", "276", "I"): "I .. NonFirm",
        ("FIX.4.4", "277", "A"): "A .. Cash",
        ("FIX.4.4", "277", "B"): "B .. AveragePriceTrade",
        ("FIX.4.4", "277", "C"): "C .. CashTrade",
        ("FIX.4.4", "277", "D"): "D .. NextDay",
        ("FIX.4.4", "277", "E"): "E .. Opening",
        ("FIX.4.4", "277", "F"): "F .. IntradayTradeDetail",
        ("FIX.4.4", "277", "G"): "G .. Rule127Trade",
        ("FIX.4.4", "277", "H"): "H .. Rule155Trade",
        ("FIX.4.4", "277", "I"): "I .. SoldLast",
        ("FIX.4.4", "277", "J"): "J .. NextDayTrade",
        ("FIX.4.4", "277", "K"): "K .. Opened",
        ("FIX.4.4", "277", "L"): "L .. Seller",
        ("FIX.4.4", "277", "M"): "M .. Sold",
        ("FIX.4.4", "277", "N"): "N .. StoppedStock",
        ("FIX.4.4", "277", "P"): "P .. ImbalanceMoreBuyers",
        ("FIX.4.4", "277", "Q"): "Q .. ImbalanceMoreSellers",
        ("FIX.4.4", "277", "R"): "R .. OpeningPrice",
        ("FIX.4.4", "279", "0"): "0 .. New",
        ("FIX.4.4", "279", "1"): "1 .. Change",
        ("FIX.4.4", "279", "2"): "2 .. Delete",
        ("FIX.4.4", "281", "0"): "0 .. UnknownSymbol",
        ("FIX.4.4", "281", "1"): "1 .. DuplicateMDReqID",
        ("FIX.4.4", "281", "2"): "2 .. InsufficientBandwidth",
        ("FIX.4.4", "281", "3"): "3 .. InsufficientPermissions",
        ("FIX.4.4", "281", "4"): "4 .. UnsupportedSubscriptionRequestType",
        ("FIX.4.4", "281", "5"): "5 .. UnsupportedMarketDepth",
        ("FIX.4.4", "281", "6"): "6 .. UnsupportedMDUpdateType",
        ("FIX.4.4", "281", "7"): "7 .. UnsupportedAggregatedBook",
        ("FIX.4.4", "281", "8"): "8 .. UnsupportedMDEntryType",
        ("FIX.4.4", "281", "9"): "9 .. UnsupportedTradingSessionID",
        ("FIX.4.4", "281", "A"): "A .. UnsupportedScope",
        ("FIX.4.4", "281", "B"): "B .. UnsupportedOpenCloseSettleFlag",
        ("FIX.4.4", "281", "C"): "C .. UnsupportedMDImplicitDelete",
        ("FIX.4.4", "285", "0"): "0 .. Cancellation",
        ("FIX.4.4", "285", "1"): "1 .. Error",
        ("FIX.4.4", "286", "0"): "0 .. DailyOpen",
        ("FIX.4.4", "286", "1"): "1 .. SessionOpen",
        ("FIX.4.4", "286", "2"): "2 .. DeliverySettlementEntry",
        ("FIX.4.4", "286", "3"): "3 .. ExpectedEntry",
        ("FIX.4.4", "286", "4"): "4 .. EntryFromPreviousBusinessDay",
        ("FIX.4.4", "286", "5"): "5 .. TheoreticalPriceValue",
        ("FIX.4.4", "291", "1"): "1 .. Bankrupt",
        ("FIX.4.4", "291", "2"): "2 .. PendingDelisting",
        ("FIX.4.4", "292", "A"): "A .. ExDividend",
        ("FIX.4.4", "292", "B"): "B .. ExDistribution",
        ("FIX.4.4", "292", "C"): "C .. ExRights",
        ("FIX.4.4", "292", "D"): "D .. New",
        ("FIX.4.4", "292", "E"): "E .. ExInterest",
        ("FIX.4.4", "297", "0"): "0 .. Accepted",
        ("FIX.4.4", "297", "1"): "1 .. CancelForSymbol",
        ("FIX.4.4", "297", "2"): "2 .. CanceledForSecurityType",
        ("FIX.4.4", "297", "3"): "3 .. CanceledForUnderlying",
        ("FIX.4.4", "297", "4"): "4 .. CanceledAll",
        ("FIX.4.4", "297", "5"): "5 .. Rejected",
        ("FIX.4.4", "297", "6"): "6 .. RemovedFromMarket",
        ("FIX.4.4", "297", "7"): "7 .. Expired",
        ("FIX.4.4", "297", "8"): "8 .. Query",
        ("FIX.4.4", "297", "9"): "9 .. QuoteNotFound",
        ("FIX.4.4", "297", "10"): "10 .. Pending",
        ("FIX.4.4", "297", "11"): "11 .. Pass",
        ("FIX.4.4", "297", "12"): "12 .. LockedMarketWarning",
        ("FIX.4.4", "297", "13"): "13 .. CrossMarketWarning",
        ("FIX.4.4", "297", "14"): "14 .. CanceledDueToLockMarket",
        ("FIX.4.4", "297", "15"): "15 .. CanceledDueToCrossMarket",
        ("FIX.4.4", "298", "1"): "1 .. CancelForOneOrMoreSecurities",
        ("FIX.4.4", "298", "2"): "2 .. CancelForSecurityType",
        ("FIX.4.4", "298", "3"): "3 .. CancelForUnderlyingSecurity",
        ("FIX.4.4", "298", "4"): "4 .. CancelAllQuotes",
        ("FIX.4.4", "300", "1"): "1 .. UnknownSymbol",
        ("FIX.4.4", "300", "2"): "2 .. Exchange",
        ("FIX.4.4", "300", "3"): "3 .. QuoteRequestExceedsLimit",
        ("FIX.4.4", "300", "4"): "4 .. TooLateToEnter",
        ("FIX.4.4", "300", "5"): "5 .. UnknownQuote",
        ("FIX.4.4", "300", "6"): "6 .. DuplicateQuote",
        ("FIX.4.4", "300", "7"): "7 .. InvalidBid",
        ("FIX.4.4", "300", "8"): "8 .. InvalidPrice",
        ("FIX.4.4", "300", "9"): "9 .. NotAuthorizedToQuoteSecurity",
        ("FIX.4.4", "300", "99"): "99 .. Other",
        ("FIX.4.4", "301", "0"): "0 .. NoAcknowledgement",
        ("FIX.4.4", "301", "1"): "1 .. AcknowledgeOnlyNegativeOrErroneousQuotes",
        ("FIX.4.4", "301", "2"): "2 .. AcknowledgeEachQuoteMessage",
        ("FIX.4.4", "303", "1"): "1 .. Manual",
        ("FIX.4.4", "303", "2"): "2 .. Automatic",
        ("FIX.4.4", "321", "0"): "0 .. RequestSecurityIdentityAndSpecifications",
        ("FIX.4.4", "321", "1"): "1 .. RequestSecurityIdentityForSpecifications",
        ("FIX.4.4", "321", "2"): "2 .. RequestListSecurityTypes",
        ("FIX.4.4", "321", "3"): "3 .. RequestListSecurities",
        ("FIX.4.4", "323", "1"): "1 .. AcceptAsIs",
        ("FIX.4.4", "323", "2"): "2 .. AcceptWithRevisions",
        ("FIX.4.4", "323", "5"): "5 .. RejectSecurityProposal",
        ("FIX.4.4", "323", "6"): "6 .. CannotMatchSelectionCriteria",
        ("FIX.4.4", "325", "Y"): "Y .. MessageIsBeingSentUnsolicited",
        ("FIX.4.4", "325", "N"): "N .. MessageIsBeingSentAsAResultOfAPriorRequest",
        ("FIX.4.4", "326", "1"): "1 .. OpeningDelay",
        ("FIX.4.4", "326", "2"): "2 .. TradingHalt",
        ("FIX.4.4", "326", "3"): "3 .. Resume",
        ("FIX.4.4", "326", "4"): "4 .. NoOpen",
        ("FIX.4.4", "326", "5"): "5 .. PriceIndication",
        ("FIX.4.4", "326", "6"): "6 .. TradingRangeIndication",
        ("FIX.4.4", "326", "7"): "7 .. MarketImbalanceBuy",
        ("FIX.4.4", "326", "8"): "8 .. MarketImbalanceSell",
        ("FIX.4.4", "326", "9"): "9 .. MarketOnCloseImbalanceBuy",
        ("FIX.4.4", "326", "10"): "10 .. MarketOnCloseImbalanceSell",
        ("FIX.4.4", "326", "12"): "12 .. NoMarketImbalance",
        ("FIX.4.4", "326", "13"): "13 .. NoMarketOnCloseImbalance",
        ("FIX.4.4", "326", "14"): "14 .. ITSPreOpening",
        ("FIX.4.4", "326", "15"): "15 .. NewPriceIndication",
        ("FIX.4.4", "326", "16"): "16 .. TradeDisseminationTime",
        ("FIX.4.4", "326", "17"): "17 .. ReadyToTrade",
        ("FIX.4.4", "326", "18"): "18 .. NotAvailableForTrading",
        ("FIX.4.4", "326", "19"): "19 .. NotTradedOnThisMarket",
        ("FIX.4.4", "326", "20"): "20 .. UnknownOrInvalid",
        ("FIX.4.4", "326", "21"): "21 .. PreOpen",
        ("FIX.4.4", "326", "22"): "22 .. OpeningRotation",
        ("FIX.4.4", "326", "23"): "23 .. FastMarket",
        ("FIX.4.4", "327", "I"): "I .. OrderImbalance",
        ("FIX.4.4", "327", "X"): "X .. EquipmentChangeover",
        ("FIX.4.4", "327", "P"): "P .. NewsPending",
        ("FIX.4.4", "327", "D"): "D .. NewsDissemination",
        ("FIX.4.4", "327", "E"): "E .. OrderInflux",
        ("FIX.4.4", "327", "M"): "M .. AdditionalInformation",
        ("FIX.4.4", "328", "Y"): "Y .. HaltWasDueToCommonStockBeingHalted",
        ("FIX.4.4", "328", "N"): "N .. HaltWasNotRelatedToAHaltOfTheCommonStock",
        ("FIX.4.4", "329", "Y"): "Y .. RelatedToSecurityHalt",
        ("FIX.4.4", "329", "N"): "N .. NotRelatedToSecurityHalt",
        ("FIX.4.4", "334", "1"): "1 .. Cancel",
        ("FIX.4.4", "334", "2"): "2 .. Error",
        ("FIX.4.4", "334", "3"): "3 .. Correction",
        ("FIX.4.4", "338", "1"): "1 .. Electronic",
        ("FIX.4.4", "338", "2"): "2 .. OpenOutcry",
        ("FIX.4.4", "338", "3"): "3 .. TwoParty",
        ("FIX.4.4", "339", "1"): "1 .. Testing",
        ("FIX.4.4", "339", "2"): "2 .. Simulated",
        ("FIX.4.4", "339", "3"): "3 .. Production",
        ("FIX.4.4", "340", "0"): "0 .. Unknown",
        ("FIX.4.4", "340", "1"): "1 .. Halted",
        ("FIX.4.4", "340", "2"): "2 .. Open",
        ("FIX.4.4", "340", "3"): "3 .. Closed",
        ("FIX.4.4", "340", "4"): "4 .. PreOpen",
        ("FIX.4.4", "340", "5"): "5 .. PreClose",
        ("FIX.4.4", "340", "6"): "6 .. RequestRejected",
        ("FIX.4.4", "347", "ISO-2022-JP"): "ISO-2022-JP .. ISO2022JP",
        ("FIX.4.4", "347", "EUC-JP"): "EUC-JP .. EUCJP",
        ("FIX.4.4", "347", "Shift_JIS"): "Shift_JIS .. ShiftJIS",
        ("FIX.4.4", "347", "UTF-8"): "UTF-8 .. UTF8",
        ("FIX.4.4", "373", "0"): "0 .. InvalidTagNumber",
        ("FIX.4.4", "373", "1"): "1 .. RequiredTagMissing",
        ("FIX.4.4", "373", "2"): "2 .. TagNotDefinedForThisMessageType",
        ("FIX.4.4", "373", "3"): "3 .. UndefinedTag",
        ("FIX.4.4", "373", "4"): "4 .. TagSpecifiedWithoutAValue",
        ("FIX.4.4", "373", "5"): "5 .. ValueIsIncorrect",
        ("FIX.4.4", "373", "6"): "6 .. IncorrectDataFormatForValue",
        ("FIX.4.4", "373", "7"): "7 .. DecryptionProblem",
        ("FIX.4.4", "373", "8"): "8 .. SignatureProblem",
        ("FIX.4.4", "373", "9"): "9 .. CompIDProblem",
        ("FIX.4.4", "373", "10"): "10 .. SendingTimeAccuracyProblem",
        ("FIX.4.4", "373", "11"): "11 .. InvalidMsgType",
        ("FIX.4.4", "373", "12"): "12 .. XMLValidationError",
        ("FIX.4.4", "373", "13"): "13 .. TagAppearsMoreThanOnce",
        ("FIX.4.4", "373", "14"): "14 .. TagSpecifiedOutOfRequiredOrder",
        ("FIX.4.4", "373", "15"): "15 .. RepeatingGroupFieldsOutOfOrder",
        ("FIX.4.4", "373", "16"): "16 .. IncorrectNumInGroupCountForRepeatingGroup",
        ("FIX.4.4", "373", "17"): "17 .. Non",
        ("FIX.4.4", "373", "99"): "99 .. Other",
        ("FIX.4.4", "374", "N"): "N .. New",
        ("FIX.4.4", "374", "C"): "C .. Cancel",
        ("FIX.4.4", "377", "Y"): "Y .. WasSolicited",
        ("FIX.4.4", "377", "N"): "N .. WasNotSolicited",
        ("FIX.4.4", "378", "0"): "0 .. GTCorporateAction",
        ("FIX.4.4", "378", "1"): "1 .. GTRenewal",
        ("FIX.4.4", "378", "2"): "2 .. VerbalChange",
        ("FIX.4.4", "378", "3"): "3 .. RepricingOfOrder",
        ("FIX.4.4", "378", "4"): "4 .. BrokerOption",
        ("FIX.4.4", "378", "5"): "5 .. PartialDeclineOfOrderQty",
        ("FIX.4.4", "378", "6"): "6 .. CancelOnTradingHalt",
        ("FIX.4.4", "378", "7"): "7 .. CancelOnSystemFailure",
        ("FIX.4.4", "378", "8"): "8 .. Market",
        ("FIX.4.4", "378", "9"): "9 .. Canceled",
        ("FIX.4.4", "378", "10"): "10 .. WarehouseRecap",
        ("FIX.4.4", "378", "99"): "99 .. Other",
        ("FIX.4.4", "380", "0"): "0 .. Other",
        ("FIX.4.4", "380", "1"): "1 .. UnknownID",
        ("FIX.4.4", "380", "2"): "2 .. UnknownSecurity",
        ("FIX.4.4", "380", "3"): "3 .. UnsupportedMessageType",
        ("FIX.4.4", "380", "4"): "4 .. ApplicationNotAvailable",
        ("FIX.4.4", "380", "5"): "5 .. ConditionallyRequiredFieldMissing",
        ("FIX.4.4", "380", "6"): "6 .. NotAuthorized",
        ("FIX.4.4", "380", "7"): "7 .. DeliverToFirmNotAvailableAtThisTime",
        ("FIX.4.4", "385", "S"): "S .. Send",
        ("FIX.4.4", "385", "R"): "R .. Receive",
        ("FIX.4.4", "388", "0"): "0 .. RelatedToDisplayedPrice",
        ("FIX.4.4", "388", "1"): "1 .. RelatedToMarketPrice",
        ("FIX.4.4", "388", "2"): "2 .. RelatedToPrimaryPrice",
        ("FIX.4.4", "388", "3"): "3 .. RelatedToLocalPrimaryPrice",
        ("FIX.4.4", "388", "4"): "4 .. RelatedToMidpointPrice",
        ("FIX.4.4", "388", "5"): "5 .. RelatedToLastTradePrice",
        ("FIX.4.4", "388", "6"): "6 .. RelatedToVWAP",
        ("FIX.4.4", "394", "1"): "1 .. NonDisclosed",
        ("FIX.4.4", "394", "2"): "2 .. Disclosed",
        ("FIX.4.4", "394", "3"): "3 .. NoBiddingProcess",
        ("FIX.4.4", "399", "1"): "1 .. Sector",
        ("FIX.4.4", "399", "2"): "2 .. Country",
        ("FIX.4.4", "399", "3"): "3 .. Index",
        ("FIX.4.4", "401", "1"): "1 .. SideValue1",
        ("FIX.4.4", "401", "2"): "2 .. SideValue2",
        ("FIX.4.4", "409", "1"): "1 .. FiveDayMovingAverage",
        ("FIX.4.4", "409", "2"): "2 .. TwentyDayMovingAverage",
        ("FIX.4.4", "409", "3"): "3 .. NormalMarketSize",
        ("FIX.4.4", "409", "4"): "4 .. Other",
        ("FIX.4.4", "411", "Y"): "Y .. True",
        ("FIX.4.4", "411", "N"): "N .. False",
        ("FIX.4.4", "414", "1"): "1 .. BuySideRequests",
        ("FIX.4.4", "414", "2"): "2 .. SellSideSends",
        ("FIX.4.4", "414", "3"): "3 .. RealTimeExecutionReports",
        ("FIX.4.4", "416", "1"): "1 .. Net",
        ("FIX.4.4", "416", "2"): "2 .. Gross",
        ("FIX.4.4", "418", "R"): "R .. RiskTrade",
        ("FIX.4.4", "418", "G"): "G .. VWAPGuarantee",
        ("FIX.4.4", "418", "A"): "A .. Agency",
        ("FIX.4.4", "418", "J"): "J .. GuaranteedClose",
        ("FIX.4.4", "419", "2"): "2 .. ClosingPriceAtMorningSession",
        ("FIX.4.4", "419", "3"): "3 .. ClosingPrice",
        ("FIX.4.4", "419", "4"): "4 .. CurrentPrice",
        ("FIX.4.4", "419", "5"): "5 .. SQ",
        ("FIX.4.4", "419", "6"): "6 .. VWAPThroughADay",
        ("FIX.4.4", "419", "7"): "7 .. VWAPThroughAMorningSession",
        ("FIX.4.4", "419", "8"): "8 .. VWAPThroughAnAfternoonSession",
        ("FIX.4.4", "419", "9"): "9 .. VWAPThroughADayExcept",
        ("FIX.4.4", "419", "A"): "A .. VWAPThroughAMorningSessionExcept",
        ("FIX.4.4", "419", "B"): "B .. VWAPThroughAnAfternoonSessionExcept",
        ("FIX.4.4", "419", "C"): "C .. Strike",
        ("FIX.4.4", "419", "D"): "D .. Open",
        ("FIX.4.4", "419", "Z"): "Z .. Others",
        ("FIX.4.4", "423", "1"): "1 .. Percentage",
        ("FIX.4.4", "423", "2"): "2 .. PerUnit",
        ("FIX.4.4", "423", "3"): "3 .. FixedAmount",
        ("FIX.4.4", "423", "4"): "4 .. Discount",
        ("FIX.4.4", "423", "5"): "5 .. Premium",
        ("FIX.4.4", "423", "6"): "6 .. Spread",
        ("FIX.4.4", "423", "7"): "7 .. TEDPrice",
        ("FIX.4.4", "423", "8"): "8 .. TEDYield",
        ("FIX.4.4", "423", "9"): "9 .. Yield",
        ("FIX.4.4", "423", "10"): "10 .. FixedCabinetTradePrice",
        ("FIX.4.4", "423", "11"): "11 .. VariableCabinetTradePrice",
        ("FIX.4.4", "427", "0"): "0 .. BookOutAllTradesOnDayOfExecution",
        ("FIX.4.4", "427", "1"): "1 .. AccumulateUntilFilledOrExpired",
        ("FIX.4.4", "427", "2"): "2 .. AccumulateUntilVerballlyNotifiedOtherwise",
        ("FIX.4.4", "429", "1"): "1 .. Ack",
        ("FIX.4.4", "429", "2"): "2 .. Response",
        ("FIX.4.4", "429", "3"): "3 .. Timed",
        ("FIX.4.4", "429", "4"): "4 .. ExecStarted",
        ("FIX.4.4", "429", "5"): "5 .. AllDone",
        ("FIX.4.4", "429", "6"): "6 .. Alert",
        ("FIX.4.4", "430", "1"): "1 .. Net",
        ("FIX.4.4", "430", "2"): "2 .. Gross",
        ("FIX.4.4", "431", "1"): "1 .. InBiddingProcess",
        ("FIX.4.4", "431", "2"): "2 .. ReceivedForExecution",
        ("FIX.4.4", "431", "3"): "3 .. Executing",
        ("FIX.4.4", "431", "4"): "4 .. Cancelling",
        ("FIX.4.4", "431", "5"): "5 .. Alert",
        ("FIX.4.4", "431", "6"): "6 .. AllDone",
        ("FIX.4.4", "431", "7"): "7 .. Reject",
        ("FIX.4.4", "433", "1"): "1 .. Immediate",
        ("FIX.4.4", "433", "2"): "2 .. WaitForInstruction",
        ("FIX.4.4", "433", "3"): "3 .. SellDriven",
        ("FIX.4.4", "433", "4"): "4 .. BuyDrivenCashTopUp",
        ("FIX.4.4", "433", "5"): "5 .. BuyDrivenCashWithdraw",
        ("FIX.4.4", "434", "1"): "1 .. OrderCancelRequest",
        ("FIX.4.4", "434", "2"): "2 .. OrderCancel",
        ("FIX.4.4", "442", "1"): "1 .. SingleSecurity",
        ("FIX.4.4", "442", "2"): "2 .. IndividualLegOfAMultiLegSecurity",
        ("FIX.4.4", "442", "3"): "3 .. MultiLegSecurity",
        ("FIX.4.4", "447", "B"): "B .. BIC",
        ("FIX.4.4", "447", "C"): "C .. GeneralIdentifier",
        ("FIX.4.4", "447", "D"): "D .. Proprietary",
        ("FIX.4.4", "447", "E"): "E .. ISOCountryCode",
        ("FIX.4.4", "447", "F"): "F .. SettlementEntityLocation",
        ("FIX.4.4", "447", "G"): "G .. MIC",
        ("FIX.4.4", "447", "H"): "H .. CSDParticipant",
        ("FIX.4.4", "447", "1"): "1 .. KoreanInvestorID",
        ("FIX.4.4", "447", "2"): "2 .. TaiwaneseForeignInvestorID",
        ("FIX.4.4", "447", "3"): "3 .. TaiwaneseTradingAcct",
        ("FIX.4.4", "447", "4"): "4 .. MalaysianCentralDepository",
        ("FIX.4.4", "447", "5"): "5 .. ChineseInvestorID",
        ("FIX.4.4", "447", "6"): "6 .. UKNationalInsuranceOrPensionNumber",
        ("FIX.4.4", "447", "7"): "7 .. USSocialSecurityNumber",
        ("FIX.4.4", "447", "8"): "8 .. USEmployerOrTaxIDNumber",
        ("FIX.4.4", "447", "9"): "9 .. AustralianBusinessNumber",
        ("FIX.4.4", "447", "A"): "A .. AustralianTaxFileNumber",
        ("FIX.4.4", "447", "I"): "I .. ISITCAcronym",
        ("FIX.4.4", "452", "1"): "1 .. ExecutingFirm",
        ("FIX.4.4", "452", "2"): "2 .. BrokerOfCredit",
        ("FIX.4.4", "452", "3"): "3 .. ClientID",
        ("FIX.4.4", "452", "4"): "4 .. ClearingFirm",
        ("FIX.4.4", "452", "5"): "5 .. InvestorID",
        ("FIX.4.4", "452", "6"): "6 .. IntroducingFirm",
        ("FIX.4.4", "452", "7"): "7 .. EnteringFirm",
        ("FIX.4.4", "452", "8"): "8 .. Locate",
        ("FIX.4.4", "452", "9"): "9 .. FundManagerClientID",
        ("FIX.4.4", "452", "10"): "10 .. SettlementLocation",
        ("FIX.4.4", "452", "11"): "11 .. OrderOriginationTrader",
        ("FIX.4.4", "452", "12"): "12 .. ExecutingTrader",
        ("FIX.4.4", "452", "13"): "13 .. OrderOriginationFirm",
        ("FIX.4.4", "452", "14"): "14 .. GiveupClearingFirm",
        ("FIX.4.4", "452", "15"): "15 .. CorrespondantClearingFirm",
        ("FIX.4.4", "452", "16"): "16 .. ExecutingSystem",
        ("FIX.4.4", "452", "17"): "17 .. ContraFirm",
        ("FIX.4.4", "452", "18"): "18 .. ContraClearingFirm",
        ("FIX.4.4", "452", "19"): "19 .. SponsoringFirm",
        ("FIX.4.4", "452", "20"): "20 .. UnderlyingContraFirm",
        ("FIX.4.4", "452", "21"): "21 .. ClearingOrganization",
        ("FIX.4.4", "452", "22"): "22 .. Exchange",
        ("FIX.4.4", "452", "24"): "24 .. CustomerAccount",
        ("FIX.4.4", "452", "25"): "25 .. CorrespondentClearingOrganization",
        ("FIX.4.4", "452", "26"): "26 .. CorrespondentBroker",
        ("FIX.4.4", "452", "27"): "27 .. Buyer",
        ("FIX.4.4", "452", "28"): "28 .. Custodian",
        ("FIX.4.4", "452", "29"): "29 .. Intermediary",
        ("FIX.4.4", "452", "30"): "30 .. Agent",
        ("FIX.4.4", "452", "31"): "31 .. SubCustodian",
        ("FIX.4.4", "452", "32"): "32 .. Beneficiary",
        ("FIX.4.4", "452", "33"): "33 .. InterestedParty",
        ("FIX.4.4", "452", "34"): "34 .. RegulatoryBody",
        ("FIX.4.4", "452", "35"): "35 .. LiquidityProvider",
        ("FIX.4.4", "452", "36"): "36 .. EnteringTrader",
        ("FIX.4.4", "452", "37"): "37 .. ContraTrader",
        ("FIX.4.4", "452", "38"): "38 .. PositionAccount",
        ("FIX.4.4", "460", "1"): "1 .. AGENCY",
        ("FIX.4.4", "460", "2"): "2 .. COMMODITY",
        ("FIX.4.4", "460", "3"): "3 .. CORPORATE",
        ("FIX.4.4", "460", "4"): "4 .. CURRENCY",
        ("FIX.4.4", "460", "5"): "5 .. EQUITY",
        ("FIX.4.4", "460", "6"): "6 .. GOVERNMENT",
        ("FIX.4.4", "460", "7"): "7 .. INDEX",
        ("FIX.4.4", "460", "8"): "8 .. LOAN",
        ("FIX.4.4", "460", "9"): "9 .. MONEYMARKET",
        ("FIX.4.4", "460", "10"): "10 .. MORTGAGE",
        ("FIX.4.4", "460", "11"): "11 .. MUNICIPAL",
        ("FIX.4.4", "460", "12"): "12 .. OTHER",
        ("FIX.4.4", "460", "13"): "13 .. FINANCING",
        ("FIX.4.4", "464", "Y"): "Y .. True",
        ("FIX.4.4", "464", "N"): "N .. Fales",
        ("FIX.4.4", "468", "0"): "0 .. RoundToNearest",
        ("FIX.4.4", "468", "1"): "1 .. RoundDown",
        ("FIX.4.4", "468", "2"): "2 .. RoundUp",
        ("FIX.4.4", "477", "1"): "1 .. CREST",
        ("FIX.4.4", "477", "2"): "2 .. NSCC",
        ("FIX.4.4", "477", "3"): "3 .. Euroclear",
        ("FIX.4.4", "477", "4"): "4 .. Clearstream",
        ("FIX.4.4", "477", "5"): "5 .. Cheque",
        ("FIX.4.4", "477", "6"): "6 .. TelegraphicTransfer",
        ("FIX.4.4", "477", "7"): "7 .. FedWire",
        ("FIX.4.4", "477", "8"): "8 .. DirectCredit",
        ("FIX.4.4", "477", "9"): "9 .. ACHCredit",
        ("FIX.4.4", "477", "10"): "10 .. BPAY",
        ("FIX.4.4", "477", "11"): "11 .. HighValueClearingSystemHVACS",
        ("FIX.4.4", "477", "12"): "12 .. ReinvestInFund",
        ("FIX.4.4", "480", "Y"): "Y .. Yes",
        ("FIX.4.4", "480", "N"): "N .. NoExecutionOnly",
        ("FIX.4.4", "480", "M"): "M .. NoWaiverAgreement",
        ("FIX.4.4", "480", "O"): "O .. NoInstitutional",
        ("FIX.4.4", "481", "Y"): "Y .. Passed",
        ("FIX.4.4", "481", "N"): "N .. NotChecked",
        ("FIX.4.4", "481", "1"): "1 .. ExemptBelowLimit",
        ("FIX.4.4", "481", "2"): "2 .. ExemptMoneyType",
        ("FIX.4.4", "481", "3"): "3 .. ExemptAuthorised",
        ("FIX.4.4", "484", "B"): "B .. BidPrice",
        ("FIX.4.4", "484", "C"): "C .. CreationPrice",
        ("FIX.4.4", "484", "D"): "D .. CreationPricePlusAdjustmentPercent",
        ("FIX.4.4", "484", "E"): "E .. CreationPricePlusAdjustmentAmount",
        ("FIX.4.4", "484", "O"): "O .. OfferPrice",
        ("FIX.4.4", "484", "P"): "P .. OfferPriceMinusAdjustmentPercent",
        ("FIX.4.4", "484", "Q"): "Q .. OfferPriceMinusAdjustmentAmount",
        ("FIX.4.4", "484", "S"): "S .. SinglePrice",
        ("FIX.4.4", "492", "1"): "1 .. CREST",
        ("FIX.4.4", "492", "2"): "2 .. NSCC",
        ("FIX.4.4", "492", "3"): "3 .. Euroclear",
        ("FIX.4.4", "492", "4"): "4 .. Clearstream",
        ("FIX.4.4", "492", "5"): "5 .. Cheque",
        ("FIX.4.4", "492", "6"): "6 .. TelegraphicTransfer",
        ("FIX.4.4", "492", "7"): "7 .. FedWire",
        ("FIX.4.4", "492", "8"): "8 .. DebitCard",
        ("FIX.4.4", "492", "9"): "9 .. DirectDebit",
        ("FIX.4.4", "492", "10"): "10 .. DirectCredit",
        ("FIX.4.4", "492", "11"): "11 .. CreditCard",
        ("FIX.4.4", "492", "12"): "12 .. ACHDebit",
        ("FIX.4.4", "492", "13"): "13 .. ACHCredit",
        ("FIX.4.4", "492", "14"): "14 .. BPAY",
        ("FIX.4.4", "492", "15"): "15 .. HighValueClearingSystem",
        ("FIX.4.4", "495", "0"): "0 .. None",
        ("FIX.4.4", "495", "1"): "1 .. MaxiISA",
        ("FIX.4.4", "495", "2"): "2 .. TESSA",
        ("FIX.4.4", "495", "3"): "3 .. MiniCashISA",
        ("FIX.4.4", "495", "4"): "4 .. MiniStocksAndSharesISA",
        ("FIX.4.4", "495", "5"): "5 .. MiniInsuranceISA",
        ("FIX.4.4", "495", "6"): "6 .. CurrentYearPayment",
        ("FIX.4.4", "495", "7"): "7 .. PriorYearPayment",
        ("FIX.4.4", "495", "8"): "8 .. AssetTransfer",
        ("FIX.4.4", "495", "9"): "9 .. EmployeePriorYear",
        ("FIX.4.4", "495", "10"): "10 .. EmployeeCurrentYear",
        ("FIX.4.4", "495", "11"): "11 .. EmployerPriorYear",
        ("FIX.4.4", "495", "12"): "12 .. EmployerCurrentYear",
        ("FIX.4.4", "495", "13"): "13 .. NonFundPrototypeIRA",
        ("FIX.4.4", "495", "14"): "14 .. NonFundQualifiedPlan",
        ("FIX.4.4", "495", "15"): "15 .. DefinedContributionPlan",
        ("FIX.4.4", "495", "16"): "16 .. IRA",
        ("FIX.4.4", "495", "17"): "17 .. IRARollover",
        ("FIX.4.4", "495", "18"): "18 .. KEOGH",
        ("FIX.4.4", "495", "19"): "19 .. ProfitSharingPlan",
        ("FIX.4.4", "495", "20"): "20 .. US401K",
        ("FIX.4.4", "495", "21"): "21 .. SelfDirectedIRA",
        ("FIX.4.4", "495", "22"): "22 .. US403b",
        ("FIX.4.4", "495", "23"): "23 .. US457",
        ("FIX.4.4", "495", "24"): "24 .. RothIRAPrototype",
        ("FIX.4.4", "495", "25"): "25 .. RothIRANonPrototype",
        ("FIX.4.4", "495", "26"): "26 .. RothConversionIRAPrototype",
        ("FIX.4.4", "495", "27"): "27 .. RothConversionIRANonPrototype",
        ("FIX.4.4", "495", "28"): "28 .. EducationIRAPrototype",
        ("FIX.4.4", "495", "29"): "29 .. EducationIRANonPrototype",
        ("FIX.4.4", "497", "Y"): "Y .. Yes",
        ("FIX.4.4", "497", "N"): "N .. No",
        ("FIX.4.4", "506", "A"): "A .. Accepted",
        ("FIX.4.4", "506", "R"): "R .. Rejected",
        ("FIX.4.4", "506", "H"): "H .. Held",
        ("FIX.4.4", "506", "N"): "N .. Reminder",
        ("FIX.4.4", "507", "1"): "1 .. InvalidAccountType",
        ("FIX.4.4", "507", "2"): "2 .. InvalidTaxExemptType",
        ("FIX.4.4", "507", "3"): "3 .. InvalidOwnershipType",
        ("FIX.4.4", "507", "4"): "4 .. NoRegDetails",
        ("FIX.4.4", "507", "5"): "5 .. InvalidRegSeqNo",
        ("FIX.4.4", "507", "6"): "6 .. InvalidRegDetails",
        ("FIX.4.4", "507", "7"): "7 .. InvalidMailingDetails",
        ("FIX.4.4", "507", "8"): "8 .. InvalidMailingInstructions",
        ("FIX.4.4", "507", "9"): "9 .. InvalidInvestorID",
        ("FIX.4.4", "507", "10"): "10 .. InvalidInvestorIDSource",
        ("FIX.4.4", "507", "11"): "11 .. InvalidDateOfBirth",
        ("FIX.4.4", "507", "12"): "12 .. InvalidCountry",
        ("FIX.4.4", "507", "13"): "13 .. InvalidDistribInstns",
        ("FIX.4.4", "507", "14"): "14 .. InvalidPercentage",
        ("FIX.4.4", "507", "15"): "15 .. InvalidPaymentMethod",
        ("FIX.4.4", "507", "16"): "16 .. InvalidAccountName",
        ("FIX.4.4", "507", "17"): "17 .. InvalidAgentCode",
        ("FIX.4.4", "507", "18"): "18 .. InvalidAccountNum",
        ("FIX.4.4", "507", "99"): "99 .. Other",
        ("FIX.4.4", "514", "0"): "0 .. New",
        ("FIX.4.4", "514", "1"): "1 .. Replace",
        ("FIX.4.4", "514", "2"): "2 .. Cancel",
        ("FIX.4.4", "517", "J"): "J .. JointInvestors",
        ("FIX.4.4", "517", "T"): "T .. TenantsInCommon",
        ("FIX.4.4", "517", "2"): "2 .. JointTrustees",
        ("FIX.4.4", "519", "1"): "1 .. CommissionAmount",
        ("FIX.4.4", "519", "2"): "2 .. CommissionPercent",
        ("FIX.4.4", "519", "3"): "3 .. InitialChargeAmount",
        ("FIX.4.4", "519", "4"): "4 .. InitialChargePercent",
        ("FIX.4.4", "519", "5"): "5 .. DiscountAmount",
        ("FIX.4.4", "519", "6"): "6 .. DiscountPercent",
        ("FIX.4.4", "519", "7"): "7 .. DilutionLevyAmount",
        ("FIX.4.4", "519", "8"): "8 .. DilutionLevyPercent",
        ("FIX.4.4", "519", "9"): "9 .. ExitChargeAmount",
        ("FIX.4.4", "519", "10"): "10 .. ExitChargePercent",
        ("FIX.4.4", "519", "11"): "11 .. FundBasedRenewalCommissionPercent",
        ("FIX.4.4", "519", "12"): "12 .. ProjectedFundValue",
        ("FIX.4.4", "519", "13"): "13 .. FundBasedRenewalCommissionOnOrder",
        ("FIX.4.4", "519", "14"): "14 .. FundBasedRenewalCommissionOnFund",
        ("FIX.4.4", "519", "15"): "15 .. NetSettlementAmount",
        ("FIX.4.4", "522", "1"): "1 .. IndividualInvestor",
        ("FIX.4.4", "522", "2"): "2 .. PublicCompany",
        ("FIX.4.4", "522", "3"): "3 .. PrivateCompany",
        ("FIX.4.4", "522", "4"): "4 .. IndividualTrustee",
        ("FIX.4.4", "522", "5"): "5 .. CompanyTrustee",
        ("FIX.4.4", "522", "6"): "6 .. PensionPlan",
        ("FIX.4.4", "522", "7"): "7 .. CustodianUnderGiftsToMinorsAct",
        ("FIX.4.4", "522", "8"): "8 .. Trusts",
        ("FIX.4.4", "522", "9"): "9 .. Fiduciaries",
        ("FIX.4.4", "522", "10"): "10 .. NetworkingSubAccount",
        ("FIX.4.4", "522", "11"): "11 .. NonProfitOrganization",
        ("FIX.4.4", "522", "12"): "12 .. CorporateBody",
        ("FIX.4.4", "522", "13"): "13 .. Nominee",
        ("FIX.4.4", "528", "A"): "A .. Agency",
        ("FIX.4.4", "528", "G"): "G .. Proprietary",
        ("FIX.4.4", "528", "I"): "I .. Individual",
        ("FIX.4.4", "528", "P"): "P .. Principal",
        ("FIX.4.4", "528", "R"): "R .. RisklessPrincipal",
        ("FIX.4.4", "528", "W"): "W .. AgentForOtherMember",
        ("FIX.4.4", "529", "1"): "1 .. ProgramTrade",
        ("FIX.4.4", "529", "2"): "2 .. IndexArbitrage",
        ("FIX.4.4", "529", "3"): "3 .. NonIndexArbitrage",
        ("FIX.4.4", "529", "4"): "4 .. CompetingMarketMaker",
        ("FIX.4.4", "529", "5"): "5 .. ActingAsMarketMakerOrSpecialistInSecurity",
        ("FIX.4.4", "529", "6"): "6 .. ActingAsMarketMakerOrSpecialistInUnderlying",
        ("FIX.4.4", "529", "7"): "7 .. ForeignEntity",
        ("FIX.4.4", "529", "8"): "8 .. ExternalMarketParticipant",
        ("FIX.4.4", "529", "9"): "9 .. ExternalInterConnectedMarketLinkage",
        ("FIX.4.4", "529", "A"): "A .. RisklessArbitrage",
        ("FIX.4.4", "530", "1"): "1 .. CancelOrdersForASecurity",
        ("FIX.4.4", "530", "2"): "2 .. CancelOrdersForAnUnderlyingSecurity",
        ("FIX.4.4", "530", "3"): "3 .. CancelOrdersForAProduct",
        ("FIX.4.4", "530", "4"): "4 .. CancelOrdersForACFICode",
        ("FIX.4.4", "530", "5"): "5 .. CancelOrdersForASecurityType",
        ("FIX.4.4", "530", "6"): "6 .. CancelOrdersForATradingSession",
        ("FIX.4.4", "530", "7"): "7 .. CancelAllOrders",
        ("FIX.4.4", "531", "0"): "0 .. CancelRequestRejected",
        ("FIX.4.4", "531", "1"): "1 .. CancelOrdersForASecurity",
        ("FIX.4.4", "531", "2"): "2 .. CancelOrdersForAnUnderlyingSecurity",
        ("FIX.4.4", "531", "3"): "3 .. CancelOrdersForAProduct",
        ("FIX.4.4", "531", "4"): "4 .. CancelOrdersForACFICode",
        ("FIX.4.4", "531", "5"): "5 .. CancelOrdersForASecurityType",
        ("FIX.4.4", "531", "6"): "6 .. CancelOrdersForATradingSession",
        ("FIX.4.4", "531", "7"): "7 .. CancelAllOrders",
        ("FIX.4.4", "532", "0"): "0 .. MassCancelNotSupported",
        ("FIX.4.4", "532", "1"): "1 .. InvalidOrUnknownSecurity",
        ("FIX.4.4", "532", "2"): "2 .. InvalidOrUnkownUnderlyingSecurity",
        ("FIX.4.4", "532", "3"): "3 .. InvalidOrUnknownProduct",
        ("FIX.4.4", "532", "4"): "4 .. InvalidOrUnknownCFICode",
        ("FIX.4.4", "532", "5"): "5 .. InvalidOrUnknownSecurityType",
        ("FIX.4.4", "532", "6"): "6 .. InvalidOrUnknownTradingSession",
        ("FIX.4.4", "532", "99"): "99 .. Other",
        ("FIX.4.4", "537", "0"): "0 .. Indicative",
        ("FIX.4.4", "537", "1"): "1 .. Tradeable",
        ("FIX.4.4", "537", "2"): "2 .. RestrictedTradeable",
        ("FIX.4.4", "537", "3"): "3 .. Counter",
        ("FIX.4.4", "544", "1"): "1 .. Cash",
        ("FIX.4.4", "544", "2"): "2 .. MarginOpen",
        ("FIX.4.4", "544", "3"): "3 .. MarginClose",
        ("FIX.4.4", "546", "1"): "1 .. LocalMarket",
        ("FIX.4.4", "546", "2"): "2 .. National",
        ("FIX.4.4", "546", "3"): "3 .. Global",
        ("FIX.4.4", "547", "Y"): "Y .. Yes",
        ("FIX.4.4", "547", "N"): "N .. No",
        ("FIX.4.4", "549", "1"): "1 .. CrossAON",
        ("FIX.4.4", "549", "2"): "2 .. CrossIOC",
        ("FIX.4.4", "549", "3"): "3 .. CrossOneSide",
        ("FIX.4.4", "549", "4"): "4 .. CrossSamePrice",
        ("FIX.4.4", "550", "0"): "0 .. None",
        ("FIX.4.4", "550", "1"): "1 .. BuySideIsPrioritized",
        ("FIX.4.4", "550", "2"): "2 .. SellSideIsPrioritized",
        ("FIX.4.4", "552", "1"): "1 .. OneSide",
        ("FIX.4.4", "552", "2"): "2 .. BothSides",
        ("FIX.4.4", "559", "0"): "0 .. Symbol",
        ("FIX.4.4", "559", "1"): "1 .. SecurityTypeAnd",
        ("FIX.4.4", "559", "2"): "2 .. Product",
        ("FIX.4.4", "559", "3"): "3 .. TradingSessionID",
        ("FIX.4.4", "559", "4"): "4 .. AllSecurities",
        ("FIX.4.4", "560", "0"): "0 .. ValidRequest",
        ("FIX.4.4", "560", "1"): "1 .. InvalidOrUnsupportedRequest",
        ("FIX.4.4", "560", "2"): "2 .. NoInstrumentsFound",
        ("FIX.4.4", "560", "3"): "3 .. NotAuthorizedToRetrieveInstrumentData",
        ("FIX.4.4", "560", "4"): "4 .. InstrumentDataTemporarilyUnavailable",
        ("FIX.4.4", "560", "5"): "5 .. RequestForInstrumentDataNotSupported",
        ("FIX.4.4", "563", "0"): "0 .. ReportByMulitlegSecurityOnly",
        ("FIX.4.4", "563", "1"): "1 .. ReportByMultilegSecurityAndInstrumentLegs",
        ("FIX.4.4", "563", "2"): "2 .. ReportByInstrumentLegsOnly",
        ("FIX.4.4", "567", "1"): "1 .. UnknownOrInvalidTradingSessionID",
        ("FIX.4.4", "567", "99"): "99 .. Other",
        ("FIX.4.4", "569", "0"): "0 .. AllTrades",
        ("FIX.4.4", "569", "1"): "1 .. MatchedTradesMatchingCriteria",
        ("FIX.4.4", "569", "2"): "2 .. UnmatchedTradesThatMatchCriteria",
        ("FIX.4.4", "569", "3"): "3 .. UnreportedTradesThatMatchCriteria",
        ("FIX.4.4", "569", "4"): "4 .. AdvisoriesThatMatchCriteria",
        ("FIX.4.4", "570", "Y"): "Y .. PerviouslyReportedToCounterparty",
        ("FIX.4.4", "570", "N"): "N .. NotReportedToCounterparty",
        ("FIX.4.4", "573", "0"): "0 .. Compared",
        ("FIX.4.4", "573", "1"): "1 .. Uncompared",
        ("FIX.4.4", "573", "2"): "2 .. AdvisoryOrAlert",
        ("FIX.4.4", "574", "A1"): "A1 .. ExactMatchPlus4BadgesExecTime",
        ("FIX.4.4", "574", "A2"): "A2 .. ExactMatchPlus4Badges",
        ("FIX.4.4", "574", "A3"): "A3 .. ExactMatchPlus2BadgesExecTime",
        ("FIX.4.4", "574", "A4"): "A4 .. ExactMatchPlus2Badges",
        ("FIX.4.4", "574", "A5"): "A5 .. ExactMatchPlusExecTime",
        ("FIX.4.4", "574", "AQ"): "AQ .. StampedAdvisoriesOrSpecialistAccepts",
        ("FIX.4.4", "574", "S1"): "S1 .. A1ExactMatchSummarizedQuantity",
        ("FIX.4.4", "574", "S2"): "S2 .. A2ExactMatchSummarizedQuantity",
        ("FIX.4.4", "574", "S3"): "S3 .. A3ExactMatchSummarizedQuantity",
        ("FIX.4.4", "574", "S4"): "S4 .. A4ExactMatchSummarizedQuantity",
        ("FIX.4.4", "574", "S5"): "S5 .. A5ExactMatchSummarizedQuantity",
        ("FIX.4.4", "574", "M1"): "M1 .. ExactMatchMinusBadgesTimes",
        ("FIX.4.4", "574", "M2"): "M2 .. SummarizedMatchMinusBadgesTimes",
        ("FIX.4.4", "574", "MT"): "MT .. OCSLockedIn",
        ("FIX.4.4", "574", "M3"): "M3 .. ACTAcceptedTrade",
        ("FIX.4.4", "574", "M4"): "M4 .. ACTDefaultTrade",
        ("FIX.4.4", "574", "M5"): "M5 .. ACTDefaultAfterM2",
        ("FIX.4.4", "574", "M6"): "M6 .. ACTM6Match",
        ("FIX.4.4", "575", "Y"): "Y .. TreatAsOddLot",
        ("FIX.4.4", "575", "N"): "N .. TreatAsRoundLot",
        ("FIX.4.4", "577", "0"): "0 .. ProcessNormally",
        ("FIX.4.4", "577", "1"): "1 .. ExcludeFromAllNetting",
        ("FIX.4.4", "577", "2"): "2 .. BilateralNettingOnly",
        ("FIX.4.4", "577", "3"): "3 .. ExClearing",
        ("FIX.4.4", "577", "4"): "4 .. SpecialTrade",
        ("FIX.4.4", "577", "5"): "5 .. MultilateralNetting",
        ("FIX.4.4", "577", "6"): "6 .. ClearAgainstCentralCounterparty",
        ("FIX.4.4", "577", "7"): "7 .. ExcludeFromCentralCounterparty",
        ("FIX.4.4", "577", "8"): "8 .. ManualMode",
        ("FIX.4.4", "577", "9"): "9 .. AutomaticPostingMode",
        ("FIX.4.4", "577", "10"): "10 .. AutomaticGiveUpMode",
        ("FIX.4.4", "577", "11"): "11 .. QualifiedServiceRepresentativeQSR",
        ("FIX.4.4", "577", "12"): "12 .. CustomerTrade",
        ("FIX.4.4", "577", "13"): "13 .. SelfClearing",
        ("FIX.4.4", "581", "1"): "1 .. CarriedCustomerSide",
        ("FIX.4.4", "581", "2"): "2 .. CarriedNonCustomerSide",
        ("FIX.4.4", "581", "3"): "3 .. HouseTrader",
        ("FIX.4.4", "581", "4"): "4 .. FloorTrader",
        ("FIX.4.4", "581", "6"): "6 .. CarriedNonCustomerSideCrossMargined",
        ("FIX.4.4", "581", "7"): "7 .. HouseTraderCrossMargined",
        ("FIX.4.4", "581", "8"): "8 .. JointBackOfficeAccount",
        ("FIX.4.4", "582", "1"): "1 .. MemberTradingForTheirOwnAccount",
        ("FIX.4.4", "582", "2"): "2 .. ClearingFirmTradingForItsProprietaryAccount",
        ("FIX.4.4", "582", "3"): "3 .. MemberTradingForAnotherMember",
        ("FIX.4.4", "582", "4"): "4 .. AllOther",
        ("FIX.4.4", "585", "1"): "1 .. StatusForOrdersForASecurity",
        ("FIX.4.4", "585", "2"): "2 .. StatusForOrdersForAnUnderlyingSecurity",
        ("FIX.4.4", "585", "3"): "3 .. StatusForOrdersForAProduct",
        ("FIX.4.4", "585", "4"): "4 .. StatusForOrdersForACFICode",
        ("FIX.4.4", "585", "5"): "5 .. StatusForOrdersForASecurityType",
        ("FIX.4.4", "585", "6"): "6 .. StatusForOrdersForATradingSession",
        ("FIX.4.4", "585", "7"): "7 .. StatusForAllOrders",
        ("FIX.4.4", "585", "8"): "8 .. StatusForOrdersForAPartyID",
        ("FIX.4.4", "589", "0"): "0 .. Auto",
        ("FIX.4.4", "589", "1"): "1 .. SpeakWithOrderInitiatorBeforeBooking",
        ("FIX.4.4", "589", "2"): "2 .. Accumulate",
        ("FIX.4.4", "590", "0"): "0 .. EachPartialExecutionIsABookableUnit",
        ("FIX.4.4", "590", "1"): "1 .. AggregatePartialExecutionsOnThisOrder",
        ("FIX.4.4", "590", "2"): "2 .. AggregateExecutionsForThisSymbol",
        ("FIX.4.4", "591", "0"): "0 .. ProRata",
        ("FIX.4.4", "591", "1"): "1 .. DoNotProRata",
        ("FIX.4.4", "626", "1"): "1 .. Calculated",
        ("FIX.4.4", "626", "2"): "2 .. Preliminary",
        ("FIX.4.4", "626", "5"): "5 .. ReadyToBook",
        ("FIX.4.4", "626", "7"): "7 .. WarehouseInstruction",
        ("FIX.4.4", "626", "8"): "8 .. RequestToIntermediary",
        ("FIX.4.4", "635", "B"): "B .. CBOEMember",
        ("FIX.4.4", "635", "C"): "C .. NonMemberAndCustomer",
        ("FIX.4.4", "635", "E"): "E .. EquityMemberAndClearingMember",
        ("FIX.4.4", "635", "F"): "F .. FullAndAssociateMember",
        ("FIX.4.4", "635", "H"): "H .. Firms106HAnd106J",
        ("FIX.4.4", "635", "I"): "I .. GIM",
        ("FIX.4.4", "635", "L"): "L .. Lessee106FEmployees",
        ("FIX.4.4", "635", "M"): "M .. AllOtherOwnershipTypes",
        ("FIX.4.4", "635", "1"): "1 .. FirstYearDelegate",
        ("FIX.4.4", "635", "2"): "2 .. SecondYearDelegate",
        ("FIX.4.4", "635", "3"): "3 .. ThirdYearDelegate",
        ("FIX.4.4", "635", "4"): "4 .. FourthYearDelegate",
        ("FIX.4.4", "635", "5"): "5 .. FifthYearDelegate",
        ("FIX.4.4", "635", "9"): "9 .. SixthYearDelegate",
        ("FIX.4.4", "636", "Y"): "Y .. Working",
        ("FIX.4.4", "636", "N"): "N .. NotWorking",
        ("FIX.4.4", "638", "0"): "0 .. PriorityUnchanged",
        ("FIX.4.4", "638", "1"): "1 .. LostPriorityAsResultOfOrderChange",
        ("FIX.4.4", "650", "Y"): "Y .. LegalConfirm",
        ("FIX.4.4", "650", "N"): "N .. DoesNotConsituteALegalConfirm",
        ("FIX.4.4", "658", "1"): "1 .. UnknownSymbol",
        ("FIX.4.4", "658", "2"): "2 .. Exchange",
        ("FIX.4.4", "658", "3"): "3 .. QuoteRequestExceedsLimit",
        ("FIX.4.4", "658", "4"): "4 .. TooLateToEnter",
        ("FIX.4.4", "658", "5"): "5 .. InvalidPrice",
        ("FIX.4.4", "658", "6"): "6 .. NotAuthorizedToRequestQuote",
        ("FIX.4.4", "658", "7"): "7 .. NoMatchForInquiry",
        ("FIX.4.4", "658", "8"): "8 .. NoMarketForInstrument",
        ("FIX.4.4", "658", "9"): "9 .. NoInventory",
        ("FIX.4.4", "658", "10"): "10 .. Pass",
        ("FIX.4.4", "658", "99"): "99 .. Other",
        ("FIX.4.4", "660", "1"): "1 .. BIC",
        ("FIX.4.4", "660", "2"): "2 .. SIDCode",
        ("FIX.4.4", "660", "3"): "3 .. TFM",
        ("FIX.4.4", "660", "4"): "4 .. OMGEO",
        ("FIX.4.4", "660", "5"): "5 .. DTCCCode",
        ("FIX.4.4", "660", "99"): "99 .. Other",
        ("FIX.4.4", "665", "1"): "1 .. Received",
        ("FIX.4.4", "665", "2"): "2 .. MismatchedAccount",
        ("FIX.4.4", "665", "3"): "3 .. MissingSettlementInstructions",
        ("FIX.4.4", "665", "4"): "4 .. Confirmed",
        ("FIX.4.4", "665", "5"): "5 .. RequestRejected",
        ("FIX.4.4", "666", "0"): "0 .. New",
        ("FIX.4.4", "666", "1"): "1 .. Replace",
        ("FIX.4.4", "666", "2"): "2 .. Cancel",
        ("FIX.4.4", "668", "1"): "1 .. BookEntry",
        ("FIX.4.4", "668", "2"): "2 .. Bearer",
        ("FIX.4.4", "690", "1"): "1 .. ParForPar",
        ("FIX.4.4", "690", "2"): "2 .. ModifiedDuration",
        ("FIX.4.4", "690", "4"): "4 .. Risk",
        ("FIX.4.4", "690", "5"): "5 .. Proceeds",
        ("FIX.4.4", "692", "1"): "1 .. Percent",
        ("FIX.4.4", "692", "2"): "2 .. PerShare",
        ("FIX.4.4", "692", "3"): "3 .. FixedAmount",
        ("FIX.4.4", "692", "4"): "4 .. Discount",
        ("FIX.4.4", "692", "5"): "5 .. Premium",
        ("FIX.4.4", "692", "6"): "6 .. Spread",
        ("FIX.4.4", "692", "7"): "7 .. TEDPrice",
        ("FIX.4.4", "692", "8"): "8 .. TEDYield",
        ("FIX.4.4", "692", "9"): "9 .. YieldSpread",
        ("FIX.4.4", "692", "10"): "10 .. Yield",
        ("FIX.4.4", "694", "1"): "1 .. Hit",
        ("FIX.4.4", "694", "2"): "2 .. Counter",
        ("FIX.4.4", "694", "3"): "3 .. Expired",
        ("FIX.4.4", "694", "4"): "4 .. Cover",
        ("FIX.4.4", "694", "5"): "5 .. DoneAway",
        ("FIX.4.4", "694", "6"): "6 .. Pass",
        ("FIX.4.4", "703", "TQ"): "TQ .. TransactionQuantity",
        ("FIX.4.4", "703", "IAS"): "IAS .. IntraSpreadQty",
        ("FIX.4.4", "703", "IES"): "IES .. InterSpreadQty",
        ("FIX.4.4", "703", "FIN"): "FIN .. EndOfDayQty",
        ("FIX.4.4", "703", "SOD"): "SOD .. StartOfDayQty",
        ("FIX.4.4", "703", "EX"): "EX .. OptionExerciseQty",
        ("FIX.4.4", "703", "AS"): "AS .. OptionAssignment",
        ("FIX.4.4", "703", "TX"): "TX .. TransactionFromExercise",
        ("FIX.4.4", "703", "TA"): "TA .. TransactionFromAssignment",
        ("FIX.4.4", "703", "PIT"): "PIT .. PitTradeQty",
        ("FIX.4.4", "703", "TRF"): "TRF .. TransferTradeQty",
        ("FIX.4.4", "703", "ETR"): "ETR .. ElectronicTradeQty",
        ("FIX.4.4", "703", "ALC"): "ALC .. AllocationTradeQty",
        ("FIX.4.4", "703", "PA"): "PA .. AdjustmentQty",
        ("FIX.4.4", "703", "ASF"): "ASF .. AsOfTradeQty",
        ("FIX.4.4", "703", "DLV"): "DLV .. DeliveryQty",
        ("FIX.4.4", "703", "TOT"): "TOT .. TotalTransactionQty",
        ("FIX.4.4", "703", "XM"): "XM .. CrossMarginQty",
        ("FIX.4.4", "703", "SPL"): "SPL .. IntegralSplit",
        ("FIX.4.4", "706", "0"): "0 .. Submitted",
        ("FIX.4.4", "706", "1"): "1 .. Accepted",
        ("FIX.4.4", "706", "2"): "2 .. Rejected",
        ("FIX.4.4", "707", "FMTM"): "FMTM .. FinalMarkToMarketAmount",
        ("FIX.4.4", "707", "IMTM"): "IMTM .. IncrementalMarkToMarketAmount",
        ("FIX.4.4", "707", "TVAR"): "TVAR .. TradeVariationAmount",
        ("FIX.4.4", "707", "SMTM"): "SMTM .. StartOfDayMarkToMarketAmount",
        ("FIX.4.4", "707", "PREM"): "PREM .. PremiumAmount",
        ("FIX.4.4", "707", "CRES"): "CRES .. CashResidualAmount",
        ("FIX.4.4", "707", "CASH"): "CASH .. CashAmount",
        ("FIX.4.4", "707", "VADJ"): "VADJ .. ValueAdjustedAmount",
        ("FIX.4.4", "709", "1"): "1 .. Exercise",
        ("FIX.4.4", "709", "2"): "2 .. DoNotExercise",
        ("FIX.4.4", "709", "3"): "3 .. PositionAdjustment",
        ("FIX.4.4", "709", "4"): "4 .. PositionChangeSubmission",
        ("FIX.4.4", "709", "5"): "5 .. Pledge",
        ("FIX.4.4", "712", "1"): "1 .. New",
        ("FIX.4.4", "712", "2"): "2 .. Replace",
        ("FIX.4.4", "712", "3"): "3 .. Cancel",
        ("FIX.4.4", "716", "ITD"): "ITD .. Intraday",
        ("FIX.4.4", "716", "RTH"): "RTH .. RegularTradingHours",
        ("FIX.4.4", "716", "ETH"): "ETH .. ElectronicTradingHours",
        ("FIX.4.4", "718", "0"): "0 .. ProcessRequestAsMarginDisposition",
        ("FIX.4.4", "718", "1"): "1 .. DeltaPlus",
        ("FIX.4.4", "718", "2"): "2 .. DeltaMinus",
        ("FIX.4.4", "718", "3"): "3 .. Final",
        ("FIX.4.4", "722", "0"): "0 .. Accepted",
        ("FIX.4.4", "722", "1"): "1 .. AcceptedWithWarnings",
        ("FIX.4.4", "722", "2"): "2 .. Rejected",
        ("FIX.4.4", "722", "3"): "3 .. Completed",
        ("FIX.4.4", "722", "4"): "4 .. CompletedWithWarnings",
        ("FIX.4.4", "723", "0"): "0 .. SuccessfulCompletion",
        ("FIX.4.4", "723", "1"): "1 .. Rejected",
        ("FIX.4.4", "723", "99"): "99 .. Other",
        ("FIX.4.4", "724", "0"): "0 .. Positions",
        ("FIX.4.4", "724", "1"): "1 .. Trades",
        ("FIX.4.4", "724", "2"): "2 .. Exercises",
        ("FIX.4.4", "724", "3"): "3 .. Assignments",
        ("FIX.4.4", "725", "0"): "0 .. Inband",
        ("FIX.4.4", "725", "1"): "1 .. OutOfBand",
        ("FIX.4.4", "728", "0"): "0 .. ValidRequest",
        ("FIX.4.4", "728", "1"): "1 .. InvalidOrUnsupportedRequest",
        ("FIX.4.4", "728", "2"): "2 .. NoPositionsFoundThatMatchCriteria",
        ("FIX.4.4", "728", "3"): "3 .. NotAuthorizedToRequestPositions",
        ("FIX.4.4", "728", "4"): "4 .. RequestForPositionNotSupported",
        ("FIX.4.4", "728", "99"): "99 .. Other",
        ("FIX.4.4", "729", "0"): "0 .. Completed",
        ("FIX.4.4", "729", "1"): "1 .. CompletedWithWarnings",
        ("FIX.4.4", "729", "2"): "2 .. Rejected",
        ("FIX.4.4", "731", "1"): "1 .. Final",
        ("FIX.4.4", "731", "2"): "2 .. Theoretical",
        ("FIX.4.4", "744", "R"): "R .. Random",
        ("FIX.4.4", "744", "P"): "P .. ProRata",
        ("FIX.4.4", "747", "A"): "A .. Automatic",
        ("FIX.4.4", "747", "M"): "M .. Manual",
        ("FIX.4.4", "749", "0"): "0 .. Successful",
        ("FIX.4.4", "749", "1"): "1 .. InvalidOrUnknownInstrument",
        ("FIX.4.4", "749", "2"): "2 .. InvalidTypeOfTradeRequested",
        ("FIX.4.4", "749", "3"): "3 .. InvalidParties",
        ("FIX.4.4", "749", "4"): "4 .. InvalidTransportTypeRequested",
        ("FIX.4.4", "749", "5"): "5 .. InvalidDestinationRequested",
        ("FIX.4.4", "749", "8"): "8 .. TradeRequestTypeNotSupported",
        ("FIX.4.4", "749", "9"): "9 .. NotAuthorized",
        ("FIX.4.4", "749", "99"): "99 .. Other",
        ("FIX.4.4", "750", "0"): "0 .. Accepted",
        ("FIX.4.4", "750", "1"): "1 .. Completed",
        ("FIX.4.4", "750", "2"): "2 .. Rejected",
        ("FIX.4.4", "751", "0"): "0 .. Successful",
        ("FIX.4.4", "751", "1"): "1 .. InvalidPartyOnformation",
        ("FIX.4.4", "751", "2"): "2 .. UnknownInstrument",
        ("FIX.4.4", "751", "3"): "3 .. UnauthorizedToReportTrades",
        ("FIX.4.4", "751", "4"): "4 .. InvalidTradeType",
        ("FIX.4.4", "751", "99"): "99 .. Other",
        ("FIX.4.4", "752", "1"): "1 .. SingleSecurity",
        ("FIX.4.4", "752", "2"): "2 .. IndividualLegOfAMultilegSecurity",
        ("FIX.4.4", "752", "3"): "3 .. MultilegSecurity",
        ("FIX.4.4", "770", "1"): "1 .. ExecutionTime",
        ("FIX.4.4", "770", "2"): "2 .. TimeIn",
        ("FIX.4.4", "770", "3"): "3 .. TimeOut",
        ("FIX.4.4", "770", "4"): "4 .. BrokerReceipt",
        ("FIX.4.4", "770", "5"): "5 .. BrokerExecution",
        ("FIX.4.4", "773", "1"): "1 .. Status",
        ("FIX.4.4", "773", "2"): "2 .. Confirmation",
        ("FIX.4.4", "773", "3"): "3 .. ConfirmationRequestRejected",
        ("FIX.4.4", "774", "1"): "1 .. MismatchedAccount",
        ("FIX.4.4", "774", "2"): "2 .. MissingSettlementInstructions",
        ("FIX.4.4", "774", "99"): "99 .. Other",
        ("FIX.4.4", "775", "0"): "0 .. RegularBooking",
        ("FIX.4.4", "775", "1"): "1 .. CFD",
        ("FIX.4.4", "775", "2"): "2 .. TotalReturnSwap",
        ("FIX.4.4", "780", "0"): "0 .. UseDefaultInstructions",
        ("FIX.4.4", "780", "1"): "1 .. DeriveFromParametersProvided",
        ("FIX.4.4", "780", "2"): "2 .. FullDetailsProvided",
        ("FIX.4.4", "780", "3"): "3 .. SSIDBIDsProvided",
        ("FIX.4.4", "780", "4"): "4 .. PhoneForInstructions",
        ("FIX.4.4", "787", "S"): "S .. Securities",
        ("FIX.4.4", "787", "C"): "C .. Cash",
        ("FIX.4.4", "788", "1"): "1 .. Overnight",
        ("FIX.4.4", "788", "2"): "2 .. Term",
        ("FIX.4.4", "788", "3"): "3 .. Flexible",
        ("FIX.4.4", "788", "4"): "4 .. Open",
        ("FIX.4.4", "792", "0"): "0 .. UnableToProcessRequest",
        ("FIX.4.4", "792", "1"): "1 .. UnknownAccount",
        ("FIX.4.4", "792", "2"): "2 .. NoMatchingSettlementInstructionsFound",
        ("FIX.4.4", "792", "99"): "99 .. Other",
        ("FIX.4.4", "794", "3"): "3 .. SellsideCalculatedUsingPreliminary",
        ("FIX.4.4", "794", "4"): "4 .. SellsideCalculatedWithoutPreliminary",
        ("FIX.4.4", "794", "5"): "5 .. WarehouseRecap",
        ("FIX.4.4", "794", "8"): "8 .. RequestToIntermediary",
        ("FIX.4.4", "796", "1"): "1 .. OriginalDetailsIncomplete",
        ("FIX.4.4", "796", "2"): "2 .. ChangeInUnderlyingOrderDetails",
        ("FIX.4.4", "796", "99"): "99 .. Other",
        ("FIX.4.4", "798", "1"): "1 .. CarriedCustomerSide",
        ("FIX.4.4", "798", "2"): "2 .. CarriedNonCustomerSide",
        ("FIX.4.4", "798", "3"): "3 .. HouseTrader",
        ("FIX.4.4", "798", "4"): "4 .. FloorTrader",
        ("FIX.4.4", "798", "6"): "6 .. CarriedNonCustomerSideCrossMargined",
        ("FIX.4.4", "798", "7"): "7 .. HouseTraderCrossMargined",
        ("FIX.4.4", "798", "8"): "8 .. JointBackOfficeAccount",
        ("FIX.4.4", "803", "1"): "1 .. Firm",
        ("FIX.4.4", "803", "2"): "2 .. Person",
        ("FIX.4.4", "803", "3"): "3 .. System",
        ("FIX.4.4", "803", "4"): "4 .. Application",
        ("FIX.4.4", "803", "5"): "5 .. FullLegalNameOfFirm",
        ("FIX.4.4", "803", "6"): "6 .. PostalAddress",
        ("FIX.4.4", "803", "7"): "7 .. PhoneNumber",
        ("FIX.4.4", "803", "8"): "8 .. EmailAddress",
        ("FIX.4.4", "803", "9"): "9 .. ContactName",
        ("FIX.4.4", "803", "10"): "10 .. SecuritiesAccountNumber",
        ("FIX.4.4", "803", "11"): "11 .. RegistrationNumber",
        ("FIX.4.4", "803", "12"): "12 .. RegisteredAddressForConfirmation",
        ("FIX.4.4", "803", "13"): "13 .. RegulatoryStatus",
        ("FIX.4.4", "803", "14"): "14 .. RegistrationName",
        ("FIX.4.4", "803", "15"): "15 .. CashAccountNumber",
        ("FIX.4.4", "803", "16"): "16 .. BIC",
        ("FIX.4.4", "803", "17"): "17 .. CSDParticipantMemberCode",
        ("FIX.4.4", "803", "18"): "18 .. RegisteredAddress",
        ("FIX.4.4", "803", "19"): "19 .. FundAccountName",
        ("FIX.4.4", "803", "20"): "20 .. TelexNumber",
        ("FIX.4.4", "803", "21"): "21 .. FaxNumber",
        ("FIX.4.4", "803", "22"): "22 .. SecuritiesAccountName",
        ("FIX.4.4", "803", "23"): "23 .. CashAccountName",
        ("FIX.4.4", "803", "24"): "24 .. Department",
        ("FIX.4.4", "803", "25"): "25 .. LocationDesk",
        ("FIX.4.4", "803", "26"): "26 .. PositionAccountType",
        ("FIX.4.4", "808", "1"): "1 .. PendingAccept",
        ("FIX.4.4", "808", "2"): "2 .. PendingRelease",
        ("FIX.4.4", "808", "3"): "3 .. PendingReversal",
        ("FIX.4.4", "808", "4"): "4 .. Accept",
        ("FIX.4.4", "808", "5"): "5 .. BlockLevelReject",
        ("FIX.4.4", "808", "6"): "6 .. AccountLevelReject",
        ("FIX.4.4", "814", "0"): "0 .. NoActionTaken",
        ("FIX.4.4", "814", "1"): "1 .. QueueFlushed",
        ("FIX.4.4", "814", "2"): "2 .. OverlayLast",
        ("FIX.4.4", "814", "3"): "3 .. EndSession",
        ("FIX.4.4", "815", "0"): "0 .. NoActionTaken",
        ("FIX.4.4", "815", "1"): "1 .. QueueFlushed",
        ("FIX.4.4", "815", "2"): "2 .. OverlayLast",
        ("FIX.4.4", "815", "3"): "3 .. EndSession",
        ("FIX.4.4", "819", "0"): "0 .. NoAveragePricing",
        ("FIX.4.4", "819", "1"): "1 .. Trade",
        ("FIX.4.4", "819", "2"): "2 .. LastTrade",
        ("FIX.4.4", "826", "0"): "0 .. AllocationNotRequired",
        ("FIX.4.4", "826", "1"): "1 .. AllocationRequired",
        ("FIX.4.4", "826", "2"): "2 .. UseAllocationProvidedWithTheTrade",
        ("FIX.4.4", "827", "0"): "0 .. ExpireOnTradingSessionClose",
        ("FIX.4.4", "827", "1"): "1 .. ExpireOnTradingSessionOpen",
        ("FIX.4.4", "828", "0"): "0 .. RegularTrade",
        ("FIX.4.4", "828", "1"): "1 .. BlockTrade",
        ("FIX.4.4", "828", "2"): "2 .. EFP",
        ("FIX.4.4", "828", "3"): "3 .. Transfer",
        ("FIX.4.4", "828", "4"): "4 .. LateTrade",
        ("FIX.4.4", "828", "5"): "5 .. TTrade",
        ("FIX.4.4", "828", "6"): "6 .. WeightedAveragePriceTrade",
        ("FIX.4.4", "828", "7"): "7 .. BunchedTrade",
        ("FIX.4.4", "828", "8"): "8 .. LateBunchedTrade",
        ("FIX.4.4", "828", "9"): "9 .. PriorReferencePriceTrade",
        ("FIX.4.4", "828", "10"): "10 .. AfterHoursTrade",
        ("FIX.4.4", "835", "0"): "0 .. Floating",
        ("FIX.4.4", "835", "1"): "1 .. Fixed",
        ("FIX.4.4", "836", "0"): "0 .. Price",
        ("FIX.4.4", "836", "1"): "1 .. BasisPoints",
        ("FIX.4.4", "836", "2"): "2 .. Ticks",
        ("FIX.4.4", "836", "3"): "3 .. PriceTier",
        ("FIX.4.4", "837", "0"): "0 .. OrBetter",
        ("FIX.4.4", "837", "1"): "1 .. Strict",
        ("FIX.4.4", "837", "2"): "2 .. OrWorse",
        ("FIX.4.4", "838", "1"): "1 .. MoreAggressive",
        ("FIX.4.4", "838", "2"): "2 .. MorePassive",
        ("FIX.4.4", "840", "1"): "1 .. Local",
        ("FIX.4.4", "840", "2"): "2 .. National",
        ("FIX.4.4", "840", "3"): "3 .. Global",
        ("FIX.4.4", "840", "4"): "4 .. NationalExcludingLocal",
        ("FIX.4.4", "841", "0"): "0 .. Floating",
        ("FIX.4.4", "841", "1"): "1 .. Fixed",
        ("FIX.4.4", "842", "0"): "0 .. Price",
        ("FIX.4.4", "842", "1"): "1 .. BasisPoints",
        ("FIX.4.4", "842", "2"): "2 .. Ticks",
        ("FIX.4.4", "842", "3"): "3 .. PriceTier",
        ("FIX.4.4", "843", "0"): "0 .. OrBetter",
        ("FIX.4.4", "843", "1"): "1 .. Strict",
        ("FIX.4.4", "843", "2"): "2 .. OrWorse",
        ("FIX.4.4", "844", "1"): "1 .. MoreAggressive",
        ("FIX.4.4", "844", "2"): "2 .. MorePassive",
        ("FIX.4.4", "846", "1"): "1 .. Local",
        ("FIX.4.4", "846", "2"): "2 .. National",
        ("FIX.4.4", "846", "3"): "3 .. Global",
        ("FIX.4.4", "846", "4"): "4 .. NationalExcludingLocal",
        ("FIX.4.4", "847", "1"): "1 .. VWAP",
        ("FIX.4.4", "847", "2"): "2 .. Participate",
        ("FIX.4.4", "847", "3"): "3 .. MininizeMarketImpact",
        ("FIX.4.4", "851", "1"): "1 .. AddedLiquidity",
        ("FIX.4.4", "851", "2"): "2 .. RemovedLiquidity",
        ("FIX.4.4", "851", "3"): "3 .. LiquidityRoutedOut",
        ("FIX.4.4", "852", "Y"): "Y .. ReportTrade",
        ("FIX.4.4", "852", "N"): "N .. DoNotReportTrade",
        ("FIX.4.4", "853", "0"): "0 .. DealerSoldShort",
        ("FIX.4.4", "853", "1"): "1 .. DealerSoldShortExempt",
        ("FIX.4.4", "853", "2"): "2 .. SellingCustomerSoldShort",
        ("FIX.4.4", "853", "3"): "3 .. SellingCustomerSoldShortExempt",
        ("FIX.4.4", "853", "4"): "4 .. QualifiedServiceRepresentative",
        ("FIX.4.4", "853", "5"): "5 .. QSROrAGUContraSideSoldShortExempt",
        ("FIX.4.4", "854", "0"): "0 .. Units",
        ("FIX.4.4", "854", "1"): "1 .. Contracts",
        ("FIX.4.4", "856", "0"): "0 .. Submit",
        ("FIX.4.4", "856", "1"): "1 .. Alleged",
        ("FIX.4.4", "856", "2"): "2 .. Accept",
        ("FIX.4.4", "856", "3"): "3 .. Decline",
        ("FIX.4.4", "856", "4"): "4 .. Addendum",
        ("FIX.4.4", "856", "5"): "5 .. No",
        ("FIX.4.4", "856", "6"): "6 .. TradeReportCancel",
        ("FIX.4.4", "856", "7"): "7 .. LockedIn",
        ("FIX.4.4", "857", "0"): "0 .. NotSpecified",
        ("FIX.4.4", "857", "1"): "1 .. ExplicitListProvided",
        ("FIX.4.4", "865", "1"): "1 .. Put",
        ("FIX.4.4", "865", "2"): "2 .. Call",
        ("FIX.4.4", "865", "3"): "3 .. Tender",
        ("FIX.4.4", "865", "4"): "4 .. SinkingFundCall",
        ("FIX.4.4", "865", "99"): "99 .. Other",
        ("FIX.4.4", "871", "1"): "1 .. Flat",
        ("FIX.4.4", "871", "2"): "2 .. ZeroCoupon",
        ("FIX.4.4", "871", "3"): "3 .. InterestBearing",
        ("FIX.4.4", "871", "4"): "4 .. NoPeriodicPayments",
        ("FIX.4.4", "871", "5"): "5 .. VariableRate",
        ("FIX.4.4", "871", "6"): "6 .. LessFeeForPut",
        ("FIX.4.4", "871", "7"): "7 .. SteppedCoupon",
        ("FIX.4.4", "871", "8"): "8 .. CouponPeriod",
        ("FIX.4.4", "871", "9"): "9 .. When",
        ("FIX.4.4", "871", "10"): "10 .. OriginalIssueDiscount",
        ("FIX.4.4", "871", "11"): "11 .. Callable",
        ("FIX.4.4", "871", "12"): "12 .. EscrowedToMaturity",
        ("FIX.4.4", "871", "13"): "13 .. EscrowedToRedemptionDate",
        ("FIX.4.4", "871", "14"): "14 .. PreRefunded",
        ("FIX.4.4", "871", "15"): "15 .. InDefault",
        ("FIX.4.4", "871", "16"): "16 .. Unrated",
        ("FIX.4.4", "871", "17"): "17 .. Taxable",
        ("FIX.4.4", "871", "18"): "18 .. Indexed",
        ("FIX.4.4", "871", "19"): "19 .. SubjectToAlternativeMinimumTax",
        ("FIX.4.4", "871", "20"): "20 .. OriginalIssueDiscountPrice",
        ("FIX.4.4", "871", "21"): "21 .. CallableBelowMaturityValue",
        ("FIX.4.4", "871", "22"): "22 .. CallableWithoutNotice",
        ("FIX.4.4", "871", "99"): "99 .. Text",
        ("FIX.4.4", "875", "1"): "1 .. Program3a3",
        ("FIX.4.4", "875", "2"): "2 .. Program42",
        ("FIX.4.4", "875", "99"): "99 .. Other",
        ("FIX.4.4", "891", "0"): "0 .. Absolute",
        ("FIX.4.4", "891", "1"): "1 .. PerUnit",
        ("FIX.4.4", "891", "2"): "2 .. Percentage",
        ("FIX.4.4", "893", "Y"): "Y .. LastMessage",
        ("FIX.4.4", "893", "N"): "N .. NotLastMessage",
        ("FIX.4.4", "895", "0"): "0 .. Initial",
        ("FIX.4.4", "895", "1"): "1 .. Scheduled",
        ("FIX.4.4", "895", "2"): "2 .. TimeWarning",
        ("FIX.4.4", "895", "3"): "3 .. MarginDeficiency",
        ("FIX.4.4", "895", "4"): "4 .. MarginExcess",
        ("FIX.4.4", "895", "5"): "5 .. ForwardCollateralDemand",
        ("FIX.4.4", "895", "6"): "6 .. EventOfDefault",
        ("FIX.4.4", "895", "7"): "7 .. AdverseTaxEvent",
        ("FIX.4.4", "896", "0"): "0 .. TradeDate",
        ("FIX.4.4", "896", "1"): "1 .. GCInstrument",
        ("FIX.4.4", "896", "2"): "2 .. CollateralInstrument",
        ("FIX.4.4", "896", "3"): "3 .. SubstitutionEligible",
        ("FIX.4.4", "896", "4"): "4 .. NotAssigned",
        ("FIX.4.4", "896", "5"): "5 .. PartiallyAssigned",
        ("FIX.4.4", "896", "6"): "6 .. FullyAssigned",
        ("FIX.4.4", "896", "7"): "7 .. OutstandingTrades",
        ("FIX.4.4", "903", "0"): "0 .. New",
        ("FIX.4.4", "903", "1"): "1 .. Replace",
        ("FIX.4.4", "903", "2"): "2 .. Cancel",
        ("FIX.4.4", "903", "3"): "3 .. Release",
        ("FIX.4.4", "903", "4"): "4 .. Reverse",
        ("FIX.4.4", "905", "0"): "0 .. Received",
        ("FIX.4.4", "905", "1"): "1 .. Accepted",
        ("FIX.4.4", "905", "2"): "2 .. Declined",
        ("FIX.4.4", "905", "3"): "3 .. Rejected",
        ("FIX.4.4", "906", "0"): "0 .. UnknownDeal",
        ("FIX.4.4", "906", "1"): "1 .. UnknownOrInvalidInstrument",
        ("FIX.4.4", "906", "2"): "2 .. UnauthorizedTransaction",
        ("FIX.4.4", "906", "3"): "3 .. InsufficientCollateral",
        ("FIX.4.4", "906", "4"): "4 .. InvalidTypeOfCollateral",
        ("FIX.4.4", "906", "5"): "5 .. ExcessiveSubstitution",
        ("FIX.4.4", "906", "99"): "99 .. Other",
        ("FIX.4.4", "910", "0"): "0 .. Unassigned",
        ("FIX.4.4", "910", "1"): "1 .. PartiallyAssigned",
        ("FIX.4.4", "910", "2"): "2 .. AssignmentProposed",
        ("FIX.4.4", "910", "3"): "3 .. Assigned",
        ("FIX.4.4", "910", "4"): "4 .. Challenged",
        ("FIX.4.4", "919", "0"): "0 .. VersusPayment",
        ("FIX.4.4", "919", "1"): "1 .. Free",
        ("FIX.4.4", "919", "2"): "2 .. TriParty",
        ("FIX.4.4", "919", "3"): "3 .. HoldInCustody",
        ("FIX.4.4", "924", "1"): "1 .. LogOnUser",
        ("FIX.4.4", "924", "2"): "2 .. LogOffUser",
        ("FIX.4.4", "924", "3"): "3 .. ChangePasswordForUser",
        ("FIX.4.4", "924", "4"): "4 .. RequestIndividualUserStatus",
        ("FIX.4.4", "926", "1"): "1 .. LoggedIn",
        ("FIX.4.4", "926", "2"): "2 .. NotLoggedIn",
        ("FIX.4.4", "926", "3"): "3 .. UserNotRecognised",
        ("FIX.4.4", "926", "4"): "4 .. PasswordIncorrect",
        ("FIX.4.4", "926", "5"): "5 .. PasswordChanged",
        ("FIX.4.4", "926", "6"): "6 .. Other",
        ("FIX.4.4", "928", "1"): "1 .. Connected",
        ("FIX.4.4", "928", "2"): "2 .. NotConnectedUnexpected",
        ("FIX.4.4", "928", "3"): "3 .. NotConnectedExpected",
        ("FIX.4.4", "928", "4"): "4 .. InProcess",
        ("FIX.4.4", "935", "1"): "1 .. Snapshot",
        ("FIX.4.4", "935", "2"): "2 .. Subscribe",
        ("FIX.4.4", "935", "4"): "4 .. StopSubscribing",
        ("FIX.4.4", "935", "8"): "8 .. LevelOfDetail",
        ("FIX.4.4", "937", "1"): "1 .. Full",
        ("FIX.4.4", "937", "2"): "2 .. IncrementalUpdate",
        ("FIX.4.4", "939", "0"): "0 .. Accepted",
        ("FIX.4.4", "939", "1"): "1 .. Rejected",
        ("FIX.4.4", "940", "1"): "1 .. Received",
        ("FIX.4.4", "940", "2"): "2 .. ConfirmRejected",
        ("FIX.4.4", "940", "3"): "3 .. Affirmed",
        ("FIX.4.4", "944", "0"): "0 .. Retain",
        ("FIX.4.4", "944", "1"): "1 .. Add",
        ("FIX.4.4", "944", "2"): "2 .. Remove",
        ("FIX.4.4", "945", "0"): "0 .. Accepted",
        ("FIX.4.4", "945", "1"): "1 .. AcceptedWithWarnings",
        ("FIX.4.4", "945", "2"): "2 .. Completed",
        ("FIX.4.4", "945", "3"): "3 .. CompletedWithWarnings",
        ("FIX.4.4", "945", "4"): "4 .. Rejected",
        ("FIX.4.4", "946", "0"): "0 .. Successful",
        ("FIX.4.4", "946", "1"): "1 .. InvalidOrUnknownInstrument",
        ("FIX.4.4", "946", "2"): "2 .. InvalidOrUnknownCollateralType",
        ("FIX.4.4", "946", "3"): "3 .. InvalidParties",
        ("FIX.4.4", "946", "4"): "4 .. InvalidTransportTypeRequested",
        ("FIX.4.4", "946", "5"): "5 .. InvalidDestinationRequested",
        ("FIX.4.4", "946", "6"): "6 .. NoCollateralFoundForTheTradeSpecified",
        ("FIX.4.4", "946", "7"): "7 .. NoCollateralFoundForTheOrderSpecified",
        ("FIX.4.4", "946", "8"): "8 .. CollateralInquiryTypeNotSupported",
        ("FIX.4.4", "946", "9"): "9 .. UnauthorizedForCollateralInquiry",
        ("FIX.4.4", "946", "99"): "99 .. Other",
        ("FIX.5.0", "4", "B"): "B .. Buy",
        ("FIX.5.0", "4", "S"): "S .. Sell",
        ("FIX.5.0", "4", "T"): "T .. Trade",
        ("FIX.5.0", "4", "X"): "X .. Cross",
        ("FIX.5.0", "5", "N"): "N .. New",
        ("FIX.5.0", "5", "C"): "C .. Cancel",
        ("FIX.5.0", "5", "R"): "R .. Replace",
        ("FIX.5.0", "13", "1"): "1 .. PerUnit",
        ("FIX.5.0", "13", "2"): "2 .. Percent",
        ("FIX.5.0", "13", "3"): "3 .. Absolute",
        ("FIX.5.0", "13", "4"): "4 .. PercentageWaivedCashDiscount",
        ("FIX.5.0", "13", "5"): "5 .. PercentageWaivedEnhancedUnits",
        ("FIX.5.0", "13", "6"): "6 .. PointsPerBondOrContract",
        ("FIX.5.0", "13", "7"): "7 .. BasisPoints",
        ("FIX.5.0", "13", "8"): "8 .. AmountPerContract",
        ("FIX.5.0", "18", "0"): "0 .. StayOnOfferSide",
        ("FIX.5.0", "18", "1"): "1 .. NotHeld",
        ("FIX.5.0", "18", "2"): "2 .. Work",
        ("FIX.5.0", "18", "3"): "3 .. GoAlong",
        ("FIX.5.0", "18", "4"): "4 .. OverTheDay",
        ("FIX.5.0", "18", "5"): "5 .. Held",
        ("FIX.5.0", "18", "6"): "6 .. ParticipateDoNotInitiate",
        ("FIX.5.0", "18", "7"): "7 .. StrictScale",
        ("FIX.5.0", "18", "8"): "8 .. TryToScale",
        ("FIX.5.0", "18", "9"): "9 .. StayOnBidSide",
        ("FIX.5.0", "18", "A"): "A .. NoCross",
        ("FIX.5.0", "18", "B"): "B .. OKToCross",
        ("FIX.5.0", "18", "C"): "C .. CallFirst",
        ("FIX.5.0", "18", "D"): "D .. PercentOfVolume",
        ("FIX.5.0", "18", "E"): "E .. DoNotIncrease",
        ("FIX.5.0", "18", "F"): "F .. DoNotReduce",
        ("FIX.5.0", "18", "G"): "G .. AllOrNone",
        ("FIX.5.0", "18", "H"): "H .. ReinstateOnSystemFailure",
        ("FIX.5.0", "18", "I"): "I .. InstitutionsOnly",
        ("FIX.5.0", "18", "J"): "J .. ReinstateOnTradingHalt",
        ("FIX.5.0", "18", "K"): "K .. CancelOnTradingHalt",
        ("FIX.5.0", "18", "N"): "N .. NonNegotiable",
        ("FIX.5.0", "18", "Q"): "Q .. CancelOnSystemFailure",
        ("FIX.5.0", "18", "S"): "S .. Suspend",
        ("FIX.5.0", "18", "U"): "U .. CustomerDisplayInstruction",
        ("FIX.5.0", "18", "V"): "V .. Netting",
        ("FIX.5.0", "18", "X"): "X .. TradeAlong",
        ("FIX.5.0", "18", "Y"): "Y .. TryToStop",
        ("FIX.5.0", "18", "Z"): "Z .. CancelIfNotBest",
        ("FIX.5.0", "18", "b"): "b .. StrictLimit",
        ("FIX.5.0", "18", "c"): "c .. IgnorePriceValidityChecks",
        ("FIX.5.0", "18", "e"): "e .. WorkToTargetStrategy",
        ("FIX.5.0", "18", "f"): "f .. IntermarketSweep",
        ("FIX.5.0", "18", "g"): "g .. ExternalRoutingAllowed",
        ("FIX.5.0", "18", "h"): "h .. ExternalRoutingNotAllowed",
        ("FIX.5.0", "18", "i"): "i .. ImbalanceOnly",
        ("FIX.5.0", "18", "j"): "j .. SingleExecutionRequestedForBlockTrade",
        ("FIX.5.0", "18", "k"): "k .. BestExecution",
        ("FIX.5.0", "18", "l"): "l .. SuspendOnSystemFailure",
        ("FIX.5.0", "18", "m"): "m .. SuspendOnTradingHalt",
        ("FIX.5.0", "18", "n"): "n .. ReinstateOnConnectionLoss",
        ("FIX.5.0", "18", "o"): "o .. CancelOnConnectionLoss",
        ("FIX.5.0", "18", "p"): "p .. SuspendOnConnectionLoss",
        ("FIX.5.0", "18", "q"): "q .. Release",
        ("FIX.5.0", "18", "r"): "r .. ExecuteAsDeltaNeutral",
        ("FIX.5.0", "18", "s"): "s .. ExecuteAsDurationNeutral",
        ("FIX.5.0", "18", "t"): "t .. ExecuteAsFXNeutral",
        ("FIX.5.0", "18", "u"): "u .. MinGuaranteedFillEligible",
        ("FIX.5.0", "18", "v"): "v .. BypassNonDisplayLiquidity",
        ("FIX.5.0", "18", "w"): "w .. Lock",
        ("FIX.5.0", "18", "x"): "x .. IgnoreNotionalValueChecks",
        ("FIX.5.0", "18", "y"): "y .. TrdAtRefPx",
        ("FIX.5.0", "21", "1"): "1 .. AutomatedExecutionNoIntervention",
        ("FIX.5.0", "21", "2"): "2 .. AutomatedExecutionInterventionOK",
        ("FIX.5.0", "21", "3"): "3 .. ManualOrder",
        ("FIX.5.0", "22", "1"): "1 .. CUSIP",
        ("FIX.5.0", "22", "2"): "2 .. SEDOL",
        ("FIX.5.0", "22", "3"): "3 .. QUIK",
        ("FIX.5.0", "22", "4"): "4 .. ISINNumber",
        ("FIX.5.0", "22", "5"): "5 .. RICCode",
        ("FIX.5.0", "22", "6"): "6 .. ISOCurrencyCode",
        ("FIX.5.0", "22", "7"): "7 .. ISOCountryCode",
        ("FIX.5.0", "22", "8"): "8 .. ExchangeSymbol",
        ("FIX.5.0", "22", "9"): "9 .. ConsolidatedTapeAssociation",
        ("FIX.5.0", "22", "A"): "A .. BloombergSymbol",
        ("FIX.5.0", "22", "B"): "B .. Wertpapier",
        ("FIX.5.0", "22", "C"): "C .. Dutch",
        ("FIX.5.0", "22", "D"): "D .. Valoren",
        ("FIX.5.0", "22", "E"): "E .. Sicovam",
        ("FIX.5.0", "22", "F"): "F .. Belgian",
        ("FIX.5.0", "22", "G"): "G .. Common",
        ("FIX.5.0", "22", "H"): "H .. ClearingHouse",
        ("FIX.5.0", "22", "I"): "I .. ISDAFpMLSpecification",
        ("FIX.5.0", "22", "J"): "J .. OptionPriceReportingAuthority",
        ("FIX.5.0", "22", "K"): "K .. ISDAFpMLURL",
        ("FIX.5.0", "22", "L"): "L .. LetterOfCredit",
        ("FIX.5.0", "22", "M"): "M .. MarketplaceAssignedIdentifier",
        ("FIX.5.0", "22", "N"): "N .. MarkitREDEntityCLIP",
        ("FIX.5.0", "22", "P"): "P .. MarkitREDPairCLIP",
        ("FIX.5.0", "22", "Q"): "Q .. CFTCCommodityCode",
        ("FIX.5.0", "22", "R"): "R .. ISDACommodityReferencePrice",
        ("FIX.5.0", "22", "S"): "S .. FinancialInstrumentGlobalIdentifier",
        ("FIX.5.0", "22", "T"): "T .. LegalEntityIdentifier",
        ("FIX.5.0", "22", "U"): "U .. Synthetic",
        ("FIX.5.0", "22", "V"): "V .. FidessaInstrumentMnemonic",
        ("FIX.5.0", "22", "W"): "W .. IndexName",
        ("FIX.5.0", "25", "H"): "H .. High",
        ("FIX.5.0", "25", "L"): "L .. Low",
        ("FIX.5.0", "25", "M"): "M .. Medium",
        ("FIX.5.0", "27", "S"): "S .. Small",
        ("FIX.5.0", "27", "M"): "M .. Medium",
        ("FIX.5.0", "27", "L"): "L .. Large",
        ("FIX.5.0", "27", "U"): "U .. UndisclosedQuantity",
        ("FIX.5.0", "28", "N"): "N .. New",
        ("FIX.5.0", "28", "C"): "C .. Cancel",
        ("FIX.5.0", "28", "R"): "R .. Replace",
        ("FIX.5.0", "29", "1"): "1 .. Agent",
        ("FIX.5.0", "29", "2"): "2 .. CrossAsAgent",
        ("FIX.5.0", "29", "3"): "3 .. CrossAsPrincipal",
        ("FIX.5.0", "29", "4"): "4 .. Principal",
        ("FIX.5.0", "29", "5"): "5 .. RisklessPrincipal",
        ("FIX.5.0", "35", "0"): "0 .. Heartbeat",
        ("FIX.5.0", "35", "1"): "1 .. TestRequest",
        ("FIX.5.0", "35", "2"): "2 .. ResendRequest",
        ("FIX.5.0", "35", "3"): "3 .. Reject",
        ("FIX.5.0", "35", "4"): "4 .. SequenceReset",
        ("FIX.5.0", "35", "5"): "5 .. Logout",
        ("FIX.5.0", "35", "6"): "6 .. IOI",
        ("FIX.5.0", "35", "7"): "7 .. Advertisement",
        ("FIX.5.0", "35", "8"): "8 .. ExecutionReport",
        ("FIX.5.0", "35", "9"): "9 .. OrderCancelReject",
        ("FIX.5.0", "35", "A"): "A .. Logon",
        ("FIX.5.0", "35", "B"): "B .. News",
        ("FIX.5.0", "35", "C"): "C .. Email",
        ("FIX.5.0", "35", "D"): "D .. NewOrderSingle",
        ("FIX.5.0", "35", "E"): "E .. NewOrderList",
        ("FIX.5.0", "35", "F"): "F .. OrderCancelRequest",
        ("FIX.5.0", "35", "G"): "G .. OrderCancelReplaceRequest",
        ("FIX.5.0", "35", "H"): "H .. OrderStatusRequest",
        ("FIX.5.0", "35", "J"): "J .. AllocationInstruction",
        ("FIX.5.0", "35", "K"): "K .. ListCancelRequest",
        ("FIX.5.0", "35", "L"): "L .. ListExecute",
        ("FIX.5.0", "35", "M"): "M .. ListStatusRequest",
        ("FIX.5.0", "35", "N"): "N .. ListStatus",
        ("FIX.5.0", "35", "P"): "P .. AllocationInstructionAck",
        ("FIX.5.0", "35", "Q"): "Q .. DontKnowTrade",
        ("FIX.5.0", "35", "R"): "R .. QuoteRequest",
        ("FIX.5.0", "35", "S"): "S .. Quote",
        ("FIX.5.0", "35", "T"): "T .. SettlementInstructions",
        ("FIX.5.0", "35", "V"): "V .. MarketDataRequest",
        ("FIX.5.0", "35", "W"): "W .. MarketDataSnapshotFullRefresh",
        ("FIX.5.0", "35", "X"): "X .. MarketDataIncrementalRefresh",
        ("FIX.5.0", "35", "Y"): "Y .. MarketDataRequestReject",
        ("FIX.5.0", "35", "Z"): "Z .. QuoteCancel",
        ("FIX.5.0", "35", "a"): "a .. QuoteStatusRequest",
        ("FIX.5.0", "35", "b"): "b .. MassQuoteAck",
        ("FIX.5.0", "35", "c"): "c .. SecurityDefinitionRequest",
        ("FIX.5.0", "35", "d"): "d .. SecurityDefinition",
        ("FIX.5.0", "35", "e"): "e .. SecurityStatusRequest",
        ("FIX.5.0", "35", "f"): "f .. SecurityStatus",
        ("FIX.5.0", "35", "g"): "g .. TradingSessionStatusRequest",
        ("FIX.5.0", "35", "h"): "h .. TradingSessionStatus",
        ("FIX.5.0", "35", "i"): "i .. MassQuote",
        ("FIX.5.0", "35", "j"): "j .. BusinessMessageReject",
        ("FIX.5.0", "35", "k"): "k .. BidRequest",
        ("FIX.5.0", "35", "l"): "l .. BidResponse",
        ("FIX.5.0", "35", "m"): "m .. ListStrikePrice",
        ("FIX.5.0", "35", "n"): "n .. XMLnonFIX",
        ("FIX.5.0", "35", "o"): "o .. RegistrationInstructions",
        ("FIX.5.0", "35", "p"): "p .. RegistrationInstructionsResponse",
        ("FIX.5.0", "35", "q"): "q .. OrderMassCancelRequest",
        ("FIX.5.0", "35", "r"): "r .. OrderMassCancelReport",
        ("FIX.5.0", "35", "s"): "s .. NewOrderCross",
        ("FIX.5.0", "35", "t"): "t .. CrossOrderCancelReplaceRequest",
        ("FIX.5.0", "35", "u"): "u .. CrossOrderCancelRequest",
        ("FIX.5.0", "35", "v"): "v .. SecurityTypeRequest",
        ("FIX.5.0", "35", "w"): "w .. SecurityTypes",
        ("FIX.5.0", "35", "x"): "x .. SecurityListRequest",
        ("FIX.5.0", "35", "y"): "y .. SecurityList",
        ("FIX.5.0", "35", "z"): "z .. DerivativeSecurityListRequest",
        ("FIX.5.0", "35", "AA"): "AA .. DerivativeSecurityList",
        ("FIX.5.0", "35", "AB"): "AB .. NewOrderMultileg",
        ("FIX.5.0", "35", "AC"): "AC .. MultilegOrderCancelReplace",
        ("FIX.5.0", "35", "AD"): "AD .. TradeCaptureReportRequest",
        ("FIX.5.0", "35", "AE"): "AE .. TradeCaptureReport",
        ("FIX.5.0", "35", "AF"): "AF .. OrderMassStatusRequest",
        ("FIX.5.0", "35", "AG"): "AG .. QuoteRequestReject",
        ("FIX.5.0", "35", "AH"): "AH .. RFQRequest",
        ("FIX.5.0", "35", "AI"): "AI .. QuoteStatusReport",
        ("FIX.5.0", "35", "AJ"): "AJ .. QuoteResponse",
        ("FIX.5.0", "35", "AK"): "AK .. Confirmation",
        ("FIX.5.0", "35", "AL"): "AL .. PositionMaintenanceRequest",
        ("FIX.5.0", "35", "AM"): "AM .. PositionMaintenanceReport",
        ("FIX.5.0", "35", "AN"): "AN .. RequestForPositions",
        ("FIX.5.0", "35", "AO"): "AO .. RequestForPositionsAck",
        ("FIX.5.0", "35", "AP"): "AP .. PositionReport",
        ("FIX.5.0", "35", "AQ"): "AQ .. TradeCaptureReportRequestAck",
        ("FIX.5.0", "35", "AR"): "AR .. TradeCaptureReportAck",
        ("FIX.5.0", "35", "AS"): "AS .. AllocationReport",
        ("FIX.5.0", "35", "AT"): "AT .. AllocationReportAck",
        ("FIX.5.0", "35", "AU"): "AU .. ConfirmationAck",
        ("FIX.5.0", "35", "AV"): "AV .. SettlementInstructionRequest",
        ("FIX.5.0", "35", "AW"): "AW .. AssignmentReport",
        ("FIX.5.0", "35", "AX"): "AX .. CollateralRequest",
        ("FIX.5.0", "35", "AY"): "AY .. CollateralAssignment",
        ("FIX.5.0", "35", "AZ"): "AZ .. CollateralResponse",
        ("FIX.5.0", "35", "BA"): "BA .. CollateralReport",
        ("FIX.5.0", "35", "BB"): "BB .. CollateralInquiry",
        ("FIX.5.0", "35", "BC"): "BC .. NetworkCounterpartySystemStatusRequest",
        ("FIX.5.0", "35", "BD"): "BD .. NetworkCounterpartySystemStatusResponse",
        ("FIX.5.0", "35", "BE"): "BE .. UserRequest",
        ("FIX.5.0", "35", "BF"): "BF .. UserResponse",
        ("FIX.5.0", "35", "BG"): "BG .. CollateralInquiryAck",
        ("FIX.5.0", "35", "BH"): "BH .. ConfirmationRequest",
        ("FIX.5.0", "35", "BO"): "BO .. ContraryIntentionReport",
        ("FIX.5.0", "35", "BP"): "BP .. SecurityDefinitionUpdateReport",
        ("FIX.5.0", "35", "BK"): "BK .. SecurityListUpdateReport",
        ("FIX.5.0", "35", "BL"): "BL .. AdjustedPositionReport",
        ("FIX.5.0", "35", "BM"): "BM .. AllocationInstructionAlert",
        ("FIX.5.0", "35", "BN"): "BN .. ExecutionAck",
        ("FIX.5.0", "35", "BJ"): "BJ .. TradingSessionList",
        ("FIX.5.0", "35", "BI"): "BI .. TradingSessionListRequest",
        ("FIX.5.0", "35", "BQ"): "BQ .. SettlementObligationReport",
        ("FIX.5.0", "35", "BR"): "BR .. DerivativeSecurityListUpdateReport",
        ("FIX.5.0", "35", "BS"): "BS .. TradingSessionListUpdateReport",
        ("FIX.5.0", "35", "BT"): "BT .. MarketDefinitionRequest",
        ("FIX.5.0", "35", "BU"): "BU .. MarketDefinition",
        ("FIX.5.0", "35", "BV"): "BV .. MarketDefinitionUpdateReport",
        ("FIX.5.0", "35", "BW"): "BW .. ApplicationMessageRequest",
        ("FIX.5.0", "35", "BX"): "BX .. ApplicationMessageRequestAck",
        ("FIX.5.0", "35", "BY"): "BY .. ApplicationMessageReport",
        ("FIX.5.0", "35", "BZ"): "BZ .. OrderMassActionReport",
        ("FIX.5.0", "35", "CA"): "CA .. OrderMassActionRequest",
        ("FIX.5.0", "35", "CB"): "CB .. UserNotification",
        ("FIX.5.0", "35", "CC"): "CC .. StreamAssignmentRequest",
        ("FIX.5.0", "35", "CD"): "CD .. StreamAssignmentReport",
        ("FIX.5.0", "35", "CE"): "CE .. StreamAssignmentReportACK",
        ("FIX.5.0", "35", "CF"): "CF .. PartyDetailsListRequest",
        ("FIX.5.0", "35", "CG"): "CG .. PartyDetailsListReport",
        ("FIX.5.0", "35", "CH"): "CH .. MarginRequirementInquiry",
        ("FIX.5.0", "35", "CI"): "CI .. MarginRequirementInquiryAck",
        ("FIX.5.0", "35", "CJ"): "CJ .. MarginRequirementReport",
        ("FIX.5.0", "35", "CK"): "CK .. PartyDetailsListUpdateReport",
        ("FIX.5.0", "35", "CL"): "CL .. PartyRiskLimitsRequest",
        ("FIX.5.0", "35", "CM"): "CM .. PartyRiskLimitsReport",
        ("FIX.5.0", "35", "CN"): "CN .. SecurityMassStatusRequest",
        ("FIX.5.0", "35", "CO"): "CO .. SecurityMassStatus",
        ("FIX.5.0", "35", "CQ"): "CQ .. AccountSummaryReport",
        ("FIX.5.0", "35", "CR"): "CR .. PartyRiskLimitsUpdateReport",
        ("FIX.5.0", "35", "CS"): "CS .. PartyRiskLimitsDefinitionRequest",
        ("FIX.5.0", "35", "CT"): "CT .. PartyRiskLimitsDefinitionRequestAck",
        ("FIX.5.0", "35", "CU"): "CU .. PartyEntitlementsRequest",
        ("FIX.5.0", "35", "CV"): "CV .. PartyEntitlementsReport",
        ("FIX.5.0", "35", "CW"): "CW .. QuoteAck",
        ("FIX.5.0", "35", "CX"): "CX .. PartyDetailsDefinitionRequest",
        ("FIX.5.0", "35", "CY"): "CY .. PartyDetailsDefinitionRequestAck",
        ("FIX.5.0", "35", "CZ"): "CZ .. PartyEntitlementsUpdateReport",
        ("FIX.5.0", "35", "DA"): "DA .. PartyEntitlementsDefinitionRequest",
        ("FIX.5.0", "35", "DB"): "DB .. PartyEntitlementsDefinitionRequestAck",
        ("FIX.5.0", "35", "DC"): "DC .. TradeMatchReport",
        ("FIX.5.0", "35", "DD"): "DD .. TradeMatchReportAck",
        ("FIX.5.0", "35", "DE"): "DE .. PartyRiskLimitsReportAck",
        ("FIX.5.0", "35", "DF"): "DF .. PartyRiskLimitCheckRequest",
        ("FIX.5.0", "35", "DG"): "DG .. PartyRiskLimitCheckRequestAck",
        ("FIX.5.0", "35", "DH"): "DH .. PartyActionRequest",
        ("FIX.5.0", "35", "DI"): "DI .. PartyActionReport",
        ("FIX.5.0", "35", "DJ"): "DJ .. MassOrder",
        ("FIX.5.0", "35", "DK"): "DK .. MassOrderAck",
        ("FIX.5.0", "35", "DL"): "DL .. PositionTransferInstruction",
        ("FIX.5.0", "35", "DM"): "DM .. PositionTransferInstructionAck",
        ("FIX.5.0", "35", "DN"): "DN .. PositionTransferReport",
        ("FIX.5.0", "35", "DO"): "DO .. MarketDataStatisticsRequest",
        ("FIX.5.0", "35", "DP"): "DP .. MarketDataStatisticsReport",
        ("FIX.5.0", "35", "DQ"): "DQ .. CollateralReportAck",
        ("FIX.5.0", "35", "DR"): "DR .. MarketDataReport",
        ("FIX.5.0", "35", "DS"): "DS .. CrossRequest",
        ("FIX.5.0", "35", "DT"): "DT .. CrossRequestAck",
        ("FIX.5.0", "35", "DU"): "DU .. AllocationInstructionAlertRequest",
        ("FIX.5.0", "39", "0"): "0 .. New",
        ("FIX.5.0", "39", "1"): "1 .. PartiallyFilled",
        ("FIX.5.0", "39", "2"): "2 .. Filled",
        ("FIX.5.0", "39", "3"): "3 .. DoneForDay",
        ("FIX.5.0", "39", "4"): "4 .. Canceled",
        ("FIX.5.0", "39", "6"): "6 .. PendingCancel",
        ("FIX.5.0", "39", "7"): "7 .. Stopped",
        ("FIX.5.0", "39", "8"): "8 .. Rejected",
        ("FIX.5.0", "39", "9"): "9 .. Suspended",
        ("FIX.5.0", "39", "A"): "A .. PendingNew",
        ("FIX.5.0", "39", "B"): "B .. Calculated",
        ("FIX.5.0", "39", "C"): "C .. Expired",
        ("FIX.5.0", "39", "D"): "D .. AcceptedForBidding",
        ("FIX.5.0", "39", "E"): "E .. PendingReplace",
        ("FIX.5.0", "40", "1"): "1 .. Market",
        ("FIX.5.0", "40", "2"): "2 .. Limit",
        ("FIX.5.0", "40", "3"): "3 .. Stop",
        ("FIX.5.0", "40", "4"): "4 .. StopLimit",
        ("FIX.5.0", "40", "6"): "6 .. WithOrWithout",
        ("FIX.5.0", "40", "8"): "8 .. LimitWithOrWithout",
        ("FIX.5.0", "40", "9"): "9 .. OnBasis",
        ("FIX.5.0", "40", "D"): "D .. PreviouslyQuoted",
        ("FIX.5.0", "40", "E"): "E .. PreviouslyIndicated",
        ("FIX.5.0", "40", "G"): "G .. ForexSwap",
        ("FIX.5.0", "40", "I"): "I .. Funari",
        ("FIX.5.0", "40", "J"): "J .. MarketIfTouched",
        ("FIX.5.0", "40", "K"): "K .. MarketWithLeftOverAsLimit",
        ("FIX.5.0", "40", "L"): "L .. PreviousFundValuationPoint",
        ("FIX.5.0", "40", "M"): "M .. NextFundValuationPoint",
        ("FIX.5.0", "40", "P"): "P .. Pegged",
        ("FIX.5.0", "40", "Q"): "Q .. CounterOrderSelection",
        ("FIX.5.0", "40", "R"): "R .. StopOnBidOrOffer",
        ("FIX.5.0", "40", "S"): "S .. StopLimitOnBidOrOffer",
        ("FIX.5.0", "43", "N"): "N .. OriginalTransmission",
        ("FIX.5.0", "43", "Y"): "Y .. PossibleDuplicate",
        ("FIX.5.0", "54", "1"): "1 .. Buy",
        ("FIX.5.0", "54", "2"): "2 .. Sell",
        ("FIX.5.0", "54", "3"): "3 .. BuyMinus",
        ("FIX.5.0", "54", "4"): "4 .. SellPlus",
        ("FIX.5.0", "54", "5"): "5 .. SellShort",
        ("FIX.5.0", "54", "6"): "6 .. SellShortExempt",
        ("FIX.5.0", "54", "7"): "7 .. Undisclosed",
        ("FIX.5.0", "54", "8"): "8 .. Cross",
        ("FIX.5.0", "54", "9"): "9 .. CrossShort",
        ("FIX.5.0", "54", "A"): "A .. CrossShortExempt",
        ("FIX.5.0", "54", "B"): "B .. AsDefined",
        ("FIX.5.0", "54", "C"): "C .. Opposite",
        ("FIX.5.0", "54", "D"): "D .. Subscribe",
        ("FIX.5.0", "54", "E"): "E .. Redeem",
        ("FIX.5.0", "54", "F"): "F .. Lend",
        ("FIX.5.0", "54", "G"): "G .. Borrow",
        ("FIX.5.0", "54", "H"): "H .. SellUndisclosed",
        ("FIX.5.0", "59", "0"): "0 .. Day",
        ("FIX.5.0", "59", "1"): "1 .. GoodTillCancel",
        ("FIX.5.0", "59", "2"): "2 .. AtTheOpening",
        ("FIX.5.0", "59", "3"): "3 .. ImmediateOrCancel",
        ("FIX.5.0", "59", "4"): "4 .. FillOrKill",
        ("FIX.5.0", "59", "5"): "5 .. GoodTillCrossing",
        ("FIX.5.0", "59", "6"): "6 .. GoodTillDate",
        ("FIX.5.0", "59", "7"): "7 .. AtTheClose",
        ("FIX.5.0", "59", "8"): "8 .. GoodThroughCrossing",
        ("FIX.5.0", "59", "9"): "9 .. AtCrossing",
        ("FIX.5.0", "59", "A"): "A .. GoodForTime",
        ("FIX.5.0", "59", "B"): "B .. GoodForAuction",
        ("FIX.5.0", "61", "0"): "0 .. Normal",
        ("FIX.5.0", "61", "1"): "1 .. Flash",
        ("FIX.5.0", "61", "2"): "2 .. Background",
        ("FIX.5.0", "63", "0"): "0 .. Regular",
        ("FIX.5.0", "63", "1"): "1 .. Cash",
        ("FIX.5.0", "63", "2"): "2 .. NextDay",
        ("FIX.5.0", "63", "3"): "3 .. TPlus2",
        ("FIX.5.0", "63", "4"): "4 .. TPlus3",
        ("FIX.5.0", "63", "5"): "5 .. TPlus4",
        ("FIX.5.0", "63", "6"): "6 .. Future",
        ("FIX.5.0", "63", "7"): "7 .. WhenAndIfIssued",
        ("FIX.5.0", "63", "8"): "8 .. SellersOption",
        ("FIX.5.0", "63", "9"): "9 .. TPlus5",
        ("FIX.5.0", "63", "B"): "B .. BrokenDate",
        ("FIX.5.0", "63", "C"): "C .. FXSpotNextSettlement",
        ("FIX.5.0", "65", "CD"): "CD .. EUCPWithLumpSumInterest",
        ("FIX.5.0", "65", "WI"): "WI .. WhenIssued",
        ("FIX.5.0", "71", "0"): "0 .. New",
        ("FIX.5.0", "71", "1"): "1 .. Replace",
        ("FIX.5.0", "71", "2"): "2 .. Cancel",
        ("FIX.5.0", "71", "6"): "6 .. Reversal",
        ("FIX.5.0", "77", "C"): "C .. Close",
        ("FIX.5.0", "77", "F"): "F .. FIFO",
        ("FIX.5.0", "77", "O"): "O .. Open",
        ("FIX.5.0", "77", "R"): "R .. Rolled",
        ("FIX.5.0", "77", "N"): "N .. CloseButNotifyOnOpen",
        ("FIX.5.0", "77", "D"): "D .. Default",
        ("FIX.5.0", "81", "0"): "0 .. Regular",
        ("FIX.5.0", "81", "1"): "1 .. SoftDollar",
        ("FIX.5.0", "81", "2"): "2 .. StepIn",
        ("FIX.5.0", "81", "3"): "3 .. StepOut",
        ("FIX.5.0", "81", "4"): "4 .. SoftDollarStepIn",
        ("FIX.5.0", "81", "5"): "5 .. SoftDollarStepOut",
        ("FIX.5.0", "81", "6"): "6 .. PlanSponsor",
        ("FIX.5.0", "87", "0"): "0 .. Accepted",
        ("FIX.5.0", "87", "1"): "1 .. BlockLevelReject",
        ("FIX.5.0", "87", "2"): "2 .. AccountLevelReject",
        ("FIX.5.0", "87", "3"): "3 .. Received",
        ("FIX.5.0", "87", "4"): "4 .. Incomplete",
        ("FIX.5.0", "87", "5"): "5 .. RejectedByIntermediary",
        ("FIX.5.0", "87", "6"): "6 .. AllocationPending",
        ("FIX.5.0", "87", "7"): "7 .. Reversed",
        ("FIX.5.0", "87", "8"): "8 .. CancelledByIntermediary",
        ("FIX.5.0", "87", "9"): "9 .. Claimed",
        ("FIX.5.0", "87", "10"): "10 .. Refused",
        ("FIX.5.0", "87", "11"): "11 .. PendingGiveUpApproval",
        ("FIX.5.0", "87", "12"): "12 .. Cancelled",
        ("FIX.5.0", "87", "13"): "13 .. PendingTakeUpApproval",
        ("FIX.5.0", "87", "14"): "14 .. ReversalPending",
        ("FIX.5.0", "88", "0"): "0 .. UnknownAccount",
        ("FIX.5.0", "88", "1"): "1 .. IncorrectQuantity",
        ("FIX.5.0", "88", "2"): "2 .. IncorrectAveragegPrice",
        ("FIX.5.0", "88", "3"): "3 .. UnknownExecutingBrokerMnemonic",
        ("FIX.5.0", "88", "4"): "4 .. CommissionDifference",
        ("FIX.5.0", "88", "5"): "5 .. UnknownOrderID",
        ("FIX.5.0", "88", "6"): "6 .. UnknownListID",
        ("FIX.5.0", "88", "7"): "7 .. OtherSeeText",
        ("FIX.5.0", "88", "8"): "8 .. IncorrectAllocatedQuantity",
        ("FIX.5.0", "88", "9"): "9 .. CalculationDifference",
        ("FIX.5.0", "88", "10"): "10 .. UnknownOrStaleExecID",
        ("FIX.5.0", "88", "11"): "11 .. MismatchedData",
        ("FIX.5.0", "88", "12"): "12 .. UnknownClOrdID",
        ("FIX.5.0", "88", "13"): "13 .. WarehouseRequestRejected",
        ("FIX.5.0", "88", "14"): "14 .. DuplicateOrMissingIndividualAllocId",
        ("FIX.5.0", "88", "15"): "15 .. TradeNotRecognized",
        ("FIX.5.0", "88", "16"): "16 .. DuplicateTrade",
        ("FIX.5.0", "88", "17"): "17 .. IncorrectOrMissingInstrument",
        ("FIX.5.0", "88", "18"): "18 .. IncorrectOrMissingSettlDate",
        ("FIX.5.0", "88", "19"): "19 .. IncorrectOrMissingFundIDOrFundName",
        ("FIX.5.0", "88", "20"): "20 .. IncorrectOrMissingSettlInstructions",
        ("FIX.5.0", "88", "21"): "21 .. IncorrectOrMissingFees",
        ("FIX.5.0", "88", "22"): "22 .. IncorrectOrMissingTax",
        ("FIX.5.0", "88", "23"): "23 .. UnknownOrMissingParty",
        ("FIX.5.0", "88", "24"): "24 .. IncorrectOrMissingSide",
        ("FIX.5.0", "88", "25"): "25 .. IncorrectOrMissingNetMoney",
        ("FIX.5.0", "88", "26"): "26 .. IncorrectOrMissingTradeDate",
        ("FIX.5.0", "88", "27"): "27 .. IncorrectOrMissingSettlCcyInstructions",
        ("FIX.5.0", "88", "28"): "28 .. IncorrectOrMissingProcessCode",
        ("FIX.5.0", "88", "99"): "99 .. Other",
        ("FIX.5.0", "94", "0"): "0 .. New",
        ("FIX.5.0", "94", "1"): "1 .. Reply",
        ("FIX.5.0", "94", "2"): "2 .. AdminReply",
        ("FIX.5.0", "97", "N"): "N .. OriginalTransmission",
        ("FIX.5.0", "97", "Y"): "Y .. PossibleResend",
        ("FIX.5.0", "98", "0"): "0 .. None",
        ("FIX.5.0", "98", "1"): "1 .. PKCS",
        ("FIX.5.0", "98", "2"): "2 .. DES",
        ("FIX.5.0", "98", "3"): "3 .. PKCSDES",
        ("FIX.5.0", "98", "4"): "4 .. PGPDES",
        ("FIX.5.0", "98", "5"): "5 .. PGPDESMD5",
        ("FIX.5.0", "98", "6"): "6 .. PEM",
        ("FIX.5.0", "102", "0"): "0 .. TooLateToCancel",
        ("FIX.5.0", "102", "1"): "1 .. UnknownOrder",
        ("FIX.5.0", "102", "2"): "2 .. BrokerCredit",
        ("FIX.5.0", "102", "3"): "3 .. OrderAlreadyInPendingStatus",
        ("FIX.5.0", "102", "4"): "4 .. UnableToProcessOrderMassCancelRequest",
        ("FIX.5.0", "102", "5"): "5 .. OrigOrdModTime",
        ("FIX.5.0", "102", "6"): "6 .. DuplicateClOrdID",
        ("FIX.5.0", "102", "7"): "7 .. PriceExceedsCurrentPrice",
        ("FIX.5.0", "102", "8"): "8 .. PriceExceedsCurrentPriceBand",
        ("FIX.5.0", "102", "18"): "18 .. InvalidPriceIncrement",
        ("FIX.5.0", "102", "99"): "99 .. Other",
        ("FIX.5.0", "103", "0"): "0 .. BrokerCredit",
        ("FIX.5.0", "103", "1"): "1 .. UnknownSymbol",
        ("FIX.5.0", "103", "2"): "2 .. ExchangeClosed",
        ("FIX.5.0", "103", "3"): "3 .. OrderExceedsLimit",
        ("FIX.5.0", "103", "4"): "4 .. TooLateToEnter",
        ("FIX.5.0", "103", "5"): "5 .. UnknownOrder",
        ("FIX.5.0", "103", "6"): "6 .. DuplicateOrder",
        ("FIX.5.0", "103", "7"): "7 .. DuplicateOfAVerballyCommunicatedOrder",
        ("FIX.5.0", "103", "8"): "8 .. StaleOrder",
        ("FIX.5.0", "103", "9"): "9 .. TradeAlongRequired",
        ("FIX.5.0", "103", "10"): "10 .. InvalidInvestorID",
        ("FIX.5.0", "103", "11"): "11 .. UnsupportedOrderCharacteristic",
        ("FIX.5.0", "103", "12"): "12 .. SurveillanceOption",
        ("FIX.5.0", "103", "13"): "13 .. IncorrectQuantity",
        ("FIX.5.0", "103", "14"): "14 .. IncorrectAllocatedQuantity",
        ("FIX.5.0", "103", "15"): "15 .. UnknownAccount",
        ("FIX.5.0", "103", "16"): "16 .. PriceExceedsCurrentPriceBand",
        ("FIX.5.0", "103", "18"): "18 .. InvalidPriceIncrement",
        ("FIX.5.0", "103", "19"): "19 .. ReferencePriceNotAvailable",
        ("FIX.5.0", "103", "20"): "20 .. NotionalValueExceedsThreshold",
        ("FIX.5.0", "103", "21"): "21 .. AlgorithRiskThresholdBreached",
        ("FIX.5.0", "103", "22"): "22 .. ShortSellNotPermitted",
        ("FIX.5.0", "103", "23"): "23 .. ShortSellSecurityPreBorrowRestriction",
        ("FIX.5.0", "103", "24"): "24 .. ShortSellAccountPreBorrowRestriction",
        ("FIX.5.0", "103", "25"): "25 .. InsufficientCreditLimit",
        ("FIX.5.0", "103", "26"): "26 .. ExceededClipSizeLimit",
        ("FIX.5.0", "103", "27"): "27 .. ExceededMaxNotionalOrderAmt",
        ("FIX.5.0", "103", "28"): "28 .. ExceededDV01PV01Limit",
        ("FIX.5.0", "103", "29"): "29 .. ExceededCS01Limit",
        ("FIX.5.0", "103", "99"): "99 .. Other",
        ("FIX.5.0", "104", "A"): "A .. AllOrNone",
        ("FIX.5.0", "104", "B"): "B .. MarketOnClose",
        ("FIX.5.0", "104", "C"): "C .. AtTheClose",
        ("FIX.5.0", "104", "D"): "D .. VWAP",
        ("FIX.5.0", "104", "E"): "E .. Axe",
        ("FIX.5.0", "104", "F"): "F .. AxeOnBid",
        ("FIX.5.0", "104", "G"): "G .. AxeOnOffer",
        ("FIX.5.0", "104", "H"): "H .. ClientNaturalWorking",
        ("FIX.5.0", "104", "I"): "I .. InTouchWith",
        ("FIX.5.0", "104", "J"): "J .. PositionWanted",
        ("FIX.5.0", "104", "K"): "K .. MarketMaking",
        ("FIX.5.0", "104", "L"): "L .. Limit",
        ("FIX.5.0", "104", "M"): "M .. MoreBehind",
        ("FIX.5.0", "104", "N"): "N .. ClientNaturalBlock",
        ("FIX.5.0", "104", "O"): "O .. AtTheOpen",
        ("FIX.5.0", "104", "P"): "P .. TakingAPosition",
        ("FIX.5.0", "104", "Q"): "Q .. AtTheMarket",
        ("FIX.5.0", "104", "R"): "R .. ReadyToTrade",
        ("FIX.5.0", "104", "S"): "S .. PortfolioShown",
        ("FIX.5.0", "104", "T"): "T .. ThroughTheDay",
        ("FIX.5.0", "104", "U"): "U .. Unwind",
        ("FIX.5.0", "104", "V"): "V .. Versus",
        ("FIX.5.0", "104", "W"): "W .. Indication",
        ("FIX.5.0", "104", "X"): "X .. CrossingOpportunity",
        ("FIX.5.0", "104", "Y"): "Y .. AtTheMidpoint",
        ("FIX.5.0", "104", "Z"): "Z .. PreOpen",
        ("FIX.5.0", "104", "1"): "1 .. QuantityNegotiable",
        ("FIX.5.0", "104", "2"): "2 .. AllowLateBids",
        ("FIX.5.0", "104", "3"): "3 .. ImmediateOrCounter",
        ("FIX.5.0", "104", "4"): "4 .. AutoTrade",
        ("FIX.5.0", "104", "a"): "a .. AutomaticSpot",
        ("FIX.5.0", "104", "b"): "b .. PlatformCalculatedSpot",
        ("FIX.5.0", "104", "c"): "c .. OutsideSpread",
        ("FIX.5.0", "104", "d"): "d .. DeferredSpot",
        ("FIX.5.0", "104", "n"): "n .. NegotiatedSpot",
        ("FIX.5.0", "113", "N"): "N .. SenderReports",
        ("FIX.5.0", "113", "Y"): "Y .. ReceiverReports",
        ("FIX.5.0", "114", "N"): "N .. No",
        ("FIX.5.0", "114", "Y"): "Y .. Yes",
        ("FIX.5.0", "121", "N"): "N .. DoNotExecuteForexAfterSecurityTrade",
        ("FIX.5.0", "121", "Y"): "Y .. ExecuteForexAfterSecurityTrade",
        ("FIX.5.0", "123", "N"): "N .. SequenceReset",
        ("FIX.5.0", "123", "Y"): "Y .. GapFillMessage",
        ("FIX.5.0", "127", "A"): "A .. UnknownSymbol",
        ("FIX.5.0", "127", "B"): "B .. WrongSide",
        ("FIX.5.0", "127", "C"): "C .. QuantityExceedsOrder",
        ("FIX.5.0", "127", "D"): "D .. NoMatchingOrder",
        ("FIX.5.0", "127", "E"): "E .. PriceExceedsLimit",
        ("FIX.5.0", "127", "F"): "F .. CalculationDifference",
        ("FIX.5.0", "127", "G"): "G .. NoMatchingExecutionReport",
        ("FIX.5.0", "127", "Z"): "Z .. Other",
        ("FIX.5.0", "130", "N"): "N .. NotNatural",
        ("FIX.5.0", "130", "Y"): "Y .. Natural",
        ("FIX.5.0", "139", "1"): "1 .. Regulatory",
        ("FIX.5.0", "139", "2"): "2 .. Tax",
        ("FIX.5.0", "139", "4"): "4 .. ExchangeFees",
        ("FIX.5.0", "139", "5"): "5 .. Stamp",
        ("FIX.5.0", "139", "6"): "6 .. Levy",
        ("FIX.5.0", "139", "7"): "7 .. Other",
        ("FIX.5.0", "139", "8"): "8 .. Markup",
        ("FIX.5.0", "139", "9"): "9 .. ConsumptionTax",
        ("FIX.5.0", "139", "10"): "10 .. PerTransaction",
        ("FIX.5.0", "139", "11"): "11 .. Conversion",
        ("FIX.5.0", "139", "12"): "12 .. Agent",
        ("FIX.5.0", "139", "13"): "13 .. TransferFee",
        ("FIX.5.0", "139", "14"): "14 .. SecurityLending",
        ("FIX.5.0", "139", "15"): "15 .. TradeReporting",
        ("FIX.5.0", "139", "16"): "16 .. TaxOnPrincipalAmount",
        ("FIX.5.0", "139", "17"): "17 .. TaxOnAccruedInterestAmount",
        ("FIX.5.0", "139", "18"): "18 .. NewIssuanceFee",
        ("FIX.5.0", "139", "19"): "19 .. ServiceFee",
        ("FIX.5.0", "139", "20"): "20 .. OddLotFee",
        ("FIX.5.0", "139", "21"): "21 .. AuctionFee",
        ("FIX.5.0", "139", "22"): "22 .. ValueAddedTax",
        ("FIX.5.0", "139", "23"): "23 .. SalesTax",
        ("FIX.5.0", "139", "24"): "24 .. ExecutionFee",
        ("FIX.5.0", "139", "25"): "25 .. OrderEntryFee",
        ("FIX.5.0", "139", "26"): "26 .. OrderModificationFee",
        ("FIX.5.0", "139", "27"): "27 .. OrdersCancellationFee",
        ("FIX.5.0", "139", "28"): "28 .. MarketDataAccessFee",
        ("FIX.5.0", "139", "29"): "29 .. MarketDataTerminalFee",
        ("FIX.5.0", "139", "30"): "30 .. MarketDataVolumeFee",
        ("FIX.5.0", "139", "31"): "31 .. ClearingFee",
        ("FIX.5.0", "139", "32"): "32 .. SettlementFee",
        ("FIX.5.0", "139", "33"): "33 .. Rebates",
        ("FIX.5.0", "139", "34"): "34 .. Discounts",
        ("FIX.5.0", "139", "35"): "35 .. Payments",
        ("FIX.5.0", "139", "36"): "36 .. NonMonetaryPayments",
        ("FIX.5.0", "141", "N"): "N .. No",
        ("FIX.5.0", "141", "Y"): "Y .. Yes",
        ("FIX.5.0", "150", "0"): "0 .. New",
        ("FIX.5.0", "150", "3"): "3 .. DoneForDay",
        ("FIX.5.0", "150", "4"): "4 .. Canceled",
        ("FIX.5.0", "150", "5"): "5 .. Replaced",
        ("FIX.5.0", "150", "6"): "6 .. PendingCancel",
        ("FIX.5.0", "150", "7"): "7 .. Stopped",
        ("FIX.5.0", "150", "8"): "8 .. Rejected",
        ("FIX.5.0", "150", "9"): "9 .. Suspended",
        ("FIX.5.0", "150", "A"): "A .. PendingNew",
        ("FIX.5.0", "150", "B"): "B .. Calculated",
        ("FIX.5.0", "150", "C"): "C .. Expired",
        ("FIX.5.0", "150", "D"): "D .. Restated",
        ("FIX.5.0", "150", "E"): "E .. PendingReplace",
        ("FIX.5.0", "150", "F"): "F .. Trade",
        ("FIX.5.0", "150", "G"): "G .. TradeCorrect",
        ("FIX.5.0", "150", "H"): "H .. TradeCancel",
        ("FIX.5.0", "150", "I"): "I .. OrderStatus",
        ("FIX.5.0", "150", "J"): "J .. TradeInAClearingHold",
        ("FIX.5.0", "150", "K"): "K .. TradeHasBeenReleasedToClearing",
        ("FIX.5.0", "150", "L"): "L .. TriggeredOrActivatedBySystem",
        ("FIX.5.0", "150", "M"): "M .. Locked",
        ("FIX.5.0", "150", "N"): "N .. Released",
        ("FIX.5.0", "156", "M"): "M .. Multiply",
        ("FIX.5.0", "156", "D"): "D .. Divide",
        ("FIX.5.0", "160", "0"): "0 .. Default",
        ("FIX.5.0", "160", "1"): "1 .. StandingInstructionsProvided",
        ("FIX.5.0", "160", "2"): "2 .. SpecificAllocationAccountOverriding",
        ("FIX.5.0", "160", "3"): "3 .. SpecificAllocationAccountStanding",
        ("FIX.5.0", "160", "4"): "4 .. SpecificOrderForASingleAccount",
        ("FIX.5.0", "160", "5"): "5 .. RequestReject",
        ("FIX.5.0", "163", "N"): "N .. New",
        ("FIX.5.0", "163", "C"): "C .. Cancel",
        ("FIX.5.0", "163", "R"): "R .. Replace",
        ("FIX.5.0", "163", "T"): "T .. Restate",
        ("FIX.5.0", "165", "1"): "1 .. BrokerCredit",
        ("FIX.5.0", "165", "2"): "2 .. Institution",
        ("FIX.5.0", "165", "3"): "3 .. Investor",
        ("FIX.5.0", "167", "EUSUPRA"): "EUSUPRA .. EuroSupranationalCoupons",
        ("FIX.5.0", "167", "FAC"): "FAC .. FederalAgencyCoupon",
        ("FIX.5.0", "167", "FADN"): "FADN .. FederalAgencyDiscountNote",
        ("FIX.5.0", "167", "PEF"): "PEF .. PrivateExportFunding",
        ("FIX.5.0", "167", "SUPRA"): "SUPRA .. USDSupranationalCoupons",
        ("FIX.5.0", "167", "CORP"): "CORP .. CorporateBond",
        ("FIX.5.0", "167", "CPP"): "CPP .. CorporatePrivatePlacement",
        ("FIX.5.0", "167", "CB"): "CB .. ConvertibleBond",
        ("FIX.5.0", "167", "DUAL"): "DUAL .. DualCurrency",
        ("FIX.5.0", "167", "EUCORP"): "EUCORP .. EuroCorporateBond",
        ("FIX.5.0", "167", "EUFRN"): "EUFRN .. EuroCorporateFloatingRateNotes",
        ("FIX.5.0", "167", "FRN"): "FRN .. USCorporateFloatingRateNotes",
        ("FIX.5.0", "167", "XLINKD"): "XLINKD .. IndexedLinked",
        ("FIX.5.0", "167", "STRUCT"): "STRUCT .. StructuredNotes",
        ("FIX.5.0", "167", "YANK"): "YANK .. YankeeCorporateBond",
        ("FIX.5.0", "167", "FXNDF"): "FXNDF .. NonDeliverableForward",
        ("FIX.5.0", "167", "FXSPOT"): "FXSPOT .. FXSpot",
        ("FIX.5.0", "167", "FXFWD"): "FXFWD .. FXForward",
        ("FIX.5.0", "167", "FXSWAP"): "FXSWAP .. FXSwap",
        ("FIX.5.0", "167", "FXNDS"): "FXNDS .. NonDeliverableSwap",
        ("FIX.5.0", "167", "CAP"): "CAP .. Cap",
        ("FIX.5.0", "167", "CDS"): "CDS .. CreditDefaultSwap",
        ("FIX.5.0", "167", "CLLR"): "CLLR .. Collar",
        ("FIX.5.0", "167", "CMDTYSWAP"): "CMDTYSWAP .. CommoditySwap",
        ("FIX.5.0", "167", "EXOTIC"): "EXOTIC .. Exotic",
        ("FIX.5.0", "167", "OOC"): "OOC .. OptionsOnCombo",
        ("FIX.5.0", "167", "FLR"): "FLR .. Floor",
        ("FIX.5.0", "167", "FRA"): "FRA .. FRA",
        ("FIX.5.0", "167", "FUT"): "FUT .. Future",
        ("FIX.5.0", "167", "FWD"): "FWD .. DerivativeForward",
        ("FIX.5.0", "167", "IRS"): "IRS .. InterestRateSwap",
        ("FIX.5.0", "167", "TRS"): "TRS .. TotalReturnSwap",
        ("FIX.5.0", "167", "LOANLEASE"): "LOANLEASE .. LoanLease",
        ("FIX.5.0", "167", "OOF"): "OOF .. OptionsOnFutures",
        ("FIX.5.0", "167", "OOP"): "OOP .. OptionsOnPhysical",
        ("FIX.5.0", "167", "OPT"): "OPT .. Option",
        ("FIX.5.0", "167", "SPOTFWD"): "SPOTFWD .. SpotForward",
        ("FIX.5.0", "167", "SWAPTION"): "SWAPTION .. SwapOption",
        ("FIX.5.0", "167", "XMISSION"): "XMISSION .. Transmission",
        ("FIX.5.0", "167", "INDEX"): "INDEX .. Index",
        ("FIX.5.0", "167", "BDBSKT"): "BDBSKT .. BondBasket",
        ("FIX.5.0", "167", "CFD"): "CFD .. ContractForDifference",
        ("FIX.5.0", "167", "CRLTNSWAP"): "CRLTNSWAP .. CorrelationSwap",
        ("FIX.5.0", "167", "DVDNDSWAP"): "DVDNDSWAP .. DiviendSwap",
        ("FIX.5.0", "167", "EQBSKT"): "EQBSKT .. EquityBasket",
        ("FIX.5.0", "167", "EQFWD"): "EQFWD .. EquityForward",
        ("FIX.5.0", "167", "RTRNSWAP"): "RTRNSWAP .. ReturnSwap",
        ("FIX.5.0", "167", "VARSWAP"): "VARSWAP .. VarianceSwap",
        ("FIX.5.0", "167", "PRTFLIOSWAP"): "PRTFLIOSWAP .. PortfolioSwaps",
        ("FIX.5.0", "167", "FUTSWAP"): "FUTSWAP .. FuturesOnASwap",
        ("FIX.5.0", "167", "FWDSWAP "): "FWDSWAP  .. ForwardsOnASwap",
        ("FIX.5.0", "167", "FWDFRTAGMT "): "FWDFRTAGMT  .. ForwardFreightAgreement",
        ("FIX.5.0", "167", "SPREADBET"): "SPREADBET .. SpreadBetting",
        ("FIX.5.0", "167", "ETC"): "ETC .. ExchangeTradedCommodity",
        ("FIX.5.0", "167", "CS"): "CS .. CommonStock",
        ("FIX.5.0", "167", "PS"): "PS .. PreferredStock",
        ("FIX.5.0", "167", "DR"): "DR .. DepositoryReceipts",
        ("FIX.5.0", "167", "REPO"): "REPO .. Repurchase",
        ("FIX.5.0", "167", "FORWARD"): "FORWARD .. Forward",
        ("FIX.5.0", "167", "BUYSELL"): "BUYSELL .. BuySellback",
        ("FIX.5.0", "167", "SECLOAN"): "SECLOAN .. SecuritiesLoan",
        ("FIX.5.0", "167", "SECPLEDGE"): "SECPLEDGE .. SecuritiesPledge",
        ("FIX.5.0", "167", "DVPLDG"): "DVPLDG .. DeliveryVersusPledge",
        ("FIX.5.0", "167", "COLLBSKT"): "COLLBSKT .. CollateralBasket",
        ("FIX.5.0", "167", "SFP"): "SFP .. StructuredFinanceProduct",
        ("FIX.5.0", "167", "BRADY"): "BRADY .. BradyBond",
        ("FIX.5.0", "167", "CAN"): "CAN .. CanadianTreasuryNotes",
        ("FIX.5.0", "167", "CTB"): "CTB .. CanadianTreasuryBills",
        ("FIX.5.0", "167", "EUSOV"): "EUSOV .. EuroSovereigns",
        ("FIX.5.0", "167", "PROV"): "PROV .. CanadianProvincialBonds",
        ("FIX.5.0", "167", "TB"): "TB .. TreasuryBill",
        ("FIX.5.0", "167", "TBOND"): "TBOND .. USTreasuryBond",
        ("FIX.5.0", "167", "TINT"): "TINT .. InterestStripFromAnyBondOrNote",
        ("FIX.5.0", "167", "TBILL"): "TBILL .. USTreasuryBill",
        ("FIX.5.0", "167", "TIPS"): "TIPS .. TreasuryInflationProtectedSecurities",
        ("FIX.5.0", "167", "TCAL"): "TCAL .. PrincipalStripOfACallableBondOrNote",
        ("FIX.5.0", "167", "TPRN"): "TPRN .. PrincipalStripFromANonCallableBondOrNote",
        ("FIX.5.0", "167", "TNOTE"): "TNOTE .. USTreasuryNote",
        ("FIX.5.0", "167", "TERM"): "TERM .. TermLoan",
        ("FIX.5.0", "167", "RVLV"): "RVLV .. RevolverLoan",
        ("FIX.5.0", "167", "RVLVTRM"): "RVLVTRM .. Revolver",
        ("FIX.5.0", "167", "BRIDGE"): "BRIDGE .. BridgeLoan",
        ("FIX.5.0", "167", "LOFC"): "LOFC .. LetterOfCredit",
        ("FIX.5.0", "167", "SWING"): "SWING .. SwingLineFacility",
        ("FIX.5.0", "167", "DINP"): "DINP .. DebtorInPossession",
        ("FIX.5.0", "167", "DEFLTED"): "DEFLTED .. Defaulted",
        ("FIX.5.0", "167", "WITHDRN"): "WITHDRN .. Withdrawn",
        ("FIX.5.0", "167", "REPLACD"): "REPLACD .. Replaced",
        ("FIX.5.0", "167", "MATURED"): "MATURED .. Matured",
        ("FIX.5.0", "167", "AMENDED"): "AMENDED .. Amended",
        ("FIX.5.0", "167", "RETIRED"): "RETIRED .. Retired",
        ("FIX.5.0", "167", "BA"): "BA .. BankersAcceptance",
        ("FIX.5.0", "167", "BDN"): "BDN .. BankDepositoryNote",
        ("FIX.5.0", "167", "BN"): "BN .. BankNotes",
        ("FIX.5.0", "167", "BOX"): "BOX .. BillOfExchanges",
        ("FIX.5.0", "167", "CAMM"): "CAMM .. CanadianMoneyMarkets",
        ("FIX.5.0", "167", "CD"): "CD .. CertificateOfDeposit",
        ("FIX.5.0", "167", "CL"): "CL .. CallLoans",
        ("FIX.5.0", "167", "CP"): "CP .. CommercialPaper",
        ("FIX.5.0", "167", "DN"): "DN .. DepositNotes",
        ("FIX.5.0", "167", "EUCD"): "EUCD .. EuroCertificateOfDeposit",
        ("FIX.5.0", "167", "EUCP"): "EUCP .. EuroCommercialPaper",
        ("FIX.5.0", "167", "LQN"): "LQN .. LiquidityNote",
        ("FIX.5.0", "167", "MTN"): "MTN .. MediumTermNotes",
        ("FIX.5.0", "167", "ONITE"): "ONITE .. Overnight",
        ("FIX.5.0", "167", "PN"): "PN .. PromissoryNote",
        ("FIX.5.0", "167", "STN"): "STN .. ShortTermLoanNote",
        ("FIX.5.0", "167", "PZFJ"): "PZFJ .. PlazosFijos",
        ("FIX.5.0", "167", "SLQN"): "SLQN .. SecuredLiquidityNote",
        ("FIX.5.0", "167", "TD"): "TD .. TimeDeposit",
        ("FIX.5.0", "167", "TLQN"): "TLQN .. TermLiquidityNote",
        ("FIX.5.0", "167", "XCN"): "XCN .. ExtendedCommNote",
        ("FIX.5.0", "167", "YCD"): "YCD .. YankeeCertificateOfDeposit",
        ("FIX.5.0", "167", "ABS"): "ABS .. AssetBackedSecurities",
        ("FIX.5.0", "167", "CMB"): "CMB .. CanadianMortgageBonds",
        ("FIX.5.0", "167", "CMBS"): "CMBS .. Corp",
        ("FIX.5.0", "167", "CMO"): "CMO .. CollateralizedMortgageObligation",
        ("FIX.5.0", "167", "IET"): "IET .. IOETTEMortgage",
        ("FIX.5.0", "167", "MBS"): "MBS .. MortgageBackedSecurities",
        ("FIX.5.0", "167", "MIO"): "MIO .. MortgageInterestOnly",
        ("FIX.5.0", "167", "MPO"): "MPO .. MortgagePrincipalOnly",
        ("FIX.5.0", "167", "MPP"): "MPP .. MortgagePrivatePlacement",
        ("FIX.5.0", "167", "MPT"): "MPT .. MiscellaneousPassThrough",
        ("FIX.5.0", "167", "PFAND"): "PFAND .. Pfandbriefe",
        ("FIX.5.0", "167", "TBA"): "TBA .. ToBeAnnounced",
        ("FIX.5.0", "167", "AN"): "AN .. OtherAnticipationNotes",
        ("FIX.5.0", "167", "COFO"): "COFO .. CertificateOfObligation",
        ("FIX.5.0", "167", "COFP"): "COFP .. CertificateOfParticipation",
        ("FIX.5.0", "167", "GO"): "GO .. GeneralObligationBonds",
        ("FIX.5.0", "167", "MT"): "MT .. MandatoryTender",
        ("FIX.5.0", "167", "RAN"): "RAN .. RevenueAnticipationNote",
        ("FIX.5.0", "167", "REV"): "REV .. RevenueBonds",
        ("FIX.5.0", "167", "SPCLA"): "SPCLA .. SpecialAssessment",
        ("FIX.5.0", "167", "SPCLO"): "SPCLO .. SpecialObligation",
        ("FIX.5.0", "167", "SPCLT"): "SPCLT .. SpecialTax",
        ("FIX.5.0", "167", "TAN"): "TAN .. TaxAnticipationNote",
        ("FIX.5.0", "167", "TAXA"): "TAXA .. TaxAllocation",
        ("FIX.5.0", "167", "TECP"): "TECP .. TaxExemptCommercialPaper",
        ("FIX.5.0", "167", "TMCP"): "TMCP .. TaxableMunicipalCP",
        ("FIX.5.0", "167", "TRAN"): "TRAN .. TaxRevenueAnticipationNote",
        ("FIX.5.0", "167", "VRDN"): "VRDN .. VariableRateDemandNote",
        ("FIX.5.0", "167", "WAR"): "WAR .. Warrant",
        ("FIX.5.0", "167", "MF"): "MF .. MutualFund",
        ("FIX.5.0", "167", "MLEG"): "MLEG .. MultilegInstrument",
        ("FIX.5.0", "167", "NONE"): "NONE .. NoSecurityType",
        ("FIX.5.0", "167", "?"): "? .. Wildcard",
        ("FIX.5.0", "167", "CASH"): "CASH .. Cash",
        ("FIX.5.0", "167", "Other"): "Other .. Other",
        ("FIX.5.0", "167", "ETN"): "ETN .. ExchangeTradedNote",
        ("FIX.5.0", "167", "SECDERIV"): "SECDERIV .. SecuritizedDerivative",
        ("FIX.5.0", "169", "0"): "0 .. Other",
        ("FIX.5.0", "169", "1"): "1 .. DTCSID",
        ("FIX.5.0", "169", "2"): "2 .. ThomsonALERT",
        ("FIX.5.0", "169", "3"): "3 .. AGlobalCustodian",
        ("FIX.5.0", "169", "4"): "4 .. AccountNet",
        ("FIX.5.0", "172", "0"): "0 .. Versus",
        ("FIX.5.0", "172", "1"): "1 .. Free",
        ("FIX.5.0", "172", "2"): "2 .. TriParty",
        ("FIX.5.0", "172", "3"): "3 .. HoldInCustody",
        ("FIX.5.0", "197", "0"): "0 .. FXNetting",
        ("FIX.5.0", "197", "1"): "1 .. FXSwap",
        ("FIX.5.0", "201", "0"): "0 .. Put",
        ("FIX.5.0", "201", "1"): "1 .. Call",
        ("FIX.5.0", "201", "2"): "2 .. Other",
        ("FIX.5.0", "201", "3"): "3 .. Chooser",
        ("FIX.5.0", "203", "0"): "0 .. Covered",
        ("FIX.5.0", "203", "1"): "1 .. Uncovered",
        ("FIX.5.0", "208", "N"): "N .. DetailsShouldNotBeCommunicated",
        ("FIX.5.0", "208", "Y"): "Y .. DetailsShouldBeCommunicated",
        ("FIX.5.0", "209", "1"): "1 .. Match",
        ("FIX.5.0", "209", "2"): "2 .. Forward",
        ("FIX.5.0", "209", "3"): "3 .. ForwardAndMatch",
        ("FIX.5.0", "216", "1"): "1 .. TargetFirm",
        ("FIX.5.0", "216", "2"): "2 .. TargetList",
        ("FIX.5.0", "216", "3"): "3 .. BlockFirm",
        ("FIX.5.0", "216", "4"): "4 .. BlockList",
        ("FIX.5.0", "216", "5"): "5 .. TargetPerson",
        ("FIX.5.0", "216", "6"): "6 .. BlockPerson",
        ("FIX.5.0", "221", "EONIA"): "EONIA .. EONIA",
        ("FIX.5.0", "221", "EUREPO"): "EUREPO .. EUREPO",
        ("FIX.5.0", "221", "FutureSWAP"): "FutureSWAP .. FutureSWAP",
        ("FIX.5.0", "221", "LIBID"): "LIBID .. LIBID",
        ("FIX.5.0", "221", "LIBOR"): "LIBOR .. LIBOR",
        ("FIX.5.0", "221", "MuniAAA"): "MuniAAA .. MuniAAA",
        ("FIX.5.0", "221", "OTHER"): "OTHER .. OTHER",
        ("FIX.5.0", "221", "Pfandbriefe"): "Pfandbriefe .. Pfandbriefe",
        ("FIX.5.0", "221", "SONIA"): "SONIA .. SONIA",
        ("FIX.5.0", "221", "SWAP"): "SWAP .. SWAP",
        ("FIX.5.0", "221", "Treasury"): "Treasury .. Treasury",
        ("FIX.5.0", "221", "FEDEFF"): "FEDEFF .. FedFundRateEffective",
        ("FIX.5.0", "221", "FEDOPEN"): "FEDOPEN .. FedOpen",
        ("FIX.5.0", "221", "EURIBOR"): "EURIBOR .. EURIBOR",
        ("FIX.5.0", "233", "AMT"): "AMT .. AlternativeMinimumTax",
        ("FIX.5.0", "233", "AUTOREINV"): "AUTOREINV .. AutoReinvestment",
        ("FIX.5.0", "233", "BANKQUAL"): "BANKQUAL .. BankQualified",
        ("FIX.5.0", "233", "BGNCON"): "BGNCON .. BargainConditions",
        ("FIX.5.0", "233", "COUPON"): "COUPON .. CouponRange",
        ("FIX.5.0", "233", "CURRENCY"): "CURRENCY .. ISOCurrencyCode",
        ("FIX.5.0", "233", "CUSTOMDATE"): "CUSTOMDATE .. CustomStart",
        ("FIX.5.0", "233", "GEOG"): "GEOG .. Geographics",
        ("FIX.5.0", "233", "HAIRCUT"): "HAIRCUT .. ValuationDiscount",
        ("FIX.5.0", "233", "INSURED"): "INSURED .. Insured",
        ("FIX.5.0", "233", "ISSUE"): "ISSUE .. IssueDate",
        ("FIX.5.0", "233", "ISSUER"): "ISSUER .. Issuer",
        ("FIX.5.0", "233", "ISSUESIZE"): "ISSUESIZE .. IssueSizeRange",
        ("FIX.5.0", "233", "LOOKBACK"): "LOOKBACK .. LookbackDays",
        ("FIX.5.0", "233", "LOT"): "LOT .. ExplicitLotIdentifier",
        ("FIX.5.0", "233", "LOTVAR"): "LOTVAR .. LotVariance",
        ("FIX.5.0", "233", "MAT"): "MAT .. MaturityYearAndMonth",
        ("FIX.5.0", "233", "MATURITY"): "MATURITY .. MaturityRange",
        ("FIX.5.0", "233", "MAXSUBS"): "MAXSUBS .. MaximumSubstitutions",
        ("FIX.5.0", "233", "MINDNOM"): "MINDNOM .. MinimumDenomination",
        ("FIX.5.0", "233", "MININCR"): "MININCR .. MinimumIncrement",
        ("FIX.5.0", "233", "MINQTY"): "MINQTY .. MinimumQuantity",
        ("FIX.5.0", "233", "PAYFREQ"): "PAYFREQ .. PaymentFrequency",
        ("FIX.5.0", "233", "PIECES"): "PIECES .. NumberOfPieces",
        ("FIX.5.0", "233", "PMAX"): "PMAX .. PoolsMaximum",
        ("FIX.5.0", "233", "PPL"): "PPL .. PoolsPerLot",
        ("FIX.5.0", "233", "PPM"): "PPM .. PoolsPerMillion",
        ("FIX.5.0", "233", "PPT"): "PPT .. PoolsPerTrade",
        ("FIX.5.0", "233", "PRICE"): "PRICE .. PriceRange",
        ("FIX.5.0", "233", "PRICEFREQ"): "PRICEFREQ .. PricingFrequency",
        ("FIX.5.0", "233", "PROD"): "PROD .. ProductionYear",
        ("FIX.5.0", "233", "PROTECT"): "PROTECT .. CallProtection",
        ("FIX.5.0", "233", "PURPOSE"): "PURPOSE .. Purpose",
        ("FIX.5.0", "233", "PXSOURCE"): "PXSOURCE .. BenchmarkPriceSource",
        ("FIX.5.0", "233", "RATING"): "RATING .. RatingSourceAndRange",
        ("FIX.5.0", "233", "REDEMPTION"): "REDEMPTION .. TypeOfRedemption",
        ("FIX.5.0", "233", "RESTRICTED"): "RESTRICTED .. Restricted",
        ("FIX.5.0", "233", "SECTOR"): "SECTOR .. MarketSector",
        ("FIX.5.0", "233", "SECTYPE"): "SECTYPE .. SecurityTypeIncludedOrExcluded",
        ("FIX.5.0", "233", "STRUCT"): "STRUCT .. Structure",
        ("FIX.5.0", "233", "SUBSFREQ"): "SUBSFREQ .. SubstitutionsFrequency",
        ("FIX.5.0", "233", "SUBSLEFT"): "SUBSLEFT .. SubstitutionsLeft",
        ("FIX.5.0", "233", "TEXT"): "TEXT .. FreeformText",
        ("FIX.5.0", "233", "TRDVAR"): "TRDVAR .. TradeVariance",
        ("FIX.5.0", "233", "WAC"): "WAC .. WeightedAverageCoupon",
        ("FIX.5.0", "233", "WAL"): "WAL .. WeightedAverageLifeCoupon",
        ("FIX.5.0", "233", "WALA"): "WALA .. WeightedAverageLoanAge",
        ("FIX.5.0", "233", "WAM"): "WAM .. WeightedAverageMaturity",
        ("FIX.5.0", "233", "WHOLE"): "WHOLE .. WholePool",
        ("FIX.5.0", "233", "YIELD"): "YIELD .. YieldRange",
        ("FIX.5.0", "233", "ORIGAMT"): "ORIGAMT .. OriginalAmount",
        ("FIX.5.0", "233", "POOLEFFDT"): "POOLEFFDT .. PoolEffectiveDate",
        ("FIX.5.0", "233", "POOLINITFCTR"): "POOLINITFCTR .. PoolInitialFactor",
        ("FIX.5.0", "233", "TRANCHE"): "TRANCHE .. Tranche",
        ("FIX.5.0", "233", "SUBSTITUTION"): "SUBSTITUTION .. Substitution",
        ("FIX.5.0", "233", "MULTEXCHFLLBCK"): "MULTEXCHFLLBCK .. MULTEXCHFLLBCK",
        ("FIX.5.0", "233", "COMPSECFLLBCK"): "COMPSECFLLBCK .. COMPSECFLLBCK",
        ("FIX.5.0", "233", "LOCLJRSDCTN"): "LOCLJRSDCTN .. LOCLJRSDCTN",
        ("FIX.5.0", "233", "RELVJRSDCTN"): "RELVJRSDCTN .. RELVJRSDCTN",
        ("FIX.5.0", "233", "INCURRCVY"): "INCURRCVY .. IncurredRecovery",
        ("FIX.5.0", "233", "ADDTRM"): "ADDTRM .. AdditionalTerm",
        ("FIX.5.0", "233", "MODEQTYDLVY"): "MODEQTYDLVY .. ModifiedEquityDelivery",
        ("FIX.5.0", "233", "NOREFOBLIG"): "NOREFOBLIG .. NoReferenceOblication",
        ("FIX.5.0", "233", "UNKREFOBLIG"): "UNKREFOBLIG .. UnknownReferenceObligation",
        ("FIX.5.0", "233", "ALLGUARANTEES"): "ALLGUARANTEES .. AllGuarantees",
        ("FIX.5.0", "233", "REFPX"): "REFPX .. ReferencePrice",
        ("FIX.5.0", "233", "REFPOLICY"): "REFPOLICY .. ReferencePolicy",
        ("FIX.5.0", "233", "SECRDLIST"): "SECRDLIST .. SecuredList",
        ("FIX.5.0", "233", "AVFICO"): "AVFICO .. AverageFICOScore",
        ("FIX.5.0", "233", "AVSIZE"): "AVSIZE .. AverageLoanSize",
        ("FIX.5.0", "233", "MAXBAL"): "MAXBAL .. MaximumLoanBalance",
        ("FIX.5.0", "233", "POOL"): "POOL .. PoolIdentifier",
        ("FIX.5.0", "233", "ROLLTYPE"): "ROLLTYPE .. TypeOfRollTrade",
        ("FIX.5.0", "233", "REFTRADE"): "REFTRADE .. ReferenceToRollingOrClosingTrade",
        ("FIX.5.0", "233", "REFPRIN"): "REFPRIN .. PrincipalOfRollingOrClosingTrade",
        ("FIX.5.0", "233", "REFINT"): "REFINT .. InterestOfRollingOrClosingTrade",
        ("FIX.5.0", "233", "AVAILQTY"): "AVAILQTY .. AvailableOfferQuantityToBeShownToTheStreet",
        ("FIX.5.0", "233", "BROKERCREDIT"): "BROKERCREDIT .. BrokerCredit",
        ("FIX.5.0", "233", "INTERNALPX"): "INTERNALPX .. OfferPriceToBeShownToInternalBrokers",
        ("FIX.5.0", "233", "INTERNALQTY"): "INTERNALQTY .. OfferQuantityToBeShownToInternalBrokers",
        ("FIX.5.0", "233", "LEAVEQTY"): "LEAVEQTY .. TheMinimumResidualOfferQuantity",
        ("FIX.5.0", "233", "MAXORDQTY"): "MAXORDQTY .. MaximumOrderSize",
        ("FIX.5.0", "233", "ORDRINCR"): "ORDRINCR .. OrderQuantityIncrement",
        ("FIX.5.0", "233", "PRIMARY"): "PRIMARY .. PrimaryOrSecondaryMarketIndicator",
        ("FIX.5.0", "233", "SALESCREDITOVR"): "SALESCREDITOVR .. BrokerSalesCreditOverride",
        ("FIX.5.0", "233", "TRADERCREDIT"): "TRADERCREDIT .. TraderCredit",
        ("FIX.5.0", "233", "DISCOUNT"): "DISCOUNT .. DiscountRate",
        ("FIX.5.0", "233", "YTM"): "YTM .. YieldToMaturity",
        ("FIX.5.0", "233", "ABS"): "ABS .. AbsolutePrepaymentSpeed",
        ("FIX.5.0", "233", "CPP"): "CPP .. ConstantPrepaymentPenalty",
        ("FIX.5.0", "233", "CPR"): "CPR .. ConstantPrepaymentRate",
        ("FIX.5.0", "233", "CPY"): "CPY .. ConstantPrepaymentYield",
        ("FIX.5.0", "233", "HEP"): "HEP .. FinalCPROfHomeEquityPrepaymentCurve",
        ("FIX.5.0", "233", "MHP"): "MHP .. PercentOfManufacturedHousingPrepaymentCurve",
        ("FIX.5.0", "233", "MPR"): "MPR .. MonthlyPrepaymentRate",
        ("FIX.5.0", "233", "PPC"): "PPC .. PercentOfProspectusPrepaymentCurve",
        ("FIX.5.0", "233", "PSA"): "PSA .. PercentOfBMAPrepaymentCurve",
        ("FIX.5.0", "233", "SMM"): "SMM .. SingleMonthlyMortality",
        ("FIX.5.0", "235", "AFTERTAX"): "AFTERTAX .. AfterTaxYield",
        ("FIX.5.0", "235", "ANNUAL"): "ANNUAL .. AnnualYield",
        ("FIX.5.0", "235", "ATISSUE"): "ATISSUE .. YieldAtIssue",
        ("FIX.5.0", "235", "AVGMATURITY"): "AVGMATURITY .. YieldToAverageMaturity",
        ("FIX.5.0", "235", "BOOK"): "BOOK .. BookYield",
        ("FIX.5.0", "235", "CALL"): "CALL .. YieldToNextCall",
        ("FIX.5.0", "235", "CHANGE"): "CHANGE .. YieldChangeSinceClose",
        ("FIX.5.0", "235", "CLOSE"): "CLOSE .. ClosingYield",
        ("FIX.5.0", "235", "COMPOUND"): "COMPOUND .. CompoundYield",
        ("FIX.5.0", "235", "CURRENT"): "CURRENT .. CurrentYield",
        ("FIX.5.0", "235", "GOVTEQUIV"): "GOVTEQUIV .. GvntEquivalentYield",
        ("FIX.5.0", "235", "GROSS"): "GROSS .. TrueGrossYield",
        ("FIX.5.0", "235", "INFLATION"): "INFLATION .. YieldWithInflationAssumption",
        ("FIX.5.0", "235", "INVERSEFLOATER"): "INVERSEFLOATER .. InverseFloaterBondYield",
        ("FIX.5.0", "235", "LASTCLOSE"): "LASTCLOSE .. MostRecentClosingYield",
        ("FIX.5.0", "235", "LASTMONTH"): "LASTMONTH .. ClosingYieldMostRecentMonth",
        ("FIX.5.0", "235", "LASTQUARTER"): "LASTQUARTER .. ClosingYieldMostRecentQuarter",
        ("FIX.5.0", "235", "LASTYEAR"): "LASTYEAR .. ClosingYieldMostRecentYear",
        ("FIX.5.0", "235", "LONGAVGLIFE"): "LONGAVGLIFE .. YieldToLongestAverageLife",
        ("FIX.5.0", "235", "MARK"): "MARK .. MarkToMarketYield",
        ("FIX.5.0", "235", "MATURITY"): "MATURITY .. YieldToMaturity",
        ("FIX.5.0", "235", "NEXTREFUND"): "NEXTREFUND .. YieldToNextRefund",
        ("FIX.5.0", "235", "OPENAVG"): "OPENAVG .. OpenAverageYield",
        ("FIX.5.0", "235", "PREVCLOSE"): "PREVCLOSE .. PreviousCloseYield",
        ("FIX.5.0", "235", "PROCEEDS"): "PROCEEDS .. ProceedsYield",
        ("FIX.5.0", "235", "PUT"): "PUT .. YieldToNextPut",
        ("FIX.5.0", "235", "SEMIANNUAL"): "SEMIANNUAL .. SemiAnnualYield",
        ("FIX.5.0", "235", "SHORTAVGLIFE"): "SHORTAVGLIFE .. YieldToShortestAverageLife",
        ("FIX.5.0", "235", "SIMPLE"): "SIMPLE .. SimpleYield",
        ("FIX.5.0", "235", "TAXEQUIV"): "TAXEQUIV .. TaxEquivalentYield",
        ("FIX.5.0", "235", "TENDER"): "TENDER .. YieldToTenderDate",
        ("FIX.5.0", "235", "TRUE"): "TRUE .. TrueYield",
        ("FIX.5.0", "235", "VALUE1_32"): "VALUE1_32 .. YieldValueOf32nds",
        ("FIX.5.0", "235", "WORST"): "WORST .. YieldToWorst",
        ("FIX.5.0", "258", "N"): "N .. NotTradedFlat",
        ("FIX.5.0", "258", "Y"): "Y .. TradedFlat",
        ("FIX.5.0", "263", "0"): "0 .. Snapshot",
        ("FIX.5.0", "263", "1"): "1 .. SnapshotAndUpdates",
        ("FIX.5.0", "263", "2"): "2 .. DisablePreviousSnapshot",
        ("FIX.5.0", "265", "0"): "0 .. FullRefresh",
        ("FIX.5.0", "265", "1"): "1 .. IncrementalRefresh",
        ("FIX.5.0", "266", "Y"): "Y .. BookEntriesToBeAggregated",
        ("FIX.5.0", "266", "N"): "N .. BookEntriesShouldNotBeAggregated",
        ("FIX.5.0", "269", "0"): "0 .. Bid",
        ("FIX.5.0", "269", "1"): "1 .. Offer",
        ("FIX.5.0", "269", "2"): "2 .. Trade",
        ("FIX.5.0", "269", "3"): "3 .. IndexValue",
        ("FIX.5.0", "269", "4"): "4 .. OpeningPrice",
        ("FIX.5.0", "269", "5"): "5 .. ClosingPrice",
        ("FIX.5.0", "269", "6"): "6 .. SettlementPrice",
        ("FIX.5.0", "269", "7"): "7 .. TradingSessionHighPrice",
        ("FIX.5.0", "269", "8"): "8 .. TradingSessionLowPrice",
        ("FIX.5.0", "269", "9"): "9 .. TradingSessionVWAPPrice",
        ("FIX.5.0", "269", "A"): "A .. Imbalance",
        ("FIX.5.0", "269", "B"): "B .. TradeVolume",
        ("FIX.5.0", "269", "C"): "C .. OpenInterest",
        ("FIX.5.0", "269", "D"): "D .. CompositeUnderlyingPrice",
        ("FIX.5.0", "269", "E"): "E .. SimulatedSellPrice",
        ("FIX.5.0", "269", "F"): "F .. SimulatedBuyPrice",
        ("FIX.5.0", "269", "G"): "G .. MarginRate",
        ("FIX.5.0", "269", "H"): "H .. MidPrice",
        ("FIX.5.0", "269", "J"): "J .. EmptyBook",
        ("FIX.5.0", "269", "K"): "K .. SettleHighPrice",
        ("FIX.5.0", "269", "L"): "L .. SettleLowPrice",
        ("FIX.5.0", "269", "M"): "M .. PriorSettlePrice",
        ("FIX.5.0", "269", "N"): "N .. SessionHighBid",
        ("FIX.5.0", "269", "O"): "O .. SessionLowOffer",
        ("FIX.5.0", "269", "P"): "P .. EarlyPrices",
        ("FIX.5.0", "269", "Q"): "Q .. AuctionClearingPrice",
        ("FIX.5.0", "269", "S"): "S .. SwapValueFactor",
        ("FIX.5.0", "269", "R"): "R .. DailyValueAdjustmentForLongPositions",
        ("FIX.5.0", "269", "T"): "T .. CumulativeValueAdjustmentForLongPositions",
        ("FIX.5.0", "269", "U"): "U .. DailyValueAdjustmentForShortPositions",
        ("FIX.5.0", "269", "V"): "V .. CumulativeValueAdjustmentForShortPositions",
        ("FIX.5.0", "269", "W"): "W .. FixingPrice",
        ("FIX.5.0", "269", "X"): "X .. CashRate",
        ("FIX.5.0", "269", "Y"): "Y .. RecoveryRate",
        ("FIX.5.0", "269", "Z"): "Z .. RecoveryRateForLong",
        ("FIX.5.0", "269", "a"): "a .. RecoveryRateForShort",
        ("FIX.5.0", "269", "b"): "b .. MarketBid",
        ("FIX.5.0", "269", "c"): "c .. MarketOffer",
        ("FIX.5.0", "269", "d"): "d .. ShortSaleMinPrice",
        ("FIX.5.0", "269", "e"): "e .. PreviousClosingPrice",
        ("FIX.5.0", "274", "0"): "0 .. PlusTick",
        ("FIX.5.0", "274", "1"): "1 .. ZeroPlusTick",
        ("FIX.5.0", "274", "2"): "2 .. MinusTick",
        ("FIX.5.0", "274", "3"): "3 .. ZeroMinusTick",
        ("FIX.5.0", "276", "A"): "A .. Open",
        ("FIX.5.0", "276", "B"): "B .. Closed",
        ("FIX.5.0", "276", "C"): "C .. ExchangeBest",
        ("FIX.5.0", "276", "D"): "D .. ConsolidatedBest",
        ("FIX.5.0", "276", "E"): "E .. Locked",
        ("FIX.5.0", "276", "F"): "F .. Crossed",
        ("FIX.5.0", "276", "G"): "G .. Depth",
        ("FIX.5.0", "276", "H"): "H .. FastTrading",
        ("FIX.5.0", "276", "I"): "I .. NonFirm",
        ("FIX.5.0", "276", "L"): "L .. Manual",
        ("FIX.5.0", "276", "J"): "J .. OutrightPrice",
        ("FIX.5.0", "276", "K"): "K .. ImpliedPrice",
        ("FIX.5.0", "276", "M"): "M .. DepthOnOffer",
        ("FIX.5.0", "276", "N"): "N .. DepthOnBid",
        ("FIX.5.0", "276", "O"): "O .. Closing",
        ("FIX.5.0", "276", "P"): "P .. NewsDissemination",
        ("FIX.5.0", "276", "Q"): "Q .. TradingRange",
        ("FIX.5.0", "276", "R"): "R .. OrderInflux",
        ("FIX.5.0", "276", "S"): "S .. DueToRelated",
        ("FIX.5.0", "276", "T"): "T .. NewsPending",
        ("FIX.5.0", "276", "U"): "U .. AdditionalInfo",
        ("FIX.5.0", "276", "V"): "V .. AdditionalInfoDueToRelated",
        ("FIX.5.0", "276", "W"): "W .. Resume",
        ("FIX.5.0", "276", "X"): "X .. ViewOfCommon",
        ("FIX.5.0", "276", "Y"): "Y .. VolumeAlert",
        ("FIX.5.0", "276", "Z"): "Z .. OrderImbalance",
        ("FIX.5.0", "276", "a"): "a .. EquipmentChangeover",
        ("FIX.5.0", "276", "b"): "b .. NoOpen",
        ("FIX.5.0", "276", "c"): "c .. RegularETH",
        ("FIX.5.0", "276", "d"): "d .. AutomaticExecution",
        ("FIX.5.0", "276", "e"): "e .. AutomaticExecutionETH",
        ("FIX.5.0", "276", "f "): "f  .. FastMarketETH",
        ("FIX.5.0", "276", "g"): "g .. InactiveETH",
        ("FIX.5.0", "276", "h"): "h .. Rotation",
        ("FIX.5.0", "276", "i"): "i .. RotationETH",
        ("FIX.5.0", "276", "j"): "j .. Halt",
        ("FIX.5.0", "276", "k"): "k .. HaltETH",
        ("FIX.5.0", "276", "l"): "l .. DueToNewsDissemination",
        ("FIX.5.0", "276", "m"): "m .. DueToNewsPending",
        ("FIX.5.0", "276", "n"): "n .. TradingResume",
        ("FIX.5.0", "276", "o"): "o .. OutOfSequence",
        ("FIX.5.0", "276", "p"): "p .. BidSpecialist",
        ("FIX.5.0", "276", "q"): "q .. OfferSpecialist",
        ("FIX.5.0", "276", "r"): "r .. BidOfferSpecialist",
        ("FIX.5.0", "276", "s"): "s .. EndOfDaySAM",
        ("FIX.5.0", "276", "t"): "t .. ForbiddenSAM",
        ("FIX.5.0", "276", "u"): "u .. FrozenSAM",
        ("FIX.5.0", "276", "v"): "v .. PreOpeningSAM",
        ("FIX.5.0", "276", "w"): "w .. OpeningSAM",
        ("FIX.5.0", "276", "x"): "x .. OpenSAM",
        ("FIX.5.0", "276", "y"): "y .. SurveillanceSAM",
        ("FIX.5.0", "276", "z"): "z .. SuspendedSAM",
        ("FIX.5.0", "276", "0"): "0 .. ReservedSAM",
        ("FIX.5.0", "276", "1"): "1 .. NoActiveSAM",
        ("FIX.5.0", "276", "2"): "2 .. Restricted",
        ("FIX.5.0", "276", "3"): "3 .. RestOfBookVWAP",
        ("FIX.5.0", "276", "4"): "4 .. BetterPricesInConditionalOrders",
        ("FIX.5.0", "276", "5"): "5 .. MedianPrice",
        ("FIX.5.0", "276", "6"): "6 .. FullCurve",
        ("FIX.5.0", "276", "7"): "7 .. FlatCurve",
        ("FIX.5.0", "277", "A"): "A .. Cash",
        ("FIX.5.0", "277", "B"): "B .. AveragePriceTrade",
        ("FIX.5.0", "277", "C"): "C .. CashTrade",
        ("FIX.5.0", "277", "D"): "D .. NextDay",
        ("FIX.5.0", "277", "E"): "E .. Opening",
        ("FIX.5.0", "277", "F"): "F .. IntradayTradeDetail",
        ("FIX.5.0", "277", "G"): "G .. Rule127Trade",
        ("FIX.5.0", "277", "H"): "H .. Rule155Trade",
        ("FIX.5.0", "277", "I"): "I .. SoldLast",
        ("FIX.5.0", "277", "J"): "J .. NextDayTrade",
        ("FIX.5.0", "277", "K"): "K .. Opened",
        ("FIX.5.0", "277", "L"): "L .. Seller",
        ("FIX.5.0", "277", "M"): "M .. Sold",
        ("FIX.5.0", "277", "N"): "N .. StoppedStock",
        ("FIX.5.0", "277", "P"): "P .. ImbalanceMoreBuyers",
        ("FIX.5.0", "277", "Q"): "Q .. ImbalanceMoreSellers",
        ("FIX.5.0", "277", "R"): "R .. OpeningPrice",
        ("FIX.5.0", "277", "S"): "S .. BargainCondition",
        ("FIX.5.0", "277", "T"): "T .. ConvertedPriceIndicator",
        ("FIX.5.0", "277", "U"): "U .. ExchangeLast",
        ("FIX.5.0", "277", "V"): "V .. FinalPriceOfSession",
        ("FIX.5.0", "277", "W"): "W .. ExPit",
        ("FIX.5.0", "277", "X"): "X .. Crossed",
        ("FIX.5.0", "277", "Y"): "Y .. TradesResultingFromManual",
        ("FIX.5.0", "277", "Z"): "Z .. TradesResultingFromIntermarketSweep",
        ("FIX.5.0", "277", "a"): "a .. VolumeOnly",
        ("FIX.5.0", "277", "b"): "b .. DirectPlus",
        ("FIX.5.0", "277", "c"): "c .. Acquisition",
        ("FIX.5.0", "277", "d"): "d .. Bunched",
        ("FIX.5.0", "277", "e"): "e .. Distribution",
        ("FIX.5.0", "277", "f"): "f .. BunchedSale",
        ("FIX.5.0", "277", "g"): "g .. SplitTrade",
        ("FIX.5.0", "277", "h"): "h .. CancelStopped",
        ("FIX.5.0", "277", "i"): "i .. CancelETH",
        ("FIX.5.0", "277", "j"): "j .. CancelStoppedETH",
        ("FIX.5.0", "277", "k"): "k .. OutOfSequenceETH",
        ("FIX.5.0", "277", "l"): "l .. CancelLastETH",
        ("FIX.5.0", "277", "m"): "m .. SoldLastSaleETH",
        ("FIX.5.0", "277", "n"): "n .. CancelLast",
        ("FIX.5.0", "277", "o"): "o .. SoldLastSale",
        ("FIX.5.0", "277", "p"): "p .. CancelOpen",
        ("FIX.5.0", "277", "q"): "q .. CancelOpenETH",
        ("FIX.5.0", "277", "r"): "r .. OpenedSaleETH",
        ("FIX.5.0", "277", "s"): "s .. CancelOnly",
        ("FIX.5.0", "277", "t"): "t .. CancelOnlyETH",
        ("FIX.5.0", "277", "u"): "u .. LateOpenETH",
        ("FIX.5.0", "277", "v"): "v .. AutoExecutionETH",
        ("FIX.5.0", "277", "w"): "w .. Reopen",
        ("FIX.5.0", "277", "x"): "x .. ReopenETH",
        ("FIX.5.0", "277", "y"): "y .. Adjusted",
        ("FIX.5.0", "277", "z"): "z .. AdjustedETH",
        ("FIX.5.0", "277", "AA"): "AA .. Spread",
        ("FIX.5.0", "277", "AB"): "AB .. SpreadETH",
        ("FIX.5.0", "277", "AC"): "AC .. Straddle",
        ("FIX.5.0", "277", "AD"): "AD .. StraddleETH",
        ("FIX.5.0", "277", "AE"): "AE .. Stopped",
        ("FIX.5.0", "277", "AF"): "AF .. StoppedETH",
        ("FIX.5.0", "277", "AG"): "AG .. RegularETH",
        ("FIX.5.0", "277", "AH"): "AH .. Combo",
        ("FIX.5.0", "277", "AI"): "AI .. ComboETH",
        ("FIX.5.0", "277", "AJ"): "AJ .. OfficialClosingPrice",
        ("FIX.5.0", "277", "AK"): "AK .. PriorReferencePrice",
        ("FIX.5.0", "277", "0"): "0 .. Cancel",
        ("FIX.5.0", "277", "AL"): "AL .. StoppedSoldLast",
        ("FIX.5.0", "277", "AM"): "AM .. StoppedOutOfSequence",
        ("FIX.5.0", "277", "AN"): "AN .. OfficalClosingPrice",
        ("FIX.5.0", "277", "AO"): "AO .. CrossedOld",
        ("FIX.5.0", "277", "AP"): "AP .. FastMarket",
        ("FIX.5.0", "277", "AQ"): "AQ .. AutomaticExecution",
        ("FIX.5.0", "277", "AR"): "AR .. FormT",
        ("FIX.5.0", "277", "AS"): "AS .. BasketIndex",
        ("FIX.5.0", "277", "AT"): "AT .. BurstBasket",
        ("FIX.5.0", "277", "AU"): "AU .. TradeThroughExempt",
        ("FIX.5.0", "277", "AV"): "AV .. QuoteSpread",
        ("FIX.5.0", "277", "AW"): "AW .. LastAuctionPrice",
        ("FIX.5.0", "277", "AX"): "AX .. HighPrice",
        ("FIX.5.0", "277", "AY"): "AY .. LowPrice",
        ("FIX.5.0", "277", "AZ"): "AZ .. SystematicInternaliser",
        ("FIX.5.0", "277", "BA"): "BA .. AwayMarket",
        ("FIX.5.0", "277", "BB"): "BB .. MidpointPrice",
        ("FIX.5.0", "277", "BC"): "BC .. TradedBeforeIssueDate",
        ("FIX.5.0", "277", "BD"): "BD .. PreviousClosingPrice",
        ("FIX.5.0", "277", "BE"): "BE .. NationalBestBidOffer",
        ("FIX.5.0", "277", "1"): "1 .. ImpliedTrade",
        ("FIX.5.0", "277", "2"): "2 .. MarketplaceEnteredTrade",
        ("FIX.5.0", "277", "3"): "3 .. MultiAssetClassMultilegTrade",
        ("FIX.5.0", "277", "4"): "4 .. MultilegToMultilegTrade",
        ("FIX.5.0", "277", "5"): "5 .. ShortSaleMinPrice",
        ("FIX.5.0", "277", "6"): "6 .. Benchmark",
        ("FIX.5.0", "279", "0"): "0 .. New",
        ("FIX.5.0", "279", "1"): "1 .. Change",
        ("FIX.5.0", "279", "2"): "2 .. Delete",
        ("FIX.5.0", "279", "3"): "3 .. DeleteThru",
        ("FIX.5.0", "279", "4"): "4 .. DeleteFrom",
        ("FIX.5.0", "279", "5"): "5 .. Overlay",
        ("FIX.5.0", "281", "0"): "0 .. UnknownSymbol",
        ("FIX.5.0", "281", "1"): "1 .. DuplicateMDReqID",
        ("FIX.5.0", "281", "2"): "2 .. InsufficientBandwidth",
        ("FIX.5.0", "281", "3"): "3 .. InsufficientPermissions",
        ("FIX.5.0", "281", "4"): "4 .. UnsupportedSubscriptionRequestType",
        ("FIX.5.0", "281", "5"): "5 .. UnsupportedMarketDepth",
        ("FIX.5.0", "281", "6"): "6 .. UnsupportedMDUpdateType",
        ("FIX.5.0", "281", "7"): "7 .. UnsupportedAggregatedBook",
        ("FIX.5.0", "281", "8"): "8 .. UnsupportedMDEntryType",
        ("FIX.5.0", "281", "9"): "9 .. UnsupportedTradingSessionID",
        ("FIX.5.0", "281", "A"): "A .. UnsupportedScope",
        ("FIX.5.0", "281", "B"): "B .. UnsupportedOpenCloseSettleFlag",
        ("FIX.5.0", "281", "C"): "C .. UnsupportedMDImplicitDelete",
        ("FIX.5.0", "281", "D"): "D .. InsufficientCredit",
        ("FIX.5.0", "285", "0"): "0 .. Cancellation",
        ("FIX.5.0", "285", "1"): "1 .. Error",
        ("FIX.5.0", "286", "0"): "0 .. DailyOpen",
        ("FIX.5.0", "286", "1"): "1 .. SessionOpen",
        ("FIX.5.0", "286", "2"): "2 .. DeliverySettlementEntry",
        ("FIX.5.0", "286", "3"): "3 .. ExpectedEntry",
        ("FIX.5.0", "286", "4"): "4 .. EntryFromPreviousBusinessDay",
        ("FIX.5.0", "286", "5"): "5 .. TheoreticalPriceValue",
        ("FIX.5.0", "291", "1"): "1 .. Bankrupt",
        ("FIX.5.0", "291", "2"): "2 .. PendingDelisting",
        ("FIX.5.0", "291", "3"): "3 .. Restricted",
        ("FIX.5.0", "292", "A"): "A .. ExDividend",
        ("FIX.5.0", "292", "B"): "B .. ExDistribution",
        ("FIX.5.0", "292", "C"): "C .. ExRights",
        ("FIX.5.0", "292", "D"): "D .. New",
        ("FIX.5.0", "292", "E"): "E .. ExInterest",
        ("FIX.5.0", "292", "F"): "F .. CashDividend",
        ("FIX.5.0", "292", "G"): "G .. StockDividend",
        ("FIX.5.0", "292", "H"): "H .. NonIntegerStockSplit",
        ("FIX.5.0", "292", "I"): "I .. ReverseStockSplit",
        ("FIX.5.0", "292", "J"): "J .. StandardIntegerStockSplit",
        ("FIX.5.0", "292", "K"): "K .. PositionConsolidation",
        ("FIX.5.0", "292", "L"): "L .. LiquidationReorganization",
        ("FIX.5.0", "292", "M"): "M .. MergerReorganization",
        ("FIX.5.0", "292", "N"): "N .. RightsOffering",
        ("FIX.5.0", "292", "O"): "O .. ShareholderMeeting",
        ("FIX.5.0", "292", "P"): "P .. Spinoff",
        ("FIX.5.0", "292", "Q"): "Q .. TenderOffer",
        ("FIX.5.0", "292", "R"): "R .. Warrant",
        ("FIX.5.0", "292", "S"): "S .. SpecialAction",
        ("FIX.5.0", "292", "T"): "T .. SymbolConversion",
        ("FIX.5.0", "292", "U"): "U .. CUSIP",
        ("FIX.5.0", "292", "V"): "V .. LeapRollover",
        ("FIX.5.0", "292", "W"): "W .. SuccessionEvent",
        ("FIX.5.0", "297", "0"): "0 .. Accepted",
        ("FIX.5.0", "297", "5"): "5 .. Rejected",
        ("FIX.5.0", "297", "6"): "6 .. RemovedFromMarket",
        ("FIX.5.0", "297", "7"): "7 .. Expired",
        ("FIX.5.0", "297", "8"): "8 .. Query",
        ("FIX.5.0", "297", "9"): "9 .. QuoteNotFound",
        ("FIX.5.0", "297", "10"): "10 .. Pending",
        ("FIX.5.0", "297", "11"): "11 .. Pass",
        ("FIX.5.0", "297", "12"): "12 .. LockedMarketWarning",
        ("FIX.5.0", "297", "13"): "13 .. CrossMarketWarning",
        ("FIX.5.0", "297", "14"): "14 .. CanceledDueToLockMarket",
        ("FIX.5.0", "297", "15"): "15 .. CanceledDueToCrossMarket",
        ("FIX.5.0", "297", "16"): "16 .. Active",
        ("FIX.5.0", "297", "17"): "17 .. Canceled",
        ("FIX.5.0", "297", "18"): "18 .. UnsolicitedQuoteReplenishment",
        ("FIX.5.0", "297", "19"): "19 .. PendingEndTrade",
        ("FIX.5.0", "297", "20"): "20 .. TooLateToEnd",
        ("FIX.5.0", "297", "21"): "21 .. Traded",
        ("FIX.5.0", "297", "22"): "22 .. TradedAndRemoved",
        ("FIX.5.0", "298", "1"): "1 .. CancelForOneOrMoreSecurities",
        ("FIX.5.0", "298", "2"): "2 .. CancelForSecurityType",
        ("FIX.5.0", "298", "3"): "3 .. CancelForUnderlyingSecurity",
        ("FIX.5.0", "298", "4"): "4 .. CancelAllQuotes",
        ("FIX.5.0", "298", "5"): "5 .. CancelSpecifiedSingleQuote",
        ("FIX.5.0", "298", "6"): "6 .. CancelByTypeOfQuote",
        ("FIX.5.0", "298", "7"): "7 .. CancelForSecurityIssuer",
        ("FIX.5.0", "298", "8"): "8 .. CancelForIssuerOfUnderlyingSecurity",
        ("FIX.5.0", "300", "1"): "1 .. UnknownSymbol",
        ("FIX.5.0", "300", "2"): "2 .. Exchange",
        ("FIX.5.0", "300", "3"): "3 .. QuoteRequestExceedsLimit",
        ("FIX.5.0", "300", "4"): "4 .. TooLateToEnter",
        ("FIX.5.0", "300", "5"): "5 .. UnknownQuote",
        ("FIX.5.0", "300", "6"): "6 .. DuplicateQuote",
        ("FIX.5.0", "300", "7"): "7 .. InvalidBid",
        ("FIX.5.0", "300", "8"): "8 .. InvalidPrice",
        ("FIX.5.0", "300", "9"): "9 .. NotAuthorizedToQuoteSecurity",
        ("FIX.5.0", "300", "10"): "10 .. PriceExceedsCurrentPriceBand",
        ("FIX.5.0", "300", "11"): "11 .. QuoteLocked",
        ("FIX.5.0", "300", "12"): "12 .. InvalidOrUnknownSecurityIssuer",
        ("FIX.5.0", "300", "13"): "13 .. InvalidOrUnknownIssuerOfUnderlyingSecurity",
        ("FIX.5.0", "300", "14"): "14 .. NotionalValueExceedsThreshold",
        ("FIX.5.0", "300", "16"): "16 .. ReferencePriceNotAvailable",
        ("FIX.5.0", "300", "17"): "17 .. InsufficientCreditLimit",
        ("FIX.5.0", "300", "18"): "18 .. ExceededClipSizeLimit",
        ("FIX.5.0", "300", "19"): "19 .. ExceededMaxNotionalOrderAmt",
        ("FIX.5.0", "300", "20"): "20 .. ExceededDV01PV01Limit",
        ("FIX.5.0", "300", "21"): "21 .. ExceededCS01Limit",
        ("FIX.5.0", "300", "99"): "99 .. Other",
        ("FIX.5.0", "301", "0"): "0 .. NoAcknowledgement",
        ("FIX.5.0", "301", "1"): "1 .. AcknowledgeOnlyNegativeOrErroneousQuotes",
        ("FIX.5.0", "301", "2"): "2 .. AcknowledgeEachQuoteMessage",
        ("FIX.5.0", "301", "3"): "3 .. SummaryAcknowledgement",
        ("FIX.5.0", "303", "1"): "1 .. Manual",
        ("FIX.5.0", "303", "2"): "2 .. Automatic",
        ("FIX.5.0", "303", "3"): "3 .. ConfirmQuote",
        ("FIX.5.0", "305", "1"): "1 .. CUSIP",
        ("FIX.5.0", "305", "2"): "2 .. SEDOL",
        ("FIX.5.0", "305", "3"): "3 .. QUIK",
        ("FIX.5.0", "305", "4"): "4 .. ISINNumber",
        ("FIX.5.0", "305", "5"): "5 .. RICCode",
        ("FIX.5.0", "305", "6"): "6 .. ISOCurrencyCode",
        ("FIX.5.0", "305", "7"): "7 .. ISOCountryCode",
        ("FIX.5.0", "305", "8"): "8 .. ExchangeSymbol",
        ("FIX.5.0", "305", "9"): "9 .. ConsolidatedTapeAssociation",
        ("FIX.5.0", "305", "A"): "A .. BloombergSymbol",
        ("FIX.5.0", "305", "B"): "B .. Wertpapier",
        ("FIX.5.0", "305", "C"): "C .. Dutch",
        ("FIX.5.0", "305", "D"): "D .. Valoren",
        ("FIX.5.0", "305", "E"): "E .. Sicovam",
        ("FIX.5.0", "305", "F"): "F .. Belgian",
        ("FIX.5.0", "305", "G"): "G .. Common",
        ("FIX.5.0", "305", "H"): "H .. ClearingHouse",
        ("FIX.5.0", "305", "I"): "I .. ISDAFpMLSpecification",
        ("FIX.5.0", "305", "J"): "J .. OptionPriceReportingAuthority",
        ("FIX.5.0", "305", "K"): "K .. ISDAFpMLURL",
        ("FIX.5.0", "305", "L"): "L .. LetterOfCredit",
        ("FIX.5.0", "305", "M"): "M .. MarketplaceAssignedIdentifier",
        ("FIX.5.0", "305", "N"): "N .. MarkitREDEntityCLIP",
        ("FIX.5.0", "305", "P"): "P .. MarkitREDPairCLIP",
        ("FIX.5.0", "305", "Q"): "Q .. CFTCCommodityCode",
        ("FIX.5.0", "305", "R"): "R .. ISDACommodityReferencePrice",
        ("FIX.5.0", "305", "S"): "S .. FinancialInstrumentGlobalIdentifier",
        ("FIX.5.0", "305", "T"): "T .. LegalEntityIdentifier",
        ("FIX.5.0", "305", "U"): "U .. Synthetic",
        ("FIX.5.0", "305", "V"): "V .. FidessaInstrumentMnemonic",
        ("FIX.5.0", "305", "W"): "W .. IndexName",
        ("FIX.5.0", "310", "EUSUPRA"): "EUSUPRA .. EuroSupranationalCoupons",
        ("FIX.5.0", "310", "FAC"): "FAC .. FederalAgencyCoupon",
        ("FIX.5.0", "310", "FADN"): "FADN .. FederalAgencyDiscountNote",
        ("FIX.5.0", "310", "PEF"): "PEF .. PrivateExportFunding",
        ("FIX.5.0", "310", "SUPRA"): "SUPRA .. USDSupranationalCoupons",
        ("FIX.5.0", "310", "CORP"): "CORP .. CorporateBond",
        ("FIX.5.0", "310", "CPP"): "CPP .. CorporatePrivatePlacement",
        ("FIX.5.0", "310", "CB"): "CB .. ConvertibleBond",
        ("FIX.5.0", "310", "DUAL"): "DUAL .. DualCurrency",
        ("FIX.5.0", "310", "EUCORP"): "EUCORP .. EuroCorporateBond",
        ("FIX.5.0", "310", "EUFRN"): "EUFRN .. EuroCorporateFloatingRateNotes",
        ("FIX.5.0", "310", "FRN"): "FRN .. USCorporateFloatingRateNotes",
        ("FIX.5.0", "310", "XLINKD"): "XLINKD .. IndexedLinked",
        ("FIX.5.0", "310", "STRUCT"): "STRUCT .. StructuredNotes",
        ("FIX.5.0", "310", "YANK"): "YANK .. YankeeCorporateBond",
        ("FIX.5.0", "310", "FXNDF"): "FXNDF .. NonDeliverableForward",
        ("FIX.5.0", "310", "FXSPOT"): "FXSPOT .. FXSpot",
        ("FIX.5.0", "310", "FXFWD"): "FXFWD .. FXForward",
        ("FIX.5.0", "310", "FXSWAP"): "FXSWAP .. FXSwap",
        ("FIX.5.0", "310", "FXNDS"): "FXNDS .. NonDeliverableSwap",
        ("FIX.5.0", "310", "CAP"): "CAP .. Cap",
        ("FIX.5.0", "310", "CDS"): "CDS .. CreditDefaultSwap",
        ("FIX.5.0", "310", "CLLR"): "CLLR .. Collar",
        ("FIX.5.0", "310", "CMDTYSWAP"): "CMDTYSWAP .. CommoditySwap",
        ("FIX.5.0", "310", "EXOTIC"): "EXOTIC .. Exotic",
        ("FIX.5.0", "310", "OOC"): "OOC .. OptionsOnCombo",
        ("FIX.5.0", "310", "FLR"): "FLR .. Floor",
        ("FIX.5.0", "310", "FRA"): "FRA .. FRA",
        ("FIX.5.0", "310", "FUT"): "FUT .. Future",
        ("FIX.5.0", "310", "FWD"): "FWD .. DerivativeForward",
        ("FIX.5.0", "310", "IRS"): "IRS .. InterestRateSwap",
        ("FIX.5.0", "310", "TRS"): "TRS .. TotalReturnSwap",
        ("FIX.5.0", "310", "LOANLEASE"): "LOANLEASE .. LoanLease",
        ("FIX.5.0", "310", "OOF"): "OOF .. OptionsOnFutures",
        ("FIX.5.0", "310", "OOP"): "OOP .. OptionsOnPhysical",
        ("FIX.5.0", "310", "OPT"): "OPT .. Option",
        ("FIX.5.0", "310", "SPOTFWD"): "SPOTFWD .. SpotForward",
        ("FIX.5.0", "310", "SWAPTION"): "SWAPTION .. SwapOption",
        ("FIX.5.0", "310", "XMISSION"): "XMISSION .. Transmission",
        ("FIX.5.0", "310", "INDEX"): "INDEX .. Index",
        ("FIX.5.0", "310", "BDBSKT"): "BDBSKT .. BondBasket",
        ("FIX.5.0", "310", "CFD"): "CFD .. ContractForDifference",
        ("FIX.5.0", "310", "CRLTNSWAP"): "CRLTNSWAP .. CorrelationSwap",
        ("FIX.5.0", "310", "DVDNDSWAP"): "DVDNDSWAP .. DiviendSwap",
        ("FIX.5.0", "310", "EQBSKT"): "EQBSKT .. EquityBasket",
        ("FIX.5.0", "310", "EQFWD"): "EQFWD .. EquityForward",
        ("FIX.5.0", "310", "RTRNSWAP"): "RTRNSWAP .. ReturnSwap",
        ("FIX.5.0", "310", "VARSWAP"): "VARSWAP .. VarianceSwap",
        ("FIX.5.0", "310", "PRTFLIOSWAP"): "PRTFLIOSWAP .. PortfolioSwaps",
        ("FIX.5.0", "310", "FUTSWAP"): "FUTSWAP .. FuturesOnASwap",
        ("FIX.5.0", "310", "FWDSWAP "): "FWDSWAP  .. ForwardsOnASwap",
        ("FIX.5.0", "310", "FWDFRTAGMT "): "FWDFRTAGMT  .. ForwardFreightAgreement",
        ("FIX.5.0", "310", "SPREADBET"): "SPREADBET .. SpreadBetting",
        ("FIX.5.0", "310", "ETC"): "ETC .. ExchangeTradedCommodity",
        ("FIX.5.0", "310", "CS"): "CS .. CommonStock",
        ("FIX.5.0", "310", "PS"): "PS .. PreferredStock",
        ("FIX.5.0", "310", "DR"): "DR .. DepositoryReceipts",
        ("FIX.5.0", "310", "REPO"): "REPO .. Repurchase",
        ("FIX.5.0", "310", "FORWARD"): "FORWARD .. Forward",
        ("FIX.5.0", "310", "BUYSELL"): "BUYSELL .. BuySellback",
        ("FIX.5.0", "310", "SECLOAN"): "SECLOAN .. SecuritiesLoan",
        ("FIX.5.0", "310", "SECPLEDGE"): "SECPLEDGE .. SecuritiesPledge",
        ("FIX.5.0", "310", "DVPLDG"): "DVPLDG .. DeliveryVersusPledge",
        ("FIX.5.0", "310", "COLLBSKT"): "COLLBSKT .. CollateralBasket",
        ("FIX.5.0", "310", "SFP"): "SFP .. StructuredFinanceProduct",
        ("FIX.5.0", "310", "BRADY"): "BRADY .. BradyBond",
        ("FIX.5.0", "310", "CAN"): "CAN .. CanadianTreasuryNotes",
        ("FIX.5.0", "310", "CTB"): "CTB .. CanadianTreasuryBills",
        ("FIX.5.0", "310", "EUSOV"): "EUSOV .. EuroSovereigns",
        ("FIX.5.0", "310", "PROV"): "PROV .. CanadianProvincialBonds",
        ("FIX.5.0", "310", "TB"): "TB .. TreasuryBill",
        ("FIX.5.0", "310", "TBOND"): "TBOND .. USTreasuryBond",
        ("FIX.5.0", "310", "TINT"): "TINT .. InterestStripFromAnyBondOrNote",
        ("FIX.5.0", "310", "TBILL"): "TBILL .. USTreasuryBill",
        ("FIX.5.0", "310", "TIPS"): "TIPS .. TreasuryInflationProtectedSecurities",
        ("FIX.5.0", "310", "TCAL"): "TCAL .. PrincipalStripOfACallableBondOrNote",
        ("FIX.5.0", "310", "TPRN"): "TPRN .. PrincipalStripFromANonCallableBondOrNote",
        ("FIX.5.0", "310", "TNOTE"): "TNOTE .. USTreasuryNote",
        ("FIX.5.0", "310", "TERM"): "TERM .. TermLoan",
        ("FIX.5.0", "310", "RVLV"): "RVLV .. RevolverLoan",
        ("FIX.5.0", "310", "RVLVTRM"): "RVLVTRM .. Revolver",
        ("FIX.5.0", "310", "BRIDGE"): "BRIDGE .. BridgeLoan",
        ("FIX.5.0", "310", "LOFC"): "LOFC .. LetterOfCredit",
        ("FIX.5.0", "310", "SWING"): "SWING .. SwingLineFacility",
        ("FIX.5.0", "310", "DINP"): "DINP .. DebtorInPossession",
        ("FIX.5.0", "310", "DEFLTED"): "DEFLTED .. Defaulted",
        ("FIX.5.0", "310", "WITHDRN"): "WITHDRN .. Withdrawn",
        ("FIX.5.0", "310", "REPLACD"): "REPLACD .. Replaced",
        ("FIX.5.0", "310", "MATURED"): "MATURED .. Matured",
        ("FIX.5.0", "310", "AMENDED"): "AMENDED .. Amended",
        ("FIX.5.0", "310", "RETIRED"): "RETIRED .. Retired",
        ("FIX.5.0", "310", "BA"): "BA .. BankersAcceptance",
        ("FIX.5.0", "310", "BDN"): "BDN .. BankDepositoryNote",
        ("FIX.5.0", "310", "BN"): "BN .. BankNotes",
        ("FIX.5.0", "310", "BOX"): "BOX .. BillOfExchanges",
        ("FIX.5.0", "310", "CAMM"): "CAMM .. CanadianMoneyMarkets",
        ("FIX.5.0", "310", "CD"): "CD .. CertificateOfDeposit",
        ("FIX.5.0", "310", "CL"): "CL .. CallLoans",
        ("FIX.5.0", "310", "CP"): "CP .. CommercialPaper",
        ("FIX.5.0", "310", "DN"): "DN .. DepositNotes",
        ("FIX.5.0", "310", "EUCD"): "EUCD .. EuroCertificateOfDeposit",
        ("FIX.5.0", "310", "EUCP"): "EUCP .. EuroCommercialPaper",
        ("FIX.5.0", "310", "LQN"): "LQN .. LiquidityNote",
        ("FIX.5.0", "310", "MTN"): "MTN .. MediumTermNotes",
        ("FIX.5.0", "310", "ONITE"): "ONITE .. Overnight",
        ("FIX.5.0", "310", "PN"): "PN .. PromissoryNote",
        ("FIX.5.0", "310", "STN"): "STN .. ShortTermLoanNote",
        ("FIX.5.0", "310", "PZFJ"): "PZFJ .. PlazosFijos",
        ("FIX.5.0", "310", "SLQN"): "SLQN .. SecuredLiquidityNote",
        ("FIX.5.0", "310", "TD"): "TD .. TimeDeposit",
        ("FIX.5.0", "310", "TLQN"): "TLQN .. TermLiquidityNote",
        ("FIX.5.0", "310", "XCN"): "XCN .. ExtendedCommNote",
        ("FIX.5.0", "310", "YCD"): "YCD .. YankeeCertificateOfDeposit",
        ("FIX.5.0", "310", "ABS"): "ABS .. AssetBackedSecurities",
        ("FIX.5.0", "310", "CMB"): "CMB .. CanadianMortgageBonds",
        ("FIX.5.0", "310", "CMBS"): "CMBS .. Corp",
        ("FIX.5.0", "310", "CMO"): "CMO .. CollateralizedMortgageObligation",
        ("FIX.5.0", "310", "IET"): "IET .. IOETTEMortgage",
        ("FIX.5.0", "310", "MBS"): "MBS .. MortgageBackedSecurities",
        ("FIX.5.0", "310", "MIO"): "MIO .. MortgageInterestOnly",
        ("FIX.5.0", "310", "MPO"): "MPO .. MortgagePrincipalOnly",
        ("FIX.5.0", "310", "MPP"): "MPP .. MortgagePrivatePlacement",
        ("FIX.5.0", "310", "MPT"): "MPT .. MiscellaneousPassThrough",
        ("FIX.5.0", "310", "PFAND"): "PFAND .. Pfandbriefe",
        ("FIX.5.0", "310", "TBA"): "TBA .. ToBeAnnounced",
        ("FIX.5.0", "310", "AN"): "AN .. OtherAnticipationNotes",
        ("FIX.5.0", "310", "COFO"): "COFO .. CertificateOfObligation",
        ("FIX.5.0", "310", "COFP"): "COFP .. CertificateOfParticipation",
        ("FIX.5.0", "310", "GO"): "GO .. GeneralObligationBonds",
        ("FIX.5.0", "310", "MT"): "MT .. MandatoryTender",
        ("FIX.5.0", "310", "RAN"): "RAN .. RevenueAnticipationNote",
        ("FIX.5.0", "310", "REV"): "REV .. RevenueBonds",
        ("FIX.5.0", "310", "SPCLA"): "SPCLA .. SpecialAssessment",
        ("FIX.5.0", "310", "SPCLO"): "SPCLO .. SpecialObligation",
        ("FIX.5.0", "310", "SPCLT"): "SPCLT .. SpecialTax",
        ("FIX.5.0", "310", "TAN"): "TAN .. TaxAnticipationNote",
        ("FIX.5.0", "310", "TAXA"): "TAXA .. TaxAllocation",
        ("FIX.5.0", "310", "TECP"): "TECP .. TaxExemptCommercialPaper",
        ("FIX.5.0", "310", "TMCP"): "TMCP .. TaxableMunicipalCP",
        ("FIX.5.0", "310", "TRAN"): "TRAN .. TaxRevenueAnticipationNote",
        ("FIX.5.0", "310", "VRDN"): "VRDN .. VariableRateDemandNote",
        ("FIX.5.0", "310", "WAR"): "WAR .. Warrant",
        ("FIX.5.0", "310", "MF"): "MF .. MutualFund",
        ("FIX.5.0", "310", "MLEG"): "MLEG .. MultilegInstrument",
        ("FIX.5.0", "310", "NONE"): "NONE .. NoSecurityType",
        ("FIX.5.0", "310", "?"): "? .. Wildcard",
        ("FIX.5.0", "310", "CASH"): "CASH .. Cash",
        ("FIX.5.0", "310", "Other"): "Other .. Other",
        ("FIX.5.0", "310", "ETN"): "ETN .. ExchangeTradedNote",
        ("FIX.5.0", "310", "SECDERIV"): "SECDERIV .. SecuritizedDerivative",
        ("FIX.5.0", "312", "CD"): "CD .. EUCPWithLumpSumInterest",
        ("FIX.5.0", "312", "WI"): "WI .. WhenIssued",
        ("FIX.5.0", "315", "0"): "0 .. Put",
        ("FIX.5.0", "315", "1"): "1 .. Call",
        ("FIX.5.0", "315", "2"): "2 .. Other",
        ("FIX.5.0", "315", "3"): "3 .. Chooser",
        ("FIX.5.0", "321", "0"): "0 .. RequestSecurityIdentityAndSpecifications",
        ("FIX.5.0", "321", "1"): "1 .. RequestSecurityIdentityForSpecifications",
        ("FIX.5.0", "321", "4"): "4 .. Symbol",
        ("FIX.5.0", "321", "5"): "5 .. SecurityTypeAndOrCFICode",
        ("FIX.5.0", "321", "6"): "6 .. Product",
        ("FIX.5.0", "321", "7"): "7 .. TradingSessionID",
        ("FIX.5.0", "321", "8"): "8 .. AllSecurities",
        ("FIX.5.0", "321", "9"): "9 .. MarketIDOrMarketID",
        ("FIX.5.0", "323", "1"): "1 .. AcceptAsIs",
        ("FIX.5.0", "323", "2"): "2 .. AcceptWithRevisions",
        ("FIX.5.0", "323", "4"): "4 .. ListOfSecuritiesReturnedPerRequest",
        ("FIX.5.0", "323", "5"): "5 .. RejectSecurityProposal",
        ("FIX.5.0", "323", "6"): "6 .. CannotMatchSelectionCriteria",
        ("FIX.5.0", "325", "N"): "N .. MessageIsBeingSentAsAResultOfAPriorRequest",
        ("FIX.5.0", "325", "Y"): "Y .. MessageIsBeingSentUnsolicited",
        ("FIX.5.0", "326", "1"): "1 .. OpeningDelay",
        ("FIX.5.0", "326", "2"): "2 .. TradingHalt",
        ("FIX.5.0", "326", "3"): "3 .. Resume",
        ("FIX.5.0", "326", "4"): "4 .. NoOpen",
        ("FIX.5.0", "326", "5"): "5 .. PriceIndication",
        ("FIX.5.0", "326", "6"): "6 .. TradingRangeIndication",
        ("FIX.5.0", "326", "7"): "7 .. MarketImbalanceBuy",
        ("FIX.5.0", "326", "8"): "8 .. MarketImbalanceSell",
        ("FIX.5.0", "326", "9"): "9 .. MarketOnCloseImbalanceBuy",
        ("FIX.5.0", "326", "10"): "10 .. MarketOnCloseImbalanceSell",
        ("FIX.5.0", "326", "12"): "12 .. NoMarketImbalance",
        ("FIX.5.0", "326", "13"): "13 .. NoMarketOnCloseImbalance",
        ("FIX.5.0", "326", "14"): "14 .. ITSPreOpening",
        ("FIX.5.0", "326", "15"): "15 .. NewPriceIndication",
        ("FIX.5.0", "326", "16"): "16 .. TradeDisseminationTime",
        ("FIX.5.0", "326", "17"): "17 .. ReadyToTrade",
        ("FIX.5.0", "326", "18"): "18 .. NotAvailableForTrading",
        ("FIX.5.0", "326", "19"): "19 .. NotTradedOnThisMarket",
        ("FIX.5.0", "326", "20"): "20 .. UnknownOrInvalid",
        ("FIX.5.0", "326", "21"): "21 .. PreOpen",
        ("FIX.5.0", "326", "22"): "22 .. OpeningRotation",
        ("FIX.5.0", "326", "23"): "23 .. FastMarket",
        ("FIX.5.0", "326", "24"): "24 .. PreCross",
        ("FIX.5.0", "326", "25"): "25 .. Cross",
        ("FIX.5.0", "326", "26"): "26 .. PostClose",
        ("FIX.5.0", "326", "27"): "27 .. NoCancel",
        ("FIX.5.0", "327", "0"): "0 .. NewsDissemination",
        ("FIX.5.0", "327", "1"): "1 .. OrderInflux",
        ("FIX.5.0", "327", "2"): "2 .. OrderImbalance",
        ("FIX.5.0", "327", "3"): "3 .. AdditionalInformation",
        ("FIX.5.0", "327", "4"): "4 .. NewsPending",
        ("FIX.5.0", "327", "5"): "5 .. EquipmentChangeover",
        ("FIX.5.0", "328", "N"): "N .. HaltWasNotRelatedToAHaltOfTheCommonStock",
        ("FIX.5.0", "328", "Y"): "Y .. HaltWasDueToCommonStockBeingHalted",
        ("FIX.5.0", "329", "N"): "N .. NotRelatedToSecurityHalt",
        ("FIX.5.0", "329", "Y"): "Y .. RelatedToSecurityHalt",
        ("FIX.5.0", "334", "1"): "1 .. Cancel",
        ("FIX.5.0", "334", "2"): "2 .. Error",
        ("FIX.5.0", "334", "3"): "3 .. Correction",
        ("FIX.5.0", "336", "1"): "1 .. Day",
        ("FIX.5.0", "336", "2"): "2 .. HalfDay",
        ("FIX.5.0", "336", "3"): "3 .. Morning",
        ("FIX.5.0", "336", "4"): "4 .. Afternoon",
        ("FIX.5.0", "336", "5"): "5 .. Evening",
        ("FIX.5.0", "336", "6"): "6 .. AfterHours",
        ("FIX.5.0", "336", "7"): "7 .. Holiday",
        ("FIX.5.0", "338", "1"): "1 .. Electronic",
        ("FIX.5.0", "338", "2"): "2 .. OpenOutcry",
        ("FIX.5.0", "338", "3"): "3 .. TwoParty",
        ("FIX.5.0", "338", "4"): "4 .. Voice",
        ("FIX.5.0", "339", "1"): "1 .. Testing",
        ("FIX.5.0", "339", "2"): "2 .. Simulated",
        ("FIX.5.0", "339", "3"): "3 .. Production",
        ("FIX.5.0", "340", "0"): "0 .. Unknown",
        ("FIX.5.0", "340", "1"): "1 .. Halted",
        ("FIX.5.0", "340", "2"): "2 .. Open",
        ("FIX.5.0", "340", "3"): "3 .. Closed",
        ("FIX.5.0", "340", "4"): "4 .. PreOpen",
        ("FIX.5.0", "340", "5"): "5 .. PreClose",
        ("FIX.5.0", "340", "6"): "6 .. RequestRejected",
        ("FIX.5.0", "368", "1"): "1 .. UnknownSymbol",
        ("FIX.5.0", "368", "2"): "2 .. Exchange",
        ("FIX.5.0", "368", "3"): "3 .. QuoteRequestExceedsLimit",
        ("FIX.5.0", "368", "4"): "4 .. TooLateToEnter",
        ("FIX.5.0", "368", "5"): "5 .. UnknownQuote",
        ("FIX.5.0", "368", "6"): "6 .. DuplicateQuote",
        ("FIX.5.0", "368", "7"): "7 .. InvalidBid",
        ("FIX.5.0", "368", "8"): "8 .. InvalidPrice",
        ("FIX.5.0", "368", "9"): "9 .. NotAuthorizedToQuoteSecurity",
        ("FIX.5.0", "368", "10"): "10 .. PriceExceedsCurrentPriceBand",
        ("FIX.5.0", "368", "11"): "11 .. QuoteLocked",
        ("FIX.5.0", "368", "12"): "12 .. InvalidOrUnknownSecurityIssuer",
        ("FIX.5.0", "368", "13"): "13 .. InvalidOrUnknownIssuerOfUnderlyingSecurity",
        ("FIX.5.0", "368", "14"): "14 .. NotionalValueExceedsThreshold",
        ("FIX.5.0", "368", "16"): "16 .. ReferencePriceNotAvailable",
        ("FIX.5.0", "368", "17"): "17 .. InsufficientCreditLimit",
        ("FIX.5.0", "368", "18"): "18 .. ExceededClipSizeLimit",
        ("FIX.5.0", "368", "19"): "19 .. ExceededMaxNotionalOrderAmt",
        ("FIX.5.0", "368", "20"): "20 .. ExceededDV01PV01Limit",
        ("FIX.5.0", "368", "21"): "21 .. ExceededCS01Limit",
        ("FIX.5.0", "368", "99"): "99 .. Other",
        ("FIX.5.0", "372", "0"): "0 .. Heartbeat",
        ("FIX.5.0", "372", "1"): "1 .. TestRequest",
        ("FIX.5.0", "372", "2"): "2 .. ResendRequest",
        ("FIX.5.0", "372", "3"): "3 .. Reject",
        ("FIX.5.0", "372", "4"): "4 .. SequenceReset",
        ("FIX.5.0", "372", "5"): "5 .. Logout",
        ("FIX.5.0", "372", "6"): "6 .. IOI",
        ("FIX.5.0", "372", "7"): "7 .. Advertisement",
        ("FIX.5.0", "372", "8"): "8 .. ExecutionReport",
        ("FIX.5.0", "372", "9"): "9 .. OrderCancelReject",
        ("FIX.5.0", "372", "A"): "A .. Logon",
        ("FIX.5.0", "372", "B"): "B .. News",
        ("FIX.5.0", "372", "C"): "C .. Email",
        ("FIX.5.0", "372", "D"): "D .. NewOrderSingle",
        ("FIX.5.0", "372", "E"): "E .. NewOrderList",
        ("FIX.5.0", "372", "F"): "F .. OrderCancelRequest",
        ("FIX.5.0", "372", "G"): "G .. OrderCancelReplaceRequest",
        ("FIX.5.0", "372", "H"): "H .. OrderStatusRequest",
        ("FIX.5.0", "372", "J"): "J .. AllocationInstruction",
        ("FIX.5.0", "372", "K"): "K .. ListCancelRequest",
        ("FIX.5.0", "372", "L"): "L .. ListExecute",
        ("FIX.5.0", "372", "M"): "M .. ListStatusRequest",
        ("FIX.5.0", "372", "N"): "N .. ListStatus",
        ("FIX.5.0", "372", "P"): "P .. AllocationInstructionAck",
        ("FIX.5.0", "372", "Q"): "Q .. DontKnowTrade",
        ("FIX.5.0", "372", "R"): "R .. QuoteRequest",
        ("FIX.5.0", "372", "S"): "S .. Quote",
        ("FIX.5.0", "372", "T"): "T .. SettlementInstructions",
        ("FIX.5.0", "372", "V"): "V .. MarketDataRequest",
        ("FIX.5.0", "372", "W"): "W .. MarketDataSnapshotFullRefresh",
        ("FIX.5.0", "372", "X"): "X .. MarketDataIncrementalRefresh",
        ("FIX.5.0", "372", "Y"): "Y .. MarketDataRequestReject",
        ("FIX.5.0", "372", "Z"): "Z .. QuoteCancel",
        ("FIX.5.0", "372", "a"): "a .. QuoteStatusRequest",
        ("FIX.5.0", "372", "b"): "b .. MassQuoteAck",
        ("FIX.5.0", "372", "c"): "c .. SecurityDefinitionRequest",
        ("FIX.5.0", "372", "d"): "d .. SecurityDefinition",
        ("FIX.5.0", "372", "e"): "e .. SecurityStatusRequest",
        ("FIX.5.0", "372", "f"): "f .. SecurityStatus",
        ("FIX.5.0", "372", "g"): "g .. TradingSessionStatusRequest",
        ("FIX.5.0", "372", "h"): "h .. TradingSessionStatus",
        ("FIX.5.0", "372", "i"): "i .. MassQuote",
        ("FIX.5.0", "372", "j"): "j .. BusinessMessageReject",
        ("FIX.5.0", "372", "k"): "k .. BidRequest",
        ("FIX.5.0", "372", "l"): "l .. BidResponse",
        ("FIX.5.0", "372", "m"): "m .. ListStrikePrice",
        ("FIX.5.0", "372", "n"): "n .. XMLnonFIX",
        ("FIX.5.0", "372", "o"): "o .. RegistrationInstructions",
        ("FIX.5.0", "372", "p"): "p .. RegistrationInstructionsResponse",
        ("FIX.5.0", "372", "q"): "q .. OrderMassCancelRequest",
        ("FIX.5.0", "372", "r"): "r .. OrderMassCancelReport",
        ("FIX.5.0", "372", "s"): "s .. NewOrderCross",
        ("FIX.5.0", "372", "t"): "t .. CrossOrderCancelReplaceRequest",
        ("FIX.5.0", "372", "u"): "u .. CrossOrderCancelRequest",
        ("FIX.5.0", "372", "v"): "v .. SecurityTypeRequest",
        ("FIX.5.0", "372", "w"): "w .. SecurityTypes",
        ("FIX.5.0", "372", "x"): "x .. SecurityListRequest",
        ("FIX.5.0", "372", "y"): "y .. SecurityList",
        ("FIX.5.0", "372", "z"): "z .. DerivativeSecurityListRequest",
        ("FIX.5.0", "372", "AA"): "AA .. DerivativeSecurityList",
        ("FIX.5.0", "372", "AB"): "AB .. NewOrderMultileg",
        ("FIX.5.0", "372", "AC"): "AC .. MultilegOrderCancelReplace",
        ("FIX.5.0", "372", "AD"): "AD .. TradeCaptureReportRequest",
        ("FIX.5.0", "372", "AE"): "AE .. TradeCaptureReport",
        ("FIX.5.0", "372", "AF"): "AF .. OrderMassStatusRequest",
        ("FIX.5.0", "372", "AG"): "AG .. QuoteRequestReject",
        ("FIX.5.0", "372", "AH"): "AH .. RFQRequest",
        ("FIX.5.0", "372", "AI"): "AI .. QuoteStatusReport",
        ("FIX.5.0", "372", "AJ"): "AJ .. QuoteResponse",
        ("FIX.5.0", "372", "AK"): "AK .. Confirmation",
        ("FIX.5.0", "372", "AL"): "AL .. PositionMaintenanceRequest",
        ("FIX.5.0", "372", "AM"): "AM .. PositionMaintenanceReport",
        ("FIX.5.0", "372", "AN"): "AN .. RequestForPositions",
        ("FIX.5.0", "372", "AO"): "AO .. RequestForPositionsAck",
        ("FIX.5.0", "372", "AP"): "AP .. PositionReport",
        ("FIX.5.0", "372", "AQ"): "AQ .. TradeCaptureReportRequestAck",
        ("FIX.5.0", "372", "AR"): "AR .. TradeCaptureReportAck",
        ("FIX.5.0", "372", "AS"): "AS .. AllocationReport",
        ("FIX.5.0", "372", "AT"): "AT .. AllocationReportAck",
        ("FIX.5.0", "372", "AU"): "AU .. ConfirmationAck",
        ("FIX.5.0", "372", "AV"): "AV .. SettlementInstructionRequest",
        ("FIX.5.0", "372", "AW"): "AW .. AssignmentReport",
        ("FIX.5.0", "372", "AX"): "AX .. CollateralRequest",
        ("FIX.5.0", "372", "AY"): "AY .. CollateralAssignment",
        ("FIX.5.0", "372", "AZ"): "AZ .. CollateralResponse",
        ("FIX.5.0", "372", "BA"): "BA .. CollateralReport",
        ("FIX.5.0", "372", "BB"): "BB .. CollateralInquiry",
        ("FIX.5.0", "372", "BC"): "BC .. NetworkCounterpartySystemStatusRequest",
        ("FIX.5.0", "372", "BD"): "BD .. NetworkCounterpartySystemStatusResponse",
        ("FIX.5.0", "372", "BE"): "BE .. UserRequest",
        ("FIX.5.0", "372", "BF"): "BF .. UserResponse",
        ("FIX.5.0", "372", "BG"): "BG .. CollateralInquiryAck",
        ("FIX.5.0", "372", "BH"): "BH .. ConfirmationRequest",
        ("FIX.5.0", "372", "BO"): "BO .. ContraryIntentionReport",
        ("FIX.5.0", "372", "BP"): "BP .. SecurityDefinitionUpdateReport",
        ("FIX.5.0", "372", "BK"): "BK .. SecurityListUpdateReport",
        ("FIX.5.0", "372", "BL"): "BL .. AdjustedPositionReport",
        ("FIX.5.0", "372", "BM"): "BM .. AllocationInstructionAlert",
        ("FIX.5.0", "372", "BN"): "BN .. ExecutionAck",
        ("FIX.5.0", "372", "BJ"): "BJ .. TradingSessionList",
        ("FIX.5.0", "372", "BI"): "BI .. TradingSessionListRequest",
        ("FIX.5.0", "372", "BQ"): "BQ .. SettlementObligationReport",
        ("FIX.5.0", "372", "BR"): "BR .. DerivativeSecurityListUpdateReport",
        ("FIX.5.0", "372", "BS"): "BS .. TradingSessionListUpdateReport",
        ("FIX.5.0", "372", "BT"): "BT .. MarketDefinitionRequest",
        ("FIX.5.0", "372", "BU"): "BU .. MarketDefinition",
        ("FIX.5.0", "372", "BV"): "BV .. MarketDefinitionUpdateReport",
        ("FIX.5.0", "372", "BW"): "BW .. ApplicationMessageRequest",
        ("FIX.5.0", "372", "BX"): "BX .. ApplicationMessageRequestAck",
        ("FIX.5.0", "372", "BY"): "BY .. ApplicationMessageReport",
        ("FIX.5.0", "372", "BZ"): "BZ .. OrderMassActionReport",
        ("FIX.5.0", "372", "CA"): "CA .. OrderMassActionRequest",
        ("FIX.5.0", "372", "CB"): "CB .. UserNotification",
        ("FIX.5.0", "372", "CC"): "CC .. StreamAssignmentRequest",
        ("FIX.5.0", "372", "CD"): "CD .. StreamAssignmentReport",
        ("FIX.5.0", "372", "CE"): "CE .. StreamAssignmentReportACK",
        ("FIX.5.0", "372", "CF"): "CF .. PartyDetailsListRequest",
        ("FIX.5.0", "372", "CG"): "CG .. PartyDetailsListReport",
        ("FIX.5.0", "372", "CH"): "CH .. MarginRequirementInquiry",
        ("FIX.5.0", "372", "CI"): "CI .. MarginRequirementInquiryAck",
        ("FIX.5.0", "372", "CJ"): "CJ .. MarginRequirementReport",
        ("FIX.5.0", "372", "CK"): "CK .. PartyDetailsListUpdateReport",
        ("FIX.5.0", "372", "CL"): "CL .. PartyRiskLimitsRequest",
        ("FIX.5.0", "372", "CM"): "CM .. PartyRiskLimitsReport",
        ("FIX.5.0", "372", "CN"): "CN .. SecurityMassStatusRequest",
        ("FIX.5.0", "372", "CO"): "CO .. SecurityMassStatus",
        ("FIX.5.0", "372", "CQ"): "CQ .. AccountSummaryReport",
        ("FIX.5.0", "372", "CR"): "CR .. PartyRiskLimitsUpdateReport",
        ("FIX.5.0", "372", "CS"): "CS .. PartyRiskLimitsDefinitionRequest",
        ("FIX.5.0", "372", "CT"): "CT .. PartyRiskLimitsDefinitionRequestAck",
        ("FIX.5.0", "372", "CU"): "CU .. PartyEntitlementsRequest",
        ("FIX.5.0", "372", "CV"): "CV .. PartyEntitlementsReport",
        ("FIX.5.0", "372", "CW"): "CW .. QuoteAck",
        ("FIX.5.0", "372", "CX"): "CX .. PartyDetailsDefinitionRequest",
        ("FIX.5.0", "372", "CY"): "CY .. PartyDetailsDefinitionRequestAck",
        ("FIX.5.0", "372", "CZ"): "CZ .. PartyEntitlementsUpdateReport",
        ("FIX.5.0", "372", "DA"): "DA .. PartyEntitlementsDefinitionRequest",
        ("FIX.5.0", "372", "DB"): "DB .. PartyEntitlementsDefinitionRequestAck",
        ("FIX.5.0", "372", "DC"): "DC .. TradeMatchReport",
        ("FIX.5.0", "372", "DD"): "DD .. TradeMatchReportAck",
        ("FIX.5.0", "372", "DE"): "DE .. PartyRiskLimitsReportAck",
        ("FIX.5.0", "372", "DF"): "DF .. PartyRiskLimitCheckRequest",
        ("FIX.5.0", "372", "DG"): "DG .. PartyRiskLimitCheckRequestAck",
        ("FIX.5.0", "372", "DH"): "DH .. PartyActionRequest",
        ("FIX.5.0", "372", "DI"): "DI .. PartyActionReport",
        ("FIX.5.0", "372", "DJ"): "DJ .. MassOrder",
        ("FIX.5.0", "372", "DK"): "DK .. MassOrderAck",
        ("FIX.5.0", "372", "DL"): "DL .. PositionTransferInstruction",
        ("FIX.5.0", "372", "DM"): "DM .. PositionTransferInstructionAck",
        ("FIX.5.0", "372", "DN"): "DN .. PositionTransferReport",
        ("FIX.5.0", "372", "DO"): "DO .. MarketDataStatisticsRequest",
        ("FIX.5.0", "372", "DP"): "DP .. MarketDataStatisticsReport",
        ("FIX.5.0", "372", "DQ"): "DQ .. CollateralReportAck",
        ("FIX.5.0", "372", "DR"): "DR .. MarketDataReport",
        ("FIX.5.0", "372", "DS"): "DS .. CrossRequest",
        ("FIX.5.0", "372", "DT"): "DT .. CrossRequestAck",
        ("FIX.5.0", "372", "DU"): "DU .. AllocationInstructionAlertRequest",
        ("FIX.5.0", "373", "0"): "0 .. InvalidTagNumber",
        ("FIX.5.0", "373", "1"): "1 .. RequiredTagMissing",
        ("FIX.5.0", "373", "2"): "2 .. TagNotDefinedForThisMessageType",
        ("FIX.5.0", "373", "3"): "3 .. UndefinedTag",
        ("FIX.5.0", "373", "4"): "4 .. TagSpecifiedWithoutAValue",
        ("FIX.5.0", "373", "5"): "5 .. ValueIsIncorrect",
        ("FIX.5.0", "373", "6"): "6 .. IncorrectDataFormatForValue",
        ("FIX.5.0", "373", "7"): "7 .. DecryptionProblem",
        ("FIX.5.0", "373", "8"): "8 .. SignatureProblem",
        ("FIX.5.0", "373", "9"): "9 .. CompIDProblem",
        ("FIX.5.0", "373", "10"): "10 .. SendingTimeAccuracyProblem",
        ("FIX.5.0", "373", "11"): "11 .. InvalidMsgType",
        ("FIX.5.0", "373", "12"): "12 .. XMLValidationError",
        ("FIX.5.0", "373", "13"): "13 .. TagAppearsMoreThanOnce",
        ("FIX.5.0", "373", "14"): "14 .. TagSpecifiedOutOfRequiredOrder",
        ("FIX.5.0", "373", "15"): "15 .. RepeatingGroupFieldsOutOfOrder",
        ("FIX.5.0", "373", "16"): "16 .. IncorrectNumInGroupCountForRepeatingGroup",
        ("FIX.5.0", "373", "17"): "17 .. Non",
        ("FIX.5.0", "373", "18"): "18 .. Invalid",
        ("FIX.5.0", "373", "99"): "99 .. Other",
        ("FIX.5.0", "374", "C"): "C .. Cancel",
        ("FIX.5.0", "374", "N"): "N .. New",
        ("FIX.5.0", "377", "N"): "N .. WasNotSolicited",
        ("FIX.5.0", "377", "Y"): "Y .. WasSolicited",
        ("FIX.5.0", "378", "0"): "0 .. GTCorporateAction",
        ("FIX.5.0", "378", "1"): "1 .. GTRenewal",
        ("FIX.5.0", "378", "2"): "2 .. VerbalChange",
        ("FIX.5.0", "378", "3"): "3 .. RepricingOfOrder",
        ("FIX.5.0", "378", "4"): "4 .. BrokerOption",
        ("FIX.5.0", "378", "5"): "5 .. PartialDeclineOfOrderQty",
        ("FIX.5.0", "378", "6"): "6 .. CancelOnTradingHalt",
        ("FIX.5.0", "378", "7"): "7 .. CancelOnSystemFailure",
        ("FIX.5.0", "378", "8"): "8 .. Market",
        ("FIX.5.0", "378", "9"): "9 .. Canceled",
        ("FIX.5.0", "378", "10"): "10 .. WarehouseRecap",
        ("FIX.5.0", "378", "11"): "11 .. PegRefresh",
        ("FIX.5.0", "378", "12"): "12 .. CancelOnConnectionLoss",
        ("FIX.5.0", "378", "13"): "13 .. CancelOnLogout",
        ("FIX.5.0", "378", "14"): "14 .. AssignTimePriority",
        ("FIX.5.0", "378", "15"): "15 .. CancelledForTradePriceViolation",
        ("FIX.5.0", "378", "16"): "16 .. CancelledForCrossImbalance",
        ("FIX.5.0", "378", "99"): "99 .. Other",
        ("FIX.5.0", "380", "0"): "0 .. Other",
        ("FIX.5.0", "380", "1"): "1 .. UnknownID",
        ("FIX.5.0", "380", "2"): "2 .. UnknownSecurity",
        ("FIX.5.0", "380", "3"): "3 .. UnsupportedMessageType",
        ("FIX.5.0", "380", "4"): "4 .. ApplicationNotAvailable",
        ("FIX.5.0", "380", "5"): "5 .. ConditionallyRequiredFieldMissing",
        ("FIX.5.0", "380", "6"): "6 .. NotAuthorized",
        ("FIX.5.0", "380", "7"): "7 .. DeliverToFirmNotAvailableAtThisTime",
        ("FIX.5.0", "380", "8"): "8 .. ThrottleLimitExceeded",
        ("FIX.5.0", "380", "9"): "9 .. ThrottleLimitExceededSessionDisconnected",
        ("FIX.5.0", "380", "10"): "10 .. ThrottledMessagesRejectedOnRequest",
        ("FIX.5.0", "380", "18"): "18 .. InvalidPriceIncrement",
        ("FIX.5.0", "385", "R"): "R .. Receive",
        ("FIX.5.0", "385", "S"): "S .. Send",
        ("FIX.5.0", "388", "0"): "0 .. RelatedToDisplayedPrice",
        ("FIX.5.0", "388", "1"): "1 .. RelatedToMarketPrice",
        ("FIX.5.0", "388", "2"): "2 .. RelatedToPrimaryPrice",
        ("FIX.5.0", "388", "3"): "3 .. RelatedToLocalPrimaryPrice",
        ("FIX.5.0", "388", "4"): "4 .. RelatedToMidpointPrice",
        ("FIX.5.0", "388", "5"): "5 .. RelatedToLastTradePrice",
        ("FIX.5.0", "388", "6"): "6 .. RelatedToVWAP",
        ("FIX.5.0", "388", "7"): "7 .. AveragePriceGuarantee",
        ("FIX.5.0", "394", "1"): "1 .. NonDisclosed",
        ("FIX.5.0", "394", "2"): "2 .. Disclosed",
        ("FIX.5.0", "394", "3"): "3 .. NoBiddingProcess",
        ("FIX.5.0", "399", "1"): "1 .. Sector",
        ("FIX.5.0", "399", "2"): "2 .. Country",
        ("FIX.5.0", "399", "3"): "3 .. Index",
        ("FIX.5.0", "401", "1"): "1 .. SideValue1",
        ("FIX.5.0", "401", "2"): "2 .. SideValue2",
        ("FIX.5.0", "409", "1"): "1 .. FiveDayMovingAverage",
        ("FIX.5.0", "409", "2"): "2 .. TwentyDayMovingAverage",
        ("FIX.5.0", "409", "3"): "3 .. NormalMarketSize",
        ("FIX.5.0", "409", "4"): "4 .. Other",
        ("FIX.5.0", "411", "N"): "N .. False",
        ("FIX.5.0", "411", "Y"): "Y .. True",
        ("FIX.5.0", "414", "1"): "1 .. BuySideRequests",
        ("FIX.5.0", "414", "2"): "2 .. SellSideSends",
        ("FIX.5.0", "414", "3"): "3 .. RealTimeExecutionReports",
        ("FIX.5.0", "416", "1"): "1 .. Net",
        ("FIX.5.0", "416", "2"): "2 .. Gross",
        ("FIX.5.0", "418", "A"): "A .. Agency",
        ("FIX.5.0", "418", "G"): "G .. VWAPGuarantee",
        ("FIX.5.0", "418", "J"): "J .. GuaranteedClose",
        ("FIX.5.0", "418", "R"): "R .. RiskTrade",
        ("FIX.5.0", "419", "2"): "2 .. ClosingPriceAtMorningSession",
        ("FIX.5.0", "419", "3"): "3 .. ClosingPrice",
        ("FIX.5.0", "419", "4"): "4 .. CurrentPrice",
        ("FIX.5.0", "419", "5"): "5 .. SQ",
        ("FIX.5.0", "419", "6"): "6 .. VWAPThroughADay",
        ("FIX.5.0", "419", "7"): "7 .. VWAPThroughAMorningSession",
        ("FIX.5.0", "419", "8"): "8 .. VWAPThroughAnAfternoonSession",
        ("FIX.5.0", "419", "9"): "9 .. VWAPThroughADayExcept",
        ("FIX.5.0", "419", "A"): "A .. VWAPThroughAMorningSessionExcept",
        ("FIX.5.0", "419", "B"): "B .. VWAPThroughAnAfternoonSessionExcept",
        ("FIX.5.0", "419", "C"): "C .. Strike",
        ("FIX.5.0", "419", "D"): "D .. Open",
        ("FIX.5.0", "419", "Z"): "Z .. Others",
        ("FIX.5.0", "423", "1"): "1 .. Percentage",
        ("FIX.5.0", "423", "2"): "2 .. PerUnit",
        ("FIX.5.0", "423", "3"): "3 .. FixedAmount",
        ("FIX.5.0", "423", "4"): "4 .. Discount",
        ("FIX.5.0", "423", "5"): "5 .. Premium",
        ("FIX.5.0", "423", "6"): "6 .. Spread",
        ("FIX.5.0", "423", "7"): "7 .. TEDPrice",
        ("FIX.5.0", "423", "8"): "8 .. TEDYield",
        ("FIX.5.0", "423", "9"): "9 .. Yield",
        ("FIX.5.0", "423", "10"): "10 .. FixedCabinetTradePrice",
        ("FIX.5.0", "423", "11"): "11 .. VariableCabinetTradePrice",
        ("FIX.5.0", "423", "12"): "12 .. PriceSpread",
        ("FIX.5.0", "423", "13"): "13 .. ProductTicksInHalves",
        ("FIX.5.0", "423", "14"): "14 .. ProductTicksInFourths",
        ("FIX.5.0", "423", "15"): "15 .. ProductTicksInEighths",
        ("FIX.5.0", "423", "16"): "16 .. ProductTicksInSixteenths",
        ("FIX.5.0", "423", "17"): "17 .. ProductTicksInThirtySeconds",
        ("FIX.5.0", "423", "18"): "18 .. ProductTicksInSixtyFourths",
        ("FIX.5.0", "423", "19"): "19 .. ProductTicksInOneTwentyEighths",
        ("FIX.5.0", "423", "20"): "20 .. NormalRateRepresentation",
        ("FIX.5.0", "423", "21"): "21 .. InverseRateRepresentation",
        ("FIX.5.0", "423", "22"): "22 .. BasisPoints",
        ("FIX.5.0", "423", "23"): "23 .. UpfrontPoints",
        ("FIX.5.0", "423", "24"): "24 .. InterestRate",
        ("FIX.5.0", "423", "25"): "25 .. PercentageNotional",
        ("FIX.5.0", "427", "0"): "0 .. BookOutAllTradesOnDayOfExecution",
        ("FIX.5.0", "427", "1"): "1 .. AccumulateUntilFilledOrExpired",
        ("FIX.5.0", "427", "2"): "2 .. AccumulateUntilVerballyNotifiedOtherwise",
        ("FIX.5.0", "429", "1"): "1 .. Ack",
        ("FIX.5.0", "429", "2"): "2 .. Response",
        ("FIX.5.0", "429", "3"): "3 .. Timed",
        ("FIX.5.0", "429", "4"): "4 .. ExecStarted",
        ("FIX.5.0", "429", "5"): "5 .. AllDone",
        ("FIX.5.0", "429", "6"): "6 .. Alert",
        ("FIX.5.0", "430", "1"): "1 .. Net",
        ("FIX.5.0", "430", "2"): "2 .. Gross",
        ("FIX.5.0", "431", "1"): "1 .. InBiddingProcess",
        ("FIX.5.0", "431", "2"): "2 .. ReceivedForExecution",
        ("FIX.5.0", "431", "3"): "3 .. Executing",
        ("FIX.5.0", "431", "4"): "4 .. Cancelling",
        ("FIX.5.0", "431", "5"): "5 .. Alert",
        ("FIX.5.0", "431", "6"): "6 .. AllDone",
        ("FIX.5.0", "431", "7"): "7 .. Reject",
        ("FIX.5.0", "433", "1"): "1 .. Immediate",
        ("FIX.5.0", "433", "2"): "2 .. WaitForInstruction",
        ("FIX.5.0", "433", "3"): "3 .. SellDriven",
        ("FIX.5.0", "433", "4"): "4 .. BuyDrivenCashTopUp",
        ("FIX.5.0", "433", "5"): "5 .. BuyDrivenCashWithdraw",
        ("FIX.5.0", "434", "1"): "1 .. OrderCancelRequest",
        ("FIX.5.0", "434", "2"): "2 .. OrderCancelReplaceRequest",
        ("FIX.5.0", "442", "1"): "1 .. SingleSecurity",
        ("FIX.5.0", "442", "2"): "2 .. IndividualLegOfAMultiLegSecurity",
        ("FIX.5.0", "442", "3"): "3 .. MultiLegSecurity",
        ("FIX.5.0", "447", "6"): "6 .. UKNationalInsuranceOrPensionNumber",
        ("FIX.5.0", "447", "7"): "7 .. USSocialSecurityNumber",
        ("FIX.5.0", "447", "8"): "8 .. USEmployerOrTaxIDNumber",
        ("FIX.5.0", "447", "9"): "9 .. AustralianBusinessNumber",
        ("FIX.5.0", "447", "A"): "A .. AustralianTaxFileNumber",
        ("FIX.5.0", "447", "J"): "J .. TaxID",
        ("FIX.5.0", "447", "1"): "1 .. KoreanInvestorID",
        ("FIX.5.0", "447", "2"): "2 .. TaiwaneseForeignInvestorID",
        ("FIX.5.0", "447", "3"): "3 .. TaiwaneseTradingAcct",
        ("FIX.5.0", "447", "4"): "4 .. MalaysianCentralDepository",
        ("FIX.5.0", "447", "5"): "5 .. ChineseInvestorID",
        ("FIX.5.0", "447", "I"): "I .. ISITCAcronym",
        ("FIX.5.0", "447", "B"): "B .. BIC",
        ("FIX.5.0", "447", "C"): "C .. GeneralIdentifier",
        ("FIX.5.0", "447", "D"): "D .. Proprietary",
        ("FIX.5.0", "447", "E"): "E .. ISOCountryCode",
        ("FIX.5.0", "447", "F"): "F .. SettlementEntityLocation",
        ("FIX.5.0", "447", "G"): "G .. MIC",
        ("FIX.5.0", "447", "H"): "H .. CSDParticipant",
        ("FIX.5.0", "447", "K"): "K .. AustralianCompanyNumber",
        ("FIX.5.0", "447", "L"): "L .. AustralianRegisteredBodyNumber",
        ("FIX.5.0", "447", "M"): "M .. CFTCReportingFirmIdentifier",
        ("FIX.5.0", "447", "N"): "N .. LegalEntityIdentifier",
        ("FIX.5.0", "447", "O"): "O .. InterimIdentifier",
        ("FIX.5.0", "447", "P"): "P .. ShortCodeIdentifier",
        ("FIX.5.0", "447", "Q"): "Q .. NationalIDNaturalPerson",
        ("FIX.5.0", "452", "1"): "1 .. ExecutingFirm",
        ("FIX.5.0", "452", "2"): "2 .. BrokerOfCredit",
        ("FIX.5.0", "452", "3"): "3 .. ClientID",
        ("FIX.5.0", "452", "4"): "4 .. ClearingFirm",
        ("FIX.5.0", "452", "5"): "5 .. InvestorID",
        ("FIX.5.0", "452", "6"): "6 .. IntroducingFirm",
        ("FIX.5.0", "452", "7"): "7 .. EnteringFirm",
        ("FIX.5.0", "452", "8"): "8 .. Locate",
        ("FIX.5.0", "452", "9"): "9 .. FundManagerClientID",
        ("FIX.5.0", "452", "10"): "10 .. SettlementLocation",
        ("FIX.5.0", "452", "11"): "11 .. OrderOriginationTrader",
        ("FIX.5.0", "452", "12"): "12 .. ExecutingTrader",
        ("FIX.5.0", "452", "13"): "13 .. OrderOriginationFirm",
        ("FIX.5.0", "452", "15"): "15 .. CorrespondantClearingFirm",
        ("FIX.5.0", "452", "16"): "16 .. ExecutingSystem",
        ("FIX.5.0", "452", "17"): "17 .. ContraFirm",
        ("FIX.5.0", "452", "18"): "18 .. ContraClearingFirm",
        ("FIX.5.0", "452", "19"): "19 .. SponsoringFirm",
        ("FIX.5.0", "452", "20"): "20 .. UnderlyingContraFirm",
        ("FIX.5.0", "452", "21"): "21 .. ClearingOrganization",
        ("FIX.5.0", "452", "22"): "22 .. Exchange",
        ("FIX.5.0", "452", "24"): "24 .. CustomerAccount",
        ("FIX.5.0", "452", "25"): "25 .. CorrespondentClearingOrganization",
        ("FIX.5.0", "452", "26"): "26 .. CorrespondentBroker",
        ("FIX.5.0", "452", "27"): "27 .. Buyer",
        ("FIX.5.0", "452", "28"): "28 .. Custodian",
        ("FIX.5.0", "452", "29"): "29 .. Intermediary",
        ("FIX.5.0", "452", "30"): "30 .. Agent",
        ("FIX.5.0", "452", "31"): "31 .. SubCustodian",
        ("FIX.5.0", "452", "32"): "32 .. Beneficiary",
        ("FIX.5.0", "452", "33"): "33 .. InterestedParty",
        ("FIX.5.0", "452", "34"): "34 .. RegulatoryBody",
        ("FIX.5.0", "452", "35"): "35 .. LiquidityProvider",
        ("FIX.5.0", "452", "36"): "36 .. EnteringTrader",
        ("FIX.5.0", "452", "37"): "37 .. ContraTrader",
        ("FIX.5.0", "452", "38"): "38 .. PositionAccount",
        ("FIX.5.0", "452", "39"): "39 .. ContraInvestorID",
        ("FIX.5.0", "452", "40"): "40 .. TransferToFirm",
        ("FIX.5.0", "452", "41"): "41 .. ContraPositionAccount",
        ("FIX.5.0", "452", "42"): "42 .. ContraExchange",
        ("FIX.5.0", "452", "43"): "43 .. InternalCarryAccount",
        ("FIX.5.0", "452", "44"): "44 .. OrderEntryOperatorID",
        ("FIX.5.0", "452", "45"): "45 .. SecondaryAccountNumber",
        ("FIX.5.0", "452", "46"): "46 .. ForeignFirm",
        ("FIX.5.0", "452", "47"): "47 .. ThirdPartyAllocationFirm",
        ("FIX.5.0", "452", "48"): "48 .. ClaimingAccount",
        ("FIX.5.0", "452", "49"): "49 .. AssetManager",
        ("FIX.5.0", "452", "50"): "50 .. PledgorAccount",
        ("FIX.5.0", "452", "51"): "51 .. PledgeeAccount",
        ("FIX.5.0", "452", "52"): "52 .. LargeTraderReportableAccount",
        ("FIX.5.0", "452", "53"): "53 .. TraderMnemonic",
        ("FIX.5.0", "452", "54"): "54 .. SenderLocation",
        ("FIX.5.0", "452", "55"): "55 .. SessionID",
        ("FIX.5.0", "452", "56"): "56 .. AcceptableCounterparty",
        ("FIX.5.0", "452", "57"): "57 .. UnacceptableCounterparty",
        ("FIX.5.0", "452", "58"): "58 .. EnteringUnit",
        ("FIX.5.0", "452", "59"): "59 .. ExecutingUnit",
        ("FIX.5.0", "452", "60"): "60 .. IntroducingBroker",
        ("FIX.5.0", "452", "61"): "61 .. QuoteOriginator",
        ("FIX.5.0", "452", "62"): "62 .. ReportOriginator",
        ("FIX.5.0", "452", "63"): "63 .. SystematicInternaliser",
        ("FIX.5.0", "452", "64"): "64 .. MultilateralTradingFacility",
        ("FIX.5.0", "452", "65"): "65 .. RegulatedMarket",
        ("FIX.5.0", "452", "66"): "66 .. MarketMaker",
        ("FIX.5.0", "452", "67"): "67 .. InvestmentFirm",
        ("FIX.5.0", "452", "68"): "68 .. HostCompetentAuthority",
        ("FIX.5.0", "452", "69"): "69 .. HomeCompetentAuthority",
        ("FIX.5.0", "452", "70"): "70 .. CompetentAuthorityLiquidity",
        ("FIX.5.0", "452", "71"): "71 .. CompetentAuthorityTransactionVenue",
        ("FIX.5.0", "452", "72"): "72 .. ReportingIntermediary",
        ("FIX.5.0", "452", "73"): "73 .. ExecutionVenue",
        ("FIX.5.0", "452", "74"): "74 .. MarketDataEntryOriginator",
        ("FIX.5.0", "452", "75"): "75 .. LocationID",
        ("FIX.5.0", "452", "76"): "76 .. DeskID",
        ("FIX.5.0", "452", "77"): "77 .. MarketDataMarket",
        ("FIX.5.0", "452", "78"): "78 .. AllocationEntity",
        ("FIX.5.0", "452", "79"): "79 .. PrimeBroker",
        ("FIX.5.0", "452", "80"): "80 .. StepOutFirm",
        ("FIX.5.0", "452", "81"): "81 .. BrokerClearingID",
        ("FIX.5.0", "452", "82"): "82 .. CentralRegistrationDepository",
        ("FIX.5.0", "452", "83"): "83 .. ClearingAccount",
        ("FIX.5.0", "452", "84"): "84 .. AcceptableSettlingCounterparty",
        ("FIX.5.0", "452", "85"): "85 .. UnacceptableSettlingCounterparty",
        ("FIX.5.0", "452", "86"): "86 .. CLSMemberBank",
        ("FIX.5.0", "452", "87"): "87 .. InConcertGroup",
        ("FIX.5.0", "452", "88"): "88 .. InConcertControllingEntity",
        ("FIX.5.0", "452", "89"): "89 .. LargePositionsReportingAccount",
        ("FIX.5.0", "452", "90"): "90 .. SettlementFirm",
        ("FIX.5.0", "452", "91"): "91 .. SettlementAccount",
        ("FIX.5.0", "452", "92"): "92 .. ReportingMarketCenter",
        ("FIX.5.0", "452", "93"): "93 .. RelatedReportingMarketCenter",
        ("FIX.5.0", "452", "94"): "94 .. AwayMarket",
        ("FIX.5.0", "452", "95"): "95 .. GiveupTradingFirm",
        ("FIX.5.0", "452", "96"): "96 .. TakeupTradingFirm",
        ("FIX.5.0", "452", "97"): "97 .. GiveupClearingFirm",
        ("FIX.5.0", "452", "98"): "98 .. TakeupClearingFirm",
        ("FIX.5.0", "452", "99"): "99 .. OriginatingMarket",
        ("FIX.5.0", "452", "100"): "100 .. MarginAccount",
        ("FIX.5.0", "452", "101"): "101 .. CollateralAssetAccount",
        ("FIX.5.0", "452", "102"): "102 .. DataRepository",
        ("FIX.5.0", "452", "103"): "103 .. CalculationAgent",
        ("FIX.5.0", "452", "104"): "104 .. ExerciseNoticeSender",
        ("FIX.5.0", "452", "105"): "105 .. ExerciseNoticeReceiver",
        ("FIX.5.0", "452", "106"): "106 .. RateReferenceBank",
        ("FIX.5.0", "452", "107"): "107 .. Correspondent",
        ("FIX.5.0", "452", "109"): "109 .. BeneficiaryBank",
        ("FIX.5.0", "452", "110"): "110 .. Borrower",
        ("FIX.5.0", "452", "111"): "111 .. PrimaryObligator",
        ("FIX.5.0", "452", "112"): "112 .. Guarantor",
        ("FIX.5.0", "452", "113"): "113 .. ExcludedReferenceEntity",
        ("FIX.5.0", "452", "114"): "114 .. DeterminingParty",
        ("FIX.5.0", "452", "115"): "115 .. HedgingParty",
        ("FIX.5.0", "452", "116"): "116 .. ReportingEntity",
        ("FIX.5.0", "452", "117"): "117 .. SalesPerson",
        ("FIX.5.0", "452", "118"): "118 .. Operator",
        ("FIX.5.0", "452", "119"): "119 .. CSD",
        ("FIX.5.0", "452", "120"): "120 .. ICSD",
        ("FIX.5.0", "452", "121"): "121 .. TradingSubAccount",
        ("FIX.5.0", "452", "122"): "122 .. InvestmentDecisionMaker",
        ("FIX.5.0", "452", "123"): "123 .. PublishingIntermediary",
        ("FIX.5.0", "456", "1"): "1 .. CUSIP",
        ("FIX.5.0", "456", "2"): "2 .. SEDOL",
        ("FIX.5.0", "456", "3"): "3 .. QUIK",
        ("FIX.5.0", "456", "4"): "4 .. ISINNumber",
        ("FIX.5.0", "456", "5"): "5 .. RICCode",
        ("FIX.5.0", "456", "6"): "6 .. ISOCurrencyCode",
        ("FIX.5.0", "456", "7"): "7 .. ISOCountryCode",
        ("FIX.5.0", "456", "8"): "8 .. ExchangeSymbol",
        ("FIX.5.0", "456", "9"): "9 .. ConsolidatedTapeAssociation",
        ("FIX.5.0", "456", "A"): "A .. BloombergSymbol",
        ("FIX.5.0", "456", "B"): "B .. Wertpapier",
        ("FIX.5.0", "456", "C"): "C .. Dutch",
        ("FIX.5.0", "456", "D"): "D .. Valoren",
        ("FIX.5.0", "456", "E"): "E .. Sicovam",
        ("FIX.5.0", "456", "F"): "F .. Belgian",
        ("FIX.5.0", "456", "G"): "G .. Common",
        ("FIX.5.0", "456", "H"): "H .. ClearingHouse",
        ("FIX.5.0", "456", "I"): "I .. ISDAFpMLSpecification",
        ("FIX.5.0", "456", "J"): "J .. OptionPriceReportingAuthority",
        ("FIX.5.0", "456", "K"): "K .. ISDAFpMLURL",
        ("FIX.5.0", "456", "L"): "L .. LetterOfCredit",
        ("FIX.5.0", "456", "M"): "M .. MarketplaceAssignedIdentifier",
        ("FIX.5.0", "456", "N"): "N .. MarkitREDEntityCLIP",
        ("FIX.5.0", "456", "P"): "P .. MarkitREDPairCLIP",
        ("FIX.5.0", "456", "Q"): "Q .. CFTCCommodityCode",
        ("FIX.5.0", "456", "R"): "R .. ISDACommodityReferencePrice",
        ("FIX.5.0", "456", "S"): "S .. FinancialInstrumentGlobalIdentifier",
        ("FIX.5.0", "456", "T"): "T .. LegalEntityIdentifier",
        ("FIX.5.0", "456", "U"): "U .. Synthetic",
        ("FIX.5.0", "456", "V"): "V .. FidessaInstrumentMnemonic",
        ("FIX.5.0", "456", "W"): "W .. IndexName",
        ("FIX.5.0", "459", "1"): "1 .. CUSIP",
        ("FIX.5.0", "459", "2"): "2 .. SEDOL",
        ("FIX.5.0", "459", "3"): "3 .. QUIK",
        ("FIX.5.0", "459", "4"): "4 .. ISINNumber",
        ("FIX.5.0", "459", "5"): "5 .. RICCode",
        ("FIX.5.0", "459", "6"): "6 .. ISOCurrencyCode",
        ("FIX.5.0", "459", "7"): "7 .. ISOCountryCode",
        ("FIX.5.0", "459", "8"): "8 .. ExchangeSymbol",
        ("FIX.5.0", "459", "9"): "9 .. ConsolidatedTapeAssociation",
        ("FIX.5.0", "459", "A"): "A .. BloombergSymbol",
        ("FIX.5.0", "459", "B"): "B .. Wertpapier",
        ("FIX.5.0", "459", "C"): "C .. Dutch",
        ("FIX.5.0", "459", "D"): "D .. Valoren",
        ("FIX.5.0", "459", "E"): "E .. Sicovam",
        ("FIX.5.0", "459", "F"): "F .. Belgian",
        ("FIX.5.0", "459", "G"): "G .. Common",
        ("FIX.5.0", "459", "H"): "H .. ClearingHouse",
        ("FIX.5.0", "459", "I"): "I .. ISDAFpMLSpecification",
        ("FIX.5.0", "459", "J"): "J .. OptionPriceReportingAuthority",
        ("FIX.5.0", "459", "K"): "K .. ISDAFpMLURL",
        ("FIX.5.0", "459", "L"): "L .. LetterOfCredit",
        ("FIX.5.0", "459", "M"): "M .. MarketplaceAssignedIdentifier",
        ("FIX.5.0", "459", "N"): "N .. MarkitREDEntityCLIP",
        ("FIX.5.0", "459", "P"): "P .. MarkitREDPairCLIP",
        ("FIX.5.0", "459", "Q"): "Q .. CFTCCommodityCode",
        ("FIX.5.0", "459", "R"): "R .. ISDACommodityReferencePrice",
        ("FIX.5.0", "459", "S"): "S .. FinancialInstrumentGlobalIdentifier",
        ("FIX.5.0", "459", "T"): "T .. LegalEntityIdentifier",
        ("FIX.5.0", "459", "U"): "U .. Synthetic",
        ("FIX.5.0", "459", "V"): "V .. FidessaInstrumentMnemonic",
        ("FIX.5.0", "459", "W"): "W .. IndexName",
        ("FIX.5.0", "460", "1"): "1 .. AGENCY",
        ("FIX.5.0", "460", "2"): "2 .. COMMODITY",
        ("FIX.5.0", "460", "3"): "3 .. CORPORATE",
        ("FIX.5.0", "460", "4"): "4 .. CURRENCY",
        ("FIX.5.0", "460", "5"): "5 .. EQUITY",
        ("FIX.5.0", "460", "6"): "6 .. GOVERNMENT",
        ("FIX.5.0", "460", "7"): "7 .. INDEX",
        ("FIX.5.0", "460", "8"): "8 .. LOAN",
        ("FIX.5.0", "460", "9"): "9 .. MONEYMARKET",
        ("FIX.5.0", "460", "10"): "10 .. MORTGAGE",
        ("FIX.5.0", "460", "11"): "11 .. MUNICIPAL",
        ("FIX.5.0", "460", "12"): "12 .. OTHER",
        ("FIX.5.0", "460", "13"): "13 .. FINANCING",
        ("FIX.5.0", "462", "1"): "1 .. AGENCY",
        ("FIX.5.0", "462", "2"): "2 .. COMMODITY",
        ("FIX.5.0", "462", "3"): "3 .. CORPORATE",
        ("FIX.5.0", "462", "4"): "4 .. CURRENCY",
        ("FIX.5.0", "462", "5"): "5 .. EQUITY",
        ("FIX.5.0", "462", "6"): "6 .. GOVERNMENT",
        ("FIX.5.0", "462", "7"): "7 .. INDEX",
        ("FIX.5.0", "462", "8"): "8 .. LOAN",
        ("FIX.5.0", "462", "9"): "9 .. MONEYMARKET",
        ("FIX.5.0", "462", "10"): "10 .. MORTGAGE",
        ("FIX.5.0", "462", "11"): "11 .. MUNICIPAL",
        ("FIX.5.0", "462", "12"): "12 .. OTHER",
        ("FIX.5.0", "462", "13"): "13 .. FINANCING",
        ("FIX.5.0", "464", "N"): "N .. False",
        ("FIX.5.0", "464", "Y"): "Y .. True",
        ("FIX.5.0", "468", "0"): "0 .. RoundToNearest",
        ("FIX.5.0", "468", "1"): "1 .. RoundDown",
        ("FIX.5.0", "468", "2"): "2 .. RoundUp",
        ("FIX.5.0", "477", "1"): "1 .. CREST",
        ("FIX.5.0", "477", "2"): "2 .. NSCC",
        ("FIX.5.0", "477", "3"): "3 .. Euroclear",
        ("FIX.5.0", "477", "4"): "4 .. Clearstream",
        ("FIX.5.0", "477", "5"): "5 .. Cheque",
        ("FIX.5.0", "477", "6"): "6 .. TelegraphicTransfer",
        ("FIX.5.0", "477", "7"): "7 .. FedWire",
        ("FIX.5.0", "477", "8"): "8 .. DirectCredit",
        ("FIX.5.0", "477", "9"): "9 .. ACHCredit",
        ("FIX.5.0", "477", "10"): "10 .. BPAY",
        ("FIX.5.0", "477", "11"): "11 .. HighValueClearingSystemHVACS",
        ("FIX.5.0", "477", "12"): "12 .. ReinvestInFund",
        ("FIX.5.0", "480", "Y"): "Y .. Yes",
        ("FIX.5.0", "480", "N"): "N .. NoExecutionOnly",
        ("FIX.5.0", "480", "M"): "M .. NoWaiverAgreement",
        ("FIX.5.0", "480", "O"): "O .. NoInstitutional",
        ("FIX.5.0", "481", "Y"): "Y .. Passed",
        ("FIX.5.0", "481", "N"): "N .. NotChecked",
        ("FIX.5.0", "481", "1"): "1 .. ExemptBelowLimit",
        ("FIX.5.0", "481", "2"): "2 .. ExemptMoneyType",
        ("FIX.5.0", "481", "3"): "3 .. ExemptAuthorised",
        ("FIX.5.0", "484", "B"): "B .. BidPrice",
        ("FIX.5.0", "484", "C"): "C .. CreationPrice",
        ("FIX.5.0", "484", "D"): "D .. CreationPricePlusAdjustmentPercent",
        ("FIX.5.0", "484", "E"): "E .. CreationPricePlusAdjustmentAmount",
        ("FIX.5.0", "484", "O"): "O .. OfferPrice",
        ("FIX.5.0", "484", "P"): "P .. OfferPriceMinusAdjustmentPercent",
        ("FIX.5.0", "484", "Q"): "Q .. OfferPriceMinusAdjustmentAmount",
        ("FIX.5.0", "484", "S"): "S .. SinglePrice",
        ("FIX.5.0", "487", "0"): "0 .. New",
        ("FIX.5.0", "487", "1"): "1 .. Cancel",
        ("FIX.5.0", "487", "2"): "2 .. Replace",
        ("FIX.5.0", "487", "3"): "3 .. Release",
        ("FIX.5.0", "487", "4"): "4 .. Reverse",
        ("FIX.5.0", "487", "5"): "5 .. CancelDueToBackOutOfTrade",
        ("FIX.5.0", "492", "1"): "1 .. CREST",
        ("FIX.5.0", "492", "2"): "2 .. NSCC",
        ("FIX.5.0", "492", "3"): "3 .. Euroclear",
        ("FIX.5.0", "492", "4"): "4 .. Clearstream",
        ("FIX.5.0", "492", "5"): "5 .. Cheque",
        ("FIX.5.0", "492", "6"): "6 .. TelegraphicTransfer",
        ("FIX.5.0", "492", "7"): "7 .. FedWire",
        ("FIX.5.0", "492", "8"): "8 .. DebitCard",
        ("FIX.5.0", "492", "9"): "9 .. DirectDebit",
        ("FIX.5.0", "492", "10"): "10 .. DirectCredit",
        ("FIX.5.0", "492", "11"): "11 .. CreditCard",
        ("FIX.5.0", "492", "12"): "12 .. ACHDebit",
        ("FIX.5.0", "492", "13"): "13 .. ACHCredit",
        ("FIX.5.0", "492", "14"): "14 .. BPAY",
        ("FIX.5.0", "492", "15"): "15 .. HighValueClearingSystem",
        ("FIX.5.0", "492", "16"): "16 .. CHIPS",
        ("FIX.5.0", "492", "17"): "17 .. SWIFT",
        ("FIX.5.0", "492", "18"): "18 .. CHAPS",
        ("FIX.5.0", "492", "19"): "19 .. SIC",
        ("FIX.5.0", "492", "20"): "20 .. euroSIC",
        ("FIX.5.0", "495", "0"): "0 .. None",
        ("FIX.5.0", "495", "1"): "1 .. MaxiISA",
        ("FIX.5.0", "495", "2"): "2 .. TESSA",
        ("FIX.5.0", "495", "3"): "3 .. MiniCashISA",
        ("FIX.5.0", "495", "4"): "4 .. MiniStocksAndSharesISA",
        ("FIX.5.0", "495", "5"): "5 .. MiniInsuranceISA",
        ("FIX.5.0", "495", "6"): "6 .. CurrentYearPayment",
        ("FIX.5.0", "495", "7"): "7 .. PriorYearPayment",
        ("FIX.5.0", "495", "8"): "8 .. AssetTransfer",
        ("FIX.5.0", "495", "9"): "9 .. EmployeePriorYear",
        ("FIX.5.0", "495", "10"): "10 .. EmployeeCurrentYear",
        ("FIX.5.0", "495", "11"): "11 .. EmployerPriorYear",
        ("FIX.5.0", "495", "12"): "12 .. EmployerCurrentYear",
        ("FIX.5.0", "495", "13"): "13 .. NonFundPrototypeIRA",
        ("FIX.5.0", "495", "14"): "14 .. NonFundQualifiedPlan",
        ("FIX.5.0", "495", "15"): "15 .. DefinedContributionPlan",
        ("FIX.5.0", "495", "16"): "16 .. IRA",
        ("FIX.5.0", "495", "17"): "17 .. IRARollover",
        ("FIX.5.0", "495", "18"): "18 .. KEOGH",
        ("FIX.5.0", "495", "19"): "19 .. ProfitSharingPlan",
        ("FIX.5.0", "495", "20"): "20 .. US401K",
        ("FIX.5.0", "495", "21"): "21 .. SelfDirectedIRA",
        ("FIX.5.0", "495", "22"): "22 .. US403b",
        ("FIX.5.0", "495", "23"): "23 .. US457",
        ("FIX.5.0", "495", "24"): "24 .. RothIRAPrototype",
        ("FIX.5.0", "495", "25"): "25 .. RothIRANonPrototype",
        ("FIX.5.0", "495", "26"): "26 .. RothConversionIRAPrototype",
        ("FIX.5.0", "495", "27"): "27 .. RothConversionIRANonPrototype",
        ("FIX.5.0", "495", "28"): "28 .. EducationIRAPrototype",
        ("FIX.5.0", "495", "29"): "29 .. EducationIRANonPrototype",
        ("FIX.5.0", "495", "999"): "999 .. Other",
        ("FIX.5.0", "497", "N"): "N .. No",
        ("FIX.5.0", "497", "Y"): "Y .. Yes",
        ("FIX.5.0", "506", "A"): "A .. Accepted",
        ("FIX.5.0", "506", "R"): "R .. Rejected",
        ("FIX.5.0", "506", "H"): "H .. Held",
        ("FIX.5.0", "506", "N"): "N .. Reminder",
        ("FIX.5.0", "507", "1"): "1 .. InvalidAccountType",
        ("FIX.5.0", "507", "2"): "2 .. InvalidTaxExemptType",
        ("FIX.5.0", "507", "3"): "3 .. InvalidOwnershipType",
        ("FIX.5.0", "507", "4"): "4 .. NoRegDetails",
        ("FIX.5.0", "507", "5"): "5 .. InvalidRegSeqNo",
        ("FIX.5.0", "507", "6"): "6 .. InvalidRegDetails",
        ("FIX.5.0", "507", "7"): "7 .. InvalidMailingDetails",
        ("FIX.5.0", "507", "8"): "8 .. InvalidMailingInstructions",
        ("FIX.5.0", "507", "9"): "9 .. InvalidInvestorID",
        ("FIX.5.0", "507", "10"): "10 .. InvalidInvestorIDSource",
        ("FIX.5.0", "507", "11"): "11 .. InvalidDateOfBirth",
        ("FIX.5.0", "507", "12"): "12 .. InvalidCountry",
        ("FIX.5.0", "507", "13"): "13 .. InvalidDistribInstns",
        ("FIX.5.0", "507", "14"): "14 .. InvalidPercentage",
        ("FIX.5.0", "507", "15"): "15 .. InvalidPaymentMethod",
        ("FIX.5.0", "507", "16"): "16 .. InvalidAccountName",
        ("FIX.5.0", "507", "17"): "17 .. InvalidAgentCode",
        ("FIX.5.0", "507", "18"): "18 .. InvalidAccountNum",
        ("FIX.5.0", "507", "99"): "99 .. Other",
        ("FIX.5.0", "514", "0"): "0 .. New",
        ("FIX.5.0", "514", "2"): "2 .. Cancel",
        ("FIX.5.0", "514", "1"): "1 .. Replace",
        ("FIX.5.0", "517", "J"): "J .. JointInvestors",
        ("FIX.5.0", "517", "T"): "T .. TenantsInCommon",
        ("FIX.5.0", "517", "2"): "2 .. JointTrustees",
        ("FIX.5.0", "519", "1"): "1 .. CommissionAmount",
        ("FIX.5.0", "519", "2"): "2 .. CommissionPercent",
        ("FIX.5.0", "519", "3"): "3 .. InitialChargeAmount",
        ("FIX.5.0", "519", "4"): "4 .. InitialChargePercent",
        ("FIX.5.0", "519", "5"): "5 .. DiscountAmount",
        ("FIX.5.0", "519", "6"): "6 .. DiscountPercent",
        ("FIX.5.0", "519", "7"): "7 .. DilutionLevyAmount",
        ("FIX.5.0", "519", "8"): "8 .. DilutionLevyPercent",
        ("FIX.5.0", "519", "9"): "9 .. ExitChargeAmount",
        ("FIX.5.0", "519", "10"): "10 .. ExitChargePercent",
        ("FIX.5.0", "519", "11"): "11 .. FundBasedRenewalCommissionPercent",
        ("FIX.5.0", "519", "12"): "12 .. ProjectedFundValue",
        ("FIX.5.0", "519", "13"): "13 .. FundBasedRenewalCommissionOnOrder",
        ("FIX.5.0", "519", "14"): "14 .. FundBasedRenewalCommissionOnFund",
        ("FIX.5.0", "519", "15"): "15 .. NetSettlementAmount",
        ("FIX.5.0", "522", "1"): "1 .. IndividualInvestor",
        ("FIX.5.0", "522", "2"): "2 .. PublicCompany",
        ("FIX.5.0", "522", "3"): "3 .. PrivateCompany",
        ("FIX.5.0", "522", "4"): "4 .. IndividualTrustee",
        ("FIX.5.0", "522", "5"): "5 .. CompanyTrustee",
        ("FIX.5.0", "522", "6"): "6 .. PensionPlan",
        ("FIX.5.0", "522", "7"): "7 .. CustodianUnderGiftsToMinorsAct",
        ("FIX.5.0", "522", "8"): "8 .. Trusts",
        ("FIX.5.0", "522", "9"): "9 .. Fiduciaries",
        ("FIX.5.0", "522", "10"): "10 .. NetworkingSubAccount",
        ("FIX.5.0", "522", "11"): "11 .. NonProfitOrganization",
        ("FIX.5.0", "522", "12"): "12 .. CorporateBody",
        ("FIX.5.0", "522", "13"): "13 .. Nominee",
        ("FIX.5.0", "522", "14"): "14 .. InstitutionalCustomer",
        ("FIX.5.0", "522", "15"): "15 .. Combined",
        ("FIX.5.0", "522", "16"): "16 .. MemberFirmEmployee",
        ("FIX.5.0", "522", "17"): "17 .. MarketMakingAccount",
        ("FIX.5.0", "522", "18"): "18 .. ProprietaryAccount",
        ("FIX.5.0", "522", "19"): "19 .. NonbrokerDealer",
        ("FIX.5.0", "522", "20"): "20 .. UnknownBeneficialOwnerType",
        ("FIX.5.0", "522", "21"): "21 .. FirmsErrorAccount",
        ("FIX.5.0", "525", "6"): "6 .. UKNationalInsuranceOrPensionNumber",
        ("FIX.5.0", "525", "7"): "7 .. USSocialSecurityNumber",
        ("FIX.5.0", "525", "8"): "8 .. USEmployerOrTaxIDNumber",
        ("FIX.5.0", "525", "9"): "9 .. AustralianBusinessNumber",
        ("FIX.5.0", "525", "A"): "A .. AustralianTaxFileNumber",
        ("FIX.5.0", "525", "J"): "J .. TaxID",
        ("FIX.5.0", "525", "1"): "1 .. KoreanInvestorID",
        ("FIX.5.0", "525", "2"): "2 .. TaiwaneseForeignInvestorID",
        ("FIX.5.0", "525", "3"): "3 .. TaiwaneseTradingAcct",
        ("FIX.5.0", "525", "4"): "4 .. MalaysianCentralDepository",
        ("FIX.5.0", "525", "5"): "5 .. ChineseInvestorID",
        ("FIX.5.0", "525", "I"): "I .. ISITCAcronym",
        ("FIX.5.0", "525", "B"): "B .. BIC",
        ("FIX.5.0", "525", "C"): "C .. GeneralIdentifier",
        ("FIX.5.0", "525", "D"): "D .. Proprietary",
        ("FIX.5.0", "525", "E"): "E .. ISOCountryCode",
        ("FIX.5.0", "525", "F"): "F .. SettlementEntityLocation",
        ("FIX.5.0", "525", "G"): "G .. MIC",
        ("FIX.5.0", "525", "H"): "H .. CSDParticipant",
        ("FIX.5.0", "525", "K"): "K .. AustralianCompanyNumber",
        ("FIX.5.0", "525", "L"): "L .. AustralianRegisteredBodyNumber",
        ("FIX.5.0", "525", "M"): "M .. CFTCReportingFirmIdentifier",
        ("FIX.5.0", "525", "N"): "N .. LegalEntityIdentifier",
        ("FIX.5.0", "525", "O"): "O .. InterimIdentifier",
        ("FIX.5.0", "525", "P"): "P .. ShortCodeIdentifier",
        ("FIX.5.0", "525", "Q"): "Q .. NationalIDNaturalPerson",
        ("FIX.5.0", "528", "A"): "A .. Agency",
        ("FIX.5.0", "528", "G"): "G .. Proprietary",
        ("FIX.5.0", "528", "I"): "I .. Individual",
        ("FIX.5.0", "528", "P"): "P .. Principal",
        ("FIX.5.0", "528", "R"): "R .. RisklessPrincipal",
        ("FIX.5.0", "528", "W"): "W .. AgentForOtherMember",
        ("FIX.5.0", "528", "M"): "M .. MixedCapacity",
        ("FIX.5.0", "529", "1"): "1 .. ProgramTrade",
        ("FIX.5.0", "529", "2"): "2 .. IndexArbitrage",
        ("FIX.5.0", "529", "3"): "3 .. NonIndexArbitrage",
        ("FIX.5.0", "529", "4"): "4 .. CompetingMarketMaker",
        ("FIX.5.0", "529", "5"): "5 .. ActingAsMarketMakerOrSpecialistInSecurity",
        ("FIX.5.0", "529", "6"): "6 .. ActingAsMarketMakerOrSpecialistInUnderlying",
        ("FIX.5.0", "529", "7"): "7 .. ForeignEntity",
        ("FIX.5.0", "529", "8"): "8 .. ExternalMarketParticipant",
        ("FIX.5.0", "529", "9"): "9 .. ExternalInterConnectedMarketLinkage",
        ("FIX.5.0", "529", "A"): "A .. RisklessArbitrage",
        ("FIX.5.0", "529", "B"): "B .. IssuerHolding",
        ("FIX.5.0", "529", "C"): "C .. IssuePriceStabilization",
        ("FIX.5.0", "529", "D"): "D .. NonAlgorithmic",
        ("FIX.5.0", "529", "E"): "E .. Algorithmic",
        ("FIX.5.0", "529", "F"): "F .. Cross",
        ("FIX.5.0", "529", "G"): "G .. InsiderAccount",
        ("FIX.5.0", "529", "H"): "H .. SignificantShareholder",
        ("FIX.5.0", "529", "I"): "I .. NormalCourseIssuerBid",
        ("FIX.5.0", "530", "1"): "1 .. CancelOrdersForASecurity",
        ("FIX.5.0", "530", "2"): "2 .. CancelOrdersForAnUnderlyingSecurity",
        ("FIX.5.0", "530", "3"): "3 .. CancelOrdersForAProduct",
        ("FIX.5.0", "530", "4"): "4 .. CancelOrdersForACFICode",
        ("FIX.5.0", "530", "5"): "5 .. CancelOrdersForASecurityType",
        ("FIX.5.0", "530", "6"): "6 .. CancelOrdersForATradingSession",
        ("FIX.5.0", "530", "7"): "7 .. CancelAllOrders",
        ("FIX.5.0", "530", "8"): "8 .. CancelOrdersForAMarket",
        ("FIX.5.0", "530", "9"): "9 .. CancelOrdersForAMarketSegment",
        ("FIX.5.0", "530", "A"): "A .. CancelOrdersForASecurityGroup",
        ("FIX.5.0", "530", "B"): "B .. CancelOrdersForSecurityIssuer",
        ("FIX.5.0", "530", "C"): "C .. CancelForIssuerOfUnderlyingSecurity",
        ("FIX.5.0", "531", "0"): "0 .. CancelRequestRejected",
        ("FIX.5.0", "531", "1"): "1 .. CancelOrdersForASecurity",
        ("FIX.5.0", "531", "2"): "2 .. CancelOrdersForAnUnderlyingSecurity",
        ("FIX.5.0", "531", "3"): "3 .. CancelOrdersForAProduct",
        ("FIX.5.0", "531", "4"): "4 .. CancelOrdersForACFICode",
        ("FIX.5.0", "531", "5"): "5 .. CancelOrdersForASecurityType",
        ("FIX.5.0", "531", "6"): "6 .. CancelOrdersForATradingSession",
        ("FIX.5.0", "531", "7"): "7 .. CancelAllOrders",
        ("FIX.5.0", "531", "8"): "8 .. CancelOrdersForAMarket",
        ("FIX.5.0", "531", "9"): "9 .. CancelOrdersForAMarketSegment",
        ("FIX.5.0", "531", "A"): "A .. CancelOrdersForASecurityGroup",
        ("FIX.5.0", "531", "B"): "B .. CancelOrdersForASecuritiesIssuer",
        ("FIX.5.0", "531", "C"): "C .. CancelOrdersForIssuerOfUnderlyingSecurity",
        ("FIX.5.0", "532", "0"): "0 .. MassCancelNotSupported",
        ("FIX.5.0", "532", "1"): "1 .. InvalidOrUnknownSecurity",
        ("FIX.5.0", "532", "2"): "2 .. InvalidOrUnkownUnderlyingSecurity",
        ("FIX.5.0", "532", "3"): "3 .. InvalidOrUnknownProduct",
        ("FIX.5.0", "532", "4"): "4 .. InvalidOrUnknownCFICode",
        ("FIX.5.0", "532", "5"): "5 .. InvalidOrUnknownSecurityType",
        ("FIX.5.0", "532", "6"): "6 .. InvalidOrUnknownTradingSession",
        ("FIX.5.0", "532", "7"): "7 .. InvalidOrUnknownMarket",
        ("FIX.5.0", "532", "8"): "8 .. InvalidOrUnkownMarketSegment",
        ("FIX.5.0", "532", "9"): "9 .. InvalidOrUnknownSecurityGroup",
        ("FIX.5.0", "532", "10"): "10 .. InvalidOrUnknownSecurityIssuer",
        ("FIX.5.0", "532", "11"): "11 .. InvalidOrUnknownIssuerOfUnderlyingSecurity",
        ("FIX.5.0", "532", "99"): "99 .. Other",
        ("FIX.5.0", "537", "0"): "0 .. Indicative",
        ("FIX.5.0", "537", "1"): "1 .. Tradeable",
        ("FIX.5.0", "537", "2"): "2 .. RestrictedTradeable",
        ("FIX.5.0", "537", "3"): "3 .. Counter",
        ("FIX.5.0", "537", "4"): "4 .. InitiallyTradeable",
        ("FIX.5.0", "538", "1"): "1 .. ExecutingFirm",
        ("FIX.5.0", "538", "2"): "2 .. BrokerOfCredit",
        ("FIX.5.0", "538", "3"): "3 .. ClientID",
        ("FIX.5.0", "538", "4"): "4 .. ClearingFirm",
        ("FIX.5.0", "538", "5"): "5 .. InvestorID",
        ("FIX.5.0", "538", "6"): "6 .. IntroducingFirm",
        ("FIX.5.0", "538", "7"): "7 .. EnteringFirm",
        ("FIX.5.0", "538", "8"): "8 .. Locate",
        ("FIX.5.0", "538", "9"): "9 .. FundManagerClientID",
        ("FIX.5.0", "538", "10"): "10 .. SettlementLocation",
        ("FIX.5.0", "538", "11"): "11 .. OrderOriginationTrader",
        ("FIX.5.0", "538", "12"): "12 .. ExecutingTrader",
        ("FIX.5.0", "538", "13"): "13 .. OrderOriginationFirm",
        ("FIX.5.0", "538", "15"): "15 .. CorrespondantClearingFirm",
        ("FIX.5.0", "538", "16"): "16 .. ExecutingSystem",
        ("FIX.5.0", "538", "17"): "17 .. ContraFirm",
        ("FIX.5.0", "538", "18"): "18 .. ContraClearingFirm",
        ("FIX.5.0", "538", "19"): "19 .. SponsoringFirm",
        ("FIX.5.0", "538", "20"): "20 .. UnderlyingContraFirm",
        ("FIX.5.0", "538", "21"): "21 .. ClearingOrganization",
        ("FIX.5.0", "538", "22"): "22 .. Exchange",
        ("FIX.5.0", "538", "24"): "24 .. CustomerAccount",
        ("FIX.5.0", "538", "25"): "25 .. CorrespondentClearingOrganization",
        ("FIX.5.0", "538", "26"): "26 .. CorrespondentBroker",
        ("FIX.5.0", "538", "27"): "27 .. Buyer",
        ("FIX.5.0", "538", "28"): "28 .. Custodian",
        ("FIX.5.0", "538", "29"): "29 .. Intermediary",
        ("FIX.5.0", "538", "30"): "30 .. Agent",
        ("FIX.5.0", "538", "31"): "31 .. SubCustodian",
        ("FIX.5.0", "538", "32"): "32 .. Beneficiary",
        ("FIX.5.0", "538", "33"): "33 .. InterestedParty",
        ("FIX.5.0", "538", "34"): "34 .. RegulatoryBody",
        ("FIX.5.0", "538", "35"): "35 .. LiquidityProvider",
        ("FIX.5.0", "538", "36"): "36 .. EnteringTrader",
        ("FIX.5.0", "538", "37"): "37 .. ContraTrader",
        ("FIX.5.0", "538", "38"): "38 .. PositionAccount",
        ("FIX.5.0", "538", "39"): "39 .. ContraInvestorID",
        ("FIX.5.0", "538", "40"): "40 .. TransferToFirm",
        ("FIX.5.0", "538", "41"): "41 .. ContraPositionAccount",
        ("FIX.5.0", "538", "42"): "42 .. ContraExchange",
        ("FIX.5.0", "538", "43"): "43 .. InternalCarryAccount",
        ("FIX.5.0", "538", "44"): "44 .. OrderEntryOperatorID",
        ("FIX.5.0", "538", "45"): "45 .. SecondaryAccountNumber",
        ("FIX.5.0", "538", "46"): "46 .. ForeignFirm",
        ("FIX.5.0", "538", "47"): "47 .. ThirdPartyAllocationFirm",
        ("FIX.5.0", "538", "48"): "48 .. ClaimingAccount",
        ("FIX.5.0", "538", "49"): "49 .. AssetManager",
        ("FIX.5.0", "538", "50"): "50 .. PledgorAccount",
        ("FIX.5.0", "538", "51"): "51 .. PledgeeAccount",
        ("FIX.5.0", "538", "52"): "52 .. LargeTraderReportableAccount",
        ("FIX.5.0", "538", "53"): "53 .. TraderMnemonic",
        ("FIX.5.0", "538", "54"): "54 .. SenderLocation",
        ("FIX.5.0", "538", "55"): "55 .. SessionID",
        ("FIX.5.0", "538", "56"): "56 .. AcceptableCounterparty",
        ("FIX.5.0", "538", "57"): "57 .. UnacceptableCounterparty",
        ("FIX.5.0", "538", "58"): "58 .. EnteringUnit",
        ("FIX.5.0", "538", "59"): "59 .. ExecutingUnit",
        ("FIX.5.0", "538", "60"): "60 .. IntroducingBroker",
        ("FIX.5.0", "538", "61"): "61 .. QuoteOriginator",
        ("FIX.5.0", "538", "62"): "62 .. ReportOriginator",
        ("FIX.5.0", "538", "63"): "63 .. SystematicInternaliser",
        ("FIX.5.0", "538", "64"): "64 .. MultilateralTradingFacility",
        ("FIX.5.0", "538", "65"): "65 .. RegulatedMarket",
        ("FIX.5.0", "538", "66"): "66 .. MarketMaker",
        ("FIX.5.0", "538", "67"): "67 .. InvestmentFirm",
        ("FIX.5.0", "538", "68"): "68 .. HostCompetentAuthority",
        ("FIX.5.0", "538", "69"): "69 .. HomeCompetentAuthority",
        ("FIX.5.0", "538", "70"): "70 .. CompetentAuthorityLiquidity",
        ("FIX.5.0", "538", "71"): "71 .. CompetentAuthorityTransactionVenue",
        ("FIX.5.0", "538", "72"): "72 .. ReportingIntermediary",
        ("FIX.5.0", "538", "73"): "73 .. ExecutionVenue",
        ("FIX.5.0", "538", "74"): "74 .. MarketDataEntryOriginator",
        ("FIX.5.0", "538", "75"): "75 .. LocationID",
        ("FIX.5.0", "538", "76"): "76 .. DeskID",
        ("FIX.5.0", "538", "77"): "77 .. MarketDataMarket",
        ("FIX.5.0", "538", "78"): "78 .. AllocationEntity",
        ("FIX.5.0", "538", "79"): "79 .. PrimeBroker",
        ("FIX.5.0", "538", "80"): "80 .. StepOutFirm",
        ("FIX.5.0", "538", "81"): "81 .. BrokerClearingID",
        ("FIX.5.0", "538", "82"): "82 .. CentralRegistrationDepository",
        ("FIX.5.0", "538", "83"): "83 .. ClearingAccount",
        ("FIX.5.0", "538", "84"): "84 .. AcceptableSettlingCounterparty",
        ("FIX.5.0", "538", "85"): "85 .. UnacceptableSettlingCounterparty",
        ("FIX.5.0", "538", "86"): "86 .. CLSMemberBank",
        ("FIX.5.0", "538", "87"): "87 .. InConcertGroup",
        ("FIX.5.0", "538", "88"): "88 .. InConcertControllingEntity",
        ("FIX.5.0", "538", "89"): "89 .. LargePositionsReportingAccount",
        ("FIX.5.0", "538", "90"): "90 .. SettlementFirm",
        ("FIX.5.0", "538", "91"): "91 .. SettlementAccount",
        ("FIX.5.0", "538", "92"): "92 .. ReportingMarketCenter",
        ("FIX.5.0", "538", "93"): "93 .. RelatedReportingMarketCenter",
        ("FIX.5.0", "538", "94"): "94 .. AwayMarket",
        ("FIX.5.0", "538", "95"): "95 .. GiveupTradingFirm",
        ("FIX.5.0", "538", "96"): "96 .. TakeupTradingFirm",
        ("FIX.5.0", "538", "97"): "97 .. GiveupClearingFirm",
        ("FIX.5.0", "538", "98"): "98 .. TakeupClearingFirm",
        ("FIX.5.0", "538", "99"): "99 .. OriginatingMarket",
        ("FIX.5.0", "538", "100"): "100 .. MarginAccount",
        ("FIX.5.0", "538", "101"): "101 .. CollateralAssetAccount",
        ("FIX.5.0", "538", "102"): "102 .. DataRepository",
        ("FIX.5.0", "538", "103"): "103 .. CalculationAgent",
        ("FIX.5.0", "538", "104"): "104 .. ExerciseNoticeSender",
        ("FIX.5.0", "538", "105"): "105 .. ExerciseNoticeReceiver",
        ("FIX.5.0", "538", "106"): "106 .. RateReferenceBank",
        ("FIX.5.0", "538", "107"): "107 .. Correspondent",
        ("FIX.5.0", "538", "109"): "109 .. BeneficiaryBank",
        ("FIX.5.0", "538", "110"): "110 .. Borrower",
        ("FIX.5.0", "538", "111"): "111 .. PrimaryObligator",
        ("FIX.5.0", "538", "112"): "112 .. Guarantor",
        ("FIX.5.0", "538", "113"): "113 .. ExcludedReferenceEntity",
        ("FIX.5.0", "538", "114"): "114 .. DeterminingParty",
        ("FIX.5.0", "538", "115"): "115 .. HedgingParty",
        ("FIX.5.0", "538", "116"): "116 .. ReportingEntity",
        ("FIX.5.0", "538", "117"): "117 .. SalesPerson",
        ("FIX.5.0", "538", "118"): "118 .. Operator",
        ("FIX.5.0", "538", "119"): "119 .. CSD",
        ("FIX.5.0", "538", "120"): "120 .. ICSD",
        ("FIX.5.0", "538", "121"): "121 .. TradingSubAccount",
        ("FIX.5.0", "538", "122"): "122 .. InvestmentDecisionMaker",
        ("FIX.5.0", "538", "123"): "123 .. PublishingIntermediary",
        ("FIX.5.0", "544", "1"): "1 .. Cash",
        ("FIX.5.0", "544", "2"): "2 .. MarginOpen",
        ("FIX.5.0", "544", "3"): "3 .. MarginClose",
        ("FIX.5.0", "546", "1"): "1 .. LocalMarket",
        ("FIX.5.0", "546", "2"): "2 .. National",
        ("FIX.5.0", "546", "3"): "3 .. Global",
        ("FIX.5.0", "547", "N"): "N .. No",
        ("FIX.5.0", "547", "Y"): "Y .. Yes",
        ("FIX.5.0", "549", "1"): "1 .. CrossAON",
        ("FIX.5.0", "549", "2"): "2 .. CrossIOC",
        ("FIX.5.0", "549", "3"): "3 .. CrossOneSide",
        ("FIX.5.0", "549", "4"): "4 .. CrossSamePrice",
        ("FIX.5.0", "549", "5"): "5 .. BasisCross",
        ("FIX.5.0", "549", "6"): "6 .. ContingentCross",
        ("FIX.5.0", "549", "7"): "7 .. VWAPCross",
        ("FIX.5.0", "549", "8"): "8 .. STSCross",
        ("FIX.5.0", "549", "9"): "9 .. CustomerToCustomer",
        ("FIX.5.0", "550", "0"): "0 .. None",
        ("FIX.5.0", "550", "1"): "1 .. BuySideIsPrioritized",
        ("FIX.5.0", "550", "2"): "2 .. SellSideIsPrioritized",
        ("FIX.5.0", "552", "1"): "1 .. OneSide",
        ("FIX.5.0", "552", "2"): "2 .. BothSides",
        ("FIX.5.0", "559", "0"): "0 .. Symbol",
        ("FIX.5.0", "559", "1"): "1 .. SecurityTypeAnd",
        ("FIX.5.0", "559", "2"): "2 .. Product",
        ("FIX.5.0", "559", "3"): "3 .. TradingSessionID",
        ("FIX.5.0", "559", "4"): "4 .. AllSecurities",
        ("FIX.5.0", "559", "5"): "5 .. MarketIDOrMarketID",
        ("FIX.5.0", "560", "0"): "0 .. ValidRequest",
        ("FIX.5.0", "560", "1"): "1 .. InvalidOrUnsupportedRequest",
        ("FIX.5.0", "560", "2"): "2 .. NoInstrumentsFound",
        ("FIX.5.0", "560", "3"): "3 .. NotAuthorizedToRetrieveInstrumentData",
        ("FIX.5.0", "560", "4"): "4 .. InstrumentDataTemporarilyUnavailable",
        ("FIX.5.0", "560", "5"): "5 .. RequestForInstrumentDataNotSupported",
        ("FIX.5.0", "563", "0"): "0 .. ReportByMulitlegSecurityOnly",
        ("FIX.5.0", "563", "1"): "1 .. ReportByMultilegSecurityAndInstrumentLegs",
        ("FIX.5.0", "563", "2"): "2 .. ReportByInstrumentLegsOnly",
        ("FIX.5.0", "564", "C"): "C .. Close",
        ("FIX.5.0", "564", "F"): "F .. FIFO",
        ("FIX.5.0", "564", "O"): "O .. Open",
        ("FIX.5.0", "564", "R"): "R .. Rolled",
        ("FIX.5.0", "564", "N"): "N .. CloseButNotifyOnOpen",
        ("FIX.5.0", "564", "D"): "D .. Default",
        ("FIX.5.0", "565", "0"): "0 .. Covered",
        ("FIX.5.0", "565", "1"): "1 .. Uncovered",
        ("FIX.5.0", "567", "1"): "1 .. UnknownOrInvalidTradingSessionID",
        ("FIX.5.0", "567", "99"): "99 .. Other",
        ("FIX.5.0", "569", "0"): "0 .. AllTrades",
        ("FIX.5.0", "569", "1"): "1 .. MatchedTradesMatchingCriteria",
        ("FIX.5.0", "569", "2"): "2 .. UnmatchedTradesThatMatchCriteria",
        ("FIX.5.0", "569", "3"): "3 .. UnreportedTradesThatMatchCriteria",
        ("FIX.5.0", "569", "4"): "4 .. AdvisoriesThatMatchCriteria",
        ("FIX.5.0", "570", "N"): "N .. NotReportedToCounterparty",
        ("FIX.5.0", "570", "Y"): "Y .. PreviouslyReportedToCounterparty",
        ("FIX.5.0", "573", "0"): "0 .. Compared",
        ("FIX.5.0", "573", "1"): "1 .. Uncompared",
        ("FIX.5.0", "573", "2"): "2 .. AdvisoryOrAlert",
        ("FIX.5.0", "574", "1"): "1 .. OnePartyTradeReport",
        ("FIX.5.0", "574", "2"): "2 .. TwoPartyTradeReport",
        ("FIX.5.0", "574", "3"): "3 .. ConfirmedTradeReport",
        ("FIX.5.0", "574", "4"): "4 .. AutoMatch",
        ("FIX.5.0", "574", "5"): "5 .. CrossAuction",
        ("FIX.5.0", "574", "6"): "6 .. CounterOrderSelection",
        ("FIX.5.0", "574", "7"): "7 .. CallAuction",
        ("FIX.5.0", "574", "8"): "8 .. Issuing",
        ("FIX.5.0", "574", "9"): "9 .. SystematicInternaliser",
        ("FIX.5.0", "574", "10"): "10 .. AutoMatchLastLook",
        ("FIX.5.0", "574", "11"): "11 .. CrossAuctionLastLook",
        ("FIX.5.0", "574", "M3"): "M3 .. ACTAcceptedTrade",
        ("FIX.5.0", "574", "M4"): "M4 .. ACTDefaultTrade",
        ("FIX.5.0", "574", "M5"): "M5 .. ACTDefaultAfterM2",
        ("FIX.5.0", "574", "M6"): "M6 .. ACTM6Match",
        ("FIX.5.0", "574", "A1"): "A1 .. ExactMatchPlus4BadgesExecTime",
        ("FIX.5.0", "574", "A2"): "A2 .. ExactMatchPlus4Badges",
        ("FIX.5.0", "574", "A3"): "A3 .. ExactMatchPlus2BadgesExecTime",
        ("FIX.5.0", "574", "A4"): "A4 .. ExactMatchPlus2Badges",
        ("FIX.5.0", "574", "A5"): "A5 .. ExactMatchPlusExecTime",
        ("FIX.5.0", "574", "AQ"): "AQ .. StampedAdvisoriesOrSpecialistAccepts",
        ("FIX.5.0", "574", "S1"): "S1 .. A1ExactMatchSummarizedQuantity",
        ("FIX.5.0", "574", "S2"): "S2 .. A2ExactMatchSummarizedQuantity",
        ("FIX.5.0", "574", "S3"): "S3 .. A3ExactMatchSummarizedQuantity",
        ("FIX.5.0", "574", "S4"): "S4 .. A4ExactMatchSummarizedQuantity",
        ("FIX.5.0", "574", "S5"): "S5 .. A5ExactMatchSummarizedQuantity",
        ("FIX.5.0", "574", "M1"): "M1 .. ExactMatchMinusBadgesTimes",
        ("FIX.5.0", "574", "M2"): "M2 .. SummarizedMatchMinusBadgesTimes",
        ("FIX.5.0", "574", "MT"): "MT .. OCSLockedIn",
        ("FIX.5.0", "575", "N"): "N .. TreatAsRoundLot",
        ("FIX.5.0", "575", "Y"): "Y .. TreatAsOddLot",
        ("FIX.5.0", "577", "0"): "0 .. ProcessNormally",
        ("FIX.5.0", "577", "1"): "1 .. ExcludeFromAllNetting",
        ("FIX.5.0", "577", "2"): "2 .. BilateralNettingOnly",
        ("FIX.5.0", "577", "3"): "3 .. ExClearing",
        ("FIX.5.0", "577", "4"): "4 .. SpecialTrade",
        ("FIX.5.0", "577", "5"): "5 .. MultilateralNetting",
        ("FIX.5.0", "577", "6"): "6 .. ClearAgainstCentralCounterparty",
        ("FIX.5.0", "577", "7"): "7 .. ExcludeFromCentralCounterparty",
        ("FIX.5.0", "577", "8"): "8 .. ManualMode",
        ("FIX.5.0", "577", "9"): "9 .. AutomaticPostingMode",
        ("FIX.5.0", "577", "10"): "10 .. AutomaticGiveUpMode",
        ("FIX.5.0", "577", "11"): "11 .. QualifiedServiceRepresentativeQSR",
        ("FIX.5.0", "577", "12"): "12 .. CustomerTrade",
        ("FIX.5.0", "577", "13"): "13 .. SelfClearing",
        ("FIX.5.0", "577", "14"): "14 .. BuyIn",
        ("FIX.5.0", "581", "1"): "1 .. CarriedCustomerSide",
        ("FIX.5.0", "581", "2"): "2 .. CarriedNonCustomerSide",
        ("FIX.5.0", "581", "3"): "3 .. HouseTrader",
        ("FIX.5.0", "581", "4"): "4 .. FloorTrader",
        ("FIX.5.0", "581", "6"): "6 .. CarriedNonCustomerSideCrossMargined",
        ("FIX.5.0", "581", "7"): "7 .. HouseTraderCrossMargined",
        ("FIX.5.0", "581", "8"): "8 .. JointBackOfficeAccount",
        ("FIX.5.0", "581", "9"): "9 .. EquitiesSpecialist",
        ("FIX.5.0", "581", "10"): "10 .. OptionsMarketMaker",
        ("FIX.5.0", "581", "11"): "11 .. OptionsFirmAccount",
        ("FIX.5.0", "582", "1"): "1 .. MemberTradingForTheirOwnAccount",
        ("FIX.5.0", "582", "2"): "2 .. ClearingFirmTradingForItsProprietaryAccount",
        ("FIX.5.0", "582", "3"): "3 .. MemberTradingForAnotherMember",
        ("FIX.5.0", "582", "4"): "4 .. AllOther",
        ("FIX.5.0", "582", "5"): "5 .. RetailCustomer",
        ("FIX.5.0", "585", "1"): "1 .. StatusForOrdersForASecurity",
        ("FIX.5.0", "585", "2"): "2 .. StatusForOrdersForAnUnderlyingSecurity",
        ("FIX.5.0", "585", "3"): "3 .. StatusForOrdersForAProduct",
        ("FIX.5.0", "585", "4"): "4 .. StatusForOrdersForACFICode",
        ("FIX.5.0", "585", "5"): "5 .. StatusForOrdersForASecurityType",
        ("FIX.5.0", "585", "6"): "6 .. StatusForOrdersForATradingSession",
        ("FIX.5.0", "585", "7"): "7 .. StatusForAllOrders",
        ("FIX.5.0", "585", "8"): "8 .. StatusForOrdersForAPartyID",
        ("FIX.5.0", "585", "9"): "9 .. StatusForSecurityIssuer",
        ("FIX.5.0", "585", "10"): "10 .. StatusForIssuerOfUnderlyingSecurity",
        ("FIX.5.0", "587", "0"): "0 .. Regular",
        ("FIX.5.0", "587", "1"): "1 .. Cash",
        ("FIX.5.0", "587", "2"): "2 .. NextDay",
        ("FIX.5.0", "587", "3"): "3 .. TPlus2",
        ("FIX.5.0", "587", "4"): "4 .. TPlus3",
        ("FIX.5.0", "587", "5"): "5 .. TPlus4",
        ("FIX.5.0", "587", "6"): "6 .. Future",
        ("FIX.5.0", "587", "7"): "7 .. WhenAndIfIssued",
        ("FIX.5.0", "587", "8"): "8 .. SellersOption",
        ("FIX.5.0", "587", "9"): "9 .. TPlus5",
        ("FIX.5.0", "587", "B"): "B .. BrokenDate",
        ("FIX.5.0", "587", "C"): "C .. FXSpotNextSettlement",
        ("FIX.5.0", "589", "0"): "0 .. Auto",
        ("FIX.5.0", "589", "1"): "1 .. SpeakWithOrderInitiatorBeforeBooking",
        ("FIX.5.0", "589", "2"): "2 .. Accumulate",
        ("FIX.5.0", "590", "0"): "0 .. EachPartialExecutionIsABookableUnit",
        ("FIX.5.0", "590", "1"): "1 .. AggregatePartialExecutionsOnThisOrder",
        ("FIX.5.0", "590", "2"): "2 .. AggregateExecutionsForThisSymbol",
        ("FIX.5.0", "591", "0"): "0 .. ProRata",
        ("FIX.5.0", "591", "1"): "1 .. DoNotProRata",
        ("FIX.5.0", "601", "CD"): "CD .. EUCPWithLumpSumInterest",
        ("FIX.5.0", "601", "WI"): "WI .. WhenIssued",
        ("FIX.5.0", "603", "1"): "1 .. CUSIP",
        ("FIX.5.0", "603", "2"): "2 .. SEDOL",
        ("FIX.5.0", "603", "3"): "3 .. QUIK",
        ("FIX.5.0", "603", "4"): "4 .. ISINNumber",
        ("FIX.5.0", "603", "5"): "5 .. RICCode",
        ("FIX.5.0", "603", "6"): "6 .. ISOCurrencyCode",
        ("FIX.5.0", "603", "7"): "7 .. ISOCountryCode",
        ("FIX.5.0", "603", "8"): "8 .. ExchangeSymbol",
        ("FIX.5.0", "603", "9"): "9 .. ConsolidatedTapeAssociation",
        ("FIX.5.0", "603", "A"): "A .. BloombergSymbol",
        ("FIX.5.0", "603", "B"): "B .. Wertpapier",
        ("FIX.5.0", "603", "C"): "C .. Dutch",
        ("FIX.5.0", "603", "D"): "D .. Valoren",
        ("FIX.5.0", "603", "E"): "E .. Sicovam",
        ("FIX.5.0", "603", "F"): "F .. Belgian",
        ("FIX.5.0", "603", "G"): "G .. Common",
        ("FIX.5.0", "603", "H"): "H .. ClearingHouse",
        ("FIX.5.0", "603", "I"): "I .. ISDAFpMLSpecification",
        ("FIX.5.0", "603", "J"): "J .. OptionPriceReportingAuthority",
        ("FIX.5.0", "603", "K"): "K .. ISDAFpMLURL",
        ("FIX.5.0", "603", "L"): "L .. LetterOfCredit",
        ("FIX.5.0", "603", "M"): "M .. MarketplaceAssignedIdentifier",
        ("FIX.5.0", "603", "N"): "N .. MarkitREDEntityCLIP",
        ("FIX.5.0", "603", "P"): "P .. MarkitREDPairCLIP",
        ("FIX.5.0", "603", "Q"): "Q .. CFTCCommodityCode",
        ("FIX.5.0", "603", "R"): "R .. ISDACommodityReferencePrice",
        ("FIX.5.0", "603", "S"): "S .. FinancialInstrumentGlobalIdentifier",
        ("FIX.5.0", "603", "T"): "T .. LegalEntityIdentifier",
        ("FIX.5.0", "603", "U"): "U .. Synthetic",
        ("FIX.5.0", "603", "V"): "V .. FidessaInstrumentMnemonic",
        ("FIX.5.0", "603", "W"): "W .. IndexName",
        ("FIX.5.0", "606", "1"): "1 .. CUSIP",
        ("FIX.5.0", "606", "2"): "2 .. SEDOL",
        ("FIX.5.0", "606", "3"): "3 .. QUIK",
        ("FIX.5.0", "606", "4"): "4 .. ISINNumber",
        ("FIX.5.0", "606", "5"): "5 .. RICCode",
        ("FIX.5.0", "606", "6"): "6 .. ISOCurrencyCode",
        ("FIX.5.0", "606", "7"): "7 .. ISOCountryCode",
        ("FIX.5.0", "606", "8"): "8 .. ExchangeSymbol",
        ("FIX.5.0", "606", "9"): "9 .. ConsolidatedTapeAssociation",
        ("FIX.5.0", "606", "A"): "A .. BloombergSymbol",
        ("FIX.5.0", "606", "B"): "B .. Wertpapier",
        ("FIX.5.0", "606", "C"): "C .. Dutch",
        ("FIX.5.0", "606", "D"): "D .. Valoren",
        ("FIX.5.0", "606", "E"): "E .. Sicovam",
        ("FIX.5.0", "606", "F"): "F .. Belgian",
        ("FIX.5.0", "606", "G"): "G .. Common",
        ("FIX.5.0", "606", "H"): "H .. ClearingHouse",
        ("FIX.5.0", "606", "I"): "I .. ISDAFpMLSpecification",
        ("FIX.5.0", "606", "J"): "J .. OptionPriceReportingAuthority",
        ("FIX.5.0", "606", "K"): "K .. ISDAFpMLURL",
        ("FIX.5.0", "606", "L"): "L .. LetterOfCredit",
        ("FIX.5.0", "606", "M"): "M .. MarketplaceAssignedIdentifier",
        ("FIX.5.0", "606", "N"): "N .. MarkitREDEntityCLIP",
        ("FIX.5.0", "606", "P"): "P .. MarkitREDPairCLIP",
        ("FIX.5.0", "606", "Q"): "Q .. CFTCCommodityCode",
        ("FIX.5.0", "606", "R"): "R .. ISDACommodityReferencePrice",
        ("FIX.5.0", "606", "S"): "S .. FinancialInstrumentGlobalIdentifier",
        ("FIX.5.0", "606", "T"): "T .. LegalEntityIdentifier",
        ("FIX.5.0", "606", "U"): "U .. Synthetic",
        ("FIX.5.0", "606", "V"): "V .. FidessaInstrumentMnemonic",
        ("FIX.5.0", "606", "W"): "W .. IndexName",
        ("FIX.5.0", "607", "1"): "1 .. AGENCY",
        ("FIX.5.0", "607", "2"): "2 .. COMMODITY",
        ("FIX.5.0", "607", "3"): "3 .. CORPORATE",
        ("FIX.5.0", "607", "4"): "4 .. CURRENCY",
        ("FIX.5.0", "607", "5"): "5 .. EQUITY",
        ("FIX.5.0", "607", "6"): "6 .. GOVERNMENT",
        ("FIX.5.0", "607", "7"): "7 .. INDEX",
        ("FIX.5.0", "607", "8"): "8 .. LOAN",
        ("FIX.5.0", "607", "9"): "9 .. MONEYMARKET",
        ("FIX.5.0", "607", "10"): "10 .. MORTGAGE",
        ("FIX.5.0", "607", "11"): "11 .. MUNICIPAL",
        ("FIX.5.0", "607", "12"): "12 .. OTHER",
        ("FIX.5.0", "607", "13"): "13 .. FINANCING",
        ("FIX.5.0", "609", "EUSUPRA"): "EUSUPRA .. EuroSupranationalCoupons",
        ("FIX.5.0", "609", "FAC"): "FAC .. FederalAgencyCoupon",
        ("FIX.5.0", "609", "FADN"): "FADN .. FederalAgencyDiscountNote",
        ("FIX.5.0", "609", "PEF"): "PEF .. PrivateExportFunding",
        ("FIX.5.0", "609", "SUPRA"): "SUPRA .. USDSupranationalCoupons",
        ("FIX.5.0", "609", "CORP"): "CORP .. CorporateBond",
        ("FIX.5.0", "609", "CPP"): "CPP .. CorporatePrivatePlacement",
        ("FIX.5.0", "609", "CB"): "CB .. ConvertibleBond",
        ("FIX.5.0", "609", "DUAL"): "DUAL .. DualCurrency",
        ("FIX.5.0", "609", "EUCORP"): "EUCORP .. EuroCorporateBond",
        ("FIX.5.0", "609", "EUFRN"): "EUFRN .. EuroCorporateFloatingRateNotes",
        ("FIX.5.0", "609", "FRN"): "FRN .. USCorporateFloatingRateNotes",
        ("FIX.5.0", "609", "XLINKD"): "XLINKD .. IndexedLinked",
        ("FIX.5.0", "609", "STRUCT"): "STRUCT .. StructuredNotes",
        ("FIX.5.0", "609", "YANK"): "YANK .. YankeeCorporateBond",
        ("FIX.5.0", "609", "FXNDF"): "FXNDF .. NonDeliverableForward",
        ("FIX.5.0", "609", "FXSPOT"): "FXSPOT .. FXSpot",
        ("FIX.5.0", "609", "FXFWD"): "FXFWD .. FXForward",
        ("FIX.5.0", "609", "FXSWAP"): "FXSWAP .. FXSwap",
        ("FIX.5.0", "609", "FXNDS"): "FXNDS .. NonDeliverableSwap",
        ("FIX.5.0", "609", "CAP"): "CAP .. Cap",
        ("FIX.5.0", "609", "CDS"): "CDS .. CreditDefaultSwap",
        ("FIX.5.0", "609", "CLLR"): "CLLR .. Collar",
        ("FIX.5.0", "609", "CMDTYSWAP"): "CMDTYSWAP .. CommoditySwap",
        ("FIX.5.0", "609", "EXOTIC"): "EXOTIC .. Exotic",
        ("FIX.5.0", "609", "OOC"): "OOC .. OptionsOnCombo",
        ("FIX.5.0", "609", "FLR"): "FLR .. Floor",
        ("FIX.5.0", "609", "FRA"): "FRA .. FRA",
        ("FIX.5.0", "609", "FUT"): "FUT .. Future",
        ("FIX.5.0", "609", "FWD"): "FWD .. DerivativeForward",
        ("FIX.5.0", "609", "IRS"): "IRS .. InterestRateSwap",
        ("FIX.5.0", "609", "TRS"): "TRS .. TotalReturnSwap",
        ("FIX.5.0", "609", "LOANLEASE"): "LOANLEASE .. LoanLease",
        ("FIX.5.0", "609", "OOF"): "OOF .. OptionsOnFutures",
        ("FIX.5.0", "609", "OOP"): "OOP .. OptionsOnPhysical",
        ("FIX.5.0", "609", "OPT"): "OPT .. Option",
        ("FIX.5.0", "609", "SPOTFWD"): "SPOTFWD .. SpotForward",
        ("FIX.5.0", "609", "SWAPTION"): "SWAPTION .. SwapOption",
        ("FIX.5.0", "609", "XMISSION"): "XMISSION .. Transmission",
        ("FIX.5.0", "609", "INDEX"): "INDEX .. Index",
        ("FIX.5.0", "609", "BDBSKT"): "BDBSKT .. BondBasket",
        ("FIX.5.0", "609", "CFD"): "CFD .. ContractForDifference",
        ("FIX.5.0", "609", "CRLTNSWAP"): "CRLTNSWAP .. CorrelationSwap",
        ("FIX.5.0", "609", "DVDNDSWAP"): "DVDNDSWAP .. DiviendSwap",
        ("FIX.5.0", "609", "EQBSKT"): "EQBSKT .. EquityBasket",
        ("FIX.5.0", "609", "EQFWD"): "EQFWD .. EquityForward",
        ("FIX.5.0", "609", "RTRNSWAP"): "RTRNSWAP .. ReturnSwap",
        ("FIX.5.0", "609", "VARSWAP"): "VARSWAP .. VarianceSwap",
        ("FIX.5.0", "609", "PRTFLIOSWAP"): "PRTFLIOSWAP .. PortfolioSwaps",
        ("FIX.5.0", "609", "FUTSWAP"): "FUTSWAP .. FuturesOnASwap",
        ("FIX.5.0", "609", "FWDSWAP "): "FWDSWAP  .. ForwardsOnASwap",
        ("FIX.5.0", "609", "FWDFRTAGMT "): "FWDFRTAGMT  .. ForwardFreightAgreement",
        ("FIX.5.0", "609", "SPREADBET"): "SPREADBET .. SpreadBetting",
        ("FIX.5.0", "609", "ETC"): "ETC .. ExchangeTradedCommodity",
        ("FIX.5.0", "609", "CS"): "CS .. CommonStock",
        ("FIX.5.0", "609", "PS"): "PS .. PreferredStock",
        ("FIX.5.0", "609", "DR"): "DR .. DepositoryReceipts",
        ("FIX.5.0", "609", "REPO"): "REPO .. Repurchase",
        ("FIX.5.0", "609", "FORWARD"): "FORWARD .. Forward",
        ("FIX.5.0", "609", "BUYSELL"): "BUYSELL .. BuySellback",
        ("FIX.5.0", "609", "SECLOAN"): "SECLOAN .. SecuritiesLoan",
        ("FIX.5.0", "609", "SECPLEDGE"): "SECPLEDGE .. SecuritiesPledge",
        ("FIX.5.0", "609", "DVPLDG"): "DVPLDG .. DeliveryVersusPledge",
        ("FIX.5.0", "609", "COLLBSKT"): "COLLBSKT .. CollateralBasket",
        ("FIX.5.0", "609", "SFP"): "SFP .. StructuredFinanceProduct",
        ("FIX.5.0", "609", "BRADY"): "BRADY .. BradyBond",
        ("FIX.5.0", "609", "CAN"): "CAN .. CanadianTreasuryNotes",
        ("FIX.5.0", "609", "CTB"): "CTB .. CanadianTreasuryBills",
        ("FIX.5.0", "609", "EUSOV"): "EUSOV .. EuroSovereigns",
        ("FIX.5.0", "609", "PROV"): "PROV .. CanadianProvincialBonds",
        ("FIX.5.0", "609", "TB"): "TB .. TreasuryBill",
        ("FIX.5.0", "609", "TBOND"): "TBOND .. USTreasuryBond",
        ("FIX.5.0", "609", "TINT"): "TINT .. InterestStripFromAnyBondOrNote",
        ("FIX.5.0", "609", "TBILL"): "TBILL .. USTreasuryBill",
        ("FIX.5.0", "609", "TIPS"): "TIPS .. TreasuryInflationProtectedSecurities",
        ("FIX.5.0", "609", "TCAL"): "TCAL .. PrincipalStripOfACallableBondOrNote",
        ("FIX.5.0", "609", "TPRN"): "TPRN .. PrincipalStripFromANonCallableBondOrNote",
        ("FIX.5.0", "609", "TNOTE"): "TNOTE .. USTreasuryNote",
        ("FIX.5.0", "609", "TERM"): "TERM .. TermLoan",
        ("FIX.5.0", "609", "RVLV"): "RVLV .. RevolverLoan",
        ("FIX.5.0", "609", "RVLVTRM"): "RVLVTRM .. Revolver",
        ("FIX.5.0", "609", "BRIDGE"): "BRIDGE .. BridgeLoan",
        ("FIX.5.0", "609", "LOFC"): "LOFC .. LetterOfCredit",
        ("FIX.5.0", "609", "SWING"): "SWING .. SwingLineFacility",
        ("FIX.5.0", "609", "DINP"): "DINP .. DebtorInPossession",
        ("FIX.5.0", "609", "DEFLTED"): "DEFLTED .. Defaulted",
        ("FIX.5.0", "609", "WITHDRN"): "WITHDRN .. Withdrawn",
        ("FIX.5.0", "609", "REPLACD"): "REPLACD .. Replaced",
        ("FIX.5.0", "609", "MATURED"): "MATURED .. Matured",
        ("FIX.5.0", "609", "AMENDED"): "AMENDED .. Amended",
        ("FIX.5.0", "609", "RETIRED"): "RETIRED .. Retired",
        ("FIX.5.0", "609", "BA"): "BA .. BankersAcceptance",
        ("FIX.5.0", "609", "BDN"): "BDN .. BankDepositoryNote",
        ("FIX.5.0", "609", "BN"): "BN .. BankNotes",
        ("FIX.5.0", "609", "BOX"): "BOX .. BillOfExchanges",
        ("FIX.5.0", "609", "CAMM"): "CAMM .. CanadianMoneyMarkets",
        ("FIX.5.0", "609", "CD"): "CD .. CertificateOfDeposit",
        ("FIX.5.0", "609", "CL"): "CL .. CallLoans",
        ("FIX.5.0", "609", "CP"): "CP .. CommercialPaper",
        ("FIX.5.0", "609", "DN"): "DN .. DepositNotes",
        ("FIX.5.0", "609", "EUCD"): "EUCD .. EuroCertificateOfDeposit",
        ("FIX.5.0", "609", "EUCP"): "EUCP .. EuroCommercialPaper",
        ("FIX.5.0", "609", "LQN"): "LQN .. LiquidityNote",
        ("FIX.5.0", "609", "MTN"): "MTN .. MediumTermNotes",
        ("FIX.5.0", "609", "ONITE"): "ONITE .. Overnight",
        ("FIX.5.0", "609", "PN"): "PN .. PromissoryNote",
        ("FIX.5.0", "609", "STN"): "STN .. ShortTermLoanNote",
        ("FIX.5.0", "609", "PZFJ"): "PZFJ .. PlazosFijos",
        ("FIX.5.0", "609", "SLQN"): "SLQN .. SecuredLiquidityNote",
        ("FIX.5.0", "609", "TD"): "TD .. TimeDeposit",
        ("FIX.5.0", "609", "TLQN"): "TLQN .. TermLiquidityNote",
        ("FIX.5.0", "609", "XCN"): "XCN .. ExtendedCommNote",
        ("FIX.5.0", "609", "YCD"): "YCD .. YankeeCertificateOfDeposit",
        ("FIX.5.0", "609", "ABS"): "ABS .. AssetBackedSecurities",
        ("FIX.5.0", "609", "CMB"): "CMB .. CanadianMortgageBonds",
        ("FIX.5.0", "609", "CMBS"): "CMBS .. Corp",
        ("FIX.5.0", "609", "CMO"): "CMO .. CollateralizedMortgageObligation",
        ("FIX.5.0", "609", "IET"): "IET .. IOETTEMortgage",
        ("FIX.5.0", "609", "MBS"): "MBS .. MortgageBackedSecurities",
        ("FIX.5.0", "609", "MIO"): "MIO .. MortgageInterestOnly",
        ("FIX.5.0", "609", "MPO"): "MPO .. MortgagePrincipalOnly",
        ("FIX.5.0", "609", "MPP"): "MPP .. MortgagePrivatePlacement",
        ("FIX.5.0", "609", "MPT"): "MPT .. MiscellaneousPassThrough",
        ("FIX.5.0", "609", "PFAND"): "PFAND .. Pfandbriefe",
        ("FIX.5.0", "609", "TBA"): "TBA .. ToBeAnnounced",
        ("FIX.5.0", "609", "AN"): "AN .. OtherAnticipationNotes",
        ("FIX.5.0", "609", "COFO"): "COFO .. CertificateOfObligation",
        ("FIX.5.0", "609", "COFP"): "COFP .. CertificateOfParticipation",
        ("FIX.5.0", "609", "GO"): "GO .. GeneralObligationBonds",
        ("FIX.5.0", "609", "MT"): "MT .. MandatoryTender",
        ("FIX.5.0", "609", "RAN"): "RAN .. RevenueAnticipationNote",
        ("FIX.5.0", "609", "REV"): "REV .. RevenueBonds",
        ("FIX.5.0", "609", "SPCLA"): "SPCLA .. SpecialAssessment",
        ("FIX.5.0", "609", "SPCLO"): "SPCLO .. SpecialObligation",
        ("FIX.5.0", "609", "SPCLT"): "SPCLT .. SpecialTax",
        ("FIX.5.0", "609", "TAN"): "TAN .. TaxAnticipationNote",
        ("FIX.5.0", "609", "TAXA"): "TAXA .. TaxAllocation",
        ("FIX.5.0", "609", "TECP"): "TECP .. TaxExemptCommercialPaper",
        ("FIX.5.0", "609", "TMCP"): "TMCP .. TaxableMunicipalCP",
        ("FIX.5.0", "609", "TRAN"): "TRAN .. TaxRevenueAnticipationNote",
        ("FIX.5.0", "609", "VRDN"): "VRDN .. VariableRateDemandNote",
        ("FIX.5.0", "609", "WAR"): "WAR .. Warrant",
        ("FIX.5.0", "609", "MF"): "MF .. MutualFund",
        ("FIX.5.0", "609", "MLEG"): "MLEG .. MultilegInstrument",
        ("FIX.5.0", "609", "NONE"): "NONE .. NoSecurityType",
        ("FIX.5.0", "609", "?"): "? .. Wildcard",
        ("FIX.5.0", "609", "CASH"): "CASH .. Cash",
        ("FIX.5.0", "609", "Other"): "Other .. Other",
        ("FIX.5.0", "609", "ETN"): "ETN .. ExchangeTradedNote",
        ("FIX.5.0", "609", "SECDERIV"): "SECDERIV .. SecuritizedDerivative",
        ("FIX.5.0", "624", "1"): "1 .. Buy",
        ("FIX.5.0", "624", "2"): "2 .. Sell",
        ("FIX.5.0", "624", "3"): "3 .. BuyMinus",
        ("FIX.5.0", "624", "4"): "4 .. SellPlus",
        ("FIX.5.0", "624", "5"): "5 .. SellShort",
        ("FIX.5.0", "624", "6"): "6 .. SellShortExempt",
        ("FIX.5.0", "624", "7"): "7 .. Undisclosed",
        ("FIX.5.0", "624", "8"): "8 .. Cross",
        ("FIX.5.0", "624", "9"): "9 .. CrossShort",
        ("FIX.5.0", "624", "A"): "A .. CrossShortExempt",
        ("FIX.5.0", "624", "B"): "B .. AsDefined",
        ("FIX.5.0", "624", "C"): "C .. Opposite",
        ("FIX.5.0", "624", "D"): "D .. Subscribe",
        ("FIX.5.0", "624", "E"): "E .. Redeem",
        ("FIX.5.0", "624", "F"): "F .. Lend",
        ("FIX.5.0", "624", "G"): "G .. Borrow",
        ("FIX.5.0", "624", "H"): "H .. SellUndisclosed",
        ("FIX.5.0", "625", "1"): "1 .. PreTrading",
        ("FIX.5.0", "625", "2"): "2 .. OpeningOrOpeningAuction",
        ("FIX.5.0", "625", "3"): "3 .. Continuous",
        ("FIX.5.0", "625", "4"): "4 .. ClosingOrClosingAuction",
        ("FIX.5.0", "625", "5"): "5 .. PostTrading",
        ("FIX.5.0", "625", "6"): "6 .. ScheduledIntradayAuction",
        ("FIX.5.0", "625", "7"): "7 .. Quiescent",
        ("FIX.5.0", "625", "8"): "8 .. AnyAuction",
        ("FIX.5.0", "625", "9"): "9 .. UnscheduledIntradayAuction",
        ("FIX.5.0", "625", "10"): "10 .. OutOfMainSessionTrading",
        ("FIX.5.0", "625", "11"): "11 .. PrivateAuction",
        ("FIX.5.0", "625", "12"): "12 .. PublicAuction",
        ("FIX.5.0", "625", "13"): "13 .. GroupAuction",
        ("FIX.5.0", "626", "1"): "1 .. Calculated",
        ("FIX.5.0", "626", "2"): "2 .. Preliminary",
        ("FIX.5.0", "626", "5"): "5 .. ReadyToBook",
        ("FIX.5.0", "626", "7"): "7 .. WarehouseInstruction",
        ("FIX.5.0", "626", "8"): "8 .. RequestToIntermediary",
        ("FIX.5.0", "626", "12"): "12 .. IncompleteGroup",
        ("FIX.5.0", "626", "13"): "13 .. CompleteGroup",
        ("FIX.5.0", "626", "15"): "15 .. ReopenGroup",
        ("FIX.5.0", "626", "16"): "16 .. CancelGroup",
        ("FIX.5.0", "626", "17"): "17 .. Giveup",
        ("FIX.5.0", "626", "18"): "18 .. Takeup",
        ("FIX.5.0", "626", "19"): "19 .. RefuseTakeup",
        ("FIX.5.0", "626", "20"): "20 .. InitiateReversal",
        ("FIX.5.0", "626", "21"): "21 .. Reverse",
        ("FIX.5.0", "626", "22"): "22 .. RefuseReversal",
        ("FIX.5.0", "626", "23"): "23 .. SubAllocationGiveup",
        ("FIX.5.0", "626", "24"): "24 .. ApproveGiveup",
        ("FIX.5.0", "626", "25"): "25 .. ApproveTakeup",
        ("FIX.5.0", "626", "26"): "26 .. NotionalValueAveragePxGroupAlloc",
        ("FIX.5.0", "635", "1"): "1 .. FirstYearDelegate",
        ("FIX.5.0", "635", "2"): "2 .. SecondYearDelegate",
        ("FIX.5.0", "635", "3"): "3 .. ThirdYearDelegate",
        ("FIX.5.0", "635", "4"): "4 .. FourthYearDelegate",
        ("FIX.5.0", "635", "5"): "5 .. FifthYearDelegate",
        ("FIX.5.0", "635", "9"): "9 .. SixthYearDelegate",
        ("FIX.5.0", "635", "B"): "B .. CBOEMember",
        ("FIX.5.0", "635", "C"): "C .. NonMemberAndCustomer",
        ("FIX.5.0", "635", "E"): "E .. EquityMemberAndClearingMember",
        ("FIX.5.0", "635", "F"): "F .. FullAndAssociateMember",
        ("FIX.5.0", "635", "H"): "H .. Firms106HAnd106J",
        ("FIX.5.0", "635", "I"): "I .. GIM",
        ("FIX.5.0", "635", "L"): "L .. Lessee106FEmployees",
        ("FIX.5.0", "635", "M"): "M .. AllOtherOwnershipTypes",
        ("FIX.5.0", "636", "N"): "N .. NotWorking",
        ("FIX.5.0", "636", "Y"): "Y .. Working",
        ("FIX.5.0", "638", "0"): "0 .. PriorityUnchanged",
        ("FIX.5.0", "638", "1"): "1 .. LostPriorityAsResultOfOrderChange",
        ("FIX.5.0", "650", "N"): "N .. DoesNotConsituteALegalConfirm",
        ("FIX.5.0", "650", "Y"): "Y .. LegalConfirm",
        ("FIX.5.0", "658", "1"): "1 .. UnknownSymbol",
        ("FIX.5.0", "658", "2"): "2 .. Exchange",
        ("FIX.5.0", "658", "3"): "3 .. QuoteRequestExceedsLimit",
        ("FIX.5.0", "658", "4"): "4 .. TooLateToEnter",
        ("FIX.5.0", "658", "5"): "5 .. InvalidPrice",
        ("FIX.5.0", "658", "6"): "6 .. NotAuthorizedToRequestQuote",
        ("FIX.5.0", "658", "7"): "7 .. NoMatchForInquiry",
        ("FIX.5.0", "658", "8"): "8 .. NoMarketForInstrument",
        ("FIX.5.0", "658", "9"): "9 .. NoInventory",
        ("FIX.5.0", "658", "10"): "10 .. Pass",
        ("FIX.5.0", "658", "11"): "11 .. InsufficientCredit",
        ("FIX.5.0", "658", "12"): "12 .. ExceededClipSizeLimit",
        ("FIX.5.0", "658", "13"): "13 .. ExceededMaxNotionalOrderAmt",
        ("FIX.5.0", "658", "14"): "14 .. ExceededDV01PV01Limit",
        ("FIX.5.0", "658", "15"): "15 .. ExceededCS01Limit",
        ("FIX.5.0", "658", "99"): "99 .. Other",
        ("FIX.5.0", "660", "1"): "1 .. BIC",
        ("FIX.5.0", "660", "2"): "2 .. SIDCode",
        ("FIX.5.0", "660", "3"): "3 .. TFM",
        ("FIX.5.0", "660", "4"): "4 .. OMGEO",
        ("FIX.5.0", "660", "5"): "5 .. DTCCCode",
        ("FIX.5.0", "660", "99"): "99 .. Other",
        ("FIX.5.0", "661", "1"): "1 .. BIC",
        ("FIX.5.0", "661", "2"): "2 .. SIDCode",
        ("FIX.5.0", "661", "3"): "3 .. TFM",
        ("FIX.5.0", "661", "4"): "4 .. OMGEO",
        ("FIX.5.0", "661", "5"): "5 .. DTCCCode",
        ("FIX.5.0", "661", "99"): "99 .. Other",
        ("FIX.5.0", "663", "1"): "1 .. Percentage",
        ("FIX.5.0", "663", "2"): "2 .. PerUnit",
        ("FIX.5.0", "663", "3"): "3 .. FixedAmount",
        ("FIX.5.0", "663", "4"): "4 .. Discount",
        ("FIX.5.0", "663", "5"): "5 .. Premium",
        ("FIX.5.0", "663", "6"): "6 .. Spread",
        ("FIX.5.0", "663", "7"): "7 .. TEDPrice",
        ("FIX.5.0", "663", "8"): "8 .. TEDYield",
        ("FIX.5.0", "663", "9"): "9 .. Yield",
        ("FIX.5.0", "663", "10"): "10 .. FixedCabinetTradePrice",
        ("FIX.5.0", "663", "11"): "11 .. VariableCabinetTradePrice",
        ("FIX.5.0", "663", "12"): "12 .. PriceSpread",
        ("FIX.5.0", "663", "13"): "13 .. ProductTicksInHalves",
        ("FIX.5.0", "663", "14"): "14 .. ProductTicksInFourths",
        ("FIX.5.0", "663", "15"): "15 .. ProductTicksInEighths",
        ("FIX.5.0", "663", "16"): "16 .. ProductTicksInSixteenths",
        ("FIX.5.0", "663", "17"): "17 .. ProductTicksInThirtySeconds",
        ("FIX.5.0", "663", "18"): "18 .. ProductTicksInSixtyFourths",
        ("FIX.5.0", "663", "19"): "19 .. ProductTicksInOneTwentyEighths",
        ("FIX.5.0", "663", "20"): "20 .. NormalRateRepresentation",
        ("FIX.5.0", "663", "21"): "21 .. InverseRateRepresentation",
        ("FIX.5.0", "663", "22"): "22 .. BasisPoints",
        ("FIX.5.0", "663", "23"): "23 .. UpfrontPoints",
        ("FIX.5.0", "663", "24"): "24 .. InterestRate",
        ("FIX.5.0", "663", "25"): "25 .. PercentageNotional",
        ("FIX.5.0", "665", "1"): "1 .. Received",
        ("FIX.5.0", "665", "2"): "2 .. MismatchedAccount",
        ("FIX.5.0", "665", "3"): "3 .. MissingSettlementInstructions",
        ("FIX.5.0", "665", "4"): "4 .. Confirmed",
        ("FIX.5.0", "665", "5"): "5 .. RequestRejected",
        ("FIX.5.0", "666", "0"): "0 .. New",
        ("FIX.5.0", "666", "1"): "1 .. Replace",
        ("FIX.5.0", "666", "2"): "2 .. Cancel",
        ("FIX.5.0", "668", "1"): "1 .. BookEntry",
        ("FIX.5.0", "668", "2"): "2 .. Bearer",
        ("FIX.5.0", "677", "EONIA"): "EONIA .. EONIA",
        ("FIX.5.0", "677", "EUREPO"): "EUREPO .. EUREPO",
        ("FIX.5.0", "677", "FutureSWAP"): "FutureSWAP .. FutureSWAP",
        ("FIX.5.0", "677", "LIBID"): "LIBID .. LIBID",
        ("FIX.5.0", "677", "LIBOR"): "LIBOR .. LIBOR",
        ("FIX.5.0", "677", "MuniAAA"): "MuniAAA .. MuniAAA",
        ("FIX.5.0", "677", "OTHER"): "OTHER .. OTHER",
        ("FIX.5.0", "677", "Pfandbriefe"): "Pfandbriefe .. Pfandbriefe",
        ("FIX.5.0", "677", "SONIA"): "SONIA .. SONIA",
        ("FIX.5.0", "677", "SWAP"): "SWAP .. SWAP",
        ("FIX.5.0", "677", "Treasury"): "Treasury .. Treasury",
        ("FIX.5.0", "677", "FEDEFF"): "FEDEFF .. FedFundRateEffective",
        ("FIX.5.0", "677", "FEDOPEN"): "FEDOPEN .. FedOpen",
        ("FIX.5.0", "677", "EURIBOR"): "EURIBOR .. EURIBOR",
        ("FIX.5.0", "680", "1"): "1 .. Percentage",
        ("FIX.5.0", "680", "2"): "2 .. PerUnit",
        ("FIX.5.0", "680", "3"): "3 .. FixedAmount",
        ("FIX.5.0", "680", "4"): "4 .. Discount",
        ("FIX.5.0", "680", "5"): "5 .. Premium",
        ("FIX.5.0", "680", "6"): "6 .. Spread",
        ("FIX.5.0", "680", "7"): "7 .. TEDPrice",
        ("FIX.5.0", "680", "8"): "8 .. TEDYield",
        ("FIX.5.0", "680", "9"): "9 .. Yield",
        ("FIX.5.0", "680", "10"): "10 .. FixedCabinetTradePrice",
        ("FIX.5.0", "680", "11"): "11 .. VariableCabinetTradePrice",
        ("FIX.5.0", "680", "12"): "12 .. PriceSpread",
        ("FIX.5.0", "680", "13"): "13 .. ProductTicksInHalves",
        ("FIX.5.0", "680", "14"): "14 .. ProductTicksInFourths",
        ("FIX.5.0", "680", "15"): "15 .. ProductTicksInEighths",
        ("FIX.5.0", "680", "16"): "16 .. ProductTicksInSixteenths",
        ("FIX.5.0", "680", "17"): "17 .. ProductTicksInThirtySeconds",
        ("FIX.5.0", "680", "18"): "18 .. ProductTicksInSixtyFourths",
        ("FIX.5.0", "680", "19"): "19 .. ProductTicksInOneTwentyEighths",
        ("FIX.5.0", "680", "20"): "20 .. NormalRateRepresentation",
        ("FIX.5.0", "680", "21"): "21 .. InverseRateRepresentation",
        ("FIX.5.0", "680", "22"): "22 .. BasisPoints",
        ("FIX.5.0", "680", "23"): "23 .. UpfrontPoints",
        ("FIX.5.0", "680", "24"): "24 .. InterestRate",
        ("FIX.5.0", "680", "25"): "25 .. PercentageNotional",
        ("FIX.5.0", "682", "S"): "S .. Small",
        ("FIX.5.0", "682", "M"): "M .. Medium",
        ("FIX.5.0", "682", "L"): "L .. Large",
        ("FIX.5.0", "682", "U"): "U .. UndisclosedQuantity",
        ("FIX.5.0", "686", "1"): "1 .. Percentage",
        ("FIX.5.0", "686", "2"): "2 .. PerUnit",
        ("FIX.5.0", "686", "3"): "3 .. FixedAmount",
        ("FIX.5.0", "686", "4"): "4 .. Discount",
        ("FIX.5.0", "686", "5"): "5 .. Premium",
        ("FIX.5.0", "686", "6"): "6 .. Spread",
        ("FIX.5.0", "686", "7"): "7 .. TEDPrice",
        ("FIX.5.0", "686", "8"): "8 .. TEDYield",
        ("FIX.5.0", "686", "9"): "9 .. Yield",
        ("FIX.5.0", "686", "10"): "10 .. FixedCabinetTradePrice",
        ("FIX.5.0", "686", "11"): "11 .. VariableCabinetTradePrice",
        ("FIX.5.0", "686", "12"): "12 .. PriceSpread",
        ("FIX.5.0", "686", "13"): "13 .. ProductTicksInHalves",
        ("FIX.5.0", "686", "14"): "14 .. ProductTicksInFourths",
        ("FIX.5.0", "686", "15"): "15 .. ProductTicksInEighths",
        ("FIX.5.0", "686", "16"): "16 .. ProductTicksInSixteenths",
        ("FIX.5.0", "686", "17"): "17 .. ProductTicksInThirtySeconds",
        ("FIX.5.0", "686", "18"): "18 .. ProductTicksInSixtyFourths",
        ("FIX.5.0", "686", "19"): "19 .. ProductTicksInOneTwentyEighths",
        ("FIX.5.0", "686", "20"): "20 .. NormalRateRepresentation",
        ("FIX.5.0", "686", "21"): "21 .. InverseRateRepresentation",
        ("FIX.5.0", "686", "22"): "22 .. BasisPoints",
        ("FIX.5.0", "686", "23"): "23 .. UpfrontPoints",
        ("FIX.5.0", "686", "24"): "24 .. InterestRate",
        ("FIX.5.0", "686", "25"): "25 .. PercentageNotional",
        ("FIX.5.0", "688", "AMT"): "AMT .. AlternativeMinimumTax",
        ("FIX.5.0", "688", "AUTOREINV"): "AUTOREINV .. AutoReinvestment",
        ("FIX.5.0", "688", "BANKQUAL"): "BANKQUAL .. BankQualified",
        ("FIX.5.0", "688", "BGNCON"): "BGNCON .. BargainConditions",
        ("FIX.5.0", "688", "COUPON"): "COUPON .. CouponRange",
        ("FIX.5.0", "688", "CURRENCY"): "CURRENCY .. ISOCurrencyCode",
        ("FIX.5.0", "688", "CUSTOMDATE"): "CUSTOMDATE .. CustomStart",
        ("FIX.5.0", "688", "GEOG"): "GEOG .. Geographics",
        ("FIX.5.0", "688", "HAIRCUT"): "HAIRCUT .. ValuationDiscount",
        ("FIX.5.0", "688", "INSURED"): "INSURED .. Insured",
        ("FIX.5.0", "688", "ISSUE"): "ISSUE .. IssueDate",
        ("FIX.5.0", "688", "ISSUER"): "ISSUER .. Issuer",
        ("FIX.5.0", "688", "ISSUESIZE"): "ISSUESIZE .. IssueSizeRange",
        ("FIX.5.0", "688", "LOOKBACK"): "LOOKBACK .. LookbackDays",
        ("FIX.5.0", "688", "LOT"): "LOT .. ExplicitLotIdentifier",
        ("FIX.5.0", "688", "LOTVAR"): "LOTVAR .. LotVariance",
        ("FIX.5.0", "688", "MAT"): "MAT .. MaturityYearAndMonth",
        ("FIX.5.0", "688", "MATURITY"): "MATURITY .. MaturityRange",
        ("FIX.5.0", "688", "MAXSUBS"): "MAXSUBS .. MaximumSubstitutions",
        ("FIX.5.0", "688", "MINDNOM"): "MINDNOM .. MinimumDenomination",
        ("FIX.5.0", "688", "MININCR"): "MININCR .. MinimumIncrement",
        ("FIX.5.0", "688", "MINQTY"): "MINQTY .. MinimumQuantity",
        ("FIX.5.0", "688", "PAYFREQ"): "PAYFREQ .. PaymentFrequency",
        ("FIX.5.0", "688", "PIECES"): "PIECES .. NumberOfPieces",
        ("FIX.5.0", "688", "PMAX"): "PMAX .. PoolsMaximum",
        ("FIX.5.0", "688", "PPL"): "PPL .. PoolsPerLot",
        ("FIX.5.0", "688", "PPM"): "PPM .. PoolsPerMillion",
        ("FIX.5.0", "688", "PPT"): "PPT .. PoolsPerTrade",
        ("FIX.5.0", "688", "PRICE"): "PRICE .. PriceRange",
        ("FIX.5.0", "688", "PRICEFREQ"): "PRICEFREQ .. PricingFrequency",
        ("FIX.5.0", "688", "PROD"): "PROD .. ProductionYear",
        ("FIX.5.0", "688", "PROTECT"): "PROTECT .. CallProtection",
        ("FIX.5.0", "688", "PURPOSE"): "PURPOSE .. Purpose",
        ("FIX.5.0", "688", "PXSOURCE"): "PXSOURCE .. BenchmarkPriceSource",
        ("FIX.5.0", "688", "RATING"): "RATING .. RatingSourceAndRange",
        ("FIX.5.0", "688", "REDEMPTION"): "REDEMPTION .. TypeOfRedemption",
        ("FIX.5.0", "688", "RESTRICTED"): "RESTRICTED .. Restricted",
        ("FIX.5.0", "688", "SECTOR"): "SECTOR .. MarketSector",
        ("FIX.5.0", "688", "SECTYPE"): "SECTYPE .. SecurityTypeIncludedOrExcluded",
        ("FIX.5.0", "688", "STRUCT"): "STRUCT .. Structure",
        ("FIX.5.0", "688", "SUBSFREQ"): "SUBSFREQ .. SubstitutionsFrequency",
        ("FIX.5.0", "688", "SUBSLEFT"): "SUBSLEFT .. SubstitutionsLeft",
        ("FIX.5.0", "688", "TEXT"): "TEXT .. FreeformText",
        ("FIX.5.0", "688", "TRDVAR"): "TRDVAR .. TradeVariance",
        ("FIX.5.0", "688", "WAC"): "WAC .. WeightedAverageCoupon",
        ("FIX.5.0", "688", "WAL"): "WAL .. WeightedAverageLifeCoupon",
        ("FIX.5.0", "688", "WALA"): "WALA .. WeightedAverageLoanAge",
        ("FIX.5.0", "688", "WAM"): "WAM .. WeightedAverageMaturity",
        ("FIX.5.0", "688", "WHOLE"): "WHOLE .. WholePool",
        ("FIX.5.0", "688", "YIELD"): "YIELD .. YieldRange",
        ("FIX.5.0", "688", "ORIGAMT"): "ORIGAMT .. OriginalAmount",
        ("FIX.5.0", "688", "POOLEFFDT"): "POOLEFFDT .. PoolEffectiveDate",
        ("FIX.5.0", "688", "POOLINITFCTR"): "POOLINITFCTR .. PoolInitialFactor",
        ("FIX.5.0", "688", "TRANCHE"): "TRANCHE .. Tranche",
        ("FIX.5.0", "688", "SUBSTITUTION"): "SUBSTITUTION .. Substitution",
        ("FIX.5.0", "688", "MULTEXCHFLLBCK"): "MULTEXCHFLLBCK .. MULTEXCHFLLBCK",
        ("FIX.5.0", "688", "COMPSECFLLBCK"): "COMPSECFLLBCK .. COMPSECFLLBCK",
        ("FIX.5.0", "688", "LOCLJRSDCTN"): "LOCLJRSDCTN .. LOCLJRSDCTN",
        ("FIX.5.0", "688", "RELVJRSDCTN"): "RELVJRSDCTN .. RELVJRSDCTN",
        ("FIX.5.0", "688", "INCURRCVY"): "INCURRCVY .. IncurredRecovery",
        ("FIX.5.0", "688", "ADDTRM"): "ADDTRM .. AdditionalTerm",
        ("FIX.5.0", "688", "MODEQTYDLVY"): "MODEQTYDLVY .. ModifiedEquityDelivery",
        ("FIX.5.0", "688", "NOREFOBLIG"): "NOREFOBLIG .. NoReferenceOblication",
        ("FIX.5.0", "688", "UNKREFOBLIG"): "UNKREFOBLIG .. UnknownReferenceObligation",
        ("FIX.5.0", "688", "ALLGUARANTEES"): "ALLGUARANTEES .. AllGuarantees",
        ("FIX.5.0", "688", "REFPX"): "REFPX .. ReferencePrice",
        ("FIX.5.0", "688", "REFPOLICY"): "REFPOLICY .. ReferencePolicy",
        ("FIX.5.0", "688", "SECRDLIST"): "SECRDLIST .. SecuredList",
        ("FIX.5.0", "688", "AVFICO"): "AVFICO .. AverageFICOScore",
        ("FIX.5.0", "688", "AVSIZE"): "AVSIZE .. AverageLoanSize",
        ("FIX.5.0", "688", "MAXBAL"): "MAXBAL .. MaximumLoanBalance",
        ("FIX.5.0", "688", "POOL"): "POOL .. PoolIdentifier",
        ("FIX.5.0", "688", "ROLLTYPE"): "ROLLTYPE .. TypeOfRollTrade",
        ("FIX.5.0", "688", "REFTRADE"): "REFTRADE .. ReferenceToRollingOrClosingTrade",
        ("FIX.5.0", "688", "REFPRIN"): "REFPRIN .. PrincipalOfRollingOrClosingTrade",
        ("FIX.5.0", "688", "REFINT"): "REFINT .. InterestOfRollingOrClosingTrade",
        ("FIX.5.0", "688", "AVAILQTY"): "AVAILQTY .. AvailableOfferQuantityToBeShownToTheStreet",
        ("FIX.5.0", "688", "BROKERCREDIT"): "BROKERCREDIT .. BrokerCredit",
        ("FIX.5.0", "688", "INTERNALPX"): "INTERNALPX .. OfferPriceToBeShownToInternalBrokers",
        ("FIX.5.0", "688", "INTERNALQTY"): "INTERNALQTY .. OfferQuantityToBeShownToInternalBrokers",
        ("FIX.5.0", "688", "LEAVEQTY"): "LEAVEQTY .. TheMinimumResidualOfferQuantity",
        ("FIX.5.0", "688", "MAXORDQTY"): "MAXORDQTY .. MaximumOrderSize",
        ("FIX.5.0", "688", "ORDRINCR"): "ORDRINCR .. OrderQuantityIncrement",
        ("FIX.5.0", "688", "PRIMARY"): "PRIMARY .. PrimaryOrSecondaryMarketIndicator",
        ("FIX.5.0", "688", "SALESCREDITOVR"): "SALESCREDITOVR .. BrokerSalesCreditOverride",
        ("FIX.5.0", "688", "TRADERCREDIT"): "TRADERCREDIT .. TraderCredit",
        ("FIX.5.0", "688", "DISCOUNT"): "DISCOUNT .. DiscountRate",
        ("FIX.5.0", "688", "YTM"): "YTM .. YieldToMaturity",
        ("FIX.5.0", "688", "ABS"): "ABS .. AbsolutePrepaymentSpeed",
        ("FIX.5.0", "688", "CPP"): "CPP .. ConstantPrepaymentPenalty",
        ("FIX.5.0", "688", "CPR"): "CPR .. ConstantPrepaymentRate",
        ("FIX.5.0", "688", "CPY"): "CPY .. ConstantPrepaymentYield",
        ("FIX.5.0", "688", "HEP"): "HEP .. FinalCPROfHomeEquityPrepaymentCurve",
        ("FIX.5.0", "688", "MHP"): "MHP .. PercentOfManufacturedHousingPrepaymentCurve",
        ("FIX.5.0", "688", "MPR"): "MPR .. MonthlyPrepaymentRate",
        ("FIX.5.0", "688", "PPC"): "PPC .. PercentOfProspectusPrepaymentCurve",
        ("FIX.5.0", "688", "PSA"): "PSA .. PercentOfBMAPrepaymentCurve",
        ("FIX.5.0", "688", "SMM"): "SMM .. SingleMonthlyMortality",
        ("FIX.5.0", "690", "1"): "1 .. ParForPar",
        ("FIX.5.0", "690", "2"): "2 .. ModifiedDuration",
        ("FIX.5.0", "690", "4"): "4 .. Risk",
        ("FIX.5.0", "690", "5"): "5 .. Proceeds",
        ("FIX.5.0", "692", "1"): "1 .. Percent",
        ("FIX.5.0", "692", "2"): "2 .. PerShare",
        ("FIX.5.0", "692", "3"): "3 .. FixedAmount",
        ("FIX.5.0", "692", "4"): "4 .. Discount",
        ("FIX.5.0", "692", "5"): "5 .. Premium",
        ("FIX.5.0", "692", "6"): "6 .. Spread",
        ("FIX.5.0", "692", "7"): "7 .. TEDPrice",
        ("FIX.5.0", "692", "8"): "8 .. TEDYield",
        ("FIX.5.0", "692", "9"): "9 .. YieldSpread",
        ("FIX.5.0", "692", "10"): "10 .. Yield",
        ("FIX.5.0", "692", "12"): "12 .. PriceSpread",
        ("FIX.5.0", "692", "13"): "13 .. ProductTicksInHalves",
        ("FIX.5.0", "692", "14"): "14 .. ProductTicksInFourths",
        ("FIX.5.0", "692", "15"): "15 .. ProductTicksInEighths",
        ("FIX.5.0", "692", "16"): "16 .. ProductTicksInSixteenths",
        ("FIX.5.0", "692", "17"): "17 .. ProductTicksInThirtySeconds",
        ("FIX.5.0", "692", "18"): "18 .. ProductTicksInSixtyFourths",
        ("FIX.5.0", "692", "19"): "19 .. ProductTicksInOneTwentyEighths",
        ("FIX.5.0", "692", "20"): "20 .. NormalRateRepresentation",
        ("FIX.5.0", "692", "21"): "21 .. InverseRateRepresentation",
        ("FIX.5.0", "692", "22"): "22 .. BasisPoints",
        ("FIX.5.0", "692", "23"): "23 .. UpFrontPoints",
        ("FIX.5.0", "692", "24"): "24 .. InterestRate",
        ("FIX.5.0", "692", "25"): "25 .. PercentageOfNotional",
        ("FIX.5.0", "694", "1"): "1 .. Hit",
        ("FIX.5.0", "694", "2"): "2 .. Counter",
        ("FIX.5.0", "694", "3"): "3 .. Expired",
        ("FIX.5.0", "694", "4"): "4 .. Cover",
        ("FIX.5.0", "694", "5"): "5 .. DoneAway",
        ("FIX.5.0", "694", "6"): "6 .. Pass",
        ("FIX.5.0", "694", "7"): "7 .. EndTrade",
        ("FIX.5.0", "694", "8"): "8 .. TimedOut",
        ("FIX.5.0", "694", "9"): "9 .. Tied",
        ("FIX.5.0", "694", "10"): "10 .. TiedCover",
        ("FIX.5.0", "695", "A"): "A .. AllOrNone",
        ("FIX.5.0", "695", "B"): "B .. MarketOnClose",
        ("FIX.5.0", "695", "C"): "C .. AtTheClose",
        ("FIX.5.0", "695", "D"): "D .. VWAP",
        ("FIX.5.0", "695", "E"): "E .. Axe",
        ("FIX.5.0", "695", "F"): "F .. AxeOnBid",
        ("FIX.5.0", "695", "G"): "G .. AxeOnOffer",
        ("FIX.5.0", "695", "H"): "H .. ClientNaturalWorking",
        ("FIX.5.0", "695", "I"): "I .. InTouchWith",
        ("FIX.5.0", "695", "J"): "J .. PositionWanted",
        ("FIX.5.0", "695", "K"): "K .. MarketMaking",
        ("FIX.5.0", "695", "L"): "L .. Limit",
        ("FIX.5.0", "695", "M"): "M .. MoreBehind",
        ("FIX.5.0", "695", "N"): "N .. ClientNaturalBlock",
        ("FIX.5.0", "695", "O"): "O .. AtTheOpen",
        ("FIX.5.0", "695", "P"): "P .. TakingAPosition",
        ("FIX.5.0", "695", "Q"): "Q .. AtTheMarket",
        ("FIX.5.0", "695", "R"): "R .. ReadyToTrade",
        ("FIX.5.0", "695", "S"): "S .. PortfolioShown",
        ("FIX.5.0", "695", "T"): "T .. ThroughTheDay",
        ("FIX.5.0", "695", "U"): "U .. Unwind",
        ("FIX.5.0", "695", "V"): "V .. Versus",
        ("FIX.5.0", "695", "W"): "W .. Indication",
        ("FIX.5.0", "695", "X"): "X .. CrossingOpportunity",
        ("FIX.5.0", "695", "Y"): "Y .. AtTheMidpoint",
        ("FIX.5.0", "695", "Z"): "Z .. PreOpen",
        ("FIX.5.0", "695", "1"): "1 .. QuantityNegotiable",
        ("FIX.5.0", "695", "2"): "2 .. AllowLateBids",
        ("FIX.5.0", "695", "3"): "3 .. ImmediateOrCounter",
        ("FIX.5.0", "695", "4"): "4 .. AutoTrade",
        ("FIX.5.0", "695", "a"): "a .. AutomaticSpot",
        ("FIX.5.0", "695", "b"): "b .. PlatformCalculatedSpot",
        ("FIX.5.0", "695", "c"): "c .. OutsideSpread",
        ("FIX.5.0", "695", "d"): "d .. DeferredSpot",
        ("FIX.5.0", "695", "n"): "n .. NegotiatedSpot",
        ("FIX.5.0", "698", "1"): "1 .. Percentage",
        ("FIX.5.0", "698", "2"): "2 .. PerUnit",
        ("FIX.5.0", "698", "3"): "3 .. FixedAmount",
        ("FIX.5.0", "698", "4"): "4 .. Discount",
        ("FIX.5.0", "698", "5"): "5 .. Premium",
        ("FIX.5.0", "698", "6"): "6 .. Spread",
        ("FIX.5.0", "698", "7"): "7 .. TEDPrice",
        ("FIX.5.0", "698", "8"): "8 .. TEDYield",
        ("FIX.5.0", "698", "9"): "9 .. Yield",
        ("FIX.5.0", "698", "10"): "10 .. FixedCabinetTradePrice",
        ("FIX.5.0", "698", "11"): "11 .. VariableCabinetTradePrice",
        ("FIX.5.0", "698", "12"): "12 .. PriceSpread",
        ("FIX.5.0", "698", "13"): "13 .. ProductTicksInHalves",
        ("FIX.5.0", "698", "14"): "14 .. ProductTicksInFourths",
        ("FIX.5.0", "698", "15"): "15 .. ProductTicksInEighths",
        ("FIX.5.0", "698", "16"): "16 .. ProductTicksInSixteenths",
        ("FIX.5.0", "698", "17"): "17 .. ProductTicksInThirtySeconds",
        ("FIX.5.0", "698", "18"): "18 .. ProductTicksInSixtyFourths",
        ("FIX.5.0", "698", "19"): "19 .. ProductTicksInOneTwentyEighths",
        ("FIX.5.0", "698", "20"): "20 .. NormalRateRepresentation",
        ("FIX.5.0", "698", "21"): "21 .. InverseRateRepresentation",
        ("FIX.5.0", "698", "22"): "22 .. BasisPoints",
        ("FIX.5.0", "698", "23"): "23 .. UpfrontPoints",
        ("FIX.5.0", "698", "24"): "24 .. InterestRate",
        ("FIX.5.0", "698", "25"): "25 .. PercentageNotional",
        ("FIX.5.0", "703", "ALC"): "ALC .. AllocationTradeQty",
        ("FIX.5.0", "703", "AS"): "AS .. OptionAssignment",
        ("FIX.5.0", "703", "ASF"): "ASF .. AsOfTradeQty",
        ("FIX.5.0", "703", "DLV"): "DLV .. DeliveryQty",
        ("FIX.5.0", "703", "ETR"): "ETR .. ElectronicTradeQty",
        ("FIX.5.0", "703", "EX"): "EX .. OptionExerciseQty",
        ("FIX.5.0", "703", "FIN"): "FIN .. EndOfDayQty",
        ("FIX.5.0", "703", "IAS"): "IAS .. IntraSpreadQty",
        ("FIX.5.0", "703", "IES"): "IES .. InterSpreadQty",
        ("FIX.5.0", "703", "PA"): "PA .. AdjustmentQty",
        ("FIX.5.0", "703", "PIT"): "PIT .. PitTradeQty",
        ("FIX.5.0", "703", "SOD"): "SOD .. StartOfDayQty",
        ("FIX.5.0", "703", "SPL"): "SPL .. IntegralSplit",
        ("FIX.5.0", "703", "TA"): "TA .. TransactionFromAssignment",
        ("FIX.5.0", "703", "TOT"): "TOT .. TotalTransactionQty",
        ("FIX.5.0", "703", "TQ"): "TQ .. TransactionQuantity",
        ("FIX.5.0", "703", "TRF"): "TRF .. TransferTradeQty",
        ("FIX.5.0", "703", "TX"): "TX .. TransactionFromExercise",
        ("FIX.5.0", "703", "XM"): "XM .. CrossMarginQty",
        ("FIX.5.0", "703", "RCV"): "RCV .. ReceiveQuantity",
        ("FIX.5.0", "703", "CAA"): "CAA .. CorporateActionAdjustment",
        ("FIX.5.0", "703", "DN"): "DN .. DeliveryNoticeQty",
        ("FIX.5.0", "703", "EP"): "EP .. ExchangeForPhysicalQty",
        ("FIX.5.0", "703", "PNTN"): "PNTN .. PrivatelyNegotiatedTradeQty",
        ("FIX.5.0", "703", "DLT"): "DLT .. NetDeltaQty",
        ("FIX.5.0", "703", "CEA"): "CEA .. CreditEventAdjustment",
        ("FIX.5.0", "703", "SEA"): "SEA .. SuccessionEventAdjustment",
        ("FIX.5.0", "703", "NET"): "NET .. NetQty",
        ("FIX.5.0", "703", "GRS"): "GRS .. GrossQty",
        ("FIX.5.0", "703", "ITD"): "ITD .. IntradayQty",
        ("FIX.5.0", "703", "NDAS"): "NDAS .. GrossLongNonDeltaAdjustedSwaptionPosition",
        ("FIX.5.0", "703", "DAS"): "DAS .. LongDeltaAdjustedPairedSwaptionPosition",
        ("FIX.5.0", "703", "EXP"): "EXP .. ExpiringQuantity",
        ("FIX.5.0", "703", "UNEX"): "UNEX .. QuantityNotExercised",
        ("FIX.5.0", "703", "REQ"): "REQ .. RequestedExerciseQuantity",
        ("FIX.5.0", "703", "CFE"): "CFE .. CashFuturesEquivalentQuantity",
        ("FIX.5.0", "706", "0"): "0 .. Submitted",
        ("FIX.5.0", "706", "1"): "1 .. Accepted",
        ("FIX.5.0", "706", "2"): "2 .. Rejected",
        ("FIX.5.0", "707", "CASH"): "CASH .. CashAmount",
        ("FIX.5.0", "707", "CRES"): "CRES .. CashResidualAmount",
        ("FIX.5.0", "707", "FMTM"): "FMTM .. FinalMarkToMarketAmount",
        ("FIX.5.0", "707", "IMTM"): "IMTM .. IncrementalMarkToMarketAmount",
        ("FIX.5.0", "707", "PREM"): "PREM .. PremiumAmount",
        ("FIX.5.0", "707", "SMTM"): "SMTM .. StartOfDayMarkToMarketAmount",
        ("FIX.5.0", "707", "TVAR"): "TVAR .. TradeVariationAmount",
        ("FIX.5.0", "707", "VADJ"): "VADJ .. ValueAdjustedAmount",
        ("FIX.5.0", "707", "SETL"): "SETL .. SettlementValue",
        ("FIX.5.0", "707", "ICPN"): "ICPN .. InitialTradeCouponAmount",
        ("FIX.5.0", "707", "ACPN"): "ACPN .. AccruedCouponAmount",
        ("FIX.5.0", "707", "CPN"): "CPN .. CouponAmount",
        ("FIX.5.0", "707", "IACPN"): "IACPN .. IncrementalAccruedCoupon",
        ("FIX.5.0", "707", "CMTM"): "CMTM .. CollateralizedMarkToMarket",
        ("FIX.5.0", "707", "ICMTM"): "ICMTM .. IncrementalCollateralizedMarkToMarket",
        ("FIX.5.0", "707", "DLV"): "DLV .. CompensationAmount",
        ("FIX.5.0", "707", "BANK"): "BANK .. TotalBankedAmount",
        ("FIX.5.0", "707", "COLAT"): "COLAT .. TotalCollateralizedAmount",
        ("FIX.5.0", "707", "LSNV"): "LSNV .. LongPairedSwapNotionalValue",
        ("FIX.5.0", "707", "SSNV"): "SSNV .. ShortPairedSwapNotionalValue",
        ("FIX.5.0", "707", "SACPN"): "SACPN .. StartOfDayAccruedCoupon",
        ("FIX.5.0", "707", "NPV"): "NPV .. NetPresentValue",
        ("FIX.5.0", "707", "SNPV"): "SNPV .. StartOfDayNetPresentValue",
        ("FIX.5.0", "707", "NCF"): "NCF .. NetCashFlow",
        ("FIX.5.0", "707", "PVFEES"): "PVFEES .. PresentValueOfFees",
        ("FIX.5.0", "707", "PV01"): "PV01 .. PresentValueOneBasisPoints",
        ("FIX.5.0", "707", "5YREN"): "5YREN .. FiveYearEquivalentNotional",
        ("FIX.5.0", "707", "UMTM"): "UMTM .. UndiscountedMarkToMarket",
        ("FIX.5.0", "707", "MTD"): "MTD .. MarkToModel",
        ("FIX.5.0", "707", "VMTM"): "VMTM .. MarkToMarketVariance",
        ("FIX.5.0", "707", "VMTD"): "VMTD .. MarkToModelVariance",
        ("FIX.5.0", "707", "UPFRNT"): "UPFRNT .. UpfrontPayment",
        ("FIX.5.0", "709", "1"): "1 .. Exercise",
        ("FIX.5.0", "709", "2"): "2 .. DoNotExercise",
        ("FIX.5.0", "709", "3"): "3 .. PositionAdjustment",
        ("FIX.5.0", "709", "4"): "4 .. PositionChangeSubmission",
        ("FIX.5.0", "709", "5"): "5 .. Pledge",
        ("FIX.5.0", "709", "6"): "6 .. LargeTraderSubmission",
        ("FIX.5.0", "709", "7"): "7 .. LargePositionsReportingSubmission",
        ("FIX.5.0", "709", "8"): "8 .. LongHoldings",
        ("FIX.5.0", "709", "9"): "9 .. InternalTransfer",
        ("FIX.5.0", "709", "10"): "10 .. TransferOfFirm",
        ("FIX.5.0", "709", "11"): "11 .. ExternalTransfer",
        ("FIX.5.0", "709", "12"): "12 .. CorporateAction",
        ("FIX.5.0", "709", "13"): "13 .. Notification",
        ("FIX.5.0", "709", "14"): "14 .. PositionCreation",
        ("FIX.5.0", "709", "15"): "15 .. Closeout",
        ("FIX.5.0", "709", "16"): "16 .. Reopen",
        ("FIX.5.0", "712", "1"): "1 .. New",
        ("FIX.5.0", "712", "2"): "2 .. Replace",
        ("FIX.5.0", "712", "3"): "3 .. Cancel",
        ("FIX.5.0", "712", "4"): "4 .. Reverse",
        ("FIX.5.0", "716", "ITD"): "ITD .. Intraday",
        ("FIX.5.0", "716", "RTH"): "RTH .. RegularTradingHours",
        ("FIX.5.0", "716", "ETH"): "ETH .. ElectronicTradingHours",
        ("FIX.5.0", "716", "EOD"): "EOD .. EndOfDay",
        ("FIX.5.0", "718", "0"): "0 .. ProcessRequestAsMarginDisposition",
        ("FIX.5.0", "718", "1"): "1 .. DeltaPlus",
        ("FIX.5.0", "718", "2"): "2 .. DeltaMinus",
        ("FIX.5.0", "718", "3"): "3 .. Final",
        ("FIX.5.0", "718", "4"): "4 .. CustomerSpecificPosition",
        ("FIX.5.0", "722", "0"): "0 .. Accepted",
        ("FIX.5.0", "722", "1"): "1 .. AcceptedWithWarnings",
        ("FIX.5.0", "722", "2"): "2 .. Rejected",
        ("FIX.5.0", "722", "3"): "3 .. Completed",
        ("FIX.5.0", "722", "4"): "4 .. CompletedWithWarnings",
        ("FIX.5.0", "723", "0"): "0 .. SuccessfulCompletion",
        ("FIX.5.0", "723", "1"): "1 .. Rejected",
        ("FIX.5.0", "723", "99"): "99 .. Other",
        ("FIX.5.0", "724", "0"): "0 .. Positions",
        ("FIX.5.0", "724", "1"): "1 .. Trades",
        ("FIX.5.0", "724", "2"): "2 .. Exercises",
        ("FIX.5.0", "724", "3"): "3 .. Assignments",
        ("FIX.5.0", "724", "4"): "4 .. SettlementActivity",
        ("FIX.5.0", "724", "5"): "5 .. BackoutMessage",
        ("FIX.5.0", "724", "6"): "6 .. DeltaPositions",
        ("FIX.5.0", "724", "7"): "7 .. NetPosition",
        ("FIX.5.0", "724", "8"): "8 .. LargePositionsReporting",
        ("FIX.5.0", "724", "9"): "9 .. ExercisePositionReportingSubmission",
        ("FIX.5.0", "724", "10"): "10 .. PositionLimitReportingSubmissing",
        ("FIX.5.0", "725", "0"): "0 .. Inband",
        ("FIX.5.0", "725", "1"): "1 .. OutOfBand",
        ("FIX.5.0", "728", "0"): "0 .. ValidRequest",
        ("FIX.5.0", "728", "1"): "1 .. InvalidOrUnsupportedRequest",
        ("FIX.5.0", "728", "2"): "2 .. NoPositionsFoundThatMatchCriteria",
        ("FIX.5.0", "728", "3"): "3 .. NotAuthorizedToRequestPositions",
        ("FIX.5.0", "728", "4"): "4 .. RequestForPositionNotSupported",
        ("FIX.5.0", "728", "99"): "99 .. Other",
        ("FIX.5.0", "729", "0"): "0 .. Completed",
        ("FIX.5.0", "729", "1"): "1 .. CompletedWithWarnings",
        ("FIX.5.0", "729", "2"): "2 .. Rejected",
        ("FIX.5.0", "731", "1"): "1 .. Final",
        ("FIX.5.0", "731", "2"): "2 .. Theoretical",
        ("FIX.5.0", "733", "1"): "1 .. Final",
        ("FIX.5.0", "733", "2"): "2 .. Theoretical",
        ("FIX.5.0", "744", "P"): "P .. ProRata",
        ("FIX.5.0", "744", "R"): "R .. Random",
        ("FIX.5.0", "747", "A"): "A .. Automatic",
        ("FIX.5.0", "747", "M"): "M .. Manual",
        ("FIX.5.0", "749", "0"): "0 .. Successful",
        ("FIX.5.0", "749", "1"): "1 .. InvalidOrUnknownInstrument",
        ("FIX.5.0", "749", "2"): "2 .. InvalidTypeOfTradeRequested",
        ("FIX.5.0", "749", "3"): "3 .. InvalidParties",
        ("FIX.5.0", "749", "4"): "4 .. InvalidTransportTypeRequested",
        ("FIX.5.0", "749", "5"): "5 .. InvalidDestinationRequested",
        ("FIX.5.0", "749", "8"): "8 .. TradeRequestTypeNotSupported",
        ("FIX.5.0", "749", "9"): "9 .. NotAuthorized",
        ("FIX.5.0", "749", "99"): "99 .. Other",
        ("FIX.5.0", "750", "0"): "0 .. Accepted",
        ("FIX.5.0", "750", "1"): "1 .. Completed",
        ("FIX.5.0", "750", "2"): "2 .. Rejected",
        ("FIX.5.0", "751", "0"): "0 .. Successful",
        ("FIX.5.0", "751", "1"): "1 .. InvalidPartyInformation",
        ("FIX.5.0", "751", "2"): "2 .. UnknownInstrument",
        ("FIX.5.0", "751", "3"): "3 .. UnauthorizedToReportTrades",
        ("FIX.5.0", "751", "4"): "4 .. InvalidTradeType",
        ("FIX.5.0", "751", "5"): "5 .. PriceExceedsCurrentPriceBand",
        ("FIX.5.0", "751", "6"): "6 .. ReferencePriceNotAvailable",
        ("FIX.5.0", "751", "7"): "7 .. NotionalValueExceedsThreshold",
        ("FIX.5.0", "751", "99"): "99 .. Other",
        ("FIX.5.0", "752", "1"): "1 .. SingleSecurity",
        ("FIX.5.0", "752", "2"): "2 .. IndividualLegOfAMultilegSecurity",
        ("FIX.5.0", "752", "3"): "3 .. MultilegSecurity",
        ("FIX.5.0", "758", "6"): "6 .. UKNationalInsuranceOrPensionNumber",
        ("FIX.5.0", "758", "7"): "7 .. USSocialSecurityNumber",
        ("FIX.5.0", "758", "8"): "8 .. USEmployerOrTaxIDNumber",
        ("FIX.5.0", "758", "9"): "9 .. AustralianBusinessNumber",
        ("FIX.5.0", "758", "A"): "A .. AustralianTaxFileNumber",
        ("FIX.5.0", "758", "J"): "J .. TaxID",
        ("FIX.5.0", "758", "1"): "1 .. KoreanInvestorID",
        ("FIX.5.0", "758", "2"): "2 .. TaiwaneseForeignInvestorID",
        ("FIX.5.0", "758", "3"): "3 .. TaiwaneseTradingAcct",
        ("FIX.5.0", "758", "4"): "4 .. MalaysianCentralDepository",
        ("FIX.5.0", "758", "5"): "5 .. ChineseInvestorID",
        ("FIX.5.0", "758", "I"): "I .. ISITCAcronym",
        ("FIX.5.0", "758", "B"): "B .. BIC",
        ("FIX.5.0", "758", "C"): "C .. GeneralIdentifier",
        ("FIX.5.0", "758", "D"): "D .. Proprietary",
        ("FIX.5.0", "758", "E"): "E .. ISOCountryCode",
        ("FIX.5.0", "758", "F"): "F .. SettlementEntityLocation",
        ("FIX.5.0", "758", "G"): "G .. MIC",
        ("FIX.5.0", "758", "H"): "H .. CSDParticipant",
        ("FIX.5.0", "758", "K"): "K .. AustralianCompanyNumber",
        ("FIX.5.0", "758", "L"): "L .. AustralianRegisteredBodyNumber",
        ("FIX.5.0", "758", "M"): "M .. CFTCReportingFirmIdentifier",
        ("FIX.5.0", "758", "N"): "N .. LegalEntityIdentifier",
        ("FIX.5.0", "758", "O"): "O .. InterimIdentifier",
        ("FIX.5.0", "758", "P"): "P .. ShortCodeIdentifier",
        ("FIX.5.0", "758", "Q"): "Q .. NationalIDNaturalPerson",
        ("FIX.5.0", "759", "1"): "1 .. ExecutingFirm",
        ("FIX.5.0", "759", "2"): "2 .. BrokerOfCredit",
        ("FIX.5.0", "759", "3"): "3 .. ClientID",
        ("FIX.5.0", "759", "4"): "4 .. ClearingFirm",
        ("FIX.5.0", "759", "5"): "5 .. InvestorID",
        ("FIX.5.0", "759", "6"): "6 .. IntroducingFirm",
        ("FIX.5.0", "759", "7"): "7 .. EnteringFirm",
        ("FIX.5.0", "759", "8"): "8 .. Locate",
        ("FIX.5.0", "759", "9"): "9 .. FundManagerClientID",
        ("FIX.5.0", "759", "10"): "10 .. SettlementLocation",
        ("FIX.5.0", "759", "11"): "11 .. OrderOriginationTrader",
        ("FIX.5.0", "759", "12"): "12 .. ExecutingTrader",
        ("FIX.5.0", "759", "13"): "13 .. OrderOriginationFirm",
        ("FIX.5.0", "759", "15"): "15 .. CorrespondantClearingFirm",
        ("FIX.5.0", "759", "16"): "16 .. ExecutingSystem",
        ("FIX.5.0", "759", "17"): "17 .. ContraFirm",
        ("FIX.5.0", "759", "18"): "18 .. ContraClearingFirm",
        ("FIX.5.0", "759", "19"): "19 .. SponsoringFirm",
        ("FIX.5.0", "759", "20"): "20 .. UnderlyingContraFirm",
        ("FIX.5.0", "759", "21"): "21 .. ClearingOrganization",
        ("FIX.5.0", "759", "22"): "22 .. Exchange",
        ("FIX.5.0", "759", "24"): "24 .. CustomerAccount",
        ("FIX.5.0", "759", "25"): "25 .. CorrespondentClearingOrganization",
        ("FIX.5.0", "759", "26"): "26 .. CorrespondentBroker",
        ("FIX.5.0", "759", "27"): "27 .. Buyer",
        ("FIX.5.0", "759", "28"): "28 .. Custodian",
        ("FIX.5.0", "759", "29"): "29 .. Intermediary",
        ("FIX.5.0", "759", "30"): "30 .. Agent",
        ("FIX.5.0", "759", "31"): "31 .. SubCustodian",
        ("FIX.5.0", "759", "32"): "32 .. Beneficiary",
        ("FIX.5.0", "759", "33"): "33 .. InterestedParty",
        ("FIX.5.0", "759", "34"): "34 .. RegulatoryBody",
        ("FIX.5.0", "759", "35"): "35 .. LiquidityProvider",
        ("FIX.5.0", "759", "36"): "36 .. EnteringTrader",
        ("FIX.5.0", "759", "37"): "37 .. ContraTrader",
        ("FIX.5.0", "759", "38"): "38 .. PositionAccount",
        ("FIX.5.0", "759", "39"): "39 .. ContraInvestorID",
        ("FIX.5.0", "759", "40"): "40 .. TransferToFirm",
        ("FIX.5.0", "759", "41"): "41 .. ContraPositionAccount",
        ("FIX.5.0", "759", "42"): "42 .. ContraExchange",
        ("FIX.5.0", "759", "43"): "43 .. InternalCarryAccount",
        ("FIX.5.0", "759", "44"): "44 .. OrderEntryOperatorID",
        ("FIX.5.0", "759", "45"): "45 .. SecondaryAccountNumber",
        ("FIX.5.0", "759", "46"): "46 .. ForeignFirm",
        ("FIX.5.0", "759", "47"): "47 .. ThirdPartyAllocationFirm",
        ("FIX.5.0", "759", "48"): "48 .. ClaimingAccount",
        ("FIX.5.0", "759", "49"): "49 .. AssetManager",
        ("FIX.5.0", "759", "50"): "50 .. PledgorAccount",
        ("FIX.5.0", "759", "51"): "51 .. PledgeeAccount",
        ("FIX.5.0", "759", "52"): "52 .. LargeTraderReportableAccount",
        ("FIX.5.0", "759", "53"): "53 .. TraderMnemonic",
        ("FIX.5.0", "759", "54"): "54 .. SenderLocation",
        ("FIX.5.0", "759", "55"): "55 .. SessionID",
        ("FIX.5.0", "759", "56"): "56 .. AcceptableCounterparty",
        ("FIX.5.0", "759", "57"): "57 .. UnacceptableCounterparty",
        ("FIX.5.0", "759", "58"): "58 .. EnteringUnit",
        ("FIX.5.0", "759", "59"): "59 .. ExecutingUnit",
        ("FIX.5.0", "759", "60"): "60 .. IntroducingBroker",
        ("FIX.5.0", "759", "61"): "61 .. QuoteOriginator",
        ("FIX.5.0", "759", "62"): "62 .. ReportOriginator",
        ("FIX.5.0", "759", "63"): "63 .. SystematicInternaliser",
        ("FIX.5.0", "759", "64"): "64 .. MultilateralTradingFacility",
        ("FIX.5.0", "759", "65"): "65 .. RegulatedMarket",
        ("FIX.5.0", "759", "66"): "66 .. MarketMaker",
        ("FIX.5.0", "759", "67"): "67 .. InvestmentFirm",
        ("FIX.5.0", "759", "68"): "68 .. HostCompetentAuthority",
        ("FIX.5.0", "759", "69"): "69 .. HomeCompetentAuthority",
        ("FIX.5.0", "759", "70"): "70 .. CompetentAuthorityLiquidity",
        ("FIX.5.0", "759", "71"): "71 .. CompetentAuthorityTransactionVenue",
        ("FIX.5.0", "759", "72"): "72 .. ReportingIntermediary",
        ("FIX.5.0", "759", "73"): "73 .. ExecutionVenue",
        ("FIX.5.0", "759", "74"): "74 .. MarketDataEntryOriginator",
        ("FIX.5.0", "759", "75"): "75 .. LocationID",
        ("FIX.5.0", "759", "76"): "76 .. DeskID",
        ("FIX.5.0", "759", "77"): "77 .. MarketDataMarket",
        ("FIX.5.0", "759", "78"): "78 .. AllocationEntity",
        ("FIX.5.0", "759", "79"): "79 .. PrimeBroker",
        ("FIX.5.0", "759", "80"): "80 .. StepOutFirm",
        ("FIX.5.0", "759", "81"): "81 .. BrokerClearingID",
        ("FIX.5.0", "759", "82"): "82 .. CentralRegistrationDepository",
        ("FIX.5.0", "759", "83"): "83 .. ClearingAccount",
        ("FIX.5.0", "759", "84"): "84 .. AcceptableSettlingCounterparty",
        ("FIX.5.0", "759", "85"): "85 .. UnacceptableSettlingCounterparty",
        ("FIX.5.0", "759", "86"): "86 .. CLSMemberBank",
        ("FIX.5.0", "759", "87"): "87 .. InConcertGroup",
        ("FIX.5.0", "759", "88"): "88 .. InConcertControllingEntity",
        ("FIX.5.0", "759", "89"): "89 .. LargePositionsReportingAccount",
        ("FIX.5.0", "759", "90"): "90 .. SettlementFirm",
        ("FIX.5.0", "759", "91"): "91 .. SettlementAccount",
        ("FIX.5.0", "759", "92"): "92 .. ReportingMarketCenter",
        ("FIX.5.0", "759", "93"): "93 .. RelatedReportingMarketCenter",
        ("FIX.5.0", "759", "94"): "94 .. AwayMarket",
        ("FIX.5.0", "759", "95"): "95 .. GiveupTradingFirm",
        ("FIX.5.0", "759", "96"): "96 .. TakeupTradingFirm",
        ("FIX.5.0", "759", "97"): "97 .. GiveupClearingFirm",
        ("FIX.5.0", "759", "98"): "98 .. TakeupClearingFirm",
        ("FIX.5.0", "759", "99"): "99 .. OriginatingMarket",
        ("FIX.5.0", "759", "100"): "100 .. MarginAccount",
        ("FIX.5.0", "759", "101"): "101 .. CollateralAssetAccount",
        ("FIX.5.0", "759", "102"): "102 .. DataRepository",
        ("FIX.5.0", "759", "103"): "103 .. CalculationAgent",
        ("FIX.5.0", "759", "104"): "104 .. ExerciseNoticeSender",
        ("FIX.5.0", "759", "105"): "105 .. ExerciseNoticeReceiver",
        ("FIX.5.0", "759", "106"): "106 .. RateReferenceBank",
        ("FIX.5.0", "759", "107"): "107 .. Correspondent",
        ("FIX.5.0", "759", "109"): "109 .. BeneficiaryBank",
        ("FIX.5.0", "759", "110"): "110 .. Borrower",
        ("FIX.5.0", "759", "111"): "111 .. PrimaryObligator",
        ("FIX.5.0", "759", "112"): "112 .. Guarantor",
        ("FIX.5.0", "759", "113"): "113 .. ExcludedReferenceEntity",
        ("FIX.5.0", "759", "114"): "114 .. DeterminingParty",
        ("FIX.5.0", "759", "115"): "115 .. HedgingParty",
        ("FIX.5.0", "759", "116"): "116 .. ReportingEntity",
        ("FIX.5.0", "759", "117"): "117 .. SalesPerson",
        ("FIX.5.0", "759", "118"): "118 .. Operator",
        ("FIX.5.0", "759", "119"): "119 .. CSD",
        ("FIX.5.0", "759", "120"): "120 .. ICSD",
        ("FIX.5.0", "759", "121"): "121 .. TradingSubAccount",
        ("FIX.5.0", "759", "122"): "122 .. InvestmentDecisionMaker",
        ("FIX.5.0", "759", "123"): "123 .. PublishingIntermediary",
        ("FIX.5.0", "761", "1"): "1 .. CUSIP",
        ("FIX.5.0", "761", "2"): "2 .. SEDOL",
        ("FIX.5.0", "761", "3"): "3 .. QUIK",
        ("FIX.5.0", "761", "4"): "4 .. ISINNumber",
        ("FIX.5.0", "761", "5"): "5 .. RICCode",
        ("FIX.5.0", "761", "6"): "6 .. ISOCurrencyCode",
        ("FIX.5.0", "761", "7"): "7 .. ISOCountryCode",
        ("FIX.5.0", "761", "8"): "8 .. ExchangeSymbol",
        ("FIX.5.0", "761", "9"): "9 .. ConsolidatedTapeAssociation",
        ("FIX.5.0", "761", "A"): "A .. BloombergSymbol",
        ("FIX.5.0", "761", "B"): "B .. Wertpapier",
        ("FIX.5.0", "761", "C"): "C .. Dutch",
        ("FIX.5.0", "761", "D"): "D .. Valoren",
        ("FIX.5.0", "761", "E"): "E .. Sicovam",
        ("FIX.5.0", "761", "F"): "F .. Belgian",
        ("FIX.5.0", "761", "G"): "G .. Common",
        ("FIX.5.0", "761", "H"): "H .. ClearingHouse",
        ("FIX.5.0", "761", "I"): "I .. ISDAFpMLSpecification",
        ("FIX.5.0", "761", "J"): "J .. OptionPriceReportingAuthority",
        ("FIX.5.0", "761", "K"): "K .. ISDAFpMLURL",
        ("FIX.5.0", "761", "L"): "L .. LetterOfCredit",
        ("FIX.5.0", "761", "M"): "M .. MarketplaceAssignedIdentifier",
        ("FIX.5.0", "761", "N"): "N .. MarkitREDEntityCLIP",
        ("FIX.5.0", "761", "P"): "P .. MarkitREDPairCLIP",
        ("FIX.5.0", "761", "Q"): "Q .. CFTCCommodityCode",
        ("FIX.5.0", "761", "R"): "R .. ISDACommodityReferencePrice",
        ("FIX.5.0", "761", "S"): "S .. FinancialInstrumentGlobalIdentifier",
        ("FIX.5.0", "761", "T"): "T .. LegalEntityIdentifier",
        ("FIX.5.0", "761", "U"): "U .. Synthetic",
        ("FIX.5.0", "761", "V"): "V .. FidessaInstrumentMnemonic",
        ("FIX.5.0", "761", "W"): "W .. IndexName",
        ("FIX.5.0", "770", "1"): "1 .. ExecutionTime",
        ("FIX.5.0", "770", "2"): "2 .. TimeIn",
        ("FIX.5.0", "770", "3"): "3 .. TimeOut",
        ("FIX.5.0", "770", "4"): "4 .. BrokerReceipt",
        ("FIX.5.0", "770", "5"): "5 .. BrokerExecution",
        ("FIX.5.0", "770", "6"): "6 .. DeskReceipt",
        ("FIX.5.0", "770", "7"): "7 .. SubmissionToClearing",
        ("FIX.5.0", "770", "8"): "8 .. TimePriority",
        ("FIX.5.0", "770", "9"): "9 .. OrderbookEntryTime",
        ("FIX.5.0", "770", "10"): "10 .. OrderSubmissionTime",
        ("FIX.5.0", "770", "11"): "11 .. PubliclyReported",
        ("FIX.5.0", "770", "12"): "12 .. PublicReportUpdated",
        ("FIX.5.0", "770", "13"): "13 .. NonPubliclyReported",
        ("FIX.5.0", "770", "14"): "14 .. NonPublicReportUpdated",
        ("FIX.5.0", "770", "15"): "15 .. SubmittedForConfirmation",
        ("FIX.5.0", "770", "16"): "16 .. UpdatedForConfirmation",
        ("FIX.5.0", "770", "17"): "17 .. Confirmed",
        ("FIX.5.0", "770", "18"): "18 .. UpdatedForClearing",
        ("FIX.5.0", "770", "19"): "19 .. Cleared",
        ("FIX.5.0", "770", "20"): "20 .. AllocationsSubmitted",
        ("FIX.5.0", "770", "21"): "21 .. AllocationsUpdated",
        ("FIX.5.0", "770", "22"): "22 .. AllocationsCompleted",
        ("FIX.5.0", "770", "23"): "23 .. SubmittedToRepository",
        ("FIX.5.0", "770", "24"): "24 .. PostTrdContntnEvnt",
        ("FIX.5.0", "770", "25"): "25 .. PostTradeValuation",
        ("FIX.5.0", "770", "26"): "26 .. PreviousTimePriority",
        ("FIX.5.0", "773", "1"): "1 .. Status",
        ("FIX.5.0", "773", "2"): "2 .. Confirmation",
        ("FIX.5.0", "773", "3"): "3 .. ConfirmationRequestRejected",
        ("FIX.5.0", "774", "1"): "1 .. MismatchedAccount",
        ("FIX.5.0", "774", "2"): "2 .. MissingSettlementInstructions",
        ("FIX.5.0", "774", "3"): "3 .. UnknownOrMissingIndividualAllocId",
        ("FIX.5.0", "774", "4"): "4 .. TransactionNotRecognized",
        ("FIX.5.0", "774", "5"): "5 .. DuplicateTransaction",
        ("FIX.5.0", "774", "6"): "6 .. IncorrectOrMissingInstrument",
        ("FIX.5.0", "774", "7"): "7 .. IncorrectOrMissingPrice",
        ("FIX.5.0", "774", "8"): "8 .. IncorrectOrMissingCommission",
        ("FIX.5.0", "774", "9"): "9 .. IncorrectOrMissingSettlDate",
        ("FIX.5.0", "774", "10"): "10 .. IncorrectOrMissingFundIDOrFundName",
        ("FIX.5.0", "774", "11"): "11 .. IncorrectOrMissingQuantity",
        ("FIX.5.0", "774", "12"): "12 .. IncorrectOrMissingFees",
        ("FIX.5.0", "774", "13"): "13 .. IncorrectOrMissingTax",
        ("FIX.5.0", "774", "14"): "14 .. IncorrectOrMissingParty",
        ("FIX.5.0", "774", "15"): "15 .. IncorrectOrMissingSide",
        ("FIX.5.0", "774", "16"): "16 .. IncorrectOrMissingNetMoney",
        ("FIX.5.0", "774", "17"): "17 .. IncorrectOrMissingTradeDate",
        ("FIX.5.0", "774", "18"): "18 .. IncorrectOrMissingSettlCcyInstructions",
        ("FIX.5.0", "774", "19"): "19 .. IncorrectOrMissingCapacity",
        ("FIX.5.0", "774", "99"): "99 .. Other",
        ("FIX.5.0", "775", "0"): "0 .. RegularBooking",
        ("FIX.5.0", "775", "1"): "1 .. CFD",
        ("FIX.5.0", "775", "2"): "2 .. TotalReturnSwap",
        ("FIX.5.0", "776", "0"): "0 .. UnknownAccount",
        ("FIX.5.0", "776", "1"): "1 .. IncorrectQuantity",
        ("FIX.5.0", "776", "2"): "2 .. IncorrectAveragegPrice",
        ("FIX.5.0", "776", "3"): "3 .. UnknownExecutingBrokerMnemonic",
        ("FIX.5.0", "776", "4"): "4 .. CommissionDifference",
        ("FIX.5.0", "776", "5"): "5 .. UnknownOrderID",
        ("FIX.5.0", "776", "6"): "6 .. UnknownListID",
        ("FIX.5.0", "776", "7"): "7 .. OtherSeeText",
        ("FIX.5.0", "776", "8"): "8 .. IncorrectAllocatedQuantity",
        ("FIX.5.0", "776", "9"): "9 .. CalculationDifference",
        ("FIX.5.0", "776", "10"): "10 .. UnknownOrStaleExecID",
        ("FIX.5.0", "776", "11"): "11 .. MismatchedData",
        ("FIX.5.0", "776", "12"): "12 .. UnknownClOrdID",
        ("FIX.5.0", "776", "13"): "13 .. WarehouseRequestRejected",
        ("FIX.5.0", "776", "14"): "14 .. DuplicateOrMissingIndividualAllocId",
        ("FIX.5.0", "776", "15"): "15 .. TradeNotRecognized",
        ("FIX.5.0", "776", "16"): "16 .. DuplicateTrade",
        ("FIX.5.0", "776", "17"): "17 .. IncorrectOrMissingInstrument",
        ("FIX.5.0", "776", "18"): "18 .. IncorrectOrMissingSettlDate",
        ("FIX.5.0", "776", "19"): "19 .. IncorrectOrMissingFundIDOrFundName",
        ("FIX.5.0", "776", "20"): "20 .. IncorrectOrMissingSettlInstructions",
        ("FIX.5.0", "776", "21"): "21 .. IncorrectOrMissingFees",
        ("FIX.5.0", "776", "22"): "22 .. IncorrectOrMissingTax",
        ("FIX.5.0", "776", "23"): "23 .. UnknownOrMissingParty",
        ("FIX.5.0", "776", "24"): "24 .. IncorrectOrMissingSide",
        ("FIX.5.0", "776", "25"): "25 .. IncorrectOrMissingNetMoney",
        ("FIX.5.0", "776", "26"): "26 .. IncorrectOrMissingTradeDate",
        ("FIX.5.0", "776", "27"): "27 .. IncorrectOrMissingSettlCcyInstructions",
        ("FIX.5.0", "776", "28"): "28 .. IncorrectOrMissingProcessCode",
        ("FIX.5.0", "776", "99"): "99 .. Other",
        ("FIX.5.0", "780", "0"): "0 .. UseDefaultInstructions",
        ("FIX.5.0", "780", "1"): "1 .. DeriveFromParametersProvided",
        ("FIX.5.0", "780", "2"): "2 .. FullDetailsProvided",
        ("FIX.5.0", "780", "3"): "3 .. SSIDBIDsProvided",
        ("FIX.5.0", "780", "4"): "4 .. PhoneForInstructions",
        ("FIX.5.0", "783", "6"): "6 .. UKNationalInsuranceOrPensionNumber",
        ("FIX.5.0", "783", "7"): "7 .. USSocialSecurityNumber",
        ("FIX.5.0", "783", "8"): "8 .. USEmployerOrTaxIDNumber",
        ("FIX.5.0", "783", "9"): "9 .. AustralianBusinessNumber",
        ("FIX.5.0", "783", "A"): "A .. AustralianTaxFileNumber",
        ("FIX.5.0", "783", "J"): "J .. TaxID",
        ("FIX.5.0", "783", "1"): "1 .. KoreanInvestorID",
        ("FIX.5.0", "783", "2"): "2 .. TaiwaneseForeignInvestorID",
        ("FIX.5.0", "783", "3"): "3 .. TaiwaneseTradingAcct",
        ("FIX.5.0", "783", "4"): "4 .. MalaysianCentralDepository",
        ("FIX.5.0", "783", "5"): "5 .. ChineseInvestorID",
        ("FIX.5.0", "783", "I"): "I .. ISITCAcronym",
        ("FIX.5.0", "783", "B"): "B .. BIC",
        ("FIX.5.0", "783", "C"): "C .. GeneralIdentifier",
        ("FIX.5.0", "783", "D"): "D .. Proprietary",
        ("FIX.5.0", "783", "E"): "E .. ISOCountryCode",
        ("FIX.5.0", "783", "F"): "F .. SettlementEntityLocation",
        ("FIX.5.0", "783", "G"): "G .. MIC",
        ("FIX.5.0", "783", "H"): "H .. CSDParticipant",
        ("FIX.5.0", "783", "K"): "K .. AustralianCompanyNumber",
        ("FIX.5.0", "783", "L"): "L .. AustralianRegisteredBodyNumber",
        ("FIX.5.0", "783", "M"): "M .. CFTCReportingFirmIdentifier",
        ("FIX.5.0", "783", "N"): "N .. LegalEntityIdentifier",
        ("FIX.5.0", "783", "O"): "O .. InterimIdentifier",
        ("FIX.5.0", "783", "P"): "P .. ShortCodeIdentifier",
        ("FIX.5.0", "783", "Q"): "Q .. NationalIDNaturalPerson",
        ("FIX.5.0", "784", "1"): "1 .. ExecutingFirm",
        ("FIX.5.0", "784", "2"): "2 .. BrokerOfCredit",
        ("FIX.5.0", "784", "3"): "3 .. ClientID",
        ("FIX.5.0", "784", "4"): "4 .. ClearingFirm",
        ("FIX.5.0", "784", "5"): "5 .. InvestorID",
        ("FIX.5.0", "784", "6"): "6 .. IntroducingFirm",
        ("FIX.5.0", "784", "7"): "7 .. EnteringFirm",
        ("FIX.5.0", "784", "8"): "8 .. Locate",
        ("FIX.5.0", "784", "9"): "9 .. FundManagerClientID",
        ("FIX.5.0", "784", "10"): "10 .. SettlementLocation",
        ("FIX.5.0", "784", "11"): "11 .. OrderOriginationTrader",
        ("FIX.5.0", "784", "12"): "12 .. ExecutingTrader",
        ("FIX.5.0", "784", "13"): "13 .. OrderOriginationFirm",
        ("FIX.5.0", "784", "15"): "15 .. CorrespondantClearingFirm",
        ("FIX.5.0", "784", "16"): "16 .. ExecutingSystem",
        ("FIX.5.0", "784", "17"): "17 .. ContraFirm",
        ("FIX.5.0", "784", "18"): "18 .. ContraClearingFirm",
        ("FIX.5.0", "784", "19"): "19 .. SponsoringFirm",
        ("FIX.5.0", "784", "20"): "20 .. UnderlyingContraFirm",
        ("FIX.5.0", "784", "21"): "21 .. ClearingOrganization",
        ("FIX.5.0", "784", "22"): "22 .. Exchange",
        ("FIX.5.0", "784", "24"): "24 .. CustomerAccount",
        ("FIX.5.0", "784", "25"): "25 .. CorrespondentClearingOrganization",
        ("FIX.5.0", "784", "26"): "26 .. CorrespondentBroker",
        ("FIX.5.0", "784", "27"): "27 .. Buyer",
        ("FIX.5.0", "784", "28"): "28 .. Custodian",
        ("FIX.5.0", "784", "29"): "29 .. Intermediary",
        ("FIX.5.0", "784", "30"): "30 .. Agent",
        ("FIX.5.0", "784", "31"): "31 .. SubCustodian",
        ("FIX.5.0", "784", "32"): "32 .. Beneficiary",
        ("FIX.5.0", "784", "33"): "33 .. InterestedParty",
        ("FIX.5.0", "784", "34"): "34 .. RegulatoryBody",
        ("FIX.5.0", "784", "35"): "35 .. LiquidityProvider",
        ("FIX.5.0", "784", "36"): "36 .. EnteringTrader",
        ("FIX.5.0", "784", "37"): "37 .. ContraTrader",
        ("FIX.5.0", "784", "38"): "38 .. PositionAccount",
        ("FIX.5.0", "784", "39"): "39 .. ContraInvestorID",
        ("FIX.5.0", "784", "40"): "40 .. TransferToFirm",
        ("FIX.5.0", "784", "41"): "41 .. ContraPositionAccount",
        ("FIX.5.0", "784", "42"): "42 .. ContraExchange",
        ("FIX.5.0", "784", "43"): "43 .. InternalCarryAccount",
        ("FIX.5.0", "784", "44"): "44 .. OrderEntryOperatorID",
        ("FIX.5.0", "784", "45"): "45 .. SecondaryAccountNumber",
        ("FIX.5.0", "784", "46"): "46 .. ForeignFirm",
        ("FIX.5.0", "784", "47"): "47 .. ThirdPartyAllocationFirm",
        ("FIX.5.0", "784", "48"): "48 .. ClaimingAccount",
        ("FIX.5.0", "784", "49"): "49 .. AssetManager",
        ("FIX.5.0", "784", "50"): "50 .. PledgorAccount",
        ("FIX.5.0", "784", "51"): "51 .. PledgeeAccount",
        ("FIX.5.0", "784", "52"): "52 .. LargeTraderReportableAccount",
        ("FIX.5.0", "784", "53"): "53 .. TraderMnemonic",
        ("FIX.5.0", "784", "54"): "54 .. SenderLocation",
        ("FIX.5.0", "784", "55"): "55 .. SessionID",
        ("FIX.5.0", "784", "56"): "56 .. AcceptableCounterparty",
        ("FIX.5.0", "784", "57"): "57 .. UnacceptableCounterparty",
        ("FIX.5.0", "784", "58"): "58 .. EnteringUnit",
        ("FIX.5.0", "784", "59"): "59 .. ExecutingUnit",
        ("FIX.5.0", "784", "60"): "60 .. IntroducingBroker",
        ("FIX.5.0", "784", "61"): "61 .. QuoteOriginator",
        ("FIX.5.0", "784", "62"): "62 .. ReportOriginator",
        ("FIX.5.0", "784", "63"): "63 .. SystematicInternaliser",
        ("FIX.5.0", "784", "64"): "64 .. MultilateralTradingFacility",
        ("FIX.5.0", "784", "65"): "65 .. RegulatedMarket",
        ("FIX.5.0", "784", "66"): "66 .. MarketMaker",
        ("FIX.5.0", "784", "67"): "67 .. InvestmentFirm",
        ("FIX.5.0", "784", "68"): "68 .. HostCompetentAuthority",
        ("FIX.5.0", "784", "69"): "69 .. HomeCompetentAuthority",
        ("FIX.5.0", "784", "70"): "70 .. CompetentAuthorityLiquidity",
        ("FIX.5.0", "784", "71"): "71 .. CompetentAuthorityTransactionVenue",
        ("FIX.5.0", "784", "72"): "72 .. ReportingIntermediary",
        ("FIX.5.0", "784", "73"): "73 .. ExecutionVenue",
        ("FIX.5.0", "784", "74"): "74 .. MarketDataEntryOriginator",
        ("FIX.5.0", "784", "75"): "75 .. LocationID",
        ("FIX.5.0", "784", "76"): "76 .. DeskID",
        ("FIX.5.0", "784", "77"): "77 .. MarketDataMarket",
        ("FIX.5.0", "784", "78"): "78 .. AllocationEntity",
        ("FIX.5.0", "784", "79"): "79 .. PrimeBroker",
        ("FIX.5.0", "784", "80"): "80 .. StepOutFirm",
        ("FIX.5.0", "784", "81"): "81 .. BrokerClearingID",
        ("FIX.5.0", "784", "82"): "82 .. CentralRegistrationDepository",
        ("FIX.5.0", "784", "83"): "83 .. ClearingAccount",
        ("FIX.5.0", "784", "84"): "84 .. AcceptableSettlingCounterparty",
        ("FIX.5.0", "784", "85"): "85 .. UnacceptableSettlingCounterparty",
        ("FIX.5.0", "784", "86"): "86 .. CLSMemberBank",
        ("FIX.5.0", "784", "87"): "87 .. InConcertGroup",
        ("FIX.5.0", "784", "88"): "88 .. InConcertControllingEntity",
        ("FIX.5.0", "784", "89"): "89 .. LargePositionsReportingAccount",
        ("FIX.5.0", "784", "90"): "90 .. SettlementFirm",
        ("FIX.5.0", "784", "91"): "91 .. SettlementAccount",
        ("FIX.5.0", "784", "92"): "92 .. ReportingMarketCenter",
        ("FIX.5.0", "784", "93"): "93 .. RelatedReportingMarketCenter",
        ("FIX.5.0", "784", "94"): "94 .. AwayMarket",
        ("FIX.5.0", "784", "95"): "95 .. GiveupTradingFirm",
        ("FIX.5.0", "784", "96"): "96 .. TakeupTradingFirm",
        ("FIX.5.0", "784", "97"): "97 .. GiveupClearingFirm",
        ("FIX.5.0", "784", "98"): "98 .. TakeupClearingFirm",
        ("FIX.5.0", "784", "99"): "99 .. OriginatingMarket",
        ("FIX.5.0", "784", "100"): "100 .. MarginAccount",
        ("FIX.5.0", "784", "101"): "101 .. CollateralAssetAccount",
        ("FIX.5.0", "784", "102"): "102 .. DataRepository",
        ("FIX.5.0", "784", "103"): "103 .. CalculationAgent",
        ("FIX.5.0", "784", "104"): "104 .. ExerciseNoticeSender",
        ("FIX.5.0", "784", "105"): "105 .. ExerciseNoticeReceiver",
        ("FIX.5.0", "784", "106"): "106 .. RateReferenceBank",
        ("FIX.5.0", "784", "107"): "107 .. Correspondent",
        ("FIX.5.0", "784", "109"): "109 .. BeneficiaryBank",
        ("FIX.5.0", "784", "110"): "110 .. Borrower",
        ("FIX.5.0", "784", "111"): "111 .. PrimaryObligator",
        ("FIX.5.0", "784", "112"): "112 .. Guarantor",
        ("FIX.5.0", "784", "113"): "113 .. ExcludedReferenceEntity",
        ("FIX.5.0", "784", "114"): "114 .. DeterminingParty",
        ("FIX.5.0", "784", "115"): "115 .. HedgingParty",
        ("FIX.5.0", "784", "116"): "116 .. ReportingEntity",
        ("FIX.5.0", "784", "117"): "117 .. SalesPerson",
        ("FIX.5.0", "784", "118"): "118 .. Operator",
        ("FIX.5.0", "784", "119"): "119 .. CSD",
        ("FIX.5.0", "784", "120"): "120 .. ICSD",
        ("FIX.5.0", "784", "121"): "121 .. TradingSubAccount",
        ("FIX.5.0", "784", "122"): "122 .. InvestmentDecisionMaker",
        ("FIX.5.0", "784", "123"): "123 .. PublishingIntermediary",
        ("FIX.5.0", "786", "1"): "1 .. Firm",
        ("FIX.5.0", "786", "2"): "2 .. Person",
        ("FIX.5.0", "786", "3"): "3 .. System",
        ("FIX.5.0", "786", "4"): "4 .. Application",
        ("FIX.5.0", "786", "5"): "5 .. FullLegalNameOfFirm",
        ("FIX.5.0", "786", "6"): "6 .. PostalAddress",
        ("FIX.5.0", "786", "7"): "7 .. PhoneNumber",
        ("FIX.5.0", "786", "8"): "8 .. EmailAddress",
        ("FIX.5.0", "786", "9"): "9 .. ContactName",
        ("FIX.5.0", "786", "10"): "10 .. SecuritiesAccountNumber",
        ("FIX.5.0", "786", "11"): "11 .. RegistrationNumber",
        ("FIX.5.0", "786", "12"): "12 .. RegisteredAddressForConfirmation",
        ("FIX.5.0", "786", "13"): "13 .. RegulatoryStatus",
        ("FIX.5.0", "786", "14"): "14 .. RegistrationName",
        ("FIX.5.0", "786", "15"): "15 .. CashAccountNumber",
        ("FIX.5.0", "786", "16"): "16 .. BIC",
        ("FIX.5.0", "786", "17"): "17 .. CSDParticipantMemberCode",
        ("FIX.5.0", "786", "18"): "18 .. RegisteredAddress",
        ("FIX.5.0", "786", "19"): "19 .. FundAccountName",
        ("FIX.5.0", "786", "20"): "20 .. TelexNumber",
        ("FIX.5.0", "786", "21"): "21 .. FaxNumber",
        ("FIX.5.0", "786", "22"): "22 .. SecuritiesAccountName",
        ("FIX.5.0", "786", "23"): "23 .. CashAccountName",
        ("FIX.5.0", "786", "24"): "24 .. Department",
        ("FIX.5.0", "786", "25"): "25 .. LocationDesk",
        ("FIX.5.0", "786", "26"): "26 .. PositionAccountType",
        ("FIX.5.0", "786", "27"): "27 .. SecurityLocateID",
        ("FIX.5.0", "786", "28"): "28 .. MarketMaker",
        ("FIX.5.0", "786", "29"): "29 .. EligibleCounterparty",
        ("FIX.5.0", "786", "30"): "30 .. ProfessionalClient",
        ("FIX.5.0", "786", "31"): "31 .. Location",
        ("FIX.5.0", "786", "32"): "32 .. ExecutionVenue",
        ("FIX.5.0", "786", "33"): "33 .. CurrencyDeliveryIdentifier",
        ("FIX.5.0", "786", "34"): "34 .. AddressCity",
        ("FIX.5.0", "786", "35"): "35 .. AddressStateOrProvince",
        ("FIX.5.0", "786", "36"): "36 .. AddressPostalCode",
        ("FIX.5.0", "786", "37"): "37 .. AddressStreet",
        ("FIX.5.0", "786", "38"): "38 .. AddressISOCountryCode",
        ("FIX.5.0", "786", "39"): "39 .. ISOCountryCode",
        ("FIX.5.0", "786", "40"): "40 .. MarketSegment",
        ("FIX.5.0", "786", "41"): "41 .. CustomerAccountType",
        ("FIX.5.0", "786", "42"): "42 .. OmnibusAccount",
        ("FIX.5.0", "786", "43"): "43 .. FundsSegregationType",
        ("FIX.5.0", "786", "44"): "44 .. GuaranteeFund",
        ("FIX.5.0", "786", "45"): "45 .. SwapDealer",
        ("FIX.5.0", "786", "46"): "46 .. MajorParticipant",
        ("FIX.5.0", "786", "47"): "47 .. FinancialEntity",
        ("FIX.5.0", "786", "48"): "48 .. USPerson",
        ("FIX.5.0", "786", "49"): "49 .. ReportingEntityIndicator",
        ("FIX.5.0", "786", "50"): "50 .. ElectedClearingRequirementException",
        ("FIX.5.0", "786", "51"): "51 .. BusinessCenter",
        ("FIX.5.0", "786", "52"): "52 .. ReferenceText",
        ("FIX.5.0", "786", "53"): "53 .. ShortMarkingExemptAccount",
        ("FIX.5.0", "786", "54"): "54 .. ParentFirmIdentifier",
        ("FIX.5.0", "786", "55"): "55 .. ParentFirmName",
        ("FIX.5.0", "786", "56"): "56 .. DealIdentifier",
        ("FIX.5.0", "786", "57"): "57 .. SystemTradeID",
        ("FIX.5.0", "786", "58"): "58 .. SystemTradeSubID",
        ("FIX.5.0", "786", "59"): "59 .. FCMCode",
        ("FIX.5.0", "786", "60"): "60 .. DlvryTrmlCode",
        ("FIX.5.0", "786", "61"): "61 .. VolntyRptEntity",
        ("FIX.5.0", "786", "62"): "62 .. RptObligJursdctn",
        ("FIX.5.0", "786", "63"): "63 .. VolntyRptJursdctn",
        ("FIX.5.0", "786", "64"): "64 .. CompanyActivities",
        ("FIX.5.0", "786", "65"): "65 .. EEAreaDomiciled",
        ("FIX.5.0", "786", "66"): "66 .. ContractLinked",
        ("FIX.5.0", "786", "67"): "67 .. ContractAbove",
        ("FIX.5.0", "786", "68"): "68 .. VolntyRptPty",
        ("FIX.5.0", "786", "69"): "69 .. EndUser",
        ("FIX.5.0", "786", "70"): "70 .. LocationOrJurisdiction",
        ("FIX.5.0", "786", "71"): "71 .. DerivativesDealer",
        ("FIX.5.0", "786", "72"): "72 .. Domicile",
        ("FIX.5.0", "786", "73"): "73 .. ExemptFromRecognition",
        ("FIX.5.0", "786", "74"): "74 .. Payer",
        ("FIX.5.0", "786", "75"): "75 .. Receiver",
        ("FIX.5.0", "786", "76"): "76 .. SystematicInternaliser",
        ("FIX.5.0", "786", "77"): "77 .. PublishingEntityIndicator",
        ("FIX.5.0", "786", "78"): "78 .. FirstName",
        ("FIX.5.0", "786", "79"): "79 .. Surname",
        ("FIX.5.0", "786", "80"): "80 .. DateOfBirth",
        ("FIX.5.0", "786", "81"): "81 .. OrderTransmittingFirm",
        ("FIX.5.0", "786", "82"): "82 .. OrderTransmittingFirmBuyer",
        ("FIX.5.0", "786", "83"): "83 .. OrderTransmitterSeller",
        ("FIX.5.0", "786", "84"): "84 .. LegalEntityIdentifier",
        ("FIX.5.0", "787", "C"): "C .. Cash",
        ("FIX.5.0", "787", "S"): "S .. Securities",
        ("FIX.5.0", "788", "1"): "1 .. Overnight",
        ("FIX.5.0", "788", "2"): "2 .. Term",
        ("FIX.5.0", "788", "3"): "3 .. Flexible",
        ("FIX.5.0", "788", "4"): "4 .. Open",
        ("FIX.5.0", "792", "0"): "0 .. UnableToProcessRequest",
        ("FIX.5.0", "792", "1"): "1 .. UnknownAccount",
        ("FIX.5.0", "792", "2"): "2 .. NoMatchingSettlementInstructionsFound",
        ("FIX.5.0", "792", "99"): "99 .. Other",
        ("FIX.5.0", "794", "2"): "2 .. PreliminaryRequestToIntermediary",
        ("FIX.5.0", "794", "3"): "3 .. SellsideCalculatedUsingPreliminary",
        ("FIX.5.0", "794", "4"): "4 .. SellsideCalculatedWithoutPreliminary",
        ("FIX.5.0", "794", "5"): "5 .. WarehouseRecap",
        ("FIX.5.0", "794", "8"): "8 .. RequestToIntermediary",
        ("FIX.5.0", "794", "15"): "15 .. Giveup",
        ("FIX.5.0", "794", "16"): "16 .. Takeup",
        ("FIX.5.0", "794", "17"): "17 .. Reversal",
        ("FIX.5.0", "794", "18"): "18 .. Alleged",
        ("FIX.5.0", "794", "19"): "19 .. SubAllocationGiveup",
        ("FIX.5.0", "796", "1"): "1 .. OriginalDetailsIncomplete",
        ("FIX.5.0", "796", "2"): "2 .. ChangeInUnderlyingOrderDetails",
        ("FIX.5.0", "796", "3"): "3 .. CancelledByGiveupFirm",
        ("FIX.5.0", "796", "99"): "99 .. Other",
        ("FIX.5.0", "798", "1"): "1 .. CarriedCustomerSide",
        ("FIX.5.0", "798", "2"): "2 .. CarriedNonCustomerSide",
        ("FIX.5.0", "798", "3"): "3 .. HouseTrader",
        ("FIX.5.0", "798", "4"): "4 .. FloorTrader",
        ("FIX.5.0", "798", "6"): "6 .. CarriedNonCustomerSideCrossMargined",
        ("FIX.5.0", "798", "7"): "7 .. HouseTraderCrossMargined",
        ("FIX.5.0", "798", "8"): "8 .. JointBackOfficeAccount",
        ("FIX.5.0", "803", "1"): "1 .. Firm",
        ("FIX.5.0", "803", "2"): "2 .. Person",
        ("FIX.5.0", "803", "3"): "3 .. System",
        ("FIX.5.0", "803", "4"): "4 .. Application",
        ("FIX.5.0", "803", "5"): "5 .. FullLegalNameOfFirm",
        ("FIX.5.0", "803", "6"): "6 .. PostalAddress",
        ("FIX.5.0", "803", "7"): "7 .. PhoneNumber",
        ("FIX.5.0", "803", "8"): "8 .. EmailAddress",
        ("FIX.5.0", "803", "9"): "9 .. ContactName",
        ("FIX.5.0", "803", "10"): "10 .. SecuritiesAccountNumber",
        ("FIX.5.0", "803", "11"): "11 .. RegistrationNumber",
        ("FIX.5.0", "803", "12"): "12 .. RegisteredAddressForConfirmation",
        ("FIX.5.0", "803", "13"): "13 .. RegulatoryStatus",
        ("FIX.5.0", "803", "14"): "14 .. RegistrationName",
        ("FIX.5.0", "803", "15"): "15 .. CashAccountNumber",
        ("FIX.5.0", "803", "16"): "16 .. BIC",
        ("FIX.5.0", "803", "17"): "17 .. CSDParticipantMemberCode",
        ("FIX.5.0", "803", "18"): "18 .. RegisteredAddress",
        ("FIX.5.0", "803", "19"): "19 .. FundAccountName",
        ("FIX.5.0", "803", "20"): "20 .. TelexNumber",
        ("FIX.5.0", "803", "21"): "21 .. FaxNumber",
        ("FIX.5.0", "803", "22"): "22 .. SecuritiesAccountName",
        ("FIX.5.0", "803", "23"): "23 .. CashAccountName",
        ("FIX.5.0", "803", "24"): "24 .. Department",
        ("FIX.5.0", "803", "25"): "25 .. LocationDesk",
        ("FIX.5.0", "803", "26"): "26 .. PositionAccountType",
        ("FIX.5.0", "803", "27"): "27 .. SecurityLocateID",
        ("FIX.5.0", "803", "28"): "28 .. MarketMaker",
        ("FIX.5.0", "803", "29"): "29 .. EligibleCounterparty",
        ("FIX.5.0", "803", "30"): "30 .. ProfessionalClient",
        ("FIX.5.0", "803", "31"): "31 .. Location",
        ("FIX.5.0", "803", "32"): "32 .. ExecutionVenue",
        ("FIX.5.0", "803", "33"): "33 .. CurrencyDeliveryIdentifier",
        ("FIX.5.0", "803", "34"): "34 .. AddressCity",
        ("FIX.5.0", "803", "35"): "35 .. AddressStateOrProvince",
        ("FIX.5.0", "803", "36"): "36 .. AddressPostalCode",
        ("FIX.5.0", "803", "37"): "37 .. AddressStreet",
        ("FIX.5.0", "803", "38"): "38 .. AddressISOCountryCode",
        ("FIX.5.0", "803", "39"): "39 .. ISOCountryCode",
        ("FIX.5.0", "803", "40"): "40 .. MarketSegment",
        ("FIX.5.0", "803", "41"): "41 .. CustomerAccountType",
        ("FIX.5.0", "803", "42"): "42 .. OmnibusAccount",
        ("FIX.5.0", "803", "43"): "43 .. FundsSegregationType",
        ("FIX.5.0", "803", "44"): "44 .. GuaranteeFund",
        ("FIX.5.0", "803", "45"): "45 .. SwapDealer",
        ("FIX.5.0", "803", "46"): "46 .. MajorParticipant",
        ("FIX.5.0", "803", "47"): "47 .. FinancialEntity",
        ("FIX.5.0", "803", "48"): "48 .. USPerson",
        ("FIX.5.0", "803", "49"): "49 .. ReportingEntityIndicator",
        ("FIX.5.0", "803", "50"): "50 .. ElectedClearingRequirementException",
        ("FIX.5.0", "803", "51"): "51 .. BusinessCenter",
        ("FIX.5.0", "803", "52"): "52 .. ReferenceText",
        ("FIX.5.0", "803", "53"): "53 .. ShortMarkingExemptAccount",
        ("FIX.5.0", "803", "54"): "54 .. ParentFirmIdentifier",
        ("FIX.5.0", "803", "55"): "55 .. ParentFirmName",
        ("FIX.5.0", "803", "56"): "56 .. DealIdentifier",
        ("FIX.5.0", "803", "57"): "57 .. SystemTradeID",
        ("FIX.5.0", "803", "58"): "58 .. SystemTradeSubID",
        ("FIX.5.0", "803", "59"): "59 .. FCMCode",
        ("FIX.5.0", "803", "60"): "60 .. DlvryTrmlCode",
        ("FIX.5.0", "803", "61"): "61 .. VolntyRptEntity",
        ("FIX.5.0", "803", "62"): "62 .. RptObligJursdctn",
        ("FIX.5.0", "803", "63"): "63 .. VolntyRptJursdctn",
        ("FIX.5.0", "803", "64"): "64 .. CompanyActivities",
        ("FIX.5.0", "803", "65"): "65 .. EEAreaDomiciled",
        ("FIX.5.0", "803", "66"): "66 .. ContractLinked",
        ("FIX.5.0", "803", "67"): "67 .. ContractAbove",
        ("FIX.5.0", "803", "68"): "68 .. VolntyRptPty",
        ("FIX.5.0", "803", "69"): "69 .. EndUser",
        ("FIX.5.0", "803", "70"): "70 .. LocationOrJurisdiction",
        ("FIX.5.0", "803", "71"): "71 .. DerivativesDealer",
        ("FIX.5.0", "803", "72"): "72 .. Domicile",
        ("FIX.5.0", "803", "73"): "73 .. ExemptFromRecognition",
        ("FIX.5.0", "803", "74"): "74 .. Payer",
        ("FIX.5.0", "803", "75"): "75 .. Receiver",
        ("FIX.5.0", "803", "76"): "76 .. SystematicInternaliser",
        ("FIX.5.0", "803", "77"): "77 .. PublishingEntityIndicator",
        ("FIX.5.0", "803", "78"): "78 .. FirstName",
        ("FIX.5.0", "803", "79"): "79 .. Surname",
        ("FIX.5.0", "803", "80"): "80 .. DateOfBirth",
        ("FIX.5.0", "803", "81"): "81 .. OrderTransmittingFirm",
        ("FIX.5.0", "803", "82"): "82 .. OrderTransmittingFirmBuyer",
        ("FIX.5.0", "803", "83"): "83 .. OrderTransmitterSeller",
        ("FIX.5.0", "803", "84"): "84 .. LegalEntityIdentifier",
        ("FIX.5.0", "805", "1"): "1 .. Firm",
        ("FIX.5.0", "805", "2"): "2 .. Person",
        ("FIX.5.0", "805", "3"): "3 .. System",
        ("FIX.5.0", "805", "4"): "4 .. Application",
        ("FIX.5.0", "805", "5"): "5 .. FullLegalNameOfFirm",
        ("FIX.5.0", "805", "6"): "6 .. PostalAddress",
        ("FIX.5.0", "805", "7"): "7 .. PhoneNumber",
        ("FIX.5.0", "805", "8"): "8 .. EmailAddress",
        ("FIX.5.0", "805", "9"): "9 .. ContactName",
        ("FIX.5.0", "805", "10"): "10 .. SecuritiesAccountNumber",
        ("FIX.5.0", "805", "11"): "11 .. RegistrationNumber",
        ("FIX.5.0", "805", "12"): "12 .. RegisteredAddressForConfirmation",
        ("FIX.5.0", "805", "13"): "13 .. RegulatoryStatus",
        ("FIX.5.0", "805", "14"): "14 .. RegistrationName",
        ("FIX.5.0", "805", "15"): "15 .. CashAccountNumber",
        ("FIX.5.0", "805", "16"): "16 .. BIC",
        ("FIX.5.0", "805", "17"): "17 .. CSDParticipantMemberCode",
        ("FIX.5.0", "805", "18"): "18 .. RegisteredAddress",
        ("FIX.5.0", "805", "19"): "19 .. FundAccountName",
        ("FIX.5.0", "805", "20"): "20 .. TelexNumber",
        ("FIX.5.0", "805", "21"): "21 .. FaxNumber",
        ("FIX.5.0", "805", "22"): "22 .. SecuritiesAccountName",
        ("FIX.5.0", "805", "23"): "23 .. CashAccountName",
        ("FIX.5.0", "805", "24"): "24 .. Department",
        ("FIX.5.0", "805", "25"): "25 .. LocationDesk",
        ("FIX.5.0", "805", "26"): "26 .. PositionAccountType",
        ("FIX.5.0", "805", "27"): "27 .. SecurityLocateID",
        ("FIX.5.0", "805", "28"): "28 .. MarketMaker",
        ("FIX.5.0", "805", "29"): "29 .. EligibleCounterparty",
        ("FIX.5.0", "805", "30"): "30 .. ProfessionalClient",
        ("FIX.5.0", "805", "31"): "31 .. Location",
        ("FIX.5.0", "805", "32"): "32 .. ExecutionVenue",
        ("FIX.5.0", "805", "33"): "33 .. CurrencyDeliveryIdentifier",
        ("FIX.5.0", "805", "34"): "34 .. AddressCity",
        ("FIX.5.0", "805", "35"): "35 .. AddressStateOrProvince",
        ("FIX.5.0", "805", "36"): "36 .. AddressPostalCode",
        ("FIX.5.0", "805", "37"): "37 .. AddressStreet",
        ("FIX.5.0", "805", "38"): "38 .. AddressISOCountryCode",
        ("FIX.5.0", "805", "39"): "39 .. ISOCountryCode",
        ("FIX.5.0", "805", "40"): "40 .. MarketSegment",
        ("FIX.5.0", "805", "41"): "41 .. CustomerAccountType",
        ("FIX.5.0", "805", "42"): "42 .. OmnibusAccount",
        ("FIX.5.0", "805", "43"): "43 .. FundsSegregationType",
        ("FIX.5.0", "805", "44"): "44 .. GuaranteeFund",
        ("FIX.5.0", "805", "45"): "45 .. SwapDealer",
        ("FIX.5.0", "805", "46"): "46 .. MajorParticipant",
        ("FIX.5.0", "805", "47"): "47 .. FinancialEntity",
        ("FIX.5.0", "805", "48"): "48 .. USPerson",
        ("FIX.5.0", "805", "49"): "49 .. ReportingEntityIndicator",
        ("FIX.5.0", "805", "50"): "50 .. ElectedClearingRequirementException",
        ("FIX.5.0", "805", "51"): "51 .. BusinessCenter",
        ("FIX.5.0", "805", "52"): "52 .. ReferenceText",
        ("FIX.5.0", "805", "53"): "53 .. ShortMarkingExemptAccount",
        ("FIX.5.0", "805", "54"): "54 .. ParentFirmIdentifier",
        ("FIX.5.0", "805", "55"): "55 .. ParentFirmName",
        ("FIX.5.0", "805", "56"): "56 .. DealIdentifier",
        ("FIX.5.0", "805", "57"): "57 .. SystemTradeID",
        ("FIX.5.0", "805", "58"): "58 .. SystemTradeSubID",
        ("FIX.5.0", "805", "59"): "59 .. FCMCode",
        ("FIX.5.0", "805", "60"): "60 .. DlvryTrmlCode",
        ("FIX.5.0", "805", "61"): "61 .. VolntyRptEntity",
        ("FIX.5.0", "805", "62"): "62 .. RptObligJursdctn",
        ("FIX.5.0", "805", "63"): "63 .. VolntyRptJursdctn",
        ("FIX.5.0", "805", "64"): "64 .. CompanyActivities",
        ("FIX.5.0", "805", "65"): "65 .. EEAreaDomiciled",
        ("FIX.5.0", "805", "66"): "66 .. ContractLinked",
        ("FIX.5.0", "805", "67"): "67 .. ContractAbove",
        ("FIX.5.0", "805", "68"): "68 .. VolntyRptPty",
        ("FIX.5.0", "805", "69"): "69 .. EndUser",
        ("FIX.5.0", "805", "70"): "70 .. LocationOrJurisdiction",
        ("FIX.5.0", "805", "71"): "71 .. DerivativesDealer",
        ("FIX.5.0", "805", "72"): "72 .. Domicile",
        ("FIX.5.0", "805", "73"): "73 .. ExemptFromRecognition",
        ("FIX.5.0", "805", "74"): "74 .. Payer",
        ("FIX.5.0", "805", "75"): "75 .. Receiver",
        ("FIX.5.0", "805", "76"): "76 .. SystematicInternaliser",
        ("FIX.5.0", "805", "77"): "77 .. PublishingEntityIndicator",
        ("FIX.5.0", "805", "78"): "78 .. FirstName",
        ("FIX.5.0", "805", "79"): "79 .. Surname",
        ("FIX.5.0", "805", "80"): "80 .. DateOfBirth",
        ("FIX.5.0", "805", "81"): "81 .. OrderTransmittingFirm",
        ("FIX.5.0", "805", "82"): "82 .. OrderTransmittingFirmBuyer",
        ("FIX.5.0", "805", "83"): "83 .. OrderTransmitterSeller",
        ("FIX.5.0", "805", "84"): "84 .. LegalEntityIdentifier",
        ("FIX.5.0", "807", "1"): "1 .. Firm",
        ("FIX.5.0", "807", "2"): "2 .. Person",
        ("FIX.5.0", "807", "3"): "3 .. System",
        ("FIX.5.0", "807", "4"): "4 .. Application",
        ("FIX.5.0", "807", "5"): "5 .. FullLegalNameOfFirm",
        ("FIX.5.0", "807", "6"): "6 .. PostalAddress",
        ("FIX.5.0", "807", "7"): "7 .. PhoneNumber",
        ("FIX.5.0", "807", "8"): "8 .. EmailAddress",
        ("FIX.5.0", "807", "9"): "9 .. ContactName",
        ("FIX.5.0", "807", "10"): "10 .. SecuritiesAccountNumber",
        ("FIX.5.0", "807", "11"): "11 .. RegistrationNumber",
        ("FIX.5.0", "807", "12"): "12 .. RegisteredAddressForConfirmation",
        ("FIX.5.0", "807", "13"): "13 .. RegulatoryStatus",
        ("FIX.5.0", "807", "14"): "14 .. RegistrationName",
        ("FIX.5.0", "807", "15"): "15 .. CashAccountNumber",
        ("FIX.5.0", "807", "16"): "16 .. BIC",
        ("FIX.5.0", "807", "17"): "17 .. CSDParticipantMemberCode",
        ("FIX.5.0", "807", "18"): "18 .. RegisteredAddress",
        ("FIX.5.0", "807", "19"): "19 .. FundAccountName",
        ("FIX.5.0", "807", "20"): "20 .. TelexNumber",
        ("FIX.5.0", "807", "21"): "21 .. FaxNumber",
        ("FIX.5.0", "807", "22"): "22 .. SecuritiesAccountName",
        ("FIX.5.0", "807", "23"): "23 .. CashAccountName",
        ("FIX.5.0", "807", "24"): "24 .. Department",
        ("FIX.5.0", "807", "25"): "25 .. LocationDesk",
        ("FIX.5.0", "807", "26"): "26 .. PositionAccountType",
        ("FIX.5.0", "807", "27"): "27 .. SecurityLocateID",
        ("FIX.5.0", "807", "28"): "28 .. MarketMaker",
        ("FIX.5.0", "807", "29"): "29 .. EligibleCounterparty",
        ("FIX.5.0", "807", "30"): "30 .. ProfessionalClient",
        ("FIX.5.0", "807", "31"): "31 .. Location",
        ("FIX.5.0", "807", "32"): "32 .. ExecutionVenue",
        ("FIX.5.0", "807", "33"): "33 .. CurrencyDeliveryIdentifier",
        ("FIX.5.0", "807", "34"): "34 .. AddressCity",
        ("FIX.5.0", "807", "35"): "35 .. AddressStateOrProvince",
        ("FIX.5.0", "807", "36"): "36 .. AddressPostalCode",
        ("FIX.5.0", "807", "37"): "37 .. AddressStreet",
        ("FIX.5.0", "807", "38"): "38 .. AddressISOCountryCode",
        ("FIX.5.0", "807", "39"): "39 .. ISOCountryCode",
        ("FIX.5.0", "807", "40"): "40 .. MarketSegment",
        ("FIX.5.0", "807", "41"): "41 .. CustomerAccountType",
        ("FIX.5.0", "807", "42"): "42 .. OmnibusAccount",
        ("FIX.5.0", "807", "43"): "43 .. FundsSegregationType",
        ("FIX.5.0", "807", "44"): "44 .. GuaranteeFund",
        ("FIX.5.0", "807", "45"): "45 .. SwapDealer",
        ("FIX.5.0", "807", "46"): "46 .. MajorParticipant",
        ("FIX.5.0", "807", "47"): "47 .. FinancialEntity",
        ("FIX.5.0", "807", "48"): "48 .. USPerson",
        ("FIX.5.0", "807", "49"): "49 .. ReportingEntityIndicator",
        ("FIX.5.0", "807", "50"): "50 .. ElectedClearingRequirementException",
        ("FIX.5.0", "807", "51"): "51 .. BusinessCenter",
        ("FIX.5.0", "807", "52"): "52 .. ReferenceText",
        ("FIX.5.0", "807", "53"): "53 .. ShortMarkingExemptAccount",
        ("FIX.5.0", "807", "54"): "54 .. ParentFirmIdentifier",
        ("FIX.5.0", "807", "55"): "55 .. ParentFirmName",
        ("FIX.5.0", "807", "56"): "56 .. DealIdentifier",
        ("FIX.5.0", "807", "57"): "57 .. SystemTradeID",
        ("FIX.5.0", "807", "58"): "58 .. SystemTradeSubID",
        ("FIX.5.0", "807", "59"): "59 .. FCMCode",
        ("FIX.5.0", "807", "60"): "60 .. DlvryTrmlCode",
        ("FIX.5.0", "807", "61"): "61 .. VolntyRptEntity",
        ("FIX.5.0", "807", "62"): "62 .. RptObligJursdctn",
        ("FIX.5.0", "807", "63"): "63 .. VolntyRptJursdctn",
        ("FIX.5.0", "807", "64"): "64 .. CompanyActivities",
        ("FIX.5.0", "807", "65"): "65 .. EEAreaDomiciled",
        ("FIX.5.0", "807", "66"): "66 .. ContractLinked",
        ("FIX.5.0", "807", "67"): "67 .. ContractAbove",
        ("FIX.5.0", "807", "68"): "68 .. VolntyRptPty",
        ("FIX.5.0", "807", "69"): "69 .. EndUser",
        ("FIX.5.0", "807", "70"): "70 .. LocationOrJurisdiction",
        ("FIX.5.0", "807", "71"): "71 .. DerivativesDealer",
        ("FIX.5.0", "807", "72"): "72 .. Domicile",
        ("FIX.5.0", "807", "73"): "73 .. ExemptFromRecognition",
        ("FIX.5.0", "807", "74"): "74 .. Payer",
        ("FIX.5.0", "807", "75"): "75 .. Receiver",
        ("FIX.5.0", "807", "76"): "76 .. SystematicInternaliser",
        ("FIX.5.0", "807", "77"): "77 .. PublishingEntityIndicator",
        ("FIX.5.0", "807", "78"): "78 .. FirstName",
        ("FIX.5.0", "807", "79"): "79 .. Surname",
        ("FIX.5.0", "807", "80"): "80 .. DateOfBirth",
        ("FIX.5.0", "807", "81"): "81 .. OrderTransmittingFirm",
        ("FIX.5.0", "807", "82"): "82 .. OrderTransmittingFirmBuyer",
        ("FIX.5.0", "807", "83"): "83 .. OrderTransmitterSeller",
        ("FIX.5.0", "807", "84"): "84 .. LegalEntityIdentifier",
        ("FIX.5.0", "808", "1"): "1 .. PendingAccept",
        ("FIX.5.0", "808", "2"): "2 .. PendingRelease",
        ("FIX.5.0", "808", "3"): "3 .. PendingReversal",
        ("FIX.5.0", "808", "4"): "4 .. Accept",
        ("FIX.5.0", "808", "5"): "5 .. BlockLevelReject",
        ("FIX.5.0", "808", "6"): "6 .. AccountLevelReject",
        ("FIX.5.0", "814", "0"): "0 .. NoActionTaken",
        ("FIX.5.0", "814", "1"): "1 .. QueueFlushed",
        ("FIX.5.0", "814", "2"): "2 .. OverlayLast",
        ("FIX.5.0", "814", "3"): "3 .. EndSession",
        ("FIX.5.0", "815", "0"): "0 .. NoActionTaken",
        ("FIX.5.0", "815", "1"): "1 .. QueueFlushed",
        ("FIX.5.0", "815", "2"): "2 .. OverlayLast",
        ("FIX.5.0", "815", "3"): "3 .. EndSession",
        ("FIX.5.0", "819", "0"): "0 .. NoAveragePricing",
        ("FIX.5.0", "819", "1"): "1 .. Trade",
        ("FIX.5.0", "819", "2"): "2 .. LastTrade",
        ("FIX.5.0", "819", "3"): "3 .. NotionalValueAveragePxGroupTrade",
        ("FIX.5.0", "819", "4"): "4 .. AveragePricedTrade",
        ("FIX.5.0", "826", "0"): "0 .. AllocationNotRequired",
        ("FIX.5.0", "826", "1"): "1 .. AllocationRequired",
        ("FIX.5.0", "826", "2"): "2 .. UseAllocationProvidedWithTheTrade",
        ("FIX.5.0", "826", "3"): "3 .. AllocationGiveUpExecutor",
        ("FIX.5.0", "826", "4"): "4 .. AllocationFromExecutor",
        ("FIX.5.0", "826", "5"): "5 .. AllocationToClaimAccount",
        ("FIX.5.0", "826", "6"): "6 .. TradeSplit",
        ("FIX.5.0", "827", "0"): "0 .. ExpireOnTradingSessionClose",
        ("FIX.5.0", "827", "1"): "1 .. ExpireOnTradingSessionOpen",
        ("FIX.5.0", "827", "2"): "2 .. SpecifiedExpiration",
        ("FIX.5.0", "828", "0"): "0 .. RegularTrade",
        ("FIX.5.0", "828", "1"): "1 .. BlockTrade",
        ("FIX.5.0", "828", "2"): "2 .. EFP",
        ("FIX.5.0", "828", "3"): "3 .. Transfer",
        ("FIX.5.0", "828", "4"): "4 .. LateTrade",
        ("FIX.5.0", "828", "5"): "5 .. TTrade",
        ("FIX.5.0", "828", "6"): "6 .. WeightedAveragePriceTrade",
        ("FIX.5.0", "828", "7"): "7 .. BunchedTrade",
        ("FIX.5.0", "828", "8"): "8 .. LateBunchedTrade",
        ("FIX.5.0", "828", "9"): "9 .. PriorReferencePriceTrade",
        ("FIX.5.0", "828", "10"): "10 .. AfterHoursTrade",
        ("FIX.5.0", "828", "11"): "11 .. ExchangeForRisk",
        ("FIX.5.0", "828", "12"): "12 .. ExchangeForSwap",
        ("FIX.5.0", "828", "13"): "13 .. ExchangeOfFuturesFor",
        ("FIX.5.0", "828", "14"): "14 .. ExchangeOfOptionsForOptions",
        ("FIX.5.0", "828", "15"): "15 .. TradingAtSettlement",
        ("FIX.5.0", "828", "16"): "16 .. AllOrNone",
        ("FIX.5.0", "828", "17"): "17 .. FuturesLargeOrderExecution",
        ("FIX.5.0", "828", "18"): "18 .. ExchangeOfFuturesForFutures",
        ("FIX.5.0", "828", "19"): "19 .. OptionInterimTrade",
        ("FIX.5.0", "828", "20"): "20 .. OptionCabinetTrade",
        ("FIX.5.0", "828", "22"): "22 .. PrivatelyNegotiatedTrades",
        ("FIX.5.0", "828", "23"): "23 .. SubstitutionOfFuturesForForwards",
        ("FIX.5.0", "828", "48"): "48 .. NonStandardSettlement",
        ("FIX.5.0", "828", "49"): "49 .. DerivativeRelatedTransaction",
        ("FIX.5.0", "828", "50"): "50 .. PortfolioTrade",
        ("FIX.5.0", "828", "51"): "51 .. VolumeWeightedAverageTrade",
        ("FIX.5.0", "828", "52"): "52 .. ExchangeGrantedTrade",
        ("FIX.5.0", "828", "53"): "53 .. RepurchaseAgreement",
        ("FIX.5.0", "828", "54"): "54 .. OTC",
        ("FIX.5.0", "828", "55"): "55 .. ExchangeBasisFacility",
        ("FIX.5.0", "828", "56"): "56 .. OpeningTrade",
        ("FIX.5.0", "828", "57"): "57 .. NettedTrade",
        ("FIX.5.0", "828", "58"): "58 .. BlockSwapTrade",
        ("FIX.5.0", "828", "59"): "59 .. CreditEventTrade",
        ("FIX.5.0", "828", "60"): "60 .. SuccessionEventTrade",
        ("FIX.5.0", "828", "61"): "61 .. GiveUpGiveInTrade",
        ("FIX.5.0", "828", "62"): "62 .. DarkTrade",
        ("FIX.5.0", "828", "63"): "63 .. TechnicalTrade",
        ("FIX.5.0", "828", "64"): "64 .. Benchmark",
        ("FIX.5.0", "828", "65"): "65 .. PackageTrade",
        ("FIX.5.0", "828", "24"): "24 .. ErrorTrade",
        ("FIX.5.0", "828", "25"): "25 .. SpecialCumDividend",
        ("FIX.5.0", "828", "26"): "26 .. SpecialExDividend",
        ("FIX.5.0", "828", "27"): "27 .. SpecialCumCoupon",
        ("FIX.5.0", "828", "28"): "28 .. SpecialExCoupon",
        ("FIX.5.0", "828", "29"): "29 .. CashSettlement",
        ("FIX.5.0", "828", "30"): "30 .. SpecialPrice",
        ("FIX.5.0", "828", "31"): "31 .. GuaranteedDelivery",
        ("FIX.5.0", "828", "32"): "32 .. SpecialCumRights",
        ("FIX.5.0", "828", "33"): "33 .. SpecialExRights",
        ("FIX.5.0", "828", "34"): "34 .. SpecialCumCapitalRepayments",
        ("FIX.5.0", "828", "35"): "35 .. SpecialExCapitalRepayments",
        ("FIX.5.0", "828", "36"): "36 .. SpecialCumBonus",
        ("FIX.5.0", "828", "37"): "37 .. SpecialExBonus",
        ("FIX.5.0", "828", "38"): "38 .. LargeTrade",
        ("FIX.5.0", "828", "39"): "39 .. WorkedPrincipalTrade",
        ("FIX.5.0", "828", "40"): "40 .. BlockTrades",
        ("FIX.5.0", "828", "41"): "41 .. NameChange",
        ("FIX.5.0", "828", "42"): "42 .. PortfolioTransfer",
        ("FIX.5.0", "828", "43"): "43 .. ProrogationBuy",
        ("FIX.5.0", "828", "44"): "44 .. ProrogationSell",
        ("FIX.5.0", "828", "45"): "45 .. OptionExercise",
        ("FIX.5.0", "828", "46"): "46 .. DeltaNeutralTransaction",
        ("FIX.5.0", "828", "47"): "47 .. FinancingTransaction",
        ("FIX.5.0", "829", "0"): "0 .. CMTA",
        ("FIX.5.0", "829", "1"): "1 .. InternalTransferOrAdjustment",
        ("FIX.5.0", "829", "2"): "2 .. ExternalTransferOrTransferOfAccount",
        ("FIX.5.0", "829", "3"): "3 .. RejectForSubmittingSide",
        ("FIX.5.0", "829", "4"): "4 .. AdvisoryForContraSide",
        ("FIX.5.0", "829", "5"): "5 .. OffsetDueToAnAllocation",
        ("FIX.5.0", "829", "6"): "6 .. OnsetDueToAnAllocation",
        ("FIX.5.0", "829", "7"): "7 .. DifferentialSpread",
        ("FIX.5.0", "829", "8"): "8 .. ImpliedSpreadLegExecutedAgainstAnOutright",
        ("FIX.5.0", "829", "9"): "9 .. TransactionFromExercise",
        ("FIX.5.0", "829", "10"): "10 .. TransactionFromAssignment",
        ("FIX.5.0", "829", "11"): "11 .. ACATS",
        ("FIX.5.0", "829", "33"): "33 .. OffHoursTrade",
        ("FIX.5.0", "829", "34"): "34 .. OnHoursTrade",
        ("FIX.5.0", "829", "35"): "35 .. OTCQuote",
        ("FIX.5.0", "829", "36"): "36 .. ConvertedSWAP",
        ("FIX.5.0", "829", "40"): "40 .. WashTrade",
        ("FIX.5.0", "829", "41"): "41 .. TradeAtSettlement",
        ("FIX.5.0", "829", "42"): "42 .. AuctionTrade",
        ("FIX.5.0", "829", "43"): "43 .. TradeAtMarker",
        ("FIX.5.0", "829", "44"): "44 .. CreditDefault",
        ("FIX.5.0", "829", "45"): "45 .. CreditRestructuring",
        ("FIX.5.0", "829", "46"): "46 .. Merger",
        ("FIX.5.0", "829", "47"): "47 .. SpinOff",
        ("FIX.5.0", "829", "48"): "48 .. MultilateralCompression",
        ("FIX.5.0", "829", "50"): "50 .. Balancing",
        ("FIX.5.0", "829", "51"): "51 .. BasisTradeIndexClose",
        ("FIX.5.0", "829", "14"): "14 .. AI",
        ("FIX.5.0", "829", "15"): "15 .. B",
        ("FIX.5.0", "829", "16"): "16 .. K",
        ("FIX.5.0", "829", "17"): "17 .. LC",
        ("FIX.5.0", "829", "18"): "18 .. M",
        ("FIX.5.0", "829", "19"): "19 .. N",
        ("FIX.5.0", "829", "20"): "20 .. NM",
        ("FIX.5.0", "829", "21"): "21 .. NR",
        ("FIX.5.0", "829", "22"): "22 .. P",
        ("FIX.5.0", "829", "23"): "23 .. PA",
        ("FIX.5.0", "829", "24"): "24 .. PC",
        ("FIX.5.0", "829", "25"): "25 .. PN",
        ("FIX.5.0", "829", "26"): "26 .. R",
        ("FIX.5.0", "829", "27"): "27 .. RO",
        ("FIX.5.0", "829", "28"): "28 .. RT",
        ("FIX.5.0", "829", "29"): "29 .. SW",
        ("FIX.5.0", "829", "30"): "30 .. T",
        ("FIX.5.0", "829", "31"): "31 .. WN",
        ("FIX.5.0", "829", "32"): "32 .. WT",
        ("FIX.5.0", "829", "37"): "37 .. CrossedTrade",
        ("FIX.5.0", "829", "38"): "38 .. InterimProtectedTrade",
        ("FIX.5.0", "829", "39"): "39 .. LargeInScale",
        ("FIX.5.0", "835", "0"): "0 .. Floating",
        ("FIX.5.0", "835", "1"): "1 .. Fixed",
        ("FIX.5.0", "836", "0"): "0 .. Price",
        ("FIX.5.0", "836", "1"): "1 .. BasisPoints",
        ("FIX.5.0", "836", "2"): "2 .. Ticks",
        ("FIX.5.0", "836", "3"): "3 .. PriceTier",
        ("FIX.5.0", "836", "4"): "4 .. Percentage",
        ("FIX.5.0", "837", "0"): "0 .. OrBetter",
        ("FIX.5.0", "837", "1"): "1 .. Strict",
        ("FIX.5.0", "837", "2"): "2 .. OrWorse",
        ("FIX.5.0", "838", "1"): "1 .. MoreAggressive",
        ("FIX.5.0", "838", "2"): "2 .. MorePassive",
        ("FIX.5.0", "840", "1"): "1 .. Local",
        ("FIX.5.0", "840", "2"): "2 .. National",
        ("FIX.5.0", "840", "3"): "3 .. Global",
        ("FIX.5.0", "840", "4"): "4 .. NationalExcludingLocal",
        ("FIX.5.0", "841", "0"): "0 .. Floating",
        ("FIX.5.0", "841", "1"): "1 .. Fixed",
        ("FIX.5.0", "842", "0"): "0 .. Price",
        ("FIX.5.0", "842", "1"): "1 .. BasisPoints",
        ("FIX.5.0", "842", "2"): "2 .. Ticks",
        ("FIX.5.0", "842", "3"): "3 .. PriceTier",
        ("FIX.5.0", "843", "0"): "0 .. OrBetter",
        ("FIX.5.0", "843", "1"): "1 .. Strict",
        ("FIX.5.0", "843", "2"): "2 .. OrWorse",
        ("FIX.5.0", "844", "1"): "1 .. MoreAggressive",
        ("FIX.5.0", "844", "2"): "2 .. MorePassive",
        ("FIX.5.0", "846", "1"): "1 .. Local",
        ("FIX.5.0", "846", "2"): "2 .. National",
        ("FIX.5.0", "846", "3"): "3 .. Global",
        ("FIX.5.0", "846", "4"): "4 .. NationalExcludingLocal",
        ("FIX.5.0", "847", "1"): "1 .. VWAP",
        ("FIX.5.0", "847", "2"): "2 .. Participate",
        ("FIX.5.0", "847", "3"): "3 .. MininizeMarketImpact",
        ("FIX.5.0", "851", "1"): "1 .. AddedLiquidity",
        ("FIX.5.0", "851", "2"): "2 .. RemovedLiquidity",
        ("FIX.5.0", "851", "3"): "3 .. LiquidityRoutedOut",
        ("FIX.5.0", "851", "4"): "4 .. Auction",
        ("FIX.5.0", "851", "5"): "5 .. TriggeredStopOrder",
        ("FIX.5.0", "851", "6"): "6 .. TriggeredContingencyOrder",
        ("FIX.5.0", "851", "7"): "7 .. TriggeredMarketOrder",
        ("FIX.5.0", "852", "N"): "N .. DoNotReportTrade",
        ("FIX.5.0", "852", "Y"): "Y .. ReportTrade",
        ("FIX.5.0", "853", "0"): "0 .. DealerSoldShort",
        ("FIX.5.0", "853", "1"): "1 .. DealerSoldShortExempt",
        ("FIX.5.0", "853", "2"): "2 .. SellingCustomerSoldShort",
        ("FIX.5.0", "853", "3"): "3 .. SellingCustomerSoldShortExempt",
        ("FIX.5.0", "853", "4"): "4 .. QualifiedServiceRepresentative",
        ("FIX.5.0", "853", "5"): "5 .. QSROrAGUContraSideSoldShortExempt",
        ("FIX.5.0", "854", "0"): "0 .. Units",
        ("FIX.5.0", "854", "1"): "1 .. Contracts",
        ("FIX.5.0", "854", "2"): "2 .. UnitsOfMeasurePerTimeUnit",
        ("FIX.5.0", "855", "0"): "0 .. RegularTrade",
        ("FIX.5.0", "855", "1"): "1 .. BlockTrade",
        ("FIX.5.0", "855", "2"): "2 .. EFP",
        ("FIX.5.0", "855", "3"): "3 .. Transfer",
        ("FIX.5.0", "855", "4"): "4 .. LateTrade",
        ("FIX.5.0", "855", "5"): "5 .. TTrade",
        ("FIX.5.0", "855", "6"): "6 .. WeightedAveragePriceTrade",
        ("FIX.5.0", "855", "7"): "7 .. BunchedTrade",
        ("FIX.5.0", "855", "8"): "8 .. LateBunchedTrade",
        ("FIX.5.0", "855", "9"): "9 .. PriorReferencePriceTrade",
        ("FIX.5.0", "855", "10"): "10 .. AfterHoursTrade",
        ("FIX.5.0", "855", "11"): "11 .. ExchangeForRisk",
        ("FIX.5.0", "855", "12"): "12 .. ExchangeForSwap",
        ("FIX.5.0", "855", "13"): "13 .. ExchangeOfFuturesFor",
        ("FIX.5.0", "855", "14"): "14 .. ExchangeOfOptionsForOptions",
        ("FIX.5.0", "855", "15"): "15 .. TradingAtSettlement",
        ("FIX.5.0", "855", "16"): "16 .. AllOrNone",
        ("FIX.5.0", "855", "17"): "17 .. FuturesLargeOrderExecution",
        ("FIX.5.0", "855", "18"): "18 .. ExchangeOfFuturesForFutures",
        ("FIX.5.0", "855", "19"): "19 .. OptionInterimTrade",
        ("FIX.5.0", "855", "20"): "20 .. OptionCabinetTrade",
        ("FIX.5.0", "855", "22"): "22 .. PrivatelyNegotiatedTrades",
        ("FIX.5.0", "855", "23"): "23 .. SubstitutionOfFuturesForForwards",
        ("FIX.5.0", "855", "48"): "48 .. NonStandardSettlement",
        ("FIX.5.0", "855", "49"): "49 .. DerivativeRelatedTransaction",
        ("FIX.5.0", "855", "50"): "50 .. PortfolioTrade",
        ("FIX.5.0", "855", "51"): "51 .. VolumeWeightedAverageTrade",
        ("FIX.5.0", "855", "52"): "52 .. ExchangeGrantedTrade",
        ("FIX.5.0", "855", "53"): "53 .. RepurchaseAgreement",
        ("FIX.5.0", "855", "54"): "54 .. OTC",
        ("FIX.5.0", "855", "55"): "55 .. ExchangeBasisFacility",
        ("FIX.5.0", "855", "56"): "56 .. OpeningTrade",
        ("FIX.5.0", "855", "57"): "57 .. NettedTrade",
        ("FIX.5.0", "855", "58"): "58 .. BlockSwapTrade",
        ("FIX.5.0", "855", "59"): "59 .. CreditEventTrade",
        ("FIX.5.0", "855", "60"): "60 .. SuccessionEventTrade",
        ("FIX.5.0", "855", "61"): "61 .. GiveUpGiveInTrade",
        ("FIX.5.0", "855", "62"): "62 .. DarkTrade",
        ("FIX.5.0", "855", "63"): "63 .. TechnicalTrade",
        ("FIX.5.0", "855", "64"): "64 .. Benchmark",
        ("FIX.5.0", "855", "65"): "65 .. PackageTrade",
        ("FIX.5.0", "855", "24"): "24 .. ErrorTrade",
        ("FIX.5.0", "855", "25"): "25 .. SpecialCumDividend",
        ("FIX.5.0", "855", "26"): "26 .. SpecialExDividend",
        ("FIX.5.0", "855", "27"): "27 .. SpecialCumCoupon",
        ("FIX.5.0", "855", "28"): "28 .. SpecialExCoupon",
        ("FIX.5.0", "855", "29"): "29 .. CashSettlement",
        ("FIX.5.0", "855", "30"): "30 .. SpecialPrice",
        ("FIX.5.0", "855", "31"): "31 .. GuaranteedDelivery",
        ("FIX.5.0", "855", "32"): "32 .. SpecialCumRights",
        ("FIX.5.0", "855", "33"): "33 .. SpecialExRights",
        ("FIX.5.0", "855", "34"): "34 .. SpecialCumCapitalRepayments",
        ("FIX.5.0", "855", "35"): "35 .. SpecialExCapitalRepayments",
        ("FIX.5.0", "855", "36"): "36 .. SpecialCumBonus",
        ("FIX.5.0", "855", "37"): "37 .. SpecialExBonus",
        ("FIX.5.0", "855", "38"): "38 .. LargeTrade",
        ("FIX.5.0", "855", "39"): "39 .. WorkedPrincipalTrade",
        ("FIX.5.0", "855", "40"): "40 .. BlockTrades",
        ("FIX.5.0", "855", "41"): "41 .. NameChange",
        ("FIX.5.0", "855", "42"): "42 .. PortfolioTransfer",
        ("FIX.5.0", "855", "43"): "43 .. ProrogationBuy",
        ("FIX.5.0", "855", "44"): "44 .. ProrogationSell",
        ("FIX.5.0", "855", "45"): "45 .. OptionExercise",
        ("FIX.5.0", "855", "46"): "46 .. DeltaNeutralTransaction",
        ("FIX.5.0", "855", "47"): "47 .. FinancingTransaction",
        ("FIX.5.0", "856", "0"): "0 .. Submit",
        ("FIX.5.0", "856", "1"): "1 .. Alleged",
        ("FIX.5.0", "856", "2"): "2 .. Accept",
        ("FIX.5.0", "856", "3"): "3 .. Decline",
        ("FIX.5.0", "856", "4"): "4 .. Addendum",
        ("FIX.5.0", "856", "5"): "5 .. No",
        ("FIX.5.0", "856", "6"): "6 .. TradeReportCancel",
        ("FIX.5.0", "856", "7"): "7 .. LockedIn",
        ("FIX.5.0", "856", "8"): "8 .. Defaulted",
        ("FIX.5.0", "856", "9"): "9 .. InvalidCMTA",
        ("FIX.5.0", "856", "10"): "10 .. Pended",
        ("FIX.5.0", "856", "11"): "11 .. AllegedNew",
        ("FIX.5.0", "856", "12"): "12 .. AllegedAddendum",
        ("FIX.5.0", "856", "13"): "13 .. AllegedNo",
        ("FIX.5.0", "856", "14"): "14 .. AllegedTradeReportCancel",
        ("FIX.5.0", "856", "15"): "15 .. AllegedTradeBreak",
        ("FIX.5.0", "856", "16"): "16 .. Verify",
        ("FIX.5.0", "856", "17"): "17 .. Dispute",
        ("FIX.5.0", "857", "0"): "0 .. NotSpecified",
        ("FIX.5.0", "857", "1"): "1 .. ExplicitListProvided",
        ("FIX.5.0", "865", "1"): "1 .. Put",
        ("FIX.5.0", "865", "2"): "2 .. Call",
        ("FIX.5.0", "865", "3"): "3 .. Tender",
        ("FIX.5.0", "865", "4"): "4 .. SinkingFundCall",
        ("FIX.5.0", "865", "5"): "5 .. Activation",
        ("FIX.5.0", "865", "6"): "6 .. Inactiviation",
        ("FIX.5.0", "865", "7"): "7 .. LastEligibleTradeDate",
        ("FIX.5.0", "865", "8"): "8 .. SwapStartDate",
        ("FIX.5.0", "865", "9"): "9 .. SwapEndDate",
        ("FIX.5.0", "865", "10"): "10 .. SwapRollDate",
        ("FIX.5.0", "865", "11"): "11 .. SwapNextStartDate",
        ("FIX.5.0", "865", "12"): "12 .. SwapNextRollDate",
        ("FIX.5.0", "865", "13"): "13 .. FirstDeliveryDate",
        ("FIX.5.0", "865", "14"): "14 .. LastDeliveryDate",
        ("FIX.5.0", "865", "15"): "15 .. InitialInventoryDueDate",
        ("FIX.5.0", "865", "16"): "16 .. FinalInventoryDueDate",
        ("FIX.5.0", "865", "17"): "17 .. FirstIntentDate",
        ("FIX.5.0", "865", "18"): "18 .. LastIntentDate",
        ("FIX.5.0", "865", "19"): "19 .. PositionRemovalDate",
        ("FIX.5.0", "865", "20"): "20 .. MinimumNotice",
        ("FIX.5.0", "865", "21"): "21 .. DeliveryStartTime",
        ("FIX.5.0", "865", "22"): "22 .. DeliveryEndTime",
        ("FIX.5.0", "865", "23"): "23 .. FirstNoticeDate",
        ("FIX.5.0", "865", "24"): "24 .. LastNoticeDate",
        ("FIX.5.0", "865", "25"): "25 .. FirstExerciseDate",
        ("FIX.5.0", "865", "26"): "26 .. RedemptionDate",
        ("FIX.5.0", "865", "27"): "27 .. TrdCntntnEfctvDt",
        ("FIX.5.0", "865", "99"): "99 .. Other",
        ("FIX.5.0", "871", "1"): "1 .. Flat",
        ("FIX.5.0", "871", "2"): "2 .. ZeroCoupon",
        ("FIX.5.0", "871", "3"): "3 .. InterestBearing",
        ("FIX.5.0", "871", "4"): "4 .. NoPeriodicPayments",
        ("FIX.5.0", "871", "5"): "5 .. VariableRate",
        ("FIX.5.0", "871", "6"): "6 .. LessFeeForPut",
        ("FIX.5.0", "871", "7"): "7 .. SteppedCoupon",
        ("FIX.5.0", "871", "8"): "8 .. CouponPeriod",
        ("FIX.5.0", "871", "9"): "9 .. When",
        ("FIX.5.0", "871", "10"): "10 .. OriginalIssueDiscount",
        ("FIX.5.0", "871", "11"): "11 .. Callable",
        ("FIX.5.0", "871", "12"): "12 .. EscrowedToMaturity",
        ("FIX.5.0", "871", "13"): "13 .. EscrowedToRedemptionDate",
        ("FIX.5.0", "871", "14"): "14 .. PreRefunded",
        ("FIX.5.0", "871", "15"): "15 .. InDefault",
        ("FIX.5.0", "871", "16"): "16 .. Unrated",
        ("FIX.5.0", "871", "17"): "17 .. Taxable",
        ("FIX.5.0", "871", "18"): "18 .. Indexed",
        ("FIX.5.0", "871", "19"): "19 .. SubjectToAlternativeMinimumTax",
        ("FIX.5.0", "871", "20"): "20 .. OriginalIssueDiscountPrice",
        ("FIX.5.0", "871", "21"): "21 .. CallableBelowMaturityValue",
        ("FIX.5.0", "871", "22"): "22 .. CallableWithoutNotice",
        ("FIX.5.0", "871", "23"): "23 .. PriceTickRulesForSecurity",
        ("FIX.5.0", "871", "24"): "24 .. TradeTypeEligibilityDetailsForSecurity",
        ("FIX.5.0", "871", "25"): "25 .. InstrumentDenominator",
        ("FIX.5.0", "871", "26"): "26 .. InstrumentNumerator",
        ("FIX.5.0", "871", "27"): "27 .. InstrumentPricePrecision",
        ("FIX.5.0", "871", "28"): "28 .. InstrumentStrikePrice",
        ("FIX.5.0", "871", "29"): "29 .. TradeableIndicator",
        ("FIX.5.0", "871", "30"): "30 .. InstrumentEligibleAnonOrders",
        ("FIX.5.0", "871", "31"): "31 .. MinGuaranteedFillVolume",
        ("FIX.5.0", "871", "32"): "32 .. MinGuaranteedFillStatus",
        ("FIX.5.0", "871", "33"): "33 .. TradeAtSettlementEligibility",
        ("FIX.5.0", "871", "34"): "34 .. TestInstrument",
        ("FIX.5.0", "871", "35"): "35 .. DummyInstrument",
        ("FIX.5.0", "871", "36"): "36 .. NegativeSettlementPriceEligibility",
        ("FIX.5.0", "871", "37"): "37 .. NegativeStrikePriceEligibility",
        ("FIX.5.0", "871", "38"): "38 .. USStdContractInd",
        ("FIX.5.0", "871", "39"): "39 .. AdmittedToTradingOnTradingVenue",
        ("FIX.5.0", "871", "40"): "40 .. AverageDailyNotionalAmount",
        ("FIX.5.0", "871", "41"): "41 .. AverageDailyNumberTrades",
        ("FIX.5.0", "871", "99"): "99 .. Text",
        ("FIX.5.0", "875", "1"): "1 .. Program3a3",
        ("FIX.5.0", "875", "2"): "2 .. Program42",
        ("FIX.5.0", "875", "3"): "3 .. Program3a2",
        ("FIX.5.0", "875", "4"): "4 .. Program3a3And3c7",
        ("FIX.5.0", "875", "5"): "5 .. Program3a4",
        ("FIX.5.0", "875", "6"): "6 .. Program3a5",
        ("FIX.5.0", "875", "7"): "7 .. Program3a7",
        ("FIX.5.0", "875", "8"): "8 .. Program3c7",
        ("FIX.5.0", "875", "99"): "99 .. Other",
        ("FIX.5.0", "877", "1"): "1 .. Program3a3",
        ("FIX.5.0", "877", "2"): "2 .. Program42",
        ("FIX.5.0", "877", "3"): "3 .. Program3a2",
        ("FIX.5.0", "877", "4"): "4 .. Program3a3And3c7",
        ("FIX.5.0", "877", "5"): "5 .. Program3a4",
        ("FIX.5.0", "877", "6"): "6 .. Program3a5",
        ("FIX.5.0", "877", "7"): "7 .. Program3a7",
        ("FIX.5.0", "877", "8"): "8 .. Program3c7",
        ("FIX.5.0", "877", "99"): "99 .. Other",
        ("FIX.5.0", "888", "AMT"): "AMT .. AlternativeMinimumTax",
        ("FIX.5.0", "888", "AUTOREINV"): "AUTOREINV .. AutoReinvestment",
        ("FIX.5.0", "888", "BANKQUAL"): "BANKQUAL .. BankQualified",
        ("FIX.5.0", "888", "BGNCON"): "BGNCON .. BargainConditions",
        ("FIX.5.0", "888", "COUPON"): "COUPON .. CouponRange",
        ("FIX.5.0", "888", "CURRENCY"): "CURRENCY .. ISOCurrencyCode",
        ("FIX.5.0", "888", "CUSTOMDATE"): "CUSTOMDATE .. CustomStart",
        ("FIX.5.0", "888", "GEOG"): "GEOG .. Geographics",
        ("FIX.5.0", "888", "HAIRCUT"): "HAIRCUT .. ValuationDiscount",
        ("FIX.5.0", "888", "INSURED"): "INSURED .. Insured",
        ("FIX.5.0", "888", "ISSUE"): "ISSUE .. IssueDate",
        ("FIX.5.0", "888", "ISSUER"): "ISSUER .. Issuer",
        ("FIX.5.0", "888", "ISSUESIZE"): "ISSUESIZE .. IssueSizeRange",
        ("FIX.5.0", "888", "LOOKBACK"): "LOOKBACK .. LookbackDays",
        ("FIX.5.0", "888", "LOT"): "LOT .. ExplicitLotIdentifier",
        ("FIX.5.0", "888", "LOTVAR"): "LOTVAR .. LotVariance",
        ("FIX.5.0", "888", "MAT"): "MAT .. MaturityYearAndMonth",
        ("FIX.5.0", "888", "MATURITY"): "MATURITY .. MaturityRange",
        ("FIX.5.0", "888", "MAXSUBS"): "MAXSUBS .. MaximumSubstitutions",
        ("FIX.5.0", "888", "MINDNOM"): "MINDNOM .. MinimumDenomination",
        ("FIX.5.0", "888", "MININCR"): "MININCR .. MinimumIncrement",
        ("FIX.5.0", "888", "MINQTY"): "MINQTY .. MinimumQuantity",
        ("FIX.5.0", "888", "PAYFREQ"): "PAYFREQ .. PaymentFrequency",
        ("FIX.5.0", "888", "PIECES"): "PIECES .. NumberOfPieces",
        ("FIX.5.0", "888", "PMAX"): "PMAX .. PoolsMaximum",
        ("FIX.5.0", "888", "PPL"): "PPL .. PoolsPerLot",
        ("FIX.5.0", "888", "PPM"): "PPM .. PoolsPerMillion",
        ("FIX.5.0", "888", "PPT"): "PPT .. PoolsPerTrade",
        ("FIX.5.0", "888", "PRICE"): "PRICE .. PriceRange",
        ("FIX.5.0", "888", "PRICEFREQ"): "PRICEFREQ .. PricingFrequency",
        ("FIX.5.0", "888", "PROD"): "PROD .. ProductionYear",
        ("FIX.5.0", "888", "PROTECT"): "PROTECT .. CallProtection",
        ("FIX.5.0", "888", "PURPOSE"): "PURPOSE .. Purpose",
        ("FIX.5.0", "888", "PXSOURCE"): "PXSOURCE .. BenchmarkPriceSource",
        ("FIX.5.0", "888", "RATING"): "RATING .. RatingSourceAndRange",
        ("FIX.5.0", "888", "REDEMPTION"): "REDEMPTION .. TypeOfRedemption",
        ("FIX.5.0", "888", "RESTRICTED"): "RESTRICTED .. Restricted",
        ("FIX.5.0", "888", "SECTOR"): "SECTOR .. MarketSector",
        ("FIX.5.0", "888", "SECTYPE"): "SECTYPE .. SecurityTypeIncludedOrExcluded",
        ("FIX.5.0", "888", "STRUCT"): "STRUCT .. Structure",
        ("FIX.5.0", "888", "SUBSFREQ"): "SUBSFREQ .. SubstitutionsFrequency",
        ("FIX.5.0", "888", "SUBSLEFT"): "SUBSLEFT .. SubstitutionsLeft",
        ("FIX.5.0", "888", "TEXT"): "TEXT .. FreeformText",
        ("FIX.5.0", "888", "TRDVAR"): "TRDVAR .. TradeVariance",
        ("FIX.5.0", "888", "WAC"): "WAC .. WeightedAverageCoupon",
        ("FIX.5.0", "888", "WAL"): "WAL .. WeightedAverageLifeCoupon",
        ("FIX.5.0", "888", "WALA"): "WALA .. WeightedAverageLoanAge",
        ("FIX.5.0", "888", "WAM"): "WAM .. WeightedAverageMaturity",
        ("FIX.5.0", "888", "WHOLE"): "WHOLE .. WholePool",
        ("FIX.5.0", "888", "YIELD"): "YIELD .. YieldRange",
        ("FIX.5.0", "888", "ORIGAMT"): "ORIGAMT .. OriginalAmount",
        ("FIX.5.0", "888", "POOLEFFDT"): "POOLEFFDT .. PoolEffectiveDate",
        ("FIX.5.0", "888", "POOLINITFCTR"): "POOLINITFCTR .. PoolInitialFactor",
        ("FIX.5.0", "888", "TRANCHE"): "TRANCHE .. Tranche",
        ("FIX.5.0", "888", "SUBSTITUTION"): "SUBSTITUTION .. Substitution",
        ("FIX.5.0", "888", "MULTEXCHFLLBCK"): "MULTEXCHFLLBCK .. MULTEXCHFLLBCK",
        ("FIX.5.0", "888", "COMPSECFLLBCK"): "COMPSECFLLBCK .. COMPSECFLLBCK",
        ("FIX.5.0", "888", "LOCLJRSDCTN"): "LOCLJRSDCTN .. LOCLJRSDCTN",
        ("FIX.5.0", "888", "RELVJRSDCTN"): "RELVJRSDCTN .. RELVJRSDCTN",
        ("FIX.5.0", "888", "INCURRCVY"): "INCURRCVY .. IncurredRecovery",
        ("FIX.5.0", "888", "ADDTRM"): "ADDTRM .. AdditionalTerm",
        ("FIX.5.0", "888", "MODEQTYDLVY"): "MODEQTYDLVY .. ModifiedEquityDelivery",
        ("FIX.5.0", "888", "NOREFOBLIG"): "NOREFOBLIG .. NoReferenceOblication",
        ("FIX.5.0", "888", "UNKREFOBLIG"): "UNKREFOBLIG .. UnknownReferenceObligation",
        ("FIX.5.0", "888", "ALLGUARANTEES"): "ALLGUARANTEES .. AllGuarantees",
        ("FIX.5.0", "888", "REFPX"): "REFPX .. ReferencePrice",
        ("FIX.5.0", "888", "REFPOLICY"): "REFPOLICY .. ReferencePolicy",
        ("FIX.5.0", "888", "SECRDLIST"): "SECRDLIST .. SecuredList",
        ("FIX.5.0", "888", "AVFICO"): "AVFICO .. AverageFICOScore",
        ("FIX.5.0", "888", "AVSIZE"): "AVSIZE .. AverageLoanSize",
        ("FIX.5.0", "888", "MAXBAL"): "MAXBAL .. MaximumLoanBalance",
        ("FIX.5.0", "888", "POOL"): "POOL .. PoolIdentifier",
        ("FIX.5.0", "888", "ROLLTYPE"): "ROLLTYPE .. TypeOfRollTrade",
        ("FIX.5.0", "888", "REFTRADE"): "REFTRADE .. ReferenceToRollingOrClosingTrade",
        ("FIX.5.0", "888", "REFPRIN"): "REFPRIN .. PrincipalOfRollingOrClosingTrade",
        ("FIX.5.0", "888", "REFINT"): "REFINT .. InterestOfRollingOrClosingTrade",
        ("FIX.5.0", "888", "AVAILQTY"): "AVAILQTY .. AvailableOfferQuantityToBeShownToTheStreet",
        ("FIX.5.0", "888", "BROKERCREDIT"): "BROKERCREDIT .. BrokerCredit",
        ("FIX.5.0", "888", "INTERNALPX"): "INTERNALPX .. OfferPriceToBeShownToInternalBrokers",
        ("FIX.5.0", "888", "INTERNALQTY"): "INTERNALQTY .. OfferQuantityToBeShownToInternalBrokers",
        ("FIX.5.0", "888", "LEAVEQTY"): "LEAVEQTY .. TheMinimumResidualOfferQuantity",
        ("FIX.5.0", "888", "MAXORDQTY"): "MAXORDQTY .. MaximumOrderSize",
        ("FIX.5.0", "888", "ORDRINCR"): "ORDRINCR .. OrderQuantityIncrement",
        ("FIX.5.0", "888", "PRIMARY"): "PRIMARY .. PrimaryOrSecondaryMarketIndicator",
        ("FIX.5.0", "888", "SALESCREDITOVR"): "SALESCREDITOVR .. BrokerSalesCreditOverride",
        ("FIX.5.0", "888", "TRADERCREDIT"): "TRADERCREDIT .. TraderCredit",
        ("FIX.5.0", "888", "DISCOUNT"): "DISCOUNT .. DiscountRate",
        ("FIX.5.0", "888", "YTM"): "YTM .. YieldToMaturity",
        ("FIX.5.0", "888", "ABS"): "ABS .. AbsolutePrepaymentSpeed",
        ("FIX.5.0", "888", "CPP"): "CPP .. ConstantPrepaymentPenalty",
        ("FIX.5.0", "888", "CPR"): "CPR .. ConstantPrepaymentRate",
        ("FIX.5.0", "888", "CPY"): "CPY .. ConstantPrepaymentYield",
        ("FIX.5.0", "888", "HEP"): "HEP .. FinalCPROfHomeEquityPrepaymentCurve",
        ("FIX.5.0", "888", "MHP"): "MHP .. PercentOfManufacturedHousingPrepaymentCurve",
        ("FIX.5.0", "888", "MPR"): "MPR .. MonthlyPrepaymentRate",
        ("FIX.5.0", "888", "PPC"): "PPC .. PercentOfProspectusPrepaymentCurve",
        ("FIX.5.0", "888", "PSA"): "PSA .. PercentOfBMAPrepaymentCurve",
        ("FIX.5.0", "888", "SMM"): "SMM .. SingleMonthlyMortality",
        ("FIX.5.0", "891", "0"): "0 .. Absolute",
        ("FIX.5.0", "891", "1"): "1 .. PerUnit",
        ("FIX.5.0", "891", "2"): "2 .. Percentage",
        ("FIX.5.0", "893", "N"): "N .. NotLastMessage",
        ("FIX.5.0", "893", "Y"): "Y .. LastMessage",
        ("FIX.5.0", "895", "0"): "0 .. Initial",
        ("FIX.5.0", "895", "1"): "1 .. Scheduled",
        ("FIX.5.0", "895", "2"): "2 .. TimeWarning",
        ("FIX.5.0", "895", "3"): "3 .. MarginDeficiency",
        ("FIX.5.0", "895", "4"): "4 .. MarginExcess",
        ("FIX.5.0", "895", "5"): "5 .. ForwardCollateralDemand",
        ("FIX.5.0", "895", "6"): "6 .. EventOfDefault",
        ("FIX.5.0", "895", "7"): "7 .. AdverseTaxEvent",
        ("FIX.5.0", "895", "8"): "8 .. TransferDeposit",
        ("FIX.5.0", "895", "9"): "9 .. TransferWithdrawal",
        ("FIX.5.0", "895", "10"): "10 .. Pledge",
        ("FIX.5.0", "896", "0"): "0 .. TradeDate",
        ("FIX.5.0", "896", "1"): "1 .. GCInstrument",
        ("FIX.5.0", "896", "2"): "2 .. CollateralInstrument",
        ("FIX.5.0", "896", "3"): "3 .. SubstitutionEligible",
        ("FIX.5.0", "896", "4"): "4 .. NotAssigned",
        ("FIX.5.0", "896", "5"): "5 .. PartiallyAssigned",
        ("FIX.5.0", "896", "6"): "6 .. FullyAssigned",
        ("FIX.5.0", "896", "7"): "7 .. OutstandingTrades",
        ("FIX.5.0", "903", "0"): "0 .. New",
        ("FIX.5.0", "903", "1"): "1 .. Replace",
        ("FIX.5.0", "903", "2"): "2 .. Cancel",
        ("FIX.5.0", "903", "3"): "3 .. Release",
        ("FIX.5.0", "903", "4"): "4 .. Reverse",
        ("FIX.5.0", "905", "0"): "0 .. Received",
        ("FIX.5.0", "905", "1"): "1 .. Accepted",
        ("FIX.5.0", "905", "2"): "2 .. Declined",
        ("FIX.5.0", "905", "3"): "3 .. Rejected",
        ("FIX.5.0", "905", "4"): "4 .. TransactionPending",
        ("FIX.5.0", "905", "5"): "5 .. TransactionCompletedWithWarning",
        ("FIX.5.0", "906", "0"): "0 .. UnknownDeal",
        ("FIX.5.0", "906", "1"): "1 .. UnknownOrInvalidInstrument",
        ("FIX.5.0", "906", "2"): "2 .. UnauthorizedTransaction",
        ("FIX.5.0", "906", "3"): "3 .. InsufficientCollateral",
        ("FIX.5.0", "906", "4"): "4 .. InvalidTypeOfCollateral",
        ("FIX.5.0", "906", "5"): "5 .. ExcessiveSubstitution",
        ("FIX.5.0", "906", "99"): "99 .. Other",
        ("FIX.5.0", "910", "0"): "0 .. Unassigned",
        ("FIX.5.0", "910", "1"): "1 .. PartiallyAssigned",
        ("FIX.5.0", "910", "2"): "2 .. AssignmentProposed",
        ("FIX.5.0", "910", "3"): "3 .. Assigned",
        ("FIX.5.0", "910", "4"): "4 .. Challenged",
        ("FIX.5.0", "912", "N"): "N .. NotLastMessage",
        ("FIX.5.0", "912", "Y"): "Y .. LastMessage",
        ("FIX.5.0", "919", "0"): "0 .. VersusPayment",
        ("FIX.5.0", "919", "1"): "1 .. Free",
        ("FIX.5.0", "919", "2"): "2 .. TriParty",
        ("FIX.5.0", "919", "3"): "3 .. HoldInCustody",
        ("FIX.5.0", "924", "1"): "1 .. LogOnUser",
        ("FIX.5.0", "924", "2"): "2 .. LogOffUser",
        ("FIX.5.0", "924", "3"): "3 .. ChangePasswordForUser",
        ("FIX.5.0", "924", "4"): "4 .. RequestIndividualUserStatus",
        ("FIX.5.0", "924", "5"): "5 .. RequestThrottleLimit",
        ("FIX.5.0", "926", "1"): "1 .. LoggedIn",
        ("FIX.5.0", "926", "2"): "2 .. NotLoggedIn",
        ("FIX.5.0", "926", "3"): "3 .. UserNotRecognised",
        ("FIX.5.0", "926", "4"): "4 .. PasswordIncorrect",
        ("FIX.5.0", "926", "5"): "5 .. PasswordChanged",
        ("FIX.5.0", "926", "6"): "6 .. Other",
        ("FIX.5.0", "926", "7"): "7 .. ForcedUserLogoutByExchange",
        ("FIX.5.0", "926", "8"): "8 .. SessionShutdownWarning",
        ("FIX.5.0", "926", "9"): "9 .. ThrottleParametersChanged",
        ("FIX.5.0", "928", "1"): "1 .. Connected",
        ("FIX.5.0", "928", "2"): "2 .. NotConnectedUnexpected",
        ("FIX.5.0", "928", "3"): "3 .. NotConnectedExpected",
        ("FIX.5.0", "928", "4"): "4 .. InProcess",
        ("FIX.5.0", "935", "1"): "1 .. Snapshot",
        ("FIX.5.0", "935", "2"): "2 .. Subscribe",
        ("FIX.5.0", "935", "4"): "4 .. StopSubscribing",
        ("FIX.5.0", "935", "8"): "8 .. LevelOfDetail",
        ("FIX.5.0", "937", "1"): "1 .. Full",
        ("FIX.5.0", "937", "2"): "2 .. IncrementalUpdate",
        ("FIX.5.0", "939", "0"): "0 .. Accepted",
        ("FIX.5.0", "939", "1"): "1 .. Rejected",
        ("FIX.5.0", "939", "2"): "2 .. Cancelled",
        ("FIX.5.0", "939", "3"): "3 .. AcceptedWithErrors",
        ("FIX.5.0", "939", "4"): "4 .. PendingNew",
        ("FIX.5.0", "939", "5"): "5 .. PendingCancel",
        ("FIX.5.0", "939", "6"): "6 .. PendingReplace",
        ("FIX.5.0", "939", "7"): "7 .. Terminated",
        ("FIX.5.0", "939", "8"): "8 .. PendingVerification",
        ("FIX.5.0", "939", "9"): "9 .. DeemedVerified",
        ("FIX.5.0", "939", "10"): "10 .. Verified",
        ("FIX.5.0", "939", "11"): "11 .. Disputed",
        ("FIX.5.0", "940", "1"): "1 .. Received",
        ("FIX.5.0", "940", "2"): "2 .. ConfirmRejected",
        ("FIX.5.0", "940", "3"): "3 .. Affirmed",
        ("FIX.5.0", "944", "0"): "0 .. Retain",
        ("FIX.5.0", "944", "1"): "1 .. Add",
        ("FIX.5.0", "944", "2"): "2 .. Remove",
        ("FIX.5.0", "945", "0"): "0 .. Accepted",
        ("FIX.5.0", "945", "1"): "1 .. AcceptedWithWarnings",
        ("FIX.5.0", "945", "2"): "2 .. Completed",
        ("FIX.5.0", "945", "3"): "3 .. CompletedWithWarnings",
        ("FIX.5.0", "945", "4"): "4 .. Rejected",
        ("FIX.5.0", "946", "0"): "0 .. Successful",
        ("FIX.5.0", "946", "1"): "1 .. InvalidOrUnknownInstrument",
        ("FIX.5.0", "946", "2"): "2 .. InvalidOrUnknownCollateralType",
        ("FIX.5.0", "946", "3"): "3 .. InvalidParties",
        ("FIX.5.0", "946", "4"): "4 .. InvalidTransportTypeRequested",
        ("FIX.5.0", "946", "5"): "5 .. InvalidDestinationRequested",
        ("FIX.5.0", "946", "6"): "6 .. NoCollateralFoundForTheTradeSpecified",
        ("FIX.5.0", "946", "7"): "7 .. NoCollateralFoundForTheOrderSpecified",
        ("FIX.5.0", "946", "8"): "8 .. CollateralInquiryTypeNotSupported",
        ("FIX.5.0", "946", "9"): "9 .. UnauthorizedForCollateralInquiry",
        ("FIX.5.0", "946", "99"): "99 .. Other",
        ("FIX.5.0", "950", "6"): "6 .. UKNationalInsuranceOrPensionNumber",
        ("FIX.5.0", "950", "7"): "7 .. USSocialSecurityNumber",
        ("FIX.5.0", "950", "8"): "8 .. USEmployerOrTaxIDNumber",
        ("FIX.5.0", "950", "9"): "9 .. AustralianBusinessNumber",
        ("FIX.5.0", "950", "A"): "A .. AustralianTaxFileNumber",
        ("FIX.5.0", "950", "J"): "J .. TaxID",
        ("FIX.5.0", "950", "1"): "1 .. KoreanInvestorID",
        ("FIX.5.0", "950", "2"): "2 .. TaiwaneseForeignInvestorID",
        ("FIX.5.0", "950", "3"): "3 .. TaiwaneseTradingAcct",
        ("FIX.5.0", "950", "4"): "4 .. MalaysianCentralDepository",
        ("FIX.5.0", "950", "5"): "5 .. ChineseInvestorID",
        ("FIX.5.0", "950", "I"): "I .. ISITCAcronym",
        ("FIX.5.0", "950", "B"): "B .. BIC",
        ("FIX.5.0", "950", "C"): "C .. GeneralIdentifier",
        ("FIX.5.0", "950", "D"): "D .. Proprietary",
        ("FIX.5.0", "950", "E"): "E .. ISOCountryCode",
        ("FIX.5.0", "950", "F"): "F .. SettlementEntityLocation",
        ("FIX.5.0", "950", "G"): "G .. MIC",
        ("FIX.5.0", "950", "H"): "H .. CSDParticipant",
        ("FIX.5.0", "950", "K"): "K .. AustralianCompanyNumber",
        ("FIX.5.0", "950", "L"): "L .. AustralianRegisteredBodyNumber",
        ("FIX.5.0", "950", "M"): "M .. CFTCReportingFirmIdentifier",
        ("FIX.5.0", "950", "N"): "N .. LegalEntityIdentifier",
        ("FIX.5.0", "950", "O"): "O .. InterimIdentifier",
        ("FIX.5.0", "950", "P"): "P .. ShortCodeIdentifier",
        ("FIX.5.0", "950", "Q"): "Q .. NationalIDNaturalPerson",
        ("FIX.5.0", "951", "1"): "1 .. ExecutingFirm",
        ("FIX.5.0", "951", "2"): "2 .. BrokerOfCredit",
        ("FIX.5.0", "951", "3"): "3 .. ClientID",
        ("FIX.5.0", "951", "4"): "4 .. ClearingFirm",
        ("FIX.5.0", "951", "5"): "5 .. InvestorID",
        ("FIX.5.0", "951", "6"): "6 .. IntroducingFirm",
        ("FIX.5.0", "951", "7"): "7 .. EnteringFirm",
        ("FIX.5.0", "951", "8"): "8 .. Locate",
        ("FIX.5.0", "951", "9"): "9 .. FundManagerClientID",
        ("FIX.5.0", "951", "10"): "10 .. SettlementLocation",
        ("FIX.5.0", "951", "11"): "11 .. OrderOriginationTrader",
        ("FIX.5.0", "951", "12"): "12 .. ExecutingTrader",
        ("FIX.5.0", "951", "13"): "13 .. OrderOriginationFirm",
        ("FIX.5.0", "951", "15"): "15 .. CorrespondantClearingFirm",
        ("FIX.5.0", "951", "16"): "16 .. ExecutingSystem",
        ("FIX.5.0", "951", "17"): "17 .. ContraFirm",
        ("FIX.5.0", "951", "18"): "18 .. ContraClearingFirm",
        ("FIX.5.0", "951", "19"): "19 .. SponsoringFirm",
        ("FIX.5.0", "951", "20"): "20 .. UnderlyingContraFirm",
        ("FIX.5.0", "951", "21"): "21 .. ClearingOrganization",
        ("FIX.5.0", "951", "22"): "22 .. Exchange",
        ("FIX.5.0", "951", "24"): "24 .. CustomerAccount",
        ("FIX.5.0", "951", "25"): "25 .. CorrespondentClearingOrganization",
        ("FIX.5.0", "951", "26"): "26 .. CorrespondentBroker",
        ("FIX.5.0", "951", "27"): "27 .. Buyer",
        ("FIX.5.0", "951", "28"): "28 .. Custodian",
        ("FIX.5.0", "951", "29"): "29 .. Intermediary",
        ("FIX.5.0", "951", "30"): "30 .. Agent",
        ("FIX.5.0", "951", "31"): "31 .. SubCustodian",
        ("FIX.5.0", "951", "32"): "32 .. Beneficiary",
        ("FIX.5.0", "951", "33"): "33 .. InterestedParty",
        ("FIX.5.0", "951", "34"): "34 .. RegulatoryBody",
        ("FIX.5.0", "951", "35"): "35 .. LiquidityProvider",
        ("FIX.5.0", "951", "36"): "36 .. EnteringTrader",
        ("FIX.5.0", "951", "37"): "37 .. ContraTrader",
        ("FIX.5.0", "951", "38"): "38 .. PositionAccount",
        ("FIX.5.0", "951", "39"): "39 .. ContraInvestorID",
        ("FIX.5.0", "951", "40"): "40 .. TransferToFirm",
        ("FIX.5.0", "951", "41"): "41 .. ContraPositionAccount",
        ("FIX.5.0", "951", "42"): "42 .. ContraExchange",
        ("FIX.5.0", "951", "43"): "43 .. InternalCarryAccount",
        ("FIX.5.0", "951", "44"): "44 .. OrderEntryOperatorID",
        ("FIX.5.0", "951", "45"): "45 .. SecondaryAccountNumber",
        ("FIX.5.0", "951", "46"): "46 .. ForeignFirm",
        ("FIX.5.0", "951", "47"): "47 .. ThirdPartyAllocationFirm",
        ("FIX.5.0", "951", "48"): "48 .. ClaimingAccount",
        ("FIX.5.0", "951", "49"): "49 .. AssetManager",
        ("FIX.5.0", "951", "50"): "50 .. PledgorAccount",
        ("FIX.5.0", "951", "51"): "51 .. PledgeeAccount",
        ("FIX.5.0", "951", "52"): "52 .. LargeTraderReportableAccount",
        ("FIX.5.0", "951", "53"): "53 .. TraderMnemonic",
        ("FIX.5.0", "951", "54"): "54 .. SenderLocation",
        ("FIX.5.0", "951", "55"): "55 .. SessionID",
        ("FIX.5.0", "951", "56"): "56 .. AcceptableCounterparty",
        ("FIX.5.0", "951", "57"): "57 .. UnacceptableCounterparty",
        ("FIX.5.0", "951", "58"): "58 .. EnteringUnit",
        ("FIX.5.0", "951", "59"): "59 .. ExecutingUnit",
        ("FIX.5.0", "951", "60"): "60 .. IntroducingBroker",
        ("FIX.5.0", "951", "61"): "61 .. QuoteOriginator",
        ("FIX.5.0", "951", "62"): "62 .. ReportOriginator",
        ("FIX.5.0", "951", "63"): "63 .. SystematicInternaliser",
        ("FIX.5.0", "951", "64"): "64 .. MultilateralTradingFacility",
        ("FIX.5.0", "951", "65"): "65 .. RegulatedMarket",
        ("FIX.5.0", "951", "66"): "66 .. MarketMaker",
        ("FIX.5.0", "951", "67"): "67 .. InvestmentFirm",
        ("FIX.5.0", "951", "68"): "68 .. HostCompetentAuthority",
        ("FIX.5.0", "951", "69"): "69 .. HomeCompetentAuthority",
        ("FIX.5.0", "951", "70"): "70 .. CompetentAuthorityLiquidity",
        ("FIX.5.0", "951", "71"): "71 .. CompetentAuthorityTransactionVenue",
        ("FIX.5.0", "951", "72"): "72 .. ReportingIntermediary",
        ("FIX.5.0", "951", "73"): "73 .. ExecutionVenue",
        ("FIX.5.0", "951", "74"): "74 .. MarketDataEntryOriginator",
        ("FIX.5.0", "951", "75"): "75 .. LocationID",
        ("FIX.5.0", "951", "76"): "76 .. DeskID",
        ("FIX.5.0", "951", "77"): "77 .. MarketDataMarket",
        ("FIX.5.0", "951", "78"): "78 .. AllocationEntity",
        ("FIX.5.0", "951", "79"): "79 .. PrimeBroker",
        ("FIX.5.0", "951", "80"): "80 .. StepOutFirm",
        ("FIX.5.0", "951", "81"): "81 .. BrokerClearingID",
        ("FIX.5.0", "951", "82"): "82 .. CentralRegistrationDepository",
        ("FIX.5.0", "951", "83"): "83 .. ClearingAccount",
        ("FIX.5.0", "951", "84"): "84 .. AcceptableSettlingCounterparty",
        ("FIX.5.0", "951", "85"): "85 .. UnacceptableSettlingCounterparty",
        ("FIX.5.0", "951", "86"): "86 .. CLSMemberBank",
        ("FIX.5.0", "951", "87"): "87 .. InConcertGroup",
        ("FIX.5.0", "951", "88"): "88 .. InConcertControllingEntity",
        ("FIX.5.0", "951", "89"): "89 .. LargePositionsReportingAccount",
        ("FIX.5.0", "951", "90"): "90 .. SettlementFirm",
        ("FIX.5.0", "951", "91"): "91 .. SettlementAccount",
        ("FIX.5.0", "951", "92"): "92 .. ReportingMarketCenter",
        ("FIX.5.0", "951", "93"): "93 .. RelatedReportingMarketCenter",
        ("FIX.5.0", "951", "94"): "94 .. AwayMarket",
        ("FIX.5.0", "951", "95"): "95 .. GiveupTradingFirm",
        ("FIX.5.0", "951", "96"): "96 .. TakeupTradingFirm",
        ("FIX.5.0", "951", "97"): "97 .. GiveupClearingFirm",
        ("FIX.5.0", "951", "98"): "98 .. TakeupClearingFirm",
        ("FIX.5.0", "951", "99"): "99 .. OriginatingMarket",
        ("FIX.5.0", "951", "100"): "100 .. MarginAccount",
        ("FIX.5.0", "951", "101"): "101 .. CollateralAssetAccount",
        ("FIX.5.0", "951", "102"): "102 .. DataRepository",
        ("FIX.5.0", "951", "103"): "103 .. CalculationAgent",
        ("FIX.5.0", "951", "104"): "104 .. ExerciseNoticeSender",
        ("FIX.5.0", "951", "105"): "105 .. ExerciseNoticeReceiver",
        ("FIX.5.0", "951", "106"): "106 .. RateReferenceBank",
        ("FIX.5.0", "951", "107"): "107 .. Correspondent",
        ("FIX.5.0", "951", "109"): "109 .. BeneficiaryBank",
        ("FIX.5.0", "951", "110"): "110 .. Borrower",
        ("FIX.5.0", "951", "111"): "111 .. PrimaryObligator",
        ("FIX.5.0", "951", "112"): "112 .. Guarantor",
        ("FIX.5.0", "951", "113"): "113 .. ExcludedReferenceEntity",
        ("FIX.5.0", "951", "114"): "114 .. DeterminingParty",
        ("FIX.5.0", "951", "115"): "115 .. HedgingParty",
        ("FIX.5.0", "951", "116"): "116 .. ReportingEntity",
        ("FIX.5.0", "951", "117"): "117 .. SalesPerson",
        ("FIX.5.0", "951", "118"): "118 .. Operator",
        ("FIX.5.0", "951", "119"): "119 .. CSD",
        ("FIX.5.0", "951", "120"): "120 .. ICSD",
        ("FIX.5.0", "951", "121"): "121 .. TradingSubAccount",
        ("FIX.5.0", "951", "122"): "122 .. InvestmentDecisionMaker",
        ("FIX.5.0", "951", "123"): "123 .. PublishingIntermediary",
        ("FIX.5.0", "954", "1"): "1 .. Firm",
        ("FIX.5.0", "954", "2"): "2 .. Person",
        ("FIX.5.0", "954", "3"): "3 .. System",
        ("FIX.5.0", "954", "4"): "4 .. Application",
        ("FIX.5.0", "954", "5"): "5 .. FullLegalNameOfFirm",
        ("FIX.5.0", "954", "6"): "6 .. PostalAddress",
        ("FIX.5.0", "954", "7"): "7 .. PhoneNumber",
        ("FIX.5.0", "954", "8"): "8 .. EmailAddress",
        ("FIX.5.0", "954", "9"): "9 .. ContactName",
        ("FIX.5.0", "954", "10"): "10 .. SecuritiesAccountNumber",
        ("FIX.5.0", "954", "11"): "11 .. RegistrationNumber",
        ("FIX.5.0", "954", "12"): "12 .. RegisteredAddressForConfirmation",
        ("FIX.5.0", "954", "13"): "13 .. RegulatoryStatus",
        ("FIX.5.0", "954", "14"): "14 .. RegistrationName",
        ("FIX.5.0", "954", "15"): "15 .. CashAccountNumber",
        ("FIX.5.0", "954", "16"): "16 .. BIC",
        ("FIX.5.0", "954", "17"): "17 .. CSDParticipantMemberCode",
        ("FIX.5.0", "954", "18"): "18 .. RegisteredAddress",
        ("FIX.5.0", "954", "19"): "19 .. FundAccountName",
        ("FIX.5.0", "954", "20"): "20 .. TelexNumber",
        ("FIX.5.0", "954", "21"): "21 .. FaxNumber",
        ("FIX.5.0", "954", "22"): "22 .. SecuritiesAccountName",
        ("FIX.5.0", "954", "23"): "23 .. CashAccountName",
        ("FIX.5.0", "954", "24"): "24 .. Department",
        ("FIX.5.0", "954", "25"): "25 .. LocationDesk",
        ("FIX.5.0", "954", "26"): "26 .. PositionAccountType",
        ("FIX.5.0", "954", "27"): "27 .. SecurityLocateID",
        ("FIX.5.0", "954", "28"): "28 .. MarketMaker",
        ("FIX.5.0", "954", "29"): "29 .. EligibleCounterparty",
        ("FIX.5.0", "954", "30"): "30 .. ProfessionalClient",
        ("FIX.5.0", "954", "31"): "31 .. Location",
        ("FIX.5.0", "954", "32"): "32 .. ExecutionVenue",
        ("FIX.5.0", "954", "33"): "33 .. CurrencyDeliveryIdentifier",
        ("FIX.5.0", "954", "34"): "34 .. AddressCity",
        ("FIX.5.0", "954", "35"): "35 .. AddressStateOrProvince",
        ("FIX.5.0", "954", "36"): "36 .. AddressPostalCode",
        ("FIX.5.0", "954", "37"): "37 .. AddressStreet",
        ("FIX.5.0", "954", "38"): "38 .. AddressISOCountryCode",
        ("FIX.5.0", "954", "39"): "39 .. ISOCountryCode",
        ("FIX.5.0", "954", "40"): "40 .. MarketSegment",
        ("FIX.5.0", "954", "41"): "41 .. CustomerAccountType",
        ("FIX.5.0", "954", "42"): "42 .. OmnibusAccount",
        ("FIX.5.0", "954", "43"): "43 .. FundsSegregationType",
        ("FIX.5.0", "954", "44"): "44 .. GuaranteeFund",
        ("FIX.5.0", "954", "45"): "45 .. SwapDealer",
        ("FIX.5.0", "954", "46"): "46 .. MajorParticipant",
        ("FIX.5.0", "954", "47"): "47 .. FinancialEntity",
        ("FIX.5.0", "954", "48"): "48 .. USPerson",
        ("FIX.5.0", "954", "49"): "49 .. ReportingEntityIndicator",
        ("FIX.5.0", "954", "50"): "50 .. ElectedClearingRequirementException",
        ("FIX.5.0", "954", "51"): "51 .. BusinessCenter",
        ("FIX.5.0", "954", "52"): "52 .. ReferenceText",
        ("FIX.5.0", "954", "53"): "53 .. ShortMarkingExemptAccount",
        ("FIX.5.0", "954", "54"): "54 .. ParentFirmIdentifier",
        ("FIX.5.0", "954", "55"): "55 .. ParentFirmName",
        ("FIX.5.0", "954", "56"): "56 .. DealIdentifier",
        ("FIX.5.0", "954", "57"): "57 .. SystemTradeID",
        ("FIX.5.0", "954", "58"): "58 .. SystemTradeSubID",
        ("FIX.5.0", "954", "59"): "59 .. FCMCode",
        ("FIX.5.0", "954", "60"): "60 .. DlvryTrmlCode",
        ("FIX.5.0", "954", "61"): "61 .. VolntyRptEntity",
        ("FIX.5.0", "954", "62"): "62 .. RptObligJursdctn",
        ("FIX.5.0", "954", "63"): "63 .. VolntyRptJursdctn",
        ("FIX.5.0", "954", "64"): "64 .. CompanyActivities",
        ("FIX.5.0", "954", "65"): "65 .. EEAreaDomiciled",
        ("FIX.5.0", "954", "66"): "66 .. ContractLinked",
        ("FIX.5.0", "954", "67"): "67 .. ContractAbove",
        ("FIX.5.0", "954", "68"): "68 .. VolntyRptPty",
        ("FIX.5.0", "954", "69"): "69 .. EndUser",
        ("FIX.5.0", "954", "70"): "70 .. LocationOrJurisdiction",
        ("FIX.5.0", "954", "71"): "71 .. DerivativesDealer",
        ("FIX.5.0", "954", "72"): "72 .. Domicile",
        ("FIX.5.0", "954", "73"): "73 .. ExemptFromRecognition",
        ("FIX.5.0", "954", "74"): "74 .. Payer",
        ("FIX.5.0", "954", "75"): "75 .. Receiver",
        ("FIX.5.0", "954", "76"): "76 .. SystematicInternaliser",
        ("FIX.5.0", "954", "77"): "77 .. PublishingEntityIndicator",
        ("FIX.5.0", "954", "78"): "78 .. FirstName",
        ("FIX.5.0", "954", "79"): "79 .. Surname",
        ("FIX.5.0", "954", "80"): "80 .. DateOfBirth",
        ("FIX.5.0", "954", "81"): "81 .. OrderTransmittingFirm",
        ("FIX.5.0", "954", "82"): "82 .. OrderTransmittingFirmBuyer",
        ("FIX.5.0", "954", "83"): "83 .. OrderTransmitterSeller",
        ("FIX.5.0", "954", "84"): "84 .. LegalEntityIdentifier",
        ("FIX.5.0", "959", "1"): "1 .. Int",
        ("FIX.5.0", "959", "2"): "2 .. Length",
        ("FIX.5.0", "959", "3"): "3 .. NumInGroup",
        ("FIX.5.0", "959", "4"): "4 .. SeqNum",
        ("FIX.5.0", "959", "5"): "5 .. TagNum",
        ("FIX.5.0", "959", "6"): "6 .. Float",
        ("FIX.5.0", "959", "7"): "7 .. Qty",
        ("FIX.5.0", "959", "8"): "8 .. Price",
        ("FIX.5.0", "959", "9"): "9 .. PriceOffset",
        ("FIX.5.0", "959", "10"): "10 .. Amt",
        ("FIX.5.0", "959", "11"): "11 .. Percentage",
        ("FIX.5.0", "959", "12"): "12 .. Char",
        ("FIX.5.0", "959", "13"): "13 .. Boolean",
        ("FIX.5.0", "959", "14"): "14 .. String",
        ("FIX.5.0", "959", "15"): "15 .. MultipleCharValue",
        ("FIX.5.0", "959", "16"): "16 .. Currency",
        ("FIX.5.0", "959", "17"): "17 .. Exchange",
        ("FIX.5.0", "959", "18"): "18 .. MonthYear",
        ("FIX.5.0", "959", "19"): "19 .. UTCTimestamp",
        ("FIX.5.0", "959", "20"): "20 .. UTCTimeOnly",
        ("FIX.5.0", "959", "21"): "21 .. LocalMktDate",
        ("FIX.5.0", "959", "22"): "22 .. UTCDateOnly",
        ("FIX.5.0", "959", "23"): "23 .. Data",
        ("FIX.5.0", "959", "24"): "24 .. MultipleStringValue",
        ("FIX.5.0", "959", "25"): "25 .. Country",
        ("FIX.5.0", "959", "26"): "26 .. Language",
        ("FIX.5.0", "959", "27"): "27 .. TZTimeOnly",
        ("FIX.5.0", "959", "28"): "28 .. TZTimestamp",
        ("FIX.5.0", "959", "29"): "29 .. Tenor",
        ("FIX.5.0", "965", "1"): "1 .. Active",
        ("FIX.5.0", "965", "2"): "2 .. Inactive",
        ("FIX.5.0", "965", "3"): "3 .. ActiveClosingOrdersOnly",
        ("FIX.5.0", "965", "4"): "4 .. Expired",
        ("FIX.5.0", "965", "5"): "5 .. Delisted",
        ("FIX.5.0", "965", "6"): "6 .. KnockedOut",
        ("FIX.5.0", "965", "7"): "7 .. KnockOutRevoked",
        ("FIX.5.0", "965", "8"): "8 .. PendingExpiry",
        ("FIX.5.0", "965", "9"): "9 .. Suspended",
        ("FIX.5.0", "965", "10"): "10 .. Published",
        ("FIX.5.0", "965", "11"): "11 .. PendingDeletion",
        ("FIX.5.0", "974", "FIXED"): "FIXED .. FIXED",
        ("FIX.5.0", "974", "DIFF"): "DIFF .. DIFF",
        ("FIX.5.0", "975", "2"): "2 .. TPlus1",
        ("FIX.5.0", "975", "4"): "4 .. TPlus3",
        ("FIX.5.0", "975", "5"): "5 .. TPlus4",
        ("FIX.5.0", "980", "A"): "A .. Add",
        ("FIX.5.0", "980", "D"): "D .. Delete",
        ("FIX.5.0", "980", "M"): "M .. Modify",
        ("FIX.5.0", "982", "1"): "1 .. AutoExercise",
        ("FIX.5.0", "982", "2"): "2 .. NonAutoExercise",
        ("FIX.5.0", "982", "3"): "3 .. FinalWillBeExercised",
        ("FIX.5.0", "982", "4"): "4 .. ContraryIntention",
        ("FIX.5.0", "982", "5"): "5 .. Difference",
        ("FIX.5.0", "992", "1"): "1 .. SubAllocate",
        ("FIX.5.0", "992", "2"): "2 .. ThirdPartyAllocation",
        ("FIX.5.0", "996", "Bcf"): "Bcf .. BillionCubicFeet",
        ("FIX.5.0", "996", "CBM"): "CBM .. CubicMeters",
        ("FIX.5.0", "996", "GJ"): "GJ .. gigajoules",
        ("FIX.5.0", "996", "kWh"): "kWh .. KilowattHours",
        ("FIX.5.0", "996", "MMBtu"): "MMBtu .. OneMillionBTU",
        ("FIX.5.0", "996", "MWh"): "MWh .. MegawattHours",
        ("FIX.5.0", "996", "thm"): "thm .. therms",
        ("FIX.5.0", "996", "Alw"): "Alw .. Allowances",
        ("FIX.5.0", "996", "Bbl"): "Bbl .. Barrels",
        ("FIX.5.0", "996", "BDFT"): "BDFT .. BoardFeet",
        ("FIX.5.0", "996", "Bu"): "Bu .. Bushels",
        ("FIX.5.0", "996", "Ccy"): "Ccy .. Currency",
        ("FIX.5.0", "996", "CDD"): "CDD .. CoolingDegreeDay",
        ("FIX.5.0", "996", "CER"): "CER .. CertifiedEmissionsReduction",
        ("FIX.5.0", "996", "CPD"): "CPD .. CriticalPrecipDay",
        ("FIX.5.0", "996", "CRT"): "CRT .. ClimateReserveTonnes",
        ("FIX.5.0", "996", "cwt"): "cwt .. Hundredweight",
        ("FIX.5.0", "996", "day"): "day .. Day",
        ("FIX.5.0", "996", "dt"): "dt .. DryMetricTons",
        ("FIX.5.0", "996", "EnvAllwnc"): "EnvAllwnc .. EnvAllwncCert",
        ("FIX.5.0", "996", "EnvCrd"): "EnvCrd .. EnvironmentalCredit",
        ("FIX.5.0", "996", "EnvOfst"): "EnvOfst .. EnvironmentalOffset",
        ("FIX.5.0", "996", "g"): "g .. Grams",
        ("FIX.5.0", "996", "Gal"): "Gal .. Gallons",
        ("FIX.5.0", "996", "GT"): "GT .. GrossTons",
        ("FIX.5.0", "996", "HDD"): "HDD .. HeatingDegreeDay",
        ("FIX.5.0", "996", "IPNT"): "IPNT .. IndexPoint",
        ("FIX.5.0", "996", "kg"): "kg .. Kilograms",
        ("FIX.5.0", "996", "kL"): "kL .. kiloliters",
        ("FIX.5.0", "996", "kW-a"): "kW-a .. KilowattYear",
        ("FIX.5.0", "996", "kW-d"): "kW-d .. KilowattDay",
        ("FIX.5.0", "996", "kW-h"): "kW-h .. KilowattHour",
        ("FIX.5.0", "996", "kW-M"): "kW-M .. KilowattMonth",
        ("FIX.5.0", "996", "kW-min "): "kW-min  .. KilowattMinute",
        ("FIX.5.0", "996", "L"): "L .. liters",
        ("FIX.5.0", "996", "lbs"): "lbs .. Pounds",
        ("FIX.5.0", "996", "MW-a"): "MW-a .. MegawattYear",
        ("FIX.5.0", "996", "MW-d"): "MW-d .. MegawattDay",
        ("FIX.5.0", "996", "MW-h"): "MW-h .. MegawattHour",
        ("FIX.5.0", "996", "MW-M"): "MW-M .. MegawattMonth",
        ("FIX.5.0", "996", "MW-min"): "MW-min .. MegawattMinute",
        ("FIX.5.0", "996", "oz_tr"): "oz_tr .. TroyOunces",
        ("FIX.5.0", "996", "PRINC"): "PRINC .. PrincipalWithRelationToDebtInstrument",
        ("FIX.5.0", "996", "t"): "t .. MetricTons",
        ("FIX.5.0", "996", "tn"): "tn .. Tons",
        ("FIX.5.0", "997", "H"): "H .. Hour",
        ("FIX.5.0", "997", "Min"): "Min .. Minute",
        ("FIX.5.0", "997", "S"): "S .. Second",
        ("FIX.5.0", "997", "D"): "D .. Day",
        ("FIX.5.0", "997", "Wk"): "Wk .. Week",
        ("FIX.5.0", "997", "Mo"): "Mo .. Month",
        ("FIX.5.0", "997", "Yr"): "Yr .. Year",
        ("FIX.5.0", "997", "Q"): "Q .. Quarter",
        ("FIX.5.0", "998", "Bcf"): "Bcf .. BillionCubicFeet",
        ("FIX.5.0", "998", "CBM"): "CBM .. CubicMeters",
        ("FIX.5.0", "998", "GJ"): "GJ .. gigajoules",
        ("FIX.5.0", "998", "kWh"): "kWh .. KilowattHours",
        ("FIX.5.0", "998", "MMBtu"): "MMBtu .. OneMillionBTU",
        ("FIX.5.0", "998", "MWh"): "MWh .. MegawattHours",
        ("FIX.5.0", "998", "thm"): "thm .. therms",
        ("FIX.5.0", "998", "Alw"): "Alw .. Allowances",
        ("FIX.5.0", "998", "Bbl"): "Bbl .. Barrels",
        ("FIX.5.0", "998", "BDFT"): "BDFT .. BoardFeet",
        ("FIX.5.0", "998", "Bu"): "Bu .. Bushels",
        ("FIX.5.0", "998", "Ccy"): "Ccy .. Currency",
        ("FIX.5.0", "998", "CDD"): "CDD .. CoolingDegreeDay",
        ("FIX.5.0", "998", "CER"): "CER .. CertifiedEmissionsReduction",
        ("FIX.5.0", "998", "CPD"): "CPD .. CriticalPrecipDay",
        ("FIX.5.0", "998", "CRT"): "CRT .. ClimateReserveTonnes",
        ("FIX.5.0", "998", "cwt"): "cwt .. Hundredweight",
        ("FIX.5.0", "998", "day"): "day .. Day",
        ("FIX.5.0", "998", "dt"): "dt .. DryMetricTons",
        ("FIX.5.0", "998", "EnvAllwnc"): "EnvAllwnc .. EnvAllwncCert",
        ("FIX.5.0", "998", "EnvCrd"): "EnvCrd .. EnvironmentalCredit",
        ("FIX.5.0", "998", "EnvOfst"): "EnvOfst .. EnvironmentalOffset",
        ("FIX.5.0", "998", "g"): "g .. Grams",
        ("FIX.5.0", "998", "Gal"): "Gal .. Gallons",
        ("FIX.5.0", "998", "GT"): "GT .. GrossTons",
        ("FIX.5.0", "998", "HDD"): "HDD .. HeatingDegreeDay",
        ("FIX.5.0", "998", "IPNT"): "IPNT .. IndexPoint",
        ("FIX.5.0", "998", "kg"): "kg .. Kilograms",
        ("FIX.5.0", "998", "kL"): "kL .. kiloliters",
        ("FIX.5.0", "998", "kW-a"): "kW-a .. KilowattYear",
        ("FIX.5.0", "998", "kW-d"): "kW-d .. KilowattDay",
        ("FIX.5.0", "998", "kW-h"): "kW-h .. KilowattHour",
        ("FIX.5.0", "998", "kW-M"): "kW-M .. KilowattMonth",
        ("FIX.5.0", "998", "kW-min "): "kW-min  .. KilowattMinute",
        ("FIX.5.0", "998", "L"): "L .. liters",
        ("FIX.5.0", "998", "lbs"): "lbs .. Pounds",
        ("FIX.5.0", "998", "MW-a"): "MW-a .. MegawattYear",
        ("FIX.5.0", "998", "MW-d"): "MW-d .. MegawattDay",
        ("FIX.5.0", "998", "MW-h"): "MW-h .. MegawattHour",
        ("FIX.5.0", "998", "MW-M"): "MW-M .. MegawattMonth",
        ("FIX.5.0", "998", "MW-min"): "MW-min .. MegawattMinute",
        ("FIX.5.0", "998", "oz_tr"): "oz_tr .. TroyOunces",
        ("FIX.5.0", "998", "PRINC"): "PRINC .. PrincipalWithRelationToDebtInstrument",
        ("FIX.5.0", "998", "t"): "t .. MetricTons",
        ("FIX.5.0", "998", "tn"): "tn .. Tons",
        ("FIX.5.0", "999", "Bcf"): "Bcf .. BillionCubicFeet",
        ("FIX.5.0", "999", "CBM"): "CBM .. CubicMeters",
        ("FIX.5.0", "999", "GJ"): "GJ .. gigajoules",
        ("FIX.5.0", "999", "kWh"): "kWh .. KilowattHours",
        ("FIX.5.0", "999", "MMBtu"): "MMBtu .. OneMillionBTU",
        ("FIX.5.0", "999", "MWh"): "MWh .. MegawattHours",
        ("FIX.5.0", "999", "thm"): "thm .. therms",
        ("FIX.5.0", "999", "Alw"): "Alw .. Allowances",
        ("FIX.5.0", "999", "Bbl"): "Bbl .. Barrels",
        ("FIX.5.0", "999", "BDFT"): "BDFT .. BoardFeet",
        ("FIX.5.0", "999", "Bu"): "Bu .. Bushels",
        ("FIX.5.0", "999", "Ccy"): "Ccy .. Currency",
        ("FIX.5.0", "999", "CDD"): "CDD .. CoolingDegreeDay",
        ("FIX.5.0", "999", "CER"): "CER .. CertifiedEmissionsReduction",
        ("FIX.5.0", "999", "CPD"): "CPD .. CriticalPrecipDay",
        ("FIX.5.0", "999", "CRT"): "CRT .. ClimateReserveTonnes",
        ("FIX.5.0", "999", "cwt"): "cwt .. Hundredweight",
        ("FIX.5.0", "999", "day"): "day .. Day",
        ("FIX.5.0", "999", "dt"): "dt .. DryMetricTons",
        ("FIX.5.0", "999", "EnvAllwnc"): "EnvAllwnc .. EnvAllwncCert",
        ("FIX.5.0", "999", "EnvCrd"): "EnvCrd .. EnvironmentalCredit",
        ("FIX.5.0", "999", "EnvOfst"): "EnvOfst .. EnvironmentalOffset",
        ("FIX.5.0", "999", "g"): "g .. Grams",
        ("FIX.5.0", "999", "Gal"): "Gal .. Gallons",
        ("FIX.5.0", "999", "GT"): "GT .. GrossTons",
        ("FIX.5.0", "999", "HDD"): "HDD .. HeatingDegreeDay",
        ("FIX.5.0", "999", "IPNT"): "IPNT .. IndexPoint",
        ("FIX.5.0", "999", "kg"): "kg .. Kilograms",
        ("FIX.5.0", "999", "kL"): "kL .. kiloliters",
        ("FIX.5.0", "999", "kW-a"): "kW-a .. KilowattYear",
        ("FIX.5.0", "999", "kW-d"): "kW-d .. KilowattDay",
        ("FIX.5.0", "999", "kW-h"): "kW-h .. KilowattHour",
        ("FIX.5.0", "999", "kW-M"): "kW-M .. KilowattMonth",
        ("FIX.5.0", "999", "kW-min "): "kW-min  .. KilowattMinute",
        ("FIX.5.0", "999", "L"): "L .. liters",
        ("FIX.5.0", "999", "lbs"): "lbs .. Pounds",
        ("FIX.5.0", "999", "MW-a"): "MW-a .. MegawattYear",
        ("FIX.5.0", "999", "MW-d"): "MW-d .. MegawattDay",
        ("FIX.5.0", "999", "MW-h"): "MW-h .. MegawattHour",
        ("FIX.5.0", "999", "MW-M"): "MW-M .. MegawattMonth",
        ("FIX.5.0", "999", "MW-min"): "MW-min .. MegawattMinute",
        ("FIX.5.0", "999", "oz_tr"): "oz_tr .. TroyOunces",
        ("FIX.5.0", "999", "PRINC"): "PRINC .. PrincipalWithRelationToDebtInstrument",
        ("FIX.5.0", "999", "t"): "t .. MetricTons",
        ("FIX.5.0", "999", "tn"): "tn .. Tons",
        ("FIX.5.0", "1000", "H"): "H .. Hour",
        ("FIX.5.0", "1000", "Min"): "Min .. Minute",
        ("FIX.5.0", "1000", "S"): "S .. Second",
        ("FIX.5.0", "1000", "D"): "D .. Day",
        ("FIX.5.0", "1000", "Wk"): "Wk .. Week",
        ("FIX.5.0", "1000", "Mo"): "Mo .. Month",
        ("FIX.5.0", "1000", "Yr"): "Yr .. Year",
        ("FIX.5.0", "1000", "Q"): "Q .. Quarter",
        ("FIX.5.0", "1001", "H"): "H .. Hour",
        ("FIX.5.0", "1001", "Min"): "Min .. Minute",
        ("FIX.5.0", "1001", "S"): "S .. Second",
        ("FIX.5.0", "1001", "D"): "D .. Day",
        ("FIX.5.0", "1001", "Wk"): "Wk .. Week",
        ("FIX.5.0", "1001", "Mo"): "Mo .. Month",
        ("FIX.5.0", "1001", "Yr"): "Yr .. Year",
        ("FIX.5.0", "1001", "Q"): "Q .. Quarter",
        ("FIX.5.0", "1002", "1"): "1 .. Automatic",
        ("FIX.5.0", "1002", "2"): "2 .. Guarantor",
        ("FIX.5.0", "1002", "3"): "3 .. Manual",
        ("FIX.5.0", "1002", "4"): "4 .. BrokerAssigned",
        ("FIX.5.0", "1008", "0"): "0 .. CMTA",
        ("FIX.5.0", "1008", "1"): "1 .. InternalTransferOrAdjustment",
        ("FIX.5.0", "1008", "2"): "2 .. ExternalTransferOrTransferOfAccount",
        ("FIX.5.0", "1008", "3"): "3 .. RejectForSubmittingSide",
        ("FIX.5.0", "1008", "4"): "4 .. AdvisoryForContraSide",
        ("FIX.5.0", "1008", "5"): "5 .. OffsetDueToAnAllocation",
        ("FIX.5.0", "1008", "6"): "6 .. OnsetDueToAnAllocation",
        ("FIX.5.0", "1008", "7"): "7 .. DifferentialSpread",
        ("FIX.5.0", "1008", "8"): "8 .. ImpliedSpreadLegExecutedAgainstAnOutright",
        ("FIX.5.0", "1008", "9"): "9 .. TransactionFromExercise",
        ("FIX.5.0", "1008", "10"): "10 .. TransactionFromAssignment",
        ("FIX.5.0", "1008", "11"): "11 .. ACATS",
        ("FIX.5.0", "1008", "33"): "33 .. OffHoursTrade",
        ("FIX.5.0", "1008", "34"): "34 .. OnHoursTrade",
        ("FIX.5.0", "1008", "35"): "35 .. OTCQuote",
        ("FIX.5.0", "1008", "36"): "36 .. ConvertedSWAP",
        ("FIX.5.0", "1008", "40"): "40 .. WashTrade",
        ("FIX.5.0", "1008", "41"): "41 .. TradeAtSettlement",
        ("FIX.5.0", "1008", "42"): "42 .. AuctionTrade",
        ("FIX.5.0", "1008", "43"): "43 .. TradeAtMarker",
        ("FIX.5.0", "1008", "44"): "44 .. CreditDefault",
        ("FIX.5.0", "1008", "45"): "45 .. CreditRestructuring",
        ("FIX.5.0", "1008", "46"): "46 .. Merger",
        ("FIX.5.0", "1008", "47"): "47 .. SpinOff",
        ("FIX.5.0", "1008", "48"): "48 .. MultilateralCompression",
        ("FIX.5.0", "1008", "50"): "50 .. Balancing",
        ("FIX.5.0", "1008", "51"): "51 .. BasisTradeIndexClose",
        ("FIX.5.0", "1008", "14"): "14 .. AI",
        ("FIX.5.0", "1008", "15"): "15 .. B",
        ("FIX.5.0", "1008", "16"): "16 .. K",
        ("FIX.5.0", "1008", "17"): "17 .. LC",
        ("FIX.5.0", "1008", "18"): "18 .. M",
        ("FIX.5.0", "1008", "19"): "19 .. N",
        ("FIX.5.0", "1008", "20"): "20 .. NM",
        ("FIX.5.0", "1008", "21"): "21 .. NR",
        ("FIX.5.0", "1008", "22"): "22 .. P",
        ("FIX.5.0", "1008", "23"): "23 .. PA",
        ("FIX.5.0", "1008", "24"): "24 .. PC",
        ("FIX.5.0", "1008", "25"): "25 .. PN",
        ("FIX.5.0", "1008", "26"): "26 .. R",
        ("FIX.5.0", "1008", "27"): "27 .. RO",
        ("FIX.5.0", "1008", "28"): "28 .. RT",
        ("FIX.5.0", "1008", "29"): "29 .. SW",
        ("FIX.5.0", "1008", "30"): "30 .. T",
        ("FIX.5.0", "1008", "31"): "31 .. WN",
        ("FIX.5.0", "1008", "32"): "32 .. WT",
        ("FIX.5.0", "1008", "37"): "37 .. CrossedTrade",
        ("FIX.5.0", "1008", "38"): "38 .. InterimProtectedTrade",
        ("FIX.5.0", "1008", "39"): "39 .. LargeInScale",
        ("FIX.5.0", "1013", "1"): "1 .. ExecutionTime",
        ("FIX.5.0", "1013", "2"): "2 .. TimeIn",
        ("FIX.5.0", "1013", "3"): "3 .. TimeOut",
        ("FIX.5.0", "1013", "4"): "4 .. BrokerReceipt",
        ("FIX.5.0", "1013", "5"): "5 .. BrokerExecution",
        ("FIX.5.0", "1013", "6"): "6 .. DeskReceipt",
        ("FIX.5.0", "1013", "7"): "7 .. SubmissionToClearing",
        ("FIX.5.0", "1013", "8"): "8 .. TimePriority",
        ("FIX.5.0", "1013", "9"): "9 .. OrderbookEntryTime",
        ("FIX.5.0", "1013", "10"): "10 .. OrderSubmissionTime",
        ("FIX.5.0", "1013", "11"): "11 .. PubliclyReported",
        ("FIX.5.0", "1013", "12"): "12 .. PublicReportUpdated",
        ("FIX.5.0", "1013", "13"): "13 .. NonPubliclyReported",
        ("FIX.5.0", "1013", "14"): "14 .. NonPublicReportUpdated",
        ("FIX.5.0", "1013", "15"): "15 .. SubmittedForConfirmation",
        ("FIX.5.0", "1013", "16"): "16 .. UpdatedForConfirmation",
        ("FIX.5.0", "1013", "17"): "17 .. Confirmed",
        ("FIX.5.0", "1013", "18"): "18 .. UpdatedForClearing",
        ("FIX.5.0", "1013", "19"): "19 .. Cleared",
        ("FIX.5.0", "1013", "20"): "20 .. AllocationsSubmitted",
        ("FIX.5.0", "1013", "21"): "21 .. AllocationsUpdated",
        ("FIX.5.0", "1013", "22"): "22 .. AllocationsCompleted",
        ("FIX.5.0", "1013", "23"): "23 .. SubmittedToRepository",
        ("FIX.5.0", "1013", "24"): "24 .. PostTrdContntnEvnt",
        ("FIX.5.0", "1013", "25"): "25 .. PostTradeValuation",
        ("FIX.5.0", "1013", "26"): "26 .. PreviousTimePriority",
        ("FIX.5.0", "1015", "0"): "0 .. False",
        ("FIX.5.0", "1015", "1"): "1 .. True",
        ("FIX.5.0", "1021", "1"): "1 .. TopOfBook",
        ("FIX.5.0", "1021", "2"): "2 .. PriceDepth",
        ("FIX.5.0", "1021", "3"): "3 .. OrderDepth",
        ("FIX.5.0", "1024", "0"): "0 .. Book",
        ("FIX.5.0", "1024", "1"): "1 .. OffBook",
        ("FIX.5.0", "1024", "2"): "2 .. Cross",
        ("FIX.5.0", "1024", "3"): "3 .. QuoteDrivenMarket",
        ("FIX.5.0", "1024", "4"): "4 .. DarkOrderBook",
        ("FIX.5.0", "1024", "5"): "5 .. AuctionDrivenMarket",
        ("FIX.5.0", "1024", "6"): "6 .. QuoteNegotiation",
        ("FIX.5.0", "1024", "7"): "7 .. VoiceNegotiation",
        ("FIX.5.0", "1024", "8"): "8 .. HybridMarket",
        ("FIX.5.0", "1031", "A"): "A .. PhoneSimple",
        ("FIX.5.0", "1031", "B"): "B .. PhoneComplex",
        ("FIX.5.0", "1031", "C"): "C .. FCMProvidedScreen",
        ("FIX.5.0", "1031", "D"): "D .. OtherProvidedScreen",
        ("FIX.5.0", "1031", "E"): "E .. ClientProvidedPlatformControlledByFCM",
        ("FIX.5.0", "1031", "F"): "F .. ClientProvidedPlatformDirectToExchange",
        ("FIX.5.0", "1031", "H"): "H .. AlgoEngine",
        ("FIX.5.0", "1031", "J"): "J .. PriceAtExecution",
        ("FIX.5.0", "1031", "W"): "W .. DeskElectronic",
        ("FIX.5.0", "1031", "X"): "X .. DeskPit",
        ("FIX.5.0", "1031", "Y"): "Y .. ClientElectronic",
        ("FIX.5.0", "1031", "Z"): "Z .. ClientPit",
        ("FIX.5.0", "1031", "ADD"): "ADD .. AddOnOrder",
        ("FIX.5.0", "1031", "AON"): "AON .. AllOrNone",
        ("FIX.5.0", "1031", "CND"): "CND .. ConditionalOrder",
        ("FIX.5.0", "1031", "CNH"): "CNH .. CashNotHeld",
        ("FIX.5.0", "1031", "CSH"): "CSH .. DeliveryInstructionsCash",
        ("FIX.5.0", "1031", "DIR"): "DIR .. DirectedOrder",
        ("FIX.5.0", "1031", "DLO"): "DLO .. DiscretionaryLimitOrder",
        ("FIX.5.0", "1031", "E.W"): "E.W .. ExchangeForPhysicalTransaction",
        ("FIX.5.0", "1031", "FOK"): "FOK .. FillOrKill",
        ("FIX.5.0", "1031", "IDX"): "IDX .. IntraDayCross",
        ("FIX.5.0", "1031", "IO"): "IO .. ImbalanceOnly",
        ("FIX.5.0", "1031", "IOC"): "IOC .. ImmediateOrCancel",
        ("FIX.5.0", "1031", "ISO"): "ISO .. IntermarketSweepOrder",
        ("FIX.5.0", "1031", "LOO"): "LOO .. LimitOnOpen",
        ("FIX.5.0", "1031", "LOC"): "LOC .. LimitOnClose",
        ("FIX.5.0", "1031", "MAO"): "MAO .. MarketAtOpen",
        ("FIX.5.0", "1031", "MAC"): "MAC .. MarketAtClose",
        ("FIX.5.0", "1031", "MOO"): "MOO .. MarketOnOpen",
        ("FIX.5.0", "1031", "MOC"): "MOC .. MarketOnClose",
        ("FIX.5.0", "1031", "MPT"): "MPT .. MergerRelatedTransferPosition",
        ("FIX.5.0", "1031", "MQT"): "MQT .. MinimumQuantity",
        ("FIX.5.0", "1031", "MTL"): "MTL .. MarketToLimit",
        ("FIX.5.0", "1031", "ND"): "ND .. DeliveryInstructionsNextDay",
        ("FIX.5.0", "1031", "NH"): "NH .. NotHeld",
        ("FIX.5.0", "1031", "OPT"): "OPT .. OptionsRelatedTransaction",
        ("FIX.5.0", "1031", "OVD"): "OVD .. OverTheDay",
        ("FIX.5.0", "1031", "PEG"): "PEG .. Pegged",
        ("FIX.5.0", "1031", "RSV"): "RSV .. ReserveSizeOrder",
        ("FIX.5.0", "1031", "S.W"): "S.W .. StopStockTransaction",
        ("FIX.5.0", "1031", "SCL"): "SCL .. Scale",
        ("FIX.5.0", "1031", "SLR"): "SLR .. DeliveryInstructionsSellersOption",
        ("FIX.5.0", "1031", "TMO"): "TMO .. TimeOrder",
        ("FIX.5.0", "1031", "TS"): "TS .. TrailingStop",
        ("FIX.5.0", "1031", "WRK"): "WRK .. Work",
        ("FIX.5.0", "1031", "F0"): "F0 .. StayOnOfferside",
        ("FIX.5.0", "1031", "F3"): "F3 .. GoAlong",
        ("FIX.5.0", "1031", "F6"): "F6 .. ParticipateDoNotInitiate",
        ("FIX.5.0", "1031", "F7"): "F7 .. StrictScale",
        ("FIX.5.0", "1031", "F8"): "F8 .. TryToScale",
        ("FIX.5.0", "1031", "F9"): "F9 .. StayOnBidside",
        ("FIX.5.0", "1031", "FA"): "FA .. NoCross",
        ("FIX.5.0", "1031", "FB"): "FB .. OKToCross",
        ("FIX.5.0", "1031", "FC"): "FC .. CallFirst",
        ("FIX.5.0", "1031", "FD"): "FD .. PercentOfVolume",
        ("FIX.5.0", "1031", "FH"): "FH .. ReinstateOnSystemFailure",
        ("FIX.5.0", "1031", "FI"): "FI .. InstitutionOnly",
        ("FIX.5.0", "1031", "FJ"): "FJ .. ReinstateOnTradingHalt",
        ("FIX.5.0", "1031", "FK"): "FK .. CancelOnTradingHalf",
        ("FIX.5.0", "1031", "FL"): "FL .. LastPeg",
        ("FIX.5.0", "1031", "FM"): "FM .. MidPricePeg",
        ("FIX.5.0", "1031", "FN"): "FN .. NonNegotiable",
        ("FIX.5.0", "1031", "FO"): "FO .. OpeningPeg",
        ("FIX.5.0", "1031", "FP"): "FP .. MarketPeg",
        ("FIX.5.0", "1031", "FQ"): "FQ .. CancelOnSystemFailure",
        ("FIX.5.0", "1031", "FR"): "FR .. PrimaryPeg",
        ("FIX.5.0", "1031", "FS"): "FS .. Suspend",
        ("FIX.5.0", "1031", "FT"): "FT .. FixedPegToLocalBBO",
        ("FIX.5.0", "1031", "FW"): "FW .. PegToVWAP",
        ("FIX.5.0", "1031", "FX"): "FX .. TradeAlong",
        ("FIX.5.0", "1031", "FY"): "FY .. TryToStop",
        ("FIX.5.0", "1031", "FZ"): "FZ .. CancelIfNotBest",
        ("FIX.5.0", "1031", "Fb"): "Fb .. StrictLimit",
        ("FIX.5.0", "1031", "Fc"): "Fc .. IgnorePriceValidityChecks",
        ("FIX.5.0", "1031", "Fd"): "Fd .. PegToLimitPrice",
        ("FIX.5.0", "1031", "Fe"): "Fe .. WorkToTargetStrategy",
        ("FIX.5.0", "1031", "G"): "G .. GOrderAndFCMAPIorFIX",
        ("FIX.5.0", "1032", "1"): "1 .. FINRAOATS",
        ("FIX.5.0", "1032", "2"): "2 .. FIAExecutionSourceCode",
        ("FIX.5.0", "1033", "A"): "A .. Agency",
        ("FIX.5.0", "1033", "AR"): "AR .. Arbitrage",
        ("FIX.5.0", "1033", "B"): "B .. BlockTrading",
        ("FIX.5.0", "1033", "C"): "C .. ConvertibleDesk",
        ("FIX.5.0", "1033", "CR"): "CR .. CentralRiskBooks",
        ("FIX.5.0", "1033", "D"): "D .. Derivatives",
        ("FIX.5.0", "1033", "EC"): "EC .. EquityCapitalMarkets",
        ("FIX.5.0", "1033", "IN"): "IN .. International",
        ("FIX.5.0", "1033", "IS"): "IS .. Institutional",
        ("FIX.5.0", "1033", "O"): "O .. Other",
        ("FIX.5.0", "1033", "PF"): "PF .. PreferredTrading",
        ("FIX.5.0", "1033", "PR"): "PR .. Proprietary",
        ("FIX.5.0", "1033", "PT"): "PT .. ProgramTrading",
        ("FIX.5.0", "1033", "S"): "S .. Sales",
        ("FIX.5.0", "1033", "SW"): "SW .. Swaps",
        ("FIX.5.0", "1033", "T"): "T .. TradingDeskSystem",
        ("FIX.5.0", "1033", "TR"): "TR .. Treasury",
        ("FIX.5.0", "1034", "1"): "1 .. FINRAOATS",
        ("FIX.5.0", "1035", "A"): "A .. PhoneSimple",
        ("FIX.5.0", "1035", "B"): "B .. PhoneComplex",
        ("FIX.5.0", "1035", "C"): "C .. FCMProvidedScreen",
        ("FIX.5.0", "1035", "D"): "D .. OtherProvidedScreen",
        ("FIX.5.0", "1035", "E"): "E .. ClientProvidedPlatformControlledByFCM",
        ("FIX.5.0", "1035", "F"): "F .. ClientProvidedPlatformDirectToExchange",
        ("FIX.5.0", "1035", "H"): "H .. AlgoEngine",
        ("FIX.5.0", "1035", "J"): "J .. PriceAtExecution",
        ("FIX.5.0", "1035", "W"): "W .. DeskElectronic",
        ("FIX.5.0", "1035", "X"): "X .. DeskPit",
        ("FIX.5.0", "1035", "Y"): "Y .. ClientElectronic",
        ("FIX.5.0", "1035", "Z"): "Z .. ClientPit",
        ("FIX.5.0", "1035", "ADD"): "ADD .. AddOnOrder",
        ("FIX.5.0", "1035", "AON"): "AON .. AllOrNone",
        ("FIX.5.0", "1035", "CND"): "CND .. ConditionalOrder",
        ("FIX.5.0", "1035", "CNH"): "CNH .. CashNotHeld",
        ("FIX.5.0", "1035", "CSH"): "CSH .. DeliveryInstructionsCash",
        ("FIX.5.0", "1035", "DIR"): "DIR .. DirectedOrder",
        ("FIX.5.0", "1035", "DLO"): "DLO .. DiscretionaryLimitOrder",
        ("FIX.5.0", "1035", "E.W"): "E.W .. ExchangeForPhysicalTransaction",
        ("FIX.5.0", "1035", "FOK"): "FOK .. FillOrKill",
        ("FIX.5.0", "1035", "IDX"): "IDX .. IntraDayCross",
        ("FIX.5.0", "1035", "IO"): "IO .. ImbalanceOnly",
        ("FIX.5.0", "1035", "IOC"): "IOC .. ImmediateOrCancel",
        ("FIX.5.0", "1035", "ISO"): "ISO .. IntermarketSweepOrder",
        ("FIX.5.0", "1035", "LOO"): "LOO .. LimitOnOpen",
        ("FIX.5.0", "1035", "LOC"): "LOC .. LimitOnClose",
        ("FIX.5.0", "1035", "MAO"): "MAO .. MarketAtOpen",
        ("FIX.5.0", "1035", "MAC"): "MAC .. MarketAtClose",
        ("FIX.5.0", "1035", "MOO"): "MOO .. MarketOnOpen",
        ("FIX.5.0", "1035", "MOC"): "MOC .. MarketOnClose",
        ("FIX.5.0", "1035", "MPT"): "MPT .. MergerRelatedTransferPosition",
        ("FIX.5.0", "1035", "MQT"): "MQT .. MinimumQuantity",
        ("FIX.5.0", "1035", "MTL"): "MTL .. MarketToLimit",
        ("FIX.5.0", "1035", "ND"): "ND .. DeliveryInstructionsNextDay",
        ("FIX.5.0", "1035", "NH"): "NH .. NotHeld",
        ("FIX.5.0", "1035", "OPT"): "OPT .. OptionsRelatedTransaction",
        ("FIX.5.0", "1035", "OVD"): "OVD .. OverTheDay",
        ("FIX.5.0", "1035", "PEG"): "PEG .. Pegged",
        ("FIX.5.0", "1035", "RSV"): "RSV .. ReserveSizeOrder",
        ("FIX.5.0", "1035", "S.W"): "S.W .. StopStockTransaction",
        ("FIX.5.0", "1035", "SCL"): "SCL .. Scale",
        ("FIX.5.0", "1035", "SLR"): "SLR .. DeliveryInstructionsSellersOption",
        ("FIX.5.0", "1035", "TMO"): "TMO .. TimeOrder",
        ("FIX.5.0", "1035", "TS"): "TS .. TrailingStop",
        ("FIX.5.0", "1035", "WRK"): "WRK .. Work",
        ("FIX.5.0", "1035", "F0"): "F0 .. StayOnOfferside",
        ("FIX.5.0", "1035", "F3"): "F3 .. GoAlong",
        ("FIX.5.0", "1035", "F6"): "F6 .. ParticipateDoNotInitiate",
        ("FIX.5.0", "1035", "F7"): "F7 .. StrictScale",
        ("FIX.5.0", "1035", "F8"): "F8 .. TryToScale",
        ("FIX.5.0", "1035", "F9"): "F9 .. StayOnBidside",
        ("FIX.5.0", "1035", "FA"): "FA .. NoCross",
        ("FIX.5.0", "1035", "FB"): "FB .. OKToCross",
        ("FIX.5.0", "1035", "FC"): "FC .. CallFirst",
        ("FIX.5.0", "1035", "FD"): "FD .. PercentOfVolume",
        ("FIX.5.0", "1035", "FH"): "FH .. ReinstateOnSystemFailure",
        ("FIX.5.0", "1035", "FI"): "FI .. InstitutionOnly",
        ("FIX.5.0", "1035", "FJ"): "FJ .. ReinstateOnTradingHalt",
        ("FIX.5.0", "1035", "FK"): "FK .. CancelOnTradingHalf",
        ("FIX.5.0", "1035", "FL"): "FL .. LastPeg",
        ("FIX.5.0", "1035", "FM"): "FM .. MidPricePeg",
        ("FIX.5.0", "1035", "FN"): "FN .. NonNegotiable",
        ("FIX.5.0", "1035", "FO"): "FO .. OpeningPeg",
        ("FIX.5.0", "1035", "FP"): "FP .. MarketPeg",
        ("FIX.5.0", "1035", "FQ"): "FQ .. CancelOnSystemFailure",
        ("FIX.5.0", "1035", "FR"): "FR .. PrimaryPeg",
        ("FIX.5.0", "1035", "FS"): "FS .. Suspend",
        ("FIX.5.0", "1035", "FT"): "FT .. FixedPegToLocalBBO",
        ("FIX.5.0", "1035", "FW"): "FW .. PegToVWAP",
        ("FIX.5.0", "1035", "FX"): "FX .. TradeAlong",
        ("FIX.5.0", "1035", "FY"): "FY .. TryToStop",
        ("FIX.5.0", "1035", "FZ"): "FZ .. CancelIfNotBest",
        ("FIX.5.0", "1035", "Fb"): "Fb .. StrictLimit",
        ("FIX.5.0", "1035", "Fc"): "Fc .. IgnorePriceValidityChecks",
        ("FIX.5.0", "1035", "Fd"): "Fd .. PegToLimitPrice",
        ("FIX.5.0", "1035", "Fe"): "Fe .. WorkToTargetStrategy",
        ("FIX.5.0", "1035", "G"): "G .. GOrderAndFCMAPIorFIX",
        ("FIX.5.0", "1036", "0"): "0 .. Received",
        ("FIX.5.0", "1036", "1"): "1 .. Accepted",
        ("FIX.5.0", "1036", "2"): "2 .. Don",
        ("FIX.5.0", "1039", "C"): "C .. CashSettlementRequired",
        ("FIX.5.0", "1039", "P"): "P .. PhysicalSettlementRequired",
        ("FIX.5.0", "1039", "E"): "E .. Election",
        ("FIX.5.0", "1043", "0"): "0 .. SpecificDeposit",
        ("FIX.5.0", "1043", "1"): "1 .. General",
        ("FIX.5.0", "1046", "D"): "D .. Divide",
        ("FIX.5.0", "1046", "M"): "M .. Multiply",
        ("FIX.5.0", "1047", "O"): "O .. Open",
        ("FIX.5.0", "1047", "C"): "C .. Close",
        ("FIX.5.0", "1047", "R"): "R .. Rolled",
        ("FIX.5.0", "1047", "F"): "F .. FIFO",
        ("FIX.5.0", "1048", "A"): "A .. Agent",
        ("FIX.5.0", "1048", "P"): "P .. Principal",
        ("FIX.5.0", "1048", "R"): "R .. RisklessPrincipal",
        ("FIX.5.0", "1049", "P"): "P .. ProRata",
        ("FIX.5.0", "1049", "R"): "R .. Random",
        ("FIX.5.0", "1050", "6"): "6 .. UKNationalInsuranceOrPensionNumber",
        ("FIX.5.0", "1050", "7"): "7 .. USSocialSecurityNumber",
        ("FIX.5.0", "1050", "8"): "8 .. USEmployerOrTaxIDNumber",
        ("FIX.5.0", "1050", "9"): "9 .. AustralianBusinessNumber",
        ("FIX.5.0", "1050", "A"): "A .. AustralianTaxFileNumber",
        ("FIX.5.0", "1050", "J"): "J .. TaxID",
        ("FIX.5.0", "1050", "1"): "1 .. KoreanInvestorID",
        ("FIX.5.0", "1050", "2"): "2 .. TaiwaneseForeignInvestorID",
        ("FIX.5.0", "1050", "3"): "3 .. TaiwaneseTradingAcct",
        ("FIX.5.0", "1050", "4"): "4 .. MalaysianCentralDepository",
        ("FIX.5.0", "1050", "5"): "5 .. ChineseInvestorID",
        ("FIX.5.0", "1050", "I"): "I .. ISITCAcronym",
        ("FIX.5.0", "1050", "B"): "B .. BIC",
        ("FIX.5.0", "1050", "C"): "C .. GeneralIdentifier",
        ("FIX.5.0", "1050", "D"): "D .. Proprietary",
        ("FIX.5.0", "1050", "E"): "E .. ISOCountryCode",
        ("FIX.5.0", "1050", "F"): "F .. SettlementEntityLocation",
        ("FIX.5.0", "1050", "G"): "G .. MIC",
        ("FIX.5.0", "1050", "H"): "H .. CSDParticipant",
        ("FIX.5.0", "1050", "K"): "K .. AustralianCompanyNumber",
        ("FIX.5.0", "1050", "L"): "L .. AustralianRegisteredBodyNumber",
        ("FIX.5.0", "1050", "M"): "M .. CFTCReportingFirmIdentifier",
        ("FIX.5.0", "1050", "N"): "N .. LegalEntityIdentifier",
        ("FIX.5.0", "1050", "O"): "O .. InterimIdentifier",
        ("FIX.5.0", "1050", "P"): "P .. ShortCodeIdentifier",
        ("FIX.5.0", "1050", "Q"): "Q .. NationalIDNaturalPerson",
        ("FIX.5.0", "1051", "1"): "1 .. ExecutingFirm",
        ("FIX.5.0", "1051", "2"): "2 .. BrokerOfCredit",
        ("FIX.5.0", "1051", "3"): "3 .. ClientID",
        ("FIX.5.0", "1051", "4"): "4 .. ClearingFirm",
        ("FIX.5.0", "1051", "5"): "5 .. InvestorID",
        ("FIX.5.0", "1051", "6"): "6 .. IntroducingFirm",
        ("FIX.5.0", "1051", "7"): "7 .. EnteringFirm",
        ("FIX.5.0", "1051", "8"): "8 .. Locate",
        ("FIX.5.0", "1051", "9"): "9 .. FundManagerClientID",
        ("FIX.5.0", "1051", "10"): "10 .. SettlementLocation",
        ("FIX.5.0", "1051", "11"): "11 .. OrderOriginationTrader",
        ("FIX.5.0", "1051", "12"): "12 .. ExecutingTrader",
        ("FIX.5.0", "1051", "13"): "13 .. OrderOriginationFirm",
        ("FIX.5.0", "1051", "15"): "15 .. CorrespondantClearingFirm",
        ("FIX.5.0", "1051", "16"): "16 .. ExecutingSystem",
        ("FIX.5.0", "1051", "17"): "17 .. ContraFirm",
        ("FIX.5.0", "1051", "18"): "18 .. ContraClearingFirm",
        ("FIX.5.0", "1051", "19"): "19 .. SponsoringFirm",
        ("FIX.5.0", "1051", "20"): "20 .. UnderlyingContraFirm",
        ("FIX.5.0", "1051", "21"): "21 .. ClearingOrganization",
        ("FIX.5.0", "1051", "22"): "22 .. Exchange",
        ("FIX.5.0", "1051", "24"): "24 .. CustomerAccount",
        ("FIX.5.0", "1051", "25"): "25 .. CorrespondentClearingOrganization",
        ("FIX.5.0", "1051", "26"): "26 .. CorrespondentBroker",
        ("FIX.5.0", "1051", "27"): "27 .. Buyer",
        ("FIX.5.0", "1051", "28"): "28 .. Custodian",
        ("FIX.5.0", "1051", "29"): "29 .. Intermediary",
        ("FIX.5.0", "1051", "30"): "30 .. Agent",
        ("FIX.5.0", "1051", "31"): "31 .. SubCustodian",
        ("FIX.5.0", "1051", "32"): "32 .. Beneficiary",
        ("FIX.5.0", "1051", "33"): "33 .. InterestedParty",
        ("FIX.5.0", "1051", "34"): "34 .. RegulatoryBody",
        ("FIX.5.0", "1051", "35"): "35 .. LiquidityProvider",
        ("FIX.5.0", "1051", "36"): "36 .. EnteringTrader",
        ("FIX.5.0", "1051", "37"): "37 .. ContraTrader",
        ("FIX.5.0", "1051", "38"): "38 .. PositionAccount",
        ("FIX.5.0", "1051", "39"): "39 .. ContraInvestorID",
        ("FIX.5.0", "1051", "40"): "40 .. TransferToFirm",
        ("FIX.5.0", "1051", "41"): "41 .. ContraPositionAccount",
        ("FIX.5.0", "1051", "42"): "42 .. ContraExchange",
        ("FIX.5.0", "1051", "43"): "43 .. InternalCarryAccount",
        ("FIX.5.0", "1051", "44"): "44 .. OrderEntryOperatorID",
        ("FIX.5.0", "1051", "45"): "45 .. SecondaryAccountNumber",
        ("FIX.5.0", "1051", "46"): "46 .. ForeignFirm",
        ("FIX.5.0", "1051", "47"): "47 .. ThirdPartyAllocationFirm",
        ("FIX.5.0", "1051", "48"): "48 .. ClaimingAccount",
        ("FIX.5.0", "1051", "49"): "49 .. AssetManager",
        ("FIX.5.0", "1051", "50"): "50 .. PledgorAccount",
        ("FIX.5.0", "1051", "51"): "51 .. PledgeeAccount",
        ("FIX.5.0", "1051", "52"): "52 .. LargeTraderReportableAccount",
        ("FIX.5.0", "1051", "53"): "53 .. TraderMnemonic",
        ("FIX.5.0", "1051", "54"): "54 .. SenderLocation",
        ("FIX.5.0", "1051", "55"): "55 .. SessionID",
        ("FIX.5.0", "1051", "56"): "56 .. AcceptableCounterparty",
        ("FIX.5.0", "1051", "57"): "57 .. UnacceptableCounterparty",
        ("FIX.5.0", "1051", "58"): "58 .. EnteringUnit",
        ("FIX.5.0", "1051", "59"): "59 .. ExecutingUnit",
        ("FIX.5.0", "1051", "60"): "60 .. IntroducingBroker",
        ("FIX.5.0", "1051", "61"): "61 .. QuoteOriginator",
        ("FIX.5.0", "1051", "62"): "62 .. ReportOriginator",
        ("FIX.5.0", "1051", "63"): "63 .. SystematicInternaliser",
        ("FIX.5.0", "1051", "64"): "64 .. MultilateralTradingFacility",
        ("FIX.5.0", "1051", "65"): "65 .. RegulatedMarket",
        ("FIX.5.0", "1051", "66"): "66 .. MarketMaker",
        ("FIX.5.0", "1051", "67"): "67 .. InvestmentFirm",
        ("FIX.5.0", "1051", "68"): "68 .. HostCompetentAuthority",
        ("FIX.5.0", "1051", "69"): "69 .. HomeCompetentAuthority",
        ("FIX.5.0", "1051", "70"): "70 .. CompetentAuthorityLiquidity",
        ("FIX.5.0", "1051", "71"): "71 .. CompetentAuthorityTransactionVenue",
        ("FIX.5.0", "1051", "72"): "72 .. ReportingIntermediary",
        ("FIX.5.0", "1051", "73"): "73 .. ExecutionVenue",
        ("FIX.5.0", "1051", "74"): "74 .. MarketDataEntryOriginator",
        ("FIX.5.0", "1051", "75"): "75 .. LocationID",
        ("FIX.5.0", "1051", "76"): "76 .. DeskID",
        ("FIX.5.0", "1051", "77"): "77 .. MarketDataMarket",
        ("FIX.5.0", "1051", "78"): "78 .. AllocationEntity",
        ("FIX.5.0", "1051", "79"): "79 .. PrimeBroker",
        ("FIX.5.0", "1051", "80"): "80 .. StepOutFirm",
        ("FIX.5.0", "1051", "81"): "81 .. BrokerClearingID",
        ("FIX.5.0", "1051", "82"): "82 .. CentralRegistrationDepository",
        ("FIX.5.0", "1051", "83"): "83 .. ClearingAccount",
        ("FIX.5.0", "1051", "84"): "84 .. AcceptableSettlingCounterparty",
        ("FIX.5.0", "1051", "85"): "85 .. UnacceptableSettlingCounterparty",
        ("FIX.5.0", "1051", "86"): "86 .. CLSMemberBank",
        ("FIX.5.0", "1051", "87"): "87 .. InConcertGroup",
        ("FIX.5.0", "1051", "88"): "88 .. InConcertControllingEntity",
        ("FIX.5.0", "1051", "89"): "89 .. LargePositionsReportingAccount",
        ("FIX.5.0", "1051", "90"): "90 .. SettlementFirm",
        ("FIX.5.0", "1051", "91"): "91 .. SettlementAccount",
        ("FIX.5.0", "1051", "92"): "92 .. ReportingMarketCenter",
        ("FIX.5.0", "1051", "93"): "93 .. RelatedReportingMarketCenter",
        ("FIX.5.0", "1051", "94"): "94 .. AwayMarket",
        ("FIX.5.0", "1051", "95"): "95 .. GiveupTradingFirm",
        ("FIX.5.0", "1051", "96"): "96 .. TakeupTradingFirm",
        ("FIX.5.0", "1051", "97"): "97 .. GiveupClearingFirm",
        ("FIX.5.0", "1051", "98"): "98 .. TakeupClearingFirm",
        ("FIX.5.0", "1051", "99"): "99 .. OriginatingMarket",
        ("FIX.5.0", "1051", "100"): "100 .. MarginAccount",
        ("FIX.5.0", "1051", "101"): "101 .. CollateralAssetAccount",
        ("FIX.5.0", "1051", "102"): "102 .. DataRepository",
        ("FIX.5.0", "1051", "103"): "103 .. CalculationAgent",
        ("FIX.5.0", "1051", "104"): "104 .. ExerciseNoticeSender",
        ("FIX.5.0", "1051", "105"): "105 .. ExerciseNoticeReceiver",
        ("FIX.5.0", "1051", "106"): "106 .. RateReferenceBank",
        ("FIX.5.0", "1051", "107"): "107 .. Correspondent",
        ("FIX.5.0", "1051", "109"): "109 .. BeneficiaryBank",
        ("FIX.5.0", "1051", "110"): "110 .. Borrower",
        ("FIX.5.0", "1051", "111"): "111 .. PrimaryObligator",
        ("FIX.5.0", "1051", "112"): "112 .. Guarantor",
        ("FIX.5.0", "1051", "113"): "113 .. ExcludedReferenceEntity",
        ("FIX.5.0", "1051", "114"): "114 .. DeterminingParty",
        ("FIX.5.0", "1051", "115"): "115 .. HedgingParty",
        ("FIX.5.0", "1051", "116"): "116 .. ReportingEntity",
        ("FIX.5.0", "1051", "117"): "117 .. SalesPerson",
        ("FIX.5.0", "1051", "118"): "118 .. Operator",
        ("FIX.5.0", "1051", "119"): "119 .. CSD",
        ("FIX.5.0", "1051", "120"): "120 .. ICSD",
        ("FIX.5.0", "1051", "121"): "121 .. TradingSubAccount",
        ("FIX.5.0", "1051", "122"): "122 .. InvestmentDecisionMaker",
        ("FIX.5.0", "1051", "123"): "123 .. PublishingIntermediary",
        ("FIX.5.0", "1054", "1"): "1 .. Firm",
        ("FIX.5.0", "1054", "2"): "2 .. Person",
        ("FIX.5.0", "1054", "3"): "3 .. System",
        ("FIX.5.0", "1054", "4"): "4 .. Application",
        ("FIX.5.0", "1054", "5"): "5 .. FullLegalNameOfFirm",
        ("FIX.5.0", "1054", "6"): "6 .. PostalAddress",
        ("FIX.5.0", "1054", "7"): "7 .. PhoneNumber",
        ("FIX.5.0", "1054", "8"): "8 .. EmailAddress",
        ("FIX.5.0", "1054", "9"): "9 .. ContactName",
        ("FIX.5.0", "1054", "10"): "10 .. SecuritiesAccountNumber",
        ("FIX.5.0", "1054", "11"): "11 .. RegistrationNumber",
        ("FIX.5.0", "1054", "12"): "12 .. RegisteredAddressForConfirmation",
        ("FIX.5.0", "1054", "13"): "13 .. RegulatoryStatus",
        ("FIX.5.0", "1054", "14"): "14 .. RegistrationName",
        ("FIX.5.0", "1054", "15"): "15 .. CashAccountNumber",
        ("FIX.5.0", "1054", "16"): "16 .. BIC",
        ("FIX.5.0", "1054", "17"): "17 .. CSDParticipantMemberCode",
        ("FIX.5.0", "1054", "18"): "18 .. RegisteredAddress",
        ("FIX.5.0", "1054", "19"): "19 .. FundAccountName",
        ("FIX.5.0", "1054", "20"): "20 .. TelexNumber",
        ("FIX.5.0", "1054", "21"): "21 .. FaxNumber",
        ("FIX.5.0", "1054", "22"): "22 .. SecuritiesAccountName",
        ("FIX.5.0", "1054", "23"): "23 .. CashAccountName",
        ("FIX.5.0", "1054", "24"): "24 .. Department",
        ("FIX.5.0", "1054", "25"): "25 .. LocationDesk",
        ("FIX.5.0", "1054", "26"): "26 .. PositionAccountType",
        ("FIX.5.0", "1054", "27"): "27 .. SecurityLocateID",
        ("FIX.5.0", "1054", "28"): "28 .. MarketMaker",
        ("FIX.5.0", "1054", "29"): "29 .. EligibleCounterparty",
        ("FIX.5.0", "1054", "30"): "30 .. ProfessionalClient",
        ("FIX.5.0", "1054", "31"): "31 .. Location",
        ("FIX.5.0", "1054", "32"): "32 .. ExecutionVenue",
        ("FIX.5.0", "1054", "33"): "33 .. CurrencyDeliveryIdentifier",
        ("FIX.5.0", "1054", "34"): "34 .. AddressCity",
        ("FIX.5.0", "1054", "35"): "35 .. AddressStateOrProvince",
        ("FIX.5.0", "1054", "36"): "36 .. AddressPostalCode",
        ("FIX.5.0", "1054", "37"): "37 .. AddressStreet",
        ("FIX.5.0", "1054", "38"): "38 .. AddressISOCountryCode",
        ("FIX.5.0", "1054", "39"): "39 .. ISOCountryCode",
        ("FIX.5.0", "1054", "40"): "40 .. MarketSegment",
        ("FIX.5.0", "1054", "41"): "41 .. CustomerAccountType",
        ("FIX.5.0", "1054", "42"): "42 .. OmnibusAccount",
        ("FIX.5.0", "1054", "43"): "43 .. FundsSegregationType",
        ("FIX.5.0", "1054", "44"): "44 .. GuaranteeFund",
        ("FIX.5.0", "1054", "45"): "45 .. SwapDealer",
        ("FIX.5.0", "1054", "46"): "46 .. MajorParticipant",
        ("FIX.5.0", "1054", "47"): "47 .. FinancialEntity",
        ("FIX.5.0", "1054", "48"): "48 .. USPerson",
        ("FIX.5.0", "1054", "49"): "49 .. ReportingEntityIndicator",
        ("FIX.5.0", "1054", "50"): "50 .. ElectedClearingRequirementException",
        ("FIX.5.0", "1054", "51"): "51 .. BusinessCenter",
        ("FIX.5.0", "1054", "52"): "52 .. ReferenceText",
        ("FIX.5.0", "1054", "53"): "53 .. ShortMarkingExemptAccount",
        ("FIX.5.0", "1054", "54"): "54 .. ParentFirmIdentifier",
        ("FIX.5.0", "1054", "55"): "55 .. ParentFirmName",
        ("FIX.5.0", "1054", "56"): "56 .. DealIdentifier",
        ("FIX.5.0", "1054", "57"): "57 .. SystemTradeID",
        ("FIX.5.0", "1054", "58"): "58 .. SystemTradeSubID",
        ("FIX.5.0", "1054", "59"): "59 .. FCMCode",
        ("FIX.5.0", "1054", "60"): "60 .. DlvryTrmlCode",
        ("FIX.5.0", "1054", "61"): "61 .. VolntyRptEntity",
        ("FIX.5.0", "1054", "62"): "62 .. RptObligJursdctn",
        ("FIX.5.0", "1054", "63"): "63 .. VolntyRptJursdctn",
        ("FIX.5.0", "1054", "64"): "64 .. CompanyActivities",
        ("FIX.5.0", "1054", "65"): "65 .. EEAreaDomiciled",
        ("FIX.5.0", "1054", "66"): "66 .. ContractLinked",
        ("FIX.5.0", "1054", "67"): "67 .. ContractAbove",
        ("FIX.5.0", "1054", "68"): "68 .. VolntyRptPty",
        ("FIX.5.0", "1054", "69"): "69 .. EndUser",
        ("FIX.5.0", "1054", "70"): "70 .. LocationOrJurisdiction",
        ("FIX.5.0", "1054", "71"): "71 .. DerivativesDealer",
        ("FIX.5.0", "1054", "72"): "72 .. Domicile",
        ("FIX.5.0", "1054", "73"): "73 .. ExemptFromRecognition",
        ("FIX.5.0", "1054", "74"): "74 .. Payer",
        ("FIX.5.0", "1054", "75"): "75 .. Receiver",
        ("FIX.5.0", "1054", "76"): "76 .. SystematicInternaliser",
        ("FIX.5.0", "1054", "77"): "77 .. PublishingEntityIndicator",
        ("FIX.5.0", "1054", "78"): "78 .. FirstName",
        ("FIX.5.0", "1054", "79"): "79 .. Surname",
        ("FIX.5.0", "1054", "80"): "80 .. DateOfBirth",
        ("FIX.5.0", "1054", "81"): "81 .. OrderTransmittingFirm",
        ("FIX.5.0", "1054", "82"): "82 .. OrderTransmittingFirmBuyer",
        ("FIX.5.0", "1054", "83"): "83 .. OrderTransmitterSeller",
        ("FIX.5.0", "1054", "84"): "84 .. LegalEntityIdentifier",
        ("FIX.5.0", "1057", "Y"): "Y .. OrderInitiatorIsAggressor",
        ("FIX.5.0", "1057", "N"): "N .. OrderInitiatorIsPassive",
        ("FIX.5.0", "1060", "6"): "6 .. UKNationalInsuranceOrPensionNumber",
        ("FIX.5.0", "1060", "7"): "7 .. USSocialSecurityNumber",
        ("FIX.5.0", "1060", "8"): "8 .. USEmployerOrTaxIDNumber",
        ("FIX.5.0", "1060", "9"): "9 .. AustralianBusinessNumber",
        ("FIX.5.0", "1060", "A"): "A .. AustralianTaxFileNumber",
        ("FIX.5.0", "1060", "J"): "J .. TaxID",
        ("FIX.5.0", "1060", "1"): "1 .. KoreanInvestorID",
        ("FIX.5.0", "1060", "2"): "2 .. TaiwaneseForeignInvestorID",
        ("FIX.5.0", "1060", "3"): "3 .. TaiwaneseTradingAcct",
        ("FIX.5.0", "1060", "4"): "4 .. MalaysianCentralDepository",
        ("FIX.5.0", "1060", "5"): "5 .. ChineseInvestorID",
        ("FIX.5.0", "1060", "I"): "I .. ISITCAcronym",
        ("FIX.5.0", "1060", "B"): "B .. BIC",
        ("FIX.5.0", "1060", "C"): "C .. GeneralIdentifier",
        ("FIX.5.0", "1060", "D"): "D .. Proprietary",
        ("FIX.5.0", "1060", "E"): "E .. ISOCountryCode",
        ("FIX.5.0", "1060", "F"): "F .. SettlementEntityLocation",
        ("FIX.5.0", "1060", "G"): "G .. MIC",
        ("FIX.5.0", "1060", "H"): "H .. CSDParticipant",
        ("FIX.5.0", "1060", "K"): "K .. AustralianCompanyNumber",
        ("FIX.5.0", "1060", "L"): "L .. AustralianRegisteredBodyNumber",
        ("FIX.5.0", "1060", "M"): "M .. CFTCReportingFirmIdentifier",
        ("FIX.5.0", "1060", "N"): "N .. LegalEntityIdentifier",
        ("FIX.5.0", "1060", "O"): "O .. InterimIdentifier",
        ("FIX.5.0", "1060", "P"): "P .. ShortCodeIdentifier",
        ("FIX.5.0", "1060", "Q"): "Q .. NationalIDNaturalPerson",
        ("FIX.5.0", "1061", "1"): "1 .. ExecutingFirm",
        ("FIX.5.0", "1061", "2"): "2 .. BrokerOfCredit",
        ("FIX.5.0", "1061", "3"): "3 .. ClientID",
        ("FIX.5.0", "1061", "4"): "4 .. ClearingFirm",
        ("FIX.5.0", "1061", "5"): "5 .. InvestorID",
        ("FIX.5.0", "1061", "6"): "6 .. IntroducingFirm",
        ("FIX.5.0", "1061", "7"): "7 .. EnteringFirm",
        ("FIX.5.0", "1061", "8"): "8 .. Locate",
        ("FIX.5.0", "1061", "9"): "9 .. FundManagerClientID",
        ("FIX.5.0", "1061", "10"): "10 .. SettlementLocation",
        ("FIX.5.0", "1061", "11"): "11 .. OrderOriginationTrader",
        ("FIX.5.0", "1061", "12"): "12 .. ExecutingTrader",
        ("FIX.5.0", "1061", "13"): "13 .. OrderOriginationFirm",
        ("FIX.5.0", "1061", "15"): "15 .. CorrespondantClearingFirm",
        ("FIX.5.0", "1061", "16"): "16 .. ExecutingSystem",
        ("FIX.5.0", "1061", "17"): "17 .. ContraFirm",
        ("FIX.5.0", "1061", "18"): "18 .. ContraClearingFirm",
        ("FIX.5.0", "1061", "19"): "19 .. SponsoringFirm",
        ("FIX.5.0", "1061", "20"): "20 .. UnderlyingContraFirm",
        ("FIX.5.0", "1061", "21"): "21 .. ClearingOrganization",
        ("FIX.5.0", "1061", "22"): "22 .. Exchange",
        ("FIX.5.0", "1061", "24"): "24 .. CustomerAccount",
        ("FIX.5.0", "1061", "25"): "25 .. CorrespondentClearingOrganization",
        ("FIX.5.0", "1061", "26"): "26 .. CorrespondentBroker",
        ("FIX.5.0", "1061", "27"): "27 .. Buyer",
        ("FIX.5.0", "1061", "28"): "28 .. Custodian",
        ("FIX.5.0", "1061", "29"): "29 .. Intermediary",
        ("FIX.5.0", "1061", "30"): "30 .. Agent",
        ("FIX.5.0", "1061", "31"): "31 .. SubCustodian",
        ("FIX.5.0", "1061", "32"): "32 .. Beneficiary",
        ("FIX.5.0", "1061", "33"): "33 .. InterestedParty",
        ("FIX.5.0", "1061", "34"): "34 .. RegulatoryBody",
        ("FIX.5.0", "1061", "35"): "35 .. LiquidityProvider",
        ("FIX.5.0", "1061", "36"): "36 .. EnteringTrader",
        ("FIX.5.0", "1061", "37"): "37 .. ContraTrader",
        ("FIX.5.0", "1061", "38"): "38 .. PositionAccount",
        ("FIX.5.0", "1061", "39"): "39 .. ContraInvestorID",
        ("FIX.5.0", "1061", "40"): "40 .. TransferToFirm",
        ("FIX.5.0", "1061", "41"): "41 .. ContraPositionAccount",
        ("FIX.5.0", "1061", "42"): "42 .. ContraExchange",
        ("FIX.5.0", "1061", "43"): "43 .. InternalCarryAccount",
        ("FIX.5.0", "1061", "44"): "44 .. OrderEntryOperatorID",
        ("FIX.5.0", "1061", "45"): "45 .. SecondaryAccountNumber",
        ("FIX.5.0", "1061", "46"): "46 .. ForeignFirm",
        ("FIX.5.0", "1061", "47"): "47 .. ThirdPartyAllocationFirm",
        ("FIX.5.0", "1061", "48"): "48 .. ClaimingAccount",
        ("FIX.5.0", "1061", "49"): "49 .. AssetManager",
        ("FIX.5.0", "1061", "50"): "50 .. PledgorAccount",
        ("FIX.5.0", "1061", "51"): "51 .. PledgeeAccount",
        ("FIX.5.0", "1061", "52"): "52 .. LargeTraderReportableAccount",
        ("FIX.5.0", "1061", "53"): "53 .. TraderMnemonic",
        ("FIX.5.0", "1061", "54"): "54 .. SenderLocation",
        ("FIX.5.0", "1061", "55"): "55 .. SessionID",
        ("FIX.5.0", "1061", "56"): "56 .. AcceptableCounterparty",
        ("FIX.5.0", "1061", "57"): "57 .. UnacceptableCounterparty",
        ("FIX.5.0", "1061", "58"): "58 .. EnteringUnit",
        ("FIX.5.0", "1061", "59"): "59 .. ExecutingUnit",
        ("FIX.5.0", "1061", "60"): "60 .. IntroducingBroker",
        ("FIX.5.0", "1061", "61"): "61 .. QuoteOriginator",
        ("FIX.5.0", "1061", "62"): "62 .. ReportOriginator",
        ("FIX.5.0", "1061", "63"): "63 .. SystematicInternaliser",
        ("FIX.5.0", "1061", "64"): "64 .. MultilateralTradingFacility",
        ("FIX.5.0", "1061", "65"): "65 .. RegulatedMarket",
        ("FIX.5.0", "1061", "66"): "66 .. MarketMaker",
        ("FIX.5.0", "1061", "67"): "67 .. InvestmentFirm",
        ("FIX.5.0", "1061", "68"): "68 .. HostCompetentAuthority",
        ("FIX.5.0", "1061", "69"): "69 .. HomeCompetentAuthority",
        ("FIX.5.0", "1061", "70"): "70 .. CompetentAuthorityLiquidity",
        ("FIX.5.0", "1061", "71"): "71 .. CompetentAuthorityTransactionVenue",
        ("FIX.5.0", "1061", "72"): "72 .. ReportingIntermediary",
        ("FIX.5.0", "1061", "73"): "73 .. ExecutionVenue",
        ("FIX.5.0", "1061", "74"): "74 .. MarketDataEntryOriginator",
        ("FIX.5.0", "1061", "75"): "75 .. LocationID",
        ("FIX.5.0", "1061", "76"): "76 .. DeskID",
        ("FIX.5.0", "1061", "77"): "77 .. MarketDataMarket",
        ("FIX.5.0", "1061", "78"): "78 .. AllocationEntity",
        ("FIX.5.0", "1061", "79"): "79 .. PrimeBroker",
        ("FIX.5.0", "1061", "80"): "80 .. StepOutFirm",
        ("FIX.5.0", "1061", "81"): "81 .. BrokerClearingID",
        ("FIX.5.0", "1061", "82"): "82 .. CentralRegistrationDepository",
        ("FIX.5.0", "1061", "83"): "83 .. ClearingAccount",
        ("FIX.5.0", "1061", "84"): "84 .. AcceptableSettlingCounterparty",
        ("FIX.5.0", "1061", "85"): "85 .. UnacceptableSettlingCounterparty",
        ("FIX.5.0", "1061", "86"): "86 .. CLSMemberBank",
        ("FIX.5.0", "1061", "87"): "87 .. InConcertGroup",
        ("FIX.5.0", "1061", "88"): "88 .. InConcertControllingEntity",
        ("FIX.5.0", "1061", "89"): "89 .. LargePositionsReportingAccount",
        ("FIX.5.0", "1061", "90"): "90 .. SettlementFirm",
        ("FIX.5.0", "1061", "91"): "91 .. SettlementAccount",
        ("FIX.5.0", "1061", "92"): "92 .. ReportingMarketCenter",
        ("FIX.5.0", "1061", "93"): "93 .. RelatedReportingMarketCenter",
        ("FIX.5.0", "1061", "94"): "94 .. AwayMarket",
        ("FIX.5.0", "1061", "95"): "95 .. GiveupTradingFirm",
        ("FIX.5.0", "1061", "96"): "96 .. TakeupTradingFirm",
        ("FIX.5.0", "1061", "97"): "97 .. GiveupClearingFirm",
        ("FIX.5.0", "1061", "98"): "98 .. TakeupClearingFirm",
        ("FIX.5.0", "1061", "99"): "99 .. OriginatingMarket",
        ("FIX.5.0", "1061", "100"): "100 .. MarginAccount",
        ("FIX.5.0", "1061", "101"): "101 .. CollateralAssetAccount",
        ("FIX.5.0", "1061", "102"): "102 .. DataRepository",
        ("FIX.5.0", "1061", "103"): "103 .. CalculationAgent",
        ("FIX.5.0", "1061", "104"): "104 .. ExerciseNoticeSender",
        ("FIX.5.0", "1061", "105"): "105 .. ExerciseNoticeReceiver",
        ("FIX.5.0", "1061", "106"): "106 .. RateReferenceBank",
        ("FIX.5.0", "1061", "107"): "107 .. Correspondent",
        ("FIX.5.0", "1061", "109"): "109 .. BeneficiaryBank",
        ("FIX.5.0", "1061", "110"): "110 .. Borrower",
        ("FIX.5.0", "1061", "111"): "111 .. PrimaryObligator",
        ("FIX.5.0", "1061", "112"): "112 .. Guarantor",
        ("FIX.5.0", "1061", "113"): "113 .. ExcludedReferenceEntity",
        ("FIX.5.0", "1061", "114"): "114 .. DeterminingParty",
        ("FIX.5.0", "1061", "115"): "115 .. HedgingParty",
        ("FIX.5.0", "1061", "116"): "116 .. ReportingEntity",
        ("FIX.5.0", "1061", "117"): "117 .. SalesPerson",
        ("FIX.5.0", "1061", "118"): "118 .. Operator",
        ("FIX.5.0", "1061", "119"): "119 .. CSD",
        ("FIX.5.0", "1061", "120"): "120 .. ICSD",
        ("FIX.5.0", "1061", "121"): "121 .. TradingSubAccount",
        ("FIX.5.0", "1061", "122"): "122 .. InvestmentDecisionMaker",
        ("FIX.5.0", "1061", "123"): "123 .. PublishingIntermediary",
        ("FIX.5.0", "1064", "1"): "1 .. Firm",
        ("FIX.5.0", "1064", "2"): "2 .. Person",
        ("FIX.5.0", "1064", "3"): "3 .. System",
        ("FIX.5.0", "1064", "4"): "4 .. Application",
        ("FIX.5.0", "1064", "5"): "5 .. FullLegalNameOfFirm",
        ("FIX.5.0", "1064", "6"): "6 .. PostalAddress",
        ("FIX.5.0", "1064", "7"): "7 .. PhoneNumber",
        ("FIX.5.0", "1064", "8"): "8 .. EmailAddress",
        ("FIX.5.0", "1064", "9"): "9 .. ContactName",
        ("FIX.5.0", "1064", "10"): "10 .. SecuritiesAccountNumber",
        ("FIX.5.0", "1064", "11"): "11 .. RegistrationNumber",
        ("FIX.5.0", "1064", "12"): "12 .. RegisteredAddressForConfirmation",
        ("FIX.5.0", "1064", "13"): "13 .. RegulatoryStatus",
        ("FIX.5.0", "1064", "14"): "14 .. RegistrationName",
        ("FIX.5.0", "1064", "15"): "15 .. CashAccountNumber",
        ("FIX.5.0", "1064", "16"): "16 .. BIC",
        ("FIX.5.0", "1064", "17"): "17 .. CSDParticipantMemberCode",
        ("FIX.5.0", "1064", "18"): "18 .. RegisteredAddress",
        ("FIX.5.0", "1064", "19"): "19 .. FundAccountName",
        ("FIX.5.0", "1064", "20"): "20 .. TelexNumber",
        ("FIX.5.0", "1064", "21"): "21 .. FaxNumber",
        ("FIX.5.0", "1064", "22"): "22 .. SecuritiesAccountName",
        ("FIX.5.0", "1064", "23"): "23 .. CashAccountName",
        ("FIX.5.0", "1064", "24"): "24 .. Department",
        ("FIX.5.0", "1064", "25"): "25 .. LocationDesk",
        ("FIX.5.0", "1064", "26"): "26 .. PositionAccountType",
        ("FIX.5.0", "1064", "27"): "27 .. SecurityLocateID",
        ("FIX.5.0", "1064", "28"): "28 .. MarketMaker",
        ("FIX.5.0", "1064", "29"): "29 .. EligibleCounterparty",
        ("FIX.5.0", "1064", "30"): "30 .. ProfessionalClient",
        ("FIX.5.0", "1064", "31"): "31 .. Location",
        ("FIX.5.0", "1064", "32"): "32 .. ExecutionVenue",
        ("FIX.5.0", "1064", "33"): "33 .. CurrencyDeliveryIdentifier",
        ("FIX.5.0", "1064", "34"): "34 .. AddressCity",
        ("FIX.5.0", "1064", "35"): "35 .. AddressStateOrProvince",
        ("FIX.5.0", "1064", "36"): "36 .. AddressPostalCode",
        ("FIX.5.0", "1064", "37"): "37 .. AddressStreet",
        ("FIX.5.0", "1064", "38"): "38 .. AddressISOCountryCode",
        ("FIX.5.0", "1064", "39"): "39 .. ISOCountryCode",
        ("FIX.5.0", "1064", "40"): "40 .. MarketSegment",
        ("FIX.5.0", "1064", "41"): "41 .. CustomerAccountType",
        ("FIX.5.0", "1064", "42"): "42 .. OmnibusAccount",
        ("FIX.5.0", "1064", "43"): "43 .. FundsSegregationType",
        ("FIX.5.0", "1064", "44"): "44 .. GuaranteeFund",
        ("FIX.5.0", "1064", "45"): "45 .. SwapDealer",
        ("FIX.5.0", "1064", "46"): "46 .. MajorParticipant",
        ("FIX.5.0", "1064", "47"): "47 .. FinancialEntity",
        ("FIX.5.0", "1064", "48"): "48 .. USPerson",
        ("FIX.5.0", "1064", "49"): "49 .. ReportingEntityIndicator",
        ("FIX.5.0", "1064", "50"): "50 .. ElectedClearingRequirementException",
        ("FIX.5.0", "1064", "51"): "51 .. BusinessCenter",
        ("FIX.5.0", "1064", "52"): "52 .. ReferenceText",
        ("FIX.5.0", "1064", "53"): "53 .. ShortMarkingExemptAccount",
        ("FIX.5.0", "1064", "54"): "54 .. ParentFirmIdentifier",
        ("FIX.5.0", "1064", "55"): "55 .. ParentFirmName",
        ("FIX.5.0", "1064", "56"): "56 .. DealIdentifier",
        ("FIX.5.0", "1064", "57"): "57 .. SystemTradeID",
        ("FIX.5.0", "1064", "58"): "58 .. SystemTradeSubID",
        ("FIX.5.0", "1064", "59"): "59 .. FCMCode",
        ("FIX.5.0", "1064", "60"): "60 .. DlvryTrmlCode",
        ("FIX.5.0", "1064", "61"): "61 .. VolntyRptEntity",
        ("FIX.5.0", "1064", "62"): "62 .. RptObligJursdctn",
        ("FIX.5.0", "1064", "63"): "63 .. VolntyRptJursdctn",
        ("FIX.5.0", "1064", "64"): "64 .. CompanyActivities",
        ("FIX.5.0", "1064", "65"): "65 .. EEAreaDomiciled",
        ("FIX.5.0", "1064", "66"): "66 .. ContractLinked",
        ("FIX.5.0", "1064", "67"): "67 .. ContractAbove",
        ("FIX.5.0", "1064", "68"): "68 .. VolntyRptPty",
        ("FIX.5.0", "1064", "69"): "69 .. EndUser",
        ("FIX.5.0", "1064", "70"): "70 .. LocationOrJurisdiction",
        ("FIX.5.0", "1064", "71"): "71 .. DerivativesDealer",
        ("FIX.5.0", "1064", "72"): "72 .. Domicile",
        ("FIX.5.0", "1064", "73"): "73 .. ExemptFromRecognition",
        ("FIX.5.0", "1064", "74"): "74 .. Payer",
        ("FIX.5.0", "1064", "75"): "75 .. Receiver",
        ("FIX.5.0", "1064", "76"): "76 .. SystematicInternaliser",
        ("FIX.5.0", "1064", "77"): "77 .. PublishingEntityIndicator",
        ("FIX.5.0", "1064", "78"): "78 .. FirstName",
        ("FIX.5.0", "1064", "79"): "79 .. Surname",
        ("FIX.5.0", "1064", "80"): "80 .. DateOfBirth",
        ("FIX.5.0", "1064", "81"): "81 .. OrderTransmittingFirm",
        ("FIX.5.0", "1064", "82"): "82 .. OrderTransmittingFirmBuyer",
        ("FIX.5.0", "1064", "83"): "83 .. OrderTransmitterSeller",
        ("FIX.5.0", "1064", "84"): "84 .. LegalEntityIdentifier",
        ("FIX.5.0", "1070", "0"): "0 .. Indicative",
        ("FIX.5.0", "1070", "1"): "1 .. Tradeable",
        ("FIX.5.0", "1070", "2"): "2 .. RestrictedTradeable",
        ("FIX.5.0", "1070", "3"): "3 .. Counter",
        ("FIX.5.0", "1070", "4"): "4 .. IndicativeAndTradeable",
        ("FIX.5.0", "1081", "0"): "0 .. SecondaryOrderID",
        ("FIX.5.0", "1081", "1"): "1 .. OrderID",
        ("FIX.5.0", "1081", "2"): "2 .. MDEntryID",
        ("FIX.5.0", "1081", "3"): "3 .. QuoteEntryID",
        ("FIX.5.0", "1081", "4"): "4 .. OriginalOrderID",
        ("FIX.5.0", "1081", "5"): "5 .. QuoteID",
        ("FIX.5.0", "1081", "6"): "6 .. QuoteReqID",
        ("FIX.5.0", "1083", "1"): "1 .. Immediate",
        ("FIX.5.0", "1083", "2"): "2 .. Exhaust",
        ("FIX.5.0", "1084", "1"): "1 .. Initial",
        ("FIX.5.0", "1084", "2"): "2 .. New",
        ("FIX.5.0", "1084", "3"): "3 .. Random",
        ("FIX.5.0", "1084", "4"): "4 .. Undisclosed",
        ("FIX.5.0", "1092", "0"): "0 .. None",
        ("FIX.5.0", "1092", "1"): "1 .. Local",
        ("FIX.5.0", "1092", "2"): "2 .. National",
        ("FIX.5.0", "1092", "3"): "3 .. Global",
        ("FIX.5.0", "1093", "1"): "1 .. OddLot",
        ("FIX.5.0", "1093", "2"): "2 .. RoundLot",
        ("FIX.5.0", "1093", "3"): "3 .. BlockLot",
        ("FIX.5.0", "1093", "4"): "4 .. RoundLotBasedUpon",
        ("FIX.5.0", "1094", "1"): "1 .. LastPeg",
        ("FIX.5.0", "1094", "2"): "2 .. MidPricePeg",
        ("FIX.5.0", "1094", "3"): "3 .. OpeningPeg",
        ("FIX.5.0", "1094", "4"): "4 .. MarketPeg",
        ("FIX.5.0", "1094", "5"): "5 .. PrimaryPeg",
        ("FIX.5.0", "1094", "7"): "7 .. PegToVWAP",
        ("FIX.5.0", "1094", "8"): "8 .. TrailingStopPeg",
        ("FIX.5.0", "1094", "9"): "9 .. PegToLimitPrice",
        ("FIX.5.0", "1094", "10"): "10 .. ShortSaleMinPricePeg",
        ("FIX.5.0", "1096", "1"): "1 .. CUSIP",
        ("FIX.5.0", "1096", "2"): "2 .. SEDOL",
        ("FIX.5.0", "1096", "3"): "3 .. QUIK",
        ("FIX.5.0", "1096", "4"): "4 .. ISINNumber",
        ("FIX.5.0", "1096", "5"): "5 .. RICCode",
        ("FIX.5.0", "1096", "6"): "6 .. ISOCurrencyCode",
        ("FIX.5.0", "1096", "7"): "7 .. ISOCountryCode",
        ("FIX.5.0", "1096", "8"): "8 .. ExchangeSymbol",
        ("FIX.5.0", "1096", "9"): "9 .. ConsolidatedTapeAssociation",
        ("FIX.5.0", "1096", "A"): "A .. BloombergSymbol",
        ("FIX.5.0", "1096", "B"): "B .. Wertpapier",
        ("FIX.5.0", "1096", "C"): "C .. Dutch",
        ("FIX.5.0", "1096", "D"): "D .. Valoren",
        ("FIX.5.0", "1096", "E"): "E .. Sicovam",
        ("FIX.5.0", "1096", "F"): "F .. Belgian",
        ("FIX.5.0", "1096", "G"): "G .. Common",
        ("FIX.5.0", "1096", "H"): "H .. ClearingHouse",
        ("FIX.5.0", "1096", "I"): "I .. ISDAFpMLSpecification",
        ("FIX.5.0", "1096", "J"): "J .. OptionPriceReportingAuthority",
        ("FIX.5.0", "1096", "K"): "K .. ISDAFpMLURL",
        ("FIX.5.0", "1096", "L"): "L .. LetterOfCredit",
        ("FIX.5.0", "1096", "M"): "M .. MarketplaceAssignedIdentifier",
        ("FIX.5.0", "1096", "N"): "N .. MarkitREDEntityCLIP",
        ("FIX.5.0", "1096", "P"): "P .. MarkitREDPairCLIP",
        ("FIX.5.0", "1096", "Q"): "Q .. CFTCCommodityCode",
        ("FIX.5.0", "1096", "R"): "R .. ISDACommodityReferencePrice",
        ("FIX.5.0", "1096", "S"): "S .. FinancialInstrumentGlobalIdentifier",
        ("FIX.5.0", "1096", "T"): "T .. LegalEntityIdentifier",
        ("FIX.5.0", "1096", "U"): "U .. Synthetic",
        ("FIX.5.0", "1096", "V"): "V .. FidessaInstrumentMnemonic",
        ("FIX.5.0", "1096", "W"): "W .. IndexName",
        ("FIX.5.0", "1100", "1"): "1 .. PartialExecution",
        ("FIX.5.0", "1100", "2"): "2 .. SpecifiedTradingSession",
        ("FIX.5.0", "1100", "3"): "3 .. NextAuction",
        ("FIX.5.0", "1100", "4"): "4 .. PriceMovement",
        ("FIX.5.0", "1100", "5"): "5 .. OnOrderEntryOrModification",
        ("FIX.5.0", "1101", "1"): "1 .. Activate",
        ("FIX.5.0", "1101", "2"): "2 .. Modify",
        ("FIX.5.0", "1101", "3"): "3 .. Cancel",
        ("FIX.5.0", "1105", "1"): "1 .. CUSIP",
        ("FIX.5.0", "1105", "2"): "2 .. SEDOL",
        ("FIX.5.0", "1105", "3"): "3 .. QUIK",
        ("FIX.5.0", "1105", "4"): "4 .. ISINNumber",
        ("FIX.5.0", "1105", "5"): "5 .. RICCode",
        ("FIX.5.0", "1105", "6"): "6 .. ISOCurrencyCode",
        ("FIX.5.0", "1105", "7"): "7 .. ISOCountryCode",
        ("FIX.5.0", "1105", "8"): "8 .. ExchangeSymbol",
        ("FIX.5.0", "1105", "9"): "9 .. ConsolidatedTapeAssociation",
        ("FIX.5.0", "1105", "A"): "A .. BloombergSymbol",
        ("FIX.5.0", "1105", "B"): "B .. Wertpapier",
        ("FIX.5.0", "1105", "C"): "C .. Dutch",
        ("FIX.5.0", "1105", "D"): "D .. Valoren",
        ("FIX.5.0", "1105", "E"): "E .. Sicovam",
        ("FIX.5.0", "1105", "F"): "F .. Belgian",
        ("FIX.5.0", "1105", "G"): "G .. Common",
        ("FIX.5.0", "1105", "H"): "H .. ClearingHouse",
        ("FIX.5.0", "1105", "I"): "I .. ISDAFpMLSpecification",
        ("FIX.5.0", "1105", "J"): "J .. OptionPriceReportingAuthority",
        ("FIX.5.0", "1105", "K"): "K .. ISDAFpMLURL",
        ("FIX.5.0", "1105", "L"): "L .. LetterOfCredit",
        ("FIX.5.0", "1105", "M"): "M .. MarketplaceAssignedIdentifier",
        ("FIX.5.0", "1105", "N"): "N .. MarkitREDEntityCLIP",
        ("FIX.5.0", "1105", "P"): "P .. MarkitREDPairCLIP",
        ("FIX.5.0", "1105", "Q"): "Q .. CFTCCommodityCode",
        ("FIX.5.0", "1105", "R"): "R .. ISDACommodityReferencePrice",
        ("FIX.5.0", "1105", "S"): "S .. FinancialInstrumentGlobalIdentifier",
        ("FIX.5.0", "1105", "T"): "T .. LegalEntityIdentifier",
        ("FIX.5.0", "1105", "U"): "U .. Synthetic",
        ("FIX.5.0", "1105", "V"): "V .. FidessaInstrumentMnemonic",
        ("FIX.5.0", "1105", "W"): "W .. IndexName",
        ("FIX.5.0", "1107", "1"): "1 .. BestOffer",
        ("FIX.5.0", "1107", "2"): "2 .. LastTrade",
        ("FIX.5.0", "1107", "3"): "3 .. BestBid",
        ("FIX.5.0", "1107", "4"): "4 .. BestBidOrLastTrade",
        ("FIX.5.0", "1107", "5"): "5 .. BestOfferOrLastTrade",
        ("FIX.5.0", "1107", "6"): "6 .. BestMid",
        ("FIX.5.0", "1108", "0"): "0 .. None",
        ("FIX.5.0", "1108", "1"): "1 .. Local",
        ("FIX.5.0", "1108", "2"): "2 .. National",
        ("FIX.5.0", "1108", "3"): "3 .. Global",
        ("FIX.5.0", "1109", "U"): "U .. Up",
        ("FIX.5.0", "1109", "D"): "D .. Down",
        ("FIX.5.0", "1111", "1"): "1 .. Market",
        ("FIX.5.0", "1111", "2"): "2 .. Limit",
        ("FIX.5.0", "1115", "1"): "1 .. Order",
        ("FIX.5.0", "1115", "2"): "2 .. Quote",
        ("FIX.5.0", "1115", "3"): "3 .. PrivatelyNegotiatedTrade",
        ("FIX.5.0", "1115", "4"): "4 .. MultilegOrder",
        ("FIX.5.0", "1115", "5"): "5 .. LinkedOrder",
        ("FIX.5.0", "1115", "6"): "6 .. QuoteRequest",
        ("FIX.5.0", "1115", "7"): "7 .. ImpliedOrder",
        ("FIX.5.0", "1115", "8"): "8 .. CrossOrder",
        ("FIX.5.0", "1115", "9"): "9 .. StreamingPrice",
        ("FIX.5.0", "1115", "A"): "A .. InternalCrossOrder",
        ("FIX.5.0", "1118", "6"): "6 .. UKNationalInsuranceOrPensionNumber",
        ("FIX.5.0", "1118", "7"): "7 .. USSocialSecurityNumber",
        ("FIX.5.0", "1118", "8"): "8 .. USEmployerOrTaxIDNumber",
        ("FIX.5.0", "1118", "9"): "9 .. AustralianBusinessNumber",
        ("FIX.5.0", "1118", "A"): "A .. AustralianTaxFileNumber",
        ("FIX.5.0", "1118", "J"): "J .. TaxID",
        ("FIX.5.0", "1118", "1"): "1 .. KoreanInvestorID",
        ("FIX.5.0", "1118", "2"): "2 .. TaiwaneseForeignInvestorID",
        ("FIX.5.0", "1118", "3"): "3 .. TaiwaneseTradingAcct",
        ("FIX.5.0", "1118", "4"): "4 .. MalaysianCentralDepository",
        ("FIX.5.0", "1118", "5"): "5 .. ChineseInvestorID",
        ("FIX.5.0", "1118", "I"): "I .. ISITCAcronym",
        ("FIX.5.0", "1118", "B"): "B .. BIC",
        ("FIX.5.0", "1118", "C"): "C .. GeneralIdentifier",
        ("FIX.5.0", "1118", "D"): "D .. Proprietary",
        ("FIX.5.0", "1118", "E"): "E .. ISOCountryCode",
        ("FIX.5.0", "1118", "F"): "F .. SettlementEntityLocation",
        ("FIX.5.0", "1118", "G"): "G .. MIC",
        ("FIX.5.0", "1118", "H"): "H .. CSDParticipant",
        ("FIX.5.0", "1118", "K"): "K .. AustralianCompanyNumber",
        ("FIX.5.0", "1118", "L"): "L .. AustralianRegisteredBodyNumber",
        ("FIX.5.0", "1118", "M"): "M .. CFTCReportingFirmIdentifier",
        ("FIX.5.0", "1118", "N"): "N .. LegalEntityIdentifier",
        ("FIX.5.0", "1118", "O"): "O .. InterimIdentifier",
        ("FIX.5.0", "1118", "P"): "P .. ShortCodeIdentifier",
        ("FIX.5.0", "1118", "Q"): "Q .. NationalIDNaturalPerson",
        ("FIX.5.0", "1119", "1"): "1 .. ExecutingFirm",
        ("FIX.5.0", "1119", "2"): "2 .. BrokerOfCredit",
        ("FIX.5.0", "1119", "3"): "3 .. ClientID",
        ("FIX.5.0", "1119", "4"): "4 .. ClearingFirm",
        ("FIX.5.0", "1119", "5"): "5 .. InvestorID",
        ("FIX.5.0", "1119", "6"): "6 .. IntroducingFirm",
        ("FIX.5.0", "1119", "7"): "7 .. EnteringFirm",
        ("FIX.5.0", "1119", "8"): "8 .. Locate",
        ("FIX.5.0", "1119", "9"): "9 .. FundManagerClientID",
        ("FIX.5.0", "1119", "10"): "10 .. SettlementLocation",
        ("FIX.5.0", "1119", "11"): "11 .. OrderOriginationTrader",
        ("FIX.5.0", "1119", "12"): "12 .. ExecutingTrader",
        ("FIX.5.0", "1119", "13"): "13 .. OrderOriginationFirm",
        ("FIX.5.0", "1119", "15"): "15 .. CorrespondantClearingFirm",
        ("FIX.5.0", "1119", "16"): "16 .. ExecutingSystem",
        ("FIX.5.0", "1119", "17"): "17 .. ContraFirm",
        ("FIX.5.0", "1119", "18"): "18 .. ContraClearingFirm",
        ("FIX.5.0", "1119", "19"): "19 .. SponsoringFirm",
        ("FIX.5.0", "1119", "20"): "20 .. UnderlyingContraFirm",
        ("FIX.5.0", "1119", "21"): "21 .. ClearingOrganization",
        ("FIX.5.0", "1119", "22"): "22 .. Exchange",
        ("FIX.5.0", "1119", "24"): "24 .. CustomerAccount",
        ("FIX.5.0", "1119", "25"): "25 .. CorrespondentClearingOrganization",
        ("FIX.5.0", "1119", "26"): "26 .. CorrespondentBroker",
        ("FIX.5.0", "1119", "27"): "27 .. Buyer",
        ("FIX.5.0", "1119", "28"): "28 .. Custodian",
        ("FIX.5.0", "1119", "29"): "29 .. Intermediary",
        ("FIX.5.0", "1119", "30"): "30 .. Agent",
        ("FIX.5.0", "1119", "31"): "31 .. SubCustodian",
        ("FIX.5.0", "1119", "32"): "32 .. Beneficiary",
        ("FIX.5.0", "1119", "33"): "33 .. InterestedParty",
        ("FIX.5.0", "1119", "34"): "34 .. RegulatoryBody",
        ("FIX.5.0", "1119", "35"): "35 .. LiquidityProvider",
        ("FIX.5.0", "1119", "36"): "36 .. EnteringTrader",
        ("FIX.5.0", "1119", "37"): "37 .. ContraTrader",
        ("FIX.5.0", "1119", "38"): "38 .. PositionAccount",
        ("FIX.5.0", "1119", "39"): "39 .. ContraInvestorID",
        ("FIX.5.0", "1119", "40"): "40 .. TransferToFirm",
        ("FIX.5.0", "1119", "41"): "41 .. ContraPositionAccount",
        ("FIX.5.0", "1119", "42"): "42 .. ContraExchange",
        ("FIX.5.0", "1119", "43"): "43 .. InternalCarryAccount",
        ("FIX.5.0", "1119", "44"): "44 .. OrderEntryOperatorID",
        ("FIX.5.0", "1119", "45"): "45 .. SecondaryAccountNumber",
        ("FIX.5.0", "1119", "46"): "46 .. ForeignFirm",
        ("FIX.5.0", "1119", "47"): "47 .. ThirdPartyAllocationFirm",
        ("FIX.5.0", "1119", "48"): "48 .. ClaimingAccount",
        ("FIX.5.0", "1119", "49"): "49 .. AssetManager",
        ("FIX.5.0", "1119", "50"): "50 .. PledgorAccount",
        ("FIX.5.0", "1119", "51"): "51 .. PledgeeAccount",
        ("FIX.5.0", "1119", "52"): "52 .. LargeTraderReportableAccount",
        ("FIX.5.0", "1119", "53"): "53 .. TraderMnemonic",
        ("FIX.5.0", "1119", "54"): "54 .. SenderLocation",
        ("FIX.5.0", "1119", "55"): "55 .. SessionID",
        ("FIX.5.0", "1119", "56"): "56 .. AcceptableCounterparty",
        ("FIX.5.0", "1119", "57"): "57 .. UnacceptableCounterparty",
        ("FIX.5.0", "1119", "58"): "58 .. EnteringUnit",
        ("FIX.5.0", "1119", "59"): "59 .. ExecutingUnit",
        ("FIX.5.0", "1119", "60"): "60 .. IntroducingBroker",
        ("FIX.5.0", "1119", "61"): "61 .. QuoteOriginator",
        ("FIX.5.0", "1119", "62"): "62 .. ReportOriginator",
        ("FIX.5.0", "1119", "63"): "63 .. SystematicInternaliser",
        ("FIX.5.0", "1119", "64"): "64 .. MultilateralTradingFacility",
        ("FIX.5.0", "1119", "65"): "65 .. RegulatedMarket",
        ("FIX.5.0", "1119", "66"): "66 .. MarketMaker",
        ("FIX.5.0", "1119", "67"): "67 .. InvestmentFirm",
        ("FIX.5.0", "1119", "68"): "68 .. HostCompetentAuthority",
        ("FIX.5.0", "1119", "69"): "69 .. HomeCompetentAuthority",
        ("FIX.5.0", "1119", "70"): "70 .. CompetentAuthorityLiquidity",
        ("FIX.5.0", "1119", "71"): "71 .. CompetentAuthorityTransactionVenue",
        ("FIX.5.0", "1119", "72"): "72 .. ReportingIntermediary",
        ("FIX.5.0", "1119", "73"): "73 .. ExecutionVenue",
        ("FIX.5.0", "1119", "74"): "74 .. MarketDataEntryOriginator",
        ("FIX.5.0", "1119", "75"): "75 .. LocationID",
        ("FIX.5.0", "1119", "76"): "76 .. DeskID",
        ("FIX.5.0", "1119", "77"): "77 .. MarketDataMarket",
        ("FIX.5.0", "1119", "78"): "78 .. AllocationEntity",
        ("FIX.5.0", "1119", "79"): "79 .. PrimeBroker",
        ("FIX.5.0", "1119", "80"): "80 .. StepOutFirm",
        ("FIX.5.0", "1119", "81"): "81 .. BrokerClearingID",
        ("FIX.5.0", "1119", "82"): "82 .. CentralRegistrationDepository",
        ("FIX.5.0", "1119", "83"): "83 .. ClearingAccount",
        ("FIX.5.0", "1119", "84"): "84 .. AcceptableSettlingCounterparty",
        ("FIX.5.0", "1119", "85"): "85 .. UnacceptableSettlingCounterparty",
        ("FIX.5.0", "1119", "86"): "86 .. CLSMemberBank",
        ("FIX.5.0", "1119", "87"): "87 .. InConcertGroup",
        ("FIX.5.0", "1119", "88"): "88 .. InConcertControllingEntity",
        ("FIX.5.0", "1119", "89"): "89 .. LargePositionsReportingAccount",
        ("FIX.5.0", "1119", "90"): "90 .. SettlementFirm",
        ("FIX.5.0", "1119", "91"): "91 .. SettlementAccount",
        ("FIX.5.0", "1119", "92"): "92 .. ReportingMarketCenter",
        ("FIX.5.0", "1119", "93"): "93 .. RelatedReportingMarketCenter",
        ("FIX.5.0", "1119", "94"): "94 .. AwayMarket",
        ("FIX.5.0", "1119", "95"): "95 .. GiveupTradingFirm",
        ("FIX.5.0", "1119", "96"): "96 .. TakeupTradingFirm",
        ("FIX.5.0", "1119", "97"): "97 .. GiveupClearingFirm",
        ("FIX.5.0", "1119", "98"): "98 .. TakeupClearingFirm",
        ("FIX.5.0", "1119", "99"): "99 .. OriginatingMarket",
        ("FIX.5.0", "1119", "100"): "100 .. MarginAccount",
        ("FIX.5.0", "1119", "101"): "101 .. CollateralAssetAccount",
        ("FIX.5.0", "1119", "102"): "102 .. DataRepository",
        ("FIX.5.0", "1119", "103"): "103 .. CalculationAgent",
        ("FIX.5.0", "1119", "104"): "104 .. ExerciseNoticeSender",
        ("FIX.5.0", "1119", "105"): "105 .. ExerciseNoticeReceiver",
        ("FIX.5.0", "1119", "106"): "106 .. RateReferenceBank",
        ("FIX.5.0", "1119", "107"): "107 .. Correspondent",
        ("FIX.5.0", "1119", "109"): "109 .. BeneficiaryBank",
        ("FIX.5.0", "1119", "110"): "110 .. Borrower",
        ("FIX.5.0", "1119", "111"): "111 .. PrimaryObligator",
        ("FIX.5.0", "1119", "112"): "112 .. Guarantor",
        ("FIX.5.0", "1119", "113"): "113 .. ExcludedReferenceEntity",
        ("FIX.5.0", "1119", "114"): "114 .. DeterminingParty",
        ("FIX.5.0", "1119", "115"): "115 .. HedgingParty",
        ("FIX.5.0", "1119", "116"): "116 .. ReportingEntity",
        ("FIX.5.0", "1119", "117"): "117 .. SalesPerson",
        ("FIX.5.0", "1119", "118"): "118 .. Operator",
        ("FIX.5.0", "1119", "119"): "119 .. CSD",
        ("FIX.5.0", "1119", "120"): "120 .. ICSD",
        ("FIX.5.0", "1119", "121"): "121 .. TradingSubAccount",
        ("FIX.5.0", "1119", "122"): "122 .. InvestmentDecisionMaker",
        ("FIX.5.0", "1119", "123"): "123 .. PublishingIntermediary",
        ("FIX.5.0", "1122", "1"): "1 .. Firm",
        ("FIX.5.0", "1122", "2"): "2 .. Person",
        ("FIX.5.0", "1122", "3"): "3 .. System",
        ("FIX.5.0", "1122", "4"): "4 .. Application",
        ("FIX.5.0", "1122", "5"): "5 .. FullLegalNameOfFirm",
        ("FIX.5.0", "1122", "6"): "6 .. PostalAddress",
        ("FIX.5.0", "1122", "7"): "7 .. PhoneNumber",
        ("FIX.5.0", "1122", "8"): "8 .. EmailAddress",
        ("FIX.5.0", "1122", "9"): "9 .. ContactName",
        ("FIX.5.0", "1122", "10"): "10 .. SecuritiesAccountNumber",
        ("FIX.5.0", "1122", "11"): "11 .. RegistrationNumber",
        ("FIX.5.0", "1122", "12"): "12 .. RegisteredAddressForConfirmation",
        ("FIX.5.0", "1122", "13"): "13 .. RegulatoryStatus",
        ("FIX.5.0", "1122", "14"): "14 .. RegistrationName",
        ("FIX.5.0", "1122", "15"): "15 .. CashAccountNumber",
        ("FIX.5.0", "1122", "16"): "16 .. BIC",
        ("FIX.5.0", "1122", "17"): "17 .. CSDParticipantMemberCode",
        ("FIX.5.0", "1122", "18"): "18 .. RegisteredAddress",
        ("FIX.5.0", "1122", "19"): "19 .. FundAccountName",
        ("FIX.5.0", "1122", "20"): "20 .. TelexNumber",
        ("FIX.5.0", "1122", "21"): "21 .. FaxNumber",
        ("FIX.5.0", "1122", "22"): "22 .. SecuritiesAccountName",
        ("FIX.5.0", "1122", "23"): "23 .. CashAccountName",
        ("FIX.5.0", "1122", "24"): "24 .. Department",
        ("FIX.5.0", "1122", "25"): "25 .. LocationDesk",
        ("FIX.5.0", "1122", "26"): "26 .. PositionAccountType",
        ("FIX.5.0", "1122", "27"): "27 .. SecurityLocateID",
        ("FIX.5.0", "1122", "28"): "28 .. MarketMaker",
        ("FIX.5.0", "1122", "29"): "29 .. EligibleCounterparty",
        ("FIX.5.0", "1122", "30"): "30 .. ProfessionalClient",
        ("FIX.5.0", "1122", "31"): "31 .. Location",
        ("FIX.5.0", "1122", "32"): "32 .. ExecutionVenue",
        ("FIX.5.0", "1122", "33"): "33 .. CurrencyDeliveryIdentifier",
        ("FIX.5.0", "1122", "34"): "34 .. AddressCity",
        ("FIX.5.0", "1122", "35"): "35 .. AddressStateOrProvince",
        ("FIX.5.0", "1122", "36"): "36 .. AddressPostalCode",
        ("FIX.5.0", "1122", "37"): "37 .. AddressStreet",
        ("FIX.5.0", "1122", "38"): "38 .. AddressISOCountryCode",
        ("FIX.5.0", "1122", "39"): "39 .. ISOCountryCode",
        ("FIX.5.0", "1122", "40"): "40 .. MarketSegment",
        ("FIX.5.0", "1122", "41"): "41 .. CustomerAccountType",
        ("FIX.5.0", "1122", "42"): "42 .. OmnibusAccount",
        ("FIX.5.0", "1122", "43"): "43 .. FundsSegregationType",
        ("FIX.5.0", "1122", "44"): "44 .. GuaranteeFund",
        ("FIX.5.0", "1122", "45"): "45 .. SwapDealer",
        ("FIX.5.0", "1122", "46"): "46 .. MajorParticipant",
        ("FIX.5.0", "1122", "47"): "47 .. FinancialEntity",
        ("FIX.5.0", "1122", "48"): "48 .. USPerson",
        ("FIX.5.0", "1122", "49"): "49 .. ReportingEntityIndicator",
        ("FIX.5.0", "1122", "50"): "50 .. ElectedClearingRequirementException",
        ("FIX.5.0", "1122", "51"): "51 .. BusinessCenter",
        ("FIX.5.0", "1122", "52"): "52 .. ReferenceText",
        ("FIX.5.0", "1122", "53"): "53 .. ShortMarkingExemptAccount",
        ("FIX.5.0", "1122", "54"): "54 .. ParentFirmIdentifier",
        ("FIX.5.0", "1122", "55"): "55 .. ParentFirmName",
        ("FIX.5.0", "1122", "56"): "56 .. DealIdentifier",
        ("FIX.5.0", "1122", "57"): "57 .. SystemTradeID",
        ("FIX.5.0", "1122", "58"): "58 .. SystemTradeSubID",
        ("FIX.5.0", "1122", "59"): "59 .. FCMCode",
        ("FIX.5.0", "1122", "60"): "60 .. DlvryTrmlCode",
        ("FIX.5.0", "1122", "61"): "61 .. VolntyRptEntity",
        ("FIX.5.0", "1122", "62"): "62 .. RptObligJursdctn",
        ("FIX.5.0", "1122", "63"): "63 .. VolntyRptJursdctn",
        ("FIX.5.0", "1122", "64"): "64 .. CompanyActivities",
        ("FIX.5.0", "1122", "65"): "65 .. EEAreaDomiciled",
        ("FIX.5.0", "1122", "66"): "66 .. ContractLinked",
        ("FIX.5.0", "1122", "67"): "67 .. ContractAbove",
        ("FIX.5.0", "1122", "68"): "68 .. VolntyRptPty",
        ("FIX.5.0", "1122", "69"): "69 .. EndUser",
        ("FIX.5.0", "1122", "70"): "70 .. LocationOrJurisdiction",
        ("FIX.5.0", "1122", "71"): "71 .. DerivativesDealer",
        ("FIX.5.0", "1122", "72"): "72 .. Domicile",
        ("FIX.5.0", "1122", "73"): "73 .. ExemptFromRecognition",
        ("FIX.5.0", "1122", "74"): "74 .. Payer",
        ("FIX.5.0", "1122", "75"): "75 .. Receiver",
        ("FIX.5.0", "1122", "76"): "76 .. SystematicInternaliser",
        ("FIX.5.0", "1122", "77"): "77 .. PublishingEntityIndicator",
        ("FIX.5.0", "1122", "78"): "78 .. FirstName",
        ("FIX.5.0", "1122", "79"): "79 .. Surname",
        ("FIX.5.0", "1122", "80"): "80 .. DateOfBirth",
        ("FIX.5.0", "1122", "81"): "81 .. OrderTransmittingFirm",
        ("FIX.5.0", "1122", "82"): "82 .. OrderTransmittingFirmBuyer",
        ("FIX.5.0", "1122", "83"): "83 .. OrderTransmitterSeller",
        ("FIX.5.0", "1122", "84"): "84 .. LegalEntityIdentifier",
        ("FIX.5.0", "1123", "0"): "0 .. TradeConfirmation",
        ("FIX.5.0", "1123", "1"): "1 .. TwoPartyReport",
        ("FIX.5.0", "1123", "2"): "2 .. OnePartyReportForMatching",
        ("FIX.5.0", "1123", "3"): "3 .. OnePartyReportForPassThrough",
        ("FIX.5.0", "1123", "4"): "4 .. AutomatedFloorOrderRouting",
        ("FIX.5.0", "1123", "5"): "5 .. TwoPartyReportForClaim",
        ("FIX.5.0", "1123", "6"): "6 .. OnePartyReport",
        ("FIX.5.0", "1123", "7"): "7 .. ThirdPtyRptForPassThrough",
        ("FIX.5.0", "1123", "8"): "8 .. OnePartyReportAutoMatch",
        ("FIX.5.0", "1124", "0"): "0 .. TradeConfirmation",
        ("FIX.5.0", "1124", "1"): "1 .. TwoPartyReport",
        ("FIX.5.0", "1124", "2"): "2 .. OnePartyReportForMatching",
        ("FIX.5.0", "1124", "3"): "3 .. OnePartyReportForPassThrough",
        ("FIX.5.0", "1124", "4"): "4 .. AutomatedFloorOrderRouting",
        ("FIX.5.0", "1124", "5"): "5 .. TwoPartyReportForClaim",
        ("FIX.5.0", "1124", "6"): "6 .. OnePartyReport",
        ("FIX.5.0", "1124", "7"): "7 .. ThirdPtyRptForPassThrough",
        ("FIX.5.0", "1124", "8"): "8 .. OnePartyReportAutoMatch",
        ("FIX.5.0", "1128", "0"): "0 .. FIX27",
        ("FIX.5.0", "1128", "1"): "1 .. FIX30",
        ("FIX.5.0", "1128", "2"): "2 .. FIX40",
        ("FIX.5.0", "1128", "3"): "3 .. FIX41",
        ("FIX.5.0", "1128", "4"): "4 .. FIX42",
        ("FIX.5.0", "1128", "5"): "5 .. FIX43",
        ("FIX.5.0", "1128", "6"): "6 .. FIX44",
        ("FIX.5.0", "1128", "7"): "7 .. FIX50",
        ("FIX.5.0", "1128", "8"): "8 .. FIX50SP1",
        ("FIX.5.0", "1128", "9"): "9 .. FIX50SP2",
        ("FIX.5.0", "1130", "0"): "0 .. FIX27",
        ("FIX.5.0", "1130", "1"): "1 .. FIX30",
        ("FIX.5.0", "1130", "2"): "2 .. FIX40",
        ("FIX.5.0", "1130", "3"): "3 .. FIX41",
        ("FIX.5.0", "1130", "4"): "4 .. FIX42",
        ("FIX.5.0", "1130", "5"): "5 .. FIX43",
        ("FIX.5.0", "1130", "6"): "6 .. FIX44",
        ("FIX.5.0", "1130", "7"): "7 .. FIX50",
        ("FIX.5.0", "1130", "8"): "8 .. FIX50SP1",
        ("FIX.5.0", "1130", "9"): "9 .. FIX50SP2",
        ("FIX.5.0", "1133", "B"): "B .. BIC",
        ("FIX.5.0", "1133", "C"): "C .. GeneralIdentifier",
        ("FIX.5.0", "1133", "D"): "D .. Proprietary",
        ("FIX.5.0", "1133", "E"): "E .. ISOCountryCode",
        ("FIX.5.0", "1133", "G"): "G .. MIC",
        ("FIX.5.0", "1137", "0"): "0 .. FIX27",
        ("FIX.5.0", "1137", "1"): "1 .. FIX30",
        ("FIX.5.0", "1137", "2"): "2 .. FIX40",
        ("FIX.5.0", "1137", "3"): "3 .. FIX41",
        ("FIX.5.0", "1137", "4"): "4 .. FIX42",
        ("FIX.5.0", "1137", "5"): "5 .. FIX43",
        ("FIX.5.0", "1137", "6"): "6 .. FIX44",
        ("FIX.5.0", "1137", "7"): "7 .. FIX50",
        ("FIX.5.0", "1137", "8"): "8 .. FIX50SP1",
        ("FIX.5.0", "1137", "9"): "9 .. FIX50SP2",
        ("FIX.5.0", "1144", "0"): "0 .. NotImplied",
        ("FIX.5.0", "1144", "1"): "1 .. ImpliedIn",
        ("FIX.5.0", "1144", "2"): "2 .. ImpliedOut",
        ("FIX.5.0", "1144", "3"): "3 .. BothImpliedInAndImpliedOut",
        ("FIX.5.0", "1159", "1"): "1 .. Preliminary",
        ("FIX.5.0", "1159", "2"): "2 .. Final",
        ("FIX.5.0", "1162", "C"): "C .. Cancel",
        ("FIX.5.0", "1162", "N"): "N .. New",
        ("FIX.5.0", "1162", "R"): "R .. Replace",
        ("FIX.5.0", "1162", "T"): "T .. Restate",
        ("FIX.5.0", "1164", "1"): "1 .. InstructionsOfBroker",
        ("FIX.5.0", "1164", "2"): "2 .. InstructionsForInstitution",
        ("FIX.5.0", "1164", "3"): "3 .. Investor",
        ("FIX.5.0", "1164", "4"): "4 .. BuyersSettlementInstructions",
        ("FIX.5.0", "1164", "5"): "5 .. SellersSettlementInstructions",
        ("FIX.5.0", "1167", "0"): "0 .. Accepted",
        ("FIX.5.0", "1167", "5"): "5 .. Rejected",
        ("FIX.5.0", "1167", "6"): "6 .. RemovedFromMarket",
        ("FIX.5.0", "1167", "7"): "7 .. Expired",
        ("FIX.5.0", "1167", "12"): "12 .. LockedMarketWarning",
        ("FIX.5.0", "1167", "13"): "13 .. CrossMarketWarning",
        ("FIX.5.0", "1167", "14"): "14 .. CanceledDueToLockMarket",
        ("FIX.5.0", "1167", "15"): "15 .. CanceledDueToCrossMarket",
        ("FIX.5.0", "1167", "16"): "16 .. Active",
        ("FIX.5.0", "1171", "Y"): "Y .. PrivateQuote",
        ("FIX.5.0", "1171", "N"): "N .. PublicQuote",
        ("FIX.5.0", "1172", "1"): "1 .. AllMarketParticipants",
        ("FIX.5.0", "1172", "2"): "2 .. SpecifiedMarketParticipants",
        ("FIX.5.0", "1172", "3"): "3 .. AllMarketMakers",
        ("FIX.5.0", "1172", "4"): "4 .. PrimaryMarketMaker",
        ("FIX.5.0", "1174", "1"): "1 .. OrderImbalance",
        ("FIX.5.0", "1174", "2"): "2 .. TradingResumes",
        ("FIX.5.0", "1174", "3"): "3 .. PriceVolatilityInterruption",
        ("FIX.5.0", "1174", "4"): "4 .. ChangeOfTradingSession",
        ("FIX.5.0", "1174", "5"): "5 .. ChangeOfTradingSubsession",
        ("FIX.5.0", "1174", "6"): "6 .. ChangeOfSecurityTradingStatus",
        ("FIX.5.0", "1174", "7"): "7 .. ChangeOfBookType",
        ("FIX.5.0", "1174", "8"): "8 .. ChangeOfMarketDepth",
        ("FIX.5.0", "1174", "9"): "9 .. CorporateAction",
        ("FIX.5.0", "1176", "1"): "1 .. ExchangeLast",
        ("FIX.5.0", "1176", "2"): "2 .. High",
        ("FIX.5.0", "1176", "3"): "3 .. AveragePrice",
        ("FIX.5.0", "1176", "4"): "4 .. Turnover",
        ("FIX.5.0", "1178", "1"): "1 .. Customer",
        ("FIX.5.0", "1178", "2"): "2 .. CustomerProfessional",
        ("FIX.5.0", "1178", "3"): "3 .. DoNotTradeThrough",
        ("FIX.5.0", "1191", "Bcf"): "Bcf .. BillionCubicFeet",
        ("FIX.5.0", "1191", "CBM"): "CBM .. CubicMeters",
        ("FIX.5.0", "1191", "GJ"): "GJ .. gigajoules",
        ("FIX.5.0", "1191", "kWh"): "kWh .. KilowattHours",
        ("FIX.5.0", "1191", "MMBtu"): "MMBtu .. OneMillionBTU",
        ("FIX.5.0", "1191", "MWh"): "MWh .. MegawattHours",
        ("FIX.5.0", "1191", "thm"): "thm .. therms",
        ("FIX.5.0", "1191", "Alw"): "Alw .. Allowances",
        ("FIX.5.0", "1191", "Bbl"): "Bbl .. Barrels",
        ("FIX.5.0", "1191", "BDFT"): "BDFT .. BoardFeet",
        ("FIX.5.0", "1191", "Bu"): "Bu .. Bushels",
        ("FIX.5.0", "1191", "Ccy"): "Ccy .. Currency",
        ("FIX.5.0", "1191", "CDD"): "CDD .. CoolingDegreeDay",
        ("FIX.5.0", "1191", "CER"): "CER .. CertifiedEmissionsReduction",
        ("FIX.5.0", "1191", "CPD"): "CPD .. CriticalPrecipDay",
        ("FIX.5.0", "1191", "CRT"): "CRT .. ClimateReserveTonnes",
        ("FIX.5.0", "1191", "cwt"): "cwt .. Hundredweight",
        ("FIX.5.0", "1191", "day"): "day .. Day",
        ("FIX.5.0", "1191", "dt"): "dt .. DryMetricTons",
        ("FIX.5.0", "1191", "EnvAllwnc"): "EnvAllwnc .. EnvAllwncCert",
        ("FIX.5.0", "1191", "EnvCrd"): "EnvCrd .. EnvironmentalCredit",
        ("FIX.5.0", "1191", "EnvOfst"): "EnvOfst .. EnvironmentalOffset",
        ("FIX.5.0", "1191", "g"): "g .. Grams",
        ("FIX.5.0", "1191", "Gal"): "Gal .. Gallons",
        ("FIX.5.0", "1191", "GT"): "GT .. GrossTons",
        ("FIX.5.0", "1191", "HDD"): "HDD .. HeatingDegreeDay",
        ("FIX.5.0", "1191", "IPNT"): "IPNT .. IndexPoint",
        ("FIX.5.0", "1191", "kg"): "kg .. Kilograms",
        ("FIX.5.0", "1191", "kL"): "kL .. kiloliters",
        ("FIX.5.0", "1191", "kW-a"): "kW-a .. KilowattYear",
        ("FIX.5.0", "1191", "kW-d"): "kW-d .. KilowattDay",
        ("FIX.5.0", "1191", "kW-h"): "kW-h .. KilowattHour",
        ("FIX.5.0", "1191", "kW-M"): "kW-M .. KilowattMonth",
        ("FIX.5.0", "1191", "kW-min "): "kW-min  .. KilowattMinute",
        ("FIX.5.0", "1191", "L"): "L .. liters",
        ("FIX.5.0", "1191", "lbs"): "lbs .. Pounds",
        ("FIX.5.0", "1191", "MW-a"): "MW-a .. MegawattYear",
        ("FIX.5.0", "1191", "MW-d"): "MW-d .. MegawattDay",
        ("FIX.5.0", "1191", "MW-h"): "MW-h .. MegawattHour",
        ("FIX.5.0", "1191", "MW-M"): "MW-M .. MegawattMonth",
        ("FIX.5.0", "1191", "MW-min"): "MW-min .. MegawattMinute",
        ("FIX.5.0", "1191", "oz_tr"): "oz_tr .. TroyOunces",
        ("FIX.5.0", "1191", "PRINC"): "PRINC .. PrincipalWithRelationToDebtInstrument",
        ("FIX.5.0", "1191", "t"): "t .. MetricTons",
        ("FIX.5.0", "1191", "tn"): "tn .. Tons",
        ("FIX.5.0", "1193", "C"): "C .. CashSettlementRequired",
        ("FIX.5.0", "1193", "P"): "P .. PhysicalSettlementRequired",
        ("FIX.5.0", "1193", "E"): "E .. Election",
        ("FIX.5.0", "1194", "0"): "0 .. European",
        ("FIX.5.0", "1194", "1"): "1 .. American",
        ("FIX.5.0", "1194", "2"): "2 .. Bermuda",
        ("FIX.5.0", "1194", "99"): "99 .. Other",
        ("FIX.5.0", "1196", "STD"): "STD .. Standard",
        ("FIX.5.0", "1196", "INX"): "INX .. Index",
        ("FIX.5.0", "1196", "INT"): "INT .. InterestRateIndex",
        ("FIX.5.0", "1196", "PCTPAR"): "PCTPAR .. PercentOfPar",
        ("FIX.5.0", "1197", "EQTY"): "EQTY .. PremiumStyle",
        ("FIX.5.0", "1197", "FUT"): "FUT .. FuturesStyleMarkToMarket",
        ("FIX.5.0", "1197", "FUTDA"): "FUTDA .. FuturesStyleWithAnAttachedCashAdjustment",
        ("FIX.5.0", "1197", "CDS"): "CDS .. CDSStyleCollateralization",
        ("FIX.5.0", "1197", "CDSD"): "CDSD .. CDSInDeliveryUseRecoveryRateToCalculate",
        ("FIX.5.0", "1198", "0"): "0 .. PreListedOnly",
        ("FIX.5.0", "1198", "1"): "1 .. UserRequested",
        ("FIX.5.0", "1209", "0"): "0 .. RegularTrading",
        ("FIX.5.0", "1209", "1"): "1 .. VariableCabinet",
        ("FIX.5.0", "1209", "2"): "2 .. FixedCabinet",
        ("FIX.5.0", "1209", "3"): "3 .. TradedAsASpreadLeg",
        ("FIX.5.0", "1209", "5"): "5 .. TradedAsSpread",
        ("FIX.5.0", "1210", "1"): "1 .. Flat",
        ("FIX.5.0", "1210", "2"): "2 .. ZeroCoupon",
        ("FIX.5.0", "1210", "3"): "3 .. InterestBearing",
        ("FIX.5.0", "1210", "4"): "4 .. NoPeriodicPayments",
        ("FIX.5.0", "1210", "5"): "5 .. VariableRate",
        ("FIX.5.0", "1210", "6"): "6 .. LessFeeForPut",
        ("FIX.5.0", "1210", "7"): "7 .. SteppedCoupon",
        ("FIX.5.0", "1210", "8"): "8 .. CouponPeriod",
        ("FIX.5.0", "1210", "9"): "9 .. When",
        ("FIX.5.0", "1210", "10"): "10 .. OriginalIssueDiscount",
        ("FIX.5.0", "1210", "11"): "11 .. Callable",
        ("FIX.5.0", "1210", "12"): "12 .. EscrowedToMaturity",
        ("FIX.5.0", "1210", "13"): "13 .. EscrowedToRedemptionDate",
        ("FIX.5.0", "1210", "14"): "14 .. PreRefunded",
        ("FIX.5.0", "1210", "15"): "15 .. InDefault",
        ("FIX.5.0", "1210", "16"): "16 .. Unrated",
        ("FIX.5.0", "1210", "17"): "17 .. Taxable",
        ("FIX.5.0", "1210", "18"): "18 .. Indexed",
        ("FIX.5.0", "1210", "19"): "19 .. SubjectToAlternativeMinimumTax",
        ("FIX.5.0", "1210", "20"): "20 .. OriginalIssueDiscountPrice",
        ("FIX.5.0", "1210", "21"): "21 .. CallableBelowMaturityValue",
        ("FIX.5.0", "1210", "22"): "22 .. CallableWithoutNotice",
        ("FIX.5.0", "1210", "23"): "23 .. PriceTickRulesForSecurity",
        ("FIX.5.0", "1210", "24"): "24 .. TradeTypeEligibilityDetailsForSecurity",
        ("FIX.5.0", "1210", "25"): "25 .. InstrumentDenominator",
        ("FIX.5.0", "1210", "26"): "26 .. InstrumentNumerator",
        ("FIX.5.0", "1210", "27"): "27 .. InstrumentPricePrecision",
        ("FIX.5.0", "1210", "28"): "28 .. InstrumentStrikePrice",
        ("FIX.5.0", "1210", "29"): "29 .. TradeableIndicator",
        ("FIX.5.0", "1210", "30"): "30 .. InstrumentEligibleAnonOrders",
        ("FIX.5.0", "1210", "31"): "31 .. MinGuaranteedFillVolume",
        ("FIX.5.0", "1210", "32"): "32 .. MinGuaranteedFillStatus",
        ("FIX.5.0", "1210", "33"): "33 .. TradeAtSettlementEligibility",
        ("FIX.5.0", "1210", "34"): "34 .. TestInstrument",
        ("FIX.5.0", "1210", "35"): "35 .. DummyInstrument",
        ("FIX.5.0", "1210", "36"): "36 .. NegativeSettlementPriceEligibility",
        ("FIX.5.0", "1210", "37"): "37 .. NegativeStrikePriceEligibility",
        ("FIX.5.0", "1210", "38"): "38 .. USStdContractInd",
        ("FIX.5.0", "1210", "39"): "39 .. AdmittedToTradingOnTradingVenue",
        ("FIX.5.0", "1210", "40"): "40 .. AverageDailyNotionalAmount",
        ("FIX.5.0", "1210", "41"): "41 .. AverageDailyNumberTrades",
        ("FIX.5.0", "1210", "99"): "99 .. Text",
        ("FIX.5.0", "1215", "CD"): "CD .. EUCPWithLumpSumInterest",
        ("FIX.5.0", "1215", "WI"): "WI .. WhenIssued",
        ("FIX.5.0", "1217", "1"): "1 .. CUSIP",
        ("FIX.5.0", "1217", "2"): "2 .. SEDOL",
        ("FIX.5.0", "1217", "3"): "3 .. QUIK",
        ("FIX.5.0", "1217", "4"): "4 .. ISINNumber",
        ("FIX.5.0", "1217", "5"): "5 .. RICCode",
        ("FIX.5.0", "1217", "6"): "6 .. ISOCurrencyCode",
        ("FIX.5.0", "1217", "7"): "7 .. ISOCountryCode",
        ("FIX.5.0", "1217", "8"): "8 .. ExchangeSymbol",
        ("FIX.5.0", "1217", "9"): "9 .. ConsolidatedTapeAssociation",
        ("FIX.5.0", "1217", "A"): "A .. BloombergSymbol",
        ("FIX.5.0", "1217", "B"): "B .. Wertpapier",
        ("FIX.5.0", "1217", "C"): "C .. Dutch",
        ("FIX.5.0", "1217", "D"): "D .. Valoren",
        ("FIX.5.0", "1217", "E"): "E .. Sicovam",
        ("FIX.5.0", "1217", "F"): "F .. Belgian",
        ("FIX.5.0", "1217", "G"): "G .. Common",
        ("FIX.5.0", "1217", "H"): "H .. ClearingHouse",
        ("FIX.5.0", "1217", "I"): "I .. ISDAFpMLSpecification",
        ("FIX.5.0", "1217", "J"): "J .. OptionPriceReportingAuthority",
        ("FIX.5.0", "1217", "K"): "K .. ISDAFpMLURL",
        ("FIX.5.0", "1217", "L"): "L .. LetterOfCredit",
        ("FIX.5.0", "1217", "M"): "M .. MarketplaceAssignedIdentifier",
        ("FIX.5.0", "1217", "N"): "N .. MarkitREDEntityCLIP",
        ("FIX.5.0", "1217", "P"): "P .. MarkitREDPairCLIP",
        ("FIX.5.0", "1217", "Q"): "Q .. CFTCCommodityCode",
        ("FIX.5.0", "1217", "R"): "R .. ISDACommodityReferencePrice",
        ("FIX.5.0", "1217", "S"): "S .. FinancialInstrumentGlobalIdentifier",
        ("FIX.5.0", "1217", "T"): "T .. LegalEntityIdentifier",
        ("FIX.5.0", "1217", "U"): "U .. Synthetic",
        ("FIX.5.0", "1217", "V"): "V .. FidessaInstrumentMnemonic",
        ("FIX.5.0", "1217", "W"): "W .. IndexName",
        ("FIX.5.0", "1220", "1"): "1 .. CUSIP",
        ("FIX.5.0", "1220", "2"): "2 .. SEDOL",
        ("FIX.5.0", "1220", "3"): "3 .. QUIK",
        ("FIX.5.0", "1220", "4"): "4 .. ISINNumber",
        ("FIX.5.0", "1220", "5"): "5 .. RICCode",
        ("FIX.5.0", "1220", "6"): "6 .. ISOCurrencyCode",
        ("FIX.5.0", "1220", "7"): "7 .. ISOCountryCode",
        ("FIX.5.0", "1220", "8"): "8 .. ExchangeSymbol",
        ("FIX.5.0", "1220", "9"): "9 .. ConsolidatedTapeAssociation",
        ("FIX.5.0", "1220", "A"): "A .. BloombergSymbol",
        ("FIX.5.0", "1220", "B"): "B .. Wertpapier",
        ("FIX.5.0", "1220", "C"): "C .. Dutch",
        ("FIX.5.0", "1220", "D"): "D .. Valoren",
        ("FIX.5.0", "1220", "E"): "E .. Sicovam",
        ("FIX.5.0", "1220", "F"): "F .. Belgian",
        ("FIX.5.0", "1220", "G"): "G .. Common",
        ("FIX.5.0", "1220", "H"): "H .. ClearingHouse",
        ("FIX.5.0", "1220", "I"): "I .. ISDAFpMLSpecification",
        ("FIX.5.0", "1220", "J"): "J .. OptionPriceReportingAuthority",
        ("FIX.5.0", "1220", "K"): "K .. ISDAFpMLURL",
        ("FIX.5.0", "1220", "L"): "L .. LetterOfCredit",
        ("FIX.5.0", "1220", "M"): "M .. MarketplaceAssignedIdentifier",
        ("FIX.5.0", "1220", "N"): "N .. MarkitREDEntityCLIP",
        ("FIX.5.0", "1220", "P"): "P .. MarkitREDPairCLIP",
        ("FIX.5.0", "1220", "Q"): "Q .. CFTCCommodityCode",
        ("FIX.5.0", "1220", "R"): "R .. ISDACommodityReferencePrice",
        ("FIX.5.0", "1220", "S"): "S .. FinancialInstrumentGlobalIdentifier",
        ("FIX.5.0", "1220", "T"): "T .. LegalEntityIdentifier",
        ("FIX.5.0", "1220", "U"): "U .. Synthetic",
        ("FIX.5.0", "1220", "V"): "V .. FidessaInstrumentMnemonic",
        ("FIX.5.0", "1220", "W"): "W .. IndexName",
        ("FIX.5.0", "1238", "Bcf"): "Bcf .. BillionCubicFeet",
        ("FIX.5.0", "1238", "CBM"): "CBM .. CubicMeters",
        ("FIX.5.0", "1238", "GJ"): "GJ .. gigajoules",
        ("FIX.5.0", "1238", "kWh"): "kWh .. KilowattHours",
        ("FIX.5.0", "1238", "MMBtu"): "MMBtu .. OneMillionBTU",
        ("FIX.5.0", "1238", "MWh"): "MWh .. MegawattHours",
        ("FIX.5.0", "1238", "thm"): "thm .. therms",
        ("FIX.5.0", "1238", "Alw"): "Alw .. Allowances",
        ("FIX.5.0", "1238", "Bbl"): "Bbl .. Barrels",
        ("FIX.5.0", "1238", "BDFT"): "BDFT .. BoardFeet",
        ("FIX.5.0", "1238", "Bu"): "Bu .. Bushels",
        ("FIX.5.0", "1238", "Ccy"): "Ccy .. Currency",
        ("FIX.5.0", "1238", "CDD"): "CDD .. CoolingDegreeDay",
        ("FIX.5.0", "1238", "CER"): "CER .. CertifiedEmissionsReduction",
        ("FIX.5.0", "1238", "CPD"): "CPD .. CriticalPrecipDay",
        ("FIX.5.0", "1238", "CRT"): "CRT .. ClimateReserveTonnes",
        ("FIX.5.0", "1238", "cwt"): "cwt .. Hundredweight",
        ("FIX.5.0", "1238", "day"): "day .. Day",
        ("FIX.5.0", "1238", "dt"): "dt .. DryMetricTons",
        ("FIX.5.0", "1238", "EnvAllwnc"): "EnvAllwnc .. EnvAllwncCert",
        ("FIX.5.0", "1238", "EnvCrd"): "EnvCrd .. EnvironmentalCredit",
        ("FIX.5.0", "1238", "EnvOfst"): "EnvOfst .. EnvironmentalOffset",
        ("FIX.5.0", "1238", "g"): "g .. Grams",
        ("FIX.5.0", "1238", "Gal"): "Gal .. Gallons",
        ("FIX.5.0", "1238", "GT"): "GT .. GrossTons",
        ("FIX.5.0", "1238", "HDD"): "HDD .. HeatingDegreeDay",
        ("FIX.5.0", "1238", "IPNT"): "IPNT .. IndexPoint",
    