# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['fakeredis', 'fakeredis.commands_mixins', 'fakeredis.stack']

package_data = \
{'': ['*']}

install_requires = \
['redis<4.5', 'sortedcontainers>=2.4.0,<3.0.0']

extras_require = \
{'json': ['jsonpath-ng>=1.5,<2.0'], 'lua': ['lupa>=1.14,<2.0']}

setup_kwargs = {
    'name': 'fakeredis',
    'version': '2.3.0',
    'description': 'Fake implementation of redis API for testing purposes.',
    'long_description': 'fakeredis: A fake version of a redis-py\n=======================================\n[![badge](https://img.shields.io/pypi/v/fakeredis)](https://pypi.org/project/fakeredis/)\n[![CI](https://github.com/cunla/fakeredis-py/actions/workflows/test.yml/badge.svg)](https://github.com/cunla/fakeredis-py/actions/workflows/test.yml)\n[![badge](https://img.shields.io/endpoint?url=https://gist.githubusercontent.com/cunla/b756396efb895f0e34558c980f1ca0c7/raw/fakeredis-py.json)](https://github.com/cunla/fakeredis-py/actions/workflows/test.yml)\n[![badge](https://img.shields.io/pypi/dm/fakeredis)](https://pypi.org/project/fakeredis/)\n[![badge](https://img.shields.io/pypi/l/fakeredis)](./LICENSE)\n--------------------\n[Intro](#intro) | [How to Use](#how-to-use) | [Contributing](.github/CONTRIBUTING.md) | [Guides](#guides)\n| [Sponsoring](#sponsor)\n\n# Intro\n\nfakeredis is a pure-Python implementation of the redis-py python client\nthat simulates talking to a redis server. This was created for a single\npurpose: **to write tests**. Setting up redis is not hard, but\nmany times you want to write tests that do not talk to an external server\n(such as redis). This module now allows tests to simply use this\nmodule as a reasonable substitute for redis.\n\nFor a list of supported/unsupported redis commands, see [REDIS_COMMANDS.md](./REDIS_COMMANDS.md).\n\n# Installation\n\nTo install fakeredis-py, simply:\n\n```bash\npip install fakeredis        # No additional modules support\n\npip install fakeredis[lua]   # Support for LUA scripts\n\npip install fakeredis[json]  # Support for RedisJSON commands\n```\n\n# How to Use\n\nFakeRedis can imitate Redis server version 6.x or 7.x.\nIf you do not specify the version, version 7 is used by default.\n\nThe intent is for fakeredis to act as though you\'re talking to a real\nredis server. It does this by storing state internally.\nFor example:\n\n```pycon\n>>> import fakeredis\n>>> r = fakeredis.FakeStrictRedis(version=6)\n>>> r.set(\'foo\', \'bar\')\nTrue\n>>> r.get(\'foo\')\n\'bar\'\n>>> r.lpush(\'bar\', 1)\n1\n>>> r.lpush(\'bar\', 2)\n2\n>>> r.lrange(\'bar\', 0, -1)\n[2, 1]\n```\n\nThe state is stored in an instance of `FakeServer`. If one is not provided at\nconstruction, a new instance is automatically created for you, but you can\nexplicitly create one to share state:\n\n```pycon\n>>> import fakeredis\n>>> server = fakeredis.FakeServer()\n>>> r1 = fakeredis.FakeStrictRedis(server=server)\n>>> r1.set(\'foo\', \'bar\')\nTrue\n>>> r2 = fakeredis.FakeStrictRedis(server=server)\n>>> r2.get(\'foo\')\n\'bar\'\n>>> r2.set(\'bar\', \'baz\')\nTrue\n>>> r1.get(\'bar\')\n\'baz\'\n>>> r2.get(\'bar\')\n\'baz\'\n```\n\nIt is also possible to mock connection errors, so you can effectively test\nyour error handling. Simply set the connected attribute of the server to\n`False` after initialization.\n\n```pycon\n>>> import fakeredis\n>>> server = fakeredis.FakeServer()\n>>> server.connected = False\n>>> r = fakeredis.FakeStrictRedis(server=server)\n>>> r.set(\'foo\', \'bar\')\nConnectionError: FakeRedis is emulating a connection error.\n>>> server.connected = True\n>>> r.set(\'foo\', \'bar\')\nTrue\n```\n\nFakeredis implements the same interface as `redis-py`, the popular\nredis client for python, and models the responses of redis 6.x or 7.x.\n\n## Use to test django-rq\n\nThere is a need to override `django_rq.queues.get_redis_connection` with\na method returning the same connection.\n\n```python\nfrom fakeredis import FakeRedisConnSingleton\n\ndjango_rq.queues.get_redis_connection = FakeRedisConnSingleton()\n```\n\n## Support for additional modules\n\n### RedisJson\n\nCurrently, Redis Json module is partially implemented (\nsee [supported commands](https://github.com/cunla/fakeredis-py/blob/master/REDIS_COMMANDS.md#json)).\n\n```pycon\n>>> import fakeredis\n>>> from redis.commands.json.path import Path\n>>> r = fakeredis.FakeStrictRedis()\n>>> assert r.json().set("foo", Path.root_path(), {"x": "bar"}, ) == 1\n>>> r.json().get("foo")\n{\'x\': \'bar\'}\n>>> r.json().get("foo", Path("x"))\n\'bar\'\n```\n\n### Lua support\n\nIf you wish to have Lua scripting support (this includes features like ``redis.lock.Lock``, which are implemented in\nLua), you will need [lupa](https://pypi.org/project/lupa/), you can simply install it using `pip install fakeredis[lua]`\n\n### JSON support\n\nSupport for JSON commands (eg, [`JSON.GET`](https://redis.io/commands/json.get/)) is implemented using\n[jsonpath-ng](https://github.com/h2non/jsonpath-ng), you can simply install it using `pip install fakeredis[json]`.\n\n## Known Limitations\n\nApart from unimplemented commands, there are a number of cases where fakeredis\nwon\'t give identical results to real redis. The following are differences that\nare unlikely to ever be fixed; there are also differences that are fixable\n(such as commands that do not support all features) which should be filed as\nbugs in GitHub.\n\n- Hyperloglogs are implemented using sets underneath. This means that the\n  `type` command will return the wrong answer, you can\'t use `get` to retrieve\n  the encoded value, and counts will be slightly different (they will in fact be\n  exact).\n- When a command has multiple error conditions, such as operating on a key of\n  the wrong type and an integer argument is not well-formed, the choice of\n  error to return may not match redis.\n\n- The `incrbyfloat` and `hincrbyfloat` commands in redis use the C `long\n  double` type, which typically has more precision than Python\'s `float`\n  type.\n\n- Redis makes guarantees about the order in which clients blocked on blocking\n  commands are woken up. Fakeredis does not honour these guarantees.\n\n- Where redis contains bugs, fakeredis generally does not try to provide exact\n  bug-compatibility. It\'s not practical for fakeredis to try to match the set\n  of bugs in your specific version of redis.\n\n- There are a number of cases where the behaviour of redis is undefined, such\n  as the order of elements returned by set and hash commands. Fakeredis will\n  generally not produce the same results, and in Python versions before 3.6\n  may produce different results each time the process is re-run.\n\n- SCAN/ZSCAN/HSCAN/SSCAN will not necessarily iterate all items if items are\n  deleted or renamed during iteration. They also won\'t necessarily iterate in\n  the same chunk sizes or the same order as redis.\n\n- DUMP/RESTORE will not return or expect data in the RDB format. Instead, the\n  `pickle` module is used to mimic an opaque and non-standard format.\n  **WARNING**: Do not use RESTORE with untrusted data, as a malicious pickle\n  can execute arbitrary code.\n\n--------------------\n\n# Local development environment\n\nTo ensure parity with the real redis, there are a set of integration tests\nthat mirror the unittests. For every unittest that is written, the same\ntest is run against a real redis instance using a real redis-py client\ninstance. In order to run these tests you must have a redis server running\non localhost, port 6379 (the default settings). **WARNING**: the tests will\ncompletely wipe your database!\n\nFirst install poetry if you don\'t have it, and then install all the dependencies:\n\n```bash\npip install poetry\npoetry install\n``` \n\nTo run all the tests:\n\n```bash\npoetry run pytest -v\n```\n\nIf you only want to run tests against fake redis, without a real redis::\n\n```bash\npoetry run pytest -m fake\n```\n\nBecause this module is attempting to provide the same interface as `redis-py`,\nthe python bindings to redis, a reasonable way to test this to take each\nunittest and run it against a real redis server. fakeredis and the real redis\nserver should give the same result. To run tests against a real redis instance\ninstead:\n\n```bash\npoetry run pytest -m real\n```\n\nIf redis is not running, and you try to run tests against a real redis server,\nthese tests will have a result of \'s\' for skipped.\n\nThere are some tests that test redis blocking operations that are somewhat\nslow. If you want to skip these tests during day to day development,\nthey have all been tagged as \'slow\' so you can skip them by running:\n\n```bash\npoetry run pytest -m "not slow"\n```\n\n# Contributing\n\nContributions are welcome.\nYou can contribute in many ways: \nOpen issues for bugs you found, implementing a command which is not yet implemented,\nimplement a test for scenario that is not covered yet, write a guide how to use fakeredis, etc.\n\nPlease see the [contributing guide](.github/CONTRIBUTING.md) for more details.\nIf you\'d like to help out, you can start with any of the issues labeled with `Help wanted`.\n\nThere are guides how to [implement a new command](#implementing-support-for-a-command) and\nhow to [write new test cases](#write-a-new-test-case).\n\nNew contribution guides are welcome.\n\n# Guides\n\n### Implementing support for a command\n\nCreating a new command support should be done in the `FakeSocket` class (in `_fakesocket.py`) by creating the method\nand using `@command` decorator (which should be the command syntax, you can use existing samples on the file).\n\nFor example:\n\n```python\nclass FakeSocket(BaseFakeSocket, FakeLuaSocket):\n    # ...\n    @command(name=\'zscore\', fixed=(Key(ZSet), bytes), repeat=(), flags=[])\n    def zscore(self, key, member):\n        try:\n            return self._encodefloat(key.value[member], False)\n        except KeyError:\n            return None\n```\n\n#### How to use `@command` decorator\n\nThe `@command` decorator register the method as a redis command and define the accepted format for it.\nIt will create a `Signature` instance for the command. Whenever the command is triggered, the `Signature.apply(..)`\nmethod will be triggered to check the validity of syntax and analyze the command arguments.\n\nBy default, it takes the name of the method as the command name.\n\nIf the method implements a subcommand (eg, `SCRIPT LOAD`), a Redis module command (eg, `JSON.GET`),\nor a python reserve word where you can not use it as the method name (eg, `EXEC`), then you can supply\nexplicitly the name parameter.\n\nIf the command implemented require certain arguments, they can be supplied in the first parameter as a tuple.\nWhen receiving the command through the socket, the bytes will be converted to the argument types\nsupplied or remain as `bytes`.\n\nArgument types (All in `_commands.py`):\n\n- `Key(KeyType)` - Will get from the DB the key and validate its value is of `KeyType` (if `KeyType` is supplied).\n  It will generate a `CommandItem` from it which provides access to the database value.\n- `Int` - Decode the `bytes` to `int` and vice versa.\n- `DbIndex`/`BitOffset`/`BitValue`/`Timeout` - Basically the same behavior as `Int`, but with different messages when\n  encode/decode fail.\n- `Hash` - dictionary, usually describe the type of value stored in Key `Key(Hash)`\n- `Float` - Encode/Decode `bytes` <-> `float`\n- `SortFloat` - Similar to `Float` with different error messages.\n- `ScoreTest` - Argument converter for sorted set score endpoints.\n- `StringTest` - Argument converter for sorted set endpoints (lex).\n- `ZSet` - Sorted Set.\n\n#### Implement a test for it\n\nThere are multiple scenarios for test, with different versions of redis server, redis-py, etc.\nThe tests not only assert the validity of output but runs the same test on a real redis-server and compares the output\nto the real server output.\n\n- Create tests in the relevant test file.\n- If support for the command was introduced in a certain version of redis-py (\n  see [redis-py release notes](https://github.com/redis/redis-py/releases/tag/v4.3.4)) you can use the\n  decorator `@testtools.run_test_if_redispy_ver` on your tests. example:\n\n```python\n@testtools.run_test_if_redispy_ver(\'above\', \'4.2.0\')  # This will run for redis-py 4.2.0 or above.\ndef test_expire_should_not_expire__when_no_expire_is_set(r):\n    r.set(\'foo\', \'bar\')\n    assert r.get(\'foo\') == b\'bar\'\n    assert r.expire(\'foo\', 1, xx=True) == 0\n```\n\n#### Updating `REDIS_COMMANDS.md`\n\nLastly, run from the root of the project the script to regenerate `REDIS_COMMANDS.md`:\n\n```bash\npython scripts/supported.py > REDIS_COMMANDS.md    \n```\n\n### Write a new test case\n\nThere are multiple scenarios for test, with different versions of python, redis-py and redis server, etc.\nThe tests not only assert the validity of the expected output with FakeRedis but also with a real redis server.\nThat way parity of real Redis and FakeRedis is ensured.\n\nTo write a new test case for a command:\n\n- Determine which mixin the command belongs to and the test file for\n  the mixin (eg, `string_mixin.py` => `test_string_commands.py`).\n- Tests should support python 3.7 and above.\n- Determine when support for the command was introduced\n    - To limit the redis-server versions it will run on use:\n      `@pytest.mark.max_server(version)` and `@pytest.mark.min_server(version)`\n    - To limit the redis-py version use `@run_test_if_redispy_ver(above/below, version)`\n- pytest will inject a redis connection to the argument `r` of the test.\n\nSample of running a test for redis-py v4.2.0 and above, redis-server 7.0 and above.\n\n```python\n@pytest.mark.min_server(\'7\')\n@testtools.run_test_if_redispy_ver(\'above\', \'4.2.0\')\ndef test_expire_should_not_expire__when_no_expire_is_set(r):\n    r.set(\'foo\', \'bar\')\n    assert r.get(\'foo\') == b\'bar\'\n    assert r.expire(\'foo\', 1, xx=True) == 0\n```\n\n# Sponsor\n\nfakeredis-py is developed for free.\n\nYou can support this project by becoming a sponsor using [this link](https://github.com/sponsors/cunla).\n\nAlternatively, you can buy me coffee using this\nlink: [!["Buy Me A Coffee"](https://www.buymeacoffee.com/assets/img/custom_images/orange_img.png)](https://buymeacoffee.com/danielmoran)\n',
    'author': 'James Saryerwinnie',
    'author_email': 'js@jamesls.com',
    'maintainer': 'Daniel Moran',
    'maintainer_email': 'daniel.maruani@gmail.com',
    'url': 'https://github.com/cunla/fakeredis-py',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.8.1,<4.0',
}


setup(**setup_kwargs)
