
from __future__ import (
    unicode_literals,
    print_function,
    absolute_import,
    division,
    )
str = type('')


from math import log, ceil
from operator import or_
try:
    from functools import reduce
except ImportError:
    pass # py2's reduce is built-in

from .exc import DeviceClosed, I2CBadArgs
from .devices import Device


class I2CDevice(Device):
    """
    Extends :class:`Device`. Represents a device that communicates via the I2C
    protocol (Inter-Integrated Circuit).

    See :ref:`i2c_args` for information on the keyword arguments that can be
    specified with the constructor.
    """
    def __init__(self, **i2c_args):
        self._i2c = None

        super(I2CDevice, self).__init__(
            pin_factory=i2c_args.pop('pin_factory', None)
        )
        self._i2c = self.pin_factory.i2c(**i2c_args)

    def close(self):
        if getattr(self, '_i2c', None):
            self._i2c.close()
            self._i2c = None
        super(I2CDevice, self).close()

    @property
    def closed(self):
        return self._i2c is None

    def __repr__(self):
        try:
            self._check_open()
            return "<gpiozero.%s object using %r>" % (self.__class__.__name__, self._spi)
        except DeviceClosed:
            return "<gpiozero.%s object closed>" % self.__class__.__name__


class DigitalIOExpanderDevice(I2CDevice):
    """
    Represents a digital IO expander device connected to I2C.
    """

    def __init__(self, ports, **i2c_args):
        self._ports = ports
        super(DigitalIOExpanderDevice, self).__init__(**i2c_args)

    @property
    def ports(self):
        """
        The expander max ports count.
        """
        return self._ports

    def _read(self):
        raise NotImplementedError

    @property
    def value(self):
        """
        The current value read from the device, each bit corresponds to a port value.
        """
        return self._read()

class MCP230xx(DigitalIOExpanderDevice):
    """
    Extends :class:`DigitalIOExpanderDevice` to implement an interface for all IO Expander
    chips with a protocol similar to the Microchip MCP23xx series of devices.
    """

    def __init__(self, ports, **i2c_args):
        super(MCP230xx, self).__init__(ports, **i2c_args)

        self._registers = {
            'IODIRA': 0x00, # Pin direction register
            'IODIRB': 0x01, # Pin direction register
            'GPPUB':  0x0C, # Pin pull register
            'GPPUB':  0x0D, # Pin pull register
            'GPIOA':  0x12, # Register for inputs
            'GPIOB':  0x13, # Register for inputs
            'OLATA':  0x14, # Register for outputs
            'OLATB':  0x15, # Register for outputs
        }

    def _read(self):
        count,data = self._i2c.read(8)
        return data


class MCP23017(MCP230xx):
    """
    Represents a MCP23017 16 ports digital IO expander device connected to I2C (Inter-Integrated Circuit).
    """

    def __init__(self, **i2c_args):
        super(MCP23017, self).__init__(16, **i2c_args)

class MCP23008(MCP230xx):
    """
    Represents a MCP23008 8 ports digital IO expander device connected to I2C (Inter-Integrated Circuit).
    """

    def __init__(self, **i2c_args):
        super(MCP23017, self).__init__(8, **i2c_args)