#!/usr/bin/env python3
# Python IR transmitter
# Requires pigpio library
# Supports NEC, RC-5 and raw IR.
# Danijel Tudek, Aug 2016

import time
import pigpio


# Since both NEC and RC-5 protocols use the same method for generating waveform,
# it can be put in a separate class and called from both protocol's classes.
class wave_generator():
    def __init__(self,protocol):
        self.protocol = protocol
        self.pulses = []
        self.pulses_array=[self.pulses]

    def add_pulse(self, gpioOn, gpioOff, usDelay):
        self.pulses.append(pigpio.pulse(gpioOn, gpioOff, usDelay))

    # Pull the specified output pin low
    def gap(self, duration):
        self.add_pulse(0, 1 << self.protocol.master.gpio_pin, duration)

    # Protocol-agnostic square wave generator
    def pulse(self, duration):
        period_time = 1000000.0 / self.protocol.frequency
        on_duration = int(round(period_time * self.protocol.duty_cycle))
        off_duration = int(round(period_time * (1.0 - self.protocol.duty_cycle)))
        total_periods = int(round(duration/period_time))
        total_pulses = total_periods * 2

        # Generate square wave on the specified output pin
        for i in range(total_pulses):
            if i % 2 == 0:
                self.add_pulse(1 << self.protocol.master.gpio_pin, 0, on_duration)
            else:
                self.add_pulse(0, 1 << self.protocol.master.gpio_pin, off_duration)

    def new_wave(self):
        self.pulses = []
        self.pulses_array.append(self.pulses)

# NEC protocol class
class NEC():
    def __init__(self,
                master,
                frequency=38000,
                duty_cycle=0.33,
                leading_pulse_duration=9000,
                leading_gap_duration=4500,
                one_pulse_duration = 562,
                one_gap_duration = 1686,
                zero_pulse_duration = 562,
                zero_gap_duration = 562,
                trailing_pulse = 0,
                separator_pulse_duration=562,
                separator_gap_duration=8000):
        self.master = master
        self.wave_generator = wave_generator(self)
        self.frequency = frequency # in Hz, 38000 per specification
        self.duty_cycle = duty_cycle # duty cycle of high state pulse
        # Durations of high pulse and low "gap".
        # The NEC protocol defines pulse and gap lengths, but we can never expect
        # that any given TV will follow the protocol specification.
        self.leading_pulse_duration = leading_pulse_duration # in microseconds, 9000 per specification
        self.leading_gap_duration = leading_gap_duration # in microseconds, 4500 per specification
        self.one_pulse_duration = one_pulse_duration # in microseconds, 562 per specification
        self.one_gap_duration = one_gap_duration # in microseconds, 1686 per specification
        self.zero_pulse_duration = zero_pulse_duration # in microseconds, 562 per specification
        self.zero_gap_duration = zero_gap_duration # in microseconds, 562 per specification
        self.trailing_pulse = trailing_pulse # trailing 562 microseconds pulse, some remotes send it, some don't
        self.separator_pulse_duration = separator_pulse_duration
        self.separator_gap_duration = separator_gap_duration

    # Send AGC burst before transmission
    def send_agc(self):
        try:
            iterator_pulse = iter(self.leading_pulse_duration)
            iterator_gap = iter(self.leading_gap_duration)
        except TypeError: # not iterable
            self.wave_generator.pulse(self.leading_pulse_duration)
            self.wave_generator.gap(self.leading_gap_duration) 
        else: # iterable
            while True:
                pulse = next(iterator_pulse, None)
                gap = next(iterator_gap, None)

                if pulse is None or gap is None:
                    break

                self.wave_generator.pulse(pulse)
                self.wave_generator.gap(gap)
            

    # Trailing pulse is just a burst with the duration of standard pulse.
    def send_trailing_pulse(self):
        self.wave_generator.pulse(self.one_pulse_duration)

    # This function is processing IR code. Leaves room for possible manipulation
    # of the code before processing it.
    def process_code(self, ircode):
        self.send_agc()
        
        for i in ircode:
            if i == "0":
                self.zero()
            elif i == "1":
                self.one()
            elif i == "|":
                self.separator()
            elif i == "*":
                self.send_agc()
            elif i == " ":
                continue
            else:
                print("ERROR! Non-binary digit!")
                return 1
        if self.trailing_pulse == 1:
            self.send_trailing_pulse()
        return 0

    # Generate zero or one in NEC protocol
    # Zero is represented by a pulse and a gap of the same length
    def zero(self):
        self.wave_generator.pulse(self.zero_pulse_duration)
        self.wave_generator.gap(self.zero_gap_duration)

    # One is represented by a pulse and a gap three times longer than the pulse
    def one(self):
        self.wave_generator.pulse(self.one_pulse_duration)
        self.wave_generator.gap(self.one_gap_duration)

    def separator(self):
        self.wave_generator.new_wave()

        try:
            iterator_pulse = iter(self.separator_pulse_duration)
            iterator_gap = iter(self.separator_gap_duration)
        except TypeError: # not iterable
            self.wave_generator.pulse(self.separator_pulse_duration)
            self.wave_generator.gap(self.separator_gap_duration)
        else: # iterable
            while True:
                pulse = next(iterator_pulse, None)
                gap = next(iterator_gap, None)

                if pulse is None or gap is None:
                    break

                self.wave_generator.pulse(pulse)
                self.wave_generator.gap(gap)

        
        

# RC-5 protocol class
# Note: start bits are not implemented here due to inconsistency between manufacturers.
# Simply provide them with the rest of the IR code.
class RC5():
    def __init__(self,
                master,
                frequency=36000,
                duty_cycle=0.33,
                one_duration=889,
                zero_duration=889):
        self.master = master
        self.wave_generator = wave_generator(self)
        self.frequency = frequency # in Hz, 36000 per specification
        self.duty_cycle = duty_cycle # duty cycle of high state pulse
        # Durations of high pulse and low "gap".
        # Technically, they both should be the same in the RC-5 protocol, but we can never expect
        # that any given TV will follow the protocol specification.
        self.one_duration = one_duration # in microseconds, 889 per specification
        self.zero_duration = zero_duration # in microseconds, 889 per specification

    # This function is processing IR code. Leaves room for possible manipulation
    # of the code before processing it.
    def process_code(self, ircode):
        for i in ircode:
            if i == "0":
                self.zero()
            elif i == "1":
                self.one()
            else:
                print("ERROR! Non-binary digit!")
                return 1
        return 0

    # Generate zero or one in RC-5 protocol
    # Zero is represented by pulse-then-low signal
    def zero(self):
        self.wave_generator.pulse(self.zero_duration)
        self.wave_generator.gap(self.zero_duration)

    # One is represented by low-then-pulse signal
    def one(self):
        self.wave_generator.gap(self.one_duration)
        self.wave_generator.pulse(self.one_duration)

# RAW IR ones and zeroes. Specify length for one and zero and simply bitbang the GPIO.
# The default values are valid for one tested remote which didn't fit in NEC or RC-5 specifications.
# It can also be used in case you don't want to bother with deciphering raw bytes from IR receiver:
# i.e. instead of trying to figure out the protocol, simply define bit lengths and send them all here.
class RAW():
    def __init__(self,
                master,
                frequency=36000,
                duty_cycle=0.33):
        self.master = master
        self.wave_generator = wave_generator(self)
        self.frequency = frequency # in Hz
        self.duty_cycle = duty_cycle # duty cycle of high state pulse

    def process_code(self, raw_codes):
        for i in range(0, len(raw_codes), 2):
            self.wave_generator.pulse(int(raw_codes[i]))

            if i+1 < len(raw_codes):
                self.wave_generator.gap(int(raw_codes[i+1]))

        return 0


class IR():
    def __init__(self, gpio_pin, protocol, protocol_config, host='localhost'):
        self.gpio = pigpio.pi(host)
        self.gpio_pin = gpio_pin
        self.gpio.set_mode(self.gpio_pin, pigpio.OUTPUT)

        if protocol == "NEC":
            self.protocol = NEC(self, **protocol_config)
        elif protocol == "RC-5":
            self.protocol = RC5(self, **protocol_config)
        elif protocol == "RAW":
            self.protocol = RAW(self, **protocol_config)
        else:
            print("Protocol not specified! Exiting...")

    # send_code takes care of sending the processed IR code to pigpio.
    # IR code itself is processed and converted to pigpio structs by protocol's classes.
    def send_code(self, ircode):
        code = self.protocol.process_code(ircode)
        if code != 0:
            print("Error in processing IR code!")
            return 1
        clear = self.gpio.wave_clear()
        if clear != 0:
            print("Error in clearing wave!")
            return 1

        for pulses in self.protocol.wave_generator.pulses_array:
            current_size = self.gpio.wave_get_micros()
            if current_size > 0:
                pulses.insert(0, pigpio.pulse(0,0,current_size))
                
            pulses = self.gpio.wave_add_generic(pulses)
            if pulses < 0:
                print("Error in adding wave!")
                return 1
                
        wave_id = self.gpio.wave_create()
        # Unlike the C implementation, in Python the wave_id seems to always be 0.
        if wave_id >= 0:
            result = self.gpio.wave_send_once(wave_id)
            if result >= 0:
                print("Success! (result: %d)" % result)
            else:
                print("Error! (result: %d)" % result)
                return 1
        else:
            print("Error creating wave: %d" % wave_id)
            return 1

        while self.gpio.wave_tx_busy():
            time.sleep(0.1)

        self.gpio.wave_delete(wave_id)

# Simply define the GPIO pin, protocol (NEC, RC-5 or RAW) and
# override the protocol defaults with the dictionary if required.
# Provide the IR code to the send_code() method.
# An example is given below.
if __name__ == "__main__":
    protocol = "RC-5"
    gpio_pin = 17
    protocol_config = dict(one_duration = 820,
                            zero_duration = 820)
    ir = IR(gpio_pin, protocol, protocol_config)
    ir.send_code("11000000001100")
    print("Exiting IR")
