import json
import re

from logger import log
from mqtt import mqtt, canonize_name
from hardware import deviceManager

DEVICE_REGEX = '(rpi|i2c)_([0-9]+)(?:_([0-9]+))?_([0-9a-zA-Z_]+)'

def configure():
	for board in filter(lambda b: hasattr(b, 'outputs'), deviceManager.boards):
		for pin in board.outputs:
			_subscribe_mqtt_light_update_request(board, pin.address)
			_publish_mqtt_light_discovery(board, pin.address)

def process_mqtt_message(message):
	if message.topic.startswith(mqtt.base_topic + '/light'):
		log.debug('Received MQTT message on topic {}'.format(message.topic))

		cmds = message.topic.split('/')
		if len(cmds) != 5:
			log.error("Invalid MQTT light command: {}".format(message.topic))
			return

		if cmds[2] == canonize_name(mqtt.hostname) and cmds[4] == 'set':
			return handle_light_update_request(cmds[3], message.payload)
	
		log.error('No handler found for light mqtt message on topic: {}'.format(message.topic))
	return False

def handle_light_update_request(device_address, request):
	log.debug('MQTT light update request: {}'.format(device_address))

	address, host = extract_device_address_from_unique_id(device_address)
	pin, board = deviceManager.get_output_pin(address, host)

	status = True if request == 'ON' else False
	if status:
		pin.on()
	else:
		pin.off()

	_publish_mqtt_light_update(board, pin.address, status)

	return True

def _publish_mqtt_light_update(board, pin, status):
	topic = mqtt.topic('light', board, pin)
	payload = 'ON' if status else 'OFF'
	
	mqtt.publish(topic + '/state', payload, retain=True)

def _publish_mqtt_light_discovery(board, pin):
	topic = mqtt.topic('light', board, pin)
	unique_id = mqtt.create_device_unique_id(board, pin)

	payload = {
		"~": topic,
		"name": 'UNIO Relay - {} - {} - {}'.format(pin, board.host, mqtt.hostname),
		"uniq_id": unique_id,
		"ret": True,
		"cmd_t": "~/set",
		"stat_t": "~/state",
		"dev": mqtt.board_hw_info(board)
	}

	mqtt.publish(topic + '/config', json.dumps(payload), retain=True)

def _subscribe_mqtt_light_update_request(board, pin):
	topic = mqtt.topic('light', board, pin)

	mqtt.subscribe(topic + '/set')

def extract_device_address_from_unique_id(unique_id):
	split = re.search(DEVICE_REGEX, unique_id).groups()
	params = [p for p in split if p is not None]
	addr_size = 2
	if 'i2c' == params[0]:
		addr_size = 3

	address = '.'.join(params[:addr_size])
	host = '_'.join(params[addr_size:])

	return address, host
