import json
import re

from logger import log
from mqtt import mqtt, canonize_name
from hardware import deviceManager

DEVICE_REGEX = '(rpi|i2c)_([0-9]+)_([0-9a-zA-Z]+)_([0-9a-zA-Z_]+)'

def configure():
	for board in filter(lambda b: hasattr(b, 'devices'), deviceManager.boards):
		for device in filter(lambda d: d.type == 'climate', board.devices):
			_subscribe_mqtt_climate_mode_update_request(board, device)
			_subscribe_mqtt_climate_fan_mode_update_request(board, device)
			_subscribe_mqtt_climate_temperature_update_request(board, device)
			_publish_mqtt_climate_discovery(board, device, board.address)

def process_mqtt_message(message):
	if message.topic.startswith(mqtt.base_topic + '/climate'):
		log.debug('Received MQTT message on topic {}'.format(message.topic))

		cmds = message.topic.split('/')
		if len(cmds) < 5 or len(cmds) > 6:
			log.error("Invalid MQTT climate command: {}".format(message.topic))
			return

		if cmds[2] != canonize_name(mqtt.hostname):
			return False

		#if cmds[4] == 'state':
		#	return handle_climate_state_request(cmds[3], message.payload)
		
		if cmds[4] == 'mode' and cmds[5] == 'set':
			return handle_climate_update_mode_request(cmds[3], message.payload)

		if cmds[4] == 'fan_mode' and cmds[5] == 'set':
			return handle_climate_update_fan_mode_request(cmds[3], message.payload)

		if cmds[4] == 'temperature' and cmds[5] == 'set':
			return handle_climate_update_temperature_request(cmds[3], message.payload)
	
		log.error('No handler found for climate mqtt message on topic: {}'.format(message.topic))
	return False

def handle_climate_update_mode_request(unique_id, request):
	log.debug('MQTT climate update mode request: {}'.format(unique_id))
	address, conf, host = extract_device_address_from_unique_id(unique_id)
	device, board = deviceManager.get_climate_device(address, conf, host)
	device.update(mode=request)
	_publish_mqtt_climate_state_update(board, board.address, device)
	return True

def handle_climate_update_fan_mode_request(unique_id, request):
	log.debug('MQTT climate update fan mode request: {}'.format(unique_id))
	address, conf, host = extract_device_address_from_unique_id(unique_id)
	device, board = deviceManager.get_climate_device(address, conf, host)
	device.update(fan_mode=request)
	_publish_mqtt_climate_state_update(board, board.address, device)
	return True

def handle_climate_update_temperature_request(unique_id, request):
	log.debug('MQTT climate update temperature request: {}'.format(unique_id))
	address, conf, host = extract_device_address_from_unique_id(unique_id)
	device, board = deviceManager.get_climate_device(address, conf, host)
	device.update(temperature=request)
	_publish_mqtt_climate_state_update(board, board.address, device)
	return True

def _publish_mqtt_climate_state_update(board, pin, device):
	topic = mqtt.topic('climate', board, pin, device=device)
	payload = json.dumps(device.state)
	mqtt.publish(topic + '/state', payload, retain=True)

def _publish_mqtt_climate_discovery(board, device, pin):
	topic = mqtt.topic('climate', board, pin, device=device)
	unique_id = mqtt.create_device_unique_id(board, pin, device=device)

	payload = {
		"~": topic,
		"name": 'UNIO Hvac IR - {} - {} - {} - {}'.format(device.name, pin, board.host, mqtt.hostname),
		"uniq_id": unique_id,
		"ret": True,
		"dev": mqtt.board_hw_info(board),

		"modes": ["off", "cool", "dry", "fan_only"],
		"fan_modes": ["auto", "low", "medium", "high"],
		"min_temp": device.min_temp,
		"max_temp": device.max_temp,

		"mode_cmd_t": "~/mode/set",
		"temp_cmd_t": "~/temperature/set",
		"fan_mode_cmd_t": "~/fan/set",
		#"swing_mode_command_topic": "~/swing/set"
		
		"mode_stat_t": "~/state",
		"fan_mode_stat_t": "~/state",
		"temp_stat_t": "~/state",
		
		"mode_stat_tpl": "{{value_json.mode}}",
		"fan_mode_stat_tpl": "{{value_json.fan_mode}}",
		"temp_stat_tpl": "{{value_json.temperature}}"
	}
	
	mqtt.publish(topic + '/config', json.dumps(payload), retain=True)

def _subscribe_mqtt_climate_mode_update_request(board, device):
	topic = mqtt.topic('climate', board, board.address, device=device)
	mqtt.subscribe(topic + '/mode/set')

def _subscribe_mqtt_climate_fan_mode_update_request(board, device):
	topic = mqtt.topic('climate', board, board.address, device=device)
	mqtt.subscribe(topic + '/fan_mode/set')

def _subscribe_mqtt_climate_temperature_update_request(board, device):
	topic = mqtt.topic('climate', board, board.address, device=device)
	mqtt.subscribe(topic + '/temperature/set')

def extract_device_address_from_unique_id(unique_id):
	split = re.search(DEVICE_REGEX, unique_id).groups()
	addr_size = 2
	if 'i2c' == split[0]:
		addr_size = 3

	address = '.'.join(split[:addr_size])
	conf = split[addr_size]
	host = '_'.join(split[addr_size+1:])

	return address, conf, host