from eventlet import import_patched
from hardware import deviceManager
from logger import log

#flask_mqtt = import_patched('flask_mqtt')
import flask_mqtt

def canonize_name(name):
	return name.replace('.', '_').replace('-', '_').lower()

class Mqtt:
	def __init__(self):
		self.mqtt = flask_mqtt.Mqtt(connect_async=True)

		self.on_connect = self.mqtt.on_connect
		self.on_message = self.mqtt.on_message
		self.on_log = self.mqtt.on_log
		self.publish = self.mqtt.publish
		self.subscribe = self.mqtt.subscribe
		
	def init_app(self, app):
		broker = app.config['MQTT_BROKER_URL']
		if broker is None or len(broker) == 0 or broker == 'None' or broker == '{MQTT_BROKER_URL}':
			log.warning('MQTT Broker not configured')
			return

		self.mqtt.init_app(app)
		self.base_topic = app.config['MQTT_BASE_TOPIC']
		self.hostname = app.config['HOSTNAME']
		self.version = app.config['UNIO_VERSION']

		from . import controllers
		controllers.configure(self.mqtt)

		input_pins = deviceManager.get_all_input_pins()
		for pin in input_pins:
			pin.when_pressed = self._on_button_toggled
			pin.when_released = self._on_button_toggled

	def topic(self, component, board, pin, device=None):
		node = canonize_name(self.hostname)
		dev_id = self.create_device_unique_id(board, pin, device)
		return '{}/{}/{}/{}'.format(self.base_topic, component, node, dev_id)

	def board_hw_info(self, board):
		return {
			"ids": self.create_board_unique_id(board),
			"name": board.name,
			"sw": self.version,
			"mdl": board.model,
			"mf": board.manufacturer,
			"via_device": self.hostname
		}

	def create_board_unique_id(self, board):
		address = board.address if hasattr(board, 'address') else None

		if address is None:
			return canonize_name('{}_{}_{}'.format(board.model, board.host, self.hostname))
		else:
			return canonize_name('{}_{}_{}_{}'.format(board.model, address, board.host, self.hostname))

	def create_device_unique_id(self, board, pin, device=None):
		dev_id = pin
		if device is not None:
			dev_id = dev_id + '_' + device.conf
		return canonize_name(dev_id + '_' + board.host + '_' + self.hostname)

	def _on_button_toggled(self, button):
		controllers.button.handle_hardware_state_trigger(button.address, button.status)

mqtt = Mqtt()

@mqtt.on_log()
def handle_logging(client, userdata, level, buf):
	log.debug("MQTT Log [{}]: {}".format(level, buf))
