from irslinger.pyslinger import IR
from logger import log

from hardware.remotes import string_inverse, bit_inverse, count_ones, convert_int_to_binary_string

class ACRemote(IR):
	def __init__(self, output_pin, host='localhost'):
		self.protocol_config = {
			'duty_cycle': 0.5,
			'frequency': 38000,
			'trailing_pulse': True,
			'leading_pulse_duration': [674, 3043],
			'leading_gap_duration': [17779, 8914],
			'separator_pulse_duration': [568, 3039],
			'separator_gap_duration': [2931, 8915],
			'one_pulse_duration': 558,
			'one_gap_duration': 1464,
			'zero_pulse_duration': 558,
			'zero_gap_duration': 436
		}
		self.host = host
		self.output_pin = output_pin
		log.debug("Output Pin = {}".format(output_pin))

	def update(self, prev_status, status, fan_speed, mode, temperature, swing=False, sleep=False):
		#self.ir = IR(26, "NEC", self.protocol_config)

		self.ir = IR(self.output_pin, "NEC", self.protocol_config, host=self.host)
		
		log.debug("processing ir command with values status={}, fan_speed={}, mode={}, temperature={}".format(status, fan_speed, mode, temperature))

		byte0 = "01000000"
		byte2to5 = "11110000 00000000 00000000 00000000"
		byte6 = "0000" + self.process_power(status) + "11"
		byte1 = "0100" + self.process_checksum1(byte0+byte2to5+byte6)
		
		byte7 = "10000000"
		byte9 = "01111111" #SWING
		byte10 = "10001110"
		byte11 = "0000" + self.process_temperature(temperature, mode)
		byte12 = self.process_fan_speed(fan_speed) + self.process_mode(mode) + "0"
		byte13 = byte6

		overflow,checksum = self.process_checksum2(byte7+byte9+byte10+byte11+byte12+byte13)
		byte8 = "0100" + checksum

		byte9 = byte9 if not overflow else "11111111"

		power_block = "10000000 01001011 11110000 00000000 00000000 00000000 00000000"
		
		command = byte0 + " " + byte1 + " " + byte2to5 + " " + byte6
		command += " | " + power_block if prev_status != status else ""
		command += " | " + byte7 + " " + byte8 + " " + byte9 + " " + byte10 + " " + byte11 + " " + byte12 + " " + byte13

		log.debug("Sending infra red code: " + command)

		self.ir.send_code(command)
		self.ir.gpio.stop()

	def process_fan_speed(self, fan_speed):
		if fan_speed == "AUTO":
			return "1000"
		elif fan_speed == "LOW":
			return "1010"
		elif fan_speed == "MED" or fan_speed == "MEDIUM":
			return "1001"
		elif fan_speed == "HIGH":
			return "1111"

		log.error("Unknown fan_speed: " + fan_speed)
		return "1000"

	def process_mode(self, mode):
		if mode == "HEAT":
			return "001" #0x4
		elif mode == "ICE" or mode == "COOL":
			return "100" #0x1
		elif mode == "DRY":
			return "010" #0x2
		elif mode == "FAN":
			return "110" #0x3
		elif mode == "AUTO":
			return "000"

		log.error("Unknown mode: " + mode)
		return "000"

	def process_temperature(self, temperature, mode):
		#if mode == 'AUTO':
		#	temperature = 16
		#elif mode not in ('ICE', 'COOL'):
		#	temperature = 24

		temperature = int(temperature)

		binary = "{0:04b}".format(temperature - 16)	
		return string_inverse(binary)

	def process_power(self, status):
		return "11" if status else "00"

	def process_swing(self, swing):
		return "11110101" if swing else "01111111"

	def process_sleep(self, sleep):
		pass

	def process_checksum1(self, string):
		result = string.count('0')
		result = result % 15
		result = result if result > 0 else 15
		result = convert_int_to_binary_string(result).rjust(4, '0')
		result = string_inverse(result)
		
		return result

	def process_checksum2(self, string):
		overflow = False
		result = string.count('0')
		#if result > 30:
		#	overflow = True
		
		result = result % 15
		result = result if result > 0 else 15
		result = convert_int_to_binary_string(result).rjust(4, '0')
		result = string_inverse(result)
		
		return overflow,result