from irslinger.pyslinger import IR
from logger import log

from hardware.remotes import string_inverse, convert_int_to_binary_string, bit_inverse

class ACRemote(IR):
	def __init__(self, output_pin, host='localhost'):
		self.protocol_config = {
			'duty_cycle': 0.5,
			'frequency': 38000,
			'trailing_pulse': True,
			'leading_pulse_duration': 3060,
			'leading_gap_duration': 2710,
			'separator_pulse_duration': 3108,
			'separator_gap_duration': 2990,
			'one_pulse_duration': 555,
			'one_gap_duration': 1634,
			'zero_pulse_duration': 562,
			'zero_gap_duration': 533
		}
		self.host = host
		self.output_pin = output_pin
		log.debug("Output Pin = {}".format(output_pin))

	def update(self, prev_status, status, fan_speed, mode, temperature):
		#self.ir = IR(26, "NEC", self.protocol_config)
		self.ir = IR(self.output_pin, "NEC", self.protocol_config, host=self.host)

		log.debug("processing ir command with values status={}, fan_speed={}, mode={}, temperature={}".format(status, fan_speed, mode, temperature))
		
		byte1to2 = "| 00001110 11110001"

		byte3 = self.process_mode(mode, status)
		byte3 += self.process_checksum(byte3)

		byte4 = self.process_fan_speed(fan_speed, mode)
		byte4 += self.process_checksum(byte4)

		byte5 = self.process_swing(False)
		byte5 += self.process_checksum(byte5)

		byte6 = self.process_temperature(temperature)
		byte6 += self.process_checksum(byte6)

		byte7 = "00001111"

		command = byte1to2 + " " + byte3 + " " + byte4 + " " + byte5 + " " + byte6 + " " + byte7

		command += " |* " + command

		log.debug("Sending infra red code: " + command)

		self.ir.send_code(command)
		self.ir.gpio.stop()

	def process_fan_speed(self, fan_speed, mode):
		if mode == 'DRY':
			return "0011"

		if fan_speed == "AUTO":
			return "0000"
		elif fan_speed == "HIGH":
			return "0001"
		elif fan_speed == "MED" or fan_speed == "MEDIUM":
			return "0010"
		elif fan_speed == "LOW":
			return "0011"

		print("Warning: unknown fan_speed: " + fan_speed)
		return "0000"

	def process_power(self, status):
		log.debug("Processing power: {}".format(status))
		return "01111111" if status else "01000000"

	def process_mode(self, mode, status):
		if not status:
			mode == "0000"

		if mode == "DRY":
			return string_inverse("0001")
		elif mode == "AUTO" or mode == "ICE" or mode == "COOL":
			return string_inverse("0010")
		elif mode == "FAN" or mode == "HEAT":
			return string_inverse("0011")

		print("Warning: unknown mode: " + mode)
		return "0010"

	def process_swing(self, swing):
		return "0100" if swing else "0000"

	def process_temperature(self, temperature):
		temperature = int(temperature)

		binary = "{0:04b}".format(temperature - 16)

		return binary

	def process_checksum(self, nibble):
		return bit_inverse(nibble)