from irslinger.pyslinger import IR
from logger import log

from hardware.remotes import string_inverse, convert_int_to_binary_string

class ACRemote(IR):
	def __init__(self, output_pin, host='localhost'):
		self.protocol_config = {
			'duty_cycle': 0.5,
			'frequency': 38000,
			'trailing_pulse': True,
			'separator_pulse_duration': 0,
			'separator_gap_duration': 0,
			'leading_pulse_duration': 8467,
			'leading_gap_duration': 4132,
			'one_pulse_duration': 585,
			'one_gap_duration': 1547,
			'zero_pulse_duration': 587,
			'zero_gap_duration': 505
		}
		self.host = host
		self.output_pin = output_pin
		log.debug("Output Pin = {}".format(output_pin))

	def update(self, prev_status, status, fan_speed, mode, temperature):
		#self.ir = IR(26, "NEC", self.protocol_config)

		self.ir = IR(self.output_pin, "NEC", self.protocol_config, host=self.host)

		log.debug("processing ir command with values status={}, fan_speed={}, mode={}, temperature={}".format(status, fan_speed, mode, temperature))

		byte1 = "01101010"
		
		byte2 = self.process_temperature(temperature) + "110"
		nibble3 = str(byte2[:4])
		nibble4 = str(byte2[4:])

		byte3to4 = "00000000 00000000"

		nibble9 = self.process_fan_speed(fan_speed)
		nibble10 = self.process_mode(mode)
		byte5 = nibble9 + nibble10

		byte6 = self.process_swing(0, status, mode) + self.process_power(status)
		nibble11 = str(byte6[:4])
		nibble12 = str(byte6[4:])

		byte7 = self.process_sleep(False) + "0000"

		byte8 = "00000000 |"

		byte9 = "0000000" + self.process_turbo(False)
		
		byte10to11 = "00000000 00000000"

		byte12to14 = "00000000 00000000 00110000" # TIME (ss mm HH)

		constantNibblesSum = 23

		byte15 = self.process_checksum(constantNibblesSum, nibble3, nibble4, nibble9, nibble10, nibble11, nibble12)

		command = byte1 + " " + byte2 + " " + byte3to4 + " " + byte5 + " " + byte6 + " " + byte7 + " " + byte8 + " " + byte9 + " " + byte10to11 + " " + byte12to14 + " " + byte15

		log.debug("Sending infra red code: " + command)

		self.ir.send_code(command)
		self.ir.gpio.stop()

	def process_fan_speed(self, fan_speed):
		if fan_speed == "AUTO":
			return "0000"
		elif fan_speed == "LOW":
			return "0100"
		elif fan_speed == "MED" or fan_speed == "MEDIUM":
			return "1100"
		elif fan_speed == "HIGH":
			return "1000"

		print("Warning: unknown fan_speed: " + fan_speed)
		return "0000"

	def process_power(self, status):
		log.debug("Processing power: {}".format(status))
		return "000" if status else "011"

	def process_turbo(self, turbo):
		return "1" if turbo else "0"

	def process_lock(self, lock):
		return "0001" if lock else "0000"

	def process_sleep(self, sleep):
		return "0001" if sleep else "0000"

	def process_mode(self, mode):
		if mode == "HEAT":
			return "1000"
		elif mode == "ICE" or mode == "COOL":
			return "0100"
		elif mode == "DRY":
			return "1100"
		elif mode == "AUTO":
			return "0010"
		elif mode == "FAN":
			return "1010"

		print("Warning: unknown mode: " + mode)
		return "0010"

	def process_swing(self, swing, status, mode):
		if not status:
			return "00000"

		bit1 = "01"

		if swing == 0: #AUTO
			return bit1 + "011"
		elif swing == 1:
			return bit1 + "100"
		elif swing == 2:
			return bit1 + "010"
		elif swing == 3:
			return bit1 + "110"
		elif swing == 4:
			return bit1 + "001"
		elif swing == 5:
			return bit1 + "101"

		print("Warning: unknown mode: " + mode)
		return bit1 + "011"

	def process_temperature(self, temperature):
		temperature = int(temperature)
		binary = "{0:05b}".format(temperature - 4)

		return string_inverse(binary)

	def process_checksum(self, constantNibblesSum, nibble3, nibble4, nibble9, nibble10, nibble11, nibble12):

		result = constantNibblesSum + int(string_inverse(nibble3), 2) + int(string_inverse(nibble4), 2) + int(string_inverse(nibble9), 2) + int(string_inverse(nibble10), 2) + int(string_inverse(nibble11), 2) + int(string_inverse(nibble12), 2)
		
		return string_inverse(convert_int_to_binary_string(result)).ljust(8, '0')

