from irslinger.pyslinger import IR
from logger import log

from hardware.remotes import string_inverse, convert_int_to_binary_string

class ACRemote(IR):
	def __init__(self, output_pin, host='localhost'):
		self.protocol_config = {
			'duty_cycle': 0.5,
			'frequency': 38000,
			'trailing_pulse': True,
			'separator_pulse_duration': 510,
			'separator_gap_duration': 10355,
			'one_pulse_duration': 460,
			'one_gap_duration': 1290,
			'zero_pulse_duration': 510,
			'zero_gap_duration': 360
		}
		self.host = host
		self.output_pin = output_pin
		log.debug("Output Pin = {}".format(output_pin))

	def update(self, prev_status, status, fan_speed, mode, temperature, swing=False, sleep=False):
		#self.ir = IR(26, "NEC", self.protocol_config)
		self.ir = IR(self.output_pin, "NEC", self.protocol_config, host=self.host)

		log.debug("processing ir command with values status={}, fan_speed={}, mode={}, temperature={}".format(status, fan_speed, mode, temperature))

		byte0to7 = "01000000 00000100 00000111 00100000 00000000 00000000 00000000 01100000"
		byte8to12 = "01000000 00000100 00000111 00100000 00000000"
		byte13 = self.process_power(prev_status != status) + self.process_mode(mode) + "000"
		byte14 = self.process_temperature(temperature)
		byte15 = " | 00000001"
		byte16 = "1111" + self.process_fan_speed(fan_speed)
		byte17to25 = "00000000 00000000 01110000 00000111 00000000 00000000 10000001 00000000 00000000"

		frame2 = byte8to12 + " " + byte13 + " " + byte14 + " " + byte15 + " " + byte16 + " " + byte17to25
		checksum = self.process_checksum(frame2)

		command = byte0to7 + " |* " + frame2 + " " + checksum

		log.debug("Sending infra red code: " + command)

		self.ir.send_code(command)
		self.ir.gpio.stop()

	def process_fan_speed(self, fan_speed):
		if fan_speed == "AUTO":
			return "0101"
		elif fan_speed == "LOW":
			return "1100"
		elif fan_speed == "MED" or fan_speed == "MEDIUM":
			return "1010"
		elif fan_speed == "HIGH":
			return "1110"

		log.error("Unknown fan_speed: " + fan_speed)
		return "0101"

	def process_power(self, status):
		return "01" if status else "00"

	def process_mode(self, mode):
		if mode == "ICE" or mode == "COOL":
			return "111"
		elif mode == "DRY":
			return "101"
		elif mode == "AUTO":
			return "100"

		log.error("Unknown mode: " + mode)
		return "100"

	def process_temperature(self, temperature):
		temperature = int(temperature)

		binary = "{0:06b}".format(temperature)	
		return "0" + binary + "00"

	def process_swing(self, swing):
		pass

	def process_sleep(self, sleep):
		pass

	def process_checksum(self, frame):
		checksum = 0
		for byte in frame.split():
			checksum += int(string_inverse(byte), 2)

		checksum = checksum % 256

		return string_inverse(convert_int_to_binary_string(checksum).rjust(8, '0'))
