from irslinger.pyslinger import IR
from logger import log
from hardware.remotes import convert_hex_to_binary_string

class LircRemote():
	def __init__(self, conf_name, output_pin, host='localhost'):
		self.commands = { }
		self.protocol_config = {
			'duty_cycle': 0.5,
			'frequency': 38000,
			#'trailing_pulse': True
		}

		self._parse(conf_name)

		self.host = host
		self.output_pin = output_pin
		log.debug("Output Pin = {}".format(output_pin))

	def update(self, command_id):
		# self.ir = IR(26, "NEC", self.protocol_config)

		log.debug("processing ir lirc command with code={}".format(command_id))
		command = self.commands.get(command_id)

		if command is None:
			log.error("IR command not found: {}".format(command_id))
			return

		if command['raw'] is not None:
			self.ir = IR(self.output_pin, "RAW", self.protocol_config)
			self.ir.send_code(command['raw'])
		else:
			cmd_str = ''
			if hasattr(self, 'header'):
				cmd_str = self.header['bin'] + ' '
			cmd_str += command['bin']

			log.debug("Sending infra red code: " + cmd_str)

			self.ir = IR(self.output_pin, "NEC", self.protocol_config, host=self.host)
			self.ir.send_code(cmd_str)

		self.ir.gpio.stop()

	def _parse(self, conf_name):
		with open('app/remote-confs/{}.conf'.format(conf_name), 'r') as conf_file:
			reading_commands = False
			reading_raw_commands = False
			processing_raw = False
			
			command_id = ''
			for line in conf_file:
				line = line.strip()

				if line.startswith('#') or len(line) == 0:
					continue

				if line.startswith('begin codes'):
					reading_commands = True
				elif line.startswith('end codes'):
					reading_commands = False
				if line.startswith('begin raw_codes'):
					reading_raw_commands = True
				elif line.startswith('end raw_codes'):
					reading_raw_commands = False
				elif reading_commands:
					self._parse_command(line)
				elif reading_raw_commands:
					if line.startswith('name'):
						command_id = line.split()[1]
						self.commands[command_id] = {
							'raw': []
							}
					else:
						self._parse_raw_command(command_id, line)
				elif line.startswith('header'):
					self._parse_header(line)
				elif line.startswith('one'):
					self._parse_one(line)
				elif line.startswith('zero'):
					self._parse_zero(line)
				elif line.startswith('ptrail'):
					self._parse_trailing_pulse(line)
				elif line.startswith('pre_data'):
					self._parse_pre_data(line)

	def _parse_command(self, command):
		command_id, code = command.split()

		if code.startswith('0x'):
			self.commands[command_id] = {
				'hex': code,
				'bin': convert_hex_to_binary_string(code),
				'raw': None
			}
		#else: is it possible other formats? binary for example?

	def _parse_raw_command(self, command_id, line):
		for code in line.split():
			self.commands.get(command_id)['raw'].append(code)

	def _parse_header(self, command):
		command, header_pulse, header_gap = command.split()

		self.protocol_config['leading_pulse_duration'] = int(header_pulse)
		self.protocol_config['leading_gap_duration'] = int(header_gap)

	def _parse_one(self, command):
		command, one_pulse, one_gap = command.split()

		self.protocol_config['one_pulse_duration'] = int(one_pulse)
		self.protocol_config['one_gap_duration'] = int(one_gap)

	def _parse_zero(self, command):
		command, zero_pulse, zero_gap = command.split()

		self.protocol_config['zero_pulse_duration'] = int(zero_pulse)
		self.protocol_config['zero_gap_duration'] = int(zero_gap)

	def _parse_trailing_pulse(self, command):
		command, pulse = command.split()

		#TODO Modify pyslinger to support custom trailing pulse duration (currently only true/false)
		self.protocol_config['trailing_pulse'] = True

	def _parse_pre_data(self, command):
		command, header = command.split()

		self.header = {
			'hex': header,
			'bin': convert_hex_to_binary_string(header)
		}
