from irslinger.pyslinger import IR
from logger import log

from hardware.remotes import string_inverse, convert_int_to_binary_string

class ACRemote(IR):
	def __init__(self, output_pin, host='localhost'):
		self.protocol_config = {
			'duty_cycle': 0.5,
			'frequency': 38000,
			'trailing_pulse': True,
			'leading_pulse_duration': 3429,
			'leading_gap_duration': 9782,
			'one_pulse_duration': 572,
			'one_gap_duration': 1484,
			'zero_pulse_duration': 572,
			'zero_gap_duration': 466
		}
		self.host = host
		self.output_pin = output_pin
		log.debug("Output Pin = {}".format(output_pin))

	def update(self, prev_status, status, fan_speed, mode, temperature, swing=False, sleep=False):
		#self.ir = IR(26, "NEC", self.protocol_config)
		self.ir = IR(self.output_pin, "NEC", self.protocol_config, host=self.host)

		log.debug("processing ir command with values status={}, fan_speed={}, mode={}, temperature={}".format(status, fan_speed, mode, temperature))

		nibble0 = "1000"
		nibble1 = "1000"
		
		nibble2 = self.process_power(status) + "00"
		nibble3 = self.process_mode(not (prev_status != status), mode)

		nibble4 = self.process_temperature(status, temperature)
		nibble5 = self.process_fan_speed(status, fan_speed)
		
		nibble6 = self.process_checksum(nibble0, nibble1, nibble2, nibble3, nibble4, nibble5)

		command = nibble0 + " " + nibble1 + " " + nibble2 + " " + nibble3 + " " + nibble4 + " " + nibble5 + " " + nibble6

		log.debug("Sending infra red code: " + command)

		self.ir.send_code(command)
		self.ir.gpio.stop()

	def process_power(self, status):
		return "00" if status else "11"

	def process_fan_speed(self, status, fan_speed):
		if not status or fan_speed == "AUTO":
			return "0101"
		elif fan_speed == "LOW":
			return "0000"
		elif fan_speed == "MED" or fan_speed == "MEDIUM":
			return "0010"
		elif fan_speed == "HIGH":
			return "0100"

		log.error("Unknown fan_speed: " + fan_speed)
		return "0000"

	def process_mode(self, status, mode):
		if not status:
			return "0000"

		if mode == "AUTO":
			return "1011"
		elif mode == "ICE" or mode == "COOL":
			return "1000"
		elif mode == "DRY":
			return "1001"
		elif mode == "FAN":
			return "1010"

		log.error("Unknown mode: " + mode)
		return "0000"

	def process_temperature(self, status, temperature):
		temperature = int(temperature)

		binary = "{0:04b}".format(temperature - 15)	
		return binary if status else "0000"

	def process_swing(self, swing):
		raise Exception

	def process_sleep(self, sleep):
		raise Exception

	def process_checksum(self, nibble0, nibble1, nibble2, nibble3, nibble4, nibble5):
		nibble0 = int(nibble0, 2)
		nibble1 = int(nibble1, 2)
		nibble2 = int(nibble2, 2)
		nibble3 = int(nibble3, 2)
		nibble4 = int(nibble4, 2)
		nibble5 = int(nibble5, 2)

		checksum = nibble0 + nibble1 + nibble2 + nibble3 + nibble4 + nibble5
		checksum = convert_int_to_binary_string(checksum).rjust(4, '0')
		return checksum[len(checksum)-4:len(checksum)]