from irslinger.pyslinger import IR
from logger import log

from hardware.remotes import string_inverse, convert_int_to_binary_string

class ACRemote(IR):
	def __init__(self, output_pin, host='localhost'):
		self.protocol_config = {
			'duty_cycle': 0.5,
			'frequency': 38000,
			'trailing_pulse': True,
			'separator_pulse_duration': 650,
			'separator_gap_duration': 20000,
			'one_pulse_duration': 650,
			'one_gap_duration': 1650,
			'zero_pulse_duration': 650,
			'zero_gap_duration': 550
		}
		self.host = host
		self.output_pin = output_pin
		log.debug("Output Pin = {}".format(output_pin))

	def update(self, prev_status, status, fan_speed, mode, temperature, swing=False, sleep=False):
		#self.ir = IR(26, "NEC", self.protocol_config)
		self.ir = IR(self.output_pin, "NEC", self.protocol_config, host=self.host)

		log.debug("processing ir command with values status={}, fan_speed={}, mode={}, temperature={}".format(status, fan_speed, mode, temperature))

		byte0 = self.process_mode(mode)
		byte0 += self.process_power(status)
		byte0 += self.process_fan_speed(fan_speed)
		byte0 += self.process_swing(swing)
		byte0 += self.process_sleep(sleep)

		byte1 = self.process_temperature(temperature)
		byte1 += "0000"

		byte2 = "00000110"
		byte3 = "00001010"
		
		byte5 = "00000000"
		byte6 = "00000100"
		byte7 = "00000000"
		byte8 = "0000" + self.process_checksum(byte0, byte1, byte2, byte3, byte5, byte6, byte7)

		command = byte0 + " " + byte1 + " " + byte2 + " " + byte3 + " " +  "010 | " + byte5 + byte6 + byte7 + byte8

		log.debug("Sending infra red code: " + command)

		self.ir.send_code(command)
		self.ir.gpio.stop()

	def process_fan_speed(self, fan_speed):
		if fan_speed == "AUTO":
			return "00"
		elif fan_speed == "LOW":
			return "10"
		elif fan_speed == "MED" or fan_speed == "MEDIUM":
			return "01"
		elif fan_speed == "HIGH":
			return "11"

		log.error("Unknown fan_speed: " + fan_speed)
		return "00"

	def process_power(self, status):
		if status:
			return "1"
		else:
			return "0"

	def process_mode(self, mode):
		if mode == "HEAT":
			return "001" #0x4
		elif mode == "ICE" or mode == "COOL":
			return "100" #0x1
		elif mode == "DRY":
			return "010" #0x2
		elif mode == "FAN":
			return "110" #0x3

		log.error("Unknown mode: " + mode)
		return "0000"

	def process_temperature(self, temperature):
		temperature = int(temperature)

		binary = "{0:04b}".format(temperature - 16)	
		return string_inverse(binary)

	def process_swing(self, swing):
		if swing:
			return "1"
		else:
			return "0"

	def process_sleep(self, sleep):
		if sleep:
			return "1"
		else:
			return "0"

	def process_checksum(self, byte0, byte1, byte2, byte3, byte5, byte6, byte7):
		byte0 = int(string_inverse(byte0), 2)
		byte1 = int(string_inverse(byte1), 2)
		byte2 = int(string_inverse(byte2), 2)
		byte3 = int(string_inverse(byte3), 2)
		byte5 = int(string_inverse(byte5), 2)
		byte6 = int(string_inverse(byte6), 2)
		byte7 = int(string_inverse(byte7), 2)

		checksum = ((byte0 & 0x0F) + (byte1 & 0x0F) + (byte2 & 0x0F) + (byte3 & 0x0F) + ((byte5 & 0xF0) >> 4) + ((byte6 & 0xF0) >> 4) + ((byte7 & 0xF0) >> 4) + 10) & 0x0F

		return string_inverse(convert_int_to_binary_string(checksum).rjust(4, '0'))